// ====================================
//  NAME    : fwUkl1RegisterInterface
// 
//  VERSION : 3
// 
//  REVISION: 1
// 
//  DATE    : 2006/10/12
// 
//  RELEASE : 2010/01/20
// 
//  PLATFORM: PVSS II
// 
//  AUTHOR  : Gareth Rogers
// 
//  EMAIL   : rogers@hep.phy.cam.ac.uk
// ====================================

/* \mainpage UKL1REG Contains the function necessary to write/read or save/load from a register.
 *
 */

// ====================
//  #USES DECLARATIONS
// ====================

// ================
//  UKL1 INTERFACE
// ================

/** @defgroup SectionInterface Read/write interface.
 *   Provides access to the either the UKL1 hardware or the recipe settings for that board.
 *
 *   In both the hardware access and register access cases maps are used to associated the
 *   register name to the data read or to be written. The data returned or take by the functions
 *   is intended to be as human reable as possible and can be directly displayed on a widget
 *   or taken directly from a widget. The for of the text is described in the fwUkl1DP control
 *   library.
 *
 *   The recipe read and writes differ from hardware writes as they are grouped by recipe types.
 *   Not all hardware registers are present in all recipe types and as a result not all the
 *   hardware registers can be read from the all, if any, of the recipes.
 *   
 */
// @{

/*!
 * Writes to a given set of the UKL1 registers on a given UKL1 board.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  inData A mapping associating the register name (string) with the data that is to be written, which is
 *           a dyn_string. The data should be in element FWUKL1_MAP_REG_DATA of the dyn_string.
 * @param  verifyWrite If this is set to TRUE write will be verified to be successful by reading back the registers
 *           and comparing it to that given.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 */
void fwUkl1_write(const string &ukl1Name, mapping inData, bool verifyWrite, dyn_string &exceptionInfo) {
  // Keep track of the exception info.
  int exInfoSize = dynlen(exceptionInfo);
  // Check we have a UKL1 name, this really upsets things if we don't.
  if (""==ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_write(): UKL1 name was empty, cannot proceed.", "2");
    return;
  }// if(""==ukl1Name)

  // Converts the data given as strings in the calling arguments to byte arguments required by the fwCcpc_write.
  // Also it creates the masks required for each write and pads the register names, such that they refer to the appropriate
  // data point elements.
  mapping outData;
  _fwUkl1_parseRegistersForSet(inData, outData, FWUKL1_HW_WRITE, exceptionInfo);
  // Check that we were actually able to parse the data. This is not necessarily fatal.
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_write(): Failed to parse input data, not all the data was written.", "2", exInfoSize);

  // This is the form of the data that we require to write.
  dyn_string regNames;
  dyn_dyn_char ddcData;
  dyn_dyn_char masks;
  dyn_string writeFunctions;
  // although we get an ordered list of the hardware types for each register we are not interested.
  dyn_int junkHwRegTypes;
  _fwUkl1_parseWriteMapping(outData, regNames, ddcData, masks, writeFunctions, junkHwRegTypes, exceptionInfo);
  // An error from here is not necessarily fatal.
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_write(): Failed to convert all the data from that was parsed to a form that can be written. Some registers will not be written.", "2", exInfoSize);

  _fwUkl1_writeSwitch(ukl1Name, regNames, writeFunctions, ddcData, masks, verifyWrite, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_write(): Failed to perform so/all of the writes to the hardware. Not all of the settings will have been updated.", "2", exInfoSize);

  // Done.
  return;
}// fwUkl1_write()

/*!
 * Reads from a given set of the UKL1 registers on a given UKL1 board.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  regList List of abstract registers that are to read.
 * @param  forStatus If the values read back are to be used to display the status of the register then it
 *           might have a different form to that to be used for the configuration. If TRUE then the
 *           values read back might not be able to be written to the configuration register, FALSE they
 *           must be.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return mapping Returns a mapping with the register name (string) as the key
 *           and a dyn_string string as the value. The dyn_string will contain the register value
 *           in the element FWUKL1_MAP_REG_DATA and a colour that identifies if the value is in
 *           a sensible range in the element FWUKL1_MAP_XTR_DATA. The input map should contain as keys the
 *           registers that are to be read. The colour FwStateAttention1 is used to indicate if the
 *           read failed and FwStateAttention2 if the value is out of bounds.
 */
mapping fwUkl1_read(const string &ukl1Name, const dyn_string &regList, bool forStatus, dyn_string &exceptionInfo) {
  // Keep track of the exception info.
  int exInfoSize = dynlen(exceptionInfo);
  // Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_read(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }// if(""==ukl1Name)

  // This is the form of the data that we require to read.
  dyn_string regNames;
  dyn_string readFunctions;
  dyn_int regTypes;
  // Just give read mapping the list of registers that are to be read.
  // We pass FALSE as we will want any alternative hardware registers.
  // The alternative registers are for those cases where the read register
  // does not match the write register. It is none standard because the DB
  // always uses the same registers for read and write which must be the
  // same as the hardware writes.
  const string registerType = (forStatus ? FWUKL1_HW_STATUS_READ:FWUKL1_HW_CONFIG_READ);
  _fwUkl1_parseReadMapping(regList, registerType, regNames, readFunctions, regTypes, exceptionInfo);
  // An error from here is not necessarily fatal.
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_read(): Failed to determine the names of the some/all of the hardware registers to read from. Read operation will cotinue, but not all the expected data will be returned.", "2", exInfoSize);

  // Now we call the function that will decide how to read the data based on the information we have just received.
  // This is the map to associate the raw data with what ever extra information we are interested in passing around.
  // At the moment, none!
  mapping rawData;
  _fwUkl1_readSwitch(ukl1Name, rawData, regNames, readFunctions, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_read(): Some or all of the registers were not read back from the hardware.", "2", exInfoSize);
  
  // Now we need to parse this data. Do it regardless of whether there was a failure, it will sort out what happened with the failures.
  mapping outData = _fwUkl1_parseRegistersForGet(regList, rawData, registerType, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_read(): Failed to parse all of the raw data read from the hardware, so registers probably failed to read correctly.", "2", exInfoSize);
  // Done.
  return outData;
}// fwUkl1_read()

/*!
 * This functions works much like read, it takes a list of abstract registers and reads their values from the hardware, but it does not
 * parse this data. Instead it just updates the HW DPs and lets a call back function handle them if necessary.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  regList List of abstract registers that are to read.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 */
void fwUkl1_refresh(const string &ukl1Name, const dyn_string &regList, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  // First loop over the abstract registers and get the related hardware function to use to read that register.
  // Just give read mapping the list of registers that are to be read.
  dyn_string hwRegNames;
  dyn_string readFunctions;
  dyn_int regTypes;
  // A refresh can only be used to update the status registers.
  _fwUkl1_parseReadMapping(regList, FWUKL1_HW_STATUS_READ, hwRegNames, readFunctions, regTypes, exceptionInfo);
  // An error from here is not necessarily fatal.
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_refresh(): Failed to determine the names of the some/all of the hardware registers to read from. Read operation will cotinue, but not all the expected data will be returned.", "2", exInfoSize);

  // Now we have to split the data to be read up such that it is read with the appropriate function.
  // This is the map to associate the raw data with what ever extra information we are interested in passing around.
  // At the moment, none!
  mapping rawData;
  _fwUkl1_readSwitch(ukl1Name, rawData, hwRegNames, readFunctions, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_refresh(): Some or all of the registers were not read back from the hardware.", "2", exInfoSize);
  
  // Done.
  return;
}// fwUkl1_refresh()

/*!
 * Parses the data from a list of DPEs for the given abstract register name.
 *
 * @param  absRegNames Abtract registers that the new values are to be retrieved for.
 * @param  dpeNames Names of the DPE that the updated hardware register data has come from.
 * @param  ddcNewValues The updated data in the hardware register(s) that need to be parsed.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return mapping The parsed data for that abstract register. In the same form as returned from fwUkl1_read.
 *
 * This function is meant to be used with callback functions that are set up for panel displays or other automated
 * responses to DPE value changes.
 *
 * DPE values passed must be from the hardware register associated with the abstract register given otherwise
 * the return value cannot be given.
 */
mapping fwUkl1_readDPEs(dyn_string absRegNames, dyn_string dpeNames, dyn_dyn_char ddcNewValues, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  // Need to put the DPE new data into a map to link it with the hardware register name.
  mapping inData;
  
  // Need to get the hardware name for each register.
  const int numDpes = dynlen(dpeNames);
  // Get the UKL1 name. This only needs to be done once.
  int endOfUkl1Name;
  for (int i=1; i<=numDpes; ++i) {
    // The UKL1 name is up to the first full stop.
    endOfUkl1Name = strpos(dpeNames[i], ".");
    // Need to get ride of anything after and including readings.
    const int startOfReadings = strpos(dpeNames[i], ".readings");
    // Now the hardware register name.
    const string hwReg = substr(dpeNames[i], endOfUkl1Name, startOfReadings-endOfUkl1Name);
    // Now setup the data.
    inData[hwReg] = makeDynMixed();
    inData[hwReg][FWUKL1_MAP_REG_DATA] = ddcNewValues[i];
  }// for i
  inData[FWUKL1_NULL_HWREG] = makeDynMixed();
  inData[FWUKL1_NULL_HWREG][FWUKL1_MAP_REG_DATA] = makeDynChar();
  mapping outData = _fwUkl1_parseRegistersForGet(absRegNames, inData, FWUKL1_HW_STATUS_READ, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_readDPEs(): Failed to parse the new register value for "+dynStringToString(absRegNames,", ")+".", "2", exInfoSize);
  // Done.
  return outData;
}// fwUkl1_readDPEs()

/*!
 * This saves a given set of registers to the configurationDB for a given UKL1 board.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  inData This is the same as that described in fwUkl1_write() documention.
 * @param  recipeName The name of the recipe to save the data to. It must be of the form "RUN_TYPE|ExtraInfo/FsmAction", where:
 *   \li RUN_TYPE is a descriptiion of the type of run the settings are saved for e.g. PHYSICS, CALIBRATION. All capitals.
 *   \li ExtraInfo is any further detail that is required to specify the name uniquely, this (and the |) are optional
 *         in the event they are present this recipe is searched for. If they are missing the recipe
 *         "RUN_TYPE/FsmAction" is searched for.
 *   \li FsmAction is the action of FSM state transition where the recipe should be loaded e.g. Configure, the first letter only should be capitalised.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 */
void fwUkl1_save(const string &ukl1Name, mapping inData, const string &recipeName, dyn_string &exceptionInfo) {
  // Keep track of the exception info.
  int exInfoSize = dynlen(exceptionInfo);
  // Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_save(): UKL1 name was empty, cannot proceed.", "2");
    return;
  }// if(""==ukl1Name)

  // Determine the recipe type from the name we are given.
  // If we can't determine the type then we can go no further.
  const string recipeType = fwUkl1_getRecipeType(recipeName, exceptionInfo);
  if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_save(): Failed to determine the recipe type from the recipe name '"+recipeName+"'.", "2", exInfoSize))
    return;

  // Converts the data given as strings in the calling arguments to byte arguments required by the fwCcpc_write.
  // Also it creates the masks required for each write and pads the register names, such that they refer to the appropriate
  // data point elements.
  mapping outData;
  _fwUkl1_parseRegistersForSet(inData, outData, FWUKL1_DB_WRITE, exceptionInfo);
  // Check that we were actually able to parse the data.
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_save(): Failed to parse input data, not all the data was written.", "2", exInfoSize);

  // This is the form of the data that we require to write.
  dyn_string regNames;
  dyn_dyn_char ddcData;
  dyn_dyn_char masks;
  // Even though we get an ordered list of write functions these are for
  // the hardware and we always use the same save function so we are not interested.
  dyn_string junkWriteFunctions;
  dyn_int hwRegTypes;
  _fwUkl1_parseWriteMapping(outData, regNames, ddcData, masks, junkWriteFunctions, hwRegTypes, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_save(): Failed to convert all the data from that was parsed to a form that can be written. Some registers will not be written.", "2", exInfoSize);
  
  // The data and registers have been parsed, so let us write it to the recipe.
  // Our save recipe function requires the masks to have already been appended to the data.
  const int numElements = dynlen(ddcData);
  for (int element = 1; element <= numElements; ++element) {
    dynAppend(ddcData[element], masks[element]);
  }// for element
  // Now save the recipe. Domain is empty as this will prevent any name mangling occuring.
  // If we are editing a confDB recipe we will be given the full name anyway.
  _fwUkl1_save(ukl1Name, recipeName, recipeType, regNames, hwRegTypes, ddcData, "", exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_save(): Failed to save recipe for " + ukl1Name + ".", "2", exInfoSize);

  // Done.
  return;
}// fwUkl1_save()

/*!
 * Loads a recipe from a given set of the UKL1 registers on a given UKL1 board.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  regList List of abstract registers that are to read.
 * @param  recipeName The name of the recipe to save the data to. It must be of the form "RUN_TYPE|ExtraInfo/FsmAction", where:
 *   \li RUN_TYPE is a descriptiion of the type of run the settings are saved for e.g. PHYSICS, CALIBRATION. All capitals.
 *   \li ExtraInfo is any further detail that is required to specify the name uniquely, this (and the |) are optional
 *         in the event they are present this recipe is searched for. If they are missing the recipe
 *         "RUN_TYPE/FsmAction" is searched for.
 *   \li FsmAction is the action of FSM state transition where the recipe should be loaded e.g. Configure, the first letter only should be capitalised.
 * @param  domain This is required if the recipe to be loaded has been loaded into the recipe cache by the FSMConfDB. In this case
 *           the name of the recipe will have been mangled. The recipe name given to this function should be the unmangled
 *           name and it will be converted as appropriate. This parameter can also be an empty string if no name mangling is required.
 *           Note in this case there can be no / present in the recipe name, only to delimit the run mode from the command (RUN_MODE/Command).
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return mapping Returns a mapping with the register name (string) as the key
 *           and a dyn_string string as the value. The dyn_string will contain the register value
 *           in the element FWUKL1_MAP_REG_DATA and a colour that identifies if the value is in
 *           a sensible range in the element FWUKL1_MAP_XTR_DATA. The input map should contain as keys the
 *           registers that are to be read. The colour FwStateAttention1 is used to indicate if the
 *           read failed and FwStateAttention2 if the value is out of bounds.
 */
mapping fwUkl1_load(const string &ukl1Name, const dyn_string &regList, const string &recipeName, string domain, dyn_string &exceptionInfo) {
  // Keep track of the exception info.
  int exInfoSize = dynlen(exceptionInfo);
  // Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_load(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }// if(""==ukl1Name)

  // Determine the recipe type from the name we are given.
  // If we can't determine the type then we can go no further.
  const string recipeType = fwUkl1_getRecipeType(recipeName, exceptionInfo);
  if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_load(): Failed to determine the recipe type from the recipe name '"+recipeName+"'.", "2", exInfoSize))
    return;
  // This is the form of the data that we require to read.
  dyn_string hwRegNames;
  dyn_dyn_char ddcData;
  // We will be given the masks back but we don't actually need them.
  dyn_dyn_char junkMasks;
  // This is the read function names that we don't care about as all are loaded with the same function.
  dyn_string junk;
  // We need to know the hardware register types in order to be able to properly retrieve the data from the recipe.
  dyn_int hwRegTypes;
  _fwUkl1_parseReadMapping(regList, FWUKL1_DB_READ, hwRegNames, junk, hwRegTypes, exceptionInfo);
  // An error from here is not necessarily fatal.
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_load(): Failed to determine the names of the some/all of the hardware registers to read from. Read operation will cotinue, but not all the expected data will be returned.", "2", exInfoSize);

  // This is the map to associate the raw data with what ever extra information we are interested in passing around.
  // At the moment, none!
  mapping rawData;
  // Now load them from the DB.
  _fwUkl1_load(ukl1Name, recipeName, recipeType, domain, hwRegNames, hwRegTypes, ddcData, junkMasks, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_load(): Failed to load from some/all of the registers for " + ukl1Name + " with the _fwUkl1_load function.", "1", exInfoSize);
  // Now append the registers to the full lists for later parsing.
  // Do this even if there is an error as we don't know how much failed.
  const int numRegs = dynlen(hwRegNames);
  for (int regNum=1; regNum<=numRegs; ++regNum) {
    const string hwRegName = hwRegNames[regNum];
    rawData[hwRegName] = makeDynMixed();
    rawData[hwRegName][FWUKL1_MAP_REG_DATA] = ddcData[regNum];
  }// regNum
  // Add the null register.
  rawData[FWUKL1_NULL_HWREG] = makeDynMixed();
  rawData[FWUKL1_NULL_HWREG][FWUKL1_MAP_REG_DATA] = makeDynChar();

  // Now we need to parse this data...
  mapping outData = _fwUkl1_parseRegistersForGet(regList, rawData, FWUKL1_DB_READ, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_load(): Failed to parse all of the raw data read from the hardware, so registers probably failed to read correctly.", "2", exInfoSize);
  // Done.
  return outData;
}// fwUkl1_load()

/*!
 * This function is responsible for starting the monitoring of the given list of abstract registers.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  absRegList List of abstract registers to monitor for this UKL1 board.
 * @param  exceptionInfo Error information. No change in size if no exception inforamtion is generated.
 * @return void.
 */
void fwUkl1_startMonitoring(const string &ukl1Name, const dyn_string &absRegList, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  // First loop through the registers and get the hw regs and their monitoring functions.
  const int numRegs = dynlen(absRegList);
  dyn_string monFuncs;
  dyn_string hwRegList;
  for (int regNum=1; regNum<=numRegs; ++regNum) {
    // Now get the hardware register list.
    const dyn_string hwRegs = fwUkl1_getMonitoringRegisterList(absRegList[regNum], exceptionInfo);
    if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_startMonitoring(): Could not identify the corresponding hardware registers to monitor for the register " + absRegList[regNum] + ". This function will not be monitored.", "2", exInfoSize)) {
      // If successful get the monitoring function.
      const string monFunc = fwUkl1_getStartMonitoringFunction(absRegList[regNum], exceptionInfo);
      if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_startMonitoring(): Could not identify the function to use to monitor the register " + absRegList[regNum] + ". This function will not be monitored.", "2", exInfoSize)) {
        for (int i=1; i<=dynlen(hwRegs); ++i) {
          if (0 < dynContains(hwRegList, hwRegs[i])) {
            dynAppend(hwRegList, hwRegs[i]);
            dynAppend(monFuncs, monFunc);
          }// if(0<dynContains(hwRegList,hwRegs[i])
        }// loop over hwRegs
      }// got mon func successfully.
    }// got hw reg successfully.
  }// for regNum
  
  // Now sort the lists for the monitoringSwitch.
  dyn_mixed lists = makeDynMixed(monFuncs, absRegList);
  fwUkl1_sort(lists, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_startMonitoring(): Failed to sort the hardware list, all registers should be monitored but it might take longer to setup.", "2", exInfoSize);
  
  _fwUkl1_monitoringSwitch(ukl1Name, hwRegList, monFuncs, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_startMonitoring(): Failed to start monitoring all of the requested registers.", "2", exInfoSize);
  // Done.
  return;
}// fwUkl1_startMonitoring()

/*!
 * This function is responsible for stopping the monitoring of the given list of abstract registers.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  absRegList List of abstract registers to monitor for this UKL1 board.
 * @param  exceptionInfo Error information. No change in size if no exception inforamtion is generated.
 * @return void.
 */
void fwUkl1_stopMonitoring(const string &ukl1Name, const dyn_string &absRegList, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  // First loop through the registers and get the hw regs and their monitoring functions.
  const int numRegs = dynlen(absRegList);
  dyn_string monFuncs;
  dyn_string hwRegList;
  for (int regNum=1; regNum<=numRegs; ++regNum) {
    // Now get the hardware register list.
    // We want to monitor those registers that are being read from so take the alternative hw reg.
    // Hence we pass false.
    const dyn_string hwRegs = fwUkl1_getAccessFunctions(absRegList[regNum], FWUKL1_HW_STATUS_READ, exceptionInfo);
    if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_stopMonitoring(): Could not identify the corresponding hardware register to stop monitoring for the register " + absRegList[regNum] + ". Monitoring will not be stopped for this function.", "2", exInfoSize)) {
      // If successful get the monitoring function.
      const string monFunc = fwUkl1_getStopMonitoringFunction(absRegList[regNum], exceptionInfo);
      if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_stopMonitoring(): Could not identify the function to use to stop monitoring the register " + absRegList[regNum] + ". Monitoring will not be stopped for this function.", "2", exInfoSize)) {
        for (int i=1; i<=dynlen(hwRegs); ++i) {
          if (0 < dynContains(hwRegList, hwRegs[i])) {
            dynAppend(hwRegList, hwRegs[i]);
            dynAppend(monFuncs, monFunc);
          }// if(0<dynContains(hwRegList,hwRegs[i])
        }// loop over hwRegs
      }// got mon func successfully.
    }// got hw reg successfully.
  }// for regNum
  
  // Now sort the lists for the monitoringSwitch.
  dyn_mixed lists = makeDynMixed(monFuncs, absRegList);
  fwUkl1_sort(lists, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_stopMonitoring(): Failed to sort the hardware list, monitoring should be stopped for all registers but it might take longer to setup.", "2", exInfoSize);
  
  _fwUkl1_monitoringSwitch(ukl1Name, hwRegList, monFuncs, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_stopMonitoring(): Failed to stop monitoring all of the requested registers.", "2", exInfoSize);
  // Done.
  return;
}// fwUkl1_stopMonitoring()

/*!
 * This function starts the monitoring of the registers that are read directly by the ccserv.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 *
 * This function is to be called when a board is added to the system and it will monitor all but the Ingress FPGA and channel
 * register using the ccserv. The Ingress FPGA and channel registers are FIFOs and hence must be readout using a predefined
 * sequence in PVSS. They cannot be monitored by the ccserv.
 */
void fwUkl1_startMonitoringAllMonitorableRegs(const string &ukl1Name, dyn_string &exceptionInfo) {
  // Keep track of the exceptionInfo.
  int exInfoSize = dynlen(exceptionInfo);
  // Get the registers from the appropriate groups. Not worth monitoring any TTCrx stuff.
  dyn_string groups = makeDynString(  FWUKL1_GBE_CONFIG_GROUP, FWUKL1_GBE_STATUS_GROUP
                                    , FWUKL1_GBE_PORT_CONFIG_GROUP, FWUKL1_GBE_PORT_STATUS_GROUP
                                    , FWUKL1_TTCRX_CONFIG_GROUP, FWUKL1_TTCRX_STATUS_GROUP
                                    , FWUKL1_EGRESS_CONFIG_GROUP, FWUKL1_EGRESS_STATUS_GROUP
                                    , FWUKL1_INGRESS_CONFIG_GROUP, FWUKL1_INGRESS_STATUS_GROUP
                                    , FWUKL1_CHANNEL_CONFIG_GROUP, FWUKL1_CHANNEL_STATUS_GROUP
                                    , FWUKL1_TEMPERATURE_SENSORS, FWUKL1_PROM_GROUP
                                    , FWUKL1_TFC_STATUS_GROUP
                                    , FWUKL1_E_IN_LINK_STATUS_GROUP
                                    , FWUKL1_ALARM_STATUS_GROUP
                                    , FWUKL1_MEP_STATUS
                                    , FWUKL1_SENSOR_GROUP
                                    , FWUKL1_OVERVIEW_GROUP
                                    );
  // These are given in the place of a recipe name and the registers that are reject due to not being in the recipe.
  // We are not interested in those for monitoring purposes.
  dyn_string dsJunk;
  string sJunk;
  dyn_string regList = fwUkl1_getRegistersInGroups(groups, sJunk, dsJunk, exceptionInfo);
  if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_startMonitoringAllMonitorableRegs(): Failed to get the registers for " + ukl1Name + ". No registers will be monitored on this UKL1.", "2", exInfoSize) ) {
    // Take the current monitoring settings that were configured when the registers were created.
    fwUkl1_startMonitoring(ukl1Name, regList, exceptionInfo);
    fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_startMonitoringAllMonitorableRegs(): Failed to start monitoring the registers for " + ukl1Name + ". These registers will not be monitored an it must be started manually on the appropriate panel.", "2", exInfoSize);
  }// if fwUkl1_getUkl1NamesList no exceptionInfo
  // Done.
  return;
}// fwUkl1_startMonitoringStatusRegs()

// ==============
//   INTERNAL
// ==============

/*!
 * Takes the list of hardware registers to write and their associated write functions and then calls the appropriate write.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  regNames The list of registers to write to.
 * @param  writeFunctions List of functions to use to write the data. This should correspond (by index) to the register names to write.
 * @param  ddcData List containing the dyn_char that contains the bytes that are to be written to the register. Should be associated
 *           by index to the register names to write.
 * @param  masks List containing the dyn_char that contains the bytes that are to be written to the register. Should be associated
 *           by index to the register names to write.
 * @param  verifyWrite TRUE if the data written is to be read back to confirm the write was successful, FALSE if not. This is mainly
 *           done on the ccserv side and provides little over head. Ingress FPGA and channel register will require an extra read.
 * @param  exceptionInfo Error information. No change in size if there is no exception information generated.
 * @return void.
 *
 * Where possible this function will group writes that use the same write function together.
 */
void _fwUkl1_writeSwitch(const string &ukl1Name, const dyn_string regNames, const dyn_string &writeFunctions, const dyn_dyn_char &ddcData, const dyn_dyn_char &masks, bool verifyWrite, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  int numRegs = dynlen(writeFunctions);
  // These hold the registers to be written to a specific function.
  dyn_string tmpRegNames;
  dyn_dyn_char tmpDdcData;
  dyn_dyn_char tmpMasks;
  // This is the function that the group is to be written with.
  string writefunc;
  // This is use to determine how many registers to write to a given function.
  int count;
  // This is an index that runs through all the element in the lists.
  int i = 1;
  // This is an index that is used to place the data in the appropriate place in the temp lists.
  // Run over write functions.
  while (i <= numRegs) {
    writefunc = writeFunctions[i];
    // Loop through the writeFunctions array and get all the matching writes.
    int j = 1;
    while (writefunc == writeFunctions[i]) {
      tmpRegNames[j] = ukl1Name + regNames[i];
      tmpDdcData[j]  = ddcData[i];
      tmpMasks[j]    = masks[i];
      ++i;
      ++j;
      if ((i>numRegs) || (j>numRegs))
        break;
    }// while(writefunc==writeFunctions[i])
    // Now write them.
    switch (writefunc) {
      case "_fwUkl1_write":
        _fwUkl1_write(tmpRegNames, tmpDdcData, masks, verifyWrite, exceptionInfo);
        fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_writeSwitch(): Failed to write to some/all of the registers for " + ukl1Name + " with the _fwUkl1_write.", "1", exInfoSize);
        break;
      case "_fwUkl1_writeIngressFpgaChannel":
        _fwUkl1_writeIngressFpgaChannel(tmpRegNames, tmpDdcData, masks, verifyWrite, exceptionInfo);
        fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_writeSwitch(): Failed to write to some/all of the registers for " + ukl1Name + " with the _fwUkl1_writeIngressFpgaChannel.", "1", exInfoSize);
        break;
      case "_fwUkl1_writeIngressFpgaChannelSoftDisable":
        _fwUkl1_writeIngressFpgaChannelSoftDisable(tmpRegNames, tmpDdcData, masks, verifyWrite, exceptionInfo);
        fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_writeSwitch(): Failed to write to some/all of the registers for " + ukl1Name + " with the _fwUkl1_writeIngressFpgaChannelSoftDisable.", "1", exInfoSize);
        break;
      case FWUKL1_READ_ONLY:
        exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_writeSwitch(): Attempted to write to read only registers for " + ukl1Name + ". Those writes will be ignored.", "2");
        break;
      default:
        exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_writeSwitch(): Did not recognise the write function " + writefunc + ". Could not write the data associated with this function (first register " + tmpRegNames[1] + ").", "2");
    }// switch(writefunc)
  }// while(i<numRegs)
  // Done.
  return;
}// _fwUkl1_writeSwitch()

/*!
 * Takes the list of hardware registers and their associated read functions and then calls the appropriate read.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  rawData This map will associate a hardware register as a key with a dyn_mixed.
 *           The element FWUKL1_MAP_REG_DATA will contain the dyn_char data read from the hardware.
 *           This map should be empty and contain no keys. Any existing entries will either be overwritten or ignored
 *           and this may produce fake warnings.
 * @param  regNames List of the hardware registers to read.
 * @param  readFunction List of the functions that should be used to read the associated (by index) hardware register.
 * @param  exceptionInfo Error information. No change in size if there is no exception information is generated.
 * @return void.
 *
 * Where possible this function will group all the reads that use the same read function together.
 *
 * The read will return an entry in the map for the null register FWUKL1_NULL_HWREG. This data is empty and effectively
 * junk. It is easier if everything can rely on this register being there.
 */
void _fwUkl1_readSwitch(const string &ukl1Name, mapping &rawData, const dyn_string &regNames, const dyn_string &readFunctions, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  // Now we have to split the data to be read up such that it is read with the appropriate function.
  int numRegs = dynlen(readFunctions);
  // These hold the registers to be read with a specific function.
  dyn_string tmpRegNames;
  dyn_string tmpRegNamesNoUkl1Name;
  dyn_dyn_char tmpDdcData;
  // This is the function that the group is to be read with.
  string readfunc;
  // This is use to determine how many registers to read with a given function.
  int count;
  // This is an index that runs through all the element in the lists.
  int i = 1;
  // This is an index that is used to place the data in the appropriate place in the temp lists.
  // Run over read functions.
  while (i <= numRegs) {
    readfunc = readFunctions[i];
    // Loop through the readFunctions array and get all the matching reads.
    int j = 1;
    while (readfunc == readFunctions[i]) {
      // Get the hardware register and add the UKL1 name before it.
      tmpRegNames[j] = ukl1Name + regNames[i];
      // Save the hardware register name without the UKL1 name. This makes parsing easier later.
      tmpRegNamesNoUkl1Name[j] = regNames[i];
      ++i;
      ++j;
      if (j > numRegs)
        break;
    }// while(readfunc==readFunctions[i])
    
    // Now read them.
    switch (readfunc) {
      case "_fwUkl1_read":
        _fwUkl1_read(tmpRegNames, tmpDdcData, exceptionInfo);
        fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_readSwitch(): Failed to read from some/all of the registers for "+ukl1Name+" with the _fwUkl1_read function.", "2", exInfoSize);
        break;
      case FWUKL1_WRITE_ONLY:
        exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_readSwitch(): Attempted to read from a write only registers for " + ukl1Name + ". Those reads will be ignored.", "2");
        break;
        
      default:
        exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_readSwitch(): Did not recognise the read function " + readfunc + ". Could not read the data associated with this function (first register " + tmpRegNames[1] + ").", "2");
        break;
    }// switch(readfunc)
    // Now append the registers to the full lists for later parsing.
    // Do this even if there is an error as we don't know how much failed.
    const int numTmpRegs = dynlen(tmpRegNames);
    for (int regNum=1; regNum<=numTmpRegs; ++regNum) {
      const string regName = tmpRegNamesNoUkl1Name[regNum];
      rawData[regName] = makeDynMixed();
      rawData[regName][FWUKL1_MAP_REG_DATA] = tmpDdcData[regNum];
    }// for regNum
  }// while(i<numRegs)
  // Add the null register.
  rawData[FWUKL1_NULL_HWREG] = makeDynMixed();
  rawData[FWUKL1_NULL_HWREG][FWUKL1_MAP_REG_DATA] = makeDynChar();
  // Done.
  return;
}// _fwUkl1_readSwitch()

/*!
 * This function looks through the list of functions given to use to start or stop the monitoring and calls the
 * the relevant function for the given registers.
 *
 * @param  regNames Names of the hardware registers to monitor.
 * @param  monFunctions List of functions to use to monitor the corresponding hardware registers.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 *
 * This function will perform more efficiently if the two lists are sorted and the registers requiring
 * the same monitoring function are sequential.
 */
void _fwUkl1_monitoringSwitch(const dyn_string &hwRegNames, const dyn_string &monFunctions, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  int numRegs = dynlen(monFunctions);
  // These hold the registers to be read with a specific function.
  dyn_string tmpRegNames;
  // This is the function that the group is to be monitored with.
  string monfunc;
  // This is use to determine how many registers to monitor with a given function.
  int count;
  // This is an index that runs through all the element in the lists.
  int i = 1;
  // This is an index that is used to place the data in the appropriate place in the temp lists.
  // Run over monitor functions.
  while ( i <= numRegs ) {
    monfunc = monFunctions[i];
    // Loop through the monFunctions array and get all the matching reads.
    int j = 1;
    while ( monfunc == monFunctions[i] ) {
      // Get the hardware register.
      tmpRegNames[j] = regNames[i];
      ++i;
      ++j;
      if ( j > numRegs )
        break;
    }// while(monfunc==monFunctions[i])
    // Now read them.
    switch (monfunc) {
      case "fwCcpc_startMonitoring":
        if ( !fwCcpc_startMonitoring(tmpRegNames) )
          exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_monitoringSwitch(): Failed to start monitoring for some/all of the registers with the fwCcpc_startMonitoring function. It does not work in a distributed system.", "2", exInfoSize);
        break;

      case "fwCcpc_stopMonitoring":
        if ( !fwCcpc_stopMonitoring(tmpRegNames) )
          exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_monitoringSwitch(): Failed to stop monitoring for some/all of the registers with the fwCcpc_stopMonitoring function. It does not work in a distributed system.", "2", exInfoSize);
        break;
      
      case "fwUkl1_startFifoMonitoring":
      case "fwUkl1_stopFifoMonitoring":
        exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_monitoringSwitch(): Monitoring of the UKL1 FIFO registers has not yet been implemented.", "2");
        
      default:
        exInfoSize = fwUkl1Exception_raise(exceptionInfo, "_fwUkl1_monitoringSwitch(): Did not recognise the monitoring function " + monfunc + ". Could not monitor the registers associated with this function (first register " + tmpRegNames[1] + ").", "2");
        break;
    }// switch(monfunc)
  }// while(i<=numRegs)
  
  // Done.
  return;
}// _fwUkl1_monitoringSwitch()

/*!
 * In order to write to specific registers on the FPGAs we must first convert the address
 * to include the address of the FPGAs from the glue card perspective.
 *
 * @param  addr Address of the register to be written to.
 * @return unsigned Full address of the FPGA register.
 */
unsigned _fwUkl1_fpgaizeAddress(unsigned addr) {
  // Based on code written by Steve Wotton, Cambridge.
  // The address sent to lb_{write|read}_hword should be above 0x20000
  // to be outside the GBE & gluecard ranges
  // The FPGA is sensitive only to bits 2-13 of the address
  // (=> the FPGA is not sensitive to the 0x2???? that indicates
  //     that an address is outside the GBE/gluecard ranges)
  // So whenever we want to address the FPGA, set the 13th bit:
  //  the FPGA will only respond to addresses for which the 13th bit has been set
  // 
  // So: << 2    translates a (32-bit) int address into (8-bit) byte address
  //    0x20000 gets us outside the address ranges of the GBE and gluecard
  //              (but will be ignored by the FPGA)
  //    0x2000  should stop the gluecard from reacting to the operation
  //              (but will be ignored by the FPGA)
  // 
  // => (int) register addresses 0x0 to 0x7ff should be usable by the FPGA
  // 
  return (0x08000000 + ( 0x800 | ( addr<<2 ) ) );
}// _fwUkl1_fpgaizeAddress()

/*!
 * This takes a hex string and converts it to a dyn_char that can be written to the UKL1 registers.
 *
 * @param  data A hex string that is to be converted into a dyn_char for writing/masking an FPGA register.
 * @return dyn_char The data that has been converted into an appropriate form for writing/masking.
 */
dyn_char _fwUkl1_fpgaizeData(const string &data) {
  // We must word swap, where a word is 16-bits, the data and add a word every other word.
  // Find the length of the string and loop over it creating a substrings for each words and swap them.
  const int numChars = strlen(data);
  const int numCharsPerWord = 4;
  // Loop over the words and create a reformed string.
  string rData = "";
  for (int index=(numChars-numCharsPerWord); index >= 0; index-=numCharsPerWord)
    rData += "0000" + substr(data, index, numCharsPerWord);// Move the end to the beginning and pad each word to 32-bits.
  // Done.
  return fwCcpc_convertHexToByte(rData);
}// _fwUkl1_fpgaizeData()

/*!
 * Writes to the PHY chip via the MDIO interface.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  phyAddr Address of the PHY chip to write to. Range 0-3.
 * @param  regAddr 5-bit address of the register on the desired PHY chip that it is to be written to.
 * @param  val Value that is to be written to this register.
 * @param  verify Read back the data written to ensure the write was successful. Note that the data written to the PHY chip
 *           is not verified, but the writes required to setup the transfer to the PHY registers are. Source documentation
 *           indicates the clear reasons for this.
 * @param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * @return void.
 */
void _fwUkl1_writeGbeMdioRegister(const string &ukl1Name, unsigned phyAddr, unsigned regAddr, unsigned val, bool verify, dyn_string &exceptionInfo) {
  // Keep track of errors.
  int exInfoSize = dynlen(exceptionInfo);
  bool error = FALSE;
  // This will be prepended to all registers that are read or written from.
  const string preRegStr = ukl1Name + ".GBE";
  // Holds the names of the registers to be written to and the data that is read back from them.
  dyn_string regs;
  dyn_dyn_char ddcData;
  dyn_dyn_char masks;
  dyn_int callStatusList;
  
  // First ensure that the MDIO control is enabled, should be enabled as part of initialisation.
  regs[1] = preRegStr + ".MDIOControl";
  _fwUkl1_read(regs, ddcData, exceptionInfo);
  if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_writeGbeMidoRegister(): Could not determine if the MDIO control has been enabled on the GBE card. Any writes to the GBE MDIO register are likely to fail and some GBE status information cannot be trusted.", "2", exInfoSize)) {
    // Were able to read the MDIO control register now check if it is enabled.
    const string MDIObit = _fwUkl1_convertByteToHex(ddcData[1], 2, 1, FALSE);
    // Check for MDIO control disabled.
    if ("00" == MDIObit) {
      dynClear(regs);
      dynClear(ddcData);
      dynClear(masks);
      // MDIO control is not enabled, enable it.
      regs[1]    = preRegStr + ".MDIOControl";
      ddcData[1] = fwCcpc_convertHexToByte("00000004");
      masks[1]   = fwCcpc_convertHexToByte("00000004");
      _fwUkl1_write(regs, ddcData, masks, verify, exceptionInfo);
      if ( (dynlen(exceptionInfo) > exInfoSize) || (-1 == exInfoSize) ) {
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_writeGbeMdioRegister(): Failed to enable the MDIO control on the GBE card. Further write attempts are not possible.", "1");
        exInfoSize = dynlen(exceptionInfo);
        error = TRUE;
      }// if failed to enable MDIO control.
    }// if MDIO control not enabled or error finding out.
  }// if(_fwUkl1_read())
  else {
    // Mark the error.
    error = TRUE;
  }// else(_fwUkl1_read())
  
  // Ensure these are all cleared for the next attempt.
  dynClear(regs);
  dynClear(ddcData);
  dynClear(masks);
  
  if (!error) {
    // The MDIO control is now enabled so we can write to the desired PHY chip.
    // Frist we write the data to be written to the first register, ensuring that the upper 16-bits are masked as requested in doc.
    masks[1]   = fwCcpc_convertHexToByte("0000ffff");
    regs[1]    = preRegStr + ".PHYData";
    string hexData = fwCcpc_convertDecToHex(val);
    // Now ensure that it is padded to the appropriate length.
    fwUkl1_padString(hexData, 4);
    ddcData[1] = fwCcpc_convertHexToByte(hexData);
    // Now write, but can't verify via this method. Would have to read from the PHY control register
    // on the relevant port and even then if performing a reset as the bit self clears it would probably fail...
    _fwUkl1_write(regs, ddcData, masks, FALSE, exceptionInfo);
    if ( (dynlen(exceptionInfo) <= exInfoSize) && (-1 != exInfoSize) ) {
      // Successfully wrote the data. Now send it.
      dynClear(regs);
      dynClear(ddcData);
      dynClear(masks);
      masks[1] = fwCcpc_convertHexToByte("ffffffff");
      regs[1]  = preRegStr + ".PHYCtrl";
      val = 0x00110000 | ((phyAddr&0x3)<<8) | (regAddr&0x1f);
      string hexData = fwCcpc_convertDecToHex(val);
      // Now ensure that it is padded to the appropriate length.
      fwUkl1_padString(hexData, 4);
      ddcData[1] = fwCcpc_convertHexToByte(hexData);
      // Now write.
      _fwUkl1_write(regs, ddcData, masks, verify, exceptionInfo);
      if ( (dynlen(exceptionInfo) > exInfoSize) || (-1 == exInfoSize) ) {
        // Failed to trigger sending of PHY data.
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_writeGbeMdioRegister(): Failed to send data to the PHY chip. Data was placed in the buffer and MDIO control enabled, a second attempt maybe succesful.", "1");
        exInfoSize = dynlen(exceptionInfo);
      }// if(!fwCcpc_write())
    }// if(fwCcpc_write())
    else {
      // Failed to write the command for the PHY.
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_writeGbeMdioRegister(): Failed to write the data to be written to the PHY chip. Write aborted.", "1");
      exInfoSize = dynlen(exceptionInfo);
    }// else(fwCcpc_read())
  }// if(!error)
  
  // Done.
  return;
}// _fwUkl1_writeGbeMdioRegister()

/*!
 * Set the Ingress Mailbox length (16-bit word).
 *
 * @param  dpName string Contains datapoint name.
 * @param  newLength unsigned that contains the new length.
 * @param  settings string The settings string before the new length was set.
 * @return void.
 */
void _fwUkl1_setMailboxLength(const string &dpName, const int newLength, string &settings) {
  dpGet(dpName,settings);
  int fifoLength;
  int mbAddress;
  sscanf(settings,"%d,%*d,%*d,%d,%*d,%*d,%*d,%*d,%*d,%*d,%*d,%*d",mbAddress,fifoLength);
//  DebugTN("Settings="+settings+" Addr="+mbAddress+" Length="+fifoLength);

  string newSettings;
  sprintf(newSettings,"%d,1,2,%d,0,0,0,0,0,0,0,1",mbAddress,newLength);
//  DebugTN("Settings="+newSettings+" Addr="+mbAddress+" Length="+fifoLength);
  dpSetWait(dpName,newSettings);
  
  return;
}// _fwUkl1_setMailboxLength()

/*!
 * Set the Ingress Mailbox settings
 *
 * @param  dpName string Contains datapoint name.
 * @param  settings string The new settings string.
 * @return void.
 */
void _fwUkl1_setMailboxSettings(const string &dpName, const string settings) {
  dpSetWait(dpName,settings);
  
  return;
}// _fwUkl1_setMailboxSettings()

/*!
 * Writes the dyn_char configuration data to a UKL1 Ingress FPGA channel configuration FIFO.
 *
 * @param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           written to, which corresponds to the appropriate hardware register.
 * @param  ddcData Each element of this array contains an array of char that make up the words that are to be written to the
 *           register in the form required by a fwCcpc_write.
 * @param  masks The masks are not used for FIFO writes and are ignored by this function.
 * @param  verifyWrite If this is set to TRUE write will be verified to be successful by reading back the registers.
 *           FALSE and no read will be done.
 * @param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * @return void.
 */

void _fwUkl1_writeIngressFpgaChannel(const dyn_string &registerNames, const dyn_dyn_char &ddcData, const dyn_dyn_char &masks, bool verifyWrite, dyn_string &exceptionInfo) {
  // Keep track of the amount of exception information generated.
  int exInfoSize = dynlen(exceptionInfo);
  // We can only write one channel at a time. The registerNames could
  // refer to multiple channels on multiple boards, so we must do everything
  // for each channel.
  const int numRegs = dynlen(registerNames);
  for (int i=1; i<=numRegs; ++i) {
    // Get the full register name which contains the channel and the UKL1 name.
    const string regName = registerNames[i];
    
    // Find the first full stop, which will indicate the end of the UKL1 name.
    const int stopPos = strpos(regName, ".");
    if (-1 < stopPos) {
      // Now we have found a full stop we must get the UKL1 name, which is between these two markers.
      // The last arugment is the length of the string to strip and hence is the difference
      // between the two markers.
      string ukl1Name = substr(regName, 0, stopPos);
      // Now we can get the UKL1 channel number from the register name.
      int chanPos = strpos(regName, "Channel");
      if (-1<chanPos) {
        // From the strpos we know where the string 'Channel' starts and the channel
        // number will be the next character.
        int channel = substr(regName, chanPos+strlen("Channel"));
        // Covert that channel number into some other forms we use.
        int inFpga = channel/9;
        // Prepare the FIFO for writing.
        _fwUkl1_setConfigForIngressFpgaChannel(ukl1Name, inFpga, verifyWrite, exceptionInfo);
        if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_writeIngressFpgaChannel(): Failed to prepare the configuration FIFO for Ingress FPGA " + inFpga + ", channel " + (channel%9) + " for writting. Data not written.", "2", exInfoSize)) {
          // Write the data.
          string fifoName;
          sprintf(fifoName,"%s.FpgaConfig.ConfigurationFifoChannel%02d",ukl1Name, channel);
          dyn_string mailboxRegs = makeDynString(fifoName);
          dyn_dyn_char mailboxData;
          dynAppend(mailboxData, ddcData[i]);
          // We actually don't use the masks as the CCPC server just writes it into the FIFO
          // and this cuases the loss of data. It must be empty.
          dyn_dyn_char mailboxMasks;
          // The given data and mask will not contain the appropriate channel to write to.
          // This must be written in the upper byte.
          // The channel number resides in bits [15..12] of the first word. Select the upper
          // byte nad shift to align with bit 12.
          // and bit shift the channel number by 4 to move it to the upper nibble of this byte.
          mailboxData[1][1] |= channel%9 << 4;
          // Can't verify the FIFO write.
          _fwUkl1_write(mailboxRegs, mailboxData, mailboxMasks, FALSE, exceptionInfo);
          if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_writeIngressFpgaChannel(): Failed to write configuration data for channel " + channel + ". Configuration not written.", "2", exInfoSize)) {
            // Now we have written the data we can trigger the sending of it.
            _fwUkl1_sendIngressChannelConfigFifo(ukl1Name, verifyWrite, exceptionInfo);
            fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_writeIngressFpgaChannel(): Triggering the transmission of the configuration data for channel " + channel + ". Data not written.", "2", exInfoSize);
          }// if _fwUkl1_write() successful.
        }// if prepared channel for writing.
      }// if(-1<chanPos)
    // If there is no channel idenfier then it can't be used to write to a channel.
    else
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_writeIngressFpgaChannel: Could not determine the UKL1 channel number from the register \'" + regName + "\'. It appears to have no .ChannelX, if it does not contain a channel number it cannot be used in this function.", "2");
    }// if(-1<stopPos)
    // If there are no full stops then this can't be a valid register name!
    else
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_writeIngressFpgaChannel: Could not determine the UKL1 name from the register \'" + regName + "\'. It has no full stops, is it correctly formed?", "2");
  }// for i
  
  // Done.
  return;
}// _fwUkl1_writeIngressFpgaChannel()

/*!
 * Writes the dyn_char configuration data to a UKL1 Ingress FPGA channel configuration FIFO.
 *
 * @param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           written to, which corresponds to the appropriate hardware register.
 * @param  ddcData Each element of this array contains an array of char that make up the words that are to be written to the
 *           register in the form required by a fwCcpc_write.
 * @param  masks The masks are not used for FIFO writes and are ignored by this function.
 * @param  verifyWrite If this is set to TRUE write will be verified to be successful by reading back the registers.
 *           FALSE and no read will be done.
 * @param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * @return void.
 */
void _fwUkl1_configureIngressFpgaChannel(const dyn_string &registerNames, const dyn_dyn_char &ddcData, const dyn_dyn_char &masks, bool verifyWrite, dyn_string &exceptionInfo) {
  // Keep track of the amount of exception information generated.
  int exInfoSize = dynlen(exceptionInfo);

  // We can write all channels on one ingress at the same time.
  // Create arrays to group all channels on each ingress
  dyn_dyn_char mailboxData0;
  dyn_dyn_char mailboxData1;
  dyn_dyn_char mailboxData2;
  dyn_dyn_char mailboxData3;
 // We actually don't use the masks as the CCPC server just writes it into the FIFO
 // and this causes the loss of data. It must be empty.
  dyn_dyn_char mailboxMasks;
 // Local copy of the data
  dyn_char tmpData;
  string ukl1Name;
  
  const int numRegs = dynlen(registerNames);
  for (int i=1; i<=numRegs; ++i) {
    // Get the full register name which contains the channel and the UKL1 name.
    const string regName = registerNames[i];
    
    // Find the first full stop, which will indicate the end of the UKL1 name.
    const int stopPos = strpos(regName, ".");
    if (-1 < stopPos) {
      // Now we have found a full stop we must get the UKL1 name, which is between these two markers.
      // The last argument is the length of the string to strip and hence is the difference
      // between the two markers.
      ukl1Name = substr(regName, 0, stopPos);
      // Now we can get the UKL1 channel number from the register name.
      int chanPos = strpos(regName, "Channel");
      if (-1<chanPos) {
        // From the strpos we know where the string 'Channel' starts and the channel
        // number will be the next character.
        int channel = substr(regName, chanPos+strlen("Channel"));
        // Covert that channel number into some other forms we use.
        int inFpga = channel/9;
        
          // The given data and mask will not contain the appropriate channel to write to.
          // This must be written in the upper byte.
          // The channel number resides in bits [15..12] of the first word. Select the upper
          // byte and shift to align with bit 12.
          // and bit shift the channel number by 4 to move it to the upper nibble of this byte.
        dynClear(tmpData);
        dynAppend(tmpData,ddcData[i]);
        tmpData[1] |= channel%9 << 4;
        switch (inFpga)
        {
          case 0:
          dynAppend(mailboxData0[1], tmpData);
          break;
          case 1:
          dynAppend(mailboxData1[1], tmpData);
          break;
          case 2:
          dynAppend(mailboxData2[1], tmpData);
          break;
          case 3:
          dynAppend(mailboxData3[1], tmpData);
          break;
        }
      }// if(-1<chanPos)
    // If there is no channel idenfier then it can't be used to write to a channel.
    else
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_writeIngressFpgaChannel: Could not determine the UKL1 channel number from the register \'" + regName + "\'. It appears to have no .ChannelX, if it does not contain a channel number it cannot be used in this function.", "2");
    }// if(-1<stopPos)
    // If there are no full stops then this can't be a valid register name!
    else
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_writeIngressFpgaChannel: Could not determine the UKL1 name from the register \'" + regName + "\'. It has no full stops, is it correctly formed?", "2");
  }// for i

// Now we only need to make one transaction per ingress FPGA
  
  dyn_string mailboxReg = makeDynString(ukl1Name + ".FpgaConfig.IngressMailbox");
    
  for ( int ing=0;ing<4;ing++)
  {
// First prepare the ingress control register
    _fwUkl1_setConfigForIngressFpgaChannel(ukl1Name, ing, FALSE, exceptionInfo);
    fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_writeIngressFpgaChannel(): Failed to prepare the configuration FIFO for Ingress FPGA " + ing + " for writting.", "2", exInfoSize);

// Write the channel configuration data for this ingress to the mailbox    
    switch(ing)
    {
      case 0:
      if (dynlen(mailboxData0)>0)
      {
        if (dynlen(mailboxData0[1])==36)
        {
          _fwUkl1_write(mailboxReg, mailboxData0, mailboxMasks, FALSE, exceptionInfo);
        }
      }
      break;
      case 1:
      if (dynlen(mailboxData1)>0)
      {
        if (dynlen(mailboxData1[1])==36)
        {
          _fwUkl1_write(mailboxReg, mailboxData1, mailboxMasks, FALSE, exceptionInfo);
        }
      }
      break;
      case 2:
      if (dynlen(mailboxData2)>0)
      {
        if (dynlen(mailboxData2[1])==36)
        {
          _fwUkl1_write(mailboxReg, mailboxData2, mailboxMasks, FALSE, exceptionInfo);
        }
      }
      break;
      case 3:
      if (dynlen(mailboxData3)>0)
      {
        if (dynlen(mailboxData3[1])==36)
        {
          _fwUkl1_write(mailboxReg, mailboxData3, mailboxMasks, FALSE, exceptionInfo);
        }
      }
      break;
    }
    fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_writeIngressFpgaChannel(): Failed to write configuration data for Ingress FPGA " + ing, "2", exInfoSize);

// Trigger the transfer of the data to the ingress    
    _fwUkl1_sendIngressChannelConfigFifo(ukl1Name, FALSE, exceptionInfo);
    fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_writeIngressFpgaChannel(): Triggering the transmission of the configuration data for Ingress FPGA " + ing, "2", exInfoSize);
  }      
  // Done.
  return;
}// _fwUkl1_configureIngressFpgaChannel()

/*!
 * Writes the dyn_char configuration data to a UKL1 Ingress FPGA channel configuration FIFO.
 *
 * @param  ukl1Name string that contains the UKL1 name.
 * @param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           written to, which corresponds to the appropriate hardware register.
 * @param  ddcData Each element of this array contains an array of char that make up the words that are to be written to the
 *           register in the form required by a fwCcpc_write.
 * @param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * @return void.
 */
void _fwUkl1_configureIngressChannels(const string& ukl1Name, const dyn_string &registerNames, const dyn_dyn_char &ddcData, dyn_string &exceptionInfo) {
  // Keep track of the amount of exception information generated.
  int exInfoSize = dynlen(exceptionInfo);

  const int numRegs = dynlen(registerNames);
  if (numRegs < 1)
  {
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_configureIngressChannels: Called with empty register list", "2");
    return;
  }
  
// Need to preserve the current settings of the ingress inhibits
  dyn_string regName = makeDynString(ukl1Name + ".FpgaConfig.IngressControl");
  dyn_dyn_char ingressControl;
  _fwUkl1_read(regName, ingressControl, exceptionInfo);
  if (fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_configureIngressChannels(): Failed to get inhibit status", "2", exInfoSize) ) return;

// We can write all channels on one ingress at the same time.
// Create arrays to group all channels on each ingress
  dyn_dyn_char mailboxData;
  
// Add sequence to reset the mailbox read/write pointers for each ingress
  for (int i=1;i<=4;i++)
  {
    mailboxData[i] = makeDynChar();
    dynAppend(mailboxData[i],24);
    dynAppend(mailboxData[i],0);
    dynAppend(mailboxData[i],(0x00 | ingressControl[1][4]));
    dynAppend(mailboxData[i],i-1);
    dynAppend(mailboxData[i],24);
    dynAppend(mailboxData[i],0);
    dynAppend(mailboxData[i],(0x30 | ingressControl[1][4]));
    dynAppend(mailboxData[i],i-1);
    dynAppend(mailboxData[i],24);
    dynAppend(mailboxData[i],0);
    dynAppend(mailboxData[i],(0x00 | ingressControl[1][4]));
    dynAppend(mailboxData[i],i-1);
  }
  
  for (int i=1; i<=numRegs; ++i) 
  {
    // Get the full register name which contains the channel and the UKL1 name.
    const string regName = registerNames[i];
    
      // Now we can get the UKL1 channel number from the register name.
    int chanPos = strpos(regName, "Channel");
    if (0>chanPos) continue;

// From the strpos we know where the string 'Channel' starts and the channel number will be the next character.
    int channel = substr(regName, chanPos+strlen("Channel"));
// Compute the Ingress FPGA index.
    int inFpga = channel/9;

// Copy the configuration data from the input and add the channel number in bits [15..12] of the first word.
    dynAppend(mailboxData[inFpga+1], 31);
    dynAppend(mailboxData[inFpga+1], 0);
    dynAppend(mailboxData[inFpga+1], ddcData[i][2]);
    dynAppend(mailboxData[inFpga+1], ddcData[i][1] | (channel%9 << 4));
    dynAppend(mailboxData[inFpga+1], 31);
    dynAppend(mailboxData[inFpga+1], 0);
    dynAppend(mailboxData[inFpga+1], ddcData[i][4]);
    dynAppend(mailboxData[inFpga+1], ddcData[i][3]);
      
  }// for i

// Add sequence to trigger the mailbox transfer
  for (int i=1;i<=4;i++)
  {
    dynAppend(mailboxData[i],24);
    dynAppend(mailboxData[i],0);
    dynAppend(mailboxData[i],(0x40 | ingressControl[1][4]));
    dynAppend(mailboxData[i],i-1);
    dynAppend(mailboxData[i],24);
    dynAppend(mailboxData[i],0);
    dynAppend(mailboxData[i],(0x00 | ingressControl[1][4]));
    dynAppend(mailboxData[i],i-1);
  }
  
// Concatenate the four ingress sequences so they can be dispatched together
  
  dyn_dyn_char ddcMailboxData;
  ddcMailboxData[1] = makeDynChar();
    
  for ( int ing=0;ing<4;ing++)
  {
      int mbSize = dynlen(mailboxData[ing+1])/4;
      if (mbSize>5)
      {
          dynAppend(ddcMailboxData[1],mailboxData[ing+1]);
      }
  }
  
// Prepend the size (number of addr/data pairs) to the beginning of sequence
  
  int totSize = dynlen(ddcMailboxData[1])/4;
  dynInsertAt(ddcMailboxData[1],0,1);
  dynInsertAt(ddcMailboxData[1],(char)(totSize&0xff),1);
  
// Write the channel configuration data for this ingress to the mailbox
// Call fwCcpc_write directly. _fwUkl1_write tries to read back which is not allowed for this.
  
  dyn_string mailboxReg = makeDynString(ukl1Name + ".FpgaConfig.ConfigurationSequence");
  dyn_int diStatus;
  if (!fwCcpc_write(mailboxReg, ddcMailboxData, makeDynChar(), diStatus))
  {
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_configureIngressChannels: Error from fwCcpc_write().", "2");
  }
  // Done.
  return;
}// _fwUkl1_configureIngressChannels()

/*!
 * Writes the dyn_char configuration data to a UKL1 Ingress FPGA channel configuration FIFO.
 *
 * @param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           written to, which corresponds to the appropriate hardware register.
 * @param  ddcData Each element of this array contains an array of char that make up the words that are to be written to the
 *           register in the form required by a fwCcpc_write.
 * @param  masks The masks are not used for FIFO writes and are ignored by this function.
 * @param  verifyWrite If this is set to TRUE write will be verified to be successful by reading back the registers.
 *           FALSE and no read will be done.
 * @param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * @return void.
 */
void _fwUkl1_writeIngressFpgaChannelSoftDisable(const dyn_string &registerNames, const dyn_dyn_char &ddcData, const dyn_dyn_char &masks, bool verifyWrite, dyn_string &exceptionInfo) {
  // Keep track of the amount of exception information generated.
  int exInfoSize = dynlen(exceptionInfo);
  // We can only write one channel at a time. The registerNames could
  // refer to multiple channels on multiple boards, so we must do everything
  // for each channel.
  const int numRegs = dynlen(registerNames);
  for (int i=1; i<=numRegs; ++i) {
    // Get the full register name which contains the channel and the UKL1 name.
    const string regName = registerNames[i];
    
    // Find the first full stop, which will indicate the end of the UKL1 name.
    const int stopPos = strpos(regName, ".");
    if (-1 < stopPos) {
      // Now we have found a full stop we must get the UKL1 name, which is between these two markers.
      // The last arugment is the length of the string to strip and hence is the difference
      // between the two markers.
      string ukl1Name = substr(regName, 0, stopPos);
      // Now we can get the UKL1 channel number from the register name.
      int chanPos = strpos(regName, "Channel");
      if (-1<chanPos) {
        // From the strpos we know where the string 'Channel' starts and the channel
        // number will be the next character.
        int channel = substr(regName, chanPos+strlen("Channel"));
        // Covert that channel number into some other forms we use.
        int inFpga = channel/9;
        // Prepare the FIFO for writing.
        _fwUkl1_setConfigForIngressFpgaChannel(ukl1Name, inFpga, verifyWrite, exceptionInfo);
        if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_writeIngressFpgaChannelSoftDisable(): Failed to prepare the configuration FIFO for Ingress FPGA " + inFpga + ", channel " + (channel%9) + " for writting. Data not written.", "2", exInfoSize)) {
          // Write the data.
          string fifoName;
          sprintf(fifoName,"%s.FpgaConfig.ConfigurationFifoChannel%02d",ukl1Name,channel);
          dyn_string mailboxRegs = makeDynString(fifoName);
          dyn_dyn_char mailboxData;
          dynAppend(mailboxData, ddcData[i]);
          // We actually don't use the masks as the CCPC server just writes it into the FIFO
          // and this causes the loss of data. It must be empty.
          dyn_dyn_char mailboxMasks;
          // The given data and mask will not contain the appropriate channel to write to.
          // This must be written in the upper byte.
          // The channel number resides in bits [15..12] of the first word. Select the upper
          // byte and shift to align with bit 12 and bit shift the channel number by 4 to move it to the upper nibble of this byte.
          mailboxData[1][1] = channel%9 << 4;
          // We must also write the command type (4) to the header
          mailboxData[1][2] = 4;
          // Can't verify the FIFO write.
          _fwUkl1_write(mailboxRegs, mailboxData, mailboxMasks, FALSE, exceptionInfo);
          if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_writeIngressFpgaChannelSoftDisable(): Failed to write configuration data for channel " + channel + ". Configuration not written.", "2", exInfoSize)) {
            // Now we have written the data we can trigger the sending of it.
            _fwUkl1_sendIngressChannelConfigFifo(ukl1Name, verifyWrite, exceptionInfo);
            fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_writeIngressFpgaChannelSoftDisable(): Triggering the transmission of the configuration data for channel " + channel + ". Data not written.", "2", exInfoSize);
          }// if _fwUkl1_write() successful.
        }// if prepared channel for writing.
      }// if(-1<chanPos)
    // If there is no channel idenfier then it can't be used to write to a channel.
    else
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_writeIngressFpgaChannelSoftDisable: Could not determine the UKL1 channel number from the register \'" + regName + "\'. It appears to have no .ChannelX, if it does not contain a channel number it cannot be used in this function.", "2");
    }// if(-1<stopPos)
    // If there are no full stops then this can't be a valid register name!
    else
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_writeIngressFpgaChannelSoftDisable: Could not determine the UKL1 name from the register \'" + regName + "\'. It has no full stops, is it correctly formed?", "2");
  }// for i
  
  // Done.
  return;
}// _fwUkl1_writeIngressFpgaChannelSoftDisable()

/*!
 * Prepares the configuration FIFO for writting by setting the appropriate Ingerss FPGA number and asserting and
 * deasserting the pointer resets. It ensures that the transmit trigger has been deasserted.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  inFpga Number of the Ingress FPGAs that are to be read, each element should correspond contain
 *           the number of the Ingress FPGA to read. Range:0 to 3.
 * @param  verifyWrite If this is set to TRUE write will be verified to be successful.
 * @param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * @return void.
 */
void _fwUkl1_setConfigForIngressFpgaChannel(const string &ukl1Name, int inFpga, bool verifyWrite, dyn_string &exceptionInfo) {
  // Holds the various bits of data that will need to be written.
  // This will only ever write to one register.
  dyn_string mailboxRegs = makeDynString(ukl1Name + ".FpgaConfig.IngressControl");
  dyn_dyn_char mailboxData;
  dyn_dyn_char mailboxMasks;
    
  // Deassert both read and write pointer resets and the transmit trigger (zeros to bits [6..4]),
  // ensuring the broadcast bit is cleared (zeros to bit 7) unless we want to broadcast,
  // while writing the Ingress FPGA to be configured (bits [9..8]). (ignored for broadcasts)
  string tmpData = "";
  // If the FPGA number is outside the normal range we want to set the broadcast FPGA bit.  
  if(inFpga > 3) {
  tmpData = fwCcpc_convertDecToHex( 1 << 7 );
  }  
  else {
  tmpData = fwCcpc_convertDecToHex(inFpga << 8);
  }
  fwUkl1_padString(tmpData, 4);
  mailboxData[1] = fwCcpc_convertHexToByte(tmpData);
  // Want to clear bits [7..4] and set bit [9..8].
  mailboxMasks[1] = fwCcpc_convertHexToByte("000003F0");
  _fwUkl1_write(mailboxRegs, mailboxData, mailboxMasks, verifyWrite, exceptionInfo);
    
  // Assert the read and write pointers' reset.
  mailboxData[1] = fwCcpc_convertHexToByte("00000030");
  // The mask is the same as the data to be written in this case.
  mailboxMasks[1] = mailboxData[1];
  _fwUkl1_write(mailboxRegs, mailboxData, mailboxMasks, verifyWrite, exceptionInfo);
    
  // Deassert the read and write pointers' reset.
  mailboxData[1] = fwCcpc_convertHexToByte("00000000");
  // The mask is the same as before so we don't need to update it.
  _fwUkl1_write(mailboxRegs, mailboxData, mailboxMasks, verifyWrite, exceptionInfo);
    
  // Done.
  return;
}// _fwUkl1_setConfigForIngressFpgaChannel()

/*!
 * Triggers the tranmission of the Ingress configuration FIFO data.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  verifyWrite If this is set to TRUE write will be verified to be successful.
 * @param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * @return void.
 */
void _fwUkl1_sendIngressChannelConfigFifo(const string& ukl1Name, bool verifyWrite, dyn_string& exceptionInfo) {
  // Trigger the FIFO tranmission to Ingress FPGA from Egress FPGA.
  dyn_string mailboxRegs;
  dyn_dyn_char mailboxData;
  dyn_dyn_char mailboxMasks;
  mailboxRegs[1]  = ukl1Name + ".FpgaConfig.IngressControl";
  mailboxData[1]  = fwCcpc_convertHexToByte("00000040");
  mailboxMasks[1] = fwCcpc_convertHexToByte("00000040");
  _fwUkl1_write(mailboxRegs, mailboxData, mailboxMasks, verifyWrite, exceptionInfo);

  // Deassert the FIFO tranmission trigger.
  mailboxData[1] = fwCcpc_convertHexToByte("00000000");
  // The mask is the same as before.
  _fwUkl1_write(mailboxRegs, mailboxData, mailboxMasks, verifyWrite, exceptionInfo);
  
  // Done.
  return;
}// _fwUkl1_sendIngressChannelConfigFifo()

/*!
 * Writes the given data to the given UKL1 board. The masks are applied when writing the data to ensure that only specific
 * bits of a register are written to.
 *
 * @param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           written to, which corresponds to the appropriate hardware register.
 * @param  ddcData Each element of this array contains an array of char that make up the words that are to be written to the
 *           register in the form required by a fwCcpc_write.
 * @param  masks Each element contains a bit mask that is applied when writing the corresponding element in the ddcData array
 *           to the corresponding register in the registerNames array. It ensure that only the desired bits are written to in
 *           the register.
 * @param  verifyWrite If this is set to TRUE write will be verified to be successful by reading back the registers.
 *           FALSE and no read will be done.
 * @param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * @return void.
 */
void _fwUkl1_write(const dyn_string &registerNames, const dyn_dyn_char &ddcData, const dyn_dyn_char &masks, bool verifyWrite, dyn_string &exceptionInfo) {
  // Check we are being asked to write something.
  if (0 >= dynlen(registerNames)) {
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_write(): List of registers to write to empty, no data written to the hardware.", "2");
    return;
  }// if(0>=dynlen(registerNames))
  // Track the exceptions.
  int exInfoSize = dynlen(exceptionInfo);
  // We will read the data back from the registers once it is written. The read data is stored here.
  dyn_dyn_char readData;
  // Holds the status of the writes.
  dyn_int callStatusList;
//  if (fwCcpc_write(registerNames, ddcData, masks, callStatusList)) {
  if (fwCcpc_writeRead(registerNames, ddcData, masks, readData, callStatusList)) {
    // See if we want to verify that the write was successfull.
    if (verifyWrite) {
      // Call the verification function.
      _fwUkl1_verifyWrite(registerNames, ddcData, masks, readData, exceptionInfo);
      fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_write(): Failed to verify data written written to some/all of the registers.", "2", exInfoSize);
    }// if(verifyWrite)
  }// if(fwCcpc_write)
  else {
    // Check where the errors occured.
    const unsigned statusLen = dynlen(callStatusList);
    if (0 != statusLen) {
      for (unsigned element=1; element<=statusLen; ++element) {
        if (0 != callStatusList[element]) {
          // Construct the error message.
          string errMsg = "_fwUkl1_write(): " + registerNames[element] + " encountered ";
          if (callStatusList[element] > 0)
            errMsg += "a problem writing to the CCPC server, return code: " + callStatusList[element] + ".";
          else if (callStatusList[element] < 0)
            errMsg += "a problem with PVSS.";
          else
            errMsg += "an unknown error.";
          exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", errMsg, "2");
        }// if(0!=callStatusList[element]
      }// for element
    }// if(0!=statusLen)
    else
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_write(): All writes failed.", "1");
  }// else(fwCcpc_write)
  
  // Done.
  return;
}// _fwUkl1_write()

/*!
 * Reads the data from the given UKL1 board.
 *
 * @param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           read from, which corresponds to the appropriate hardware register.
 * @param  ddcData Each element of this array contains an array of char that make up the words that are read from the
 *           register in the form returned by a fwCcpc_read. Returned by reference.
 * @param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * @return void.
 *
 * This function can be called to update the values in the datapoint elements and the data read back ignored. In this case the function
 * is used to convert the fwCcpc exception information into a fwUkl1 exceptionInfo and DP connects to the DPEs can parse the
 * read data if successful.
 */
void _fwUkl1_read(const dyn_string &registerNames, dyn_dyn_char &ddcData, dyn_string &exceptionInfo) {
  // Check we are being asked to read something.
  if (0 >= dynlen(registerNames)) {
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_read(): List of registers to read is empty, no data read from the hardware.", "2");
    return;
  }// if(0>=dynlen(registerNames))
  
  // Track the exceptions.
  int exInfoSize = dynlen(exceptionInfo);
  
  // In order to read the status information we must first assert and then deassert the read pointer
  // reset to ensure that we always start from the beginning of the FIFO. Always reset all the pointers.
  // Do it for all reads as it is only two extra writes and means we don't need a separate function
  // for the status FIFOs.
  // Need to get theUKL1 name from the register name. Can include the system name if we wish.
  // Simply remove all the characters from the beginning to the first full stop.
  // We must have at least one register name as we return if the dyn_string has zero length earlier.
  const string ukl1Name = substr(registerNames[1], 0, strpos(registerNames[1], "."));
  fwUkl1_resetIngressFpgaStatusBufPointers(ukl1Name, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_resetIngressFpgaStatusBufPointers(): Failed to reset status buffer pointers, any read status information will be invalid.", "2", exInfoSize);
  // Note what the status of the reads is.
  dyn_int callStatusList;
  // Just read don't bother checking returns the panels will indicate the problems soon enough.
  // We do this as it appears to crash if we try and use ddcData directly. If anyone ever reads
  // this comment they should try using ddcData directly and if it doesn't work figure out why!
  dyn_dyn_char tmpDdcData;
  if (!fwCcpc_read(registerNames, tmpDdcData, callStatusList)) {
    // There has been an error, check its seriousness
    const int sizeList = dynlen(callStatusList);
    if (0 != sizeList) {
      for (int element=1; element<=sizeList; ++element) {
        if ( 0 != callStatusList[element] ) {
          // Construct the error message.
          string errMsg = "_fwUkl1_read(): Read failed for " + registerNames[element] + ". Encountered ";
          if (callStatusList[element] > 0)
            errMsg += "a problem writing to the CCPC server, return code: " + callStatusList[element] + ".";
          else if (callStatusList[element] < 0)
            errMsg += "a problem with PVSS.";
          else
            errMsg += "an unknown error.";
          // Now raise the exception.
          exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", errMsg, "2");
        }// if(0!=callStatusList[element])
      }// for element.
    }// if(0!=sizeList)
    else {
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_read(): Failed to read, no information read back.", "1");
    }// else(0!=sizeList)
  }// if(!status)
  ddcData = tmpDdcData;
  // Done.
  return;
}// _fwUkl1_read()

/*!
 * Save a list of recipes to the given recipe, which is of the given type. The data should contain the actual value that would be
 * written to the hardware and include the mask that should be used when writing the data.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  recipeName Name of the recipe. The structure for this is described in fwUkl1_save().
 * @param  recipeType Recipe type that is to be saved. The structure for this is described in fwUkl1_save().
 * @param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           written to, which corresponds to the appropriate hardware register.
 * @param  registerTypes dyn_int containing the hardware register types for each of the registers in the registerNames variable.
 * @param  ddcDataAndMask Each element of this array contains an array of char that make up the words that are to be written to the
 *           register in the form required by a fwCcpc_write and this includes the mask that is to be used for the data.
 * @param  domain This is required if the recipe to be saved has been loaded into the recipe cache by the FSMConfDB. In this case
 *           the name of the recipe will have been mangled. The recipe name given to this function should be the unmangled
 *           name and it will be converted as appropriate. This parameter can also be an empty string if no name mangling is required.
 *           Note in this case there can be no / present in the recipe name, only to delimit the run mode from the command (RUN_MODE/Command).
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 */
void _fwUkl1_save(const string &ukl1Name, string recipeName, const string &recipeType, const dyn_string &registerNames, const dyn_int &registerTypes, const dyn_dyn_char &ddcDataAndMask, string domain, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  if ("" == ukl1Name) {
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_save(): No UKL1 name give to retrieve the recipe from, no data will be retrieved.", "2");
    return;
  }// if(""==ukl1Name)
  
  // Check that we are given a recipeType and a recipeName.
  if (("" != recipeType) && ("" != recipeName)) {
    // We must write to all the registers in the recipe otherwise it will populate them with the last values written to the board.
    // We must therefore get the recipe first and then update the relevant values and write it back to the recipe.
    // Holds a list of all the registers in the recipe to be written.
    dyn_string fullRegisterNames;
    // Holds a list of all the data, including masks, to be written into the recipe.
    dyn_dyn_char fullDdcData;
    dyn_int fullHwRegTypes;
    // Before we try and get the recipe we should check that it exists, as if it doesn't trying to retrieve it will fail.
    dyn_string recipes = fwHw_getRecipes(ukl1Name);
    // Also note whether or not the recipe already exists.
    bool recipeAlreadyExists;
    // Check to see if the register we are going to write to exists.
    // dynContains returns either the index of the element that the string is contained within, 0 if it does not exists or -1 in the event of an error.
    if (0 < dynContains(recipes, recipeName)) {
      // The recipe does exist, thus we should get it and overwrite the relevant elements.
      recipeAlreadyExists = TRUE;
      if (-1 != fwHw_getRecipe(ukl1Name, recipeName, fullRegisterNames, fullDdcData, domain)) {
        // Once we have gotten the recipe we also need to retrieve the hardware types in order
        // to write the recipe properly. These aren't given by getRecipe but are required by
        // saveRecipe. Get the system name by removing the device name.
        const string dpeName = fwUkl1_removeDeviceName(ukl1Name)+"RecipeCache/"+recipeName+".Values.DPETypes";
        if (dpExists(dpeName)) {
          dpGet(dpeName, fullHwRegTypes);
        }// if(dpExists(dpeName))
        else {
          // In the case that we don't know the type we can check what the value is for each element individually.
          const int numFullRegs = dynlen(fullRegisterNames);
          for (int i=1; i<=numFullRegs; ++i) {
            // This section of code was taken from the fwHw.ctl. It wasn't in a useful
            // function so just copied, pasted, renamed a few variables and added documentation.
            // The HW register type is determined from it DPT. Find out the DPT of our register.
            // It will contain the system and UKL1 name, that is how it is stored in the recipe.
          		const string dpt = dpTypeRefName(fullRegisterNames[i]);
            // The DPT will be of the form HwRegisterXXX where XXX is the HW register type.
            const string hwRegister = "HwRegister";
            // Strip the type from the name. HwRegister must be at the beinging of the string.
            const int hwRegisterPos = strpos(dpt, hwRegister);
          		if (0 == pos) {
              // By taking a substring starting at the end of the HwRegister section
              // (which must be at the beginning!) we are left with the string name
              // of the hardware type. We need the type number, which a function is
              // provided for by the fwHw library.
              const string hwRegTypeName = substr(ref, pos+strlen(hwRegister));
              fullHwRegTypes[i] = _fwHw_getRegisterTypeId(regTypeName);
            }// if(0==pos)
          }// for i
        }// else(dpExists(dpeName))
        
        const int numElements = dynlen(registerNames);
        for (int element = 1; element <= numElements; ++element) {
          // This take the specific register name and returns the index of its occurance in the full list.
          // It is this index that we can use to over write the appropriate data element.
          // It assumes there is only one occurrance of a register in the recipe. This should always be the case!
          int index = dynContains(patternMatch("*"+ukl1Name+registerNames[element], fullRegisterNames), TRUE);
          // Now we know where the appropriate recipe data is copy it into the ddcData array converting from its string format.
          // It is possible that this data does not occur in this recipe type, hence we must protect against this.
          if (0 != index) {
            fullDdcData[index] = ddcDataAndMask[element];
            fullHwRegTypes[index] = registerTypes[element];
          }// if(0!=index)
          // else
            // fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_save(): " + registerNames[element] + " was not found in recipe type. This data has not been saved.", "1");
          // Don't bother warning about this it generates an unnecessary error, just let it happen silently.
        }// for element
      }// if(-1!=fwHw_getRecipe())
      else {
       exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_save(): Failed to retrieve the recipe `" + recipeName + "' for " + ukl1Name + ". Writing the settings anyway. This could create a recipe with only partially initialised values and potentially a corrupt name.", "1");
       recipeAlreadyExists = FALSE;
     }// else(-1!=fwHw_getRecipe())      
    }// if(dynContains)
    
    if (!recipeAlreadyExists) {
      // As the recipe doesn't exist or couldn't retrieve it we are creating it. Therefore we just give the recipe all the data we have.
      // The save recipe function will place the last values written to the hardware in the other recipe registers.
      // These can be updated by other function calls.
      fullRegisterNames = registerNames;
      fullDdcData = ddcDataAndMask;
    }// if(!recipeAlreadyExists)

    // If we are given a domain then we should mangle the recipe name to write to the FSMConfigDB cached recipe.
    if ("" != domain)
      recipeName = _fwHw_getRecipeName(fwUkl1_removeSystemName(ukl1Name), recipeName, domain, fwUkl1_removeDeviceName(ukl1Name));
    // If this function fails and the recipe already exists then there has been an error, however this function could still apparently fail if the recipe
    // doesn't already exist, but it does write the appropriate values to the recipe. Require it fails and the recipe already exists for us to indicate an error.
    if ( (-1 == fwHw_saveRecipe(ukl1Name, recipeType, recipeName, fullRegisterNames, fullHwRegTypes, fullDdcData)) && recipeAlreadyExists )
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_save(): Failed to save the recipe " + recipeName + " for " + ukl1Name + ".", "1");
  }// if recipe name and type not empty.
  else {
    // Create more useful error message.
    emptyName = ("" == recipeName ? "empty":recipeName);
    emptyType = ("" == recipeType ? "emtpy":recipeType);
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_save(): Recipe name (" + emptyName + ") and/or type (" + emptyType + ") not given for " + ukl1Name + ". Cannot save recipe.", "1");
  }// else recipe name or type empty.
  
  // Done.
  return;
}// _fwUkl1_save()

/*!
 * Loads the data from the given UKL1 board.
 *
 * @param  [in] ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  [in] recipeName Name of the recipe. The structure for this is described in fwUkl1_save().
 * @param  [in] recipeType Recipe type that is to be saved.
 * @param  [in] domain This is required if the recipe to be loaded has been loaded into the recipe cache by the FSMConfDB. In this case
 *           the name of the recipe will have been mangled. The recipe name given to this function should be the unmangled
 *           name and it will be converted as appropriate. This parameter can also be an empty string if no name mangling is required.
 *           Note in this case there can be no / present in the recipe name, only to delimit the run mode from the command (RUN_MODE/Command).
 * @param  [in] registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           read from, which corresponds to the appropriate hardware register.
 * @param  [in] registerTypes The register types of the hardware register we are loading the data for.
 *           Reqired to properly split the data into a data and mask.
 * @param  [out] ddcData Each element of this array contains an array of char that make up the words that are load from the
 *           recipe. Returned by reference.
 * @param  [out] masks The masks that are to be used with this data are returned by reference.
 * @param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * @return void.
 *
 * The data in the recipe may contain both the register data and a mask. If this is not the case and only the data is present
 * then a mask of 0xFF for each byte is assumed and returned.
 */
void _fwUkl1_load(const string &ukl1Name, const string &recipeName, const string &recipeType, const string &domain, const dyn_string &registerNames, const dyn_int &registerTypes, dyn_dyn_char &ddcData, dyn_dyn_char &masks, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  if ("" == ukl1Name) {
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_load(): No UKL1 name give to retrieve the recipe from, no data will be retrieved.", "2");
    return;
  }// if(""==ukl1Name)
  // Check that we are given a recipeType and a recipeName.
  if ((""!=recipeType) && (""!=recipeName)) {
    // When we retrieve the recipe data it will generally contain the data and a mask.
    // We need to know how large the data section of this register is so we can decide
    // whether we want to split it into mask and data or not.
    mapping sizes = fwUkl1_getHwRegistersSizeInBytes(registerNames, registerTypes, exceptionInfo);
    fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_load(): Failed to determine the size of the data part of the recipe value stored. Will assume the data stored contains no mask and it will not be split. Mask of 0xFF was returned for each data byte.", "2", exInfoSize);
    // When we get the recipe data it will give us all registers in the recipe and also data.
    // This will hold the registers we are given and then we can check them against the list of requested registers to see what data to parse.
    dyn_string tmpRegisterNames;
    // Holds the data from the recipe, but it will be all the data stored in the recipe and will later be extracted.
    dyn_dyn_char tmpDdcData;
    if (-1 != fwHw_getRecipe(ukl1Name, recipeName, tmpRegisterNames, tmpDdcData, domain)) {
      const int numRegs = dynlen(registerNames);
      for (int element=1; element<=numRegs; ++element) {
        // This takes the specific register name and returns the index of its occurance in the full list.
        // Relies on the fact that the list is unique. The returned register names will have the UKL1 name
        // prepended to them i.e. they are the absolute path of the DPE. Its possible that we haven't been
        // given the system name as part of the ukl1Name variable so use a leading wildcard just in case.
        // It is this index that we can use to over write the appropriate data element.
        int index = dynContains(patternMatch("*"+ukl1Name+registerNames[element], tmpRegisterNames), TRUE);
        // Now we know where the appropriate recipe data is copy it into the ddcData array.
        // It is possible that this data does not occur in this recipe type, hence we must protect against this.
        if (0 != index) {
          // We check that the data we have read from the DB to see if it is larger
          // than the size the register is expecting. If it is then it must contain
          // the map and we will have to split the data. Unless we didn't recognise
          // the register type and we have -1. The treat it as though we don't have
          // a mask in the recipe.
          const int expectedSize = sizes[registerNames[element]];
          if ((-1!=expectedSize) && (expectedSize<dynlen(tmpDdcData[index]))) {
            fwUkl1_splitRecipeData(tmpDdcData[index], ddcData[element], masks[element]);
          }// if(expectedSize<actualSize)
          else {
            ddcData[element] = tmpDdcData[index];
            const int numElements = dynlen(ddcData[element]);
            masks[element] = makeDynChar();
            for (int i=1; i<=numElements; ++i) {
              masks[element][i] = 0xFF;
            }// for i
          }// else(expectedSize<actualSize)
        }// if(0!=index)
        // If the element doesn't exist then pad the ddcData array with an empty array.
        // This allows the parse data function to pick up on it and put a null value
        // into the into the return data.
        else {
          ddcData[element] = makeDynChar();
          masks[element] = makeDynChar();
        }// else(0!=index)
      }// for element
    }// if(-1!=fwhw_getRecipe())
    else {
      // It is most likely that the recipe just didn't exist.
      // Populate the ddcData with nothing such that it can return nothing, but it will contain the appropriate number of elements of nothing.
      for (int element=1; element<=dynlen(registerNames); ++element) {
        ddcData[element] = makeDynChar();
        masks[element] = makeDynChar();
      }// for element
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_load(): Failed to retrieve the recipe " + recipeName + " for " + ukl1Name + ". Recipe may not exist.", "2");
    }// else(-1!=fwHw_getRecipe())
  }// if recipe name and type not empty.
  else {
    // Create more useful error message.
    emptyName = (""==recipeName ? "empty":recipeName);
    emptyType = (""==recipeType ? "emtpy":recipeType);
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_load(): Recipe name (" + emptyName + ") and/or type (" + emptyType + ") not given for " + ukl1Name + ". Cannot load recipe.", "2");
  }// else recipe name or type empty.
    
  // Done.
  return;
}// _fwUkl1_load()
  
/*!
 * Reads from the registers on the L1 board to ensure that the data just written has been written successfully.
 *
 * @param  registerNames Names of the registers that were written to.
 * @param  writtenData The data that was written to the registers that the read back values should be compared against.
 * @param  masks Any masks that should be applied to the read data, such that only specific bits in a register are compared
 *           against the write bits.
 * @param  readData The data that was read back from the registers to compare against the written data.
 * @param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * @return void.
 *
 * This function is designed to check the data written and read using the fwCcpc_writeRead function.
 */
void _fwUkl1_verifyWrite(dyn_string registerNames, dyn_dyn_char writtenData, dyn_dyn_char masks, dyn_dyn_char readData, dyn_string& exceptionInfo) {
  // Number of elements of data to loop through.
  const unsigned numRegs = dynlen(registerNames);
  // Check to see if any masks are supplied and if so flag them for use.
  bool applyMasks = ( numRegs==dynlen(masks) ? TRUE:FALSE );
  // Check the data.
  for (unsigned element = 1; element <= numRegs; ++element) {
    // Number of bytes written.
    const int dataLen  = dynlen(writtenData[element]);
    // Number of bytes read back.
    const int checkLen = dynlen(readData[element]);
    // Check they are the same size.
    if (checkLen != dataLen) {
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_verifyWrite(): " + registerNames[element] + " written and read data lengths differ when verifying write, " + dataLen + " and " + checkLen + " bytes big respectively. Read data 0x" + fwCcpc_convertByteToHex(readData[element]) + ", and written data 0x" + fwCcpc_convertByteToHex(writtenData[element]) + " (premask).", "2");
    }// if(checkLen!=data)
    else {
      // Only check the data if they are the same size.
      // Use to note if an element in the read array differ.
      bool different = FALSE;
      // This will provide us with our loop index. The two arrays have been found to be the same size, so just set to either.
      for (unsigned index = 1; index <= dataLen; ++index) {
        // Apply the masks that are given to the read data, if masks are supplied.
        if (applyMasks)
          readData[element][index] &= masks[element][index];
        // Now check the data against the read values.
        if ( readData[element][index] != writtenData[element][index] ) {
          different = TRUE; 
          // We can't break here as otherwise the readData will not be fully masked and that value included in the error will make no sense.
          // break;
        }
      }// for index
      // Will only raise the exception displaying the whole value and not for each differing element.
      if (different)
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_verifyWrite(): " + registerNames[element] + " read data, " + fwCcpc_convertByteToHex(readData[element]) + ", and written data, " + fwCcpc_convertByteToHex(writtenData[element]) + ", differ.", "2");
    }// else(checkLen!=dataLen)
  }// for element

  // Done.
  return;
}// _fwUkl1_verifyWrite()

// @}
// ==========================
//   DATA PARSING FUNCTIONS
// ==========================

/** @defgroup SectionInterface DataParsingFunctions
 *   These functions are all provided to assist with converting the data from that read from
 *   the hardware to that display to the User or vice versa.
 */
// @{

/*!
 * Takes a dyn_char and converts it to a hex string assuming the input data is little endian.
 *
 * @param  dcData A dyn_char containing the data read back from the hardware register.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return string Returns the hex string.
 *
 * The fwCcpc_convertByteToHex function assumes big endian data.
 */
string fwUkl1_convertByteToHexLittleEndian(const dyn_char &dcData, dyn_string &exceptionInfo) {
  // We must reverse the byte ordering of the array.
  // For this we will need to know its size.
  const int numBytes = dynlen(dcData);
  // This stores the reversed bytes.
  dyn_char swappedDcData;
  if (0<numBytes) {
    // Have a none zero number of bytes.
    // Loop through them swapping the first element for last, 2nd for 2nd to last etc..
    for (int byte=1; byte<=numBytes; ++byte)
      swappedDcData[numBytes-(byte-1)] = dcData[byte];
  }// if(0<numBytes)
  else if (0 == numBytes)
    return "";
  else {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_convertByteToHexLittleEndian(): PVSS error, failed to determine size of byte array, could not convert to hex string.", "1");
    return "Fail";
  }// else failed
  
  // Done.
  return fwCcpc_convertByteToHex(swappedDcData);
}// fwUkl1_convertByteToHexLittleEndian()

/*!
 * Takes a string and ensures that it contains enough characters for the number of bytes specified
 * by numBytes. Adds `0' to the beginnig if it is too short and removes characters from the beginning
 * if it is too long. Must be in hex.
 *
 * @param  sData The string that contains the data to be checked. Returned by reference.
 * @param  numBytes The number of bytes that the returned character string should return.
 * @return void
 *
 * This is for use with strings that represent a hex number that should be a set number of bits long.
 * It assumes that the lowest order bit is the last (reading left to right) character and hence its behaviour is
 * to move the last character always towards the final character in the padded string (the 2nd, 4th, 6th etc character).
 * This is achieved by padding or removing at the beginning of the string.
 *
 * \b Example:
 * \code
 *   (...)
 *   // Declare some data e.g. User input from a panel.
 *   const string hexData1 = "1234";
 *   // Wish to use the fwCcpc library to write this to a 32-bit registers hence is needs to be zero padded
 *   // to be 4-bytes big (8-characters).
 *   fwUkl1_padString(hexData1, 4);
 *   DebugTN(hexData1); // "00001234"
 *
 *   // Declare some more data e.g. User input from a panel.
 *   const string hexData2 = "1234";
 *   // Wish to use the fwCcpc library to write this to an 8-bit registers hence it need to be 1-byte (2-characters) big.
 *   fwUkl1_padString(hexData2, 1);
 *   // Here the data is misformed and it is assumed that the least significant bit (last character) in the given number
 *   // should remain the least significant bit.
 *   DebugTN(hexData2); // "34"
 *   (...)
 * \endcode
 */
void fwUkl1_padString(string& sData, unsigned numBytes) {
  // First determine the length.
  const int datalen = strlen(sData);
  // Create a temporary string to store the original string.
  const string tmpOrig = sData;
  // Determine the number of characters that must be in the final string,
  // each character represents half a byte or nibble.
  const int numChar = numBytes*2;
  // Perform the checking and padding/trimming.
  if ( numChar > datalen ) {
    // The data string is too short so increase its length by padding at the beginning, highest order bytes.
    // A temporary string to hold the amount of padding.
    string tmpPad;
    for (int character = 1; character <= (numChar-datalen); ++character) {
      tmpPad += "0";
    }
    // Now insert the padding into the data string.
    sData = tmpPad + tmpOrig;
  }// if(numChar>datalen)
  // Remove the initial characters of the string until it reaches the appropriate length, highest order bytes.
  else if ( numChar < datalen )
    sData = substr(sData, (datalen-numChar), datalen);
  // No else do nothing, it is the right length.

  // Return the resulting string.
  return;
}// fwUkl1_padString()

/*!
 * This takes a string and will bit shift it left by the requested number of bits. It will pad the end of the string with
 * with zero bits.
 *
 * @param  theString The string to be left shifted. Returned by reference.
 * @param  shift The number of bits to shift the string by.
 * @return void.
 *
 * The primary assumption of this algorithm is that the given string is a hex representation of a number. Thus each character
 * is 4 bits when considered as an int. As in PVSS each character is actually 8 bits the shift it doesn't represent a direct
 * bit shift of the binary representation.
 *
 * Padding will be added at the end of the string, which is considered the beginning of the number, and extra characters will
 * be added at the beginning of the string, which is considered the end of the number, if the bits overflow the current length
 * of the string. No bits are lost.
 */
void fwUkl1_leftShiftString(string &theString, int shift) {
  // The alogrithm works by considering first how many whole characters to shift (1 character = 4 bits there)
  // and then how many bits to shift within each character.
  const int charToShift       = shift / 4;
  const int bitsInCharToShift = shift % 4;
  // Current number of characters in string.
  const int startLenString = strlen(theString);
  // We will break each character into a char shift each one individually.
  // Each element has enough bits to prevent overflow as we only need to shift by less bits than a character.
  // We need to keep the previous result from the loop to put the over flow in it.
  char tmpCPrevious;
  char tmpCCurrent;
  string tmpString;
  // First we will shift by the appropriate number of bits within each character.
  // We are starting at the end of the number.
  for (int character = 1; character <= startLenString; ++character) {
    // First store the character as a byte, which is has plenty of room for overflow.
    tmpCCurrent = fwCcpc_convertHexToByte(theString[character-1])[1];
    // Now shift it.
    tmpCCurrent <<= bitsInCharToShift;
    if (1 != character) {
      // Now we OR the last 4 bits of the current element with the first 4 bits of the previous element.
      // This puts any overflow into the next character.
      tmpCPrevious |= tmpCCurrent >> 4;
      // Now erase the last 4 bit of the char as we are only interested in 4 bit chars.
      tmpCCurrent   &= 0x0f;
      
      // Now build the string back up from the previous value.
      // Generally we only want the last character from the returned string as this is the 4 bytes we have OR'd to
      // make the bit shift value. The first character is just padding as we give the conversion function 8 bits.
      if (2 != character)
        tmpString += fwCcpc_convertByteToHex(makeDynChar(tmpCPrevious))[1];
      // In the event we are dealing with the first created character we want all 8bits as the first character in the
      // returned string contains the overflow for the shifted string.
      else
        tmpString += fwCcpc_convertByteToHex(makeDynChar(tmpCPrevious));
      // If it is the last element then we want to save it as well.
    }// if(1!=character)
    // Save the current char for the next loop.
    tmpCPrevious = tmpCCurrent;
  }// for character
  // Need to put the last character in the temp string too. It will have not be done in the loop.
  // Always only interested in the last character.
  tmpString += fwCcpc_convertByteToHex(makeDynChar(tmpCPrevious))[1];
  
  // Overwrite theString with that we just created.
  theString = tmpString;
  // Add to the end of the string the number of whole characters we need to shift.
  int addedChar = 0;
  while (addedChar++ < charToShift)
    theString += "0";
  
  // Done.
  return;
}// fwUkl1_leftShiftString()
  
/*!
 * Splits the data read from the recipe into the data half and the mask half.
 *
 * @param  recipeData The data as read from the recipe.
 * @param  data Data removed from the data given from the recipe. Returned by reference.
 * @param  mask Mask removed from the data given from the recipe. Returned by reference.
 * @return void.
 */
void fwUkl1_splitRecipeData(const dyn_char &recipeData, dyn_char &data, dyn_char &mask) {
  // Data from the recipe is the value to write and the mask concatonated into a single dyn_char.
  // By construction the data and mask are equal sizes. We must split the array in half in order
  // to return the data and mask separately.
  const int numElements = dynlen(recipeData);
  for (int i=1; i<=numElements; ++i) {
    if (i <= numElements/2)
      dynAppend(data, recipeData[i]);
    else
      dynAppend(mask, recipeData[i]);
  }// for i
  // Done.
  return;
}// fwUkl1_splitRecipeData()

/*!
 * Retrieves the size of a hardware register from the default settings of the UKL1 HW type.
 *
 * @param  [in] hwRegList List of the hardware registers to look up.
 * @param  [in] regTypes List of HW types for the registers to look up. Size of register is saved in words and
 *           this function knows how many bytes are in a word for the specific type to convert it.
 *           It recognsises the following types:
 *           - FWHW_LBUS
 *           - FWHW_GBE
 *           - FWHW_I2C
 *           - FWHW_UserSpecific
 * @param  [in,out] exceptionInfo Error information. No change in size if no exception information is generated.
 * @return mapping Use the hardware register as a key which links to the size of the register
 *           in bytes.
 *
 * For FWHW_LBUS registers we assume 4 bytes big, which is the size of the data actually stored in the
 * recipes and read back from the hardware. They hardware registers are infact 2 bytes and it is upto
 * the caller to decide whether to divide the result by 2 or not.
 */
mapping fwUkl1_getHwRegistersSizeInBytes(dyn_string hwRegList, const dyn_int &regTypes, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  dyn_dyn_anytype settingsList;
  mapping sizes;
  const int numHwRegs = dynlen(hwRegList);
  for (int i=1; i<=numHwRegs; ++i) {
    // We have to remove any leading full stops from the hwRegName otherwise it can't
    // be found by this function call. As this must be done for each the we might as well
    // do it in this loop and get each one individually.
    // Of course the mapping must contain the original name.
    string strippedHwRegName = strltrim(strrtrim(hwRegList[i],"."),".");
    // Look in the default settings for this register, they should always be the same as the values for each board and then we don't
    // have to specify the UKL1 board specifically. 
    dynClear(settingsList);
    fwHw_getDefaultSpecificSettings(FWUKL1_HW_TYPE, makeDynString(strippedHwRegName), settingsList);
    // Where we look for it depends on the register type.
    switch (regTypes[i]) {
      case FWHW_LBUS:
        // The FPGA registers are given in 32-bit words we are dealing with bytes.
        // As the hardware tool is concerned they are 32-bit words, but the hardware
        // considers them to be 16-bits. The user will have handle this distinction
        // it documented that the function is doing this.
        sizes[hwRegList[i]] = (int)settingsList[1][3] * 4;// 3rd item in first register.
        break;
      case FWHW_GBE:
        // The GBE registers are given in 32-bit words but we are dealing with bytes.
        sizes[hwRegList[i]] = (int)settingsList[1][2] * 4;// 2nd item in first register.
        break;
      case FWHW_I2C:
        // The I2C registers are given in bytes and we are dealing with bytes.
        sizes[hwRegList[i]] = (int)settingsList[1][4];// 1st item in first register.
        break;
      case FWHW_UserSpecific:
        // The user specific registers are FIFOs in our case and we need to know how many bytes in
        // total to get, which is specified in element 4. This is the number of words and must
        // be multiplied by the size of the words.
        sizes[hwRegList[i]] = (int)settingsList[1][4] * (int)settingsList[1][3];
        break;
      default:
        sizes[hwRegList[i]] = -1;
        exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_convertAbsRegHexToByte(): Failed to recognise the register type "+regType+". Could not determine number of bytes required for the given register.", "2");
        break;
    }// switch(regType)
  }// for entry in hwRegList
  return sizes;
}// fwUkl1_getHwRegisterSizeInBytes()

// ===============
//    INTERNAL
// ===============

/*!
 * Converts a dyn_char to a hex string. Adds the functionality to remove only a limit range from the input dyn_char.
 *
 * @param  dcData Data read from the register as a byte array.
 * @param  startBit Number of the first bit to be read from the register, where bit 0 is the least significant bit.
 * @param  length Number of bits to be read, set to 1 if only startBit is to be read. If -1 all the data is returned.
 * @param  fpga Boolean value to indicate whether we are parsing FPGA registers or not. This must be set correctly otherwise
 *           the data will not be properly interpreted.
 * @return string Hex representation of the requested bits. Will contain no leading 0s.
 *
 * The length and startBit will be cohereced to fit into the size of the dyn_char is necessary. startBit will be made less
 * than length or length greater than startBit as appropriate.
 * The reason for the FPGA flag is that the UKL1 FPGA registers are read as if they were 32-bit registers and not the 16-bit registers
 * that they are physically (this is because sequential addresses are 32-bits apart and not 16!). This function will remove the extra two
 * junk words that are read and parse only the remaining data in this case. Do not remove these extra words manually before hand.
 */
string _fwUkl1_convertByteToHex(dyn_char dcData, unsigned startBit, unsigned length, bool fpga) {
  // Check there is actually some data.
  if (0 >= dynlen(dcData))
    return "";

  // If we are dealing with an FPGA the we need to convert from 16- to 32-bit words.
  if (fpga)
    _fwUkl1_convert32To16BitWords(dcData);

  // In order to be able to span byte boundaries we need to know where we are going to start and finish in terms
  // of the appropriate byte. These constants will ease the navigation of the bit-field.
  // Determine how many bytes we have been given.
  const int numBytes = dynlen(dcData);
  // Determine how many bits we have, there are eight bits in each element as each element contains a char.
  const unsigned numBits = 8 * numBytes;
  // Perform a sanity check on the starting bit.
  if ((numBits-(unsigned)1)<startBit)
    startBit = numBits-1;
  
  // The last bit is the number of bits we wish to remove offset by the starting bit number.
  // We subtract 1 as if we just remove 1 bit then the start and end bit are the same.
  unsigned endBit = startBit + length - 1;
  // Can't end at a point passed the number of bits that we have.
  // The last bit is 1 less than the number of bits in the register as we start counting at 0.
  // numBits is starts at 1, whereas endBit starts at 0.
  if ((numBits-(unsigned)1)<endBit)
    endBit = numBits -1;
  
  // The first byte that we need to read from. Remembering that we are using a big endian numbering scheme.
  const unsigned startReadByte = numBytes - (unsigned)(startBit/8);
  // This is the last byte that we need to read from.
  const unsigned endReadByte = numBytes - (unsigned)endBit/8;
  // Lets us know which byte we are currently reading from the dcData.
  // Remember to decrement this number as we move backwards through
  // the array.
  unsigned readByte = startReadByte;
  // When reading we will may need to mask off the end of the read bits
  // there are more beyond the end read bit.
  char readMask;
  // This is the region to remove the data from.
  // Marks the bit in the byte to start the selection from.
  // It is initially offset into the first byte to read from.
  // After the first byte it is always the beginning of a byte.
  unsigned startBitInReadByte = (unsigned)startBit%8;
  // This is the last bit in the byte that we need to read from.
  // It will start as the last bit in the first byte, unless
  // we are only reading from a single byte in which case
  // we must only read until the last bit. The last bit must
  // be less than 8 in this case, otherwise the start and end
  // bytes cannot be the same!
  unsigned endBitInReadByte = 7;
  if (startReadByte == endReadByte)
    endBitInReadByte = endBit%8;
  
  // The selected elements from the given dyn_char are stored here before being converted to hex.
  dyn_char selectedData;
  // This the number of bytes we need to select. Make sure we round up when converting to an int.
  const int numBytesToWrite = ceil(length/8.0);
  // When we add the selected data we need to know where in the selectedData dyn_char to write it.
  // This keeps track of the byte bearing in mind we are moving from end to start of the arrays.
  unsigned writeByte = numBytesToWrite;
  // When pulling that data out of the given dyn_char we need to know where we are writing the
  // bits we have removed. This will always correspond to the position after the last number
  // of bits written. When we start we haven't written any bits.
  unsigned startWriteBit = 0;
  // When we start we can assume we are writing to all the bits.
  unsigned endWriteBit = 7;
  
  
  // Loop through all the bytes we want to select, starting at the end and
  // moving to the beginning and remove the appropriate bits for each byte
  // from the given data remove the appropriate bits.
  // Keep track of the total number of bits we have written.
  // Can stop when they reach the requested.
  unsigned totalWrittenBits = 0;
  // Need to keep track of how many bits we have written to the selectedData
  // in the current byte.
  unsigned writtenBits = 0;
  // Need to know how many bits we just wrote.
  unsigned currentWrittenBits = 0;
  // We need to keep track of how many bits we have read from the source
  // dcData in the current operation.
  unsigned readBits = 0;
  do {
    // Find the last bit in a byte using the modulo operator with the number of bits
    // in a byte. Shift 0's into a stream of 1's until the last byte. A bit wise not
    // gives us a mask that is 0 except for bits we are reading. Bit wise and with
    // the data masks off any byte we don't want.
    readMask = ~(0xff << (endBitInReadByte+1));
    // Remove the interesting bits and then save them,
    // taking care not to overwrite any previously written bits.
    selectedData[writeByte] |= ((dcData[readByte]&readMask)>>startBitInReadByte)<<startWriteBit;
    // We need to count the total number of bits we have read, including
    // any 'lost' as they were before the start bit.
    readBits += endBitInReadByte+1;
    // We need to count the total number of bits we have written.
    // Add 1 to the end bit to go to the nth bit, end bit starts
    // counting at zero.
    currentWrittenBits = endBitInReadByte+1-startBitInReadByte;
    writtenBits += currentWrittenBits;
    // As the writtenBits is an accumulation until we reach 8
    // we must calculate the totalWrittenBits from the amount
    // we just read and not the total in this byte.
    totalWrittenBits += currentWrittenBits;
    // The new start bit is at the end of the number of bits written
    // for this write only. This must be done before we update the
    // endBitInReadByte.
    startBitInReadByte += currentWrittenBits;
    // Modulo 8 to keep us within the byte.
    startBitInReadByte %= 8;
    // In order to work out where to read until in the next operation
    // we must figure out how many byte we have left to read.
    startWriteBit += currentWrittenBits;
    // Modulo 8 to keep us within the byte.
    startWriteBit %= 8;
    // If we have read 8 bits then we can move onto the next byte to read.
    if (7<readBits) {
      --readByte;
      readBits = 0;
    }// if(8>readBits)
    if (7<writtenBits) {
      --writeByte;
      writtenBits = 0;
    }// if(8>writtenBits)
    // The end bit in the read byte is the current start bit plus the
    // remain bits to write to the selectedData byte.
    endBitInReadByte = startBitInReadByte + 8 - currentWrittenBits - 1;
    if ((readByte==endReadByte) && ((endBit%8)<endBitInReadByte))
      endBitInReadByte = endBit%8;
  } while(length>totalWrittenBits);
  
  // Remove any leading zeros.
  string rtrn = strltrim(fwCcpc_convertByteToHex(selectedData), "0");
  // If it is zero then it will now be empty.
  if ("" == rtrn)
    rtrn = "0";
  // Done.
  return rtrn;
}// _fwUkl1_convertByteToHex()

/*!
 * Takes an optionally colon delimited hex MAC address, as a string, and returns a dyn_char containing
 * the MAC address in a hardware/DB compatible form.
 *
 * @param  macAddress The string that contains the MAC address to be parsed. Should be of the form XX:XX:XX:XX:XX:XX
 *           or XX:XX:XX:XX. If the final two bytes are omitted the returned dyn_char is formatted for the source MAC
 *           address, otherwise the dyn_char is formatted for the destination MAC address.
 * @param  exceptionInfo Error information. No change in size if there are no errors.
 * @return dyn_char The MAC address converted to a dyn_char suitable for writing to either the destination
 *           or source address register, which is determined by the input data format. An empty array if the
 *           data cannot be parsed.
 *
 * Expects the MAC address to be of the form 00:0e:0c:b0:2d:19 or 000e0cb02d19 i.e. 12 characters long.
 */
dyn_char _fwUkl1_convertMacToByte(string mac, dyn_string& exceptionInfo)
{
  dyn_int oct = makeDynInt(0,0,0,0,0,0,0,0,0,0,0,0);

  // First case handles full 6 octet MAC with colon separators
  int matched = sscanf(mac,"%x:%x:%x:%x:%x:%x",oct[11],oct[12],oct[7],oct[8],oct[3],oct[4]);

  // Same again but in case MAC does not use colon separators
  if ( matched < 6 )
  {
    matched = sscanf(mac,"%2x%2x%2x%2x%2x%2x",oct[11],oct[12],oct[7],oct[8],oct[3],oct[4]);
  }

  // Try case when passed only highest four octets of MAC
  if ( matched < 6 )
  {
    matched = sscanf(mac,"%x:%x:%x:%x",oct[11],oct[12],oct[7],oct[8]);
  }

  // And again but without colons
  if ( matched < 4 )
  {
    matched = sscanf(mac,"%2x%2x%2x%2x",oct[11],oct[12],oct[7],oct[8]);
  }

  // Convert int to char
  dyn_char dcoct = makeDynChar(0,0,0,0,0,0,0,0,0,0,0,0);
  for ( int i=1;i<=12;i++)
  {
    dcoct[i] = oct[i];
  }

  // Return empty dyn_char on error or the parsed array on success
  if (matched == 4 || matched == 6)
  {
    return dcoct;
  }
  else
  {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_convertMacToByte(): Failed to convert MAC address.", "1");
    return makeDynChar();
  }

}// _fwUkl1_convertMacToByte()

/*!
 * Takes an array of bytes and returns them as a colon delimited MAC address.
 *
 * @param  macAddress The array containing the bytes to be parsed.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return dyn_string Contains the MAC address in element FWUKL1_MAP_REG_DATA and a colour to indicate status in FWUKL1_MAP_XTR_DATA.
 *           Returns `Fail' and FwStateAttention1 if the input data is missformed.
 *
 * This code requires that the extra bytes generated from a fwCcpc_read() are first removed, via _fwUkl1_convert32To16BitWords().
 */
dyn_string _fwUkl1_convertByteToMac(dyn_char macAddress, dyn_string& exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  // First convert it from the 32bit to 16bit words. We don't care about the loss of mask
  // element from the confDB as we ignore them anyway.
  // Do this before we calculate the size of the MAC address. It also makes the later indexing more natural.
  _fwUkl1_convert32To16BitWords(macAddress);
  
  // If the masks are in the data then they will be at the front i.e. lowest element numbers.
  // Determine the number of elements in the array and count backwards from there.
  // This will allow absolute indexing whether the masks are present or not.
  const int size = dynlen(macAddress);
  // Check that it contains the appropriate number of elements.
  // Technically if it came from the configuration database it could contain the write mask tagged on the end.
  // This doesn't cause a problem as we use absolute indexes later, just require there to be 6 elements minimum (and the MAC to be in those).
  if (6>size) {
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_convertByteToMac(): Input array contains " + size + " elements. Should be 6 or 12 elements big.", "1");
    return makeDynString("Fail", "FwStateAttention1", "");
  }// if(6>size)

  // Now we need to put the data into the appropriate format.
  string output;
  // This should be in a big endian order once it reaches here.
  output =  fwCcpc_convertByteToHex(macAddress[size - 5]);
  output += ":";
  output += fwCcpc_convertByteToHex(macAddress[size - 4]);
  output += ":";
  output += fwCcpc_convertByteToHex(macAddress[size - 3]);
  output += ":";
  output += fwCcpc_convertByteToHex(macAddress[size - 2]);
  output += ":";
  output += fwCcpc_convertByteToHex(macAddress[size - 1]);
  output += ":";
  output += fwCcpc_convertByteToHex(macAddress[size]);

  // Done.
  return makeDynString(output, "FwBackgroundInputText", "");
}// _fwUkl1_convertByteToMac()

/*!
 * Takes a standard formed IP(4) address, as a string, and convert it to a continuous hex
 * string. Also performs syntacs checking on the given string.
 *
 * @param  ipAddress The string that contains the IP address to be parsed. Should be of the form:
 *           XXX.XXX.XXX.XXX, where X represents a decimal number, e.g. 192.168.192.192. It is possible
 *           to miss the final byte i.e. 192.168.192, this is required for the source IP address.
 *           In this case where the final byte is missing it will be replaced with zeros in the resulting hex string.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return dyn_char Contains the data in a form that can be written directly to the hardware or to the confDB.
 *           Empty if there is an error.
 */
dyn_char _fwUkl1_convertIpToByte(string ip, dyn_string& exceptionInfo)
{
  dyn_int oct = makeDynInt(0,0,0,0,0,0,0,0);

  // First try case when all four IP octets are present
  int matched = sscanf(ip,"%d.%d.%d.%d",oct[7],oct[8],oct[3],oct[4]);

  // Try case when passed only highest three octets of IP
  if ( matched < 4 )
  {
    matched = sscanf(ip,"%d.%d.%d",oct[7],oct[8],oct[3]);
  }

  // Convert int to char
  dyn_char dcoct = makeDynChar(0,0,0,0,0,0,0,0);
  for ( int i=1;i<=8;i++)
  {
    dcoct[i] = oct[i];
  }

  // Return empty dyn_char on error or the parsed array on success
  if (matched == 4 || matched == 3)
  {
    return dcoct;
  }
  else
  {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_convertIpToByte(): Failed to convert IP address.", "1");
    return makeDynChar();
  }

}// _fwUkl1_convertIpToByte()

/*!
 * Takes an array of bytes, and returns them as a full stop delimited IP address.
 *
 * @param  ipAddress The array containing the bytes to be parsed.
 * @param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * @return dyn_string Contains the IP in element FWUKL1_MAP_REG_DATA and a colour to indicate status in FWUKL1_MAP_XTR_DATA.
 *           Returns `Fail' and FwStateAttention1 if the input data is missformed.
 *
 * This code requires that the extra bytes generated from a fwCcpc_read() are first removed, via _fwUkl1_convert32To16BitWords().
 *
 */
dyn_string _fwUkl1_convertByteToIp(dyn_char ipAddress, dyn_string& exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  // First convert it from the 32bit to 16bit words. We don't care about the loss of mask
  // element from the confDB as we ignore them anyway.
  // Do this before we calculate the size of the MAC address. It also makes the later indexing more natural.
  _fwUkl1_convert32To16BitWords(ipAddress);
  
  // If the masks are in the data then they will be at the front i.e. lowest element numbers.
  // Determine the number of elements in the array and count backwards from there.
  // This will allow absolute indexing whether the masks are present or not.
  const int size = dynlen(ipAddress);
  // Check that it contains the appropriate number of elements.
  // Technically if it came from the configuration database it could contain the write mask tagged on the end.
  // This doesn't cause a problem as we use absolute indexes later, just require there to be 4 elements minimum (and the IP to be in those).
  if (4>size) {
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_convertByteToIp(): Input array contains " + size + " elements. Should be 4 elements big.", "1");
    return makeDynString("Fail", "FwStateAttention1", "");
  }// if(4>size)

  // Now we need to put the data into the appropriate format.
  string output;
  output += fwCcpc_convertByteToDec(ipAddress[size - 3]);
  output += ".";
  output += fwCcpc_convertByteToDec(ipAddress[size - 2]);
  output += ".";
  output += fwCcpc_convertByteToDec(ipAddress[size - 1]);
  output += ".";
  output += fwCcpc_convertByteToDec(ipAddress[size]);

  // Done.
  return makeDynString(output, "FwBackgroundInputText", "");
}// _fwUkl1_convertByteToIp()

/*!
 * This takes a description of the value to be set to the hardware and 
 * This takes the data, removes the appropriate bit using _fwUkl1_convertHwRegtoAbsRegHex() and then converts that bit to the
 * appropriate text description of the register. This is then returned.
 *
 * @param  absRegName The abstract register name that the data is being read for.
 * @param  hwRegName Name of the hardware register that is to be written to.
 * @param  description An dyn_string containing in the element FWUKL1_MAP_REG_DATA the description of the setting.
 *           In the element FWUKL1_MAP_XTR_DATA it must contain the element type to parse the data for, HW or DB.
 * @param  hexMask This is the hex string that is the mask for the given data. It is applied to the data before it is added to the dcData
 *           to ensure it does not go out of bounds. The mask is also added to the given mask such that it can be written to the hardware.
 * @param  parserInfo This information allows the data to be retrieve and formatted from the hardware register.
 *           It should have been returned from the fwUkl1_getParserInfo function in order to be properly formatted.
 * @param  dcData The dyn_char that the new value is to be written into. Returned by reference.
 * @param  dcMask The dyn_char that is the mask for the given data. Returned by reference.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 *
 * The dyn_char's for the data and mask can already contain data for that hardware register and only the bit relevant to the given
 * data and mask will be updated. These dyn_char's can also be empty and the function will create them such that they are the correct size.
 *
 * No changes are made to the data or mask unless the parsing is successful.
 */
void _fwUkl1_convertAbsRegDescriptionToByte(const string &absRegName, const string &hwRegName, const dyn_string &description, string hexMask, const dyn_dyn_string &parserInfo, dyn_char &dcData, dyn_char &dcMask, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);

  // First the description needs to be converted a hex value.
  // Initialise it from the description dyn_string. It is only
  // the first element we need to convert and ensure we preserve
  // all the extra data this way.
  dyn_string hexData = description;
  // Must have at least three list of arugments.
  // Number arguments.
  const int numParserArgs = dynlen(parserInfo);
  if (2<numParserArgs) {
    // Loop over all the arguments given from the second element onwards.
    // The first element gives the information for parsing to a hex number.
    bool matched = FALSE;
    for (int infoEle=2; infoEle<=numParserArgs; ++infoEle) {
      if (1<dynlen(parserInfo[infoEle])) {
        const string comparison = parserInfo[infoEle][1];
        const string value = parserInfo[infoEle][2];
        if (comparison == description[FWUKL1_MAP_REG_DATA]) {
          hexData[FWUKL1_MAP_REG_DATA] = value;
          matched = TRUE;
          break;
        }// if(comparison==description[FWUKL1_MAP_REG_DATA])
      }// if(1<dynlen(parserInfo[infoEle]))
      else
        exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_convertAbsRegDescriptionToByte(): For comparison to convert the description '" + description[FWUKL1_MAP_REG_DATA] + "' to a hex value for the register "+ absRegName + " not enough arguments found. Require 2, given " + dynlen(parserInfo[infoEle]) + ".", "2");
    }// for infoEle
    if (!matched) {
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_convertAbsRegDescriptionToByte(): Failed to convert the setting " + description[FWUKL1_MAP_REG_DATA] + " for the register " + absRegName + " as the description was not recognised.", "2");
    }// if(!matched)
  }// if(2<dynlen(listOfArgs))
  else
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_convertAbsRegDescriptionToByte(): When trying to parse the description (" + description[FWUKL1_MAP_REG_DATA] + ") did not retrieve enough arguments from the list to parse the data for the register " + absRegName + ". Required 3, given " + numParserArgs + ".", "2");
  
  // Now we should have some hex data lets convert it to a dyn_char.
  _fwUkl1_convertAbsRegHexToByte(absRegName, hwRegName, hexData, hexMask, parserInfo, dcData, dcMask, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_convertAbsRegDescriptionToByte(): Failed to convert the hex data (" + hexData[FWUKL1_MAP_REG_DATA] + ") for the register " + absRegName + " to a dyn_char to write to the hardware.", "2", exInfoSize);
  // Done.
  return;
}// _fwUkl1_convertAbsRegDescriptionToByte()

/*!
 * This takes the data, removes the appropriate bit using _fwUkl1_convertHwRegtoAbsRegHex() and then converts that bit to the
 * appropriate text description of the register. This is then returned.
 *
 * @param  absRegName The abstract register name that the data is being converted for.
 * @param  hwRegName The specific hardware register from the list associated with the abstract register
 *           to parse the data for.
 * @param  parserInfo This information allows the data to be retrieve and formatted from the hardware register.
 *           It should have been returned from the fwUkl1_getParserInfo function in order to be properly formatted.
 * @param  validationInfo This information allows the formatted data to be checked for certain conditions and highlighted
 *           according. It should have been returned from the fwUkl1_getValidationInfo function for proper formatting.
 * @param  rawData A dyn_mixed containing in the element FWUKL1_MAP_REG_DATA the data read back from the hardware register.
 *           In the element FWUKL1_MAP_XTR_DATA the register type should be include i.e. on of FWUKL1_HW_CONFIG_READ,
 *           FWUKL1_HW_STATUS_READ, FWUKL1_DB_READ.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return dyn_string Returns a text description and colour to associate with the description. Will be Fail and FwStateAttention1
 *           in the event of an error.
 */
dyn_string _fwUkl1_convertByteToAbsRegDescription(const string &absRegName, const string &hwRegName, const dyn_dyn_string &parserInfo, const dyn_dyn_string &validationInfo, const dyn_mixed &rawData, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  // This holds the output. First get the hex value, don't pad it. It confuses the comparison.
  dyn_string dsData = _fwUkl1_convertByteToAbsRegHex(absRegName, hwRegName, parserInfo, validationInfo, rawData, FALSE, exceptionInfo);
  if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_convertByteToAbsRegDescription(): Failed to retrieve the hex value from the hardware. Cannot convert it to a description.", "2", exInfoSize)) {
    // Must have at least three list of arguments.
    const int numArgLists = dynlen(parserInfo);
    if (2<numArgLists) {
      // Loop through the remaining list of arguments until we find a match.
      bool matched = FALSE;
      for (int argListNum=2; argListNum<=numArgLists; ++argListNum) {
        dyn_string arguments = parserInfo[argListNum];
        // We must have at least two argument to make the decision on.
        if (1<dynlen(arguments)) {
          // We take the first argument and compare it to the value we have read back.
          // This is the hex number.
          // If the value read back matches then assign the description from the second
          // argument to the value to return for this register and stop looking for
          // further comparisons.
          string comparison  = arguments[1];
          string description = arguments[2];
          if (dsData[FWUKL1_MAP_REG_DATA] == comparison) {
            dsData[FWUKL1_MAP_REG_DATA] = description;
            matched = TRUE;
            break;
          }// if matched
        }// if not enough arguments.
        else
          exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_convertByteToAbsRegDescription(): Not given a large enough list of arguments. Require at least 2, given " + numArgLists + ". Hex value of data read back has been returned.", "2");
      }// for argListNum
      if (!matched) {
        exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_convertByteToAbsRegDescription(): Failed to convert the hex value 0x" + dsData[FWUKL1_MAP_REG_DATA] + " for the register " + absRegName + " to a text description as the value was not recognised. The value was returned.", "2");
        dsData[FWUKL1_MAP_XTR_DATA] = "FwStateAttention1";
        dsData[FWUKL1_MAP_XTR_DATA+1] = "Failed to match value to allowed descriptions.";
      }
    }// if not enough arguments.
    else
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_convertByteToAbsRegDescription(): Not given a large enough list of arguments. Require at least 3, given " + numArgLists + ". Hex value of data read back has been returned.", "2");
  }// if got hex value.
  // Done.
  return dsData;
}// _fwUkl1_convertByteToAbsRegDescription()

/*!
 * This takes the data as a hex string and converts it to a dyn_char which can be written to the hardware register. It also converts the mask
 * from the given string to a dyn_char which can be sent to the server. The string is treated as little endian.
 *
 * @param  regName The abstract register name that the data is being read for.
 * @param  hwRegName Name of the hardware register that is to be written to.
 * @param  hexData An dyn_string containing in the element FWUKL1_MAP_REG_DATA the hex string representing the data.
 *           In the element FWUKL1_MAP_XTR_DATA it must contain the element type to parse the data for, HW or DB.
 *           The hex string need not be padded.
 * @param  hexMask This is the hex string that is the mask for the given data. It is applied to the data before it is added to the dcData
 *           to ensure it does not go out of bounds. The mask is also added to the given mask such that it can be written to the hardware.
 * @param  parserInfo This information allows the data to be retrieve and formatted from the hardware register.
 *           It should have been returned from the fwUkl1_getParserInfo function in order to be properly formatted.
 * @param  dcData The dyn_char that the new value is to be written into. Returned by reference.
 * @param  dcMask The dyn_char that is the mask for the given data. Returned by reference.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 *
 * The dyn_char's for the data and mask can already contain data for that hardware register and only the bit relevant to the given
 * data and mask will be updated. These dyn_char's can also be empty and the function will create them such that they are the correct size.
 *
 * No changes are made to the data or mask unless the parsing is successful.
 */
void _fwUkl1_convertAbsRegHexToByteLittleEndian(const string &regName, const string &hwRegName, dyn_string hexData, string hexMask, const dyn_dyn_string &parserInfo, dyn_char &dcData, dyn_char &dcMask, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  // Must have at least one list of arugments.
  if (0<dynlen(parserInfo)) {
    // Need one argument.
    if (0<dynlen(parserInfo[1])) {
      // The start bit is the position in the register and as we will need to place the bits within a dyn_char need to know
      // the byte (element) to put it in plus the bit within that element.
      const int startBit = parserInfo[1][1];
      // See what register type it is.
      int regType = fwUkl1_getHwRegType(makeDynString(regName,hwRegName), hexData[FWUKL1_MAP_XTR_DATA], exceptionInfo);
      if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_convertAbsRegHexToByte(): Failed to determine the hardware register type for " + regName + ". Cannot continue with the data parsing.", "2", exInfoSize)) {
        // Get the number of bytes that are required for the register.
        int numBytes;
        // Look in the default settings for this register, they should always be the same as the values for each board and then we don't
        // have to specify the UKL1 board specifically. 
        dyn_dyn_anytype settingsList;
        // Have to strip the leading full stop from the hardware name if it is there.
        string strippedHwRegName = strltrim(strrtrim(hwRegName,"."),".");
        fwHw_getDefaultSpecificSettings(FWUKL1_HW_TYPE, makeDynString(strippedHwRegName), settingsList);
        // Where we look for it depends on the register type.
        switch (regType) {
          case FWHW_LBUS:
            // The FPGA registers are given in 16-bit words we are dealing with bytes.
            // Actually as far as the hardware tool is concerned they are 32-bit words, but we will take that into account later
            // as it is more complex than just adding padding to one end. Can be considered 16-bit words.
            numBytes = (int)settingsList[1][3] * 2;// 3rd item in first register, only ever ask for 1 reg.
            break;
          case FWHW_GBE:
            // The GBE registers are given in 32-bit words but we are dealing with bytes.
            numBytes = (int)settingsList[1][2] * 4;// 2nd item in first register, only ever ask for 1 reg.
            break;
          case FWHW_I2C:
            // The I2C registers are given in bytes and we are dealing with bytes.
            numBytes = (int)settingsList[1][4];// 1st item in first register, only ever ask for 1 reg.
            break;
          case FWHW_UserSpecific:
            // The user specific registers are FIFOs in our case and we need to know how many bytes in
            // total to get, which is specified in element 4. This is the number of words and must
            // be multiplied by the size of the words.
            numBytes = (int)settingsList[1][4] * (int)settingsList[1][3];// Only ever ask for 1 reg.
            break;
          default:
            numBytes = -1;
            exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_convertAbsRegHexToByte(): Failed to recognise the register type " + regType + ". Could not determine number of bytes required for the given register. Cannot write to register without possibly corrupting it.", "2");
            break;
        }// switch(regType)
        if (-1!=numBytes) {
          // Bit shift the hex data and mask.
          fwUkl1_leftShiftString(hexData[FWUKL1_MAP_REG_DATA], startBit);
          fwUkl1_leftShiftString(hexMask, startBit);
          // Now ensure there are enough 0's for the entire word by adding them to the beginning of the string.
          fwUkl1_padString(hexData[FWUKL1_MAP_REG_DATA], numBytes);
          fwUkl1_padString(hexMask, numBytes);
          // Need to swap the bytes of the data and the mask around as we are dealing with little endian data.
          string swappedHexData;
          string swappedHexMask;
          // Must be padded to a word boundary so can gauranty we have an even number of characters.
          const int numCharacters = strlen(hexData[FWUKL1_MAP_REG_DATA]);
          // Strings are zero based!
          for (int i=numCharacters-1; i>=0; i-=2) {
            // Remember a pair of characters is a byte and we preserve the bit order in the bytes.
            swappedHexData += hexData[FWUKL1_MAP_REG_DATA][i-1];
            swappedHexData += hexData[FWUKL1_MAP_REG_DATA][i];
            swappedHexMask += hexMask[i-1];
            swappedHexMask += hexMask[i];
          }// for i
          // Now convert to a dyn_char. Exactly how depends on the register type.
          dyn_char tmpData;
          dyn_char tmpMask;
          if (FWHW_LBUS==regType) {
            tmpData = _fwUkl1_fpgaizeData(swappedHexData);
            tmpMask = _fwUkl1_fpgaizeData(swappedHexMask);
          }// if(FWHW_LBUS==regType)
          else {
            tmpData = fwCcpc_convertHexToByte(swappedHexData);
            tmpMask = fwCcpc_convertHexToByte(swappedHexMask);
          }// else(FWHW_LBUS==regType)

          // Now we need to correct for the FPGA data after _fwUkl1_fpgaizeData, which expects normal data!
          if (FWHW_LBUS==regType)
            numBytes *= 2;

          // We need to ensure the data is of the correct type if empty and has at least 1 element.
          if (0==dynlen(dcData))
            dcData = makeDynChar();
          // and that the dcData is the correct size
          while (numBytes > dynlen(dcData))
            dynAppend(dcData, 0);
          
          // Same for the mask.
          if (0==dynlen(dcMask))
            dcMask = makeDynChar();
          // and that the dcData is the correct size
          while (numBytes > dynlen(dcMask))
            dynAppend(dcMask, 0);
          
          // Now loop over all the data and insert it into the existing data ensure that it doesn't exceed its bounds.
          // Same goes for the mask.
          for (int i=1; i<=numBytes; ++i) {
            // The data must be prepared first by zeroing the bits we intend to over write. AND with NOT the mask to do this.
            // NOT mask gives you zeros where you want to write and ones where you don't. When you AND with the data the zeros
            // cause the bits you want to write to become zero. The ones means you leave the rest of the data untouched.
            dcData[i] &= ~tmpMask[i];
            // Now we can OR with the data (with mask taken into account) and it will set the zeros to what ever it is.
            dcData[i] |= tmpData[i] & tmpMask[i];
            // The mask is simplier as we just want to ensure the mask bits are 1.
            dcMask[i] |= tmpMask[i];
          }// for i
        }// if(-1!=numBytes)
        // No else already flagged the error.
      }// if fwUkl1_getHwRegType successful.
    }// if(1<dynlen(listOfArgs[1]))
    else
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_convertAbsRegHexToByte(): Not given enough arguments to determine the starting bit for the hex value. Require 1, given " + dynlen(parserInfo[1]) + ".", "2");
  }// if(0<dynlen(listOfArgs))
  else
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_convertAbsRegHexToByte(): Not given a large enough list of arguments. Require 1, given " + dynlen(parserInfo) + ".", "2");
  // Done.
  return;
}// _fwUkl1_convertAbsRegHexToByteLittleEndian()

/*!
 * This takes the data from the hardware register and converts it to a hex number based on the requirements
 * for the abstract hardware register. Ensures that the bytes are treated as little endian.
 *
 * @param  absRegName The abstract register name that the data is being read for.
 * @param  hwRegName The specific hardware register from the list associated with the abstract register
 *           to parse the data for.
 * @param  parserInfo This information allows the data to be retrieve and formatted from the hardware register.
 *           It should have been returned from the fwUkl1_getParserInfo function in order to be properly formatted.
 * @param  validationInfo This information allows the formatted data to be checked for certain conditions and highlighted
 *           according. It should have been returned from the fwUkl1_getValidationInfo function for proper formatting.
 * @param  rawData A dyn_mixed containing in the element FWUKL1_MAP_REG_DATA the data read back from the hardware register.
 *           In the element FWUKL1_MAP_XTR_DATA the register type should be include i.e. on of FWUKL1_HW_CONFIG_READ,
 *           FWUKL1_HW_STATUS_READ, FWUKL1_DB_READ.
 * @param  pad Optionally the returned hex value can be padded to the maxmimum number of bytes it can take.
 * @param  forStatus If the description to be retrieved for the status display or configuration display.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return dyn_string Returns a dyn_string containing the value read back from the hardware register for the given
 *           abstract register in FWUKL1_MAP_REG_DATA element and the colour (if any) associated with that value
 *           in FWUKL1_MAP_XTR_DATA. Will be padded to the nearest byte to the length. Will be Fail and FwStateAttention1
 *           in the event of a read failure.
 */
dyn_string _fwUkl1_convertByteToAbsRegHexLittleEndian(const string &absRegName, const string &hwRegName, const dyn_dyn_string &parserInfo, const dyn_dyn_string &validationInfo, const dyn_mixed &rawData, bool pad, dyn_string &exceptionInfo) {
  // We must reverse the byte ordering of the array.
  // For this we will need to know its size.
  const int numBytes = dynlen(rawData[FWUKL1_MAP_REG_DATA]);
  // This stores the reversed bytes.
  // Must intialise it from the input raw data as we
  // need to ensure we get the rest of the elements
  // from the dym structure.
  dyn_mixed swappedRawData = rawData;
  if (0 < numBytes) {
    // Have a none zero number of bytes.
    // Loop through them swapping the first element for last, 2nd for 2nd to last etc..
    for (int byte=1; byte<=numBytes; ++byte)
      swappedRawData[FWUKL1_MAP_REG_DATA][numBytes-(byte-1)] = rawData[FWUKL1_MAP_REG_DATA][byte];
  }// if(0<numBytes)
  else if (0 == numBytes)
    return "";
  else {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_convertByteToAbsRegHexLittleEndian(): PVSS error, failed to determine size of byte array, could not convert to hex string.", "1");
    return "Fail";
  }// else failed
  
  // Done.
  return _fwUkl1_convertByteToAbsRegHex(absRegName, hwRegName, parserInfo, validationInfo, swappedRawData, pad, exceptionInfo);
}// _fwUkl1_convertByteToAbsRegHexLittleEndian()

/*!
 * This takes the data as a hex string and converts it to a dyn_char which can be written to the hardware register. It also converts the mask
 * from the given string to a dyn_char which can be sent to the server.
 *
 * @param  absRegName The abstract register name that the data is being read for.
 * @param  hwRegName Name of the hardware register that is to be written to.
 * @param  hexData An dyn_string containing in the element FWUKL1_MAP_REG_DATA the hex string representing the data.
 *           In the element FWUKL1_MAP_XTR_DATA it must contain the element type to parse the data for, HW or DB.
 *           The hex string need not be padded.
 * @param  hexMask This is the hex string that is the mask for the given data. It is applied to the data before it is added to the dcData
 *           to ensure it does not go out of bounds. The mask is also added to the given mask such that it can be written to the hardware.
 * @param  parserInfo This information allows the data to be retrieve and formatted from the hardware register.
 *           It should have been returned from the fwUkl1_getParserInfo function in order to be properly formatted.
 * @param  dcData The dyn_char that the new value is to be written into. Returned by reference.
 * @param  dcMask The dyn_char that is the mask for the given data. Returned by reference.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 *
 * The dyn_char's for the data and mask can already contain data for that hardware register and only the bit relevant to the given
 * data and mask will be updated. These dyn_char's can also be empty and the function will create them such that they are the correct size.
 *
 * No changes are made to the data or mask unless the parsing is successful.
 */
void _fwUkl1_convertAbsRegHexToByte(const string &regName, const string &hwRegName, dyn_string hexData, string hexMask, const dyn_dyn_string &parserInfo, dyn_char &dcData, dyn_char &dcMask, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  // Must have at least one list of arguments.
  if (0<dynlen(parserInfo)) {
    // Need one argument.
    if (0<dynlen(parserInfo[1])) {
      // The start bit is the position in the register and as we will need to place the bits within a dyn_char need to know
      // the byte (element) to put it in plus the bit within that element.
      const int startBit = parserInfo[1][1];
      // See what register type it is.
      int regType = fwUkl1_getHwRegType(makeDynString(regName,hwRegName), hexData[FWUKL1_MAP_XTR_DATA], exceptionInfo);
      if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_convertAbsRegHexToByte(): Failed to determine the hardware register type for " + regName + ". Cannot continue with the data parsing.", "2", exInfoSize)) {
        // Get the number of bytes that are required for the register.
        int numBytes = fwUkl1_getHwRegistersSizeInBytes(makeDynString(hwRegName), makeDynInt(regType), exceptionInfo)[hwRegName];
        if (-1!=numBytes) {
          // Now convert to a dyn_char. Exactly how depends on the register type.
          dyn_char tmpData;
          dyn_char tmpMask;
          // Bit shift the hex data and mask, this is the same for all register types.
          fwUkl1_leftShiftString(hexData[FWUKL1_MAP_REG_DATA], startBit);
          fwUkl1_leftShiftString(hexMask, startBit);
          // Now ensure there are enough 0's for the entire word by adding them to the beginning of the string.
          // We have not yet converted the LBUS registers from 32- to 16-bit words
          // and the hardware tool considers them to be 32-bit, the number of
          // bytes is therefore incorrect at this stage and we should not pad
          // the data by so much.
          if (FWHW_LBUS==regType) {
            fwUkl1_padString(hexData[FWUKL1_MAP_REG_DATA], numBytes/2);
            fwUkl1_padString(hexMask, numBytes/2);
            tmpData = _fwUkl1_fpgaizeData(hexData[FWUKL1_MAP_REG_DATA]);
            tmpMask = _fwUkl1_fpgaizeData(hexMask);
          }// if(FWHW_LBUS==regType)
          else {
            fwUkl1_padString(hexData[FWUKL1_MAP_REG_DATA], numBytes);
            fwUkl1_padString(hexMask, numBytes);
            tmpData = fwCcpc_convertHexToByte(hexData[FWUKL1_MAP_REG_DATA]);
            tmpMask = fwCcpc_convertHexToByte(hexMask);
          }// else(FWHW_LBUS==regType)

          // We need to ensure the data is of the correct type if empty and has at least 1 element.
          if (0==dynlen(dcData))
            dcData = makeDynChar();
          // and that the dcData is the correct size
          while (numBytes > dynlen(dcData))
            dynAppend(dcData, 0);
          
          // Same for the mask.
          if (0==dynlen(dcMask))
            dcMask=makeDynChar();
          // and that the dcData is the correct size
          while (numBytes > dynlen(dcMask))
            dynAppend(dcMask, 0);
          
          // Now loop over all the data and insert it into the existing data ensure that it doesn't exceed its bounds.
          // Same goes for the mask.
          for (int i=1; i<=numBytes; ++i) {
            // The data must be prepared first by zeroing the bits we intend to over write. AND with NOT the mask to do this.
            // NOT mask gives you zeros where you want to write and ones where you don't. When you AND with the data the zeros
            // cause the bits you want to write to become zero. The ones means you leave the rest of the data untouched.
            dcData[i] &= ~tmpMask[i];
            // Now we can OR with the data (with mask taken into account) and it will set the zeros to what ever it is.
            dcData[i] |= tmpData[i] & tmpMask[i];
            // The mask is simplier as we just want to ensure the mask bits are 1.
            dcMask[i] |= tmpMask[i];
          }// for i
        }// if(-1!=numBytes)
        // No else already flagged the error.
      }// if fwUkl1_getHwRegType successful.
    }// if(1<dynlen(parserInfo[1]))
    else
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_convertAbsRegHexToByte(): Not given enough arguments to determine the starting bit for the hex value. Require 1, given " + dynlen(parserInfo[1]) + ".", "2");
  }// if(0<dynlen(listOfArgs))
  else
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_convertAbsRegHexToByte(): Not given a large enough list of arguments. Require 1, given " + dynlen(parserInfo) + ".", "2");
  // Done.
  return;
}// _fwUkl1_convertAbsRegHexToByte()

/*!
 * This takes the data from the hardware register and converts it to a hex number based on the requirements
 * for the abstract hardware register.
 *
 * @param  regName The abstract register name that the data is being read for.
 * @param  hwRegName The specific hardware register from the list associated with the abstract register
 *           to parse the data for.
 * @param  parserInfo This information allows the data to be retrieve and formatted from the hardware register.
 *           It should have been returned from the fwUkl1_getParserInfo function in order to be properly formatted.
 * @param  validationInfo This information allows the formatted data to be checked for certain conditions and highlighted
 *           according. It should have been returned from the fwUkl1_getValidationInfo function for proper formatting.
 * @param  rawData A dyn_mixed containing in the element FWUKL1_MAP_REG_DATA the data read back from the hardware register.
 *           In the element FWUKL1_MAP_XTR_DATA the register type should be include i.e. on of FWUKL1_HW_CONFIG_READ,
 *           FWUKL1_HW_STATUS_READ, FWUKL1_DB_READ.
 * @param  pad Optionally the returned hex value can be padded to the maxmimum number of bytes it can take.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return dyn_string Returns a dyn_string containing the value read back from the hardware register for the given
 *           abstract register in FWUKL1_MAP_REG_DATA element and the colour (if any) associated with that value
 *           in FWUKL1_MAP_XTR_DATA. Will be padded to the nearest byte to the length. Will be Fail and FwStateAttention1
 *           in the event of a read failure.
 */
dyn_string _fwUkl1_convertByteToAbsRegHex(const string &absRegName, const string &hwRegName, const dyn_dyn_string &parserInfo, const dyn_dyn_string &validationInfo, const dyn_mixed &rawData, bool pad, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  // This will hold the output data.
  dyn_string dsData = makeDynString("Fail", "FwStateAttention1", "");
  // Must have at least one list of arugments.
  if (0 < dynlen(parserInfo)) {
    // Need two arguments.
    if (1 < dynlen(parserInfo[1])) {
      int startBit = parserInfo[1][1];
      int length   = parserInfo[1][2];
      // Need to decide if it is for an LBUS register or not.
      int regType = fwUkl1_getHwRegType(makeDynString(absRegName,hwRegName), rawData[FWUKL1_MAP_XTR_DATA], exceptionInfo);
      if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_convertByteToAbsRegHex(): Failed to determine the hardware register type for " + absRegName + ". Cannot continue with the data parsing.", "2", exInfoSize)) {
        bool fpga = (regType==FWHW_LBUS ? TRUE:FALSE);
        dsData[FWUKL1_MAP_REG_DATA] = _fwUkl1_convertByteToHex(rawData[FWUKL1_MAP_REG_DATA], startBit, length, fpga);
        if (0 == dynlen(validationInfo))
          dsData[FWUKL1_MAP_XTR_DATA] = "FwBackgroundInputText";
        else {
          // Start at the lowest limit and break once we have found the appropriate range.
          for (int arg=1; arg<=dynlen(validationInfo); ++arg) {
            // Need at least 2 arguments for the value and colour.
            if (1 < dynlen(validationInfo[arg])) {
              string value = validationInfo[arg][1];
              string colour = validationInfo[arg][2];
              if (dsData[FWUKL1_MAP_REG_DATA] <= value) {
                dsData[FWUKL1_MAP_XTR_DATA] = colour;
                break;
              }// if lower than limit.
            }// if got value and colour.
            else
              exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_convertByteToAbsRegHex(): When checking the range limit for " + absRegName + " found only " + dynlen(validationInfo[arg]) + " arguments, require 2. One for limit and the other for colour associated with range.", "");
          }// for arg
          // If after the loop we still haven't assigned a colour then give it one.
          if ("FwStateAttention1" == dsData[FWUKL1_MAP_XTR_DATA])
            dsData[FWUKL1_MAP_XTR_DATA] = "FwBackgroundInputText";
        }// else choose colour from range.
        
        // Now pad the data before returning it.
        if (pad) {
          // Divide by length in bits 8 and then round up. This will give the minimum number of bytes that can contain this number of bits.
          int padding = ceil(length/8.0);
          fwUkl1_padString(dsData[FWUKL1_MAP_REG_DATA], padding);
        }// if(pad)
      }// if got reg type
    }// if(1<dynlen(parserInfo[1]))
    else
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_convertByteToAbsRegHex(): Not given enough arguments to pass to _fwUkl1_convertByteToHex(). Require 2, given " + dynlen(parserInfo[1]) + ".", "2");
  }// if(0<dynlen(parserInfo))
  else
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_convertByteToAbsRegHex(): Not given a large enough list of arguments. Require 1, given " + dynlen(parserInfo) + ".", "2");
  // Done.
  return dsData;
}// _fwUkl1_convertByteToAbsRegHex()

/*!
 * Takes a string representing a hex number and converts it to a dyn_char assuming the input data is little endian.
 *
 * @param  dcData A dyn_char containing the data read back from the hardware register.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return dyn_char Returns the dyn_char.
 *
 * The fwCcpc_convertHexToByte function assumes big endian data.
 */
dyn_char fwUkl1_convertHexToByteLittleEndian(const string &sData, dyn_string &exceptionInfo) {
  // Convert the hex data to a dyn_char.
  dyn_char dcData = fwCcpc_convertHexToByte(sData);
  // We must reverse the byte ordering of the array.
  // For this we will need to know its size.
  const int numBytes = dynlen(dcData);
  // This stores the reversed bytes.
  dyn_char swappedDcData;
  if (0 <= numBytes) {
    // Have a none zero number of bytes.
    // Loop through them swapping the first element for last, 2nd for 2nd to last etc..
    for (int byte=1; byte<=numBytes; ++byte)
      swappedDcData[numBytes-(byte-1)] = dcData[byte];
  }// if(0<numBytes)
  else
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_convertHexToByteLittleEndian(): PVSS error, failed to determine size of byte array, could not convert to hex string.", "1");
  
  // Done.
  return swappedDcData;
}// fwUkl1_convertHexToByteLittleEndian()

/*!
 * This takes the data that is read by from the L1 board FPGAs (LBUS), which is read as a series of 32-bit words,
 * and converts it to a series of 16-bit words can then be decoded to return the register value.
 *
 * @param  dcData The dyn_char array that contains the bytes read back from the hardware, will have the upper 2-bytes
 *           removed from the array. Assume byte ordering that matches that done read by the FW CCPC library. Return
 *           by reference.
 * @return dyn_char The dyn_char that now has its bytes in an order that can be decoded by the fwCcpc library
 *           to return a string representing the hex number.
 *
 * The function performs two tasks. First it considers the byte array as a group of 32-bit numbers and removes
 * the upper two bytes to reduce them to a 16-bit number. The resulting array must then be word swapped, however
 * the words must maintain their internal byte ordering. If the register is not a mutliple of 32, which if it is a
 * valid LBUS read it must be, then it will be padded.
 */
void _fwUkl1_convert32To16BitWords(dyn_char& dcData) {
  // Convert the read data from a series of 32-bit words to a series of 16-bit words.

  // Get the number of bytes.
  int numBytes = dynlen(dcData);
  // Check that we have a multiple of 32-bits.
  while (0 != numBytes%4) {
    // The most significant word should be in the last four bytes. We need to pad this word.
    // The most significant byte of this word is the at size-4.
    int position = numBytes - numBytes%4 + 1;
    dynInsertAt(dcData, 0x00, position);
    numBytes = dynlen(dcData);
  }// while(0!=(numBytes%4))

  // The loop must run over the number of 32-bit words in the read data.
  const unsigned total32Words = dynlen(dcData)/4;
  for (unsigned word=0; word<total32Words; ++word) {
    // We need to remove the first two elements in that word.
    // This will follow the sequence of odd numbers.
    // After we delete the first element the second element becomes the first, so we always delete the same one.
    dynRemove(dcData, 2*word+1);
    dynRemove(dcData, 2*word+1);
  }// for word

  // In the event that we have read back more than one word then we will need to swap the ordering of the 16-bit
  // words contained within the array.
  const unsigned total16Words = dynlen(dcData)/2;
  if (1 < total16Words) {
    // Copy the data into a temporary array.
    const dyn_char tmpData = dcData;
    // Loop over all the 16-bit words of the array and put them back into the original in the reverse order.
    const unsigned totalElements = dynlen(dcData);
    for (unsigned word=0; word<totalElements; word+=2) {
      // Need to keep the byte ordering within the word, but word swap the array. 16-bit words.
      dcData[word+1] = tmpData[totalElements-word-1];
      dcData[word+2] = tmpData[totalElements-word];
    }// for word
  }// if(1<total16Words)
  
  // Done.
  return;
}// _fwUkl1_convert32To16BitWords()

/*!
 * Takes a group of parsed hardware register values and returns the majority value.
 *
 * @param  absRegName The abstract register name that the data is being read for.
 * @param  allData Each element should contain the parsed data from each hardware register associated
 *           with the given abstract register.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return dyn_string Returns a text description, colour and message to associate with the description.
 *
 * If not all the values are the same the colour will be set to FwStateAttention1 and a message
 * constructed to indicate the values that were different. In the event of a tie then the value
 * that first became the majoritory value will be kept.
 */
dyn_string _fwUkl1_combineAbsValueMajorityVote(const string &absRegName, const dyn_dyn_string &allData, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  dyn_string combinedOutput = makeDynString("Fail", "FwStateAttention1", "");
  // We will use a map to count the number of times each parsed
  // value occurs in allData in order to see which occurs most often.
  mapping parsedValueCount;
  // We will keep a note of the current number of times the majoritory
  // value has been seen. We can then look to see if the value we just
  // added in the loop is the new majoritory and save it.
  int majoritoryCount;
  string majoritoryValue;
  // Need to loop over all the data we have been given.
  const int numEntries = dynlen(allData);
  for (int i=1; i<=numEntries; ++i) {
    // We use the value found in the allData element as a key into
    // the map and then count how many times this key was seen.
    // PVSS will correctly initialise the map value to be 0 the first
    // time the key is seen.
    const string key = allData[i][FWUKL1_MAP_REG_DATA];
    parsedValueCount[key] += 1;
    if (parsedValueCount[key]>majoritoryCount) {
      majoritoryCount = parsedValueCount[key];
      majoritoryValue = key;
    }// if(parsedValueCount[key]>majoritoryCount)
  }// for i
  // Check to see if the majoritory value was unanimous.
  combinedOutput[FWUKL1_MAP_REG_DATA] = majoritoryValue;
  if (1<dynlen(mappingKeys(parsedValueCount))) {
    // No unanimous, but have a majoritory.
    combinedOutput[FWUKL1_MAP_XTR_DATA] = "FwStateAttention1";
    combinedOutput[FWUKL1_MAP_XTR_DATA+1] = "WARNING - Majoritory vote was not unanimous.";
  }// if not unanimous
  else {
    // Success.
    combinedOutput[FWUKL1_MAP_XTR_DATA] = "FwBackgroundInputText";
    combinedOutput[FWUKL1_MAP_XTR_DATA+1] = fwUkl1_getToolTipText(absRegName, exceptionInfo);
    fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_combineAbsValueMajorityVote(): Failed to retrieve the tool tip text for the register "+absRegName+". Tool tip text could be out of date.", "2", exInfoSize);
  }// else unanimous vote
  // Done.
  return combinedOutput;
}// _fwUkl1_combineAbsValueMajorityVote()

/*!
 * Takes a group of parsed hardware register values and subtracts all subsequent values from the first.
 *
 * @param  absRegName The abstract register name that the data is being read for.
 * @param  allData Each element should contain the parsed data from each hardware register associated
 *           with the given abstract register. The parsed data must be a decimal value.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return dyn_string Returns a decimal value, colour and message to associate.
 */
dyn_string _fwUkl1_combineAbsRegSubtraction(const string &absRegName, const dyn_dyn_string &allData, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  dyn_string combinedOutput = makeDynString("Fail", "FwStateAttention1", "");
  // Loop through all the given data elements and substract from the first element all subsequent elements.
  const int numValues = dynlen(allData);
  if (0 < numValues) {
    // Save the first element.
    // We can converting from a string here, which represents a decimal number.
    int result = (int)allData[1][FWUKL1_MAP_REG_DATA];
    // Subtract all subsequent values.
    for (int i=2; i<=numValues; ++i) {
      // We can converting from a string here, which represents a decimal number.
      result -= (int)allData[i][FWUKL1_MAP_REG_DATA];
    }// for i
    // Now save the result in the combinedOutput.
    // We want the data as a string, so do an explicit cast here.
    combinedOutput[FWUKL1_MAP_REG_DATA] = (string)result;
    combinedOutput[FWUKL1_MAP_XTR_DATA] = "FwBackgroundInputText";
    combinedOutput[FWUKL1_MAP_XTR_DATA+1] = fwUkl1_getToolTipText(absRegName, exceptionInfo);
    fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_combineAbsValueMajorityVote(): Failed to retrieve the tool tip text for the register "+absRegName+". Tool tip text could be out of date.", "2", exInfoSize);
  }// if(0<numValues)
  else {
    combinedOutput[FWUKL1_MAP_XTR_DATA+1] = "No data present to subtract.";
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_combineAbsRegSubtraction(): No data was given to the combiner. Could not return a value.", "2");
  }// else(0<numValues)
  // Done.
  return combinedOutput;
}// _fwUkl1_combineAbsRegSubtraction()

/*!
 * Take a group of parsed hardware register values and ORs the values. Disable is considered high.
 *
 * @param  absRegName The abstract register name that the data is being read for.
 * @param  allData Each element should contain the parsed data from each hardware register associated
 *           with the given abstract register. The parsed data must be either Disable(d) or Enable(d).
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return dyn_string Returns a decimal value, colour and message to associate.
 *
 * Either Disable or Disabled can be given, allowing it to handle status and config parsing. If no match
 * is made the value in the first element of allData is returned.
 */
dyn_string _fwUkl1_combineAbsRegOrDisableHigh(const string &absRegName, const dyn_dyn_string &allData, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  dyn_string combinedOutput = makeDynString("Fail", "FwStateAttention1", "");
  const int numValues = dynlen(allData);
  if (0 < numValues) {
    // Initialise the return value to what we are given in the first element of allData
    // and then check for a disable(d) in the given element and return that.
    int value = 1;
    do {
      combinedOutput = allData[value];
      ++value;
      if (value>numValues) {
        break;
      }// if(value>numValues)
    } while (!patternMatch("Disable*", combinedOutput[FWUKL1_MAP_REG_DATA]));
  }// if(0<numValues)
  // Done.
  return combinedOutput;
}// _fwUkl1_combineAbsRegOrDisableHigh()

/*!
 * Combines the data parsed from each hardware register for a given abstract register into a single value.
 *
 * @param  regName The abstract register name that the data is being read for.
 * @param  allData Each element should contain the parsed data from each hardware register associated
 *           with the given abstract register.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return dyn_string Returns a text description, colour and message to associate with the description.
 *
 * If the abstract register is not to have its data combined it will simply return the first entry
 * in the allData variable.
 */
dyn_string _fwUkl1_absRegDataCombiner(const string &absRegName, const dyn_dyn_string &allData, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  dyn_string combinedOutput = makeDynString("Fail", "FwStateAttention1", "");
  if (1>dynlen(allData))
    return combinedOutput;
  // We need to know how to combine this data so ask the data point.
  const string combinerFunc = fwUkl1_getCombinerFunction(absRegName, exceptionInfo);
  if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_absRegDataCombiner(): Failed to retrieve the combiner function for the abstract register '"+absRegName+"'. Returned first read register value only.", "2", exInfoSize)) {
    switch (combinerFunc) {
      case "_fwUkl1_combineAbsValueMajorityVote":
        combinedOutput = _fwUkl1_combineAbsValueMajorityVote(absRegName, allData, exceptionInfo);
        break;
      case "_fwUkl1_combineAbsRegSubtraction":
        combinedOutput = _fwUkl1_combineAbsRegSubtraction(absRegName, allData, exceptionInfo);
        break;
      case "_fwUkl1_combineAbsRegOrDisableHigh":
        combinedOutput = _fwUkl1_combineAbsRegOrDisableHigh(absRegName, allData, exceptionInfo);
        break;
      case FWUKL1_NO_COMBINER:
        // In this case we are not to do anything so just return the first element of the allData.
        combinedOutput = allData[1];
        break;
      default:
        exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_absRegDataCombiner(): Unrecognised combiner function '"+combinerFunc+"'given for abstract register '"+absRegName+". Returned first read register value only.", "2");
        combinedOutput = makeDynString(allData[1][FWUKL1_MAP_REG_DATA], "FwStateAttention1", "WARANING - Don't know how to combine data, first value only.");
        break;
    }// switch (combinerFunc)
    fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_absRegDataCombiner(): Failed to combine the data for '"+absRegName+"' using '"+combinerFunc+"'.", "2", exInfoSize);
  }// if combiner function retrieved.
  else {
    // Have at least one entry to get this far.
    combinedOutput = makeDynString(allData[1][FWUKL1_MAP_REG_DATA], "FwStateAttention1", "WARNING - Don't know how to combine data, first value only.");
  }// else combiner function not retrieved.
  // Done.
  return combinedOutput;
}// _fwUkl1_absRegDataCombiner()

/*!
 * Parses the given input data map and converts it to values that can be written to the hardware or saved in the ConfDB.
 *
 * @param  inData This is a map that associates an absrtact register name with the data that is to be
 *           written and any extra information required to write that data. In the inData key should be
 *           an abstract register and the value a dyn_string. In the element FWUKL1_MAP_REG_DATA is the
 *           value to be written and this function will add to FWUKL1_MAP_XTR_DATA the register type.
 * @param  outData This is a map that associates a hardware register name with the dyn_char data to be written
 *           and the dyn_char mask to be applied to that data on the server end. FWUKL1_MAP_REG_DATA contains the
 *           hardware register data and FWUKL1_MAP_XTR_DATA contains the mask. These are kept separate deliberately.
 *           In addition the write function is in FWUKL1_MAP_XTR_DATA+1 and register type in FWUKL1_MAP_XTR_DATA+2.
 * @param  registerType This is the register type that the data is to be parsed for. It should be either
 *           FWUKL1_HW_WRITE or FWUKL1_DB_WRITE.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 */
void _fwUkl1_parseRegistersForSet(mapping &inData, mapping &outData, const string &registerType, dyn_string &exceptionInfo) {
  // Keep track of the exception information.
  int exInfoSize = dynlen(exceptionInfo);
  // We use the outData mapping to associate the register names with the data that is to be written.
  // By using a mapping we ensure that multiple entries for a given hardware register don't occur
  // and look ups should be faster than searching dyn structures and counting the index.
  // We will use the same structure as the inData with register values in the FWUKL1_MAP_REG_DATA element
  // and put the mask in the FWUKL1_MAP_XTR_DATA element. In this case everything will be dyn_char's.
  // The name of the function to use to write the data is given in FWUKL1_MAP_XTR_DATA+1.
  // Get all the register names were given and loop over them parsing the data as necessary.
  const dyn_string absRegNames = mappingKeys(inData);
  const int numAbsRegs = dynlen(absRegNames);
  // Now loop over the keys and parse all the data.
  for (int absRegNum=1; absRegNum<=numAbsRegs; ++absRegNum) {
    const string absRegName = absRegNames[absRegNum];
    // This is the name of the hardware registers that are being used. Retrieve it from the DP.
    dyn_string hwRegNames = fwUkl1_getAbsRegHwRegNames(absRegName, registerType, exceptionInfo);
    if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_parseRegistersForSet(): Failed to retrieve the hardware registers associated with the register "  + absRegName + ". Will not be able to write this data, skipping register.", "2", exInfoSize))
      continue;
    // These are the types of the hardware registers that we have just retrieved.
    dyn_int hwRegTypes = fwUkl1_getHwRegTypes(absRegName, exceptionInfo);
    if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_parseRegistersForSet(): Failed to retrieve the hardware registers' types with the register "  + absRegName + ". Will not be able to write this data, skipping register.", "2", exInfoSize))
      continue;
    // This is the mask as a hex string that needs to be retrieved from the DP.
    dyn_string masks = fwUkl1_getMasks(absRegName, exceptionInfo);
    if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_parseRegistersForSet(): Failed to retrieve the masks associated with the register "  + absRegName + ". Will not be able to write this data, skipping register.", "2", exInfoSize))
      continue;
    // This is the name of the function to use to parse the information we are given. Retrieve it from the DP.
    dyn_string parserFunctions = fwUkl1_getParserFunctions(absRegName, registerType, exceptionInfo);
    if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_parseRegistersForSet(): Failed to retrieve the parser functions associated with the register "  + absRegName + ". Will not be able to write this data, skipping register.", "2", exInfoSize))
      continue;
    // This is the name of the function that we will use to access the hardware register.
    dyn_string accessFunctions = fwUkl1_getAccessFunctions(absRegName, registerType, exceptionInfo);
    if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_parseRegistersForSet(): Failed to retrieve the functions for accessing the hardware registers associated with the register "  + absRegName + ". Will not be able to write this data, skipping register.", "2", exInfoSize))
      continue;
    // Add to the inData map the flag to indicate that we want to indicate what register
    // type we are parsing this data for.
    inData[absRegName][FWUKL1_MAP_XTR_DATA] = registerType;
    // Some functions will return the dcData and dcMask by reference so it is useful to define these.
    dyn_char dcData, dcMask;

    // Get a reference to the information required to parse this data.
    // The mapping's dyn_string will contain the data to write and any other information
    // that is require, such as the register type (HW or DB).
    // For abstract registers with multiple hardware registers we copy the data from the inData
    // for the abstract register and then write it to each hardware register.
    dyn_string dsData = inData[absRegName];
    // Need to loop over all the hardware registers that we have retrieved.
    const int numHwRegs = dynlen(hwRegNames);
    for (int hwRegNum=1; hwRegNum<=numHwRegs; ++hwRegNum) {
      // Get the hardware register name we are interested in.
      const string hwRegName = hwRegNames[hwRegNum];
      // If the hardware register we have been given is the null register than we should not enter it into the map.
      if (FWUKL1_NULL_HWREG!=hwRegName) {
        // Check to see if the hardware register exists already and if not create it with blank defaults.
        // This is best done here and not in the case statements as it means that those that share a HW register
        // can always rely on it existing. They never have to check.
        if (!mappingHasKey(outData, hwRegName)) {
          outData[hwRegName] = makeDynMixed();
          outData[hwRegName][FWUKL1_MAP_REG_DATA]   = makeDynChar();
          outData[hwRegName][FWUKL1_MAP_XTR_DATA]   = makeDynChar();
          outData[hwRegName][FWUKL1_MAP_XTR_DATA+1] = accessFunctions[hwRegNum];
          outData[hwRegName][FWUKL1_MAP_XTR_DATA+2] = hwRegTypes[hwRegNum];
        }// if(!mappingHasKey())
        // Must populate the values with what ever we had before such that we have add in the data correctly.
        // If the key didn't exist then we are ensured that it will be empty.
        dcData = outData[hwRegName][FWUKL1_MAP_REG_DATA];
        dcMask = outData[hwRegName][FWUKL1_MAP_XTR_DATA];
      }// if(FWUKL1_NULL_HWREG!=hwRegName)
    
      // Each of the parser functions requires the parser information.
      // Get the information from these for the specific hardware register.
      dyn_dyn_string parserInfo = fwUkl1_getParserInfo(absRegName, registerType, hwRegNum, exceptionInfo);
      if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_parseRegistersForSet(): Failed to retrieve the parser information associated with the register "  + absRegName + ". Will not be able to understand this data, skipping register.", "2", exInfoSize))
        continue;
      // Use a switch statement to decide what to do.
      switch (parserFunctions[hwRegNum]) {
        case "_fwUkl1_convertMacToByte":
          // Always over write any previous information here, it just doesn't matter.
          // We also use the same for hardware and high level.
          // regName is not necessarily the name we switched on.
          outData[hwRegName][FWUKL1_MAP_REG_DATA] = _fwUkl1_convertMacToByte(dsData[FWUKL1_MAP_REG_DATA], exceptionInfo);
          if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_parseRegistersForSet(): Failed to parse the MAC address removed '" + absRegName + "' from the list of parsed registers.", "2", exInfoSize))
            outData[hwRegName][FWUKL1_MAP_XTR_DATA] = _fwUkl1_fpgaizeData(masks[hwRegNum]);// Don't need to bother with the mask building here only one mask ever.
          // Done.
          break;
        
        case "_fwUkl1_convertIpToByte":
          // Always over write any previous information here, it just doesn't matter.
          // We also use the same for hardware and high level.
          // regName is not necessarily the name we switched on.
          outData[hwRegName][FWUKL1_MAP_REG_DATA] = _fwUkl1_convertIpToByte(dsData[FWUKL1_MAP_REG_DATA], exceptionInfo);
          if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_parseRegistersForSet(): Failed to parse the IP address removed " + absRegName + " from the list of parsed registers.", "2", exInfoSize))
            outData[hwRegName][FWUKL1_MAP_XTR_DATA] = _fwUkl1_fpgaizeData(masks[hwRegNum], exceptionInfo);// Don't need to bother with the mask building here only one mask ever.
          // Done.
          break;

        // For the decimal numbers we first convert them to hex and then
        // will have them dealt with the hex to byte.
        // There is no actual function at the moment, but one day we might create one.
        case "_fwUkl1_convertAbsRegDecToByte":
          // Convert from decimal to hex.
          dsData[FWUKL1_MAP_REG_DATA] = fwCcpc_convertDecToHex(dsData[FWUKL1_MAP_REG_DATA]);
          // No break now goes to the hex dealing with code.
        case "_fwUkl1_convertAbsRegHexToByte":
          _fwUkl1_convertAbsRegHexToByte(absRegName, hwRegName, dsData, masks[hwRegNum], parserInfo, dcData, dcMask, exceptionInfo);
          if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_parseRegistersForSet(): Failed to parse the given data '" + dsData[FWUKL1_MAP_REG_DATA] + "' (may have been converted to hex) for the register '" + absRegName + "' no data will be written for this register.", "2", exInfoSize)) {
            outData[hwRegName][FWUKL1_MAP_REG_DATA] = dcData;
            outData[hwRegName][FWUKL1_MAP_XTR_DATA] = dcMask;
          }// if converted data
          // Done.
          break;
        
        case "_fwUkl1_convertAbsRegHexToByteLittleEndian":
          _fwUkl1_convertAbsRegHexToByteLittleEndian(absRegName, hwRegName, dsData, masks[hwRegNum], parserInfo, dcData, dcMask, exceptionInfo);
          if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_parseRegistersForSet(): Failed to parse the given data hex '" + dsData[FWUKL1_MAP_REG_DATA] + "' for the register '" + absRegName + "' no data will be written for this register.", "2", exInfoSize)) {
            outData[hwRegName][FWUKL1_MAP_REG_DATA] = dcData;
            outData[hwRegName][FWUKL1_MAP_XTR_DATA] = dcMask;
          }// if converted data
          // Done.
          break;
        
        case "_fwUkl1_convertAbsRegDescriptionToByte":
          _fwUkl1_convertAbsRegDescriptionToByte(absRegName, hwRegName, dsData, masks[hwRegNum], parserInfo, dcData, dcMask, exceptionInfo);
          if ( !fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_parseRegistersForSet(): Failed to parse the given description '" + dsData[FWUKL1_MAP_REG_DATA] + "' for the register '" + absRegName + "' no data will be written for this register.", "2", exInfoSize)) {
            outData[hwRegName][FWUKL1_MAP_REG_DATA] = dcData;
            outData[hwRegName][FWUKL1_MAP_XTR_DATA] = dcMask;
          }// if converted data
          // Done.
          break;
        
        default: 
          exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_parseRegistersForSet(): Did not recognise the parser function '" + parserFunctions[hwRegNum] + "'. Will not parse the register '" + absRegName + "'.", "2");
          break;
      }// switch(parserFunction)
      // If at the end of the switch we haven't added any data for the register then we should remove it.
      // This indicates that there was an error and we didn't add any data. Don't want to remove data that
      // another abstract register is writing so only remove if empty.
      if (mappingHasKey(outData, hwRegName) && (0==dynlen(outData[hwRegName]))) {
        mappingRemove(outData, hwRegName);
      }// removed any registers with no data
    }// for hwRegNum
  }// for absRegNum
}// _fwUkl1_parseRegistersForSet()

/*!
 * Parses the given input data map of data read from the hardware/DB and converts it to values that
 * are human readable and associated with an abstract register.
 *
 * @param  inData This is the raw data that has been read from the hardware or out of the DB. The map
 *           should contain as a key the hardware register and a dyn_anytype as the value. The dyn_anytype
 *           should contain the data to be written in FWUKL1_MAP_REG_DATA and this function will add
 *           The register type to be written to the FWUKL1_MAP_XTR_DATA element.
 * @param  registerType This is the register type that the data is to be parsed for. It should be either
 *           FWUKL1_HW_CONFIG_READ, FWUKL1_HW_STATUS_READ or FWUKL1_DB_READ.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return mapping This is a map that associates an abstract register name with the string containing the human readable
 *           value of the register and a colour representing its state. FWUKL1_MAP_REG_DATA contains the string
 *           with the register data and FWUKL1_MAP_XTR_DATA contains the colour and in certain cases
 *           FWUKL1_MAP_XTR_DATA+1 will contain a string usually with an error description. The outData should
 *           be populated with keys such that the function knows which registers to attempted to retrieve from
 *           the inData.
 *
 * The given inData map should have an empty entry for the FWUKL1_NULL_HWREG constant. The data associated with
 * the entry will never be used, but the key should at least exist.
 */
mapping _fwUkl1_parseRegistersForGet(const dyn_string &absRegNames, const mapping &inData, const string &registerType, dyn_string &exceptionInfo) {
  // Keep track of the exception information.
  int exInfoSize = dynlen(exceptionInfo);
  // It is in this mapping that we will save the parsed data which is returned to the caller.
  mapping outData;
  const int numAbsRegs = dynlen(absRegNames);
  // Now loop over the keys and parse all the data.
  string absRegName;
  for (int j=1; j<=numAbsRegs; ++j) {
    absRegName = absRegNames[j];
    // This is the name of the hardware register that is being read from. Retrieve it from the DP.
    // Here we always take the alternative register as those are the ones used for reading from the
    // status registers.
    dyn_string hwRegNames = fwUkl1_getAbsRegHwRegNames(absRegName, registerType, exceptionInfo);
    if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_parseRegistersForGet(): Failed to retrieve the hardware registers associated with the register "  + absRegName + ". Will not be able to read this data, skipping register.", "2", exInfoSize))
      continue;
    // This is the information that is required to know how to deal with the registers.
    dyn_string parserFunctions = fwUkl1_getParserFunctions(absRegName, registerType, exceptionInfo);
    if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_parseRegistersForGet(): Failed to retrieve the parser functions associated with the register "  + absRegName + ". Will not be able to understand this data, skipping register.", "2", exInfoSize))
      continue;
    // As we can have multiple hardware registers for the status we will build up
    // a dyn_dyn_string which contains a dyn_string for each hardware register read.
    // The dyn_string for the hardware register is the parsed data.
    // At the end we use a combiner function to convert this dyn_dyn_string with
    // data for each hardware register to one parsed value for the abstract register.
    dyn_dyn_string allOutput;
    
    // Need to loop over all the hardware registers that we have retrieve and parse the data for each one.
    const int numHwRegs = dynlen(hwRegNames);
    for (int regNum=1; regNum<=numHwRegs; ++regNum) {
      string hwRegName = hwRegNames[regNum];
      dyn_string output = makeDynString("Fail", "FwStateAttention1", "");
      // Now we have the hwRegName lets check to see if we have data for it.
      if (!mappingHasKey(inData, hwRegName))
        exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_parseRegistersForGet(): The hardware register, " + hwRegName + ", associated with the register, " + absRegName + ", was not found.", "2");
      else {
        if ((FWUKL1_NULL_HWREG!=hwRegName) && (1>dynlen(inData[hwRegName])))
          exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_parseRegistersForGet(): There is no raw data associated with the hardware register " + hwRegName + ". Cannot parse the data for the register " + absRegName + ", thus cannot retrieve any data for this register.", "2");
        else {
          // We should never operate on the inData directly. It will be parsed multi times, need a local copy each time.
          dyn_mixed rawData = inData[hwRegName];
          rawData[FWUKL1_MAP_XTR_DATA] = registerType;
          // Each of the parser functions requires the parser information and the validation information.
          // Get the information from these for the specific hardware register.
          dyn_dyn_string parserInfo = fwUkl1_getParserInfo(absRegName, registerType, regNum, exceptionInfo);
          if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_parseRegistersForGet(): Failed to retrieve the parser information associated with the register "  + absRegName + ". Will not be able to understand this data, skipping register.", "2", exInfoSize))
            continue;
          dyn_dyn_string validationInfo = fwUkl1_getValidationInfo(absRegName, registerType, regNum, exceptionInfo);
          fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_parseRegistersForGet(): Failed to retrieve the validation information associated with the register "  + absRegName + ". Will not be able to check the parsed data is valid, continuing.", "2", exInfoSize);
          // Use a switch statement to decide what to do.
          // Deal with errors at the end.
          switch (parserFunctions[regNum]) {
            case "_fwUkl1_convertByteToMac":
              output = _fwUkl1_convertByteToMac(rawData[FWUKL1_MAP_REG_DATA], exceptionInfo);
              break;
            
            case "_fwUkl1_convertByteToIp":
              output = _fwUkl1_convertByteToIp(rawData[FWUKL1_MAP_REG_DATA], exceptionInfo);
              break;
            
            case "_fwUkl1_convertByteToAbsRegHex":
              // Do pad this hex data such that it is the maximum length in bytes it can be.
              output = _fwUkl1_convertByteToAbsRegHex(absRegName, hwRegName, parserInfo, validationInfo, rawData, TRUE, exceptionInfo);
              break;
            
            case "_fwUkl1_convertByteToAbsRegHexLittleEndian":
              output = _fwUkl1_convertByteToAbsRegHexLittleEndian(absRegName, hwRegName, parserInfo, validationInfo, rawData, TRUE, exceptionInfo);
              break;
            
            case "_fwUkl1_convertByteToAbsRegDec":
              // Actually there is no real function to do this. We first convert it to hex and then from hex to dec.
              // Don't bother to pad the data it will be converted to decimal anyway.
              output = _fwUkl1_convertByteToAbsRegHex(absRegName, hwRegName, parserInfo, validationInfo, rawData, FALSE, exceptionInfo);
              if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_parseRegistersForGet(): Failed to parse the data from '"+hwRegName+"' for '"+absRegName+"'.", "2", exInfoSize))
                output[FWUKL1_MAP_REG_DATA] = fwCcpc_convertHexToDec(output[FWUKL1_MAP_REG_DATA]);
              break;
            
            case "_fwUkl1_convertByteToAbsRegDescription":
              output = _fwUkl1_convertByteToAbsRegDescription(absRegName, hwRegName, parserInfo, validationInfo, rawData, exceptionInfo);
              break;
            
            default:
              exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_parseRegistersForGet(): Unrecognised parser function '"+parserFunctions[regNum]+"' for the register '"+absRegName+"'.", "2");
          }// switch(parserFunction)
          // Check for any error that occured.
          fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_parseRegistersForGet(): Failed to parse '"+absRegName+"' with the function '"+parserFunctions[regNum]+"'.", "2", exInfoSize);
        }// else there is data.
      }// else there is no key!
      // Save this output to our hardware register result collection.
      dynAppend(allOutput, output);
    }// for regNum

    // Now we must run a combiner on the given result to ensure we get a single value.
    outData[absRegName] = _fwUkl1_absRegDataCombiner(absRegName, allOutput, exceptionInfo);
    fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_parseRegistersForGet(): Failed to combine the data for '"+absRegName+"'.", "2", exInfoSize);
  }// for j absRegNames loop
  // Done.
  return outData;
}// _fwUkl1_parseConfigRegistersForGet()
  
/*!
 * This takes an outData mapping from the one of the register parsing functions and converts them to two dyn_dyn_chars
 * for the register data and the mask.
 *
 * @param  outData The mapping from the parse register function.
 * @param  regNames Names of the hardware registers that are to be written to.
 * @param  regData The register data.
 * @param  masks The mask.
 * @param  writeFunctions This is a list that contains the functions that the register data and mask in the
 *           corresponding element should be written with. The writeFunctions dyn_string is sorted such
 *           that matching write functions are guarantied to be consequtive.
 * @param  hwRegType The hardware register type for each hardware register in the list.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 *
 * The returned regData and masks are sorted with respect to the writeFunction that is to be used.
 * The writeFunctions arrays is sorted such that all matching write functions are sequential and it
 * matches the corresponding regData and mask. This is to make it easier to separate the all the data
 * and masks that are written in the same way.
 */
void _fwUkl1_parseWriteMapping(const mapping &writeData, dyn_string &regNames, dyn_dyn_char &regData, dyn_dyn_char &masks, dyn_string &writeFunctions, dyn_int &hwRegTypes, dyn_string &exceptionInfo) {
  // Get all the keys.
  const dyn_string keys = mappingKeys(writeData);
  const int numKeys = dynlen(keys);
  string key = "";
  for (int keyNum=1; keyNum<=numKeys; ++keyNum) {
    key = keys[keyNum];
    // Check we have the data.
    const int numDataEle = dynlen(writeData[key]);
    const int requiredEle = 3;
    if (requiredEle <= numDataEle) {
      regNames[keyNum]       = key;
      regData[keyNum]        = writeData[key][FWUKL1_MAP_REG_DATA];
      masks[keyNum]          = writeData[key][FWUKL1_MAP_XTR_DATA];
      writeFunctions[keyNum] = writeData[key][FWUKL1_MAP_XTR_DATA+1];
      hwRegTypes[keyNum]     = writeData[key][FWUKL1_MAP_XTR_DATA+2];
    }// if(1<numDataEle)
    else
      fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_parseOutMapping(): Not enough data was given for the hardware register " + key + ", this register will not be written. Require " + requiredEle + " elements given " + numDataEle + ".", "2");
  }// for keyNum
  
  // Now we need to sort the 4 arrays relative to the write function to use.
  // Hence this must be first in the array.
  dyn_mixed lists = makeDynMixed(writeFunctions, regNames, regData, masks, hwRegTypes);
  fwUkl1_sort(lists, exceptionInfo);
  // Now copy the sorted lists back again.
  writeFunctions = lists[1];
  regNames       = lists[2];
  regData        = lists[3];
  masks          = lists[4];
  hwRegTypes     = lists[5];
  
  // Done.
  return;
}// _fwUkl1_parseWriteMapping()

/*!
 * This retrieves the hardware register names that are required to read the given registers in the outData mapping.
 *
 * @param  [in] absRegNames This is the list of registers that are to be read. It is for each of these that the associated
 *           hardware registers are retrieved.
 * @param  [in] registerType One of the constants FWUKL1_DB_READ, FWUKL1_DB_WRITE, FWUKL1_HW_CONFIG_READ,
 *           FWUKL1_HW_STATUS_READ or FWUKL1_HW_WRITE. The value indicates which type of register
 *           the value should be retrieved for e.g. the database write register, hardware status register.
 * @param  [out] hwRegNames This is the returned list of hardware register names that are to be read. It will contain
 *           no duplicate entries.
 * @param  [out] readFunctions This is the name of the function that should be used to read the hardware register.
 *           Its index in this array corresponds to the hardware register in the equivalent element in the
 *           hwRegNames array.
 * @param  [out] hwRegTypes List of the types of the hardware registers that are to be read e.g. FWHW_LBUS.
 * @param  [in,out] exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 *
 * The hardware registers are sorted with respect to the read functions this is required by the _fwUkl1_readSwitch
 * function.
 */
void _fwUkl1_parseReadMapping(const dyn_string &absRegNames, const string &registerType, dyn_string &hwRegNames, dyn_string &readFunctions, dyn_int  &hwRegTypes, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  // These must be clear to start otherwise it could muck up the counting.
  dynClear(hwRegNames);
  dynClear(readFunctions);
  dynClear(hwRegTypes);
  const int numRegs = dynlen(absRegNames);
  for (int regNum=1; regNum<=numRegs; ++regNum) {
    const dyn_string retrievedHwRegNames = fwUkl1_getAbsRegHwRegNames(absRegNames[regNum], registerType, exceptionInfo);
    if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_parseReadMapping(): Failed to get the hardware register name for the register " + absRegNames[regNum] + ". This register will not be read.", "2", exInfoSize)) {
      // Get the name of the read function.
      const dyn_string readFuncs = fwUkl1_getAccessFunctions(absRegNames[regNum], registerType, exceptionInfo);
      if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_parseReadMapping(): Failed to get the hardware register read functions for the register " + absRegNames[regNum] + ". This register will not be read.", "2", exInfoSize)) {
        continue;
      }// if couldn't get read functions.
      // Get the list of hardware types.
      const dyn_string types = fwUkl1_getHwRegTypes(absRegNames[regNum], exceptionInfo);
      if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_parseReadMapping(): Failed to get the hardware register types for the register " + absRegNames[regNum] + ". This register will not be read.", "2", exInfoSize)) {
        continue;
      }// if couldn't get hw register types.
      const int totalRetrievedRegs = dynlen(retrievedHwRegNames);
      for (int hwRegNum=1; hwRegNum<=totalRetrievedRegs; ++hwRegNum) {
        // If we have the null register then we do not want to read from it.
        if (FWUKL1_NULL_HWREG == retrievedHwRegNames[hwRegNum])
          continue;
        // Now check if it is already in the array.
        if (1 > dynContains(hwRegNames, retrievedHwRegNames[hwRegNum])) {
          if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_parseReadMapping(): Failed to retrieve the function to use to read the hardware register for " + absRegNames[regNum] + ", this register cannot be read.", "2", exInfoSize)) {
            dynAppend(hwRegNames, retrievedHwRegNames[hwRegNum]);
            dynAppend(readFunctions, readFuncs[hwRegNum]);
            dynAppend(hwRegTypes, types[hwRegNum]);
          }// if got read function.
        }// if not already added the hw reg
      }// for hwRegNum
    }// if couldn't get the HW registers
  }// for regNum

  // Now we need to sort the arrays relative to the read function to use.
  // Hence this must be first in the array.
  dyn_mixed lists = makeDynMixed(readFunctions, hwRegNames, hwRegTypes);
  fwUkl1_sort(lists, exceptionInfo);
  // Now copy the sorted lists back again.
  readFunctions = lists[1];
  hwRegNames    = lists[2];
  hwRegTypes    = lists[3];
  // Done.
  return;
}// _fwUkl1_parseReadMapping()

/*!
 * Takes the given mapping element containing the data in the form for writing to the hardware register
 * and ORs the data and mask into the appropriate place in the dyn_char entries.
 *
 * @param  outDataElement The element from the mapping that corresponds to the hardware registers that the data
 *           should be added for. The data and mask elements are checked must contain at least 1 element.
 * @param  startBit First bit to write the data to in the bit field.
 * @param  sData The data that is to be written to the register.
 * @param  mask String containing the mask to be written to the register. It will appropriately aligned with the
 *           start of the data and padded to the appropriate length.
 * @param  fpga This field indicates if the data is to be written to an FPGA or not. They have a slightly different
 *           register structure that needs to be taken into account.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 *
 * Due to the way the function works we are limited to 32-bit data if the fpga is FALSE and only 16-bit if fpga is TRUE.
 */
void _fwUkl1_addWriteDataAndMask(dyn_mixed &outDataElement, int startBit, int data, int mask, bool fpga, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  // Check that we have a sensible startBit. As we don't know how long the mask or data is then can't force the
  // start to be sensible.
  const int maxBit = (fpga ? 15:31);
  if (startBit >= maxBit) {
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_addWriteDataAndMask(): The startBit to write the data and mask to, " + startBit + ", is outside the allowed range, " + maxBit + ". Can't continue. Data and mask left unchanged.", "2");
    return;
  }// if(startBit>=maxBit);
  
  // We only want to write to the first 2 elements so we need at least that many. Not interested in the rest of the extra data.
  if (2 > dynlen(outDataElement)) {
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_addWriteDataAndMask(): Not enough elements to write the data and mask to. Require at least two element given dyn_mixed contains only " + dynlen(outDataElement) + ". Inserting two elements at the beginning. Assuming 32-bit registers.", "2");
    outDataElement[FWUKL1_MAP_REG_DATA] = makeDynChar(0x00, 0x00, 0x00, 0x00);
    outDataElement[FWUKL1_MAP_XTR_DATA] = makeDynChar(0x00, 0x00, 0x00, 0x00);
  }// if(2>dynlen(outDataElement))
  // These will be used a lot.
  int datalen = dynlen(outDataElement[FWUKL1_MAP_REG_DATA]);
  int masklen = dynlen(outDataElement[FWUKL1_MAP_XTR_DATA]);
  // Need at least a byte to write to.
  if (1 > datalen) {
    // Either make it the same size as the mask or default to 32-bit.
    if (1 > masklen) {
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_addWriteDataAndMask(): Write element contains no data. Assuming a 32-bit register and adding 4 bytes.", "2");
      outDataElement[FWUKL1_MAP_REG_DATA] = makeDynChar(0x00,0x00,0x00,0x00);
      datalen = dynlen(outDataElement[FWUKL1_MAP_REG_DATA]);
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_addWriteDataAndMask(): Mask element contains no data. Assuming a 32-bit register and adding 4 bytes.", "2");
      outDataElement[FWUKL1_MAP_XTR_DATA] = makeDynChar(0x00,0x00,0x00,0x00);
      masklen = dynlen(outDataElement[FWUKL1_MAP_XTR_DATA]);
    }// if not mask data.
    else {
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_addWriteDataAndMask(): Write element contains no data. Setting to the same size as the mask, " + masklen + ".", "2");
      int loopStart = datalen;
      int loopEnd   = masklen;
      for (int i=loopStart; i<=loopEnd; ++i)
        dynAppend(outDataElement[FWUKL1_MAP_REG_DATA], 0x00);
      datalen = dynlen(outDataElement[FWUKL1_MAP_REG_DATA]);
    }// else mask data.
  }// if no elements in write data.
  // Require to have the same number of mask elements as in the write register.
  // This will also catch the condition where there is no mask data but some write data.
  if (datalen != masklen) {
    // If the data is shorter than the mask then add elements to the data till it is the same length.
    if (datalen < masklen) {
      int loopStart = datalen;
      int loopEnd   = masklen;
      for (int i=loopStart; i<=loopEnd; ++i)
        dynAppend(outDataElement[FWUKL1_MAP_REG_DATA], 0x00);
      datalen = dynlen(outDataElement[FWUKL1_MAP_REG_DATA]);
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_addWriteDataAndMask(): Appended " + (loopEnd-loopStart) + " elements to the register data such that it is the same length as the mask data (" + masklen + ").", "2");
    }// if(datalen<masklen)
    // If the mask is shorter than the data then add elements to the mask until it is the same length.
    else if (masklen < datalen) {
      int loopStart = datalen;
      int loopEnd   = masklen;
      for (int i=loopStart; i<=loopEnd; ++i)
        dynAppend(outDataElement[FWUKL1_MAP_XTR_DATA], 0x00);
      masklen = dynlen(outDataElement[FWUKL1_MAP_XTR_DATA]);
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_addWriteDataAndMask(): Appended " + (loopEnd-loopStart) + " elements to the mask data such that it is the same length as the register data (" + datalen + ").", "2");
    }// if(masklen<datalen)
  }// if(datalen!=masklen)
  
  // Now we have everything must be setup appropriately lets OR in the data and mask!
  // Arrange the data.
  data = data << startBit;
  // Arrange the mask.
  mask = mask << startBit;
  // Make sure neither over step their bounds.
  if (fpga)
    mask = mask & 0x0000ffff;
  // else is 32 the size of the reigster.
  data = data & mask;
  // Now convert it to a dyn_char, go via a hex string to ensure it is the correct size.
  string sData = fwCcpc_convertDecToHex(data);
  string sMask = fwCcpc_convertDecToHex(mask);
  // By padding the string out to a 32-bit register we will ensure that the FPGA registers are the correct lenght.
  fwUkl1_padString(sData, 4);
  fwUkl1_padString(sMask, 4);
  // Now convert to dyn_char.
  outDataElement[FWUKL1_MAP_REG_DATA] = fwCcpc_convertHexToByte(sData);
  outDataElement[FWUKL1_MAP_XTR_DATA] = fwCcpc_convertHexToByte(sMask);
  
  // Done.
  return;
}// _fwUkl1_addWriteDataAndMask()

/*!
 * Swaps two elements of a dyn structure. No operation is performed if they are the same or if one of the
 * indexes is out of bounds.
 *
 * @param  dyn The dyn structure containing the elements to be swapped.
 * @param  ele1 Index of one of the elements to swapped.
 * @param  ele2 Index of the second element to be swapped.
 * @return void.
 *
 * Not that the input elements are unsigned. If a negative number is given it will likely be
 * converted to an int be very large and in most cases out of bounds.
 */
void _fwUkl1_swapElements(dyn_anytype dyn, unsigned ele1, unsigned ele2) {
  const int len = dynlen(dyn);
  if ( (ele1 != ele2) && (ele1 <= len) && (ele2 <= len) ) {
    // Read ele1 and save it.
    anytype tmpValue  = dyn[ele1];
    // Write ele2 to ele1.
    dyn[ele1] = dyn[ele2];
    // Now write the saved ele1 to ele2.
    dyn[ele2] = tmpValue;
  }// if indexes not the same and inbounds.
  // Done.
  return;
}// _fwUkl1_swapElements()

// @}

// ==========
//   RESETS
// ==========

/** @defgroup SectionResets
 *    These are the resets that have been provided for the UKL1 board.
 *    Most of these resets are for expert use only and should be used with caution.
 *    The exception is fwUkl1_reload, which is used by the FSM for the reset
 *    command.
 */
// @{

/*!
 * Performs a reload of the FPGA firmware on the L1 board. It will reset all values back to their defaults.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return int. 0=Ingress OK.
 */
int fwUkl1_reload(const string &ukl1Name, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  _fwUkl1_toggleGpioLine(ukl1Name, FWCCPC_GPIO_LINE_6, exceptionInfo);
  if ( fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_reload(): Failed to reload the UKL1 firmware. There was a problem with the hardware access.", "2", exInfoSize) )
  {
    return -1;
  }

  return 0;
  
  // Done.
}// fwUkl1_reload()

/*!
 * Performs a reset of the FPGAs on the L1 board. Effects still under discussion.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 */
void fwUkl1_reset(const string &ukl1Name, dyn_string& exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  _fwUkl1_toggleGpioLine(ukl1Name, FWCCPC_GPIO_LINE_5, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_reset(): Failed to reset the UKL1 logic. There was a problem with the hardware access.", "2", exInfoSize);
  // Done.
  return;
}// fwUkl1_reset()

/*!
 * Performs a reset of the GBE card. If this fails it may prevent access to the GBE card.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 */
void fwUkl1_resetGBE(const string &ukl1Name, dyn_string& exceptionInfo) {
  // Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_resetGBE(): UKL1 name was empty, cannot proceed.", "2");
    return;
  }// if(""==ukl1Name)

  int exInfoSize = dynlen(exceptionInfo);
  int callStatus;
  // I am not certain if this GPIO line is enable for output as the others are by default, thus attempt to enable it.
  callStatus = fwCcpc_GPIOEnable(ukl1Name, FWCCPC_GPIO_LINE_4, FWCCPC_GPIO_OUTPUT);
  if (0 != callStatus) {
    // Construct the error message.
    string errMsg = "fwUkl1_resetGBE(): Failed to set the GPIO line 4 enabled for output during the reset, will continue to attempt a reset but it may fail. Encountered";
    if (callStatus > 0)
      errMsg += " a problem writing to the CCPC server, return code " + callStatus + ".";
    else if (callStatus < 0)
      errMsg += " a problem with PVSS.";
    else
      errMsg += " an unknown error.";
    // Now raise the exception.
    fwUkl1Exception_raise(exceptionInfo, "ERROR", errMsg, "1");
  }// Failed to enable the GPIO line.
  
  _fwUkl1_toggleGpioLine(ukl1Name, FWCCPC_GPIO_LINE_4, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_resetGBE(): Failed to reset the GBE card, hardware acess problems.", "2", exInfoSize);
  
  // Give the GBE card time to recover.
  delay(0,10);
  // Done.
  return;
}// fwUkl1_resetGBE()

/*!
 * Perform a partial reset of the UKL1 boards.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 *
 * Resets everything except the configuration registers e.g. logic, readout buffers, status registers etc..
 */
void fwUkl1_resetUkl1Partial(string ukl1Name, dyn_string& exceptionInfo) {
  // Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_resetUkl1Partial(): UKL1 name was empty, cannot proceed.", "2");
    return;
  }// if(""==ukl1Name)
  
  // Used to track changes in the exception info array.
  int exInfoSize = dynlen(exceptionInfo);
  // The reset must first be asserted and then deasserted for the pointers to reset.
  // Assert the reset.
  const string fifoResetReg = ukl1Name + ".Resets.PartialL1";
  // data and mask are the same in this case.
  dyn_dyn_char dataAndMask;
  dataAndMask[1] = fwCcpc_convertHexToByte("00000020");
  _fwUkl1_write(fifoResetReg, dataAndMask, dataAndMask, TRUE, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_resetPartialUkl1(): Failed to assert reset, attempting to deassert the reset anyway.", "2", exInfoSize);

  // Deassert the reset of the FIFO pointer.
  dyn_dyn_char clearData;
  clearData[1] = fwCcpc_convertHexToByte("00000000");
  // dataAndMask now just contains the mask as we want to write something different for the data.
  // The mask is the same as before however.
  _fwUkl1_write(fifoResetReg, clearData, dataAndMask, FALSE, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_resetPartialUkl1(): Failed to de-assert status reset.", "2", exInfoSize);
  
  // Done.
  return;
}// fwUkl1_resetPartialUkl1()

/*!
 * Re-enable any disabled HPDs.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 *
 * Re-enables any channels disabled by the automatic HPD disabling.
 */
void fwUkl1_resetDisabledHPDs(const string ukl1Name, dyn_string &exceptionInfo) {

  // Keep track of the amount of exception information generated.
  int exInfoSize = dynlen(exceptionInfo);
  
    // Find out what sort of action to take (RESET or SQUASH)
    int disableCode = 2;
    if (dpExists("UKL1Constants.DisableAction"))
    {
      string disableAction;
      dpGet("UKL1Constants.DisableAction",disableAction);
      if (disableAction=="SQUASH")
      {
        disableCode = 8;
      }
    }
    
  // The ingress FPGA number to be passed, this is set to 4 which causes a broadcast to be sent.
  int ing = 4;
  
  // First prepare the ingress control register setting the broadcast flag
  _fwUkl1_setConfigForIngressFpgaChannel(ukl1Name, ing, FALSE, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_resetDisabledHPDs(): Failed to prepare the configuration FIFO for Ingress FPGA " + ing + " for writting.", "2", exInfoSize);
  
  // The name of the hardware register  
  dyn_string mailboxReg = makeDynString(ukl1Name + ".FpgaConfig" + ".IngressMailbox2");
  // The data to be written to the Ingress Configuration Mailbox
  dyn_dyn_char mailboxData;
  string action;
  sprintf(action,"f004000%1x",disableCode);
  mailboxData[1] = fwCcpc_convertHexToByte(action);
 // We don't use the masks as the CCPC server just writes it into the FIFO and this causes the loss of data. It must be empty.
  dyn_dyn_char mailboxMasks;
  
    // Write the data to the hardware register
    _fwUkl1_write(mailboxReg, mailboxData, mailboxMasks, FALSE, exceptionInfo);
    fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_resetDisabledHPDs(): Failed to write configuration data for Ingress FPGA " + ing, "2", exInfoSize);

   // Trigger the transfer of the data to the ingress    
   _fwUkl1_sendIngressChannelConfigFifo(ukl1Name, FALSE, exceptionInfo);
   fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_resetDisabledHPDs(): Triggering the transmission of the configuration data for Ingress FPGA " + ing, "2", exInfoSize);
  
  // Done.
  return;
}// fwUkl1_resetDisabledHPDs()

/*!
 * Asserts and then deasserts a reset of the status buffer pointers. This must be done in order
 * to read any information from the Ingress status FIFOs.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 *
 * This will reset all the status buffer pointers simulataneously.
 */
void fwUkl1_resetIngressFpgaStatusBufPointers(string ukl1Name, dyn_string& exceptionInfo) {
  // Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_resetIngressFpgaStatusBufPointers(): UKL1 name was empty, cannot proceed.", "2");
    return;
  }// if(""==ukl1Name)
  
  // Used to track changes in the exception info array.
  int exInfoSize = dynlen(exceptionInfo);
  // The reset must first be asserted and then deasserted for the pointers to reset.
  // Assert the reset of FIFO pointer.
  const string fifoResetReg = ukl1Name + ".Resets.IngressFpgaStatusBufferPointers";
  // data and mask are the same in this case.
  dyn_dyn_char dataAndMask;
  dataAndMask[1] = fwCcpc_convertHexToByte("0000000f");
  _fwUkl1_write(fifoResetReg, dataAndMask, dataAndMask, TRUE, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_resetIngressFpgaStatusBufPointers(): Failed to assert status buffer pointer reset, status information will likely be invalid.", "2", exInfoSize);

  // Deassert the reset of the FIFO pointer.
  dyn_dyn_char clearData;
  clearData[1] = fwCcpc_convertHexToByte("00000000");
  // dataAndMask now just contains the mask as we want to write something different for the data.
  // The mask is the same as before however.
  _fwUkl1_write(fifoResetReg, clearData, dataAndMask, FALSE, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_resetIngressFpgaStatusBufPointers(): Failed to de-assert status buffer pointer reset, status information will likely be invalid.", "2", exInfoSize);
  
  // Done.
  return;
}// fwUkl1_resetIngressFpgaStatusBufPointers()

/*!
 * Asserts and then deasserts a reset of the SPI3 RX read pointer.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * @return void.
 *
 * This is for a special test feature and will almost never need to be used.
 */
void fwUkl1_resetSPI3RXReadPointer(string ukl1Name, dyn_string& exceptionInfo) {
  // Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_resetSPI3RXReadPointer(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }// if(""==ukl1Name)
  
  // Used to track changes in the exception info array.
  int exInfoSize = dynlen(exceptionInfo);
  // The reset must first be asserted and then deasserted for the pointers to reset.
  // Assert the reset.
  const string fifoResetReg = ukl1Name + ".Resets.SPI3RxReadPointer";
  // data and mask are the same in this case.
  dyn_dyn_char dataAndMask;
  dataAndMask[1] = fwCcpc_convertHexToByte("00000010");
  _fwUkl1_write(fifoResetReg, dataAndMask, dataAndMask, TRUE, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_resetSPI3RXReadPointer(): Failed to assert reset.", "2", exInfoSize);

  // Deassert the reset of the FIFO pointer.
  dyn_dyn_char clearData;
  clearData[1] = fwCcpc_convertHexToByte("00000000");
  // dataAndMask now just contains the mask as we want to write something different for the data.
  // The mask is the same as before however.
  _fwUkl1_write(fifoResetReg, clearData, dataAndMask, FALSE, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_resetSPI3RXReadPointer(): Failed to de-assert status reset.", "2", exInfoSize);
  
  // Done.
  return;
}// fwUkl1_resetSPI3RXReadPointer()

/*!
 * Perform a reset of the DCMs.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * @return void.
 *
 * Resets the Egress clock manager and generally will never need to be used except in special test cases.
 */
void fwUkl1_resetDCMs(string ukl1Name, dyn_string& exceptionInfo) {
  // Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_resetDCMs(): UKL1 name was empty, cannot proceed.", "2");
    return;
  }// if(""==ukl1Name)
  
  // Used to track changes in the exception info array.
  int exInfoSize = dynlen(exceptionInfo);
  // The reset must first be asserted and then deasserted for the pointers to reset.
  // Assert the reset.
  const string fifoResetReg = ukl1Name + ".Resets.DCMs";
  // data and mask are the same in this case.
  dyn_dyn_char dataAndMask;
  dataAndMask[1] = fwCcpc_convertHexToByte("00000030");
  _fwUkl1_write(fifoResetReg, dataAndMask, dataAndMask, FALSE, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_resetDCMs(): Failed to assert reset.", "2", exInfoSize);

  // Deassert the reset of the FIFO pointer.
  dyn_dyn_char clearData;
  clearData[1] = fwCcpc_convertHexToByte("00000000");
  // dataAndMask now just contains the mask as we want to write something different for the data.
  // The mask is the same as before however.
  _fwUkl1_write(fifoResetReg, clearData, dataAndMask, FALSE, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_resetDCMs(): Failed to de-assert status reset.", "2", exInfoSize);
  
  // Done.
  return;
}// fwUkl1_resetDCMs()

/*!
 * Performs a reset of the GBE PHY chips.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * @return void.
 *
 * Required during the GBE initialisation procedure.
 */
void fwUkl1_resetPhyChips(const string &ukl1Name, dyn_string &exceptionInfo) {
  // Keep track of errors.
  int exInfoSize = dynlen(exceptionInfo);
  // Loop over the PHY chips.
  for (int phy = 0; phy < 4; ++phy) {
    // Write the reset command to the PHY chip, don't bother to verify the writes.
    _fwUkl1_writeGbeMdioRegister(ukl1Name, phy, 0x00, 0x8140, FALSE, exceptionInfo);
    fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_resetPhyChips(): Failed to perform a reset of PHY chip number " + phy + ". GBE card may not function correctly if this was part of the initialisation.", "2", exInfoSize);
  }// for phy
}// fwUkl1_resetPhyChip()

// ==============
//    INTERNAL
// ==============

/*!
 * This function toggles the given GPIO function from low to high.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  gpioLine The line to toggle.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 */
void _fwUkl1_toggleGpioLine(const string &ukl1Name, int gpioLine, dyn_string &exceptionInfo) {
  // Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_reload(): UKL1 name was empty, cannot proceed.", "2");
    return;
  }// if(""==ukl1Name)
  
  int exInfoSize = dynlen(exceptionInfo);
  int callStatus;
  // The reload is performed by setting the GPIO line 6 low for a few 10s of milliseconds and then setting it back to high.
  // First set it low.
  callStatus = fwCcpc_GPIOSet(ukl1Name, gpioLine, FWCCPC_GPIO_LOW);
  // Check for errors.
  if (0 == callStatus) {
    // We have driven it low, wait a few milli-seconds for the reload to take affect.
    delay(0,20);
    // Only worth trying to do this if it succeeded.
    callStatus = fwCcpc_GPIOSet(ukl1Name, gpioLine, FWCCPC_GPIO_HIGH);
    // Check for errors.
    if (0 != callStatus) {
      // Construct the error message.
      string errMsg = "_fwUkl1_toggleGpioLine(): Failed to set the GPIO line " + gpioLine + " high during a reload, unable to desassert the reload. Encountered";
      if (callStatus > 0)
        errMsg += " a problem writing to the CCPC server, return code " + callStatus + ".";
      else if (callStatus < 0)
        errMsg += " a problem with PVSS.";
      else
        errMsg += " an unknown error.";
      // Now raise the exception.
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", errMsg, "2");
    }// if set the GPIO line high.
  }// if set GPIO line low
  else {
    // Construct the error message.
    string errMsg = "_fwUkl1_toggleGpioLine(): Failed to set the GPIO line " + gpioLine + " low during a reload, unable to reload. Encountered";
    if (callStatus > 0)
      errMsg += " a problem writing to the CCPC server, return code " + callStatus + ".";
    else if (callStatus < 0)
      errMsg += " a problem with PVSS.";
    else
      errMsg += " an unknown error.";
    // Now raise the exception.
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", errMsg, "2");
  }// else could not set the GPIO line low.
  // Done.
  return;
}// _fwUkl1_toggleGpioLine()

// @}

/*!
 * Masks noisy pixels.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  runType  The run type. e.g. PHYSICS, ALICE etc.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 *
 * Masks noisy pixels given in a list in the UKL1Constants.MaskedPixelsLHCb dpe.
 */
void fwUkl1_maskPixels(const string ukl1Name, const string runType, dyn_string &exceptionInfo) {
   
//  DebugTN(ukl1Name + ": Pixel masking initiated!");
  // Keep track of the amount of exception information generated.
  int exInfoSize = dynlen(exceptionInfo);
 
  // Get the system name
  string systemName = fwUkl1_removeDeviceName(ukl1Name);
  // Get the ukl1 name
  string ukl1Board = fwUkl1_removeSystemName(ukl1Name);

  // Get the rich number from the board name.
  string richNumber = substr(ukl1Board, 1, 1);
  string boardNumber = substr(ukl1Board, 6, 2);
//  DebugTN(ukl1Board + ": systemName = " + systemName + ", richNumber = " + richNumber + ", boardNumber = " + boardNumber);

  // Get the list of pixels to be disabled from the datapoint.
  // The format is: (#1 rich number, #2 board number, #3 ingress number, #4 channel number, #5 pixel address, #6 pixel mask).
  string pixelListUnformatted;
  // Check the run mode and get the masked pixel list from the appropriate data point.
  if(runType != "ALICE") {
    dpGet(systemName + "UKL1Constants.MaskedPixelsLHCb", pixelListUnformatted);
  }
  else {
    dpGet(systemName + "UKL1Constants.MaskedPixelsALICE", pixelListUnformatted);
  }
  // Split the unformatted list into the individual entries.
  dyn_string pixelList = strsplit(pixelListUnformatted, " "); 
  // Get the length of the pixelList.
  int length = dynlen(pixelList);
  // Create a dyn_dyn_string to hold each list element entry.
  dyn_dyn_string boardPixelList;
  // Loop over the pixel list and remove the brackets and split up each element, trimming any unintended spaces.
  for(int i = 1; i < length+1; i++) {
    // Strip off any spaces on the left and right.
    pixelList[i] = strltrim(pixelList[i], " ");
    pixelList[i] = strrtrim(pixelList[i], " "); 
    // Strip off the brackets from the left and right.
    pixelList[i] = strltrim(pixelList[i], "(");     
    pixelList[i] = strrtrim(pixelList[i], ")");
    // Split each list element up into its individual entries.
    dyn_string splitPixelList = strsplit(pixelList[i], ",");
    // Loop over the splitPixelList removing any leading or trailing spaces.
    int splitLength = dynlen(splitPixelList);
    for(int j = 1; j < splitLength+1; j++) {
      splitPixelList[j] = strltrim(splitPixelList[j], " ");
      splitPixelList[j] = strrtrim(splitPixelList[j], " ");
    } // Loop over the splitPixelList. 
    // Check if the rich and board numbers match this particular board.
    // If so, append the details to the boardPixelList.
    if(splitPixelList[1] == richNumber && splitPixelList[2] == boardNumber) {
      dynAppend(boardPixelList, splitPixelList);
    } // Check if the rich and board numbers match this particular board. 
  } // Loop over the pixel list.

  // The name of the hardware register  
  dyn_string mailboxReg = makeDynString(ukl1Name + ".FpgaConfig" + ".IngressMailbox3");
  
  // Now we have a list of channels and pixels to be disabled for this UKL1 board.
  // Get the length of this list.  
  int boardLength = dynlen(boardPixelList);
  if (boardLength > 0) DebugTN(ukl1Name + ": Pixels to mask = " + boardPixelList);
  
  // Loop over the board list
  for(int list = 1; list < boardLength+1; list++){
    
    // The data to be written to the Ingress Configuration Mailbox. 
    // We don't need mailboxData to be a dyn_dyn_char but the _fwUkl1_write function does.
    // We only use the first element of mailboxData.
    dyn_dyn_char mailboxData;
    // Create a string to hold the mailbox data in hexadecimal form and add the ingress configuration mailbox header.      
    string hexMailboxData = boardPixelList[list][4] + "001";
    // Add the address and mask.
    hexMailboxData = hexMailboxData + boardPixelList[list][5] + boardPixelList[list][6];
    // Add the end of command footer.
    hexMailboxData = hexMailboxData + "8000";
    // Convert to a binary number.        
    mailboxData[1] = fwCcpc_convertHexToByte(hexMailboxData);
    int mailboxDataLength = dynlen(mailboxData[1])/2;        
  
    // Convert the ingress fpga number to an integer.    
    int ing = (int)boardPixelList[list][3];
    // Prepare the ingress control register.
    _fwUkl1_setConfigForIngressFpgaChannel(ukl1Name, ing, FALSE, exceptionInfo);
    fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_maskPixels(): Failed to prepare the configuration FIFO for Ingress FPGA " + ing + " for writting.", "2", exInfoSize);  

    // We don't use the masks as the CCPC server just writes it into the FIFO and this causes the loss of data. 
    // It must be empty.
    dyn_dyn_char mailboxMasks;

    // Write the data to the hardware register
    _fwUkl1_write(mailboxReg, mailboxData, mailboxMasks, FALSE, exceptionInfo);
    fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_maskPixels(): Failed to write configuration data for Ingress FPGA " + ing, "2", exInfoSize);

    // Trigger the transfer of the data to the ingress    
    _fwUkl1_sendIngressChannelConfigFifo(ukl1Name, FALSE, exceptionInfo);
    fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_maskPixels(): Triggering the transmission of the configuration data for Ingress FPGA " + ing, "2", exInfoSize);
  }   
  // Done.
  return;
}// fwUkl1_maskPixels()
