// ====================================
//  NAME    : FWUKL1 FSM action Library
// 
//  VERSION : 3
// 
//  REVISION: 1
// 
//  DATE    : 2010/08/27
// 
//  RELEASE : 2010/08/27
// 
//  PLATFORM: PVSS II
// 
//  AUTHOR  : Gareth Rogers, Steve Wotton, Harry Cliff
// 
//  EMAIL   : wotton@hep.phy.cam.ac.uk
// ====================================

/* \mainpage UKL1 Library Documentation
 *
 * This library contains UKL1 FSM actions.
 */

// ====================
//  #USES DECLARATIONS
// ====================

// fwUkl1Ext is DLL containing fwUkl1 ctrl extensions
#uses "fwUkl1Ext"

/*! This UKL1 library contains all the functions to configure a UKL1 board. */
#uses "fwUkl1/fwUkl1DPInterface.ctl"

/*! Provides access to the registers, both hardware and database. */
#uses "fwUkl1/fwUkl1RegisterInterface.ctl"

/*! Contains the call back functions that are available for the UKL1s. */
#uses "fwUkl1/fwUkl1CB.ctl"

/*! These are the callback functions that have been created for use in the UKL1 panels. */
#uses "fwUkl1/fwUkl1PanelCB.ctl"

/*! fwUkl1ExcepionHandling library is used to store and display errors that occur with fwUkl1 function calls. */
#uses "fwUkl1ExceptionHandling/fwUkl1ExceptionHandling.ctl"

/*! fwHw library is used for recipe and data point manipulation. */
#uses "fwHw.ctl"

/*! fwCcpc library is used for accessing the UKL1 registers. */
#uses "fwCcpc.ctl"

/*! fwFSMConfDB library is used to add and remove the FSM confDB object to the FSM tree. */
#uses "fwFSMConfDB/fwFsmConfDB.ctl"

/*! fwUkl1.ctl library */
#uses "fwUkl1/fwUkl1.ctl"

/*!
 * The Configure command for the UKL1 boards
 *
 * @param  domain
 * @param  device
 * @param  command
 * @param  exceptionInfo
 * @param  exInfoSize
 */
void fwUkl1_HwTypeCCPCUKL1_CONFIGURE(string domain, string device, string command, dyn_string &exceptionInfo, int &exInfoSize) {
 
  // Inform the world that we are configuring.
  dpSet(device+".status","CONFIGURING");
    
  string sysName = fwUkl1_removeDeviceName(device);
  
  // Holds the name of the recipe that is to be loaded.
  string recipeName = "";
  // Determine what the parameter was given and hence the recipe name.
  string runType;
  fwDU_getCommandParameter(domain, device, "RUN_TYPE", runType);
  if ("" == runType)
    fwCU_getParameter(domain, "RUN_TYPE", runType);
      
//  DebugTN("runType = " + runType + ", domain = " + domain + ", device = " + device);
     
  recipeName += runType;
  recipeName += "/";
  recipeName += command;

/*  
  // Determine the partition ID. Will retrieve a decimal number stored in a string.
  string sPartId;
  fwDU_getCommandParameter(domain, device, "PART_ID", sPartId);
  if ("" == sPartId)
    fwCU_getParameter(domain, "PART_ID", sPartId);
  // Convert the string to an int so it can be made into a hex number.
  sPartId = fwCcpc_convertDecToHex( (int)sPartId );
      
  // Determine the MTU from the run info.
  string runInfo;
  fwDU_getCommandParameter(domain, device, "RUN_INFO_DP", runInfo);
  int mtu = fwUkl1_getMtuFromRunInfo(runInfo, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_doCommand(): Failed to retrieve the ethernet MTU size from the given run info data point. Taking value from the recipe.", "2", exInfoSize);
*/
  
// Most egress settings have already been applied (at RESET).
// Now set all common ingress settings

  dyn_dyn_anytype tab;      
  dpQuery("SELECT '_online.._value,_general.._dyn_string_01' FROM '"+sysName+"DefaultConfiguration.ChannelConfiguration.*' WHERE _DPT= \"FwUkl1Configuration\" ", tab);

// Get the list of enabled channels
  dyn_dyn_anytype enbtab;      
  dpQuery("SELECT '_online.._value' FROM '"+sysName+"FwUkl1Settings/"+device+".*.Enable' WHERE _DPT= \"FwUkl1Settings\" ", enbtab);

// Get the list of masked pixels
  dyn_dyn_anytype masktab;
  if (runType == "ALICE")
  {
    dpQuery("SELECT '_online.._value' FROM '"+sysName+"FwUkl1Settings/"+device+".ALICEPixelMasks' WHERE _DPT= \"FwUkl1Settings\" ", masktab);
  }
  else
  {
    dpQuery("SELECT '_online.._value' FROM '"+sysName+"FwUkl1Settings/"+device+".LHCbPixelMasks' WHERE _DPT= \"FwUkl1Settings\" ", masktab);
  }

  dyn_char mailboxData = fwUkl1Ext_makeConfigureSequence(tab,enbtab,masktab,runType);
  dyn_dyn_char ddcMailboxData;
  ddcMailboxData[1] = makeDynChar();
  dynAppend(ddcMailboxData[1],mailboxData);
  dyn_string mailboxReg = makeDynString(device + ".FpgaConfig.ConfigurationSequence");
  dyn_int diStatus;
  if (!fwCcpc_write(mailboxReg, ddcMailboxData, makeDynChar(), diStatus))
  {
    DebugTN(device + " _CONFIGURE: Error from fwCcpc_write().");
  }
  
  // Check the GBE link status.  The function returns 0 if no critical errors occur, -1 in case of a critical error when the board FSM state is set to ERROR.
  bool autoGBEDisable;
  dpGet(fwUkl1_removeDeviceName(device) + "UKL1Constants.GBEAutoDisableFlag", autoGBEDisable);
  if(autoGBEDisable) {      
    int errorGBE = fwUkl1_checkForBrokenGBELinks(device, exceptionInfo);
    fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_configure(): An problem was detected while checking the GBE links.", "2", exInfoSize);
    if(errorGBE == -1) {
      dpSet(device+".status","ERROR");
      return;
    }   
  } 
  else{
    DebugTN("WARNING: Automatic disabling of disconnected GBE ports is switched off, this may lead to errors in the case of a damaged linecard.");      
    dpSet(fwUkl1_removeDeviceName(device) + "UKL1Constants.GBELinkAlert", -2);
  }
      
  // Mask the pixels.
  bool mask;
  dpGet(fwUkl1_removeDeviceName(device) + "UKL1Constants.MaskPixels", mask);
  if(mask == TRUE) {
     fwUkl1_maskPixels(device, runType, exceptionInfo);
  }
  return;
}// fwUkl1_HwTypeCCPCUKL1_CONFIGURE() 
  
/*!
 * The Stop command for the UKL1 boards
 *
 * @param  domain
 * @param  device
 * @param  command
 * @param  exceptionInfo
 * @param  exInfoSize
 */
void fwUkl1_HwTypeCCPCUKL1_STOP(string domain, string device, string command, dyn_string &exceptionInfo, int &exInfoSize) {
  
  dyn_char rData;
  int rc = fwCcpc_LBUSRead(device,0x0800083C,FWCCPC_LBUS_BITS_32,1,rData);
  int regin = rData[1]<<24 | rData[2]<<16 | rData[3]<<8 | rData[4];
  
  dyn_char mailboxData = fwUkl1Ext_makeStopSequence(regin);
  dyn_dyn_char ddcMailboxData;
  ddcMailboxData[1] = makeDynChar();
  dynAppend(ddcMailboxData[1],mailboxData);
  dyn_string mailboxReg = makeDynString(device + ".FpgaConfig.ConfigurationSequence");
  dyn_int diStatus;
  if (!fwCcpc_write(mailboxReg, ddcMailboxData, makeDynChar(), diStatus))
  {
    dpSet(device+".status","ERROR");
    DebugTN(device + " _STOP: Error from fwCcpc_write().");
  }
  else
  {
    dpSet(device+".status","READY");
  }
  // Start and stop are configured from hardcoded recipe names. The FSM provides no choice in the matter.
  //  fwUkl1_stop(device, exceptionInfo);
//  if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_doCommand(): Failed to perform the stop action for " + device + ", transition to READY state failed.", "1", exInfoSize))
//    dpSet(device+".status","READY");
//  else
//    dpSet(device+".status","ERROR");
  return;

}// fwUkl1_HwTypeCCPCUKL1_STOP()

/*!
 * The Start command for the UKL1 boards
 *
 * @param  domain
 * @param  device
 * @param  command
 * @param  exceptionInfo
 * @param  exInfoSize
 */
void fwUkl1_HwTypeCCPCUKL1_START(string domain, string device, string command, dyn_string &exceptionInfo, int &exInfoSize) {

  // Reset any HPDs that have been disabled by the automatic HPD disabling.
  fwUkl1_resetDisabledHPDs(device, exceptionInfo); 

  dyn_char rData;
  int rc = fwCcpc_LBUSRead(device,0x0800083C,FWCCPC_LBUS_BITS_32,1,rData);
  int regin = rData[1]<<24 | rData[2]<<16 | rData[3]<<8 | rData[4];
  
  dyn_char mailboxData = fwUkl1Ext_makeStartSequence(regin);
  dyn_dyn_char ddcMailboxData;
  ddcMailboxData[1] = makeDynChar();
  dynAppend(ddcMailboxData[1],mailboxData);
  dyn_string mailboxReg = makeDynString(device + ".FpgaConfig.ConfigurationSequence");
  dyn_int diStatus;
  if (!fwCcpc_write(mailboxReg, ddcMailboxData, makeDynChar(), diStatus))
  {
    dpSet(device+".status","ERROR");
    DebugTN(device + " _START: Error from fwCcpc_write().");
  }
  else
  {
    dpSet(device+".status","RUNNING");
  }

  // Start and stop are configured from hardcoded recipe names. The FSM provides no choice in the matter.
//  fwUkl1_start(device, exceptionInfo);
//  if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_doCommand(): Failed to perform the start action for " + device + ", transition to RUNNING state failed.", "1", exInfoSize)) {
//    dpSet(device+".status","RUNNING");
//  }
//  else {
//    dpSet(device+".status","ERROR");
//  }
  return;
}// fwUkl1_HwTypeCCPCUKL1_START()

/*!
 * The Reload command for the UKL1 boards
 *
 * @param  domain
 * @param  device
 * @param  command
 * @param  exceptionInfo
 * @param  exInfoSize
 */
void fwUkl1_HwTypeCCPCUKL1_RELOAD(string domain, string device, string command, dyn_string &exceptionInfo, int &exInfoSize) {

  // Reloads the firmware for the FPGAs.
  fwUkl1_reload(device, exceptionInfo);
  if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_doCommand(): Failed to perform the reload action for " + device + ", transition to NOT_READY state failed.", "1", exInfoSize))
    dpSet(device+".status","NOT_READY");
  else
    dpSet(device+".status","ERROR");
  return;
}//fwUkl1_HwTypeCCPCUKL1_RELOAD()

/*!
 * The Reset command for the UKL1 boards
 *
 * @param  domain
 * @param  device
 * @param  command
 * @param  exceptionInfo
 * @param  exInfoSize
 */
void fwUkl1_HwTypeCCPCUKL1_RESET(string domain, string device, string command, dyn_string &exceptionInfo, int &exInfoSize) {

  string sysName = fwUkl1_removeDeviceName(device);
  
// Write a crash dump file containing state of config and status regs.
  fwUkl1_writeStatusToFile(device,"");

// Trigger firmware reload.

  _fwUkl1_toggleGpioLine(device, FWCCPC_GPIO_LINE_6, exceptionInfo);
  if ( fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_RESET(): Failed to reload the UKL1 firmware. There was a problem with the hardware access.", "2", exInfoSize) )
  {
    DebugTN(device+"_RESET(): fwukl1_reload() failed.");
    dpSet(device+".status","ERROR");
    return;
  }

// Register read loop to check FPGAs ready
  
  delay(4);
  int regin = 0;
  for (unsigned i=0;i<10;i++)
  {
    delay(0,500);
    dyn_char rData;
    int rc = fwCcpc_LBUSRead(device,0x08000840,FWCCPC_LBUS_BITS_32,1,rData);
    if ( rc == -1 ) continue;
    regin = rData[3]<<8 | rData[4];
    if (regin == 0xface) break;
  }

  if (regin != 0xface)
  {
    DebugTN(device + "_RESET(): firmware reload failed.");
    dpSet(device+".status","ERROR");
    return;
  }

  bool init = TRUE;
  
  if ( init )
  {

// Enable MDIO control to allow checking of PHY status (during CONFIGURE).
    fwUkl1_enableMDIOCtrl(device);

// Need to mask off ingresses that are not configured. They may not even exist on v2p2 boards.

// Read the configuration.
    dyn_dyn_char inhbVal;
    _fwUkl1_read(makeDynString(device+".FpgaConfig.IngressInhibit"),inhbVal,exceptionInfo);
    if ( fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_reload(): Failed to get ingress inhibit configuration.", "2", exInfoSize) )
    {
      dpSet(device+".status","ERROR");
      return;
    }
    char inhbCfg = ~((inhbVal[1][4] & 0xf) | ((inhbVal[1][4] & 0xf) << 4));
// Check the ingress-egress links.
    dyn_string regName = makeDynString(device + ".FpgaStatus.GigabitTransceiverInhibits");
    dyn_dyn_char readVal;
    _fwUkl1_read(regName, readVal, exceptionInfo);
    if ( fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_reload(): Failed to get ingress inhibit status.", "2", exInfoSize) )
    {
      dpSet(device+".status","ERROR");
      return;
    }
// Check the inhibits but remember to mask off the part that is ORed with the configuration bits.
    char inhbStatusOr = readVal[1][4] | (readVal[1][3] & 0xf0);
    if ((inhbStatusOr & inhbCfg)!=0)
    {
      DebugTN(device+":fwUkl1_reload() Not all egress-ingress links ready.");
      dpSet(device+".status","ERROR");
      return;
    }

// This performs some configuration routines for the GBE, setting the GBE into its running mode.
    _fwUkl1_configureHardcodedSettings(device, exceptionInfo);
    fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_HwTypeCCPCUKL1_RESET(): Failed to initialise GBE or TTCrx.", "2", exInfoSize);

// Settings common to all boards.
    _fwUkl1_configureCommonSettings(device, exceptionInfo);
    fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_HwTypeCCPCUKL1_RESET(): Failed to initialise some/all of the common settings.", "2", exInfoSize);

    dpSet(device+".status","NOT_READY");
  }
  else
  {
    dpSet(device+".status","ERROR");
  }
  
  return;
}//fwUkl1_HwTypeCCPCUKL1_RESET()

/*!
 * The Recover command for the UKL1 boards
 *
 * @param  domain
 * @param  device
 * @param  command
 * @param  exceptionInfo
 * @param  exInfoSize
 */
void fwUkl1_HwTypeCCPCUKL1_RECOVER(string domain, string device, string command, dyn_string &exceptionInfo, int &exInfoSize) {
  
  // Write a crash dump file containing state of config and status regs.
  fwUkl1_writeStatusToFile(device,"");

  // A recover should do the same as a reset, but only operate on those UKL1s that are in the error state.
  int status = fwUkl1_reload(device, exceptionInfo);
    
  if ( status == 0 )
  {
    dpSet(device+".status","NOT_READY");
  }
  else if ( status == 1 )
  {
    int status2 = fwUkl1_reload(device, exceptionInfo);
    if ( status2 != 0)
    {
      dpSet(device+".status","ERROR");
    }
    else
    {
      dpSet(device+".status","NOT_READY");
    }
  }
  else
  {
    dpSet(device+".status","ERROR");
  }
  return;
  
}//fwUkl1_HwTypeCCPCUKL1_RECOVER()

/*!
 * The Configure command for the UKL1 HPDDisabling FSM
 *
 * @param  domain
 * @param  device
 * @param  command
 * @param  exceptionInfo
 * @param  exInfoSize
 */
void fwUkl1_HPDDisabling_CONFIGURE(string domain, string device, string command, dyn_string &exceptionInfo, int &exInfoSize) {
      // Inform the world that we are configuring.
      dpSet(device+".status","CONFIGURING");
    
      // Determine the partition name from the run info.
      string runInfo;
      fwDU_getCommandParameter(domain, device, "RUN_INFO_DP", runInfo);
      
      string partitionName = fwUkl1_getPartitionNameFromRunInfo(runInfo, exceptionInfo);
      fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "FwUkl1HPDDisabling_doCommand(): Failed to retrieve the partition name from the run info DP. Have set an empty partition ID for this run.", "2", exInfoSize);
      // Even if we are given an empty partition name from the function we should set it anyway.
      // The partition ID should be cleared rather that 'remembering' its value from the
      // previous run.
      if(dpSet(device+".partitionID", partitionName)!=0) {
        exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "FwUkl1HPDDisabling_doCommand(): Failed to set the partition name `"+partitionName+"' to the UKL1 DP constant.", "2");
        dpSet(device+".status", "ERROR");
      }
      return;
}//fwUkl1_HPDDisabling_CONFIGURE()

/*!
 * The Stop command for the UKL1 HPDDisabling FSM
 *
 * @param  domain
 * @param  device
 * @param  command
 * @param  exceptionInfo
 * @param  exInfoSize
 */
void fwUkl1_HPDDisabling_STOP(string domain, string device, string command, dyn_string &exceptionInfo, int &exInfoSize) {
      // Check to see if the thread is still running
      // Get the thread ID  
      int ThreadID; 
      dpGet(device+".threadID", ThreadID); 
      // If the thread is still running (ThreadID is greater than or equal to zero) we need to stop the thread before 
      // we unsubscribe from the HPD disabling DIM service
      if(ThreadID >= 0) {
        // stopThread returns -1 in the event of failure
        DebugTN("The fwUkl1_subscribeHPDDisableDimService is still running with thread id = " + ThreadID);
        int StoppedThread = stopThread(ThreadID);
        if(StoppedThread == -1) {
          exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "FwUkl1HPDDisabling_doCommand(): Failed to stop the fwUkl1_subscribeHPDDisableDimService thread.", "2");
        }
        else {
          DebugTN("The fwUkl1_subscribeHPDDisableDimService thread with id = " + ThreadID + " has been stopped");
          // Set the thread ID to be -3, denoting that the thread terminated 
          dpSet(device+".threadID", -3);
        }  
      }
      // Now we need to unsubscribe from the HPD disabling DIM service    
      fwUkl1_unsubscribeHPDDisableDimService();     
    
      // Check to see if the HPD counting thread is still running.      
      int counterThreadID; 
      dpGet(device+".counterThreadID", counterThreadID); 
      // If the thread is still running (counterThreadID is greater than or equal to zero) we need to stop the thread.
      if(counterThreadID >= 0) {
        // stopThread returns -1 in the event of failure
        DebugTN("The fwUkl1_countDisabledHPDs thread is still running with thread id = " + counterThreadID);
        int StoppedCounterThread = stopThread(counterThreadID);
        if(StoppedCounterThread == -1) {
          exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "FwUkl1HPDDisabling_doCommand(): Failed to stop the fwUkl1_countDisabledHPDs thread.", "2");
        }
        else {
          DebugTN("The fwUkl1_fwUkl1_countDisabledHPDs thread with id = " + counterThreadID + " has been stopped");
          // Set the thread ID to be -3, denoting that the thread terminated 
          dpSet(device+".counterThreadID", -3);
        }  
      }
    
      dpSet(device+".status","READY");
      // Done     
      return;
}// fwUkl1_HwTypeHPDDisabling_STOP()

/*!
 * The Start command for the UKL1 HPDDisabling FSM
 *
 * @param  domain
 * @param  device
 * @param  command
 * @param  exceptionInfo
 * @param  exInfoSize
 */
void fwUkl1_HPDDisabling_START(string domain, string device, string command, dyn_string &exceptionInfo, int &exInfoSize) {
      // As we are leaving the READY state for RUNNING we should try and subscribe to the DIM services.
      // Once we have removed the device name from the UKL1 we should be left with a valid system name,
      // which could of course be an emtpy string!
        string partitionName;
        string sysName = fwUkl1_removeDeviceName(device);
//        DebugTN("device = " + device);
        
        // Start the thread which counts the number of disabled HPDs and save the thread ID.
        dpSet(device+".counterThreadID", startThread("fwUkl1_countDisabledHPDs", sysName)); 
        
        if(dpGet(device+".partitionID", partitionName)==0) {
           // Save the thread ID to the dp
           dpSet(device+".threadID", startThread("fwUkl1_subscribeHPDDisableDimService", partitionName, exceptionInfo));
           dpSet(device+".status", "RUNNING");        
         }
        else {
          exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "FwUkl1HPDDisabling_doCommand(): Failed to subscribe to the DIM service which will disable the UKL1 channels. HPDs in error will not be suppressed for this run.", "2");
          dpSet(device+".status", "ERROR");   
        }
        return;
}// fwUkl1_HwTypeHPDDisabling_START()

/*!
 * The Recover command for the UKL1 HPDDisabling FSM
 *
 * @param  domain
 * @param  device
 * @param  command
 * @param  exceptionInfo
 * @param  exInfoSize
 */
void fwUkl1_HPDDisabling_RECOVER(string domain, string device, string command, dyn_string &exceptionInfo, int &exInfoSize) {
      
      // Check to see if the thread is still running
      // Get the thread ID  
      int ThreadID; 
      dpGet(device+".threadID", ThreadID); 
      // If the thread is still running (ThreadID is greater than or equal to zero) we need to stop the thread before 
      // we unsubscribe from the HPD disabling DIM service
      if(ThreadID >= 0) {
        // stopThread returns -1 in the event of failure
        DebugTN("The fwUkl1_subscribeHPDDisableDimService is still running with thread id = " + ThreadID);
        int StoppedThread = stopThread(ThreadID);
        if(StoppedThread == -1) {
          exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "FwUkl1HPDDisabling_doCommand(): Failed to stop the fwUkl1_subscribeHPDDisableDimService thread.", "2");
        }
        else {
          DebugTN("The fwUkl1_subscribeHPDDisableDimService thread with id = " + ThreadID + " has been stopped");
          // Set the thread ID to be -3, denoting that the thread terminated 
          dpSet(device+".threadID", -3);
        }  
      }
      // Now we need to unsubscribe from the HPD disabling DIM service    
      fwUkl1_unsubscribeHPDDisableDimService();     

      // Check to see if the HPD counting thread is still running.      
      int counterThreadID; 
      dpGet(device+".counterThreadID", counterThreadID); 
      // If the thread is still running (counterThreadID is greater than or equal to zero) we need to stop the thread.
      if(counterThreadID >= 0) {
        // stopThread returns -1 in the event of failure
        DebugTN("The fwUkl1_countDisabledHPDs thread is still running with thread id = " + counterThreadID);
        int StoppedCounterThread = stopThread(counterThreadID);
        if(StoppedCounterThread == -1) {
          exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "FwUkl1HPDDisabling_doCommand(): Failed to stop the fwUkl1_countDisabledHPDs thread.", "2");
        }
        else {
          DebugTN("The fwUkl1_fwUkl1_countDisabledHPDs thread with id = " + counterThreadID + " has been stopped");
          // Set the thread ID to be -3, denoting that the thread terminated 
          dpSet(device+".counterThreadID", -3);
        }  
      }
      
      dpSet(device+".status","NOT_READY");
      // Done.
      return;
}// fwUkl1_HwTypeHPDDisabling_RECOVER()

/*!
 * The Reset command for the UKL1 HPDDisabling FSM
 *
 * @param  domain
 * @param  device
 * @param  command
 * @param  exceptionInfo
 * @param  exInfoSize
 */
void fwUkl1_HPDDisabling_RESET(string domain, string device, string command, dyn_string &exceptionInfo, int &exInfoSize) {

  fwUkl1_HPDDisabling_RECOVER(domain,device,command,exceptionInfo,exInfoSize);
  
  // Clear the HPD disabling data points
  string sysName = fwUkl1_removeDeviceName(device);
  dpSet(sysName+"HPD_Disabling.totalNumberOfDisabledHPDs", 0);
  dpSet(sysName+"HPD_Disabling.totalWeightOfDisabledHPDs", 0);
  dpSet(sysName+"HPD_Disabling.numberOfMonitoringDisabledHPDs", 0);
  dpSet(sysName+"HPD_Disabling.weightOfMonitoringDisabledHPDs", 0);
  dpSet(sysName+"HPD_Disabling.numberOfHardwareDisabledHPDs", 0);
  dpSet(sysName+"HPD_Disabling.weightOfHardwareDisabledHPDs", 0);
  dpSet(sysName+"HPD_Disabling.HPDsDisabledByMonitoring", "");
  dpSet(sysName+"HPD_Disabling.HPDsDisabledByHardware", "");

  return;
}// fwUkl1_HwTypeHPDDisabling_RESET()
/*!
 * Configures settings common to all boards.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 */
void _fwUkl1_configureCommonSettings(const string& ukl1Name, dyn_string& exceptionInfo) {
  dyn_dyn_anytype tab;
  string sysName = fwUkl1_removeDeviceName(ukl1Name);
  
  dpQuery("SELECT '_online.._value,_general.._dyn_string_01' FROM '"+sysName+"DefaultConfiguration.*' WHERE _DPT= \"FwUkl1Configuration\" ", tab);
  
// Get the board personality data from the PROM

  dyn_char promData;
  int rc = _fwCcpc_read(ukl1Name,"1,0,80,0,36,8,0,1",promData,FWCCPC_I2C);

  dyn_char mailboxData = fwUkl1Ext_makeResetSequence(tab,promData);

  // Write the channel configuration data for this ingress to the mailbox
// Call fwCcpc_write directly. _fwUkl1_write tries to read back which is not allowed for this.
  
  dyn_dyn_char ddcMailboxData;
  ddcMailboxData[1] = makeDynChar();
  dynAppend(ddcMailboxData[1],mailboxData);
  dyn_string mailboxReg = makeDynString(ukl1Name + ".FpgaConfig.ConfigurationSequence");
  dyn_int diStatus;
  if (!fwCcpc_write(mailboxReg, ddcMailboxData, makeDynChar(), diStatus))
  {
    DebugTN(ukl1Name + " _fwUkl1_configureCommonSettings: Error from fwCcpc_write().");
  }
  
  return;
}// _fwUkl1_configureCommonSettings()

