
//  NAME    : fwUkl1DPInterface
// 
//  VERSION : 3
// 
//  REVISION: 1
// 
//  DATE    : 2008/07/27
// 
//  RELEASE : 2010/01/20
// 
//  PLATFORM: PVSS II
// 
//  AUTHOR  : Gareth Rogers
// 
//  EMAIL   : rogers@hep.phy.cam.ac.uk
// ====================================

/* \mainpage UKL1 datapoint interface library documentation
 *
 * This library provides the all the functions that interface to the UKL1 project datapoints.
 *
 * The library provides an interface to the UKL1 constants datapoint and the software representation
 * of the UKL1 boards, which is a collection of several datapoint structures.
 */

// ====================
//  #USES DECLARATIONS
// ====================

/*! fwUkl1ExcepionHandling library is used to store and display errors that occur with fwUkl1 function calls. */
#uses "fwUkl1ExceptionHandling/fwUkl1ExceptionHandling.ctl"

/*! fwHw library is used for recipe and data point manipulation. */
#uses "fwHw.ctl"

/*! fwCcpc library is used for accessing the UKL1 registers. */
#uses "fwCcpc.ctl"

/*! fwFSMConfDB library is used to add and remove the FSM confDB object to the FSM tree. */
#uses "fwFSMConfDB/fwFsmConfDB.ctl"

/* Progress bar */
#uses "fwGeneral/fwProgressBar.ctl"

/*! The FSM libraries are used to create and configure the UKL1 FSM tree. */
#uses "fwCU"
#uses "fwDU"

// =========================
//  CONSTANTs & GLOBAL VARs
// =========================

/** @defgroup SectionConstants Constants
 *
 *    These are the constants used by the library and are available for library users. Primaryly
 *    they define the DP structures in use and provide a means to interact with the DPEs without
 *    the need to hardcode names.
 *
 *  @{
 */

/*! These constants define the possible system numbers that are used in the fwUkl1 projects. */
const int FWUKL1_RICH1_SYS_NUM = "136";
const int FWUKL1_RICH2_SYS_NUM = "176";
const int FWUKL1_RICH3_SYS_NUM = "176";

/*! These constants define the values that the UKL1 CU should take in the different RICH systems. */
const string FWUKL1_RICH1_CU = "RICH1_L1";
const string FWUKL1_RICH2_CU = "RICH2_L1";
const string FWUKL1_RICH3_CU = "RICH3_L1";

/*! Defines the RICH ID to be used for different RICH systems. RICH3 is the SSB2 lab and test systems. */
const int FWUKL1_RICH1_ID = 1;
const int FWUKL1_RICH2_ID = 2;
const int FWUKL1_RICH3_ID = 3;

/*! Defines the name of the data point type of the UKL1 hardware type. All UKL1 board data points are created of this type. */
const string FWUKL1_HW_TYPE = "UKL1";

/*! Broadcast FE FPGA number. Value used to write to all the FE FPGAs. */
const unsigned FWUKL1_BROADCAST_FPGA = 5;
/*! Broadcast FE FPGA channel number. Value used to write to all of a FE FPGA's channels. */
const unsigned FWUKL1_BROADCAST_CHANNEL = 10;
/*! Number of Front end (ingress) FPGAs on the RICH L1 board. */
const unsigned FWUKL1_FRONT_FPGAS = 4;
/*! Number of channels that are on each front end FPGA. */
const unsigned FWUKL1_FPGA_CHANNELS = 9;
/*! Number of channels that are on the RICH L1 board. */
const unsigned FWUKL1_BOARD_CHANNELS = 36;
/*! Number of ports on the gigabit ethernet card. */
const unsigned FWUKL1_GBE_PORTS = 4;
/*! Number of temperature sensors present on the board. */
const unsigned FWUKL1_TEMPERATURE_SENSORS = 3;

/*! This numeric constant is used to identify a production UKL1 board. */
const unsigned FWUKL1_PRODUCTION = 2;
/*! This numeric constant is used to identify a prototype UKL1 board. */
const unsigned FWUKL1_PROTOTYPE = 1;
/*! This numeric constant is used to identify a board found in the DIM server that is not a UKL1. */
const unsigned FWUKL1_NOT_UKL1 = 0;

/*!
 * This defines the name of the Hugin to be used in the FSM tree. This is typically only valid in RICH1 or 2.
 * Therefore the RICH3 Hugin name is an empty string and this forms the special case where it is not added.
 */
const string FWUKL1_RICH1_HUGIN_NAME = "r1hugin01";
const string FWUKL1_RICH2_HUGIN_NAME = "r2hugin02";
const string FWUKL1_RICH3_HUGIN_NAME = "No hugin";

/*! This is the name of the Hugin FSM type. */
const string FWUKL1_HUGIN_TYPE = "HUGIN";
/*! This is the TFC system name. */
const string FWUKL1_TFC_SYS_NAME = "TFC";

/*!
 * Constant to indicate if the FSM configurator object should apply the recipes itself.
 * Unless there is a change in the hardware tool this must be FALSE. Not sure exactly
 * what the simplified AR does, but it is related to having the configurator applying the
 * recipes.
 */
const bool FWUKL1_CONFIGURATOR_APPLY_RECIPES = FALSE;
const bool FWUKL1_CONFIGURATOR_SIMPLIFIED_AR = FALSE;

/*!
 * Constant to define the length of the time for the FSM configurator object should it be
 * applying the recipes itself. This only matters if FWUKL1_CONFIGURATOR_APPLY_RECIPES is
 * TRUE. See its documentation for comments on that.
 */
const int FWUKL1_CONFIGURATOR_TIMEOUT = 0;

/*! The name of the DPT that the UKL1 constants DP is created from. */
const string FWUKL1_CONSTS_DPT = "FwUkl1Constants";
/*!
 * This is the name of the datapoint that contains the various constants that should be defined
 * during post installation.
 */
const string FWUKL1_CONSTS_DP_BASENAME = "UKL1Constants";
/*!
 * FwUkl1Constants contains a bool to indicate if the FSM configurator object should retrieve the
 * recipes from the configuration database or from the recipe cache. This should be TRUE for all systems
 * where the configuration database is present. The following constant defines the name of the DPE
 * in FwUkl1Constants.
 */
const string FWUKL1_CONFIGURATOR_USE_CONFDB = "ConfiguratorUseConfDB";
/*!
 * The name of the top level control unit (CU) that is used in the L1 FSM tree is stored in FwUkl1Constants.
 * This is the name of the DPE that it is stored under.
 */
const string FWUKL1_CU = "FSMCUName";
/*!
 * The name of the Hugin that is in the UKL1 FSM, if any. An empty string would indicate that no Hugin is
 * to be placed in the tree.
 * This is the name of the DPE that it is stored under.
 */
const string FWUKL1_HUGIN_NAME = "HuginName";
/*!
 * If TRUE the UKL1 channels can be disabled via a DIM service, FALSE and if connected to that DIM service
 * requests will be ignored. This is the name of the DP element that stores this value.
 */
const string FWUKL1_DISABLE_UKL1_CHANNEL = "DisableUkl1Channel";

/*!
 * Name of the datapoint type that manages the DIM subscription for the UKL1 channel disabling.
 */
const string FWUKL1_DISABLE_UKL1_CHANNEL_DPT = "FwUkl1HPDDisabling";
/*! Name of the data point element that holds the current value published by DIM. */
const string FWUKL1_DISBALE_HPD_DPE = ".DisableHPD";
/*! Name of the data point that stores information about the disabled channels and how frequently they were disabled. */
const string FWUKL1_DISABLE_HPD_DPE_STATS = ".DisableHPDStats";
/*!
 * The last partition name to have been sent during configuration. Only valid after a configure command.
 * For READY and RUNNING states this should be valid. Any other state and it could be before a configure
 * command changes it.
 */
const string FWUKL1_DISABLE_HPD_PART_ID = ".PartitionID";

/*! The name of the DPT that contains the information about the various abstract registers. */
const string FWUKL1_UKL1_BOARD_DPT_NAME = "FwUkl1Board";
/*! The name of the DPT that the DP points for each of the abstract registers are created from. */
const string FWUKL1_ABSTRACT_REGISTER_DPT_NAME = "FwUkl1AbstractRegister";
/*! The name of the DP that contains the information about the various abstract registers. */
const string FWUKL1_UKL1_BOARD_DP_NAME  = "UKL1Board";
/*! Name of the element that holds the list of abstract register names. */
const string FWUKL1_ABSTRACT_REG_LIST   = ".AbstractRegisterList";
/*! The tool tip text that should be displayed by the shape displaying the register data. */
const string FWUKL1_TOOLTIP_TEXT    = ".ToolTipText";
/*! The function if any that should be used to combine the hardware register values for an abstract register. */
const string FWUKL1_COMBINER_FUNCTION = ".CombinerFunction";
/*! Values to use to check the combined value.*/
const string FWUKL1_COMBINER_VALIDATION = ".CombinerValidation";
/*! Group that the register is associated with. */
const string FWUKL1_REGISTER_GROUPS = ".Groups";
/*! The mask that should be applied to the data when it is written as a hex string.
    Does not take into account any perculiarities of the register e.g. the FPGAs. */
const string FWUKL1_MASKS = ".Masks";
/*! Name of the function that should be used to start the monitoring for this register. */
const string FWUKL1_MONITORING_START_FUNCTION = ".MonitoringInfo.StartFunction";
/*! Name of the function that should be used to stop the monitoring for this register. */
const string FWUKL1_MONITORING_STOP_FUNCTION = ".MonitoringInfo.StopFunction";
/*! Name of the function that should be used to start the monitoring for this register. */
const string FWUKL1_MONITORING_REGISTERS = ".MonitoringInfo.Registers";
/*! DPE element for the hardware read register use to display the register for configuration. */
const string FWUKL1_HW_CONFIG_READ       = ".HwConfigReadRegister";
/*! DPE element for the hardware read register use to display the register for status. */
const string FWUKL1_HW_STATUS_READ       = ".HwStatusReadRegister";
/*! DPE element for the hardware write register. */
const string FWUKL1_HW_WRITE             = ".HwWriteRegister";
/*! DPE element for the database read register. */
const string FWUKL1_DB_READ              = ".DbReadRegister";
/*! DPE element for the database write register. */
const string FWUKL1_DB_WRITE             = ".DbWriteRegister";
/*! Hardware register type number. Associated with either a write or read hardware register. */
const string FWUKL1_HW_REGISTER_TYPES    = ".HwRegisterTypes";
/*! Recipe types that contain this database register. */
const string FWUKL1_DB_RECIPE_TYPES      = ".DbRecipeTypes";
/*! List of registers associated with either a hardware or database register. */
const string FWUKL1_ACCESS_REGISTERS     = ".Registers";
/*! The function used to parse the register input. */
const string FWUKL1_PARSER_FUNCTIONS     = ".ParserFunctions";
/*! The function used to parse the register input. */
const string FWUKL1_PARSER_INFO          = ".ParserInfo";
/*! This is the function to use to read or write the information to or from the hardware or database. */
const string FWUKL1_ACCESS_FUNCTIONS     = ".AccessFunctions";
/*! This information is used to valid date the resulting register data to see if any warnings are associated with the value. */
const string FWUKL1_VALIDATION           = ".Validation";

/*! This is used to define the name of the group of registers that are in the Egress FPGA configuration group.*/
const string FWUKL1_EGRESS_CONFIG_GROUP = "EgressConfig";
/*! This is used to define the name of the group of registers that are in the Egress FPGA status group.*/
const string FWUKL1_EGRESS_STATUS_GROUP = "EgressStatus";
/*! This is used to define the name of the group of registers that are in the Ingress FPGA configuration group.*/
const string FWUKL1_INGRESS_CONFIG_GROUP = "IngressConfig";
/*! This is used to define the name of the group of registers that are in the Ingress FPGA status group.*/
const string FWUKL1_INGRESS_STATUS_GROUP = "IngressStatus";
/*!
 * This group forms the basis of any channel group. There are 36 input channels to a UKL1 board and these
 * can logically be group in many ways. Starting with the base group appending one of logical group constants
 * and the number for that group will allow that collection of registers to be displayed.
 */
const string FWUKL1_CHANNEL_CONFIG_GROUP = "ChannelConfig";
const string FWUKL1_CHANNEL_STATUS_GROUP = "ChannelStatus";
/*! This is used to define the name of the group of registers that are in the all input channel configuration group.*/
const string FWUKL1_ALL_CHANNEL_GROUP = "All";
const string FWUKL1_INGRESS_CHANNEL_GROUP = "Ingress";
const string FWUKL1_INDIVIDUAL_CHANNEL_GROUP = "Individual";
/*! This is used to define the name of the group of registers that are in the GBE configuration group.*/
const string FWUKL1_GBE_CONFIG_GROUP = "GbeConfig";
/*! This is used to define the name of the group of registers that are in the GBE status group.*/
const string FWUKL1_GBE_STATUS_GROUP = "GbeStatus";
/*! This is used to define the name of the group of registers that are in the GBE port configuration group.*/
const string FWUKL1_GBE_PORT_CONFIG_GROUP = "GbePortConfig";
/*! This is used to define the name of the group of registers that are in the GBE port status group.*/
const string FWUKL1_GBE_PORT_STATUS_GROUP = "GbePortStatus";
/*! This is used to define the name of the group of registers that are in the TTCrx configuration group.*/
const string FWUKL1_TTCRX_CONFIG_GROUP = "TtcrxConfig";
/*! This is used to define the name of the group of registers that are in the TTCrx status group.*/
const string FWUKL1_TTCRX_STATUS_GROUP = "TtcrxStatus";
/*! This is a sub-group which contains all the relevant TFC status bits. */
const string FWUKL1_TFC_STATUS_GROUP = "TfcStatus";
/*! This is a sub-group all the status registers that are archived and should trigger alarms. */
const string FWUKL1_ALARM_STATUS_GROUP = "AlarmStatus";
/*! This is a sub-group for the status registers for the Ingress to Egress link on the Egress FPGA. */
const string FWUKL1_E_IN_LINK_STATUS_GROUP = "EgressIngressLinkStatus";
/*! This is a sub-group for the MEP building and transmission status. */
const string FWUKL1_MEP_STATUS = "EgressMEPStatus";
/*! This is the sub-group for the UKL1 EPROM registers. */
const string FWUKL1_PROM_GROUP = "Prom";
/*! This is the sub-group for the sensors on the UKL1 board. */
const string FWUKL1_SENSOR_GROUP = "Sensor";
/*! This is the sub-group for the registers that are displayed on the overview panel. */
const string FWUKL1_OVERVIEW_GROUP = "Overview";
/*! This is the sub-group for the registers that are displayed on the optical receiver panel.*/
const string FWUKL1_OPTICAL_RX_CHANNEL_GROUP = "OpticalReceiver";
/*! This is the sub-group for the registers that are displayed on the zero suppression panel.*/
const string FWUKL1_ZERO_SUPPRESSION_CHANNEL_GROUP = "ZeroSuppression";

/*! This is a NULL hardware register for those that do not have them. */
const string FWUKL1_NULL_HWREG = "NullRegister";
/*! This is a function to indicate that no combination of hardware values should be done for that abstract register. */
const string FWUKL1_NO_COMBINER = "NoCombiner";
/*! This is the name of the DP node for the FGPGA configuration settings. */
const string FWUKL1_FPGA_CONFIG_HWTYPE = ".FpgaConfig";
/*! This is the name of the DP node for the FGPGA status settings. */
const string FWUKL1_FPGA_STATUS_HWTYPE = ".FpgaStatus";
/*! This is the name of the DP node for the UKL1 PROM register settings. */
const string FWUKL1_PROM_HWTYPE = ".PROM";
/*! This is the name of the DP node for the UKL1 temperature sensors. Needs to have the sensor number appended. */
const string FWUKL1_SENSOR_HWTYPE = ".Sensors";
/*! This is the name of the DP node for the UKL1 TTCrx registers. */
const string FWUKL1_TTCRX_HWTYPE = ".TTCRX";
/*! This is the name of the DP node for the UKL1 GBE registers. */
const string FWUKL1_GBE_HWTYPE = ".GBE";
/*! This is the name that must be added to the GBE type to get the GBE port DP node. It must have the port number appended. */
const string FWUKL1_GBE_PORT_HWTYPE = ".PORT";
/*! This is the name we use to indicate a function is write only and present read attempts. */
const string FWUKL1_WRITE_ONLY = "fwUkl1_writeOnly";
/*! This is the name we use to indicate a function is ready only and prevent write attempts. */
const string FWUKL1_READ_ONLY = "fwUkl1_readOnly";

// @}

// ==================
//  UKL1 BOARD SETUP
// ==================

/** @defgroup SectionSetup Board setup
 *    The functions in this section are responsible for setting up the software object that represents a UKL1 board.
 *    It provides a function that is capable of dynamically adding or removing the UKL1 board(s) from the system.
 *    An added board is then ready to use via the FSM and is visible in the appropriate places.
 *    The ability to manually add/delete UKL1 boards from the system is also provided, by the User must take care that
 *    they are adding/removing a valid UKL1 board to or from the system. None existent boards can be added to the system.\n
 *    UKL1 boards can be identified via their ID number and a function is provided that is capable of identifying UKL1
 *    production and prototype boards. It is via this function that the fwUkl1_findAndAddUkl1Boards() identifies UKL1
 *    boards from all the CCPCs present in the DIM server.
 *  @{
 */

/*!
 * Gets the names off all the CCPCs found in the DIM server and then identifies the UKL1 boards based on their ID number.
 * It will then add boards that are found to the system. It is also capable of removing boards that are found in the system
 * but no longer present in DIM.
 *
 * @param  refresh If TRUE the list existing list of UKL1 boards will be updated with any new boards found and those
 *           no longer present will be removed. FALSE all existing boards will be deleted along with their associated
 *           settings and the whole list created from new. The Hugin, FSM CU, FSMConfDB object and the abstract register
 *           list data point will all be left untouched.
 * @param  forceReconfigure If TRUE it will recreate the default settings from those hardcoded into the file and
 *           reconfigures the harware data point from these settings. If FALSE the settings are just loaded from the defaults.
 *           In the case of the FSM tree if TRUE it will force the DU, CU and FSMConfDB settings to all be updated.
 *           This won't cause the FSM tree to be restart unless there are DU's added to the tree. The Hugins will always
 *           be left alone. Finally if true then it will cause the abstract register list's data point to be reloaded
 *           from the hardcoded settings.
 * @param  showProgress If TRUE then the fwShowProgressBar() will be called indicating the progress of the function.
 *           If FALSE it will not be called, this is useful when running from a PVSS00ctrl or similar manager which do not have
 *           access to graphics functions.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @param  sysName The system name.
 * @return void.
 *
 * forceReconfigure and refresh are independent and forceReconfigure=True does not mean that all the existing boards will be
 * updated, only those new boards found in the system.
 *
 * This function will not add the Hugin to the FSM tree. The Hugin must be added manually and as a result it will not be
 * removed automatically either. Once the Hugin has been installed there are no functions in the fwUkl1 libraries to
 * remove it.
 *
 * This function reports its progress via fwShowProgressBar() and thus fwOpenProgressBar() can be called to monitor its progress.
 * It does not submit a percentage completion however and can only be used with types 1 and 3 with fwOpenProgressBar().
 */
void fwUkl1_findAndAddUkl1Boards(bool refresh, bool forceReconfigure, bool showProgress, dyn_string& exceptionInfo, string sysName = "") {
  // Note the size of the exception info.
  int exInfoSize = dynlen(exceptionInfo);
  // Stores a string for the progress bar or logfile
  string progress;
  // Holds the list of CCPC names retrieved.
  // First get a list of presently registered CCPCs in the DIM server, retrieving the appropriate system name.
  dyn_string ccpcList;
  fwCcpc_get(ccpcList);
  // Save the total number found for later use.
  const int numCcpc = dynlen(ccpcList);
  // Filter the CCPC names to determine, which of these are actually UKL1 boards.
  // Holds the names of the found UKL1 boards.
  dyn_string toBeAddedUkl1List;
  // Keep track of the number of UKL1 boards added to the list, start from 1 due to PVSS array silliness.
  unsigned numUkl1Added = 1;
  // Loop over all the found CCPCs.
  for (unsigned ccpc=1; ccpc<=numCcpc; ++ccpc) {
    // Name of the CCPC currently being checked.
    const string ccpcName = ccpcList[ccpc];
    // Check to see if it is a production or prototype.
    // Ignore the return as it will probably just indicate a prototype anyway...
    int isUkl1;
    string serialNumber;
    fwUkl1_isUkl1Board(ccpcName, isUkl1, serialNumber, exceptionInfo);
    fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_findAndAddBoards(): Adding board without a serial number to the system.", "2", exInfoSize);

// Only production boards are supported now    
    if (FWUKL1_PRODUCTION==isUkl1) {
      // Add the name of the board to the list of UKL1 names.
      toBeAddedUkl1List[numUkl1Added] = ccpcName;
      // Increment the number added.
      ++numUkl1Added;
      // Indicate our progress.
      progress = "Identified " + ccpcName + " with serial number 0x" + serialNumber + " as a UKL1 board.";
      if (showProgress)
        fwShowProgressBar(progress);
      else
        DebugTN(progress);
    }// if(0!=isUkl1) Don't do anything for else as we are just not interested in this board.
   }// CCPC for

  // Produce a warning message if we failed to find any UKL1s or even CCPCs.
  if (0==numCcpc) {
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_findAndAddBoards(): No CCPCs found in the DIM server.", "2");
  }// if(0==numCcpc)
  else if (0==dynlen(toBeAddedUkl1List)) {
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_findAndAddBoards(): No UKL1 boards found, but found " + numCcpc + " CCPCs in the DIM server.", "2");
  }// elseif(0==dynlen(toBeAddedList))
  // If we didn't find anything then the rest of this section is doesn't really need to be done, but it won't do anything anyway.

  // Now we have a list of UKL1 boards in the system lets deal with them.
  // The variable toBeAddedList holds the names of the UKL1 boards that need to be added to the library.
  // This always need to contain all the found boards as we always need to check the datapoints for consistency
  // and ensure the relevant DIM and FSM functionality is established.
  // The function fwUkl1_addUkl1Board will ensure that the minimum possible is done depending on calling arguments.

  // Holds the names of the UKL1 boards that need to be removed from the library. It will either be those that are
  // no longer present after a refresh or all of them in the case of a reload.
  dyn_string toBeDeletedUkl1List;
  // Just find the new boards in the system and those that have been lost.
  if (refresh) {
    // Reset the number added.
    numUkl1Added = 1;
    // Look through the list of existing names and check for those that need to be deleted.
    // Get the list.
    const dyn_string currentList = fwUkl1_getUkl1NamesList(exceptionInfo, sysName);
    const int curListSize = dynlen(currentList);
    for (int board=1; board<=curListSize; ++board) {
      // Check all the names in the library and check if they are still present in the new server list.
      if (0==dynContains(toBeAddedUkl1List, currentList[board])) {
        // If they are not present they need to be deleted.
        toBeDeletedUkl1List[numUkl1Added] = currentList[board];
        progress = "Deleting "+currentList[board]+".";
        if (showProgress)
          fwShowProgressBar(progress);
        else
          DebugTN(progress);
      }// if(0==dynContains())
    }// for board
  }// if(refresh).
  else {
    // In this case we are starting afresh and can just remove any existing settings from 
    toBeDeletedUkl1List = fwUkl1_getUkl1NamesList(exceptionInfo, sysName);
    progress = "Deleting "+toBeDeletedUkl1List;
    if (showProgress)
      fwShowProgressBar(progress);
    else
      DebugTN(progress);
    // Will return an empty list in the event of an error. We will just have to see why later.
  }// else(refresh)

  // We will also delete any incomplete boards that are found in the system. They can always go.
  // Give all the incomplete names to the delete FSM DU and HW DP functions as they can handle a name
  // not exisiting.
  // Just return empty arrays in the event of failure so don't worry about errors.
  dynAppend(toBeDeletedUkl1List, fwUkl1_getUkl1HwDpNamesOnlyList(exceptionInfo, sysName));
  dynAppend(toBeDeletedUkl1List, fwUkl1_getUkl1FsmTreeNamesOnlyList(exceptionInfo, sysName));

  DebugTN("The following will be deleted: "+toBeDeletedUkl1List);
  DebugTN("The following will be added: "+toBeAddedUkl1List);
  
  // Delete those boards that need to be deleted.
  if (0!=dynlen(toBeDeletedUkl1List)) {
    // Exception info won't be checked after this call as we will continue anyway, just let the panel deal with the exceptions.
    fwUkl1_deleteUkl1Boards(toBeDeletedUkl1List, showProgress, exceptionInfo);
    fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_findAndAddBoards(): Failed to delete all the requested boards from the system.", "2", exInfoSize);
  }// if(0!=dynlen(toBeDeletedUkl1List))
  
  DebugTN("Delete done.");

  // Add those boards that need to be added and add them in ascending order so they look nicer in the FSM.
  dynSortAsc(toBeAddedUkl1List);
  if (0!=dynlen(toBeAddedUkl1List)) {
    // Exception info won't be checked after this call as we will continue anyway, just let the panel deal with the exceptions.
    fwUkl1_addUkl1Boards(toBeAddedUkl1List, forceReconfigure, showProgress, exceptionInfo);
    fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_findAndAddBoards(): Failed to all the requested boards to the system.", "2", exInfoSize);
  }// if(0!=dynlen(toBeAddedUkl1List))
  
  DebugTN("Add done.");
  
  // Finally check the abstract register DP and update if necessary.
  _fwUkl1_updateAbstractRegisterDP(forceReconfigure, showProgress, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_findAndAddBoards(): Failed to create/update the abstract register data point. This could prevent access to some/all of the registers.", "2", exInfoSize);

  // Found and added all the boards!
  progress = "Created UKL1 list.";
  if (showProgress)
    fwShowProgressBar(progress);
  else
    DebugTN(progress);
  return;
}// fwUkl1_findAndAddUkl1Boards()

/*!
 * Checks the given CCPC to see if it attached to UKL1 board. It identifies the board by the serial number accessible via
 * the I2C bus.
 *
 * @param  ccpcName Name of the CCPC as found in the DIM server.
 * @param  isUkl1 Returned by reference, there are three possible values:
 *   \li FWUKL1_PRODUCTION The board is a UKL1 production board.
 *   \li FWUKL1_PROTOTYPE The board is a UKL1 prototype board.
 *   \li FWUKL1_NOT_UKL1 The board is not a UKL1.
 * @param  serialNumber Returned by reference, the serial number of the found board. Returns:
 *   \li Serial number of found board, whether UKL1 or not.
 *   \li Prototype in the event that it is a prototype without a working PROM. This will occur in preference to fail in the event of an error!
 *   \li Fail if an error occurred.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 *
 * There exists a UKL1 prototype that did not have a working PROM and as such no serial number can be read. In this case the
 * an exception is returned, but isUkl1 will indicate that it is a prototype and will return `PROTOTYPE' for the serial number.
 * This should be treated with suspicion especially if it is known that there are no prototype boards in the system. It will
 * eventually be `retired' and this function will need to be updated.
 */
void fwUkl1_isUkl1Board(string ccpcName, unsigned& isUkl1, string& serialNumber, dyn_string& exceptionInfo) {
  // Attempt to read the serial number from the given CCPC. All boards should have this identifier in the same place.
  // This cannot be done through a fwUkl1_read as that assumes the existence of the relevant hardware data point.
  // We can't assume it exists as we want to check if the board is a UKL1 before creating it!
  // Use the fwCcpc library I2C read function. It should be on bus 0 at address 0x50, sub-address 0. It is 32-bits big (4-bytes)
  // and we will assume an infinite page size as it doesn't really matter.
  // The bus is not constant as when dealing with prototypes it could be 1, therefore run twice if it isn't found the first time.
  bool found = FALSE;
  int bus = 0;
  const int address=0x50, subAddress=0, size=4, pageSize=-1;
  // We wish the negative acknowledge on address to be enabled.
  const bool nack = TRUE;
  // Holds the read data.
  dyn_char dcData;
  // Holds the status of the read.
  int status;
  do {
    status = fwCcpc_I2CReadSub(ccpcName, bus, address, 0, size, pageSize, FWCCPC_I2C_COMBINED, nack, dcData);
    if (0==status) {
      // Successfully read the serial number from the CCPC.
      // Indicate we found the board.
      found = TRUE;
      // Is it an UKL1?
      string tmpstr = fwUkl1_convertByteToHexLittleEndian(dcData, exceptionInfo);
      const unsigned ccpcSerNum = fwCcpc_convertHexToDec(tmpstr);
      // If the word is byte swapped then the revision will occupy the 3rd nibble.
      // This we don't care about when identifying the board, hence mask it.
      if ((0x601 == (ccpcSerNum>>20)) || (0x1060 == (ccpcSerNum&0xF0FF))) {
        // Check whether the serial number indicates that it is a production or prototype board.
        if ((0x00020000 == (ccpcSerNum&0x000F0000)) || (0x0200 == (ccpcSerNum&0x0F00)))
          isUkl1 = FWUKL1_PRODUCTION;
        else
          isUkl1 = FWUKL1_PROTOTYPE;
        // Convert the serial number to a string.
        serialNumber = fwCcpc_convertDecToHex(ccpcSerNum);
      }// if UKL1 serial number.
      else {
        // The CCPC is not attached to a UKL1, so we must tell everyone.
        isUkl1 = FWUKL1_NOT_UKL1;
        serialNumber = fwCcpc_convertDecToHex(ccpcSerNum);
      }// else not a UKL1 serial number.
    }// if(0==status1)
    else if (0==bus) {
      // Didn't find anything this bus try reading from the next bus.
      ++bus;
    }// elseif(0==bus)
    else {
      // No buses found in the range we are searching. Must not be a UKL1.
      // We no longer support the prototype board without a PROM.
      isUkl1 = FWUKL1_NOT_UKL1;
      serialNumber = "Fail";
      fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_isUkl1Board(): " + ccpcName + " did not have a PROM on I2C buses 0 or 1 at address 0x50 and is not a UKL1 board.","2");
      // Either we break or set found to TRUE. This seems more honest :)
      break;
    }// else no 
  } while (!found);
  
  // Done.
  return;
}// fwUkl1_isUkl1Board()

/*!
 * Adds a list of UKL1 boards to the system. It creates the hardware datapoint, subscribing it to the DIM server;
 * and adding it to the FSM tree.
 *
 * @param  toBeAddedUkl1List A dyn_string of names of the UKL1 boards as they appear in the DIM server that need to
 *           be added to the system. If a name is already found in the system it will be removed and then readded.
 * @param  forceReconfigure If TRUE it will force the settings to be manually set and not to be loaded from the defaults
 *           for the fwHw datapoint type. FALSE and the settings are just loaded from the defaults.
 * @param  showProgress If TRUE then the fwShowProgressBar() will be called indicating the progress of the function.
 *           If FALSE it will not be called, this is useful when running from a PVSS00ctrl or similar manager which do not have
 *           access to graphics functions.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 *
 * This function reports its progress via fwShowProgressBar() and thus fwOpenProgressBar() can be called to monitor its progress.
 * It does not submit a percentage completion however and can only be used with types 1 and 3 with fwOpenProgressBar().
 */
void fwUkl1_addUkl1Boards(dyn_string toBeAddedUkl1List, bool forceReconfigure, bool showProgress, dyn_string &exceptionInfo) {
  // Current length of the exception info, used to determine if more errors have appeared.
  int exInfoSize = dynlen(exceptionInfo);
// String to store progress info
  string progress;
  // Store the number of elements in the list.
  const unsigned numToBeAdded = dynlen(toBeAddedUkl1List);
  
  DebugTN("fwUkl1_addUkl1Boards: adding "+toBeAddedUkl1List);

  // If the list was empty return!
  if (0==numToBeAdded)
    return;

  if (!dpExists("_defaultSettingsCCPCUKL1"))
  {
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_addUkl1Boards(): _defaultSettingsCCPCUKL1 DP does not exist", "2");
    return;
  }
  
  // Loop over each board adding them to the system.
//  string errMsg;
//  for (unsigned board=1; board<=numToBeAdded; ++board) {
//    const string ukl1Name = toBeAddedUkl1List[board];
//    // Create the hardware type datapoint and subscribe it to the DIM server.
//    // We want to just load the new datapoint from the HW type defaults and if it does exist then lets delete it.
//    _fwUkl1_createHwTypeCCPCUKL1Datapoint(ukl1Name, forceReconfigure, exceptionInfo);
//    // Apparent bug where variables placed in strings are not updated if past directly to functions.
//    errMsg = "fwUkl1_addUkl1Boards(): Failed to setup " + ukl1Name + "'s connection with the DIM server. Communication with this UKL1 is not possible.";
//    fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", errMsg, "2", exInfoSize);
//    progress = "Added HW DP for " + ukl1Name + ".";
//    if (showProgress)
//      fwShowProgressBar(progress);
//    else
//      DebugTN(progress);
//  }// for each board.

  for (unsigned board=1; board<=numToBeAdded; ++board) {
    const string ukl1Name = toBeAddedUkl1List[board];
//    // Create the hardware type datapoint and subscribe it to the DIM server.
    if (forceReconfigure) dpDelete(ukl1Name);
    if (!dpExists(ukl1Name))
    {
      dpCreate(ukl1Name,"HwTypeCCPCUKL1");
      dpCopyBufferClear();
      int rc;
      dpCopy("_defaultSettingsCCPCUKL1",ukl1Name,rc);
      dpCopyOriginal("_defaultSettingsCCPCUKL1",ukl1Name,rc);
  // Configure those settings that cannot be loaded from the defaults.
  // The common settings are for example the name of the UKL1 boards to communicate with.
      if (!fwHw_setCommonSettings(ukl1Name, makeDynString(ukl1Name), TRUE))
        exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_addUkl1Boards(): Failed to set the CCPC name for " + ukl1Name + ", this will prevent correct communication with this UKL1.", "1");
  // Initialise the FSM state.
      if (0!=dpSet(ukl1Name+".status", "NOT_READY"))
        exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_addUkl1Boards(): Failed to initialise the UKL1 FSM state to NOT_READY for, problems accessing the DP were encountered. Board will be in a UNKNOWN state when FSM is started.", "1");
  // Ensure the DIM services are setup. If they are it will be ignored otherwise they will be set up.
  // Must be done last to ensure that we have configured the HW DPs. Only do it if they are being created.
      if (!fwHw_subscribe(ukl1Name))
        exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_addUkl1Boards(): Failed to subscribe " + ukl1Name + " to the DIM server, communication with this UKL1 board will not be possible.", "1");
    }
    if (showProgress)
      fwShowProgressBar(progress);
    else
      DebugTN(progress);
  }// for each board.

// Add the boards as device units in the FSM tree.
  
  DebugTN("fwUkl1_addUkl1Boards: Add done. Now creating DU tree");
  _fwUkl1_createUkl1FsmTreeDU(toBeAddedUkl1List, forceReconfigure, showProgress, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_addUkl1Boards(): Failed to add the boards to the FSM tree list.", "2", exInfoSize);
  
  // Done.
  return;
}// fwUkl1_addUkl1Boards()

/*!
 * Deletes a list of UKL1 boards from the system. It removes the hardware datapoint, unsubscribing it from the DIM server;
 * and removes it from the FSM tree.
 *
 * @param  toBeDeletedUkl1List A dyn_string of names of the UKL1 boards as they appear in the DIM server that need to
 *           be removed from the system.
 * @param  showProgress If TRUE then the fwShowProgressBar() will be called indicating the progress of the function.
 *           If FALSE it will not be called, this is useful when running from a PVSS00ctrl or similar manager which do not have
 *           access to graphics functions.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 *
 * This function reports its progress via fwShowProgressBar() and thus fwOpenProgressBar() can be called to monitor its progress.
 * It does not submit a percentage completion however and can only be used with types 1 and 3 with fwOpenProgressBar().
 */
void fwUkl1_deleteUkl1Boards(dyn_string toBeDeletedUkl1List, bool showProgress, dyn_string& exceptionInfo) {
  // Keep track of the size of the exceptionInfo.
  int exInfoSize = dynlen(exceptionInfo);
  // Store the number of elements in the list.
  const unsigned numToBeDeleted = dynlen(toBeDeletedUkl1List);
  // If the list was empty return!
  if ( 0 == numToBeDeleted )
    return;

  // Loop over each board removing them from the system.
  for (unsigned board=1; board<=numToBeDeleted; ++board) {
    const string ukl1Name = toBeDeletedUkl1List[board];

    // Remove the hardware type datapoint and unsubscribes it from the DIM server.
    // Only remove it if it is not the FSMConfDB object.
    if (!patternMatch("*_ConfDB", ukl1Name) && ("HPD_Disabling"!=ukl1Name) && !patternMatch("*hugin*", ukl1Name)) {
      bool bStatus = fwHw_remove(FWUKL1_HW_TYPE, ukl1Name);
      if (!bStatus)
        exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_deleteUkl1Board(): Failed to remove " + ukl1Name + "'s DIM server subscriptions.", "2");
      if (showProgress)
        fwShowProgressBar("Removed HW DP for " + ukl1Name + ".");
    }// if UKL1 is present in list.
    else
      if (showProgress)
        fwShowProgressBar("Did not unsubscribe the HW DP for " + ukl1Name + " as we believe it is the FSMConfDB object and hence has no subscriptions to unsubscribe.");
  }// for each board.

  // Remove the list of boards to the FSM tree.
  _fwUkl1_deleteUkl1FsmTreeDU(toBeDeletedUkl1List, showProgress, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_deleteUkl1Boards(): Failed to delete the boards from the FSM tree list.", "2", exInfoSize);
  
  // Done.
  return;
}// fwUkl1_deleteUkl1Boards()

/*!
 * Returns the full list of names of UKL1 boards that are present in both the FSM tree and as a HWTypeCCPCUKL1 data point
 * and hence are fully accessible in the system.
 *
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @param  sysName The system name required by fwHw_get
 * @return dyn_string List of all the names of the UKL1 board that has been retrieved.
 */
dyn_string fwUkl1_getUkl1NamesList(dyn_string& exceptionInfo, string sysName = "") {
  // This will hold the list to that is returned.
  dyn_string namesList;

  // Get the list of HW DPs in the system.
  dyn_string hwList;
  if (fwHw_get(FWUKL1_HW_TYPE, hwList, sysName)) {
    // Now we have the hardware list check to see if it is a node. We don't care if it is local or remote, both are >0 so just check that.
    const int numHw = dynlen(hwList);
    // This keeps track of the index in the names list that we wish to add elements for.
    int nameIndex = 1;
    for (int hw=1; hw<=numHw; ++hw) {
      if (0<fwFsmTree_isNode(hwList[hw])) {
        // Add it to the list.
        namesList[nameIndex] = hwList[hw];
        ++nameIndex;
      }// if isNode.
      // No need for else we don't keep track here of incomplete names.
    }// for hw.
  }// if(fwHw_get())
  else
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_getUkl1NamesList(): Failed to retrieve list of UKL1 hardware datapoints in the system, assuming there are none.", "2");

  // Put what ever was retrieved in ascending order to make it look nice.
  dynSortAsc(namesList);
  // Done.
  return namesList;
}// fwUkl1_getUkl1NamesList()

/*!
 * Returns number of UKL1 board names that are currently in the system.
 *
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @param  sysName The system name.
 * @return unsigned Number of UKL1 boards that are stored in the list, -1 in the event of an error.
 */
int fwUkl1_getNamesListSize(dyn_string& exceptionInfo, string sysName = "") {
  int exInfoSize = dynlen(exceptionInfo);
  const int size = dynlen(fwUkl1_getUkl1NamesList(exceptionInfo, sysName));
  if (-1!=size) {
    fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_getNamesListSize(): Error occurred while retrieving names list.", "1", exInfoSize);
    // Set the size to -1 to indicate an error.
    size = -1;
  }// if(-1!=size)
  else {
    // Indicate PVSS error.
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getNamesListSize(): PVSS encountered an error while determining the size of the list.", "1");
    // Size is -1 so don't worry about it.
  }// else(-1!=size)
  // Done.
  return size;   
}// fwUkl1_getNamesListSize()

/*!
 * Returns the names of the UKL1 board names that are current stored as hardware data points but are not present in the FSM tree.
 *
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @param  sysName The system name required by fwHw_get.
 * @return dyn_string List of all the names of the UKL1 board that are present only as hardware data points. Empty if there are none.
 */
dyn_string fwUkl1_getUkl1HwDpNamesOnlyList(dyn_string& exceptionInfo, string sysName = "") {
  // This holds the list that is to be returned.
  dyn_string namesList;

  // Get the list of HW DPs in the system.
  dyn_string hwList;
  if (fwHw_get(FWUKL1_HW_TYPE, hwList, sysName)) {
    // Now we have the hardware list check to see if it is a node. 0 indicates that it is not.
    const int numHw = dynlen(hwList);
    // This keeps track of the index in the names list that we wish to add elements for.
    int nameIndex = 1;
    for (int hw = 1; hw <= numHw; ++hw) {
      if (0==fwFsmTree_isNode(hwList[hw])) {
        // Add it to the list.
        namesList[nameIndex] = hwList[hw];
        ++nameIndex;
      }// if is not node.
      // No need for else we don't keep track here of complete names.
    }// for hw.
  }// if(fwHw_get())
  else
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_getUkl1HwDpNamesOnlyList(): Failed to retrieve list of UKL1 hardware datapoints in the system, assuming there are none.", "2");

  // Hopefully this is empty.
  return namesList;
}// fwUkl1_getUkl1HwDpNamesOnlyList()

/*!
 * Returns the names of the UKL1 board names that are current present in the FSM tree, but are not stored as hardware data points.
 *
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @param  sysName The system name required by fwHw_get.
 * @return dyn_string List of all the names of the UKL1 board that are present only as hardware data points. Empty if there are none.
 */
dyn_string fwUkl1_getUkl1FsmTreeNamesOnlyList(dyn_string& exceptionInfo, string sysName = "") {
  // This holds the list that is to be returned.
  dyn_string namesList;

  // Get the list of children in the FSM tree node.
  // The function also returns by reference the type of each child, CU, LU or DU. We don't care, should only ever be a DU.
  dyn_int junkChildrenTypes;
  dyn_string fsmList = fwCU_getChildren(junkChildrenTypes, fwUkl1_getFSMCUName() );
  // There is a special FSM child called `fwUk1_getFSMCUName()'_confDB, which should not be returned.
  // It will never exist in the hardware, but should always exist in the FSM tree.
  int confDbIndex = dynContains(fsmList, (fwUkl1_getFSMCUName() + "_ConfDB"));
  dynRemove(fsmList, confDbIndex);
  // Now get the list of HW DPs for comparison with.
  dyn_string hwList;
  if (fwHw_get(FWUKL1_HW_TYPE, hwList, sysName)) {
    // How many FSM children are there? Required to know if we are to loop over them.
    const int numChildren = dynlen(fsmList);
    // This keeps track of the index in the names list that we wish to add elements for.
    int nameIndex = 1;
    for (int child=1; child<=numChildren; ++child) {
      if (0==dynContains(hwList, fsmList[child])) {
        // The FSM child is not present as a harware DP.
        namesList[nameIndex] = fsmList[child];
        ++nameIndex;
      }// if is not in the hardware list.
      // No need for else we don't keep track here of complete names.
    }// for children.
  }// if(fwHw_get())
  else
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getUkl1FsmTreeNamesOnlyList(): Failed to retrieve list of UKL1 hardware datapoints in the system, assuming there are none.", "2");

  // Hopefully this is empty.
  return namesList;
}// fwUkl1_getUkl1FsmTreeNamesOnlyList()

// ============
//   INTERNAL
// ============

/*!
 * This is used to add list of UKL1 boards to the appropriate FSM tree. It will add the UKL1 boards as device units (DUs) under
 * the control unit (CU) defined by the FwUkl1Constants DP.
 *
 * @param  ukl1List Names of UKL1 boards to be added to the FSM tree. Should be as they appear in the DIM server.
 * @param  forceReconfigure If true then it will force the settings to be updated for all of the DU's, the CU and the FSMConfDB
 *           whether they already exist or not. It will not necessarily restart the FSM tree if no boards are added.
 * @param  showProgress If TRUE then the fwShowProgressBar() will be called indicating the progress of the function.
 *           If FALSE it will not be called, this is useful when running from a PVSS00ctrl or similar manager which do not have
 *           access to graphics functions.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 *
 * The DUs added to the tree will appear as their DIM server name, with the panel fwUkl1DUOverview.pnl added for
 * User interaction with the hardware. The label for the DUs will be the name of the UKL1 board as it appears in the DIM server.\n
 * The FSM CU node may have to be stopped for this function to complete and the CU node will be restarted once the
 * operation is complete. No other nodes in the tree will be affected.\n
 * The behaviour of this function is outlined below below:
 *   \li If the DU does not exist it will be created. The existing DU's also the CU and FSMConfDB object will be
 *         left alone. If a DU is added the FSM tree node will have been modified and requires it to be stopped and restarted.
 *   \li If forceReconfigure is TRUE the settings for the DUs will be updated as will those for the CU and
 *         FSMConfDB object. This does not require a restart of the FSM tree, only the panels to be reopened. It will also add
 *         any boards that are not already present in the FSM tree.
 *
 * Note that if the CU (as defined by FwUkl1Constants) does not exist then it will be created and the function will proceed as for creation.
 * The function will also refresh the FSM tree list in the device editor and navigator (DEN) and also will display the CU panel which
 * will refresh it if the list changed.
 */
void _fwUkl1_createUkl1FsmTreeDU(dyn_string ukl1List, bool forceReconfigure, bool showProgress, dyn_string& exceptionInfo) {
  string progress;
  // Don't do anything if the list is empty and we are not updating the existing boards.
  if (!forceReconfigure && (0 == dynlen(ukl1List)))
    return;
  progress = "Configuring FSM tree.";
  if (showProgress)
    fwShowProgressBar(progress);
  else
    DebugTN(progress);
  // Get the name of the FSM CU, exit if it is NULL.
  const string fsmCu = fwUkl1_getFSMCUName();
  if (""==fsmCu) {
    // If the FSM CU is a NULL string then we should not try and add it to the tree whether it exists or not!
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_createUkl1FsmTreeDU(): The name of the FSM CU stored in the UKL1 constants datapoint is empty. The tree has not been created. Please use the Constants configuration panel in the UKL1 administration panel to set the appropriate FSM CU name and try again.", "1");
    return;
  }// if(""!=fsmCu)
  
  // Keep track of the exception information.
  int exInfoSize = dynlen(exceptionInfo);
  // It is possible to call this function and for it to have no effect on the FSM tree node. In this case there is
  // no point in stopping, regenerating and restarting the FSM tree node. This variable is set to true if a restart
  // is required.
  bool restart = FALSE;
  // This flag is used to determine if we should create the DU or not.
  bool createDU = FALSE;
  // This is used to mark if we created a DU so we know we have to update the settings.
  bool addedDU = FALSE;
  // The CU settings will need to be updated if forceReconfigure is set or we add the CU. Determine the latter later.
  bool updateCU = forceReconfigure ? TRUE:FALSE;
  // This flag is used to signal if there is no CU.
  bool noCU = FALSE;
  // Determine if the FSM CU exists.
  const int nodeExists = fwFsmTree_isNode(fwUkl1_getFSMCUName());
  // If the node does not exist then we must create it.
  if (0==nodeExists) {
    // Stop the FSM tree to do this. Can't have been stopped by this stage.
    fwFsmTree_stopTree();
    // Tree was stopped will need to regenerate and restart.
    restart = TRUE;
    // Create the node as a control unit (last arg=1) and put it under the root node `FSM'.
    const string createdNode = fwFsmTree_addNode("FSM", fsmCu, "DAQ_Domain_v1", 1);
    if (""!=createdNode)
      updateCU = TRUE;
    else {
      // Tell everyone we have no CU.
      noCU = TRUE;
      // Failed to create the node, update the status value and tell everyone.
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_createUkl1FsmTreeDU(): Failed to create the control unit \'" + fwUkl1_getFSMCUName() + "\' in the FSM state machine.", "1");
    }// else(""!=createNode)
  }// if(0==nodeExists)

  // The next section can be done regardless of whether the node exist section was called or not,
  // but if it failed then we don't have a CU FSM tree node to add the DUs to so check the status is still good.
  if (!noCU) {
    // Loop over all the names given in the ukl1List and determine what is the appropriate behaviour for each.
    const int numUkl1s = dynlen(ukl1List);
    for (int ukl1 = 1; ukl1 <= numUkl1s; ++ukl1) {
      // Get the name of the UKL1 board as it will appear in the FSM.
      const string ukl1FsmName = ukl1List[ukl1];
      // Check whether the DU exists.
      const int duExists = fwFsmTree_isNode(ukl1FsmName);
      // If the node does not exist then we must create it.
      if (0 == duExists)
        createDU = TRUE;
      // If we need to create the DU then we best had.
      if (createDU) {
        // We need to be stopped by here if we are not already.
        if (!restart) {
          // Must always stop everything.
          fwFsmTree_stopTree();
          // We are going to have to restart now.
          restart = TRUE;
        }// if(!restart)
        // Add the board to the tree as a HwTypeCCPCUKL1 type and as a device unit (last arg=0).
        fwFsmTree_addNode(fwUkl1_getFSMCUName(), ukl1FsmName, "HwTypeCCPCUKL1", 0);
        // Check to see if the node was added.
        const int duPresent = fwFsmTree_isNode(ukl1FsmName);
        if (0!=duPresent)
          addedDU = TRUE;
        else
          exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_createUkl1FsmTreeDU(): Failed to add " + ukl1FsmName + " to the FSM tree. Communication with this board will not be possible via FSM.", "2");
      }// if(createDu)
      
      // Update the settings for this DU if we added it or if we are forcing a reconfigure.
      if (forceReconfigure || addedDU) {
        fwFsmTree_setNodeLabel(ukl1FsmName, ukl1FsmName);
        fwFsmTree_setNodePanel(ukl1FsmName, "objects/fwUkl1/fwUkl1DUOverview.pnl");
        progress = "Added " + ukl1FsmName + " to the FSM.";
        if (showProgress)
          fwShowProgressBar(progress);
        else
          DebugTN(progress);
      }// if(forceReconfigure || addedDU)
    }// for(numUkl1s)

    // Harry :: Add the HPD disabling DU    
    _fwUkl1_addHPDdisablingDU(restart, forceReconfigure, showProgress);
    
    // This function will add the configurator object if it does not already exist, and will only update its settings if it does.
    // Do this after creating all the UKL1s incase it adds it to the middle of an existing list. Generally it is going to find its
    // way to the top of all lists.
    DebugTN("Creating Ukl1 configurator");
    _fwUkl1_createUkl1FsmConfiguratorObject(forceReconfigure, exceptionInfo);
    fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_createUkl1FsmTreeDU(): Failed to add the FSM configurator object the FSM tree, this will prevent interaction with the configurationDB and recipes may not be loaded correctly.", "2", exInfoSize);
    
    // Check to see if we need to update the CU settings.
    if (updateCU) {
      DebugTN("Updating CU settings");
      fwFsmTree_setNodeLabel(fwUkl1_getFSMCUName(), fwUkl1_getFSMCUName());
      fwFsmTree_setNodePanel(fwUkl1_getFSMCUName(), "fwUkl1/fwUkl1CUOverview.pnl");
    }// if(updateCU)
  }// if(!noCU) Don't bother with an else we already have the appropriate status and will have signaled the error.

  // We should only regenerate and restart the FSM tree node if it has actually been stopped and modified.
  if (restart) {
    DebugTN("Generate CU FSM");
    // Generate the FSM for this CU only.
    fwFsmTree_generateTreeNode( fwUkl1_getFSMCUName() );
    // Start FSM processes for the CU sub-tree
    fwFsmTree_startTree();
  }// if(restart)

  DebugTN("Refresh tree");
  // Refresh the Tree in case the DEN is running, this can be done regardless. It ensures that the tree display is correct.
  fwFsmTree_refreshTree();
  // Don't try and view the CU panel as if it is run in a script in the console it has no concept of a UI and ths can't start it.

  // Now we are done.
  return;
}// _fwUkl1_createUkl1FsmTreeDU()

/*!
 * This is used to add the HPD_Disabling DU to the FSM tree.
 * @param  Restart Boolean indicating whether a restart is required. If TRUE then a restart is required.
 * @param  ForceReconfigure If true then it will force the settings to be updated for all of the DU's, the CU and the FSMConfDB
 *                         whether they already exist or not. It will not necessarily restart the FSM tree if no boards are added.
 * @param  ShowProgress If TRUE then the fwShowProgressBar() will be called indicating the progress of the function.
 *                      If FALSE it will not be called, this is useful when running from a PVSS00ctrl or similar manager which do not have
 *                      access to graphics functions.
 * @return void.
 */

void _fwUkl1_addHPDdisablingDU(bool &Restart, bool ForceReconfigure, bool ShowProgress) {
// String to store progress
  string progress;
  // Boolean to see if a DU was added  
  bool AddedDU = FALSE;
  // Check whether the DU exists.
  const int duExists = fwFsmTree_isNode("HPD_Disabling");
  // If the node does not exist then we must create it.
  // Tells us if we should create the DU
  bool CreateDU;
  if (0 == duExists)
       CreateDU = TRUE;
  // If we need to create the DU then we best had.
  if (CreateDU) {
    // We need to be stopped by here if we are not already.
    if (!Restart) {
      // Must always stop everything.
      fwFsmTree_stopTree();
      // We are going to have to restart now.
      Restart = TRUE;
    }// if(!Restart)
    // Add the board to the tree as a FwUkl1HPDDisabling type and as a device unit (last arg=0).
    fwFsmTree_addNode(fwUkl1_getFSMCUName(), "HPD_Disabling", "FwUkl1HPDDisabling", 0);
    // Check to see if the node was added.
    const int dupresent = fwFsmTree_isNode("HPD_Disabling");
    if (0!=dupresent) 
      AddedDU = TRUE;
    else
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_addHPDdisablingDU(): Failed to add HPDDisabling to the FSM tree.", "2");
  }// if(CreateDu)
   // Update the settings for this DU if we added it or if we are forcing a reconfigure.
      if (ForceReconfigure || AddedDU) {
        fwFsmTree_setNodeLabel("HPD_Disabling", "HPD_Disabling");
        fwFsmTree_setNodePanel("HPD_Disabling", "objects/fwUkl1/fwUkl1HPDDisablingDUOverview.pnl");
        progress = "Added HPD_Disabling to the FSM.";
        if (ShowProgress)
          fwShowProgressBar(progress);
        else
          DebugTN(progress);
      }// if(ForceReconfigure || AddedDU) 
    return;
}// _fwUkl1_addHPDdisablingDU()

/*!
 * This is used to delete a list UKL1 boards from the appropriate FSM tree. It will look in the tree fwUkl1_getFSMCUName()
 *
 * @param  ukl1List List of names of UKL1 boards to be deleted from the FSM tree. Names should be as they appear in the DIM server.
 * @param  showProgress If TRUE then the fwShowProgressBar() will be called indicating the progress of the function.
 *           If FALSE it will not be called, this is useful when running from a PVSS00ctrl or similar manager which do not have
 *           access to graphics functions.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 */
void _fwUkl1_deleteUkl1FsmTreeDU(dyn_string ukl1List, bool showProgress, dyn_string& exceptionInfo) {
  if (0 == dynlen(ukl1List))
    return;
  
  int exInfoSize = dynlen(exceptionInfo);
  // First we must determine if the fwUkl1_getFSMCUName() node exists in the FSM tree.
  const int nodeExists = fwFsmTree_isNode(fwUkl1_getFSMCUName());
  // Does the node we wish to delete from exist?
  if (0!=nodeExists) {
    // The node exists, either locally or remotely.
    // Stop it so we can edit it, accept the default of stopping all children.
    fwFsmTree_stopTree();
    // Loop over all the names given in the ukl1List and determine what is the appropriate behaviour for each.
    const int numUkl1s = dynlen(ukl1List);
    const string fsmConfDBName = fwUkl1_getFSMCUName() + "_ConfDB";
    const string huginName     = fwUkl1_getHuginName();
    for (int ukl1 = 1; ukl1 <= numUkl1s; ++ukl1) {
      // Convert the ukl1Name into the name as it will appear in the FSM.
      const string ukl1FsmName = ukl1List[ukl1];
      // Check whether the DU exists.
      const int duExists = fwFsmTree_isNode(ukl1FsmName);
      if (0!=duExists) {
        // Check to see if we are dealing with the FSMConfDB object, hugin or a normal board.
        switch (ukl1FsmName) {
          case fsmConfDBName:
            if (-1 == fwFSMConfDB_removeConfigurator(fwUkl1_getFSMCUName(), fwUkl1_getFSMCUName() + "_ConfBD"))
              exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_deleteUkl1FsmTreeDU(): Failed to delete " + ukl1FsmName + " from the FSM tree.", "2");
            break;
          
          case huginName:
            // Do nothing here, leave the Hugin alone as it has to be added manually.
            // The Hugin is an object in another system and my knowledge of the FSM
            // libraries is not sufficient to retrieve and add the board.
            break;
            
          case "HPD_Disabling":           
            // Delete the HPD disabling DU
            fwFsmTree_removeNode(fwUkl1_getFSMCUName(), "HPD_Disabling");           
            // Check the DU was actually deleted.
            if (0==fwFsmTree_isNode("HPD_Disabling"))
              if (showProgress)
                fwShowProgressBar("Removed HPD_Disabling from FSM.");
            else
              exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_deleteUkl1FsmTreeDU(): Failed to delete HPD_Disabling from the FSM tree.", "2");
            break;
            
          default:
            // The default case is a UKL1 board.
            // Remove the DU from the tree, accept the default of removing a node recursively from the tree as the DU shouldn't have any children!
            fwFsmTree_removeNode(fwUkl1_getFSMCUName(), ukl1FsmName);
            // Check the DU was actually deleted.
            if (0==fwFsmTree_isNode(ukl1FsmName))
              if (showProgress)
                fwShowProgressBar("Removed " + ukl1FsmName + " from FSM.");
            else
              exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_deleteUkl1FsmTreeDU(): Failed to delete " + ukl1FsmName + " from the FSM tree. There is no physical board in the crate, but it will still appear in the FSM tree.", "2");
            break;
        }// switch(ukl1FSMName)
      }// if(0==duExists)
      else
        if (showProgress)
          fwShowProgressBar(ukl1FsmName + " is not present in the FSM tree to remove.");
    }// for(numUkl1s)

    // Only any point of doing this if the node exists.
    // Generate the FSM for CU.
    fwFsmTree_generateTreeNode(fwUkl1_getFSMCUName());
    // Start FSM processes for the CU sub-tree. We could have deleted only a couple of boards.
    fwFsmTree_startTreeNode(fwUkl1_getFSMCUName());
    // Refresh the Tree in case the DEN is running, this can be done regardless.
    fwFsmTree_refreshTree();
    // Don't try and reopen the CU, we might not be able to achieve that if we are running in a console script.
  }// if(0!=nodeExists)
  else {
    // The node does not exist so we cannot delete any boards. Warn people, it will happen on a fresh install though.
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_deleteUkl1FsmTreeDU(): UKL1 control unit `" + fwUkl1_getFSMCUName() + "' not found. UKL1 FSM tree does not exist, cannot remove UKL1 boards from a none existant FSM tree.", "1");
  }// else(0!=nodeExists)

  // Done.
  return;
}// _fwUkl1_deleteUkl1FsmTreeDU()

/*!
 * This will create and configure an FSM configurationDB configurator object for the UKL1 control unit in the FSM tree.
 *
 * @param  forceReconfigure If this is TRUE then the setting for the configurator object will be updated even if it already
 *           exists. If it is FALSE then if the configurator exists then no changes will be made to it.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void
 *
 * If the object already exists then this function will only update the settings for the object. There is no single function
 * provided by the framework to do this and as such some code is taken straight from the FSMConfDB configuration panel.
 */
void _fwUkl1_createUkl1FsmConfiguratorObject(bool forceReconfigure, dyn_string &exceptionInfo) {
  // Used to keep track of errors.
  int exInfoSize = dynlen(exceptionInfo);
  // This is the name of the DP that will hold the confDB object.
  const string dpName = fwFSMConfDB_getConfiguratorName(fwUkl1_getFSMCUName(), "_ConfDB");
  // This is used to indicate if the configurator exists.
  bool fsmConfDBExists = TRUE;
  // This is used to indicate if it has been created.
  bool fsmConfDBCreated = FALSE;
  
  // Lets check to see if the object already exists, if it doesn't then try and create it.
  if (!dpExists(dpName)) {
    // Call the function to add the configurator object. The domain is just the name of the CU.
    // If it is added potentially over ride the forceReconfigure value to ensure the settings are updated.
    if(0==fwFSMConfDB_addConfigurator(fwUkl1_getFSMCUName(), FWUKL1_FSMCONFDB_TYPE)) {
      fsmConfDBCreated = TRUE;
    }// if added configurator
    else {
      fsmConfDBExists = FALSE;
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_createUkl1FsmConfiguratorObject(): Failed to add the configurator object to the UKL1 FSM tree. Call fwFSMConfDB_addConfigurator returned -1.", "1");
    }// else failed to add configurator
  }// if(!dpExists())
  // The exists flag is already set to TRUE and created to FALSE.
  // If there was problem making it exist then it will have been flagged.

  // If the FSMConfDB object exists then we should configure it.
  if (fsmConfDBExists && (fsmConfDBCreated || forceReconfigure)) {
    // These are the recipes that are used for start and stop. They are entered as user recipes, so they
    // are not associated with any mode.
    const dyn_string userRecipes = makeDynString("Start", "Stop");
    // Now update the settings for the configurator object.
    if (0 != dpSet(dpName + ".applyRecipes.usingConfigurator", FWUKL1_CONFIGURATOR_APPLY_RECIPES,
                    dpName + ".applyRecipes.timeout", FWUKL1_CONFIGURATOR_TIMEOUT,
                    dpName + ".applyRecipes.simplifiedAR", FWUKL1_CONFIGURATOR_SIMPLIFIED_AR,
                    dpName + ".useConfDB", fwUkl1_getConfiguratorUseConfDBFlag(),
                    dpName + ".userRecipes", userRecipes)) {
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_createUkl1FsmConfiguratorObject(): Failed to update the configuration settings for the FSMConfDB object. It may not be properly configured, please check these manually. dpSet failed.", "2");
    }// if dpSet failed.
  }// if(fsmConfDBExists)

  // Done.
  return;
}// _fwUkl1_createUkl1FsmConfiguratorObject()

/*!
 * This create if necessary the abstract register data point, which contains the information about how to convert
 * the abstract register displayed in the panel into the hardware registers written with DIM to the UKL1s.
 *
 * @param  forceReconfigure If TRUE then the data point elements will be updated from the library settings,
 *           if FALSE then if the data point elements will not be updated.
 * @param  showProgress If TRUE then the fwShowProgressBar() will be called indicating the progress of the function.
 *           If FALSE it will not be called, this is useful when running from a PVSS00ctrl or similar manager which do not have
 *           access to graphics functions.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @param  sys System Name.
 * @return void.
 *
 * In the event that the DP does not exist then it will always be created regardless of the forceReconfigure.
 */
void _fwUkl1_updateAbstractRegisterDP(bool forceReconfigure, bool showProgress, dyn_string &exceptionInfo, string sys = "") {
  int exInfoSize = dynlen(exceptionInfo);
  // Done.
  return;
}// _fwUkl1_updateAbstractRegisterDP()

/*!
 * This sets the DPE for the specific register in the UKL1 abstract registers' interface DP.
 *
 * @param  regName The name of the register to update.
 * @param  regValues A map containing the list of registers to set. The key is the DPE name (all are defined as contants
 *           in this library) and it corresponds to the value to be set. Only those values given as a key will
 *           be set. No notification of those elements not present is given.
 * @param  forceReconfigure If TRUE it will first delete the abstract register DP before recreating it.
 *           If the delete fails then the values will be updated anyway.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 */
void _fwUkl1_updateAbstractRegisterDPE(const string &absRegName, mapping &regValues, bool forceReconfigure, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  // Try to create the register. If it already exists nothing will happen.
  fwUkl1_createAbstractRegisterDP(absRegName, exceptionInfo);
  if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_updateAbstractRegisterDPE(): Failed to create the abstract register '"+absRegName+"'.", "2", exInfoSize)) {
    // Get the list of keys to write.
    const dyn_string keys = mappingKeys(regValues);
    // Now loop over them.
    const int numKeys = dynlen(keys);
    for (int keyNum=1; keyNum<=numKeys; ++keyNum) {
      // The key.
      const string key = keys[keyNum];
      // Now determine the element.
      const string dpeName = absRegName + key;
      // Now determine the element.
      // const string dpeName = FWUKL1_UKL1_BOARD_DP_NAME + "." + regName + key;
      // Check it exists.
      if (dpExists(dpeName)) {
        // Set it.
        if (0!=dpSet(dpeName, regValues[key]))
          exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_updateAbstractRegisterDPE(): Failed to update/set the abstract register " + absRegName + " setting " + key + ". Attempted to write to the DPE " + dpeName + ". The settings for this abstract register will either be incomplete, out of data or at worse missing. This could affect the ability to write to this register.", "2");
      }// if (dpExists(dpeName))
      else
        exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_updateAbstractRegisterDPE(): Failed to find the DPE " + dpeName + ". Could not update the settings " + key + " for the register " + absRegName + ". Please check that this register should exists.", "2");
    }// for keyNum
  }// if fwUkl1_createAbstractRegisterDP failed.
  // Done.
  return;
}// _fwUkl1_updateAbstractRegisterDPE()


// @}


// ==========================
//  UKL1 CONSTANTS INTERFACE
// ==========================

/*! @defgroup ConstantsInterface UKL1 constants interface
 *     These functions are all used to set or get the DPE of the UKL1 constants datapoint.
 *
 *     The constants datapoint contains the values that are RICH specific and are used at run time to configure
 *     the project appropriately.
 *  @{
 */

/*!
 * Retrieves the name that is to be used for the UKL1 FSM CU.
 *
 * @param  sysName This is the name of the system that the FwUkl1Constant DP has been stored on. It must contain the trailing
 *           semi-colon e.g. R2DAQL1: or it can be an empty string if the system name is not known or it is running locally.
 *           Defaults to an empty string.
 * @return string The name that is to be used for the UKL1 FSM CU. Returns an empty string in the event of an error.
 *
 * Errors are reported directly to the UKL1 log.
 */
string fwUkl1_getFSMCUName(string sysName="") {
  // Holds the exception information.
  dyn_string exceptionInfo;
  int exInfoSize = 0;
  // Holds the name of the CU.
  string cu = fwUkl1_getUkl1Constant(FWUKL1_CU, sysName, exceptionInfo);
  if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_getFSMCUName(): Failed to retrieve the FSM CU name. Returning an emtpy string for the name.", "2", exInfoSize)) {
    fwUkl1ExceptionLog_submit(exceptionInfo, FWUKL1_CONSTS_DP_BASENAME);
    cu = "";
  }// if error.
  // Done.
  return cu;
}// fwUkl1_getFSMCUName()

/*!
 * Retrieves the flag that indicates if the recipes are to be loaded from the database or the recipe cache.
 *
 * @param  sysName This is the name of the system that the FwUkl1Constant DP has been stored on. It must contain the trailing
 *           semi-colon e.g. R2DAQL1: or it can be an empty string if the system name is not known or it is running locally.
 *           Defaults to an empty string.
 * @return bool Value of the flag. FALSE is returned in the event of an error as the recipe cache should always be present.
 *
 * Errors are reported directly to the UKL1 log.
 */
string fwUkl1_getConfiguratorUseConfDBFlag(string sysName="") {
  // Holds the exception information.
  dyn_string exceptionInfo;
  // Holds the value of the flag.
  bool flag = fwUkl1_getUkl1Constant(FWUKL1_CONFIGURATOR_USE_CONFDB, sysName, exceptionInfo);
  int exInfoSize = 0;
  if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_getConfiguratorUseConfDBFlag(): Failed to retrieve the value of the flag that determines if the FSM configurator object should retrieve recipes from the cache or DB. Setting to use from the cache.", "2", exInfoSize)) {
    fwUkl1ExceptionLog_submit(exceptionInfo, FWUKL1_CONSTS_DP_BASENAME);
    flag = FALSE;
  }// if error.
  // Done.
  return flag;
}// fwUkl1_getFSMCUName()

/*!
 * Retrieves the name that is to be used for the Hugin in the FSM tree.
 *
 * @param  sysName This is the name of the system that the FwUkl1Constant DP has been stored on. It must contain the trailing
 *           semi-colon e.g. R2DAQL1: or it can be an empty string if the system name is not known or it is running locally.
 *           Defaults to an empty string.
 * @return string The name that is to be used for the Hugin. Returns an empty string in the event of an error.
 *
 * Errors are reported directly to the UKL1 log.
 */
string fwUkl1_getHuginName(string sysName="") {
  // Holds the exception information.
  dyn_string exceptionInfo;
  int exInfoSize = 0;
  // Holds the name of the Hugin.
  string hugin = fwUkl1_getUkl1Constant(FWUKL1_HUGIN_NAME, sysName, exceptionInfo);
  if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_getHuginName(): Failed to retrieve the name of the Hugin that is to be used in the FSM tree. Returning an emtpy string for the name.", "2", exInfoSize)) {
    fwUkl1ExceptionLog_submit(exceptionInfo, FWUKL1_CONSTS_DP_BASENAME);
    hugin = "";
  }// if error.
  // Done.
  return hugin;
}// fwUkl1_getHuginName()

/*!
 * Retrieves the flag that indicates if the UKL1 channels are to be disabled or not when requested to do so by the HPDUKL1DisableTool.
 *
 * @param  sysName This is the name of the system that the FwUkl1Constant DP has been stored on. It must contain the trailing
 *           semi-colon e.g. R2DAQL1: or it can be an empty string if the system name is not known or it is running locally.
 *           Defaults to an empty string.
 * @return bool Value of the flag. FALSE is returned in the event of an error as this currently is considered to be the default behaviour.
 *
 * Errors are reported directly to the UKL1 log.
 */
bool fwUkl1_getDisableUkl1ChannelsFlag(string sysName = "") {
  // Holds the exception information.
  dyn_string exceptionInfo;
  // Holds the value of the flag.
  bool flag = fwUkl1_getUkl1Constant(FWUKL1_DISABLE_UKL1_CHANNEL, sysName, exceptionInfo);
  int exInfoSize = 0;
  if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_getDisableUkl1ChannelsFlag(): Failed to retrieve the value of the flag that determines if the UKL1 channels should be disabled when requested to do via the UKL1 HPD disable tool. Setting to not disable the channels.", "2", exInfoSize)) {
    fwUkl1ExceptionLog_submit(exceptionInfo, FWUKL1_CONSTS_DP_BASENAME);
    flag = FALSE;
  }// if error.
  // Done.
  return flag;
}// fwUkl1_getDisableUkl1ChannelsFlag()



/*!
 * This sets a specific constant in the FwUkl1Constants DP.
 *
 * @param  dpeName This is the name of the DPE element that is to be set. These are all defined as library constants.
 * @param  value This is the value to be set to the constant. It is an anytype parameter so the user must ensure that
 *           the appropriate type is set for the given constant.
 * @param  sysName This is the name of the system that the FwUkl1Constant DP has been stored on. It must contain the trailing
 *           semi-colon e.g. R2DAQL1: or it can be an empty string if the system name is not known or it is running locally.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 *
 * This function should only be used as part of the post install process. It purely sets the value of the constant and does
 * not trigger any of the required side effects that changing the constants entails.
 */
void fwUkl1_setUkl1Constant(const string& dpeName, anytype& value, const string& sysName, dyn_string& exceptionInfo) {
  // Keeps track of the exceptionInfo size.
  int exInfoSize = dynlen(exceptionInfo);
  // Create the full DPE name.
  const string fullName = sysName + FWUKL1_CONSTS_DP_BASENAME + "." + dpeName;
  // Now check it exists.
  if (dpExists(fullName)) {
    // DP exists so set the value.
    if (0!=dpSet(fullName, value)) {
      // Failed to retrieve the value, note this.
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setUkl1Constant(): Failed to set the requested constant \'" + fullName + "\'. dpSet() failed.", "2");
    }// if(0!=dpSet())
  }// if(dpExists(fullName))
  else {
    // If the DPE didn't exist, just check that the base DP exists.
    if (!dpExists(FWUKL1_CONSTS_DP_BASENAME)) {
      // Try and create the constants DP.
      fwUkl1_createUkl1ConstantsDP(sysName, exceptionInfo);
      // If no error call ourselves again.
      if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "FATAL", "fwUkl1_setUkl1Constant(): The UKL1 constants DP does not exists and could not be created. Could not set the DPE \'" + fullName + "\' as the DP does not exist and could not be created.", "1", exInfoSize))
        fwUkl1_setUkl1Constant(dpeName, value, sysName, exceptionInfo);
    }// if(!dpExists())
    else {
      // The base DP did existed so there must be a problem with the DPE.
      // Raise an exception to note that the requested DP element did not exist.
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setUkl1Constant(): Failed to set the requested constant as the DPE \'" + fullName + "\'. Does not exist.", "2");
    }// else(!dpExists())
  }// else(dpExists(fullName))
  
  // Done.
  return;
}// fwUkl1_setUkl1Constant()

/*!
 * This retrieves a specific constant from the FwUk1Constants DP.
 *
 * @param  dpeName This is the name of the DPE element that is to be retrieved. These are all defined as library constants.
 * @param  sysName This is the name of the system that the FwUkl1Constant DP has been stored on. It must contain the trailing
 *           semi-colon e.g. R2DAQL1: or it can be an empty string if the system name is not known or it is running locally.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return anytype The constant that is requested from the DP. It is an any type so must be converted to the appropriate
 *           value for the constant. In the event that an error occurs the value of the return can be anything.
 */
anytype fwUkl1_getUkl1Constant(const string& dpeName, const string& sysName, dyn_string& exceptionInfo) {
  const string fullName = sysName + FWUKL1_CONSTS_DP_BASENAME + "." + dpeName;
  // Holds the value that is to be returned.
  anytype value;
  // Now check it exists.
  if (dpExists(fullName)) {
    // DP exists so retrieve the data.
    if (0!=dpGet(fullName, value)) {
      // Failed to retrieve the value, note this.
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getUkl1Constant(): Failed to retrieve the requested constant as the DPE \'" + fullName + "\'. dpGet() failed.", "2");
    }// if(0!=dpGet())
  }// if(dpExists(fullName))
  else {
    // Raise an exception to note that the requested DP element did not exist.
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getUkl1Constant(): Failed to retrieve the requested constant as the DPE \'" + fullName + "\'. Does not exist.", "2");
  }// else(dpExists(fullName))
  // Done.
  return value;
}// fwUkl1_getUkl1Constant()

/*!
 * Creates the DP that stores the UKL1 constants.
 *
 * @param  sysName This is the name of the system that the FwUkl1Constant DP has been stored on. It must contain the trailing
 *           semi-colon e.g. R2DAQL1: or it can be an empty string if the system name is not known or it is running locally.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 */
void fwUkl1_createUkl1ConstantsDP(const string& sysName, dyn_string& exceptionInfo) {
  // Keeps track of the exceptionInfo size.
  int exInfoSize = dynlen(exceptionInfo);
  const string fullDpName = sysName + FWUKL1_CONSTS_DP_BASENAME;
  if (!dpExists(fullDpName)) {
    // If it doesn't exist then we must create it.
    // Quote PVSS help: "dpCreate() returns 0, in the event of a failure returns -1. 
    //                  The function possibly returns 0 also in the event of a failure.
    //                  In this case errors can only be retrieved with getLastError()!"
    // Nice. Use getLastError() to determine if the DP creation was successful.
    // Don't bother to try and create it on a specific system, it is not necessarily correct
    // when this is called. For example during the init routine where the library won't have
    // been reloaded with the appropriate system number.
    dpCreate(FWUKL1_CONSTS_DP_BASENAME, FWUKL1_CONSTS_DPT, getSystemId(sysName));
    // Doubt it is thread safe...
    dyn_errClass error = getLastError();
    if (0!=dynlen(error)) {
      // Couldn't create the data point.
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "FATAL", "fwUkl1_createUkl1ConstantsDP(): Failed to create the UKL1 constants DP, the UKL1 library and panels will not function correctly. Check the DPT \'" + FWUKL1_CONSTS_DPT + "\' exists.", "1");
    }// if dpCreate() failed
  }// if(!dpExists())
  // No else as we just return if the data point already exists.
  // Done.
  return;
}// fwUkl1_createUkl1ConstantsDP()

// @}

// ==========================
//  UKL1 DISABLE HPD DP
// ==========================

/*! @defgroup DisableHpdDpInterface Interface to UKL1 channel disabling DP.
 *     These functions are all used to create, delete, set or get the DP and its DPEs.
 *
 *  This datapoint is subscribed to the DIM service that publishes the UKL1
 *  channels that are to be disabled to suppress the HPDs.
 *  @{
 */

/*!
 * Creates the DP that is subscribed to the DIM service that publishes UKL1 channels to be disabled.
 *
 * @param  serviceName A string that contains the name of the DIM service that this DP will be subscribed to.
 * @param  exceptionInfo Error information. No change in size if no error information is generated.
 * @return string Name of the DP created.
 *
 * The name of the datapoint is constructed from the service name. Illegal characters are removed.
 */
string fwUkl1_createChannelDisableDP(const string& serviceName, dyn_string& exceptionInfo) {
  // Keeps track of the exceptionInfo size.
  int exInfoSize = dynlen(exceptionInfo);
  // Split the name by /
  const dyn_string splitName = strsplit(serviceName, "/");
  // Take the first part of the name which identifies the node and partition for name of the DP.
  // Should be unique for each service.
  const string fullDpName = splitName[1];
  if (!dpExists(fullDpName)) {
    // If it doesn't exist then we must create it.
    // Quote PVSS help: "dpCreate() returns 0, in the event of a failure returns -1. 
    //                   The function possibly returns 0 also in the event of a failure.
    //                   In this case errors can only be retrieved with getLastError()!"
    // Nice. Use getLastError() to determine if the DP creation was successful.
    // Don't bother to try and create it on a specific system, it is not necessarily correct
    // when this is called. For example during the init routine where the library won't have
    // been reloaded with the appropriate system number.
    dpCreate(fullDpName, FWUKL1_DISABLE_UKL1_CHANNEL_DPT);
    // Doubt it is thread safe...
    dyn_errClass error = getLastError();
    if (0 != dynlen(error)) {
      // Couldn't create the data point.
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_createChannelDisableDP(): Failed to create the UKL1 DP channel disable datapoint for the corresponding service " + serviceName + ". Cannot receive information about which HPDs to disable from this service. Check the DPT \'" + "FwUkl1DisableUkl1Channel" + "\' exists.", "1");
    }// if dpCreate() failed
  }// if(!dpExists())
  // No else as we just return if the data point already exists.
  // Done.
  return fullDpName;
}// fwUkl1_createChannelDisableDP()

/*!
 * Deletes the DP that is subscribed to the DIM service with the channel disable flag.
 *
 * @param  serviceName A string that contains the name of the DIM service that this DP has been unsubscribed from.
 * @param  exceptionInfo Error information. No change in size if no error information is generated.
 * @return string Name of the DP unsubscribed.
 *
 * Should only be done once the service has been unsubscribed.
 * The name of the datapoint is constructed from the service name. Illegal characters are removed.
 */
string fwUkl1_deleteChannelDisableDP(const string& serviceName, dyn_string& exceptionInfo) {
  // Keeps track of the exceptionInfo size.
  int exInfoSize = dynlen(exceptionInfo);
  // Split the name by /
  const dyn_string splitName = strsplit(serviceName, "/");
  // Take the first part of the name which identifies the node that the node and partition for name of the DP.
  // Should be unique for each service.
  const string fullDpName = splitName[1];
  if (dpExists(fullDpName)) {
    // If the DP exists then it must be deleted.
    dpDelete(fullDpName);
  }// if(dpExists())
  // No else as we just return if the data point already exists.
  // Done.
  return;
}// fwUkl1_deleteChannelDisableDP()

//@}

// =========================
//  UKL1 BOARD DP INTERFACE
// =========================

/*! @defgroup Ukl1BoardDpInterface These function provide the interface to the UKL1 board datapoint.
 *     These functions are all used to get the DPE of the UKL1 board datapoint.
 *
 *     The UKL1 board datapoint provides the information on how to deal with the abstract registers
 *     that exist for the UKL1 board. One or more abstract registers can be associated with a single
 *     hardware register. The fwHw library does provide the means to do this also, however DIM communication
 *     is the bottle neck in terms of time in the system so it is better to have a minimum of registers
 *     that DIM is used to communicate with. The abstraction layer is therefore created in the software
 *     above DIM using this datapoint to manage these 'registers'.
 *
 *     The functions used here can only be used to read values from the data point. The configuration
 *     functions are provided as part of the board setup interface and are all internal functions.
 *  @{
 */

/*!
 * Creates the DP that represents an abstract register.
 *
 * @param  abstractRegisterName of the abstract register to create the DP for. If the regsiter already exists
 *            no changes are made.
 * @param  exceptionInfo Error information. No change in size if no error information is generated.
 * @return void.
 */
void fwUkl1_createAbstractRegisterDP(const string& abstractRegisterName, dyn_string& exceptionInfo) {
  // Keeps track of the exceptionInfo size.
  int exInfoSize = dynlen(exceptionInfo);
  if (!dpExists(abstractRegisterName)) {
    // If it doesn't exist then we must create it.
    // Quote PVSS help: "dpCreate() returns 0, in the event of a failure returns -1. 
    //                   The function possibly returns 0 also in the event of a failure.
    //                   In this case errors can only be retrieved with getLastError()!"
    // Nice. Use getLastError() to determine if the DP creation was successful.
    // Don't bother to try and create it on a specific system, it is not necessarily correct
    // when this is called. For example during the init routine where the library won't have
    // been reloaded with the appropriate system number.
    const string sysName = fwUkl1_removeSystemName(abstractRegisterName);
    dpCreate(abstractRegisterName, FWUKL1_ABSTRACT_REGISTER_DPT_NAME);
    // Doubt it is thread safe...
    dyn_errClass error = getLastError();
    if (0 != dynlen(error)) {
      // Couldn't create the data point.
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "FATAL", "_fwUkl1_updateAbstractRegisterDP(): Failed to create the UKL1 abstract register DP, it is not possible to access the UKL1 register "+abstractRegisterName+" or recipes via the panels. The FSM will still work. Check the DPT \'" + FWUKL1_ABSTRACT_REGISTER_DPT_NAME + "\' exists.", "1");
      // Without the datapoint there is little point trying to do anything.
      return;
    }// if dpCreate() failed
  }// if(!dpExists())
}// fwUkl1_createAbstractRegisterDP()

/*!
 * Deletes the DP that represents the abstract register.
 *
 * @param  abstractRegisterName of the abstract register to create the DP for. If the register
 *           does not exist then no changes are made.
 * @param  exceptionInfo Error information. No change in size if no error information is generated.
 * @return void.
 *
 * Should only be done once the service has been unsubscribed.
 * The name of the datapoint is constructed from the service name. Illegal characters are removed.
 */
string fwUkl1_deleteAbstractRegisterDP(const string& abstractRegisterName, dyn_string& exceptionInfo) {
  // Keeps track of the exceptionInfo size.
  int exInfoSize = dynlen(exceptionInfo);
  if (dpExists(abstractRegisterName)) {
    // If the DP exists then it must be deleted.
    dpDelete(abstractRegisterName);
  }// if(dpExists())
  // No else as we just return if the data point already exists.
  // Done.
  return;
}// fwUkl1_deleteAbstractRegisterDP()

/*!
 * Deletes all the abstract register DPs from the system.
 *
 * @param  Error information. No change in size if no exception information is generated.
 * @param  sys System Name.
 * @return void.
 *
 * This asks the DPT for all its registers and deletes them. This way registers that no longer
 * exist can be removed from the system.
 */
void fwUkl1_deleteAllAbstractRegisterDPs(dyn_string &exceptionInfo, string sys = "") {
  int exInfoSize = dynlen(exceptionInfo);
  dyn_string absRegList = fwUkl1_getRegisterList(exceptionInfo, sys);
  if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_deleteAllAbstractRegisterDPs(): Could not get the list of abstract registers to delete. No registers were deleted.", "2", exInfoSize)) {
    const int numAbsRegs = dynlen(absRegList);
    for (int absRegNum=1; absRegNum<=numAbsRegs; ++absRegNum) {
      fwUkl1_deleteAbstractRegisterDP(absRegList[absRegNum], exceptionInfo);
      fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_deleteAllAbstractRegisterDPs(): Failed to delete the abstract register "+absRegList[absRegNum]+".", "2", exInfoSize);
    }// for absRegNum
  }// if got register list
  // Done.
  return;
}// fwUkl1_deleteAllAbstractRegisterDPs()

/*!
 * This retrieves the names of the abstract registers that are in use in the system.
 *
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @param  sys The System Name.
 * @return dyn_string List of the register names, empty in the event of an error.
 */
dyn_string fwUkl1_getRegisterList(dyn_string &exceptionInfo, string sys = "") {
  int exInfoSize = dynlen(exceptionInfo);
  // Holds the hardware register name.
  dyn_string rList = dpNames(sys+"*", FWUKL1_ABSTRACT_REGISTER_DPT_NAME);
  // Done.
  return rList;
}// fwUkl1_getRegisterList()

/*!
 * Returns all the registers that are in the given group and are present in the recipe type as keys.
 *
 * @param  groups List of groups that the registers should be part of. Can be an empty dyn_string if it is not to be checked.
 * @param  recipeType Name of the recipe type that the registers should be in. Can be empty string if it is not to be checked.
 * @param  rejectedFromType This is a list of the registers that were rejected because they were not in the recipe type,
 *           but they were in the group. This is useful if panel elements are to be disabled when they are not in the type.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @param  sys The System Name.
 * @return dyn_string List of the registers that matched the search criteria.
 *
 * The map need not be empty, registers not present are added. Those already present are ignored. Newly added
 * keys have their associated data iniitalised to an empty dyn_string.
 */
dyn_string fwUkl1_getRegistersInGroups(const dyn_string &groups, const string &recipeType, dyn_string &rejectedFromType, dyn_string &exceptionInfo, string sys = "") {
  int exInfoSize = dynlen(exceptionInfo);
  // Need to create a string for the error message as it isn't updated in loops properly!
  string errmsg = "";
  // This will hold the selected list of registers.
  dyn_string regList;
  // First we must get the list of registers in the system.
  const dyn_string absRegList = fwUkl1_getRegisterList(exceptionInfo, sys);
  if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_getRegistersInGroups(): Failed to get the list of registers that are in the system, thus we cannot retrieve those that are in the specified group and recipe type.", "2", exInfoSize)) {
    // Used to note if we passed the cut.
    bool passed = FALSE;
    // Now loop over those registers and retrieve the group.
    const int numRegs = dynlen(absRegList);
    for (int regNum=1; regNum<=numRegs; ++regNum) {
      // Default to not passed.
      passed = FALSE;
      // Name of the register that we are looking at.
      const string reg = absRegList[regNum];
      
      // Only check the group list if it is not empty.
      if (0<dynlen(groups)) {
        const dyn_string regGroups = fwUkl1_getRegisterGroups(reg, exceptionInfo);
        errmsg = "fwUkl1_getRegistersInGroups(): Failed to retrieve the group for the register " + reg + ". This register will not be returned in the list.";
        if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", errmsg, "2", exInfoSize)) {
          // Loop over the given groups and compare to the list that this register is in.
          for (int i=1; i<=dynlen(groups); ++i) {
            if (0<dynContains(regGroups, groups[i]))
              passed = TRUE;
            else
              passed = FALSE;
          }// for i
        }// if no error
      }// if(0<dynlen(groups))
      
      // Only check the recipe type if it is not empty and we are already saving this recipe.
      // We only record a failed register if it was because it wasn't in the type.
      bool failedRecipeType = FALSE;
      if (passed && (""!=recipeType)) {
        // This returns a list of the recipe types that contain this.
        // It doesn't matter if we use read or write DB registers they are the same by definition.
        const dyn_string regRTypes = fwUkl1_getDbRecipeTypes(reg, exceptionInfo);
        errmsg = "fwUkl1_getRegistersInGroups(): Failed to retrieve the recipe type for the register " + reg + ". This register will not be returned in the list.";
        if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", errmsg, "2", exInfoSize)) {
          if (0<dynContains(regRTypes, recipeType)) {
            passed = TRUE;
          }// if type in approved list
          else {
            passed = FALSE;
            failedRecipeType = TRUE;
          }// else type not in approved list
        }// if get group
      }// if check recipeType

      // See if we should add the register to the map.
      if (passed) {
        dynAppend(regList, reg);
      }// if(passed)
      else {
        // Add it to the list of rejected registers.
        if (failedRecipeType) {
          dynAppend(rejectedFromType, reg);
        }// if(failedRecipeType)
      }// else(passed)
    }// for regNum
  }// if got register list
  // Done.
  return regList;
}// fwUkl1_getRegistersInGroups()

/*!
 * This retrieves the tool tip that should be associated with shape used to display the data.
 *
 * @param  regName Name of the register to look up.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return string Tool tip text information.
 */
string fwUkl1_getToolTipText(const string &regName, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  string tooltip = _fwUkl1_getUkl1AbstractRegisterProperty(regName, FWUKL1_TOOLTIP_TEXT, exceptionInfo);
  if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_getToolTipText(): Failed to retrieve the tool tip text associated with " + regName + ".", "2", exInfoSize))
    tooltip = "";
  // Done.
  return tooltip;
}// fwUkl1_getToolTipText()

/*!
 * This retrieves the function that should be used to combine the parsed data from each
 * hardware register associated with abstract register.
 *
 * @param  regName Name of the register to look up.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return string Combiner function
 */
string fwUkl1_getCombinerFunction(const string &regName, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  string combiner = _fwUkl1_getUkl1AbstractRegisterProperty(regName, FWUKL1_COMBINER_FUNCTION, exceptionInfo);
  if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_getCombinerFunction(): Failed to retrieve the combiner function associated with " + regName + ".", "2", exInfoSize))
    combiner = "";
  // Done.
  return combiner;
}// fwUkl1_getToolTipText()
    
/*!
 * This retrieves the masks for the hardware registers of this abstract register as of hex strings.
 *
 * @param  regName Name of the register to look up.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return dyn_string Masks as hex strings.
 */
dyn_string fwUkl1_getMasks(const string &regName, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  dyn_string masks = _fwUkl1_getUkl1AbstractRegisterProperty(regName, FWUKL1_MASKS, exceptionInfo);
  if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_getWriteFunction(): Failed to retrieve the masks associated with " + regName + ".", "2", exInfoSize))
    masks = makeDynString();
  // Done.
  return masks;
}// fwUkl1_getMasks()
    
/*!
 * Retrieves the groups associated with the given register.
 *
 * @param  regName Name of the register to find the associated hardware register.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return dyn_string List of group names which this register is a member of or empty in the event of failure.
 */
dyn_string fwUkl1_getRegisterGroups(const string &regName, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  dyn_string groups = _fwUkl1_getUkl1AbstractRegisterProperty(regName, FWUKL1_REGISTER_GROUPS, exceptionInfo);
  if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_getRegisterGroups(): Failed to retrieve the group associated with " + regName + ".", "2", exInfoSize))
    groups = makeDynString();
  // Done.
  return groups;
}// fwUkl1_getRegisterGroups()

/*!
 * Retrieves the names of the hardware registers associated with the a list of abstract regiseters.
 *
 * @param  registerList Names of the registers to find the associated hardware register(s).
 * @param  registerTypes One of the constants FWUKL1_DB_READ, FWUKL1_DB_WRITE, FWUKL1_HW_CONFIG_READ,
 *           FWUKL1_HW_STATUS_READ or FWUKL1_HW_WRITE. The value indicates which type of register
 *           the value should be retrieved for e.g. the database write register, hardware status register.
 *           Each element must correspond to an element in the registers list.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return dyn_dyn_string Each element corresponds to the equivalent element in the input registers
 *           dyn_string and contains the hardware registers associated with it.
 */
dyn_dyn_string fwUkl1_getAbsRegsHwRegNames(const dyn_string &registerList, const dyn_string &registerTypes, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  // Holds the list of hardware registers once retrieved from the DP.
  dyn_dyn_string hwRegNames;
  // Holds the individual hw reg while we check whether to add it.
  string hwRegName;
  // Number of registers to search for.
  const int numRegs = dynlen(registerList);
  // Name and type of the abstract register we are interested in.
  // The type allows us to find the appropriate hardware/DB register
  // as it typically varies between the register types.
  string regName;
  string regType;
  for (int regNum=1; regNum<=numRegs; ++regNum) {
    regName = registerList[regNum];
    regType = registerTypes[regNum];
    hwRegName = fwUkl1_getAbsRegHwRegNames(regName, regType, exceptionInfo);
    if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_getAbsRegsHwRegNames(): Failed to retrieve the hardware register associated with the abstract register " + regName + ". Returned list of hardware registers may not be complete.", "2", exInfoSize)) {
      dynAppend(hwRegNames, hwRegName);
    }// if got hardware reg name.
  }// for regNum
  // Done.
  return hwRegNames;
}// fwUkl1_getHwRegNames()

/*!
 * Retrieves the names of the hardware registers associated with the given abstract register.
 *
 * @param  regName Name of the register to find the associated hardware register.
 * @param  registerType One of the constants FWUKL1_DB_READ, FWUKL1_DB_WRITE, FWUKL1_HW_CONFIG_READ,
 *           FWUKL1_HW_STATUS_READ or FWUKL1_HW_WRITE. The value indicates which type of register
 *           the value should be retrieved for e.g. the database write register, hardware status register.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return dyn_string Empty dyn_string in the event of an error.
 */
dyn_string fwUkl1_getAbsRegHwRegNames(const string &regName, const string &registerType, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  dyn_string hwRegs = _fwUkl1_getUkl1AbstractRegisterProperty(regName, registerType+FWUKL1_ACCESS_REGISTERS, exceptionInfo);
  if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_getAbsRegHwRegNames(): Failed to retrieve the hardware registers associated with " + regName + ".", "2", exInfoSize))
    hwRegs = makeDynString();
  // Done.
  return hwRegs;
}// fwUkl1_getHwRegNames()

/*!
 * This retrieves the name of the functions that should be used to parse the data.
 *
 * @param  regName Name of the register to look up.
 * @param  registerType One of the constants FWUKL1_DB_READ, FWUKL1_DB_WRITE, FWUKL1_HW_CONFIG_READ,
 *           FWUKL1_HW_STATUS_READ or FWUKL1_HW_WRITE. The value indicates which type of register
 *           the value should be retrieved for e.g. the database write register, hardware status register.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return dyn_string Names of the functions to use.
 */
dyn_string fwUkl1_getParserFunctions(const string &regName, const string &registerType, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  dyn_string funcs = _fwUkl1_getUkl1AbstractRegisterProperty(regName, registerType+FWUKL1_PARSER_FUNCTIONS, exceptionInfo);
  if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_getParserFunction(): Failed to retrieve the function associated with " + regName + " for " + registerType + ".", "2", exInfoSize))
    funcs = makeDynString();
  // Done.
  return funcs;
}// fwUkl1_getParserFunctions()

/*!
 * This retrieves the information that should be given to the parser function to parse the data.
 *
 * @param  regNames Two register names are required in order to retrieve the appropriate function.
 *           Element 1 should contain the abstract register and element 2 the specific hardware
 *           register associated with that register to retrieve the data for.
 * @param  registerType One of the constants FWUKL1_DB_READ, FWUKL1_DB_WRITE, FWUKL1_HW_CONFIG_READ,
 *           FWUKL1_HW_STATUS_READ or FWUKL1_HW_WRITE. The value indicates which type of register
 *           the value should be retrieved for e.g. the database write register, hardware status register.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return dyn_dyn_string Parsed parser information. Each of element of the dyn_dyn_string contains the
 *           list of arguments to the data parsing functions required to get the data into a displayable
 *           or writeable format. Each element of the dyn_string contains the argument itself. No
 *           indication is given as to what functions should be called this is required to be known.
 *           Empty in the event of an error.
 */
dyn_dyn_string fwUkl1_getParserInfo(const dyn_string &absRegNames, const string &registerType, int index, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  dyn_dyn_string dds = _fwUkl1_getExtraRegisterInformation(absRegNames, registerType+FWUKL1_PARSER_INFO, index, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_getParserInfo(): Failed to retrieve the information required to parse the function for " + absRegNames + " for " + registerType + ".", "2", exInfoSize);
  // Done.
  return dds;
}// fwUkl1_getParserInfo()

/*!
 * This retrieves the information required to ensure the parsed value is within limits.
 *
 * @param  absRegName The abstract register to retrieve the data for.
 * @param  registerType One of the constants FWUKL1_DB_READ, FWUKL1_DB_WRITE, FWUKL1_HW_CONFIG_READ,
 *           FWUKL1_HW_STATUS_READ or FWUKL1_HW_WRITE. The value indicates which type of register
 *           the value should be retrieved for e.g. the database write register, hardware status register.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return dyn_dyn_string Parsed validation information. Each of element of the dyn_dyn_string contains the
 *           list of arguments to the data parsing functions required to get the data into a displayable
 *           or writeable format. Each element of the dyn_string contains the argument itself. No
 *           indication is given as to what functions should be called this is required to be known.
 *           Empty in the event of an error.
 */
dyn_dyn_string fwUkl1_getValidationInfo(const string &absRegName, const string &registerType, int index, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  // This is what we will return.
  dyn_dyn_string dds = _fwUkl1_getExtraRegisterInformation(absRegName, registerType+FWUKL1_VALIDATION, index, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_getValidationInfo(): Failed to retrieve the information required to validate the returned data for " + absRegName + " for " + registerType + ".", "2", exInfoSize);
  // Done.
  return dds;
}// fwUkl1_getValidationInfo()

/*!
 * Retrieves the types of the specific hardware registers associated with the given abstract register.
 *
 * @param  absRegNames Should contain the abstract register register to retrieve the HW types for.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return dyn_int Each element contains the hardware type.
 */
dyn_int fwUkl1_getHwRegTypes(const string &absRegName, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  dyn_int hwTypes = _fwUkl1_getUkl1AbstractRegisterProperty(absRegName, FWUKL1_HW_REGISTER_TYPES, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_getHwRegType(): Failed to retrieve the types of the hardware registers associated with " + absRegName + ".", "2", exInfoSize);
  return hwTypes;
}// fwUkl1_getHwRegTypes()

/*!
 * Retrieves the type of the specific hardware register associated with the given abstract register.
 *
 * @param  regNames Two register names are required in order to retrieve the appropriate function.
 *           Element 1 should contain the abstract register and element 2 the specific hardware
 *           register associated with that register to retrieve the data for.
 * @param  registerType One of the constants FWUKL1_DB_READ, FWUKL1_DB_WRITE, FWUKL1_HW_CONFIG_READ,
 *           FWUKL1_HW_STATUS_READ or FWUKL1_HW_WRITE. The value indicates which type of register
 *           the value should be retrieved for e.g. the database write register, hardware status register.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return int -1 in the event of failure or the hardware type.
 *
 * This function relies on the fact that if there are duplicates of the hardware register used by
 * this abstract register they must be of the same type. It will return the type of the first occurance
 * of the given hardware register in the hardware register list.
 */
int fwUkl1_getHwRegType(const dyn_string &regNames, const string &registerType, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  // Useful for indexing the regNames dyn_string.
  const int ABS_REG = 1;
  const int HW_REG = 2;
  // This is what we return.
  int hwType = -1;
  // We need to get both the list of hardware registers and the hw reg types list.
  const dyn_string hwNames = _fwUkl1_getUkl1AbstractRegisterProperty(regNames[ABS_REG], registerType+FWUKL1_ACCESS_REGISTERS, exceptionInfo);
  if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_getHwRegType(): Failed to retrieve the list of available hardware registers. Cannot determine which hardware type to retrieve for this abstract register.", "2", exInfoSize)) {
    // Now find the element index of the hardware registers we want.
    int index = dynContains(hwNames, regNames[HW_REG]);
    if (-1<index) {
      dyn_int hwTypes = _fwUkl1_getUkl1AbstractRegisterProperty(regNames[ABS_REG], FWUKL1_HW_REGISTER_TYPES, exceptionInfo);
      if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_getHwRegType(): Failed to retrieve the types of the hardware registers associated with " + regNames[HW_REG] + " for " + registerType + ".", "2", exInfoSize)) {
        if (index <= dynlen(hwTypes))
          hwType = hwTypes[index];
        else
          exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getHwRegType(): The hardware register '"+regNames[HW_REG]+"' does not have a corresponding type. Required hardware type at index "+index+" but we have only "+dynlen(hwTypes)+" hardware type entries.", "2");
      }// if got hwTypes
    }// if hw reg found.
    else
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getHwRegType(): The hardware register '"+regNames[HW_REG]+"' was not found in the list of available harware registers for '"+regNames[ABS_REG]+"'. Cannnot retrieve the hardware register type for this register.", "2");
  }// if got list of hardware registers
  
  // Done.
  return hwType;
}// fwUkl1_getHwRegType()

/*!
 * Retrieves the names of the recipe types associated with the given abstract register.
 *
 * @param  regName Name of the register to find the recipe types for.
 * @param  registerType One of the constants FWUKL1_DB_READ or FWUKL1_DB_WRITE. The value indicates
 *           which type of register the value should be retrieved for e.g. the database write register.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return dyn_string Empty dyn_string in the event of an error.
 */
dyn_string fwUkl1_getDbRecipeTypes(const string &regName, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  dyn_string recipeTypes = _fwUkl1_getUkl1AbstractRegisterProperty(regName, FWUKL1_DB_RECIPE_TYPES, exceptionInfo);
  if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_getDbRecipeTypes(): Failed to retrieve the recipe types associated with " + regName + ".", "2", exInfoSize))
    recipeTypes = makeDynString();
  // Done.
  return recipeTypes;
}// fwUkl1_getDbRecipeTypes()

/*!
 * This retrieves the names of the functions that should be used to access the hardware or database.
 *
 * @param  regName Name of the register to look up.
 * @param  registerType One of the constants FWUKL1_DB_READ, FWUKL1_DB_WRITE, FWUKL1_HW_CONFIG_READ,
 *           FWUKL1_HW_STATUS_READ or FWUKL1_HW_WRITE. The value indicates which type of register
 *           the value should be retrieved for e.g. the database write register, hardware status register.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return dyn_string Names of the function to use.
 */
dyn_string fwUkl1_getAccessFunctions(const string &regName, const string &registerType, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  dyn_string funcs = _fwUkl1_getUkl1AbstractRegisterProperty(regName, registerType+FWUKL1_ACCESS_FUNCTIONS, exceptionInfo);
  if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_getAccessFunction(): Failed to retrieve the access function associated with " + regName + " for " + registerType + ".", "2", exInfoSize))
    funcs = makeDynString();
  // Done.
  return funcs;
}// fwUkl1_getAccessFunctions()

/*!
 * This retrieves the function that should be used to start monitoring the register.
 *
 * @param  regName Name of the register to look up.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return string Function name.
 */
string fwUkl1_getStartMonitoringFunction(const string &regName, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  string func = _fwUkl1_getUkl1AbstractRegisterProperty(regName, FWUKL1_MONITORING_START_FUNCTION, exceptionInfo);
  if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_getStartMonitoringFunction(): Failed to retrieve the monitoring start function associated with " + regName + ".", "2", exInfoSize))
    func = "";
  // Done.
  return func;
}// fwUkl1_getStartMonitoringFunction()

/*!
 * This retrieves the function that should be used to stop monitoring the register.
 *
 * @param  regName Name of the register to look up.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return string Function name.
 */
string fwUkl1_getStopMonitoringFunction(const string &regName, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  string func = _fwUkl1_getUkl1AbstractRegisterProperty(regName, FWUKL1_MONITORING_STOP_FUNCTION, exceptionInfo);
  if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_getStopMonitoringFunction(): Failed to retrieve the monitoring start function associated with " + regName + ".", "2", exInfoSize))
    func = "";
  // Done.
  return func;
}// fwUkl1_getStopMonitoringFunction()

/*!
 * This retrieves the list of hardware registers that should be monitored for this function.
 *
 * @param  regName Name of the register to look up.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return dyn_string List of registers to monitor.
 */
dyn_string fwUkl1_getMonitoringRegisterList(const string &regName, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  dyn_string list = _fwUkl1_getUkl1AbstractRegisterProperty(regName, FWUKL1_MONITORING_REGISTERS, exceptionInfo);
  if (fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_getMonitoringRegisterList(): Failed to retrieve the list of hardware registers to monitor associated with " + regName + ".", "2", exInfoSize))
    list = makeDynString();
  // Done.
  return list;
}// fwUkl1_getMonitoringRegisterList()

// ==========
// INTERNAL
// =========

/*!
 * This returns the extra information for a specified action and returns it in a parsed form.
 *
 * @param  regName Abstract register name to retrieve the extra information for.
 * @param  hwRegName Name of the specific hardware registers 
 * @param  action Takes on of the three library constants FWUKL1_HW_WRITE+FWUKL1_PARSER_INFO, FWUKL1_HW_CONFIG_READ+FWUKL1_PARSER_INFO
 *           FWUKL1_HW_STATUS_READ+FWUKL1_PARSER_INFO or FWUKL1_HW_CONFIG_READ+FWUKL1_VALIDATION.
 * @param  index The extra information is stored in a list with each element corresponding to one of the
 *           hardware registers. The index gives us specific extra information element to retrieve.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return dyn_dyn_string Each of element of the dyn_dyn_string contains the list of arguments to the data parsing
 *           functions required to get the data into a displayable or writeable format. Each element of the
 *           dyn_string contains the argument itself. No indication is given as to what functions should be
 *           called this is required to be known. Empty in the event of an error.
 *
 * The library provides the functions fwUkl1_parseRegistersForGet/Set which are capable of interpreting this
 * information.
 */
dyn_dyn_string _fwUkl1_getExtraRegisterInformation(const string &regName, const string &action, int index, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  // Store the outcome here.
  dyn_dyn_string dds;
  // Get the extra information element.
  string xtrInfo = _fwUkl1_getUkl1AbstractRegisterProperty(regName, action, exceptionInfo)[index];
  if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_getExtraRegisterInformation(): Failed to retrieve the extra register information associated with " + regName + " for " + action + ".", "2", exInfoSize)) {
    // Now we have the information we need to parse it.
    if (""!=xtrInfo) {
      dyn_string semiColonDelim = strsplit(xtrInfo, ";");
      const int numSemiSec = dynlen(semiColonDelim);
      // Now split the sections by commas.
      for (int secNum=1; secNum<=numSemiSec; ++secNum) {
        dynAppend(dds, strsplit(semiColonDelim[secNum], ","));
      }// for secNum
    }// if got some extra info
  }// if got extra info successfully
  return dds;
}// fwUkl1_getExtraRegisterInformation()

/*!
 * This retrieves a value associated with a register from the FwUkl1Board DP.
 *
 * @param  regName The abstract register name that we want to retrieve the information for.
 * @param  dpeName This is the name of the DPE element that is to be retrieved. These are all defined as library constants.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return anytype The value that is requested from the DP. It is an any type so must be converted to the appropriate
 *           value for the constant. In the event that an error occurs the value of the return can be anything.
 */
anytype _fwUkl1_getUkl1AbstractRegisterProperty(string regName, const string& dpeName, dyn_string& exceptionInfo) {
  const string fullName = regName + dpeName;
  // Holds the value that is to be returned.
  anytype value;
  // Now check it exists.
  if (dpExists(fullName)) {
    // DP exists so retrieve the data.
    if (0!=dpGet(fullName, value)) {
      // Failed to retrieve the value, note this.
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_getUkl1AbstractRegisterProperty(): Failed to retrieve the requested DPE \'"+fullName+"\'. dpGet() failed.", "2");
    }// if(0!=dpGet())
  }// if(dpExists(fullName))
  else {
    // Raise an exception to note that the requested DP element did not exist.
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_getUkl1AbstractRegisterProperty(): Failed to retrieve the requested constant as the DPE \'"+fullName+"\' does not exist.", "2");
  }// else(dpExists(fullName))
  // Done.
  return value;
}// _fwUkl1_getUkl1AbstractRegisterProperty()

// @}
