//  ====================================
//   NAME    : fwUkl1CB
// 
//   VERSION : 3
// 
//   REVISION: 0
// 
//   DATE    : 2008/07/28
// 
//   RELEASE : 2009/10/12
// 
//   PLATFORM: PVSS II
// 
//   AUTHOR  : Gareth Rogers
// 
//   EMAIL   : rogers@hep.phy.cam.ac.uk
//  ====================================

/* \mainpage UKL1CB Contains all the callback functions that are available for the UKL1 library.
 *
 *   All callback functions that be connected to are in this library. It includes the callback functions
 *   that are to be used to keep the abstract registers panel shapes update when monitoring the hardware registers.
 */

// ====================
//  #USES DECLARATIONS
// ====================

/*! Used to retrieve the information associated with the abstract registers. */
#uses "fwUkl1/fwUkl1DPInterface.ctl"

/*! Provides access to the DB and hardware. */
#uses "fwUkl1/fwUkl1RegisterInterface.ctl"

/*! fwUkl1ExcepionHandling library is used to store and display errors that occur with fwUkl1 function calls. */
#uses "fwUkl1ExceptionHandling/fwUkl1ExceptionHandling.ctl"

/*! fwHw library is used for recipe and data point manipulation. */
#uses "fwHw.ctl"

/*! fwCcpc library is used for accessing the UKL1 registers. */
#uses "fwCcpc.ctl"

/*! The FSM libraries are used to create and configure the UKL1 FSM tree. */
#uses "fwCU"
#uses "fwDU"

// =========================
//  CONSTANTs & GLOBAL VARs
// =========================

/** @defgroup SectionConstants Constants
 *
 *    These are the constants used by the library and are available for library users. Primaryly
 *    they define the DP structures in use and provide a means to interact with the DPEs without
 *    the need to hardcode names.
 *
 *  @{
 */

/*! Defines the name of the callback function that manipulates the FSM state depending on the CCSERV state. */
const string FWUKL1_MONITOR_CCSERV_CB = "fwUkl1_monitorCcservStateCB";

/*! Defines the name of the callback function that will disable the UKL1 channels if request via DIM. */
const string FWUKL1_DISABLE_UKL1_CHANNEL_CB = "fwUkl1_disableHpdCB";
// @}

// ========================
//  UKL1 CONNECT FUNCTIONS
// ========================

/*! @defgroup ConnectFunctions Connect functions
 *    These are a group of functions that automate the calling of the connect function for call back registers.
 * // @{
 */

/*!
 * This function takes a list of abstract registers and connects the appropriate function to the appropriate hardware datapoint.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  regList List of abstract registers that are to be updated.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 */
void fwUkl1_connectAbstractRegsToHwDps(const string &ukl1Name, const dyn_string &regList, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  // Need to know how many registers to loop over.
  const int numRegs = dynlen(regList);
  // This will store the name of the callback function we are connecting to.
  string cb;
  // Name of the associate HW registers.
  dyn_string hwRegs;
  // The tool tip text associated with this register.
  string tooltip;
  // The DPE to connect to all have an extension that need to be added after the hw reg.
  const string extension = ".readings";
  // The name of the DPE we wish to connect to.
  string dpe;
  // Name of the abstract register we are connecting.
  string absReg;
  // The abstract register without the system name.
  // This is required for shape retrieval and callback function creation.
  string absRegNoSysName;
  // Loop over all the registers given.
  for (int regNum = 1; regNum <= numRegs; ++regNum) {
    absReg = regList[regNum];
    absRegNoSysName = fwUkl1_removeSystemName(absReg);
    // Some of the register names contain full stops and these can't be part of a function name.
    // Just remove them from the string.
    cb = "fwUkl1_" + absRegNoSysName + "CB";
    strreplace(cb, ".", "");
    if (isFunctionDefined(cb)) {
      // Get the list of hardware registers to monitor.
      hwRegs = fwUkl1_getMonitoringRegisterList(absReg, exceptionInfo);
      if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_connectAbstractRegsToHwDps(): Failed to connect the abstract register " + absRegNoSysName + " to the associated hardware register as it could not be found.", "2", exInfoSize)) {
        // This will contain the DPEs that we wish to connect to.
        dyn_string dpes;
        // Create a DPE to connect to for each of the hardware registers given.
        for (int i=1; i<=dynlen(hwRegs); ++i) {
          string dpe = ukl1Name+hwRegs[i]+extension;
          if (dpExists(dpe))
            dynAppend(dpes, dpe);
          else
            exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_connectAbstractRegsToHwDps(): Failed to connect the abstract register " + absRegNoSysName + " to the datapoint element " + dpe + " because the DPE did not exists.", "2");
        }// for i
        
        // Now connect the callback function to these DPEs.
        if (-1 == dpConnect(cb, FALSE, dpes))
          exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_connectAbstractRegsToHwDps(): Failed to connect the abstract register " + absRegNoSysName + " to the requested datapoint elements. The dpConnect failed.", "2");
      }// if got hw regs
      // Get the tool tip text.
      tooltip = fwUkl1_getToolTipText(absReg, exceptionInfo);
      if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_connectAbstractRegsToHwDps(): Failed to retrieve the tool tip text for the register " + absRegNoSysName + ". No information about this register can be found by hovering over the display shape.", "2", exInfoSize)) {
        // We know what shape name we want to put the data on.
        const string shapeName = absRegNoSysName + "Field";
        // Check if it exists.
        if (shapeExists(shapeName)) {
          shape sh = getShape(shapeName);
          sh.toolTipText(tooltip);
        }// if shapeExists()
      }// if got tool tip text.
    }// if(isFunctionDefined())
    else
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_connectAbstractRegsToHwDps(): Failed to connect the abstract register " + absRegNoSysName + " to its associated hardware datapoint as the callback function " + cb + " did not exist.", "2");
  }// for regNum
  
  // Done.
  return;
}// fwUkl1_connectAbstractRegsToHwDps()

/*!
 * This function takes a list of abstract registers and disconnects the appropriate function from the hardware datapoint.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  regList List of abstract registers that are to be updated.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 *
 * This will only work if it is called in the same scope as the connect function.
 */
void fwUkl1_disconnectAbstractRegsToHwDps(const string &ukl1Name, const dyn_string &regList, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  // Need to know how many registers to loop over.
  const int numRegs = dynlen(regList);
  // This will store the name of the callback function we are connecting to.
  string cb;
  // Name of the abstract register we are connecting.
  string absReg;
  // The abstract register without the system name.
  // This is required for shape retrieval and callback function creation.
  string absRegNoSysName;
  // Name of the associate HW registers.
  dyn_string hwRegs;
  // The DPE to connect to all have an extension that need to be added after the hw reg.
  const string extension = ".readings";
  // The name of the DPE we wish to connect to.
  string dpe;
  // Loop over all the registers given.
  for (int regNum = 1; regNum <= numRegs; ++regNum) {
    absReg = regList[regNum];
    absRegNoSysName = fwUkl1_removeSystemName(absReg);
    // Some of the register names contain full stops and these can't be part of a function name.
    // Just remove them from the string.
    cb = "fwUkl1_" + absRegNoSysName + "CB";
    strreplace(cb, ".", "");
    if (isFunctionDefined(cb)) {
      // This will contain the DPEs that we wish to connect to.
      dyn_string dpes;
      // Get the hardware reg name
      hwRegs = fwUkl1_getMonitoringRegisterList(absReg, exceptionInfo);
      if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_disconnectAbstractRegsToHwDps(): Failed to disconnect the abstract register " + absRegNoSysName + " from the associated hardware register as it it could not be found.", "2", exInfoSize)) {
        // Create a DPE to connect to for each of the hardware registers given.
        for (int i=1; i<=dynlen(hwRegs); ++i) {
          string dpe = ukl1Name+hwRegs[i]+extension;
          if (dpExists(dpe))
            dynAppend(dpes, dpe);
          else
            exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_disconnectAbstractRegsToHwDps(): Failed to disconnect the abstract register " + absRegNoSysName + " to the datapoint element " + dpe + " because the DPE did not exists.", "2");
        }// for i
        
        // Now disconnect the callback function to these DPEs.
        if (-1 == dpDisconnect(cb, dpes))
          exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_disconnectAbstractRegsToHwDps(): Failed to disconnect the abstract register " + absRegNoSysName + " from the datapoint elements. The dpDisconnect failed.", "2");
      }// if got hardware regs
    }// if callback function defined
  }// for regNum
  // Done.
  return;
}// fwUkl1_disconnectAbstractRegsToHwDps()

// @}


// =========================
//  UKL1 CALLBACK FUNCTIONS
// =========================

/*! @defgroup CallbackFunctions Callback functions
 *    The call back functions that are available.
 * // @{
 */

/*!
 * This function is connected to a UKL1 ccserv state and is called when ever this changes.
 * It will set the UKL1 FSM state to UNKNOWN if the ccserv process disconnects from the
 * PVSS system and to NOT_READY when it reconnects.
 *
 * @param  domain FSM domain that triggered the action.
 * @param  device Name of the FSM DU that triggered the action, corresponds to the UKL1 name.
 * @param  state State of ccserv process.
 * @return void.
 *
 * An error message is sent to the exception log if the ccserv disconnects, but not if it reconnects.
 * This behaviour is overidden for the case where the UKL1 is in the RUNNING state.
 * In this case there will be no change to the UKL1 state, but a message is still sent to the
 * exception log. The connection is setup via the fwHw library function fwHw_connectServerState.
 *
 * Once a board is connected it will be setup to start monitoring those registers that can be monitored
 * on the ccserv.
 */
void fwUkl1_monitorCcservStateCB(const string &domain, const string &device, int state) {
  dyn_string exceptionInfo;
  int exInfoSize = dynlen(exceptionInfo);
  // Perform the necessary bits when we disconnect, except state manipulation.
  if (0 == state)
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_monitorCcservStateCB(): ccserv process for the " + device + " disconnected. Communication is impossible with this board until the ccserv process reconnects. The board can continue to operate as the firmware is likey unaffected by the event.", "2");
  // Now perform the necessary bit when we connect.
  else if (1 == state) {
    fwUkl1_startMonitoringAllMonitorableRegs(device, exceptionInfo);
    fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_monitorCcservStateCB(): Failed to start monitoring the registers for " + device + " when it reconnected.", "2", exInfoSize);
  }// else(1==state)
  
  // Now set the appropriate state for the UKL1.
  const string dpeName = device + ".status";
  if (dpExists(dpeName)) {
    // The FSM state does exist for this device, now check its state.
    const string fsmState = fwUkl1_getUkl1State(device);
    if ("RUNNING" != fsmState) {
      // We are not in the running state so set the new state accordingly.
      // Note that we have already checked for the existence of the DPE so don't need to worry about
      // an empty string return from the fwUkl1_getUkl1State function.
      if (0 == state)
        dpSet(dpeName, "UNKNOWN");
      else
        dpSet(dpeName, "NOT_READY");
    }// if("RUNNING"!=fsmState)
    // no else, don't change the state when in the RUNNING state.
  }// if(dpExists(dpeName))
  else
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_monitorCcservStateCB(): Cannot find the FSM state datapoint element for " + device + ", cannot update its state to reflect the ccserv connection status. ccserv for this UKL1 " + (0==state ? "diconnected":"connected") + ".", "2");
  
  fwUkl1ExceptionLog_submit(exceptionInfo, device);
  // Done.
  return;
}// fwUkl1_monitorCcservStateCB()

/*!
 * This callback function will disable the requested channel.
 *
 * @param  dpeName Name of the data point element, DPE, that changed.
 * @param  sNewValue New value of the given DPE. A string in this case.
 *
 * The channels requested to be disabled are monitored and updated
 * by DIM. It will not disable the channel if the UKL1 constant
 * FWUKL1_DISABLE_UKL1_CHANNEL is set to FALSE. A message is always
 * sent to the exception log when a request is made.
 */
void fwUkl1_disableHpdCB(const string &dpeName, const string &sNewValue) {
  // Check to see if we have been given something, otherwise exit silently.
  if ("" != sNewValue) {
    dyn_string exceptionInfo;
    int exInfoSize = dynlen(exceptionInfo);
    // See if we can get the system name from the DPE name. Will make this more reliable in a distributed system.
    const string sysName = fwUkl1_removeDeviceName(dpeName);
    DebugTN(dpeName, sysName);
    // Get the list of UKL1s that this script can access.
    const dyn_string availableUkl1s = fwUkl1_getUkl1NamesList(exceptionInfo, sysName);
    fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_disableHpdCB(): Failed to get the list of available UKL1 boards. Will try to disable channels whether the UKL1s are available or not, but this will likely lead to a lot of error messages that can probably be ignored.", "2", exInfoSize);
    const int numUkl1sAvailable = dynlen(availableUkl1s);
    // First parse the given string to get the details about which channel to disable.
    // Initialise this to the HPD disable string, any error from failing to parse this value will appear under this string in the.
    // Otherwise the UKL1 name will be found and they will appear under that as expected.
    dyn_string ukl1Names;
    dyn_int iFpgas;
    dyn_int channels;
    fwUkl1_parseHpdDisableString(sNewValue, ukl1Names, iFpgas, channels, exceptionInfo);
    if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_disableHpdCB(): Failed to determine the UKL1 name, the Ingress FPGA and the channel number to disable. Cannot disable the given HPD channel.", "2", exInfoSize)) {
      //Now loop over all the returned channels to disable. All the returned dyn's will have the same number of entries, just use channels for fun.
      const int numChans = dynlen(channels);
        for (int index=1; index<=numChans; ++index) {
        const string ukl1Name = ukl1Names[index];
      	// Check to see if we have access to this board.
       	if (0 < dynContains(availableUkl1s, ukl1Name)) {
      	  // Now check to see if we should disable the channel or not.
         if (fwUkl1_getDisableUkl1ChannelsFlag()) {
      	    // We can't just update the soft channel inhibit flag, we must write to all the settings at once.
            // Therefore read back the current settings.
            const string channel = "Channel"+(string)(FWUKL1_FPGA_CHANNELS*iFpgas[index]+channels[index]);
            // We will use the SoftDisable register many times so save this key.
            const string softDisable = sysName+channel+"SoftDisable";
            const dyn_string channelRegNames = makeDynString(softDisable);
            // We want the configure form of the return as we will write it
            // again in a moment. forStatus argument is FALSE.
            mapping data = fwUkl1_read(ukl1Name, channelRegNames, FALSE, exceptionInfo);
      	    if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_disableHpdCB(): Failed to read current settings for the channel that must be disabled. Cannot update the disable setting for this channel as all channel configuration bits must be set at once. Don't know what the other configuration values were!", "2", exInfoSize)) {
              //Now check what the disable setting for the channel. No point in disabling a disabled channel.
              if ("Disable" != data[softDisable]) {
                //OK now we want to disable the channel setting.
                data[softDisable][FWUKL1_MAP_REG_DATA] = "Disable";
                // Now update the disable setting.
                // Verify that the write was successful.
                fwUkl1_write(ukl1Name, data, FALSE, exceptionInfo);
                fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_disableHpdCB(): On " + ukl1Names[index] + " failed to disable channel " + channels[index] + " on Ingress " + iFpgas[index] + ".", "2", exInfoSize); 
             }// if channel not already disabled or don't want it disabled.
            }// if got current channel configuration.
          }// if disable the channel
          else {
            // Stupid PVSS bug where you have to create the message first and then pass it to the function.
            const string msg = "fwUkl1_disableHpdCB(): Have been asked to disable channel " + channels[index] + " on Ingress " + iFpgas[index] + " UKL1 " + ukl1Names[index] + ". Disabling of the HPD is suppressed so not action taken. Set UKL1 constant Disable UKL1 Channel to TRUE.";
            exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", msg, "2");
          }// else requested not to disable the HPDs.
        }// if board available or no boards available.
      }// for index
    }// if got the channel identifiers.

    // Submit any errors, create a different node for this as it requires some thought to get them in an appropriate place.
    fwUkl1ExceptionLog_submit(exceptionInfo, "HPDDisableErrors");
  }// if(""!=sNewValue)
}// fwUkl1_disableHpdCB()

// @}

// =================================
//  UKL1 CALLBACK SUPPORT FUNCTIONS
// =================================

/*! @defgroup CallbackSupportFunctions Used by the callback functions.
 *
 * These functions are used to extract some common functionality that the callback
 * function use.
 * // @{
 */

/*!
 * This decodes the string that is published by the DIM server and returns the information to identify a channel uniquely.
 *
 * @param  disableHpd String published by the DIM server that contains the RICH, UKL1 ID, Ingress FPGA
 *           and channel number (within the Ingress) that is to be disabled.
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server. Returned by reference.
 * @param  iFpga Ingress FPGA number that contains the channel to be disabled. Returned by reference.
 * @param  channel Channel number within the given Ingress FPGA to be disabled. Returned by reference.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 */
void fwUkl1_parseHpdDisableString(string disableHpd, dyn_string &ukl1Names, dyn_int &iFpgas, dyn_int &channels, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  // The given string should have the form.
  // (r,u,i,c) (r,u,i,c) (r,u,i,c)... where each bracket section refers uniquely to a channel. r=RICH base 0, u=L1Id, i=IngressId, c=channel number on Ingress.
  // We need to first time the leading space to save confusing the splitter.
  disableHpd = strrtrim( strltrim(disableHpd) );
  dyn_string splitValues = strsplit(disableHpd, " ");
  // See how many sections we have.
  const int numSecs = dynlen(splitValues);
  // The number of sections we require.
  const int requiredSecs = 4;
  for (int sec=1; sec<=numSecs; ++sec) {
    // Separate out the comma delimited sections first removing the brackets.
    const string trimmed = strrtrim( strltrim( splitValues[sec], "(" ), ")" );
    dyn_string commaSplit = strsplit(trimmed, ",");
    if (requiredSecs == dynlen(commaSplit)) {
    // Now we have the Ingress and channel IDs we must convert the RICH and L1IDs to the UKL1 name.
    // Increment the RICH by 1 as the string uses a 0 base.
    string sRich = commaSplit[1];
     if((sRich == "0") || (sRich == "1")) { 
       sRich = (int)sRich + 1;    
     }
      // Ensure the UKL1 ID has two digits.
      string sL1Id = commaSplit[2];
      fwUkl1_padString(sL1Id, 1);//1 means 1 byte. Normally this would be used to deal with hex strings but it works here as the UKL1 ID is less than 100.
      const string ukl1Name = "r" + sRich + "ukl1" + sL1Id;
      // Now append the values to the given arrays.
      dynAppend(ukl1Names, ukl1Name);
      dynAppend(iFpgas, (int)commaSplit[4]);
      dynAppend(channels, (int)commaSplit[3]);
    }// if found all the parts.
    // If we don't find all the bits then don't add any entries to the lists.
    else
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_parseHpdDisableString(): Failed to find enough comma delimited sections in the disable HPD string " + splitValues[sec] + ". Require " + requiredSecs + " but only found " + dynlen(commaSplit) + " comma spearated values.", "2");
  }// for sec
  // Done.
  return;
}// _fwUkl1_parseHpdDisableString()

// @}
