// ====================================
//  NAME    : FWUKL1 Library
// 
//  VERSION : 3
// 
//  REVISION: 1
// 
//  DATE    : 2006/10/12
// 
//  RELEASE : 2010/01/20
// 
//  PLATFORM: PVSS II
// 
//  AUTHOR  : Gareth Rogers
// 
//  EMAIL   : rogers@hep.phy.cam.ac.uk
// ====================================

/* \mainpage UKL1 Library Documentation
 *
 * This library is used to control the UKL1 boards.
 */

// ====================
//  #USES DECLARATIONS
// ====================

/*! This UKL1 library contains all the functions to configure a UKL1 board. */
#uses "fwUkl1/fwUkl1DPInterface.ctl"


/*! Provides access to the registers, both hardware and database. */
#uses "fwUkl1/fwUkl1RegisterInterface.ctl"

/*! Contains the call back functions that are available for the UKL1s. */
#uses "fwUkl1/fwUkl1CB.ctl"

/*! These are the callback functions that have been created for use in the UKL1 panels. */
#uses "fwUkl1/fwUkl1PanelCB.ctl"

/*! fwUkl1ExcepionHandling library is used to store and display errors that occur with fwUkl1 function calls. */
#uses "fwUkl1ExceptionHandling/fwUkl1ExceptionHandling.ctl"

/*! fwHw library is used for recipe and data point manipulation. */
#uses "fwHw.ctl"

/*! fwCcpc library is used for accessing the UKL1 registers. */
#uses "fwCcpc.ctl"

/*! fwFSMConfDB library is used to add and remove the FSM confDB object to the FSM tree. */
#uses "fwFSMConfDB/fwFsmConfDB.ctl"


// =========================
//  CONSTANTs & GLOBAL VARs
// =========================

/** @defgroup SectionConstants Constants
 *   List of constants define for various parameters used by the UKL1 controls software.
 *   All are intended for library and user use, there are no 'internal' constants.
 *  @{
 */

/*! Defines the name of the data point that holds the UKL1 exception log. */
const string FWUKL1_EXCEPTION_LOG = "Ukl1Exceptions";

/*! Defines the name of the recipe type that is associated with the configure action. */
const string FWUKL1_CONFIGURE_TYPE = "UKL1_Configure";
/*! Defines the name of the recipe type that is associated with the start and stop actions. */
const string FWUKL1_START_TYPE = "UKL1_Start";

/*! Constant that defines the size of a 32-bit data word. */
const unsigned FWUKL1_32BIT_WORD = 32;
/*! Constant that defines the size of a 16-bit data word. */
const unsigned FWUKL1_16BIT_WORD = 16;
/*! Constant that defines the size of an 8-bit data word. */
const unsigned FWUKL1_8BIT_WORD = 8;

/*! Constant to specify the type of the FSM configurator object to be added to the UKL1 tree. */
const string FWUKL1_FSMCONFDB_TYPE = "FwFSMConfDB_DAQ";

/*!
 * The constants specify the position of the register data and the extra data that is associated
 * with a register name in the map.
 */
const int FWUKL1_MAP_REG_DATA = 1;
const int FWUKL1_MAP_XTR_DATA = 2;

// @}

// =========================
//  STATE MACHINE FUNCTIONS
// =========================

/** @defgroup SectionState FSM interface.
 *    A group of functions that can be used to perform state machine actions
 *    for a UKL1 board. The following actions are defined:
 *    \li Configure - Loads of the configuration registers on the UKL1 board with the settings defined in the database.
 *          It will leave the board in the READY state. It is slow transition.
 *    \li Start - Moves the board into the RUNNING state. Enables data to flow through the UKL1 board. Fast transition.
 *    \li Stop  - Moves the board into the READY state. Prevents data flow through the UKL1 board. It will ignore TFC
 *          commands and cannot output over the GBE card.
 *    \li Reset  - Performs a reset of the UKL1. Currently the actions of this are ill defined.
 *    \li Reload - Reloads the FPGA (Ingress/FE and Egress/BE) firmware from the onboard memory.
 *    \li Recover - Currently does nothing.
 *
 *    None of the defined functions change the FSM state, this is left to the User to ensure the FSM is in the appropriate
 *    state after the function has been called.
 *  @{
 */

/*!
 * Gets the state of the current UKL1.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @return string State that the UKL1 board is currently in, empty string if the board is not present
 *           and hence there is no associated state.
 */
string fwUkl1_getUkl1State(const string& ukl1Name) {
  // Holds the state to be returned.
  string sState = "";
  // Check the DPE exists.
  const string dpeName = ukl1Name + ".status";
  if (dpExists(dpeName))
    dpGet(dpeName, sState);
  // No else as we have initialised ourselves to the appropriate case for DPE not existing.
  return sState;
}// fwUkl1_getUkl1State()

/*!
 * Loads a specific recipe to a UKL1 board from the configuration database.
 *
 * @param  ukl1Name Name of the UKL1 board as defined in the DIM server.
 * @param  recipe Name of the recipe that is to be applied to the UKL1 board.
 * @param  domain Name of the FSM domain that the action was triggered from.
 *           If specified the recipe loaded by the "Configurator", if it exists, will be used.
 * @param  partitionId ID of the partition that this UKL1 board is within. Should be a hex string.
 *           If an empty string is passed the value stored in the recipe will be used.
 * @param  ethMtu Size of the ethernet maximum transmission unit that is to be used. If set to -1 then the value in the recipe
 *           will be used, otherwise it is overwritten.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void
 *
 * This will move the UKL1 board to a configured state, it is typically a long operation and
 * it is recommended that a fwUkl1_stop() is performed once completed. It will not update the
 * state of the UKL1, this must be done by the User.
 */
void fwUkl1_configure(const string &ukl1Name, string recipe, const string &domain, const string &partitionId, int ethMtu, dyn_string& exceptionInfo) {
  // Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_configure(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }// if(""==ukl1Name)
  // Holds the registers that can be set with this recipe.
  dyn_string regs;
  // Holds the recipe data.
  dyn_dyn_char data;
  // Keeps track of the number of errors that occured.
  int exInfoSize = dynlen(exceptionInfo);

  // When performing the configure we will always configure as much as possible and indicate any errors via
  // exception info. Boards may still be mostly usable.
  
  // See if the UKL1 name contains the system name. If it does, get it and then remove the system name.
  // We must pass the UKL1 name with the system name present otherwise it will appear not to be in the list
  // of hardware in the recipes. This causes lots and lots of difficulties!!!
  string sysName = fwUkl1_removeDeviceName(ukl1Name);
  string sysUkl1Name = ukl1Name;
  
  // Convert the recipe name that we were given into a FSMConfDB compatible name.
  // don't! getRecipe does that all for you!!!
  
  // For the moment we will still configure the hardcoded settings of the GBE from value defined settings and not DB.
  // Probably will never use the DB.
  // This performs some configuration routines for the GBE, setting the GBE into its running mode.
  _fwUkl1_configureHardcodedSettings(sysUkl1Name, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_configure(): Failed to initialise GBE or TTCrx.", "2", exInfoSize);
  
  // Perform a reset of the UKL1 logic here. Doesn't affect the configuration register,
  // but it must be done before configuring the Ingress inhibits at the moment.
  // SW. 14/11/2009 Remove this reset. It is implicated in misbehaviour of data transmission logic and is probably unnecessary because configure is usually preceeded by L1 reload.
  //fwUkl1_resetUkl1Partial(sysUkl1Name, exceptionInfo);
  //fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_configure(): Failed to perform a reset of the UKL1 logic.", "2", exInfoSize);

  // The source IP and MAC addresses must be configured dynamically from the serial number.
  _fwUkl1_configureSourceAddresses(sysUkl1Name, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_configure(): Failed to initialise some/all of the source IP and MAC addresses, and L1 ID.", "2", exInfoSize);
  
  dyn_string regs;
  dyn_dyn_char ddcData;
  dyn_dyn_char masks;
  // Get the recipe that we wish to apply, from the cache hence domain!
  if (-1 != fwHw_getRecipe(sysUkl1Name, recipe, regs, ddcData, domain)) {
    // Now we have the recipe we must remove the channel configuration from it for later application.
    // If successful this will modify the regs and data such that fwHw_applyRecipe() will only apply the Egress
    // and GBE settings. If it is not successful everything will be applied, but the channel configuration data
    // will not be set.
    _fwUkl1_configureIngressChannelSettings(sysUkl1Name, regs, ddcData, exceptionInfo);
    fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "fwUkl1_configure(): Failed to configure the UKL1 input channel specific settings from the recipe `" + recipe + "'.", "2", exInfoSize);
    
    // Apply the recipe with the Egress and GBE data.
    if (-1 == fwHw_applyRecipe(sysUkl1Name, regs, ddcData)) {
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_configure(): Failed to apply the recipe `" + recipe + "' for the Egress and GBE settings.", "2");
    }// if(-1==fwHw_applyRecipe())
  }// if(-1!=fwHw_getRecipe())
  else {
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_configure(): Failed to get the recipe `" + recipe + "'. Cannot configure the recipe specific settings.", "2");
  }// else(-1!=fwHw_getRecipe())

  // Set the MTU and the partition ID as they are not in the recipe and instead given to the configure function.
  mapping writeData;
  // Enforce minimum MTU size
  if (1024>ethMtu) {
    ethMtu = 1024;
  }// if(1024>ethMtu)
  writeData["MTU"] = makeDynString(ethMtu);
  writeData["PartitionId"] = makeDynString(partitionId);
  fwUkl1_write(sysUkl1Name, writeData, TRUE, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1_configure(): Failed to set the partition ID and/or the MTU size during the configure.", "2", exInfoSize);

   // Apply the pixel masks recipe by here when we have some.
  // Don't apply them in the channel configuration as we have to get them from a recipe.
  
  // Done.
  return;
}// fwUkl1_configure()

/*!
 * Start does only two things and that is to enable the TFC decoding on the UKL1 and enable the link from
 * the BE FPGA to the GBE card for data output.
 *
 * @param  ukl1Name Name of the UKL1 board as defined in the DIM server.
 * @param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * @return void
 *
 * This will move the UKL1 to the start state where it is ready to take data. It will not update the state
 * once complete this is upto the User.
 */
void fwUkl1_start(string ukl1Name, dyn_string &exceptionInfo) {
  // Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_start(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }// if(""==ukl1Name)
  
  int exInfoSize = dynlen(exceptionInfo);
  // Holds the registers that can be set with this recipe.
  dyn_string regs;
  // Holds the recipe data.
  dyn_dyn_anytype data;
  
  // Get and apply the recipe.
  // We just hardcode the recipe, as we always use the same one. There is no domain argument given as it is
  // a user recipe and hence the FSMConfDB object will not mangle its name.
  const string recipe = "Start";
  if (-1 == fwHw_getApplyRecipe(ukl1Name, recipe)) {
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_start(): Failed to get and apply the recipe `" + recipe + "'.", "1");
  }// if(-1==fw_getApplyRecipe())

  // Clear any of the soft channel disables before we start.
  //fwUkl1_resetUkl1Partial(ukl1Name, exceptionInfo);
  
  // Done.
  return;
}// fwUkl1_start()

/*!
 * Stop does only two things and that is to disable the TFC decoding on the UKL1 and disable the link from
 * the BE FPGA to the GBE card for data output.
 *
 * @param  ukl1Name Name of the UKL1 board as defined in the DIM server.
 * @param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * @return void
 *
 * This will move the UKL1 to the start state where it will not be able to see triggers sent from the TFC and also
 * is not capable of outputting data. It will not update the state once complete this is upto the User.
 */
void fwUkl1_stop(string ukl1Name, dyn_string &exceptionInfo) {
  // Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_stop(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }// if(""==ukl1Name)
  // Holds the registers that can be set with this recipe.
  dyn_string regs;
  // Holds the recipe data.
  dyn_dyn_anytype data;
  
  // Get and apply the recipe.
  // We just hardcode the recipe, as we always use the same one. There is no domain argument given as it is
  // a user recipe and hence the FSMConfDB object will not mangle its name.
  const string recipe = "Stop";
  if ( -1 == fwHw_getApplyRecipe(ukl1Name, recipe) ) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_stop(): Failed to get and apply the recipe `" + recipe + "'.", "1");
  }// if(-1==fw_getApplyRecipe())

  // Done.
  return;
}// fwUkl1_stop()

/*!
 * Recovers the UKL1 board from an error state. This will return the UKL1 to an unconfigured state and the values in
 * the registers cannot be guarantied to be the same as before recovering. The User should update the FSM state appropriately.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * @return void
 *
 * The interface to this function is unlikely to change, but it might.
 */
void fwUkl1_recover(string ukl1Name, dyn_string& exceptionInfo) {
  // Do something to recover from an error...
  // this is actually quite useful as it is, it allows us to return to NOT_READY without doing anything.
  return;
}// fwUkl1_recover()

/*!
 * Configures the last byte of the source IP address and the fifth byte of the source MAC address from the ID number in the
 * EPROM. It also sets the L1 ID to the least significant byte of the serial number.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 */
void _fwUkl1_configureSourceAddresses(const string& ukl1Name, dyn_string& exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  // Get the logical ID and serial number from the board.
  // These are used to set the source MAC and IP addresses correctly
  // using the logical numbering system and to ensure the UKL1 knows
  // which logical number it is.
  dyn_string regList = makeDynString("LogicalId", "SerialNumber");
  mapping readData = fwUkl1_read(ukl1Name, regList, TRUE, exceptionInfo);
  int logicalId = readData["LogicalId"][FWUKL1_MAP_REG_DATA];
  string serialNumber = readData["SerialNumber"][FWUKL1_MAP_REG_DATA];
  
  // This prefixes the register names.
  const string prefix = ukl1Name + FWUKL1_FPGA_CONFIG_HWTYPE;
  // These are required when writing to the board registers.
  dyn_string regNames;
  dyn_dyn_char data;
  dyn_dyn_char masks;
  
  // Convert the logical number to a string. For the IP this is fine, as it is a decimal number.
  // For the MAC address this is also fine as we wish the MAC address hex numbers to appear visually as the same
  // as the IP address i.e. if the logical number is 10, both the IP and MAC address will appear with 10
  // (192.168.14.10 and 00:BB:CC:DD:10:00) rather than a normal display (192.168.14.10 and 00:BB:CC:DD:0A:00).
  // This was requested by Niko and the Online group.
  const string boardNum = logicalId;
  // Construct the MAC address. Pad the first 4 bytes with 0s and the last byte also.
  // It must have extra 16-bit words added in order to convert the data from 16- to 32-bit word for writing to the registers.
  regNames[1] = prefix + ".MacSourceAddress";
  data[1] = fwCcpc_convertHexToByte("0000" + boardNum + "000000000000000000");
  // Need a mask for a 96-bit word write, 2 32-bit words.
  // The last two 32-bit words (actually last two bytes but 16- to 32-bit funniness comes into play)
  // are reserved. The first of these bytes we must configure, the other must be set to zero and the UKL1 firmware will deal with it.
  masks[1] = makeDynChar(0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00);
  // Now construct the IP address. Need to pad the first 3 bytes with 0s, plus the extra padding to convert from 16- to 32-bit
  // words for the register writes.
  regNames[2] = prefix + ".IpSourceAddress";
  data[2] = fwCcpc_convertHexToByte("000000" + boardNum + "00000000");
  // Need a 48-bit mask for this.
  // The last 8-bits are reserved and not User configurable. These are written what we now write.
  masks[2] = makeDynChar(0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00);
  // Now get the last two characters (byte) from this number for the L1 ID.
  regNames[3] = prefix + ".L1Id";
  string l1Id = substr(serialNumber, strlen(serialNumber)-2, 2);
  fwUkl1_padString(l1Id, 4);
  data[3] = fwCcpc_convertHexToByte(l1Id);
  // Need an 8-bit mask for this.
  masks[3] = makeDynChar(0x00, 0x00, 0x00, 0xff);

  // Now write this data. Use the low level write function, saves a lot of unnecessary function calls.
  // Verify the write.
  _fwUkl1_write(regNames, data, masks, TRUE, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_configureSourceAddresses(): Failed to configure the source IP and MAC addresses, and the L1 ID. Events may not be routed correctly.", "1", exInfoSize);
  
  // Done.
  return;
}// _fwUkl1_configureSourceAddresses()

/*!
 * Configures all the settings on the GBE that are never changed by the User and it calls a function directly on the ccserv to
 * perform this initialisation. It also setups the single TTCrx register that is hardcoded to a UKL1 specific setting.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 */
void _fwUkl1_configureHardcodedSettings(const string& ukl1Name, dyn_string& exceptionInfo) {
  // Used to track the size of the exception info array to see if new exception have been added.
  int exInfoSize = dynlen(exceptionInfo);

  // These are used just to make it a bit clearer in the function calls what we are doing.
  const bool writeRecipe = FALSE;
  dyn_string recipeNameAndType = makeDynString();
  const bool verifyWrite = FALSE;
  
  // Contains the registers and data that are to be written.
  mapping data;

  // Set up the TTCrx control register as desired.
  data["TtcrxControl"] = makeDynString("a1");
  // Add the MDIO enable here also.
  fwUkl1_write(ukl1Name, data, TRUE, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "_fwUkl1_configureHardcodedSettings(): Some of the hardcoded settings have not been set correctly.", "2", exInfoSize);

  // Reset the GBE card via the GPIO line.
  fwUkl1_resetGBE(ukl1Name, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_configureHardcodedSettings(): Failed to reset the GBE card.", "2", exInfoSize);
    
  // Reset the PHY chips first.
  fwUkl1_resetPhyChips(ukl1Name, exceptionInfo);
  fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_configureHardcodedSettings(): Failed to reset the GBE PHY chips, they may not function correct.", "2", exInfoSize);
  
  // Now ask the ccserv to perform the GBE setup routine. Does lots of initialisation that has been hardcoded into the ccserv.
  if (_fwCcpc_commonOperation(ukl1Name, "231", true) != "0")
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_configureHardcodedSettings(): Failed to initialise the GBE it is very unlikely that it will function correctly. There was a problem sending the command to the ccserv.", "2");
  
  // Done.
  return;
}// _fwUkl1_configureHardcodedSettings()

/*!
 * This writes the Ingress input channel configuration settings to the UKL1s.
 *
 * @param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * @param  regs This should contain all the registers that are found in a UKL1_Configure type recipe.
 *           The function will separate out the Channel settings and the other settings returning, by reference,
 *           the other settings that this function does not write. These can then be applied by fwHw_applyRecipe().
 * @param  data This should contain all the data that is retrieved from a recipe and the function will, as for
 *         regs, separate out the data it wants, apply it, and return the remaining data by reference for use with
 *         fwHw_applyRecipe().
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 */
void _fwUkl1_configureIngressChannelSettings(const string& ukl1Name, dyn_string& regs, dyn_dyn_char& data, dyn_string& exceptionInfo) {
  // Keep track of the number of exceptions that have been generated.
  int exInfoSize = dynlen(exceptionInfo);
  // Determine where in the full register list are the registers that are set for channels.
  dyn_bool chanRegPos = patternMatch("*.ConfigurationFifoChannel*", regs);
  if (0 < dynlen(chanRegPos)) {
    // Holds the names of the registers that we wish to apply and the corresponding data.
    dyn_string channelRegs;
    dyn_dyn_char channelData;
    dyn_dyn_char channelMasks;
    // Note where we put the register and data into the arrays.
    int index = 0;
    // Now loop through the returned bool array and determine where the channel data is.
    for (int reg=1; reg<=dynlen(chanRegPos); ++reg) {
      if (chanRegPos[reg]) {
        ++index;
        // Copy the desired register to the channel register names array and then remove the register.
        channelRegs[index] = regs[reg];
        dynRemove(regs, reg);
        dynRemove(chanRegPos, reg);
        // also the data, which must be split in half to give the data and mask.
        dyn_char theData, theMask;
        fwUkl1_splitRecipeData(data[reg], theData, theMask);
        channelData[index] = theData;
        channelMasks[index] = theMask;
        dynRemove(data, reg);
        // Now decrement the reg number.
        --reg;
      }// if(chanRegPos[reg])
      // No need for an else as we want to leave the other register untouched.
    }// for reg
    // We can just apply the settings to the appropriate channel.
    const int numChanRegs = dynlen(channelRegs);
    // Check to see if we matched against anything.
    if (0 == numChanRegs)
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_configureIngressChannelSettings(): Failed to find any configuration settings for the Ingress FPGA channels. These have not been configured.", "2");
    
    // The FALSE is because we are not going to verify the writes. Want recipe application to be as fast as possible.
    // It would take a lot of reads to verify these writes.
//    _fwUkl1_configureIngressFpgaChannel(channelRegs, channelData, channelMasks, FALSE, exceptionInfo);
    _fwUkl1_configureIngressChannels(ukl1Name, channelRegs, channelData, exceptionInfo);
    fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "_fwUkl1_configureIngressChannelSettings(): Encountred a problem when trying to configure the FPGA channel settings for `"+ukl1Name+"'.", "2", exInfoSize);
  }// if(0<dynlen(chanRegPos)
  // In this case we couldn't find any channel configuration registers!
  // Warn people and then apply the full recipe anyway.
  else
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_configureIngressChannelSettings(): Could not find any channel configuration settings in this recipe. No input channel configuration was done.", "2");
  // Done.
  return;
}// _fwUkl1_configureIngressChannelSettings()

/*!
 * Finds the appropriate HPD disable DIM service and subscribes to that service.
 *
 * @param  partitionName Name of the partition we are running in.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return void.
 *
 * This function first unsubscribes all HPD disabling DIM services and then subscribes to the service
 * for the partition that is running.
 *
 * The function should be called only once the UKL1 board has been configured and the partitionId
 * has been set. This information is required to ensure we subscribe to the data from the correct
 * partition.
 */
void fwUkl1_subscribeHPDDisableDimService(const string &partitionName, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  string dimConfig = "MonitoringConfig";
  // For safety, unsubscribe from all HPD Disabling DIM services.
  string searchString = "*_x_RichDAQMon_00/RICHHPDDisableUKL1";  
  fwDim_unSubscribeServices(dimConfig, searchString); 
  // Construct the name of the DIM service we want to subscribe to.
  string serviceName = partitionName + "_x_RichDAQMon_00/RICHHPDDisableUKL1";   
  // Start a loop to attempt to subscribe to the appropriate HPD disabling DIM service.
  const int maxAttempts = 5;  
  const int delayBetweenAttempts = 3;
  for(int attempt = 0; attempt < maxAttempts; attempt++)  
  {
    const int success = fwDim_subscribeService(dimConfig, serviceName, "HPD_Disabling"+FWUKL1_DISBALE_HPD_DPE, "");
    // Returns the number of services subscribed I believe. Not documented so had a browse of the code.
    if (success < 1) 
    {
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_subscribeHPDDisableDimService(): Failed to subscribe to the " + serviceName + "DIM service.", "2");
    }    
    else
    {
      // Set up the call back function for the DPE that will contain the latest value from the DIM service.
      dpConnect(FWUKL1_DISABLE_UKL1_CHANNEL_CB, TRUE, "HPD_Disabling"+FWUKL1_DISBALE_HPD_DPE);
      // We have subscribed to the DIM service and setup the call back function so we can break out of the loop.      
      break;
    }
    //Wait a couple of seconds before trying again.
    delay(delayBetweenAttempts);
  }//for(int attempt)
  // Set the thread ID dpe to -3 showing that the thread has terminated
  dpSet("HPD_Disabling.threadID", -3);
}// fwUkl1_subscribeHPDDisableDimService()

/*!
 * Unsubscribes from the automatic HPD disabling DIM service.
 *
 * @return void.
 *
 * The function should be called in Stop
 */
void fwUkl1_unsubscribeHPDDisableDimService() {
  // Get the partition name, this should have been set in Configure.  
  string partitionName;
  dpGet("HPD_Disabling.partitionID", partitionName);
  string dimConfig = "MonitoringConfig";
  // Construct the name of the service we want to unsubscribe from.  
  string serviceName = partitionName + "_x_RichDAQMon_00/RICHHPDDisableUKL1";
  // We have to pass a dyn_string to the unSubscribe function so lets make one.
  dyn_string serviceNames;  
  dynAppend(serviceNames, serviceName);  
  // Unsubscribe from the automatic HPD disabling service.
  fwDim_unSubscribeServicesNoWild(dimConfig, serviceNames);
} //fwUkl1_unsubscribeHPDDisableDimService()

// @}

// ===================
//  SUPPORT FUNCTIONS
// ===================

/** @defgroup SectionSupport Support funtions
 *     These are miscellaneous functions that provide functionality that does not fit into any of the other categories.
 *  @{
 */

/*!
 * This takes a dyn_mixed where each element should be a list. The list in the first element is sorted
 * with the other lists also being sorted also being sorted in the same order.
 *
 * @param  lists This is the list of dyn types that are to be sorted. They are sorted relative to the
 *           first element of this dyn_mied type.
 * @param  exceptionInfo Error information. No change in size if there is no exception information generated.
 * @return void.
 *
 * The reference list must contain elements that can be compared and the other lists must
 * have at least as many elements. In the event there are too few elements in one list compared to the
 * reference then the function will fail with no sorting having taken place. In there event are too many
 * elements then those extra elements will not be sorted but the function will not fail nor warn of this.
 */
void fwUkl1_sort(dyn_mixed &lists, dyn_string &exceptionInfo) {
  // Just see who is first in the list then place ever function with the same next to it until there
  // are no more, by swapping with the appropriate element. When there are no more move onto the next type.
  // This is the list we are using as the reference for the sorting.
  const int ref = 1;
  // Need to now the number of lists to sort.
  const int numLists = dynlen(lists);
  // Need to know the size of the list we are sorting.
  const int numItems = dynlen(lists[ref]);
  // Start with the first element.
  anytype sortVal;
  // This is the element that we need to put the function in.
  int writeIndex = 1;
  while ( writeIndex < numItems ) {
    // Now we have a function to look for.
    sortVal = lists[ref][writeIndex];
    // Shuffle on one as we want to write matching functions in the next element.
    ++writeIndex;
    // Read all the remaining elements and swap in the occurances of this function.
    // Move on the write index and keep reading the rest of the elements looking for something to swap.
    for (int readIndex = writeIndex; readIndex <= numItems; ++readIndex) {
      if ( sortVal == lists[ref][readIndex] ) {
        // Swap the elements in each of the lists.
        for (int listNum = 1; listNum <= numLists; ++listNum)
          _fwUkl1_swapElements(lists[listNum], writeIndex, readIndex);
        // Now move on our write pointer.
        ++writeIndex;
      }// if (func===writeFunctions[readIndex])
    }// for readIndex
    // Once we have read through once we should move on one to where we will look for our next comparison function.
    ++writeIndex;
  }// while(writeIndex<numItems)
  // Done.
  return;
}// fwUkl1_sort()
  
/*!
 * Retrieves the ethernet MTU setting from the given run info data point. In the event of an error returns -1.
 *
 * @param  runInfo The run info data point to retrieve the value from. The DPE and necessary full stops will be added
 *           by the function.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return int The ethernet maximum transmission unit size as given by the FSM.
 */
int fwUkl1_getMtuFromRunInfo(string runInfo, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  // We will return -1 if everything is not successful.
  int mtu = -1;
  // Save the full path to the DPE.
  const string runInfoMTU = runInfo + ".SubDetectors.ethMTU";
  // Check it exists.
  if ( dpExists(runInfoMTU) ) {
    // Retrieve the value from the DP.
    dpGet(runInfoMTU, mtu);
    // Couldn't get the MTU.
    dyn_errClass error = getLastError();
    if ( 0 != dynlen(error) )
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getMtuFromRunInfo(): Could not retrieve the ethernet MTU size from the run info data point (" + runInfoMTU + "), it must exist at least. More details will be in the PVSS log.", "1");
  }// if(dpExists())
  else
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getMtuFromRunInfo(): Failed to retrieve the ethernet MTU from the run info datapoint as the DPE '" + runInfoMTU + "' did not exist.", "1");
  
  // Done.
  return mtu;
}// fwUkl1_getMtuFromRunInfo()

/*!
 * Retrieves the partition name setting from the given run info data point.
 *
 * @param  runInfo The run info data point to retrieve the value from. The DPE and necessary full stops will be added
 *           by the function.
 * @param  exceptionInfo Error information. No change in size if no exception information is generated.
 * @return string Name of the partition we are currently running in.
 */
string fwUkl1_getPartitionNameFromRunInfo(string runInfo, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  string partition;
  // Save the full path to the DPE.
  const string runInfoPartName = runInfo + ".general.partName";
  // Check it exists.
  if ( dpExists(runInfoPartName) ) {
    // Retrieve the value from the DP.
    dpGet(runInfoPartName, partition);
    // Couldn't get the partition name.
    dyn_errClass error = getLastError();
    if (0 != dynlen(error))
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getPartitionNameFromRunInfo(): Could not retrieve the partition name from the run info data point (" + runInfoPartName + "), it must exist at least. More details will be in the PVSS log.", "1");
  }// if(dpExists())
  else
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getPartitionNameFromRunInfo(): Failed to retrieve the partition name from the run info datapoint as the DPE '" + runInfoPartName + "' did not exist.", "1");
  
  // Done.
  return partition;
}// fwUkl1_getPartitionNameFromRunInfo()

/*!
 * Based on the name of the CU it guess what the system number should be and then asks for the
 * name of that system.
 *
 * @param  cu Name of the control unit for this group of UKL1s.
 * @return string System name with an ending colon or empty string.
 *
 * Basically this function uses the following assumptions
 *   \li RICH1_L1 - This is RICH1 and has a system number of 136.
 *   \li RICH2_L1 - This is RICH2 and has a system number of 176.
 *   \li RICH3_L1 - This is any test system and has a system number of 176.
 *   \li Anything else will be assigned a system number of 0.
 * As the PVSS function getSystemName is used it will return the name with a ending colon.
 * If there is no connection to the guessed system number an empty string will be return, which
 * can be safely used for the system name and it will work provided everything is running on the
 * same local system.
 */
string fwUkl1_guessSystemName(const string &cu) {
  // Holds the guessed system ID.
  int sysId = 0;
  switch (cu) {
    case FWUKL1_RICH1_CU:
      sysId = FWUKL1_RICH1_SYS_NUM;
      break;
    case FWUKL1_RICH2_CU:
      sysId = FWUKL1_RICH2_SYS_NUM;
      break;
    case FWUKL1_RICH3_CU:
      sysId = FWUKL1_RICH3_SYS_NUM;
      break;
    default:
      // Do nothing it was initialised to 0.
      break;
  }// switch(cu)
  // Done.
  return getSystemName(sysId);
}// fwUkl1_guessSystemName()

/*!
 * This takes a recipe name and tries to determine if there is a domain name present in the recipe. If it is
 * found then it is removed and the recipe name is returned in a demanagled state.
 *
 * @param  recipeName Name of the recipe to retrieve the domain from and to demanagle.
 * @return string A string containing the domain name. It is an empty string if no domain name is found.
 *
 * The aim of this function is to take a recipe name that has been managled by the FSMConfDB object and demangle it.
 * The demangling is reversible as the domain name is returned by the function and this is with the demanagled
 * recipe name is all that is required to managle the name such that it is suitable to be used with the FSMConfDB
 * recipes.
 *
 * If the domain cannot be found then an empty string is returned and the recipeName left unchanged. If the recipe
 * name is believed to be valid then no error is raised. If the recipe name is believed to be invalid then
 * the exceptionInfo dyn_string will have information exception information added to it.
 */
string fwUkl1_removeDomainName(string &recipeName, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  string domain = "";
  // Split the recipe name using / as a delimiter.
  const dyn_string splitName = strsplit(recipeName, "/");
  const int splitLen = dynlen(splitName);
  if ( 4 == splitLen ) {
    // In this case return the domain and put the recipe name back in its original form.
    domain = splitName[2];
    for ( int i = 3; i <= splitLen; ++i ) {
      recipeName += splitName[i];
      if ( i != splitLen )
        recipeName += "/";
    }// for i
  }// if domain present.
  // No domain present and the recipe name appears to be invalid.
  else if ( 2 != dynlen(splitName) )
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getDomainNameFromRecipeName(): Failed to find the required number of / delimited sections, 4. Found only " + splitLen + " sections. Cannot retrieve the domain name and the recipe name appears invalid. Given recipe name was " + recipeName + ".", "2");
  // Ignore the case where we have two parts as this is a correctly formed none managled recipe name.

  // Done.
  return domain;
}// fwUkl1_getDomainNameFromRecipeName()

/*!
 * Strips the system name from the given string. Will remove only the system name. If no system name is present
 * then the effects are undefined, unless there are no semi-colons present in the string.
 *
 * @param  dpName String containing the DP name that the system name is to be removed from.
 * @return string dpName minus the beginning system name.
 */
string fwUkl1_removeSystemName(string dpName) {
  // Will hold the stripped DP name, will rely on it being initialised to NULL.
  string strippedDpName = "";

  // First break up the string into its semi-colon parts.
  const dyn_string semiColonDelimited = strsplit(dpName, ':');
  // Determine how many parts we have.
  const int numSec = dynlen(semiColonDelimited);
  if ( 1 == numSec ) {
    // No semi-colons found just return the original string.
    strippedDpName = dpName;
  }// if(1==numSec)
  else if ( 2 <= numSec ) {
    // Start at the second element, bypassing the system name and then append the other elements.
    for (int sec = 2; sec <= numSec; ++sec) {
      strippedDpName += semiColonDelimited[sec];
    }// for sec
  }// elseif(2<=numSec)
  else {
    // Must be no elements, just return the original string which is probably NULL :S
    strippedDpName = dpName;
  }// else

  // Done.
  return strippedDpName; 
}// fwUkl1_removeSystemName()

/*!
 * Strips the rest of the details from a DP name leaving only the system name and a colon.
 * If there is no system name present an empty string is returned.
 *
 * @param  dpName String containing the DP name that the system name is to selected from.
 * @return string The system name with a colon e.g. "R2DAQL1:", empty string is not present.
 */
string fwUkl1_removeDeviceName(string dpName) {
  // Will hold the stripped DP name, will rely on it being initialised to NULL.
  string strippedDpName = "";

  // First break up the string into its semi-colon parts.
  const dyn_string semiColonDelimited = strsplit(dpName, ':');
  // Determine how many parts we have.
  const int numSec = dynlen(semiColonDelimited);
  // Found at least two sections so the first should be the system name and then add the colon. Lost during the splitting.
  if ( 2 <= numSec )
    strippedDpName = semiColonDelimited[1] + ":";

  // Done.
  return strippedDpName; 
}// fwUkl1_removeDeviceName()

/*!
 * Takes the recipe name in the form RUN_MODE/Action and returns the recipe type the recipe should be save as.
 *
 * @param  recipeName The name of the recipe as it is save in the configuration data base.
 * @param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * @return string The recipe type that the recipe should be saved under. An empty string is returned
 *           if the recipe type cannot be determined, along with exception information detailing the problem.
 *
 * The recipe type required is determine from the Action part of the recipe name, if this is not present it
 * cannot be determined. The return value can be used directly in any recipe saving function. The only exception
 * to this are the start and stop recipes, which take the name "Start" and "Stop" respectively. These are user
 * recipes and their type is already known.
 * \n
 * This function works entirely from the recipe name and no gauranties are given that it is the actual type.
 * If the recipe is not named correctly or is currently saved under the wrong type this function cannot
 * know this.
 */
string fwUkl1_getRecipeType(const string& recipeName, dyn_string& exceptionInfo) {
  // Holds the recipe type.
  string recipeType = "";
  // Check to see if we have a name.
  if ("" != recipeName) {
    // Try and split the recipe name using a `/' as a delimiter.
    dyn_string tmpRunModeAndType = strsplit(recipeName, "/");
    // Check to see we have a name and type.
    const int numEle = dynlen(tmpRunModeAndType);
    if (1 < numEle) {
      // It must have at least two elements for it to be of the appropriate form
      // (taking into account the fwFSMConfDB form with many /).
      // The type is always in the last element regardless of form.
      if ("Configure" == tmpRunModeAndType[numEle])
        recipeType = FWUKL1_CONFIGURE_TYPE;
      else if (("Start" == tmpRunModeAndType[numEle]) || ("Stop" == tmpRunModeAndType[numEle]))
        recipeType = FWUKL1_START_TYPE;
      // Only need to mark the error with else as the element is already an empty string as promised.
      else
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getRecipeType(): Could not determine the recipe type from the recipe name `" + recipeName + "'. The action is believed to be `" + tmpRunModeAndType[2] + "'. No recognisable action present in the name (must be of the form RUN_MODE/Action).", "1");
    }// if(1<numEle)
    else if (1 == numEle) {
      // This could be the user recipes Start and Stop, we know these types. Otherwise it is an error.
      if (("Start" == recipeName) || ("Stop" == recipeName))
        recipeType = FWUKL1_START_TYPE;
      else
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getRecipeType(): Could not determine the recipe type from the recipe name `" + recipeName + "'. The recipe name does not appear to cotain an action and it is not recognised `" + recipeName+ "'. No recognisable action present in the name (must be of the form RUN_MODE/Action or called Start or Stop).", "1");
    }// elseif(1==numEle)
    else {
      // Can't do anything with either, mark the error and move on. The return values are already initialised to their error states.
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getRecipeType(): Could not determine the recipe type from the recipe name `" + recipeName + "'. No forward slash, /, present in the name to delimit the run mode and action (must be of the form RUN_MODE/Action).", "1");
    }// else
  }// if(""!=recipeName)
  else {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getRecipeType(): Could not identify recipe type as the recipe name string was empty.", "1");
  }// else(""!=recipeName)
  
  // Done.
  return recipeType;
}// fwUkl1_getRecipeType()

/*!
 * Checks that the given recipe name already exists.
 *
 * @param  recipeName Name of the recipe that is to be checked for.
 * @param  systemName Name of the system to check the recipe cache on.
 * @return bool TRUE if the recipe exists, FALSE if it does not.
 */
bool fwUkl1_checkRecipeExists(const string &recipeName, const string &systemName) {
  // Need to construct the name of the DP that we will look for.
  // If the systemName is not empty then we can look on a remote
  // system, if not then the local system is checked.
  string dpName;
  if ("" != systemName) {
    // The system name must be proceeded by a colon in order for the
    // name to be valid. The user could give us the form with or without
    // so need to put the appropriate punctuation in.
    dpName += systemName;
    if (0 > strpos(dpName,":"))
      dpName += ":";
  }// if(""!=systemName)
  dpName += "RecipeCache/";
  dpName += recipeName;
  
  return dpExists(dpName);
}// fwUkl1_checkRecipeExists()


/*! --- !!!DEPRECIATED!!! ---
 * Updates the library constant FWUKL1_SYS_NUMBER with the value given to this function. If the value given
 * is equal to the current system number no changes are made.
 *
 * @param  sysNum The new number of the PVSS system.
 * @param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * @return void.
 *
 * This function updates the text in this library (const int FWUKL1_SYS_NUMBER = ?;), so it should not be called frequently
 * as it has the potential to corrupt the PVSS code in the file. If the system number set is not the current system number
 * of the PVSS project then all functions that require DP access will likely fail as they can no longer retrieve the system name.
 * The library will have to be reloaded after this function has been called in order for any changes to propagate.
 *
 * This function has been depreciated, it is just no longer needed. The system number can never be set anywhere
 * as the RICH1 and RICH2 libraries have the same name, thus the top level FSM will only ever load them once,
 * thus RICH1 and 2 will always use the same copy and see the same system number. Same goes for globals!
 * It is kept simply because a lot of effort went into writing it!!!!
 */
// void fwUkl1_setSystemNumber(int sysNum, dyn_string &exceptionInfo) {
  // Tracks the exception information.
//   int exInfoSize = dynlen(exceptionInfo);
  // This returns the full path and name of the library we are looking for.
//   const string libPath = getPath(LIBS_REL_PATH, "fwUkl1/fwUkl1.ctl");
  // Attempt to load the file into a string.
//   string sLib;
//   if ( fileToString(libPath, sLib) ) {
    // So we have the file as a rather large string.
    // We don't know the current value of the so ask sscans for it.
//     const string baseStr = "const int FWUKL1_SYS_NUMBER = ";
//     int curSysNum;
    // sscanf doesn't seem to like wildcards at the beginning of the string and hence instead we have to create
    // a substring from the point where the constant definition start onwards.
    // Don't worry about substr and strpos failing as sscanf will indicate an error if they fail.
//     if ( -1 != sscanf( substr(sLib, strpos(sLib, baseStr)), baseStr + "%d;*", curSysNum) ) {
      // There is no need to continue as the system number requested is the same as that to be set.
      // Note there are no file handles to clean up at this point so we can just return.
//       if (sysNum == curSysNum)
//         return;
      // Need to find the occurance of the constant
//       const string searchStr    = "const int FWUKL1_SYS_NUMBER = " + curSysNum + ";";
      // and replace it with this.
//       const string replacement  = "const int FWUKL1_SYS_NUMBER = " + sysNum + ";";
      // Perform the search and check the result.
//       int numReplacements = strreplace(sLib, searchStr, replacement);
//       if ( 1 == numReplacements ) {
        // Now try and write the new version of the library to file.
//         const string tmpPath = libPath + ".tmp";
//         file tmpFile = fopen(tmpPath, "w");
//         if ( 0 == ferror(tmpFile) ) {
          // Now we have opened a temporary file we should write the updated library to it.
//           if ( 0 == fputs(sLib, tmpFile) ) {
            // Flush the file and close it.
//             fflush(tmpFile);
//             fclose(tmpFile);
            // Now copy the temporary file across.
//             if ( !copyFile(tmpPath, libPath) )
//               exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setSystemNumber(): Failed to update the library file fwUkl1.ctl with the new system number. Over write \'" + libPath + "\' with manually \'" + tmpPath + "\'.", "1");
//           }// if(0==fputs())
//           else
//             exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setSystemNumber(): Could not write to the temporary file, aborting update of the system number.", "1");
          // Try and delete the temporary file.
//           if ( -1 == remove(tmpPath) )
//             exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_setSystemNumber(): Could not delete the temporary file \'" + tmpPath + "\'. Please delete it manually.","2");
//         }// if(0==ferror())
//         else
//           exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setSystemNumber(): Could not open a temporary file to save the updated library to. Failed to update the system number.", "1");
//       }// if 1 replacement
//       else if ( 1 < numReplacements )
//         exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setSystemNumber(): Replace more than one entry of the system number in the library. It is likely that the library has been corrupted hence the update has not been saved. Pleasew updated the system number manually.", "1");
//       else
//         exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setSystemNumber(): Could not find current entry of the system number in the file and hence could not update it.", "1");
//     }// if(-1!=sscanf())
//     else
//       exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setSystemNumber(): Failed to identify the current system number from the library. Cannot update the number as we could not find it.", "1");
//   }// if(fileToString())
//   else
//     exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setSystemNumber(): Failed to open the library \'" + libPath + "\' into a string to update the system number. This must be done manually.", "1");
//   
  // Done.
//   return;
// }// fwUkl1_setSystemNumber()

/*! --- !!!DEPRECIATED!!! ---
 * Starts a PVSS manager that is defined by the inputs manager and options. It is based on pmonStartManager(), which is provided
 * by PVSS but does not appear to work from within a PVSS00ctrl manager. The function has been modified to allow this.
 *
 * @param  manager The name of the PVSS manager to be started e.g. PVSS00ui.
 * @param  options The options that are passed to that manager in the PVSS console, sometime refered to as the command.
 * @param  showErrDialog If true PVSS will produce an error dialog in the event that the attempt to start the manager fails (due to an
 *            error connecting to pmon). This cannot work from within a PVSS00ctrl script, only a PVSS00ui script.
 * @param  openTcp Opens a TCP port in order to communicate with pmon if TRUE. This defaults to FALSE in pmonStartManager(),
 *            however it appears to be necessary for this to be TRUE when running inside a PVSS00ctrl manager. Perhaps it works
 *            within a PVSS00ui manager when FALSE.
 * @param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * @return void.
 *
 * This function appears to work, but often still causes errors to be produced in the PVSS log. A fix is being worked on.
 * This function is probably no longer going to be maintained. It is left here simply as a starting point if this is ever
 * needed again... It also now won't work until the system name is properly determined...
 */
// void fwUkl1_startManager(const string &manager, const string &options, bool showErrDialog, bool openTcp, dyn_string &exceptionInfo) {
  // Keeps track of the errors.
//   int exInfoSize = dynlen(exceptionInfo);
//   
  // First we must identify the manager to start, ask the fwInstallation library to figure it out.
  // The postion of the manager that it found, this also doubles as an error code.
  // The function does have a return code but it is couple to an error in the position.
//   int manPos = -1;
//   fwInstallation_getManagerIndex(manager, options, manPos);
//   if ( -1 != manPos ) {
    // It appears that the PVSS console is counting from 0 and getMangerIndex returns the position not in the console
    // but the index in an array of all the entries. Hence it is out by 1!
//     --manPos;
    // Name of the host that is running the PVSS project.
//     string host;
    // Port that pmon is running on the host.
//     int port;
    // Get the host and port details for this project.
//     string sysName = "";// THIS NEEDS TO BE FOUND!!!
//     int iErr = paGetProjHostPort(sysName, host, port);
//     if (!iErr) {
      // Get the username and password for the project.
//       string sUser = "";
//       string sPassword = "";
//       fwInstallation_getPmonInfo(sUser, sPassword);
      // The command to be sent to pmon.
//       string command = "";
//       sprintf(command, "SINGLE_MGR:START %d", manPos);
//       command = sUser+"#"+sPassword+"#"+command;
//   
      // Don't show the error dialog, it can't be used in 
//       if ( pmon_command(command, host, port, showErrDialog, true) )
//         exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_startManager(): Failed to start the manager at position " + manPos + ". Tried to connect to pmon on host \'" + host + "\' on port " + port + ". The manager will have to be started manually.", "1");
        // Start the manager.
//         bool err = FALSE;
//         pmonStartManager(err, fwUkl1_getSystemName(), manPos, sUser, sPassword);
//         if (err) {
//           exInfoSize = fwUkl1Exception_raise(exceptionInfo, "FATAL", "fwUkl1_startManager(): Failed to start the post installation manager. Could not start the post install PVSS panel, this must now be done manually. The FSM will also need to be configured manually as it cannot be setup correctly until the constants have been set.", "1");
//         }// if(err)
//     }// if(!iErr)
//     else {
//       exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_startManager(): Failed to retrieve host and port for the project " + fwUkl1_getSystemName() + ". The manager could not be started and will need to be started manually.", "1");
//     }// else(!iErr)
//   }// if(-1!=pos)
//   else {
//     exInfoSize = fwUkl1Exception_raise(exceptionInfo, "FATAL", "fwUkl1.postInstall(): Failed to identify the post installation manager to start. Could not start the post install PVSS panel, this must now be done manually. The FSM will also need to be configured manually as it cannot be setup correctly until the constants have been set.", "1");
//   }// else(-1!=pos)
// 
  // Done.
//   return;
// }// fwUkl1_startManager()

// @}
