// ======================================
//  NAME    : fwTell1 Library
//
//  VERSION : 1
//
//  REVISION: 0
//
//  RELEASE : 2006/Oct/07
//
//  PLATFORM: PVSS II
//
//  AUTHOR  : Stefan Koestner
//
//  EMAIL   : stefan.koestner@cern.ch
// ======================================


#uses "fwCcpc.ctl"
#uses "fwHw.ctl"
#uses "fwTell1.ctl"



/*! \mainpage fwTell1 Library Documentation
 *
 *  Library of PVSS functions regularly used for the operation of the TELL1-board
 *
 *  Modifications: none
 *
 *  Usage: JCOP framework internal, public
 *
 *  PVSS manager usage: VISION, CTRL
 *
 *  \n
 *  \n
 *  \li \ref SectionConstants
 *  \n
 *  \n
 *  \li \ref SectionArbitrary
 *  \n
 *  \n
 *  \li \ref SectionHardware
*/






// ======================================
//  CONSTANTs & GLOBAL VARs
// ======================================

/** @defgroup SectionConstants Constants
 *  List of constants that can be used by the user.
 *  @{
*/



/** This constant represents nothing.
 *  \n
*/
const int					FWTELL1_NOTHING = 0;



/** This constant represents nothing.
 *  \n
*/
global string				FWTELL1_BOARDTYPE = "NOT_DEF";





/** @} */


//const string			FWCCPC_CONFIG_NAME = "CcpcConfig";
//const string			FWCCPC_PREFIX = "ccpc/";
//const int         FWCCPC_SERVER_VERSION = 220;
//const int					FWCCPC_TIME_OUT = 10;
//global dyn_string fwCcpcList;	
//global int				fwCcpcStateManager;
//global bool				fwCcpcChangeScriptsFlag;







// ======================================
//  ARBITRARY FUNCTIONs
// ======================================

/** @defgroup SectionArbitrary Arbitrary Functions
 *  List of regulary used functions.
 *  @{
*/



/** This function cuts out a bit-array from a hexadecimal string.
 *
 *  \param [in] pHex hexadecimal number (represented as a string) from which an array of bits is cut out
 *  \param [in] pUpper the most significant bit (counted from zero - conform with Guido's ECS docu!)
 *  \param [in] pLower the least significant bit (counted from zero - conform with Guido's ECS docu!)
 *  \return \e the bit array represented as a hexadecimal number (string not integer!)
 *
 *  \b Example: - under construction (returns 0 by default)
 *
 *  \code
 *  (...)
 * // read a register and represent a few adjacent bits out of the full register as a number on the panel
 * dyn_dyn_char data;  
 * dyn_int status;
 * dyn_string registersDP;
 * dynClear(registersDP);
 * dynAppend(registersDP, "dist_1:myTell1_OT.SL.CMNCTRL.SLCtrlReg1.readings"); 
 * fwCcpc_read(registersDP, data, status)
 *  SLCtrlReg1.text = fwTell1_BitsFromString(fwCcpc_convertByteToHex(data[2]),31,24);
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/

string fwTell1_BitsFromString(string pHex, int pUpper, int pLower){
   string casted;
   int number;
   int dummy;
   int mask = 0;
   int i;
 
   number = fwCcpc_convertHexToDec(pHex);
   number = number >> pLower;   
   
   for(i = 0; i <= (pUpper - pLower); i++){
      dummy = 1 << i;
      mask = dummy + mask;
   }
   dummy = number & mask;
   casted = fwCcpc_convertDecToHex(dummy);
   return casted;
}



/** This function cuts out a bit-array from an integer.
 *
 *  \param [in] pNumber an integer number from which an array of bits is cut out
 *  \param [in] pUpper the most significant bit (counted from zero - conform with Guido's ECS docu!)
 *  \param [in] pLower the least significant bit (counted from zero - conform with Guido's ECS docu!)
 *  \return \e the bit array represented as an integer number 
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 * // read a register and represent a few adjacent bits out of the full register as a number on the panel
 * dyn_dyn_char data;  
 * dyn_int status;
 * dyn_string registersDP;
 * dynClear(registersDP);
 * dynAppend(registersDP, "dist_1:myTell1_OT.SL.CMNCTRL.SLCtrlReg1.readings"); 
 * fwCcpc_read(registersDP, data, status)
 *  SLCtrlReg1.text = fwTell1_BitsFromString(fwCcpc_convertByteToDec(data[2]),31,24);
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/

int fwTell1_BitsFromInt(int pNumber, int pUpper, int pLower){
   int dummy;
   int mask = 0;
   int i;
   pNumber = pNumber >> pLower;   
   
   for(i = 0; i <= (pUpper - pLower); i++){
      dummy = 1 << i;
      mask = dummy + mask;
   }
   dummy = pNumber & mask;
   return dummy;  
}






/** This function is used currently by Cedric and will be implemented propperly soon.
 *
 *  \param [in] imput decimal number to be converted.
 *  \param [in] total decimal number to be converted.
 *  \return \e string hexadecimal number.
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  string result;
 *
 *  result = fwCcpc_convertDecToHex(1127);   // convert decimal number '1127' to the corresponding hexadecimal number
 *  DebugTN("Hexadecimal: " + result);   // output the result of the conversion
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/

string complStr(string imput, int total)
{	
	 string zeros ;
		if(strlen(imput) != total){
	    for(int h = 1 ; h <= total - strlen(imput); h++){
		       
		    zeros = zeros + "0";
		 }
    }
		
	  string output = zeros + imput;
		
		return output;
}



/** @} */






// ======================================
//  COMMON HARDWARE FUNCTIONs
// ======================================

/** @defgroup SectionHardware Hardware Functions
 *  List of common functions operating with the Tell1 board.
 *  @{
*/



/** TELL1 boards of different subdetectors can be recognised by reading various bits 
 *  of a special register. This function allows to identify the board type without previously
 *  subscribing to this register. The global variable FWTELL1_BOARDTYPE can be used to store the
 *  returned string which is a short acronym for the board type. By default this variable is "NOT_DEF".
 *
 *  \param [in] CCPC name of the credit card PC.
 *  \param [out] status represents the return value of the read function used inside the code.
 *  \return \e string returns the abriations of the board type
 *
 *  \b Example:
 *  
 *  \code
 *  (...)
 *  int status;
 *  if (FWTELL1_BOARDTYPE == "NOT_DEF")
 *     FWTELL1_BOARDTYPE = fwTell1_TypeOfBoard("pctell07",status);
 *  if (status == 0)
 *     DebugN("Reading of register successfull!");
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/

string fwTell1_TypeOfBoard(string CCPC, int &status){
   dyn_char data;
   string dummy;
   int address;
   string TypeOfBoard;

   address = fwCcpc_convertHexToDec("4001000");
   status = fwCcpc_LBUSRead(CCPC, address, FWCCPC_LBUS_BITS_32,1, data);
   if(status == 0){
      dummy = fwTell1_BitsFromString(fwCcpc_convertByteToHex(data),7,4);
      //DebugTN("***********dummy:");
      //DebugN(dummy);      
      if(dummy == "1"){
         TypeOfBoard = "VELO";
      }   
      else if(dummy == "2"){
         TypeOfBoard = "ST";
      }
      else if(dummy == "3"){
         TypeOfBoard = "OT";
      }   
      else if(dummy == "4"){
         TypeOfBoard = "EHCAL";
      }
      else if(dummy == "5"){
         TypeOfBoard = "MUON";
      }
      else if(dummy == "6"){
         TypeOfBoard = "L0MUON";
      }
      else if(dummy == "7"){
         TypeOfBoard = "L0DU";
      } 
      else if(dummy == "8"){
         TypeOfBoard = "L0PUS";
      }
      else if(dummy == "9"){
         TypeOfBoard = "RICH";
      }
       else if(dummy == "A"){
         TypeOfBoard = "PSSPD";
      }
      else {
         TypeOfBoard = "NOT_DEF";
      }     
   } else{
        DebugTN("Warning: LBUSRead failed!");
        TypeOfBoard = "ERR";
   }
   //DebugTN(TypeOfBoard);
   return TypeOfBoard;
}








/** This function sets bit Nr.1 in the MDIO-Control register (0x683) of the MAC chip to enable/disable the austoscan feature. 
 *  The flag 'true' (=default) enables the autoscan, 'false' will disable the autoscan.
 *  It checks if the WriteRead function terminated successfully and compares the written bit with the value that was read back.
 *
 *  \param [in] CCPC name of the credit card PC.
 *  \param [in] pFlag to enable or disable autoscan feature of MAC chip.
 *  \return \e int the errorcode of this library: 0 if no error occured
 *
 *  \b Example:
 *  \code
 *  (...)
 *  string CCPC;
 *  flag = true;
 *  //get the ccpc of the Tell1 and enable the autoscan feature
 *  dummy = DeviceUnitDP + ".settings";
 *  dpGet(dummy,CCPC);
 *  fwTell1_enableMDIOCtrl(CCPC,true);
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/

int fwTell1_enableMDIOCtrl(string CCPC, bool pFlag = true){
   int error = 0;
   dyn_char data;
   dyn_char mask;
dyn_char dataRead;
int dummy;
int bit;

   int address = fwCcpc_convertHexToDec("683");
   if(pFlag){
      data = fwCcpc_convertHexToByte("00000002");
      mask = fwCcpc_convertHexToByte("00000002");
   }
   else{
      dyn_char data = fwCcpc_convertHexToByte("00000000");
      dyn_char mask = fwCcpc_convertHexToByte("00000002");
   }  
   
   if(fwCcpc_GBEWriteRead(CCPC, address, 1, data, mask, dataRead )){
      DebugTN("Warning: GBEWrite failed!");
      error |= 1;
   }     
   else{
      error = 0;
   }


 dummy = fwCcpc_convertByteToDec(dataRead);
 bit = fwTell1_BitsFromInt(dummy,1,1);


 DebugTN("MDIO Autoscan - Bit value: ");
 DebugN(bit); 



 if(pFlag){
    if(bit != 1){
        error |= 1;
        DebugTN("Error when reading back register - MDIO Autoscan may be not set!");
    }
 }
 else{ 
   if(bit != 0){
        error |= 1;
        DebugTN("Error when reading back register - MDIO Autoscan may be not disabled!");
   }
 }

   return error;
}





/** This function loads the FPGA from EPC16 (just for TELL1).
 *
 *  \param [in] pCCPC name of the CCPC.
 *  \return \e true no error found.
 *  \return \e false error found.
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  if (fwTell1_EPC16ToFPGA("pclbcecs03") == 0)   // load the FPGA from EPC16
 *     (...)
 *  else
 *     DebugTN("Error found!");
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/
bool fwTell1_EPC16ToFPGA(string pCCPC)
{

	if (_fwCcpc_commonOperation(pCCPC, "5", true) == "-1")
		return false;		// ~OK
	else
		return true;		// OK	
		
}







/** Checks if the .cfg file was written propperly by evaluating a part of a certain register .
 *
 *  \param [in] CCPC name of the credit card PC.
 *  \return \e int the errorcode of this library: 0 if no error occured
 *                                               -1 LBUS access failed
 *                                                or positive value (5-bit number)
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  string ccpc = "pctell07";
 *
 *  if(fwTell1_chkCFG(ccpc))
 *     DebugTN("FPGAs are not configured propperly!"
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/

int fwTell1_chkCFG(string CCPC){
   int status = 0;
   int error = 0;
   dyn_char data;
   int address;
   int myhex;

   address = fwCcpc_convertHexToDec("1000004");   
   if(fwCcpc_LBUSRead(CCPC, address, FWCCPC_LBUS_BITS_32, 1, data)){
      DebugTN("Warning: LBUSRead failed!");
      return -1;
   }      

   myhex = fwTell1_BitsFromInt(fwCcpc_convertByteToDec(data), 15, 11);
   DebugTN("Check CFG: Returned 5-bit number (hex):");
   DebugN(fwCcpc_convertDecToHex(myhex));
 
   if( myhex == 0x1F){
      //FPGA init OK
      return 0;
   }else{
      //FPGA init NOT OK
      return myhex;
   }
}




/** Checks if the CCPC is running by checking the invalid bit of the 'common.operationStatus' service.
 *
 *  \param [in] CCPC name of the credit card PC.
 *  \return \e int the errorcode of this library: 0 if no error occured 
 *                                                1 requested CCPC was not found
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  string ccpc = "pctell07";
 *
 *  if(fwTell1_chkCCPC(ccpc))
 *     DebugTN("Checked CCPC is not running!"
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/



int fwTell1_chkCCPC(string CCPC){
 
   //dyn_string tmp0;
   //dyn_string tmp1;
   //dynClear(tmp0);
   //dynClear(tmp1);
   //int i;	
   //CcpcOn = 0
   bool invalidBit;

//a temporary hack to set the invalid bit - later service is published immediately on start up
   
   int a, b, c; 
   fwCcpc_getServerVersion(CCPC, a, b, c); 
   DebugTN(a, b, c); 


   DebugTN("Check if server is connected!");
   dpGet("CCPC/"+CCPC+".common.operationStatus:_original.._invalid",invalidBit);
   //DebugTN(invalidBit);
   if(invalidBit == true){
      DebugTN("Server "+CCPC+" is down!");
      return 1;
   } else{
      DebugTN("Server "+CCPC+" found!");
      return 0;
   }
}
   


/*
   DebugTN("chkCCPC - looking for command:"); 
   if (isFunctionDefined("fwDim_createConfig")){
         fwDim_findCommands(FWCCPC_CONFIG_NAME, FWCCPC_PREFIX  + CCPC + "/CmndCommonOperation", tmp0, tmp1);	
                  DebugN(FWCCPC_CONFIG_NAME, FWCCPC_PREFIX + CCPC + "/CmndCommonOperation");

      if(dynlen(tmp0) == 1){
         DebugTN("ccpc found!");
         return 0;}
      else{
	 DebugTN("ccpc not found!");
         return 1;}	
    }
*/




/** Checks if GBE is operational - currently under construction (returns 0 by default!)
 *
 *  \return \e int the errorcode of this library: 0 if no error occured
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  dyn_string myList;
 *  int        i;
 *
 *  fwCcpc_get(myList);   // get a list of CCPC servers running in the moment
 *  DebugTN("CCPC servers: " + dynlen(myList));   // display number of CCPC servers running in the moment
 *  for(i = 1; i <= dynlen(myList); i++)
 *     DebugTN(myList[i]);
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/


int fwTell1_chkGBE(string Tell1Name, string CCPC){
   int status = 0;
   int status2 = 0;
   
   fwTell1_enableMDIOCtrl(CCPC); 

   
   dyn_dyn_char data;  
   dyn_int statusG;
   dyn_string registers;
   dyn_string registersDP;
  

   
   dynClear(registers);
   dynClear(registersDP);
   
   dynAppend(registers, "SL.CMNCTRL.Spi3TxCtrlReg");  
   dynAppend(registers, "GBE.PORT0.PHYStatus");  
   dynAppend(registers, "GBE.PORT1.PHYStatus");  
   dynAppend(registers, "GBE.PORT2.PHYStatus");  
   dynAppend(registers, "GBE.PORT3.PHYStatus"); 
   
    
   for(int i = 1; i <= dynlen(registers); i++){
      dynAppend(registersDP, Tell1Name + "." + registers[i]); 
   }
   
   if(fwCcpc_read(registersDP, data, statusG)) {
   

   dyn_int dummy;
   for(int i = 1; i<= dynlen(data); i++)
       dummy[i] = fwCcpc_convertByteToDec(data[i]);
  
   for(int j = 1; j<=4;j++)  
     if((fwTell1_BitsFromInt(dummy[j+1],2,2) != 1)  && (fwTell1_BitsFromInt(dummy[1],3+j,3+j) == 1))
       status = 1;
      //DATA LOSS, the port is not plugged, but the link bwt sl and gbe is enabled

   for(int j = 1; j<=4;j++)  
     if((fwTell1_BitsFromInt(dummy[1],3+j,3+j) == 0))
       status2 += 1;
   
   if(status2 == 4)
     status =2;
   
   
   return status;
   
 }
   else{
     
       DebugN("Warning: GBERead failed!");
       return 1; 
     
   }

}




/** Checks if TTC is operational (fibre is plugged). If fibre is unplugged the I2C interface is pertubated. This function is reading 3-times from the status register. In addition the last two bits from the status register are checked.
 *
 *  \param [in] CCPC name of the CCPC.
 *  \return \e int the errorcode of this library: 0 if no error occured
 *                                                1  clock or data recovery circuit not locked on incoming data stream
 *                                                2  high-resolution phase shifters not working propperly!
 *                                                3  both bits are set
 *                                               -1  I2C-read of status register failed (minimum once out of three trials)
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  string ccpc = "pctell07";
 *
 *  if(fwTell1_chkTTC(ccpc))
 *     DebugTN("WARNING: Fibre might not be plugged on TTC!"
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/

int fwTell1_chkTTC(string CCPC){
   int i;
   int error = 0;
   dyn_char data;
   int dllReady;
   //indicates that the high-resolution phase shifters are working propperly
   int pllReady;
   //indicates that the clock and data recovery circuit is locked on the incoming data stream
   
   //int address = fwCcpc_convertHexToDec("58");
   int address = 0x58;
   for(i=1; i <= 3; i++){
      if(fwCcpc_I2CReadSub(CCPC, 0, address, 22, 1, -1, FWCCPC_I2C_SEPARATED, 0, data)){
         DebugN("Warning: TTCRead failed!");
         error += 1;
      }     
   }
   if(error){
      DebugTN("Could not read I2C interface - check clock and fibre of TTCRx!");
      return -1;
   }
      
   error = 0;
   pllReady = fwTell1_BitsFromInt(fwCcpc_convertByteToDec(data), 7, 7);
   dllReady = fwTell1_BitsFromInt(fwCcpc_convertByteToDec(data), 6, 6);
   if((pllReady == 0)){
      DebugTN("TTC: Clock or data recovery circuit not locked on incoming data stream!");
      error += 1;
   }
   if(dllReady==0){
      DebugTN("TTC: High-resolution phase shifters not working propperly!");
      error += 2;
   }
   return error;
}





/** Checks if ORx is operational - under construction (returns 0 by default)
 *
 *  \return \e int the errorcode of this library: 0 if no error occured
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  dyn_string myList;
 *  int        i;
 *
 *  fwCcpc_get(myList);   // get a list of CCPC servers running in the moment
 *  DebugTN("CCPC servers: " + dynlen(myList));   // display number of CCPC servers running in the moment
 *  for(i = 1; i <= dynlen(myList); i++)
 *     DebugTN(myList[i]);
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/


int fwTell1_chkORx(string Tell1Name){
   int status = 0;
   int disableLink = 0; 

   dyn_dyn_char data;  
   dyn_int statusG;
   dyn_string registers;
   dyn_string registersDP;
  
   string dummy1;
   string dummy2;
   string dummy3;
   
   dynClear(registers);
   dynClear(registersDP);
   
   dynAppend(registers, "PP0.ORX.link0");  
   dynAppend(registers, "PP0.ORX.link1");  
   dynAppend(registers, "PP0.ORX.link2");  
   dynAppend(registers, "PP0.ORX.link3");  
   dynAppend(registers, "PP0.ORX.link4");  
   dynAppend(registers, "PP0.ORX.link5");  
   dynAppend(registers, "PP1.ORX.link0");  
   dynAppend(registers, "PP1.ORX.link1");  
   dynAppend(registers, "PP1.ORX.link2");  
   dynAppend(registers, "PP1.ORX.link3");  
   dynAppend(registers, "PP1.ORX.link4");  
   dynAppend(registers, "PP1.ORX.link5");  
   dynAppend(registers, "PP2.ORX.link0");  
   dynAppend(registers, "PP2.ORX.link1");  
   dynAppend(registers, "PP2.ORX.link2");  
   dynAppend(registers, "PP2.ORX.link3");  
   dynAppend(registers, "PP2.ORX.link4");  
   dynAppend(registers, "PP2.ORX.link5");  
   dynAppend(registers, "PP3.ORX.link0");  
   dynAppend(registers, "PP3.ORX.link1");  
   dynAppend(registers, "PP3.ORX.link2");  
   dynAppend(registers, "PP3.ORX.link3");  
   dynAppend(registers, "PP3.ORX.link4");  
   dynAppend(registers, "PP3.ORX.link5"); 
   
    
   for(int i = 1; i <= dynlen(registers); i++){
      dynAppend(registersDP, Tell1Name + "." + registers[i]); 
   }
   
   if(fwCcpc_read(registersDP, data, statusG)){

   

   for(int i = 1; i <=24; i++){
   //fwCcpc_read ....
     
    
    if (fwTell1_BitsFromInt(fwCcpc_convertByteToDec(data[i]), 23, 23) != "0"){  
           disableLink++;
	  }else if(fwTell1_BitsFromInt(fwCcpc_convertByteToDec(data[i]), 14, 14) == "0" && fwTell1_BitsFromInt(fwCcpc_convertByteToDec(data[i]), 12, 12) == "0"){  
           status++;
    }
 
   }
   
   DebugTN("   "+disableLink +" ORx link disabled");
   int en = 24 - disableLink;
   DebugTN("   "+en +" ORx link enabled and "+status+" are(is) in error state");
   if(disableLink ==  24)     
      return 2;
   if(status !=  0)     
      return 1;
   
   
   return 0; 
   } else{
      
       DebugN("Warning: ORxRead failed!"); 
   }
   return 1;
}


/** Checks if ARx is operational - under construction (returns 0 by default)
 *
 *  \return \e int the errorcode of this library: 0 if no error occured
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  dyn_string myList;
 *  int        i;
 *
 *  fwCcpc_get(myList);   // get a list of CCPC servers running in the moment
 *  DebugTN("CCPC servers: " + dynlen(myList));   // display number of CCPC servers running in the moment
 *  for(i = 1; i <= dynlen(myList); i++)
 *     DebugTN(myList[i]);
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/

int fwTell1_chkARx(string Tell1Name){
   int status = 0; 
   
   
   
   dyn_dyn_char data;  
   dyn_int statusG;
   dyn_string registers;
   dyn_string registersDP;
   dyn_int dummy;
   dyn_int limit;

   
   dynClear(registers);
   dynClear(registersDP);
   
  dynAppend(registers, "ARX.DACPP0.BurstWrite_Data");  //freeze the cnts
  dynAppend(registers, "ARX.DACPP1.BurstWrite_Data");      
  dynAppend(registers, "ARX.DACPP2.BurstWrite_Data");      
  dynAppend(registers, "ARX.DACPP3.BurstWrite_Data");      
   
   
    
   for(int i = 1; i <= dynlen(registers); i++){
      dynAppend(registersDP, Tell1Name + "." + registers[i]); 
   }
   
  fwCcpc_read(registersDP, data, statusG);
   
  if(statusG[1] == 4){
   status += 1; 
   DebugTN("DAC 0 not plugged");
 }
   
 if(statusG[2] == 4){
   status += 2;
    DebugTN("DAC 1 not plugged");
 }
 if(statusG[3] == 4){
   status += 4;
    DebugTN("DAC 2 not plugged");
 }
 if(statusG[4] == 4) {
   status += 8; 
   DebugTN("DAC 3 not plugged");
 } 

       

            

  
 return status;
}









/** Checks if processing on Tell1 board is ok  - under construction (returns 0 by default)
 *
 *  \return \e int the errorcode of this library: 0 if no error occured
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  dyn_string myList;
 *  int        i;
 *
 *  fwCcpc_get(myList);   // get a list of CCPC servers running in the moment
 *  DebugTN("CCPC servers: " + dynlen(myList));   // display number of CCPC servers running in the moment
 *  for(i = 1; i <= dynlen(myList); i++)
 *     DebugTN(myList[i]);
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/


int fwTell1_chkProcessing(string Tell1Name){
   int status = 0; 
   
   
   
   dyn_dyn_char data;  
   dyn_int statusG;
   dyn_string registers;
   dyn_string registersDP;
   dyn_int dummy;
   dyn_int limit;

   
   dynClear(registers);
   dynClear(registersDP);
   
  dynAppend(registers, "SL.REALTIME.SLTrigCntReg");  //freeze the cnts
  dynAppend(registers, "SL.CMNMON.SLEvtInCntReg0");      
  dynAppend(registers, "SL.CMNMON.SLEvtInCntReg1");      
  dynAppend(registers, "SL.CMNMON.SLEvtInCntReg2");      
  dynAppend(registers, "SL.CMNMON.SLEvtInCntReg3");      
   
    
   for(int i = 1; i <= dynlen(registers); i++)
      dynAppend(registersDP, Tell1Name + "." + registers[i]);   
   
 if(fwCcpc_read(registersDP, data, statusG)){  
   
  if(fwCcpc_convertByteToHex(data[2]) == fwCcpc_convertByteToHex(data[3]) == fwCcpc_convertByteToHex(data[4]) == fwCcpc_convertByteToHex(data[5]))
    status = 0;
   
  }else{
     DebugN("Warning: PROCCESSINGRead failed!");
     status = 1; 
     
   }
   
   
   
   return status;
}




/** Checks if data flow on Tell1 board is ok  - under construction (returns 0 by default)
 *
 *  \return \e int the errorcode of this library: 0 if no error occured
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  dyn_string myList;
 *  int        i;
 *
 *  fwCcpc_get(myList);   // get a list of CCPC servers running in the moment
 *  DebugTN("CCPC servers: " + dynlen(myList));   // display number of CCPC servers running in the moment
 *  for(i = 1; i <= dynlen(myList); i++)
 *     DebugTN(myList[i]);
 *  (...)
 *  \endcode
 *(string Tell1Name){
 *  \n
 *  \n
*/



int fwTell1_chkFlow(string Tell1Name){
   int status = 0;
   
   
   //chk if throttle ae coming
   
   

   dyn_dyn_char data1, data2;  
   dyn_int statusG;
   dyn_string registers;
   dyn_string registersDP;


   
   dynClear(registers);
   dynClear(registersDP);
   
  dynAppend(registers, "SL.CMNMON.ThroCntReg0");  

   
    
   for(int i = 1; i <= dynlen(registers); i++){
      dynAppend(registersDP, Tell1Name + "." + registers[i]); 
   }
   
 if(fwCcpc_read(registersDP, data1, statusG)){  
  delay(0,100);
  if(fwCcpc_read(registersDP, data2, statusG)){   
   
  
  if (fwCcpc_convertByteToHex(data1[1]) != fwCcpc_convertByteToHex(data2[1]))
    status =1;
  
  
   
   return status;
   
   
   
   }else{
     
       DebugN("Warning: GBERead failed!");
       return 1; 
     
   }
   }else{
     
       DebugN("Warning: GBERead failed!");
       return 1; 
     
   }
}



/** Checks if memory on Tell1 board is ok  - under construction (returns 0 by default)
 *
 *  \return \e int the errorcode of this library: 0 if no error occured
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  dyn_string myList;
 *  int        i;
 *
 *  fwCcpc_get(myList);   // get a list of CCPC servers running in the moment
 *  DebugTN("CCPC servers: " + dynlen(myList));   // display number of CCPC servers running in the moment
 *  for(i = 1; i <= dynlen(myList); i++)
 *     DebugTN(myList[i]);
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/



int fwTell1_chkMemory(string Tell1Name){
   int status = 0;
   

   
   
   dyn_dyn_char data;  
   dyn_int statusG;
   dyn_string registers;
   dyn_string registersDP;
   dyn_int dummy;
   dyn_int limit;

   
   dynClear(registers);
   dynClear(registersDP);
   
  dynAppend(registers, "SL.CMNMON.SLTriggerFifoUsedReg0");  
  dynAppend(registers, "SL.CMNMON.SLTriggerFifoUsedReg1");      
  dynAppend(registers, "SL.REALTIME.TrigInfoFifoMonReg0");  
  dynAppend(registers, "SL.REALTIME.TrigInfoFifoMonReg1");   
  dynAppend(registers, "SL.CMNMON.MepMaxUsageReg");  
  dynAppend(registers, "SL.CMNMON.SLMaxUseReg");  
  dynAppend(registers, "SL.CMNMON.FramerMaxUseReg");  
  dynAppend(registers, "SL.CMNMON.MepGt16KCntReg");     
  dynAppend(registers, "PP0.CMNMON.MemMaxUsageReg0");  
  dynAppend(registers, "PP1.CMNMON.MemMaxUsageReg0");  
  dynAppend(registers, "PP2.CMNMON.MemMaxUsageReg0");
  dynAppend(registers, "PP3.CMNMON.MemMaxUsageReg0");  
  dynAppend(registers, "PP0.CMNMON.MemMaxUsageReg1");
  dynAppend(registers, "PP1.CMNMON.MemMaxUsageReg1");  
  dynAppend(registers, "PP2.CMNMON.MemMaxUsageReg1");  
  dynAppend(registers, "PP3.CMNMON.MemMaxUsageReg1");  

   
    
   for(int i = 1; i <= dynlen(registers); i++){
      dynAppend(registersDP, Tell1Name + "." + registers[i]); 
   }
   
   if(fwCcpc_read(registersDP, data, statusG)){  
   
   
  for(int i = 1; i <= dynlen(data); i++)
    dummy[i] = fwCcpc_convertByteToDec(data[i]);
   
 
  limit[1] = fwCcpc_convertHexToDec("1F");   //5bit
  limit[2] = fwCcpc_convertHexToDec("3F");    //7bit 
  limit[3] = fwCcpc_convertHexToDec("FF");    //8bit  
  limit[4] = fwCcpc_convertHexToDec("3FF");    //10bit
  limit[5] = fwCcpc_convertHexToDec("7FF");     //11bit
  limit[6] = fwCcpc_convertHexToDec("FFFF");    //16bit  
  limit[7] = fwCcpc_convertHexToDec("7FFFF");   //19bit   
  limit[8] = fwCcpc_convertHexToDec("FFFFFFFF");    //32bit   
   
  
  if(fwTell1_BitsFromInt(dummy[3],31,24) == limit[3])
    status ++;
  if(fwTell1_BitsFromInt(dummy[3],23,16) == limit[3])
    status ++;
  if(fwTell1_BitsFromInt(dummy[3],15,8) == limit[3])
    status ++;
  if(fwTell1_BitsFromInt(dummy[4],31,24) == limit[3])
    status ++;
  if(fwTell1_BitsFromInt(dummy[4],23,16) == limit[3])
    status ++;
  
  for(int j = 1; j<=4; j++){
  if(fwTell1_BitsFromInt(dummy[8+j],31,16) == limit[6])
    status ++;
  if(fwTell1_BitsFromInt(dummy[8+j],15,0) == limit[6])
    status ++;
  if(fwTell1_BitsFromInt(dummy[12+j],29,20) == limit[4])
    status ++;
  if(fwTell1_BitsFromInt(dummy[12+j],19,10) == limit[4])
    status ++;
  if(fwTell1_BitsFromInt(dummy[12+j],9,0) == limit[4])
   status ++;

 }
  
  
  if(fwTell1_BitsFromInt(dummy[5],18,0) == limit[7])
    status ++;
  if(fwTell1_BitsFromInt(dummy[6],9,0) == limit[4])
    status ++;
  if(fwTell1_BitsFromInt(dummy[6],20,10) == limit[5])
    status ++;
  if(fwTell1_BitsFromInt(dummy[6],27,21) == limit[2])
    status ++;
  if(fwTell1_BitsFromInt(dummy[7],4,0) == limit[1])
    status ++;
  if(fwTell1_BitsFromInt(dummy[7],11,5) == limit[2])
    status ++;
  if(fwTell1_BitsFromInt(dummy[7],19,12) == limit[3])
    status ++;
  if(dummy[8]!=0)
    status ++;

  
   return status;
   
   
   }else{
     
       DebugN("Warning: GBERead failed!");
       return 1; 
     
   }
}



/** Checks if data on Tell1 board is ok  - under construction (returns 0 by default)
 *
 *  \return \e int the errorcode of this library: 0 if no error occured
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  dyn_string myList;
 *  int        i;
 *
 *  fwCcpc_get(myList);   // get a list of CCPC servers running in the moment
 *  DebugTN("CCPC servers: " + dynlen(myList));   // display number of CCPC servers running in the moment
 *  for(i = 1; i <= dynlen(myList); i++)
 *     DebugTN(myList[i]);
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/



int fwTell1_chkData(){
   int status = 0;
   return status;
}











/** Checks if Triggers are sent to Tell1 board. Accessing the board e.g. applying recipes should be prohibitted then.
 *
 *  \return \e int return 1 if triggers are applied and 0 if not. -1 if an error occured.
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  if(!chk_Triggers("myTell1Board")
 *     fwHw_applyRecipe(DeviceUnitDP_noDist, regs, data);
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/


int fwTell1_chkTriggers(string Tell1Name){
  
   dyn_dyn_char data;  
   dyn_int status;
   dyn_string registers;
   dyn_string registersDP;
 
   int dummy;
   
   //dynClear(registers);
   dynClear(registersDP);   
  // dynAppend(registers, "SL.CMNMON.RealRateReg1.readings");   
   
   //for(int i = 1; i <= dynlen(registers); i++)
   dynAppend(registersDP, Tell1Name + "." + "SL.CMNMON.RealRateReg1");   
   
   if(fwCcpc_read(registersDP, data, status)){  
   
      dummy = fwCcpc_convertByteToDec(data[1]);   
      if (dummy != 0){
        // setValue("attention1","visible",true);
        // setValue("daqStat","visible",true);
        // setValue("apply","enabled",false);
        return 1;
      }
      else{
        //setValue("attention1","visible",false);
        //setValue("daqStat","visible",false);
        //setValue("apply","enabled",true);
        return 0; 
      }
 
   }
   else   
  return status;
}












/** @}*/










// ======================================
//  INTERNAL FUNCTIONs (NOT TO BE USED)
// ======================================


/*
string _fwCcpc_getConfigName()
{

	return FWCCPC_CONFIG_NAME;
	
}




string _fwCcpc_getPrefix()
{

	return FWCCPC_PREFIX;
	
}
*/













 
 
//**********************NEW FUNCTION******************************** 
// 
//   functions that are calling hard-coded sequences on the server 
//   this is needed for e.g. configuring the Tell1 board 
 
 
 
 
 
 
 
/** This function initializes the glue-card  
 * 
 *  \param [in] pCCPC name of the CCPC. 
 *  \return \e int the errorcode of this library: 0 if no error occured 
 * 
 *  \b Example: 
 * 
 *  \code 
 *  (...) 
 *  if (fwTell1_iniGlueCard("pclbcecs03") == 0)  
 *     (...) 
 *  else 
 *     DebugTN("Error found!"); 
 *  (...) 
 *  \endcode 
 * 
 *  \n 
 *  \n 
*/ 
int fwTell1_iniGlueCard(string pCCPC) 
{ 
 
	if (_fwCcpc_commonOperation(pCCPC, "20", true) == "-1") 
		return -1;		// ~OK 
	else 
		return 0;		// OK	 
		 
} 
 
 
 
 
 
 
 
 
 
 
/** This function resets the Tell1 board by pulling some GPIO lines  
 * 
 *  \param [in] pCCPC name of the CCPC. 
 *  \return \e int the errorcode of this library: 0 if no error occured 
 * 
 *  \b Example: 
 * 
 *  \code 
 *  (...) 
 *  if (fwTell1_resetTell1("pclbcecs03") == 0)  
 *     (...) 
 *  else 
 *     DebugTN("Error found!"); 
 *  (...) 
 *  \endcode 
 * 
 *  \n 
 *  \n 
*/ 
int fwTell1_resetTell1(string pCCPC) 
{ 
 
	if (_fwCcpc_commonOperation(pCCPC, "21", true) == "-1") 
		return -1;		// ~OK 
	else 
		return 0;		// OK	 
		 
} 
 
 
 
 
 
 
 
/** This function sets a bit in SL_RESET_REG to reset the TELL1 board  
 * 
 *  \param [in] pCCPC name of the CCPC. 
 *  \return \e int the errorcode of this library: 0 if no error occured 
 * 
 *  \b Example: 
 * 
 *  \code 
 *  (...) 
 *  if (fwTell1_resetTell1Soft("pclbcecs03") == 0)  
 *     (...) 
 *  else 
 *     DebugTN("Error found!"); 
 *  (...) 
 *  \endcode 
 * 
 *  \n 
 *  \n 
*/ 
int fwTell1_resetTell1Soft(string pCCPC) 
{ 
 
	if (_fwCcpc_commonOperation(pCCPC, "22", true) == "-1") 
		return -1;		// ~OK 
	else 
		return 0;		// OK	 
		 
} 
 
 
 
 
 
 
 
 
/** This function initializes the Gigabit Ethernet card 
 * 
 *  \param [in] pCCPC name of the CCPC. 
 *  \return \e int the errorcode of this library: 0 if no error occured 
 * 
 *  \b Example: 
 * 
 *  \code 
 *  (...) 
 *  if (fwTell1_iniGBE("pclbcecs03") == 0)  
 *     (...) 
 *  else 
 *     DebugTN("Error found!"); 
 *  (...) 
 *  \endcode 
 * 
 *  \n 
 *  \n 
*/ 
int fwTell1_iniGBE(string pCCPC) 
{ 
 
	if (_fwCcpc_commonOperation(pCCPC, "23", true) == "-1") 
		return -1;		// ~OK 
	else 
		return 0;		// OK	 
		 
} 
 
 
 
 
 
 
 
/** This function clears the MEP buffer  
 * 
 *  \param [in] pCCPC name of the CCPC. 
 *  \return \e int the errorcode of this library: 0 if no error occured  
 * 
 *  \b Example: 
 * 
 *  \code 
 *  (...) 
 *  if (fwTell1_clearMEP("pclbcecs03") == 0)  
 *     (...) 
 *  else 
 *     DebugTN("Error found!"); 
 *  (...) 
 *  \endcode 
 * 
 *  \n 
 *  \n 
*/ 
int fwTell1_clearMEP(string pCCPC) 
{ 
 
	if (_fwCcpc_commonOperation(pCCPC, "24", true) == "-1") 
		return -1;		// ~OK 
	else 
		return 0;		// OK	 
		 
} 
 
 
 
 
 
 
 
 
/** This function initializes the complete Tell1 board. It calls all the necessary hardcoded sequences 
 *  directly on the server. For propper configuration it is necessary to call this function before the 
 *  recipes are applied  
 * 
 *  \param [in] pCCPC name of the CCPC. 
 *  \return \e int the errorcode of this library: 0 if no error occured 
 * 
 *  \b Example: 
 * 
 *  \code 
 *  (...) 
 *  if (fwTell1_iniTell1("pclbcecs03") == 0)  
 *     (...) 
 *  else 
 *     DebugTN("Error found!"); 
 *  (...) 
 *  \endcode 
 * 
 *  \n 
 *  \n 
*/ 
int fwTell1_iniTell1(string pCCPC) 
{ 
 
	if (_fwCcpc_commonOperation(pCCPC, "25", true) == "-1") 
		return -1;		// ~OK 
	else 
		return 0;		// OK	 
		 
} 
 
 



/*

fwTell1_AssembleAndCreateRecipes(string hardware, dyn_string recipes, string RecipeName, string dpType1, string RecipeType){ 
  //dyn_dyn_string Registers; 
  //dyn_dyn_int Registers_Size; 
  //dyn_dyn_int Registers_Type; 
  //dyn_string  HardwRegisters_ALL, regs;  
  //dyn_int   Types_ALL;
  //dyn_dyn_anytype RecipeData1, RecipeData2,RecipeDataALL; 
  //dyn_dyn_anytype data1, mask1; 
  //dyn_dyn_anytype data2, mask2; 
  //dyn_dyn_anytype dummy; 
  int j,i; 
  int num; 
  bool SameRegistersFlag; 
   
  dyn_string kRegs, kRegsAll, regs;
  dyn_dyn_anytype kData, kDataAll;
  dyn_int kRegSize, kRegType, kRegSizeAll, kRegTypeAll;
  dyn_anytype data1, mask1, dummy; 
  dyn_anytype data2, mask2; 
  
  DebugTN("=========================================================================="); 
  DebugTN("====================ASSEMBLE=THE=DIFFERENT=RECIPES========================");  
  DebugN(recipes);
  if(dynlen(recipes) > 1){     
	fwHw_getRecipe(hardware, recipes[1], kRegsAll, kDataAll);
        //fwTell1_getRegInfo  givs the size and the type of each registers 
        fwTell1_getRegInfo(hardware, kRegsAll, kRegSizeAll, kRegTypeAll); 
        DebugN(kRegsAll); 
 	for(num = 2; num <= dynlen(recipes); num++){ 
	                fwHw_getRecipe(hardware, recipes[num], kRegs, kData); 
                        fwTell1_getRegInfo(hardware, kRegs, kRegSize, kRegType);
                        DebugN(kRegs);
                        DebugN(kRegSize);
                        DebugN(kRegType);
	                if(dynlen(kRegs)>0){ 
	   			for(i = 1; i <= dynlen(kRegs); i++){ 
	   				SameRegistersFlag = false; 
	     				for(j = 1; j<=  dynlen(kRegsAll); j++){	      					  
	                                        if(kRegsAll[j] == kRegs[i]){ 							 
							SameRegistersFlag = true; 
							break ; 
						}               
					} 
					if(SameRegistersFlag){ 
						//fwTell1_updateVar gives back data and mask 
	    					fwTell1_updateVar(kDataAll[j], kRegSizeAll[j], data1, mask1); 
					        fwTell1_updateVar(kData[i], kRegSize[i], data2, mask2);    
						DebugTN("re-create data for register: "+kData[i]); 
						//update of the data and the mask (according to the two old ones) 
                                                dynClear(dummy);
					        fwTell1_writeToVar(data1,mask1,data2,mask2) ; 
						dynAppend(dummy, data1); 
						dynAppend(dummy, mask1);    
                                                kDataAll[i] = dummy;
					}else{ 
                                                dynAppend(kDataAll, kData[i]);
                                                dynAppend(kRegSizeAll, kRegSize[i]);  
                                                dynAppend(kRegTypeAll, kRegType[i]); 
                                                //for the Recipes Type ".SL.blabla" 
					        //dynAppend(kRegAll, substr(kRegs[i],strlen(hardware))); 
                                                dynAppend(kRegsAll, kRegs);
					} 
				}
		}else{ 
 		DebugTN("No registers found in : "+recipes[num]+"!"); 
		} 
	} 
   }else{ 
   DebugTN("Less than two recipes passed as argument!"); 
   } 
   
   
     for(i=1; i<=dynlen(kRegsAll); i++){
    dum = strsplit(kRegsAll[i],":");
            if(dynlen(dum) >= 2)
              kRegsAll[i] = dum[2];
      kRegsAll[i] = substr(kRegsAll[i],strlen(hardware));
          }
   
   
   
   //DebugTN(HardwRegisters_ALL);
   fwHw_createRecipeType(RecipeType,dpType1,kRegsAll);
   DebugN("Registers belonging to recipe type: " + RecipeType);  
   //fwHw_getRecipeTypeInfo(RecipeType, dpType1, regs); 
   DebugN(kRegsAll);    
   fwHw_saveRecipe(hardware, RecipeType, RecipeName, kRegsAll, kRegTypeAll, kDataAll);
   // DebugN(hardware, RecipeType, RecipeName); 
   // DebugN(hardware, RecipeType, RecipeName); 
   DebugTN("=========================END=OF=THE=ASSEMBLY=============================="); 
   DebugTN("=========================================================================="); 
 }

*/



 

 
   
   //this is Cedrics code
   
fwTell1_AssembleAndCreateRecipes(string hardware, dyn_string recipes, string RecipeName, string dpType1, string RecipeType){ 
   
 
  dyn_dyn_string Registers; 
  dyn_dyn_int Registers_Size; 
  dyn_dyn_int Registers_Type; 
  dyn_string  HardwRegisters_ALL, regs;  
  dyn_int   Types_ALL;
  dyn_dyn_anytype RecipeData1, RecipeData2,RecipeDataALL; 
  dyn_dyn_anytype data1, mask1; 
  dyn_dyn_anytype data2, mask2; 
  dyn_dyn_anytype dummy; 
  int j,i; 
  int num; 
  bool SameRegistersFlag; 
  dyn_string dum; 
  
  DebugTN("=========================================================================="); 
  DebugTN("====================ASSEMBLE=THE=DIFFERENT=RECIPES========================");  
 
   DebugN(recipes);
  if(dynlen(recipes) > 1){     
	fwHw_getRecipe(hardware, recipes[1], Registers[1], RecipeData1);  
        DebugN(Registers[1]); 
 	for(num = 1; num <= dynlen(recipes) - 1 ; num++){ 
	   	if(dynlen(Registers[num])>0){ 
	     		fwTell1_getRegInfo(hardware, Registers[num], Registers_Size[num], Registers_Type[num]); 
                        DebugN(Registers_Type);
	                fwHw_getRecipe(hardware, recipes[num+1], Registers[num+1], RecipeData2); 
	                if(dynlen(Registers[num])>0){ 
 
				//fwTell1_getRegInfo  givs the size and the type of each registers 
				fwTell1_getRegInfo(hardware, Registers[num+1], Registers_Size[num+1], Registers_Type[num+1]); 
				 
	   			for(i = 1; i <= dynlen(Registers[num]); i++){ 
                                        dynClear(dummy); 
	   				SameRegistersFlag = false; 
	     				for(j = 1; j<=  dynlen(Registers[num+1]); j++){	      					  
	                                        if(Registers[num][i] == Registers[num+1][j]){ 							 
							SameRegistersFlag = true; 
							break ; 
						}               
					} 
					if(SameRegistersFlag){ 
 
						dynClear(dummy); 
						dynClear(data1); 
	    					dynClear(mask1); 
						dynClear(data2); 
						dynClear(mask2); 
 
						//fwTell1_updateVar gives back data and mask 
	    					fwTell1_updateVar(RecipeData1[i], Registers_Size[num][i], data1[i], mask1[i]); 
					        fwTell1_updateVar(RecipeData2[j], Registers_Size[num+1][j], data2[j], mask2[j]);    
    
						DebugTN("re-create recipes for register: "+Registers[num][i]); 
 
						//update of the data and the mask (according to the two old ones) 
					        fwTell1_writeToVar(data1[i],mask1[i],data2[j],mask2[j]) ; 
 
						dynAppend(dummy[i], data1[i]); 
						dynAppend(dummy[i], mask1[i]);    
 
						//remove in the other dyn_string num+1 
					        dynRemove(Registers[num+1],j); 
 					        dynRemove(Registers_Type[num+1],j); 
					        dynRemove(RecipeData2,j); 
 
 
					}else{ 
						dynAppend(dummy[i], RecipeData1[i]); 
					} 
   
					dynAppend(HardwRegisters_ALL, Registers[num][i]); 
					dynAppend(Types_ALL,Registers_Type[num][i]); 
					dynAppend(RecipeDataALL, dummy[i]); 
 
					//for the Recipes Type ".SL.blabla" 
					dynAppend(Registers[dynlen(recipes)+1], substr(Registers[num][i],strlen(hardware)));     
			    		 
				} 
				for(i = 1; i <= dynlen(Registers[num+1]); i++){       
					dynAppend(HardwRegisters_ALL, Registers[num+1][i]); 
					dynAppend(Types_ALL,Registers_Type[num+1][i]); 
					dynAppend(RecipeDataALL, RecipeData2[i]); 
					dynAppend(Registers[dynlen(recipes)+1], substr(Registers[num+1][i],strlen(hardware))); 
				} 
				 
			} 
			dynClear(Registers[num+1]); 
			dynClear(Registers_Size[num+1]); 
			dynClear(Registers_Type[num+1]); 
			dynClear(RecipeData2);  
			Registers[num+1] = HardwRegisters_ALL; 
			RecipeData1 = RecipeDataALL; 
                        //warning
			if(num < dynlen(recipes) - 1 ){ 
				dynClear(HardwRegisters_ALL); 
				dynClear(Types_ALL); 
				dynClear(RecipeDataALL); 
				dynClear(Registers[dynlen(recipes)+1]); 
			} 
 
		}else{ 
 
 		DebugTN("there is not register in this recipes: "+recipes[num]+" ?!?!"); 
 
		} 
 
	} 
 
   }else{ 
 
   DebugTN("you've given only less than two recipes"); 
 
   } 
 
   DebugTN(HardwRegisters_ALL);
  
   
   //this is a dirty bugfix - the problem arrises elsewhere!!!
   for(i=1; i<=dynlen(HardwRegisters_ALL); i++){
    dum = strsplit(HardwRegisters_ALL[i],":");
            if(dynlen(dum) >= 2)
              HardwRegisters_ALL[i] = dum[2];
      HardwRegisters_ALL[i] = substr(HardwRegisters_ALL[i],strlen(hardware));
          }
   
          
          
     //DebugTN(Registers[dynlen(recipes)+1]);  
     //DebugTN("new");      
            
   //fwHw_createRecipeType(RecipeType,dpType1,Registers[dynlen(recipes)+1]);
    fwHw_createRecipeType(RecipeType,dpType1,HardwRegisters_ALL);
   DebugN("Registers belonging to recipe type: " + RecipeType);  

   fwHw_getRecipeTypeInfo(RecipeType, dpType1, regs); 
   DebugN(regs);    
  
   fwHw_saveRecipe(hardware, RecipeType, RecipeName, HardwRegisters_ALL, Types_ALL, RecipeDataALL);
   
  // DebugN(hardware, RecipeType, RecipeName); 
  // DebugN(hardware, RecipeType, RecipeName); 
   DebugTN("=========================END=OF=THE=ASSEMBLY=============================="); 
   DebugTN("=========================================================================="); 
 
 }


/*
 
fwTell1_AssembleAndCreateRecipes(string hardware, dyn_string recipes, string RecipeName, string dpType1, string RecipeType){ 

dyn_string AllRegisters;
dyn_int AllRegistersSize;
dyn_string AllRegistersType;   
dyn_dyn_anytype AllRegistersDataCombined;
dyn_dyn_anytype AllRegistersData;
dyn_dyn_anytype AllRegistersMask;

dyn_string registers;
dyn_dyn_anytype data;
dyn_int size;
dyn_string type;

dyn_dyn_anytype  splitdata;
dyn_dyn_anytype  splitmask;
int existing = 0;
int i, j, num;

  DebugTN("=========================================================================="); 
  DebugTN("====================ASSEMBLE=THE=DIFFERENT=RECIPES========================");  
 
   
  if(dynlen(recipes) > 1){                                              //loop over recipes 
 	for(num = 1; num <= dynlen(recipes); num++){ 
            dynClear(registers);
            dynClear(data);
            dynClear(type);
            dynClear(size);
            fwHw_getRecipe(hardware, recipes[num], registers, data);          //load info of recipe's regs   
	   	if(dynlen(registers)>0){             
	     	   fwTell1_getRegInfo(hardware, registers, size, type);     //load info of recipe's regs
	          
	         for(i = 1; i <= dynlen(registers); i++){
                    existing = 0; 
	   	    for(j = 1; j<=  dynlen(AllRegisters); j++){	   //check if register is already existing: updates or appends	
		        
	                if(AllRegisters[j] == registers[i]){ 							 
			   	dynClear(splitdata); 
	    			dynClear(splitmask); 
				fwTell1_updateVar(data[i], size[i], splitdata, splitmask);                   //fwTell1_updateVar gives back data and mask 		 
				fwTell1_writeToVar(AllRegistersData[i], AllRegistersMask1[i], splitdata, splitmask);	   //update of the data and the mask (according to the two old ones) 
				AllRegistersDataCombined[i] = AllRegistersData[i] + AllRegistersMask[i];
                                existing = 1;
                                break; 
			}
                     }
                   if(!existing){
                        fwTell1_updateVar(data[i], size[i], splitdata, splitmask);  
                        dynAppend(AllRegistersData, splitdata);
                        dynAppend(AllRegistersMask, splitmask);
                        dynAppend(AllRegistersType, type[i]);
                        dynAppend(AllRegistersSize, size[i]);
	           }
                }
	      }else{ 
 		   DebugTN("Recipe parsed without registers!?"); 
		} 
     	} //loop over recipes
   }else{ 
      DebugTN("Less than two recipes given as argument!?"); 
   } 
 
   //DebugTN(HardwRegisters_ALL);
   dynClear(registers);
   if (fwHw_createRecipeType(RecipeType,dpType1,AllRegistersType)){
      DebugN("Registers belonging to recipe type: " + RecipeType);  
      fwHw_getRecipeTypeInfo(RecipeType, dpType1, registers); 
      DebugN(registers);
   }else{                                                             //type already exists - check if order of recipes is correct
      fwHw_getRecipeTypeInfo(RecipeType, dpType1, registers); 
      for(i = 1; i <= dynlen(registers); i++){
          if(registers[i] != AllRegisters[i])
             DebugN("registers not in expected order!!!  -  " + registers[i]);   //in case it matters?
      }
   }
        

   fwHw_saveRecipe(hardware, RecipeType, RecipeName, AllRegisters, AllRegistersType, AllRegistersDataCombined); 
    
   DebugTN("=========================END=OF=THE=ASSEMBLY=============================="); 
   DebugTN("=========================================================================="); 
 
} 
 
 
 */
 
 
fwTell1_updateVar(dyn_anytype loadData,int size,dyn_anytype &Data,dyn_anytype &Mask)   
{ 
  	int i; 
	if (dynlen(loadData) == (2 * size)){ 
		for (i=1; i <= size; i++){ 
			Data[i] = loadData[i]; 
			Mask[i] = loadData[i+size]; 
		} 
 
	}else{   
		Data = loadData;       
		Mask = loadData; 
	} 
}	 
 
 
 
 
fwTell1_writeToVar(dyn_anytype &RegVarData, dyn_anytype &RegVarMask, dyn_anytype NewData, dyn_anytype RegMask){ //, dyn_anytype &DataOut, dyn_anytype &MaskOut){ 
  
 int i; 
 dyn_anytype maskedRead; 
 dyn_anytype maskedWrite; 
	 
	 
 bool DEBUG = false; 
	 
	if(DEBUG){ 
		DebugTN("-----------------------------------------"); 
		DebugTN("old data : "+fwHw_convertByteToHex(RegVarData)); 
		DebugTN("old mask : "+fwHw_convertByteToHex(RegVarMask)); 
		DebugTN("-----------------------------------------"); 
		DebugTN("add data : "+fwHw_convertByteToHex(NewData)); 
		DebugTN("add mask : "+fwHw_convertByteToHex(RegMask)); 
		DebugTN("-----------------------------------------"); 
	} 
 
	if(dynlen(RegMask) != dynlen(RegVarData)) 
		DebugN("WARNING: Length of mask does not match register length!!!"); 
   
   
	for(i = 1; i <= dynlen(NewData); i++){ 
		maskedRead[i]  = RegVarData[i] & (~RegMask[i]);  //read_tmp[i] &= ~mask_tmp[i]; 
		maskedWrite[i] = NewData[i]    & RegMask[i];       //write_tmp[i] &= mask_tmp[i]; 
		RegVarData[i]  = maskedRead[i] | maskedWrite[i]; //write_tmp[i] |=  read_tmp[i];       
		RegVarMask[i]  = RegVarMask[i] | RegMask[i]; 
	} 
	 
	if(DEBUG){ 
		DebugTN("new data : "+fwHw_convertByteToHex(RegVarData)); 
		DebugTN("new mask : "+fwHw_convertByteToHex(RegVarMask)); 
		DebugTN("-----------------------------------------"); 
		DebugTN("-----------------------------------------");  
	}  
  
 //DataOut = RegVarData; 
 //MaskOut = RegVarMask; 
   
} 
 
 
 
fwTell1_getRegInfo(string hardware, dyn_string Registers, dyn_int &Registers_Size, dyn_string &MyregType) 
{ 
	dyn_dyn_anytype list6789;  //Find out yourself list Nr 1    (specific settings) 
        dyn_string newRegisters; 
        dyn_string typeReg; 
        dyn_string dum;
       
        
        for(int i =1; i<= dynlen(Registers); i++) {
            //dynClear(dum);
            dum = strsplit(Registers[i],":");
            if(dynlen(dum) >= 2)
              Registers[i] = dum[2];
            dynAppend(newRegisters,substr(Registers[i],strlen(hardware+"."))); 
        }
  
	fwHw_getSpecificSettings(hardware, newRegisters, list6789); 
       // DebugTN("fwTell1_getRegInfo:");
       // DebugN("hardware: " + hardware);
       // DebugN("newRegisters: " + newRegisters);
       // DebugN("specific Settings: " + list6789);
        
        
        for(int i = 1; i <= dynlen(Registers); i++){  
                  if(dynlen(list6789[i]) == 5){ 
			dynAppend(MyregType,FWHW_LBUS); 
			dynAppend(Registers_Size, (int) list6789[i][2] * (int) list6789[i][3]); 
		  } 
                  if(dynlen(list6789[i]) == 9){ 
			dynAppend(MyregType,FWHW_I2C); 
			dynAppend(Registers_Size, list6789[i][4]); 
		  }           
          } 
}  




