//-*-c++-*-

// ======================================
//  NAME    : FWUKL1 Library
//
//  VERSION : 0
//
//  REVISION: 9
//
//  DATE    : 2006/10/12
//
//  RELEASE : 2006/01/19
//
//  PLATFORM: PVSS II
//
//  AUTHOR  : Gareth Rogers
//
//  EMAIL   : rogers@hep.phy.cam.ac.uk
// ======================================

/* \mainpage UKL1 Library Documentation
 *
 * This library is used to manipulate the state of the L1 board for the RICH detector.
 * Currently there are four possible states the board can exist in:
 *   \li Unconfigured This is the initial state of the board after turning on, a reset or if any register is
 *         not correctly configured. Its behaviour in this state is undefined, however it is may act vaguley
 *         sensibly depending on what caused this state to be reached.
 *   \li Ready All the registers on the L1 board have been configured and the board is ready to receive data.
 *         All input channels, front end (FE) FPGAs and gigabit ethernet (GBE) ports are disabled,no data
 *         can there for be sent or received. This state can be reached from any of the other defined states.
 *   \li Start This state can only be reached from the ready state. It configures the enable settings for the
 *         input channels, FE FPGAs and GBE ports. Once this has been called data can be receieved, processed
 *         and transmitted. If the start state is reached and ready has not been called at least once then the
 *         state will be unconfigured.
 *   \li Stop This state is effectively the same as ready, as it disables the input channels, FE FPGAs and GBE
 *         ports. It can only be reached from either the ready or start states otherwise the board state will
 *         be unconfigured.
 * \n
 * The library has been designed such that all setting of hardware registers can be done through the function
 * 'ukl1SetCcpcState', using the states defined in the constants and global variables section
 * (e.g. UKL1_START_STATE). It will call all the necessary functions to set the hardware to the appropriate state,
 * logging any errors that occur and returning a status number to indicate whether the state change was successful.
 * This is the only function that a panel will need to call to configure the board.
 * \n
 * Functionality is provided to create a Ukl1Board datapoint will sensible default settings for the each datapoint
 * element value. The datapoint will be called RichL1 followed by the name of the RICH L1 board as it appears in the
 * DIM name server, allowing a unique datapoint to be created for each of RICH L1 boards found in a system. The
 * state machine will access the values stored in the appropriate datapoint by setting the global variable
 * ukl1CcpcName. This allows both the appropriate datapoint settings to be accessed and the appropriate L1 board
 * to be written to. The state of each of these boards is maintained through the global mapping ukl1CcpcList, which
 * maintains a knowledge of each L1 board state, associating it with its name. It is up to the User to ensure that
 * this always contains a valid set of L1 boards.
 * \n
 * A second datapoint, RichL1Log, contains the status, warning and error messages that are generated by the
 * library function calls and this datapoint should be monitored to ensure that all library messages are seen.
 *
 */// ======================================

// ======================================
//  CONSTANTs & GLOBAL VARs
// ======================================

/** @defgroup SectionConstants Constants
 *  List of constants that can be used by the user.
 *  @{
 */

/*! Defines the numeric value of the message level . */
const unsigned FWUKL1_ALWAYS_MESSAGE_LEVEL = 0;

/*! Defines the numeric value of the message level . */
const unsigned FWUKL1_INFO_MESSAGE_LEVEL = 1;

/*! Defines the numeric value of the message level . */
const unsigned FWUKL1_DEBUG_MESSAGE_LEVEL = 2;

/*! Defines the numeric value of the message level . */
const unsigned FWUKL1_VERBOSE_MESSAGE_LEVEL = 3;

/*! Broadcast FE FPGA number. Value used to write to all the FE FPGAs. */
const unsigned FWUKL1_BROADCAST_FPGA = 5;
/*! Broadcast FE FPGA channel number. Value used to write to all of a FE FPGA's channels. */
const unsigned FWUKL1_BROADCAST_CHANNEL = 10;
/*! Number of Front end (ingress) FPGAs on the RICH L1 board. */
const unsigned FWUKL1_FRONT_FPGAS = 4;
/*! Number of channels that are on each front end FPGA. */
const unsigned FWUKL1_FPGA_CHANNELS = 9;
/*! Number of channels that are on the RICH L1 board. */
const unsigned FWUKL1_BOARD_CHANNELS = 36;
/*! Number of ports on the gigabit ethernet card. */
const unsigned FWUKL1_GBE_PORTS = 4;

/*! Defines a string, which represents the unconfigured state. */
const unsigned FWUKL1_NOT_READY_STATE = 0;
/*! Defines a string, which represents the ready state. */
const unsigned FWUKL1_CONFIGURING_STATE = 1;
/*! Defines a string, which represents the stop state. */
const unsigned FWUKL1_STOP_STATE = 2;
/*! Defines a string, which represents the start state. */
const unsigned FWUKL1_START_STATE = 3;

/*! Constant that defines the size of a 32-bit data word. */
const unsigned FWUKL1_32BIT_WORD = 32;

/*! Constant that defines the size of a 16-bit data word. */
const unsigned FWUKL1_16BIT_WORD = 16;

/*! Constant that defines the size of an 8-bit data word. */
const unsigned FWUKL1_8BIT_WORD = 8;

/*! Constant that defines the prefix to all the datapoints created from the DPT HwTypeCCPCUKL1. */
const string FWUKL1_HW_DPNAME_PREFIX = "HWCCPCUKL1/";

/*! Constant that defines the prefix to all the datapoints created from the DPT FwLHCbDAQDevice. */
const string FWUKL1_FSM_DPNAME_PREFIX = "FSMCCPCUKL1/";

/*! Name of the data point element that stores the list of names of found L1 boards.*/
const string FWUKL1_L1_LIST_DPE = "Ukl1BoardDimNamesList.DimServerNameList";

/*! A level that indicates what status messages should be written to the buffer. */
global unsigned gufwUkl1_StatusMessageLevel = FWUKL1_INFO_MESSAGE_LEVEL;

/*! A level that indicates what warning messages should be written to the buffer. */
global unsigned gufwUkl1_WarningMessageLevel = FWUKL1_INFO_MESSAGE_LEVEL;

/*! A level that indicates what error messages should be written to the buffer. */
global unsigned gufwUkl1_ErrorMessageLevel = FWUKL1_INFO_MESSAGE_LEVEL;

/*! A boolean to control whether or not the hardware writes are verified by reading back from that register. */
global bool gbfwUkl1_verifyWrite = TRUE;

global bool bfwUkl1_SendMessageToDataPoint = TRUE;
//@}


// ==================
//  UKL1 BOARD SETUP
// ==================

/** @defgroup SectionSetup
 *  Provides functions that will find and configure the UKL1 boards that are present in the DIM server.
 *  These will be used only when starting the PVSS project or when changes are made to the hardware
 *  configuration in the DIM server.
 *  @{
 */

/*!
 * Searches for UKL1s found on the DIM server and then identifies the UKL1 boards based on a name filter
 * creating the software instance of the boards.
 *
 * \param  nameFilter Part of the UKL1 board name that is used to identify the name. Each element in the dyn_string
 *           can represent a different filter. Supports the use of * for wild cards.
 * \param  refresh If TRUE the list existing list of UKL1 boards will be updated with any new boards found and those
 *           no longer present will be removed. FALSE all existing boards will be deleted along with their associated
 *           settings and the whole list created from new. Default: TRUE.
 * \param  forceReconfigure If TRUE it will force the settings to be manually set and not to be loaded from the defaults
 *           for the fwHw datapoint type. FALSE and the settings are just loaded from the defaults. Default: FALSE.
 * \param  deleteExisting If the datapoint is found to exist it will be deleted and then recreated with the relevant
 *           settings. Default: FALSE.
 * \return int The following return codes are used:
 *    \li  0 Execution proceeded with a problem and all boards are now present in the system.
 *    \li -1 Failed to find and add boards to the system.
 *
 * It will then create a HW type datapoint for the found board, subscribe it to the DIM server allowing hardware
 * access and connect the board to the finite state machine.\n
 * It can either update the existing list of known UKL1 boards or remove the existing list and start afresh.
 * In the case of starting afresh any existing datapoints will be deleted, DIM services unsubscribed and finite state
 * machine call back functions disconnected.
 */
int fwUkl1_FindAndAddUkl1Boards(dyn_string nameFilter, bool refresh = TRUE, bool forceReconfigure = FALSE, bool deleteExisting = FALSE) {
  //Holds the list of CCPC names retrieved.
  dyn_string ccpcList;
  //First get a list of presently registered CCPCs in the DIM server.
  fwCcpc_get(ccpcList);
  //Save the total number found for later use.
  const int numCcpc = dynlen(ccpcList);
  fwUkl1_StatusMessage("Found " + numCcpc + " CCPCs in DIM server.",
			 FWUKL1_VERBOSE_MESSAGE_LEVEL);
  //Filter the CCPC names to determine, which of these are actually UKL1 boards.
  //Holds the names of the found UKL1 boards.
  dyn_string fullUkl1List;
  //Keep track of the number of UKL1 boards added to the list, start from 1 due to PVSS array silliness.
  unsigned numUkl1Added = 1;
  //Loop over all the found CCPCs.
  for (unsigned ccpc = 1; ccpc <= numCcpc; ++ccpc) {
    //Name of the CCPC currently being checked.
    const string ccpcName = ccpcList[ccpc];
    //Now loop over all the filters to check if it matches any.
    for (unsigned filter = 1; filter <= dynlen(nameFilter); ++filter) {
      //Check against the patterns.
      if (patternMatch(nameFilter[filter], ccpcName) ) {
	fullUkl1List[numUkl1Added] = ccpcName;
	++numUkl1Added;
	//Don't bother checking the other patterns.
	break;
      }//filter if
    }//filter for
  }//CCPC for

  //check we have actually matched some boards.
  if ( 0 == dynlen(fullUkl1List) ) {
    fwUkl1_WarningMessage("Failed to match any UKL1 boards against the filter[s]: " + nameFilter + ". Compared to " + numCcpc + " found in DIM.",
			  FWUKL1_INFO_MESSAGE_LEVEL);
  }

  //Now we have a list of UKL1 boards in the system lets deal with them.
  //Holds the names of the UKL1 boards that need to be added to the library. This always need to contain all the found boards
  //as we always need to check the datapoints for consistency and ensure the relevant DIM and FSM functionality is established.
  //The function fwUkl1_AddUkl1Board will ensure that the minimum possible is done depending on calling arguments.
  dyn_string toBeAddedUkl1List = fullUkl1List;

  //Holds the names of the UKL1 boards that need to be removed from the library. It will either be those that are
  //no longer present after a refresh or all of them in the case of a reload.
  dyn_string toBeDeletedUkl1List;
  //Just find the new boards in the system and those that have been lost.
  if (refresh) {
    //Reset the number added.
    numUkl1Added = 1;
    //Look through the list of existing names and check for those that need to be deleted.
    for (unsigned board = 1; board <= fwUkl1_GetNamesListSize(); ++board) {
      //Check all the names in the library and check if they are still present in the new server list.
      if ( "" == dynPatternMatch(fwUkl1_GetName(board), fullUkl1List) ) {
	//If they are not present they need to be deleted.
	toBeDeletedUkl1List[numUkl1Added] = fwUkl1_GetName(board);
      }
    }

  }// if (refresh).
  //In this case we are starting afresh and can just remove any existing settings from 
  else {
    toBeDeletedUkl1List = fwUkl1_GetNamesList(FWUKL1_VERBOSE_MESSAGE_LEVEL);
  }// else (refresh)

  //Delete those boards that need to be deleted.
  //Must be done before creation as we need to ensure for the reload case the list is empty.
  int status = fwUkl1_DeleteUkl1Board(toBeDeletedUkl1List);
  if ( 0 == status ) {
    fwUkl1_StatusMessage("Removed all existing boards from the system.",
			 FWUKL1_DEBUG_MESSAGE_LEVEL);
  } else if ( -1 == status ) {
    fwUkl1_WarningMessage("Removed some of the existing boards from the system.",
			  FWUKL1_INFO_MESSAGE_LEVEL);
  } else if ( -2 == status ) {
    fwUkl1_ErrorMessage("Failed to completely remove the existing UKL1 list from the system.",
			  FWUKL1_INFO_MESSAGE_LEVEL);
    return -1;
  } else if ( -3 == status ) {
    fwUkl1_StatusMessage("No existing UKL1 boards found in list to remove.",
			FWUKL1_INFO_MESSAGE_LEVEL);
  } else {
    fwUkl1_ErrorMessage("Unrecognised return function while deleting the existing UKL1 boards from the system.",
			FWUKL1_INFO_MESSAGE_LEVEL);
    return -1;
  }

  //Add those boards that need to be added.
  int status = fwUkl1_AddUkl1Board(toBeAddedUkl1List, forceReconfigure, deleteExisting);
  if ( 0 == status ) {
    fwUkl1_StatusMessage("Added UKL1 boards to the system.",
			 FWUKL1_DEBUG_MESSAGE_LEVEL);
  } else if ( -1 == status ) {
    fwUkl1_WarningMessage("Failed to all of the UKL1 boards to the system.",
			  FWUKL1_INFO_MESSAGE_LEVEL);
  } else if ( -2 == status ) {
    fwUkl1_ErrorMessage("Failed to add all of the UKL1 list to the system.",
			FWUKL1_INFO_MESSAGE_LEVEL);
    return -1;
  } else if ( -3 == status ) {
    fwUkl1_StatusMessage("No existing UKL1 boards found in list to add.",
			FWUKL1_INFO_MESSAGE_LEVEL);
  } else {
    fwUkl1_ErrorMessage("Unrecognised return function while adding UKL1 boards to the system.",
			FWUKL1_INFO_MESSAGE_LEVEL);
    return -1;
  }

  //Found and added all the boards!
  return 0;
}

/*!
 * Adds a list of UKL1 boards to the system. It creates the hardware datapoint, subscribing it to the DIM server;
 * and the finite statemachine datapoint, connecting the callback function for state changes. The names of those
 * added are put in the library list of DIM server names.
 *
 * \param  tobeAddedUkl1List A dyn_string of names of the UKL1 boards as they appear in the DIM server that need to
 *           be added to the system. If a name is already found in the system it will be removed and then readded.
 * \param  forceReconfigure If TRUE it will force the settings to be manually set and not to be loaded from the defaults
 *           for the fwHw datapoint type. FALSE and the settings are just loaded from the defaults. Default: FALSE.
 * \param  deleteExisting If the datapoint is found to exist it will be deleted and then recreated with the relevant
 *           settings. Default: FALSE.
 * \return int The following return codes are used:
 *    \li  0 Execution proceeded without a problem.
 *    \li -1 Added some of the UKL1 boards to the list.
 *    \li -2 Failed to add any of the UKL1 boards to the list.
 *    \li -3 List was empty and no boards could be added to the system.
 */
int fwUkl1_AddUkl1Board(dyn_string toBeAddedUkl1List, bool forceReconfigure = FALSE, bool deleteExisting = FALSE) {
  //Store the number of elements in the list.
  const unsigned numToBeAdded = dynlen(toBeAddedUkl1List);
  //If the list was empty return!
  if ( 0 == numToBeAdded )
    return -3;

  //Used to hold the return status of the function.
  int status = 0;

  //Loop over each board adding them to the system.
  for (unsigned board = 1; board <= numToBeAdded; ++board) {
    const string ukl1Name = toBeAddedUkl1List[board];
    //Create the hardware type datapoint and subscribe it to the DIM server.
    //We want to just load the new datapoint from the HW type defaults and if it does exist then lets delete it.
    if ( 0 != fwUkl1_CreateHwTypeCCPCUKL1Datapoint(ukl1Name, forceReconfigure, deleteExisting) ) {
      fwUkl1_ErrorMessage("Failed to add " + toBeAddedUkl1List[board] + "'s as a hardware type to the system.",
			  FWUKL1_INFO_MESSAGE_LEVEL);
      if ( 1 < board )
	status = -1;
      else
	status = -2;
    }

    //Create the FSM type datapoints and connect the callback functions.
    if ( 0 != fwUkl1_CreateFwLHCbDAQDeviceDatapoint(ukl1Name, deleteExisting) ) {
      fwUkl1_ErrorMessage("Failed to add " + toBeAddedUkl1List[board] + "'s as a FSM type to the system.",
			  FWUKL1_INFO_MESSAGE_LEVEL);
      if ( 1 < board )
	status = -1;
      else
	status = -2;
    }

    //Once we have created all the appropriate datapoints and their associated connections add the names to the list.
    if ( -1 == fwUkl1_SetName(ukl1Name) ) {
      fwUkl1_ErrorMessage("Failed to add " + toBeAddedUkl1List[board] + "'s to the name list.",
			  FWUKL1_INFO_MESSAGE_LEVEL);
      if ( 1 < board )
	status = -1;
      else
	status = -2;
    }

  }//for each board.

  return status;

}

/*!
 * Deletes a list of UKL1 boards from the system. It removes the hardware datapoint, unsubscribing it from the DIM server;
 * and the finite statemachine datapoint, disconnecting the callback function for state changes. The names of those
 * added are removed from the list of UKL1 names stored in the library.
 *
 * \param  tobeDeletedUkl1List A dyn_string of names of the UKL1 boards as they appear in the DIM server that need to
 *           be removed from the system.
 * \return int The following return codes are used:
 *    \li  0 Execution proceeded without a problem.
 *    \li -1 Removed some of the UKL1 boards from the system.
 *    \li -2 Failed to remove any of the UKL1 boards from the list.
 *    \li -3 List was empty and no boards could be deleted from the system.
 */
int fwUkl1_DeleteUkl1Board(dyn_string toBeDeletedUkl1List) {
  //Store the number of elements in the list.
  const unsigned numToBeDeleted = dynlen(toBeDeletedUkl1List);
  //If the list was empty return!
  if ( 0 == numToBeDeleted )
    return -3;

  //Used to hold the return status of the function.
  int status = 0;

  //Loop over each board removing them from the system.
  for (unsigned board = 1; board <= numToBeDeleted; ++board) {
    const string ukl1Name = toBeDeletedUkl1List[board];
    //Some datapoint and datapoint type related constants...
    //Name of the hardware type, as considered by the fwHw library.
    const string fwHwType = "UKL1";
    //First convert the UKL1 board name into the datapoint name.
    const string dpName = fwUkl1_ConvertDimServerNameToHwDpName(ukl1Name);

    //Remove the hardware type datapoint and unsubscribe it to the DIM server.
    bool bStatus = fwHw_remove(fwHwType, dpName);
    if ( !bStatus ) {
      fwUkl1_ErrorMessage("Failed to remove " + toBeDeletedUkl1List[board] + "'s hardware type from the system.",
			  FWUKL1_INFO_MESSAGE_LEVEL);
      if ( 1 < board )
	status = -1;
      else
	status = -2;
    }

    //Create the FSM type datapoints and connect the callback functions.
    int status = fwUkl1_DeleteFwLHCbDAQDeviceDatapoint(ukl1Name);
    if ( (0 != status) && (-2 != status) ) {
      fwUkl1_ErrorMessage("Failed to remove " + toBeDeletedUkl1List[board] + "'s FSM type from the system.",
			  FWUKL1_INFO_MESSAGE_LEVEL);
      if ( 1 < board )
	status = -1;
      else
	status = -2;
    }

    //Once we have created all the appropriate datapoints and their associated connections add the name to the list.
    if ( 0 != fwUkl1_DeleteName(ukl1Name) ) {
      fwUkl1_ErrorMessage("Failed to remove " + toBeDeletedUkl1List[board] + "'s from the name list.",
			  FWUKL1_INFO_MESSAGE_LEVEL);
      if ( 1 < board )
	status = -1;
      else
	status = -2;
    }

  }//for each board.

  return status;
}

//@}


// ======================================
//  DATAPOINT MANIPULATION FUNCTIONS
// ======================================

/** @defgroup SectionDatapoint datapoint functions.
 *    Provides functions for the creation and manipulation of the FSM and hardware datapoints, either through hardcoded
 *    default settings or the saved default settings. The manipulation that is allowed is for the settings of the
 *    registers that are to be written to. It configures the datapoints to allow the hardware access and state machine
 *    functionality.
 *  @{
 */

/*!
 * This will create a datapoint from the HwTypeCCPCUKL1 datapoint type, which represents a UKL1 board, with all
 * the appropriate settings and subscribe these registers to the DIM server.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  forceReconfigure If TRUE it will force the settings to be manually set and not to be loaded from the defaults
 *           for the fwHw datapoint type. FALSE and the settings are just loaded from the defaults. Default: FALSE.
 * \param  deleteExisting If the datapoint is found to exist it will be deleted and then recreated with the relevant
 *           settings. Default: FALSE.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *    \li  0 Execution proceeded without a problem.
 *    \li -1 Execution failed and the datapoint is in an unuseable state due to a PVSS problem.
 *    \li -2 The datapoint was created but it was not configured correctly.
 *    \li -3 The datapoint was created and configured, but the registers could not be subscribed to the DIM server.
 *
 *
 * This function has the following possible ways of operating:
 *    \li Hard creation - Datapoint does not exist and the datapoint is created, the settings are loaded and saved
 *          to defaults, all registers are subscribed to the DIM server. forceReconfigure=TRUE, deleteExisting has
 *          no relevance in this case.
 *    \li Soft creation - Datapoint does not exist and the datapoint is created, the settings are from the
 *          defaults, all registers are subscribed to the DIM server. forceReconfigure=FALSE, deleteExisting has
 *          no relevance in this case.
 *    \li Full hard recreation - Datapoint exists, deleteExisting=TRUE, forceReconfigure=TRUE the datapoint registers
 *          are unsubscribed from the DIM server and then the datapoint is deleted. It then proceeds as for a
 *          hard creation.
 *    \li Full soft recreation - Datapoint exists, deleteExisting=TRUE, forceReconfigure=FALSE the datapoint registers
 *          are unsubscribed from the DIM server and then the datapoint is deleted. It then proceeds as for a
 *          soft creation.
 *    \li Hard recreation - Datapoint exists, deleteExisting=FALSE, forceReconfigure=TRUE, all the settings and their
 *          defaults are reset to the settings as for a full creation and DIM server subscriptions are recreated.
 *          The datapoint is never deleted.
 *    \li Soft recreation - Datapoint exists, deleteExisting=FALSE, forceReconfigure=FALSE, all the settings are reloaded
 *          from their default values. The datapoint is never deleted and the DIM server subscriptions are not recreated.
 *
 * The name of the created datapoint will be the concatantion of FWUKL1_DPNAME_PREFIX and ukl1Name. The setting placed in this
 * datapoint set the name of the UKL1 board as in the DIM server and the address of all the registers that
 * the datapoint can access. None of the registers are set up for monitoring and the datapoints do not contain
 * any values, hence none are written to the board.
 */
int fwUkl1_CreateHwTypeCCPCUKL1Datapoint(string ukl1Name, bool forceReconfigure=FALSE, bool deleteExisting=FALSE) {
  //Some datapoint and datapoint type related constants...
  //Name of the hardware type, as considered by the fwHw library.
  const string fwHwType = "UKL1";
  //Name of the datapoint type we are creating from.
  const string dptName = FWHW_TYPE_DPT + FWHW_CCPC_DESCRIPTION + fwHwType;
  //First convert the UKL1 board name into the datapoint name.
  const string dpName = fwUkl1_ConvertDimServerNameToHwDpName(ukl1Name);

  //Check to see if the default settings datapoint actually exists. If it doesn't then we should add it
  //to the system, otherwise it will cause much confusion later... especially as there is no error checking
  //associated with loading/saving to the default values!
  if ( !dpExists(FWHW_DEFAULT_SETTINGS_DP + FWHW_CCPC_DESCRIPTION + "UKL1") ) {
    //Potentially override the User choice, but then we must...
    forceReconfigure = TRUE;
    //Create the hardware datapoint, but make sure we don't try and load the default settings as they were not found!
    if ( fwHw_create(fwHwType, dpName, FALSE) ) {
      //Successfully recreated the defaults datapoint, better warning people it happened as it is a bit odd.
      fwUkl1_WarningMessage("Recreated the datapoint to store UKL1 hardware type default settings.",
			    FWUKL1_INFO_MESSAGE_LEVEL);
    } else {
      fwUkl1_ErrorMessage("Failed to recreate the datapoint to store the UKL1 hardware type default settings. This is not fatal in itself, but could be an indicator to more serious problems. Have other datapoint failure messages appeared?",
			  FWUKL1_INFO_MESSAGE_LEVEL);
    }
  }

  //Now check to see if the datapoint exists.
  if ( dpExists(dpName) ) {
    //Datapoint exists. There are three possible cases that need to be handled here...

    //Soft recreation.
    if ( !deleteExisting && !forceReconfigure ) {
      //In this case we just need to reload from the default settings and ensure that the common settings are set correctly.
      //At present it appears that this function just returns 1, update this to check for errors once it can actually show them!
      if ( 1 == fwHw_applyDefaultSettings(fwHwType, dpName) ) {
	fwUkl1_StatusMessage("Successfully applied the default settings.",
			     FWUKL1_DEBUG_MESSAGE_LEVEL);
      }
      if ( 0 != _fwUkl1_ConfigureHwTypeCCPCUKL1CommonSettings(ukl1Name) ) {
	fwUkl1_ErrorMessage("Failed to update the common settings while reloading existing datapoint from default settings.",
			    FWUKL1_DEBUG_MESSAGE_LEVEL);
      }

      //Ensure the DIM services are setup.
      if ( !fwHw_subscribe(dpName) ) {
	fwUkl1_ErrorMessage("Failed to resubscribe to the DIM server during a creation of UKL1 board: " + ukl1Name,
			    FWUKL1_INFO_MESSAGE_LEVEL);
	return -3;
      }

    }//if ( !deleteExisting && !forceReconfigure )

    //Hard recreation.
    else if ( !deleteExisting && forceReconfigure ) {
      //Reconfigure all the settings from fwUkl1 lib values and save as defaults.
      int status = _fwUkl1_ConfigureHwTypeCCPCUKL1Settings(ukl1Name, TRUE);
      if ( 0 == status ) {
	fwUkl1_StatusMessage("Succesfully reloaded UKL1 board " + ukl1Name + " and saved as defaults.",
			     FWUKL1_DEBUG_MESSAGE_LEVEL);
      } else if ( -1 == status ) {
	fwUkl1_ErrorMessage("Failed to reload the UKL1 board " + ukl1Name,
			    FWUKL1_INFO_MESSAGE_LEVEL);
	return -2;
      } else if ( -2 == status ) {
	fwUkl1_ErrorMessage("Reloaded UKL1 board " + ukl1Name + ", but failed to save to defaults.",
			      FWUKL1_INFO_MESSAGE_LEVEL);
	return -3;
      } else {
	fwUkl1_ErrorMessage("Unrecognised return from reloading of UKL1 board " + ukl1Name + ", assuming the worst.",
			    FWUKL1_INFO_MESSAGE_LEVEL);
	return -1;
      }
      //Now resubscribe to the server.
      if ( !fwHw_subscribe(dpName) ) {
	fwUkl1_ErrorMessage("Failed to resubscribe to the DIM server during a hard recreation of UKL1 board: " + ukl1Name,
			    FWUKL1_INFO_MESSAGE_LEVEL);
	return -3;
      }

    }//else if ( !deleteExisting && forceReconfigure )

    //Full hard&soft recreation, hardness determined by forceReconfigure during recreation.
    else if ( deleteExisting ) {
      //Here we want to delete the existing datapoint and recreate it.
      bool status = fwHw_remove(fwHwType, dpName);//fwUkl1_DeleteHwTypeCCPCUKL1Datapoint(ukl1Name);
      if ( status ) {
	//Successfully deleted the datapoint, recall this function to recreate everything.
	if ( 0 == fwUkl1_CreateHwTypeCCPCUKL1Datapoint(ukl1Name, forceReconfigure) ) {
	  fwUkl1_StatusMessage("Succesfully delete and recreated the UKL1 board " + ukl1Name,
			       FWUKL1_DEBUG_MESSAGE_LEVEL);
	} else {
	  fwUkl1_ErrorMessage("Failed to recreate the UKL1 board " + ukl1Name,
			       FWUKL1_DEBUG_MESSAGE_LEVEL);
	  return -1;
	}
      } else if ( -1 == status ) {
	fwUkl1_ErrorMessage("Failed to unsubscribe the existing UKL1 board from the DIM server, before recreating it.",
			    FWUKL1_INFO_MESSAGE_LEVEL);
      } else if ( -2 == status ) {
	//Datapoint doesn't exist. It should never happen as delete performs dpExists as a check which has already
	//been done before else if...
	//If it does happen it means the wrong datapoint has been requested to be deleted, hence the original datapoint exists
	//upon recalling of this function. We will find the datapoint exists and it should be deleted before recalling itself
	//to recreate it. However as the datapoint was never actually deleted, because the second call thought it didn't exist
	//then we will keep calling ourselves. This is a symptom of trying to delete a different datapoint to what we want to
	//operate on. Its the fwUkl1_DeleteDatapoint just after else if ( deleteExisting=TRUE) that causes the trouble.
	fwUkl1_ErrorMessage("Failed to delete the existing UKL1 board from the DIM server, before recreating it. It should have existed at this point, something has gone very wrong.",
			    FWUKL1_INFO_MESSAGE_LEVEL);
	return -1;
      } else if ( -3 == status ) {
	fwUkl1_ErrorMessage("UKL1 board failed to delete due to a problem with PVSS",
			    FWUKL1_INFO_MESSAGE_LEVEL);
	return -1;
      } else {
	fwUkl1_ErrorMessage("Failed to delete the UKL1 board " + ukl1Name + ", hence recreation of datapoint failed.",
			    FWUKL1_INFO_MESSAGE_LEVEL);
	return -1;
      }
    }// else if ( deleteExisting )

  }//if ( dpExists(dpName) )

  else {
    //Hard creation.
    if ( forceReconfigure ) {
      //DP doesn't exist so create it, but don't reload from defaults we need to set those up first.
      if ( fwHw_create(fwHwType, dpName, FALSE) ) {
	//Set all the values on the datapoint to something sensible.
	int status = _fwUkl1_ConfigureHwTypeCCPCUKL1Settings(ukl1Name, TRUE);
	if ( 0 == status ) {
	  fwUkl1_StatusMessage("Succesfully configured datapoint " + dpName + " and saved as defaults.",
			       FWUKL1_DEBUG_MESSAGE_LEVEL);
	} else if ( -1 == status ) {
	  fwUkl1_ErrorMessage("Failed to configure the datapoint " + dpName,
			      FWUKL1_INFO_MESSAGE_LEVEL);
	  return -2;
	} else if ( -2 == status ) {
	  fwUkl1_ErrorMessage("Configure data point " + dpName + ", but failed to save to defaults.",
			      FWUKL1_INFO_MESSAGE_LEVEL);
	  return -3;
	} else {
	  fwUkl1_ErrorMessage("Unrecognised return from configuring of datapoint " + dpName + ", assuming the worst.",
			      FWUKL1_INFO_MESSAGE_LEVEL);
	  return -1;
	}
      } else {
	fwUkl1_ErrorMessage("Failed to create the datapoint: " + dpName,
			    FWUKL1_INFO_MESSAGE_LEVEL);
	return -1;
      }
    }// if ( forceReconfigure )

    //Soft creation
    else {
      //DP doesn't exist so create it and reload from the defaults.
      if ( fwHw_create(fwHwType, dpName, TRUE) ) {
	if ( 0 != _fwUkl1_ConfigureHwTypeCCPCUKL1CommonSettings(ukl1Name) ) {
	  fwUkl1_ErrorMessage("Failed to update the common settings while reloading existing datapoint from default settings.",
			      FWUKL1_DEBUG_MESSAGE_LEVEL);
	}
      }
    }// else (forceReconfigure)

    //Now subscribe all the registers to the DIM server. DIM server will just unsubscribe and resubscribe if the subscriptions exist.
    if ( !fwHw_subscribe(dpName) ) {
      fwUkl1_ErrorMessage("Failed to resubscribe to the DIM server during a creation of UKL1 board: " + ukl1Name,
			  FWUKL1_INFO_MESSAGE_LEVEL);
      return -3;
    }

  }// else ( dpExists(dpName) //if ( 0 == dpCreate(dpName, dptName) )

  //If we got this far then everything must have been created and subscribed without issue.
  return 0;
}

/*!
 * This will remove a HwTypeCCPCUKL1 datapoint from the system that corresponds to a given UKL1 board. It will also unsubscribe
 * all the connections to the DIM server associated with this datapoint.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *    \li  0 Execution proceeded without a problem.
 *    \li -1 The datapoint could not be unsubscribed from the DIM server and was not deleted.
 *    \li -2 The datapoint was unsubscribed from the DIM server, but could not be deleted as it did not exist.
 *    \li -3 The datapoint was unsubscribed from the DIM server, but could not be deleted due to a PVSS issue.
 */
int fwUkl1_DeleteHwTypeCCPCUKL1Datapoint(string ukl1Name) {
  //Figure out what the datapoint name is from the ukl1Name.
  const string dpName = fwUkl1_ConvertDimServerNameToHwDpName(ukl1Name);

  //Detele the datapoint.
  int status = _fwUkl1_DeleteDatapoint(dpName);
  //Check the return status and return the appropriate value.
  if ( 0 == status ) {
    fwUkl1_StatusMessage("Sucessfully delete the datapoint: " + dpName,
			 FWUKL1_VERBOSE_MESSAGE_LEVEL);
  } else if ( -2 == status ) {
    fwUkl1_ErrorMessage("Failed to delete the datapoint " + dpName + " as it did not exist.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -2;
  } else if ( -1 == status ) {
    fwUkl1_ErrorMessage("Failed to delete the datapoint " + dpName + " due to a PVSS problem.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -3;
  } else {
    fwUkl1_ErrorMessage("Unrecognised return value from fwUkl1 deletion function.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -3;
  }
  //Try to unsubscribe from the DIM server.
  if ( !fwHw_unsubscribe(dpName) ) {
    fwUkl1_ErrorMessage("Failed to unsubscribe the datapoint: " + dpName + ", from the DIM server.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -1;
  }


  return 0;
}

/*!
 * This will create a datapoint from the FwLHCbDAQDevice datapoint type, which represents a UKL1 board in the FSM,
 * and connect it to the appropriate callback function.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  deleteExisting If the datapoint is found to exist it will be deleted and then recreated with the relevant
 *           settings. Default: FALSE.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *    \li  0 Execution proceeded without a problem.
 *    \li -1 Execution failed and the datapoint is in an unuseable state due to a PVSS problem.
 *    \li -2 The datapoint was created but it was not configured correctly.
 *    \li -3 The datapoint was created and configured, but the state machine callback functions could not be connected.
 *
 * This function has the following ways of creating an FSM setup:
 *    \li Creation - Datapoint does not exist and the datapoint is created, set to a not ready state and the callback
 *          functions are connected to it. deleteExisting has no effect in this case.
 *    \li Recreation - Datapoint exists, deleteExisting=TRUE. Callback function is disconnected from the datapoint
 *          and then the datapoint is deleted. It will then proceed as for creation.
 *    \li Reset - Datapoint exists, deleteExisting=FALSE. Datapoint is not deleted, it is reset to a not ready state
 *          and the callback function disconnected and reconnected.
 *
 * The name of the created datapoint will be the concatonation of FWUKL1_FSM_DPNAME_PREFIX and ukl1Name and will always
 * be set to the not ready state. These changes will not affect any of the hardware settings.
 */
int fwUkl1_CreateFwLHCbDAQDeviceDatapoint(string ukl1Name, bool deleteExisting=FALSE) {
  //Stores the datapoint type name that the datapoint is to be created from.
  const string dptName = "FwLHCbDAQDevice";
  //Convert the UKL1 board name into the datapoint name.
  const string dpName = fwUkl1_ConvertDimServerNameToFsmDpName(ukl1Name);
  //Stores the name of the callback function.
  const string callback = "fwUkl1_SetUkl1StateCB";

  //Now check to see if the datapoint exists.
  if ( dpExists(dpName) ) {
    //Datapoint exists.
    //Stores the element of the datapoint we are connect/disconnecting to/from.
    const string dpeName = dpName + ".status";
    //Reset.
    if ( !deleteExisting ) {
      //First disconnect the callback function so changes won't get propagated to the hardware.
      if ( 0 != dpDisconnect(callback, dpeName) ) {
					fwUkl1_ErrorMessage("Failed to disconnect the callback function during a reset of FSM functionality. Callback function is probably not yet connected e.g. system been restart but DP exists, if connect also fails there is a problem.",
			    FWUKL1_INFO_MESSAGE_LEVEL);
      }
      //Now update the status element in datapoint.
      if ( 0 != dpSet(dpeName, FWUKL1_NOT_READY_STATE) ) {
					fwUkl1_ErrorMessage("Failed to reset the FSM status element during a reset datapoint.",
			    FWUKL1_DEBUG_MESSAGE_LEVEL);
					return -2;
      }
      //Now reconnect the callback function. Don't want it to be called on connection however.
      if ( 0 != dpConnect(callback, FALSE, dpeName) ) {
					fwUkl1_ErrorMessage("Failed to connect the callback function during a reset of FSM functionality.",
			    FWUKL1_DEBUG_MESSAGE_LEVEL);
					return -3;
      }
      //Done.
    }//if ( !deleteExisting )

    //Recreation.
    else  {
      //Here we want to delete the existing datapoint and recreate it.
      int status = fwUkl1_DeleteFwLHCbDAQDeviceDatapoint(ukl1Name);
      if ( 0 == status ) {
	//Successfully deleted the datapoint, recall this function to recreate everything.
	if ( 0 == fwUkl1_CreateFwLHCbDAQDeviceDatapoint(ukl1Name) ) {
	  fwUkl1_StatusMessage("Succesfully delete and recreated the UKL1 FSM for " + ukl1Name,
			       FWUKL1_DEBUG_MESSAGE_LEVEL);
	} else {
	  fwUkl1_ErrorMessage("Failed to recreate the UKL1 FSM for " + ukl1Name,
			      FWUKL1_DEBUG_MESSAGE_LEVEL);
	  return -1;
	}
      } else if ( -1 == status ) {
	fwUkl1_ErrorMessage("Failed to unsubscribe the existing UKL1 board from the DIM server, before recreating it.",
			    FWUKL1_INFO_MESSAGE_LEVEL);
      } else if ( -2 == status ) {
	//Datapoint doesn't exist. It should never happen as delete performs dpExists as a check which has already
	//been done before else if...
	//If it does happen it means the wrong datapoint has been requested to be deleted, hence the original datapoint exists
	//upon recalling of this function. We will find the datapoint exists and it should be deleted before recalling itself
	//to recreate it. However as the datapoint was never actually deleted, because the second call thought it didn't exist
	//then we will keep calling ourselves. This is a symptom of trying to delete a different datapoint to what we want to
	//operate on. Its the fwUkl1_DeleteDatapoint just after else if ( deleteExisting=TRUE) that causes the trouble.
	fwUkl1_ErrorMessage("Failed to delete the existing UKL1 board from the DIM server, before recreating it. It should have existed at this point, something has gone very wrong.",
			    FWUKL1_INFO_MESSAGE_LEVEL);
	return -1;
      } else if ( -3 == status ) {
	fwUkl1_ErrorMessage("UKL1 board failed to delete due to a problem with PVSS",
			    FWUKL1_INFO_MESSAGE_LEVEL);
	return -1;
      } else {
	fwUkl1_ErrorMessage("Failed to delete the UKL1 board " + ukl1Name + ", hence recreation of datapoint failed.",
			    FWUKL1_INFO_MESSAGE_LEVEL);
	return -1;
      }
    }//else ( deleteExisting )

  }//if ( dpExists(dpName) )

  else {
    //The datapoint does not exist, so we will need to first create it.
    if ( 0 == dpCreate(dpName, dptName) ) {
      //Note the datapoint element that we want to deal with.
      const string dpeName = dpName + ".status";
      //Now update the status element in datapoint to ensure it is in the not ready state.
      if ( 0 != dpSet(dpeName, FWUKL1_NOT_READY_STATE) ) {
	fwUkl1_ErrorMessage("Failed to reset the FSM status element during a reset datapoint.",
			    FWUKL1_DEBUG_MESSAGE_LEVEL);
      }
      //Now reconnect the callback function. Don't want it to be called on connection however.
      if ( 0 != dpConnect(callback, FALSE, dpeName) ) {
	fwUkl1_ErrorMessage("Failed to connect the callback function during a reset of FSM functionality.",
			    FWUKL1_DEBUG_MESSAGE_LEVEL);
      }
    }// if ( 0 == dpCreate(dpName, dptName) )

    else {
      //Failed to create the datapoint so indicate an error.
      fwUkl1_ErrorMessage("Failed to create the datapoint:" + dpName,
			  FWUKL1_INFO_MESSAGE_LEVEL);
      return -1;
    }// else ( 0 == dpCreate(dpName, dptName) )

  }// else ( dpExists(dpName) )

  //If we got this far then everything must have been created and subscribed without issue.
  return 0;
}

/*!
 * This will remove a FwLHCbDAQDevice datapoint from the system that corresponds to a given UKL1 board. It will also
 * disconnect the callback functions associated with the datapoint.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *    \li  0 Execution proceeded without a problem.
 *    \li -1 The datapoint could not be callback function and was not deleted.
 *    \li -2 The datapoint was disconnected from the callback but could not be deleted as it did not exist.
 *    \li -3 The datapoint was disconnected from the callback but could not be deleted due to a PVSS issue.
 */
int fwUkl1_DeleteFwLHCbDAQDeviceDatapoint(string ukl1Name) {
  //Name of the datapoint to be deleted.
  const string dpName = fwUkl1_ConvertDimServerNameToFsmDpName(ukl1Name);
  //Name of the datapoint element to be disconnected from callback function.
  const string dpeName = dpName + ".status";
  //Stores the name of the callback function.
  const string callback = "fwUkl1_SetUkl1StateCB";

  //First disconnect the callback function so changes won't get propagated to the hardware.
  if ( 0 != dpDisconnect(callback, dpeName) ) {
    fwUkl1_ErrorMessage("Failed to disconnect the callback function during deletion of " + dpName + " datapoint.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
  }
  //Detele the datapoint.
  int status = _fwUkl1_DeleteDatapoint(dpName);
  //Check the return status and return the appropriate value.
  if ( 0 == status ) {
    fwUkl1_StatusMessage("Sucessfully delete the datapoint: " + dpName,
			 FWUKL1_VERBOSE_MESSAGE_LEVEL);
    return 0;
  } else if ( -2 == status ) {
    fwUkl1_ErrorMessage("Failed to delete the datapoint " + dpName + " as it did not exist.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -2;
  } else if ( -1 == status ) {
    fwUkl1_ErrorMessage("Failed to delete the datapoint " + dpName + " due to a PVSS problem.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -3;
  } else {
    fwUkl1_ErrorMessage("Unrecognised return value from fwUkl1 deletion function.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -3;
  }

}

// =============================================
//  INTERNAL
// =============================================

/*!
 * This is used to configure the register settings of the HwTypeCCPCUKL1 datapoint type.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server. The appropriate
 *           datapoint to be set will be found from this.
 * \param  saveAsDefaults If true then the settings loaded into the datapoint are also saved as
 *           its default values.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *    \li  0 Execution proceeded without a problem.
 *    \li -1 Execution failed and datapoint is in an unuseable state due to a PVSS problem.
 */
int _fwUkl1_ConfigureHwTypeCCPCUKL1Settings(string ukl1Name, bool saveAsDefaults) {
  //Base name of the fwHwType that is being set.
  const string fwHwType = "UKL1";
  //Convert the UKL1 board name into the datapoint name.
  const string dpName = fwUkl1_ConvertDimServerNameToHwDpName(ukl1Name);

  //Write the common settings to the all the datapoints on the board.
  if ( 0 != _fwUkl1_ConfigureHwTypeCCPCUKL1CommonSettings(ukl1Name) ) {
    fwUkl1_ErrorMessage("Failed to configure the common settings for the UKL1 board.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -1;
  }
  fwUkl1_StatusMessage("Common settings configured.",
		       FWUKL1_VERBOSE_MESSAGE_LEVEL);
  //Now we must do the specific settings for all of the hardware type included in the UKL1 type.
  //GBE
  if ( 0 != _fwUkl1_ConfigureHwTypeCCPCGbeSpecificSettings(ukl1Name) ) {
    fwUkl1_ErrorMessage("Failed to configure the GBE settings for the UKL1 board.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -1;
  }
  fwUkl1_StatusMessage("GBE specific settings configured.",
		       FWUKL1_VERBOSE_MESSAGE_LEVEL);
  //TTCrx
  if ( 0 != _fwUkl1_ConfigureHwTypeCCPCTtcrxSpecificSettings(ukl1Name) ) {
    fwUkl1_ErrorMessage("Failed to configure the TTCrx settings for the UKL1 board.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -1;
  }
  fwUkl1_StatusMessage("TTCrx specific settings configured.",
		       FWUKL1_VERBOSE_MESSAGE_LEVEL);
  //FE FPGA
  if ( 0 != _fwUkl1_ConfigureHwTypeCCPCFeFpgaSpecificSettings(ukl1Name) ) {
    fwUkl1_ErrorMessage("Failed to configure the FE FPGA settings for the UKL1 board.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -1;
  }
  fwUkl1_StatusMessage("FE FPGA specific settings configured.",
		       FWUKL1_VERBOSE_MESSAGE_LEVEL);
  //BE FPGA
  if ( 0 != _fwUkl1_ConfigureHwTypeCCPCBeFpgaSpecificSettings(ukl1Name) ) {
    fwUkl1_ErrorMessage("Failed to configure the BE FPGA settings for the UKL1 board.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -1;
  }
  fwUkl1_StatusMessage("BE FPGA specific settings configured.",
		       FWUKL1_VERBOSE_MESSAGE_LEVEL);

  //BE FPGA status
  if ( 0 != _fwUkl1_ConfigureHwTypeCCPCBeFpgaStatusSpecificSettings(ukl1Name) ) {
    fwUkl1_ErrorMessage("Failed to configure the BE FPGA status settings for the UKL1 board.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -1;
  }
  fwUkl1_StatusMessage("BE FPGA status specific settings configured.",
		       FWUKL1_VERBOSE_MESSAGE_LEVEL);

  //Having done all that do we want to save them as the defaults?
  if (saveAsDefaults) {
    //Will get the settings that we have just set and save them to the default values.
    //At present it appears that this function just returns 1, update this to check for errors once it can actually show them!
    if ( 1 == fwHw_saveSettingsAsDefault(fwHwType, dpName) ) {
      fwUkl1_StatusMessage("Successfully applied the default settings.",
			   FWUKL1_DEBUG_MESSAGE_LEVEL);
    }
  }

  //Configured the datapoint!
  return 0;

}

/*!
 * This will set the common settings for the fwHw type UKL1 and all included fwHw types.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *    \li  0 Execution proceeded without a problem.
 *    \li -1 Execution failed and datapoint is in an unuseable state due to a PVSS problem.
 */
int _fwUkl1_ConfigureHwTypeCCPCUKL1CommonSettings(string ukl1Name) {
  //Convert the UKL1 board name into the datapoint name.
  const string dpName = fwUkl1_ConvertDimServerNameToHwDpName(ukl1Name);
  //The common setting to be written.
  dyn_string commonSettings;
  commonSettings[1] = ukl1Name;
  //Write the common setting, this is done recursively so all the subtypes have their common setting
  //updated to.
  if ( !fwHw_setCommonSettings(dpName, commonSettings, TRUE) ) {
    fwUkl1_ErrorMessage("Failed to update the common settings for the datapoint: " + dpName,
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -1;
  }
  return 0;
}

/*!
 * This will set the specific settings for the fwHw type GBE that has been inserted into the
 * UKL1 fwHw type.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *    \li  0 Execution proceeded without a problem.
 *    \li -1 Execution failed and datapoint is in an unuseable state due to a PVSS problem.
 */
int _fwUkl1_ConfigureHwTypeCCPCGbeSpecificSettings(string ukl1Name) {
  //Base name of the fwHwType that is being set.
  const string fwHwType = "UKL1.Gbe";
  //Convert the UKL1 board name into the datapoint name.
  const string dpName = fwUkl1_ConvertDimServerNameToHwDpName(ukl1Name) + ".Gbe";

  //Set the specific settings.
  //This contains a list of all the register names that are to be set.
  dyn_string registerList;
  registerList[1]  = "JTAGID";
  registerList[2]  = "MACSoftReset";
  registerList[3]  = "RxFifoPrtReset";
  registerList[4]  = "TxFifoPrtReset";
  registerList[5]  = "PortEnable";
  registerList[6]  = "RxFifoEnable";
  registerList[7]  = "Mode";
  registerList[8]  = "Clock";
  registerList[9]  = "SPI3ConfigTrnmtGlobal";
  registerList[10] = "SPI3ConfigRcv";
  registerList[11] = "RxStatus";
  registerList[12] = "PHYCtrl";
  registerList[13] = "PHYData";
  registerList[14] = "MDIOControl";
  //This contains the address of all the registers to be set.
  dyn_uint regAddrList;
  regAddrList[1]  = 0x50c;
  regAddrList[2]  = 0x505;
  regAddrList[3]  = 0x59e;
  regAddrList[4]  = 0x620;
  regAddrList[5]  = 0x500;
  regAddrList[6]  = 0x5b3;
  regAddrList[7]  = 0x501;
  regAddrList[8]  = 0x794;
  regAddrList[9]  = 0x700;
  regAddrList[10] = 0x701;
  regAddrList[11] = 0x793;
  regAddrList[12] = 0x680;
  regAddrList[13] = 0x681;
  regAddrList[14] = 0x683;
  //Now we create the settings, these are actually almost the same for each except for the address.
  //1st element is the address to be filled in later.
  //2nd element is the number of 32-bit words for command.
  //3rd element is the check for change rate, set to 2 to match Tell1 change rate and to be compatible with their panels.
  //4th element is send on change only setting.
  dyn_int specificSetting = makeDynAnytype(0x00, 1, 2, 0);
  dyn_dyn_int specificSettings;
  //Loop over all the registers
  for (unsigned count = 1; count <= dynlen(regAddrList); ++count) {
    //Put the appropriate register address in the settings array.
    specificSetting[1] = regAddrList[count];
    //Now put the specific settings in the appropriate element of the array containing all the settings for each register.
    specificSettings[count] = specificSetting;
  }

  //Now write all these to the datapoint.
  if ( !fwHw_setSpecificSettings(dpName, registerList, specificSettings) ) {
    fwUkl1_ErrorMessage("Failed to set the GBE specific settings for the datapoint: " + dpName + ".",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -1;
  }

  //Now setup the PROM.
  if ( 0 != _fwUkl1_ConfigureHwTypeCCPCGBEPORTSpecificSettings(ukl1Name) ) {
    fwUkl1_ErrorMessage("Failed to configure the GBE PORT settings for the UKL1 board.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -1;
  }

  //Now setup the four ports.
  if ( 0 != _fwUkl1_ConfigureHwTypeCCPCGBEPORTSpecificSettings(ukl1Name) ) {
    fwUkl1_ErrorMessage("Failed to configure the GBE PORT settings for the UKL1 board.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -1;
  }

  //And we are done.
  return 0;
}

/*!
 * This will set the specific settings for the fwHw GBEPORM type that has been inserted into the
 * UKL1 fwHw type.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *    \li  0 Execution proceeded without a problem.
 *    \li -1 Execution failed and datapoint is in an unuseable state due to a PVSS problem.
 */
int _fwUkl1_ConfigureHwTypeCCPCGBEPORMSpecificSettings(string ukl1Name) {
  //Base name of the fwHwType that is being set.
  const string fwHwType = "UKL1.Gbe.PROM";
  //Convert the UKL1 board name into the datapoint name.
  const string dpName = fwUkl1_ConvertDimServerNameToHwDpName(ukl1Name) + ".Gbe.PROM";

    //Set the specific settings.
    //This contains a list of all the register names that are to be set.
    dyn_string registerList;
    registerList[1] = "Id";
    //Now we create the settings, these are actually almost the same for each except for the address.
    //1st element is the bus number that the PROM is on, always 1.
    //2nd element is the address of the PROM on the bus, always 0x57.
    //3rd element is the address of the PROM registers at register 0x57, always 0.
    //4th element is the number of bytes that should be written to that register, 46 for compatibility with Tell1.
    //5th element is the page size, again 46 for compatibility with Tell1.
    //6th element is the the type of I2C interface to use, always FWCCPC_I2C_COMBINED for the PROM.
    //7th element is the negative acknowledge, NACK, leave it enabled as I think it tells us if it failed. Always 1. 
    //8th element is the refresh rate, always 2.
    //9th element is the on data change settings always 0.
    dyn_int specificSetting = makeDynAnytype(1, 0x57, 0, 46, 46, FWCCPC_I2C_COMBINED, 0, 2, 0);
    dyn_dyn_int specificSettings;
    
    //Now write all these to the datapoint.
    if ( !fwHw_setSpecificSettings(dpName, registerList, specificSettings) ) {
      fwUkl1_ErrorMessage("Failed to set the FE FPGA specific settings for the datapoint: " + dpName + ".",
			  FWUKL1_DEBUG_MESSAGE_LEVEL);
      return -1;
    }

    //Now setup the four ports.
    if ( 0 != _fwUkl1_ConfigureHwTypeCCPCChannelSpecificSettings(ukl1Name, feFpga) ) {
      fwUkl1_ErrorMessage("Failed to configure the channel settings for the UKL1 board.",
			  FWUKL1_DEBUG_MESSAGE_LEVEL);
      return -1;
    }

  return 0;
}

/*!
 * This will set the specific settings for the fwHw GBEPORT type that has been inserted into the
 * UKL1 fwHw type.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *    \li  0 Execution proceeded without a problem.
 *    \li -1 Execution failed and datapoint is in an unuseable state due to a PVSS problem.
 */
int _fwUkl1_ConfigureHwTypeCCPCGBEPORTSpecificSettings(string ukl1Name) {
  //Loop over the four ports.
  for (unsigned port = 0; port < FWUKL1_GBE_PORTS; ++port) {
    //Base name of the fwHwType that is being set.
    const string fwHwType = "UKL1.Gbe.PORT" + port;
    //Convert the UKL1 board name into the datapoint name.
    const string dpName = fwUkl1_ConvertDimServerNameToHwDpName(ukl1Name) + ".Gbe.PORT" + port;

    //Set the specific settings.
    //This contains a list of all the register names that are to be set.
    dyn_string registerList;
    registerList[1]  = "Duplex";
    registerList[2]  = "MAC";
    registerList[3]  = "Config";
    registerList[4]  = "TXFifoThreshold";
    registerList[5]  = "TXFifoLowWtrmrk";
    registerList[6]  = "TXFifoHighWtrmrk";
    registerList[7]  = "PHYControl";
    registerList[8]  = "PHYStatus";
    registerList[9]  = "TXStat";
    registerList[10] = "RXStat";

    //This contains the address of all the registers to be set.
    dyn_uint regAddrList;
    //Calculate the start address in the block that contains the per port settings.
    const unsigned offset = (0x80*(port));
    regAddrList[1]  = offset + 0x02;
    regAddrList[2]  = offset + 0x10;
    regAddrList[3]  = offset + 0x18;
    regAddrList[4]  = 0x614 + port;
    regAddrList[5]  = 0x60a + port;
    regAddrList[6]  = 0x600 + port;
    regAddrList[7]  = offset + 0x60;
    regAddrList[8]  = offset + 0x61;
    regAddrList[9]  = offset + 0x40;
    regAddrList[10] = offset + 0x20;

    //This contains the size in bytes of the register to be read.
    dyn_uint regSizeList;
    regSizeList[1]  = 1;
    regSizeList[2]  = 1;
    regSizeList[3]  = 1;
    regSizeList[4]  = 1;
    regSizeList[5]  = 1;
    regSizeList[6]  = 1;
    regSizeList[7]  = 1;
    regSizeList[8]  = 1;
    regSizeList[9]  = 25;
    regSizeList[10] = 26;

    //Now we create the settings, these are actually almost the same for each except for the address.
    //1st element is the address to be filled in later.
    //2nd element is the number of 32-bit words for command.
    //3rd element is the check for change rate, 4th is send on change only setting.
    dyn_int specificSetting = makeDynAnytype(0x00, 1, 2, 0);
    dyn_dyn_int specificSettings;
    //Loop over all the registers
    for (unsigned count = 1; count <= dynlen(regAddrList); ++count) {
      //Put the appropriate register address in the settings array.
      specificSetting[1] = regAddrList[count];
      //Put the appropriate size of the register in.
      specificSetting[2] = regSizeList[count];
      //Now put the specific settings in the appropriate element of the array containing all the settings for each register.
      specificSettings[count] = specificSetting;
    }
    
    //Now write all these to the datapoint.
    if ( !fwHw_setSpecificSettings(dpName, registerList, specificSettings) ) {
      fwUkl1_ErrorMessage("Failed to set the GBE PORT specific settings for the datapoint: " + dpName + ".",
			  FWUKL1_DEBUG_MESSAGE_LEVEL);
      return -1;
    }
  }

  return 0;
}
  
/*!
 * This will set the specific settings for the fwHw IngressFPGA type that has been inserted into the
 * UKL1 fwHw type.
 * \todo
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *    \li  0 Execution proceeded without a problem.
 *    \li -1 Execution failed and datapoint is in an unuseable state due to a PVSS problem.
 */
int _fwUkl1_ConfigureHwTypeCCPCFeFpgaSpecificSettings(string ukl1Name) {
  //Loop over the ingress/FE FPGAs.
  for (unsigned feFpga = 0; feFpga < FWUKL1_FRONT_FPGAS; ++feFpga) {
    //Base name of the fwHwType that is being set.
    const string fwHwType = "UKL1.FeFpga" + feFpga;
    //Convert the UKL1 board name into the datapoint name.
    const string dpName = fwUkl1_ConvertDimServerNameToHwDpName(ukl1Name) + ".FeFpga" + feFpga;

    //Set the specific settings.
    //This contains a list of all the register names that are to be set.
    dyn_string registerList;
    registerList[1] = "Disable";
    registerList[2] = "Emulator";
    registerList[3] = "EmulatorHpdPixelMode";
    registerList[4] = "EmulatorEvent";
    registerList[5] = "ResetStatusBufferPointer";
    //This contains the address of all the registers to be set.
    dyn_uint regAddrList;
    regAddrList[1] = fwUkl1_FpgaizeAddress(24);
    regAddrList[2] = fwUkl1_FpgaizeAddress(20);
    regAddrList[3] = fwUkl1_FpgaizeAddress(20);
    regAddrList[4] = fwUkl1_FpgaizeAddress(20);
    regAddrList[5] = fwUkl1_FpgaizeAddress(21);
    //Now we create the settings, these are actually almost the same for each except for the address.
    //1st element is the address to be filled in later.
    //2nd element is the number of 32-bit words for command.
    //3rd element is the check for change rate, 4th is send on change only setting.
    dyn_int specificSetting = makeDynAnytype(0x00, FWCCPC_LBUS_BITS_32, 1, 0, 0);
    dyn_dyn_int specificSettings;
    //Loop over all the registers
    for (unsigned count = 1; count <= dynlen(regAddrList); ++count) {
      //Put the appropriate register address in the settings array.
      specificSetting[1] = regAddrList[count];
      //Now put the specific settings in the appropriate element of the array containing all the settings for each register.
      specificSettings[count] = specificSetting;
    }
    
    //Now write all these to the datapoint.
    if ( !fwHw_setSpecificSettings(dpName, registerList, specificSettings) ) {
      fwUkl1_ErrorMessage("Failed to set the FE FPGA specific settings for the datapoint: " + dpName + ".",
			  FWUKL1_DEBUG_MESSAGE_LEVEL);
      return -1;
    }

    //Now setup the four ports.
    if ( 0 != _fwUkl1_ConfigureHwTypeCCPCChannelSpecificSettings(ukl1Name, feFpga) ) {
      fwUkl1_ErrorMessage("Failed to configure the channel settings for the UKL1 board.",
			  FWUKL1_DEBUG_MESSAGE_LEVEL);
      return -1;
    }

  }

  return 0;
}

/*!
 * This will set the specific settings for the fwHw InputChannel type that has been inserted into the
 * UKL1 fwHw type.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  feFpga Number of the FE/ingress FPGA that the channels are being configured for.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *    \li  0 Execution proceeded without a problem.
 *    \li -1 Execution failed and datapoint is in an unuseable state due to a PVSS problem.
 */
int _fwUkl1_ConfigureHwTypeCCPCChannelSpecificSettings(string ukl1Name, unsigned feFpga) {
  //Loop over the input channels.
  for (unsigned channel = 0; channel < FWUKL1_FPGA_CHANNELS; ++channel) {
    //Base name of the fwHwType that is being set.
    const string fwHwType = "UKL1.FeFpga" + feFpga + ".Channel"+ channel;
    //Convert the UKL1 board name into the datapoint name.
    const string dpName = fwUkl1_ConvertDimServerNameToHwDpName(ukl1Name) + ".FeFpga" + feFpga + ".Channel"+ channel;

    //Set the specific settings.
    //This contains a list of all the register names that are to be set.
    dyn_string registerList;
    registerList[1] = "Disable";

    //This contains the address of all the registers to be set.
    dyn_uint regAddrList;
    regAddrList[1] = fwUkl1_FpgaizeAddress(25+feFpga);
    //Now we create the settings, these are actually almost the same for each except for the address.
    //1st element is the address to be filled in later.
    //2nd element is the number of 32-bit words for command.
    //3rd element is the check for change rate, 4th is send on change only setting.
    dyn_int specificSetting = makeDynAnytype(0x00, FWCCPC_LBUS_BITS_32, 1, 0, 0);
    dyn_dyn_int specificSettings;
    //Loop over all the registers
    for (unsigned count = 1; count <= dynlen(regAddrList); ++count) {
      //Put the appropriate register address in the settings array.
      specificSetting[1] = regAddrList[count];
      //Now put the specific settings in the appropriate element of the array containing all the settings for each register.
      specificSettings[count] = specificSetting;
    }
    
    //Now write all these to the datapoint.
    if ( !fwHw_setSpecificSettings(dpName, registerList, specificSettings) ) {
      fwUkl1_ErrorMessage("Failed to set the channel specific settings for the datapoint: " + dpName + ".",
			  FWUKL1_DEBUG_MESSAGE_LEVEL);
      return -1;
    }
  }

  return 0;
}

/*!
 * This will set the specific settings for the fwHw EgressFPGA type that has been inserted into the
 * UKL1 fwHw type.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *    \li  0 Execution proceeded without a problem.
 *    \li -1 Execution failed and datapoint is in an unuseable state due to a PVSS problem.
 */
int _fwUkl1_ConfigureHwTypeCCPCBeFpgaSpecificSettings(string ukl1Name) {
//Base name of the fwHwType that is being set.
  const string fwHwType = "UKL1.BeFpga";
  //Convert the UKL1 board name into the datapoint name.
  const string dpName = fwUkl1_ConvertDimServerNameToHwDpName(ukl1Name) + ".BeFpga";

  //Set the specific settings.
  //This contains a list of all the register names that are to be set.
  dyn_string registerList;
  registerList[1]  = "MacDestinationAddress";
  registerList[2]  = "MacSourceAddress";
  registerList[3]  = "IpDestinationAddress";
  registerList[4]  = "IpSourceAddress";
  registerList[5]  = "Protocol";
  registerList[6]  = "TypeOfService";
  registerList[7]  = "TimeToLive";
  registerList[8]  = "Type";
  registerList[9]  = "Version";
  registerList[10] = "FixedMepDestinationAddress";
  registerList[11] = "DontWaitForMepDestBrdcst";
  registerList[12] = "Throttle";
  registerList[13] = "ThrottlePolarity";
  registerList[14] = "TfcDecoder";
  registerList[15] = "PollGbePorts";
  registerList[16] = "PartitionId";
  registerList[17] = "MepPackingFactor";
  registerList[18] = "ZeroSuppression";
  registerList[19] = "HpdPixelMode";
  registerList[20] = "Latency";
  registerList[21] = "L1Id";
  registerList[22] = "DataFragmentSize";

  //This contains the address of all the registers to be set.
  dyn_int regAddrList;
  regAddrList[1]  = fwUkl1_FpgaizeAddress(0);
  regAddrList[2]  = fwUkl1_FpgaizeAddress(3);
  regAddrList[3]  = fwUkl1_FpgaizeAddress(6);
  regAddrList[4]  = fwUkl1_FpgaizeAddress(8);
  regAddrList[5]  = fwUkl1_FpgaizeAddress(10);
  regAddrList[6]  = fwUkl1_FpgaizeAddress(11);
  regAddrList[7]  = fwUkl1_FpgaizeAddress(12);
  regAddrList[8]  = fwUkl1_FpgaizeAddress(13);
  regAddrList[9]  = fwUkl1_FpgaizeAddress(14);
  regAddrList[10] = fwUkl1_FpgaizeAddress(15);
  regAddrList[11] = fwUkl1_FpgaizeAddress(15);
  regAddrList[12] = fwUkl1_FpgaizeAddress(15);
  regAddrList[13] = fwUkl1_FpgaizeAddress(15);
  regAddrList[14] = fwUkl1_FpgaizeAddress(15);
  regAddrList[15] = fwUkl1_FpgaizeAddress(15);
  regAddrList[16] = fwUkl1_FpgaizeAddress(16);
  regAddrList[17] = fwUkl1_FpgaizeAddress(18);
  regAddrList[18] = fwUkl1_FpgaizeAddress(19);
  regAddrList[19] = fwUkl1_FpgaizeAddress(19);
  regAddrList[20] = fwUkl1_FpgaizeAddress(22);
  regAddrList[21] = fwUkl1_FpgaizeAddress(23);
  regAddrList[22] = fwUkl1_FpgaizeAddress(29);

  //This contains a list of the number of 16-bit words per register.
  dyn_uint byteNumList;
  byteNumList[1]  = 3;
  byteNumList[2]  = 3;
  byteNumList[3]  = 2;
  byteNumList[4]  = 2;
  byteNumList[5]  = 1;
  byteNumList[6]  = 1;
  byteNumList[7]  = 1;
  byteNumList[8]  = 1;
  byteNumList[9]  = 1;
  byteNumList[10] = 1;
  byteNumList[11] = 1;
  byteNumList[12] = 1;
  byteNumList[13] = 1;
  byteNumList[14] = 1;
  byteNumList[15] = 1;
  byteNumList[16] = 2;
  byteNumList[17] = 1;
  byteNumList[18] = 1;
  byteNumList[19] = 1;
  byteNumList[20] = 1;
  byteNumList[21] = 1;
  byteNumList[22] = 1;

  //Now we create the settings, these are actually almost the same for each except for the address.
  //1st element is the address to be filled in later.
  //2nd element is the number of 32-bit words for command.
  //3rd element is the check for change rate, 4th is send on change only setting.
  dyn_uint specificSetting = makeDynAnytype(0x00, FWCCPC_LBUS_BITS_32, 1, 0, 0);
  dyn_dyn_uint specificSettings;
  //Loop over all the registers
  for (unsigned count = 1; count <= dynlen(regAddrList); ++count) {
    //Put the appropriate register address in the settings array.
    specificSetting[1] = regAddrList[count];
    //Put the appropriate size of register into the array.
    specificSetting[3] = byteNumList[count];
    //Now put the specific settings in the appropriate element of the array containing all the settings for each register.
    specificSettings[count] = specificSetting;
  }

  //Now write all these to the datapoint.
  if ( !fwHw_setSpecificSettings(dpName, registerList, specificSettings) ) {
    fwUkl1_ErrorMessage("Failed to set the BE FPGA specific settings for the datapoint: " + dpName + ".",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -1;
  }

  return 0;

}

/*!
 * This will set the specific settings for the fwHw EgressFPGAStatus type that has been inserted into the
 * UKL1 fwHw type.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *    \li  0 Execution proceeded without a problem.
 *    \li -1 Execution failed and datapoint is in an unuseable state due to a PVSS problem.
 */
int _fwUkl1_ConfigureHwTypeCCPCBeFpgaStatusSpecificSettings(string ukl1Name) {
//Base name of the fwHwType that is being set.
  const string fwHwType = "UKL1.BeFpgaStatus";
  //Convert the UKL1 board name into the datapoint name.
  const string dpName = fwUkl1_ConvertDimServerNameToHwDpName(ukl1Name) + ".BeFpgaStatus";

  //Set the specific settings.
  //This contains a list of all the register names that are to be set.
  dyn_string registerList;
  registerList[1]  = "TTCId";
  registerList[2]  = "TfcFifoRequest";
  registerList[3]  = "EventMergerReadState";

  registerList[4]  = "SPI3TxBusy";
  registerList[5]  = "MepFifoFull";
  registerList[6]  = "Throttled";

  registerList[7]  = "TfcL0Reset";
  registerList[8]  = "TfcL1Reset";
  registerList[9]  = "TfcBShort";
  registerList[11] = "Tpa";
  registerList[12] = "FragmentReady";
  registerList[13] = "MepReady";
  registerList[14] = "FragmentRequest";
  registerList[15] = "EventQValid";

  registerList[16] = "FragmenterState";

  registerList[17] = "EthernetQValid";
  registerList[18] = "SequenceError";
  registerList[19] = "MepOccupancy";
  registerList[20] = "TfcFifoOccupany";


  //This contains the address of all the registers to be set.
  dyn_int regAddrList;
  regAddrList[1]  = fwUkl1_FpgaizeAddress(32);
  regAddrList[2]  = fwUkl1_FpgaizeAddress(32);
  regAddrList[3]  = fwUkl1_FpgaizeAddress(32);

  regAddrList[4]  = fwUkl1_FpgaizeAddress(34);
  regAddrList[5]  = fwUkl1_FpgaizeAddress(34);
  regAddrList[6]  = fwUkl1_FpgaizeAddress(34);

  regAddrList[7]  = fwUkl1_FpgaizeAddress(35);
  regAddrList[8]  = fwUkl1_FpgaizeAddress(35);
  regAddrList[9]  = fwUkl1_FpgaizeAddress(35);
  regAddrList[10] = fwUkl1_FpgaizeAddress(35);
  regAddrList[11] = fwUkl1_FpgaizeAddress(35);
  regAddrList[12] = fwUkl1_FpgaizeAddress(35);
  regAddrList[13] = fwUkl1_FpgaizeAddress(35);
  regAddrList[14] = fwUkl1_FpgaizeAddress(35);
  regAddrList[15] = fwUkl1_FpgaizeAddress(35);

  regAddrList[16] = fwUkl1_FpgaizeAddress(36);

  regAddrList[17] = fwUkl1_FpgaizeAddress(37);
  regAddrList[18] = fwUkl1_FpgaizeAddress(37);
  regAddrList[19] = fwUkl1_FpgaizeAddress(37);
  regAddrList[20] = fwUkl1_FpgaizeAddress(37);
  //Now we create the settings, these are actually almost the same for each except for the address.
  //1st element is the address to be filled in later.
  //2nd element is the number of 32-bit words for command.
  //3rd element is the rate at which it checks for change, 1sec always
  //4th is send on change only setting, again set.
  dyn_uint specificSetting = makeDynAnytype(0x00, FWCCPC_LBUS_BITS_32, 1, 1, 1);
  dyn_dyn_uint specificSettings;
  //Loop over all the registers
  for (unsigned count = 1; count <= dynlen(regAddrList); ++count) {
    //Put the appropriate register address in the settings array.
    specificSetting[1] = regAddrList[count];
    //Now put the specific settings in the appropriate element of the array containing all the settings for each register.
    specificSettings[count] = specificSetting;
  }

  //Now write all these to the datapoint.
  if ( !fwHw_setSpecificSettings(dpName, registerList, specificSettings) ) {
    fwUkl1_ErrorMessage("Failed to set the BE FPGA status specific settings for the datapoint: " + dpName + ".",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -1;
  }

  return 0;

}

/*!
 * This will set the specific settings for the fwHw TTCRX type that has been inserted into the
 * UKL1 fwHw type.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *    \li  0 Execution proceeded without a problem.
 *    \li -1 Execution failed and datapoint is in an unuseable state due to a PVSS problem.
 */
int _fwUkl1_ConfigureHwTypeCCPCTtcrxSpecificSettings(string ukl1Name) {
  //Base name of the fwHwType that is being set.
  const string fwHwType = "UKL1.Ttcrx";
  //Convert the UKL1 board name into the datapoint name.
  const string dpName = fwUkl1_ConvertDimServerNameToHwDpName(ukl1Name) + ".Ttcrx";

  //Set the specific settings.
  //This contains a list of all the register names that are to be set.
  dyn_string registerList;
  registerList[1]  = "IACId";
  registerList[2]  = "I2CId";
  registerList[3]  = "FineDelay1";
  registerList[4]  = "FineDelay2";
  registerList[5]  = "CoarseDelay";
  registerList[6]  = "Control";
  registerList[7]  = "Status";
  registerList[8]  = "Config1";
  registerList[9]  = "Config2";
  registerList[10] = "Config3";
  registerList[11] = "SingleErrCnt";
  registerList[12] = "DoubleErrCnt";
  registerList[13] = "SEUErrCnt";
  registerList[14] = "BunchCnt";
  registerList[15] = "EventCnt";
  //This contains the address of all the registers to be set.
  dyn_uint regAddrList;
  regAddrList[1]  = 16;
  regAddrList[2]  = 18;
  regAddrList[3]  = 0;
  regAddrList[4]  = 1;
  regAddrList[5]  = 2;
  regAddrList[6]  = 3;
  regAddrList[7]  = 22;
  regAddrList[8]  = 19;
  regAddrList[9]  = 20;
  regAddrList[10] = 21;
  regAddrList[11] = 8;
  regAddrList[12] = 10;
  regAddrList[13] = 11;
  regAddrList[14] = 24;
  regAddrList[15] = 26;
  //This contains the number of bytes that need to be set for the each of the registers.
  dyn_uint numBytes;
  numBytes[1]  = 2;
  numBytes[2]  = 1;
  numBytes[3]  = 1;
  numBytes[4]  = 1;
  numBytes[5]  = 1;
  numBytes[6]  = 1;
  numBytes[7]  = 1;
  numBytes[8]  = 1;
  numBytes[9]  = 1;
  numBytes[10] = 1;
  numBytes[11] = 2;
  numBytes[12] = 1;
  numBytes[13] = 1;
  numBytes[14] = 2;
  numBytes[15] = 3;

  //Now we create the settings, these are actually almost the same for each except for the address.
  //1st element is the bus number that the TTCrx is on, always 1.
  //2nd element is the address of the TTCrx on the bus, again always 0x42 the address the register addresses should be written to.
  //3rd element is the address of the TTCrx registers on the TTCrx chip, variable.
  //4th element is the number of bytes that should be written to that register, again variable.
  //5th element is the page size, don't really know but if -1 it is infinite and we can ignore it.
  //6th element is the the type of I2C interface to use, always FWCCPC_I2C_SEPARATED for the TTCrx chip.
  //7th element is the negative acknowledge, NACK, leave it enabled as I think it tells us if it failed. Always 1. 
  //8th element is the refresh rate, always 0.
  //9th element is the on data change settings always 0.
  dyn_int specificSetting = makeDynAnytype(1, 0x42, 0, 0, -1, FWCCPC_I2C_SEPARATED, 1, 0, 0);
  dyn_dyn_int specificSettings;
  //Loop over all the registers
  for (unsigned count = 1; count <= dynlen(regAddrList); ++count) {
    //Put the appropriate register address in the settings array.
    specificSetting[3] = regAddrList[count];
    specificSetting[4] = numBytes[count];
    //Now put the specific settings in the appropriate element of the array containing all the settings for each register.
    specificSettings[count] = specificSetting;
  }

  //Now write all these to the datapoint.
  if ( !fwHw_setSpecificSettings(dpName, registerList, specificSettings) ) {
    fwUkl1_ErrorMessage("Failed to set the BE FPGA specific settings for the datapoint: " + dpName + ".",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -1;
  }

  return 0;
}

/*!
 * This is used to delete a given datapoint type. If that type has any elements subscribed to the DIM server they
 * must first be unsubscribed using fwHw_unsubscribe.
 *
 * \param  dpName Name of the data point to be deleted.
 * \return int Returns the following values:
 *    \li  0 Successfully deleted datapoint.
 *    \li -1 Datapoint could not be deleted due to a PVSS error.
 *    \li -2 Datapoint did not exist, although technically a failure as function did not perform required task,
 *             the specified datapoint does not exist in the system which is what was requested.
 */
int _fwUkl1_DeleteDatapoint(string dpName) {
  //First check to see if it exists.
  if ( dpExists(dpName) ) {
    //Lets delete it then...
    if ( 0 == dpDelete(dpName) ) {
      fwUkl1_StatusMessage("Successfully deleted the datapoint: " + dpName,
			   FWUKL1_DEBUG_MESSAGE_LEVEL);
      return 0;
    } else {
      fwUkl1_ErrorMessage("Failed to delete datapoint: " + dpName,
			  FWUKL1_INFO_MESSAGE_LEVEL);
      return -1;
    }
  } else {
    //If it doesn't exist then there is no point in trying to delete it!
    fwUkl1_WarningMessage("Datapoint requested for deletion: " + dpName + ", didn't exist.",
			  FWUKL1_INFO_MESSAGE_LEVEL);
    return -2;
  }
}

/*!
 * Provides a wrapper for dpSet, which allows the value from a single datapoint value to be set.
 *
 * \param  dpName Name of the datapoint whose value is to be set.
 * \param  data Data that is to be set to the datapoint. It of type anytype to allow the
 *           setting of any datapoint value.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *
 * Further to the standard dpSet functionality it will also log any errors that occur to the UKL1
 * critical error datapoint and indicate a problem through the return value.
 */
int _fwUkl1_DpSet(string dpName, anytype data) {
  //Write the data to the dp.
  int pvssErr = dpSet(dpName, data);
  //Now check the return value.
  if (0 == pvssErr) {
    //Everything is ok.
    fwUkl1_StatusMessage("Set datapoint: " + dpName, FWUKL1_VERBOSE_MESSAGE_LEVEL);
    return 0;
  } else {
    //There was an error reading from the datapoint. Indicate this.
    string msg = "Failed to write to " + dpName;
    _fwUkl1_LogDpError(msg);
    return -1;
  }
}

/*!
 * Provides a wrapper for dpGet, which allows the value from a single datapoint value to be
 * retrieved.
 *
 * \param  dpName Name of the datapoint whose value is to be retrieved.
 * \param  data Data that is to be retrieved from the datapoint. It of type anytype to allow the
 *         retrival of any datapoint value. Returned by reference.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *
 * Further to the standard dpGet functionality it will also log any errors that occur to the UKL1
 * critical error datapoint and indicate the problem through the return value.
 */
int _fwUkl1_DpGet(string dpName, anytype& data) {
  //Write the data to the dp.
  int pvssErr = dpGet(dpName, data);
  //Now check the return value.
  if (0 == pvssErr) {
    //Everything is ok.
    return 0;
  } else {
    //There was an error reading from the datapoint. Indicate this.
    string msg = "Failed to read from " + dpName;
    _fwUkl1_LogDpError(msg);
    return -1;
  }
}

//@}


// ======================
//  UKL1 RESET FUNCTIONS
// ======================

/** @defgroup SectionReset Provides the functions for various levels of resets of the UKL1 board.
 *  @{
 */

/*!
 * Performs a reset of the FPGAs on the L1 board. Effects still under discussion.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \return int Defines the state of the function after execution. The states are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int fwUkl1_ResetFpgas(string ukl1Name) {
  //The reset is performed by setting the GPIO line 5 low for a few 10s of milliseconds and then setting it back to high.
  //First it must be enabled for output.
  //Note: we could check that it is enabled for output, but it would always require at least 1 hardware
  //  access, so that 1 access might as well be the setting.
  int callStatus = fwUkl1_GpioEnable(ukl1Name, FWCCPC_GPIO_LINE_5, FWCCPC_GPIO_OUTPUT, "Set GPIO output in reset");
  //Check for errors.
  if (0 != callStatus) {
    fwUkl1_ErrorMessage("Failed to set GPIO line 5 to output during reset.", FWUKL1_DEBUG_MESSAGE_LEVEL);
    return callStatus;
  }
  //Now we set the GPIO line to be low to active the reset.
  int callStatus = fwUkl1_GpioSet(ukl1Name, FWCCPC_GPIO_LINE_5, FWCCPC_GPIO_LOW, "Set GPIO low in reset");
  //Check for errors.
  if (0 != callStatus) {
    fwUkl1_ErrorMessage("Failed to set GPIO line 5 to low during reset.", FWUKL1_DEBUG_MESSAGE_LEVEL);
    return callStatus;
  }
  //Now wait to allow reset to take affect.
  delay(0,20);
  //Now return it to the high state.
  callStatus = fwUkl1_GpioSet(ukl1Name, FWCCPC_GPIO_LINE_5, FWCCPC_GPIO_HIGH, "Set GPIO high in reset");
  //Check for errors.
  if (0 != callStatus) {
    fwUkl1_ErrorMessage("Failed to set GPIO line 5 to high during reset.", FWUKL1_DEBUG_MESSAGE_LEVEL);
  }
  //Now return.
  return callStatus;
}

/*!
 * Performs a reload of the FPGA firmware on the L1 board. It will reset all values back to their defaults.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \return int Defines the state of the function after execution. The states are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int fwUkl1_ReloadFpgas(string ukl1Name) {
  //The reload is performed by setting the GPIO line 6 low for a few 10s of milliseconds and then setting it back to high.
  //First it must be enabled for output.
  //Note: we could check that it is enabled for output, but it would always require at least 1 hardware
  //  access, so that 1 access might as well be the setting.
  int callStatus = fwUkl1_GpioEnable(ukl1Name, FWCCPC_GPIO_LINE_6, FWCCPC_GPIO_OUTPUT, "Set GPIO output in reload");
  //Check for errors.
  if (0 != callStatus) {
    fwUkl1_ErrorMessage("Failed to set GPIO line 6 to output during reload.", FWUKL1_DEBUG_MESSAGE_LEVEL);
    return callStatus;
  }
  //Now we set the GPIO line to be low to active the reload.
  callStatus = fwUkl1_GpioSet(ukl1Name, FWCCPC_GPIO_LINE_6, FWCCPC_GPIO_LOW, "Set GPIO low in reload");
  //Check for errors.
  if (0 != callStatus) {
    fwUkl1_ErrorMessage("Failed to set GPIO line 6 to low during reload.", FWUKL1_DEBUG_MESSAGE_LEVEL);
    return callStatus;
  }
  //Now wait to allow reset to take affect.
  delay(0,20);
  //Now return it to the high state.
  callStatus = fwUkl1_GpioSet(ukl1Name, FWCCPC_GPIO_LINE_6, FWCCPC_GPIO_HIGH, "Set GPIO high in reload");
  //Check for errors.
  if (0 != callStatus) {
    fwUkl1_ErrorMessage("Failed to set GPIO line 6 to high during reload.", FWUKL1_DEBUG_MESSAGE_LEVEL);
  }
  //Now return.
  return callStatus;
}

//@}


// ==============================
//  FPGA CONFIGURATION FUNCTIONS
// ==============================

/** @defgroup SectionFpgaConfiguration
 *  @{
 */

/*!
 * This writes to a given set of registers on the BE FPGAs.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           written to, which corresponds to the appropriate hardware register. The following values are permitted:
 *   \li MacDestinationAddress A 12 character string that gives the hex number for the MAC address, can either be a continuous
 *         string of the address or may posses a colon delimiter every two characters,
 *         e.g. 000e0cb02d19 or 00:0e:0c:b0:2d:19.
 *   \li MacSourceAddress A 12 character string that gives the hex number for the MAC address, can either be a continuous
 *         string of the address or may posses a colon delimiter every two characters,
 *         e.g. 000e0cb02d19 or 00:0e:0c:b0:2d:19.
 *   \li IpDestinationAddress A string that contains either the address in hex format e.g. c0a0c0c0 or in a typical IP4 format
 *         e.g. 192.168.192.192.
 *   \li IpSourceAddress A string that contains either the address in hex format e.g. c0a0c0c0 or in a typical IP4 format
 *         e.g. 192.168.192.192.
 *   \li Protocol Default format described below.
 *   \li TypeOfService Default format described below.
 *   \li TimeToLive Default format described below.
 *   \li Type Default format described below.
 *   \li Version Default format described below.
 *   \li PartitionId Must contain an 8 character string representing the two 16-bit words of data for the partition ID.
 *   \li L1Id An 8-bit value, 2-character string. This will be present in the output data packet and MUST be unique for each L1.
 *         This is set correctly by the FSM and generally should not be User defined, except maybe when the User is an expert.
 *   \li ZeroSuppression String should be either `Enable' or `Disable' to enable or disable zero suppression across
 *         all the Ingress/FE FPGAs. Case sensitive.
 *   \li HpdPixelMode String should be either `ALICE' or `LHCb', which should correspond to the mode of operation
 *         required from the pixel chips. Case sensitive.
 *   \li Latency Default format described below.
 *   \li MepPackingFactor Default format described below.
 *   \li FixedMepDestinationAddress `Enable' or `Disable' to enable or disable a fixed multi-event packet destination address.
 *   \li DontWaitForMepDestBrdcst `Don't wait' or `Wait' to enable or disable the waiting for a multi-event packet destination
 *         address broadcast, respectively.
 *   \li Throttle `Enable' or `Disable' to enable or disable the throttle.
 *   \li ThrottlePolarity `Invert' or `Normal' to select the polarity of the throttle output.
 *   \li TfcDecoder `Enable' or `Disable' to enable or disable the decoding of TFC commands. The board will not operate without
 *         this setting enabled.
 *   \li PollGbePorts `Enable' or `Disable' to enable the Egress FPGA to send data to the GBE card for transmission.
 *   \li DataFragmentSize A decimal number specifying the size of the data packet that is transmitted via the GBE card.
 *         It is the size of the fragment of the multi-event packet that is transmitted. Max 9000.
 * \param  data By default a 16-bit value that is to be written to the register represented as a 4-character hex string,
 *           it need not be zero padded. Certain settings accept or require an alternative format as described for each register above.
 * \param  callStatusList This array of ints stores the return code from each write call to the fwCcpc library. The return code from this
 *           function will indicate if any of the writes failed and by checking the array the specific write failures can be
 *           identified. The order of return codes will match the ordering of the list of register names. Returned by reference.
 *           The following are the possible return codes (see fwCcpc lib):
 *   \li  0 Execution proceeded without a problem.
 *   \li >0 Error found and related with CCPC Server.
 *   \li -1 Error found and related with PVSS.
 * \param  verifyWrite If this is set to TRUE then the register will be read back from to ensure that
 *           the data written corresponds to that read. Default: FALSE.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *            The error log contains information about which register failed to write.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *            The error log contains information about which register failed to write.
 *   \li -2 Execution failed due to a problem with data or name format and no hardware/datapoint access was performed.
 */
int fwUkl1_BeFpgaWrite(string ukl1Name, dyn_string registerNames, dyn_string data, dyn_int& callStatusList, bool verifyWrite=FALSE) {
  //Check that the register names and data arrays are the same size.
  //Save only the lenght of one of the arrays as the other must be the same size, in this case it is the registerNames.
  const unsigned numRegs = dynlen(registerNames);
  if ( numRegs != dynlen(data) ) {
    fwUkl1_ErrorMessage("registerNames size: " + numRegs + " is not equal to the data size: " + dynlen(data) + ", when writting to BE FPGA",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -2;
  }

  //The overall status of the function.
  int status = 0;

  //Holds the data in the form that the fwCcpc_write function expects.
  dyn_dyn_char ddcData;
  //Holds the mask for each element.
  dyn_dyn_char masks;

  //Converts the data given as strings in the calling arguments to byte arguments required by the fwCcpc_write.
  //Also it creates the masks required for each write and pads the register names, such that they refer to the appropriate
  //data point elements.
  status = _fwUkl1_ParseBeFpgaWriteRegisters(ukl1Name, registerNames, data, ddcData, masks);
  //Check that we were actually able to parse the data.
  if ( 0 == status ) {
    //The data and registers have been parsed, so let us write to the hardware.
    //We will wait for each element to be written before moving on.
    const bool waitFlag = TRUE;
    if ( fwCcpc_write(registerNames, ddcData, masks, callStatusList, waitFlag) ) {
      status = 0;
    } else {
      //Check where the errors occured.
      const unsigned statusLen = dynlen(callStatusList);
      if ( 0 != statusLen ) {
	for (unsigned element = 1; element <= statusLen; ++element) {
	  if ( 0 != callStatusList[element] ) {
	    //Construct the error message.
	    string errMsg = "While writing to \'" + registerNames[element] + "\' BeFpgaWrite encountered ";
	    if (callStatusList[element] > 0) {
	      errMsg += "a problem writing to the CCPC server.";
	      status = 1;
	    } else if (callStatusList[element] < 0) {
	      errMsg += "a problem with PVSS.";
	      status = -1;
	    } else {
	      errMsg += "an unknown error.";
	      status = -1;
	    }
	    //Now log the error.
	    _fwUkl1_LogFwCcpcError(errMsg);
	  }
	}
      } else {
	fwUkl1_ErrorMessage("BeFpgaWrite encountered a fatal error and an unknown number of writes were successful. L1 is in a highly inconsistent state.",
			    FWUKL1_DEBUG_MESSAGE_LEVEL);
      }
    }

    if (verifyWrite) {
      //Call the verification function.
      status = _fwUkl1_VerifyWrite(registerNames, ddcData, masks, "fwUkl1_BeFpgaWrite");
    }
  } else {
    fwUkl1_ErrorMessage("Failed to parse the data given to write, did not access hardware.", FWUKL1_DEBUG_MESSAGE_LEVEL);
    status = -2;
  }

  //Return the value of the last status failure. It is likely that they all failed for the same reason.
  return status;
}

/*!
 * This reads from a given set of registers on the BE FPGAs.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           read from, which corresponds to the appropriate hardware register.
 * \param  data dyn_string returned by reference that contains a string representing the value in the register.
 *           It will be zero padded to the nearest 16-bit word, except in the following cases:
 *   \li MacDestinationAddress A 12 character string that gives the hex number for the MAC address, can either be a continuous
 *         string of the address or may posses a colon delimiter every two characters,
 *         e.g. 000e0cb02d19 or 00:0e:0c:b0:2d:19.
 *   \li MacSourceAddress A 12 character string that gives the hex number for the MAC address, can either be a continuous
 *         string of the address or may posses a colon delimiter every two characters,
 *         e.g. 000e0cb02d19 or 00:0e:0c:b0:2d:19.
 *   \li IpDestinationAddress A string that contains either the address in hex format e.g. c0a0c0c0 or in a typical IP4 format
 *         e.g. 192.168.192.192.
 *   \li IpSourceAddress A string that contains either the address in hex format e.g. c0a0c0c0 or in a typical IP4 format
 *         e.g. 192.168.192.192.
 *   \li Protocol Default format described below.
 *   \li TypeOfService Default format described below.
 *   \li TimeToLive Default format described below.
 *   \li Type Default format described below.
 *   \li Version Default format described below.
 *   \li PartitionId Must contain an 8 character string representing the two 16-bit words of data for the partition ID.
 *   \li L1Id An 8-bit value, 2-character string. This will be present in the output data packet and MUST be unique for each L1.
 *         This is set correctly by the FSM and generally should not be User defined, except maybe when the User is an expert.
 *   \li ZeroSuppression String should be either `Enable' or `Disable' to enable or disable zero suppression across
 *         all the Ingress/FE FPGAs. Case sensitive.
 *   \li HpdPixelMode String should be either `ALICE' or `LHCb', which should correspond to the mode of operation
 *         required from the pixel chips. Case sensitive.
 *   \li Latency Default format described below.
 *   \li MepPackingFactor Default format described below.
 *   \li FixedMepDestinationAddress `Enable' or `Disable' to enable or disable a fixed multi-event packet destination address.
 *   \li DontWaitForMepDestBrdcst `Don't wait' or `Wait' to enable or disable the waiting for a multi-event packet destination
 *         address broadcast, respectively.
 *   \li Throttle `Enable' or `Disable' to enable or disable the throttle.
 *   \li ThrottlePolarity `Invert' or `Normal' to select the polarity of the throttle output.
 *   \li TfcDecoder `Enable' or `Disable' to enable or disable the decoding of TFC commands. The board will not operate without
 *         this setting enabled.
 *   \li PollGbePorts `Enable' or `Disable' to enable the Egress FPGA to send data to the GBE card for transmission.
 *   \li DataFragmentSize A decimal number specifying the size of the data packet that is transmitted via the GBE card.
 *         It is the size of the fragment of the multi-event packet that is transmitted. Max 9000.
 * \param  callStatusList This array of ints stores the return code from each write call to the fwCcpc library. The return code from this
 *           function will indicate if any of the writes failed and by checking the array the specific write failures can be
 *           identified. The order of return codes will match the ordering of the list of register names. Returned by reference.
 *           The following are the possible return codes (see fwCcpc lib):
 *   \li  0 Execution proceeded without a problem.
 *   \li >0 Error found and related with CCPC Server.
 *   \li -1 Error found and related with PVSS.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *            The error log contains information about which register failed to write.
 *   \li -1 Execution failed and due to a PVSS problem.
 *            The error log contains information about which register failed to write.
 *   \li -2 Exectution failed due to a read value having an unprocessable structure.
 */
int fwUkl1_BeFpgaRead(string ukl1Name, dyn_string registerNames, dyn_string& data, dyn_int& callStatusList) {
  //Make sure we have an empty data array.
  dynClear(data);
  //Define the string to be added to each of the register names such that they are written to the appropriate datapoint element.
  const string prefix = FWUKL1_HW_DPNAME_PREFIX + ukl1Name + ".BeFpga.";
  //Useful to know how many elements we may need to loop over.
  const unsigned numRegs = dynlen(registerNames);
  //Add prefix to each name and convert the data into a byte array.
  for (unsigned element = 1; element <= numRegs; ++element) {
    //Add the prefix.
    registerNames[element] = prefix + registerNames[element];
  }

  //Do the reads...
  //Holds the data in the form that the fwCcpc_read function returns.
  dyn_dyn_char ddcData;
  //The read status will need to be checked to ensure that the function succeeded and there is a valid call list.
  const bool readStatus = fwCcpc_read(registerNames, ddcData, callStatusList);
  //Size of the callStatusList. Important for checking it actually exists.
  const unsigned callStatusListSize = dynlen(callStatusList);

  //The overall status of the read, when combining callStatusList values.
  int status = 0;
  //Check that the callStatusList size is none zero otherwise there is no point in checking the element states.
  if ( 0 != callStatusListSize ) {
    //Put the data into a single string for return.
    for (unsigned element = 1; element <= numRegs; ++element) {
      //Convert the data from a series of 32-bit words to a series of 16-bit words.
      ddcData[element] = _fwUkl1_ConvertFrom32To16BitWords(ddcData[element]);

      //See how to parse the data.
      //First check for errors.
      if ( 0 != callStatusList[element] ) {
	//Construct the error message.
	string errMsg = "While reading from \'" + registerNames[element] + "\' BeFpgaRead encountered ";
	if (callStatusList[element] > 0) {
	  errMsg += "a problem writing to the CCPC server.";
	  status = 1;
	} else if (callStatusList[element] < 0) {
	  errMsg += "a problem with PVSS.";
	  status = -1;
	} else {
	  errMsg += "an unknown error.";
	  status = -1;
	}
	//Now log the error.
	_fwUkl1_LogFwCcpcError(errMsg);
	//Mark the error.
	data[element] = "Fail";
      }
      //Parse the MAC addresses.
      else if ( (prefix + "MacDestinationAddress" == registerNames[element]) || (prefix + "MacSourceAddress" == registerNames[element]) ) {
	//Now convert the remaining 48-bits into a MAC address.
	data[element] = fwUkl1_ByteToMacAddress(ddcData[element]);
	if ( "failed" == data[element] ) {
	  fwUkl1_ErrorMessage("Failed parse the " + registerNames[element] + ", return value not recognised: ",
			      FWUKL1_INFO_MESSAGE_LEVEL);
	  for (unsigned i = 1; i <= dynlen(ddcData[element]); ++i) {
	    fwUkl1_ErrorMessage(" " + (unsigned)ddcData[element][i],
				FWUKL1_INFO_MESSAGE_LEVEL);
	  }
	  status = -2;
	}
      }
      //Parse the IP addresses.
      else if ( (prefix + "IpDestinationAddress" == registerNames[element]) || (prefix + "IpSourceAddress" == registerNames[element]) ) {
	//This is potentially in decimal with full stop delimiters.
	data[element] = fwUkl1_ByteToIpAddress(ddcData[element]);
	if ( "failed" == data[element] ) {
	  fwUkl1_ErrorMessage("Failed parse the " + registerNames[element] + ", check argument structure.",
			      FWUKL1_INFO_MESSAGE_LEVEL);
	  status = -2;
	}
      }
      //Parse the zero suppression data.
      else if ( (prefix + "ZeroSuppression") == registerNames[element] ) {
	//Convert the requested setting into a padded hex string.
	if ( 0x0f == (ddcData[element][2] & 0x0f) ) {
	  data[element] = "Disable";
	} else if ( 0x00 == ddcData[element][2] ) {
	  data[element] = "Enable";
	} else {
	  fwUkl1_ErrorMessage(registerNames[element] + " has unrecognised value: " + fwCcpc_convertByteToHex(ddcData[element]) + ". Perhaps the setting is not consistent across all FE FPGAs, this will result in undefined behaviour.",
			      FWUKL1_INFO_MESSAGE_LEVEL);
	  status = -2;
	}
      }
      //Parse LHCb mode settings.
      else if ( (prefix + "HpdPixelMode") == registerNames[element] ) {
	//Convert the requested setting into a padded hex string.
	if ( 0xf0 == (ddcData[element][2] & 0xf0) ) {
	  data[element] = "ALICE";
	} else if ( 0x00 == ddcData[element][2] ) {
	  data[element] = "LHCb";
	} else {
	  fwUkl1_ErrorMessage(registerNames[element] + " has unrecognised value: " + fwCcpc_convertByteToHex(ddcData[element]) + ". Perhaps the setting is not consistent across all FE FPGAs, this will result in undefined behaviour.",
			      FWUKL1_INFO_MESSAGE_LEVEL);
	  status = -2;
	}
      }
      //Parse enable forced fixed MEP destination address settings.
      else if ( (prefix + "FixedMepDestinationAddress") == registerNames[element] ) {
	//Is it enabled or disabled.
	if ( 0x01 == (ddcData[element][2] & 0x01) ) {
	  data[element] = "Enable";
	} else if ( 0x00 == (ddcData[element][2] & 0x01) ) {
	  data[element] = "Disable";
	} else {
	  fwUkl1_ErrorMessage(registerNames[element] + " has unrecognised value: " + fwCcpc_convertByteToHex(ddcData[element]) + ".",
			      FWUKL1_INFO_MESSAGE_LEVEL);
	  status = -2;
	}
      }
      //Parse don't wait for MEP destination broadcast settings.
      else if ( (prefix + "DontWaitForMepDestBrdcst") == registerNames[element] ) {
	//Is it waiting for not.
	if ( 0x02 == (ddcData[element][2] & 0x02) ) {
	  data[element] = "Don't wait";
	} else if ( 0x00 == (ddcData[element][2] & 0x02) ) {
	  data[element] = "Wait";
	} else {
	  fwUkl1_ErrorMessage(registerNames[element] + " has unrecognised value: " + fwCcpc_convertByteToHex(ddcData[element]) + ".",
			      FWUKL1_INFO_MESSAGE_LEVEL);
	  status = -2;
	}
      }
      //Check to see if the throttle is enabled.
      else if ( (prefix + "Throttle") == registerNames[element] ) {
	//Is it enabled or disabled.
	if ( 0x0c == (ddcData[element][2] & 0x0c) ) {
	  data[element] = "Enable";
	} else if ( 0x00 == (ddcData[element][2] & 0x0c) ) {
	  data[element] = "Disable";
	} else {
	  fwUkl1_ErrorMessage(registerNames[element] + " has unrecognised value: " + fwCcpc_convertByteToHex(ddcData[element]) + ".",
			      FWUKL1_INFO_MESSAGE_LEVEL);
	  status = -2;
	}
      }
      //Check polarity of throttle.
      else if ( (prefix + "ThrottlePolarity") == registerNames[element] ) {
	//Is it enabled or disabled.
	if ( 0x30 == (ddcData[element][2] & 0x30) ) {
	  data[element] = "Invert";
	} else if ( 0x00 == (ddcData[element][2] & 0x30) ) {
	  data[element] = "Normal";
	} else {
	  fwUkl1_ErrorMessage(registerNames[element] + " has unrecognised value: " + fwCcpc_convertByteToHex(ddcData[element]) + ".",
			      FWUKL1_INFO_MESSAGE_LEVEL);
	  status = -2;
	}
      }
      //Check to see if the TFC is enabled.
      else if ( (prefix + "TfcDecoder") == registerNames[element] ) {
	//Is it enabled or disabled.
	if ( 0x40 == (ddcData[element][2] & 0x40) ) {
	  data[element] = "Enable";
	} else if ( 0x00 == (ddcData[element][2] & 0x40) ) {
	  data[element] = "Disable";
	} else {
	  fwUkl1_ErrorMessage(registerNames[element] + " has unrecognised value: " + fwCcpc_convertByteToHex(ddcData[element]) + ".",
			      FWUKL1_INFO_MESSAGE_LEVEL);
	  status = -2;
	}
      }
      //Check to see if the transmission of data to the GBE is enabled.
      else if ( (prefix + "PollGbePorts") == registerNames[element] ) {
	//Is it enabled or disabled.
	if ( 0x80 == (ddcData[element][2] & 0x80) ) {
	  data[element] = "Enable";
	} else if ( 0x00 == (ddcData[element][2] & 0x80) ) {
	  data[element] = "Disable";
	} else {
	  fwUkl1_ErrorMessage(registerNames[element] + " has unrecognised value: " + fwCcpc_convertByteToHex(ddcData[element]) + ".",
			      FWUKL1_INFO_MESSAGE_LEVEL);
	  status = -2;
	}
      }
      //Check the size of the data fragment size, it needs to be in decimal.
      else if ( (prefix + "DataFragmentSize") == registerNames[element] ) {
	data[element] = fwCcpc_convertByteToDec(ddcData[element]);
      
      }
      //This is only 8-bits big.
      else if ( (prefix + "L1Id") == registerNames[element] ) {
	//Remove elements until only 1 is left.
	const unsigned maxCount = dynlen(ddcData[element]);
	//Notice that the loop deliberately goes to less than element count. This is to ensure the whole loop is not deleted.
	for (unsigned count = 1; count < maxCount; ++count) {
	  //Remove the first element until there is only one left.
	  dynRemove(ddcData[element], 1);
	}
	data[element] = fwCcpc_convertByteToDec(ddcData[element]);
      
      }
      //Can just return the read data to a hex string as the extra words have already been removed.
      else {
	data[element] = fwCcpc_convertByteToHex(ddcData[element]);
      }
    }//for size of arrays.
  }//if ( 0 != callStatusListSize )
  else {
    //The function didn't execute properly as the callStatusList size is zero.
    fwUkl1_ErrorMessage("Failed to read from L1 board. The read did not return a populated array. It is unknown if any reads succeeded.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    status = -1;
  }

  //Return the value of the last status failure. It is likely that they all failed for the same reason.
  return status;
}

/*!
 * This writes to a given set of registers on the FE FPGAs.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  feFpga Number of the FE FPGA that is to be written. Range:0 to 3, FWUKL1_BROADCAST_FPGA.  FWUKL1_BROADCAST_FPGA is a 
 *           software creatation that allows all the FE FPGAs to be written to in a single command. It save on the number
 *           of hardware accesses by a factor of four.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           written to, which corresponds to the appropriate hardware register. The following values are permitted:
 *   \li Disable Accepts the text `Disable' or `Enable' to disable or enable a FE FPGA.
 *   \li Emulator Accepts the text `Disable' or `Enable' to disable or enable input data emulation on that FE FPGA, respectively.
 *         This setting disables the input data from the optical inputs on the 9 associated inputs channels and then sends emulated
 *         data through the L1 data processing chain.
 *   \li EmulatorHpdPixelMode Accepts `LHCb' or `ALICE'. The type of emulated data that will be sent. This will not automatically
 *         enable the event emulation.
 *   \li EmulatorEvent Accepts `Send' or `Don't send' to enable the sending of emulated events. This can only occur if the emulator is
 *         enabled via the `Emulator' setting. `Don't send' will prevent events being sent through the chain even if the emulator is
 *         enabled.
 *   \li ResetStatusBufferPointer Accepts `Assert' or `Deassert', which will either assert or deassert the reset for the buffer pointer
 *         for that FE FPGA. This must be done before reading the status information for a FE FPGA or it will produce undefined data.
 * \param  data By default a 16-bit value that is to be written to the register represented as a hex string, it need not be zero padded.
 *           Certain settings accept or require an alternative format as described for each register above. For a single FPGA write
 *           the data should be in the format above with one element for each register. If broadcasting then the data must be comma
 *           delimited and four values MUST be present e.g. To disable a single front end `Enable,Enable,Disable,Enable'.
 * \param  callStatusList This array of ints stores the return code from each write call to the fwCcpc library. The return code from this
 *           function will indicate if any of the writes failed and by checking the array the specific write failures can be
 *           identified. The order of return codes will match the ordering of the list of register names. Returned by reference.
 *           The following are the possible return codes (see fwCcpc lib):
 *   \li  0 Execution proceeded without a problem.
 *   \li >0 Error found and related with CCPC Server.
 *   \li -1 Error found and related with PVSS.
 * \param  verifyWrite If this is set to TRUE then the register will be read back from to ensure that
 *           the data written corresponds to that read. Default: FALSE.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *            The error log contains information about which register failed to write.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *            The error log contains information about which register failed to write.
 *   \li -2 Execution failed due to a problem with data or name format and no hardware/datapoint access was performed.
 */
int fwUkl1_FeFpgaWrite(string ukl1Name, unsigned feFpga, dyn_string registerNames, dyn_string data, dyn_int& callStatusList, bool verifyWrite=FALSE) {
  //Check that the register names and data arrays are the same size.
  //Save only the lenght of one of the arrays as the other must be the same size, in this case it is the registerNames.
  const unsigned numRegs = dynlen(registerNames);
  if ( numRegs != dynlen(data) ) {
    fwUkl1_ErrorMessage("registerNames size: " + numRegs + " is not equal to the data size: " + dynlen(data) + ", when writting to FE FPGA",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -2;
  }

  //Holds the return status of the function.
  int status = 0;

  //Holds the data in the form that the fwCcpc_write function expects.
  dyn_dyn_char ddcData;
  //Holds the mask for each element.
  dyn_dyn_char masks;

  //Converts the data given as strings in the calling arguments to byte arguments required by the fwCcpc_write.
  //Also it creates the masks required for each write and pads the register names, such that they refer to the appropriate
  //data point elements.
  status = _fwUkl1_ParseFeFpgaWriteRegisters(ukl1Name, feFpga, registerNames, data, ddcData, masks);

  //Check that we were able to validate and parse the data before attempting to write.
  if ( 0 == status ) {
    //The data and registers have been parsed, so let us write to the hardware.
    //We will wait for each element to be written before moving on.
    const bool waitFlag = TRUE;
    if ( fwCcpc_write(registerNames, ddcData, masks, callStatusList, waitFlag) ) {
      status = 0;
    } else {
      //Check where the errors occured.
      const unsigned statusLen = dynlen(callStatusList);
      if ( 0 != statusLen ) {
	for (unsigned element = 1; element <= statusLen; ++element) {
	  if ( 0 != callStatusList[element] ) {
	    //Construct the error message.
	    string errMsg = "While writing to \'" + registerNames[element] + "\' FeFpgaWrite encountered ";
	    if (callStatusList[element] > 0) {
	      errMsg += "a problem writing to the CCPC server.";
	      status = 1;
	    } else if (callStatusList[element] < 0) {
	      errMsg += "a problem with PVSS.";
	      status = -1;
	    } else {
	      errMsg += "an unknown error.";
	      status = -1;
	    }
	    //Now log the error.
	    _fwUkl1_LogFwCcpcError(errMsg);
	  }
	}
      } else {
	fwUkl1_ErrorMessage("FeFpgaWrite encountered a fatal error and an unknown number of writes were successful. L1 is in a highly inconsistent state.",
			    FWUKL1_DEBUG_MESSAGE_LEVEL);
      }
    }

    if (verifyWrite) {
      //Call the verification function.
      status = _fwUkl1_VerifyWrite(registerNames, ddcData, masks, "fwUkl1_FeFpgaWrite");
    }
  } else {
    fwUkl1_ErrorMessage("Failed to write data as data given to write was not valid.", FWUKL1_DEBUG_MESSAGE_LEVEL);
    status = -2;
  }

  //Return the value of the last status failure. It is likely that they all failed for the same reason.
  return status;
}

/*!
 * This reads from a given set of registers on the FE FPGAs. It is capable of reading from either an individual FE FPGA
 * or all FE FPGAs in broadcast mode.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  feFpga FE FPGA number that is to be written to. Range: 0 to 3, FWUKL1_BROADCAST_FPGA.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           read from, which corresponds to the appropriate hardware register.
 * \param  data dyn_string returned by reference that contains a string representing the value in the register.
 *           It will be zero padded to the nearest 16-bit word, except in the following cases:
 *   \li Disable `Enable' or `Disable' depending on the state of the FPGA.
 *   \li Emulator `Disable' or `Enable' depending on whether the L0 emulation is active or not.
 *   \li EmulatorHpdPixelMode `LHCb' or `ALICE' depending on which HPD pixel mode is being simulated in the events.
 *   \li EmulatorEvent `Send' or `Don't send' depending on whether or not events are being sent when in L0 emulation mode.
 *   \li ResetStatusBufferPointer `Assert' or `Deassert' depending on whether the reset is asserted or not.
 *           In broadcast mode the returned data elements will contain a comma delimited list of the settings for each
 *           individual FE FPGA.
 * \param  callStatusList This array of ints stores the return code from each write call to the fwCcpc library. The return code from this
 *           function will indicate if any of the writes failed and by checking the array the specific write failures can be
 *           identified. The order of return codes will match the ordering of the list of register names. Returned by reference.
 *           The following are the possible return codes (see fwCcpc lib):
 *   \li  0 Execution proceeded without a problem.
 *   \li >0 Error found and related with CCPC Server.
 *   \li -1 Error found and related with PVSS.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *            The error log contains information about which register failed to write.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *            The error log contains information about which register failed to write.
 *   \li -2 Exectution failed due to a read value having an unprocessable structure.
 */
int fwUkl1_FeFpgaRead(string ukl1Name, unsigned feFpga, dyn_string registerNames, dyn_string& data, dyn_int& callStatusList) {
  //Make sure we have an empty data array.
  dynClear(data);
  //Decide which FE FPGA we are going to read from. In the event of a broadcast because of the way the registers are set up in hardware
  //and implemented as datapoints we can just read from FE FPGA 0 and get all the FE FPGA settings provided we decode it properly.
  unsigned fpga = 0;
  //Only need to changed from initial value if not broadcasting.
  if ( FWUKL1_BROADCAST_FPGA != feFpga )
    fpga = feFpga;
  //Define the string to be added to each of the register names such that they are written to the appropriate datapoint element.
  const string prefix = FWUKL1_HW_DPNAME_PREFIX + ukl1Name + ".FeFpga" + fpga + ".";
  //Useful to know how many elements we may need to loop over.
  const unsigned numRegs = dynlen(registerNames);
  //Add prefix to each name and convert the data into a byte array.
  for (unsigned element = 1; element <= numRegs; ++element) {
    //Add the prefix.
    registerNames[element] = prefix + registerNames[element];
  }

  //Do the reads...
  //Holds the data in the form that the fwCcpc_read function returns.
  dyn_dyn_char ddcData;
  fwCcpc_read(registerNames, ddcData, callStatusList);

  //The overall status of the write.
  int status = 0;
  //Check that we do have some elements in the callStatusList.
  const unsigned callStatusListSize = dynlen(callStatusList);
  if ( 0 != callStatusListSize ) {
    //Put the data into a single string for return.
    for (unsigned element = 1; element <= numRegs; ++element) {
      //Convert the data from a series of 32-bit words to a series of 16-bit words.
      ddcData[element] = _fwUkl1_ConvertFrom32To16BitWords(ddcData[element]);

      if ( 0 != callStatusList[element] ) {
	//Construct the error message.
	string errMsg = "While reading from \'" + registerNames[element] + "\' FeFpgaRead encountered ";
	if (callStatusList[element] > 0) {
	  errMsg += "a problem writing to the CCPC server.";
	  status = 1;
	} else if (callStatusList[element] < 0) {
	  errMsg += "a problem with PVSS.";
	  status = -1;
	} else {
	  errMsg += "an unknown error.";
	  status = -1;
	}
	//Now log the error.
	_fwUkl1_LogFwCcpcError(errMsg);
	//Mark the error, either one or all FE FPGAs.
	if ( FWUKL1_BROADCAST_FPGA != feFpga )
	  data[element] = "Fail";
	else
	  data[element] = "Fail,Fail,Fail,Fail";
      }
      else if ( ((prefix + "Disable") == registerNames[element]) ) {
	//Is it enabled or disabled?
	//First convert it to a single number.
	unsigned fpgaStates = fwCcpc_convertByteToDec(ddcData[element]);
	//This is to mark whether it is the first time round the loop.
	bool firstTime = TRUE;
	//Now loop over the appropriate range of FPGAs.
	//First determine the starting number.
	const unsigned startFpga = FWUKL1_BROADCAST_FPGA==feFpga ? 0:feFpga;
	//Determine the end of the loop.
	const unsigned endFpga = FWUKL1_BROADCAST_FPGA==feFpga ? (FWUKL1_FRONT_FPGAS-1):feFpga;
	for (unsigned loopFpga = startFpga; loopFpga <= endFpga; ++loopFpga) {
	  //If it is not the first time round the loop we should add a comma before we write the next setting.
	  if ( !firstTime )
	    data[element] += ",";
	  if ( 0x01 == ((fpgaStates>>loopFpga) & 0x01) ) {
	    data[element] += "Disable";
	  } else if ( 0x00 == ((fpgaStates>>loopFpga) & 0x01) ) {
	    data[element] += "Enable";
	  } else {
	    fwUkl1_ErrorMessage(registerNames[element] + " has unrecognised value: " + ddcData[element] + ".",
				FWUKL1_INFO_MESSAGE_LEVEL);
	    status = -2;
	  }
	  //If it is the first time, it isn't any more.
	  if ( firstTime )
	    firstTime = FALSE;
	}
      }
      else if ( ((prefix + "Emulator") == registerNames[element]) ) {
	//Is it enabled or disabled?
	//First convert it to a single number.
	unsigned fpgaStates = fwCcpc_convertByteToDec(ddcData[element]);
	//This is to mark whether it is the first time round the loop.
	bool firstTime = TRUE;
	//Now loop over the appropriate range of FPGAs.
	//First determine the starting number.
	const unsigned startFpga = FWUKL1_BROADCAST_FPGA==feFpga ? 0:feFpga;
	//Determine the end of the loop.
	const unsigned endFpga = FWUKL1_BROADCAST_FPGA==feFpga ? (FWUKL1_FRONT_FPGAS-1):feFpga;
	for (unsigned loopFpga = startFpga; loopFpga <= endFpga; ++loopFpga) {
	  //If it is not the first time round the loop we should add a comma before we write the next setting.
	  if ( !firstTime )
	    data[element] += ",";
	  if ( 0x01 == (fpgaStates>>(4*loopFpga) & 0x01) ) {
	    data[element] += "Disable";
	  } else if ( 0x00 == (fpgaStates>>(4*loopFpga) & 0x01) ) {
	    data[element] += "Enable";
	  } else {
	    fwUkl1_ErrorMessage(registerNames[element] + " has unrecognised value: " + ddcData[element] + ".",
				FWUKL1_INFO_MESSAGE_LEVEL);
	    status = -2;
	  }
	  //If it is the first time, it isn't any more.
	  if ( firstTime )
	    firstTime = FALSE;
	}
      }
      else if ( (prefix + "EmulatorHpdPixelMode") == registerNames[element] ) {
	//Is it in ALICE or LHCb mode?
	//First convert it to a single number.
	unsigned fpgaStates = fwCcpc_convertByteToDec(ddcData[element]);
	//This is to mark whether it is the first time round the loop.
	bool firstTime = TRUE;
	//Now loop over the appropriate range of FPGAs.
	//First determine the starting number.
	const unsigned startFpga = FWUKL1_BROADCAST_FPGA==feFpga ? 0:feFpga;
	//Determine the end of the loop.
	const unsigned endFpga = FWUKL1_BROADCAST_FPGA==feFpga ? (FWUKL1_FRONT_FPGAS-1):feFpga;
	for (unsigned loopFpga = startFpga; loopFpga <= endFpga; ++loopFpga) {
	  //If it is not the first time round the loop we should add a comma before we write the next setting.
	  if ( !firstTime )
	    data[element] += ",";
	  if ( 0x02 == (fpgaStates>>(4*loopFpga) & 0x02) ) {
	    data[element] += "ALICE";
	  } else if ( 0x00 == (fpgaStates>>(4*loopFpga) & 0x02) ) {
	    data[element] += "LHCb";
	  } else {
	    fwUkl1_ErrorMessage(registerNames[element] + " has unrecognised value: " + ddcData[element] + ".",
				FWUKL1_INFO_MESSAGE_LEVEL);
	    status = -2;
	  }
	  //If it is the first time, it isn't any more.
	  if ( firstTime )
	    firstTime = FALSE;
	}
      }
      else if ( (prefix + "EmulatorEvent") == registerNames[element] ) {
	//Is it send events or not?
	//First convert it to a single number.
	unsigned fpgaStates = fwCcpc_convertByteToDec(ddcData[element]);
	//This is to mark whether it is the first time round the loop.
	bool firstTime = TRUE;
	//Now loop over the appropriate range of FPGAs.
	//First determine the starting number.
	const unsigned startFpga = FWUKL1_BROADCAST_FPGA==feFpga ? 0:feFpga;
	//Determine the end of the loop.
	const unsigned endFpga = FWUKL1_BROADCAST_FPGA==feFpga ? (FWUKL1_FRONT_FPGAS-1):feFpga;
	for (unsigned loopFpga = startFpga; loopFpga <= endFpga; ++loopFpga) {
	  //If it is not the first time round the loop we should add a comma before we write the next setting.
	  if ( !firstTime )
	    data[element] += ",";
	  if ( 0x04 == (fpgaStates>>(4*loopFpga) & 0x04) ) {
	    data[element] += "Send";
	  } else if ( 0x00 == (fpgaStates>>(4*loopFpga) & 0x04) ) {
	    data[element] += "Don't send";
	  } else {
	    fwUkl1_ErrorMessage(registerNames[element] + " has unrecognised value: " + ddcData[element] + ".",
				FWUKL1_INFO_MESSAGE_LEVEL);
	    status = -2;
	  }
	  //If it is the first time, it isn't any more.
	  if ( firstTime )
	    firstTime = FALSE;
	}
      }
      //Is the FE status buffer pointer reset asserted or deasserted.
      else if ( (prefix + "ResetStatusBufferPointer") == registerNames[element] ) {
	//First convert it to a single number.
	unsigned fpgaStates = fwCcpc_convertByteToDec(ddcData[element]);
	//This is to mark whether it is the first time round the loop.
	bool firstTime = TRUE;
	//Now loop over the appropriate range of FPGAs.
	//First determine the starting number.
	const unsigned startFpga = FWUKL1_BROADCAST_FPGA==feFpga ? 0:feFpga;
	//Determine the end of the loop.
	const unsigned endFpga = FWUKL1_BROADCAST_FPGA==feFpga ? (FWUKL1_FRONT_FPGAS-1):feFpga;
	for (unsigned loopFpga = startFpga; loopFpga <= endFpga; ++loopFpga) {
	  //If it is not the first time round the loop we should add a comma before we write the next setting.
	  if ( !firstTime )
	    data[element] += ",";
	  if ( 0x01 == ((fpgaStates>>loopFpga) & 0x01) ) {
	    data[element] += "Assert";
	  } else if ( 0x00 == ((fpgaStates>>loopFpga) & 0x01) ) {
	    sData[element] += "Deassert";
	  } else {
	    fwUkl1_ErrorMessage(registerNames[element] + " has unrecognised value: " + ddcData[element] + ".",
				FWUKL1_INFO_MESSAGE_LEVEL);
	    status = -2;
	  }
	  //If it is the first time, it isn't any more.
	  if ( firstTime )
	    firstTime = FALSE;
	}
      }
      //Can just return the read data to a hex string as the extra words have already been removed.
      else {
	//Assume the same data should be repeated four times if broadcasting.
	//This is to mark whether it is the first time round the loop.
	bool firstTime = TRUE;
	//Now loop over the appropriate range of FPGAs.
	//First determine the starting number.
	const unsigned startFpga = FWUKL1_BROADCAST_FPGA==feFpga ? 0:feFpga;
	//Determine the end of the loop.
	const unsigned endFpga = FWUKL1_BROADCAST_FPGA==feFpga ? (FWUKL1_FRONT_FPGAS-1):feFpga;
	for (unsigned loopFpga = startFpga; loopFpga <= endFpga; ++loopFpga) {
	  //If it is not the first time round the loop we should add a comma before we write the next setting.
	  if ( !firstTime )
	    data[element] += ",";
	  data[element] = fwCcpc_convertByteToHex(ddcData[element]);
	  //If it is the first time, it isn't any more.
	  if ( firstTime )
	    firstTime = FALSE;
	}
      }
    }//for element
  }//if ( 0 != callStatusListSize )
  else {
    //The function didn't execute properly as the callStatusList size is zero.
    fwUkl1_ErrorMessage("Failed to read from L1 board. The read did not return a populated array. It is unknown if any reads succeeded.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    status = -1;
  }

  //Return the value of the last status failure. It is likely that they all failed for the same reason.
  return status;
}

/*!
 * This writes to a given set of registers on a FE FPGA channel.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  channel Input channel number on a FE FPGA that is to be written to.
 * \param  feFpga FE FPGA number that is to be written to.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           written to, which corresponds to the appropriate hardware register. The following are accepted.
 *   \li Disable Accepts the text `Disable' or `Enable' to disable or enable a FE FPGA.
 * \param  data By default a 16-bit value that is to be written to the register represented as a hex string, it need not be zero padded.
 *           Certain settings accept or require an alternative format as described for each register above.
 * \param  callStatusList This array of ints stores the return code from each write call to the fwCcpc library. The return code from this
 *           function will indicate if any of the writes failed and by checking the array the specific write failures can be
 *           identified. The order of return codes will match the ordering of the list of register names. Returned by reference.
 *           The following are the possible return codes (see fwCcpc lib):
 *   \li  0 Execution proceeded without a problem.
 *   \li >0 Error found and related with CCPC Server.
 *   \li -1 Error found and related with PVSS.
 * \param  verifyWrite If this is set to TRUE then the register will be read back from to ensure that
 *           the data written corresponds to that read. Default: FALSE.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *            The error log contains information about which register failed to write.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *            The error log contains information about which register failed to write.
 *   \li -2 Execution failed due to a problem with data or name format and no hardware/datapoint access was performed.
 *
 * The disabled registers are all stored on the same UKL1 register. This is handled internally by this function call and
 * only ever the zeroth bit need be toggled for any of the FE FPGA disabled settings.
 */
int fwUkl1_FeFpgaChannelWrite(string ukl1Name, unsigned channel, unsigned feFpga, dyn_string registerNames, dyn_string data, dyn_int& callStatusList, bool verifyWrite=FALSE) {
  //Check that the register names and data arrays are the same size.
  //Save only the lenght of one of the arrays as the other must be the same size, in this case it is the registerNames.
  const unsigned numRegs = dynlen(registerNames);
  if ( numRegs != dynlen(data) ) {
    fwUkl1_ErrorMessage("registerNames size: " + numRegs + " is not equal to the data size: " + dynlen(data) + ", when writting to FE FPGA " + feFpga + " channel " + channel,
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -2;
  }

  //The overall status of the function.
  int status = 0;

  //Holds the data in the form that the fwCcpc_write function expects.
  dyn_dyn_char ddcData;
  //Holds the mask for each element.
  dyn_dyn_char masks;

  //Converts the data given as strings in the calling arguments to byte arguments required by the fwCcpc_write.
  //Also it creates the masks required for each write and pads the register names, such that they refer to the appropriate
  //data point elements.
  status = _fwUkl1_ParseFeFpgaChannelWriteRegisters(ukl1Name, channel, feFpga, registerNames, data, ddcData, masks);
  //Check that we were able to parse the data.
  if ( 0 == status ) {
    //The data and registers have been parsed, so let us write to the hardware.
    //We will wait for each element to be written before moving on.
    const bool waitFlag = TRUE;
    if ( fwCcpc_write(registerNames, ddcData, masks, callStatusList, waitFlag) ) {
      status = 0;
    } else {
      //Check where the errors occured.
      const unsigned statusLen = dynlen(callStatusList);
      if ( 0 != statusLen ) {
	for (unsigned element = 1; element <= statusLen; ++element) {
	  if ( 0 != callStatusList[element] ) {
	    //Construct the error message.
	    string errMsg = "While writing to \'" + registerNames[element] + "\' FeFpgaChannelWrite encountered ";
	    if (callStatusList[element] > 0) {
	      errMsg += "a problem writing to the CCPC server.";
	      status = 1;
	    } else if (callStatusList[element] < 0) {
	      errMsg += "a problem with PVSS.";
	      status = -1;
	    } else {
	      errMsg += "an unknown error.";
	      status = -1;
	    }
	    //Now log the error.
	    _fwUkl1_LogFwCcpcError(errMsg);
	  }
	}
      } else {
	fwUkl1_ErrorMessage("FeFpgaChannelWrite encountered a fatal error and an unknown number of writes were successful. L1 is in a highly inconsistent state.",
			    FWUKL1_DEBUG_MESSAGE_LEVEL);
      }
    }

    if (verifyWrite) {
      //Call the verification function.
      status = _fwUkl1_VerifyWrite(registerNames, ddcData, masks, "fwUkl1_FeFpgaWrite");
    }
  } else {
    fwUkl1_ErrorMessage("Failed to write data as data given to write was not valid.", FUKL1_DEBUG_MESSAGE_LEVEL);
    status = -2;
  }

  //Return the value of the last status failure. It is likely that they all failed for the same reason.
  return status;
}

/*!
 * This reads from a given set of registers on a FE FPGA channel.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  channel Input channel number on a FE FPGA that is to be written to.
 * \param  feFpga FE FPGA number that is to be written to.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           read from, which corresponds to the appropriate hardware register.
 * \param  data dyn_string returned by reference that contains a string representing the value in the register.
 *           It will be zero padded to the nearest 16-bit word, except in the following cases:
 *   \li Disable `Enable' or `Disable' depending on the state of the channel.
 * \param  callStatusList This array of ints stores the return code from each write call to the fwCcpc library. The return code from this
 *           function will indicate if any of the writes failed and by checking the array the specific write failures can be
 *           identified. The order of return codes will match the ordering of the list of register names. Returned by reference.
 *           The following are the possible return codes (see fwCcpc lib):
 *   \li  0 Execution proceeded without a problem.
 *   \li >0 Error found and related with CCPC Server.
 *   \li -1 Error found and related with PVSS.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *            The error log contains information about which register failed to write.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *            The error log contains information about which register failed to write.
 *   \li -2 Exectution failed due to a read value having an unprocessable structure.
 */
int fwUkl1_FeFpgaChannelRead(string ukl1Name, unsigned channel, unsigned feFpga, dyn_string registerNames, dyn_string& data, dyn_int& callStatusList) {
  //Make sure we have an empty data array.
  dynClear(data);
  //Define the string to be added to each of the register names such that they are written to the appropriate datapoint element.
  //Either we need to write to a specific FE FPGA channel or in the case of broadcasting we can write to FE FPGA channel 0 as all
  //FE FPGA channel settings are at the same register address, we merely need to write to the first channel with the appropriate data.
  //This is why broadcasting over the UKL1 channels is not any better, as you still have to write to the four registers anyway.
  const unsigned selectedChannel = FWUKL1_BROADCAST_CHANNEL==channel ? 0:channel;
  //Define the string to be added to each of the register names such that they are written to the appropriate datapoint element.
  const string prefix = FWUKL1_HW_DPNAME_PREFIX + ukl1Name + ".FeFpga" + feFpga + ".Channel" + selectedChannel + ".";
  //Useful to know how many elements we may need to loop over.
  const unsigned numRegs = dynlen(registerNames);
  //Add prefix to each name and convert the data into a byte array.
  for (unsigned element = 1; element <= numRegs; ++element) {
    //Add the prefix.
    registerNames[element] = prefix + registerNames[element];
  }

  //Do the reads...
  //Holds the data in the form that the fwCcpc_read function returns.
  dyn_dyn_char ddcData;
  //Records the return status of each of the writes.
  dyn_int callStatusList;
  fwCcpc_read(registerNames, ddcData, callStatusList);

  //The overall status of the write.
  int status = 0;

  //Check that we have some well formed arrays to read.
  if ( 0 != dynlen(callStatusList) ) {
    //Put the data into a single string for return.
    for (unsigned element = 1; element <= numRegs; ++element) {
      //Convert the data from a series of 32-bit words to a series of 16-bit words.
      ddcData[element] = _fwUkl1_ConvertFrom32To16BitWords(ddcData[element]);
  
      if ( 0 != callStatusList[element] ) {
	//Construct the error message.
	string errMsg = "While reading from \'" + registerNames[element] + "\' FeFpgaChannelRead encountered ";
	if (callStatusList[element] > 0) {
	  errMsg += "a problem writing to the CCPC server.";
	  status = 1;
	} else if (callStatusList[element] < 0) {
	  errMsg += "a problem with PVSS.";
	  status = -1;
	} else {
	  errMsg += "an unknown error.";
	  status = -1;
	}
	//Now log the error.
	_fwUkl1_LogFwCcpcError(errMsg);
	//Mark the error, either one or all channels on this FE FPGA.
	if ( FWUKL1_BROADCAST_CHANNEL != channel )
	  data[element] = "Fail";
	else
	  data[element] = "Fail,Fail,Fail,Fail,Fail,Fail,Fail,Fail,Fail";
      }
      else if ( ((prefix + "Disable") == registerNames[element]) ) {
	//Is it enabled or disabled?
	//First convert it to a single number.
	unsigned channelStates = fwCcpc_convertByteToDec(ddcData[element]);
	//This is to mark whether it is the first time round the loop.
	bool firstTime = TRUE;
	//Now loop over the appropriate range of channels.
	//Start at the selected channel.
	//Determine the end of the loop.
	const unsigned endChannel = FWUKL1_BROADCAST_CHANNEL==channel ? (FWUKL1_FPGA_CHANNELS-1):channel;
	for (unsigned loopChannel = selectedChannel; loopChannel <= endChannel; ++loopChannel) {
	  //If it is not the first time round the loop we should add a comma before we write the next setting.
	  if ( !firstTime )
	    data[element] += ",";
	  if ( 0x01 == ((channelStates>>loopChannel) & 0x01) ) {
	    data[element] += "Disable";
	  } else if ( 0x00 == ((channelStates>>loopChannel) & 0x01) ) {
	    data[element] += "Enable";
	  } else {
	    fwUkl1_ErrorMessage(registerNames[element] + " has unrecognised value: " + channelStates + ".",
				FWUKL1_INFO_MESSAGE_LEVEL);
	    status = -2;
	  }
	  //If it is the first time, it isn't any more.
	  if ( firstTime )
	    firstTime = FALSE;
	}
      }
      //Can just return the read data to a hex string as the extra words have already been removed.
      else {
	data[element] = fwCcpc_convertByteToHex(ddcData[element]);
      }

    }
  }//if ( 0 != dynlen(callStatusList) )
  else {
    //The function didn't execute properly as the callStatusList size is zero.
    fwUkl1_ErrorMessage("Failed to read from L1 board. The read did not return a populated array. It is unknown if any reads succeeded.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    status = -1;
  }


  //Return the value of the last status failure. It is likely that they all failed for the same reason.
  return status;
}

// =============================
//  INTERNAL FUNCTIONS
// =============================

/*!
 * This is an internal function that will convert the data and register names given to the fwUkl1_BeFpgaWrite function
 * and converts them into a form that can be understood by the hardware writes. It also creates the appropriate masks
 * for the data functions.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param registerName List of BE FPGA register names that are to be written to. Returned with the appropriate name
 *          prefix by reference.
 * \param sData Data to be written to the registers in the form of a string. The allowed values for these are described
 *          in the fwUkl1_BeFpgaWrite function documentation.
 * \param ddcData Data converted into a dyn_dyn_char for use with the fwCcpc_write function. Returned by reference.
 * \param masks Masks that are associated with the write data.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *            The error log contains information about which register failed to write.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *            The error log contains information about which register failed to write.
 *   \li -2 Execution failed due to a problem with data or name format and no hardware/datapoint access was performed.
 */
int _fwUkl1_ParseBeFpgaWriteRegisters(string ukl1Name, dyn_string& registerNames, dyn_string sData, dyn_dyn_char& ddcData, dyn_dyn_char& masks) {

  //Define the string to be added to each of the register names such that they are written to the appropriate datapoint element.
  const string prefix = FWUKL1_HW_DPNAME_PREFIX + ukl1Name + ".BeFpga.";

  //Holds the status of the function.
  int status = 0;

  //Add prefix to each name and convert the data into a byte array.
  for (unsigned element = 1; element <= dynlen(registerNames); ++element) {

    //Check for the MAC addresses being written.
    if ( ("MacDestinationAddress" == registerNames[element]) || ("MacSourceAddress" == registerNames[element]) ) {
      //Need a mask for a 96-bit word write, 3 32-bit words.
      masks[element] = makeDynChar(0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0xff, 0xff);
      //This is potentially has colons in it that may need to be removed.
      sData[element] = fwUkl1_ParseMacAddress(sData[element]);
      if ( "failed" == sData[element] ) {
	fwUkl1_ErrorMessage("Failed parse the " + registerNames[element] + ", check argument structure.",
			    FWUKL1_INFO_MESSAGE_LEVEL);
	status = -2;
      }
      //Need to make sure the three 16-bit words that make up the data are padded to 32-bit words.
      //This relies on a deep knowledge of the way the fwCcpc library and L1 hardware works.
      //It is explained in the main library documentation.
      sData[element] = "0000" + substr(sData[element], 8, 4) + "0000" + substr(sData[element], 4, 4) +
	"0000" + substr(sData[element], 0, 4);
    }
    //Check for the IP addresses.
    else if ( ("IpDestinationAddress" == registerNames[element]) || ("IpSourceAddress" == registerNames[element]) ) {
       //Need a 64-bit mask for this.
      masks[element] = makeDynChar(0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0xff, 0xff);
      //This is potentially in decimal with full stop delimiters.
      sData[element] = fwUkl1_ParseIpAddress(sData[element]);
      if ( "failed" == sData[element] ) {
	fwUkl1_ErrorMessage("Failed parse the " + registerNames[element] + ", check argument structure.",
			    FWUKL1_INFO_MESSAGE_LEVEL);
	status = -2;
      }
      //Need to make sure the two 16-bit words that make up the data are padded to 32-bit words.
      //This relies on a deep knowledge of the way the fwCcpc library and L1 hardware works.
      //It is explained in the main library documentation.
      sData[element] = "0000" + substr(sData[element], 4, 4) + "0000" + substr(sData[element], 0, 4);
    }
    //Check for the partition ID.
    else if ( "PartitionId" == registerNames[element] ) {
      //Must be 8 characters big to be a valid ID, which is 4 bytes.
      sData[element] = _fwUkl1_PadString(sData[element], 4);
      //This needs to be written to two 32-bit registers with the two 16-bits words of the given partition ID
      //in each. Need to pad the beginning with four 0 characters and between the 1st and 2nd words.
      sData[element] = "0000" + substr(sData[element], 4, 4) + "0000" + substr(sData[element], 0, 4);
      //Here we need a 64-bit mask.
      masks[element] = makeDynChar(0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0xff, 0xff);
    }
    //Check for the zero suppression mode.
    else if ( "ZeroSuppression" == registerNames[element] ) {
      //For this we need to write to only the first four bits.
      masks[element] = makeDynChar(0x00, 0x00, 0x00, 0x0f);
      //Convert the requested setting into a padded hex string.
      if ( "Disable" == sData[element] ) {
	sData[element] = "0000000f";
      } else if ( "Enable" == sData[element] ) {
	sData[element] = "00000000";
      } else {
	fwUkl1_ErrorMessage(registerNames[element] + " has misformed argument: " + sData[element] + ". Should be `disabled' or `enabled', case sensitive.",
			    FWUKL1_INFO_MESSAGE_LEVEL);
	status = -2;
      }
    }
    //Check for a change of pixel mode.
    else if ( "HpdPixelMode" == registerNames[element] ) {
      //For this we need to write to only bits 4 to 7.
      masks[element] = makeDynChar(0x00, 0x00, 0x00, 0xf0);
      //Convert the requested setting into a padded hex string.
      if ( "ALICE" == sData[element] ) {
	sData[element] = "000000f0";
      } else if ( "LHCb" == sData[element] ) {
	sData[element] = "00000000";
      } else {
	fwUkl1_ErrorMessage(registerNames[element] + " has misformed argument: " + sData[element] + ". Should be `ALICE' or `LHCb', case sensitive.",
			    FWUKL1_INFO_MESSAGE_LEVEL);
	status = -2;
      }
    }
    //Check for fixing MEP destination address.
    else if ( "FixedMepDestinationAddress" == registerNames[element] ) {
      //Write to bit 1.
      masks[element] = makeDynChar(0x00, 0x00, 0x00, 0x01);
      if ( "Disable" == sData[element] ) {
	sData[element] = "00000000";
      } else if ( "Enable" == sData[element] ) {
	sData[element] = "00000001";
      } else {
	fwUkl1_ErrorMessage(registerNames[element] + " has misformed argument: " + sData[element] + ". Should be `Disable' or `Enable', case sensitive.",
			    FWUKL1_INFO_MESSAGE_LEVEL);
	status = -2;
      }
    }
    //Check for MEP destination broadcast wait setting.
    else if ( "DontWaitForMepDestBrdcst" == registerNames[element] ) {
      //Write to bit 2.
      masks[element] = makeDynChar(0x00, 0x00, 0x00, 0x02);
      if ( "Don't wait" == sData[element] ) {
	sData[element] = "00000002";
      } else if ( "Wait" == sData[element] ) {
	sData[element] = "00000000";
      } else {
	fwUkl1_ErrorMessage(registerNames[element] + " has misformed argument: " + sData[element] + ". Should be `Don't wait' or `Wait', case sensitive.",
			    FWUKL1_INFO_MESSAGE_LEVEL);
	status = -2;
      }
    }
    //Check for throttle enabling.
    else if ( "Throttle" == registerNames[element] ) {
      //Write to bits 3 and 4.
      masks[element] = makeDynChar(0x00, 0x00, 0x00, 0x0c);
      if ( "Disable" == sData[element] ) {
	sData[element] = "00000000";
      } else if ( "Enable" == sData[element] ) {
	sData[element] = "0000000c";
      } else {
	fwUkl1_ErrorMessage(registerNames[element] + " has misformed argument: " + sData[element] + ". Should be `Disable' or `Enable', case sensitive.",
			    FWUKL1_INFO_MESSAGE_LEVEL);
	status = -2;
      }
    }
    //Check for throttle invertion.
    else if ( "ThrottlePolarity" == registerNames[element] ) {
      //Write to bits 5 and 6.
      masks[element] = makeDynChar(0x00, 0x00, 0x00, 0x30);
      if ( "Invert" == sData[element] ) {
	sData[element] = "00000030";
      } else if ( "Normal" == sData[element] ) {
	sData[element] = "00000000";
      } else {
	fwUkl1_ErrorMessage(registerNames[element] + " has misformed argument: " + sData[element] + ". Should be `Invert' or `Normal', case sensitive.",
			    FWUKL1_INFO_MESSAGE_LEVEL);
	status = -2;
      }
    }
    //Check if TFC decoder is enabled.
    else if ( "TfcDecoder" == registerNames[element] ) {
      //Write to bit 6.
      masks[element] = makeDynChar(0x00, 0x00, 0x00, 0x40);
      if ( "Disable" == sData[element] ) {
	sData[element] = "00000000";
      } else if ( "Enable" == sData[element] ) {
	sData[element] = "00000040";
      } else {
	fwUkl1_ErrorMessage(registerNames[element] + " has misformed argument: " + sData[element] + ". Should be `Disable' or `Enable', case sensitive.",
			    FWUKL1_INFO_MESSAGE_LEVEL);
	status = -2;
      }
    }
    //Check if transmission to the GBE output ports is enabled.
    else if ( "PollGbePorts" == registerNames[element] ) {
      //Write to bit 7.
      masks[element] = makeDynChar(0x00, 0x00, 0x00, 0x80);
      if ( "Disable" == sData[element] ) {
	sData[element] = "00000000";
      } else if ( "Enable" == sData[element] ) {
	sData[element] = "00000080";
      } else {
	fwUkl1_ErrorMessage(registerNames[element] + " has misformed argument: " + sData[element] + ". Should be `Disable' or `Enable', case sensitive.",
			    FWUKL1_INFO_MESSAGE_LEVEL);
	status = -2;
      }
    }
    //Data fragment size is in decimal.
    else if ( "DataFragmentSize" == registerNames[element] ) {
      //Convert from decimal to hex.
      sData[element] = fwCcpc_convertDecToHex(sData[element]);
      //Pad the string.
      sData[element] = _fwUkl1_PadString(sData[element], 4);
      //Create a 16-bit mask.
      masks[element]   = makeDynChar(0x00, 0x00, 0xff, 0xff);
    }
    //L1 ID is only 2-characters big.
    else if ( "L1Id" == registerNames[element] ) {
      //Create an 8-bit mask.
      masks[element]   = makeDynChar(0x00, 0x00, 0x00, 0xff);
      //Pad the string.
      sData[element] = _fwUkl1_PadString(sData[element], 4);
    }
    //All the others just require a 16-bit mask and padding to 32-bit word.
    else {
      //Need a 16-bit mask.
      masks[element]   = makeDynChar(0x00, 0x00, 0xff, 0xff);
      //Pad the string, it still must be 32-bits long, the mask ensures we ignore the upper 16-bits.
      sData[element] = _fwUkl1_PadString(sData[element], 4);
    }
    //Put the data and register names in a format that the write can use.
    //Add the prefix to convert it to a datapoint name.
    registerNames[element] = prefix + registerNames[element];

    //Convert from a hex string to a byte array.
    ddcData[element] = fwCcpc_convertHexToByte( sData[element] );
  }


  return status;

}

/*!
 * This is an internal function that will convert the data and register names given to the fwUkl1_FeFpgaWrite function
 * and converts them into a form that can be understood by the hardware writes. It also creates the appropriate masks
 * for the data functions. It can be used for either writing to a single FE FPGA register or to broadcast to all FE FPGA
 * registers.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  feFpga Number of the FE FPGA that is to be written. Range:0 to 3, FWUKL1_BROADCAST_FPGA.
 * \param  registerName List of FE FPGA register names that are to be written to. Returned with the appropriate name
 *           prefix by reference.
 * \param  sData Data to be written to the registers in the form of a string. The allowed values for these are described
 *           in the fwUkl1_FeFpgaWrite function documentation. These should be comma separated values in the broadcast case.
 * \param  ddcData Data converted into a dyn_dyn_char for use with the fwCcpc_write function. Returned by reference.
 * \param  masks Masks that are associated with the write data.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *            The error log contains information about which register failed to write.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *            The error log contains information about which register failed to write.
 *   \li -2 Execution failed due to a problem with data or name format and no hardware/datapoint access was performed.
 */
int _fwUkl1_ParseFeFpgaWriteRegisters(string ukl1Name, unsigned feFpga, dyn_string& registerNames, dyn_string sData, dyn_dyn_char& ddcData, dyn_dyn_char& masks) {
  //Define the string to be added to each of the register names such that they are written to the appropriate datapoint element.
  //Either we need to write to a specific FE FPGA or in the case of broadcasting we can  write to FE FPGA 0 as all FE FPGA settings
  //exist in the same BE FPGA register and just set different bits. We can always write to the FE FPGA 0 DP and with the
  //appropriate mask and data structure it can write them all at the same time.
  const unsigned selectedFpga = FWUKL1_BROADCAST_FPGA==feFpga ? 0:feFpga;
  const string prefix = FWUKL1_HW_DPNAME_PREFIX + ukl1Name + ".FeFpga" + selectedFpga + ".";

  //First we need to parse the broadcast data to break it up into its component FPGA settings.
  dyn_dyn_string sParsedData = fwUkl1_ParseBroadcastData(sData);

  //Holds the status of the function.
  int status = 0;

  //Add prefix to each name and convert the data into a byte array.
  for (unsigned element = 1; element <= dynlen(registerNames); ++element) {
    //Disabling the FPGA.
    if ( "Disable" == registerNames[element] ) {
      //These are useful for holding the mask and data to be written as it is determined.
      unsigned mask = 0;
      unsigned disable = 0;
      //Now loop over the appropriate range of FPGAs.
      //Loop should start at the FE FPGA we selected in name prefix.
      //Determine the end of the loop.
      const unsigned endFpga = FWUKL1_BROADCAST_FPGA==feFpga ? (FWUKL1_FRONT_FPGAS-1):feFpga;
      for (unsigned loopFpga = selectedFpga; loopFpga <= endFpga; ++loopFpga) {
	//Create the mask. Store it as a single number and convert it to a hex string at the end.
	mask |= 1<<loopFpga;
	//Create the data to write. In the event that we are only writing to 1 FPGA then we must always take
	//the data from the 1st element, as this is the only element that will contain any data.
	if ( "Disable" == sParsedData[element][loopFpga+1-selectedFpga] ) {
	  //Set the relevant bit high.
	  disable |= 1<<loopFpga;
	} else if ( "Enable" == sParsedData[element][loopFpga+1-selectedFpga] ) {
	  //Do nothing the bit is already zero.
	} else {
	  fwUkl1_ErrorMessage(registerNames[element] + " has misformed argument: " + sParsedData[element][loopFpga+1-selectedFpga] + ". Should be `Disable' or `Enable', case sensitive.",
			      FWUKL1_INFO_MESSAGE_LEVEL);
	  status = -2;
	}
      }
      //Convert the mask data to a string.
      masks[element] = fwCcpc_convertHexToByte(_fwUkl1_PadString(fwCcpc_convertDecToHex(mask),4));
      //Convert the register data to a string.
      sData[element] = _fwUkl1_PadString( fwCcpc_convertDecToHex(disable), 4 );
    }
    //Enable/disable the L0 emulator setting.
    else if ( "Emulator" == registerNames[element] ) {
      //These are useful for holding the mask and data to be written as it is determined.
      unsigned mask = 0;
      unsigned disable = 0;
      //Now loop over the appropriate range of FPGAs.
      //Loop should start at the FE FPGA we selected in name prefix.
      //Determine the end of the loop.
      const unsigned endFpga = FWUKL1_BROADCAST_FPGA==feFpga ? (FWUKL1_FRONT_FPGAS-1):feFpga;
      for (unsigned loopFpga = selectedFpga; loopFpga <= endFpga; ++loopFpga) {
	//Create the mask. Store it as a single number and convert it to a hex string at the end.
	mask |= 1<<(4*loopFpga);
	//Create the data to write. In the event that we are only writing to 1 FPGA then we must always take
	//the data from the 1st element, as this is the only element that will contain any data.
	if ( "Disable" == sParsedData[element][loopFpga+1-selectedFpga] ) {
	  //Set the relevant bit high.
	  disable |= 1<<(4*loopFpga);
	} else if ( "Enable" == sParsedData[element][loopFpga+1-selectedFpga] ) {
	  //Do nothing the bit is already zero.
	} else {
	  fwUkl1_ErrorMessage(registerNames[element] + " has misformed argument: " + sParsedData[element][loopFpga+1-selectedFpga] + ". Should be `Disable' or `Enable', case sensitive.",
			      FWUKL1_INFO_MESSAGE_LEVEL);
	  status = -2;
	}
      }
      //Convert the mask data to a string.
      masks[element] = fwCcpc_convertHexToByte(_fwUkl1_PadString(fwCcpc_convertDecToHex(mask),4));
      //Convert the register data to a string.
      sData[element] = _fwUkl1_PadString( fwCcpc_convertDecToHex(disable), 4 );
    }
    //Set the pixel mode for the emulated event.
    else if ( "EmulatorHpdPixelMode" == registerNames[element] ) {
      //These are useful for holding the mask and data to be written as it is determined.
      unsigned mask = 0;
      unsigned mode = 0;
      //Now loop over the appropriate range of FPGAs.
      //Loop should start at the FE FPGA we selected in name prefix.
      //Determine the end of the loop.
      const unsigned endFpga = FWUKL1_BROADCAST_FPGA==feFpga ? (FWUKL1_FRONT_FPGAS-1):feFpga;
      for (unsigned loopFpga = selectedFpga; loopFpga <= endFpga; ++loopFpga) {
	//Create the mask. Store it as a single number and convert it to a hex string at the end.
	mask |= 2<<(4*loopFpga);
	//Create the data to write. In the event that we are only writing to 1 FPGA then we must always take
	//the data from the 1st element, as this is the only element that will contain any data.
	if ( "ALICE" == sParsedData[element][loopFpga+1-selectedFpga] ) {
	  //Set the relevant bit high.
	  mode |= 2<<(4*loopFpga);
	} else if ( "LHCb" == sParsedData[element][loopFpga+1-selectedFpga] ) {
	  //Do nothing the bit is already zero.
	} else {
	  fwUkl1_ErrorMessage(registerNames[element] + " has misformed argument: " + sParsedData[element][loopFpga+1-selectedFpga] + ". Should be `ALICE' or `LHCb', case sensitive.",
			      FWUKL1_INFO_MESSAGE_LEVEL);
	  status = -2;
	}
      }
      //Convert the mask data to a string.
      masks[element] = fwCcpc_convertHexToByte(_fwUkl1_PadString(fwCcpc_convertDecToHex(mask),4));
      //Convert the register data to a string.
      sData[element] = _fwUkl1_PadString( fwCcpc_convertDecToHex(mode), 4 );
    }
    //Enable/disable the L0 emulator setting.
    else if ( "EmulatorEvent" == registerNames[element] ) {
      //These are useful for holding the mask and data to be written as it is determined.
      unsigned mask = 0;
      unsigned send = 0;
      //Now loop over the appropriate range of FPGAs.
      //Loop should start at the FE FPGA we selected in name prefix.
      //Determine the end of the loop.
      const unsigned endFpga = FWUKL1_BROADCAST_FPGA==feFpga ? (FWUKL1_FRONT_FPGAS-1):feFpga;
      for (unsigned loopFpga = selectedFpga; loopFpga <= endFpga; ++loopFpga) {
	//Create the mask. Store it as a single number and convert it to a hex string at the end.
	mask |= 4<<(4*loopFpga);
	//Create the data to write. In the event that we are only writing to 1 FPGA then we must always take
	//the data from the 1st element, as this is the only element that will contain any data.
	if ( "Send" == sParsedData[element][loopFpga+1-selectedFpga] ) {
	  //Set the relevant bit high.
	  send |= 4<<(4*loopFpga);
	} else if ( "Don't send" == sParsedData[element][loopFpga+1-selectedFpga] ) {
	  //Do nothing the bit is already zero.
	} else {
	  fwUkl1_ErrorMessage(registerNames[element] + " has misformed argument: " + sParsedData[element][loopFpga+1-selectedFpga] + ". Should be `Send' or `Don't send', case sensitive.",
			      FWUKL1_INFO_MESSAGE_LEVEL);
	  status = -2;
	}
      }
      //Convert the mask data to a string.
      masks[element] = fwCcpc_convertHexToByte(_fwUkl1_PadString(fwCcpc_convertDecToHex(mask),4));
      //Convert the register data to a string.
      sData[element] = _fwUkl1_PadString( fwCcpc_convertDecToHex(send), 4 );
    }
    //Asserts or deasserts the reset for the FE status buffer pointer.
    else if ( "ResetStatusBufferPointer" == registerNames[element] ) {
      //These are useful for holding the mask and data to be written as it is determined.
      unsigned mask = 0;
      unsigned assert = 0;
      //Now loop over the appropriate range of FPGAs.
      //Loop should start at the FE FPGA we selected in name prefix.
      //Determine the end of the loop.
      const unsigned endFpga = FWUKL1_BROADCAST_FPGA==feFpga ? (FWUKL1_FRONT_FPGAS-1):feFpga;
      for (unsigned loopFpga = selectedFpga; loopFpga <= endFpga; ++loopFpga) {
	//Create the mask. Store it as a single number and convert it to a hex string at the end.
	mask |= 1<<loopFpga;
	//Create the data to write. In the event that we are only writing to 1 FPGA then we must always take
	//the data from the 1st element, as this is the only element that will contain any data.
	if ( "Assert" == sParsedData[element][loopFpga+1-selectedFpga] ) {
	  //Set the relevant bit high.
	  assert |= 1<<loopFpga;
	} else if ( "Deassert" == sParsedData[element][loopFpga+1-selectedFpga] ) {
	  //Do nothing the bit is already zero.
	} else {
	  fwUkl1_ErrorMessage(registerNames[element] + " has misformed argument: " + sParsedData[element][loopFpga+1-selectedFpga] + ". Should be `Assert' or `Deassert', case sensitive.",
			      FWUKL1_INFO_MESSAGE_LEVEL);
	  status = -2;
	}
      }
      //Convert the mask data to a string.
      masks[element] = fwCcpc_convertHexToByte(_fwUkl1_PadString(fwCcpc_convertDecToHex(mask),4));
      //Convert the register data to a string.
      sData[element] = _fwUkl1_PadString( fwCcpc_convertDecToHex(assert), 4 );
    }
    else {
       //These are useful for holding the mask and data to be written as it is determined.
      unsigned mask = 0;
      unsigned number = 0;
      //Now loop over the appropriate range of FPGAs.
      //Loop should start at the FE FPGA we selected in name prefix.
      //Determine the end of the loop.
      const unsigned endFpga = FWUKL1_BROADCAST_FPGA==feFpga ? (FWUKL1_FRONT_FPGAS-1):feFpga;
      for (unsigned loopFpga = selectedFpga; loopFpga <= endFpga; ++loopFpga) {
	//Create the mask. Store it as a single number and convert it to a hex string at the end.
	mask |= 0xf<<loopFpga;
	//This is a bit of a funny way to do things, but there are no generic registers for FE specific commands,
	//so this might make sense one day. At least it will do something sensible one day...
	//Convert the four numbers got from the string to decimal and perform a bitwise OR. once they have been aligned
	//to the appropriate byte boundary.
	number |= fwCcpc_convertHexToDec(sParsedData[element][loopFpga+1-selectedFpga])<<feFpga;
      }
      //Convert the mask data to a string.
      masks[element] = fwCcpc_convertHexToByte(_fwUkl1_PadString(fwCcpc_convertDecToHex(mask),4));
      //Convert the register data to a string.
      sData[element] = _fwUkl1_PadString( fwCcpc_convertDecToHex(assert), 4 );
    }

    //Put the data and register names in a format that the write can use.
    //Add the prefix to convert it to a datapoint name.
    registerNames[element] = prefix + registerNames[element];

    //Convert from a hex string to a byte array.
    ddcData[element] = fwCcpc_convertHexToByte( sData[element] );
  }

  return 0;
}

/*!
 * This is an internal function that will convert the data and register names given to the fwUkl1_FeFpgaChannelWrite function
 * and converts them into a form that can be understood by the hardware writes. It also creates the appropriate masks
 * for the data functions. It can also deal with comma separated lists in the event that broadcasts are done to all channels
 * on a FE FPGA. In this case all the settings for the 9 channels on a given FE FPGA can be given as a comma separated list.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  channel Input channel number on a FE FPGA that is to be written to. Range: 0 to 8, FWUKL1_BROADCAST_CHANNEL.
 * \param  feFpga Number of the FE FPGA that is to be written. Range:0 to 3.
 * \param  registerName List of BE FPGA register names that are to be written to. Returned with the appropriate name
 *           prefix by reference.
 * \param  sData Data to be written to the registers in the form of a string. The allowed values for these are described
 *           in the fwUkl1_BeFpgaWrite function documentation.
 * \param  ddcData Data converted into a dyn_dyn_char for use with the fwCcpc_write function. Returned by reference.
 * \param  masks Masks that are associated with the write data.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *            The error log contains information about which register failed to write.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *            The error log contains information about which register failed to write.
 *   \li -2 Execution failed due to a problem with data or name format and no hardware/datapoint access was performed.
 */
int _fwUkl1_ParseFeFpgaChannelWriteRegisters(string ukl1Name, unsigned channel, unsigned feFpga, dyn_string& registerNames, dyn_string sData, dyn_dyn_char& ddcData, dyn_dyn_char& masks) {
  //Define the string to be added to each of the register names such that they are written to the appropriate datapoint element.
  //Either we need to write to a specific FE FPGA channel or in the case of broadcasting we can write to FE FPGA channel 0 as all
  //FE FPGA channel settings are at the same register address, we merely need to write to the first channel with the appropriate data.
  //This is why broadcasting over the UKL1 channels is not any better, as you still have to write to the four registers anyway.
  const unsigned selectedChannel = FWUKL1_BROADCAST_CHANNEL==channel ? 0:channel;
  //Define the string to be added to each of the register names such that they are written to the appropriate datapoint element.
  const string prefix = FWUKL1_HW_DPNAME_PREFIX + ukl1Name + ".FeFpga" + feFpga + ".Channel" + selectedChannel + ".";
  //First we need to parse the broadcast data to break it up into its component channel settings.
  dyn_dyn_string sParsedData = fwUkl1_ParseBroadcastData(sData);

  //Holds the overall return status of the function.
  int status = 0;

  //Add prefix to each name and convert the data into a byte array.
  for (unsigned element = 1; element <= dynlen(registerNames); ++element) {
    if ( "Disable" == registerNames[element] ) {
      //These are useful for holding the mask and data to be written as it is determined.
      unsigned mask = 0;
      unsigned disable = 0;
      //Now loop over the appropriate range of FPGAs.
      //Loop should start at the FE FPGA we selected in name prefix.
      //Determine the end of the loop.
      const unsigned endChannel = FWUKL1_BROADCAST_CHANNEL==channel ? (FWUKL1_FPGA_CHANNELS-1):channel;
      for (unsigned loopChannel = selectedChannel; loopChannel <= endChannel; ++loopChannel) {
	//Create the mask. Store it as a single number and convert it to a hex string at the end.
	mask |= 1<<loopChannel;
	//Create the data to write. In the event that we are only writing to 1 channel then we must always take
	//the data from the 1st element, as this is the only element that will contain any data.
	if ( "Disable" == sParsedData[element][loopChannel+1-selectedChannel] ) {
	  //Set the relevant bit high.
	  disable |= 1<<loopChannel;
	} else if ( "Enable" == sParsedData[element][loopChannel+1-selectedChannel] ) {
	  //Do nothing the bit is already zero.
	} else {
	  fwUkl1_ErrorMessage(registerNames[element] + " has misformed argument: " + sParsedData[element][channel+1-selectedChannel] + ". Should be `Disable' or `Enable', case sensitive.",
			      FWUKL1_INFO_MESSAGE_LEVEL);
	  status = -2;
	}
      }
      //Convert the mask data to a string.
      masks[element] = fwCcpc_convertHexToByte(_fwUkl1_PadString(fwCcpc_convertDecToHex(mask),4));
      //Convert the register data to a string.
      sData[element] = _fwUkl1_PadString( fwCcpc_convertDecToHex(disable), 4 );

    }
    else {
      //This doesn't really have any meaning...
      //Need a 16-bit mask.
      masks[element] = makeDynChar(0x00, 0x00, 0xff, 0xff);
      //Pad the string.
      sData[element] = _fwUkl1_PadString(sData[element], 4);
    }
    //Put the data and register names in a format that the write can use.
    //Add the prefix to convert it to a datapoint name.
    registerNames[element] = prefix + registerNames[element];

    //Convert from a hex string to a byte array.
    ddcData[element] = fwCcpc_convertHexToByte( sData[element] );
  }

  return status;
}

//@}


// =============================
//  GBE CONFIGURATION FUNCTIONS
// =============================

/** @defgroup SectionGbeConfiguration
 *  @{
 */

/*!
 * This writes the general GBE settings.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           written to, which corresponds to the appropriate hardware register. The following are accepted.
 *   \li JTAGID Default data format, see below.
 *   \li MACSoftReset Default data format, see below.
 *   \li RxFifoPrtReset Default data format, see below.
 *   \li TxFifoPrtReset Default data format, see below.
 *   \li PortEnable Default data format, see below.
 *   \li RxFifoEnable Default data format, see below.
 *   \li Mode Default data format, see below.
 *   \li Clock Default data format, see below.
 *   \li SPI3ConfigTrnmtGlobal Default data format, see below.
 *   \li SPI3ConfigRcv Default data format, see below.
 *   \li RxStatus Default data format, see below.
 *   \li PHYCtrl Default data format, see below.
 *   \li PHYData Default data format, see below.
 *   \li MDIOControl Default data format, see below.
 * \param  data By default a 32-bit value that is to be written to the register represented as a hex string, it need not be zero padded.
 *           Certain settings accept or require an alternative format as described for each register above.
 * \param  callStatusList This array of ints stores the return code from each write call to the fwCcpc library. The return code from this
 *           function will indicate if any of the writes failed and by checking the array the specific write failures can be
 *           identified. The order of return codes will match the ordering of the list of register names. Returned by reference.
 *           The following are the possible return codes (see fwCcpc lib):
 *   \li  0 Execution proceeded without a problem.
 *   \li >0 Error found and related with CCPC Server.
 *   \li -1 Error found and related with PVSS.
 * \param  verifyWrite If this is set to TRUE then the register will be read back from to ensure that
 *           the data written corresponds to that read. Default: FALSE.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *            The error log contains information about which register failed to write.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *            The error log contains information about which register failed to write.
 *   \li -2 Execution failed due to a problem with data or name format and no hardware/datapoint access was performed.
 */
int fwUkl1_GbeWrite(string ukl1Name, dyn_string registerNames, dyn_string data, dyn_int& callStatusList, bool verifyWrite=FALSE) {
  //Check that the register names and data arrays are the same size.
  //Save only the lenght of one of the arrays as the other must be the same size, in this case it is the registerNames.
  const unsigned numRegs = dynlen(registerNames);
  if ( numRegs != dynlen(data) ) {
    fwUkl1_ErrorMessage("registerNames size: " + numRegs + " is not equal to the data size: " + dynlen(data) + ", when writting to the GBE.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -2;
  }

  //Holds the data in the form that the fwCcpc_write function expects.
  dyn_dyn_char ddcData;
  //Holds the mask for each element.
  dyn_dyn_char masks;

  //Converts the data given as strings in the calling arguments to byte arguments required by the fwCcpc_write.
  //Also it creates the masks required for each write and pads the register names, such that they refer to the appropriate
  //data point elements.
  _fwUkl1_ParseGbeWriteRegisters(ukl1Name, registerNames, data, ddcData, masks);

  //The data and registers have been parsed, so let us write to the hardware.
  //The overall status of the write.
  int status = 0;
  //We will wait for each element to be written before moving on.
  const bool waitFlag = TRUE;
  if ( fwCcpc_write(registerNames, ddcData, masks, callStatusList, waitFlag) ) {
    status = 0;
  } else {
    //Check where the errors occured.
    const unsigned statusLen = dynlen(callStatusList);
    if ( 0 != statusLen ) {
      for (unsigned element = 1; element <= statusLen; ++element) {
	if ( 0 != callStatusList[element] ) {
	  //Construct the error message.
	  string errMsg = "While writing to \'" + registerNames[element] + "\' GbeWrite encountered ";
	  if (callStatusList[element] > 0) {
	    errMsg += "a problem writing to the CCPC server.";
	    status = 1;
	  } else if (callStatusList[element] < 0) {
	    errMsg += "a problem with PVSS.";
	    status = -1;
	  } else {
	    errMsg += "an unknown error.";
	    status = -1;
	  }
	  //Now log the error.
	  _fwUkl1_LogFwCcpcError(errMsg);
	}
      }
    } else {
      fwUkl1_ErrorMessage("GbeWrite encountered a fatal error and an unknown number of writes were successful. L1 is in a highly inconsistent state.",
			  FWUKL1_DEBUG_MESSAGE_LEVEL);
    }
  }

  if (verifyWrite) {
    //Call the verification function.
    status = _fwUkl1_VerifyWrite(registerNames, ddcData, masks, "fwUkl1_GbeWrite");
  }

  //Return the value of the last status failure. It is likely that they all failed for the same reason.
  return status;
}

/*!
 * This reads from a given set of general GBE registers.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           read from, which corresponds to the appropriate hardware register.
 * \param  data dyn_string returned by reference that contains a string representing the value in the register.
 *           It will be zero padded to the nearest 32-bit word.
 * \param  callStatusList This array of ints stores the return code from each write call to the fwCcpc library. The return code from this
 *           function will indicate if any of the writes failed and by checking the array the specific write failures can be
 *           identified. The order of return codes will match the ordering of the list of register names. Returned by reference.
 *           The following are the possible return codes (see fwCcpc lib):
 *   \li  0 Execution proceeded without a problem.
 *   \li >0 Error found and related with CCPC Server.
 *   \li -1 Error found and related with PVSS.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *            The error log contains information about which register failed to write.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *            The error log contains information about which register failed to write.
 */
int fwUkl1_GbeRead(string ukl1Name, dyn_string registerNames, dyn_string& data, dyn_int& callStatusList) {
  //Make sure we have an empty data array.
  dynClear(data);
  //Define the string to be added to each of the register names such that they are written to the appropriate datapoint element.
  const string prefix = FWUKL1_HW_DPNAME_PREFIX + ukl1Name + ".Gbe.";
  //Useful to know how many elements we may need to loop over.
  const unsigned numRegs = dynlen(registerNames);
  //Add prefix to each name and convert the data into a byte array.
  for (unsigned element = 1; element <= numRegs; ++element) {
    //Add the prefix.
    registerNames[element] = prefix + registerNames[element];
  }

  //Do the reads...
  //Holds the data in the form that the fwCcpc_read function returns.
  dyn_dyn_char ddcData;
  fwCcpc_read(registerNames, ddcData, callStatusList);

  //The overall status of the write.
  int status = 0;
  //Check we have well formed arrays.
  if ( 0 != dynlen(callStatusList) ) {
    //Put the data into a single string for return.
    for (unsigned element = 1; element <= numRegs; ++element) {
      if ( 0 != callStatusList[element] ) {
	//Construct the error message.
	string errMsg = "While reading from \'" + registerNames[element] + "\' GbeRead encountered ";
	if (callStatusList[element] > 0) {
	  errMsg += "a problem writing to the CCPC server.";
	  status = 1;
	} else if (callStatusList[element] < 0) {
	  errMsg += "a problem with PVSS.";
	  status = -1;
	} else {
	  errMsg += "an unknown error.";
	  status = -1;
	}
	//Now log the error.
	_fwUkl1_LogFwCcpcError(errMsg);
	//Mark the error.
	data[element] = "Fail";
      }
      else {
	//Currently no exceptions.
	//Can just return the read data to a hex string as the extra words have already been removed.
	data[element] = fwCcpc_convertByteToHex(ddcData[element]);
      }
    }
  }//if ( 0 != dynlen(callStatusList) )
  else {
    //The function didn't execute properly as the callStatusList size is zero.
    fwUkl1_ErrorMessage("fwUkl1_GbeRead failed. The read did not return a populated array. It is unknown if any reads succeeded.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    status = -1;
  }

  //Return the value of the last status failure. It is likely that they all failed for the same reason.
  return status;
}

/*!
 * This writes the GBE port specific settings.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  port Output port on the GBE card that is to be set.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           written to, which corresponds to the appropriate hardware register. The following are accepted.
 *   \li Duplex Default data format, see below.
 *   \li MAC Default data format, see below.
 *   \li Config Default data format, see below.
 *   \li TXFifoThreshold Default data format, see below.
 *   \li TXFifoLowWtrmrk Default data format, see below.
 *   \li TXFifoHighWtrmrk Default data format, see below.
 *   \li PHYControl Default data format, see below.
 *   \li PHYStatus Default data format, see below.
 *   \li TXStat Default data format, see below.
 *   \li RXStat Default data format, see below.
 * \param  data By default a 32-bit value that is to be written to the register represented as a hex string, it need not be zero padded.
 *           Certain settings accept or require an alternative format as described for each register above.
 * \param  callStatusList This array of ints stores the return code from each write call to the fwCcpc library. The return code from this
 *           function will indicate if any of the writes failed and by checking the array the specific write failures can be
 *           identified. The order of return codes will match the ordering of the list of register names. Returned by reference.
 *           The following are the possible return codes (see fwCcpc lib):
 *   \li  0 Execution proceeded without a problem.
 *   \li >0 Error found and related with CCPC Server.
 *   \li -1 Error found and related with PVSS.
 * \param  verifyWrite If this is set to TRUE then the register will be read back from to ensure that
 *           the data written corresponds to that read. Default: FALSE.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *            The error log contains information about which register failed to write.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *            The error log contains information about which register failed to write.
 *   \li -2 Execution failed due to a problem with data or name format and no hardware/datapoint access was performed.
 */
int fwUkl1_GbePortWrite(string ukl1Name, unsigned port, dyn_string registerNames, dyn_string data, dyn_int& callStatusList, bool verifyWrite=FALSE) {
  //Check that the register names and data arrays are the same size.
  //Save only the lenght of one of the arrays as the other must be the same size, in this case it is the registerNames.
  const unsigned numRegs = dynlen(registerNames);
  if ( numRegs != dynlen(data) ) {
    fwUkl1_ErrorMessage("registerNames size: " + numRegs + " is not equal to the data size: " + dynlen(data) + ", when writting to the GBE port " + port + ".",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -2;
  }

  //Holds the data in the form that the fwCcpc_write function expects.
  dyn_dyn_char ddcData;
  //Holds the mask for each element.
  dyn_dyn_char masks;

  //Converts the data given as strings in the calling arguments to byte arguments required by the fwCcpc_write.
  //Also it creates the masks required for each write and pads the register names, such that they refer to the appropriate
  //data point elements.
  _fwUkl1_ParseGbePortWriteRegisters(ukl1Name, port, registerNames, data, ddcData, masks);

  //The data and registers have been parsed, so let us write to the hardware.
  //The overall status of the write.
  int status = 0;
  //We will wait for each element to be written before moving on.
  const bool waitFlag = TRUE;
  const bool returnStatus = fwCcpc_write(registerNames, ddcData, masks, callStatusList, waitFlag);
  if ( returnStatus ) {
    status = 0;
  } else {
    //Check where the errors occured.
    const unsigned statusLen = dynlen(callStatusList);
    if ( 0 != statusLen ) {
      for (unsigned element = 1; element <= statusLen; ++element) {
	if ( 0 != callStatusList[element] ) {
	  //Construct the error message.
	  string errMsg = "While writing to \'" + registerNames[element] + "\' GbePortWrite encountered ";
	  if (callStatusList[element] > 0) {
	    errMsg += "a problem writing to the CCPC server.";
	    status = 1;
	  } else if (callStatusList[element] < 0) {
	    errMsg += "a problem with PVSS.";
	    status = -1;
	  } else {
	    errMsg += "an unknown error.";
	    status = -1;
	  }
	  //Now log the error.
	  _fwUkl1_LogFwCcpcError(errMsg);
	}
      }
    } else {
      fwUkl1_ErrorMessage("GbePortWrite encountered a fatal error and an unknown number of writes were successful. L1 is in a highly inconsistent state.",
			  FWUKL1_DEBUG_MESSAGE_LEVEL);
    }
  }
  if (verifyWrite) {
    //Call the verification function.
    status = _fwUkl1_VerifyWrite(registerNames, ddcData, masks, "fwUkl1_GbePortWrite");
  }
  //Return the value of the last status failure. It is likely that they all failed for the same reason.
  return status;
}

/*!
 * This reads from a GBE port's registers.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  port Output port on the GBE card that is to be set.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           read from, which corresponds to the appropriate hardware register.
 * \param  data dyn_string returned by reference that contains a string representing the value in the register.
 *           It will be zero padded to the nearest 32-bit word.
 * \param  callStatusList This array of ints stores the return code from each write call to the fwCcpc library. The return code from this
 *           function will indicate if any of the writes failed and by checking the array the specific write failures can be
 *           identified. The order of return codes will match the ordering of the list of register names. Returned by reference.
 *           The following are the possible return codes (see fwCcpc lib):
 *   \li  0 Execution proceeded without a problem.
 *   \li >0 Error found and related with CCPC Server.
 *   \li -1 Error found and related with PVSS.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *            The error log contains information about which register failed to write.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *            The error log contains information about which register failed to write.
 */
int fwUkl1_GbePortRead(string ukl1Name, unsigned port, dyn_string registerNames, dyn_string& data, dyn_int& callStatusList) {
  //Make sure we have an empty data array.
  dynClear(data);
  //Define the string to be added to each of the register names such that they are written to the appropriate datapoint element.
  const string prefix = FWUKL1_HW_DPNAME_PREFIX + ukl1Name + ".Gbe.PORT" + port + ".";
  //Useful to know how many elements we may need to loop over.
  const unsigned numRegs = dynlen(registerNames);
  //Add prefix to each name and convert the data into a byte array.
  for (unsigned element = 1; element <= numRegs; ++element) {
    //Add the prefix.
    registerNames[element] = prefix + registerNames[element];
  }

  //Do the reads...
  //Holds the data in the form that the fwCcpc_read function returns.
  dyn_dyn_char ddcData;
  fwCcpc_read(registerNames, ddcData, callStatusList);

  //The overall status of the write.
  int status = 0;
  //Check that the arrays are well formed.
  if ( 0 != dynlen(callStatusList) ) {
    //Put the data into a single string for return.
    for (unsigned element = 1; element <= numRegs; ++element) {
      if ( 0 != callStatusList[element] ) {
	//Construct the error message.
	string errMsg = "While reading from \'" + registerNames[element] + "\' GbePortRead encountered ";
	if (callStatusList[element] > 0) {
	  errMsg += "a problem writing to the CCPC server.";
	  status = 1;
	} else if (callStatusList[element] < 0) {
	  errMsg += "a problem with PVSS.";
	  status = -1;
	} else {
	  errMsg += "an unknown error.";
	  status = -1;
	}
	//Now log the error.
	_fwUkl1_LogFwCcpcError(errMsg);
	//Mark the error.
	data[element] = "Fail";
      }
      //Deal with the different formats.
      else if ( ( (prefix + "TXFifoThreshold") == registerNames[element] ) ||
		( (prefix + "TXFifoLowWtrmrk") == registerNames[element] ) ||
		( (prefix + "TXFifoHighWtrmrk") == registerNames[element] ) ) {
	//These want to be decimal numbers.
	data[element] = fwCcpc_convertByteToDec(ddcData[element]);
      }
      else {
	//Can just return the read data to a hex string.
	data[element] = fwCcpc_convertByteToHex(ddcData[element]);
      }
    }
  }//if ( 0 != dynlen(callStatusList) )
  else {
    //The function didn't execute properly as the callStatusList size is zero.
    fwUkl1_ErrorMessage("fwUkl1_GbePortRead failed. The read did not return a populated array. It is unknown if any reads succeeded.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    status = -1;
  }

  //Return the value of the last status failure. It is likely that they all failed for the same reason.
  return status;
}

// =============================
//  INTERNAL FUNCTIONS
// =============================

/*!
 * This is an internal function that will convert the data and register names given to the fwUkl1_GbeWrite function
 * and converts them into a form that can be understood by the hardware writes. It also creates the appropriate masks
 * for the data functions.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  registerName List of GBE register names that are to be written to. Returned with the appropriate name
 *           prefix by reference.
 * \param  sData Data to be written to the registers in the form of a string. The allowed values for these are described
 *           in the fwUkl1_GbeWrite function documentation.
 * \param  ddcData sData converted into a dyn_dyn_char for use with the fwCcpc_write function. Returned by reference.
 * \param  masks Masks that are associated with the write data.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *            The error log contains information about which register failed to write.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *            The error log contains information about which register failed to write.
 *   \li -2 Execution failed due to a problem with data or name format and no hardware/datapoint access was performed.
 */
int _fwUkl1_ParseGbeWriteRegisters(string ukl1Name, dyn_string& registerNames, dyn_string sData, dyn_dyn_char& ddcData, dyn_dyn_char& masks) {
  //Define the string to be added to each of the register names such that they are written to the appropriate datapoint element.
  const string prefix = FWUKL1_HW_DPNAME_PREFIX + ukl1Name + ".Gbe.";

  //Add prefix to each name and convert the data into a byte array.
  for (unsigned element = 1; element <= dynlen(registerNames); ++element) {
    //Currently there are no exceptions.
    //Need a 32-bit mask.
    masks[element]   = makeDynChar(0xff, 0xff, 0xff, 0xff);
    //Pad the string.
    sData[element] = _fwUkl1_PadString(sData[element], 4);
    //Put the data and register names in a format that the write can use.
    //Add the prefix to convert it to a datapoint name.
    registerNames[element] = prefix + registerNames[element];

    //Convert from a hex string to a byte array.
    ddcData[element] = fwCcpc_convertHexToByte( sData[element] );
  }

  return 0;
}

/*!
 * This is an internal function that will convert the data and register names given to the fwUkl1_GbePortWrite function
 * and converts them into a form that can be understood by the hardware writes. It also creates the appropriate masks
 * for the data functions.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  port Output port on the GBE card that is to be set.
 * \param  registerName List of GBE register names that are to be written to. Returned with the appropriate name
 *           prefix by reference.
 * \param  sData Data to be written to the registers in the form of a string. The allowed values for these are described
 *           in the fwUkl1_GbePortWrite function documentation.
 * \param  ddcData sData converted into a dyn_dyn_char for use with the fwCcpc_write function. Returned by reference.
 * \param  masks Masks that are associated with the write data.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *            The error log contains information about which register failed to write.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *            The error log contains information about which register failed to write.
 *   \li -2 Execution failed due to a problem with data or name format and no hardware/datapoint access was performed.
 */
int _fwUkl1_ParseGbePortWriteRegisters(string ukl1Name, unsigned port, dyn_string& registerNames, dyn_string sData, dyn_dyn_char& ddcData, dyn_dyn_char& masks) {
  //Define the string to be added to each of the register names such that they are written to the appropriate datapoint element.
  const string prefix = FWUKL1_HW_DPNAME_PREFIX + ukl1Name + ".Gbe.PORT" + port + ".";
  //Add prefix to each name and convert the data into a byte array.
  for (unsigned element = 1; element <= dynlen(registerNames); ++element) {
    //Need a 32-bit mask, the same for all.
    masks[element]   = makeDynChar(0xff, 0xff, 0xff, 0xff);
    //Deal with the different formats.
    if ( ("TXFifoThreshold" == registerNames[element]) ||
	 ("TXFifoLowWtrmrk" == registerNames[element]) ||
	 ("TXFifoHighWtrmrk" == registerNames[element]) ) {
      //These will be decimal numbers.
      sData[element] = fwCcpc_convertDecToHex(sData[element]);
    }
    //Everything needs to be padded the string.
    sData[element] = _fwUkl1_PadString(sData[element], 4);
    //Put the data and register names in a format that the write can use.
    //Add the prefix to convert it to a datapoint name.
    registerNames[element] = prefix + registerNames[element];

    //Convert from a hex string to a byte array.
    ddcData[element] = fwCcpc_convertHexToByte( sData[element] );
    
  }

  return 0;
}

//@}


// ===============================
//  TTCRX CONFIGURATION FUNCTIONS
// ===============================

/** @defgroup SectionTtcrxConfiguration
 *  @{
 */

/*!
 * This writes the general TTCrx settings.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           written to, which corresponds to the appropriate hardware register. The following are accepted.
 *   \li IACId Requires a two byte word.
 *   \li I2CId Default data format, see below.
 *   \li FineDelay1 Default data format, see below.
 *   \li FineDelay2 Default data format, see below.
 *   \li CoarseDelay Default data format, see below.
 *   \li Control Default data format, see below.
 *   \li Status Default data format, see below.
 *   \li Config1 Default data format, see below.
 *   \li Config2 Default data format, see below.
 *   \li Config3 Default data format, see below.
 *   \li SingleErrCnt Default data format, see below.
 *   \li DoubleErrCnt Default data format, see below.
 *   \li SEUErrCnt Requires a two byte word, read only.
 *   \li BunchCnt Requires a two byte word, read only.
 *   \li EventCnt Requires a three byte word, read only.
 * \param  data By default an 8-bit value that is to be written to the register represented as a hex string, it need not be zero padded.
 *           Certain settings accept or require an alternative format as described for each register above.
 * \param  callStatusList This array of ints stores the return code from each write call to the fwCcpc library. The return code from this
 *           function will indicate if any of the writes failed and by checking the array the specific write failures can be
 *           identified. The order of return codes will match the ordering of the list of register names. Returned by reference.
 *           The following are the possible return codes (see fwCcpc lib):
 *   \li  0 Execution proceeded without a problem.
 *   \li >0 Error found and related with CCPC Server.
 *   \li -1 Error found and related with PVSS.
 * \param  verifyWrite If this is set to TRUE then the register will be read back from to ensure that
 *           the data written corresponds to that read. Default: FALSE.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *            The error log contains information about which register failed to write.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *            The error log contains information about which register failed to write.
 *   \li -2 Execution failed due to a problem with data or name format and no hardware/datapoint access was performed.
 */
int fwUkl1_TtcrxWrite(string ukl1Name, dyn_string registerNames, dyn_string data, dyn_int& callStatusList, bool verifyWrite=FALSE) {
  //Check that the register names and data arrays are the same size.
  //Save only the lenght of one of the arrays as the other must be the same size, in this case it is the registerNames.
  const unsigned numRegs = dynlen(registerNames);
  if ( numRegs != dynlen(data) ) {
    fwUkl1_ErrorMessage("registerNames size: " + numRegs + " is not equal to the data size: " + dynlen(data) + ", when writting to the TTCrx.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -2;
  }

  //Holds the data in the form that the fwCcpc_write function expects.
  dyn_dyn_char ddcData;
  //Holds the mask for each element.
  dyn_dyn_char masks;

  //Converts the data given as strings in the calling arguments to byte arguments required by the fwCcpc_write.
  //Also it creates the masks required for each write and pads the register names, such that they refer to the appropriate
  //data point elements.
  _fwUkl1_ParseTtcrxWriteRegisters(ukl1Name, registerNames, data, ddcData, masks);

  //The data and registers have been parsed, so let us write to the hardware.
  //The overall status of the write.
  int status = 0;
  //We will wait for each element to be written before moving on.
  const bool waitFlag = TRUE;
  if ( fwCcpc_write(registerNames, ddcData, masks, callStatusList, waitFlag) ) {
    status = 0;
  } else {
    //Check where the errors occured.
    const unsigned statusLen = dynlen(callStatusList);
    if ( 0 != statusLen ) {
      for (unsigned element = 1; element <= statusLen; ++element) {
	if ( 0 != callStatusList[element] ) {
	  //Construct the error message.
	  string errMsg = "While writing to \'" + registerNames[element] + "\' TtcrxWrite encountered ";
	  if (callStatusList[element] > 0) {
	    errMsg += "a problem writing to the CCPC server.";
	    status = 1;
	  } else if (callStatusList[element] < 0) {
	    errMsg += "a problem with PVSS.";
	    status = -1;
	  } else {
	    errMsg += "an unknown error.";
	    status = -1;
	  }
	  //Now log the error.
	  _fwUkl1_LogFwCcpcError(errMsg);
	}
      }
    } else {
      fwUkl1_ErrorMessage("TtcrxWrite encountered a fatal error and an unknown number of writes were successful. L1 is in a highly inconsistent state.",
			  FWUKL1_DEBUG_MESSAGE_LEVEL);
    }
  }

  if (verifyWrite) {
    //Call the verification function.
    status = _fwUkl1_VerifyWrite(registerNames, ddcData, masks, "fwUkl1_TtcrxWrite");
  }

  //Return the value of the last status failure. It is likely that they all failed for the same reason.
  return status;
}

/*!
 * This reads from a given set of general TTCrx registers.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           read from, which corresponds to the appropriate hardware register.
 * \param  data dyn_string returned by reference that contains a string representing the value in the register.
 *           It will be zero padded to the nearest byte boundary.
 * \param  callStatusList This array of ints stores the return code from each write call to the fwCcpc library. The return code from this
 *           function will indicate if any of the writes failed and by checking the array the specific write failures can be
 *           identified. The order of return codes will match the ordering of the list of register names. Returned by reference.
 *           The following are the possible return codes (see fwCcpc lib):
 *   \li  0 Execution proceeded without a problem.
 *   \li >0 Error found and related with CCPC Server.
 *   \li -1 Error found and related with PVSS.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *            The error log contains information about which register failed to write.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *            The error log contains information about which register failed to write.
 */
int fwUkl1_TtcrxRead(string ukl1Name, dyn_string registerNames, dyn_string& data, dyn_int& callStatusList) {
  //Make sure we have an empty data array.
  dynClear(data);
  //Define the string to be added to each of the register names such that they are written to the appropriate datapoint element.
  const string prefix = FWUKL1_HW_DPNAME_PREFIX + ukl1Name + ".Ttcrx.";
  //Useful to know how many elements we may need to loop over.
  const unsigned numRegs = dynlen(registerNames);
  //Add prefix to each name and convert the data into a byte array.
  for (unsigned element = 1; element <= numRegs; ++element) {
    //Add the prefix.
    registerNames[element] = prefix + registerNames[element];
  }

  //Do the reads...
  //Holds the data in the form that the fwCcpc_read function returns.
  dyn_dyn_char ddcData;
  //The overall status of the write.
  int status = 0;
  fwCcpc_read(registerNames, ddcData, callStatusList);

  //Check the arrays are well formed.
  if ( 0 != dynlen(callStatusList) ) {
    //Put the data into a single string for return.
    for (unsigned element = 1; element <= numRegs; ++element) {
      if ( 0 != callStatusList[element] ) {
	//Construct the error message.
	string errMsg = "While reading from \'" + registerNames[element] + "\' TtcrxRead encountered ";
	if (callStatusList[element] > 0) {
	  errMsg += "a problem writing to the CCPC server.";
	  status = 1;
	} else if (callStatusList[element] < 0) {
	  errMsg += "a problem with PVSS.";
	  status = -1;
	} else {
	  errMsg += "an unknown error.";
	  status = -1;
	}
	//Now log the error.
	_fwUkl1_LogFwCcpcError(errMsg);
	//Mark the error.
	data[element] = "Fail";
	//Skip the rest of the loop, there is no need to do any further data checking.
	continue;
      }
      else {
	//Currently no exceptions.
	data[element] = fwCcpc_convertByteToHex(ddcData[element]);
      }
    }
  }//if ( 0 != dynlen(callStatusList) )
  else {
    //The function didn't execute properly as the callStatusList size is zero.
    fwUkl1_ErrorMessage("fwUkl1_TtcrxRead failed. The read did not return a populated array. It is unknown if any reads succeeded.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    status = -1;
  }

  //Return the value of the last status failure. It is likely that they all failed for the same reason.
  return status;
}

// =============================
//  INTERNAL FUNCTIONS
// =============================

/*!
 * This is an internal function that will convert the data and register names given to the fwUkl1_TtcrxWrite function
 * and converts them into a form that can be understood by the hardware writes. It also creates the appropriate masks
 * for the data functions.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  registerName List of TTCrx register names that are to be written to. Returned with the appropriate name
 *           prefix by reference.
 * \param  sData Data to be written to the registers in the form of a string. The allowed values for these are described
 *           in the fwUkl1_TtcrxWrite function documentation.
 * \param  ddcData sData converted into a dyn_dyn_char for use with the fwCcpc_write function. Returned by reference.
 * \param  masks Masks that are associated with the write data.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *            The error log contains information about which register failed to write.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *            The error log contains information about which register failed to write.
 *   \li -2 Execution failed due to a problem with data or name format and no hardware/datapoint access was performed.
 */
int _fwUkl1_ParseTtcrxWriteRegisters(string ukl1Name, dyn_string& registerNames, dyn_string sData, dyn_dyn_char& ddcData, dyn_dyn_char& masks) {
  //Define the string to be added to each of the register names such that they are written to the appropriate datapoint element.
  const string prefix = FWUKL1_HW_DPNAME_PREFIX + ukl1Name + ".Ttcrx.";

  //Add prefix to each name and convert the data into a byte array.
  for (unsigned element = 1; element <= dynlen(registerNames); ++element) {
    //The execptions are those that require more than one byte and hence a larger mask.
    if ( "IACId" == registerNames[element] ) {
      //This requires a two byte mask and padding.
      masks[element]   = makeDynChar(0xff, 0xff);
      sData[element] = _fwUkl1_PadString(sData[element], 2);
    }
    else if ( "SingleErrCnt" == registerNames[element] ) {
      //This requires a two byte mask and padding.
      masks[element]   = makeDynChar(0xff, 0xff);
      sData[element] = _fwUkl1_PadString(sData[element], 2);
    }
    else if ( "BunchCnt" == registerNames[element] ) {
      //This requires a two byte mask and padding.
      masks[element]   = makeDynChar(0xff, 0xff);
      sData[element] = _fwUkl1_PadString(sData[element], 2);
    }
    else if ( "EventCnt" == registerNames[element] ) {
      //This requires a three byte mask and padding.
      masks[element]   = makeDynChar(0xff, 0xff, 0xff);
      sData[element] = _fwUkl1_PadString(sData[element], 3);
    }
    else {
      //The standard only require an 8-bit mask.
      masks[element]   = makeDynChar(0xff);
      //Pad the string, such that it is only 1-byte big.
      sData[element] = _fwUkl1_PadString(sData[element], 1);
    }
    
    //Put the data and register names in a format that the write can use.
    //Add the prefix to convert it to a datapoint name.
    registerNames[element] = prefix + registerNames[element];
    //Convert from a hex string to a byte array.
    ddcData[element] = fwCcpc_convertHexToByte( sData[element] );
  }

  return 0;
}

//@}


// ===============================
//  GPIO CONFIGURATION FUNCTIONS
// ===============================

/** @defgroup SectionGpioConfiguration
 *  @{
 */

/*!
 * This enables a GPIO line for input or output.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  line Number of the GPIO line to be enabled. Range:FWCCPC_GPIO_LINE_0 to FWCCPC_GPIO_LINE_8, as defined in fwCcpc.ctl.
 * \param  output True if the line is to be enabled for output, false if for input.
 * \param  regName Name of the register to be written to. This is used only for error logging and
 *           is defaulted to an empty string, in which case no logging will be performed.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *   \li -2 Execution failed due to an out of range GPIO line number.
 *
 * In addition to returning an error code the function will also write any information to the library error log, which
 * can be identified by the register name that is given as an argument. If no error occurs no error information is recorded.
 */
int fwUkl1_GpioEnable(string ukl1Name, int line, bool output, string regName="") {
  //Perform a range check.
  if ( (FWCCPC_GPIO_LINE_0 > line) || (FWCCPC_GPIO_LINE_8 < line) ) {
    fwUkl1_StatusMessage("GPIO line number " + line + " is out of range. Min: " + FWCCPC_GPIO_LINE_0 + ", Max: " + FWCCPC_GPIO_LINE_8,
			 FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -2;
  }

  //Decide whether they wish to enable for output or input.
  //Set for input and then change if output.
  unsigned lineOutputs = FWCCPC_GPIO_INPUT;
  if (output) {
    lineOutputs = FWCCPC_GPIO_OUTPUT;
  }
  fwUkl1_StatusMessage("output: " + output + " lineOutput: " + lineOutputs + " line: " + line, FWUKL1_VERBOSE_MESSAGE_LEVEL);
  int callStatus = fwCcpc_GPIOEnable(ccpcName, line, lineOutputs);
  //Check for errors.
  if ( (0 != callStatus) && ("" != regName) ) {
    //Construct the error message.
    string errMsg = "While writing to \'"+regName+"\' GpioEnabled encountered";
    if (callStatus > 0) {
      errMsg += " a problem writing to the CCPC server.";
    } else if (callStatus < 0) {
      errMsg += " a problem with PVSS.";
    } else {
      errMsg += " an unknown error.";
    }
    //Now log the error.
    _fwUkl1_LogFwCcpcError(errMsg);
  }

  //Return the call status to indicate whether we were successful or not.
  if (0 < callStatus) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Otherwise the return values match mine.
    return callStatus;
  }
}

/*!
 * This sets the GPIO line to either high or low.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  line Number of the GPIO line to be set. Range:FWCCPC_GPIO_LINE_0 to FWCCPC_GPIO_LINE_8, as defined in fwCcpc.ctl.
 * \param  high True if the line is to be set high and false if it is to be set low.
 * \param  regName Name of the register to be written to. This is used only for error logg  //Now configure the other general settings.ing and
 *           is defaulted to an empty string, in which case no logging will be performed.
 * \param  verifyWrite If this is set to TRUE then the register will be read back from to ensure that
 *           the data written corresponds to that read. Default: FALSE.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *   \li -2 Execution failed due to an out of range GPIO line number.
 *
 * In addition to returning an error code the function will also write any information to the library error log, which
 * can be identified by the register name that is given as an argument. If no error occurs no error information is recorded.
 */
int fwUkl1_GpioSet(string ccpcName, unsigned line, bool high, string regName="", bool verifyWrite=FALSE) {
  //Perform a range check.
  if ( (FWCCPC_GPIO_LINE_0 > line) || (FWCCPC_GPIO_LINE_8 < line) ) {
    fwUkl1_StatusMessage("GPIO line number " + line + " is out of range. Min: " + FWCCPC_GPIO_LINE_0 + ", Max: " + FWCCPC_GPIO_LINE_8,
			 FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -2;
  }

  //Decide whether it is to be set high or low.
  //Set it to low and then change if it is high.
  unsigned lineHigh = FWCCPC_GPIO_LOW;
  if (high) {
    lineHigh = FWCCPC_GPIO_HIGH;
  }
  int callStatus = fwCcpc_GPIOSet(ccpcName, line, lineHigh);
  //Check for errors.
  if ( (0 != callStatus) && ("" != regName) ) {
    //Construct the error message.
    string errMsg = "While writing to \'"+regName+"\' GpioSet encountered";
    if (callStatus > 0) {
      errMsg += " a problem writing to the CCPC server.";
    } else if (callStatus < 0) {
      errMsg += " a problem with PVSS.";
    } else {
      errMsg += " an unknown error.";
    }
    //Now log the error.
    _fwUkl1_LogFwCcpcError(errMsg);
  }

  if (verifyWrite) {
    //We wish to check the values written to the L1 board are the same as those we read back.
    //For now we will naively assume that if the write works then so will the read.
    unsigned checkHigh;
    fwCcpc_GPIOGet(ccpcName, line, checkHigh);
    //Now check the return value matches the written value.
    if ( checkHigh != lineHigh ) {
      fwUkl1_WarningMessage("read value, "+checkHigh+", does not match written value, "+lineHigh+", for GPIO set.", FWUKL1_INFO_MESSAGE_LEVEL);
    }
  }

  //Return the call status to indicate whether we were successful or not.
  if (0 < callStatus) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Otherwise the return values match mine.
    return callStatus;
  }
}

/*!
 * Returns the high or low state of a GPIO line.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  line Number of the GPIO line to be read. Range:FWCCPC_GPIO_LINE_0 to FWCCPC_GPIO_LINE_8, as defined in fwCcpc.ctl.
 * \param  high True if the line is set high and false if low. Returned by reference.
 * \param  regName Name of the register to be read. This is used only for error logging and
 *           is defaulted to an empty string, in which case no logging will be performed.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *   \li -2 Execution failed due to an out of range GPIO line number.
 *
 * In addition to returning an error code the function will also write any information to the library error log, which
 * can be identified by the register name that is given as an argument. If no error occurs no error information is recorded.
 */
int fwUkl1_GpioGet(string ukl1Name, unsigned line, bool& high, string regName="") {
  //Perform a range check.
  if ( (FWCCPC_GPIO_LINE_0 > line) || (FWCCPC_GPIO_LINE_8 < line) ) {
    fwUkl1_StatusMessage("GPIO line number " + line + " is out of range. Min: " + FWCCPC_GPIO_LINE_0 + ", Max: " + FWCCPC_GPIO_LINE_8,
			 FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -2;
  }

  //Holds the high/low value of the GPIO line.
  unsigned lineHigh = FWCCPC_GPIO_LOW;
  int callStatus = fwCcpc_GPIOSet(ccpcName, line, lineHigh);

  //Set the boolean to the appropriate value.
  if ( FWCCPC_GPIO_HIGH == lineHigh ) {
    high = TRUE;
  } else {
    high = FALSE;
  }

  //Check for errors.
  if ( (0 != callStatus) && ("" != regName) ) {
    //Construct the error message.
    string errMsg = "While writing to \'"+regName+"\' GpioGet encountered";
    if (callStatus > 0) {
      errMsg += " a problem writing to the CCPC server.";
    } else if (callStatus < 0) {
      errMsg += " a problem with PVSS.";
    } else {
      errMsg += " an unknown error.";
    }
    //Now log the error.
    _fwUkl1_LogFwCcpcError(errMsg);
  }

  //Return the call status to indicate whether we were successful or not.
  if (0 < callStatus) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Otherwise the return values match mine.
    return callStatus;
  }
}

//@}


// ======================================
//  FPGA STATUS FUNCTIONS
// ======================================

/** @defgroup SectionFpgaStatus FPGA status functions.
 *  @{
 */

/*!
 * This reads the BE FPGA status registers.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint elements to be
 *           read from, which corresponds to the appropriate hardware register. The accepted names are:
 *    \li TTCId Returns the first byte of the TTCrx ID as a hex string.
 *    \li SPI3TxBusy Returns `Busy' if the SPI3 transmit is busy or `Not busy' if it is not. Busy indicates
 *          data is being transmited from BE FPGA to the GBE.
 * \param  data dyn_string returned by reference that contains a string representing the value in the register.
 *           It will either be a hex representation, zero padded to the nearest byte boundary or a text description
 *           of the settings.
 * \param  callStatusList This array of ints stores the return code from each write call to the fwCcpc library. The return code from this
 *           function will indicate if any of the writes failed and by checking the array the specific write failures can be
 *           identified. The order of return codes will match the ordering of the list of register names. Returned by reference.
 *           The following are the possible return codes (see fwCcpc lib):
 *   \li  0 Execution proceeded without a problem.
 *   \li >0 Error found and related with CCPC Server.
 *   \li -1 Error found and related with PVSS.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *            The error log contains information about which register failed to write.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *            The error log contains information about which register failed to write.
 */
int fwUkl1_GetBeFpgaStatus(string ukl1Name, dyn_string registerNames, dyn_string& data, dyn_int& callStatusList) {
  //Make sure we have an empty data array and status list.
  dynClear(data);
  dynClear(callStatusList);
  //Define the string to be added to each of the register names such that they are read from the appropriate datapoint element.
  const string prefix = FWUKL1_HW_DPNAME_PREFIX + ukl1Name + ".BeFpgaStatus.";
  //Useful to know how many elements we may need to loop over.
  const unsigned numRegs = dynlen(registerNames);
  //Add prefix to each name and convert the data into a byte array.
  for (unsigned element = 1; element <= numRegs; ++element) {
    //Add the prefix.
    registerNames[element] = prefix + registerNames[element];
  }

  //Do the reads...
  //Holds the data in the form that the fwCcpc_read function returns.
  dyn_dyn_char ddcData;
  //The overall status of the write.
  int status = 0;
  fwCcpc_read(registerNames, ddcData, callStatusList);

  status = fwUkl1_ParseBeFpgaStatusData(ukl1Name, registerNames, ddcData, data, callStatusList);

  //Return the value of the last status failure. It is likely that they all failed for the same reason.
  return status;
}

/*!
 * This takes the data read from the BE FPGA status register in the format as read from a data point and
 * converts it to a string value for display.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint elements to be
 *           read from, which corresponds to the appropriate hardware register. The accepted names are:
 *    \li TTCId Returns the first byte of the TTCrx ID as a hex string.
 *    \li SPI3TxBusy Returns `Busy' if the SPI3 transmit is busy or `Not busy' if it is not. Busy indicates
 *          data is being transmited from BE FPGA to the GBE.
 *    \li MepFifoFull Returns `Full' if the MEP FIFO is full or `Not full' if it is not. Indicates the amount of data
 *          present in the MEP FIFO.
 *    \li Throttled Returns `Throttled' or `Not throttled' depending on the state of the throttle.
 * \param  ddcData This contains the raw data that has been read back from the PVSS datapoint.
 * \param  data dyn_string returned by reference that contains a string representing the value in the register.
 *           It will either be a hex representation, zero padded to the nearest byte boundary or a text description
 *           of the settings. In the event that the read failed it will contain the text `Fail'.
 * \param  callStatusList This array of ints stores the return code from each write call to the fwCcpc library. The return code from this
 *           function will indicate if any of the writes failed and by checking the array the specific write failures can be
 *           identified. The order of return codes will match the ordering of the list of register names. Returned by reference.
 *           The following are the possible return codes (see fwCcpc lib):
 *   \li  0 Execution proceeded without a problem.
 *   \li >0 Error found and related with CCPC Server.
 *   \li -1 Error found and related with PVSS.
 *           This array is not required in all situation. As a result it is defaulted to an empty array, this will cause
 *           the given data to be parsed regardless, even if it is likely to be junk. This can result in an undefined
 *           return value.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *            The error log contains information about which register failed to write.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *            The error log contains information about which register failed to write.
 */
int fwUkl1_ParseBeFpgaStatusData(string ukl1Name, dyn_string registerNames, dyn_dyn_char ddcData, dyn_string& data, dyn_int callStatusList=makeDynInt()) {
  //Define the string to be added to each of the register names such that they are read from the appropriate datapoint element.
  const string prefix = FWUKL1_HW_DPNAME_PREFIX + ukl1Name + ".BeFpgaStatus.";
  //Determine the number of registers for which we are reading the data.
  const unsigned numRegs = dynlen(registerNames);
  //Determine whether we should be checking the status list. If it is empty don't look in it.
  const bool checkStatus = (0==dynlen(callStatusList) ? FALSE:TRUE);
  //Used to check whether the function executed without a problem.
  int status = 0;
  //Loop over all the given registers and put the data into a single string for return.
  for (unsigned element = 1; element <= numRegs; ++element) {
    //This series of `if' statements determines how to handle the possible raw status register data we may receive.
    //Check for errors. As we are using AND the second case will only be evaluated provided checkStatus is not empty
    //and hence we should avoid out of bounds errors.
    //Check for both the case when the register names has the appropriate prefix and for the case where it doesn't.
    if ( checkStatus && (0 != callStatusList[element]) && (0 != dynlen(ddcData[element])) ) {
      //Construct the error message.
      string errMsg = "While reading from \'" + registerNames[element] + "\' BeFpgaStatus encountered ";
      if (callStatusList[element] > 0) {
	errMsg += "a problem writing to the CCPC server.";
	status = 1;
      } else if (callStatusList[element] < 0) {
	errMsg += "a problem with PVSS.";
	status = -1;
      } else {
	errMsg += "an unknown error.";
	status = -1;
      }
      //Now log the error.
      _fwUkl1_LogFwCcpcError(errMsg);
      //Mark the error.
      data[element] = "Fail";
    }
    else if ( ((prefix + "TTCId") == registerNames[element]) || ("TTCId" == registerNames[element]) ) {
      //Return the TTCId as an 8 bit number by removing the extra bytes.
      //Note the deliberate use of `<', this is so we don't delete all the elements!
      //Calculate the maximum number of iterations for the loop now, as otherwise the length calculation will change
      //with each loop.
      const unsigned totalBytes = dynlen(ddcData[element]);
      for (unsigned byte = 1; byte < totalBytes; ++byte) {
	//Always remove the first element, until we only have one left.
	dynRemove(ddcData[element], 1);
      }
      //Now convert it to hex.
      data[element] = fwCcpc_convertByteToHex(ddcData[element]);
    }
    else if ( ((prefix + "SPI3TxBusy") == registerNames[element]) || ("SPI3TxBusy" == registerNames[element]) ) {
      //Wish to look at the LSB.
      //This is the index of the element that should contain the appropriate status bit.
      const int index = dynlen(ddcData[element]);
      if ( 0x01 == (ddcData[element][index]&0x01) ) {
	data[element] = "Busy";
      } else if ( 0x00 == (ddcData[element][index]&0x01) ){
	data[element] = "Not busy";
      } else {
	data[element] = "Fail";
	status = -2;
	fwUkl1_ErrorMessage(registerNames[element] + " has unrecognised value: " + fwCcpc_convertByteToHex(ddcData[element][1]) + ".",
			    FWUKL1_INFO_MESSAGE_LEVEL);
      }
    }
    else if ( ((prefix + "MepFifoFull") == registerNames[element]) || ("MepFifoFull" == registerNames[element]) ) {
      //Wish to look at the LSB.
      //This is the index of the element that should contain the appropriate status bit.
      const int index = dynlen(ddcData[element]);
      if ( 0x02 == (ddcData[element][index]&0x02) ) {
	data[element] = "Full";
      } else if ( 0x00 == (ddcData[element][index]&0x02) ){
	data[element] = "Not full";
      } else {
	data[element] = "Fail";
	status = -2;
	fwUkl1_ErrorMessage(registerNames[element] + " has unrecognised value: " + fwCcpc_convertByteToHex(ddcData[element][1]) + ".",
			    FWUKL1_INFO_MESSAGE_LEVEL);
      }
    }
    else if ( ((prefix + "Throttled") == registerNames[element]) || ("Throttled" == registerNames[element]) ) {
      //Wish to look at the LSB.
      //This is the index of the element that should contain the appropriate status bit.
      const int index = dynlen(ddcData[element]);
      if ( 0x0c == (ddcData[element][index]&0x0c) ) {
	data[element] = "Throttled";
      } else if ( 0x00 == (ddcData[element][index]&0x0c) ){
	data[element] = "Not throttled";
      } else {
	data[element] = "Fail";
	status = -2;
	fwUkl1_ErrorMessage(registerNames[element] + " has unrecognised value: " + fwCcpc_convertByteToHex(ddcData[element][1]) + ".",
			    FWUKL1_INFO_MESSAGE_LEVEL);
      }
    }
    else {
      //For those values that don't need to be treated.
      data[element] = fwCcpc_convertByteToHex(ddcData[element]);
    }
  }
  return status;
}

//@}


// ======================================
//  STATE MACHINE FUNCTIONS
// ======================================

/** @defgroup SectionState State machine functions.
 *  @{
 */

/*!
 * Sets the state of the current UKL1, checking that the proposed change is allowed. Only certain
 * state transitions are viable.
 *
 * \param  newState The state the UKL1 is to be changed to. Use the strings defined by UKL1.ctl library
 *           to ensure recognised state transitions e.g. FWUKL1_CONFIGURING_STATE.
 * \return int An error code indicating if the operation was successful. The codes are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with the CCPC server.
 *   \li  2 Change was prevented as state change was not compatible with previous state.
 *   \li  3 Unrecognised state was requested.
 *
 * A more detailed explanation of what state transitions are permitted can be found in the library documentation.
 * The board that will be addressed is that selected by the global variable ukl1CcpcName.
 */
int fwUkl1_SetUkl1StateCB(string dpe, int newState) {
  //Try to manipulate the dpe to remove _online.._value bit, might help with the setting.
  const string dpName = dpSubStr(dpe, DPSUB_DP_EL);
  fwUkl1_StatusMessage("Accessing: " + dpName, FWUKL1_VERBOSE_MESSAGE_LEVEL);
  //Get the name of the UKL1 board from the data point by first stripping to just datapoint name
  //and then extracting it from the FSM DP name.
  const string ukl1Name = fwUkl1_ConvertFsmDpNameToDimServerName( dpSubStr(dpe, DPSUB_DP) );
  fwUkl1_StatusMessage("UKL1 name: " + ukl1Name, FWUKL1_VERBOSE_MESSAGE_LEVEL);
  //We must change to the appropriate state and check that the change was successful.
  //Return with the appropriate return code, after logging an error.
  //Deal with the NOT_READY_STATE, this might actual be worth calling the reset function here.
  if (FWUKL1_NOT_READY_STATE == newState) {
    //For the moment we don't do anything, but we did it successfully :)
    return 0;
  }
  //Requested configuration of the board, do that and leave stopped.
  if (FWUKL1_CONFIGURING_STATE == newState) {
    int changeStatus = fwUkl1_Configure(ukl1Name);
    if (0 == changeStatus) {
      _fwUkl1_DpSet(dpName, FWUKL1_STOP_STATE);
    } else {
      fwUkl1_ErrorMessage("Failed to change to ready state.", FWUKL1_INFO_MESSAGE_LEVEL);
      //The state change was unsuccessful and we no longer know how many register were changed,
      //hence state is now not ready.
      _fwUkl1_DpSet(dpName, FWUKL1_NOT_READY_STATE);
    }
    return changeStatus;
  }
  //Put the board in a mode that will allow it to take physics data i.e. enable the appropriate in/outputs.
  if (FWUKL1_START_STATE == newState) {
    int changeStatus = fwUkl1_Start(ukl1Name);
    if (0 != changeStatus) {
      fwUkl1_ErrorMessage("Failed to change to start state.", FWUKL1_INFO_MESSAGE_LEVEL);
      //The state change was unsuccessful and we no longer know how many register were changed,
      //hence state is now not ready.
      _fwUkl1_DpSet(dpName, FWUKL1_NOT_READY_STATE);
    }
    return changeStatus;
  }
  //Stop all the in/outputs to prevent the board handling data, now in ready state.
  if (FWUKL1_STOP_STATE == newState) {
    int changeStatus = fwUkl1_Stop(ukl1Name);
    if (0 != changeStatus) {
      fwUkl1_ErrorMessage("Failed to change to stop state.", FWUKL1_INFO_MESSAGE_LEVEL);
      //The state change was unsuccessful and we no longer know how many register were changed,
      //hence state is now not ready.
      _fwUkl1_DpSet(dpName, FWUKL1_NOT_READY_STATE);
    }
    return changeStatus;
  }
  
  //If we get to here it means that an invalid state was requested, as we haven't returned anything yet.
  //Hence we must return the appropriate code to signal that.
  fwUkl1_WarningMessage("Unrecognised state \'" + newState + "\' attempted to be selected.", FWUKL1_INFO_MESSAGE_LEVEL);
  return 3;
}

/*!
 * Gets the state of the current UKL1.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \return unsigned Value of the state that the board is currently in. Can be compared to the FWUKL1_*_STATE defined by the library.
 */
unsigned fwUkl1_GetUkl1State(string ukl1Name) {
  //Retrieve the state from the datapoint.
  unsigned uState = 0;
  _fwUkl1_DpGet(FWUKL1_FSM_DPNAME_PREFIX + ukl1Name + ".status", uState);
  fwUkl1_StatusMessage("Returned state: " + uState, FWUKL1_VERBOSE_MESSAGE_LEVEL);
  return uState;
}

/*!
 * This performs the configuring action require to make the board ready for operation, 
 *
 * \return int Defines the state of the function after execution. The states are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int fwUkl1_Configure(string ukl1Name) {
  //Configure the FPGAs.
  int fpgaStatus = _fwUkl1_FpgaConfigureNormalMode(ukl1Name);
  fwUkl1_StatusMessage("FPGA config return status: " + fpgaStatus, FWUKL1_VERBOSE_MESSAGE_LEVEL);
  //Check for errors.
  if (0 != fpgaStatus) {
    fwUkl1_ErrorMessage("Failed to configure FPGAs.", FWUKL1_DEBUG_MESSAGE_LEVEL);
    return fpgaStatus;
  }

  //Configure the gigabit ethernet.
  int gbeStatus = _fwUkl1_GbeConfigureNormalMode(ukl1Name);
  fwUkl1_StatusMessage("GBE config return status: " + gbeStatus, FWUKL1_VERBOSE_MESSAGE_LEVEL);
  //Check for errors.
  if (0 != gbeStatus) {
    fwUkl1_ErrorMessage("Failed to configure GBE.", FWUKL1_DEBUG_MESSAGE_LEVEL);
    return gbeStatus;
  }

  //Configure the TTCrx.
  int ttcrxStatus = _fwUkl1_TtcrxConfigureNormalMode(ukl1Name);
  fwUkl1_StatusMessage("TTCrx config return status: " + ttcrxStatus, FWUKL1_VERBOSE_MESSAGE_LEVEL);
  if (0 != gbeStatus) {
    fwUkl1_ErrorMessage("Failed to configure TTCrx.", FWUKL1_DEBUG_MESSAGE_LEVEL);
  }
  //TTCrx status will define the final return status of this function,
  //as we would have returned already if the otheres were bad.
  return ttcrxStatus;
}

/*!
 * Sets the board into the start state. All inputs and outputs will be enabled.
 *
 * \return int Defines the state of the function after execution. The states are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *
 * This must be called only when the board is in a configured state i.e. ready or stopped. Behaviour is undefined otherwise.
 */
int fwUkl1_Start(string ukl1Name) {
  //Set FPGA states, also does the channel settings and enables the TFC decoding.
  int fpgaStatus = _fwUkl1_ConfigureAllFpgaEnableSettings(ukl1Name);
  //Check for errors.
  if (0 != fpgaStatus) {
    fwUkl1_ErrorMessage("Failed to start FPGAs.", FWUKL1_DEBUG_MESSAGE_LEVEL);
    return fpgaStatus;
  }
  //Set GBE ports states, also enable the GBE ports on the Egress FPGA side.
  int portStatus = _fwUkl1_ConfigureAllGbePortEnableSettings(ukl1Name);
  //GBE port status will define the final return status of this function.
  if (0 != fpgaStatus) {
    fwUkl1_ErrorMessage("Failed to start GBE ports.", FWUKL1_DEBUG_MESSAGE_LEVEL);
  }
  return portStatus;
}


/*!
 * Stops the board from taking data by disabling all front end FPGAs and input channels.
 *
 * \return int Defines the state of the function after execution. The states are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *
 * This must be called only when the board is in a configured state i.e. ready or stopped. Behaviour is undefined otherwise.
 */
int fwUkl1_Stop(string ukl1Name) {
  //Disable all the FPGAs and channels.
  int fpgaStatus = _fwUkl1_DisableAllFpgas(ukl1Name);
  //Check for errors.
  if (0 != fpgaStatus) {
    fwUkl1_ErrorMessage("Failed to stop FPGAs.", FWUKL1_DEBUG_MESSAGE_LEVEL);
    return fpgaStatus;
  }
  //Disable all the ports, also from the Egress FPGA side.
  int portStatus = _fwUkl1_DisableAllGbePorts(ukl1Name);
  //GBE port status will define the final return status of this function.
  if (0 != fpgaStatus) {
    fwUkl1_ErrorMessage("Failed to stop GBE ports.", FWUKL1_DEBUG_MESSAGE_LEVEL);
  }
  return portStatus;
}


//@}


// ==========
//  INTERNAL
// ==========

/** @defgroup SectionConfiguration High level configuration functions.
 *     These internal functions are used by the FSM to load the configuration settings to the L1 board.
 *  @{
 */

/*!
 * Loads all the settings for the Egress FPGA from the database.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \return int Defines the state of the function after execution. The states are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *
 */
int _fwUkl1_FpgaConfigureNormalMode(string ukl1Name) {
  //This lists all the registers that are to be written to for the BE FPGA.
  dyn_string registerList;
  registerList[1]  = "MacDestinationAddress";
  registerList[2]  = "MacSourceAddress";
  registerList[3]  = "IpDestinationAddress";
  registerList[4]  = "IpSourceAddress";
  registerList[5]  = "Protocol";
  registerList[6]  = "TypeOfService";
  registerList[7]  = "TimeToLive";
  registerList[8]  = "Type";
  registerList[9]  = "Version";
  registerList[10] = "PartitionId";
  registerList[11] = "ZeroSuppression";
  registerList[12] = "HpdPixelMode";
  registerList[13] = "Latency";
  registerList[14] = "MepPackingFactor";
  registerList[15] = "FixedMepDestinationAddress";
  registerList[16] = "DontWaitForMepDestBrdcst";
  registerList[17] = "Throttle";
  registerList[18] = "ThrottlePolarity";
  registerList[19] = "DataFragmentSize";
  //This will hold all the data retrieved from the datapoints.
  dyn_string sData;
  //Name of the data point to read from.
  const string dpBase = "Ukl1Defaults.";
  //Loop over every register name given to us and get the value for it.
  for (unsigned element = 1; element <= dynlen(registerList); ++element) {
    //For the moment we will hardcode the IP values, do this rather than getting from the datapoint.
    if ( "IpSourceAddress" == registerList[element] ) {
      //Make it unqiue for the two CERN boards only.
      if ( "pcukl101" == ukl1Name ) {
	sData[element] = "192.168.33.22";
      } else if ( "pcukl1102" == ukl1Name ) {
	sData[element] = "192.168.33.23";
      } else {
	//Get from the data point.
	//Retrieve the default from the database data point.
	int pvssErr = _fwUkl1_DpGet(dpBase + registerList[element], sData[element]);
	//Return if we saw a problem with either read from dp or write to hardware.
	if (0 > pvssErr) {
	  //Blame software before hardware, software is easier to debug.
	  return -1;
	}
      }
    } else {
      //Anything else we can get from the data point.
      //Retrieve the default from the database data point.
      int pvssErr = _fwUkl1_DpGet(dpBase + registerList[element], sData[element]);
      //Return if we saw a problem with either read from dp or write to hardware.
      if (0 > pvssErr) {
	//Blame software before hardware, software is easier to debug.
	return -1;
      }
    }
  }
  //Write these to the board.
  dyn_int callStatusList;
  int callStatus = fwUkl1_BeFpgaWrite(ukl1Name, registerList, sData, callStatusList, gbfwUkl1_verifyWrite);

  //Now write the per FE FPGA settings.
  dynClear(registerList);
  dynClear(sData);
  registerList[1]  = "Emulator";
  registerList[2]  = "EmulatorHpdPixelMode";
  registerList[3]  = "EmulatorEvent";
  //Retrieve the default from the database data point, it will be in broadcast format, for each register we want.
  int pvssErr = 0;
  for (unsigned element = 1; element <= dynlen(registerList); ++element) {
    pvssErr = _fwUkl1_DpGet(dpBase + "FeFpga." + registerList[element], sData[element]);
    if (0 > pvssErr) {
      //Blame software before hardware, software is easier to debug.
      fwUkl1_ErrorMessage("Failed to read from database for register: " + registerList[element] + ".",
			  FWUKL1_DEBUG_MESSAGE_LEVEL);
      return -1;
    }
  }
  //Write these to the board.
  dynClear(callStatusList);
  callStatus = fwUkl1_FeFpgaWrite(ukl1Name, FWUKL1_BROADCAST_FPGA, registerList, sData, callStatusList, gbfwUkl1_verifyWrite);

  //Done.
  return callStatus;
}

/*!
 * Configures the GBE card for running in the standard configuration mode.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \return int Defines the state of the function after execution. The states are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *
 */
int _fwUkl1_GbeConfigureNormalMode(string ukl1Name) {
  //This performs some configuration routines for the GBE, setting the GBE into its running mode.
  int callStatus = _fwUkl1_GbeConfigureHardcodedSettings(ukl1Name);
  //If error return.
  if (0 != callStatus) {
    return callStatus;
  }

  //Set up the settings for each of the ports.
  //Name of the data point to read from.
  const string dpBase = "Ukl1Defaults.GbeConfiguration.Port";
  //List of registers to set for each port.
  dyn_string registerList;
  registerList[1] = "TXFifoThreshold";
  registerList[2] = "TXFifoLowWtrmrk";
  registerList[3] = "TXFifoHighWtrmrk";
  //Holds the data to be written.
  dyn_string sData;
  sData[1] = fwCcpc_convertDecToHex(448);
  sData[2] = fwCcpc_convertDecToHex(208);
  sData[3] = fwCcpc_convertDecToHex(960);
  for (unsigned port = 0; port < FWUKL1_GBE_PORTS; ++port) {
    //Loop over every register name given to us and get the value for it.
    for (unsigned element = 1; element <= dynlen(registerList); ++element) {
      //Retrieve the default from the database data point.
      int pvssErr = _fwUkl1_DpGet(dpBase + port + "." + registerList[element], sData[element]);
      //Return if we saw a problem with either read from dp or write to hardware.
      if (0 > pvssErr) {
	//Blame software before hardware, software is easier to debug.
	return -1;
      }
    }
    //Configure Tx FIFO thresholds.
    dyn_int callStatusList;
    callStatus = fwUkl1_GbePortWrite(ukl1Name, port, registerList, sData, callStatusList, gbfwUkl1_verifyWrite);
    //Delaying for the shortest time possible.
    delay(0,1);
    //If error return.
    if (0 != callStatus) {
      return callStatus;
    }
  }

  //Just return, that was the last function call.
  return callStatus;
}

/*!
 * Configures all the settings on the GBE that are never changed by the User and all values are hardcoded into this function.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \return int Returns the execution status of the function. The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *
 * The values set here are defined by the LHCb technical note Quad Gigabit Ethernet plug-in card and are correct as of
 * issue 2.2 revision 0 of that document. They must be performed in a specific order.
 */
int _fwUkl1_GbeConfigureHardcodedSettings(string ukl1Name) {
  //Used to keep track of the error conditions. This needs to be upgraded.
  int writeStatus = 0;

  //This holds the list of registers that to be written
  dyn_string registerList;
  //and their corresponding data.
  dyn_string sData;
  //both MUST be cleared after each write.

  //Place MAC, SPI3 TX and RX, RX and TX FIFOs in reset.
  registerList[1] = "MACSoftReset";
  sData[1] = "0000000f";
  registerList[2] = "SPI3ConfigTrnmtGlobal";
  sData[2] = "000c000f";
  registerList[3] = "RxFifoPrtReset";
  sData[3] = "0000000f";
  registerList[4] = "TxFifoPrtReset";
  sData[4] = "0000000f";
  //Perform the write.
  dyn_int callStatusList;
  writeStatus = fwUkl1_GbeWrite(ukl1Name, registerList, sData, callStatusList, gbfwUkl1_verifyWrite);
  //Clear the arrays for next use.
  dynClear(registerList);
  dynClear(sData);
  dynClear(callStatusList);
  //Delaying for the shortest time possible.
  delay(0,1);

  //Disable all ports.
  registerList[1] = "PortEnable";
  sData[1] = "00000000";
  registerList[2] = "Clock";
  sData[2] = "00000000";
  //Perform the write.
  writeStatus = fwUkl1_GbeWrite(ukl1Name, registerList, sData, callStatusList, gbfwUkl1_verifyWrite);
  //Clear the arrays for next use.
  dynClear(registerList);
  dynClear(sData);
  dynClear(callStatusList);
  //Delaying for the shortest time possible.
  delay(0,1);

  //Select copper mode
  registerList[1] = "Mode";
  sData[1] = "0000000f";
  //Perform the write.
  writeStatus = fwUkl1_GbeWrite(ukl1Name, registerList, sData, callStatusList, gbfwUkl1_verifyWrite);
  //Clear the arrays for next use.
  dynClear(registerList);
  dynClear(sData);
  dynClear(callStatusList);
  //Delaying for the shortest time possible.
  delay(0,1);
  
  for (unsigned port = 0; port < FWUKL1_GBE_PORTS; ++port) {
    // Select copper mode for each port.
    registerList[1] = "MAC";
    sData[1] = "00000002";
    //Perform the write.
    writeStatus = fwUkl1_GbePortWrite(ukl1Name, port, registerList, sData, callStatusList, gbfwUkl1_verifyWrite);
    //Clear the arrays for next use.
    dynClear(registerList);
    dynClear(sData);
    dynClear(callStatusList);
    //Delaying for the shortest time possible.
    delay(0,1);

    //Set full duplex mode for each port.
    registerList[1] = "Duplex";
    sData[1] = "00000001";
    //Perform the write.
    writeStatus = fwUkl1_GbePortWrite(ukl1Name, port, registerList, sData, callStatusList, gbfwUkl1_verifyWrite);
    //Clear the arrays for next use.
    dynClear(registerList);
    dynClear(sData);
    dynClear(callStatusList);
    //Delaying for the shortest time possible.
    delay(0,1);
  }

  //Enable clocks for all active channels
  registerList[1] = "Clock";
  sData[1] = "0000000f";
  //Perform the write.
  writeStatus = fwUkl1_GbeWrite(ukl1Name, registerList, sData, callStatusList, gbfwUkl1_verifyWrite);
  //Clear the arrays for next use.
  dynClear(registerList);
  dynClear(sData);
  dynClear(callStatusList);
  //Delaying for the shortest time possible.
  delay(0,1);

  //Remove resets
  registerList[1] = "MACSoftReset";
  sData[1] = "00000000";
  registerList[2] = "SPI3ConfigTrnmtGlobal";
  sData[2] = "0000000f";
  registerList[3] = "RxFifoPrtReset";
  sData[3] = "00000000";
  registerList[4] = "TxFifoPrtReset";
  sData[4] = "00000000";
  //Perform the write.
  writeStatus = fwUkl1_GbeWrite(ukl1Name, registerList, sData, callStatusList, gbfwUkl1_verifyWrite);
  //Clear the arrays for next use.
  dynClear(registerList);
  dynClear(sData);
  dynClear(callStatusList);
  //Delaying for the shortest time possible.
  delay(0,1);

  // Enable automatic padding and CRC generation for all active channels
  registerList[1] = "Config";
  sData[1] = "0000114d";
  for (unsigned port = 0; port < FWUKL1_GBE_PORTS; ++port) {
    //Perform the write.
    writeStatus = fwUkl1_GbePortWrite(ukl1Name, port, registerList, sData, callStatusList, gbfwUkl1_verifyWrite);
    dynClear(callStatusList);
  }
  //Clear the arrays for next use.
  dynClear(registerList);
  dynClear(sData);
  //Delaying for the shortest time possible.
  delay(0,1);

  // Setup Rx interface: configure SPI3 width 32 bit, RVAL pause and active channel
  registerList[1] = "SPI3ConfigRcv";
  sData[1] = "00ffff80";
  //Perform the write.
  writeStatus = fwUkl1_GbeWrite(ukl1Name, registerList, sData, callStatusList, gbfwUkl1_verifyWrite);
  //Clear the arrays for next use.
  dynClear(registerList);
  dynClear(sData);
  dynClear(callStatusList);
  //Delaying for the shortest time possible.
  delay(0,1);

  // Setup Rx interface: configure RX CRC stripping and check
  registerList[1] = "RxFifoEnable";
  sData[1] = "000000f0";
  //Perform the write.
  writeStatus = fwUkl1_GbeWrite(ukl1Name, registerList, sData, callStatusList, gbfwUkl1_verifyWrite);
  //Clear the arrays for next use.
  dynClear(registerList);
  dynClear(sData);
  dynClear(callStatusList);
  //Delaying for the shortest time possible.
  delay(0,1);

  //Return the appropriate number.
  if (0 == writeStatus) {
    //Everything was successful.
    return 0;
  } else if (-1 == writeStatus) {
    //PVSS error.
    return -1;
  } else if (writeStatus > 0) {
    //CCPC server error.
    return 1;
  }

}

/*!
 * Configures the TTCrx card for running in the standard configuration mode.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \return int Defines the state of the function after execution. The states are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *
 * Changes only the value of the 
 */
int _fwUkl1_TtcrxConfigureNormalMode(string ukl1Name) {
  //Set the TTCrx register that we wish to write to.
  dyn_int callStatusList;
  int callStatus = fwUkl1_TtcrxWrite(ukl1Name, "Control", "a1", callStatusList, gbfwUkl1_verifyWrite);
  //callStatus now contains the appropriate error code.
  return callStatus;

}

/*!
 * This will configure the FPGAs and their channels enable settings
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int _fwUkl1_ConfigureAllFpgaEnableSettings(string ukl1Name) {
  //Holds the name of the registers.
  dyn_string registerList;
  //Need to deal with the TFC decoder and the FE FPGA setting. First deal with TFC decoder.
  registerList[1] = "TfcDecoder";
  //Will hold the data that is read for each FPGA and channel.
  dyn_string sData;

  //Base name for the datapoint.
  string dpBase = "Ukl1Defaults.";
  //Used to check the return status.
  int fwCcpcErr = 0;

  //Get the TFC decoder setting from the database, will be in broadcast format.
  int pvssErr = _fwUkl1_DpGet(dpBase + registerList[1], sData[1]);
  if (0 != pvssErr) {
    //Error getting datapoints, return a PVSS problem.
    return -1;
  }
  //Write them to the register.
  dyn_int callStatusList;
  fwCcpcErr = fwUkl1_BeFpgaWrite(ukl1Name, registerList, sData, callStatusList, gbfwUkl1_verifyWrite);
  //Our return status now depends on the return value of the hardware write.
  if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  }
  if (0 > fwCcpcErr) {
    //Otherwise the return values match mine.
    return fwCcpcErr;
}

  //Get the FE FPGA enable setting from the database, will be in broadcast format.
  registerList[1] = "Disable";
  pvssErr = _fwUkl1_DpGet(dpBase + "FeFpga." + registerList[1], sData[1]);
  if (0 != pvssErr) {
    //Error getting datapoints, return a PVSS problem.
    return -1;
  }
  //Write them to the register.
  dynClear(callStatusList);
  fwCcpcErr = fwUkl1_FeFpgaWrite(ukl1Name, FWUKL1_BROADCAST_FPGA, registerList, sData, callStatusList, gbfwUkl1_verifyWrite);
  //Our return status now depends on the return value of the hardware write.
  if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  }
  if (0 > fwCcpcErr) {
    //Otherwise the return values match mine.
    return fwCcpcErr;
  }

  //Loop over all the FPGAs and set the channel disable settings from the configuration database.
  for (unsigned fpga = 0; fpga < FWUKL1_FRONT_FPGAS; ++fpga) {
    //Also set the channel disable settings to save multiple loops over everything.
    _fwUkl1_ConfigureAllFeFpgaChannelEnableSettings(ukl1Name, fpga);
  }


  return 0;
}

/*!
 * Sets all the front end FPGAs and their channel states to disabled. It also disables the TFC decoding.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int _fwUkl1_DisableAllFpgas(string ukl1Name) {
  //Holds the name of the registers.
  dyn_string registerList;
  registerList[1] = "TfcDecoder";
  //Holds the data to be written.
  dyn_string sData;
  //Broadcast disable everything.
  sData[1] = "Disable";
  //Used to check the return status.
  int fwCcpcErr = 0;

  //Write them to the register.
  dyn_int callStatusList;
  fwCcpcErr = fwUkl1_BeFpgaWrite(ukl1Name, registerList, sData, callStatusList, gbfwUkl1_verifyWrite);
  //Our return status now depends on the return value of the hardware write.
  if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  }
  if (0 > fwCcpcErr) {
    //Otherwise the return values match mine.
    return fwCcpcErr;
  }

  //Holds the name of the registers.
  registerList[1] = "Disable";
  //Will hold the data that is read for each FPGA and channel.
  dynClear(sData);
  //Broadcast disable everything.
  sData[1] = "Disable,Disable,Disable,Disable";
  //Used to check the return status.
  fwCcpcErr = 0;

  //Write them to the register.
  dynClear(callStatusList);
  fwCcpcErr = fwUkl1_FeFpgaWrite(ukl1Name, FWUKL1_BROADCAST_FPGA, registerList, sData, callStatusList, gbfwUkl1_verifyWrite);
  //Our return status now depends on the return value of the hardware write.
  if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  }
  if (0 > fwCcpcErr) {
    //Otherwise the return values match mine.
    return fwCcpcErr;
  }

  //Loop over all the FPGAs and disable all the input channels.
  for (unsigned fpga = 0; fpga < FWUKL1_FRONT_FPGAS; ++fpga) {
    fwCcpcErr = _fwUkl1_DisableAllFeFpgaChannelEnableSettings(ukl1Name, fpga);
    //Our return status now depends on the return value of the hardware write.
    if (0 < fwCcpcErr) {
      //They only promise that the error will be greater than 1, whereas I promise it will be 1.
      return 1;
    }
    if (0 > fwCcpcErr) {
      //Otherwise the return values match mine.
      return fwCcpcErr;
    }
  }

  return 0;
}

/*!
 * This will configure the channel enable settings.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  feFpga Number of the FE FPGA whose channels are to be disabled.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int _fwUkl1_ConfigureAllFeFpgaChannelEnableSettings(string ukl1Name, unsigned feFpga) {
  //Holds the name of the registers.
  dyn_string registerList;
  registerList[1] = "Disable";
  //Will hold the data that is read for each FPGA and channel.
  dyn_string sData;

  //Base name for the datapoint.
  const string dpBase = "Ukl1Defaults.FeFpga.Channel.";
  //Used to check the return status.
  int fwCcpcErr = 0;

  //Get the disable setting from the database in broadcast format.
  int pvssErr = _fwUkl1_DpGet(dpBase + registerList[1], sData[1]);
  if (0 != pvssErr) {
    //Error getting datapoints, return a PVSS problem.
    return -1;
  }
  //Write them to the register.
  dyn_int callStatusList;
  int fwCcpcErr = fwUkl1_FeFpgaChannelWrite(ukl1Name, FWUKL1_BROADCAST_CHANNEL, feFpga, registerList, sData, callStatusList, gbfwUkl1_verifyWrite);
  //Our return status now depends on the return value of the hardware write.
  if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  }
  if (0 > fwCcpcErr) {
    //Otherwise the return values match mine.
    return fwCcpcErr;
  }

  return 0;
}

/*!
 * Sets all the front end FPGAs states to disabled.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  feFpga Number of the FE FPGA whose channels are to be disabled.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int _fwUkl1_DisableAllFeFpgaChannelEnableSettings(string ukl1Name, unsigned feFpga) {
  //Holds the name of the registers.
  dyn_string registerList;
  registerList[1] = "Disable";
  //Holds the data to be written to the channels. Disable all FE FPGA channels using a broadcast.
  dyn_string sData;
  sData[1] = "Disable,Disable,Disable,Disable,Disable,Disable,Disable,Disable,Disable";

  //Broadcast to all channels.
  dyn_int callStatusList;
  //Write them to the register.
  int fwCcpcErr = fwUkl1_FeFpgaChannelWrite(ukl1Name, FWUKL1_BROADCAST_CHANNEL, feFpga, registerList, sData, callStatusList, gbfwUkl1_verifyWrite);
  //Our return status now depends on the return value of the hardware write.
  if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  }
  if (0 > fwCcpcErr) {
    //Otherwise the return values match mine.
    return fwCcpcErr;
  }

  //If we made it to here then everything is good.
  return 0;
}

/*!
 * This will configure the port settings on the GBE.
 *
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int _fwUkl1_ConfigureAllGbePortEnableSettings(string ukl1Name) {
  int status = 0;
  //Holds all the settings for the GBE ports.
  unsigned uValue = 0;
  //Get the setting from the database. Stored in one datapoint.
  int pvssErr = _fwUkl1_DpGet("Ukl1Defaults.GbeConfiguration.PortEnable", uValue);
  if (0 != pvssErr) {
    //There was a problem getting one of the datapoints, so there is no use writting to the hardware.
    status = -1;
  }
  //Write the setting to the ports.
  dyn_int callStatusList;
  int fwCcpcErr = fwUkl1_GbeWrite(ukl1Name, "PortEnable", fwCcpc_convertDecToHex(uValue), callStatusList, gbfwUkl1_verifyWrite);
  //Our return status now depends on the return value of the hardware write.
  if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    status = 1;
  } else {
    //Otherwise the return values match mine.
    status = fwCcpcErr;
  }
  //Now we must enable the output ports on the Egress FPGA, don't bother getting this from a data point, it must be enabled to take data.
  dynClear(callStatusList);
  int fwUkl1Err = fwUkl1_BeFpgaWrite(ukl1Name, "PollGbePorts", "Enable", callStatusList, gbfwUkl1_verifyWrite);

  if ( -2 != fwUkl1Err ) {
    //There is only one case where the return value from the write does not match that stated for this function, and this isn't it.
    status = fwUkl1Err;
  } else {
    //This is not allowed by this functions return code, just return -1 it is the closest.
    status = -1;
  }

}

/*!
 * This will disable all the ports on the GBE.
 *
 * \return int Returns an error code to indicate the execution status of the function. The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int _fwUkl1_DisableAllGbePorts(string ukl1Name) {
  int status = 0;
  //Just write 0 to the register.
  dyn_int callStatusList;
  int fwCcpcErr = fwUkl1_GbeWrite(ukl1Name, "PortEnable", 0, callStatusList, gbfwUkl1_verifyWrite);
  //Our return status now depends on the return value of the hardware write.
  if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    status = 1;
  } else {
    //Otherwise the return values match mine.
    status = fwCcpcErr;
  }
  //Now we must disable the output ports on the Egress FPGA.
  dynClear(callStatusList);
  int fwUkl1Err = fwUkl1_BeFpgaWrite(ukl1Name, "PollGbePorts", "Disable", callStatusList, gbfwUkl1_verifyWrite);

  if ( -2 != fwUkl1Err ) {
    //There is only one case where the return value from the write does not match that stated for this function, and this isn't it.
    status = fwUkl1Err;
  } else {
    //This is not allowed by this functions return code, just return -1 it is the closest.
    status = -1;
  }

  return status;
}

//@}


// ==========================
//  UKL1 NAME LIST FUNCTIONS
// ==========================

/** @defgroup SectionCcpc UKL1 name list functions.
 *  These functions allow the manipulation of the global list that the library uses
 *  to store the names of the UKL1 boards accessed by the system.
 *  @{
 */

/*!
 * Updates the list of the name of the UKL1 boards that are used. Overwrites all the old entries.
 *
 * \param  ukl1List A dyn_string containing the names of the UKL1s as they appear on the DIM server.
 * \return int The number of UKL1 names that were added, -1 in the event of errors.
 */
int fwUkl1_SetNamesList(dyn_string ukl1List) {
  //First clear the list.
  int status = fwUkl1_ClearNamesList();
  if (0 != status) {
    fwUkl1_ErrorMessage("Failed to clear the names list, while setting new list.", FWUKL1_DEBUG_MESSAGE_LEVEL);
    return status;
  }
	dpSet(FWUKL1_L1_LIST_DPE, ukl1List);
  //gasfwUkl1_Ukl1NamesList = ukl1List;
  //Now return the number that we just added.
  return dynlen(ukl1List);
}

/*!
 * Prints all the names in the UKL1 names list.
 *
 * \param  messageLevel The level of importance of printing the UKL1 board name to the screen. Default: FWUKL1_VERBOSE_MESSAGE_LEVEL.
 * \return dyn_string List of all the names of the UKL1 board that has been retrieved.
 */
dyn_string fwUkl1_GetNamesList() {
  for (unsigned name = 1; name <= fwUkl1_GetNamesListSize(); ++name) {
    fwUkl1_GetName(name, messageLevel);
  }
  dyn_string namesList;
  dpGet(FWUKL1_L1_LIST_DPE, namesList);
  return namesList;
}

/*!
 * This will add a UKL1 name to the gasfwUkl1_Ukl1NamesList. All the old entries are kept and
 * only unique entries are allowed. Adding a name that already exists will have no effect.
 *
 * \param  ukl1Name A string containing the name of the UKL1 as it appears in the DNS server.
 * \return int The total number of UKL1s now present in the list, -1 in the event of errors.
 */
int fwUkl1_SetName(string ukl1Name) {
  //Append the entries to the list and return the new total number using the control function.
  dyn_string currentList;
  int dpStatus = dpGet(FWUKL1_L1_LIST_DPE, currentList);
  if (0 == dpStatus) {
    //Successfully accessed the datapoint.
    //Check the name we have been given is not already in the list.
    if ( "" == dynPatternMatch(ukl1Name, currentList) ) {
      //Now append the new message to this.
      dynAppend(currentList, ukl1Name);
      //and write it back to the datapoint.
      dpStatus = dpSet(FWUKL1_L1_LIST_DPE, currentList);
      return dpStatus;
    } else {
      //Name already there just return.
      return dynlen(currentList);
    }
  } else {
    DebugTN("Failed to add the UKL1 board: " + ukl1Name);
    return -1;
  }

}

/*!
 * Prints the name of the requested UKL1 name list element and also returns that value.
 *
 * \param  elementNumber An unsigned giving the element number that is to be retrieved.
 * \return string Name of the UKL1 board that has been retrieved.
 */
string fwUkl1_GetName(unsigned elementNumber) {
  dyn_string currentList;
  int dpStatus = dpGet(FWUKL1_L1_LIST_DPE, currentList);
  if (0 == dpStatus) {
    return currentList[elementNumber];
  } else {
    return "Failed";
  }
}

/*!
 * Returns number of UKL1 board names that are stored in the list.
 *
 * \return unsigned Number of UKL1 boards that are stored in the list, -1 in the event of errors.
 */
int fwUkl1_GetNamesListSize() {
  dyn_string currentList;// = gasfwUkl1_Ukl1NamesList[elementNumber];
  int dpStatus = dpGet(FWUKL1_L1_LIST_DPE, currentList);
  if (0 == dpStatus) {
    return dynlen(currentList);
  } else {
    return -1;
  }
}

/*!
 * Removes all the UKL1s from the list.
 *
 * \return Returns 0 in the event of success, -1 otherwise.
 */
int fwUkl1_ClearNamesList() {
  //Write a 1 element dyn_string to the datapoint. Prevents problems accessing the zeroth element.
  dyn_string clearedString = makeDynString("");
  if ( 0 == dpSet(FWUKL1_L1_LIST_DPE, clearedString) ) {
    return 0;
  } else {
    return 1;
  }
}

/*!
 * Removes the specified UKL1 board name from the list.
 *
 * \return int Return codes are as follows:
 *    \li  0 Successfully removed the name.
 *    \li -1 Failed to remove the name.
 *    \li -2 Name not present to be removed.
 */
int fwUkl1_DeleteName(string ukl1Name) {
  dyn_string currentList;// = gasfwUkl1_Ukl1NamesList[elementNumber];
  dpGet(FWUKL1_L1_LIST_DPE, currentList);
  //First find the instance of this name.
  int index = dynContains(currentList, ukl1Name);
  if (0 == index) {
    fwUkl1_WarningMessage("Name not present in list to remove.",
			  FWUKL1_DEBUG_MESSAGE_LEVEL);
    return -2;
  }
  //Will return 0 if successful, -1 if fails as required by our documentation.
  dynRemove(currentList, index);
  return dpSet(FWUKL1_L1_LIST_DPE, currentList);
}

//@}


// ======================================
//  ERROR FUNCTIONS
// ======================================

/** @defgroup SectionError Error functions.
 *  These functions are used for logging errors that occur when the various read/write
 *  operations are performed.
 *  @{
 */

/*!
 * Sends a message to the FSM message box and optionally the RichL1Log datapoint,
 * also appends 'STATUS:' to the beginning of all messages.
 *
 * \param  message Debug message that is to be sent to the datapoint.
 * \param  level This is the level of message that is to be logged. It will only be logged if it is
 *           greater or equal to the global message level. Should be a value specified by the
 *           FWUKL1_*_MESSAGE_LEVEL constants.
 *
 * It can also send a message to the Ukl1MessageLog datapoint, this depends on the library constant
 * FWUKL1_SEND_MESSAGE_TO_DATAPOINT. This will be set to false in versions released with the FSM.
 */
void fwUkl1_StatusMessage(string message, unsigned level) {
  //First check the level of the warning against the global level.
  if (gufwUkl1_StatusMessageLevel >= level) {
    //Append the appropriate text to indicate this is a warning.
    message = "STATUS: " + message;
    //Send the message to the FSM message box.
    DebugTN(message);
    if (bfwUkl1_SendMessageToDataPoint) {
      //First we must get the existing debug messages.
      dyn_string dbgMsg;
      int dpStatus = dpGet("RichL1Log.Status", dbgMsg);
      if (0 == dpStatus) {
	//Successfully accessed the datapoint.
	//Now append the new message to this.
	dynAppend(dbgMsg, message);
	//and write it back to the datapoint.
	dpStatus = dpSet("RichL1Log.Status", dbgMsg);
      } else {
	DebugTN("Failed to access the status datapoint:");
	DebugTN("RichL1Log.Status");
	DebugTN("Status messages diverted to PVSS log, the following message was recieved:");
	DebugTN(message);
      }
    }
  }
}

/*!
 * Sends a message to the FSM message box and optionally the RichL1Log datapoint,
 * also appends 'WARNING:' to the beginning of all messages.
 *
 * \param  message Warning message that is to be logged.
 * \param  level This is the level of message that is to be logged. It will only be logged if it is
 *           greater or equal to the global message level. Should be a value specified by the
 *           FWUKL1_*_MESSAGE_LEVEL constants.
 *
 * It can also send a message to the Ukl1MessageLog datapoint, this depends on the library constant
 * FWUKL1_SEND_MESSAGE_TO_DATAPOINT. This will be set to false in versions released with the FSM.
 */
void fwUkl1_WarningMessage(string message, unsigned level) {
  //First check the level of the warning against the global level.
  if (gufwUkl1_WarningMessageLevel >= level) {
    //Append the appropriate text to indicate this is a warning.
    message = "WARNING: " + message;
    //Send the message to the FSM message box.
    DebugTN(message);
    if (bfwUkl1_SendMessageToDataPoint) {
      //First we must get the existing debug messages.
      dyn_string dbgMsg;
      int dpStatus = dpGet("RichL1Log.Warning", dbgMsg);
      if (0 == dpStatus) {
	//Successfully accessed the datapoint.
	//Now append the new message to this.
	dynAppend(dbgMsg, message);
	//and write it back to the datapoint.
	dpStatus = dpSet("RichL1Log.Warning", dbgMsg);
      } else {
	DebugTN("Failed to access the status datapoint:");
	DebugTN("RichL1Log.Warning");
	DebugTN("Warning messages diverted to PVSS log, the following message was recieved:");
	DebugTN(message);
      }
    }
  }
}

/*!
 * Sends a message to the FSM message box and optionally the RichL1Log datapoint,
 * also appends 'ERROR:' to the beginning of all messages.
 *
 * \param  message Warning message that is to be logged.
 * \param  level This is the level of message that is to be logged. It will only be logged if it is
 *           greater or equal to the global message level. Should be a value specified by the
 *           FWUKL1_*_MESSAGE_LEVEL constants.
 *
 * It can also send a message to the Ukl1MessageLog datapoint, this depends on the library constant
 * FWUKL1_SEND_MESSAGE_TO_DATAPOINT. This will be set to false in versions released with the FSM.
 */
void fwUkl1_ErrorMessage(string message, unsigned level) {
  if (gufwUkl1_ErrorMessageLevel >= level) {
    //Append the appropriate text to indicate this is a warning.
    message = "ERROR: " + message;
    //Send the message to the FSM message box.
    DebugTN(message);
    if (bfwUkl1_SendMessageToDataPoint) {
      //First we must get the existing debug messages.
      dyn_string dbgMsg;
      int dpStatus = dpGet("RichL1Log.Error", dbgMsg);
      if (0 == dpStatus) {
	//Successfully accessed the datapoint.
	//Now append the new message to this.
	dynAppend(dbgMsg, message);
	//and write it back to the datapoint.
	dpStatus = dpSet("RichL1Log.Error", dbgMsg);
      } else {
	DebugTN("Failed to access the status datapoint:");
	DebugTN("RichL1Log.Error");
	DebugTN("Error messages diverted to PVSS log, the following message was recieved:");
	DebugTN(message);
      }
    }
  }
}

/**
 * Clears all the messages from the message log.
 *
 * \return int Returns an error code that indicates the execution status of the function. They are as follows:
 *   \li  0 Function completed successfully.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int fwUkl1_ClearAllLogMessages() {
  //Check the return status for errors.
  //Write to the appropriate log if we failed to clear it (it will default to the PVSS log if there is a datapoint issue).
  //Return the appropriate state.
  if ( 0 != fwUkl1_ClearStatusMessages() ) {
    fwUkl1_StatusMessage("Failed to clear status log, due to a PVSS error.", FWUKL1_INFO_MESSAGE_LEVEL);
    return -1;
  }
  if ( 0 != fwUkl1_ClearWarningMessages() ) {
    fwUkl1_WarningMessage("Failed to clear warning log, due to a PVSS error.", FWUKL1_INFO_MESSAGE_LEVEL);
    return -1;
  }
  if ( 0 == fwUkl1_ClearCriticalMessages() ) {
    return 0;
  } else {
    fwUkl1_ErrorMessage("Failed to clear error log, due to a PVSS error.", FWUKL1_INFO_MESSAGE_LEVEL);
    return -1;
  }
}

/**
 * Clears all the status messages from the message log.
 *
 * \return int Returns an error code that indicates the execution status of the function. They are as follows:
 *   \li  0 Function completed successfully.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int fwUkl1_ClearStatusMessages() {
  //Write a 1 element dyn_string to the datapoint. Prevents problems accessing the zeroth element.
  dyn_string clearedString = makeDynString("Cleared status messages.");
  if ( 0 == dpSet("RichL1Log.Status", clearedString) ) {
    return 0;
  } else {
    return 1;
  }
}

/**
 * Clears all the warning messages from the message log.
 *
 * \return int Returns an error code that indicates the execution status of the function. They are as follows:
 *   \li  0 Function completed successfully.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int fwUkl1_ClearWarningMessages() {
  //Write a 1 element dyn_string to the datapoint. Prevents problems accessing the zeroth element.
  dyn_string clearedString = makeDynString("Cleared warning messages.");
  if ( 0 == dpSet("RichL1Log.Warning", clearedString) ) {
    return 0;
  } else {
    return 1;
  }
}

/**
 * Clears all the error messages from the message log.
 *
 * \return int Returns an error code that indicates the execution status of the function. They are as follows:
 *   \li  0 Function completed successfully.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int fwUkl1_ClearErrorMessages() {
  //Write a 1 element dyn_string to the datapoint. Prevents problems accessing the zeroth element.
  dyn_string clearedString = makeDynString("Cleared error messages.");
  if ( 0 == dpSet("RichL1Log.Error", clearedString) ) {
    return 0;
  } else {
    return 1;
  }
}

/**
 * Logs an error that occurred with while using the framework library for writing/reading from the
 * CCPC.
 *
 * \param  messageLevel Level that the message is defined as.
 * \param  message Error message that is to be logged.
 */
void _fwUkl1_LogFwCcpcError(string message) {
  //Append a note specifying it occurred within the framework and then log it as a critical error.
  string newMessage = "fwCcpc.ctl lib reported error: " + message;
  //Just call the appropriate logging function return its state.
  fwUkl1_ErrorMessage(newMessage, FWUKL1_DEBUG_MESSAGE_LEVEL);
}

/**
 * Logs an error that occurred with while retrieving/setting a datapoint.
 *
 * \param  messageLevel Level that the message is defined as.
 * \param  message Error message that is to be logged.
 */
void _fwUkl1_LogDpError(string message) {
  //Append a note specifying it occurred within the framework and then log it as a critical error.
  string newMessage = "datapoint error: " + message;
  //Just call the appropriate logging function return its state.
  fwUkl1_ErrorMessage(newMessage, FWUKL1_DEBUG_MESSAGE_LEVEL);
}

//@}


// ======================================
//  SUPPORT FUNCTIONS
// ======================================

/** @defgroup SectionSupport Support funtions
 *  List of constants that can be used by the user.
 *  @{
 */

/*!
 * Reads from the registers on the L1 board to ensure that the data just written has been written successfully.
 *
 * \param  registerNames Names of the registers that were written to.
 * \param  writtenData The data that was written to the registers that the read back values should be compared against.
 * \param  masks Any masks that should be applied to the read data, such that only specific bits in a register are compared
 *           against the write bits.
 * \param  callingFunctionName Name of the function that called the write verification. Allows error messages to track
 *           the failing point more easily.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Verification failed due to a problem with reading from the CCPC server.
 *   \li -1 Verification failed due to a PVSS problem.
 *   \li -2 Verification failed as there was a mismatch in size of the written and read data.
 *   \li -3 Verification failed due to a mismatch in on of the data elements.
 */
int _fwUkl1_VerifyWrite(dyn_string registerNames, dyn_dyn_char writtenData, dyn_dyn_char masks, string callingFunctionName) {
  //Holds the return state of the function. Indicates either success or failure.
  int status = 0;
  //This holds the read data.
  dyn_dyn_char ddcCheckData;
  //This holds the status from the register reads.
  dyn_int callStatusList;
  //Number of elements of data to loop through.
  const unsigned numRegs = dynlen(registerNames);
  //Read back from all registers.
  if ( fwCcpc_read(registerNames, ddcCheckData, callStatusList) ) {
    //Check the data.
    for (unsigned element = 1; element <= numRegs; ++element) {
      const int checkLen = dynlen(ddcCheckData[element]);
      const int dataLen  = dynlen(writtenData[element]);
      if (checkLen  != dataLen) {
	fwUkl1_WarningMessage(callingFunctionName + ": " + registerNames[element] + ": Check data and data lengths differ when verifying write.",
			      FWUKL1_INFO_MESSAGE_LEVEL);
	fwUkl1_WarningMessage(callingFunctionName + ": " + registerNames[element] + ": " + checkLen + ", data: " + dataLen,
			      FWUKL1_INFO_MESSAGE_LEVEL);
	status = -2;
      } else {
	//Only check the data if they are the same size.
	//This will provide us with our loop index. The two arrays have been found to be the same size, so just set to either.
	const int dataLen = dataLen;
	for (unsigned index = 1; index <= dataLen; ++index) {
	  //Apply the masks that are given to the read data.
	  ddcCheckData[element][index] &= masks[element][index];
	  //Now check the data against the read values.
	  if ( ddcCheckData[element][index] != writtenData[element][index] ) {
	    fwUkl1_WarningMessage(registerNames[element] + " BE FGPA write: CheckData (" + (unsigned)ddcCheckData[element][index] +
				  ") and written data (" + (unsigned)writtenData[element][index] + ") differ in element " + index + ".",
				  FWUKL1_INFO_MESSAGE_LEVEL);
	    status = -3;
	  }
	}
      }
    }
  } else {
    //Check where the errors occured.
    const unsigned statusLen = dynlen(callStatusList);
    if ( 0 != statusLen ) {
      for (unsigned element = 1; element <= statusLen; ++element) {
	if ( 0 != callStatusList[element] ) {
	  //Construct the error message.
	  string errMsg = "While verifying \'" + registerNames[element] + "\' " + callingFunctionName + " encountered ";
	  if (callStatusList[element] > 0) {
	    errMsg += "a problem writing to the CCPC server.";
	    status = 1;
	  } else if (callStatusList[element] < 0) {
	    errMsg += "a problem with PVSS.";
	    status = -1;
	  } else {
	    errMsg += "an unknown error.";
	    status = -1;
	  }
	  //Now log the error.
	  _fwUkl1_LogFwCcpcError(errMsg);
	}
      }
    } else {
      fwUkl1_ErrorMessage("While verifying write a fatal error was encountered. Verfiying won't have put L1 into an inconsistent state, but it is likely that other error messages will have said that the board is likely in an inconsistent state.",
			  FWUKL1_DEBUG_MESSAGE_LEVEL);
    }
  }
  //This will get set to a none zero value if an error path is followed, otherwise will still be 0 the initialised value.
  return status;
}


/*!
 * In order to write to specific registers on the FPGAs we must first convert the address
 * to include the address of the FPGAs from the glue card perspective.
 *
 * \param  addr Address of the register to be written to.
 * \return unsigned Full address of the FPGA register.
 */
unsigned fwUkl1_FpgaizeAddress(unsigned addr) {
  // Based on code written by Steve Wotton, Cambridge.
  // The address sent to lb_{write|read}_hword should be above 0x20000
  // to be outside the GBE & gluecard ranges
  // The FPGA is sensitive only to bits 2-13 of the address
  //  (=> the FPGA is not sensitive to the 0x2???? that indicates
  //      that an address is outside the GBE/gluecard ranges)
  // So whenever we want to address the FPGA, set the 13th bit:
  //   the FPGA will only respond to addresses for which the 13th bit has been set
  //
  // So: << 2    translates a (32-bit) int address into (8-bit) byte address
  //     0x20000 gets us outside the address ranges of the GBE and gluecard
  //               (but will be ignored by the FPGA)
  //     0x2000  should stop the gluecard from reacting to the operation
  //               (but will be ignored by the FPGA)
  //
  // => (int) register addresses 0x0 to 0x7ff should be usable by the FPGA
  //

  return (0x08000000 + ( 0x800 | ( addr<<2 ) ) );
}

/*!
 * This will take a UKL1 board name as it appears in the DIM server and return the HW type datapoint
 * name that is used to identify it in the PVSS system.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \return string Name of the UKL1 board HW type datapoint.
 *
 * It is of the form FWUKL1_HW_DPNAME_PREFIX/ukl1Name.
 */
string fwUkl1_ConvertDimServerNameToHwDpName(string ukl1Name) {
  //Add a prefix from that is defined by the library for creating the datapoints.
  return FWUKL1_HW_DPNAME_PREFIX + ukl1Name;
}

/*!
 * This will take a HW datapoint UKL1 board name and return the name as it appears in the DIM server.
 *
 * \param  hwDpName Name of the datapoint of type HwTypeCCPCUKL1 whose DIM server name is to be retrieved.
 * \return string Name of the UKL1 board as it would appear in the DIM server.
 */
string fwUkl1_ConvertHwDpNameToDimServerName(string hwDpName) {
  //Create a substr starting from after the prefix.
  const int startPos = strlen( FWUKL1_HW_DPNAME_PREFIX );
  //substr cuts from startPos to end as no length for substring is specified.
  return substr(hwDpName, startPos);
}

/*!
 * This will take a UKL1 board name as it appears in the DIM server and return the FSM type datapoint
 * name that is used to identify it in the PVSS system.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \return string Name of the UKL1 board FSM type datapoint.
 *
 * It is of the form FWUKL1_FSM_DPNAME_PREFIX/ukl1Name.
 */
string fwUkl1_ConvertDimServerNameToFsmDpName(string ukl1Name) {
  //Add a prefix from that is defined by the library for creating the datapoints.
  return FWUKL1_FSM_DPNAME_PREFIX + ukl1Name;
}

/*!
 * This will take a FSM datapoint UKL1 board name and return the name as it appears in the DIM server.
 *
 * \param  fsmDpName Name of the datapoint of type FwLHCbDAQDevice whose DIM server name is to be retrieved.
 * \return string Name of the UKL1 board as it would appear in the DIM server.
 */
string fwUkl1_ConvertFsmDpNameToDimServerName(string fsmDpName) {
  //Check to see if the system name is present and if so remove it.
  //Colons are can only be used with system name.
  //Either the return will contain two element, the first being the system name or it will contain
  //only one which is the data point name.
  dyn_string tmpFsmDpName = strsplit(fsmDpName, ":");
  //Create a substr starting from after the prefix.
  const int startPos = strlen( FWUKL1_FSM_DPNAME_PREFIX );
  //substr cuts from startPos to end as no length for substring is specified.
  //Always want to look at the last element of the split string, should be either 1 if system name is not
  //present or 2 if it is.
  return substr(tmpFsmDpName[dynlen(tmpFsmDpName)], startPos);
}

/*!
 * Takes a string and ensures that it contains enough characters for the number of bytes specified
 * by numBytes. Adds `0' to the beginnig if it is too short and removes characters from the beginning
 * if it is too long. Must be in hex.
 *
 * \param  sData The string that contains the data to be checked.
 * \param  numBytes The number of bytes that the returned character string should return.
 * \return string The formatted string.
 *
 * This is for use with strings that represent a hex number that should be a set number of bits long.
 * It assumes that the lowest order bit is the 2nd, 4th, 6th etc character and hence its behaviour is
 * to move the last character always towards the final character in the padded string. This is achieved
 * by padding or removing at the beginning of the string. e.g. if 4-bytes are required (32-bit word) and
 * the input string was 123456789abcd -> 6789abcd or again for 4-bytes: 1234 -> 00001234.
 * However if only 1-byte was specified the same inputs would give cd and 34 respectively.
 * This is quite effective for guessing the structure of misformed data for when writing to
 * the L1 board using the fwCcpc library as they use a big endian format for storing data.
 */
string _fwUkl1_PadString(string sData, unsigned numBytes) {
  //First determine the length.
  const int datalen = strlen(sData);
  //Create a temporary string to store the original string.
  const string tmpOrig = sData;
  //Determine the number of characters that must be in the final string,
  //each character represents half a byte or nibble.
  const int numChar = numBytes*2;
  //Perform the checking and padding/trimming.
  if ( numChar > datalen ) {
    //The data string is too short so increase its length by padding at the beginning, highest order bytes.
    //A temporary string to hold the amount of padding.
    string tmpPad;
    for (int character = 1; character <= (numChar-datalen); ++character)
      tmpPad += "0";
    //Now insert the padding into the data string.
    sData = tmpPad + tmpOrig;
  }
  else if ( numChar < datalen ) {
    //Remove the initial characters of the string until it reaches the appropriate length, highest order bytes.
    sData = substr(sData, (datalen-numChar), datalen);
  }
  else {
    //Do nothing, it is the right length.
  }
  //Return the resulting string.
  return sData;
}

/*!
 * This takes the data that is read by from the L1 board FPGAs (LBUS), which is read as a series of 32-bit words,
 * and converts it to a series of 16-bit words that is then decoded to return the register value.
 *
 * \param dcData The dyn_char array that contains the bytes read back from the hardware.
 * \return dyn_char The dyn_char that now has its bytes in an order that can be decoded by the fwCcpc library
 *           to return a string representing the hex number.
 *
 * The function performs two tasks. First it considers the byte array as a group of 32-bit numbers and removes
 * the upper two bytes to reduce them to a 16-bit number. The resulting array must then be word swapped, however
 * the words must maintain their internal byte ordering.
 */
dyn_char _fwUkl1_ConvertFrom32To16BitWords(dyn_char dcData) {
  //Convert the read data from a series of 32-bit words to a series of 16-bit words.
  //The loop must run over the number of 32-bit words in the read data.
  const unsigned total32Words = dynlen(dcData)/4;
  for (unsigned word = 0; word < total32Words; ++word) {
    //We need to remove the first two elements in that word.
    //This will follow the sequence of odd numbers.
    //After we delete the first element the second element becomes the first, so we always delete the same one.
    dynRemove(dcData, 2*word+1);
    dynRemove(dcData, 2*word+1);
  }
  //In the event that we have read back more than one word then we will need to swap the ordering of the 16-bit
  //words contained within the array.
  const unsigned total16Words = dynlen(dcData)/2;
  if (1 < total16Words) {
    //Copy the data into a temporary array.
    const dyn_char tmpData = dcData;
    //Loop over all the 16-bit words of the array and put them back into the original in the reverse order.
    const unsigned totalElements = dynlen(dcData);
    for (unsigned word = 0; word < totalElements; word+=2) {
      //Need to keep the byte ordering within the word, but word swap the array. 16-bit words.
      dcData[word+1] = tmpData[totalElements-word-1];
      dcData[word+2] = tmpData[totalElements-word];
    }
  }
  return dcData;
}

/*!
 * This takes a array of string where each element of the array contains the relevant settings for either
 * the FE FPGAs, the channels on a FE FPGA or the GBE ports and converts these to an array of strings.
 * The returned result is an array of array of strings that contain all the relevant settings.
 *
 * \param  origData The dyn_string that contains the data to be parsed. Each element should contain a
 *           comma separated list of settings.
 * \return dyn_dyn_string Each element contains a dyn_string, where each element contains the setting for
 *           an individual unit.
 *
 * The input strings must contain a comma separated list of the unit settings. The function checks that
 * there are either four or nine settings found, if there are no then the data format is considered to be
 * invalid and for that specific element an empty string is return. It will continue to parse the rest of
 * the data if an error is found. The ordering of the data is maintained from the input to output array.
 */
dyn_dyn_string fwUkl1_ParseBroadcastData(dyn_string origData) {
  //First determine the number strings we will need to parse.
  const unsigned totalLength = dynlen(origData);
  //Holds the result that we will be returning.
  dyn_dyn_string parsedData;
  //Loop over the array and deal with each element.
  for (unsigned element = 1; element <= totalLength; ++element) {
    //Holds the result of the string parsing.
    //Split the string up using comma's as a deliminator.
    parsedData[element] = strsplit(origData[element], ',');
    //Check we have enough elements for either all the FE FPGAs, GBE ports (both 4) or FE FPGA
    //channels (9).
    const unsigned foundParts = dynlen(parsedData[element]);
    if ( (1 != foundParts) && (4 != foundParts) && (9 != foundParts) ) {
      fwUkl1_ErrorMessage("Failed to parse the broadcast string for element number " + element + ". Found " + foundParts + " sections in the string. Must have 1, 4 or 9. We were given " + origData[element] + " and require it to be comma delimited.",
			  FWUKL1_DEBUG_MESSAGE_LEVEL);
      //Parsed data wasn't good so return an empty array instead.
      parsedData[element] = makeDynChar();
    }
  }
  //Finally return the parsed strings.
  return parsedData;
}

/*!
 * Takes a colon delimited hex MAC address, as a string, and removes the colons
 * to return a continuous hex number. It also performs checks on the data format.
 *
 * \param  macAddress The string that contains the MAC address to be parsed.
 * \return string MAC address without the colons. If the MAC address contains no colons
 *           it will be returned as it was. If the MAC address does not look like a MAC
 *           address `failed' will be returned.
 *
 * Expects the MAC address to be of the form 00:0e:0c:b0:2d:19 or 000e0cb02d19
 * i.e. 12 characters long.
 * Examples:
 *   00:0e:0c:b0:2d:19 -> 000e0cb02d19, 000e0cb02d19 -> 000e0cb02d19, 00:0e:0c:b0 -> failed.
 */
string fwUkl1_ParseMacAddress(string macAddress) {
  //Check that it contains the appropriate number of characters, either with or without colons.
  if ( (12 != strlen(macAddress)) && (17 != strlen(macAddress)) ) {
    fwUkl1_ErrorMessage("MAC address is not of the expected length: " + strlen(macAddress) + ". Should be 12 or 17 characters long.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return "failed";
  }
  //Try to split the string using a colon as a delimiter.
  dyn_string tmpMac = strsplit(macAddress, ":");
  //Check it has found the appropriate number of parts.
  const int lenMac = dynlen(tmpMac);
  if ( 1 == lenMac ) {
    //String contained no colons, it is the right lenght so it can be returned as it was given.
    return macAddress;
  }
  else if ( 6 == lenMac ) {
    //Contains 6 parts, put them all into one string.
    string output;
    int status = sprintf(output, "%02s%02s%02s%02s%02s%02s", tmpMac[1], tmpMac[2], tmpMac[3], tmpMac[4], tmpMac[5], tmpMac[6]);
    //Check for errors.
    if ( -1 != status ) {
      fwUkl1_StatusMessage("Parsed MAC address: " + output, FWUKL1_VERBOSE_MESSAGE_LEVEL);
      return output;
    } else {
      fwUkl1_ErrorMessage("Remove colons from MAC address: " + output,
			  FWUKL1_DEBUG_MESSAGE_LEVEL);
      return "failed";
    }
  }
  else {
    //No properly formed.
    fwUkl1_ErrorMessage("Didn't find the appropriate number of colon delimited sections: " + lenMac + ". Should be 1 or 6.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return "failed";
  }

}

/*!
 * Takes an array of bytes and returns them as a colon delimited MAC address.
 *
 * \param  macAddress The array containing the bytes to be parsed.
 * \return string Expects a 6 element array containing 3 elements with 16-bit words of data in each.
 *           Returns `failed' if the input data is misformed.
 *
 * Example return structure: 00:0e:0c:b0:2d:19. It is expected that the array will have already undergone
 * the first stage of processing after being read by the from the L1 board and have been converted from
 * a series of 32-bit words to a series of 16-bit words.
 */
string fwUkl1_ByteToMacAddress(dyn_char macAddress) {
  //This will be repeatedly useful.
  const int sizeOfInput = dynlen(macAddress);
  //Check that it contains the appropriate number of elements.
  if ( 6 != sizeOfInput ) {
    fwUkl1_ErrorMessage("Input MAC address is not of the expected size: " + sizeOfInput + ". Should be 6 elements big.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return "failed";
  }

  //Now we need to put the data into the appropriate format.
  string output;
  //This should be in a little endian order once it reaches here
  output += fwCcpc_convertByteToHex(macAddress[1]);
  output += ":";
  output += fwCcpc_convertByteToHex(macAddress[2]);
  output += ":";
  output += fwCcpc_convertByteToHex(macAddress[3]);
  output += ":";
  output += fwCcpc_convertByteToHex(macAddress[4]);
  output += ":";
  output += fwCcpc_convertByteToHex(macAddress[5]);
  output += ":";
  output += fwCcpc_convertByteToHex(macAddress[6]);

  return output;

}

/*!
 * Takes a standard formed IP(4) address, as a string, and convert it to a continuous hex
 * string. Also performs syntacs checking on the given string.
 *
 * \param  ipAddress The string that contains the IP address to be parsed. Should be of the form:
 *           XXX.XXX.XXX.XXX, where X represents a decimal number, e.g. 192.168.192.192.
 * \return string If the IP address does not look like a IP address `failed' will be returned.
 *
 * Examples:
 *   192.168.192.192 -> c0a8c0c0, 192.168. -> failed.
 */
string fwUkl1_ParseIpAddress(string ipAddress) {
  //Check that it is less than the maximum length of an IP(4) address.
  if ( 16 < strlen(ipAddress) ) {
    fwUkl1_ErrorMessage("IP address is not of the expected length: " + strlen(ipAddress) + ". Should be no more than 16 characters long.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return "failed";
  }
  //Try to split the string using a . as a delimiter.
  dyn_string tmpIp = strsplit(ipAddress, ".");
  //Check it has found the appropriate number of parts.
  const int lenIp = dynlen(tmpIp);
  if ( 4 == lenIp ) {
    //Contains 4 parts, put them all into one string.
    string output;
    for (unsigned element = 1; element <= dynlen(tmpIp); ++element) {
      //Take a section of the IP and convert it to a hex string.
      string tmpSection = fwCcpc_convertDecToHex(tmpIp[element]);
      //If it does not contain 2 characters pad the string.
      if ( 2 != strlen(tmpSection) )
	tmpSection = "0" + tmpSection;
      output += tmpSection;
    }
    return output;
  }
  else {
    //Not properly formed.
    fwUkl1_ErrorMessage("Didn't find the appropriate number of full stop delimited sections: " + lenIp + ". Should be 4.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return "failed";
  }

}

/*!
 * Takes an array of bytes, and returns them as a full stop delimited IP address.
 *
 * \param  ipAddress The array containing the bytes to be parsed.
 * \return string Expects a 4 element array. Returns `failed' if the input data is missformed.
 *
 * Example return structure: 192.168.192.192. It is expected that the array will have already undergone
 * the first stage of processing after being read by the from the L1 board and have been converted from
 * a series of 32-bit words to a series of 16-bit words.
 */
string fwUkl1_ByteToIpAddress(dyn_char ipAddress) {
  //This will be repeatedly useful.
  const int sizeOfInput = dynlen(ipAddress);
  //Check that it contains the appropriate number of elements.
  if ( 4 != sizeOfInput ) {
    fwUkl1_ErrorMessage("Input IP address is not of the expected size: " + sizeOfInput + ". Should be 4 elements big.",
			FWUKL1_DEBUG_MESSAGE_LEVEL);
    return "failed";
  }

  //Now we need to put the data into the appropriate format.
  string output;
  output += fwCcpc_convertByteToDec(ipAddress[1]);
  output += ".";
  output += fwCcpc_convertByteToDec(ipAddress[2]);
  output += ".";
  output += fwCcpc_convertByteToDec(ipAddress[3]);
  output += ".";
  output += fwCcpc_convertByteToDec(ipAddress[4]);

  return output;
}


//@}
