// ====================================
//  NAME    : fwUkl1ExceptionHandling.ctl
//
//  VERSION : 1
//
//  REVISION: 8
//
//  DATE    : 2007/06/07
//
//  RELEASE : 2009/10/12
//
//  PLATFORM: PVSS II
//
//  AUTHOR  : Gareth Rogers
//
//  EMAIL   : rogers@hep.phy.cam.ac.uk
// ====================================

/*! \mainpage UKL1 Exception Library Documentation
 *
 * This library is designed to extend the capablilty of the framework exception handling to make the submission and
 * display more relevant to the UKL1 project. In order to provide an analogy with the framework exception handling
 * functions all functions that are intended for use by the User to create an exception chain are prefixed fwUkl1Exception_
 * and those functions that are for the display of exception chains are prefixed fwUkl1ExcpetionHandling_. Finally
 * the exception log is provided as an extension over the framework capabilities and functions for utilising the
 * exception log are prefixed fwUkl1ExceptionLog_. Functions used for internal library use have a leading underscore '_'.
 *
 * The UKL1 exceptions utilise the same structure as that defined by the framework, for obvious compatibility reasons
 * and use the fwException_raise() function as a base for constructing the exception chains. As a consequence of this
 * compatibility the library provides a fwUkl1Exception_raise function that takes the exact same arguments as the framework
 * equivalent. It provided as a wrapper for fwException_raise() which ensures that the structure of the exception message
 * is in a format that is required by the display functions (`Function(): Message', defined in the framework guidelines)
 * and will coerce it into this form if necessary. This is simply to reduce the number of cases that the display functions
 * cannot deal with. It is not require that this function be used and fwException_raise() can be used to create exception
 * chains that this libraries display functions can handle. fwUkl1Exception_raise returns the size of the exception array
 * after the exception has been appended, this is useful for keeping track of the size of the exception array as function
 * progress.
 *
 * A second function, fwUkl1Exception_checkAndRaise, is also provided. This performs the additional function that it takes
 * the size of the exception array before some function was called and if and only if that function was called then the
 * exception information is appended (via a call to fwUkl1Exception_raise()). The parameter that indicates the size of
 * the exception information is passed by reference and will indicate the size of the exception array once the call to this
 * function is finished. The function returns true to indicate if more exception information was added and thus if an error
 * occured when the function generating the error was called. Both fwUkl1Exception_raise and _checkAndRaise will if given
 * an empty exception type attempt to determine the exception type of the daughter exception and then take that for the
 * parent. This allows the calling function to decide how serious the error is depending on the function it called.
 *
 * The library also provides a replacement for fwExceptionHandling_display() called fwUkl1ExceptionHandling_display().
 * This function is not however a wrapper for fwExceptionHandling_display() and instead is provided as a replacement.
 * It displays the exception chain in a tree structure using the fwUkl1ExceptionTree_ functions to format it.
 * This function is intended for immediate display of a single exception chain in an easy to read manor. The chains
 * are lost once the panel is closed.
 *
 * The function fwUkl1ExceptionLog_submit() is provided as a means of saving exception chains to a log (a PVSS data point)
 * that can then be displayed how the User desires. The exception saved will have an additional exception adding the
 * logical node to group the exception with and the time and date that the exception was submit. Functions are also
 * provided to manage the log, for example creation and deletion.
 *
 * A group of functions prefixed fwUkl1ExceptionTree_ are provided to manipulate the exception chains into a tree
 * structure in the PVSS tree widget. They are used by the fwUkl1ExceptionHandlingDisplay panel and can be used
 * by any User panel to display the exception log. The package provides a fwUkl1ExceptionLogViewer panel that uses
 * functions to do just that.
 *
 * An exception chains are grouped together under logical nodes in the tree. Distinct systems submitting to the same
 * log can group themselves under these logical nodes to collect there exception chains. The chains are converted into
 * trees by starting with the last submitted exception and add it to the tree under its function name. Any functions
 * it called will have been added afterwards. It subsequent exceptions are added to the tree below the last added node.
 * If a function is already in the tree then they are placed under the same node and subsequent functions are placed
 * under this new node. This simply mimics the logical flow of PVSS code with a given function making a function
 * call, which in turns calls further functions. Once this is done the original function will possible call another
 * function with its own chain. It is important to have error checking at all stages in a function to ensure that
 * any exception chain is built up correctly. Of course the hope is that no exception information is generated
 * and the tree will be empty.
 *
 * It should also be noted that the exception log converts the dyn_string exception chain into a string and the log stores
 * each exception chain as an element in a dyn_string. The conversion is done using fwGeneral_stringToDynString() and
 * fwGeneral_dynStringToString() with `|' separators and white space left in will reconstruct the exception chain in a
 * form fully compatible with fwUkl1ExceptionHandling_display() and fwExceptionHandling_display().
 */

// =========================
//  CONSTANTs & GLOBAL VARs
// =========================

/** @defgroup SectionConstants Constants and global variables.
 *   List of constants and global variables that are used by the library. Functions are provided
 *   to set/get all global variables as part of the library interface.
 *  @{
 */

/*! Defines the name of the data point type that holds the exception log. */
const string FWUKL1_EXCEPTION_LOG_DPT = "FwUkl1ExceptionLog";

/*! Defines a default name for the log that is to be created an used. Can be overriden. */
const string FWUKL1_EXCEPTION_LOG_DEFAULT = "Ukl1Exceptions";
/*! Defines the name of the data point element that the log is stored in. */
const string FWUKL1_EXCEPTION_LOG_DPE = "exceptionLog";

/*! Defines the full path and name of the panel that is to be opened by fwUkl1ExceptionHandling_display(). */
const string FWUKL1_EXCEPTION_HANDLING_DISPLAY_PANEL = "objects/fwUkl1ExceptionHandling/fwUkl1ExceptionHandlingDisplay.pnl";

//@}


// =====================
//  EXCEPTION FUNCTIONS
// =====================

/** @defgroup SectionExecption Exception functions.
 *   Contains the functions that are used to create an exception chain.
 *  @{
 */

/*!
 * A wrapper for fwException_raise(), fwUkl1Exception_raise() takes the same arguments, but will check
 * that the exceptionText is of the form `Function(): Message', as defined by the framework, and will
 * prepend `Unknown function(): ' to any exceptionText that does not contain `(): '.
 *
 * \param  exceptionInfo An array of strings that the exception information should be
 *           appended to.
 * \param  exceptionType WARNING, ERROR and FATAL are permitted for this field. It is upto the user
 *           of the library to decide which is the most appropriate to use. It is possible to pass an
 *           empty string here. In this case the exception type of the last submitted exception (in the
 *           given exceptionInfo) will be used. If a previous exception type cannot be found it will default
 *           to WARNING and an exception will be submitted to indicate the error. In a complex function chain
 *           this could lead to confusing results as it is the last exception submitted that defines the 
 *           exception type and this could override earlier (potentially more serious) exception types.
 * \param  exceptionText A string to describe the exceptional condition that occured. Must be
 *           of the form `Function(): Message' where:
 *   \li Function - the name of the function/button/thing that was executing when the
 *         when the exceptional condition arose.
 *   \li Message - description of the exceptional condition that occured, a message to the function caller.
 * \param  exceptionCode A number that indicates the exception that occured. This library reserves the error
 *           code value "0" for internal use.
 * \return int The size of the exceptionInfo array after the exceptionInfo has been added.
 *
 * \b Example 1
 * \code
 *   (...)
 *   //Check some condition.
 *   if ( trueIfErrorCondition ) {
 *     //The current size of the exceptionInfo array is returned.
 *     exInfoSize = fwUkl1Exception_raise(exceptionInfo, "WARNING", "thisFunction(): A error condition was detected.", "1");
 *   }//if(trueIfErrorCondition)
 *   else
 *     //Normal path of execution.
 *   (...)
 * \endcode
 *
 * \b Example 2
 * \code
 *   (...)
 *   //Call function 1 which returns exceptionInfo.
 *   functionOne(exceptionInfo);
 *   //Determine if there is an error condition.
 *   if ( trueIfErrorCondition ) {
 *     //The exception type is dependent on that which functionOne sets, thus leave it empty.
 *     exInfoSize = fwUkl1Exception_raise(exceptionInfo, "", "thisFunction(): A error condition was detected.", "1");
 *   }//if(trueIfErrorCondition)
 *   else
 *     //Normal path of execution.
 *   (...)
 * \endcode
 *
 * In example 2 the function fwUkl1Exception_checkAndRaise could be called with "" for the exception type and it would
 * determine if there was an error condition based on a change in size of exceptionInfo compared to before the call to
 * functionOne. This significantly reduces the amount of code that needs to be written.
 */
int fwUkl1Exception_raise(dyn_string &exceptionInfo, string exceptionType, string exceptionText, const string &exceptionCode) {
  //Search the exceptionText for the colon space marker.
  //Add it if it is not present.
  if (-1 == strpos(exceptionText, ": "))
    exceptionText = "Unknown function(): " + exceptionText;
	
  //If the exceptionType is an empty string, then check to see if there is a previous exception type and use that instead.
  if ("" == exceptionType) {
    const int exlen = dynlen(exceptionInfo);
    //There should be at least 1 exception entry already present.
    if (2 < exlen)
      exceptionType = exceptionInfo[exlen-2];
    else {
      exceptionType = "WARNING";
      fwException_raise(exceptionInfo, "WARNING", "fwUkl1Exception_raise(): No exception type given and no previous exceptions to take the type from. Setting exception type to WARNING and raising the given exception.", "");
    }//if(0<exlen)
  }//if(""==exceptionType)

  //Now raise the exception.
  fwException_raise(exceptionInfo, exceptionType, exceptionText, exceptionCode);
  
  //Done.
  return dynlen(exceptionInfo);
}//fwUkl1Exception_raise()

/*!
 * A wrapper for fwUkl1Exception_raise(), fwUkl1Exception_checkAndRaise() takes the same arguments, but will check
 * the given exInfoSize and compare it to the current size of the exceptionInfo array.
 *
 * \param  exceptionInfo An array of strings that the exception information should be
 *           appended to.
 * \param  exceptionType WARNING, ERROR and FATAL are permitted for this field. It is upto the user
 *           of the library to decide which is the most appropriate to use. It can be left as an empty
 *           string which means the exception type from the previous exception will be used. See the
 *           fwUkl1Exception_raise function for more details.
 * \param  exceptionText A string to describe the exceptional condition that occured. Must be
 *           of the form `Function(): Message' where:
 *   \li Function - the name of the function/button/thing that was executing when the
 *         when the exceptional condition arose.
 *   \li Message - description of the exceptional condition that occured, a message to the function caller.
 * \param  exceptionCode A number that indicates the exception that occured. This library reserves the error
 *           code value "0" for internal use.
 * \param  exInfoSize A typical use for fwUkl1Exception_raise() is for it to be called after a call to `some
 *           function' generated an error message detected by a change in size of the exceptionInfo array.
 *           If the size of the exceptionInfo array before the call to `some function' is given to fwUkl1Exception_raise()
 *           it will check the current size of exceptionInfo and only append the given error message if there
 *           has been a new error message added. A value of -2 will prevent this check occuring and force the
 *           error message to be added anyway. Note that exInfoSize will NOT be updated with the new size of
 *           the array, that is given in the return value of this function. Note PVSS uses -1 to indicate an error
 *           condition when calculating the size of arrays.
 * \return bool TRUE if exception information has been added to the array, FALSE if not.
 *
 * If the exception array is larger than the exInfoSize given then it will append the given exception information to the
 * exception string (using fwUkl1Exception_raise()), otherwise the exception array is left untouched. exInfoSize will correspond
 * to the size of the exceptionInfo array after a call to this function (it will either be unchanged or match the new size).
 * Further the function returns a boolean to indicate if the exception information has been added.
 *
 * \b Example
 * \code
 *   (...)
 *   //Assume we are in the code body of a function with the prototype `void thisFunction(dyn_string& exceptionInfo)',
 *   //hence exceptionInfo has already been defined.
 *   //Check the size of exceptionInfo so we can see exception information is added by a function call.
 *   int exInfoSize = dynlen(exceptionInfo);
 *   //Call a function.
 *   someFunction(exceptionInfo);
 *   if ( !fwUkl1Exception_checkAndRaise(exceptionInfo, "WARNING", "thisFunction(): someFunction did not execute appropriate.", "1", exInfoSize) ) {
 *     //... no error occured continue with normal function execution path.
 *   }//if exception not raised.
 *   else {
 *     //Execute any error code.
 *   }//else exception raised.
 *
 *   (...)
 * \endcode
 */
bool fwUkl1Exception_checkAndRaise(dyn_string &exceptionInfo, const string &exceptionType, const string &exceptionText, const string &exceptionCode, int &exInfoSize) {
  //Indicate if an exception was raised.
  bool raised = FALSE;
  //First check to see if we wish to raise an exception, remembering -2 means we must.
  if ((-2 == exInfoSize) || (exInfoSize < dynlen(exceptionInfo))) {
    //Raise the exception by simply calling the UKL1 raise function.
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, exceptionType, exceptionText, exceptionCode);
    raised = TRUE;
  }//if raise an exception.
  //Done.
  return raised;
}//fwUkl1Exception_checkAndRaise()

//@}


// ==============================
//  EXCEPTION HANDLING FUNCTIONS
// ==============================

/** @defgroup SectionExceptionHandling Exception handling functions.
 *   Replacement for the fwExceptionHandling_display function.
 *  @{
 */

/*!
 * Opens the panel defined by FWUKL1_EXCEPTION_HANDLING_DISPLAY_PANEL displaying the exception
 * chain given by exceptionInfo. If the dyn_string is empty no panel is displayed.
 *
 * \param  exceptionInfo Exception chain that is to be displayed. It will be cleared by a call to this function.
 * \return int Size of the exceptionInfo array. Should be zero if the function was successful.
 *
 * The form of exceptionInfo is checked to ensure that it is of an appropriate form to be displayed in a tree
 * with fwUkl1ExceptionHandlingDisplay.pnl, if it cannot be displayed with this panel then it is displayed
 * via fwExceptionHandling_display(). In both cases a call to fwUkl1ExceptionHandling_display() will clear the
 * contents of exceptionInfo, as defined in the framework guidelines.
 *
 * \b Example
 * \code
 *   (...)
 *   //Create a dyn_string to hold the exception information.
 *   dyn_string exceptionInfo;
 *   //Call a function, that could generate an error.
 *   thisFunction(exceptionInfo);
 *   //If an error has been generated it will be displayed otherwise nothing will occur.
 *   fwUkl1ExceptionHandling_display(FWUKL1_EXCEPTION_HANDLING_DISPLAY_PANEL, exceptionInfo);
 *   //Note that exception info is now definately empty, for example a call to fwUkl1ExceptionLog_submit() would do nothing.
 *   (...)
 *  \endcode
 */
int fwUkl1ExceptionHandling_display(dyn_string &exceptionInfo) {
  //Determine if we have a valid exception info and return if it is empty.
  const int exceptSize = dynlen(exceptionInfo);
  if (0 == exceptSize)
    return 0;
  
  if ((-1 != exceptSize) && (0 == exceptSize%3)) {
    //Add a final exception that is used to create the root node for this exception tree.
    //The final string is a unique identifier in the tree widget. It will only ever display
    //one entry so it doesn't matter what this number is. Ensure that the time is stamped with this
    //number.
    fwUkl1Exception_raise(exceptionInfo, "", ": ", "1_" + (string)getCurrentTime());
    //PVSS doesn't seem to like arrays being passed as dollar params.
    //Convert to from dyn_string to string.
    //Used to use the fwGeneral_dynStringToString function here but it is very
    //slow once you have a more than several thousand elements. The PVSS native
    //function doesn't encounter this problem.
    //The GUI doesn't like such large exceptions though!
    //Really should limit the size of the exception sent.
    const string sExceptionInfo = dynStringToString(exceptionInfo, "|");
    
    //Add a random number as it allows multiple exception window to be open at the same time.
    //There should never be enough open via this function to worry about it being the same.
    //Suspect that PVSS wouldn't care anyway.
    string panelName = "ExceptionInfo" + rand();
    dyn_string panelParams;
    //The only parameter it takes is the exception info.
    panelParams[1] = "$sExceptionInfo:" + sExceptionInfo;
    ChildPanelOnCentral(FWUKL1_EXCEPTION_HANDLING_DISPLAY_PANEL, panelName, panelParams);
    dynClear(exceptionInfo);
  }//if((-1!=exceptSize)&&(0==exceptSize%3))
  else {
    //If we couldn't determine the size of the exception it is probably not a valid array. Warn.
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1ExceptionHandling_display(): Unrecognised formatting of the exception info. Using framework exception display mechanism.", "");
    //Note this is not fw"Ukl1"Exception_display, thus we don't have an infinite loop.
    fwExceptionHandling_display(exceptionInfo);
  }//else(!formatError)
  
  //Done.
  return dynlen(exceptionInfo);
}//fwUkl1ExceptionHandling_display()

//@}


// =========================
//  EXCEPTION LOG FUNCTIONS
// =========================

/** @defgroup SectionExecptionLog Exception log manipulation functions.
 *   Through this group of functions the exception log can be created, deleted, cleared and submitted to.
 *   The only prerequesit is the data point type is already installed, however it does provide the
 *   functionality to delete the data point type.
 *  @{
 */

/*!
 * Submits an exception chain to the exception log. Additional formatting is added to be used with fwUkl1ExceptionTree_ functions.
 *
 * \param  exceptionInfo Exception chain that is to be displayed. It will be cleared by a call to this function.
 * \param  baseNode This is an optional parameter. It can be used to group exceptions under specific node in the
 *           exception tree and could be used, for example, to have all errors relating to a specific UKL1 board
 *           always occur beneath that node in the tree. Defaults to an empty string, in this case a unique identifier
 *           is generated for the base node.
 * \param  logName Name of the log that is to be submitted to, if it is being used on a distributed system it should
 *           contain the system. This is defaulted to FWUKL1_EXCEPTION_LOG_DEFAULT which is usually present in the system.
 *           If submitting to a remote system then the log name must contain the system name. If it does not it will
 *           be submitted locally and the exception log likely created on the fly!
 * \return int Size of the exceptionInfo array. Should be zero if the function was successful.
 *
 * If this function fails to complete an exception will be raised to note why it failed and then
 * fwUkl1ExceptionHandling_display() will used to display the exception chain. In all cases the exceptionInfo dyn_string
 * will be cleared by a call to this function. This function is synchronized to prevent multiple accesses to the
 * exception log DP.
 *
 * \b Example
 * \code
 *   (...)
 *   //Create a dyn_string to hold the exception information.
 *   dyn_string exceptionInfo;
 *   //Call a function, that could generate an error.
 *   thisFunction(exceptionInfo);
 *   //If an error has been generated it will be submitted to the log, otherwise nothing will occur.
 *   //All future exceptions that are submit()ted with "myNode" will be placed under the "myNode" entry in the exception tree.
 *   //We are submitting to a remote system so will have to specify the log name.
 *   const string logName = "R2DAQL1:" + FWUKL1_EXCEPTION_LOG_DEFAULT;
 *   fwUkl1ExceptionLog_submit(exceptionInfo, "myNode", logName);
 *   //Note that exception info is now definately empty, for example a call to fwUkl1ExceptionHandling_display() would do nothing.
 *   (...)
 *  \endcode
 */
synchronized int fwUkl1ExceptionLog_submit(dyn_string &exceptionInfo, string baseNode="", string logName = "Ukl1Exceptions") {
  //Don't try and submit something that does not exist.
  int exInfoSize = dynlen(exceptionInfo);
  if ( 0 >= exInfoSize )
    return 0;
  
  //Now check that the exception log DP exists. fwUkl1ExceptionLog_create will just return if the log already exists,
  //so there will be no exception info.
  //If there is an error it must have tried to created it and failed.
  fwUkl1ExceptionLog_create(logName, exceptionInfo);
  if (!fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1ExceptionLog_submit(): Checked to see if the exception log \'" + logName + "\' existed. It did not and it could not be created.", "", exInfoSize)) {
    //We now definately have the data point to submit the information to, can only be here if there is exception information to submit.
    //Get the existing exceptionInfo, such that we can add more information to it.
    dyn_string currentExInfo;
    int dpStatus = dpGet(logName + "." + FWUKL1_EXCEPTION_LOG_DPE, currentExInfo);
    if (0 == dpStatus) {
      //Successfully accessed the datapoint.
      //Now we have to generate a unique ID for the new exception to be added.
      //This will be used to store the unique ID given to this exception chain.
      string id = fwUkl1ExceptionTree_generateExceptionCode(currentExInfo);
      //If ID is 0 the generating a unique ID for the exception code failed.
      if ("0" != id) {
        //Add an exception to the end of the exceptionInfo which stores the ID number and who generated the error.
        //if ( "" == baseNode )
          //baseNode = id;
        //The message submits `: ' as otherwise it would be tagged as an unknown function.
        exInfoSize = fwUkl1Exception_raise(exceptionInfo, "", baseNode + ": ", id + "_" + (string)getCurrentTime());
        //PVSS doesn't seem to like arrays being passed as dollar params.
        //Convert to from dyn_string to string.
        //Used to use the fwGeneral_dynStringToString function here but it is very
        //slow once you have a more than several thousand elements. The PVSS native
        //function doesn't encounter this problem.
        //The GUI doesn't like such large exceptions though!
        //Really should limit the size of the exception sent.
        const string sExceptionInfo = dynStringToString(exceptionInfo, "|");
        //Append the stringized exception information to the existing exception information.
        dynAppend(currentExInfo, sExceptionInfo);
        //and write it back to the datapoint. Clear exceptionInfo if we are successful, indicate if we failed.
        if (0 == dpSet(logName + "." + FWUKL1_EXCEPTION_LOG_DPE, currentExInfo)) {
           dynClear(exceptionInfo);
           exInfoSize = 0;
         }//if(0==dpSet())
        else
          exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1ExceptionLog_submit(): Failed to write the exception information to the exception log.", "");
      }//if("0"!=id)
      else
        exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1ExceptionLog_submit(): Failed to generate a unique identifier for the exception chain. Exception information not added to the log.", "");
    }//if(0==dpStatus)
    else
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1ExceptionLog_submit(): Failed to retrieve current exception log. Exception information not added to log.", "");
  }//Exception log did not exist and could not be created.
  
  //Just incase we did not submit the exceptions.
  exInfoSize = fwUkl1ExceptionHandling_display(exceptionInfo);
  //Done.
  return exInfoSize;
}//fwUkl1ExceptionLog_submit()

/*!
 * Removes all the exceptions in the given exception log.
 *
 * \param  logName Name of the log that is to be cleared, if it is being used on a distributed system it should
 *           contain the system.
 * \param  exceptionInfo Contains information about exceptions that may have occured during the function execution.
 * \return void.
 */
void fwUkl1ExceptionLog_clear(const string &logName, dyn_string &exceptionInfo) {
  //Check to see if the exception log exists.
  if (dpExists(logName)) {
    //There is only any point trying to clear the log if it actually exists.
    //Write an empty array to the datapoint.
    if (0 != dpSet(logName + "." + FWUKL1_EXCEPTION_LOG_DPE, makeDynString()))
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1ExceptionLog_clear(): Failed to clear exception log.", "");
  }//if(dpExists(log))
  else
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1ExceptionLog_clear(): The exception log, " + logName + ", did not exist. No exception information to clear.", "");
  //Done.
  return;
}//fwUkl1ExceptionLog_clear()

/*!
 * Used to create an exception log for use with the library functions.
 *
 * \param  logName Name of the log that is to be created. If used on a distributed system then the system name
 *           should preceed the log name and they must be separated by a semi-colon e.g. R2DAQL1:ExceptionLog.
 * \param  exceptionInfo Error information. No change in size if no exception information is generated.
 * \return void.
 *
 * If the log already exists it will return with no errors and make no changes to the existing log.
 */
void fwUkl1ExceptionLog_create(const string &logName, dyn_string &exceptionInfo) {
  int exInfoSize = dynlen(exceptionInfo);
  //Check we are given something sensible.
  if ("" == logName) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1ExceptionLog_create(): An empty log name has been given cannot create or even check for existance of log.", "");
    return;
  }//if(""==logName)
  
  //Check that it doesn't exist.
  if (!dpExists(logName)) {
    //If it doesn't exist then we must create it.
    string logNameNoSys = logName;
    string sysName = "";
    if (patternMatch("*:*", logName)) {
      logNameNoSys = strsplit(logName, ":")[2];
      sysName      = strsplit(logName, ":")[1];
    }//if sys name present
    //Quote PVSS help: "dpCreate() returns 0, in the event of a failure returns -1. 
    //                  The function possibly returns 0 also in the event of a failure.
    //                  In this case errors can only be retrieved with getLastError()!"
    //Nice. Use getLastError() to determine if the DP creation was successful.
    //Initialise the log name without the system name present to be the given log name.
    //The we can check to see if there is a system name present. If so separate out the log
    //and the system name. We cannot create a DP with the system name in the name.
    //It must be given separately.
    dpCreate(logNameNoSys, FWUKL1_EXCEPTION_LOG_DPT, getSystemId(sysName + ":") );
    //Doubt it is thread safe...
    dyn_errClass error = getLastError();
    if (0 != dynlen(error))
      exInfoSize = fwUkl1Exception_raise(exceptionInfo, "FATAL", "fwUkl1ExceptionLog_create(): Exception log data point \'" + logName + "\'cannot be created.", "");
  }//if(!dpExists())
  //Done.
  return;
}//fwUkl1ExceptionLog_create()

/*!
 * Deletes the given exception log.
 *
 * \param  logName Name of the log that is to be deleted, if it is being used on a distributed system it should
 *           contain the system.
 * \param  exceptionInfo Contains information about any exceptions that may occur during the function call.
 * \return void.
 */
void fwUkl1ExceptionLog_delete(const string &logName, dyn_string &exceptionInfo) {
  //Check that it doesn't exist.
  if (dpExists(logName)) {
    //Now delete the log.
    dpDelete(logName);
    //Doubt it is thread safe...
    dyn_errClass error = getLastError();
    if (0 != dynlen(error)) {
      //Couldn't create the data point.
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1ExceptionLog_delete(): Failed to delete the log  '" + logName + "'.", "");
    }//if dpCreate() failed
  }//if(dpExists())
  //No else as if it doesn't exist then we are done.
  //Done.
  return;
}//fwUkl1ExceptionLog_delete()

/*!
 * Removes the exception log data point type from the system.
 *
 * \param  exceptionInfo Error information. No change in size if no exception information is generated.
 * \return void.
 *
 * The data point type that is removed is identifed by FWUKL1_EXCEPTION_LOG_DPT. All data points of this
 * type myst be removed before it can be deleted, and this will be done by this function.
 */
void fwUkl1ExceptionLog_deleteDPT(dyn_string &exceptionInfo) {
  //Keeps track of the exceptionInfo.
  int exInfoSize = dynlen(exceptionInfo);
  //First get a list of DPs that exist for the DPT.
  dyn_string logs = dpNames("*", FWUKL1_EXCEPTION_LOG_DPT);
  for (int i = 1; i <= dynlen(logs); ++i) {
    fwUkl1ExceptionLog_delete(logs[i], exceptionInfo);
    fwUkl1Exception_checkAndRaise(exceptionInfo, "ERROR", "fwUkl1ExceptionLog_deleteDPT(): Failed to delete the exception log '" + logs[i] + "'. This may prevent the exception log DPT from being deleted.", "2", exInfoSize);
  }//for i
  //Now we should have cleared the exception log, we can try and delete the data point type.
  int rc = dpTypeDelete(FWUKL1_EXCEPTION_LOG_DPT);
  if (0 != rc)
    exInfoSize = fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1ExceptionLog_deleteDPT(): Failed to delete the DPT. dpTypeDelete returned the error code " + rc + ".", "1");
  //Done.
  return;
}//fwUkl1ExceptionLog_deleteDpt()

//@}

// ==================================
//  EXCEPTION TREE DISPLAY FUNCTIONS
// ==================================

/** @defgroup SectionExecptionTreeDisplay Functions for the exception display in a tree widget.
 *   These functions are provided to display the exception information in a PVSS tree widget.
 *   They require the last exception added with a call to fwUkl1Exception_raise similar to this
 *   'fwUkl1Exception_raise(exceptionInfo, "", baseNode + ": ", id + "_" + time;'
 *   where:
 *     baseNode -- This is the node to add the exceptions under in the tree. It can already exist
 *       and is used to provide a logical group for multiple exception chains. It can be an empty
 *       string to be placed under the tree's root node.
 *     id -- This is a unique ID (provided by fwUkl1ExceptionTree_generateExceptionCode()).
 *       It can be anything if there will only be a single entry in the tree.
 *     time -- This is the time that the exception was generated. Typically the PVSS function
 *       getCurrentTime can be called. Note time is a PVSS reserved word.
 *  @{
 */

/*!
 * Takes an exception chain in the form of a `|' delimited string and converts this into a tree structure
 * displayed on a PVSS tree shape.
 *
 * \param  exceptionTree The tree widget that the exception tree is to be displayed on.
 * \param  sExceptionInfo An array of strings that contains an exception tree built up with the UKL1 library.
 * \return void.
 *
 * In order for the exception info to be displayed on the tree correctly the exception its final entry should be
 * of the form:
 *  \li exceptionText -- 'base node' + ": ".
 *  \li exceptionCode -- 'id' + "_" + 'time';
 * where base node is the node to enter the exception text into the tree; id is a unique ID for the tree entry;
 * and time is the time that the exception was submitted.
 *
 * In the event that display cannot parse the exception information it is handled via fwExceptionHandling_display()
 * if it is believed to be still in a form it can handle, otherwise it is sent to the PVSS log. This will not display
 * the last exception in the tree as this is used to create the base node for the tree. It should be ensured that
 * this has been created appropriately.
 */
void fwUkl1ExceptionTree_display(shape exceptionTree, string sExceptionInfo) {
  //This will hold the exception info once it have been converted back to a dyn_string.
  dyn_string exceptionInfo;
  //Search the string and replace all entries for `: ' with a `|'. This will ensure that the `Function(): Message'
  //will be split into two different elements of the array, allowing them to be more easily be displayed.
  const int numReplacements = strreplace(sExceptionInfo, ": ", "|");
  //Now we have the information convert it back to an array of strings for easier manipulation.
  fwGeneral_stringToDynString(sExceptionInfo, exceptionInfo, "|", FALSE);
  //Get the size of the exception information we have been given.
  const int sizeExInfo = dynlen(exceptionInfo);
  //Check that it is a multiple of 4, otherwise we cannot handle it (could try, but it will likely make a mess).
  //Also that there is some information to handle.
  if ((0 == sizeExInfo%4) && (0 < sizeExInfo)) {
    //The code of the last exception to be entered is the unique identifier. Can't be const because PVSS can't accept const IDs!
    //It is encoded along with the time in the last element of the exception array.
    dyn_string exCodeAndTime = strsplit(exceptionInfo[dynlen(exceptionInfo)], "_");
    string exceptId = "";
    if (0 < dynlen(exCodeAndTime))
      exceptId = exCodeAndTime[1];
    //Now check to see whether this exception is present in the tree.
    if (!exceptionTree.itemExists(exceptId)) {
      //Create/determine who the parent should be for this exception tree.
      string parentId = _fwUkl1ExceptionTree_createTreeBaseNode(exceptionTree, exceptionInfo);
      //Now convert the exceptionInfo array into a tree displayed on exceptionTree.
      _fwUkl1ExceptionTree_createTree(exceptionTree, exceptionInfo, parentId, exceptId);
    }//if(!exceptionTree.itemExists(exceptId))
  }//if((0==sizeExInfo%4)&&(0>sizeExInfo))
  else {
    //Check to see if we thing the framework can deal with it.
    if (0 == sizeExInfo%3) {
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1ExceptionTree_display(): Structure of the exception chain was invalid and could not be displayed in a tree.", "");
      fwExceptionHandling_display(exceptionInfo);
    }//if(0==sizeExInfo%3)
    else {
      //Framework can't deal with it, well it would make a mess, send to log.
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1ExceptionTree_display(): Failed to parse exception information.", "");
      DebugTN(exceptionInfo);
    }//else(0==sizeExInfo%3)
  }//else(0==sizeExInfo%4)
  
  //Done.
  return;
}//fwUkl1ExceptionTree_display()

/*!
 * Returns an unique identifier for that can be used to label an exception chain before it is submitted
 * to the exception log.
 *
 * \param  exceptionLog The array of strings that contains the exception log. This
 *           can only handle exceptions that have been stored within the HW DP exception
 *           log. This is the only way to gauranty that the exception information has been
 *           tagged with a unique ID, which it is the aim of this function to return.
 * \return string The unique ID that can be used as a string. It must be submitted as a string
 *           for the exception code.
 *
 * The function is quite robust and will return a unique ID in almost all cases. It will return a postive
 * ID if the function completes normally. It will return a negative ID if the function failed to determine
 * the previous unique ID and was forced to generate a new one randomly. In the event that it cannot find
 * a unique ID then it will return 0.
 */
string fwUkl1ExceptionTree_generateExceptionCode(const dyn_string &exceptionLog) {
  //The ID code to be returned as a number. Initialise it to -1. If it is still -1 at the end we need to
  //generate a new number randomly!
  int id = -1;
  
  //Get the number of entries in the log.
  const int exLogSize = dynlen(exceptionLog);
  //If there is no existing exception information then we should start at 1 for the new exception code.
  if (0 == exLogSize)
    id = 1;
  //If there is an error determining the exception log size then indicate this with a -ve exception code.
  else if (-1 == exLogSize)
    id = -1;
  //Otherwise we have to determine what the exception code is.
  else {
    //We have some data. The last error code must be contained in the last entry, by definition :)
    const string lastException = exceptionLog[exLogSize];
    //Now search backwards for the first occurance of a | which are used to delimit each entry in the string.
    int index;
    for (index = strlen(lastException)-1; index >= 0; --index) {
      //PVSS needs and reverse find...
      if ("|" == lastException[index]) {
        //We know the position of the last | to occur in the string. Any characters from one past here to the end will be the ID.
        id = substr(lastException, index+1);
        //Check to see if the ID was able to find an number. PVSS returns 0 if the string to int conversion failed.
        //We start the ID number from 1.
        if (0 == id)
          id = -1;
        else
          ++id;
        //Now need to loop any further we will just find the other | delimited elements.
        break;
      }//if found |
    }//for index
  }//else existing exceptions.
  
  //If there was an error and the ID is -1 generate a new ID from a random number.
  if (-1 == id) {
    //If the ID number is -1 then we need to create a random number to store instead.
    //Seed the random number generator with the current time.
    srand();
    //rand returns a 16-bit number that is made -ve to avoid conflicting with normal ID number chain.
    id = -rand();
  }//if(-1==id)
  
  //We need to check that the ID we have got is unique. Generally we will always increment from the previous
  //ID, but if an error occured and we start counting from a random negative number funny effects could occur.
  const int maxLoop = 100;
  int loop = 0;
  bool unique;
  do {
    //Check whether the randomly generated number is present already as an ID.
    dyn_bool matches = patternMatch(("*|" + (string)id + "*"), exceptionLog);
    //Check to see if TRUE is contained in the returned set of matches.
    if (0 != dynContains(matches, TRUE))
      unique = FALSE;
    else
      unique = TRUE;
    
    if (!unique) {
      if (loop == (maxLoop/2)) {
        //In this case we are failing to find a unique number, some how we must have started in the middle of
        //a run of numbers. We will substract a large number from these values to move us in ID space to
        //hopefully somewhere empty.
        id -= 65536;
      }//if(loop==(maxLoop/2))
      else if (loop > maxLoop) {
        //We have looped far too many times and hence should just give up!
        id = 0;
        break;
      }//else if(loop>maxLoop)
      else {
        //Try incrementing the ID and hope the next one is available.
        ++id;
      }//else
    }//if(!unique)
    //Increment the loop counter.
    ++loop;
  } while(!unique);

  //The ID as a string.
  string sId = id;
  
  //Done.
  return sId;
}//fwUkl1ExceptionTree_generateExceptionCode()

// ============
//   INTERNAL
// ============

/*!
 * This adds the base node to the tree for this exception chain.
 *
 * \param  exceptionTree Reference to the shape that will display the exception tree. No checks are made on
 *           the existance of this shape, so calling functions must ensure that it exists.
 * \param  exceptionInfo An array of strings that contains an exception tree built up with the UKL1 library.
 * \return string ID of the node that has been created. It should be the base node for the rest of the exception
 *           chain in the tree.
 *
 * If the base node already exists then this will be returned for the parent ID.
 * Otherwise the function adds the node and returns the ID of this node.
 */
string _fwUkl1ExceptionTree_createTreeBaseNode(shape &exceptionTree, dyn_string &exceptionInfo) {
  //Get the last function name that has been entered into the exception chain. This identifies the
  //root node and will either be a UKL1 board name or the name of the UKL1 CU. It is this that is
  //is used as the base ID.
  string baseId = exceptionInfo[dynlen(exceptionInfo)-2];
  //Check to see if this is present as the base node, note if it is an empty string then this is
  //the root node, but itemExists() doesn't seem to pick up on that.
  //As the item doesn't exist then we must add it to the node, where its parent is the root node, "".
  //It is identified by the base ID and it also displays the base ID.
  if (!exceptionTree.itemExists(baseId) && ("" != baseId))
    exceptionTree.appendItem("", baseId, baseId);

  //The exception code for the base node of the tree contains the unique identifier and the time the exception and
  //was submitted, delimited by an underscore.
  dyn_string exCodeAndTime = strsplit(exceptionInfo[dynlen(exceptionInfo)], "_");
  //Will hold the ID of the node that is being added plus the text that will be displayed.
  string nodeId, nodeText;
  const int lenExCodeAndTime = dynlen(exCodeAndTime);
  if (2 == lenExCodeAndTime) {
    //Everything is normal and we can assign things as desired.
    nodeId = exCodeAndTime[1];
    nodeText = exCodeAndTime[2];
  }//if(2==lenExCodeAndTime)
  else if (1 == lenExCodeAndTime) {
    //Don't have either the time or ID, just use what we have. The library should ensure we have a unique ID.
    nodeId = exCodeAndTime[1];
    nodeText = exCodeAndTime[1];
  }//else if(1==lenExCodeAndTime)
  else {
    //Something is really wrong. Just tag it with a random number for now.
    srand();
    nodeId = -rand();
    nodeText = nodeId;
  }//else no exCodeAndTimeFound
  //The parent node for all the exception will be the exception ID itself.
  //The text it is labelled with is the time the exception was submitted.
  //By this point it will have already been checked that it does not exist.
  exceptionTree.appendItem(baseId, nodeId, nodeText);
  
  //Done.
  return nodeId;
}//_fwUkl1ExceptionTree_createTreeBaseNode()

/*!
 * This adds the exception chain to the given node as a base not in the tree.
 *
 * \param  exceptionTree Reference to the shape that will display the exception tree. No checks are made on
 *           the existance of this shape, so calling functions must ensure that it exists.
 * \param  exceptionInfo An array of strings that contains an exception tree built up with the UKL1 library.
 * \param  baseNodeId ID of the node that the exception tree is to descend from. If an empty string is given
 *           the root node of the exception tree shape will be used.
 * \param  exceptId This is the number that uniquely identifies the exception.
 * \return void.
 *
 * A check must have been made that the exception does not already exist in the tree.
 * This function assumes that it does not.
 */
void _fwUkl1ExceptionTree_createTree(shape &exceptionTree, dyn_string &exceptionInfo, string &baseNodeId, string &exceptId) {
  //Indicates the parent of the next node to be added. Starts as the base node.
  string parentId = baseNodeId;
  //Indicates the identifier of the child that is being added to the tree. Initialise to NULL, create in loop.
  string childId = "";
  //Used to ensure that all the IDs for a given exception ID are unique.
  unsigned idNumber = 0;
  //This map uses the function name from the exception as a key to its parents ID in the tree.
  //If when we get a new function name we access the map and see if there is an ID associated with the
  //function name. If there is we place the following entries below the same parent in the tree,
  //and update the ID in the map for that function. This ensure that there is a level in the tree for each
  //function name and if a function generates multiple errors itself they appear in the same level.
  mapping functionMap;
  //Loop over the exception array, backwards so we see the highest level functions messages first.
  //Skip the first four levels as these contain the base node identifier and the unique code, which
  //have already been dealt with.
  const int firstException = dynlen(exceptionInfo) - 4;
  //Move in multiples of 4 as this is the number of bits of information per exception. Stop at element 4
  //as we index backwards from the number of exInfo.
  for (int exInfo = firstException; exInfo >= 4; exInfo -= 4) {
    //First get the function name (second element of the exception group) and check if it is in the map.
    string functionName = exceptionInfo[exInfo-2];
    //Check to see if our function is already in this exception tree. Do this before adding the function to the
    //map otherwise it will be executed everytime round the loop.
    if (mappingHasKey(functionMap, functionName)) {
      //The function exists in the map.
      //Get the ID number from the map.
      string mapId = functionMap[functionName];
      //Check the IDs of the current parent's parent etc. to see if this mapId is in the chain.
      string ancestorsId = parentId;
      //If the ID from the map is not an ancestor of the logical parent of the exception then the
      //exception stays with its logical parent.
      do {
        if (mapId == ancestorsId) {
          //The mapId is infact our ancestor and so it should become our parent.
          parentId = mapId;
          break;
        }//if(mapId==ancestorsId)
        //Update the ancestorsId to be the parent of the ancestor that is currently under investigation
        //and check this against the map ID.
        ancestorsId = exceptionTree.parent(ancestorsId);
      } while (ancestorsId != "");
    }//if(mappingHasKey())
    
    //Whether the function exists in the map or not it must occur there with its latest parent discovered.
    functionMap[functionName] = parentId;
    
    //Now create the ID of the child.
    childId = exceptId + "_" + (idNumber++);
    
    //Create an entry in the tree with under the appropriate parent.
    string text = "";
    if (baseNodeId == parentId)
      text = exceptId;
    exceptionTree.appendItem(parentId, childId, text);
    //Set the type FATAL, ERROR or WARNING.
    exceptionTree.setText(childId, 0, exceptionInfo[exInfo-3]);
    //Set the function.
    exceptionTree.setText(childId, 1, functionName);
    //Set the message.
    exceptionTree.setText(childId, 2, exceptionInfo[exInfo-1]);
    //Not bothering with the error code any more.
    //Update the current parent.
    parentId = childId;
  }//for exInfo
  //Done.
  return;
}//_fwUkl1ExceptionTree_createTree()

//@}
