// ====================================
//  NAME    : fwUkl1ExceptionHandling.ctl
//
//  VERSION : 1
//
//  REVISION: 2
//
//  DATE    : 2007/06/07
//
//  RELEASE : 2007/10/10
//
//  PLATFORM: PVSS II
//
//  AUTHOR  : Gareth Rogers
//
//  EMAIL   : rogers@hep.phy.cam.ac.uk
// ====================================

/*! \mainpage UKL1 Exception Library Documentation
 *
 * This library is designed to extend the capablilty of the framework exception handling to make the submission and
 * display more relevant to the UKL1 project. In order to provide an analogy with the framework exception handling
 * functions all functions that are intended for use by the User to create an exception chain are prefixed fwUkl1Exception_
 * and those functions that are for the display of exception chains are prefixed fwUkl1ExcpetionHandling_. Finally
 * the exception log is provided as an extension over the framework capabilities and functions for utilising the
 * exception log are prefixed fwUkl1ExceptionLog_. Functions used for internal library use have a leading _.
 *
 * The UKL1 exceptions utilise the same structure as that defined by the framework, for obvious compatibility reasons
 * and use the fwException_raise() function as a base for constructing the exception chains. As a consequence of this
 * compatibility the library provides a fwUkl1Exception_raise function that takes the exact same arguments as the framework
 * equivalent. It provided as a wrapper for fwException_raise() which ensures that the structure of the exception message
 * is in a format that is required by the display functions (`Function(): Message', defined in the framework guidelines)
 * and will coerce it into this form if necessary. This is simply to reduce the number of cases that the display functions
 * cannot deal with. It is not require that this function be used and fwException_raise() can be used to create exception
 * chains that this libraries display functions can handle.
 *
 * The library also provides a replacement for fwExceptionHandling_display() called fwUkl1ExceptionHandling_display().
 * This function is not however a wrapper for fwExceptionHandling_display() and instead is provided as a replacement.
 * It displays the exception chain in a tree structure with the last exception being displayed as the first branch of the
 * tree each time a new function fails it will be placed under the exception generated by its calling function.
 * In order to ensure the tree is displayed in a logical way each function call that may generate an execption should have
 * an exception raise()d by the calling function if one occurs. If this is not the case the it is unlikely the exception
 * will be able to identify its parent in the tree correct and unexpected tree structure may occur. The base node in the
 * tree is the exception code that is given for the last exception entered in the tree. Each subsequent node will have the
 * exception code to identify it.
 *
 * The panel opened by fwUkl1ExceptionHandling_display() is User configurable and it can be set to a User created panel
 * but it must take the exception chain as a string via the parameter $sExceptionInfo. No requirements are set on how
 * the panel displays the element and the string can be converted back to a framework compatible dyn_string via
 * fwGeneral_stringToDynString() with a `|' as a delimiter. A panel is provided by the library and is set by default.
 *
 * fwUkl1ExceptionHandling_display() generates a pop up panel and is intended to show exception information immediately after
 * a function chain has completed. The generated information will be lost as soon as the panel has been closed. The function
 * fwUkl1ExceptionLog_submit() is provided as a means of saving exception chains to a log for later display. This also
 * allows more complex exception tree displays to be created. fwUkl1ExceptionLog_submit() takes not only the exception
 * chain to be displayed but the name of a base node that the exception chain should be displayed under. This allows the
 * grouping of exception chains for example a base node name may be the name of a UKL1 board for example, all failures of
 * functions relating to this board could be tagged and would be displayed under the hardware name regardless of when they
 * occured. A function is also provided to display the exception log, fwUkl1ExceptionLog_displayCB(), and it is intended
 * for use as a call back function. A PVSS tree shape can be connected to the log using the function
 * fwUkl1ExceptionLog_connect() and then all exception chains that are submitted to the log are automatically displayed
 * once they occur. fwUkl1ExceptionLog_displayCB() does not need be used as a call back function, but the User must
 * create their own panels for use in this case.
 *
 * The library provides all the functionality to create the exception log, including the data point type. The name of the
 * exception log data point is User configurable and should be set before the first exception is submitted, if it is to
 * be changed from the default. It is possible to change the log data point at any time, but the call back function must
 * be recreated by the User if the new exception log data point is to be monitored for submissions.
 *
 * It should also be noted that the exception log converts the dyn_string exception chain into a string and the log stores
 * each exception chain as an element in a dyn_string. The conversion is done using fwGeneral_stringToDynString() and
 * fwGeneral_dynStringToString() with `|' separators and white space left in will reconstruct the exception chain in a
 * form fully compatible with fwUkl1ExceptionHandling_display() and fwExceptionHandling_display().
 *
 * Two panels are also included with this library in objects/fwUkl1Exception/fwUkl1ExceptionHandlingDisplay.pnl and
 * fwUkl1ExceptionLog_display.pnl. fwUkl1ExceptionHandling_display.pnl is opened by calls to fwUkl1ExceptionHandlingDisplay()
 * and is intended to display single exceptions chains immediately after they occur. Exceptions chains are passed as a
 * string reference parameter $sExceptionInfo and the panel displays the given exception chain in a PVSS tree shape as
 * described. fwUkl1/fwUkl1ExceptionLogDisplay.pnl is a panel that is a standalone panel that can be entered as a manager in the
 * PVSS console if desired or launched from an existing panel. It provides an interface to the exception log and will
 * display any exceptions it finds in the log that have not already been displayed. It is connected to the exception log
 * and will thus automatically update with new exceptions as they are generated.
 * 
 */

// =========================
//  CONSTANTs & GLOBAL VARs
// =========================

/** @defgroup SectionConstants Constants and global variables.
 *   List of constants and global variables that are used by the library. Functions are provided
 *   to set/get all global variables as part of the library interface.
 *  @{
 */

/*! Defines the name of the data point type that holds the exception log. */
const string FWUKL1_EXCEPTION_LOG_DPT = "FwUkl1ExceptionLog";

/*! Defines the name of the data point that is the instantiation of the exception log, defaults to `Ukl1Exceptions' */
global string gsFwUkl1ExceptionLogDp = "Ukl1Exceptions";

/*! Defines the name of the data point element that the log is stored in, includes the `.' required to access the element. */
const string FWUKL1_EXCEPTION_LOG_DPE = "exceptionLog";

/*! Defines the full path and name of the panel that is to be opened by fwUkl1ExceptionHandling_display(). */
global string gsFwUkl1ExceptionHandlingDisplayPanel = "objects/fwUkl1ExceptionHandling/fwUkl1ExceptionHandlingDisplay.pnl";

/*! Name of the shape that fwUkl1ExceptionLog_displayCB() should use to display the exception information. */
global string gsFwUkl1ExceptionLogTree = "ExceptionTree";

//@}


// =====================
//  EXCEPTION FUNCTIONS
// =====================

/** @defgroup SectionExecption Exception functions.
 *   Contains the functions that are used to create an exception chain.
 *  @{
 */

/*!
 * A wrapper for fwException_raise(), fwUkl1Exception_raise() takes the same arguments, but will check
 * that the exceptionText is of the form `Function(): Message', as defined by the framework, and will
 * prepend `Unknown function(): ' to any exceptionText that does not contain `(): '.
 *
 * \param  exceptionInfo An array of strings that the exception information should be
 *           appended to.
 * \param  exceptionType WARNING, ERROR and FATAL are permitted for this field. It is upto the user
 *           of the library to decide which is the most appropriate to use.
 * \param  exceptionText A string to describe the exceptional condition that occured. Must be
 *           of the form `Function(): Message' where:
 *   \li Function - the name of the function/button/thing that was executing when the
 *         when the exceptional condition arose.
 *   \li Message - description of the exceptional condition that occured, a message to the function caller.
 * \param  exceptionCode A number that indicates the exception that occured. This library reserves the error
 *           code value "0" for internal use.
 * \return void.
 *
 * \b Example
 * \code
 *   (...)
 *   //Assume we are in the code body of a function with the prototype `void thisFunction(dyn_string& exceptionInfo)',
 *   //hence exceptionInfo has already been defined.
 *   //Check the size of exceptionInfo so we can see exception information is added by a function call.
 *   int exInfoSize = dynlen(exceptionInfo);
 *   //Call a function.
 *   someFunction(exceptionInfo);
 *   //Check size of the exceptionInfo to see if an error occured.
 *   if ( (exInfoSize == dynlen(exceptionInfo)) && (-1 != exInfoSize) ) {
 *     //... no error occured continue with normal function execution path.
 *   }//if(0==dynlen(exceptionInfo))
 *   else {
 *     //An error occured while calling someFunction. It is this might not cause thisFunction to fail so mark as a WARNING.
 *     //Details are provided and it is given a code of -1.
 *     fwUkl1Exception_raise(exceptionInfo, "WARNING", "thisFunction(): someFunction did not execute appropriate.", "-1");
 *   }//else(0==dynlen(exceptionInfo))
 *
 *   (...)
 *  \endcode
 */
void fwUkl1Exception_raise(dyn_string& exceptionInfo, string exceptionType, string exceptionText, string exceptionCode) {
  //Search the exceptionText for the colon space marker.
  //Add it if it is not present.
  if ( -1 == strpos(exceptionText, ": ") )
    exceptionText = "Unknown function(): " + exceptionText;

  //Now raise the exception.
  fwException_raise(exceptionInfo, exceptionType, exceptionText, exceptionCode);
  //Done.
  return;
}//fwUkl1Exception_raise()

//@}

// ==============================
//  EXCEPTION HANDLING FUNCTIONS
// ==============================

/** @defgroup SectionExecptionHandling Exception handling functions.
 *   Contains the functions that are used to convert the dyn_string form of the exception chain
 *   into an exception tree and display it on a PVSS tree shape.
 *  @{
 */

/*!
 * Opens the panel defined by gsFwUkl1ExceptionHandlingDisplayPanel displaying the exception
 * chain given by exceptionInfo. If the dyn_string is empty no panel is displayed.
 *
 * \param  exceptionInfo Exception chain that is to be displayed. It will be cleared by a call to this function.
 * \return void.
 *
 * The form of exceptionInfo is checked to ensure that it is of an appropriate form to be displayed in a tree
 * with fwUkl1ExceptionHandlingDisplay.pnl, if it cannot be displayed with this panel then it is displayed
 * via fwExceptionHandling_display(). In both cases a call to fwUkl1ExceptionHandling_display() will clear the
 * contents of exceptionInfo, as defined in the framework guidelines.
 *
 * \b Example
 * \code
 *   (...)
 *   //Create a dyn_string to hold the exception information.
 *   dyn_string exceptionInfo;
 *   //Call a function, that could generate an error.
 *   thisFunction(exceptionInfo);
 *   //If an error has been generated it will be displayed otherwise nothing will occur.
 *   fwUkl1ExceptionHandling_display(exceptionInfo);
 *   //Note that exception info is now definately empty, for example a call to fwUkl1ExceptionLog_submit() would do nothing.
 *   (...)
 *  \endcode
 */
void fwUkl1ExceptionHandling_display(dyn_string& exceptionInfo) {
  //Determine if we have a valid exception info.
  const int exceptSize = dynlen(exceptionInfo);
  if ( 0 == exceptSize ) {
    //If it is empty just return.
    return;
  }//if(0==exceptSize)
  
  if ( (-1 != exceptSize) && (0 == exceptSize%3) ) {
    //Add a final exception that is used to create the root node for this exception tree.
    //The final string is a unique identifier in the tree widget. It will only ever display
    //one entry so it doesn't matter what this number is. Ensure that the time is stamped with this
    //number.
    fwUkl1Exception_raise(exceptionInfo, "", ": ", "1_" + (string)getCurrentTime());
    //PVSS doesn't seem to like arrays being passed as dollar params.
    //Convert to from dyn_string to string.
    string sExceptionInfo;
    fwGeneral_dynStringToString(exceptionInfo, sExceptionInfo, "|");
    
    //We have a valid exception array so give it to the exception viewer.
    string panelLocation = gsFwUkl1ExceptionHandlingDisplayPanel;
    //Add a random number as it allows multiple exception window to be open at the same time.
    //There should never be enough open via this function to worry about it being the same.
    string panelName = "ExceptionInfo" + rand();
    dyn_string panelParams;
    //The only parameter it takes is the exception info.
    panelParams[1] = "$sExceptionInfo:" + sExceptionInfo;
    ChildPanelOnCentral(panelLocation, panelName, panelParams);
    dynClear(exceptionInfo);
  }//if((-1!=exceptSize)&&(0==exceptSize%3))
  else {
    //If we couldn't determine the size of the exception it is probably not a valid array. Warn.
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1ExceptionHandling_display(): Unrecognised formatting of the exception info. Using framework exception display mechanism.", "");
    fwExceptionHandling_display(exceptionInfo);
  }//else(!formatError)
  
  //Done.
  return;
}//fwUkl1ExceptionHandling_display()

/*!
 * Sets the panel used to display the exceptionInfo dyn_string given to fwUkl1ExceptionHandling_display().
 *
 * \param  panel The full path and name of the panel that is opened by fwUkl1ExceptionHandling_display().
 * \return void.
 */
void fwUkl1ExceptionHandling_setDisplayPanel(const string& panel) {
  gsFwUkl1ExceptionHandlingDisplayPanel = panel;
  //Done.
  return;
}//fwExceptionHandling_setDisplayPanel()

/*!
 * Returns the full path and name of the panel that is used to display the exception chain by fwUkl1ExceptionHandling_display().
 *
 * \return string Full path and name of the panel.
 */
string fwUkl1ExceptionHandling_getDisplayPanel() {
  //Only one thing to do.
  return gsFwUkl1ExceptionHandlingDisplayPanel();
}//fwUkl1ExceptionHandling_getDisplayPanel()

// ============
//   INTERNAL
// ============

/*!
 * Takes an exception chain in the form of a `|' delimited string and converts this into a tree structure
 * displayed on a PVSS tree shape.
 *
 * \param  exceptionTreeName Name of the tree widget shape that the exception tree is to be displayed on.
 * \param  sExceptionInfo An array of strings that contains an exception tree built up with the UKL1 library.
 * \return void.
 *
 * In the event that display cannot parse the exception information it is handled via fwExceptionHandling_display()
 * if it is believed to be still in a form it can handle, otherwise it is sent to the PVSS log. This will not display
 * the last exception in the tree as this is used to create the base node for the tree. It should be ensured that
 * this has been created appropriately.
 */
void _fwUkl1ExceptionHandling_display(const string& exceptionTreeName, string sExceptionInfo) {
  //This will hold the exception info once it have been converted back to a dyn_string.
  dyn_string exceptionInfo;
  //First we must check the shape exists.
  if (shapeExists(exceptionTreeName)) {
    //Search the string and replace all entries for `: ' with a `|'. This will ensure that the `Function(): Message'
    //will be split into two different elements of the array, allowing them to be more easily be displayed.
    const int numReplacements = strreplace(sExceptionInfo, ": ", "|");
    //Now we have the information convert it back to an array of strings for easier manipulation.
    fwGeneral_stringToDynString(sExceptionInfo, exceptionInfo, "|", FALSE);
    //Get the size of the exception information we have been given.
    const int sizeExInfo = dynlen(exceptionInfo);
    //Check that it is a multiple of 4, otherwise we cannot handle it (could try, but it will likely make a mess).
    //Also that there is some information to handle.
    if ( (0 == sizeExInfo%4) && (0 < sizeExInfo) ) {
      //Get the shape that is to display the exception tree.
      shape exceptionTree = getShape(exceptionTreeName);
      //The code of the last exception to be entered is the unique identifier. Can't be const because PVSS can't accept const IDs!
      //It is encoded along with the time in the last element of the exception array.
      dyn_string exCodeAndTime = strsplit(exceptionInfo[dynlen(exceptionInfo)], "_");
      string exceptId = exCodeAndTime[1];
      //Now check to see whether this exception is present in the tree.
      if ( !exceptionTree.itemExists(exceptId) ) {
        //Create/determine who the parent should be for this exception tree.
        string parentId = _fwUkl1ExceptionHandling_createTreeBaseNode(exceptionTree, exceptionInfo);
        //Now convert the exceptionInfo array into a tree displayed on exceptionTree.
        _fwUkl1ExceptionHandling_createTree(exceptionTree, exceptionInfo, parentId, exceptId);
      }
    }//if((0==sizeExInfo%4)&&(0>sizeExInfo))
    else {
      //Check to see if we thing the framework can deal with it.
      if (0 == sizeExInfo%3) {
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1ExceptionHandling_display(): Structure of the exception chain was invalid and could not be displayed in a tree.", "");
        fwExceptionHandling_display(exceptionInfo);
      }//if(0==sizeExInfo%3)
      else {
        //Framework can't deal with it, well it would make a mess, send to log.
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1ExceptionHandling_display(): Failed to parse exception information.", "");
        DebugTN(exceptionInfo);
      }//else(0==sizeExInfo%3)
    }//else(0==sizeExInfo%4)
  }//if(shapeExists(exceptionTreeName))
  else {
    //The tree for displaying the exception info doesn't exist.
    //Convert back to a dyn_string and use fw exception handling.
    fwGeneral_stringToDynString(sExceptionInfo, exceptionInfo, "|", FALSE);
    fwUkl1ExceptionHandling_display(exceptionInfo);
  }//else(shapeExists(exceptionTreeName))
  
  //Done.
  return;
}//_fwUkl1ExceptionHandling_display()

/*!
 * This creates the base node to be used in the tree for this exception chain. If the base node it wishes
 * to use already exists then this will be returned for the parent ID and no changes are made to it. Otherwise
 * the function creates a node and returns the ID of this node.
 *
 * \param  exceptionTree Reference to the shape that will display the exception tree. No checks are made on
 *           the existance of this shape, so calling functions must ensure that it exists.
 * \param  exceptionInfo An array of strings that contains an exception tree built up with the UKL1 library.
 * \return string ID of the node that has been created. It should be the base node for the rest of the exception
 *           chain in the tree.
 */
string _fwUkl1ExceptionHandling_createTreeBaseNode(shape& exceptionTree, dyn_string& exceptionInfo) {
  //Get the last function name that has been entered into the exception chain. This identifies the
  //root node and will either be a UKL1 board name or the name of the UKL1 CU. It is this that is
  //is used as the base ID.
  string baseId = exceptionInfo[dynlen(exceptionInfo)-2];
  //Check to see if this is present as the base node, note if it is an empty string then this is
  //the root node, but itemExists() doesn't seem to pick up on that.
  if ( !exceptionTree.itemExists(baseId) && ("" != baseId) ) {
    //As the item doesn't exist then we must add it to the node, where its parent is the root node, "".
    //It is identified by the base ID and it also displays the base ID.
    exceptionTree.appendItem("", baseId, baseId);
  }//if(!exceptionTree.itemExists())
  //Nothing to be done for else.

  //The exception code for the base node of the tree contains the unique identifier and the time the exception and
  // was submitted, delimited by an underscore.
  dyn_string exCodeAndTime = strsplit(exceptionInfo[dynlen(exceptionInfo)], "_");
  //Will hold the ID of the node that is being added plus the text that will be displayed.
  string nodeId, nodeText;
  const int lenExCodeAndTime = dynlen(exCodeAndTime);
  if ( 2 == lenExCodeAndTime ) {
    //Everything is normal and we can assign things as desired.
    nodeId = exCodeAndTime[1];
    nodeText = exCodeAndTime[2];
  }//if(2==lenExCodeAndTime)
  else if ( 1 == lenExCodeAndTime ) {
    //Don't have either the time or ID, just use what we have. The library should ensure we have a unique ID.
    nodeId = exCodeAndTime[1];
    nodeText = exCodeAndTime[1];
  }//else if(1==lenExCodeAndTime)
  else {
    //Something is really wrong. Just tag it with a random number for now.
    srand();
    nodeId = -rand();
    nodeText = nodeId();
  }//else no exCodeAndTimeFound
  //The parent node for all the exception will be the exception ID itself.
  //The text it is labelled with is the time the exception was submitted.
  //By this point it will have already been checked that it does not exist.
  exceptionTree.appendItem(baseId, nodeId, nodeText);
  
  //Done.
  return nodeId;
}//fwUkl1ExceptionHandling_createTreeBaseNode()

/*!
 * This creates the exception tree with the given node as a base. A check must have been made that the
 * exception does not already exist in the tree. This function assumes that it does not.
 *
 * \param  exceptionTree Reference to the shape that will display the exception tree. No checks are made on
 *           the existance of this shape, so calling functions must ensure that it exists.
 * \param  exceptionInfo An array of strings that contains an exception tree built up with the UKL1 library.
 * \param  baseNodeId ID of the node that the exception tree is to descend from. If an empty string is given
 *           the root node of the exception tree shape will be used.
 * \param  exceptId This is the number that uniquely identifies the exception.
 * \return void.
 */
void _fwUkl1ExceptionHandling_createTree(shape& exceptionTree, dyn_string& exceptionInfo, string& baseNodeId, string& exceptId) {
  //Indicates the parent of the next node to be added. Starts as the base node.
  string parentId = baseNodeId;
  //Indicates the identifier of the child that is being added to the tree. Initialise to NULL, create in loop.
  string childId = "";
  //Used to ensure that all the IDs for a given exception ID are unique.
  unsigned idNumber = 0;
  //This map uses the function name from the exception as a key to its parents ID in the tree.
  //If when we get a new function name we access the map and see if there is an ID associated with the
  //function name. If there is we place the following entries below the same parent in the tree,
  //and update the ID in the map for that function. This ensure that there is a level in the tree for each
  //function name and if a function generates multiple errors itself they appear in the same level.
  mapping functionMap;
  //Loop over the exception array, backwards so we see the highest level functions messages first.
  //Skip the first four levels as these contain the base node identifier and the unique code, which
  //have already been dealt with.
  const int firstException = dynlen(exceptionInfo) - 4;
  //Move in multiples of 4 as this is the number of bits of information per exception. Stop at element 4
  //as we index backwards from the number of exInfo.
  for (int exInfo = firstException; exInfo >= 4; exInfo -= 4) {
    //First get the function name (second element of the exception group) and check if it is in the map.
    string functionName = exceptionInfo[exInfo-2];
    //Check to see if our function is already in this exception tree. Do this before adding the function to the
    //map otherwise it will be executed everytime round the loop.
    if ( mappingHasKey(functionMap, functionName) ) {
      //The function exists in the map.
      //Get the ID number from the map.
      string mapId = functionMap[functionName];
      //Check the IDs of the current parent's parent etc. to see if this mapId is in the chain.
      string ancestorsId = parentId;
      //If the ID from the map is not an ancestor of the logical parent of the exception then the
      //exception stays with its logical parent.
      do {
        if (mapId == ancestorsId) {
          //The mapId is infact our ancestor and so it should become our parent.
          parentId = mapId;
          break;
        }//if(mapId==ancestorsId)
        //Update the ancestorsId to be the parent of the ancestor that is currently under investigation
        //and check this against the map ID.
        ancestorsId = exceptionTree.parent(ancestorsId);
      } while (ancestorsId != "");
    }//if(mappingHasKey())
    
    //Whether the function exists in the map or not it must occur there with its latest parent discovered.
    functionMap[functionName] = parentId;
    
    //Now create the ID of the child.
    childId = exceptId + "_" + (idNumber++);
    
    //Create an entry in the tree with under the appropriate parent.
    string text = "";
    if ( baseNodeId == parentId )
      text = exceptId;
    exceptionTree.appendItem(parentId, childId, text);
    //Set the type FATAL, ERROR or WARNING.
    exceptionTree.setText(childId, 0, exceptionInfo[exInfo-3]);
    //Set the function.
    exceptionTree.setText(childId, 1, functionName);
    //Set the message.
    exceptionTree.setText(childId, 2, exceptionInfo[exInfo-1]);
    //Not bothering with the error code any more.
    //Update the current parent.
    parentId = childId;
  }//for exInfo
  //Done.
  return;
}//_fwUkl1_createExceptionTree()

//@}


// =========================
//  EXCEPTION LOG FUNCTIONS
// =========================

/** @defgroup SectionExecptionLog Exception log manipulation functions.
 *   Contains the functions that are used to create the exception log, including the data point type.
 *   Functions are provided for submitting to the log as well as displaying the exceptions that are
 *   generated by the log. The display mechanism can be set up as a call back routine to monitor changes
 *   in the exception log and then display them appropriately.
 *  @{
 */

// /*!
//  * Used to create the data point type for the exception log. The name of the type is taken from the library constant
//  * FWUKL1_EXCEPTION_LOG_DPT. It checks whether the data point type exists before it tries to create it.
//  *
//  * \param  exceptionInfo Contains information about exceptional conditions that may have occured.
//  * \return void.
//  */
// void fwUkl1ExceptionLog_createDpt(dyn_string& exceptionInfo) {
//   //Holds the size of the exceptionInfo so we can see if exceptions have been generated.
//   int exInfoSize = dynlen(exceptionInfo);
//   
//   //Check to see if the DPT already exists.
//   if ( 0 == dynlen(dpTypes(FWUKL1_EXCEPTION_LOG_DPT)) ) {
//     //The DTP doesn't exist, so create it. Names of the elements in the DPT.
//     dyn_dyn_string dptElementNames;
//     dptElementNames[1] = makeDynString(FWUKL1_EXCEPTION_LOG_DPT, "");
//     dptElementNames[2] = makeDynString("", FWUKL1_EXCEPTION_LOG_DPE);
//     //Types of the elements in the DPT.
//     dyn_dyn_int dptElementTypes;
//     dptElementTypes[1] = makeDynInt(DPEL_STRUCT, 0);
//     dptElementTypes[2] = makeDynInt(0, DPEL_DYN_STRING);
//     //Now create it.
//     dpTypeCreate(dptElementNames, dptElementTypes);
//     //Check for errors.
//     dyn_errClass error = getLastError();
//     if ( 0 < dynlen(error) ) {
//       DebugTN(error);
//       fwUkl1Exception_raise(exceptionInfo, "FATAL", "fwUkl1ExceptionLog_createDpt(): Failed to create the data point type, cannot use exception log.", "");
//     }//if dpTypeCreate failed.
//   }//if DPT does not exist.
//   //no else as if it already exists and we were asked to create it then we are done.
//   //Done.
//   return;
// }//fwUkl1ExceptionLog_createDpt()

// /*!
//  * Removes the exception log data point type from the system. The data point type that is removed is identifed by
//  * FWUKL1_EXCEPTION_LOG_DPT. All data points of this tpye myst be removed before it can be deleted, and this will
//  * be done by this function.
//  *
//  * \param  exceptionInfo Contains information about exceptional conditions that may have occured.
//  * \return void.
//  */
// void fwUkl1ExceptionLog_deleteDpt(dyn_string& exceptionInfo) {
//   //To be added...
// }//fwUkl1ExceptionLog_deleteDpt()

/*!
 * Sets the name of the data point that is to be used for the exception log. Can also be used to create the data point
 * if desired.
 *
 * \param  logName Name of the log that is to be created.
 * \param  deleteOld If TRUE this deletes the existing exception log, if one can be found. If FALSE any existing
 *           log is not checked for.
 * \param  create If TRUE then the new exception log is created, if FALSE then it is not. It is important to note that
 *           the next call to submit will create the new exception log anyway.
 * \param  exceptionInfo Contains information about any exceptions that may occur during the function call.
 * \return void.
 */
void fwUkl1ExceptionLog_setLogName(const string& logName, bool deleteOld, bool create, dyn_string& exceptionInfo) {
  //Used to check if exception information is generated.
  int exInfoSize = dynlen(exceptionInfo);
  
  //Check to see if we should delete the existing log.
  if (deleteOld) {
    fwUkl1ExceptionLog_delete(exceptionInfo);
    if ( (exInfoSize != dynlen(exceptionInfo)) || (-1 == exInfoSize) ) {
      fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1ExceptionLog_setLogName(): Failed to delete existing log, " + gsFwUkl1ExceptionLogDp + ", before updating log name.", "");
      exInfoSize = dynlen(exceptionInfo);
    }//if exception info generated.
  }//if(deleteOld)
  
  //Set the new log name.
  gsFwUkl1ExceptionLogDp = logName;
  
  //Check to see if we want to create a new exception log.
  if (create) {
    fwUkl1ExceptionLog_create(exceptionInfo);
    if ( (exInfoSize != dynlen(exceptionInfo)) || (-1 == exInfoSize) ) {
      fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1ExceptionLog_setLogName(): Failed to create new log, " + gsFwUkl1ExceptionLogDp + ".", "");
      exInfoSize = dynlen(exceptionInfo);
    }//if exception info generated.
  }//if(create)
  
  //Done.
  return;
}//fwUkl1ExceptionLog_setLogName()

/*!
 * Returns the name of the current exception log.
 *
 * \return string Name of the current exception log.
 */
string fwUkl1ExceptionLog_getLogName() {
  //Only one thing to do.
  return gsFwUkl1ExceptionLogDpName;
}//fwUkl1ExceptionLog_getLogName()

/*!
 * Used to create an exception log for use with the library functions. If the log already exists it will return
 * with no errors. The log name is taken from the global variable gsFwUkl1ExceptionLogDp, which can accessed through
 * set/getFwUkl1ExceptionLog_getLogName().
 *
 * \param  exceptionInfo Contains details about any exception information that occured during the creation of the log.
 *           if exception information is present then the log does not exist.
 * \return void.
 */
void fwUkl1ExceptionLog_create(dyn_string& exceptionInfo) {
  //Check that it doesn't exist.
  if (!dpExists(gsFwUkl1ExceptionLogDp)) {
    //If it doesn't exist then we must create it.
    //Quote PVSS help: "dpCreate() returns 0, in the event of a failure returns -1. 
    //                  The function possibly returns 0 also in the event of a failure.
    //                  In this case errors can only be retrieved with getLastError()!"
    //Nice. Use getLastError() to determine if the DP creation was successful.
    dpCreate(gsFwUkl1ExceptionLogDp, FWUKL1_EXCEPTION_LOG_DPT);
    //Doubt it is thread safe...
    dyn_errClass error = getLastError();
    if ( 0 != dynlen(error) ) {
      //Couldn't create the data point.
      fwUkl1Exception_raise(exceptionInfo, "FATAL", "fwUkl1ExceptionLog_create(): Exception log data point does not exist and cannot be created.", "");
    }//if dpCreate() failed
  }//if(!dpExists())
  //no else as we just return if the data point already exists.
  //Done.
  return;
}//fwUkl1_createExceptionLog()

/*!
 * Deletes the existing exception log.
 *
 * \param  exceptionInfo Contains information about any exceptions that may occur during the function call.
 * \return void.
 *
 * If the log is delete and none recreated then the next call to submit or any attempt to set up a call back function
 * will recreate the log.
 */
void fwUkl1ExceptionLog_delete(dyn_string& exceptionInfo) {
  //Check to see if the log exists.
  if (dpExists(gsFwUkl1ExceptionLogDp)) {
    dpDelete(gsFwUkl1ExceptionLogDp);
    //Check for errors.
    dyn_errClass error = getLastError();
    if ( 0 != dynlen(error) ) {
      //Couldn't delete the data point.
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1ExceptionLog_delete(): Failed to delete the existing log, " + gsFwUkl1ExceptionLogDp + ".", "");
    }//if dpDelete was successful.
  }//if(dpExists(gsFwUkl1ExceptionLogDp))
  //no else as if it doesn't exist then we are done.
  //Done.
  return;
}//fwUkl1ExceptionLog_delete()

/*!
 * Removes all the exceptions in the current exception log.
 *
 * \param  exceptionInfo Contains information about exceptions that may have occured during the function execution.
 * \return void.
 */
void fwUkl1ExceptionLog_clear(dyn_string& exceptionInfo) {
  //Check to see if the exception log exists.
  if (dpExists(gsFwUkl1ExceptionLogDp)) {
    //There is only any point trying to clear the log if it actually exists.
    //Write an empty array to the datapoint.
    if ( 0 != dpSet(gsFwUkl1ExceptionLogDp + "." + FWUKL1_EXCEPTION_LOG_DPE, makeDynString()) )
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1ExceptionLog_clear(): Failed to clear exception log.", "");
  }//if(dpExists())
  else
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1ExceptionLog_clear(): The exception log, " + gsFwUkl1ExceptionLogDp + ", did not exist. No exception information to clear.", "");
  //Done.
  return;
}//fwUkl1_clearExceptionLog()

/*!
 * Submits an exception chain to the exception log. Takes a name for the base node that the exception chain
 * should appear under in the exception tree. If exceptionInfo is empty then the function will just return.
 *
 * \param  exceptionInfo Exception chain that is to be displayed. It will be cleared by a call to this function.
 * \param  baseNode This is an optional parameter. It can be used to group exceptions under specific node in the
 *           exception tree and could be used, for example, to have all errors relating to a specific UKL1 board
 *           always occur beneath that node in the tree. Defaults to an empty string, in this case a unique identifier
 *           is generated for the base node.
 * \return void.
 *
 * If the exception log does not exist then one will be created, the name of which is a User defined variable in
 * the library. If this function fails to complete an exception will be raise()d to note why it failed and then
 * fwUkl1ExceptionHandling_display() will used to display the exception chain. In all cases the exceptionInfo dyn_string
 * will be cleared by a call to this function. This function is synchronized to prevent multiple accesses to the
 * exception log DP.
 *
 * \b Example
 * \code
 *   (...)
 *   //Create a dyn_string to hold the exception information.
 *   dyn_string exceptionInfo;
 *   //Call a function, that could generate an error.
 *   thisFunction(exceptionInfo);
 *   //If an error has been generated it will be submitted to the log, otherwise nothing will occur.
 *   //All future exceptions that are submit()ted with "myNode" will be placed under the "myNode" entry in the exception tree.
 *   fwUkl1ExceptionLog_submit(exceptionInfo, "myNode");
 *   //Note that exception info is now definately empty, for example a call to fwUkl1ExceptionHandling_display() would do nothing.
 *   (...)
 *  \endcode
 */
synchronized fwUkl1ExceptionLog_submit(dyn_string& exceptionInfo, string baseNode="") {
  //Don't try and submit something that does not exist.
  if ( 0 >= dynlen(exceptionInfo) )
    return;
  
  //Now check that the exception log DP exists. fwUkl1_createExceptionLog will just return if the log already exists.
  int exInfoSize = dynlen(exceptionInfo);
  fwUkl1ExceptionLog_create(exceptionInfo);
  if ( (exInfoSize == dynlen(exceptionInfo)) && (exInfoSize != -1) ) {
    //We now definately have the data point to submit the information to, can only be here if there is exception information to submit.
    //Get the existing exceptionInfo, such that we can add more information to it.
    dyn_string currentExInfo;
    int dpStatus = dpGet(gsFwUkl1ExceptionLogDp + "." + FWUKL1_EXCEPTION_LOG_DPE, currentExInfo);
    if (0 == dpStatus) {
      //Successfully accessed the datapoint.
      //Now we have to generate a unique ID for the new exception to be added.
      //This will be used to store the unique ID given to this exception chain.
      string id = _fwUkl1ExceptionLog_generateCode(currentExInfo);
      //If ID is 0 the generating a unique ID for the exception code failed.
      if ( "0" != id ) {
        //Add an exception to the end of the exceptionInfo which stores the ID number and who generated the error.
        //if ( "" == baseNode )
          //baseNode = id;
        //The message submits `: ' as otherwise it would be tagged as an unknown function.
        fwUkl1Exception_raise(exceptionInfo, "", baseNode + ": ", id + "_" + (string)getCurrentTime());
        //Now we must convert the exceptionInfo array into a | separated string.
        string stringizedExInfo;
        fwGeneral_dynStringToString(exceptionInfo, stringizedExInfo, "|");
        //Append the stringized exception information to the existing exception information.
        dynAppend(currentExInfo, stringizedExInfo);
        //and write it back to the datapoint. Clear exceptionInfo if we are successful, indicate if we failed.
        if ( 0 == dpSet(gsFwUkl1ExceptionLogDp + "." + FWUKL1_EXCEPTION_LOG_DPE, currentExInfo) )
           dynClear(exceptionInfo);
        else {
          fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1ExceptionLog_submit(): Failed to write the exception information to the exception log.", "");
          fwUkl1ExceptionHandling_display(exceptionInfo);
          exInfoSize = dynlen(exceptionInfo);
        }//else(0==dpSet())
      }//if("0"!=id)
      else {
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1ExceptionLog_submit(): Failed to generate a unique identifier for the exception chain.", "");
        fwUkl1ExceptionHandling_display(exceptionInfo);
        exInfoSize = dynlen(exceptionInfo);
      }//else("0"!=id)
  
    }//if(0==dpStatus)
    else {
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1ExceptionLog_submit(): Failed to retrieve current exception log. Exception information not added to log.", "");
      fwUkl1ExceptionHandling_display(exceptionInfo);
      exInfoSize = dynlen(exceptionInfo);
    }//else(0==dpStatus)
  }//if the log exists.
  else {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1ExceptionLog_submit(): Failed to create the exception log.", "");
    fwUkl1ExceptionHandling_display(exceptionInfo);
    exInfoSize = dynlen(exceptionInfo);
  }//else the log does not exist.
  //Done.
  return;
}//fwUkl1_submitExceptionInfo()

/*!
 * Sets the name of the PVSS tree shape that the fwUkl1ExceptionLog_displayCB() should use to display the exception data.
 *
 * \param  treeName Name of the PVSS tree shape that is to be used. A check is made for the existance of this shape,
 *           if the shape is found not to exist then it will not be set and the tree name will be left unchanged.
 *           An exception is raised in this case.
 * \param  exceptionInfo Contains information about exceptions that may have occured.
 * \return void.
 */
void fwUkl1ExceptionLog_setDisplayTreeName(const string& treeName, dyn_string& exceptionInfo) {
  //Check if the shape exists.
  if (shapeExists(treeName))
    gsFwUkl1ExceptionLogTree = treeName;
  else
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1ExceptionLog_setDisplayTreeName(): The shape " + treeName + " did not exists and thus was not set. Display tree remains `" + gsFwUkl1ExceptionLogTree + "'.", "");
}//fwUkl1ExceptionLog_setDisplayTreeName()

/*!
 * Returns the name of the PVSS tree shape that will be used by the fwUkl1ExceptionLog_displayCB function to display the
 * exception log data.
 *
 * \param  exceptionInfo Contains information about exceptions that are encountered during the function execution.
 * \return string Name of the PVSS tree shape that will be used. This always returns the name of the current tree
 *           that has been set, however a check is made as to whether the shape exists. A none existent shape is
 *           signaled via the exceptionInfo parameter.
 *
 * \b Example:
 * \code
 *   (...)
 *   //Wish to get the current name of the tree being used by the fwExceptionLog_displayCB().
 *   //exceptionInfo will tell us if it is valid.
 *   dyn_string exceptionInfo;
 *   const string currentTreeDisplayName = fwUkl1ExceptionLog_getDisplayTreeName(exceptionInfo);
 *   //The tree name will always be what is currently in use, check for errors to see if it will work.
 *   if ( 0 != dynlen(exceptionInfo) ) {
 *     //We can use the name in functions that require a PVSS tree shape.
 *   }//if(0!=dynlen(exceptionInfo))
 *   else {
 *     //Tree shape is not recognised by PVSS need to take action...
 *   }//else(0!=dynlen(exceptionInfo))
 *   (...)
 * \endcode
 */
string fwUkl1ExceptionLog_getDisplayTreeName(dyn_string& exceptionInfo) {
  //Check to see if the shape exists.
  if (!shapeExists(gsFwUkl1ExceptionLogTree))
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1ExceptionLog_getDisplayTreeName(): The shape " + gsFwUkl1ExceptionLogTree + " does not exist and should not be used for displaying exception information.", "");
  //Done.
  return gsFwUkl1ExceptionLogTree;
}//fwUkl1ExceptionLog_getDisplayTreeName()

/*!
 * Sets up a connect()ion between the exception log data point and the function that is used to parse the exception log
 * display.
 *
 * \param  treeName Name of the PVSS tree shape that the exception log should be displayed on.
 * \return void.
 *
 * If the exception log is found not to exist then it will be created. In the event that the exception log
 * cannot be connected to or the tree shape does not exist, an exception will be displayed via fwUkl1ExceptionHandling_display().
 *
 * \b Example
 * \code
 *   (...)
 *   //Part of a PVSS tree shape initialisation function, created in GEDI.
 *   //Get the name of the shape that we are initialising.
 *   const string treeName = this.name();
 *   //Now call the connect function.
 *   fwUkl1ExceptionLog_connect(treeName);
 *   //Will either have successfully connected to the log or will display an exception chain in this tree.
 *   (...)
 *  \endcode
 */
void fwUkl1ExceptionLog_connect(const string& treeName) {
  //Holds any exception information that may occur.
  dyn_string exceptionInfo;
  int exInfoSize = dynlen(exceptionInfo);
  
  //Set the name of the tree to display the exception log.
  fwUkl1ExceptionLog_setDisplayTreeName(treeName, exceptionInfo);
  if ( (exInfoSize == dynlen(exceptionInfo)) && (-1 != exInfoSize) ) {
    //Check for the existance of the relevant datapoint to connect to. If it exists then no exception information will be generated.
    fwUkl1ExceptionLog_create(exceptionInfo);
    if ( (exInfoSize == dynlen(exceptionInfo)) && (-1 != exInfoSize) ) {
      //The datapoint element that we wish to connect to exists, so connect to it!
      dpConnect("fwUkl1ExceptionLog_displayCB", gsFwUkl1ExceptionLogDp + "." + FWUKL1_EXCEPTION_LOG_DPE);
      //Check for errors.
      dyn_errClass err = getLastError();
      if (0 < dynlen(err)) {
        //Datapoint element doesn't exist can't check for errors!
        //Raise an exception.
        fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1ExceptionLog_connect(): Failed to establish a call back function for the exception log data point. Future submitted errors will not be display in this tree.", "");
        //Now display the exception on given tree name.
        string sExInfo;
        fwGeneral_dynStringToString(exceptionInfo, sExInfo, "|");
        _fwUkl1ExceptionHandling_display(treeName, sExInfo);
        exInfoSize = dynlen(exceptionInfo);
      }//if(0<dynlen(err))
    }//if log exists or was created successfully.
    else {
      //Raise an exception.
      fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1ExceptionLog_connect(): Failed to establish the existance of the exception log data point. Future submitted errors will not be display in this tree.", "");
      //Now display the exception on given tree name.
      string sExInfo;
      fwGeneral_dynStringToString(exceptionInfo, sExInfo, "|");
      _fwUkl1ExceptionHandling_display(treeName, sExInfo);
      exInfoSize = dynlen(exceptionInfo);
    }//else log exists or was created successfully.
  }//if display tree shape exists.
  else {
    //Raise an exception.
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1ExceptionLog_connect(): Shape " + treeName + " does not exist, therefore exception log cannot be display on this tree shape.", "");
    //Display the exception on a new panel.
    fwUkl1ExceptionHandling_display(exceptionInfo);
    exInfoSize = dynlen(exceptionInfo);
  }//else display tree shape exists.  
  //Done.
  return;
}//fwUkl1ExceptionLog_connect()

/*!
 * Updates the exception tree display on the given exception tree shape.
 *
 * \param  treeName Name of the PVSS tree shape that the exception log should be displayed on.
 * \return void.
 *
 * \b Example
 * \code
 *   (...)
 *   //Could be a refresh button on a log viewer.
 *   //Holds any exception information.
 *   dyn_string exceptionInfo;
 *   //Set the name of the shape that we are initialising.
 *   const string treeName = "MyExceptionViewerTree";
 *   //Now call the connect function.
 *   fwUkl1ExceptionLog_refreshDisplay(treeName, exceptionInfo);
 *   //Check for errors in exceptionInfo, can't submit them to the log as it is the log we are trying to
 *   //refresh from.
 *   fwUkl1ExceptionHandling_display(exceptionInfo);
 *   (...)
 *  \endcode
 */
void fwUkl1ExceptionLog_refreshDisplay(const string& treeName, dyn_string& exceptionInfo) {
  //Check to see if the shape exists.
  if (shapeExists(treeName)) {
    //Clear all items from the tree to ensure we update it from afresh from the DP list.
    shape tree = getShape(treeName);
    tree.clear();
    //Check the log exists.
    if (dpExists(gsFwUkl1ExceptionLogDp)) {
      //Both the shape to display the exception log on and the exception log itself exist.
      //Get the current exception log data.
      dyn_string currentExInfo;
      int dpStatus = dpGet(gsFwUkl1ExceptionLogDp + "." + FWUKL1_EXCEPTION_LOG_DPE, currentExInfo);
      if (0 == dpStatus) {
        //Got the current exception log, lets see if any need updating.
        const int sizeExInfoLog = dynlen(currentExInfo);
        for ( int ex = 1; ex <= sizeExInfoLog; ++ex ) {
          //The internal exception handling display function takes the exception information in the form of a string
          //as stored in the log and it will do what is necessary to convert the data into a form that it can use.
          _fwUkl1ExceptionHandling_display(treeName, currentExInfo[ex]);
        }//for ex
      }//if(0==dpStatus)
      else {
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1ExceptionLog_refreshDisplay(): Failed to retrieve current exception log. Exception information display could not be updated.", "");
      }//else(0==dpStatus)
    }//if(dpExists())
    else {
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1ExceptionLog_refreshDisplay(): Could not find exception log to update the displaying of exception information from.", "");
    }//else(dpExists())
  }//if(shapeExists())
  else {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1ExceptionLog_refreshDisplay(): The shape " + treeName + " does not exist, could not update the exception information display on a none existant shape.", "");
  }//else(shapeExists())  
  //Done.
  return;
}//fwUkl1ExceptionLog_refreshDisplay()

/*!
 * A call back function that can be used to display the exceptions contained within the exception log.
 *
 * \param  dpe Name of the data point element that the call back function is connected to.
 * \param  asNewData A dyn_string containing the information that is contained in the data point.
 * \return void.
 *
 * \b Example
 * \code
 *   (...)
 *   //First set the name of the tree that we wish to use to display the exception log.
 *   //Holds any exception information that may occur.
 *   dyn_string exceptionInfo;
 *   int exInfoSize = dynlen(exceptionInfo);
 *   //Set the name of the tree to display the exception log.
 *   fwUkl1ExceptionLog_setDisplayTreeName(treeName, exceptionInfo);
 *   //Check for errors here, only continue if everything is ok.
 *
 *   //...assuming everything is good, check for the existance of the exception log.
 *   fwUkl1ExceptionLog_create(exceptionInfo);
 *   //Again check for errors here, only continue if everything is ok.
 *
 *   //...assuming everything is good connect this function to the exception log.
 *   dpConnect("fwUkl1ExceptionLog_displayCB", gsFwUkl1ExceptionLogDp + "." + FWUKL1_EXCEPTION_LOG_DPE);
 *   //Check for errors.
 *   dyn_errClass err = getLastError();
 *   if (0 < dynlen(err)) {
 *     //Datapoint element doesn't exist can't check for errors!
 *     //Raise an exception.
 *     fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1ExceptionLog_connect(): Failed to establish a call back function for the exception log data point. Future submitted errors will not be display in this tree.", "");
 *     //Pop up an exception demostrating the error that occured.
 *     fwUkl1ExceptionHandling_display(exceptionInfo);
 *     exInfoSize = dynlen(exceptionInfo);
 *   }//if(0<dynlen(err))
 *   (...)
 * \endcode
 */
void fwUkl1ExceptionLog_displayCB(string dpe, dyn_string asNewData) {
  //Number of elements in the UKL1 exception info log.
  const int sizeExInfoLog = dynlen(asNewData);
  //Loop over all the elements of the exception information that we have been given and see if they need adding.
  for ( int ex = 1; ex <= sizeExInfoLog; ++ex ) {
    //Get the name of the tree shape that is to be used.
    dyn_string exceptionInfo;
    const string treeName = fwUkl1ExceptionLog_getDisplayTreeName(exceptionInfo);
    if ( 0 == dynlen(exceptionInfo) ) {
      //The internal exception handling display function takes the exception information in the form of a string
      //as stored in the log and it will do what is necessary to convert the data into a form that it can use.
      _fwUkl1ExceptionHandling_display(treeName, asNewData[ex]);
    }//if(0==dynlen(exceptionInfo)
    else {
      //Tree requested to be displayed on does not exist. Create an exception.
      //First convert the current exception chain back to a dyn_string.
      dyn_string currentExInfo;
      fwGeneral_stringToDynString(asNewData[ex], currentExInfo, "|", FALSE);
      //Now append any exception information added by this function.
      dynAppend(currentExInfo, exceptionInfo);
      //Raise an exception to indicate what happened.
      fwUkl1Exception_raise(currentExInfo, "ERROR", "fwUkl1ExceptionLog_display(): Failed to find the shape to display the exception log data. Only displaying the last entry in the exception log.", "");
      //Now try and display this exception. This has the potential to flood the User with exception information.
      //It won't connect to this function if the User specifies a none existent shape, changing of the shape name is
      //also suppressed if the library finds the name doesn't work. The only problem should be if the User changes the
      //global variable directly and there is little that can be done about that.
      //By only showing the last entry in the log we prevent the User being bombarded with every exception that has ever occured.
      if ( sizeExInfoLog == ex )
        fwUkl1ExceptionHandling_display(currentExInfo);
    }//else(0==dynlen(exceptionInfo)
  }//for ex
  
  //Done.
  return;
}//fwUkl1_parseExceptionInfoLogCB

// ============
//   INTERNAL
// ============

/*!
 * Returns an unique identifier for that can be used to label an exception chain before it is submitted
 * to the exception log.
 *
 * \param  exceptionLog The array of strings that contains the exception log. This
 *           can only handle exceptions that have been stored within the HW DP exception
 *           log. This is the only way to gauranty that the exception information has been
 *           tagged with a unique ID, which it is the aim of this function to return.
 * \return string The unique ID that can be used as a string. It must be submitted as a string
 *           for the exception code.
 *
 * The function is quite robust and will return a unique ID in almost all cases. It will return a postive
 * ID if the function completes normally. It will return a negative ID if the function failed to determine
 * the previous unique ID and was forced to generate a new one randomly. In the event that it cannot find
 * a unique ID then it will return 0.
 */
string _fwUkl1ExceptionLog_generateCode(const dyn_string& exceptionLog) {
  //The ID code to be returned as a number. Initialise it to -1. If it is still -1 at the end we need to
  //generate a new number randomly!
  int id = -1;
  
  //Get the number of entries in the log.
  const int exLogSize = dynlen(exceptionLog);
  //If there is no existing exception information then we should start at 1 for the new exception code.
  if ( 0 == exLogSize ) {
    id = 1;
  }//if(0==exLogSize)
  //If there is an error determining the exception log size then indicate this with a -ve exception code.
  else if ( -1 == exLogSize ) {
    id = -1;
  }//else if(-1==exLogSize)
  //Otherwise we have to determine what the exception code is.
  else {
    //We have some data. The last error code must be contained in the last entry, by definition :)
    const string lastException = exceptionLog[exLogSize];
    //Now search backwards for the first occurance of a | which are used to delimit each entry in the string.
    int index;
    for (index = strlen(lastException)-1; index >= 0; --index) {
      //PVSS needs and reverse find...
      if ( "|" == lastException[index] ) {
        //We know the position of the last | to occur in the string. Any characters from one past here to the end will be the ID.
        id = substr(lastException, index+1);
        //Check to see if the ID was able to find an number. PVSS returns 0 if the string to int conversion failed.
        //We start the ID number from 1.
        if ( 0 == id )
          id = -1;
        else
          ++id;
        //Now need to loop any further we will just find the other | delimited elements.
        break;
      }//if found |
    }//for index
  }//else existing exceptions.
  
  //If there was an error and the ID is -1 generate a new ID from a random number.
  if ( -1 == id ) {
    //If the ID number is -1 then we need to create a random number to store instead.
    //Seed the random number generator with the current time.
    srand();
    //rand returns a 16-bit number that is made -ve to avoid conflicting with normal ID number chain.
    id = -rand();
  }//if(-1==id)
  
  //We need to check that the ID we have got is unique. Generally we will always increment from the previous
  //ID, but if an error occured and we start counting from a random negative number funny effects could occur.
  const int maxLoop = 100;
  int loop = 0;
  bool unique;
  do {
    //Check whether the randomly generated number is present already as an ID.
    dyn_bool matches = patternMatch( ("*|" + (string)id + "*"), exceptionLog );
    //Check to see if TRUE is contained in the returned set of matches.
    if ( 0 != dynContains(matches, TRUE) )
      unique = FALSE;
    else
      unique = TRUE;
    
    if (!unique) {
      if ( loop == (maxLoop/2) ) {
        //In this case we are failing to find a unique number, some how we must have started in the middle of
        //a run of numbers. We will substract a large number from these values to move us in ID space to
        //hopefully somewhere empty.
        id -= 65536;
      }//if(loop==(maxLoop/2))
      else if ( loop > maxLoop ) {
        //We have looped far too many times and hence should just give up!
        id = 0;
        break;
      }//else if(loop>maxLoop)
      else {
        //Try incrementing the ID and hope the next one is available.
        ++id;
      }//else
    }//if(!unique)
    //Increment the loop counter.
    ++loop;
  } while(!unique);

  //The ID as a string.
  string sId = id;
  
  //Done.  
  return sId;
}//_fwUkl1_generateExceptionCode()

//@}
