//-*-c++-*-

// ======================================
//  NAME    : UKL1 Library
//
//  VERSION : 0
//
//  REVISION: 2
//
//  DATE    : 2006/10/12
//
//  RELEASE : 2006/11/16
//
//  PLATFORM: PVSS II
//
//  AUTHOR  : Gareth Rogers
//
//  EMAIL   : rogers@hep.phy.cam.ac.uk
// ======================================

/*! \mainpage UKL1 Library Documentation
 *
 * This library is used to manipulate the state of the L1 board for the RICH detector.
 * Currently there are four possible states the board can exist in:
 *   \li Unconfigured This is the initial state of the board after turning on, a reset or if any register is
 *         not correctly configured. Its behaviour in this state is undefined, however it is may act vaguley
 *         sensibly depending on what caused this state to be reached.
 *   \li Ready All the registers on the L1 board have been configured and the board is ready to receive data.
 *         All input channels, front end (FE) FPGAs and gigabit ethernet (GBE) ports are disabled,no data
 *         can there for be sent or received. This state can be reached from any of the other defined states.
 *   \li Start This state can only be reached from the ready state. It configures the enable settings for the
 *         input channels, FE FPGAs and GBE ports. Once this has been called data can be receieved, processed
 *         and transmitted. If the start state is reached and ready has not been called at least once then the
 *         state will be unconfigured.
 *   \li Stop This state is effectively the same as ready, as it disables the input channels, FE FPGAs and GBE
 *         ports. It can only be reached from either the ready or start states otherwise the board state will
 *         be unconfigured.
 * \n
 * The library has been designed such that all setting of hardware registers can be done through the function
 * 'ukl1SetCcpcState', using the states defined in the constants and global variables section
 * (e.g. ukl1StartState). It will call all the necessary functions to set the hardware to the appropriate state,
 * logging any errors that occur and returning a status number to indicate whether the state change was successful.
 * This is the only function that a panel will need to call to configure the board.
 * \n
 * Functionality is provided to create a Ukl1Board data point will sensible default settings for the each data point
 * element value. The data point will be called RichL1 followed by the name of the RICH L1 board as it appears in the
 * DIM name server, allowing a unique data point to be created for each of RICH L1 boards found in a system. The
 * state machine will access the values stored in the appropriate data point by setting the global variable
 * ukl1CcpcName. This allows both the appropriate data point settings to be accessed and the appropriate L1 board
 * to be written to. The state of each of these boards is maintained through the global mapping ukl1CcpcList, which
 * maintains a knowledge of each L1 board state, associating it with its name. It is up to the User to ensure that
 * this always contains a valid set of L1 boards.
 * \n
 * A second data point, RichL1Log, contains the status, warning and error messages that are generated by the
 * library function calls and this data point should be monitored to ensure that all library messages are seen.
 *
 */// ======================================

// ======================================
//  CONSTANTs & GLOBAL VARs
// ======================================

/** @defgroup SectionConstants Constants
 *  List of constants that can be used by the user.
 *  @{
 */

/*! The name of the CCPC that is to be communicated with. */
global string ukl1CcpcName = "";

/*! A mapping of all the CCPCs found to their current state.*/
global mapping ukl1CcpcList;

/*! Number of Front end (ingress) FPGAs on the RICH L1 board. */
const unsigned UKL1_FRONT_FPGAS = 4;
/*! Number of channels that are on each front end FPGA. */
const unsigned UKL1_FPGA_CHANNELS = 9;
/*! Number of channels that are on the RICH L1 board. */
const unsigned UKL1_BOARD_CHANNELS = UKL1_FRONT_FPGAS * UKL1_FPGA_CHANNELS;
/*! Number of ports on the gigabit ethernet card. */
const unsigned UKL1_GBE_PORTS = 4;

/*! Defines a string, which represents the unconfigured state. */
const string ukl1UnconfiguredState = "Not ready";
/*! Defines a string, which represents the ready state. */
const string ukl1ReadyState = "Ready";
/*! Defines a string, which represents the start state. */
const string ukl1StartState = "Start";
/*! Defines a string, which represents the stop state. */
const string ukl1StopState = "Stop";

/*! A boolean to control whether or not the hardware writes are verified by reading back from that register. */
global bool UKL1_VERIFY_WRITE = FALSE;
//@}


// ======================================
//  CCPC LIST FUNCTIONS
// ======================================

/** @defgroup SectionCcpc CCPC list functions.
 *  These functions allow the manipulation of the global list that the library uses
 *  to keep track of the CCPCs that it can access and their states.
 *  @{
 */

/*!
 * This will initialise the global mapping so that it contains the given list of CCPC server names.
 *
 * \param  ccpcList A dyn_string containing the names of the CCPCs as they appear on the DNS server.
 * \return int The number of CCPCs that were added to the mapping.
 *
 * This will add each entry as a key in the mapping and will set their state to 'Not ready'
 * (ukl1UnconfiguredState), which can be updated by calling ukl1SetState with a CCPC name selected
 * (set in ukl1CcpcName). This will over write any key that is already in the list. Use ukl1AddCcpcToList
 * if the existing values are to be kept.
 */
int ukl1SetCcpcList(dyn_string ccpcList) {
  //First clear the list.
  ukl1ClearCcpcList();
  //Now we have emptied our list of CCPCs we should add some new ones and set their state to unconfigured.
  for (unsigned ccpcNum = 1; ccpcNum <= dynlen(ccpcList); ++ccpcNum) {
    ukl1CcpcList[ccpcList[ccpcNum]] = ukl1UnconfiguredState;
  }
  //Now return the number that we just added.
  return dynlen(ccpcList);
}

/*!
 * This will add a set of CCPC names to the ukl1CcpcList.
 *
 * \param  ccpcList A dyn_string containing the names of the CCPCs as they appear on the DNS server.
 * \return int The number of CCPCs that were added to the mapping.
 *
 * This will add each entry as a key in the mapping and will set their state to 'Not ready'
 * (ukl1UnconfiguredState), which can be updated by calling ukl1SetState with a CCPC name selected
 * (set in ukl1CcpcName). If a CCPC is already found to be in the list then no change will be made
 * to the current entry, including its state. It will show up as a reduction in the number of added
 * entries in return value.
 */
int ukl1AddCcpcToList(dyn_string ccpcList) {
  //Loop over all of the new CCPC names we have been given.
  string newName = "";
  for (unsigned ccpcNum = 1; ccpcNum <= dynlen(ccpcList); ++ccpcNum) {
    //First get the name and check to see if it is already in the mapping.
    newName = ccpcList[ccpcNum];
    if ( !mappingHasKey(ukl1CcpcList, newName) ) {
      ukl1CcpcList[newName] = ukl1UnconfiguredState;
    }
  }
  //Now return the number that we just added.
  return dynlen(ccpcList);
}

/*!
 * Returns the names of the CCPCs that we have stored in the list.
 *
 * \return dyn_string List of all the names of the CCPCs that are in the ukl1CcpcList and are used
 *           as keys for the mapping.
 */
dyn_string ukl1GetCcpcList() {
  //Loop over the full length of the mapping and record each of the keys to the map.
  dyn_string ccpcList;
  for (unsigned keyNum = 1; keyNum <= mappinglen(ukl1CcpcList); ++keyNum) {
    dynAppend(ccpcList, mappingGetKey(ukl1CcpcList, keyNum) );
  }
  //Now return our list of keys.
  return ccpcList;
}

/*!
 * Removes all the CCPCs and their state from the ukl1CcpcList.
 *
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *    \li  0 Execution proceeded without a problem.
 *    \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1ClearCcpcList() {
  //First find all the values in the list and remove them.
  string keyName = "";
  for (unsigned keyNum = 1; keyNum <= mappinglen(ukl1CcpcList); ++keyNum) {
    keyName = mappingGetKey(ukl1CcpcList, keyNum);
    mappingRemove(ukl1CcpcList, keyName);
  }
  return 0;
}

//@}


// ======================================
//  DATA POINT MANIPULATION FUNCTIONS
// ======================================

/** @defgroup SectionDatapoint Data point functions.
 *  These functions are used to manipulate the data points and to create
 *  data points from the data points type for each CCPC that is found and
 *  also for the message logs.
 *  @{
 */

/*!
 * This will create a data point from the UKL1Board data point type for a given L1 board. If the data point
 * exists no changes will be made to the existing data point.
 *
 * \param  ccpcName Name of the CCPC as it appears in the DIM name server, this will form part of
 *           the data point name.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *    \li  0 Execution proceeded without a problem.
 *    \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *    \li -2 The data point was created but the default values could not be changed, hence the User will need
 *             to set these.
 *    \li -3 The data point already exists, no data point values were changed.
 *
 * The name of the data point will be a concatonation of RichL1 and ccpcName. For example if ccpcName
 * were given as "pclbcc01" then the data point created would have the name "RichL1pclbcc01".
 * All the data point values are created with default values that would result in the stable running of
 * the L1 board.
 */
int ukl1CreateUkl1BoardDataPoint(string ccpcName) {
  //Create the name of the data point to be created.
  const string dpName = "RichL1" + ccpcName;
  //Now check to see if the data point exists.
  if ( !dpExists(dpName) ) {
    if ( 0 == dpCreate(dpName, "Ukl1Board") ) {
      //Set all the values on the data point to something sensible.
      if ( 0 == ukl1SetUkl1BoardDpDefaultValues(dpName) ) {
	//Everything has been set up fine.
	return 0;
      } else {
	//Couldn't set the default values.
	return -2;
      }
    } else {
      //Failed to create the data point so indicate an error.
      ukl1ErrorMessage("Failed to create the data point \'" + dpName + "\'");
      return -1;
    }
  } else {
    ukl1StatusMessage("The data point that was attempted to be created already exists.");
    return -3;
  }
  //Return a good state, although probably can never get to here.
  return 0;
}

/*!
 * This is used to set all the default values of the elements of the Ukl1Board data point type.
 *
 * \param  dpBaseName Name of the data point type instantiation that is to be set, i.e. data point name.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *    \li  0 Execution proceeded without a problem.
 *    \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1SetUkl1BoardDpDefaultValues(string dpBaseName) {
  //Work our way through the settings, only checking for errors every few data points or so.
  //If the first one can be set then the rest will also be set, unless the connection to the data
  //point is lost during the function call.

  //This is used to check for errors in the setting.
  int pvssErr = 0;

  //Set all the port settings.
  //Will hold the name of the data point containing any port settings.
  string dpPortName = "";
  for (unsigned port = 0; port < UKL1_GBE_PORTS; ++port) {
    sprintf(dpPortName, dpBaseName+".GbeConfiguration.Port%u.baseAddress", port);
    pvssErr += ukl1DpSet(dpPortName, 128*port);

    sprintf(dpPortName, dpBaseName+".GbeConfiguration.Port%u.TxFifoThreshold.address", port);
    pvssErr += ukl1DpSet(dpPortName, 1556+port);
    sprintf(dpPortName, dpBaseName+".GbeConfiguration.Port%u.TxFifoThreshold.value", port);
    pvssErr += ukl1DpSet(dpPortName, 208);

    sprintf(dpPortName, dpBaseName+".GbeConfiguration.Port%u.TxFifoLowWatermark.address", port);
    pvssErr += ukl1DpSet(dpPortName, 1546+port);
    sprintf(dpPortName, dpBaseName+".GbeConfiguration.Port%u.TxFifoLowWatermark.value", port);
    pvssErr += ukl1DpSet(dpPortName, 448);

    sprintf(dpPortName, dpBaseName+".GbeConfiguration.Port%u.TxFifoHighWatermark.address", port);
    pvssErr += ukl1DpSet(dpPortName, 1536+port);
    sprintf(dpPortName, dpBaseName+".GbeConfiguration.Port%u.TxFifoHighWatermark.value", port);
    pvssErr += ukl1DpSet(dpPortName, 960);

    sprintf(dpPortName, dpBaseName+".GbeConfiguration.Port%u.enabled", port);
    pvssErr += ukl1DpSet(dpPortName, TRUE);
  }
  if (0 != pvssErr) {
    //If we couldn't get the data point then return as there is no point trying to write.
    return -1;
  }

  //Set the 3 16-bit words of the destination and source MAC address, although there is no sensible value for these.
  pvssErr += ukl1DpSet(dpBaseName+".GbeConfiguration.Common.EthernetDestAddress.regAddress.bits0to15", 0);
  pvssErr += ukl1DpSet(dpBaseName+".GbeConfiguration.Common.EthernetDestAddress.destAddress.bits0to15", 0);
  pvssErr += ukl1DpSet(dpBaseName+".GbeConfiguration.Common.EthernetDestAddress.regAddress.bits16to31",1);
  pvssErr += ukl1DpSet(dpBaseName+".GbeConfiguration.Common.EthernetDestAddress.destAddress.bits16to31",0);
  pvssErr += ukl1DpSet(dpBaseName+".GbeConfiguration.Common.EthernetDestAddress.regAddress.bits32to47",2);
  pvssErr += ukl1DpSet(dpBaseName+".GbeConfiguration.Common.EthernetDestAddress.destAddress.bits32to47",0);

  pvssErr += ukl1DpSet(dpBaseName+".GbeConfiguration.Common.EthernetSourceAddress.regAddress.bits0to15", 3);
  pvssErr += ukl1DpSet(dpBaseName+".GbeConfiguration.Common.EthernetSourceAddress.sourceAddress.bits0to15", 0);
  pvssErr += ukl1DpSet(dpBaseName+".GbeConfiguration.Common.EthernetSourceAddress.regAddress.bits16to31", 4);
  pvssErr += ukl1DpSet(dpBaseName+".GbeConfiguration.Common.EthernetSourceAddress.sourceAddress.bits16to31", 0);
  pvssErr += ukl1DpSet(dpBaseName+".GbeConfiguration.Common.EthernetSourceAddress.regAddress.bits32to47", 5);
  pvssErr += ukl1DpSet(dpBaseName+".GbeConfiguration.Common.EthernetSourceAddress.sourceAddress.bits32to47", 0);

  //Also do the 2 16-bit words of the destination and source IP address, again no sensible value for these.
  pvssErr += ukl1DpSet(dpBaseName+".GbeConfiguration.Common.IpDestAddress.regAddress.bits0to15", 6);
  pvssErr += ukl1DpSet(dpBaseName+".GbeConfiguration.Common.IpDestAddress.destAddress.bits0to15", 0);
  pvssErr += ukl1DpSet(dpBaseName+".GbeConfiguration.Common.IpDestAddress.regAddress.bits16to31", 7);
  pvssErr += ukl1DpSet(dpBaseName+".GbeConfiguration.Common.IpDestAddress.destAddress.bits16to31", 0);

  pvssErr += ukl1DpSet(dpBaseName+".GbeConfiguration.Common.IpSourceAddress.regAddress.bits0to15", 8);
  pvssErr += ukl1DpSet(dpBaseName+".GbeConfiguration.Common.IpSourceAddress.sourceAddress.bits0to15", 0);
  pvssErr += ukl1DpSet(dpBaseName+".GbeConfiguration.Common.IpSourceAddress.regAddress.bits16to31",9);
  pvssErr += ukl1DpSet(dpBaseName+".GbeConfiguration.Common.IpSourceAddress.sourceAddress.bits16to31",0);

  if (0 != pvssErr) {
    //Problem retrieving values from the datapoints, no point in writing to CCPC server.
    return -1;
  }

  //These are some of the more advanced settings on the L1 board and need to be set correctly
  //otherwise some strange behaviour may result...
  pvssErr += ukl1DpSet(dpBaseName+".ProtocolType.address", 10);
  pvssErr += ukl1DpSet(dpBaseName+".ProtocolType.value", 0xf2);
  pvssErr += ukl1DpSet(dpBaseName+".TypeOfService.address", 11);
  pvssErr += ukl1DpSet(dpBaseName+".TypeOfService.value", 0);
  pvssErr += ukl1DpSet(dpBaseName+".TimeToLive.address", 12);
  pvssErr += ukl1DpSet(dpBaseName+".TimeToLive.value", 64);
  pvssErr += ukl1DpSet(dpBaseName+".Type.address", 13);
  pvssErr += ukl1DpSet(dpBaseName+".Type.value", 0x800);
  pvssErr += ukl1DpSet(dpBaseName+".Version.address", 14);
  pvssErr += ukl1DpSet(dpBaseName+".Version.value", 69);
  pvssErr += ukl1DpSet(dpBaseName+".PartitionId.address.bits0to15", 15);
  pvssErr += ukl1DpSet(dpBaseName+".PartitionId.value.bits0to15", 0xface);
  pvssErr += ukl1DpSet(dpBaseName+".PartitionId.address.bits16to31", 16);
  pvssErr += ukl1DpSet(dpBaseName+".PartitionId.value.bits16to31", 0xdead);
  pvssErr += ukl1DpSet(dpBaseName+".BunchCountPresetValue.address", 17);
  pvssErr += ukl1DpSet(dpBaseName+".BunchCountPresetValue.value", 0);

  if (0 != pvssErr) {
    //Perform an error check here and exit if there is a problem.
    return -1;
  }

  //Enable all the FPGAs and their channels.
  //Will hold the name of the data point containing any channel settings.
  string dpChanName = "";
  //Will hold the name of the data point containing any FPGA settings.
  string dpFpgaName = "";
  for (unsigned fpga = 0; fpga < UKL1_FRONT_FPGAS; ++fpga) {
    //Each FPGA has a register that contains 8 bits represeting the enable start of its channels.
    //This holds its address.
    //Determine the data point name.
    sprintf(dpChanName, dpBaseName+".Fpga%uChannels.address", fpga);
    //Now set the address that the channel enable register is at.
    pvssErr = ukl1DpSet(dpChanName, 25+fpga);
    //Loop over each channel.
    for (unsigned channel = 0; channel < UKL1_FPGA_CHANNELS; ++channel) {
      //Now the bit that will set the channel enable value.
      sprintf(dpChanName, dpBaseName+".Fpga%uChannels.Channel%u.bit", fpga, channel);
      //Set all the channels enabled.
      pvssErr = ukl1DpSet(dpChanName, channel);

      //Finally the defaul value.
      sprintf(dpChanName, dpBaseName+".Fpga%uChannels.Channel%u.enabled", fpga, channel);
      //Set all the channels enabled.
      pvssErr = ukl1DpSet(dpChanName, TRUE);
    }

    //First set the bit that the FPGA enable setting exists in, the same as its number.
    sprintf(dpFpgaName, dpBaseName+".Fpga%u.bit", fpga);
    pvssErr += ukl1DpSet(dpFpgaName, fpga);
    //Set the address that the enable register is located at.
    sprintf(dpFpgaName, dpBaseName+".Fpga%u.Enabled.address", fpga);
    pvssErr += ukl1DpSet(dpFpgaName, 24);
    //Now turn them all on by default.
    sprintf(dpFpgaName, dpBaseName+".Fpga%u.Enabled.value", fpga);
    pvssErr += ukl1DpSet(dpFpgaName, TRUE);

    //Now set the address that the run mode configuration is done at.
    sprintf(dpFpgaName, dpBaseName+".Fpga%u.RunMode.address", fpga);
    //This is always at 19.
    pvssErr += ukl1DpSet(dpFpgaName, 19);
    
    //Zero suppression settings, defaulted to enabled.
    sprintf(dpFpgaName, dpBaseName+".Fpga%u.RunMode.zeroSuppressed", fpga);
    pvssErr += ukl1DpSet(dpFpgaName, TRUE);

    //HPD setting. Here we use an inverted logic to the L1 board.
    sprintf(dpFpgaName, dpBaseName+".Fpga%u.RunMode.lhcb", fpga);
    pvssErr += ukl1DpSet(dpFpgaName, TRUE);
  }

  if (0 != pvssErr) {
    //Perform an error check here and exit if there is a problem.
    return -1;
  }

  //The MEP builder is a single register with the different settings held in different bits.
  //These define which bits control which settings.
  pvssErr += ukl1DpSet(dpBaseName+".MepBuilder.address", 15);
  pvssErr += ukl1DpSet(dpBaseName+".MepBuilder.ForceFixMepDestAddr.bits", 0);
  pvssErr += ukl1DpSet(dpBaseName+".MepBuilder.DontWaitForDestBroadcast.bits", 1);
  pvssErr += ukl1DpSet(dpBaseName+".MepBuilder.EnableThrottleOutput.bits", 2);
  pvssErr += ukl1DpSet(dpBaseName+".MepBuilder.ThrottlePolarity.bits", 4);
  //All the settings are defauled to true.
  pvssErr += ukl1DpSet(dpBaseName+".MepBuilder.ForceFixMepDestAddr.fixedDestAddr", TRUE);
  pvssErr += ukl1DpSet(dpBaseName+".MepBuilder.DontWaitForDestBroadcast.dontWait", TRUE);
  pvssErr += ukl1DpSet(dpBaseName+".MepBuilder.EnableThrottleOutput.enabled", TRUE);
  pvssErr += ukl1DpSet(dpBaseName+".MepBuilder.ThrottlePolarity.inverted", TRUE);
  //This has its own register.
  pvssErr += ukl1DpSet(dpBaseName+".MepBuilder.MepEventCount.address", 18);
  pvssErr += ukl1DpSet(dpBaseName+".MepBuilder.MepEventCount.value", 2);
  if (0 != pvssErr) {
    //Perform an error check here and exit if there is a problem.
    return -1;
  }

  //If we got to here then some magic has happened :)
  return 0;
}

/*!
 * Provides a wrapper for dpSet, which allows the value from a single data point value to be set.
 *
 * \param  dpName Name of the data point whose value is to be set.
 * \param  data Data that is to be set to the data point. It of type anytype to allow the
 *           setting of any data point value.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *
 * Further to the standard dpSet functionality it will also log any errors that occur to the UKL1
 * critical error data point and indicate a problem through the return value.
 */
int ukl1DpSet(string dpName, anytype data) {
  //Write the data to the dp.
  int pvssErr = dpSet(dpName, data);
  //Now check the return value.
  if (0 == pvssErr) {
    //Everything is ok.
    return 0;
  } else {
    //There was an error reading from the data point. Indicate this.
    string msg = "Failed to write to " + dpName;
    ukl1LogDpError(msg);
    return -1;
  }
}

/*!
 * Provides a wrapper for dpGet, which allows the value from a single data point value to be
 * retrieved.
 *
 * \param  dpName Name of the data point whose value is to be retrieved.
 * \param  data Data that is to be retrieved from the data point. It of type anytype to allow the
 *         retrival of any data point value. Returned by reference.
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *
 * Further to the standard dpGet functionality it will also log any errors that occur to the UKL1
 * critical error data point and indicate the problem through the return value.
 */
int ukl1DpGet(string dpName, anytype& data) {
  //Write the data to the dp.
  int pvssErr = dpGet(dpName, data);
  //Now check the return value.
  if (0 == pvssErr) {
    //Everything is ok.
    return 0;
  } else {
    //There was an error reading from the data point. Indicate this.
    string msg = "Failed to read from " + dpName;
    ukl1LogDpError(msg);
    return -1;
  }
}

//@}


// ======================================
//  STATE MACHINE FUNCTIONS
// ======================================

/** @defgroup SectionState State machine functions.
 *  These functions will move the L1 board into its various running states.
 *  @{
 */

/*!
 * Sets the state of the current CCPC, checking that the proposed change is allowed. Only certain
 * state transitions are viable.
 *
 * \param  newState The state the CCPC is to be changed to. Use the strings defined by UKL1.ctl library
 *           to ensure recognised state transitions e.g. ukl1ReadyState.
 * \return int An error code indicating if the operation was successful. The codes are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with the CCPC server.
 *   \li  2 Change was prevented as state change was not compatible with previous state.
 *   \li  3 Unrecognised state was requested.
 *
 * A more detailed explanation of what state transitions are permitted can be found in the library documentation.
 * The board that will be addressed is that selected by the global variable ukl1CcpcName.
 */
int ukl1SetCcpcState(string newState) {
  //Determine the current state of the board.
  string currentState = ukl1GetCcpcState();
  ukl1StatusMessage("currentState: " + currentState + ", newState: " + newState);
  //Check that the transition requested is valid.
  if ( (ukl1UnconfiguredState == currentState) && (ukl1ReadyState != newState) ) {
    //Transition not allowed.
    ukl1WarningMessage("from not ready change not allowed.");
    return 2;
  } 
  if ( (ukl1ReadyState == currentState) && ((ukl1ReadyState != newState) && (ukl1StartState != newState) && (ukl1StopState != newState)) ) {
    //Transition not allowed.
    ukl1WarningMessage("from ready change not allowed.");
    return 2;
  }
  if ( (ukl1StartState == currentState) && ((ukl1StartState != newState) && (ukl1StopState != newState) && (ukl1ReadyState != newState)) ) {
    //Transition not allowed.
    ukl1WarningMessage("from start change not allowed.");
    return 2;
  }
  if ( (ukl1StopState == currentState) &&  ((ukl1StopState != newState) && (ukl1StartState != newState) && (ukl1ReadyState != newState)) ) {
    //Transition not allowed.
    ukl1WarningMessage("from stop change not allowed.");
    return 2;
  }

  //We have a valid transition, now we must change to the appropriate state and check that the change was successful.
  //Return with the appropriate return code, after logging an error.
  if (ukl1ReadyState == newState) {
    int changeStatus = ukl1Ready();
    if (0 == changeStatus) {
      ukl1CcpcList[ukl1CcpcName] = newState;
    } else {
      ukl1ErrorMessage("Failed to change to ready state.");
      //The state change was unsuccessful and we no longer know how many register were changed,
      //hence state is now not ready.
      ukl1CcpcList[ukl1CcpcName] = ukl1UnconfiguredState;
    }
    return changeStatus;
  }
  if (ukl1StartState == newState) {
    int changeStatus = ukl1Start();
    if (0 == changeStatus) {
      ukl1CcpcList[ukl1CcpcName] = newState;
    } else {
      ukl1ErrorMessage("Failed to change to start state.");
      //The state change was unsuccessful and we no longer know how many register were changed,
      //hence state is now not ready.
      ukl1CcpcList[ukl1CcpcName] = ukl1UnconfiguredState;
    }
    return changeStatus;
  }
  if (ukl1StopState == newState) {
    int changeStatus = ukl1Stop();
    if (0 == changeStatus) {
      ukl1CcpcList[ukl1CcpcName] = newState;
    } else {
      ukl1ErrorMessage("Failed to change to stop state.");
      //The state change was unsuccessful and we no longer know how many register were changed,
      //hence state is now not ready.
      ukl1CcpcList[ukl1CcpcName] = ukl1UnconfiguredState;
    }
    return changeStatus;
  }

  //If we get to here it means that an invalid state was requested, as we haven't returned anything yet.
  //Hence we must return the appropriate code to signal that.
  ukl1WarningMessage("Unrecognised state \'" + newState + "\' attempted to be selected.");
  return 3;
}

/*!
 * Gets the state of the current CCPC.
 *
 * \return string Name of the state that board is currently in.
 *
 * The board that will be addressed is that selected by the global variable ukl1CcpcName.
 */
string ukl1GetCcpcState() {
  return ukl1CcpcList[ukl1CcpcName];
}

/*!
 * Sets the board into the ready state. In this state all registers have been configured, but the
 * input channels and front end FPGAs are left in disabled state.
 *
 * \return int Defines the state of the function after execution. The states are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1Ready() {
  //First stop all data coming into the board while we move to the ready state.
  int stopStatus = ukl1Stop();
  ukl1StatusMessage("Data throughput disabling return status: " + stopStatus);
  //Check for errors.
  if (0 != stopStatus) {
    ukl1ErrorMessage("Failed to disable inputs and outputs.");
    return stopStatus;
  }

  //Configure the FPGAs.
  int fpgaStatus = ukl1FpgaConfigureNormalMode();
  ukl1StatusMessage("FPGA config return status: " + fpgaStatus);
  //Check for errors.
  if (0 != fpgaStatus) {
    ukl1ErrorMessage("Failed to configure FPGAs.");
    return fpgaStatus;
  }

  //Configure the gigabit ethernet.
  int gbeStatus = ukl1GbeConfigureNormalMode();
  ukl1StatusMessage("GBE config return status: " + gbeStatus);
  //Check for errors.
  if (0 != gbeStatus) {
    ukl1ErrorMessage("Failed to configure GBE.");
    return gbeStatus;
  }

  //Configure the TTCrx.
  int ttcrxStatus = ukl1TtcrxConfigureNormalMode();
  ukl1StatusMessage("TTCrx config return status: " + ttcrxStatus);
  if (0 != gbeStatus) {
    ukl1ErrorMessage("Failed to configure TTCrx.");
  }
  //TTCrx status will define the final return status of this function,
  //as we would have returned aready if the otheres were bad.
  return ttcrxStatus;
}

/*!
 * Sets the board into the start state. The input channels and front end FPGAs will be set to their appropriate state.
 *
 * \return int Defines the state of the function after execution. The states are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *
 * This must be called only when the board is in a configured state i.e. ready or stopped. Behaviour is undefined otherwise.
 */
int ukl1Start() {
  //Set channel states.
  int chanStatus = ukl1ConfigureAllChannelEnableSettings();
  //Check for errors.
  if (0 != chanStatus) {
    ukl1ErrorMessage("Failed to start channels.");
    return chanStatus;
  }
  //Set FPGA states.
  int fpgaStatus = ukl1ConfigureAllFpgaEnableSettings();
  //Check for errors.
  if (0 != fpgaStatus) {
    ukl1ErrorMessage("Failed to start FPGAs.");
    return fpgaStatus;
  }
  //Set GBE ports states.
  int portStatus = ukl1ConfigureAllGbePortEnableSettings();
  //GBE port status will define the final return status of this function.
  if (0 != fpgaStatus) {
    ukl1ErrorMessage("Failed to start GBE ports.");
  }
  return portStatus;
}


/*!
 * Stops the board from taking data by disabling all front end FPGAs and input channels.
 *
 * \return int Defines the state of the function after execution. The states are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *
 * This must be called only when the board is in a configured state i.e. ready or stopped. Behaviour is undefined otherwise.
 */
int ukl1Stop() {
  //Disable all the channels.
  int chanStatus = ukl1DisableAllChannels();
  //Check for errors.
  if (0 != chanStatus) {
    ukl1ErrorMessage("Failed to stop channels.");
    return chanStatus;
  }
  //Disable all the FPGAs.
  int fpgaStatus = ukl1DisableAllFpgas();
  //Check for errors.
  if (0 != fpgaStatus) {
    ukl1ErrorMessage("Failed to stop FPGAs.");
    return fpgaStatus;
  }
  //Disable all the ports.
  int portStatus = ukl1DisableAllGbePorts();
  //GBE port status will define the final return status of this function.
  if (0 != fpgaStatus) {
    ukl1ErrorMessage("Failed to stop GBE ports.");
  }
  return portStatus;
}


//@}

// ======================================
//  CONFIGURATION FUNCTIONS
// ======================================

/** @defgroup SectionConfiguration Configuration functions.
 *  These functions configure the various settings on the board at a varying level of granuality.
 *  @{
 */

/*!
 * Configures the FPGAs for running in the standard configuration mode.
 *
 * \return int Defines the state of the function after execution. The states are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *
 * This configures all settings, except for the channel and FPGA enable/disable settings, putting the FPGA
 * in the ready state.
 */
int ukl1FpgaConfigureNormalMode() {
  //First configure the run mode.
  int callStatus = ukl1FpgaConfigureRunMode();
  //If error return.
  if (0 != callStatus) {
    delay(1);
    return callStatus;
  }
  //Configure the MEP settings.
  callStatus = ukl1MepEventBuilderConfigure();
  //If error return.
  if (0 != callStatus) {
    delay(1);
    return callStatus;
  }
  //Now configure the other general settings.
  callStatus = ukl1FpgaConfigureGeneralSettings();
  //Last call, just return.
  return callStatus;
}

/*!
 * Configures the GBE card for running in the standard configuration mode.
 *
 * \return int Defines the state of the function after execution. The states are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *
 * This configures all settings, except for the port enable/disable settings, putting the GBE
 * in the ready state.
 */
int ukl1GbeConfigureNormalMode() {
  //Sets all the register settings that never need to be changed and are just hardcoded into library.
  int callStatus = ukl1GbeConfigureHardcodedSettings();
  //If error return.
  if (0 != callStatus) {
    return callStatus;
  }

  //Set up the settings for each of the ports.
  for (unsigned port = 0; port < UKL1_GBE_PORTS; ++port) {
    //Configure Tx FIFO thresholds.
    callStatus = ukl1GbeConfigureThresholds(port);
    //Delaying for the shortest time possible.
    delay(0,1);
    //If error return.
    if (0 != callStatus) {
      return callStatus;
    }

    //Configure Tx FIFO high watermarks.
    callStatus = ukl1GbeConfigureHighWatermarks(port);
    //Delaying for the shortest time possible.
    delay(0,1);
    //If error return.
    if (0 != callStatus) {
      return callStatus;
    }

    //Configure Tx FIFO low watermarks.
    callStatus = ukl1GbeConfigureLowWatermarks(port);
    //Delaying for the shortest time possible.
    delay(0,1);
    //If error return.
    if (0 != callStatus) {
      return callStatus;
    }
  }

  //Set the destination ethernet adaptor's MAC address.
  callStatus = ukl1GbeConfigureDestEthernetMacAddress();
  //If error return.
  if (0 != callStatus) {
    return callStatus;
  }
  //Set the source ethernet adaptor's MAC address.
  callStatus = ukl1GbeConfigureSourceEthernetMacAddress();
  //If error return.
  if (0 != callStatus) {
    return callStatus;
  }
  //Set the destination ethernet adaptor's IP address.
  callStatus = ukl1GbeConfigureDestEthernetIpAddress();
  //If error return.
  if (0 != callStatus) {
    return callStatus;
  }
  //Set the source ethernet adaptor's IP address.
  callStatus = ukl1GbeConfigureSourceEthernetIpAddress();
  //Just return, that was the last function call.
  return callStatus;
}

/*!
 * Configures the TTCrx card for running in the standard configuration mode.
 *
 * \return int Defines the state of the function after execution. The states are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *
 * Changes only the value of the 
 */
int ukl1TtcrxConfigureNormalMode() {
  //Set the TTCrx register that we wish to write to.
  int callStatus = ukl1TtcrxWrite(0x03, 0xa1, "The TTCrx register.", UKL1_VERIFY_WRITE);
  //callStatus now contains the appropriate error code.
  return callStatus;

}

// ==============================
//  FPGA CONFIGURATION FUNCTIONS
// ==============================

/*!
 * Performs a reset of the FPGAs on the L1 board. It will reset all values back to their defaults.
 *
 * \return int Defines the state of the function after execution. The states are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1ResetFpgas() {
  //The reset is performed by setting the GPIO line 5 low for a few 10s of milliseconds and then setting it back to high.
  //First it must be enabled for output.
  //Note: we could check that it is enabled for output, but it would always require at least 1 hardware
  //  access, so that 1 access might as well be the setting.
//   int callStatus = ukl1GpioEnable(FWCCPC_GPIO_LINE_5, FWCCPC_GPIO_OUTPUT, "Set GPIO output in reset");
//   //Check for errors.
//   if (0 != callStatus) {
//     ukl1WarningMessage("Failed to set GPIO line 5 to output during reset.");
//     return callStatus;
//   }
  //Now we set the GPIO line to be low to active the reset.
  int callStatus = ukl1GpioSet(FWCCPC_GPIO_LINE_5, FWCCPC_GPIO_LOW, "Set GPIO low in reset");
  //Check for errors.
  if (0 != callStatus) {
    ukl1WarningMessage("Failed to set GPIO line 5 to low during reset.");
    return callStatus;
  }
  //Now wait to allow reset to take affect.
  delay(0,20);
  //Now return it to the high state.
  callStatus = ukl1GpioSet(FWCCPC_GPIO_LINE_5, FWCCPC_GPIO_HIGH, "Set GPIO high in reset");
  //Check for errors.
  if (0 != callStatus) {
    ukl1WarningMessage("Failed to set GPIO line 5 to high during reset.");
  }
  //Now return.
  return callStatus;
}

/*!
 * Performs a reload of the FPGA firmware on the L1 board. It will reset all values back to their defaults.
 *
 * \return int Defines the state of the function after execution. The states are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1ReloadFpgas() {
  //The reload is performed by setting the GPIO line 6 low for a few 10s of milliseconds and then setting it back to high.
  //First it must be enabled for output.
  //Note: we could check that it is enabled for output, but it would always require at least 1 hardware
  //  access, so that 1 access might as well be the setting.
  int callStatus = ukl1GpioEnable(FWCCPC_GPIO_LINE_6, FWCCPC_GPIO_OUTPUT, "Set GPIO output in reload");
  //Check for errors.
  if (0 != callStatus) {
    ukl1WarningMessage("Failed to set GPIO line 6 to output during reload.");
    return callStatus;
  }
  //Now we set the GPIO line to be low to active the reload.
  callStatus = ukl1GpioSet(FWCCPC_GPIO_LINE_6, FWCCPC_GPIO_LOW, "Set GPIO low in reload");
  //Check for errors.
  if (0 != callStatus) {
    ukl1WarningMessage("Failed to set GPIO line 6 to low during reload.");
    return callStatus;
  }
  //Now wait to allow reset to take affect.
  delay(0,20);
  //Now return it to the high state.
  callStatus = ukl1GpioSet(FWCCPC_GPIO_LINE_6, FWCCPC_GPIO_HIGH, "Set GPIO high in reload");
  //Check for errors.
  if (0 != callStatus) {
    ukl1WarningMessage("Failed to set GPIO line 6 to high during reload.");
  }
  //Now return.
  return callStatus;
}

/*!
 * Configures the FPGAs for running with zero suppression enabled or disabled and
 * the mode that the HPD are running in, ALICE or LHCb.
 *
 * \return int Defines the state of the function after execution. The states are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1FpgaConfigureRunMode() {
  //Holds some of the values.
  unsigned uValue = 0;
  unsigned fpga0Bit = 0;
  unsigned fpga1Bit = 0;
  unsigned fpga2Bit = 0;
  unsigned fpga3Bit = 0;
  bool bValue = FALSE;
  char cValue = 0;
  dyn_char dcValue;

  //First set the 'run mode' settings for the board.
  //Determine the number of bits to be shifted by.
  //Record the return state and bit shift the value so as not to lose any previous error state.
  int pvssErr = ukl1DpGet("RichL1" + ukl1CcpcName + ".Fpga0.bit", fpga0Bit);
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".Fpga1.bit", fpga1Bit) << 1;
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".Fpga2.bit", fpga2Bit) << 2;
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".Fpga3.bit", fpga3Bit) << 3;
  if (0 != pvssErr) {
    //Perform an error check here and exit if there is a problem.
    return -1;
  }

  //Zero suppression settings.
  pvssErr = ukl1DpGet("RichL1" + ukl1CcpcName + ".Fpga0.RunMode.zeroSuppressed", bValue);
  if (TRUE == bValue) {
    cValue |= 1<<fpga0Bit;
  }
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".Fpga1.RunMode.zeroSuppressed", bValue) << 1;
  if (TRUE == bValue) {
    cValue |= 1<<fpga1Bit;
  }
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".Fpga2.RunMode.zeroSuppressed", bValue) << 2;
  if (TRUE == bValue) {
    cValue |= 1<<fpga2Bit;
  }
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".Fpga3.RunMode.zeroSuppressed", bValue) << 3;
  if (TRUE == bValue) {
    cValue |= 1<<fpga3Bit;
  }
  if (0 != pvssErr) {
    //Perform an error check here and exit if there is a problem.
    return -1;
  }

  //HPD setting. Here we use an inverted logic to the L1 board.
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".Fpga0.RunMode.lhcb", bValue);
  if (FALSE == bValue) {
    cValue |= 1<<(fpga0Bit+4);
  }
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".Fpga1.RunMode.lhcb", bValue) << 1;
  if (FALSE == bValue) {
    cValue |= 1<<(fpga1Bit+4);
  }
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".Fpga2.RunMode.lhcb", bValue) << 2;
  if (FALSE == bValue) {
    cValue |= 1<<(fpga2Bit+4);
  }
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".Fpga3.RunMode.lhcb", bValue) << 3;
  if (FALSE == bValue) {
    cValue |= 1<<(fpga3Bit+4);
  }
  if (0 != pvssErr) {
    //Perform an error check here and exit if there is a problem.
    return -1;
  }

  //Now make the array to write the values.
  dcValue = makeDynChar(0x00,0x00,0x00,cValue);
  int fpgaErr = ukl1FpgaWrite(19, dcValue, "LHCb and zero suppression register.", UKL1_VERIFY_WRITE);
  //Our return status depends now simply on the return value of the FPGA write.
  return fpgaErr;
}

/*!
 * Configures the MEP event builder settings on the L1 board.
 *
 * \return int Defines the state of the function after execution. The states are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1MepEventBuilderConfigure() {
  //Holds some of the values.
  unsigned uValue = 0;
  bool bValue = FALSE;
  dyn_char dcValue;
  unsigned bit0 = 0;
  unsigned bit1 = 0;
  unsigned bit2 = 0;
  unsigned bit3 = 0;

  //The following settings are written to multiple bits on register 15.
  //First the bits that the commands belong in are found.
  int pvssErr = ukl1DpGet("RichL1" + ukl1CcpcName + ".MepBuilder.ForceFixMepDestAddr.bits", bit0);
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".MepBuilder.DontWaitForDestBroadcast.bits", bit1) << 1;
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".MepBuilder.EnableThrottleOutput.bits", bit2) << 2;
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".MepBuilder.ThrottlePolarity.bits", bit3) << 3;
  if (0 != pvssErr) {
    //Perform an error check here and exit if there is a problem.
    return -1;
  }

  //Now determine the settings for each bit.
  pvssErr = ukl1DpGet("RichL1" + ukl1CcpcName + ".MepBuilder.ForceFixMepDestAddr.fixedDestAddr", bValue);
  if (TRUE == bValue) {
    uValue |= 1<<bit0;
  }
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".MepBuilder.DontWaitForDestBroadcast.dontWait", bValue) << 1;
  if (TRUE == bValue) {
    uValue |= 1<<bit1;
  }
  //The next two settings must be set to two bits each in the register.
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".MepBuilder.EnableThrottleOutput.enabled", bValue) << 2;
  if (TRUE == bValue) {
    uValue |= 3<<bit2;
  }
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".MepBuilder.ThrottlePolarity.inverted", bValue) << 3;
  if (TRUE == bValue) {
    uValue |= 3<<bit3;
  }
  if (0 != pvssErr) {
    //Perform an error check here and exit if there is a problem.
    return -1;
  }

  //Write those settings to the appropriate register.
  dcValue = makeDynChar(0x00,0x00, 0x00, uValue&0x3f);
  int fwCcpcErr = ukl1FpgaWrite(15, dcValue, "throttle polarity", UKL1_VERIFY_WRITE);
  //Our return status now depends on the return value of the hardware write.
  if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else if (0 > fwCcpcErr) {
    //Otherwise the return values match mine.
    return fwCcpcErr;
  } else {
    //The final setting is a value written to a register.
    ukl1DpGet("RichL1" + ukl1CcpcName + ".MepBuilder.MepEventCount.value", uValue);
    dcValue = makeDynChar(0x00,0x00, (uValue>>8)&0xff, uValue&0xff);
    fwCcpcErr = ukl1FpgaWrite(18, dcValue, "MEP event count", UKL1_VERIFY_WRITE);

    //Our return status now depends on the return value of the hardware write.
    if (0 < fwCcpcErr) {
      //They only promise that the error will be greater than 1, whereas I promise it will be 1.
      return 1;
    } else {
      //Otherwise the return values match mine.
      return fwCcpcErr;
    }
  }
}

/*!
 * Configures the FPGAs and L1 board general settings that are specific only to an
 * L1 board and not to any of the individual FPGAs or channels.
 *
 * \return int Defines the state of the function after execution. The states are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1FpgaConfigureGeneralSettings() {
  //Some variables to hold the values read from the data points.
  unsigned uValue = 0;
  dyn_char dcValue;

  int pvssErr = ukl1DpGet("RichL1" + ukl1CcpcName + ".ProtocolType.value", uValue);
  dcValue = makeDynChar(0x00,0x00, (uValue>>8)&0xff, uValue&0xff);
  int fwCcpcErr = ukl1FpgaWrite(10, dcValue, "protocol type", UKL1_VERIFY_WRITE);
  //Return if we saw a problem with either read from dp or write to hardware.
  if (0 > fwCcpcErr || 0 > pvssErr) {
    //Blame software before hardware, software is easier to debug.
    return -1;
  } else if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Don't do anything, just move on.
  }

  pvssErr = ukl1DpGet("RichL1" + ukl1CcpcName + ".TypeOfService.value", uValue);
  dcValue = makeDynChar(0x00,0x00, (uValue>>8)&0xff, uValue&0xff);
  fwCcpcErr = ukl1FpgaWrite(11, dcValue, "type of service", UKL1_VERIFY_WRITE);
  //Return if we saw a problem with either read from dp or write to hardware.
  if (0 > fwCcpcErr || 0 > pvssErr) {
    //Blame software before hardware, software is easier to debug.
    return -1;
  } else if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Don't do anything, just move on.
  }

  pvssErr = ukl1DpGet("RichL1" + ukl1CcpcName + ".TimeToLive.value", uValue);
  dcValue = makeDynChar(0x00,0x00, (uValue>>8)&0xff, uValue&0xff);
  fwCcpcErr = ukl1FpgaWrite(12, dcValue, "time to live", UKL1_VERIFY_WRITE);
  //Return if we saw a problem with either read from dp or write to hardware.
  if (0 > fwCcpcErr || 0 > pvssErr) {
    //Blame software before hardware, software is easier to debug.
    return -1;
  } else if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Don't do anything, just move on.
  }

  pvssErr = ukl1DpGet("RichL1" + ukl1CcpcName + ".Type.value", uValue);
  dcValue = makeDynChar(0x00,0x00, (uValue>>8)&0xff, uValue&0xff);
  fwCcpcErr = ukl1FpgaWrite(13, dcValue, "type", UKL1_VERIFY_WRITE);
  //Return if we saw a problem with either read from dp or write to hardware.
  if (0 > fwCcpcErr || 0 > pvssErr) {
    //Blame software before hardware, software is easier to debug.
    return -1;
  } else if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Don't do anything, just move on.
  }

  pvssErr = ukl1DpGet("RichL1" + ukl1CcpcName + ".Version.value", uValue);
  dcValue = makeDynChar(0x00,0x00, (uValue>>8)&0xff, uValue&0xff);
  fwCcpcErr = ukl1FpgaWrite(14, dcValue, "version", UKL1_VERIFY_WRITE);
  //Return if we saw a problem with either read from dp or write to hardware.
  if (0 > fwCcpcErr || 0 > pvssErr) {
    //Blame software before hardware, software is easier to debug.
    return -1;
  } else if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Don't do anything, just move on.
  }

  pvssErr = ukl1DpGet("RichL1" + ukl1CcpcName + ".PartitionId.value.bits0to15", uValue);
  dcValue = makeDynChar(0x00,0x00, (uValue>>8)&0xff, uValue&0xff);
  fwCcpcErr = ukl1FpgaWrite(16, dcValue, "partition ID bits 0 to 15", UKL1_VERIFY_WRITE);
  //Return if we saw a problem with either read from dp or write to hardware.
  if (0 > fwCcpcErr || 0 > pvssErr) {
    //Blame software before hardware, software is easier to debug.
    return -1;
  } else if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Don't do anything, just move on.
  }

  pvssErr = ukl1DpGet("RichL1" + ukl1CcpcName + ".PartitionId.value.bits16to31", uValue);
  dcValue = makeDynChar(0x00,0x00, (uValue>>8)&0xff, uValue&0xff);
  fwCcpcErr = ukl1FpgaWrite(17, dcValue, "partition ID bits 16 to 31", UKL1_VERIFY_WRITE);
  //Return if we saw a problem with either read from dp or write to hardware.
  if (0 > fwCcpcErr || 0 > pvssErr) {
    //Blame software before hardware, software is easier to debug.
    return -1;
  } else if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Don't do anything, just move on.
  }

  pvssErr = ukl1DpGet("RichL1" + ukl1CcpcName + ".BunchCountPresetValue.value", uValue);
  dcValue = makeDynChar(0x00,0x00, (uValue>>8)&0xff, uValue&0xff);
  fwCcpcErr = ukl1FpgaWrite(31, dcValue, "bunch count preset value", UKL1_VERIFY_WRITE);
  //Return if we saw a problem with either read from dp or write to hardware.
  if (0 > fwCcpcErr || 0 > pvssErr) {
    //Blame software before hardware, software is easier to debug.
    return -1;
  }
  if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  }

  //If we haven't already, just return.
  return fwCcpcErr;

}


// =============================
//  GBE CONFIGURATION FUNCTIONS
// =============================

/*!
 * Configures all the settings on the GBE that are never changed by the User and all values are hardcoded into this function.
 *
 * \return int Returns the execution status of the function. The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *
 * The values set here are defined by the LHCb technical note Quad Gigabit Ethernet plug-in card and are correct as of
 * issue 2.2 revision 0 of that document.
 */
int ukl1GbeConfigureHardcodedSettings() {
  //Used to keep track of the error conditions. This needs to be upgraded.
  int writeStatus = 0;
  //The function will ignore all error conditions until the end,
  //make this more sophisticated so you can tell where an error occured.

  // Resets
  writeStatus = ukl1GbeWrite(0x505, makeDynChar(0x0f,0x00,0x00,0x00), "MAC in reset", UKL1_VERIFY_WRITE); // Place MAC in reset
  writeStatus = ukl1GbeWrite(0x700, makeDynChar(0x0f,0x00,0x0c,0x00), "SPI3 TX and RX in reset", UKL1_VERIFY_WRITE); // Place SPI3 TX and RX in reset
  writeStatus = ukl1GbeWrite(0x59e, makeDynChar(0x0f,0x00,0x00,0x00), "RX FIFO in reset", UKL1_VERIFY_WRITE); // Place RX FIFO in reset
  writeStatus = ukl1GbeWrite(0x620, makeDynChar(0x0f,0x00,0x00,0x00), "TX FIFO in reset", UKL1_VERIFY_WRITE); // Place TX FIFO in reset
  //Delaying for the shortest time possible.
  delay(0,1);

  // Disable all ports
  writeStatus = ukl1GbeWrite(0x500, makeDynChar(0x00,0x00,0x00,0x00), "disable all ports during hardcoded config", UKL1_VERIFY_WRITE);

  // Disable mode sampling clocks for all active channels
  writeStatus = ukl1GbeWrite(0x794, makeDynChar(0x00,0x00,0x00,0x00), "disable clocks", UKL1_VERIFY_WRITE);
  //Delaying for the shortest time possible.
  delay(0,1);

  // Select copper mode
  writeStatus = ukl1GbeWrite(0x501, makeDynChar(0x0f,0x00,0x00,0x00), "set copper mode", UKL1_VERIFY_WRITE);
  //Delaying for the shortest time possible.
  delay(0,1);
  
  writeStatus = ukl1GbeWrite(0x010, makeDynChar(0x02,0x00,0x00,0x00), "set GMII 1000MB 1st", UKL1_VERIFY_WRITE); // Set GMII 1000Mb mode
  writeStatus = ukl1GbeWrite(0x090, makeDynChar(0x02,0x00,0x00,0x00), "set GMII 1000MB 2nd", UKL1_VERIFY_WRITE);
  writeStatus = ukl1GbeWrite(0x110, makeDynChar(0x02,0x00,0x00,0x00), "set GMII 1000MB 3rd", UKL1_VERIFY_WRITE);
  writeStatus = ukl1GbeWrite(0x190, makeDynChar(0x02,0x00,0x00,0x00), "set GMII 1000MB 4th", UKL1_VERIFY_WRITE);
  //Delaying for the shortest time possible.
  delay(0,1);

  writeStatus = ukl1GbeWrite(0x002, makeDynChar(0x01,0x00,0x00,0x00), "set full duplex mode 1st", UKL1_VERIFY_WRITE); // Set full duplex
  writeStatus = ukl1GbeWrite(0x082, makeDynChar(0x01,0x00,0x00,0x00), "set full duplex mode 2st", UKL1_VERIFY_WRITE);
  writeStatus = ukl1GbeWrite(0x102, makeDynChar(0x01,0x00,0x00,0x00), "set full duplex mode 3st", UKL1_VERIFY_WRITE);
  writeStatus = ukl1GbeWrite(0x182, makeDynChar(0x01,0x00,0x00,0x00), "set full duplex mode 4st", UKL1_VERIFY_WRITE);

  // Enable clocks for all active channels
  writeStatus = ukl1GbeWrite(0x794, makeDynChar(0x0f,0x00,0x00,0x00), "enable clocks", UKL1_VERIFY_WRITE);
  //Delaying for the shortest time possible.
  delay(0,1);

  // Remove resets
  writeStatus = ukl1GbeWrite(0x505, makeDynChar(0x00,0x00,0x00,0x00), "remove MAC reset", UKL1_VERIFY_WRITE);
  writeStatus = ukl1GbeWrite(0x700, makeDynChar(0x0f,0x00,0x00,0x00), "remove SPI3 TX and RX reset", UKL1_VERIFY_WRITE);
  writeStatus = ukl1GbeWrite(0x59e, makeDynChar(0x00,0x00,0x00,0x00), "remove RX FIFO reset", UKL1_VERIFY_WRITE);
  writeStatus = ukl1GbeWrite(0x620, makeDynChar(0x00,0x00,0x00,0x00), "remove TX FIFO reset", UKL1_VERIFY_WRITE);
  //Delaying for the shortest time possible.
  delay(0,1);

  // Enable automatic padding and CRC generation for all active channels
  writeStatus = ukl1GbeWrite(0x018, makeDynChar(0x4d,0x11,0x00,0x00), "enable auto padding and CRC generation 1st", UKL1_VERIFY_WRITE);
  writeStatus = ukl1GbeWrite(0x098, makeDynChar(0x4d,0x11,0x00,0x00), "enable auto padding and CRC generation 2st", UKL1_VERIFY_WRITE);
  writeStatus = ukl1GbeWrite(0x118, makeDynChar(0x4d,0x11,0x00,0x00), "enable auto padding and CRC generation 3st", UKL1_VERIFY_WRITE);
  writeStatus = ukl1GbeWrite(0x198, makeDynChar(0x4d,0x11,0x00,0x00), "enable auto padding and CRC generation 4st", UKL1_VERIFY_WRITE);
  //Delaying for the shortest time possible.
  delay(0,1);

  // Setup Rx interface: configure SPI3 width 32 bit, RVAL pause and active channel
  writeStatus = ukl1GbeWrite(0x701, makeDynChar(0x80,0xff,0xff,0x00), "set SPI3 width to 32 bit, RVAL pause and active channel", UKL1_VERIFY_WRITE);
  //Delaying for the shortest time possible.
  delay(0,1);

  // Setup Rx interface: configure RX CRC stripping and check
  writeStatus = ukl1GbeWrite(0x5b3, makeDynChar(0xff,0x00,0x00,0x00), "config RX CRC stripping and check", UKL1_VERIFY_WRITE);
  //Delaying for the shortest time possible.
  delay(0,1);

  //Return the appropriate number.
  if (0 == writeStatus) {
    //Everything was successful.
    return 0;
  } else if (-1 == writeStatus) {
    //PVSS error.
    return -1;
  } else if (writeStatus > 0) {
    //CCPC server error.
    return 1;
  }

}

/*!
 * Configures the Tx FIFO thresholds for a given GBE port number.
 *
 * \param  port The port number that the thresholds are to be written to.
 * \return int Returns the execution status of the function. The possible values are as follows:
 *   \li  0 Execution proceeded without a problem and thresholds are now configured.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1GbeConfigureThresholds(unsigned port) {
  //Create the data point depending on what port we are given.
  string dpName;
  int pvssErr = sprintf(dpName, "RichL1" + ukl1CcpcName + ".GbeConfiguration.Port%u.TxFifoThreshold.value", port);
  //Not worth checking this, it causes more problems that it would detect.
//   if (58 != pvssErr) {
//     //If the data point name is not written correctly, determined by whether the appropriate number of variables
//     //were written into the string, then we must exit as there is no hope for successful completion.
//     ukl1WarningMessage("Error creating DP name in GBE thresholds, constructed name:");
//     ukl1WarningMessage(dpName);
//     return -1;
//   }

  // Retrieve them from the dp, but hardcode address to save on dp accesses.
  unsigned uValue;
  pvssErr = ukl1DpGet(dpName, uValue);
  if (0 != pvssErr) {
    //If we couldn't get the data point then return as there is no point trying to write.
    ukl1ErrorMessage("Error retrieving value from data point");
    return -1;
  }
  //Actually this is limited to a 16bit number, but it doesn't hurt to write everything.
  dyn_char dcValue = makeDynChar( (uValue) & 0xff, (uValue>>8) & 0xff, (uValue>>16) & 0xff, (uValue) & 0xff );
  int fwCcpcErr = ukl1GbeWrite(1556, 1, dcValue, "config GBE thresholds", UKL1_VERIFY_WRITE);
  //Our return status now depends on the return value of the hardware write.
  if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Otherwise the return values match mine.
    return fwCcpcErr;
  }

}

/*!
 * Configures the Tx FIFO low watermarks for a given GBE port number.
 *
 * \param  port The port number that the low watermarks are to be written to.
 * \return int Returns the execution status of the function. The possible values are as follows:
 *   \li  0 Execution proceeded without a problem and low watermarks are now configured.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1GbeConfigureLowWatermarks(unsigned port) {
  //Create the data point depending on what port we are given.
  string dpName;
  int pvssErr = sprintf(dpName, "RichL1" + ukl1CcpcName + ".GbeConfiguration.Port%u.TxFifoLowWatermark.value", port);
  //Not worth checking this, it causes more problems that it would detect.
//   if (61 != pvssErr) {
//     //If the data point name is not written correctly, determined by whether the appropriate number of variables
//     //were written into the string, then we must exit as there is no hope for successful completion.
//     ukl1WarningMessage("Error creating DP name in GBE thresholds, constructed name:");
//     ukl1WarningMessage(dpName);
//     return -1;
//   }

  // Retrieve them from the dp, but hardcode address to save on dp accesses.
  unsigned uValue;
  pvssErr = ukl1DpGet(dpName, uValue);
  if (0 != pvssErr) {
    //If we couldn't get the data point then return as there is no point trying to write.
    return -1;
  }
  //Actually this is limited to a 16bit number, but it doesn't hurt to write everything.
  dyn_char dcValue = makeDynChar( (uValue) & 0xff, (uValue>>8) & 0xff, (uValue>>16) & 0xff, (uValue) & 0xff );
  int fwCcpcErr = ukl1GbeWrite(1556, 1, dcValue, "config GBE low watermarks", UKL1_VERIFY_WRITE);
  //Our return status now depends on the return value of the hardware write.
  if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Otherwise the return values match mine.
    return fwCcpcErr;
  }

}

/*!
 * Configures the Tx FIFO high watermarks for a given GBE port number.
 *
 * \param  port The port number that the high watermarks are to be written to.
 * \return int Returns the execution status of the function. The possible values are as follows:
 *   \li  0 Execution proceeded without a problem and high watermarks are now configured.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1GbeConfigureHighWatermarks(unsigned port) {
  //Create the data point depending on what port we are given.
  string dpName;
  int pvssErr = sprintf(dpName, "RichL1" + ukl1CcpcName + ".GbeConfiguration.Port%u.TxFifoHighWatermark.value", port);
//   if (62 != pvssErr) {
//     //If the data point name is not written correctly, determined by whether the appropriate number of variables
//     //were written into the string, then we must exit as there is no hope for successful completion.
//     ukl1WarningMessage("Error creating DP name in GBE thresholds, constructed name:");
//     ukl1WarningMessage(dpName);
//     return -1;
//   }

  // Retrieve them from the dp, but hardcode address to save on dp accesses.
  unsigned uValue;
  pvssErr = ukl1DpGet(dpName, uValue);
  if (0 != pvssErr) {
    //If we couldn't get the data point then return as there is no point trying to write.
    return -1;
  }
  //Actually this is limited to a 16bit number, but it doesn't hurt to write everything.
  dyn_char dcValue = makeDynChar( (uValue) & 0xff, (uValue>>8) & 0xff, (uValue>>16) & 0xff, (uValue) & 0xff );
  int fwCcpcErr = ukl1GbeWrite(1556, 1, dcValue, "config GBE high watermarks", UKL1_VERIFY_WRITE);
  //Our return status now depends on the return value of the hardware write.
  if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Otherwise the return values match mine.
    return fwCcpcErr;
  }

}

/*!
 * This sets the destination ethernet adaptor's MAC address.
 *
 * \return int Returns the execution status of the function. The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1GbeConfigureDestEthernetMacAddress() {
  //Retrieve the settings for each of the bits from the database and then write them to CCPC.
  unsigned bits0to15 = 0;
  unsigned bits16to31 = 0;
  unsigned bits32to47 = 0;

  int pvssErr = ukl1DpGet("RichL1" + ukl1CcpcName + ".GbeConfiguration.Common.EthernetDestAddress.destAddress.bits0to15", bits0to15);
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".GbeConfiguration.Common.EthernetDestAddress.destAddress.bits16to31", bits16to31) << 1;
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".GbeConfiguration.Common.EthernetDestAddress.destAddress.bits32to47", bits32to47) << 2;
  if (0 != pvssErr) {
    //Problem retrieving values from the datapoints, no point in writing to CCPC server.
    return -1;
  }

  dyn_char dcValue = makeDynChar( 0x00, 0x00, (bits0to15>>8)&0xff, bits0to15&0xff );
  int fwCcpcErr = ukl1FpgaWrite(0x00, dcValue, "MAC dest address, bits 0 to 15", UKL1_VERIFY_WRITE);
  //Check for errors
  if (0 < fwCcpcErr) {
    return 1;
  }
  if (-1 == fwCcpcErr) {
    return -1;
  }

  dcValue = makeDynChar( 0x00, 0x00, (bits16to31>>8)&0xff, bits16to31 );
  fwCcpcErr = ukl1FpgaWrite(0x01, dcValue, "MAC dest address, bits 16 to 31", UKL1_VERIFY_WRITE);
  //Check for errors
  if (0 < fwCcpcErr) {
    return 1;
  }
  if (-1 == fwCcpcErr) {
    return -1;
  }

  dcValue = makeDynChar( 0x00, 0x00, (bits32to47>>8)&0xff, bits32to47&0xff );
  fwCcpcErr = ukl1FpgaWrite(0x02, dcValue, "MAC dest address, bits 32 to 47", UKL1_VERIFY_WRITE);
  //Our return status now depends on the return value of the hardware write.
  if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Otherwise the return values match mine.
    return fwCcpcErr;
  }
}

/*!
 * This sets the source ethernet adaptor's MAC address.
 *
 * \return int Returns the execution status of the function. The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1GbeConfigureSourceEthernetMacAddress() {
  //Retrieve the settings for each of the bits from the database and then write them to CCPC.
  unsigned bits0to15 = 0;
  unsigned bits16to31 = 0;
  unsigned bits32to47 = 0;
  int pvssErr = ukl1DpGet("RichL1" + ukl1CcpcName + ".GbeConfiguration.Common.EthernetSourceAddress.sourceAddress.bits0to15", bits0to15);
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".GbeConfiguration.Common.EthernetSourceAddress.sourceAddress.bits16to31", bits16to31) << 1;
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".GbeConfiguration.Common.EthernetSourceAddress.sourceAddress.bits32to47", bits32to47) << 2;
  if (0 != pvssErr) {
    //Problem retrieving values from the datapoints, no point in writing to CCPC server.
    return -1;
  }

  dyn_char dcValue = makeDynChar( (bits0to15>>24)&0xff, (bits0to15>>16)&0xff, (bits0to15>>8)&0xff, bits0to15&0xff );
  int fwCcpcErr = ukl1FpgaWrite(0x03, dcValue, "MAC source address, bits 0 to 15", UKL1_VERIFY_WRITE);
  //Check for errors
  if (0 < fwCcpcErr) {
    return 1;
  }
  if (-1 == fwCcpcErr) {
    return -1;
  }

  dcValue = makeDynChar( (bits16to31>>24)&0xff, (bits16to31>>16)&0xff, (bits16to31>>8)&0xff, bits16to31 );
  fwCcpcErr = ukl1FpgaWrite(0x04, dcValue, "MAC source address, bits 16 to 31", UKL1_VERIFY_WRITE);
  //Check for errors
  if (0 < fwCcpcErr) {
    return 1;
  }
  if (-1 == fwCcpcErr) {
    return -1;
  }

  dcValue = makeDynChar( (bits32to47>>24)&0xff, (bits32to47>>16)&0xff, (bits32to47>>8)&0xff, bits32to47&0xff );
  fwCcpcErr = ukl1FpgaWrite(0x05, dcValue, "MAC source address, bits 32 to 47", UKL1_VERIFY_WRITE);

  //Our return status now depends on the return value of the hardware write.
  if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Otherwise the return values match mine.
    return fwCcpcErr;
  }
}

/*!
 * This sets the destination ethernet adaptor's IP address.
 *
 * \return int Returns the execution status of the function. The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1GbeConfigureDestEthernetIpAddress() {
  //Retrieve the settings for each of the bits from the database and then write them to CCPC.
  unsigned bits0to15 = 0;
  unsigned bits16to31 = 0;
  int pvssErr = ukl1DpGet("RichL1" + ukl1CcpcName + ".GbeConfiguration.Common.IpDestAddress.destAddress.bits0to15", bits0to15);
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".GbeConfiguration.Common.IpDestAddress.destAddress.bits16to31", bits16to31) << 1;
  if (0 != pvssErr) {
    //Problem retrieving values from the datapoints, no point in writing to CCPC server.
    return -1;
  }

  dyn_char dcValue = makeDynChar( 0x00, 0x00, (bits0to15>>8)&0xff, bits0to15&0xff );
  int fwCcpcErr = ukl1FpgaWrite(0x06, dcValue, "IP dest address, bits 0 to 15", UKL1_VERIFY_WRITE);
  //Check for errors
  if (0 < fwCcpcErr) {
    return 1;
  }
  if (-1 == fwCcpcErr) {
    return -1;
  }

  dcValue = makeDynChar( 0x00, 0x00, (bits16to31>>8)&0xff, bits16to31 );
  fwCcpcErr = ukl1FpgaWrite(0x07, dcValue, "IP dest address, bits 16 to 31", UKL1_VERIFY_WRITE);

  //Our return status now depends on the return value of the hardware write.
  if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Otherwise the return values match mine.
    return fwCcpcErr;
  }
}

/*!
 * This sets the source ethernet adaptor's IP address.
 *
 * \return int Returns the execution status of the function. The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1GbeConfigureSourceEthernetIpAddress() {
  //Retrieve the settings for each of the bits from the database and then write them to CCPC.
  unsigned bits0to15 = 0;
  unsigned bits16to31 = 0;
  int pvssErr = ukl1DpGet("RichL1" + ukl1CcpcName + ".GbeConfiguration.Common.IpSourceAddress.sourceAddress.bits0to15", bits0to15);
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".GbeConfiguration.Common.IpSourceAddress.sourceAddress.bits16to31", bits16to31) << 1;
  if (0 != pvssErr) {
    //Problem retrieving values from the datapoints, no point in writing to CCPC server.
    return -1;
  }

  dyn_char dcValue = makeDynChar( 0x00, 0x00, (bits0to15>>8)&0xff, bits0to15&0xff );
  int fwCcpcErr = ukl1FpgaWrite(0x08, dcValue, "IP source address bits 0 to 15", UKL1_VERIFY_WRITE);

  dcValue = makeDynChar( 0x00, 0x00, (bits16to31>>8)&0xff, bits16to31 );
  fwCcpcErr = ukl1FpgaWrite(0x09, dcValue, "IP source address bits 16 to 31", UKL1_VERIFY_WRITE);

  //Our return status now depends on the return value of the hardware write.
  if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Otherwise the return values match mine.
    return fwCcpcErr;
  }
}

// ============================================
//  ENABLING/DISABLING CONFIGURATION FUNCTIONS
// ============================================

/*!
 * This will configure the channel enable settings.
 *
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1ConfigureAllChannelEnableSettings() {
  //These variables will hold the relevant information from the data points.
  unsigned bit = 0;
  bool bValue = TRUE;
  dyn_char dcValue;
  string dataPointName;
  //Hold the function return status values.
  int pvssErr = 0;
  int fwCcpcErr = 0;
  //Loop over each FPGA.
  for (unsigned fpga = 0; fpga < UKL1_FRONT_FPGAS; ++fpga) {
    //This will hold the setting for each FPGA.
    unsigned uValue = 0;
    //Loop over each channel.
    for (unsigned channel = 0; channel < UKL1_FPGA_CHANNELS; ++channel) {
      //Determine the data point name.
      pvssErr = sprintf(dataPointName, "RichL1" + ukl1CcpcName + ".Fpga%uChannels.Channel%u.enabled", fpga, channel);
      //Not worth checking this, it causes more problems that it would detect.
//       //Check that the returned string is of the correct length.
//       if (44 != pvssErr) {
// 	ukl1WarningMessage("Error creating DP name in configuring FPGA channel's enable setting, constructed name:");
// 	ukl1WarningMessage(dpName);
// 	//Something went wrong with PVSS while creating the data point name.
// 	return -1;
//       }
      //Get the data point value.
      pvssErr = ukl1DpGet(dataPointName, bValue);
      if (0 != pvssErr) {
	//Problem retrieving the data point.
	return -1;
      }
      //Record the setting for later writing.
      if (!bValue) {
	uValue |= 1<<channel;
      }
    }
    //Write it to the appropriate address.
    dcValue = makeDynChar(0x00,0x00, (uValue>>8)&0xff, uValue&0xff);
    fwCcpcErr = ukl1FpgaWrite(25+fpga, dcValue, "enable channel settings", UKL1_VERIFY_WRITE);
    if (0 < fwCcpcErr) {
      //They only promise that the error will be greater than 1, whereas I promise it will be 1.
      return 1;
    }
    if (-1 == fwCcpcErr) {
      //Return -1 as there was a PVSS error.
      return -1;
    }
  }
  //If we made it to here then everything must be okay.
  return 0;

}

/*!
 * Sets all the front end FPGAs states to disabled.
 *
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1DisableAllChannels() {
  //This will hold any errors that we find.
  int fwCcpcErr = 0;
  //Loop over the four FPGAs and write 0 (disables all channels) to the appropriate register.
  for (unsigned fpga = 0; fpga < UKL1_FRONT_FPGAS; ++fpga) {
    fwCcpcErr = ukl1FpgaWrite(25+fpga, makeDynChar(0x00,0x00,0x01,0xff), "disabling all channels" , UKL1_VERIFY_WRITE);
    //If an error was found return, else carry on.
    if (0 < fwCcpcErr) {
      //They only promise that the error will be greater than 1, whereas I promise it will be 1.
      return 1;
    }
    if (-1 == fwCcpcErr) {
      //This indicates there was an error with PVSS.
      return -1;
    }
  }
  //If we made it to here then everything is good.
  return 0;
}

/*!
 * This will configure the FPGA enable settings.
 *
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1ConfigureAllFpgaEnableSettings() {

  //Some variables to hold the values read from the data points.
  unsigned uValue  = 0;
  bool bValue      = TRUE;
  unsigned bit0    = 0;
  unsigned bit1    = 1;
  unsigned bit2    = 2;
  unsigned bit3    = 3;
  dyn_char dcValue = 0;

  //Determine which bits the FPGA settings need be set for.
  int pvssErr = ukl1DpGet("RichL1" + ukl1CcpcName + ".Fpga0.bit", bit0);
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".Fpga1.bit", bit1) << 1;
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".Fpga2.bit", bit2) << 2;
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".Fpga3.bit", bit3) << 3;
  if (0 != pvssErr) {
    //Error getting data points, return a PVSS problem.
    return -1;
  }

  //Now get the FPGA settings and combine them to be written to the appropriate register.
  pvssErr = ukl1DpGet("RichL1" + ukl1CcpcName + ".Fpga0.Enabled.value", bValue);
  if (!bValue) {
    uValue |= 1<<bit0;
  }
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".Fpga1.Enabled.value", bValue) << 1;
  if (!bValue) {
    uValue |= 1<<bit1;
  }
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".Fpga2.Enabled.value", bValue) << 2;
  if (!bValue) {
    uValue |= 1<<bit2;
  }
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".Fpga3.Enabled.value", bValue) << 3;
  if (!bValue) {
    uValue |= 1<<bit3;
  }
  if (0 != pvssErr) {
    //Error getting data points, return a PVSS problem.
    return -1;
  }

  //Write them to the register.
  dcValue = makeDynChar(0x00,0x00, 0x00, uValue&0xff);
  int fwCcpcErr = ukl1FpgaWrite(24, dcValue, "set fpga enable settings", UKL1_VERIFY_WRITE);
  //Our return status now depends on the return value of the hardware write.
  if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Otherwise the return values match mine.
    return fwCcpcErr;
  }
}

/*!
 * Sets all the front end FPGAs states to disabled.
 *
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1DisableAllFpgas() {
  //Write a zero to the appropriate address to disable the FPGAs.
  int fwCcpcErr = ukl1FpgaWrite(24, makeDynChar(0x00,0x00,0x00,0x0f), "disable all FPGAs" , UKL1_VERIFY_WRITE);
  //Our return status now depends on the return value of the hardware write.
  if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Otherwise the return values match mine.
    return fwCcpcErr;
  }
}

/*!
 * This will configure the port settings on the GBE.
 *
 * \return int Returns an error code to indicate the execution status of the function.
 *           The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1ConfigureAllGbePortEnableSettings() {
  //Retrieve the status for each port and combine these into a single byte representing the total status.
  char cValue = 0;
  bool bValue;

  int pvssErr = ukl1DpGet("RichL1" + ukl1CcpcName + ".GbeConfiguration.Port0.enabled", bValue);
  if (bValue) {
    cValue |= 1;
  }
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".GbeConfiguration.Port1.enabled", bValue) << 1;
  if (bValue) {
    cValue |= 2;
  }
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".GbeConfiguration.Port2.enabled", bValue) << 2;
  if (bValue) {
    cValue |= 4;
  }
  pvssErr |= ukl1DpGet("RichL1" + ukl1CcpcName + ".GbeConfiguration.Port3.enabled", bValue) << 3;
  if (bValue) {
    cValue |= 8;
  }
  if (0 != pvssErr) {
    //There was a problem getting one of the data points, so there is no use writting to the hardware.
    return -1;
  }

  int fwCcpcErr = ukl1GbeWrite(0x500, makeDynChar(cValue, 0x00, 0x00, 0x00), "port enable settings", UKL1_VERIFY_WRITE);
  //Delaying for the shortest time possible, there is no point checking for an error here.
  //The act of calling the function is probably enough to ensure that it was delayed long enough.
  delay(0,1);
  //Our return status now depends on the return value of the hardware write.
  if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Otherwise the return values match mine.
    return fwCcpcErr;
  }
}

/*!
 * This will disable all the ports on the GBE.
 *
 * \return int Returns an error code to indicate the execution status of the function. The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1DisableAllGbePorts() {
  //Just write 0 to the register.
  int fwCcpcErr = ukl1GbeWrite(0x500, makeDynChar(0x00, 0x00, 0x00, 0x00), "disabling GBE ports", UKL1_VERIFY_WRITE);
  //Delaying for the shortest time possible, there is no point checking for an error here.
  //The act of calling the function is probably enough to ensure that it was delayed long enough.
  delay(0,1);
  //Our return status now depends on the return value of the hardware write.
  if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Otherwise the return values match mine.
    return fwCcpcErr;
  }
}

//@}


// ======================================
//  ERROR FUNCTIONS
// ======================================

/** @defgroup SectionError Error functions.
 *  These functions are used for logging errors that occur when the various read/write
 *  operations are performed.
 *  @{
 */

/**
 * Logs debug text to the UKL1 debug message log data point.
 *
 * \param  message Debug message that is to be sent to the data point.
 * \return unsigned Number of messages in the debug message data point.
 *
 * If a zero is returned this indicates that the message data point is still empty after
 * streaming the message. There must have been a problem accessing the data point and the
 * status message will have been set to the PVSS project log.
 */
unsigned ukl1StatusMessage(string message) {
  //First we must get the existing debug messages.
  dyn_string dbgMsg;
  int dpStatus = ukl1DpGet("RichL1Log.Status", dbgMsg);
  if (0 == dpStatus) {
    //Successfully accessed the data point.
    //Now append the new message to this.
    dynAppend(dbgMsg, message);
    //and write it back to the data point.
    dpStatus = dpSet("RichL1Log.Status", dbgMsg);
    //Now return the number of messages in the data point.
    return dynlen(dbgMsg);
  } else {
    DebugN("Failed to access the status data point:");
    DebugN("RichL1Log.Status");
    DebugN("Status messages diverted to PVSS log, the following message was recieved:");
    DebugN(message);
    //Return zero to indicate there was a problem and that we couldn't access the data point.
    return 0;
  }
}

/**
 * Logs a general warning that occurred while running the program.
 *
 * \param  message Warning message that is to be logged.
 * \return unsigned This is the index in the message map value that the message was written to.
 *
 * If a zero is returned this indicates that the message data point is still empty after
 * streaming the message. There must have been a problem accessing the data point and the
 * warning message will have been set to the PVSS project log.
 */
unsigned ukl1WarningMessage(string message) {
  //First we must get the existing error messages.
  dyn_string errMsg;
  int dpStatus = ukl1DpGet("RichL1Log.Warning", errMsg);
  if (0 == dpStatus) {
    //Successfully accessed the data point.
    //Now append the new message to this.
    dynAppend(errMsg, message);
    //and write it back to the data point.
    dpSet("RichL1Log.Warning", errMsg);
    //Now return the number of messages in the data point.
    return dynlen(errMsg);
  } else {
    DebugN("Failed to access the status data point:");
    DebugN("RichL1Log.Warning");
    DebugN("Warning messages diverted to PVSS log, the following message was recieved:");
    DebugN(message);
    //Return zero to indicate there was a problem and that we couldn't access the data point.
    return 0;
  }
}

/**
 * Logs a general critical error that occurred while running the program.
 *
 * \param  message Critical error message that is to be logged.
 * \return unsigned This is the index in the message map value that the message was written to.
 *
 * If a zero is returned this indicates that the message data point is still empty after
 * streaming the message. There must have been a problem accessing the data point and the
 * error message will have been set to the PVSS project log.
 */
unsigned ukl1ErrorMessage(string message) {
  //First we must get the existing error messages.
  dyn_string errMsg;
  int dpStatus = ukl1DpGet("RichL1Log.Error", errMsg);
  if (0 == dpStatus) {
    //Successfully accessed the data point.
    //Now append the new message to this.
    dynAppend(errMsg, message);
    //and write it back to the data point.
    dpSet("RichL1Log.Error", errMsg);
    //Now return the number of messages in the data point.
    return dynlen(errMsg);
  } else {
    DebugN("Failed to access the status data point:");
    DebugN("RichL1Log.Error");
    DebugN("Error messages diverted to PVSS log, the following message was recieved:");
    DebugN(message);
    //Return zero to indicate there was a problem and that we couldn't access the data point.
    return 0;
  }
}

/**
 * Logs an error that occurred with while using the framework library for writing/reading from the
 * CCPC.
 *
 * \param  messageLevel Level that the message is defined as.
 * \param  message Error message that is to be logged.
 * \return unsigned This is the index in the message map value that the message was written to.
 */
int ukl1LogFwCcpcError(string message) {
  //Append a note specifying it occurred within the framework and then log it as a critical error.
  string newMessage = "fwCcpc.ctl lib reported error: " + message;
  //Just call the appropriate logging function return its state.
  return ukl1ErrorMessage(newMessage);
}

/**
 * Logs an error that occurred with while retrieving/setting a data point.
 *
 * \param  messageLevel Level that the message is defined as.
 * \param  message Error message that is to be logged.
 * \return unsigned This is the index in the message map value that the message was written to.
 */
int ukl1LogDpError(string message) {
  //Append a note specifying it occurred within the framework and then log it as a critical error.
  string newMessage = "Data point error: " + message;
  //Just call the appropriate logging function return its state.
  return ukl1ErrorMessage(newMessage);
}

/**
 * Clears all the messages from the message log.
 *
 * \return int Returns an error code that indicates the execution status of the function. They are as follows:
 *   \li  0 Function completed successfully.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1ClearAllLogMessages() {
  //Check the return status for errors.
  //Write to the appropriate log if we failed to clear it (it will default to the PVSS log if there is a data point issue).
  //Return the appropriate state.
  if ( 0 != ukl1ClearStatusMessages() ) {
    ukl1StatusMessage("Failed to clear status log, due to a PVSS error.");
    return -1;
  }
  if ( 0 != ukl1ClearWarningMessages() ) {
    ukl1WarningMessage("Failed to clear warning log, due to a PVSS error.");
    return -1;
  }
  if ( 0 == ukl1ClearCriticalMessages() ) {
    return 0;
  } else {
    ukl1ErrorMessage("Failed to clear error log, due to a PVSS error.");
    return -1;
  }
}

/**
 * Clears all the status messages from the message log.
 *
 * \return int Returns an error code that indicates the execution status of the function. They are as follows:
 *   \li  0 Function completed successfully.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1ClearStatusMessages() {
  //To clear the messages just write an empty dyn_string to the data point.
  dyn_string emptyString = makeDynString();
  if ( 0 == dpSet("RichL1Log.Status", emptyString) ) {
    return 0;
  } else {
    return 1;
  }
}

/**
 * Clears all the warning messages from the message log.
 *
 * \return int Returns an error code that indicates the execution status of the function. They are as follows:
 *   \li  0 Function completed successfully.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1ClearWarningMessages() {
  //To clear the messages just write an empty dyn_string to the data point.
  dyn_string emptyString = makeDynString();
  if ( 0 == dpSet("RichL1Log.Warning", emptyString) ) {
    return 0;
  } else {
    return 1;
  }
}

/**
 * Clears all the critical messages from the message log.
 *
 * \return int Returns an error code that indicates the execution status of the function. They are as follows:
 *   \li  0 Function completed successfully.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1ClearErrorMessages() {
  //To clear the messages just write an empty dyn_string to the data point.
  dyn_string emptyString = makeDynString();
  if ( 0 == dpSet("RichL1Log.Error", emptyString) ) {
    return 0;
  } else {
    return 1;
  }
}

//@}


// ======================================
//  SUPPORT FUNCTIONS
// ======================================

/** @defgroup SectionSupport Support funtions
 *  List of constants that can be used by the user.
 *  @{
 */

/*!
 * This writes to a given register on the FPGAs.
 *
 * \param  address The address of the register on the FPGA to be accessed.
 * \param  data The data that is to be written to the FPGA register.
 * \param  regName Name of the register to be written to. This is used only for error logging and
 *           is defaulted to an empty string, in which case no logging will be performed.
 * \param  verifyWrite If this is set to TRUE then the register will be read back from to ensure that
 *           the data written corresponds to that read. Default: FALSE.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *
 * In addition to returning an error code the function will also write any information to the library error log, which
 * can be identified by the register name that is given as an argument. If no error occurs no error information is recorded.
 */
int ukl1FpgaWrite(unsigned address, dyn_char data, string regName="", bool verifyWrite=FALSE) {
  //Perform the write to the local bus, FPGAs, of the desired data using the fwCcpc library.
  //We will always be using 32-bit registers and write one 32-bit word of data.
  int callStatus = fwCcpc_LBUSWrite(ukl1CcpcName, ukl1FpgaizeAddress(address), FWCCPC_LBUS_BITS_32, 1, data);
  //Check for errors.
  if ( (0 != callStatus) && ("" != regName) ) {
    //Construct the error message.
    string errMsg = "While writing to \'"+regName+"\' FpgaWrite encountered";
    if (callStatus > 0) {
      errMsg += " a problem writing to the CCPC server.";
    } else if (callStatus < 0) {
      errMsg += " a problem with PVSS.";
    } else {
      errMsg += " an unknown error.";
    }
    //Now log the error.
    ukl1LogFwCcpcError(errMsg);
  }
//   if (verifyWrite) {
//     //We wish to check the values written to the L1 board are the same as those we read back.
//     //For now we will naively assume that if the write works then so will the read.
//     dyn_char checkData;
//     fwCcpc_LBUSRead(ukl1CcpcName, ukl1FpgaizeAddress(address), FWCCPC_LBUS_BITS_32, 1, checkData);
//     const int checkLen = dynlen(checkData);
//     const int dataLen  = dynlen(data);
//     //This will provide us with our loop index. It must be set to the shorter of the two arrays in order that
//     //we don't go out of bounds. Initialise it to either one and only change it if their lengths differ.
//     int shortestDataLen = dataLen;
//     if (checkLen  != dataLen) {
//       ukl1WarningMessage("check data and data lengths differ, when verifying FPGA write.");
//       ukl1WarningMessage("check data: " + checkLen + ", data: " + dataLen);
//       //Already set to dataLen, so just need to change if checkLen is shorter. 
//       if (checkLen < dataLen) {
// 	shortestDataLen = checkLen;
//       }
//     }
//     for (unsigned index = 1; index <= shortestDataLen; ++index) {
//       if ( checkData[index] != data[index] ) {
// 	ukl1WarningMessage("checkData ("+(unsigned)checkData[index]+") and data ("+(unsigned)data[index]+") differ in element "+index+".");
//       }
//     }
//   }

  //Return the call status to indicate whether we were successful or not.
  if (0 < callStatus) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Otherwise the return values match mine.
    return callStatus;
  }
}

/*!
 * This writes to a given register on the GBE controller.
 *
 * \param  address The address of the register on the GBE to be written to.
 * \param  data The data that is to be written to the GBE register.
 * \param  regName This is the name of the register that is to be written to. It is used for identification purposes
 *           in the error log and should be unique to the register. It is defaulted to an empty string if no logging
 *           is to be used.
 * \param  verifyWrite If this is set to TRUE then the register will be read back from to ensure that
 *           the data written corresponds to that read. Default: FALSE.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1GbeWrite(unsigned address, dyn_char data, string regName="", bool verifyWrite=FALSE) {
  //Perform the write to the GBE of the desired data using the fwCcpc library.
  //We will always be writing one 32-bit word of data.
  int callStatus = fwCcpc_GBEWrite(ukl1CcpcName, address, 1, data);
  //Check for errors.
  if ( (0 != callStatus) && ("" != regName) ) {
    //Construct the error message.
    string errMsg = "While writing to \'"+regName+"\' GbeWrite encountered";
    if (callStatus > 0) {
      errMsg += " a problem writing to the CCPC server.";
    } else if (callStatus < 0) {
      errMsg += " a problem with PVSS.";
    } else {
      errMsg += " an unknown error.";    return callStatus;

    }
    //Now log the error.
    ukl1LogFwCcpcError(errMsg);
  }
//   if (verifyWrite) {
//     //We wish to check the values written to the L1 board are the same as those we read back.
//     //For now we will naively assume that if the write works then so will the read.
//     dyn_char checkData;
//     fwCcpc_GBERead(ukl1CcpcName, address, 1, checkData);
//     const int checkLen = dynlen(checkData);
//     const int dataLen  = dynlen(data);
//     //This will provide us with our loop index. It must be set to the shorter of the two arrays in order that
//     //we don't go out of bounds. Initialise it to either one and only change it if their lengths differ.
//     int shortestDataLen = dataLen;
//     if (checkLen  != dataLen) {
//       ukl1WarningMessage("check data and data lengths differ, when verifying FPGA write.");
//       ukl1WarningMessage("check data: " + checkLen + ", data: " + dataLen);
//       //Already set to dataLen, so just need to change if checkLen is shorter. 
//       if (checkLen < dataLen) {
// 	shortestDataLen = checkLen;
//       }
//     }
//     for (unsigned index = 1; index <= shortestDataLen; ++index) {
//       if ( checkData[index] != data[index] ) {
// 	ukl1WarningMessage("checkData ("+(unsigned)checkData[index]+") and data ("+(unsigned)data[index]+") differ in element "+index+".");
//       }
//     }
//   }
  //Return the call status to indicate whether we were successful or not.
  if (0 < callStatus) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Otherwise the return values match mine.
    return callStatus;
  }
}

/*!
 * This writes to a given register on the TTCrx chip the desired data.
 *
 * \param  address The address of the register on the TTCrx chip you wish to access.
 * \param  data The data that is to be written to the TTCrx control register.
 * \param  regName This is the name of the register that is to be written to. It is used for identification purposes
 *           in the error log and should be unique to the register. It is defaulted to an empty string if no logging
 *           is to be used.
 * \param  verifyWrite If this is set to TRUE then the register will be read back from to ensure that
 *           the data written corresponds to that read. Default: FALSE.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *
 * This function calls both ukl1SetTtcrxRegister and ukl1SetTtcrxData in order to provide simple access to the registers
 * on the TTCrx and should be used in preference to the two functions ukl1SetTtcrxRegister and ukl1SetTtcrxData.
 */
int ukl1TtcrxWrite(unsigned address, dyn_char data, string regName = "", bool verifyWrite=FALSE) {
  //First set the address to write to.
  int callStatus = ukl1SetTtcrxRegister(address);
  //Check for errors.
  if ( (0 != callStatus) && ("" != regName) ) {
    //Construct the error message.
    string errMsg = "While setting up TTCrx to write to \'"+regName+"\' TTCrxWrite encountered";
    if (callStatus > 0) {
      errMsg += " a problem writing to the CCPC server.";
    } else if (callStatus < 0) {
      errMsg += " a problem with PVSS.";
    } else {
      //Technically this covers only the case zero, but as the outer if should not be called
      //if callStatus is zero to reach here is truely an unknown error.
      errMsg += " an unknown error.";
    }
    //Now log the error.
    ukl1LogFwCcpcError(errMsg);
    //If we don't set up this register properly then we should not write the data as it could over write any register on the board.
    if (0 < callStatus) {
      //They only promise that the error will be greater than 1, whereas I promise it will be 1.
      return 1;
    } else {
      //Otherwise the return values match mine.
      return callStatus;
    }
  }

  //Now write the data.
  callStatus = ukl1SetTtcrxData(data);
  //Check for errors.
  if ( (0 != callStatus) && ("" != regName) ) {
    //Construct the error message.
    string errMsg = "While writing to \'"+regName+"\' TTCrxRead encountered";
    if (callStatus > 0) {
      errMsg += " a problem writing to the CCPC server.";
    } else if (callStatus < 0) {
      errMsg += " a problem with PVSS.";
    } else {
      errMsg += " an unknown error.";
    }
    //Now log the error.
    ukl1LogFwCcpcError(errMsg);
  }
//   if (verifyWrite) {
//     //We wish to check the values written to the L1 board are the same as those we read back.
//     //For now we will naively assume that if the write works then so will the read.
//     dyn_char checkData;
//     //Address must be set up correctly due to above write.
//     ukl1GetTtcrxData(checkData);
//     const int checkLen = dynlen(checkData);
//     const int dataLen  = dynlen(data);
//     //This will provide us with our loop index. It must be set to the shorter of the two arrays in order that
//     //we don't go out of bounds. Initialise it to either one and only change it if their lengths differ.
//     int shortestDataLen = dataLen;
//     if (checkLen  != dataLen) {
//       ukl1WarningMessage("check data and data lengths differ, when verifying FPGA write.");
//       ukl1WarningMessage("check data: " + checkLen + ", data: " + dataLen);
//       //Already set to dataLen, so just need to change if checkLen is shorter. 
//       if (checkLen < dataLen) {
// 	shortestDataLen = checkLen;
//       }
//     }
//     for (unsigned index = 1; index <= shortestDataLen; ++index) {
//       if ( checkData[index] != data[index] ) {
// 	ukl1WarningMessage("checkData ("+(unsigned)checkData[index]+") and data ("+(unsigned)data[index]+") differ in element "+index+".");
//       }
//     }
//   }

  //Now return our final status.
  if (0 < callStatus) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Otherwise the return values match mine.
    return callStatus;
  }
}

/*!
 * This reads from a given register on the TTCrx chip.
 *
 * \param  address The address of the register on the TTCrx chip you wish to access.
 * \param  data The data that is to be read from the TTCrx control register.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *
 * This function calls both ukl1SetTtcrxRegister and ukl1GetTtcrxData in order to provide simple access to the registers
 * on the TTCrx and should be used in preference to the two functions ukl1SetTtcrxRegister and ukl1GetTtcrxData.
 */
int ukl1TtcrxRead(unsigned address, dyn_char& data, string regName="") {
  //First set the address to write to.
  int callStatus = ukl1SetTtcrxRegister(address);
  //Check for errors.
  if ( (0 != callStatus) && ("" != regName) ) {
    //Construct the error message.
    string errMsg = "While setting up TTCrx to write to \'"+regName+"\' TTCrxRead encountered";
    if (callStatus > 0) {
      errMsg += " a problem writing to the CCPC server.";
    } else if (callStatus < 0) {
      errMsg += " a problem with PVSS.";
    } else {
      errMsg += " an unknown error.";
    }
    //Now log the error.
    ukl1LogFwCcpcError(errMsg);
    //If we don't set up this register properly then we should not write the data as it could over write any register on the board.
    if (0 < callStatus) {
      //They only promise that the error will be greater than 1, whereas I promise it will be 1.
      return 1;
    } else {
      //Otherwise the return values match mine.
      return callStatus;
    }
  }

  //Now read the data.
  callStatus = ukl1GetTtcrxData(data);
  //Check for errors.
  if ( (0 != callStatus) && ("" != regName) ) {
    //Construct the error message.
    string errMsg = "While reading from \'"+regName+"\' TTCrxRead encountered";
    if (callStatus > 0) {
      errMsg += " a problem writing to the CCPC server.";
    } else if (callStatus < 0) {
      errMsg += " a problem with PVSS.";
    } else {
      errMsg += " an unknown error.";
    }
    //Now log the error.
    ukl1LogFwCcpcError(errMsg);
  }

  //Now return our final status.
  if (0 < callStatus) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Otherwise the return values match mine.
    return callStatus;
  }
}

/*!
 * This will set the TTCrx register that is to read/written from.
 *
 * \param  address The address of the register on the TTCrx chip you wish to access.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *
 * It is not possible to read back from this register. It will merely return the value stored at the register on the
 * TTCrx chip that has been selected.
 */
int ukl1SetTtcrxRegister(unsigned address) {
  //Write to address 0x42 on the I2C bus a value of 0x03.
  //The tells the TTCrx that the data written/read to the I2C address 0x43 should 
  //go to/come from the TTCrx address 0x03.
  //Some of the I2C properties that need to be written.
  const unsigned bus = 1;
  const unsigned dataSize = 1;
  int fwCcpcErr = fwCcpc_I2CWrite(ukl1CcpcName, bus, 0x42, dataSize, -1, FWCCPC_I2C_SEPARATED, TRUE, makeDynChar(address));
  //Our return status now depends on the return value of the hardware write.
  if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Otherwise the return values match mine.
    return fwCcpcErr;
  }
}

/*!
 * This will write to a TTCrx control register
 *
 * \param  data The data that is to be written to the TTCrx control register.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1SetTtcrxData(dyn_char data) {
  //Write to address 0x43 on the I2C bus.
  //The address on the TTCrx chip that this should be written to must have previously been
  //selected using ukl1SetTtcrxRegister.
  //Some of the I2C properties that need to be written.
  const unsigned bus = 1;
  const unsigned dataSize = dynlen(data);
  int fwCcpcErr = fwCcpc_I2CWrite(ukl1CcpcName, bus, 0x43, dataSize, -1, FWCCPC_I2C_SEPARATED, TRUE, makeDynChar(data));
  //Our return status now depends on the return value of the hardware write.
  if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Otherwise the return values match mine.
    return fwCcpcErr;
  }
}

/*!
 * This will read from a TTCrx control register
 *
 * \param  data The data that is to be read from the TTCrx control register.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 */
int ukl1GetTtcrxData(dyn_char& data) {
  //Read from address 0x43 on the I2C bus.
  //The address on the TTCrx chip that this should be read from must have previously been
  //selected using ukl1SetTtcrxRegister.
  //Some of the I2C properties that need to be written.
  const unsigned bus = 1;
  const unsigned dataSize = 1;
  int fwCcpcErr = fwCcpc_I2CRead(ukl1CcpcName, bus, 0x43, dataSize, -1, FWCCPC_I2C_SEPARATED, TRUE, data);
  //Our return status now depends on the return value of the hardware write.
  if (0 < fwCcpcErr) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Otherwise the return values match mine.
    return fwCcpcErr;
  }
}

/*!
 * This enables a GPIO line.
 *
 * \param  line Number of the GPIO line to be enabled. Range:0 to 8.
 * \param  output True if the line is to be enabled for output, false if for input.
 * \param  regName Name of the register to be written to. This is used only for error logging and
 *           is defaulted to an empty string, in which case no logging will be performed.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *
 * In addition to returning an error code the function will also write any information to the library error log, which
 * can be identified by the register name that is given as an argument. If no error occurs no error information is recorded.
 */
int ukl1GpioEnable(unsigned line, bool output, string regName="") {
  //Decide whether they wish to enable for output or input.
  //Set for input and then change if output.
  unsigned lineOutputs = FWCCPC_GPIO_INPUT;
  if (output) {
    lineOutputs = FWCCPC_GPIO_OUTPUT;
  }
  ukl1StatusMessage("output: "+output+" lineOutput: "+lineOutputs+" line: "+line);
  int callStatus = fwCcpc_GPIOEnable(ukl1CcpcName, line, lineOutputs);
  //Check for errors.
  if ( (0 != callStatus) && ("" != regName) ) {
    //Construct the error message.
    string errMsg = "While writing to \'"+regName+"\' GpioEnabled encountered";
    if (callStatus > 0) {
      errMsg += " a problem writing to the CCPC server.";
    } else if (callStatus < 0) {
      errMsg += " a problem with PVSS.";
    } else {
      errMsg += " an unknown error.";
    }
    //Now log the error.
    ukl1LogFwCcpcError(errMsg);
  }

  //Return the call status to indicate whether we were successful or not.
  if (0 < callStatus) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Otherwise the return values match mine.
    return callStatus;
  }
}

/*!
 * This sets the GPIO line to either high or low.
 *
 * \param  line Number of the GPIO line to be enabled. Range:0 to 8.
 * \param  high True if the line is to be set high and false if it is to be set low.
 * \param  regName Name of the register to be written to. This is used only for error logging and
 *           is defaulted to an empty string, in which case no logging will be performed.
 * \param  verifyWrite If this is set to TRUE then the register will be read back from to ensure that
 *           the data written corresponds to that read. Default: FALSE.
 * \return int Returns an error code to indicate the execution status of the function.
 *   The possible values are as follows:
 *   \li  0 Execution proceeded without a problem.
 *   \li  1 Execution failed and the board is in an unuseable state due to a problem with writing to the CCPC server.
 *   \li -1 Execution failed and the board is in an unuseable state due to a PVSS problem.
 *
 * In addition to returning an error code the function will also write any information to the library error log, which
 * can be identified by the register name that is given as an argument. If no error occurs no error information is recorded.
 */
int ukl1GpioSet(unsigned line, bool high, string regName="", bool verifyWrite=FALSE) {
  //Decide whether it is to be set high or low.
  //Set it to low and then change if it is high.
  unsigned lineHigh = FWCCPC_GPIO_LOW;
  if (high) {
    lineHigh = FWCCPC_GPIO_HIGH;
  }
  int callStatus = fwCcpc_GPIOSet(ukl1CcpcName, line, lineHigh);
  //Check for errors.
  if ( (0 != callStatus) && ("" != regName) ) {
    //Construct the error message.
    string errMsg = "While writing to \'"+regName+"\' GpioSet encountered";
    if (callStatus > 0) {
      errMsg += " a problem writing to the CCPC server.";
    } else if (callStatus < 0) {
      errMsg += " a problem with PVSS.";
    } else {
      errMsg += " an unknown error.";
    }
    //Now log the error.
    ukl1LogFwCcpcError(errMsg);
  }

//   if (verifyWrite) {
//     //We wish to check the values written to the L1 board are the same as those we read back.
//     //For now we will naively assume that if the write works then so will the read.
//     unsigned checkHigh;
//     fwCcpc_GPIOGet(ukl1CcpcName, line, checkHigh);
//     //Now check the return value matches the written value.
//     if ( checkHigh != lineHigh ) {
// 	ukl1WarningMessage("read value, "+checkHigh+", does not match written value, "+lineHigh+", for GPIO set.");
//     }
//   }

  //Return the call status to indicate whether we were successful or not.
  if (0 < callStatus) {
    //They only promise that the error will be greater than 1, whereas I promise it will be 1.
    return 1;
  } else {
    //Otherwise the return values match mine.
    return callStatus;
  }
}

/*!
 * Takes text that is either Enabled or Disabled and converts that to a TRUE
 * and FALSE respectivily.
 *
 * \param  enabled A string containing the text enabled or disabled.
 * \param  trueString The string that represents true value.
 * \return bool Conversion value of the string into a boolean type.
 *
 * Only a true string need be given. If it does not match the true string a false will be returned.
 */
bool ukl1StringToBool(string enabled, string trueString) {
  if (trueString == enabled) {
    return TRUE;
  } else {
    return FALSE;
  }
}

/*!
 * Takes a boolean and converts it to text that is either Enabled or Disabled corresponding to
 * TRUE and FALSE respectivily.
 *
 * \param  enabled A boolean representing the enabled/disabled stated.
 * \param  trueString The string that is to be returned in the event of a true value.
 * \param  falseString The string to be returned in the event of false.
 * \return string Conversion value of the boolean into string type.
 */
string ukl1BoolToString(bool enabled, string trueString, string falseString) {
  if (enabled) {
    return trueString;
  } else {
    return falseString;
  }
}

/*!
 * In order to write to specific registers on the FPGAs we must first convert the address
 * to include the address of the FPGAs on the CCPC perspective.
 *
 * \param  addr Address of the register to be written to.
 * \return unsigned Full address of the FPGA register.
 */
unsigned ukl1FpgaizeAddress(unsigned addr) {
  // Based on code written by Steve Wotton, Cambridge.
  // The address sent to lb_{write|read}_hword should be above 0x20000
  // to be outside the GBE & gluecard ranges
  // The FPGA is sensitive only to bits 2-13 of the address
  //  (=> the FPGA is not sensitive to the 0x2???? that indicates
  //      that an address is outside the GBE/gluecard ranges)
  // So whenever we want to address the FPGA, set the 13th bit:
  //   the FPGA will only respond to addresses for which the 13th bit has been set
  //
  // So: << 2    translates a (32-bit) int address into (8-bit) byte address
  //     0x20000 gets us outside the address ranges of the GBE and gluecard
  //               (but will be ignored by the FPGA)
  //     0x2000  should stop the gluecard from reacting to the operation
  //               (but will be ignored by the FPGA)
  //
  // => (int) register addresses 0x0 to 0x7ff should be usable by the FPGA
  //

  return (0x08000000 + ( 0x800 | ( addr<<2 ) ) );
}

/*!
 * A function that is used to convert a 12 hex digit MAC address and convert it
 * into three 16-bit numbers.
 *
 * \param  macAddress The string that contains the MAC address to be parsed.
 * \return dyn_uint A three element array, each element containing 16-bits
 * of the MAC address. The elements are arranged in the following order:
 * \li 1 bits 0 to 15.
 * \li 2 bits 16 to 31.
 * \li 3 bits 32 to 47.
 */
dyn_int ukl1ParseMacAddress(string macAddr) {
  //Variable to hold output.
  dyn_int outputAddr;
  //Use sscanf to parse the MAC address and remove the sections.
  int matchedCases = sscanf(macAddr, "%04x%04x%04x", outputAddr[3], outputAddr[2], outputAddr[1]);
  if (3 != matchedCases) {
    string errMsg;
    sprintf(errMsg, "Matched only %d sections of MAC address (%s, should be 3).", matchedCases, macAddr);
    setValue("GbeMessages", "text", errMsg);
  }

  return outputAddr;
}

/*!
 * A function that is used to convert an IP address in the format xxx.xxx.xxx.xxx and convert it
 * into two 16-bit numbers.
 *
 * \param  ipAddress The string that contains the IP address to be parsed.
 * \return dyn_uint A two element array, each element containing 16-bits
 * of the IP address. The elements are arranged in the following order:
 * \li 1 bits 0 to 15.
 * \li 2 bits 16 to 31.
 *
 * The address is expected as a decimal number, where leading zeros are optionally suppressed.
 * x=0-9 or omitted. e.g. 192.168.000.001 or 192.168.0.1 are both accepted.
 */
dyn_int ukl1ParseIpAddress(string ipAddr) {
  //Variables to hold output.
  dyn_int tmpAddr;
  dyn_int outputAddr;
  //Get sscanf to parse the IP address and determine which sections contain what numbers.
  int matchedCases = sscanf(ipAddr, "%d.%d.%d.%d", tmpAddr[4], tmpAddr[3], tmpAddr[2], tmpAddr[1]);
  if (4 != matchedCases) {
    string errMsg;
    sprintf(errMsg, "Matched only %d sections of IP (%s, should be 4).", matchedCases, ipAddr);
    setValue("GbeMessages", "text", errMsg);
  }

  outputAddr[1] = tmpAddr[2]<<8 | tmpAddr[1];
  outputAddr[2] = tmpAddr[4]<<8 | tmpAddr[3];

  return outputAddr;
}

//@}
