#include <sys/stat.h>

#include <iostream>
#include <fstream>
#include <string>
#include <cstring>
#include <memory>

#include <stdio.h>

#include <getopt.h>

#include "configipc.h"

void createGlobal();
void createCrate(int partition, int crate);
void createROD(int partition, int crate, int rod);
void createMUR(int part, int crate, int rod, int order, int number);
void createModule(int MUR, int order, unsigned long i);
void printProc();

// The real values
static int numPartitions = 4;
static int cratesPerPartition = 2;
static int rodsPerCrate = 11;
static int MURsPerRod = 8;
static int modulesPerMUR = 6;

std::auto_ptr<SctConfiguration::Configuration> config;

void usage();

int main(int argc, char **argv) {
  config.reset(new SctConfiguration::ConfigIPC());
  std::cout << "Welcome to configuration creator\n";

  while(1) {
    int opt = getopt(argc, argv, "p:c:r:u:m:h");
    if(opt == -1) break;

    switch(opt) {
    case 'p': numPartitions = atoi(optarg); break;
    case 'c': cratesPerPartition = atoi(optarg); break;
    case 'r': rodsPerCrate = atoi(optarg); break;
    case 'u': MURsPerRod = atoi(optarg); break;
    case 'm': modulesPerMUR = atoi(optarg); break;
    case 'h': usage(); exit(1); break;
    }
  }

  try {
    config->clearAll();
    createGlobal();
  } catch(SctConfiguration::ConfigurationException &e) {
    std::cout << "Exception creating configuration\n";
    std::cout << e.what() << std::endl;
  }

  config->saveConfiguration("newConfiguration.xml");

  std::cout << "Wrote Configuration\n";

  //  printProc();
}

void createGlobal() {
  for(int i=0; i<numPartitions; i++) {
    std::string name;

    switch(i) {
    case 0: name = "barrel0"; break;
    case 1: name = "barrel1"; break;
    case 2: name = "forward0"; break;
    case 3: name = "forward1"; break;
    }

    std::cout << "Partition " << name << std::endl;

    config->namePartition(i, name);

    for(int cr=0; cr<cratesPerPartition; cr++) {
      createCrate(i, cr);
    }
  }
}

void createCrate(int part, int crate) {
  static int lastROD = -1;

  SctConfiguration::TimConfig timConf;

  timConf.baseAddress = 13 * 0x10000000UL;
  timConf.trigFrequency = 100;
  timConf.resetFrequency = 10;

  config->configureTIM(part, crate, timConf);

  for(int r=0; r<rodsPerCrate; r++) {
    lastROD++;
    createROD(part, crate, lastROD);
  }
}

void createROD(int part, int crate, int rod) {
  static int lastMUR = 0;

  if(part < 2 && lastMUR > 175)
    return;

  std::cout << "ROD " << rod << std::endl;

  SctConfiguration::RodConfig rConf;

  rConf.baseAddress = 0x1000000 * rod;

  for(int s=0; s<4; s++) {
    rConf.slaves[s].ipramFile = "../SlaveDspCode1/slaveRunBios_IPRAM.bin";
    rConf.slaves[s].idramFile = "../SlaveDspCode1/slaveRunBios_IDRAM.bin";
    rConf.slaves[s].extFile = "../SlaveDspCode1/slaveRunBios_xcode.bin";
  }

  config->configureROD(part, crate, rod, rConf);

  for(int m=0; m<MURsPerRod; m++) {
    if(part < 2 && lastMUR > 175)
      return;

    createMUR(part, crate, rod, m, lastMUR);
    lastMUR ++;
  }
}

void createMUR(int part, int crate, int rod, int order, int number) {
  static int lastModule = -1;
  static int offset = 1;
  static int lastPart = 0;

  std::cout << "MUR " << number << std::endl;

  config->mapRODMUR(part, crate, rod, order, number);

  for(int m=0; m<modulesPerMUR; m++) {
    createModule(number, m+1, lastModule+1);

    lastModule ++;
  }

  if(part!=lastPart) {
    lastPart = part;
    offset = number+1;
  }

  if(part < 2) {
    int barrel;
    int row;
    if(number < 56) {
      barrel = 6;
      row = number;
    } else if (number < 56+48) {
      barrel = 5;
      row = number - 56;
    } else if (number < 56+48+40) {
      barrel = 4;
      row = number - (56+48);
    } else if (number < 56+48+40+32) {
      barrel = 3;
      row = number - (56+48+40);
    } else {
      barrel = 2;
      row = 10;
    }

    int position = part * 2 - 1;
    config->mapBarrelMUR(number, barrel, row, position);
  } else {
    config->mapEndcapMUR(number, 1 * (part*2)-7, number % 4, number / 4);
  }
}

void createModule(int MUR, int order, unsigned long sn) {
  ABCDModule mConf;

  //  meta stuff
//    sprintf(id, "%015d", number);
//    xmlNewChild(root, NULL, (unsigned char *)"sn", (unsigned char *)id);

//    xmlNewChild(root, NULL, (unsigned char *)"source", (unsigned char *)"dummy generated by create using libxml");
//    xmlNewChild(root, NULL, (unsigned char *)"location", (unsigned char *)"RAL");

//    xmlNodePtr config = xmlNewChild(root, NULL, (unsigned char *)"config", NULL);

  mConf.active = 1;
  mConf.select = 0;

  for(int chipNum=0; chipNum<12; chipNum++) {
//      snprintf(buffer, 100, "%02d", chipNum);
//      xmlSetProp(chip, (unsigned char *)"id", (unsigned char *)buffer);

    ABCDChip &chip = (mConf.chip[chipNum]);

    chip.address = chipNum + 32;
    chip.active = 1;
    chip.basic.config.readoutMode = 2;
    chip.basic.config.calibMode = 3;
    chip.basic.config.trimRange = 0;
    chip.basic.config.edgeDetect = 1;
    chip.basic.config.mask = 0;
    chip.basic.config.accumulate = 0;
    chip.basic.config.inputBypass = 0;
    chip.basic.config.outputBypass = 0;
    chip.basic.config.master = 1;
    chip.basic.config.end = 0;
    chip.basic.config.feedThrough = 0;

    chip.basic.mask[0] = 0xffffffff;
    chip.basic.mask[1] = 0xffffffff;
    chip.basic.mask[2] = 0xffffffff;
    chip.basic.mask[3] = 0xffffffff;

    chip.basic.vthr = 0x29;
    chip.basic.vcal = 0xa0;
    chip.basic.delay = 0x0d;
    chip.basic.preamp = 0x18;
    chip.basic.shaper = 0x19;

    chip.caldata.rc_function = 3;
    chip.caldata.rc_params[0] = 1.66e+003;
    chip.caldata.rc_params[1] = 7.88e+000;
    chip.caldata.rc_params[2] = -7.82e+002;

    chip.caldata.c_factor = 1.00;

    chip.target = 0;

    for(int ch=0; ch<128; ch+=8) {
      unsigned long trimValues;
      switch(ch/8) {
      case 0: trimValues = 0x75b87a78; break;
      case 1: trimValues = 0xa8876b77; break;
      case 2: trimValues = 0x9777b6aa; break;
      case 3: trimValues = 0x88578b7c; break;

      case 4: trimValues = 0xb9a9ab79; break;
      case 5: trimValues = 0x9d8979a9; break;
      case 6: trimValues = 0x9a98a898; break;
      case 7: trimValues = 0x95969cb9; break;

      case 8: trimValues = 0x79b78768; break;
      case 9: trimValues = 0x65098b88; break;
      case 10: trimValues = 0x9a569658; break;
      case 11: trimValues = 0x97899768; break;

      case 12: trimValues = 0x75999665; break;
      case 13: trimValues = 0x76787a95; break;
      case 14: trimValues = 0xa97a7a68; break;
      case 15: trimValues = 0x689b5745; break;
      default: trimValues = 0x00000000; break;
      }
      for(int i=0; i<8; i++) {
        unsigned char val = (trimValues >> ((7-i)*4)) & 0x0000000f;

        chip.trim[(ch/8)*8+i] = val;
      }
    }
  }    
  
  char buffer[20];
  sprintf(buffer, "%014ld", sn);

  config->configureModuleFromStructure(buffer, mConf);

  config->mapModuleMUR(MUR, order, MUR, (order%6)+1, buffer);
  config->mapPowerChannel(MUR, order, 0, sn/48, sn%48);

  SctConfiguration::BOCChannelConfig bConf;

  bConf.current = 0x0; bConf.delay = 0x1; bConf.markSpace = 0x13;
  bConf.threshold0 = 0xff; bConf.delay0 = 0x2;
  bConf.threshold1 = 0xfe; bConf.delay1 = 0x3;

  config->configureBOCChannel(MUR, order, bConf);

}

void printProc() {
  std::ifstream status("/proc/self/status");

  while(status) {
    std::string line;
    getline(status, line);

    std::cout << line << std::endl;
  }
}

void usage() {
  std::cout << "usage: create [options]\n";
  std::cout << "\t p #   Set num partitions [4]\n";
  std::cout << "\t c #   Set num crates per partition [2]\n";
  std::cout << "\t r #   Set num rods per crate [11]\n";
  std::cout << "\t u #   Set num MURs per ROD [8]\n";
  std::cout << "\t m #   Set num modules per MUR [6]\n";
  std::cout << "\t h     Print this help\n";    
}

