#include "SctData/FitScanResult.h"
#include "SctData/FitObject.h"

#include "../../AnalysisTests/AnalysisTestFramework.h"
#include "../../AnalysisTests/CutUtils.h"

using namespace Sct;
using namespace Sct::IS;
using namespace boost;
using namespace SctData;
using namespace SctTest;
using namespace std;

struct Info {
    unsigned int runNumber;
    unsigned int scanNumber;
    unsigned int channel;
    char serial[15];    
};
string InfoStr = "Run/i:Scan:Channel:Serial/C";

struct ThresholdData {
    double Mean;
    double Sigma;
    double ChiSq;
};
string ThresholdDataStr = "Mean/D:Sigma:ChiSq";

ThresholdData root;
ThresholdData nag;
Info info;

class NagRootCompare : public AnalysisTestFramework<FitScanResult> {
public:
    NagRootCompare();
    virtual void doModule(string serialNumber);
    virtual void publishData(SctTestApiStatus* status);    
    virtual void setup();
    virtual void summaryOutput();
    
private:
    void setFitStrategy(string strategy);
    void compareFits(FitScanResult& nagFit, FitScanResult& rootFit);
    void fillData(FitObject& fitOb, ThresholdData& data);
    void removeAll();
};

NagRootCompare::NagRootCompare() : AnalysisTestFramework<FitScanResult>(".*FitScanResult", "FittedData") {
    nData = 10;
}

void NagRootCompare::removeAll() {
    ISInfoIterator it( SctNames::getPartition(), "FittedData", ".*" );
    ISInfoDictionary dict(SctNames::getPartition());
    while (it()) {
	dict.remove(it.name());
    }
    ISInfoIterator it2( SctNames::getPartition(), "EventData", ".*" );
    while (it2()) {
	dict.remove(it2.name());
    }
    system("is_ls -p SCT");
}

void NagRootCompare::doModule(string serialNumber) {
    SctTestApiStatus status; 
    status.returnCode = 0;
    
    highLevelApi->setModuleSerial(&status, copyStringToILU(serialNumber));
    
    try {
	setFitStrategy("NagFitStrategy");
	publishData(&status);
	if (status.returnCode != 0) {
	    cerr << "Error publishing data: " << status.returnCode << endl;
	    return;
	}	
	vector<shared_ptr<FitScanResult> > nagFits = readData(serialNumber);
	
	removeAll();
	setFitStrategy("RootFitStrategy");
	
	publishData(&status);
	if (status.returnCode != 0) {
	    cerr << "Error publishing data: " << status.returnCode << endl;
	    return;
	}	
	vector<shared_ptr<FitScanResult> > rootFits = readData(serialNumber);
	
	if (nagFits.size() != rootFits.size()) {
	    cout << "Numbers of fits different!!  Nag: " << nagFits.size() << " Root: " << rootFits.size() << endl;
	    ++exitCode;
	    return;
	}
	
	for (unsigned int i=0; i<nagFits.size(); ++i) {
	    compareFits(*nagFits[i], *rootFits[i]);
	}
    } catch(Throwable& e) {
	e.sendToMrs(MRS_ERROR);
    }
}



void NagRootCompare::fillData(FitObject& fitOb, ThresholdData& data) {
    data.Mean = fitOb.getParameter("Mean");
    data.Sigma = fitOb.getParameter("Sigma");
    data.ChiSq = fitOb.getChiSquared() / (fitOb.getNDF() + fitOb.getNPar());
}


void NagRootCompare::compareFits(FitScanResult& nagFit, FitScanResult& rootFit) {
    info.runNumber = nagFit.getHeader().getRunNumber();
    info.scanNumber = nagFit.getHeader().getScanNumber();
    strncpy(info.serial, nagFit.getHeader().getModuleName().c_str(), 14);
			          
    
    for (unsigned int channel = 0; channel<nChannelModule; ++channel) {	
	info.channel = channel;
	fillData(nagFit.getChannelFit(channel), nag);    
	fillData(rootFit.getChannelFit(channel), root);
	
	tree->Fill();	
    }
}


/**
  Create the TFile and TTree
  */
void NagRootCompare::setup() {
    string name = Env::substituteVariables("${SCT_DAQ_ROOT}/SystemTests/logs/NagRootComp.root");
    file = new TFile(name.c_str(), "RECREATE");
    tree = new TTree("Data", "Threshold Scan Comparison Data");
    tree->Branch("Info", &info, InfoStr.c_str());
    tree->Branch("Nag", &nag, ThresholdDataStr.c_str());
    tree->Branch("Root", &root, ThresholdDataStr.c_str());
    info.serial[14] = '\0';
}

void NagRootCompare::publishData(SctTestApiStatus* status) {
    highLevelApi->responseCurve(status);
}

/**
  Generates some summary output from the TTree
  */
void NagRootCompare::summaryOutput() {
    if (cut(*tree, "mean", "(Nag.Mean-Root.Mean)/Nag.Mean*100", 0.1, 0.5, 5) > 0) {
	++exitCode;
	cout << "Failed mean tail check" << endl;
    }
    if (cut(*tree, "sigma", "(Nag.Sigma-Root.Sigma)/Nag.Sigma*100", 1, 2, 10) > 0) {
	++exitCode;
	cout << "Failed sigma tail check" << endl;
    }
    if (cut(*tree, "chi", "(Nag.ChiSq-Root.ChiSq)", 0.3, 0.5, 2, false) > 0) {
	++exitCode;
	cout << "Failed ChiSq tail check" << endl;
    }
}

void NagRootCompare::setFitStrategy(string strategy) {
    string command = "java -Dipc.ref.file=$IPC_REF_FILE bsh.Interpreter $SCT_DAQ_ROOT/SystemTests/scripts/SetFitStrategy.bsh " + strategy;
    system(command.c_str());
}
		
int main(int argc, char** argv) {
    TH1::AddDirectory(true);
    NagRootCompare sdc;
    sdc.analyzeAll(argc, argv);
    return sdc.getExitCode();
}
