#include "SctData/TimeWalkTestResult.h"
#include "../../AnalysisTestFramework.h"
#include "../../CutUtils.h"
#include "../../TestFileUtil.h"

struct Info {
    unsigned int runNumber;
    unsigned int scanNumber;
    unsigned int chip;
    unsigned int nScans;
    char serial[15];    
};
string InfoStr = "Run/i:Scan:Chip:NScans:Serial/C";

struct SDData {
    double timeWalk;
    double calibration;
    int pass;
};
string DataStr = "TimeWalk/D:Calibration:Pass/I";

//Globals variables:
SDData RodDaq;                    //Our data
SDData SctDaq;                    //SCTDAQ data
Info info;                          //Some info

//Noise Occupancy stuff
string Ext = "_TimeWalk.dat";
string Output = "${SCT_DAQ_ROOT}/SystemTests/logs/TimeWalk.root";
string Arg = "-9";

class TimeWalkCompare : public AnalysisTestFramework<TimeWalkTestResult> {
public:
    virtual void publishData(SctTestApiStatus* status);
    virtual void downloadData(string serialNumber);
    virtual void compare(const TimeWalkTestResult& t); 
    virtual void summaryOutput();
    virtual void setup();
};


string getFileName(string serial) {
    ostringstream oss;
    oss << serial << Ext;
    
    return oss.str();
}

void TimeWalkCompare::compare(const TimeWalkTestResult& sd) {
    strncpy(info.serial, sd.getModuleName().c_str(), 14);
    info.runNumber = sd.getRunNumber();
    info.scanNumber = sd.getScanNumberAt(0);
    info.nScans = sd.getNScans();
    
    string fileName = getFileName(sd.getModuleName());
    ifstream file (fileName.c_str());
    if (!file.good())
        throw IllegalStateError("Failed to open comparison file: " + fileName, __FILE__, __LINE__);
    
    SctDaq.pass = TestFileUtil::getPass(file);
    TestFileUtil::skipHeader(file);

    for (unsigned int chip = 0; chip<nChipModule; ++chip) {
	string chipStr;
	info.chip = chip;
	file >> chipStr >> SctDaq.timeWalk >> SctDaq.calibration;
	//Ignore rest of line
	file.ignore(256, '\n');

	RodDaq.timeWalk = sd.getChipResult(chip).timewalk;
	RodDaq.calibration = sd.getChipResult(chip).calibration;
	RodDaq.pass = sd.getPassed();

        tree->Fill();
    }
}


void TimeWalkCompare::summaryOutput() {
    if (cut(*tree, "timewalk", "(RodDaq.TimeWalk-SctDaq.TimeWalk)/SctDaq.TimeWalk*100", 1, 2, 10) > 0) {
	++exitCode;
	cout << "Failed timewalk tail check" << endl;
    }
    if (cut(*tree, "calibration", "(RodDaq.Calibration-SctDaq.Calibration)/SctDaq.Calibration*100", 0.3, 0.5, 2) > 0) {
	++exitCode;
	cout << "Failed calibration tail check" << endl;
    }
    exitCode += errorCode;
}


/**
  Create the TFile and TTree
  */
void TimeWalkCompare::setup() {
    string name = Env::substituteVariables(Output);
    file = new TFile(name.c_str(), "RECREATE");
    tree = new TTree("TWData", "TimeWalk Comparison Data");
    tree->Branch("RodDaq", &RodDaq, DataStr.c_str());
    tree->Branch("SctDaq", &SctDaq, DataStr.c_str());
    tree->Branch("Info", &info, InfoStr.c_str());
    info.serial[14] = '\0';
}

/**
  Downloads the response curve data
  */
void TimeWalkCompare::downloadData(string serialNumber) {
    ostringstream oss;
    TestInfo info = moduleData.getTimeWalkInfo(serialNumber);
    oss << "java ProdDatabase/getDBfiles " << Arg << " -d -r " << info.runNumber << " -s " << info.scanNumber << "  " << serialNumber;
    system(oss.str().c_str());
}

/**
  Publishes all the data we need, looping over modules,
  then waits for it all to be fitted, analyzed
  */
void TimeWalkCompare::publishData(SctTestApiStatus* status) {
    highLevelApi->timeWalk(status);
}


int main(int argc, char** argv) {
    TimeWalkCompare sdc;
    sdc.analyzeAll(argc, argv);
    return sdc.getExitCode();
}












