#include "SummaryManager.h"
#include "NPtGainSummaryWriter.h"

#include "SctData/NPtGainTestResult.h"
#include "SctData/ResponseCurve.h"
#include "SctData/Stat.h"

#include <iomanip>
#include <TF1.h>

namespace SctData {
    namespace TestSummary{
	using namespace std;

	void NPtGainSummaryWriter::write(const TestResult& t, ostream& out) const throw(Sct::IoError, Sct::LogicError)
	{
	    t.getNScans();
	    const NPtGainTestResult& result  = dynamic_cast<const NPtGainTestResult&>(t);
	    switch ( result.getNScans() ){
	    case (3) : out << "%ThreePointGain" << endl; break;
	    case (10) : out << "%ResponseCurve" << endl; break;
	    default : out <<"%NPtGain"<<endl; break;
	    }

	    out << "#\n#LoopA - Fit"<<endl;
	    out << "#      func    p0     p1       p2" << endl;
	    for (short unsigned ichip=0; ichip<nChipModule; ++ichip){
		out << "#"<<getChipName(ichip) << endl;
		shared_ptr<TF1> fit = result.getChipData(ichip).rc->getFunction();
		out << setfill(' ');
		out << setw(6) << result.getChipData(ichip).rc->getIndex();
		for (short ipar=0; ipar<fit->GetNpar(); ++ipar){
		    out << "  " << setfill(' ') << setw(6) << fit->GetParameter(ipar);
		}
		out << endl;
	    }

	    /** cout << endl << "Defects: " << endl;
		const vector< shared_ptr<const ModuleDefect> >& defects = result.getDefects().getAllDefects();
		for (unsigned int i=0; i<defects.size(); ++i) {
		cout << setfill(' ') << setw(20) << defects[i]->getName() 
		<<  "\tStart channel: " << defects[i]->getModuleElement().getFirst() 
		<< "\t# of channels: " << defects[i]->getModuleElement().getNChannels() 
		<< "\t" << defects[i]->getDescription() << endl;
		}
		cout << endl;
	    */

	    out << "#\n#LoopB - Gain, Offset, Noise at 2.00fC"<<endl;
	    out << "#     vt50    rms     gain    rms   offset   rms   outnse   innse   rms" << endl;

	    for (short unsigned ichip=0; ichip<nChipModule; ++ichip){
		Stats<float> vt50(nChannelChip);
		Stats<float> gain(nChannelChip);
		Stats<float> offset(nChannelChip);
		Stats<float> outnse(nChannelChip);
		Stats<float> innse(nChannelChip);
		
		for (unsigned ichan=0; ichan<nChannelChip; ++ichan){
		    unsigned ichannel = ichan + ichip*nChannelChip;
		    bool valid=result.getDefects().defectSeverityEncompassingElement(ModuleElement::Channel(ichannel)) == NONE;		    

		    vt50.modifyAt(ichan).valid = valid;
		    gain.modifyAt(ichan).valid = valid;
		    offset.modifyAt(ichan).valid = valid;
		    outnse.modifyAt(ichan).valid = valid;
		    innse.modifyAt(ichan).valid = valid;
		    
		    vt50.modifyAt(ichan).value = result.getChannelData(ichannel).rc->getFunction()->Eval(result.getSpecialScanPointValue());
		    gain.modifyAt(ichan).value = result.getChannelData(ichannel).gain;
		    innse.modifyAt(ichan).value = result.getChannelData(ichannel).noise;
		    offset.modifyAt(ichan).value = result.getChannelData(ichannel).offset;
		}
		
		out << "#"<<getChipName(ichip) << endl;
		out << setfill(' ')  << setw(6) << vt50.mean() << "  " << sqrt(vt50.var()) << "  " 
		    << setw(6) << gain.mean() << "  " <<  setw(6) << sqrt(gain.var()) << "  " 
		    << setw(6) << offset.mean() << "  " << setw(6) << sqrt(offset.var()) << "  "
		    << setw(6) << innse.mean()*gain.mean()/6250. << "  "
		    << setw(6) << innse.mean() << "  " << setw(6) << sqrt(innse.var()) << endl;
	    }
	    

	    out << "#\n#LoopC - comment"<<endl;
	    out << "#";
	    for (short unsigned ichip=0; ichip<nChipModule/2; ++ichip){
		out << getChipName(ichip) << " ";
	    }
	    out << endl;

	    for (short unsigned ichip=0; ichip<nChipModule/2; ++ichip){
		outputDefectOnChip(ichip, result, out);
	    }
	    out << endl << "#";
	    
	    for (short unsigned ichip=nChipModule/2; ichip<nChipModule; ++ichip){
		out << getChipName(ichip) << " ";
	    }
	    out << endl;
	    for (short unsigned ichip=nChipModule/2; ichip<nChipModule; ++ichip){
		outputDefectOnChip(ichip, result, out);
	    }
	    out << endl << "#";

	    SummaryWriter::write(t.getDefects(), out);
	    
	}

	void NPtGainSummaryWriter::outputDefectOnChip(unsigned ichip, const NPtGainTestResult& result, ostream& out) throw() {
	    auto_ptr<DefectList> defects = result.getDefects().getDefectsEncompassingElement(ModuleElement::Chip(ichip) );
	    
	    short unsigned ndefect=0;
	    for (list<Defect>::const_iterator i = defects->getAllDefects().begin() ;
		 i != defects->getAllDefects().end(); ++i ) {

	      if ( (*i).getModuleElement().isChip() ) {
		if (ndefect!=0) out << ",";
		++ndefect;
		out << (*i).getPrototype().getName() ;
		}
	    }
	    if (ndefect==0) out <<  "\"OK\"  " ;
	}

	//------------------------------------------------------

	bool NPtGainSummaryWriter::inMap = SummaryManager::instance().addWriter("SctData::NPtGainTestResult", shared_ptr<SummaryWriter>(new NPtGainSummaryWriter()));
    }
}
