#include "NPtGainDisplayer.h"
#include "DisplayManager.h"
#include "DisplayInfo.h"

#include "SctData/NPtGainTestResult.h"
#include "SctData/FitScanResult.h"
#include "SctData/ResponseCurve.h"
#include "Sct/SctParameters.h"

#include <TGraph.h>
#include <TCanvas.h>
#include <TF1.h>
#include <TPolyLine.h>

#include <algorithm>

using namespace Sct;
using namespace SctData;
using namespace std;
using namespace boost;

namespace SctDataDisplay {

class NPtGainDisplayData : public DisplayData {
public:
    vector<shared_ptr<TCanvas> > canvas;
    shared_ptr<const NPtGainTestResult> test;
    vector<shared_ptr<TF1> > funcs;
    vector<shared_ptr<TGraph> > graphs;
    vector<shared_ptr<TPolyLine> > lines;
};        
    
bool NPtGainDisplayer::inMap = DisplayManager::addToMap("SctData::NPtGainTestResult", shared_ptr<Displayer>(new NPtGainDisplayer()));
    
    
shared_ptr<DisplayData> NPtGainDisplayer::display(shared_ptr<const Sct::Serializable> serial, const DisplayInfo& info, std::ostream& os) {
    shared_ptr<NPtGainDisplayData> data (new NPtGainDisplayData());
    data->test = dynamic_pointer_cast<const NPtGainTestResult>(serial);
        
    //Construct gain, noise and offset graphs
    shared_ptr<TGraph> gain (new TGraph(nChannelModule));
    shared_ptr<TGraph> noise (new TGraph(nChannelModule));
    shared_ptr<TGraph> offset (new TGraph(nChannelModule));
    data->graphs.push_back(gain);
    data->graphs.push_back(noise);
    data->graphs.push_back(offset);

    for (unsigned int i=0; i<nChannelModule; ++i) {
        const NPtGainTestResultData& chanData = data->test->getChannelData(i);
        gain->SetPoint(i, i, chanData.gain);
        noise->SetPoint(i, i, chanData.noise);
        offset->SetPoint(i, i, chanData.offset);
    }

    plotData("Gain", 0, 80, *data, *gain);
    plotData("Noise", 0, 2500, *data, *noise);
    plotData("Offset", 0, 200, *data, *offset);

    if (info.displayChips.size() > 0)
	displayChips(*data, info, os);
    
    if (info.displayChannels.size() > 0)
	displayChannels(*data, info, os);
    
    os << "Defects:" << std::endl;
    printDefectList(data->test->getDefects(), os);
    
    return data;
}

void NPtGainDisplayer::plotData(string name, double min, double max, NPtGainDisplayData& data, TGraph& graph) {
    shared_ptr<TCanvas> c = createCanvas(data.test->getModuleName() + name, name + " for " + data.test->getModuleName());
    data.canvas.push_back(c);
    graph.SetMaximum(max);
    graph.SetMinimum(min);
    graph.SetTitle(name.c_str());
    graph.GetXaxis()->SetLimits(0, nChannelModule);
    graph.GetXaxis()->SetTitle("Channel Number");
    graph.Draw("alp");
    for (unsigned int i=0; i<nChipModule; ++i) {
        shared_ptr<TPolyLine> p (new TPolyLine(2));
        data.lines.push_back(p);
        p->SetLineStyle(2);
        p->SetPoint(0, nChannelChip*i-0.5, min);
        p->SetPoint(1, nChannelChip*i-0.5, max);
        p->Draw();
    }
}

void NPtGainDisplayer::displayChips(NPtGainDisplayData& data, const DisplayInfo& info, std::ostream& os) {
    shared_ptr<TCanvas> c = createCanvas(data.test->getModuleName() + "_chip", data.test->getModuleName() + " Chip Fits");
    data.canvas.push_back(c);
    divideCanvas(info.displayChips.size(), *c);
    for (unsigned int i = 0; i < info.displayChips.size(); ++i) {
        c->cd(i + 1);
	displayData(data, data.test->getChipData(info.displayChips[i]));
    }
    
    //Print out chip fit parameters
    os << "Chip parameters" << endl;
    for (unsigned int i = 0; i < 12; ++i) {
        shared_ptr<TF1> f = data.test->getChipData(i).rc->getFunction();
        os << f->GetParameter(0) << "  " << f->GetParameter(1) << "  " << f->GetParameter(2) << endl;
    }    
}

void NPtGainDisplayer::displayChannels(NPtGainDisplayData& data, const DisplayInfo& info, std::ostream& os) {
    shared_ptr<TCanvas> c;
    unsigned int nMax = min((unsigned int)25, info.displayChannels.size());
    int j=0;
    
    for (unsigned int i=0; i<info.displayChannels.size(); ++i, ++j) {	
        if (j%nMax == 0) {
	    c = createCanvas("channelRC" + data.test->getModuleName(), "Channel Response Curves for module " + data.test->getModuleName());
	    data.canvas.push_back(c);
	    divideCanvas(nMax, *c);
            j=0;
        }   
        c->cd(i%nMax + 1);
	displayData(data, data.test->getChannelData(info.displayChannels[i]));
    }
}

void NPtGainDisplayer::displayData(NPtGainDisplayData& data, const NPtGainTestResultData& testData) {
    TGraph & g = *testData.graph;
    g.SetMarkerStyle(20);
    g.Draw("ap");
    g.GetHistogram()->SetXTitle("Q_th / fC");
    g.GetHistogram()->SetYTitle("V_th / mV");
    shared_ptr<TF1> f = testData.rc->getFunction();
    f->SetRange(0, 8);
    f->SetFillStyle(0);
    f->Draw("same");
    data.funcs.push_back(f);
}


}
