#define USELIBRARY

#include <iostream>
#include <vector>

#include <boost/bind.hpp>
#include <boost/thread.hpp>
#include <boost/timer.hpp>

#include "../TApi.h"

#include "memoryPartitions.h"


// Get intel processor instruction counter!
static __inline__ unsigned long long int rdtsc()
{
  unsigned long long int x;
  __asm__ volatile (".byte 0x0f, 0x31" : "=A" (x));
  return x;
}

using namespace std;

std::list<boost::thread *> threads;

struct { 
  unsigned long *blockData;
  int repetitions;
  unsigned int length;
  int multiRod;
  long address;
  int dsp;
  int usePrimitive;
  int useThreads;
  int threadStart;
  int checkData;

  int doneCount;

  std::list<SctApi::RodLabel> rodList;
} globalConf;

void usage();
void parseArgs(int argc, char **argv);
void buildData();
void setupThreads(TApi &tapi);
void startThreads();
void writeData(TApi &tapi);
bool readDataNoThread(TApi &tapi);
bool readData(TApi &tapi, const SctApi::RodLabel &rl);
bool checkResponse(unsigned long *body, unsigned long length);

void threadLoop(TApi &tapi, const SctApi::RodLabel &rl);

int main (int argc, char **argv) {
  TApi tapi;

  bool cont = true;

  globalConf.repetitions = 1;
  globalConf.length = 2000;
  globalConf.multiRod = 0;
  globalConf.address = 0xb0000000;
  globalConf.dsp = -1;
  globalConf.usePrimitive = 0;
  globalConf.useThreads = 0;
  globalConf.threadStart = 0;
  globalConf.checkData = 0;

  globalConf.doneCount = 0;

  parseArgs(argc, argv);

  buildData();

  tapi.initialiseAll(0);

//   tapi.setDebugOption("print_in_prim");
//   tapi.setDebugOption("print_out_prim");
//   tapi.setDebugOption("print_raw");

  cout << "Config:\n";
  cout << " no.  repetitions: " << globalConf.repetitions << "\n";
  cout << " data length: " << globalConf.length << endl;
  cout << " use all RODs: " << (globalConf.multiRod?"yes":"no") << endl;
  cout << " data address: 0x" << hex << globalConf.address << dec << endl;
  cout << " dsp to use: " << globalConf.dsp << endl;
  cout << " use primitive: " << (globalConf.usePrimitive?"yes":"no") << endl;
  cout << " use threads: " << (globalConf.useThreads?"yes":"no") << endl;
  cout << " check valid: " << (globalConf.checkData?"yes":"no") << endl;

  if(globalConf.multiRod) {
    globalConf.rodList = tapi.listRods();
  } else {
    SctApi::RodLabel rl(0, 0, 0);
    globalConf.rodList.push_back(rl);
  }

  cout << "Testing block reads on " << globalConf.rodList.size() << " RODs\n";

  writeData(tapi);

  if(globalConf.useThreads) {
    setupThreads(tapi);
  }

  double startTime = rdtsc() / 1e9; // timer t;
  double finishTime = startTime - 10;

  try {
    if(globalConf.useThreads) {
      startThreads();
      for(std::list<boost::thread *>::iterator iter = threads.begin();
          iter != threads.end();
          iter++) {
        (*iter)->join();
      }
    } else {
      for(int rep=0; rep<globalConf.repetitions && cont == true; rep++) {
        try {
          if(cont) {
            if(!readDataNoThread(tapi)) {
              //             cout << "Read or check failed on rep: " << rep << endl;
            }
          }
        } catch(...) {
          cout << "Exception sending primitive\n";
          cont = false;
        }
      }
    }

    finishTime = rdtsc() / 1e9; // t.elapsed();

    cout << "Complete\n";
  } catch(...) { // SctPixelRod::BaseException &b) {
    cout << "Got exception!" << endl;
  }

  if(globalConf.repetitions > 1) {
    cout << "Did " << (globalConf.doneCount/globalConf.rodList.size()) << " successful repetitions\n";
  }

  int words = globalConf.length * (globalConf.doneCount/globalConf.rodList.size());
  cout << " " << words << " words in " << (finishTime - startTime) << " seconds\n";
  cout << " " << ((words/(finishTime - startTime))/1000) << " kWords/sec\n";

  delete [] globalConf.blockData;

  tapi.shutdownAll();
}

bool checkResponse(unsigned long *outBody, unsigned long outLength) {
  bool correct = true;
  if(globalConf.checkData) {
    if(outLength != globalConf.length) {
//     cout << "Length check failed\n";
      correct = false;
    } else {
      for(unsigned int i=0; i<globalConf.length; i++) {
        if(outBody[i] != globalConf.blockData[i]) {
          correct = false;
//         cout << "Check failed at word " << i << "\n";
//         cout << "\t0x" << hex << outBody[i] << " != 0x" << globalConf.blockData[i] << dec << "\n";
          break;
        }
      }
    }
  }

  if(correct) cerr << ".";

  return correct;
}

void buildData() {
  // Create the data
  globalConf.blockData = new unsigned long[globalConf.length];

  if(globalConf.length < 3) {
    if(globalConf.length > 0) globalConf.blockData[0] = 0xDEADF00D;
    if(globalConf.length > 1) globalConf.blockData[1] = 0x0;
  } else {
    for (unsigned int i=0; i<globalConf.length; i++) {
      globalConf.blockData[i] = 0xBEAD0000 + i;
    }
  }
}

void writeData(TApi &tapi) {
  if(globalConf.multiRod) {
    for(list<SctApi::RodLabel>::const_iterator ri = globalConf.rodList.begin(); 
        ri != globalConf.rodList.end(); 
        ri++) {
      tapi.dspBlockWrite(ri->partition, ri->crate, ri->rod, 
                         globalConf.blockData, globalConf.address, globalConf.length, 
                         globalConf.dsp, globalConf.usePrimitive);
    }
  } else {
    tapi.dspBlockWrite(0, 0, 0,
                       globalConf.blockData, globalConf.address, globalConf.length, 
                       globalConf.dsp, globalConf.usePrimitive);
  }
}

bool readDataNoThread(TApi &tapi) {
  bool success = true;

  if(globalConf.multiRod) {
    for(list<SctApi::RodLabel>::const_iterator ri = globalConf.rodList.begin(); 
        ri != globalConf.rodList.end(); 
        ri++) {

      readData(tapi, *ri);
    }
  } else {
    static SctApi::RodLabel rl(0, 0, 0);
    readData(tapi, rl);
  }

  return success;
}

bool readData(TApi &tapi, const SctApi::RodLabel &rl) {
  bool success = true;

  unsigned long returnLength;
  unsigned long *result = tapi.dspBlockRead(rl.partition, rl.crate, rl.rod, 
                                            globalConf.address, globalConf.length, globalConf.dsp,
                                            &returnLength, globalConf.usePrimitive);
//       tapi.dspBlockDump(rl->partition, rl->crate, rl->rod, 
//                                                 globalConf.address, globalConf.length, globalConf.dsp);

//     tapi.dspBlockDumpFile(0, 0, 0, 
//                           globalConf.address, globalConf.length, globalConf.dsp, "blockDump.bin");

  
  if(result) {
    if(!checkResponse(result, returnLength)) {
      //          cout << "Check failed repetition " << rep << "\n";
      success = false;
    } else {
      globalConf.doneCount ++;
    }
    delete [] result;
  } else {
    cout << "Response was null\n";
  }

  return success;
}

void parseArgs(int argc, char **argv) {
  if(argc > 1) {
    for(int i=1; i<argc; i++) {
      if(argv[i][0] == '-') {
        switch(argv[i][1]) {
        case 'r':
          globalConf.repetitions = atoi(&argv[i][2]);
          break;
        case 'l':
          globalConf.length = atoi(&argv[i][2]);
          break;
        case 'd':
          globalConf.dsp = atoi(&argv[i][2]);
          break;
        case 'm':
          globalConf.multiRod = 1;
          break;
        case 'p':
          globalConf.usePrimitive = 1;
          break;
        case 't':
          globalConf.useThreads = 1;
          break;
        case 'c':
          globalConf.checkData = 1;
          break;
        case 'h':
          usage();
          exit(1);
          break;
        }
      }
    }
  }
}

void usage() {
  cout << "BlockReads [opts]\n";
  cout << "\t-r# Repeat # times\n";
  cout << "\t-l# Use buffer with length #\n";
  cout << "\t-d# Use dsp number # (-1 for master)\n";
  //  cout << "\t-a# Read/write address #\n";
  cout << "\t-m  Use all RODs\n";
  cout << "\t-p  Use RW_SLAVE_PRIM instead of direct\n";
  cout << "\t-t  Use threads to do reads concurrently?\n";
  cout << "\t-c  Check the return data\n";
  cout << "\t-h  Show this help\n";
}

void setupThreads(TApi &tapi) {
  if(globalConf.multiRod) {
    for(list<SctApi::RodLabel>::const_iterator ri = globalConf.rodList.begin(); 
        ri != globalConf.rodList.end(); 
        ri++) {
      threads.push_back(new boost::thread(boost::bind(&threadLoop, boost::ref(tapi), *ri)));
    }
  } else {
    static SctApi::RodLabel rl(0, 0, 0);
    threads.push_back(new boost::thread(boost::bind(&threadLoop, boost::ref(tapi), rl)));
  }
}

void startThreads() {
  globalConf.threadStart = 1;
}

void threadLoop(TApi &tapi, const SctApi::RodLabel &rl) {
  while(globalConf.threadStart == 0);

  bool cont = true;

  for(int rep=0; rep<globalConf.repetitions && cont == true; rep++) {
    try {
      if(cont) {
        if(!readData(tapi, rl)) {
          //             cout << "Read or check failed on rep: " << rep << endl;
        }
      }
    } catch(...) {
      cout << "Exception sending primitive\n";
      cont = false;
    }
  }
}
