#ifndef SCTAPI_CRATE
#define SCTAPI_CRATE

// Make sure all compilations have same SctApi structure!
#include "config.h"

#include <boost/shared_ptr.hpp>

namespace SctPixelRod {
  class RodOutList;
  class RodModule;
  class TimModule;
  class VmeInterface;
}

namespace SctConfiguration {
  class Configuration;
  class BOCChannelConfig;
  class BOCGlobalConfig;
}

#include "SctApiFwd.h"

namespace SctApi {
  class PrimListWrapper;

  /**
     Exception to be thrown by public API methods
  */
  class CrateException : public std::exception {
    std::string desc;
  public:
    /// Provide a description on construction
    CrateException(const std::string &str) throw() : desc(str) {}

    ~CrateException() throw() {}

    /// Override std::exception method
    virtual const char* what () const throw() { return desc.c_str(); }
  };

/**
   A representation of the crate controller.

   This is used by SCTAPI to send commands to RodModule
   objects. These are stored in crate memory. 

   Module configuration will eventually be stored here 
   also?
 */
class Crate {
  // Don't use copy constructor or assignment operator
  Crate(const Crate&);
  Crate &operator=(const Crate&);
 public:
  /// An empty default constructor. 
  Crate() {};

  /// The destructor
  /**
     Delete all the RodModules and VmeInterface 
  */
  virtual ~Crate() {};

  /************** Interaction with Configuration **************/

  /// Initialise the TIM, RODs and BOCs from the configuration.
  /**
     Return number of RODs initialised
   */ 
  virtual int initialiseCrate() = 0;

  /// Initialise the TIM from the configuration.
  /**
     Return 0 for success
   */ 
  virtual int initialiseTim() = 0;

  /// Initialise a ROD from the configuration.
  /**
     Store object in local map.
     Return 0 for success
   */ 
  virtual int initialiseRod(unsigned int rod) = 0;

  /// Initialise a BOC from the configuration.
  /**
     Return 0 for success
  */ 
  virtual int initialiseBOC(unsigned int rod) = 0;

  /// Configure BOC parameters from the configuration
  /**
     Return 0 for success
   */
  virtual int configureBOC(unsigned int rod) = 0;

  /*************** Primlists *********************/

  /// Send a primitive list to a ROD
  /**
     Return 0 for success
  */
  virtual int sendPrimList(unsigned int rod, boost::shared_ptr<PrimListWrapper> prim) = 0;

  /// Send a primitive list to all RODs in a crate
  /**
     Return 0 for success
  */
  virtual int sendPrimListAll(boost::shared_ptr<PrimListWrapper> prim) = 0;

  /// Await a response from a ROD
  /**
     Await response from a ROD within this crate.
     Handle text messages sent by the ROD (print to stdout)

     If waiting longer than timeout seconds then return anyway
  */
  virtual int awaitResponse(unsigned int rod, int timeout) = 0;

  /// Await a response from all RODs in a crate
  /**
     Handle text messages sent by the RODs (print to stdout)

     If waiting longer than timeout seconds then return anyway
  */
  virtual int awaitResponseAll(int timeout) = 0;

  /// Get the response from a ROD
  /**
     Return the response to the last primlist (a wrapped copy)
   */
  virtual boost::shared_ptr<SctPixelRod::RodOutList> getResponse(unsigned int rod) = 0;

  virtual void setMrsStream(MRSStream *stream) = 0;

  /***************** Messages ******************/

  /// Get a message in the text buffer
  /**
     @param rod Index of ROD
     @param buffer Buffer to write text into
     @param length Length of buffer allocated, on return contains amount of data copied.

     Return message from ROD if present
   */
  virtual bool getRodMessage(unsigned int rod, char *buffer, unsigned long &length) = 0;

  /************** Misc read/writes *****************/

  /// Read data from the master DSP's memory
  /**
     Proxy for RodModule method
  */
  virtual int mdspBlockRead(unsigned int rod, long dspStart, unsigned long *buffer, unsigned long numWords) = 0;

  /// Read data from a slave DSP's memory
  /**
     Proxy for RodModule method
  */
  virtual int slvBlockRead(unsigned int rod, long dspStart, unsigned long *buffer, unsigned long numWords, long slaveNumber) = 0;

  /// Write data direct to a slave DSP's memory
  /**
     Proxy for RodModule method
  */
  virtual int slvBlockWrite(unsigned int rod, long dspStart, unsigned long* buffer, unsigned long numWords, long dspNumber) = 0;

  /// Write block of master dsp memory
  /**
     Proxy for RodModule method
   */
  virtual int mdspBlockWrite(unsigned int rod, long dspStart, unsigned long *buffer, unsigned long numWords) = 0;

  /// Read Status register 
  /**
     Proxy for RodModule method
  */
  virtual unsigned long readRodStatusReg(unsigned int rod, long regNumber) = 0;


  /// Read command register
  /**
     Proxy for RodModule method
  */
  virtual unsigned long readRodCommandReg(unsigned int rod, long regNumber) = 0;

  /// Read single word from ROD
  /**
     Proxy for RodModule method
  */
  virtual unsigned long dspSingleRead(unsigned int rod, const unsigned long dspAddr, long dspNumber) = 0;

  /// Write single word
  /**
     Proxy for RodModule method
  */
  virtual int dspSingleWrite(unsigned int rod, unsigned long dspAddr, unsigned long buffer, long dspNumber) = 0;

  /// Print status of rods on crate 
  virtual void status() = 0;

  /// Primitive list control constants 
  /**
     C_PL_STOP        Abort the current primitiveList
     C_PL_PAUSE       Pause the current primitiveList
     C_PL_RESUME      Resume the current primitiveList
     C_PL_CANCEL      Cancel the recently sent primitiveList (only if not started execution)
  */
  enum PLControl {
    C_PL_STOP,
    C_PL_PAUSE,
    C_PL_RESUME,
    C_PL_CANCEL };

  /// Control the primitive list running on a rod (Untested)
  virtual void primListControl(unsigned int rod, PLControl function) = 0;

  /// Print all BOC paramters as set up
  virtual void printBOCSetup(unsigned int rod) = 0;

  /// Return channel parameters as stored in BOC
  virtual std::vector<SctConfiguration::BOCChannelConfig> currentBOCSetup(unsigned int rod) = 0;

  /// Print all BOC registers
  virtual void printBOCRegisters(unsigned int rod) = 0;

  /// Return global registers as stored in BOC
  virtual SctConfiguration::BOCGlobalConfig currentBOCRegisters(unsigned int rod) = 0;

  /// Save BOC channel setup
  virtual void saveBOCSetup(unsigned int rod, BankType bank) = 0;
  /// Save BOC register setup
  virtual void saveBOCRegisters(unsigned int rod, BankType bank) = 0;

  /// Restore BOC channel setup
  virtual void restoreBOCSetup(unsigned int rod, BankType bank) = 0;
  /// Restore BOC register setup
  virtual void restoreBOCRegisters(unsigned int rod, BankType bank) = 0;

  /// Check BOC laser interlock
  virtual bool checkBOCLasersOn(unsigned int rod) = 0;

  /// Setup BOC to sample at 20MHz
  virtual void enterBOCClockBy2Mode(unsigned int rod) = 0;

  /// Restore BOC to 40MHz mode
  virtual void leaveBOCClockBy2Mode(unsigned int rod) = 0;

  /// Return 12 monitor values from BOC ADCs
  virtual std::vector<double> getBOCMonitorArray(unsigned int rod) = 0;

  /// Modify parameter on all BOC channels
  /**
     @param raw If false then map module channel to BOC, otherwise channel refers to BOC channel
   */
  virtual void modifyBOCParam(unsigned int type, unsigned int val, bool raw) = 0;

  /// Modify parameter on one BOC channel
  /**
     @param raw If false then map module channel to BOC, otherwise channel refers to BOC channel
   */
  virtual void modifyBOCParam(unsigned int rod, unsigned int channel, unsigned int type, unsigned int val, bool raw) = 0;

  /// Turn all BOC laser currents to 0
  virtual void lasersOff() = 0;

  /* ******************* TIM Functions ***************************/

  /// Set the TIM trigger frequency
  /**
     @param trigFrequency Frequency of triggers
     @param rstFrequency Frequency of resets
   */
  virtual void timSetFrequency(double trigFreq, double rstFreq) = 0;

  /// Start the TIM generating regular triggers 
  virtual void freeTriggers() = 0;

  /// Stop the TIM generating regular triggers
  virtual void stopTriggers() = 0;

  /// Tell the TIM to send an L1A 
  virtual void timL1A() = 0;

  /// Tell TIM to send Cal + L1A
  virtual void timCalL1A(int delay) = 0;

  virtual void timECR() = 0;

  virtual void timBCR() = 0;

  virtual void timFER() = 0;

  /// Tell TIM to send a burst of triggers
  virtual void sendTimBurst(int count) = 0;

  /// Print all the Tim registers
  virtual void timVerbose() = 0;

  /// Debug, load TIM register
  virtual void timRegLoad(int reg, UINT16 val) = 0;

  /// Debug, read TIM register
  virtual UINT16 timRegRead(int reg) = 0;

  /// Check if a slave has been started
  virtual bool slavePresent(int rod, int index) const = 0;

  /// Notify that slave was started externally...
  virtual void slaveStarted(int rod, int slave) = 0;

  /// Check if a ROD has been configured
  virtual bool RODPresent(int rod) const = 0;

  virtual long getRodSlot(int rod) const = 0;

  /// Get ROD Revision
  virtual int getRodRevision(int rod) const = 0;

  /// Stop the primitive list polling 
  virtual void stopPolling() = 0;

  /// Restart the primitive list polling
  virtual void resumePolling() = 0;
};

}  // End of namespace SctApi
#endif
