// File: TApi.cxx

#include <iostream>
#include <list>

#include "TApi.h"
#include "autoConfig.h"

#include "SctApiImpl.h"
// For BOCChannelConfig
#include "sctConf/configuration.h"

#include <boost/bind.hpp>

// Template to check for exceptions 
//  ::result_type is implemented by the boost bind stuff

template <class Call> typename Call::result_type checkCall(const Call &call) {
  try {
    return call();
  } catch(SctApi::SctApiException &e) {
    std::cout << "SctApiException " << e.what() << " thrown\n";
  }
}

ClassImp(TApi)
ClassImp(TScanDef)
ClassImp(TTrigger)

// TApi *tapi;

// Constructor
TApi::TApi(){
  worker = new SctApi::SctApi();
}

// Destructor
/*
  The destructor.
*/
TApi::~TApi(){
  delete worker;
}

// ROD Diagnostics

void TApi::initialiseAll(int scanNumber) {
  worker->initialiseAll(scanNumber);
}

void TApi::shutdownAll() {
  worker->shutdownAll();
}

void TApi::setRunNumber(int newRun) {
  worker->setRunNumber(newRun);
}

void TApi::setScanNumber(int newScan) {
  worker->setScanNumber(newScan);
}

void TApi::flashLED(unsigned int partition, unsigned int crate, unsigned int rod,
                    long slaveNumber, long period, long flashes) {
  worker->flashLED(partition, crate, rod, slaveNumber, period, flashes);
}

void TApi::echo(unsigned int partition, unsigned int crate, unsigned int rod,
                unsigned int length, unsigned long *data) {
  worker->echo(partition, crate, rod, length, data);
}

void TApi::echoAll(unsigned int length, unsigned long *data) {
  worker->echoAll(length, data);
}

void TApi::echoSlave(unsigned int partition, unsigned int crate, unsigned int rod,
                     unsigned int slave, unsigned int length, unsigned long *data) {
  worker->echoSlave(partition, crate, rod, slave, length, data);
}

void TApi::awaitResponse(unsigned int partition, unsigned int crate, unsigned int rod, int timeout) {
  worker->awaitResponse(partition, crate, rod, timeout);
}

unsigned long *TApi::getResponse(unsigned int partition, unsigned int crate, unsigned int rod) {
  unsigned long length;
  unsigned long *result = worker->getResponse(partition, crate, rod, length);
  if(length < 2) result = 0;
  else {
    if(result[0] != length) {
      std::cout << "Result length mismatch!\n";
    }
  }
  return result;
}

unsigned long *TApi::getResponse(unsigned int partition, unsigned int crate, unsigned int rod, 
                                 unsigned long *length) {
  unsigned long myLength;
  unsigned long *result = worker->getResponse(partition, crate, rod, myLength);
  *length = myLength;
  return result;
}

// Get a message from a ROD text buffer
bool TApi::getRodMessage(unsigned int partition, unsigned int crate, unsigned int rod, 
                         char *buffer, unsigned long &length) {
  return worker->getRodMessage(partition, crate, rod, buffer, length);
}

UINT32 TApi::findModule(const char *sn) {
  return worker->findModule(sn);
}

UINT32 TApi::findModule(INT32 mur, INT32 module) {
  return worker->findModule(mur, module);
}

void TApi::sendABCDModules(UINT32 bank, UINT32 type) {
  try {
    worker->sendAllABCDModules(SctApi::BankType(bank), SctApi::ConfigType(type));
  } catch(SctApi::SctApiException &e) {
    std::cout << "Got exception: " << e.what() << std::endl;
  }
}

void TApi::sendABCDModule(UINT32 mid, UINT32 bank, UINT32 type) {
  try {
    worker->sendABCDModule(mid, SctApi::BankType(bank), SctApi::ConfigType(type));
  } catch(SctApi::SctApiException &e) {
    std::cout << "Got exception: " << e.what() << std::endl;
  }
}

void TApi::getABCDModules(UINT32 bank) {
  worker->getABCDModules(SctApi::BankType(bank));
}

void TApi::getABCDModule(UINT32 mid, UINT32 bank) {
  worker->getABCDModule(mid, SctApi::BankType(bank));
}

void TApi::modifyABCDMask(UINT32 mid, UINT32* mask) {
  worker->modifyABCDMask(mid, mask);
}

void TApi::modifyABCDTrims(UINT32 mid, UINT8* trims) {
  worker->modifyABCDTrims(mid, trims);
}

void TApi::modifyABCDVar(UINT32 typ, FLOAT32 var) {
  worker->modifyABCDVar(typ, var);
}

void TApi::modifyABCDVar(UINT32 mid, UINT32 typ, FLOAT32 var) {
  worker->modifyABCDVar(mid, typ, var);
}

void TApi::modifyABCDVar(UINT32 mid, UINT32 c, UINT32 typ, FLOAT32 var) {
  worker->modifyABCDVar(mid, c, typ, var);
}

void TApi::modifyABCDVarROD(UINT32 mid, UINT32 chip, UINT32 typ, FLOAT32 var, UINT32 bank) {
  worker->modifyABCDVarROD(mid, chip, typ, var, SctApi::BankType(bank));
}

void TApi::modifyABCDVarROD(UINT32 typ, FLOAT32 var, UINT32 bank) {
  worker->modifyABCDVarROD(typ, var, SctApi::BankType(bank));
}

void TApi::modifyBOCParam(unsigned int partition, unsigned int crate, unsigned int rod,
                            unsigned int channel, unsigned int type, unsigned int val) {
  checkCall(boost::bind(&SctApi::SctApi::modifyBOCParam, worker, 
                        partition, crate, rod, channel, type, val));
}

void TApi::loadConfiguration() {
  worker->loadConfiguration();
}

void TApi::loadModuleConfigurations() {
  worker->loadModuleConfigurations();
}

void TApi::configureBOC(unsigned int partition, unsigned int crate, unsigned int rod) {
  worker->configureBOC(partition, crate, rod);
}

UINT32 TApi::findBarrelModule(INT32 barrel, INT32 row, INT32 number) {
  return worker->findBarrelModule(barrel, row, number);
}

UINT32 TApi::findEndcapModule(INT32 disk, INT32 ring, INT32 number) {
  return worker->findEndcapModule(disk, ring, number);
}

void TApi::addDebugPrimList(unsigned long length, long index, long id, long version,
                            unsigned long * body) {
  worker->addDebugPrimList(length, index, id, version, body);
}

void TApi::sendDebugPrimList(unsigned int partition, unsigned int crate, unsigned int rod) {
  worker->sendDebugPrimList(partition, crate, rod);
}

void TApi::sendDebugPrimListAll() {
  worker->sendDebugPrimListAll();
}

void TApi::sendDebugSlavePrimList(unsigned int partition, unsigned int crate, unsigned int rod,
                                  unsigned int slave, bool await, bool response) {
  worker->sendDebugSlavePrimList(partition, crate, rod, slave, await, response);
}

void TApi::createDebugPrimList() {
  worker->createDebugPrimList();
}

void TApi::debugPrimListFromFile(const char *fileName) {
  worker->debugPrimListFromFile(fileName);
}

void TApi::dumpDebugPrimList() {
  worker->dumpDebugPrimList();
}

std::list<SctApi::RodLabel> TApi::listRods() {
  return worker->listRods();
}

/*
  Dump contents of a block of DSP memory to cout.
*/
int TApi::dspBlockDump(unsigned int partition, unsigned int crate, unsigned int rod,
                       long dspStart, long numWords, long dspNumber, bool usePrimitive){
  return worker->dspBlockDump(partition, crate, rod, dspStart, numWords, dspNumber, usePrimitive);
}

int TApi::dspBlockDumpFile(unsigned int partition, unsigned int crate, unsigned int rod,
                           long dspStart, long numWords, long dspNumber, const char *filename, bool usePrimitive)
{
  return worker->dspBlockDumpFile(partition, crate, rod, dspStart, numWords, dspNumber, filename, usePrimitive);
}

/*
  Read contents of a block of DSP memory.
*/
unsigned long *TApi::dspBlockRead(unsigned int partition, unsigned int crate, unsigned int rod,
                                  long dspStart, long numWords, long dspNumber, 
                                  unsigned long *length, bool usePrimitive) {
  unsigned long thing;
  unsigned long *result = worker->dspBlockRead(partition, crate, rod, dspStart, numWords, 
                                               dspNumber, thing, usePrimitive);
  *length = thing;
  return result;
}

int TApi::dspBlockWrite(unsigned int partition, unsigned int crate, unsigned int rod,
                        unsigned long *buffer, unsigned long dspAddress, long numWords, 
                        long dspNumber, bool usePrimitive) {
  return worker->dspBlockWrite(partition, crate, rod, 
                               buffer, dspAddress, numWords, dspNumber, usePrimitive);
}

unsigned long TApi::dspSingleRead(unsigned int partition, unsigned int crate, unsigned int rod,
                                  const unsigned long dspAddr, long dspNumber) {
  return worker->dspSingleRead(partition, crate, rod, dspAddr, dspNumber);
}

/// Write a single 32b word to MasterDSP SDRAM via HPI
void TApi::dspSingleWrite(unsigned int partition, unsigned int crate, unsigned int rod,
                   unsigned long dspAddr, unsigned long value, long dspNumber) {
  worker->dspSingleWrite(partition, crate, rod, dspAddr, value, dspNumber);
}

void TApi::defaultScan(int type) {
  worker->defaultScan(type);
}

void TApi::tidyHistogramming() {
  worker->tidyHistogramming();
}


void TApi::doScan(TScanDef tScan) {
  tScan.update();
  boost::shared_ptr<SctApi::Scan> scan = tScan.getScan();

  try {
    if(scan)
      worker->doScan(scan);
    else 
      std::cout << "Bad scan at Root interface level\n";
  } catch(SctApi::SctApiException &s) {
    std::cout << "Exception thrown by doScan:\n";
    std::cout << s.what() << std::endl;
  }
}

void TApi::awaitScan() {
  worker->awaitScan();
}

void TApi::doRawScan(TScanDef tScan, int delay, bool configure, bool clkBy2) {
  tScan.update();
  boost::shared_ptr<SctApi::Scan> scan = tScan.getScan();

  try {
    if(scan)
      worker->doRawScan(scan, delay, 768, configure, clkBy2);
    else 
      std::cout << "Bad scan at Root interface level\n";
  } catch(SctApi::SctApiException &s) {
    std::cout << "SctApi Exception thrown by doRawScan:\n";
    std::cout << s.what() << std::endl;
  } catch(...) {
    std::cout << "Unknown Exception thrown by doRawScan:\n";
  }
}

void TApi::sendTrigger(unsigned int partition, unsigned int crate, unsigned int rod, TTrigger tTrig) {
  tTrig.update();

  boost::shared_ptr<SctApi::Trigger> trig = tTrig.getTrigger();

  if(trig)
    worker->sendTrigger(partition, crate, rod, trig.get());
  else 
    std::cout << "Bad trigger at Root interface level\n";
}

void TApi::status() {
  worker->status();
}

unsigned long TApi::readRodCommandReg(unsigned int partition, unsigned int crate, unsigned int rod,
                                      long regNumber) {
  return worker->readRodCommandReg(partition, crate, rod, regNumber);
}

unsigned long TApi::readRodStatusReg(unsigned int partition, unsigned int crate, unsigned int rod,
                                     long regNumber) {
  return worker->readRodStatusReg(partition, crate, rod, regNumber);
}

void TApi::setABCDModule(UINT32 mid, UINT32 bank) {
  try {
    worker->setABCDModule(mid, SctApi::BankType(bank));
  } catch(SctApi::SctApiException &s) {
    std::cout << "Exception thrown by setABCDModule:\n";
    std::cout << s.what() << std::endl;
  }
}

void TApi::setABCDModules(UINT32 bank) {
  try {
    worker->setABCDModules(SctApi::BankType(bank));
  } catch(SctApi::SctApiException &s) {
    std::cout << "Exception thrown by setABCDModules:\n";
    std::cout << s.what() << std::endl;
  }
}

unsigned long *TApi::retrieveModule(UINT32 mid) {
  return (unsigned long*)worker->retrieveModule(mid);
}

void TApi::printABCDModule(int mid) {
  worker->printABCDModule(mid);
}

void TApi::printABCDRodModule(int mid, int bank) {
  worker->printABCDRodModule(mid, SctApi::BankType(bank));
}

void TApi::printBOCSetup(unsigned int partition, unsigned int crate, unsigned int rod) {
  checkCall(boost::bind(&SctApi::SctApi::printBOCSetup, worker, 
                        partition, crate, rod));
}

void TApi::currentBOCSetup(unsigned int partition, unsigned int crate, unsigned int rod) {
  try {
    std::vector<SctConfiguration::BOCChannelConfig> setup = worker->currentBOCSetup(partition, crate, rod);

    std::cout << "Got setup length " << setup.size() << std::endl;

    for(unsigned int i=0; i<setup.size(); i++) {
      SctConfiguration::BOCChannelConfig conf = setup[i];

      std::cout << conf.current << " " << conf.delay << " " << conf.markSpace << std::endl;
      std::cout << conf.threshold0 << " " << conf.delay0 << std::endl;
      std::cout << conf.threshold1 << " " << conf.delay1 << std::endl;
    }
  } catch(SctApi::SctApiException &s) {
    std::cout << "Exception in currentBOCSetup " << s.what() << std::endl;
  }
}

void TApi::printBOCRegisters(unsigned int partition, unsigned int crate, unsigned int rod) {
  checkCall(boost::bind(&SctApi::SctApi::printBOCRegisters, worker, partition, crate, rod));
}

void TApi::currentBOCRegisters(unsigned int partition, unsigned int crate, unsigned int rod) {
  checkCall(boost::bind(&SctApi::SctApi::currentBOCRegisters, worker, partition, crate, rod));
}

void TApi::saveBOCSetup(unsigned int partition, unsigned int crate, unsigned int rod, UINT32 bank) {
  worker->saveBOCSetup(partition, crate, rod, SctApi::BankType(bank));
}

void TApi::saveBOCRegisters(unsigned int partition, unsigned int crate, unsigned int rod, UINT32 bank) {
  worker->saveBOCRegisters(partition, crate, rod, SctApi::BankType(bank));
}

void TApi::restoreBOCSetup(unsigned int partition, unsigned int crate, unsigned int rod, UINT32 bank) {
  worker->restoreBOCSetup(partition, crate, rod, SctApi::BankType(bank));
}

void TApi::restoreBOCRegisters(unsigned int partition, unsigned int crate, unsigned int rod, UINT32 bank) {
  worker->restoreBOCRegisters(partition, crate, rod, SctApi::BankType(bank));
}

void TApi::rawData(unsigned int partition, unsigned int crate, unsigned int rod, 
                   int delay, int units, bool setMask, TTrigger tTrig) {
  tTrig.update();

  boost::shared_ptr<SctApi::Trigger> trig = tTrig.getTrigger();

  if(trig)
    worker->rawData(partition, crate, rod, delay, units, setMask, trig.get());
  else 
    std::cout << "Bad trigger to raw data at Root interface level\n";
}

char *TApi::probe(unsigned int partition, unsigned int crate, unsigned int rod, signed int harness) {
  std::vector<char> result = worker->probe(partition, crate, rod, harness);
  int length = result.size();
  char *buffer = new char[length + 1];

  for (int i=0; i<length; i++) {
    buffer[i] = result[i];
  }

  buffer[length] = 0;

  return buffer;
}

char *TApi::probeWithTrigger(unsigned int partition, unsigned int crate, unsigned int rod, 
                             TTrigger tTrig, signed int harness) {
  tTrig.update();

  boost::shared_ptr<SctApi::Trigger> trig = tTrig.getTrigger();

  if(trig) {
    std::vector<char> result = worker->probeWithTrigger(partition, crate, rod, trig.get(), harness);
    int length = result.size();
    char *buffer = new char[length + 1];

    for (int i=0; i<length; i++) {
      buffer[i] = result[i];
    }

    buffer[length] = 0;

    return buffer;
  } else {
    std::cout << "Bad trigger to raw data at Root interface level\n";
    return 0;
  }
}

void TApi::probeScan(unsigned int partition, unsigned int crate, unsigned int rod, 
                     TScanDef tScan, signed int harness) {
  tScan.update();
  boost::shared_ptr<SctApi::Scan> scan = tScan.getScan();

  if(!scan) {
    std::cout << "Bad scan at Root interface level\n";
    return;
  }

  std::cout << "Summary\n";
  std::vector< std::vector <char> > result = worker->probeScan(partition, crate, rod, scan, harness);
  int outerLength = result.size();
  int innerLength = 0;
  if(outerLength > 0) {
    innerLength = result[0].size();
  }
  for(int i=0; i<innerLength; i++) {
    for(int j=0; j<outerLength; j++) {
      std::cout << result[j][i];
    }
    std::cout << std::endl;
  }
}

bool TApi::checkAllModulesProbe(const char *value) {
  return worker->checkAllModulesProbe(value);
}

void TApi::autoConfigure() {
  ::SctApi::AutoConf::AutoConfigurer configurer(*worker);
  configurer.run();

  std::list< ::SctApi::AutoConf::AutoResult> results = configurer.getResults();

  std::cout << results.size() << " auto-configure results:\n";

  for(std::list< ::SctApi::AutoConf::AutoResult>::const_iterator autoIter = results.begin();
      autoIter != results.end(); autoIter ++) {
    const ::SctApi::AutoConf::AutoResult &autoResult = *autoIter;

    std::cout << "r: " << autoResult.rod << " tx: " << autoResult.tx << " rx: " << autoResult.rx << std::endl;
  }
}

void TApi::rodMode(unsigned int partition, unsigned int crate, unsigned int rod,
                   int mode, int flag, int fifoSetup, int nBins, int delay, int message) {
  worker->rodMode(partition, crate, rod, 
                  mode, flag, fifoSetup, nBins, delay, message);
}

void TApi::bocHistogram(unsigned int partition, unsigned int crate, unsigned int rod, 
                        unsigned int samples, unsigned int numLoops)
{
  worker->bocHistogram(partition, crate, rod, samples, numLoops);
}

void TApi::testLinkOutSelect(unsigned int partition, unsigned int crate, unsigned int rod,
                             unsigned int link) {
  worker->testLinkOutSelect(partition, crate, rod, link);
}

void TApi::setDebugOption(const char *opt) {
  worker->setDebugOption(opt);
}

void TApi::unsetDebugOption(const char *opt) {
  worker->unsetDebugOption(opt);
}

void TApi::listEnabledDebugOptions() {
  std::list<std::string> theList = worker->listEnabledDebugOptions();
  for(std::list<std::string>::const_iterator i = theList.begin();
      i != theList.end();
      i++) {
    std::cout << *i << std::endl;
  }
}

void TApi::listDebugOptions() {
  std::vector<std::string> theList = worker->listDebugOptions();
  for(std::vector<std::string>::const_iterator i = theList.begin();
      i != theList.end();
      i++) {
    std::cout << *i << std::endl;
  }
}

void TApi::debugStepHistogram() 
{
  worker->debugStepHistogram();
}

void TApi::debugContinueHistogram()
{
  worker->debugContinueHistogram();
}

void TApi::debugAbortHistogram()
{
  worker->debugAbortHistogram();
}

void TApi::standardRegisterDump(unsigned int partition, unsigned int crate, unsigned int rod) 
{
  worker->standardRegisterDump(partition, crate, rod);
}

void TApi::lasersOff() {
  worker->lasersOff();
}

void TApi::scanEvents(unsigned int partition, unsigned int crate, unsigned int rod, int sl, 
                      bool extFlag, bool errorType) {
  worker->scanEvents(partition, crate, rod, sl, extFlag, errorType);
}

void TApi::decodeEvent(unsigned int partition, unsigned int crate, unsigned int rod, 
                       int sl, int index, bool extFlag, bool errorType) {
  worker->decodeEvent(partition, crate, rod, sl, index, extFlag, errorType);
}

void TApi::decodeConfig(unsigned int partition, unsigned int crate, unsigned int rod, 
                        bool skipTrim, bool bypass) {
  worker->decodeConfig(partition, crate, rod, skipTrim, bypass);
}

void TApi::timSetFrequency(unsigned int partition, unsigned int crate, 
                           double trigFreq, double rstFreq) {
  worker->timSetFrequency(partition, crate, trigFreq, rstFreq);
}

void TApi::freeTriggers(unsigned int partition, unsigned int crate) {
  worker->freeTriggers(partition, crate);
}

void TApi::stopTriggers(unsigned int partition, unsigned int crate) {
  worker->stopTriggers(partition, crate);
}

void TApi::timL1A(unsigned int partition, unsigned int crate) {
  worker->timL1A(partition, crate);
}

void TApi::timCalL1A(unsigned int partition, unsigned int crate, int delay) {
  worker->timCalL1A(partition, crate, delay);
}

void TApi::timSoftReset(unsigned int partition, unsigned int crate) {
  worker->timSoftReset(partition, crate);
}

void TApi::timBCReset(unsigned int partition, unsigned int crate) {
  worker->timBCReset(partition, crate);
}

void TApi::sendTimBurst(unsigned int partition, unsigned int crate, int count) {
  checkCall(boost::bind(&SctApi::SctApi::sendTimBurst, worker, partition, crate, count));
}

void TApi::timVerbose(unsigned int partition, unsigned int crate) {
  worker->timVerbose(partition, crate);
}

void TApi::timRegLoad(unsigned int partition, unsigned int crate, int reg, UINT16 val) {
  worker->timWriteRegister(partition, crate, reg, val);
}

UINT16 TApi::timReadRegister(unsigned int partition, unsigned int crate, int reg) {
  return worker->timReadRegister(partition, crate, reg);
}

void TApi::requestHardReset(UINT32 mid) {
  worker->requestHardReset(mid);
}

void TApi::resumePolling() {
  worker->resumePolling();
}

void TApi::stopPolling() {
  worker->stopPolling();
}


TScanDef::TScanDef() :
  trigsPerBurst(200),
  scanVariable(1),  scanVariable2(1),
  full(0), bits32(0), loopCalLine(0),
  distSlave(0), debug(0), tim(0), nth(0), nth_rem(0)
{
  worker.reset(new SctApi::ScanDefImpl);
  // A large threshold scan
  configure(1, 0.0, 400.0, 5.0);

  // With a single L1A trigger
  trigSequence.singleL1A();

  // So the worker has pointers to the TriggerImpl's in the trigsequences
  update();
}

TScanDef::~TScanDef() {
//   delete worker;
}

TScanDef::TScanDef(const TScanDef &other) :
  trigsPerBurst(other.trigsPerBurst),

  scanVariable(other.scanVariable),
  scanVariable2(other.scanVariable2),

  trigSequence(other.trigSequence),
  trigSequence2(other.trigSequence2),

  full(other.full), 
  bits32(other.bits32),
  loopCalLine(other.loopCalLine),

  distSlave(other.distSlave),
  debug(other.debug),
  tim(other.tim),
  nth(other.nth),
  nth_rem(other.nth_rem)
{
  worker.reset(new SctApi::ScanDefImpl(*other.worker));
}

void TScanDef::update() {
  // Only copy the things which are defined as parameters
  // (ScanPoints are set up in worker by the configure method)
  if(trigsPerBurst != -1)
    worker->setNTrigs(trigsPerBurst);

  worker->setScanVariable1(scanVariable);
  worker->setScanVariable2(scanVariable2);

  // Copy triggers
  trigSequence.update();
  worker->setTrigger1(trigSequence.getTrigger());
  trigSequence2.update();
  worker->setTrigger2(trigSequence2.getTrigger());

  // Set options
  worker->setOption(SctApi::Scan::FULL, full);
  worker->setOption(SctApi::Scan::BITS32, bits32);
  worker->setOption(SctApi::Scan::LOOPCALLINE, loopCalLine);
  worker->setOption(SctApi::Scan::DISTSLAVE, distSlave);
  worker->setOption(SctApi::Scan::DEBUG, debug);
  worker->setOption(SctApi::Scan::TIM, tim);
  worker->setOption(SctApi::Scan::NTH, nth);
  worker->setOption(SctApi::Scan::NTH_REM, nth_rem);
}

void TScanDef::print() {
  update();
  worker->print(); // copyScan(this).print();
}

void TScanDef::configure(UINT16 type, FLOAT32 start, FLOAT32 stop, FLOAT32 step) {
  scanVariable = type;
  worker->configure(type, start, stop, step);
}

void TScanDef::configure2(UINT16 type, FLOAT32 start, FLOAT32 stop, FLOAT32 step) {
  scanVariable2 = type;
  worker->configure2(type, start, stop, step);
}

void TScanDef::setScanPoint(int index, FLOAT32 value) {
  worker->scanPoints[index] = value;
}

void TScanDef::setScanPoint2(int index, FLOAT32 value) {
  worker->scanPoints2[index] = value;
}

void TScanDef::setTriggersPoint(int index, UINT32 nTrigs) {
  if(trigsPerBurst != -1) {
    worker->setNTrigs(trigsPerBurst);
  }
  trigsPerBurst = -1;
  worker->allTrigsSame = false;
  worker->trigPoints[index] = nTrigs;
}

// This is the default for both set 0 and set 1
TTrigger::TTrigger() {
  worker.reset(new SctApi::TriggerImpl);
  incCmd = 0;
  incData = 0;

  update();
}

TTrigger::TTrigger(const TTrigger &other) {
  worker.reset(new SctApi::TriggerImpl(*other.worker));
  incCmd = other.incCmd;
  incData = other.incData;
}

TTrigger::~TTrigger() {
  // This is deleted by ScanDefImpl not via TScanDef
  //  delete worker;
}

void TTrigger::singleL1A() {
  worker->singleL1A();
}

void TTrigger::doubleL1A(int delay) {
  worker->doubleL1A(delay);
}

void TTrigger::delayedL1A(int delay) {
  worker->delayedL1A(delay);
}

void TTrigger::calL1A(int delay) {
  worker->calL1A(delay);
}

void TTrigger::pulseL1A(int delay) {
  worker->pulseL1A(delay);
}

void TTrigger::softL1A(int delay) {
  worker->softL1A(delay);
}

void TTrigger::softCalL1A(int delay, int delay2) {
  worker->softCalL1A(delay, delay2);
}

void TTrigger::softPulseL1A(int delay, int delay2) {
  worker->softPulseL1A(delay, delay2);
}

void TTrigger::bcL1A(int delay) {
  worker->bcL1A(delay);
}

void TTrigger::bcCalL1A(int delay, int delay2) {
  worker->bcCalL1A(delay, delay2);
}

void TTrigger::bcPulseL1A(int delay, int delay2) {
  worker->bcPulseL1A(delay, delay2);
}

void TTrigger::update() {
  worker->setCommIncr(incCmd, incData);
}

boost::shared_ptr<SctApi::Trigger> TriggerWrapper::getTrigger() {
  return worker;
}

// Copy TScanDef scan to implementation of Scan interface 
//   update(); must be called on the scan first
boost::shared_ptr<SctApi::Scan> ScanDefWrapper::getScan() {
//   if((tScan.trigSequence.trigCommand.GetSize() != tScan.trigSequence.trigData.GetSize()) ||
//      (tScan.trigSequence2.trigCommand.GetSize() != tScan.trigSequence2.trigData.GetSize()) ||
//      ((tScan.trigSequence.trigCommand.GetSize() != tScan.trigSequence2.trigCommand.GetSize()) && 
//       (tScan.trigSequence2.trigData.GetSize() != 0))) {
//     cout << "Trigger arrays not compatible! Aborting scan...\n";
//     return;
//   }

  return worker;
}
