#include "primListWrapper.h"
#include "PrimBuilder.h"
#include "SctApi.h"
#include "SctApiRodInfo.h"
#include "crate.h"
#include "utility.h"

#include "CommonWithDsp/primParams.h"
#include "CommonWithDsp/registerIndices.h"

using namespace std;
using namespace SctPixelRod;

namespace SctApi {

void SctApi::getABCDModules(BankType bank) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "getABCDModule (all). Bank: " << bank << endl;
  }
  for(map<UINT32, ABCDModule> ::const_iterator iter = moduleMap.begin();
      iter != moduleMap.end();
      iter ++) {
    getABCDModule(iter->first, bank);
  }
}

void SctApi::getABCDModule(UINT32 mid, BankType bank) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "getABCDModule " << mid << " Bank: " << bank << endl;
  }

  // Find module channel in ROD
  unsigned int partition, crate, rod, channel;
  Utility::getpcrc(mid, partition, crate, rod, channel);

  // Which bank id to use
  int realBank = Utility::translateBank(bank);

  boost::shared_ptr<PrimListWrapper> primList(new PrimListWrapper(1));
  PrimBuilder::instance().readModuleData(primList, realBank, channel);

  sendPrimList(partition, crate, rod, primList);

  int responseCode = awaitResponse(partition, crate, rod, 5);

  unsigned long length;
  unsigned long *outBody = 0;
  if(responseCode == 0) {
    outBody = getResponse(partition, crate, rod, length);
  }

  if(outBody) {
//      unsigned long outLength = outBody[0];
//      unsigned long outIndex = outBody[1];
    unsigned long outNumPrims = outBody[2];
//      unsigned long outPrimVersion = outBody[3];

    if(outNumPrims == 1) {
      unsigned long primLength = outBody[4];
//        unsigned long primIndex = outBody[5];
      unsigned long primID = outBody[6];
//        unsigned long primVersion = outBody[7];

      if(primID == RW_MODULE_DATA) {
        if(primLength != (sizeof(ABCDModule)/sizeof(UINT32) + 4)) { // Prim length includes header
          cout << "Output primitive was " << primLength << " not " << sizeof(ABCDModule)/sizeof(UINT32) << endl;
        }

        // Data is at outBody[8]
        ABCDModule result;

        // Copy data
        for(unsigned int i=0; i<sizeof(ABCDModule)/sizeof(UINT32); i++) {
          ((unsigned long *)&result)[i] = outBody[8 + i];
        }

        ABCDModule *previous = lookupConfig(mid);

        if(previous) {
          cout << "Replacing previous module config\n";

          *previous = result;
        } else {
          cout << "New module config inserted\n";
          moduleMap.insert(make_pair(mid, result));
        }
      } else {
        cout << "Returned primitive was not RW_MODULE_DATA\n";
      }
    } else {
      cout << "Too many primitives in output list\n";
    }

    delete [] outBody;
  } else {
    cout << "No response to primitive list!\n";
  }
}

boost::shared_ptr<ABCDModule> SctApi::getABCDModuleRaw(unsigned int partition, unsigned int crate, unsigned int rod, 
                                                       UINT32 slot, BankType bank) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "getABCDModuleRaw Slot: " << slot << " Bank: " << bank << endl;
  }

  boost::shared_ptr<ABCDModule> result;

  // Which bank id to use
  int realBank = Utility::translateBank(bank);

  boost::shared_ptr<PrimListWrapper> primList(new PrimListWrapper(1));
  PrimBuilder::instance().readModuleData(primList, realBank, slot);
  sendPrimList(partition, crate, rod, primList);

  int responseCode = awaitResponse(partition, crate, rod, 5);

  unsigned long length;
  unsigned long *outBody = 0;
  if(responseCode == 0) {
    outBody = getResponse(partition, crate, rod, length);
  }

  if(outBody) {
    unsigned long outNumPrims = outBody[2];

    if(outNumPrims == 1) {
      unsigned long primLength = outBody[4];
      unsigned long primID = outBody[6];

      if(primID == RW_MODULE_DATA) {
        unsigned int expectedSize = (sizeof(ABCDModule)/sizeof(UINT32) + 4);
        if(primLength != expectedSize) { // Prim length includes header
          cout << "Output primitive was " << primLength << " not " << expectedSize << endl;
        }

        result.reset(new ABCDModule);

        // Data is at outBody[8]
        // Copy data
        for(unsigned int i=0; i<sizeof(ABCDModule)/sizeof(UINT32); i++) {
          ((unsigned long *)result.get())[i] = outBody[8 + i];
        }
      } else {
        cout << "Returned primitive was not RW_MODULE_DATA\n";
      }
    } else {
      cout << "Too many primitives in output list\n";
    }

    delete [] outBody;
  } else {
    cout << "No response to primitive list!\n";
  }

  return result;
}

/* Store config in ROD */
void SctApi::setABCDModules(BankType bank) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "setABCDModule called for all modules. Bank " << bank << endl;
  }
  for(map<UINT32, ABCDModule>::const_iterator iter = moduleMap.begin();
      iter != moduleMap.end();
      iter ++) {
    cout << "Set Config " << iter->first << endl;
    setABCDModule(iter->first, bank);
  }

  for(list<RodLabel>::const_iterator rl = rodList.begin();
      rl!=rodList.end();
      rl++){
    RodInfo info = getRodInfo(*rl);

    for(std::map<unsigned long, unsigned long>::const_iterator iter = info.offRODRX.begin();
        iter != info.offRODRX.end(); iter++) {
      cout << "Set config " << hex << iter->first << " (off ROD pair of " << iter->second << ")" << dec << endl;
      setABCDModule(iter->first, bank);
    }
  }

  return;
}

/* Store config in ROD */
void SctApi::setABCDModule(UINT32 mid, BankType bank) {
  if(mid==0xffffffff){
    cout << "setABCDModule with 0xffffffff\n";
    throw SctApiException("mid out of range");
  }

  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "setABCDModule " << mid << " Bank: " << bank << endl;
  }

  unsigned int partition, crate, rod, channel;
  Utility::getpcrc(mid, partition, crate, rod, channel);

  int realBank = Utility::translateBank(bank);

  std::string moduleName;
  ABCDModule *moduleConfig;

  moduleConfig = lookupConfig(mid);

  if(!moduleConfig) {
    cout << "*** NO module configuration for mid = " << mid << " in set module" << endl;
    return;
  }

  cout << "\nModule config\n";
  cout << " primary channel " << (int)moduleConfig->pTTC 
       << " red channel " << (int)moduleConfig->rTTC 
       << " group " << (int)moduleConfig->groupId
       << " select " << (int)moduleConfig->select << endl;
  cout << " target (chip 0) " << (int)moduleConfig->chip[0].target << endl;

  boost::shared_ptr<PrimListWrapper> primList(new PrimListWrapper(1));
  PrimBuilder::instance().writeModuleData(primList, moduleConfig, realBank, channel);

  // This will be a null op if it doesn't exist
#ifdef CMD_BUFFER_0
  setupModuleMask(CMD_BUFFER_0, 0xf, primList);
#else
  setupModuleMask(SP0, 0xf, primList);
#endif
  sendPrimList(partition, crate, rod, primList);

  int responseCode = awaitResponse(partition, crate, rod, 10);

  if(responseCode!=0) {
    cout << "Set module unsuccessful!\n";
  }
}

void SctApi::sendAllABCDModules(BankType bank, ConfigType type) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "sendABCDModule called for all modules. Bank: " << bank << endl;
  }

  cout << "Send config to all modules\n";

  PrimBuilder &builder = PrimBuilder::instance();

  // Set ROD mode to something appropriate (may be in TIM mode!)
  boost::shared_ptr<PrimListWrapper> rodModeList1(new PrimListWrapper(1));
  // Should CALIBRATION_SLINK_OVERRIDE_MODE be added to this (it works without!)
  builder.rodMode(rodModeList1, CALIBRATION_MODE, 0, 1, 1, 1, 1);

  synchSendPrimListAllCrates(rodModeList1);


#if (R_SEND_CONFIG == 104) || (R_SEND_CONFIG == 105)
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "\tUsing one SEND_CONFIG primitive\n";
  }

  {
    for(list<RodLabel>::const_iterator rl = rodList.begin();
        rl!=rodList.end();
        rl++){

      if(!getCrate(rl->partition, rl->crate)->checkBOCLasersOn(rl->rod)) {
        cout << "Trying to send module configuration using BOC that has its lasers cut out\n";
        if(mrs) {
          *mrs << "BOC_INTERLOCKED" << MRS_ERROR << MRS_QUALIF("SCTAPI") 
               << MRS_PARAM<int>("crate", rl->crate) 
               << MRS_PARAM<int>("rod", rl->rod)
               << MRS_TEXT("Can't send all module config (BOC interlocked)") << ENDM;
        }

        throw SctApiException("Send modules aborted due to interlock status");
      }
    }
  }

  boost::shared_ptr<PrimListWrapper> sendConfigList(new PrimListWrapper(1));

  // Write "Disable trigger decoder" to primList
#ifdef RRIF_CMND_1
  builder.writeRegister(sendConfigList, RRIF_CMND_1, 18, 1, 0);
#else
#error "Unsupported no registers"
#endif

  // Disable all formatters
#ifdef FMT_LINK_EN
  for(int i=0; i<8; i++) {
    builder.writeRegister(sendConfigList, FMT_LINK_EN(i), 0, 32, 0);
  }
#else
#warning "Probably don't need to turn off formatters!?"
#endif

  int realBank = Utility::translateBank(bank);

  //@todo SP_BOTH doesn't work with current DSP code
//  builder.sendConfig(sendConfigList, SP_BOTH, 0, ALL_MODULES, ALL_MODULES, ALL_CHIPS, 1, 1,
//                                     realBank, MODULE_GROUP_ALL, type, 0, 1);
  builder.sendConfig(sendConfigList, SP0, 0, ALL_MODULES, ALL_MODULES, ALL_CHIPS, 1, 1,
                     realBank, MODULE_GROUP_ALL, type, 0, 1);

  // Write "Re-enable trigger decoder" to primList
#ifdef RRIF_CMND_1
  builder.writeRegister(sendConfigList, RRIF_CMND_1, 18, 1, 1);
#else
#error "Unsupported no registers"
#endif

  // setupModuleMask to reenable appropriate formatters??

  for(map<pair<unsigned int, unsigned int>, Crate* >::const_iterator ci = crateMap.begin();
      ci != crateMap.end();
      ci++) {
    sendPrimListAll(ci->first.first, ci->first.second, sendConfigList);
  }

  for(map<pair<unsigned int, unsigned int>, Crate* >::const_iterator ci = crateMap.begin();
      ci != crateMap.end();
      ci++) {
    int responseCode = awaitResponseAll(ci->first.first, ci->first.second, 10);

    if(responseCode != 0) {
      cout << "Send modules unsuccessful\n";
    }
  }

  boost::shared_ptr<PrimListWrapper> rodModeList(new PrimListWrapper(1));
  // Should CALIBRATION_SLINK_OVERRIDE_MODE be added to this (it works without!)
  builder.rodMode(rodModeList, CALIBRATION_MODE, 0, 1, 1, 1, 1);

  for(map<pair<unsigned int, unsigned int>, Crate* >::const_iterator ci = crateMap.begin();
      ci != crateMap.end();
      ci++) {
    sendPrimListAll(ci->first.first, ci->first.second, rodModeList);
  }

  for(map<pair<unsigned int, unsigned int>, Crate* >::const_iterator ci = crateMap.begin();
      ci != crateMap.end();
      ci++) {
    int responseCode = awaitResponseAll(ci->first.first, ci->first.second, 10);

    if(responseCode != 0) {
      cout << "Send modules rod mode unsuccessful\n";
    }
  }
#else
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "\tUsing multiple SEND_CONFIG primitives\n";
  }

#warning "Slow module sending"

  // Disable all formatters to avoid bus error!
#ifdef FMT_LINK_EN
  boost::shared_ptr<PrimListWrapper> clearFormatList(new PrimListWrapper(1));

  for(int i=0; i<8; i++) {
    builder.writeRegister(clearFormatList, FMT_LINK_EN(i), 0, 32, 0);
  }

  for(map<pair<unsigned int, unsigned int>, Crate* >::const_iterator ci = crateMap.begin();
      ci != crateMap.end();
      ci++) {
    sendPrimListAll(ci->first.first, ci->first.second, clearFormatList);

    int responseCode = awaitResponseAll(ci->first.first, ci->first.second, 10);

    if(responseCode != 0) {
      cout << "Clear formatters unsuccessful\n";
    }
  }
#else
#warning "Probably don't need to turn off formatters! (old version)"
#endif

  for(map<UINT32, ABCDModule>::const_iterator iter = moduleMap.begin();
      iter != moduleMap.end();
      iter ++) {
    cout << "SendConfig " << iter->first << endl;
    sendABCDModule(iter->first, bank, type);
  }
#endif   // End of slow config sending

  // Restore link masks etc
  boost::shared_ptr<PrimListWrapper> setupMasks(new PrimListWrapper(1));

#ifdef CMD_BUFFER_0
  setupModuleMask(CMD_BUFFER_0, 0xf, setupMasks);
#else
  setupModuleMask(SP0, 0xf, setupMasks);
#endif

  for(map<pair<unsigned int, unsigned int>, Crate* >::const_iterator ci = crateMap.begin();
      ci != crateMap.end();
      ci++) {
    sendPrimListAll(ci->first.first, ci->first.second, setupMasks);

    int responseCode = awaitResponseAll(ci->first.first, ci->first.second, 10);

    if(responseCode != 0) {
      cout << "Setup masks unsuccessful\n";
    }
  }

  // Reset everything so its not confused by the clk/2
  boost::shared_ptr<PrimListWrapper> resetList(new PrimListWrapper(1));

  cout << "Resetting front end interface\n";
#ifdef RRIF_CMND_0
  builder.writeRegister(resetList, RRIF_CMND_0, 0, 8, 0xff);  // Reset everything
  builder.writeRegister(resetList, RRIF_CMND_0, 0, 8, 0x78);  // Leave debug FIFO in reset
#else
#error "Unsupported no registers"
#endif

  for(map<pair<unsigned int, unsigned int>, Crate* >::const_iterator ci = crateMap.begin();
      ci != crateMap.end();
      ci++) {
    sendPrimListAll(ci->first.first, ci->first.second, resetList);

    int responseCode = awaitResponseAll(ci->first.first, ci->first.second, 10);

    if(responseCode != 0) {
      cout << "ROD front end reset unsuccessful\n";
    }
  }

  return;
}

void SctApi::sendABCDModule(UINT32 mid, BankType bank, ConfigType type) {
  if(mid==0xffffffff){
    cout << "sendABCDModule with 0xffffffff\n";
    throw SctApiException("mid out of range");
  }

  cout << "Send config to module " << mid << endl;

  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "sendABCDModule: Mid: " << mid << " Bank: " << bank << " Type: " << type << endl;
  }

  unsigned int partition, crate, rod, channel;
  Utility::getpcrc(mid, partition, crate, rod, channel);

  if(!getCrate(partition, crate)) {
    cout << "Send module for non-existent crate " << partition << " " << crate << endl;
    return;
  }

  int realBank = Utility::translateBank(bank);

  if(realBank > SPARE_MODULE_CONFIG) {
    cout << "sendABCDModule with illegal bank (translated) " << realBank << endl;
    throw SctApiException("Illegal Bank");
  }

  if(!getCrate(partition, crate)->checkBOCLasersOn(rod)) {
    cout << "Trying to send module configuration using BOC that has its lasers cut out\n";
    if(mrs) {
      *mrs << "BOC_INTERLOCKED" << MRS_ERROR << MRS_QUALIF("SCTAPI") 
           << MRS_PARAM<int>("module", mid) << MRS_PARAM<int>("crate", crate) 
           << MRS_PARAM<int>("rod", rod)
           << MRS_TEXT("Can't send module config (BOC interlocked)") << ENDM;
    }

    return;
  }

  boost::shared_ptr<PrimListWrapper> primList(new PrimListWrapper(1));
  PrimBuilder &builder = PrimBuilder::instance();

  // Write "Disable trigger decoder" to primList
#ifdef RRIF_CMND_1
  builder.writeRegister(primList, RRIF_CMND_1, 18, 1, 0);
#else
#error "Unsupported no registers"
#endif

  builder.sendConfig(primList, SP0, 0, channel, NO_MODULE, ALL_CHIPS, 1, 1,
                     realBank, MODULE_GROUP_ALL, type, 0, 1);

  // Write "Re-enable trigger decoder" to primList
#ifdef RRIF_CMND_1
  builder.writeRegister(primList, RRIF_CMND_1, 18, 1, 1);
#else
#error "Unsupported no registers"
#endif

  sendPrimList(partition, crate, rod, primList);

  int responseCode = awaitResponse(partition, crate, rod, 10);

  if(responseCode != 0) {
    cout << "Send module unsuccessful\n";
  }
}

void SctApi::modifyABCDVarROD(UINT32 mid, UINT32 chip, UINT32 type, FLOAT32 value, BankType bank) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "modifyABCDVarROD (module " << mid << ": chip " << chip  << ") " << type << " " << value << "\n";
  }

  cout << "Change variable macro\n";

  unsigned int partition, crate, rod, channel;
  Utility::getpcrc(mid, partition, crate, rod, channel);

  if(!isRODPresent(partition, crate, rod)) {
    cout << "Trying to change module configuration on a non existant ROD\n";
    if(mrs) {
      *mrs << "ROD_UNCONFIGURED" << MRS_ERROR << MRS_QUALIF("SCTAPI") 
           << MRS_PARAM<int>("module", mid) << MRS_PARAM<int>("crate", crate) 
           << MRS_PARAM<int>("rod", rod)
           << MRS_TEXT("Can't modify module config (ROD non-existant)") << ENDM;
    }

    return;
  }

  boost::shared_ptr<PrimListWrapper> primList(new PrimListWrapper(1));

  int realBank = Utility::translateBank(bank);

  PrimBuilder::instance().writeModuleVariable(primList, realBank, MODULE_GROUP_ALL, channel, chip, type, value);

  sendPrimList(partition, crate, rod, primList);
  awaitResponse(partition, crate, rod, 5);
}

void SctApi::modifyABCDVarROD(UINT32 mid, UINT32 type, FLOAT32 value, BankType bank) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "modifyABCDVarROD (module " << mid << ") " << type << " " << value << "\n";
  }

  cout << "Change variable macro\n";

  unsigned int partition, crate, rod, channel;
  Utility::getpcrc(mid, partition, crate, rod, channel);

  if(!isRODPresent(partition, crate, rod)) {
    cout << "Trying to change module configuration on a non existant ROD\n";
    if(mrs) {
      *mrs << "ROD_UNCONFIGURED" << MRS_ERROR << MRS_QUALIF("SCTAPI") 
           << MRS_PARAM<int>("module", mid) << MRS_PARAM<int>("crate", crate) 
           << MRS_PARAM<int>("rod", rod)
           << MRS_TEXT("Can't modify module config (ROD non-existant)") << ENDM;
    }

    return;
  }

  boost::shared_ptr<PrimListWrapper> primList(new PrimListWrapper(1));

  int realBank = Utility::translateBank(bank);

  PrimBuilder::instance().writeModuleVariable(primList, realBank, MODULE_GROUP_ALL, channel, ALL_CHIPS, type, value);

  sendPrimList(partition, crate, rod, primList);
  awaitResponse(partition, crate, rod, 5);
}

void SctApi::modifyABCDVarROD(UINT32 type, FLOAT32 value, BankType bank) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "modifyABCDVarROD (all modules, all chips) " << type << " " << value << " Bank: " << bank << "\n";
  }

  cout << "Change variable macro\n";

  boost::shared_ptr<PrimListWrapper> primList(new PrimListWrapper(1));

  int realBank = Utility::translateBank(bank);

  PrimBuilder::instance().writeModuleVariable(primList, realBank, MODULE_GROUP_ALL, ALL_MODULES, ALL_CHIPS, type, value);

  for(list<RodLabel>::const_iterator rl = rodList.begin();
      rl!=rodList.end();
      rl++){

    unsigned int partition = rl->partition, crate = rl->crate, rod = rl->rod;

    if(!isRODPresent(partition, crate, rod)) {
      cout << "Trying to change module configuration on a non existant ROD\n";
      if(mrs) {
        *mrs << "ROD_UNCONFIGURED" << MRS_ERROR << MRS_QUALIF("SCTAPI") 
             << MRS_PARAM<int>("crate", crate) 
             << MRS_PARAM<int>("rod", rod)
             << MRS_TEXT("Can't modify module config (ROD non-existant)") << ENDM;
      }

      return;
    }

    sendPrimList(partition, crate, rod, primList);
    awaitResponse(partition, crate, rod, 2);
  }
}

} // Close namespace
