// Low level ROD access, no access to modules...
#include <iostream>

#include "SctApi.h"
#include "SctApiDebug.h"
#include "primListWrapper.h"
#include "PrimBuilder.h"

#include "RodCrate/RodPrimitive.h"

#include "CommonWithDsp/primParams.h"

using namespace std;
using namespace SctPixelRod;

namespace SctApi {

void SctApi::writeRODRegister(unsigned int partition, unsigned int crate, unsigned int rod,
                              int reg, int off, int width, int value) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "Write ROD register 0x" << hex << reg << dec << endl;
  }

  boost::shared_ptr<PrimListWrapper> regList(new PrimListWrapper(2));

  PrimBuilder::instance().writeRegister(regList, reg, off, width, value);

  sendPrimList(partition, crate, rod, regList);

  int responseCode = awaitResponse(partition, crate, rod, 1);

  if(responseCode != 0) {
    cout << "Register write 0x" << hex << reg << dec << " failed!\n";
  }
}

unsigned int SctApi::readRODRegister(unsigned int partition, unsigned int crate, unsigned int rod, int r) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "Read ROD reg 0x" << hex << r << dec << "\n";
  }

  boost::shared_ptr<PrimListWrapper> readList(new PrimListWrapper(2));

  PrimBuilder::instance().readRegister(readList, r);
 
  sendPrimList(partition, crate, rod, readList);

  int responseCode = awaitResponse(partition, crate, rod, 2);

  if(responseCode != 0) {
    cout << "Read ROD reg (0x" << hex << r << dec << ") primitive failed!\n";
  }

  unsigned long length;
  unsigned long *result = getResponse(partition, crate, rod, length);
  unsigned int val = 0;
  if(result) {
    val = result[8];

    delete [] result;
  } else {
    cout << "ReadRegister 0x" << hex << r << dec << " failed (no result)\n";
  }

  return val;
}

int SctApi::pollRegister(unsigned int partition, unsigned int crate, unsigned int rod, 
                         int r, int off, int width, int val, int timeout) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "Poll ROD register\n";
  }

  boost::shared_ptr<PrimListWrapper> pollList(new PrimListWrapper(2));
 
  PrimBuilder::instance().pollRegister(pollList, r, off, width, val, timeout);

  sendPrimList(partition, crate, rod, pollList);

  int responseCode = awaitResponse(partition, crate, rod, 5);

  if(responseCode != 0) {
    cout << "POLL ROD reg 0x" << hex << r << dec << " failed!\n";
    return 0;
  }

  unsigned long length;

  unsigned long *response = getResponse(partition, crate, rod, length);

  int result = 0;

  if(response) {
    if(response[8] == 1) {
      cout << "Success\n";
      result = 1;
    } else {
      cout << "Failure (timeout)\n";
      result = 0;
    }

    delete [] response;
  }

  return result;
}

unsigned long *SctApi::readFifo(unsigned int partition, unsigned int crate, unsigned int rod, int id, int bank, int elems) {
  {
    boost::mutex::scoped_lock lock(logMutex); 
    log << "readFIFO\n";
  }

  boost::shared_ptr<PrimListWrapper> fifoList(new PrimListWrapper(2));

  PrimBuilder::instance().readFifo(fifoList, id, bank, elems);

  sendPrimList(partition, crate, rod, fifoList);

  int responseCode = awaitResponse(partition, crate, rod, 2);

  if(responseCode != 0) {
    cout << "Read FIFO failed!\n";
  }

  unsigned long length;
  unsigned long *response = getResponse(partition, crate, rod, length);

  unsigned long *result;
  if(response) {
    unsigned long myLength = length-9-2;

    result = new unsigned long[myLength];

    for(unsigned int i=0; i<myLength; i++) {
      result[i] = response[i+9];
    }
  } else {
    result = 0;
  }

  return result;
}

void SctApi::echo(unsigned int partition, unsigned int crate, unsigned int rod,
                  unsigned int length, const unsigned long *data) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "Echo primitive\n";
  }

  boost::shared_ptr<PrimListWrapper> primList(new PrimListWrapper(1));

  // Create and Send a simple ECHO primitive
#if (R_ECHO != 100) 
#error "ECHO revision changed!"
#endif

  long *myData = new long[length];
  copy(data, data + length, myData);
  primList->addPrimitive(RodPrimitive(4+length, 1, ECHO, R_ECHO, myData),
                         myData);

  sendPrimList(partition, crate, rod, primList);

  if(checkDebugOption(DEBUG_DIAG)) {
    cout << "Echo primList sent\n";
  }
}

void SctApi::echoAll(unsigned int length, const unsigned long *data) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "Echo primitive\n";
  }

  boost::shared_ptr<PrimListWrapper> primList(new PrimListWrapper(1));

  // Create and Send a simple ECHO primitive
#if (R_ECHO != 100) 
#error "ECHO revision changed!"
#endif

  long *myData = new long[length];
  copy(data, data + length, myData);
  primList->addPrimitive(RodPrimitive(4+length, 1, ECHO, R_ECHO, myData),
                         myData);

  for(map<pair<unsigned int, unsigned int>, Crate* >::const_iterator crate = crateMap.begin();
      crate != crateMap.end();
      crate++) {
    sendPrimListAll(crate->first.first, crate->first.second, primList);
  }

  if(checkDebugOption(DEBUG_DIAG)) {
    cout << "Echo primList sent\n";
  }
}

void SctApi::echoSlave(unsigned int partition, unsigned int crate, unsigned int rod,
                       unsigned int slave, unsigned int length, const unsigned long *data) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "Send Echo primitive to slave\n";
  }

  boost::shared_ptr<PrimListWrapper> primList(new PrimListWrapper(1));

  // Create and Send a simple ECHO primitive
#if (R_ECHO != 100) 
#error "ECHO revision changed!"
#endif

  long *myData = new long[length];
  copy(data, data + length, myData);
  primList->addPrimitive(RodPrimitive(4+length, 1, ECHO, R_ECHO, myData),
                         myData);

  sendSlavePrimList(partition, crate, rod, primList, slave, 1, 1);

  if(checkDebugOption(DEBUG_DIAG)) {
    cout << "Echo primList sent\n";
  }
}

/*
    Flash the LED associated with a Slave DSP.
    This creates its own primitive list and sends it to the ROD
*/
void SctApi::flashLED(unsigned int partition, unsigned int crate, unsigned int rod,
                      long slaveNumber, long period, long flashes) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "FlashLED\n" << flush;
  }

  boost::shared_ptr<PrimListWrapper> primList(new PrimListWrapper(1));

  // Create a FLASH_LED primitive to be sent to a slave DSP
  long *ledData = new long[sizeof(FLASH_LED_IN)/4];

  FLASH_LED_IN &ledParam = *(FLASH_LED_IN *)ledData;

#if (R_FLASH_LED == 103) 
  ledParam.ledNum = YELLOW_LED;
  ledParam.period = period;
  ledParam.numTimes = flashes;
#else 
#error "FLASH_LED not compiled (revision change)"
#endif

  primList->addPrimitive(sizeof(FLASH_LED_IN)/4, 0, FLASH_LED, R_FLASH_LED, ledData);

  if(slaveNumber == -1) {
    // Send to master
    sendPrimList(partition, crate, rod, primList);
  } else {
    sendSlavePrimList(partition, crate, rod, primList, slaveNumber, true, false);
  }

  return;
}

pair<UINT32, UINT32> SctApi::sendData(unsigned int partition, unsigned int crate, unsigned int rod, 
                                      int type, int dsp) {
  // type was HISTOGRAM_DATA
  SEND_DATA_IN primData;
  primData.dataType = type;
  primData.auxVal = 0;
  primData.repBufferFlag = 0;
  primData.timeout = 0x20000;

  boost::shared_ptr<PrimListWrapper> primList(new PrimListWrapper(1));

  primList->addPrimitive(sizeof(primData)/4, 0, SEND_DATA, R_SEND_DATA, (const long *)&primData);

  if(dsp == -1) {
    sendPrimList(partition, crate, rod, primList);
  } else {
    sendSlavePrimList(partition, crate, rod, primList, dsp, true, true);
  }

  pair<UINT32, UINT32> result;

  int responseCode = awaitResponse(partition, crate, rod, 5);
  if(responseCode != 0) {
    cout << "Send_data failed!\n";

    throw SctApiException("SEND DATA didn't respond");
  } else {
    unsigned long dataLength;
    unsigned long *dataBody = getResponse(partition, crate, rod, dataLength);

    if(dataBody) {
      if(dsp == -1) {
        cout << "SEND DATA says ptr 0x" << hex << dataBody[8] << " length 0x" << dataBody[9] << dec << endl;

        result.first = dataBody[8];
        result.second = dataBody[9];
      } else {
        cout << "SEND DATA from dsp " << dsp 
             << " says ptr 0x" << hex << dataBody[16] << " length 0x" << dataBody[17] << dec << endl;

        result.first = dataBody[16];
        result.second = dataBody[17];
      }
    } else {
      cout << "SEND DATA returned nothing!\n";
      throw SctApiException("SEND DATA returned no data");
    }
  }

  return result;
}

}  // Close namespace
