// Anything that has to do with Scans

#include "SctApi.h"
#include "SctApiConsts.h"
#include "SctApiDebug.h"
#include "SctApiHisto.h"
#include "crate.h"
#include "primListWrapper.h"
#include "PrimBuilder.h"

#include "RodCrate/RodModule.h"

#include "CommonWithDsp/primParams.h"
#include "CommonWithDsp/registerIndices.h"

#include "ABCD/ABCDscans.h"
#include "ABCD/ABCDchip.h"

#include "ScanResultWriter/scan.h"
#include "utility.h"

#ifdef USE_IS
#include "Sct/IoExceptions.h"
#include "ScanResultWriter/ScanResultWriter.h"
#endif
#include "Sct/SctNames.h"

#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/lexical_cast.hpp>

#include <algorithm>

using namespace std;
using namespace Sct;
using namespace SctPixelRod;
using namespace boost::posix_time;
using boost::shared_ptr;

namespace SctApi {
void SctApi::doScan(boost::shared_ptr<Scan> scan) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "doScan\n";
  }

  scan->setStartTime(second_clock::universal_time());
  
  // Set up parameters
  boost::shared_ptr<ScanEx> extra(new ScanEx);
  extra->diagnosticReg = 0;

  extra->trimScan = 0;
  if(scan->getScanVariable1() == ST_TRIM) {
    // We need to download the trim configuration
    extra->trimScan = 1;
  }

  cout << "Do scan " << scanNumber << ":\n";
  scan->setRunNumber(runNumber);
  scan->setScanNumber(scanNumber);

  // Increment the scan number before doing anything else
  if(scan->getOption(Scan::FULL)) {
    // A full scan is three consecutive histograms
    scanNumber += 3;
  } else {
    scanNumber += 1;
  }

  if(mrs) {
    *mrs << "SCAN_REQUESTED" << MRS_INFORMATION
         << MRS_PARAM<int>("run", runNumber) << MRS_PARAM<int>("scan", scan->getScanNumber()) << MRS_TEXT("Scan requested") << ENDM;
  }

  scan->print();

  // Do checks on scan
  if(scan->getScanPoints2().size() != 0 && scan->getScanPoints2().size() != scan->getScanPoints1().size()) {
    cout << "Can't have two different sized scan points lists!\n";
    if(mrs) {
      *mrs << "SCAN_MISMATCH" << MRS_ERROR 
           << MRS_QUALIF("SCTAPI") << MRS_QUALIF("doScan") 
           << MRS_TEXT("doScan supplied with two different sized scan points lists!")
           << ENDM;
    }
    throw SctApiException("Invalid scan. Scan points mismatch");
  }

  if(!checkModuleListsForScan()) 
    throw SctApiException("Invalid scan. Module lists check failed");

  setupScanMasks(*extra, scan->getOption(Scan::DISTSLAVE), scan->getScanPoints2().size() != 0);

  RodLabel zeroRod = extra->rodInfo.begin()->first;

  // Tell the scan which modules are in which groups
  for(unsigned int i = 0;
      i<extra->groupLists.size();
      i++) {
    scan->setModuleList(i, extra->groupLists[i]);
  }
  scan->setNGroups(extra->groupLists.size());

  // Report module masks
  if(checkDebugOption(DEBUG_DIAG)) {
    RodScanEx &rodInfo = extra->rodInfo.find(zeroRod)->second;
    cout << hex << "Channels in use: 0x" << rodInfo.channels.mask0 << " 0x" << rodInfo.channels.mask1 << dec << endl;

    for(int i=0; i<numSlaves * 2; i++) {
      cout << hex << "Group channels: 0x" << rodInfo.groupChannels[i].mask0 << " 0x" << rodInfo.groupChannels[i].mask1 << dec << endl;
    }

    for(int i=0; i<numSlaves; i++) {
      cout << hex << "Slave channels: 0x" << rodInfo.slaveChannels[i].mask0 << " 0x" << rodInfo.slaveChannels[i].mask1 << dec << endl;
    }

    cout << "Slave DSP bit-field: " << hex << (int)rodInfo.bitFieldDSP << dec << " nSlaves: " << rodInfo.slaves << endl;

    hex(cout);
    for(int i=0; i<numSlaves; i++) {
      cout << "Groups on DSP " << i << ": 0x" << (int)extra->groupDspMap[i] << endl;
    }

    for(int i=0; i<2; i++) {
      cout << "Groups to serial port " << i << ": 0x" << (int)extra->groupSpMap[i] << endl;
    }

    for(int i=0; i<2; i++) {
      cout << "Groups using range list " << i << ": 0x" << (int)extra->groupRangeMap[i] << endl;
    }
    dec(cout);
  }

  for(ScanEx::RodInfoMap::const_iterator ri = extra->rodInfo.begin();
      ri != extra->rodInfo.end();
      ri++) {
    if(ri->second.bitFieldDSP & 0xf == 0) {
#warning "Should only abort if no slaves on all RODs?"
      cout << "No slaves have been deemed involved in histogramming aborting\n";
      if(mrs) {
        *mrs << MRS_ERROR 
             << "SCAN_NODSP" 
             << MRS_QUALIF("SCTAPI") << MRS_QUALIF("doScan") 
             << MRS_TEXT("No DSP matched to modules for scan")
             << ENDM;
      }
      // Harshly abort if one ROD doesn't have modules in a scan...
      throw SctApiException("Invalid scan. A ROD doesn't have any modules attached");
    }
  }

  if (!preScanHardwareCheck(*scan.get(), *extra)) throw SctApiException("Pre scan hardware check failed");

  preScanModuleSetup(*scan.get());

  { 
    // Check for soft reset and update diagnostic reg accordingly.
    Trigger::RODTriggers points = scan->getTrigger1()->getRODTriggers();

    for(unsigned int i=0; i<points.size(); i++) {
      if(points[i].first == SOFT_RESET) {
        cout << "Do extra soft reset before trigger to mask soft reset in trigger\n";
        extra->diagnosticReg |= 1 << DR_SOFT_BC_RESET;
      }
    }
  }

  bool setupGood = true;

  try {
    cout << "Setup histogramming tasks ...\n";
    doHistogramSetup(*scan.get(), *extra);
    cout << " ... done setup histogramming tasks\n";
  } catch(SctApiException &s) {
    setupGood = false;
  }

  // Send MRS message first so timing not involved
  if(mrs) {
    *mrs << "SCAN_STARTED" << MRS_INFORMATION 
         << MRS_QUALIF("SCTAPI") << MRS_QUALIF("doScan") 
         << MRS_TEXT("Scan started") 
         << MRS_PARAM<int>("ScanNumber", scan->getScanNumber())
         << MRS_PARAM<int>("ScanType1", scan->getScanVariable1())
         << MRS_PARAM<int>("ScanType2", scan->getScanVariable2())
         << ENDM;
  }

  try {
    if(setupGood) {
      // Do polling in separate thread, first decide which controller to use
      if(scan->getOption(Scan::TIM)) {
        scanController.reset(new ScanControlTIMHisto(*this, scan, extra));
      } else {
        scanController.reset(new ScanControlRODHisto(*this, scan, extra));
      }

      cout << "Starting histogramming ...\n";
      scanController->startHistogramming();
      cout << " ... histogramming started\n";
    }
  } catch(SctApiException &s) {
    setupGood = false;
  }

#ifndef USE_SCAN_THREAD
  if(setupGood) {
    // But not for now
    scanLoop();
  }
#endif
}

void SctApi::awaitScan() {
  while(scanController || m_inScanLoop) {
    sleep(1);
  }
}

/* Called by scanPollingThread */
void SctApi::scanLoop() {
  // Do this before getting the scan out
  m_inScanLoop = true;

  boost::shared_ptr<ScanControl> control;

  swap(control, scanController);

  // For the debug methods to talk to
  lastScanController = control;

  if(!control) {
    cout << "Given null scan aborting!\n";
    return;
  }

  control->initialisePolling();

  int pollReturn = pollHistogramming(control, 3);

  cout << "pollHistogramming complete (" << pollReturn << ")\n";

  control->finishHistogram(pollReturn == 0);

  // Do this before getting the scan out
  m_inScanLoop = false;
}

void SctApi::scanPollingThread() {
  cout << "SCAN: Start polling thread\n";

  while(!m_stopPolling) {
    sleep(1);
    if(scanController) {
      scanLoop();
    }
  }
}

void SctApi::setupEventTrapping(const Scan &scan, const ScanEx &ex, const RodLabel rod, boost::shared_ptr<PrimListWrapper> primList) {
  const RodScanEx &rodInfo = ex.getRodScanInfo(rod);

  // Set up each slave individually
  for(int slaveNumber=0; slaveNumber<numSlaves; slaveNumber++) {
    if((rodInfo.bitFieldDSP & (1<<slaveNumber)) && getCrate(rod.partition, rod.crate)->slavePresent(rod.rod, slaveNumber)) {
      int match;
      int modulus;            // Trap if (count % mod) == rem
      int remainder;

      switch(scan.getOption(Scan::DISTSLAVE)) {
      case 0:
        // Single slave
        match = 0;
        modulus = 1;
        remainder = 0;

        if(scan.getOption(Scan::NTH) > 1) {
          // Only trap every nth trigger
          modulus = scan.getOption(Scan::NTH);
          remainder = scan.getOption(Scan::NTH_REM);
        }
        break;
      case 1:
        // Group distribution
        match = slaveNumber;  // Match number set by histo control to distribute to slaves
        modulus = 1;
        remainder = 0;
        break;
      case 2:
        // Router distribution, between 4 DSPs
        match = 0;
        modulus = rodInfo.slaves;
        remainder = slaveNumber;

        if(scan.getOption(Scan::NTH) > 1) {
          modulus = rodInfo.slaves * scan.getOption(Scan::NTH);
          remainder = slaveNumber * scan.getOption(Scan::NTH) + scan.getOption(Scan::NTH_REM);
        }
        break;
      default:
        match = 0;
        modulus = 1;
        remainder = 0;
        break;
      }

      eventTrapSetup(0x1 << slaveNumber, 
                     match, modulus, remainder,
                     scan.getOption(Scan::TIM), checkDebugOption(DEBUG_SCAN_ERROR_TRAP_ALL), primList);
    }
  }
}

void SctApi::setupEventDump(const Scan &scan, const ScanEx &ex, const RodLabel rod, boost::shared_ptr<PrimListWrapper> primList) {
  if(!(checkDebugOption(DEBUG_SCAN_ERROR_TRAP) && scan.getOption(Scan::DEBUG))) return;

  int slaveNumber = 2;

  // No difference depending where its run...
  {
    // Trap all events as errors
    long *evData = new long[sizeof(EVENT_TRAP_SETUP_IN)/4];

    EVENT_TRAP_SETUP_IN &primData = *(EVENT_TRAP_SETUP_IN*)evData;

#if R_EVENT_TRAP_SETUP == 103
    primData.slvBits = 0x1 << slaveNumber;
    primData.numberOfEvents = COLLECT_FOREVER;
    primData.timeoutInUsec = 0x30000;

    primData.extRouterSetup = 0;

    // This tells the setup code to change the trapRemainder
    //  Only matters if setting up more than one slave
    primData.distribute = TRAP_DISTRIB_NEITHER;

    primData.releaseFrames = 0;
    primData.permitBackPressure = 1;
    primData.dataMode = 0;                        // Data or Event mode
    primData.sLink = 0;                           // Where to pick events (dsp or router)

    primData.format = TRAP_FMT_ERROR;             // Slink format
    primData.trapStray = 1;                       // Trap stray events
    primData.iterLimit = 2;                       // Don't keep them long

    // Trap events from ROD triggers
    primData.trapConfig[0] = TRAP_CFG_ROD_EVT;
    primData.trapExclusionFlag[0] = 0;            // Don't exclude anything
    primData.trapFunction[0] = TRAP_FXN_RESYNCH;

    primData.trapMatch[0] = 0;                    // = slave number??
    primData.trapModulus[0] = 1;                  // Trap if (count % mod) == rem
    primData.trapRemainder[0] = 0;

    // Trap one is idle
    primData.trapConfig[1] = TRAP_CFG_IDLE;
    primData.trapExclusionFlag[1] = 0;
    primData.trapFunction[1] = 0;

    primData.trapMatch[1] = 0; 
    primData.trapModulus[1] = 0;
    primData.trapRemainder[1] = 0;
#else 
#error "Event trapping not compiled (new Primitive definition)!"
#endif
    primList->addPrimitive(RodPrimitive(sizeof(EVENT_TRAP_SETUP_IN)/sizeof(UINT32) + 4, 
                                        0, EVENT_TRAP_SETUP, R_EVENT_TRAP_SETUP, evData),
                           evData);
  }

  // Start event trapping on slave 2
  startEventTrap(slaveNumber, primList);

  // Start a dump to remove the events...
  {
    long *taskData = new long[sizeof(START_TASK_IN)/4];

    for(unsigned int i=0; i<sizeof(START_TASK_IN)/4; i++) {
      taskData[i] = 0;
    }

    START_TASK_IN &taskPrim = *(START_TASK_IN *)taskData;

    taskPrim.taskType = RESYNCH_TASK;
    taskPrim.taskRevision = R_RESYNCH_TASK;
    taskPrim.priority = 1; // Top???
    taskPrim.completionFlag = 1; 

    RESYNCH_TASK_IN &resynch = taskPrim.taskStruct.resynchTaskIn;

    resynch.errorType = 0;

    shared_ptr<PrimListWrapper> myList(new PrimListWrapper(4));

    myList->addPrimitive(RodPrimitive(4 + sizeof(START_TASK_IN)/4,
                                      3, START_TASK, R_START_TASK, taskData),
                         taskData);

    PrimBuilder::instance().slavePrimList(primList, myList, slaveNumber, true, true);
  }
}

void SctApi::startEventTrapping(const Scan &scan, const ScanEx &ex, const RodLabel rod, shared_ptr<PrimListWrapper> primList) {
  const RodScanEx &rodInfo = ex.getRodScanInfo(rod);

  for(int slaveNumber=0; slaveNumber<numSlaves; slaveNumber++) {
    if((rodInfo.bitFieldDSP & (1<<slaveNumber)) && getCrate(rod.partition, rod.crate)->slavePresent(rod.rod, slaveNumber)) {
      startEventTrap(slaveNumber, primList);
    }
  }
}

void  SctApi::setupHistogramming(const Scan &scan, const ScanEx &ex, const RodLabel rod, bool tim, 
                                 boost::shared_ptr<PrimListWrapper> primList) {
  const RodScanEx &rodInfo = ex.getRodScanInfo(rod);

  int nBins = scan.getScanPoints1().size();
  if(!scan.getOption(Scan::BITS32)) {
    if(nBins%2 == 1) {
      nBins += 1;
    }
  }

  for(int slaveNumber = 0; slaveNumber < 4; slaveNumber ++) {
    if((rodInfo.bitFieldDSP & (1<<slaveNumber)) && getCrate(rod.partition, rod.crate)->slavePresent(rod.rod, slaveNumber)) {
      // Primitive
      long *histoData = new long[sizeof(HISTOGRAM_SETUP_IN)/4];

      HISTOGRAM_SETUP_IN &primData = *(HISTOGRAM_SETUP_IN*)histoData;

#if (R_HISTOGRAM_SETUP == 108)
      primData.base = (UINT32 *)0xffffffff;//  HISTOGRAM_DEFAULT_BASE;    // 0x20d2000
      primData.nBins = nBins; 

      primData.dataType[0] = scan.getScanVariable1();
      primData.dataType[1] = scan.getScanVariable2();

      primData.binSize = scan.getOption(Scan::BITS32)?HISTOGRAM_32BIT:HISTOGRAM_16BIT;

      primData.routineType = HISTO_ROUTINE_C;

      // arrangement
      primData.opt[0] = 0; // HISTOGRAM_SLICE; // scan.getOption(Scan::FORMAT)?HISTOGRAM_SLICE:HISTOGRAM_BLOCK;
      // dataFormat
      primData.opt[1] = scan.getOption(Scan::FULL)?HISTOGRAM_FULL:HISTOGRAM_CONDENSED;          // Get all three data sets
      // unused
      primData.opt[2] = 0;
      primData.opt[3] = 0;

      UINT8 groupMap = ex.groupDspMap[slaveNumber];

      primData.validModules[0] = 0;
      primData.validModules[1] = 0;

      primData.moduleRangeMap[0][0] = 0;
      primData.moduleRangeMap[0][1] = 0;
      primData.moduleRangeMap[1][0] = 0;
      primData.moduleRangeMap[1][1] = 0;

      for(int i=0; i<8; i++) {
        if(groupMap & (1<<i)) {
          primData.validModules[0] |= rodInfo.groupChannels[i].mask0;          // Modules to this slave
          primData.validModules[1] |= rodInfo.groupChannels[i].mask1;          //    32-47 in low 16 bits

          if(ex.groupSpMap[0] & (1<<i)) {
            primData.moduleRangeMap[0][0] |= rodInfo.groupChannels[i].mask0;
            primData.moduleRangeMap[0][1] |= rodInfo.groupChannels[i].mask1;
          }
          if(ex.groupSpMap[1] & (1<<i)) {
            primData.moduleRangeMap[1][0] |= rodInfo.groupChannels[i].mask0;
            primData.moduleRangeMap[1][1] |= rodInfo.groupChannels[i].mask1;
          }
        }
      }

      primData.xPtr[0] = (FLOAT32 *)0xffffffff; // 0x3f00000;                    // Where the fit data will be stored
      primData.xPtr[1] = (FLOAT32 *)0xffffffff; // 0x3f01000; 
#elif (R_HISTOGRAM_SETUP == 105) || (R_HISTOGRAM_SETUP == 107)
      primData.base = (UINT32 *)0xffffffff;//  HISTOGRAM_DEFAULT_BASE;    // 0x20d2000
      primData.nBins = nBins; 

      primData.padding[0] = 0;
      primData.padding[1] = 0;

      primData.dataType[0] = scan.getScanVariable1();
      primData.dataType[1] = scan.getScanVariable2();

      primData.binSize = scan.getOption(Scan::BITS32)?HISTOGRAM_32BIT:HISTOGRAM_16BIT;

      // arrangement
      primData.opt[0] = HISTOGRAM_SLICE; // scan.getOption(Scan::FORMAT)?HISTOGRAM_SLICE:HISTOGRAM_BLOCK;
      // dataFormat
      primData.opt[1] = scan.getOption(Scan::FULL)?HISTOGRAM_FULL:HISTOGRAM_CONDENSED;          // Get all three data sets
      // unused
      primData.opt[2] = 0;
#if(R_HISTOGRAM_SETUP == 107)
      primData.opt[3] = 0;
#endif

      UINT8 groupMap = ex.groupDspMap[slaveNumber];

      primData.validModules[0] = 0;
      primData.validModules[1] = 0;

      primData.moduleRangeMap[0][0] = 0;
      primData.moduleRangeMap[0][1] = 0;
      primData.moduleRangeMap[1][0] = 0;
      primData.moduleRangeMap[1][1] = 0;

      for(int i=0; i<8; i++) {
        if(groupMap & (1<<i)) {
          primData.validModules[0] |= rodInfo.groupChannels[i].mask0;          // Modules to this slave
          primData.validModules[1] |= rodInfo.groupChannels[i].mask1;          //    32-47 in low 16 bits

          if(ex.groupSpMap[0] & (1<<i)) {
            primData.moduleRangeMap[0][0] |= rodInfo.groupChannels[i].mask0;
            primData.moduleRangeMap[0][1] |= rodInfo.groupChannels[i].mask1;
          }
          if(ex.groupSpMap[1] & (1<<i)) {
            primData.moduleRangeMap[1][0] |= rodInfo.groupChannels[i].mask0;
            primData.moduleRangeMap[1][1] |= rodInfo.groupChannels[i].mask1;
          }
        }
      }

      primData.xPtr[0] = (FLOAT32 *)0xffffffff; // 0x3f00000;                    // Where the fit data will be stored
      primData.xPtr[1] = (FLOAT32 *)0xffffffff; // 0x3f01000; 
#elif R_HISTOGRAM_SETUP == 104
      primData.base = (UINT32 *)0xffffffff; // HISTOGRAM_DEFAULT_BASE;    // 0x20d2000
      primData.rodType = ROD_TYPE_SCT;
      primData.nBins = nBins;

      primData.padding[0] = 0;
      primData.padding[1] = 0;

      primData.dataType[0] = scan.getScanVariable1();
      primData.dataType[1] = scan.getScanVariable2();

      /** Use slice (faster with caching)
          condensed will be in assembler, 
          full (3 bits) in c so slower 
      */

      primData.arrangement = HISTOGRAM_SLICE;
      primData.dataFormat = scan.getOption(Scan::FULL)?HISTOGRAM_FULL:HISTOGRAM_CONDENSED;          // Get all three data sets
      primData.binSize = scan.getOption(Scan::BITS32)?HISTOGRAM_32BIT:HISTOGRAM_16BIT;
      primData.unused = 0;

      if(scan.getOption(Scan::DISTSLAVE) == 1) {
        primData.validModules[0] = rodInfo.groupChannels[slaveNumber][0];          // Modules to this slave
        primData.validModules[1] = rodInfo.groupChannels[slaveNumber][1];          //    32-47 in low 16 bits
        primData.moduleRangeMap[0][0] = rodInfo.groupChannels[slaveNumber][0];     // All modules use rangemap 0
        primData.moduleRangeMap[0][1] = rodInfo.groupChannels[slaveNumber][1];
      } else {
        primData.validModules[0] = rodInfo.channels[0];                          // Modules to this slave
        primData.validModules[1] = rodInfo.channels[1];                          //    32-47 in low 16 bits
        primData.moduleRangeMap[0][0] = rodInfo.channels[0];                     // All modules use rangemap 0
        primData.moduleRangeMap[0][1] = rodInfo.channels[1];
      }
      primData.moduleRangeMap[1][0] = 0;                                    // No modules for rangemap 1
      primData.moduleRangeMap[1][1] = 0;

      primData.xPtr[0] = (FLOAT32 *)0x3f00000;                              // Where the fit data will be stored
      primData.xPtr[1] = (FLOAT32 *)0x3f01000; 
#else 
#error "Histogram setup not compiled (new Primitive definition)!"
#endif

      long *taskData = new long[sizeof(START_TASK_IN)/4];

      for(unsigned int i=0; i<sizeof(START_TASK_IN)/4; i++) {
        taskData[i] = 0;
      }

      START_TASK_IN &taskPrim = *(START_TASK_IN *)taskData;

      taskPrim.taskType = HISTOGRAM_TASK;
      taskPrim.taskRevision = R_HISTOGRAM_TASK;
      taskPrim.priority = 1; // Top???
      taskPrim.completionFlag = 1; 

      HISTOGRAM_TASK_IN &histoTaskData = taskPrim.taskStruct.histogramTaskIn;

      histoTaskData.nEvents = COLLECT_FOREVER;
      if(tim) {
        histoTaskData.controlFlag = LOCAL_SET_TRIG;
      } else {
        histoTaskData.controlFlag = MASTER_HREG;
      }

      boost::shared_ptr<PrimListWrapper> myList(new PrimListWrapper(4));

      myList->addPrimitive(RodPrimitive(4 + sizeof(HISTOGRAM_SETUP_IN)/sizeof(UINT32), 
                                        2, HISTOGRAM_SETUP, R_HISTOGRAM_SETUP, histoData),
                           histoData);
      myList->addPrimitive(RodPrimitive(4 + sizeof(START_TASK_IN)/4,
                                        3, START_TASK, R_START_TASK, taskData),
                           taskData);

      PrimBuilder::instance().slavePrimList(primList, myList, slaveNumber, true, true);
    }
  }
}

void SctApi::startHistogramTask(const Scan &scan, const ScanEx &ex, const RodLabel rod, 
                                unsigned int startBin, unsigned int nBins, unsigned int nTrigs,
                                boost::shared_ptr<PrimListWrapper> primList) {
  const RodScanEx &rodInfo = ex.getRodScanInfo(rod);

  // The number of bins in the variable list has to be nBins...
  unsigned int allBins = nBins;

  long *ctrlData = new long[sizeof(START_TASK_IN)/4 + allBins * 2];

  for(unsigned int i=0; i<sizeof(START_TASK_IN)/4 + allBins * 2; i++) {
    ctrlData[i] = 0;
  }

  struct START_TASK_IN &primData = *(START_TASK_IN*)ctrlData;
  struct HISTOGRAM_CTRL_TASK_IN &taskData = primData.taskStruct.histoCtrlTaskIn; 

#if (R_HISTOGRAM_CTRL_TASK == 107)
  primData.taskType = HISTOGRAM_CTRL_TASK;
  primData.taskRevision = R_HISTOGRAM_CTRL_TASK;
  primData.priority = 1;
  primData.completionFlag = 1; 

  taskData.slvBits = rodInfo.bitFieldDSP;   // Which DSPs to use

  if(scan.getScanPoints2().size() > 0) {
    taskData.port = SP_BOTH;
  } else {
    //@todo SP_BOTH doesn't work with current DSP code
    //taskData.port = SP_BOTH;
    taskData.port = SP0;
  }

  taskData.configRegister[0] = scan.getScanVariable1(); // ST_VTHR;
  taskData.configRegister[1] = scan.getScanVariable2(); // ST_VTHR;

  taskData.configSctSet = Utility::translateBank(SCTAPI_BANK_SCAN);

  // Which bit of configuration to send, unless this is a trim scan then basic should be enough
  if(!ex.trimScan) 
    taskData.dataSet = MODULE_BASIC; 
  else 
    taskData.dataSet = MODULE_TRIM;               // Which bit of configuration to send

  taskData.groupRangeMap[0] = ex.groupRangeMap[0];
  taskData.groupRangeMap[1] = ex.groupRangeMap[1];

  taskData.groupDSPMap[0] = ex.groupDspMap[0];
  taskData.groupDSPMap[1] = ex.groupDspMap[1];
  taskData.groupDSPMap[2] = ex.groupDspMap[2];
  taskData.groupDSPMap[3] = ex.groupDspMap[3];

  taskData.groupSPMap[0] = ex.groupSpMap[0];
  taskData.groupSPMap[1] = ex.groupSpMap[1];

  taskData.globalCtrl = 0;              // Don't want to take control for internal triggers
  taskData.syncLevel = 0;               //  So don't need synchronisation 

  taskData.histoBase = (UINT32*)0xffffffff; // HISTOGRAM_DEFAULT_BASE;    // 0x20d2000

  taskData.doChipOcc = 1;
  taskData.dataFormat = scan.getOption(Scan::FULL)?HISTOGRAM_FULL:HISTOGRAM_CONDENSED;
  taskData.binSize = scan.getOption(Scan::BITS32)?HISTOGRAM_32BIT:HISTOGRAM_16BIT;
  taskData.unused1 = 0;

  taskData.extSetup = 0xff;                    // This is specifies which setup tasks are done externally (to the control task)
  taskData.dataPath = DATA_PATH_NORMAL;        // Other option is INMEM (presumably TestBench based)
  taskData.capture = 0;
  taskData.useRangeList = 0;                   // Major difference from Doug's example!

  taskData.repetitions = nTrigs;
  taskData.nBins = nBins;
  taskData.bin0 = startBin;

  // Ignore range list data 
  for(int i=0; i<5; i++) {
    taskData.rangeList[0].xPair[i].x0 = 0.0;
    taskData.rangeList[1].xPair[i].x0 = 0.0;
    taskData.rangeList[0].xPair[i].delta_x = 0.0;
    taskData.rangeList[1].xPair[i].delta_x = 0.0;
  }

  // X data lists
  taskData.dataPtr[0] = (FLOAT32 *)0xffffffff;   // Data for group 1               Append to end of primitive
  taskData.dataPtr[1] = (FLOAT32 *)0xffffffff;   // Data for group 2

  // Clear trigger sequences
  for(int i=0; i<N_CMD_LIST_CMDS; i++) {            // from primParams.h
    taskData.triggerSequence[0].cmd[i] = NO_CMD;    // Set a trigger sequence
    taskData.triggerSequence[1].cmd[i] = NO_CMD;    // Set a trigger sequence
    taskData.triggerSequence[0].data[i] = 0;        // Set a trigger sequence
    taskData.triggerSequence[1].data[i] = 0;        // Set a trigger sequence
  }


//    // Two L1As (illegal without delay?!)
//    taskData.triggerSequence[0].cmd[0] = L1_TRIGGER;
//    taskData.triggerSequence[1].cmd[0] = L1_TRIGGER;
//    taskData.triggerSequence[0].cmd[1] = L1_TRIGGER;
//    taskData.triggerSequence[1].cmd[1] = L1_TRIGGER;

  Trigger::RODTriggers points = scan.getTrigger1()->getRODTriggers();

  for(unsigned int i=0; i<points.size(); i++) {
    taskData.triggerSequence[0].cmd[i] = points[i].first;        // Set a trigger sequence
    taskData.triggerSequence[0].data[i] = points[i].second;      // Set a trigger sequence
  }

  if(scan.getScanPoints2().size() > 0) {
    points = scan.getTrigger2()->getRODTriggers();

    for(unsigned int i=0; i<points.size(); i++) {
      taskData.triggerSequence[1].cmd[i] = points[i].first;      // Set a trigger sequence 2
      taskData.triggerSequence[1].data[i] = points[i].second;    // Set a trigger sequence 2
    }
  } else {
    // Leave as default
  }

//    taskData.triggerSequence[0].cmd[0] = CALIBRATION_PULSE;
//    taskData.triggerSequence[0].cmd[1] = DELAY;
//    taskData.triggerSequence[0].cmd[2] = L1_TRIGGER;

//    taskData.triggerSequence[0].data[1] = 0x81;

  unsigned short cmd, data;

  scan.getTrigger1()->getCommIncr(cmd, data);
  taskData.incCmd[0] = cmd;
  taskData.incData[0] = data;
  scan.getTrigger2()->getCommIncr(cmd, data);
  taskData.incCmd[1] = cmd;
  taskData.incData[1] = data;

  taskData.calLineLoop = scan.getOption(Scan::LOOPCALLINE);
  if(scan.getOption(Scan::DISTSLAVE) == 2) {
    taskData.distributionToggle = ROUTER_DISTRIB;
  } else {
    taskData.distributionToggle = MODULE_GROUP;
  }

  for(int i=0; i<10; i++) {
    taskData.genData[i] = 0;
  }

  // Defined groups
  taskData.genData[0] = 0xff;

  if(scan.getScanPoints2().size() > 0) {
    taskData.genData[1] = 1;
    taskData.genData[2] = 0x0200; // 0301;
  }

#elif (R_HISTOGRAM_CTRL_TASK == 105)
  primData.taskType = HISTOGRAM_CTRL_TASK;
  primData.taskRevision = R_HISTOGRAM_CTRL_TASK;
  primData.priority = 0; // Top???
  primData.completionFlag = 1; 

  taskData.slvBits = 0xf;             // Use all slaves
  taskData.configSctSet = translateBank(SCTAPI_BANK_PHYSICS);
  // Not an array yet
//    taskData.configRegister[0] = 0;     // V_THR ??
//    taskData.configRegister[1] = 0;     // V_THR ??
  taskData.dataSet = 0;               // ??
  taskData.useRangeList = 0;          // ??
  taskData.groupRangeMap[0] = 0;      // No range map
  taskData.groupRangeMap[1] = 0;      //  ditto
  taskData.cmdBuff = CMD_BUFFER_BOTH;     // Which serial port to use
  taskData.dataPtr[0] = group1Data;   // Data for group 1
  taskData.dataPtr[1] = group2Data;   // Data for group 2
  taskData.repetitions = 10;          // Make it stop quickly 10 events
  taskData.nBins = 10;                // 10 bins
  taskData.bin0 = 0;                  // Start at 0

  taskData.triggerSequence[0];        // Set a trigger sequence
  taskData.triggerSequence[1];        // Set a trigger sequence

  taskData.incCmd[0] = 0;
  taskData.incCmd[1] = 0;
  taskData.calLineLoop = 0;  // Doesn't work yet??

  taskData.distributionToggle = 0;  // ??
  taskData.incData[0] = 1;
  taskData.incData[1] = 1;

  taskData.groupSPMap[0] = 0;
  taskData.groupSPMap[1] = 0;
#else  // Unknown R_HISTOGRAM_CTRL_TASK
#error "Histogram Task not compiled (new Primitive definition)!" 
#endif

  int offset = sizeof(START_TASK_IN)/4;

  // Copy real bins
  for(unsigned int i=0; i<allBins; i++) {
    ctrlData[offset + i] = *(UINT32 *)&(scan.getScanPoints1()[i]);
    if(scan.getScanPoints2().size() == 0) {
      ctrlData[offset + allBins + i] = *(UINT32*)&(scan.getScanPoints1()[i]);
    } else {
      ctrlData[offset + allBins + i] = *(UINT32*)&(scan.getScanPoints2()[i]);
    }
  }

  primList->addPrimitive(RodPrimitive(4 + sizeof(START_TASK_IN)/sizeof(UINT32) + allBins * 2, 
                                      0, START_TASK, R_START_TASK, ctrlData),
                         ctrlData);
}

/*
  Stop histogram control in master
  Histogram and event trapping in slaves

  *********** XXX Send them individually as it doesn't seem to work otherwise ***************
*/
void SctApi::stopHistogramming(const ScanEx &ex) {
  for(ScanEx::RodInfoMap::const_iterator ri = ex.rodInfo.begin();
      ri != ex.rodInfo.end();
      ri++) {
    const RodLabel &rod = ri->first;

    {
      boost::mutex::scoped_lock lock(logMutex);
      log << "Stop histogramming (controlled)\n";
    }
//  stopEvTrap(0);

//  taskOp(16, 0, 0, -1);   // Stop histogram control task

//  taskOp(32, 0, 0, 0);     // Stop histogram task

    boost::shared_ptr<PrimListWrapper> primList(new PrimListWrapper(2));

    PrimBuilder &builder = PrimBuilder::instance();
    builder.taskOp(primList, HISTOGRAM_CTRL_TASK, TASK_STOP, 0);

    {
      boost::mutex::scoped_lock lock(logMutex);
      log << "Stop histo control task\n";
    }
    sendPrimList(rod.partition, rod.crate, rod.rod, primList);

    awaitResponse(rod.partition, rod.crate, rod.rod, 5);
    primList->clear();

    cout << "Done kill control task\n";

    for(int slaveNumber=0; slaveNumber<numSlaves; slaveNumber++) {
      // Kill all tasks!!!!
      boost::shared_ptr<PrimListWrapper> slList(new PrimListWrapper(1));

      // Stop Slave histogramming task

      builder.taskOp(slList, HISTOGRAM_TASK, TASK_STOP, 0);

      PrimBuilder::instance().slavePrimList(primList, slList, slaveNumber, true, true);
      {
        boost::mutex::scoped_lock lock(logMutex);
        log << "Stop slave histogram task " << slaveNumber << endl;
      }
      sendPrimList(rod.partition, rod.crate, rod.rod, primList);
      awaitResponse(rod.partition, rod.crate, rod.rod, 5);
      slList->clear();
      primList->clear();

      cout << "Done kill histogram task\n";

      stopEventTrap(slaveNumber, primList);

      {
        boost::mutex::scoped_lock lock(logMutex);
        log << "Stop Event trapping on slave " << slaveNumber << endl;
      }
      sendPrimList(rod.partition, rod.crate, rod.rod, primList);

      awaitResponse(rod.partition, rod.crate, rod.rod, 5);
      primList->clear();

      cout << "Done kill event trapping\n";
    }
  }

  //  sendPrimList(ex.partition, ex.crate, ex.rod, primList);
  //  awaitResponse(ex.partition, ex.crate, ex.rod, 2);
  cout << "Done kill histogramming tasks\n";
}

ScanControl::ScanControl(SctApi &api) : api(api) {}

void ScanControlRODHisto::readHistograms() {
  time_t saveTime;
  saveTime = time(NULL);

  // For each module
  for(unsigned int group = 0; group < scan->getNGroups(); group ++) {
    const list<string> &groupList = scan->getModuleList(group);
    cout << "Modules in group " << group << " count: " << groupList.size() << endl;
    for(list<string>::const_iterator mi = groupList.begin();
        mi!=groupList.end();
        mi ++) {

      string sn = *mi;
      UINT32 mid = api.findModule(sn);

      cout << "Module:  " << mid << " " << sn << endl;

      try {
        if(scan->getOption(Scan::FULL)) {
          for(int f=0; f<3; f++) {
            scan_result_ptrs result = readHistogramData(mid, f);

            result.header.scanNumber += f;

            // Files need discriminating because they're all the same time
            if(checkDebugOption(DEBUG_SAVE_HISTOGRAM)) {
              readHistogramToFile(*scan, *scanEx, mid, api.convertToString(mid), result, saveTime, f);
            }

#ifdef USE_IS
            // IS based on Run/Scan number so nothing more needed
            readHistogramToIS(*scan, *scanEx, mid, result);
#endif

            delete [] (char*)result.data;
            delete [] result.points;
            delete [] result.nEvents;
            delete [] result.nErrorEvents;
          }
        } else {
          scan_result_ptrs result = readHistogramData(mid, 0);

          if(checkDebugOption(DEBUG_SAVE_HISTOGRAM)) {
            readHistogramToFile(*scan, *scanEx, mid, api.convertToString(mid), result, saveTime);
          }
#ifdef USE_IS
          readHistogramToIS(*scan, *scanEx, mid, result);
#endif
          delete [] (char*)result.data;
          delete [] result.points;
          delete [] result.nEvents;
          delete [] result.nErrorEvents;
        }
      } catch(SctApiException &s) {
        if(api.mrs) {
          *api.mrs << "SCAN_READOUT_FAIL" << MRS_ERROR
                   << MRS_QUALIF("SctApi")
                   << MRS_PARAM<int>("run", scan->getRunNumber()) 
                   << MRS_PARAM<int>("scan", scan->getScanNumber()) 
                   << MRS_PARAM<int>("mid", mid) 
                   << MRS_TEXT("Module readout failed") << ENDM;
        }
      }
      // Histogram for next module
    }
  }
}

scan_result_ptrs ScanControlRODHisto::readHistogramData(UINT32 mid, int frame) {
  scan_result_ptrs scanResult;

  //We do this first deliberately.
  //The dummy api will need to mess around with some things
  scanResult.data = readHistogramRawData(mid, frame);

  int nBins = scan->getScanPoints1().size();

  // Read the module configuration from the end of the scan...
  api.getABCDModule(mid, SCTAPI_BANK_SCAN);

  // Save copy of configuration from end of scan
  ABCDModule moduleConfig = *api.retrieveModule(mid);

  if(checkDebugOption(DEBUG_MODULE_CONFIG)) {
    cout << "Configuration after scan:\n";
    api.printABCDModule(mid);
  }

  // Restore cached configuration from ROD
  //getABCDModule(mid, SCTAPI_BANK_PHYSICS);

  ScanHeader &header = scanResult.header;

  header.version = CURRENT_SCANHEADER_VERSION;
  header.length = sizeof(ScanHeader);
  header.runNumber = scan->getRunNumber();
  header.scanNumber = scan->getScanNumber();
  
  strncpy(header.moduleName, api.convertToString(mid).c_str(), 16);
  strncpy(header.startTime, to_iso_string(scan->getStartTime()).c_str(), 16);
  strncpy(header.endTime, to_iso_string(scan->getEndTime()).c_str(), 16);
  header.moduleName[15] = 0;    // Null terminate the string
  header.startTime[15] = 0;
  header.endTime[15] = 0;
  
//   snprintf(header.moduleName, 16, "%s", convertToString(mid).c_str());
//   cout << "Module name in Scan Results (" << header.moduleName << ")\n" ;
  header.scanType = scan->getScanVariable1();
  header.npoints = nBins;
  header.size = nBins * 2 * 1024 * (scan->getOption(Scan::BITS32)?2:1);
  header.dataType = SR_DT_SLICE;

  header.width = scan->getOption(Scan::BITS32)?SR_WD_32:SR_WD_16;

  header.config = moduleConfig;

  header.pntPoints = sizeof(ScanHeader);
  header.pntEvents = header.pntPoints + nBins * sizeof(FLOAT32);
  header.pntErrors = header.pntEvents + nBins * sizeof(UINT32);
  header.pntData = header.pntErrors + nBins * sizeof(UINT32);

  // The points
  FLOAT32 *points = new FLOAT32[nBins];
  for(int i=0; i<nBins; i++) {
    points[i] = scan->getScanPoints1()[i];
  }
  scanResult.points = points;

  // The event counts
  UINT32 *events = new UINT32[nBins];

  std::pair<RodLabel, unsigned int> moduleSlave = findModuleSlave(mid);

#if 0 // This is the better way of counting (especially with errors)
  unsigned long *countChunk = getEventCountChunk(moduleSlave.first, moduleSlave.second);

  for(int i=0; i<nBins; i++) {
    events[i] = countChunk[i];
  }
#else
  // But this way does get the count right with the callineloop on!
  Scan::TrigPoints trigs = scan->getVariableTrigs();
 
  for(int i=0; i<nBins; i++) {
    if(i<lastBin[moduleSlave.first]) {
      events[i] = trigs[i];
    } else if(i == lastBin[moduleSlave.first]) {
      events[i] = lastEvent[moduleSlave.first];
    } else {
      events[i] = 0;
    }
  }
#endif

  scanResult.nEvents = events;

  // The error counts
  UINT32 *errors = new UINT32[nBins];
  for(int i=0; i<nBins; i++) {
#warning "Get real error count (if possible!)"
    errors[i] = 0;
  }
  scanResult.nErrorEvents = errors;

  return scanResult;
}

char* ScanControlRODHisto::readHistogramRawData(UINT32 mid, int frame) {
  int nBins = scan->getScanPoints1().size();

  // The number of bins the ROD thinks there are...
  int rodBins = nBins;
  if(!scan->getOption(Scan::BITS32))
    if(rodBins%2 == 1) rodBins++;

  // The data!
  int binSize = 0x800*(scan->getOption(Scan::BITS32)?4:2);   // In bytes, so works for 16bit histograms

  char *data = new char[nBins * binSize];

  bool moduleBlock = false;
//    int baseAddress = index*(binSize*nBins);

  unsigned int slaveNumber = 4;

  RodLabel rodLabel;
  unsigned int channel;
  {
    unsigned int partition, crate, rod;
    Utility::getpcrc(mid, partition, crate, rod, channel);
    RodLabel temp(partition, crate, rod);
    rodLabel = temp;
  }

  const RodScanEx &rodInfo = scanEx->getRodScanInfo(rodLabel);

  for(int slave=0; slave<4; slave++) {
    if((channel<32 && (rodInfo.slaveChannels[slave].mask0 & 1<<channel))
       || (channel >= 32 && (rodInfo.slaveChannels[slave].mask1 & 1<<(channel-32)))) {
      if(slaveNumber == 4) {
        slaveNumber = slave;
      } else {
        cerr << " ****** Module found on multiple slaves this is an unsupported configuration, the first one will be downloaded??? *****\n";
        cerr << " ****** Well you don't expect me to add them all together do you??? *****\n";
#warning "But maybe I should any way"
      }
    }
  }

  if(slaveNumber == 4) {
    cerr << " ****** Current module not found on any slave, skipping histogram read out *****\n";

    throw SctApiException("No mapping from module to DSP found");
  }

  // Find the module index on this slave
  int index = 0;
  int nModules = 0; // moduleMap.size();

  for(unsigned int ch=0; ch<48; ch++) {
    if(channel == ch) {
      index = nModules;
    }
    if(ch<32) {
      if(rodInfo.slaveChannels[slaveNumber].mask0 & 1<<ch) {
        nModules ++;
      }
    } else {
      if(rodInfo.slaveChannels[slaveNumber].mask1 & 1<<(ch-32)) {
        nModules ++;
      }
    }
  }

  // Get data from place defined by DSP
  unsigned long slaveOffset = 0;

  if(moduleBlock) {
    slaveOffset += binSize * rodBins;
  } else {
    slaveOffset += binSize * index;
  }

  // For reading expanded data, skip to next block of histograms...
  slaveOffset += ((binSize * rodBins * nModules) + binSize) * frame;

  bool error = false;

  for(int i=0; i<nBins; i++) {
    unsigned long *chunk = getHistogramChunk(rodLabel, slaveNumber, slaveOffset, binSize/4);
    unsigned long *buffer = (unsigned long *)&(data[i*binSize]);

    if(chunk) {
      for(int b=0; b<binSize/4; b++) {
        buffer[b] = chunk[b];
      }

      // Check End of bin xx Module xx
      int bin, module;
      int count = sscanf(&((char*)buffer)[binSize-32], "End of bin %d Module %d", &bin, &module);
      if(count == 2) {
//          printf("Found End of bin %d module %d\n", bin, module);
      } else {
        if(!error)
          printf("Found bad end of histogram bin (%25s)\n", &((char*)buffer)[binSize-32]);
        error = true;
      }

#warning "Memory leak if sendData didn't work"
//       delete [] chunk;
    } else {
      for(int b=0; b<binSize/4; b++) {
        buffer[b] = 0;
      }
    }

    if(moduleBlock) {
      slaveOffset += binSize;
    } else {
      slaveOffset += binSize * nModules;
    }
  }

  return data;
}

pair<RodLabel, unsigned int> ScanControlRODHisto::findModuleSlave(unsigned int mid) {
  RodLabel rodLabel;
  unsigned int channel;
  {
    unsigned int partition, crate, rod;
    Utility::getpcrc(mid, partition, crate, rod, channel);
    RodLabel temp(partition, crate, rod);
    rodLabel = temp;
  }

  unsigned int slaveNumber = 4;

  const RodScanEx &rodInfo = scanEx->getRodScanInfo(rodLabel);

  for(int slave=0; slave<4; slave++) {
    if((channel<32 && (rodInfo.slaveChannels[slave].mask0 & 1<<channel))
       || (channel >= 32 && (rodInfo.slaveChannels[slave].mask1 & 1<<(channel-32)))) {
      if(slaveNumber == 4) {
        slaveNumber = slave;
      } else {
        cerr << " ****** Module found on multiple slaves this is an unsupported configuration, the first one will be downloaded??? *****\n";
        cerr << " ****** Well you don't expect me to add them all together do you??? *****\n";
#warning "But maybe I should any way"
      }
    }
  }

  if(slaveNumber == 4) {
    cerr << " ****** Current module not found on any slave, skipping histogram read out *****\n";

    throw SctApiException("No mapping from module to DSP found");
  }

  return make_pair(rodLabel, slaveNumber);
}

void readHistogramToFile(const Scan &scan, const ScanEx &ex, 
                         UINT32 mid, std::string sn, scan_result_ptrs scanData, time_t saveTime, int frame) {
  const int BUFF_SIZE = 50;

  struct tm broke;
  broke = *(gmtime(&saveTime));

  string tempDir = SctNames::getTempDir() + "/";
  int bufferLength = BUFF_SIZE + tempDir.length();
  char *filename = new char[bufferLength];
  strcpy(filename, tempDir.c_str());

  cout << "Module:  " << mid << " " << sn << endl;

  int written = tempDir.length();
  if(scan.getOption(Scan::FULL)) {
    written += snprintf(filename + written, bufferLength - written, "Histogram%d_%s", frame, sn.c_str());
    //                                                                    20021112143712
    strftime(filename + written, bufferLength - written, "_%Y%m%d%H%M%S.bin", &broke);
  } else {
    written += snprintf(filename + written, bufferLength - written, "Histogram_%s", sn.c_str());
    //                                                                    20021112143712
    strftime(filename + written, bufferLength - written, "_%Y%m%d%H%M%S.bin", &broke);
  }

  cout << "Writing histogram data to " << filename << endl;

  saveHistogramToFile(scanData, filename);
    
  cout << "done.\n";

  delete [] filename;
}

void saveHistogramToFile(scan_result_ptrs histo, std::string filename) {
  ofstream histoout(filename.c_str(), ios::binary);

  //    assert(histo.header.length == sizeof(ScanHeader));
  histoout.write((char*)&histo.header, histo.header.length);

  //    assert(histo.header.pntEvents - histo.header.pntPoints == nBins * sizeof(FLOAT32));
  histoout.write((char*)histo.points, histo.header.pntEvents-histo.header.pntPoints);

  //    assert(histo.header.pntErrors - histo.header.pntEvents == nBins * sizeof(FLOAT32));
  histoout.write((char*)histo.nEvents, histo.header.pntErrors-histo.header.pntEvents);

  //    assert(histo.header.pntData - histo.header.pntErrors == nBins * sizeof(FLOAT32));
  histoout.write((char*)histo.nErrorEvents, histo.header.pntData-histo.header.pntErrors);

  //    assert(histo.header.size*2 == (nBins * 0x800 * 4));

  // header.size is a count in 16bit words!
  histoout.write((char*)histo.data, histo.header.size*2);
}

#ifdef USE_IS

void readHistogramToIS(const Scan &scan, const ScanEx &ex, UINT32 mid, scan_result_ptrs scanData) {
//   {
//     boost::mutex::scoped_lock lock(logMutex);
//     log << "Read Histogram for " << mid << " to IS\n";
//   }

  cout << "Writing histogram data to IS\n";

//   string sn = convertToString(mid);

  cout << "Module:  " << mid << /* " " << sn << */ endl;

  try {
    SctData::ScanResultWriter::publish(scanData);
  } catch(Sct::IoException &e) {
    cout << "ScanResult publish failed:\n" << e.what() << endl;
    e.sendToMrs(MRS_ERROR);
  }
}

#endif

int SctApi::pollHistogramming(boost::shared_ptr<ScanControl> controller, int timeout) {
  int returnVal = 0;

  bool timedout = false;

  time_t start_time = time(0);

  while(!timedout) {
    bool progressMade = false;
    bool newBin = false;
    if(controller->checkScanComplete(progressMade, newBin)) {
      // Finished, exit while
      break;
    }

    if(progressMade) {
      start_time = time(0);
    }

    if(newBin) {
      controller->nextBin();
    } else {
      // Check text buffers (no longer a side effect of polling)
#if USE_THREADS
      // Text buffers already checked by primThread
      sleep(1);
#else
      awaitResponse(partition, crate, rod, 0);
#endif

      if((time(0) - start_time) > timeout) {
        timedout = true;
        returnVal = -1;

        controller->reportTimeout();
        controller->reportEventErrors();

        break;
      }
    }
  }

#if USE_IS
  if(m_isDict) {
    m_isDict->remove("SCTAPIServer.currentBin");
    m_isDict->remove("SCTAPIServer.maxBin");
  }
#endif

  return returnVal;
}

void SctApi::doHistogramSetup(const Scan &scan, const ScanEx &extra) {
  for(ScanEx::RodInfoMap::const_iterator ri = extra.rodInfo.begin();
      ri != extra.rodInfo.end();
      ri++) {
    const RodScanEx &rodInfo = ri->second;
    const RodLabel &rod = ri->first;

    // Clear event memory on all slaves
    // 8192 words from what was 0x80008000
    for(int slaveNumber=0; slaveNumber<numSlaves; slaveNumber++) {
      if(getCrate(rod.partition, rod.crate)->slavePresent(rod.rod, slaveNumber)) {
        if(getCrate(rod.partition, rod.crate)->getRodRevision(rod.rod) == 0xe) {
          setSlaveBlock(rod.partition, rod.crate, rod.rod, slaveNumber, 0x18000, 0x2000, 0);
        } else {
          setSlaveBlock(rod.partition, rod.crate, rod.rod, slaveNumber, 0x80008000, 0x2000, 0);
        }
      }
    }

    cout << "Setup event trapping...\n";

    boost::shared_ptr<PrimListWrapper> primList(new PrimListWrapper(1));

    setupEventTrapping(scan, extra, rod, primList);
    startEventTrapping(scan, extra, rod, primList);

    setupEventDump(scan, extra, rod, primList);

    if(checkDebugOption(DEBUG_EXTRA_DUMPS)) {
      standardRegisterDump(rod.partition, rod.crate, rod.rod);

      sendPrimList(rod.partition, rod.crate, rod.rod, primList);
      awaitResponse(rod.partition, rod.crate, rod.rod, 5);

      primList->clear(); // .reset(new PrimListWrapper(2));
    }

    cout << "Histogramming setup...\n";

    setupHistogramming(scan, extra, rod, scan.getOption(Scan::TIM), primList);

    // Wait for things to settle before starting histogramming
    sendPrimList(rod.partition, rod.crate, rod.rod, primList);

    int responseCode = awaitResponse(rod.partition, rod.crate, rod.rod, 5);
    if(responseCode != 0) {
      cout << "Histogram setup list failed! (ROD " << rod.rod << ")\n";
      throw SctApiException("Histogram setup list failed! (ROD " + boost::lexical_cast<string>(rod.rod) + ")");
    }
  }
}

void ScanControlRODHisto::startHistogramming() {
  for(ScanEx::RodInfoMap::const_iterator ri = scanEx->rodInfo.begin();
      ri != scanEx->rodInfo.end();
      ri++) {
    const RodLabel &rod = ri->first;

    // Setup debug register
    if(scan->getOption(Scan::DEBUG)) {
      if(checkDebugOption(DEBUG_SCAN_AUTO_STALL)) {
        cout << "** Debug mode setting AUTO_STALL\n"; 
        scanEx->diagnosticReg |= (1<<DR_AUTO_STALL);
      }

      if(checkDebugOption(DEBUG_SCAN_STEP_MODE)) {
        cout << "** Debug mode setting STEP_CTRL\n"; 
        scanEx->diagnosticReg |= (1<<DR_STEP_CTRL);
      }

      if(checkDebugOption(DEBUG_SCAN_PAUSE_PULSE)) {
        cout << "** Debug mode setting PAUSE_PULSE\n"; 
        scanEx->diagnosticReg |= (1<<DR_PULSE_PAUSE);
      }
    }

    if(checkDebugOption(DEBUG_SCAN_ROD_MODE_BITS)) {
      cout << "Setting up to use ROD mode bits\n"; 
      scanEx->diagnosticReg |= (1<<DR_USE_ROD_MASK_LUT);
    }

    if(scanEx->diagnosticReg != 0x0) {
      cout << "Writing master diag reg: 0x" << hex << scanEx->diagnosticReg << dec << endl;
      api.dspSingleWrite(rod.partition, rod.crate, rod.rod, 0x80000010, scanEx->diagnosticReg, -1);
    } else {
      cout << "Clearing master diag reg\n";
      api.dspSingleWrite(rod.partition, rod.crate, rod.rod, 0x80000010, scanEx->diagnosticReg, -1);
    }

    cout << "Histogram starting...\n";

    boost::shared_ptr<PrimListWrapper> primList(new PrimListWrapper(1));
    api.startHistogramTask(*scan, *scanEx, rod, 
                           sectionStartBin, 1 + sectionEndBin - sectionStartBin, sectionEndTrigger, 
                           primList);

//     {
//       boost::mutex::scoped_lock lock(logMutex);
//       log << "Send start histogram task list\n";
//     }
    api.sendPrimList(rod.partition, rod.crate, rod.rod, primList);
    int responseCode = api.awaitResponse(rod.partition, rod.crate, rod.rod, 5);

    if(responseCode != 0) {
      cout << "Histogram primitive failed! (ROD " << rod.rod << ")\n";
      throw SctApiException("Histogram primitive failed! (ROD " + boost::lexical_cast<string>(rod.rod) + ")");
    }
  }
}

void SctApi::eventTrapSetup(int slaveMask, int trapMatch, int trapMod, int trapRemain, bool tim, bool error,
                            boost::shared_ptr<PrimListWrapper> primList) {
  long *evData = new long[sizeof(EVENT_TRAP_SETUP_IN)/4];

  EVENT_TRAP_SETUP_IN &primData = *(EVENT_TRAP_SETUP_IN*)evData;

#if R_EVENT_TRAP_SETUP == 103
  primData.slvBits = slaveMask;
  primData.numberOfEvents = COLLECT_FOREVER;
  primData.timeoutInUsec = 0x30000;

  primData.extRouterSetup = 0;

  // This tells the setup code to change the trapRemainder
  //  Only matters if setting up more than one slave
  primData.distribute = TRAP_DISTRIB_NEITHER;

  primData.releaseFrames = 0;
  if(tim) {
    primData.permitBackPressure = 0;
  } else {
    primData.permitBackPressure = 1;
  }
  primData.dataMode = 0;                        // Data or Event mode
  if(tim) {                           // Where to pick events (dsp or router)
    primData.sLink = 1;
  } else {
    primData.sLink = 0;
  }

  primData.format = error?TRAP_FMT_ERROR:TRAP_FMT_NORMAL; // Slink format
  primData.trapStray = 1;                       // Trap stray events
  primData.iterLimit = 2;                       // Don't keep them long

  if(tim) {
    // Trap events from the SLINK
    primData.trapConfig[0] = TRAP_CFG_SLINK_EVT;
  } else {
    // Trap events from ROD triggers
    primData.trapConfig[0] = TRAP_CFG_ROD_EVT;
  }
  primData.trapExclusionFlag[0] = 0;            // Don't exclude anything
  primData.trapFunction[0] = TRAP_FXN_HISTOGRAM;

  primData.trapMatch[0] = trapMatch;
  primData.trapModulus[0] = trapMod;
  primData.trapRemainder[0] = trapRemain;

  // Trap one is idle
  primData.trapConfig[1] = TRAP_CFG_IDLE;
  primData.trapExclusionFlag[1] = 0;
  primData.trapFunction[1] = 0;

  primData.trapMatch[1] = 0; 
  primData.trapModulus[1] = 0;
  primData.trapRemainder[1] = 0;
#else 
#error "Event trapping not compiled (new Primitive definition)!"
#endif
  primList->addPrimitive(RodPrimitive(sizeof(EVENT_TRAP_SETUP_IN)/4 + 4, 
                                      0, EVENT_TRAP_SETUP, R_EVENT_TRAP_SETUP, evData),
                         evData);
}

void SctApi::startEventTrap(int slaveNumber, boost::shared_ptr<PrimListWrapper> primList) {
  shared_ptr<PrimListWrapper> startList(new PrimListWrapper(1));

  PrimBuilder::instance().startEvTrap(startList);

  PrimBuilder::instance().slavePrimList(primList, startList, slaveNumber, true, true);
}

void SctApi::stopEventTrap(int slaveNumber, boost::shared_ptr<PrimListWrapper> primList) {
  shared_ptr<PrimListWrapper> stopList(new PrimListWrapper(1));

  PrimBuilder::instance().stopEvTrap(stopList);

  PrimBuilder::instance().slavePrimList(primList, stopList, slaveNumber, true, true);
}

unsigned long *ScanControlRODHisto::getHistogramChunk(RodLabel rodLabel, int slaveNumber,
                                                      unsigned long offset, unsigned long size) {
  try {
    pair<RodLabel, int> key = make_pair(rodLabel, slaveNumber);

    Utility::MemoryBlock chunk;
    chunk = scanEx->histoMap[key];
    if(!chunk) {
      pair<UINT32, UINT32> chunkPos = api.sendData(rodLabel.partition, rodLabel.crate, rodLabel.rod, HISTOGRAM_DATA, slaveNumber);

      cout << "SendData returned " << chunkPos.first << " " << chunkPos.second << endl;
      cout << " need offset " << offset << " length " << size << endl;

      unsigned long *block = api.primReadSlaveDsp(rodLabel.partition, rodLabel.crate, rodLabel.rod, 
                                              slaveNumber, chunkPos.first, chunkPos.second/4);
      chunk = Utility::MemoryBlock(chunkPos.second/4, block);

      scanEx->histoMap[key] = chunk;
    }

    if((offset + size*4) > (chunk.size() * 4)) {
      cout << "Asked for chunk +" << offset << " length " << size << " in " << chunk.size() << "words (continuing)\n";
    }

    // Offset is in bytes
    return chunk.address() + (offset / 4);
  } catch(SctApiException &e) {
    unsigned long slaveAddress;

    if(api.getRodRevision(rodLabel) == 0xe) {
      slaveAddress = 0xa00e2000;
    } else {
      slaveAddress = HISTOGRAM_DEFAULT_BASE;     // 0x20d2000;
    }

    return api.primReadSlaveDsp(rodLabel.partition, rodLabel.crate, rodLabel.rod, 
                                slaveNumber, slaveAddress + offset, size);
  }
}

unsigned long *ScanControlRODHisto::getEventCountChunk(RodLabel rodLabel, int slaveNumber) {
  try {
    pair<RodLabel, int> key = make_pair(rodLabel, slaveNumber);

    Utility::MemoryBlock chunk = scanEx->evCountMap[key];
    if(!chunk) {
      pair<UINT32, UINT32> chunkPos = api.sendData(rodLabel.partition, rodLabel.crate, rodLabel.rod, BIN_DATA, slaveNumber);

      cout << "SendData bin returned " << chunkPos.first << " " << chunkPos.second << endl;

      unsigned long *block = api.primReadSlaveDsp(rodLabel.partition, rodLabel.crate, rodLabel.rod, 
                                                  slaveNumber, chunkPos.first, chunkPos.second);
      chunk = Utility::MemoryBlock(chunkPos.second, block);

      scanEx->evCountMap[key] = chunk;
    }

    return chunk.address();
  } catch(SctApiException &e) {
    return 0;
  }
}


ScanControlRODHisto::ScanControlRODHisto(SctApi &api, boost::shared_ptr<Scan> aScan, boost::shared_ptr<ScanEx> aScanEx) : ScanControl(api), scan(aScan), scanEx(aScanEx) {
  int nBins = scan->getScanPoints1().size();

  Scan::TrigPoints trigPoints = scan->getVariableTrigs();

  finalTrigger = trigPoints[nBins-1];

  finalBin = nBins - 1;

  UINT32 initialNTrigs = trigPoints[0];

  unsigned int i = 0;
  while(i<trigPoints.size() && trigPoints[i] >= initialNTrigs) 
    i++;

  // i is now the index of the first different point
  sectionStartBin = 0;
  sectionEndBin = i - 1;
  if(i==trigPoints.size()) {
    sectionEndTrigger = scan->getVariableTrigs()[i-1];
  } else {
    sectionEndTrigger = scan->getVariableTrigs()[i-1] - scan->getVariableTrigs()[i];
  }
  cout << "Final bin = " << finalBin << " trigger = " << finalTrigger << endl;

  cout << "Initial sectionEndBin = " << sectionEndBin << " sectionEndTrigger = " << sectionEndTrigger << endl;
}

void ScanControlRODHisto::initialisePolling() { 
  scanStart = time(0);

  cout << "Histogram polling started\n";
  // Let it get going so the bin counter gets reset
  sleep(1);

  cout << "Polling histogramming task\n";
  cout << "scan point count " << scan->getScanPoints1().size() << endl;

  for(ScanEx::RodInfoMap::const_iterator ri = scanEx->rodInfo.begin();
      ri != scanEx->rodInfo.end();
      ri++) {
    const RodLabel &rodLabel = ri->first;

    if(api.mrs) {
      *api.mrs << "POLL_ROD" << MRS_DIAGNOSTIC << MRS_QUALIF("SCTAPI") 
               << MRS_PARAM<int>("partition", rodLabel.partition)
               << MRS_PARAM<int>("crate", rodLabel.crate)
               << MRS_PARAM<int>("rod", rodLabel.rod)
               << MRS_PARAM<int>("run", scan->getRunNumber())
               << MRS_PARAM<int>("scan", scan->getScanNumber())
               << MRS_TEXT("ROD polling started")
               << ENDM;
    }
  }

#if USE_IS
  if(api.m_isDict) {
    ISInfoInt binNumber(0);
    api.m_isDict->insert("SCTAPIServer.currentBin", binNumber);

    ISInfoInt maxNumber(finalBin);
    api.m_isDict->insert("SCTAPIServer.maxBin", maxNumber);
  }
#endif
}

void ScanControlRODHisto::reportTimeout() {
  cout << "Histogram polling timeout\n";

  RodLabel minROD;
  if(scanEx->rodInfo.size() > 0) {
    minROD = scanEx->rodInfo.begin()->first;
  }

  cout << "Expecting: " << finalBin << " " << finalTrigger << endl;

  // Find the ROD that stopped everything
  for(ScanEx::RodInfoMap::const_iterator ri = scanEx->rodInfo.begin();
      ri != scanEx->rodInfo.end();
      ri++) {
    const RodLabel &rodLabel = ri->first;
    cout << "   ROD (" << rodLabel.partition << ", " << rodLabel.crate << ", " << rodLabel.rod << "): ";
    cout << " Event " << lastEvent[rodLabel] << "    Bin " << lastBin[rodLabel] << "\n";

    if(lastEvent[rodLabel] < lastEvent[minROD]) {
      minROD = rodLabel;
    } else if(lastEvent[rodLabel] == lastEvent[minROD] && lastBin[rodLabel] < lastBin[minROD]) {
      minROD = rodLabel;
    }
  }

  if(api.mrs) {
    *api.mrs << "SCAN_ABORTED" << MRS_ERROR
             << MRS_PARAM<int>("partition", minROD.partition)
             << MRS_PARAM<int>("crate", minROD.crate)
             << MRS_PARAM<int>("rod", minROD.rod)
             << MRS_PARAM<int>("run", scan->getRunNumber())
             << MRS_PARAM<int>("scan", scan->getScanNumber())
             << MRS_PARAM<int>("bin", lastBin[minROD]) << MRS_PARAM<int>("event", lastEvent[minROD]) 
             << MRS_TEXT("Scan aborted (histogramming stalled)") << ENDM;
  }
}

void ScanControlRODHisto::nextBin() {
  // Fix ending variables if reached the end
  if(findNextSection()) {
    finalTrigger = sectionEndTrigger;
    finalBin = sectionEndBin;
    return;
  }

  for(ScanEx::RodInfoMap::const_iterator ri = scanEx->rodInfo.begin();
      ri != scanEx->rodInfo.end();
      ri++) {
    const RodLabel &rod = ri->first;

    cout << "New histogram control task...\n";

    boost::shared_ptr<PrimListWrapper> primList(new PrimListWrapper(1));
    api.startHistogramTask(*scan, *scanEx, rod, 
                           sectionStartBin, 1 + sectionEndBin - sectionStartBin, sectionEndTrigger, 
                           primList);

    api.sendPrimList(rod.partition, rod.crate, rod.rod, primList);
    int responseCode = api.awaitResponse(rod.partition, rod.crate, rod.rod, 5);

    if(responseCode != 0) {
      cout << "Histogram primitive failed! (ROD " << rod.rod << ")\n";
      throw SctApiException("Histogram primitive failed! (ROD " + boost::lexical_cast<string>(rod.rod) + ")");
    }
  }  
}

bool ScanControlRODHisto::findNextSection() {
  // Find next section to do
  Scan::TrigPoints trigPoints = scan->getVariableTrigs();

  unsigned int i = sectionEndBin + 1; // StartBin + 1;

  if(i==trigPoints.size()) {
    // Nothing more to do
    // If we're done, the previous section was last so don't overwrite the parameters
    return true;
  }

  UINT32 initialNTrigs = trigPoints[i];

  while(i<trigPoints.size() && trigPoints[i] >= initialNTrigs)
    i ++; 

  bool done = false;

  sectionStartBin = 0;
  sectionEndBin = i - 1;

  if(i == trigPoints.size()) {
    sectionEndTrigger = scan->getVariableTrigs()[i-1];
  } else {
    sectionEndTrigger = scan->getVariableTrigs()[i-1] - scan->getVariableTrigs()[i];

//     sectionStartBin = i;
//     sectionEndBin = trigPoints.size() - 1;
//     sectionEndTrigger = scan->getVariableTrigs()[i] - completedTriggers[i];
  }

#if 0 
  // The old way
  unsigned int i = sectionEndBin + 1;
  UINT32 initialNTrigs = trigPoints[i];
  while(i<trigPoints.size() && trigPoints[i] == initialNTrigs) 
    i++;

  sectionStartBin = sectionEndBin + 1;
  sectionEndBin = i - 1;
  sectionEndTrigger = scan->getVariableTrigs()[i-1];

  done = false;
#endif

  cout << "Next sectionStartBin = " << sectionStartBin << " sectionEndBin = " << sectionEndBin << " sectionEndTrigger = " << sectionEndTrigger << endl;

  return done;
}

bool ScanControlRODHisto::checkScanComplete(bool &progressMade, bool &newBin) {
  progressMade = false;
  newBin = false;

  bool result = true;

  int totalBin = 0;

  for(ScanEx::RodInfoMap::const_iterator ri = scanEx->rodInfo.begin();
      ri != scanEx->rodInfo.end();
      ri++) {
    const RodLabel &rodLabel = ri->first;

    if(!checkScanCompleteROD(rodLabel, progressMade, newBin, totalBin)) {
      result = false;
    }
  }

  if(progressMade) {
    int progressBin = totalBin / scanEx->rodInfo.size();

#if USE_IS
    if(api.m_isDict) {
      ISInfoInt binNumber(progressBin);
      api.m_isDict->update("SCTAPIServer.currentBin", binNumber);
    }
#endif
  }

  return result;
}

bool ScanControlRODHisto::checkScanCompleteROD(const RodLabel &rodLabel, bool &progressMade, bool &newBin, int &totalBin) {
  int partition = rodLabel.partition;
  int crate = rodLabel.crate;
  int rod = rodLabel.rod;

  int currBin = 0;
  int currEvent = 0;

  // Read bin number from master dsp
  long reg = api.dspSingleRead(partition, crate, rod, 0x80000020, -1);
  currBin = (reg & 0xff);

#warning "This is slave 0's event count!"
  // Read trigger number from master dsp copy (Was reading 0x8000002c on slave)
  reg = api.dspSingleRead(partition, crate, rod, 0x8000002c, -1);
  currEvent = (int) reg & 0xffff;

  if(currBin != lastBin[rodLabel]) {
    lastBin[rodLabel] = currBin;
    lastEvent[rodLabel] = currEvent;
    cout << "Next bin " << currBin << endl;

    progressMade = true;
  } else if(currEvent != lastEvent[rodLabel]) { 
    lastEvent[rodLabel] = currEvent;
    cout << "Next event " << currEvent << " (bin " << currBin << ")\n";

    progressMade = true;
  }

  totalBin += currBin;

  // End of current section
  if(currBin == sectionEndBin && currEvent == sectionEndTrigger) {
    newBin = true;
  } else if(currBin == sectionEndBin && currEvent == (sectionEndTrigger%65536)) {
    // Should check trigger count on slave (not 16bit?)

    // If progress was made we'll wait for the next iteration
    if(!progressMade) {
      // Otherwise its most likely finished properly...
      newBin = true;
    }
  }

  if(((currBin == finalBin) && (currEvent == finalTrigger))
     || ((currBin == finalBin) && (currEvent == (finalTrigger%65536)) && !progressMade)) {
    if(api.mrs) {
      *api.mrs << "HISTO_SUCCESS" << MRS_DIAGNOSTIC << MRS_QUALIF("SCTAPI") 
               << MRS_PARAM<int>("partition", rodLabel.partition)
               << MRS_PARAM<int>("crate", rodLabel.crate)
               << MRS_PARAM<int>("rod", rodLabel.rod)
               << MRS_TEXT("Histogramming successfully complete")
               << ENDM;
    }

    cout << "Histogramming completed successfully\n";

    return true;
  }

  return false;
}

void ScanControlRODHisto::finishHistogram(bool success) { 
  time_t scanEnd = time(0);

  scan->setEndTime(second_clock::universal_time());

  // Read out if successful (even in debug mode!)
  //  but don't readout in debug mode if polling failed
  if(!scan->getOption(Scan::DEBUG) || success == true) {
    if(api.mrs) {
      *api.mrs << "SCAN_FINISHED" << MRS_INFORMATION 
               << MRS_QUALIF("SCTAPI") << MRS_QUALIF("doScan") 
               << MRS_TEXT("Scan finished") 
               << MRS_PARAM<int>("ScanNumber", scan->getScanNumber())
               << MRS_PARAM<int>("ScanType1", scan->getScanVariable1())
               << MRS_PARAM<int>("ScanType2", scan->getScanVariable2())
               << MRS_PARAM<int>("ScanTime", scanEnd - scanStart)
               << ENDM;
    }

    cout << "Histogramming complete, " << (scanEnd - scanStart) << " seconds, tidying up...\n";
//     {
//       boost::mutex::scoped_lock lock(logMutex);
//       log << "Histogramming complete, " << (scanEnd - scanStart) << " seconds\n";
//     }

    cout << "Reading histograms...\n";

    readHistograms();

    cout << "... histograms read\n";
  } else {
    if(!scan->getOption(Scan::DEBUG)) {
      cout << "Histogram polling failed\n";

      if(api.mrs) {
        *api.mrs << "SCAN_FAILED" << MRS_INFORMATION 
                 << MRS_QUALIF("SCTAPI") << MRS_QUALIF("doScan") 
                 << MRS_TEXT("Scan polling failed") 
                 << MRS_PARAM<int>("ScanNumber", scan->getScanNumber())
                 << MRS_PARAM<int>("ScanTime", scanEnd - scanStart)
                 << ENDM;
      }
    }
  }

  if(checkDebugOption(DEBUG_PRINT_CALIB)) {
    RodLabel zeroRod = scanEx->rodInfo.begin()->first;

    api.print_calib(zeroRod.partition, zeroRod.crate, zeroRod.rod);
  }

  if(!scan->getOption(Scan::DEBUG)) {
    api.stopHistogramming(*scanEx);

    // Only restore module config if not debug mode
    postScanModuleSetup();
  } else {
    if(api.mrs) {
      *api.mrs << "SCAN_DEBUG" << MRS_INFORMATION 
               << MRS_QUALIF("SCTAPI") << MRS_QUALIF("doScan") 
               << MRS_TEXT("Scan done in debug mode, probably best to be at the command line to sort it out!") 
               << ENDM;
    }

    cout << "** In debug mode, you're responsible for terminating histogram and for readout\n";
  }

  if(checkDebugOption(DEBUG_PRINT_CALIB)) {
    RodLabel zeroRod = scanEx->rodInfo.begin()->first;

    api.print_calib(zeroRod.partition, zeroRod.crate, zeroRod.rod);
  }

  // This should now have been updated with the modules in the scan
  scan->print();
  
#if USE_IS
  if(api.m_isDict) 
    api.m_isDict->remove("SCTAPIServer.scanning");
#endif

  if(api.mrs)
    *api.mrs << "SCAN_COMPLETE" << MRS_INFORMATION << MRS_QUALIF("SctApi") << MRS_TEXT("Scan has completed") << ENDM;
}

void ScanControlRODHisto::postScanModuleSetup() {
  PrimBuilder &builder = PrimBuilder::instance();

  if(scan->getScanVariable1() == ST_TOKEN
     || scan->getScanVariable1() == ST_BYPASS) {
    // Restore chip sequence check in EFB
    shared_ptr<PrimListWrapper> chipSeqList(new PrimListWrapper(1));
    for(int i=0; i<48; i++) {
      builder.writeRegister(chipSeqList, ERROR_MASK(0, i), 10, 1, 0);
      builder.writeRegister(chipSeqList, ERROR_MASK(1, i), 10, 1, 0);
    }

    if(api.synchSendPrimListAllCrates(chipSeqList)) {
      cout << "Chip sequence error list failed!\n";
    }

    cout << "EFB chip sequence detection turned back on\n";
  }

  // Restore checks cancelled for double triggers which aren't supported by DSP code
  {
    Trigger::RODTriggers points = scan->getTrigger1()->getRODTriggers();

    int triggerCount = 0;

    for(unsigned int i=0; i<points.size(); i++) {
      // Count triggers
      if(points[i].first == L1_TRIGGER) {
        triggerCount++;
      }
    }

    // Double triggers... none of current options will cope...
    if(triggerCount > 1 && scan->getOption(Scan::DISTSLAVE) < 3) {
      shared_ptr<PrimListWrapper> dblIdCheckList(new PrimListWrapper(1));

      // Mask L1 and BCID checks
      for(int f=0; f<48; f++) {
        // L1ID
        builder.writeRegister(dblIdCheckList, ERROR_MASK(0, f), 5, 1, 0);
        builder.writeRegister(dblIdCheckList, ERROR_MASK(1, f), 5, 1, 0);

        // BCID
        builder.writeRegister(dblIdCheckList, ERROR_MASK(0, f), 6, 1, 0);
        builder.writeRegister(dblIdCheckList, ERROR_MASK(1, f), 6, 1, 0);
      }

      if(api.synchSendPrimListAllCrates(dblIdCheckList)) {
        cout << "Double trigger BC check list failed!\n";
      }
    }
  }

  //    getABCDModules(SCTAPI_BANK_PHYSICS);
  //    sendAllABCDModules(SCTAPI_BANK_PHYSICS);
}

bool ScanControlRODHisto::checkDebugOption(int opt) {
  return Debug::getInstance()->checkDebugOption((DebugOptions)opt);
}

void ScanControlRODHisto::reportEventErrors() {
  api.reportEventErrors();
}

unsigned int ScanControlRODHisto::getProcTime(const RodLabel rlabel, int dsp) {
  int procTime;

  if(dsp == -1) {
    throw SctApiException("Can't getProcTime on MDSP");
  } else {
    unsigned long SDSP_IDRAM_BASE;
    if(api.getRodRevision(rlabel) == 0xe) {
      SDSP_IDRAM_BASE = 0x10000;        // Rev E
    } else {
      SDSP_IDRAM_BASE = 0x80000000;     // Rev C
    }

    unsigned long length;
    unsigned long *regs = api.dspBlockRead(rlabel.partition, rlabel.crate, rlabel.rod, 
                                           SDSP_IDRAM_BASE + 0x10, 20, dsp, length);

    if(regs) {
      long reg;

      // H stat reg 0
      reg = regs[6];
      procTime = ((reg >> 16) & 0xffff);
    }
  }

  return procTime;
}

ScanControlRODHisto::TrapBuffers ScanControlRODHisto::getTrapBuffers(const RodLabel rlabel, int dsp) {
  int ihead, itail, xhead, xtail;

  if(dsp == -1) {
    throw;
  } else {
    unsigned long SDSP_IDRAM_BASE;
    if(api.getRodRevision(rlabel) == 0xe) {
      SDSP_IDRAM_BASE = 0x10000;        // Rev E
    } else {
      SDSP_IDRAM_BASE = 0x80000000;     // Rev C
    }

    unsigned long length;
    unsigned long *regs = api.dspBlockRead(rlabel.partition, rlabel.crate, rlabel.rod, 
                                           SDSP_IDRAM_BASE + 0x10, 20, dsp, length);

    if(regs) {
      long reg;

      // Trap stat 1
      reg = regs[1];
      itail = ((reg & 0xff0000) >> 16);
      xtail = ((reg & 0xff000000) >> 24);

      // Trap stat 1
      reg = regs[2];
      ihead = ((reg & 0xff0000) >> 16);
      xhead = ((reg & 0xff000000) >> 24);
    }
  }

  return make_pair(make_pair(ihead, itail), make_pair(xhead, xtail));
}

void ScanControlRODHisto::dumpHistoStatus(const RodLabel rlabel, int dsp) {
  if(dsp == -1) {
    const unsigned long MDSP_IDRAM_BASE = 0x80000000;

    // Histogram command stat 0
    long reg = api.dspSingleRead(0, 0, 0, MDSP_IDRAM_BASE + 0x20, -1);
    cout << "Bin: " << (reg & 0xff) 
         << " Cal: " << ((reg & 0xff00) >> 8) 
         << " Errs: " << ((reg & 0xffff0000) >> 16) << endl;

    // Histogram command stat 1
    reg = api.dspSingleRead(0, 0, 0, MDSP_IDRAM_BASE + 0x24, -1);
    cout << "avg. trans " << (reg & 0xff)
         << " avg. len " << ((reg & 0xff00) >> 8)
         << " avg. proc. " << ((reg & 0xffff0000) >> 16) << endl;

    // Histogram status 1
    reg = api.dspSingleRead(0, 0, 0, MDSP_IDRAM_BASE + 0x2c, -1);
    cout << "Slave 0: " << (reg & 0xffff)
         << " Slave 1: " << ((reg & 0xffff0000) >> 16) << endl;

    // Histogram status 0
    reg = api.dspSingleRead(0, 0, 0, MDSP_IDRAM_BASE + 0x28, -1);
    cout << "Slave 2: " << (reg & 0xffff)
         << " Slave 3: " << ((reg & 0xffff0000) >> 16) << endl;
  } else {
    unsigned long SDSP_IDRAM_BASE;
    if(api.getRodRevision(rlabel) == 0xe) {
      SDSP_IDRAM_BASE = 0x10000;        // Rev E
    } else {
      SDSP_IDRAM_BASE = 0x80000000;     // Rev C
    }

    unsigned long length;
    unsigned long *regs = api.dspBlockRead(rlabel.partition, rlabel.crate, rlabel.rod, 
                                           SDSP_IDRAM_BASE + 0x10, 20, dsp, length);

    if(regs) {
      long reg;

      cout << "TrapStatus: \n";


      // Trap stat 0
      reg = regs[1];
      int words = ((reg &0xffff) >> 0);
      int itail = ((reg & 0xff0000) >> 16);
      int xtail = ((reg & 0xff000000) >> 24);

      cout << "Event word count: " << words;
      cout << " IFrame tail: " << itail;
      cout << " XFrame tail: " << xtail << endl;

      // Trap stat 1
      reg = regs[2];
      int ihead = ((reg & 0xff0000) >> 16);
      int xhead = ((reg & 0xff000000) >> 24);
      cout << " IFrame head: " << ihead;
      cout << " XFrame head: " << xhead << endl;

      // Trap cmd stat
      reg = regs[17];
      cout << "Trap Command/Status: " 
           << ((reg & 0x1)?"Trailer ":"")            // TCSR_TRAILER
           << ((reg & 0x2)?"Transmit ":"")
           << ((reg & 0x4)?"Header ":"")
           << ((reg & 0x4)?"ISR active ":"");
      cout << ((reg & 0x10)?"\"Data Error\" ":"")
           << ((reg & 0x20)?"\"Header Error\" ":"")
           << ((reg & 0x40)?"\"Trailer Error\" ":"")
           << ((reg & 0x80)?"\"Link Error\" ":"");
      cout << ((reg & 0x100)?"\"Error\" ":"")
           << ((reg & 0x200)?"\"Overflow(old)\" ":"")
           << ((reg & 0x800)?"\"ISR pending\" ":"");
      cout << ((reg & 0x4000)?"\"Overflow error\" ":"")
           << ((reg & 0x8000)?"\"Overflow\" ":"");
      cout << endl;

      int errCount = ((reg >> 16) & 0xff);
      cout << "\"Error count\": " << errCount << endl;
      int evCount = ((reg >> 24) & 0xff);
      cout << "\"Event count\": " << evCount << endl;

      // Hcmd stat reg 0
      reg = regs[4];
      int currentBin = (reg & 0xff);
      cout << "Bin: " << currentBin
           << " Cal: " << ((reg & 0x1f00) >> 8) 
           << ((reg & 0x800)?" Cal enabled":"")
           << ((reg & 0x1000)?" New bin":"")
           << ((reg & 0x20000)?" New cal":"") << endl;

      // Hcmd stat reg 1
      reg = regs[5];
      int numEvents = reg;
      cout << "Num events: " << numEvents << endl;

      // H stat reg 0
      reg = regs[6];
      cout << "Histogram status: 0x" << hex << (reg & 0xffff) << dec << " " << ((reg & 0x1)?"Ready ":"")
           << ((reg & 0x2)?"Expecting ":"")
           << ((reg & 0x4)?"Processing ":"")
           << ((reg & 0x8)?"Done ":"") << endl;

      int procTime = ((reg >> 16) & 0xffff);
      cout << "Bin err: " << ((reg >> 8) & 0xff) 
           << " proc time: " << ((reg >> 16) & 0xffff) << endl;

      // H stat reg 1
      reg = regs[7];
      int evRecvd = reg;
      cout << "Events recvd: " << evRecvd << endl;

      ofstream statusOut("TIMStatus.txt", ios_base::out | ios_base::app);
      statusOut << currentBin << "\t" 
                << ihead  << "\t" << itail << "\t" 
                << xhead << "\t" << xtail << "\t"
                << errCount << "\t" << evCount << "\t" 
                << evRecvd << "\t" << words << "\t" << procTime << endl;
    }
  }
}

ScanControlTIMHisto::ScanControlTIMHisto(SctApi &api, boost::shared_ptr<Scan> aScan, boost::shared_ptr<ScanEx> aScanEx) : ScanControlRODHisto(api, aScan, aScanEx), binCount(0) {
}

void ScanControlTIMHisto::startHistogramming() {
  // Set BCID offset to 3
  api.timWriteRegister(0, 0, 0x14, 0x3000);

  shared_ptr<PrimListWrapper> preTimList(new PrimListWrapper(4));
  PrimBuilder::instance().rodMode(preTimList, 0x20000, 0, 1, 0, 0, 1); 

  PrimBuilder::instance().writeRegister(preTimList, RTR_CMND_STAT, 7, 1, 1);   // Mask LDown#

  for(int f=0; f<8; f++) 
//     //    PrimBuilder::instance().writeRegister(preTimList, DM_DFLT_LUT(f), 0, 16, 0xaaaa);   // Decrement L1 for check
//     PrimBuilder::instance().writeRegister(preTimList, DM_DFLT_LUT(f), 0, 16, 0x5555);   // Increment L1 for check
    PrimBuilder::instance().writeRegister(preTimList, DM_DFLT_LUT(f), 0, 16, 0x0);   // Clear L1 inc

//   for(int ch=0; ch<48; ch++) {
//     PrimBuilder::instance().writeRegister(preTimList, ERROR_MASK(0, ch), 5, 1, 1);  // Mask L1ID check
//     PrimBuilder::instance().writeRegister(preTimList, ERROR_MASK(1, ch), 5, 1, 1);  // Mask L1ID check
//   }

  cout << "Sending pre TIM setup list\n";
  api.synchSendPrimListAllCrates(preTimList);
  cout << " done sending pre TIM setup list\n";

  hex(cout);
  cout << "0x" << RTR_CMND_STAT << ": 0x" << api.readRODRegister(0, 0, 0, RTR_CMND_STAT) << endl;
  cout << "0x" << ERROR_MASK(0, 0) << ": 0x" << api.readRODRegister(0, 0, 0, ERROR_MASK(0, 0)) << endl;
  cout << "0x" << ERROR_MASK(1, 0) << ": 0x" << api.readRODRegister(0, 0, 0, ERROR_MASK(1, 0)) << endl;
  cout << "0x" << DM_DFLT_LUT(0) << ": 0x" << api.readRODRegister(0, 0, 0, DM_DFLT_LUT(0)) << endl;
  cout << "0x" << DM_DFLT_LUT(1) << ": 0x" << api.readRODRegister(0, 0, 0, DM_DFLT_LUT(1)) << endl;
  dec(cout);

//   // Do first bin...
//   nextBin();
}

void ScanControlTIMHisto::nextBin() {
  cout << "TIM histo, next bin\n";

  api.modifyABCDVarROD(scan->getScanVariable1(), scan->getScanPoints1()[binCount], SCTAPI_BANK_SCAN);
  api.sendAllABCDModules(SCTAPI_BANK_SCAN);

  // SET_TRIGGER (ie tell histograms about next bin)
  shared_ptr<PrimListWrapper> setTriggerList(new PrimListWrapper(4));
  PrimBuilder::instance().setTrigger(setTriggerList, binCount, Utility::translateBank(SCTAPI_BANK_SCAN));

  shared_ptr<PrimListWrapper> binTimList(new PrimListWrapper(4));

  // Send to all slaves
  for(int s=0; s<4; s++) {
    PrimBuilder::instance().slavePrimList(binTimList, setTriggerList, 
                                          s, 1, 0);
  }

  PrimBuilder::instance().rodMode(binTimList, 0x20000, 0, 1, 0, 0, 1); 
  PrimBuilder::instance().writeRegister(binTimList, RTR_CMND_STAT, 7, 1, 1);   // Mask LDown#

  cout << "Sending TIM setup list\n";
  api.synchSendPrimListAllCrates(binTimList);
  cout << " done sending TIM setup list\n";

  if(checkDebugOption(DEBUG_TIM_SCAN_STATUS)) {
    cout << "******************* Pre triggers status ********************\n";
    dumpHistoStatus(RodLabel(), 0);
  }

#warning "Reset one crate only"
  api.timSoftReset(0, 0);
  api.timBCReset(0, 0);

  Scan::TrigPoints trigs = scan->getVariableTrigs();

  cout << "Sending " << trigs[binCount] << " triggers to one crate " << endl;

  switch(scan->getOption(Scan::TIM)) {
  case 1:
    // Just send lots of triggers, relies on user to set up frequency/nth appropriately
    api.sendTimBurst(0, 0, trigs[binCount]);
    break;
  case 2:
    // Send one trigger at a time, then wait maximum of 1 second for trap buffers to equalize
    for(int i=0; i<trigs[binCount]; i++) {
      api.timL1A(0, 0);

      TrapBuffers tb = getTrapBuffers(RodLabel(), 0);

      if (!(tb.first.first == tb.first.second && tb.second.first == tb.second.second))
        sleep(1);
    }
    break;
  case 3:
    // Send in bursts of triggers, wait for them to finish
    // Designed to be run at 100kHz
    {
      unsigned int sentTriggers = 0;
      const unsigned int trigsPerBurst = 20;

      while(sentTriggers < trigs[binCount]) {
        if(sentTriggers % 1000 < trigsPerBurst) {
          cout << "Done " << sentTriggers << " triggers\n";
          if(checkDebugOption(DEBUG_TIM_SCAN_STATUS))
             dumpHistoStatus(RodLabel(), 0);
        }
        int ntrigs = min(trigs[binCount]-sentTriggers, trigsPerBurst);
        api.sendTimBurst(0, 0, ntrigs);
        sentTriggers += ntrigs;

        // Have they been processed yet?
        TrapBuffers tb = getTrapBuffers(RodLabel(), 0);

        TrapBuffers oldtb;
        if (!(tb.first.first == tb.first.second && tb.second.first == tb.second.second))
          do {
            oldtb = tb;
            usleep(trigsPerBurst * 10);  // 7 * 10us at 100kHz 
            tb = getTrapBuffers(RodLabel(), 0);

            if(tb.first.first == tb.first.second && tb.second.first == tb.second.second) break;
            if(tb == oldtb) break;
          } while(1); 
      }
    }
    break;
  case 4:
    // Adjust frequency depending on how long the previous set of triggers 
    //  takes to histogram, first 1 then 10 then the rest.
    {
      unsigned int sentTriggers = 0;

      api.timL1A(0, 0);
      sentTriggers ++;

//   // Set 100kHz triggers
//   api.timSetFrequency(0, 0, 100, 0);

      unsigned int procTime = getProcTime(RodLabel(), 0);
      double timFrequency = 0.25 * 1e6/procTime;
      // Don't go faster than 6kHz
      api.timSetFrequency(0, 0, min(6.0, timFrequency/1000), 0);

      cout << "*** Proc Time for 1 trigger was " << procTime << " setting frequency to " << timFrequency << "Hz\n";

      procTime = getProcTime(RodLabel(), 0);
      timFrequency = 0.25 * 1e6/procTime;
      api.timSetFrequency(0, 0, min(6.0, timFrequency/1000), 0);

      cout << "*** Proc Time after 10 trigs was " << procTime << " setting frequency to " << timFrequency << "Hz\n";

      if(trigs[binCount] > sentTriggers)
        api.sendTimBurst(0, 0, trigs[binCount]-sentTriggers);

      if(checkDebugOption(DEBUG_TIM_SCAN_STATUS)) {
        cout << "******************* Post triggers status ********************\n";
        dumpHistoStatus(RodLabel(), 0);
      }

      TrapBuffers tb = getTrapBuffers(RodLabel(), 0);

      TrapBuffers oldtb;
      if (!(tb.first.first == tb.first.second && tb.second.first == tb.second.second))
        do {
          oldtb = tb;
          sleep(1);
          tb = getTrapBuffers(RodLabel(), 0);

          if(tb.first.first == tb.first.second && tb.second.first == tb.second.second) break;
          if(tb == oldtb) break;
        } while(1); 
      break;
    }
  } // End switch on TIM histogramming types

  if(checkDebugOption(DEBUG_TIM_SCAN_STATUS)) {
    cout << "******************* Post triggers status 2 ******************\n";
    dumpHistoStatus(RodLabel(), 0);
  }

  //  TTrigger soft;
  //soft.softL1A(30);

  //  api.timL1A(0, 0);

  binCount ++;
}

// TIM histos don't run Histo Control so event counters not mirrored to master
// A sort of dummy function at the moment...
bool ScanControlTIMHisto::checkScanCompleteROD(const RodLabel &rodLabel, bool &progressMade, bool &newBin, int &totalBin) {

  cout << "Checking TIM scan complete\n";

  progressMade = true;

  // If the triggers are sent synchronously in newBin then they must be done...
  newBin = true;

  totalBin += binCount-1;

  //  cout << "Are we at the end: " << (binCount-1) << " " << finalBin << endl;

  if((binCount-1 >= finalBin)) {
    return true;
  }

  return false;
}

// bool ScanControlTIMHisto::checkScanComplete(bool &progressMade) {
//   return false;
// }

}
