// File: SctApi.cxx

#include <iostream>
#include <stdexcept>
#include <cmath>
#include <cstdio>  // For tempnam (and ScanResultWriter)
#include <unistd.h>

#include <sys/stat.h>   // mkdir etc
#include <sys/types.h>

#include <boost/bind.hpp>

#include "SctApi.h"
#include "SctApiConsts.h"
#include "SctApiDebug.h"
#include "SctApiHisto.h"
#include "sctConf/configXMLImpl.h"
#include "SctApiImpl.h"
#include "crateImpl.h"
#include "autoConfig.h"

#include "primUtils.h"
#include "primListWrapper.h"
#include "PrimBuilder.h"

#include "extraScans.h"
#include "SctApiRodInfo.h"

#include "ScanResultWriter/scan.h"
#include "Sct/SctNames.h"

#include "registerIndices.h"

#include "ABCD/ABCDscans.h"
#include "ABCD/ABCDchip.h"

#include "CommonWithDsp/primParams.h"

#include "RodCrate/RodModule.h"

#include "VmeDecode.h"

#include <boost/date_time/posix_time/posix_time.hpp>

#ifdef USE_IS
#include "Sct/IoExceptions.h"
#include "ScanResultWriter/dataTypes.h"
#endif

#include "utility.h"

using namespace std;
using namespace SctPixelRod;
using namespace SctConfiguration;
using namespace SctApi::Utility;
using namespace boost::posix_time;
using boost::shared_ptr;

// Some static methods
static void handle_unexpected(void);

namespace SctApi {

const unsigned int SctApi::BAD_MODULE = 0xffffffff;

/*
  The default constructor.
  This sets the unexpected handler to interpret all sorts of exceptions
*/
SctApi::SctApi() 
  : mrs(0),
#if USE_IS
    m_isDict(0), 
#endif
    rodList(), rodInfoList(), moduleMap(), scanController(), lastScanController(),
    config(), 
    ddcPartition(0), ddcCmd(0), dcsAccess(new DCSAccessDummy), debugPrimList(), crateMap(), 
    log(), lastDebugScan(), lastDebugScanEx(),
    scanNumber(0), runNumber(0), 
#if USE_SCAN_THREAD
    scanQueue_notEmpty(), scanQueue_mutex(), scanPollThread(),
    m_stopPolling(false), m_inScanLoop(),
#endif
    logMutex()
{
  cout << "Load configuration\n";
  config.reset(new ConfigurationXMLImpl());
  setup();
}

SctApi::SctApi(boost::shared_ptr<Configuration> newConf) 
  : mrs(0),
#if USE_IS
    m_isDict(0), 
#endif
    rodList(), rodInfoList(), moduleMap(), scanController(), lastScanController(),
    config(), 
    ddcPartition(0), ddcCmd(0), dcsAccess(new DCSAccessDummy), debugPrimList(), crateMap(), 
    log(), lastDebugScan(), lastDebugScanEx(),
    scanNumber(0), runNumber(0), 
#if USE_SCAN_THREAD
    scanQueue_notEmpty(), scanQueue_mutex(), scanPollThread(),
    m_stopPolling(false), m_inScanLoop(),
#endif
    logMutex()
{
  config = newConf;
  setup();
}

/* The destructor. */
SctApi::~SctApi() {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "SctApi destructor\n";
    log << "\t" << second_clock::universal_time() << endl;
  }

#if USE_SCAN_THREAD
  m_stopPolling = true;
  scanPollThread->join();
#endif

  for(map<pair<unsigned int,unsigned int>, Crate*>::const_iterator i=crateMap.begin();i!=crateMap.end();i++){
    delete i->second;
  }
}

void SctApi::setup() 
{
  Debug::getInstance(); // Make sure setupDebugOptions() is called early

  debugPrimList.reset(new PrimListWrapper);

  // Examine exceptions on the way to a crash...
  set_unexpected(handle_unexpected);
  cout << "Installed unexpected handler\n";

  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "Initial setup complete\n";
  }

#ifdef USE_SCAN_THREAD
  scanPollThread.reset(new boost::thread(boost::bind(&SctApi::scanPollingThread, this)));
#endif
}

void SctApi::setMrsStream(MRSStream *stream) 
{
  mrs = stream;
}

#if USE_IS
void SctApi::setIsDictionary(ISInfoDictionary *dict)
{
  m_isDict = dict;
}
#endif

// ROD Diagnostics

// Read one ROD Status Register via HPI
unsigned long SctApi::readRodStatusReg(unsigned int partition, unsigned int crate, unsigned int rod,
                                       long regNumber) {
  unsigned long result = 0;

  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "readRodStatusReg " << regNumber << endl;
  }

  Crate *myCrate = getCrate(partition, crate);
  if(myCrate) {
    result = myCrate->readRodStatusReg(rod, regNumber);
  }

  return result;
}
  
// Read one ROD Command Register via HPI
unsigned long SctApi::readRodCommandReg(unsigned int partition, unsigned int crate, unsigned int rod,
                                        long regNumber) {
  unsigned long result = 0;

  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "readRodCommandReg\n";
  }

  Crate *myCrate = getCrate(partition, crate);
  if(myCrate) {
    result = myCrate->readRodCommandReg(rod, regNumber);
  }

  return result;
}

// Read a single 32b word from MasterDSP SDRAM via HPI
unsigned long SctApi::dspSingleRead(unsigned int partition, unsigned int crate, unsigned int rod,
                                    const unsigned long dspAddr, long dspNumber) {
  unsigned long result = 0;

  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "dspSingleRead 0x" << hex << dspAddr << dec << endl;
  }
  Crate *myCrate = getCrate(partition, crate);
  if(myCrate) {
    result = myCrate->dspSingleRead(rod, dspAddr, dspNumber);
  }

  return result;
}

// Write a single 32b word to MasterDSP SDRAM via HPI
void SctApi::dspSingleWrite(unsigned int partition, unsigned int crate, unsigned int rod,
                            unsigned long dspAddr, unsigned long value, long dspNumber) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "dspSingleWrite 0x" << hex << dspAddr << " 0x" << value << dec << endl;
  }
  Crate *myCrate = getCrate(partition, crate);
  if(myCrate) {
    myCrate->dspSingleWrite(rod, dspAddr, value, dspNumber);
  }
}

ABCDModule *SctApi::lookupConfig(UINT32 mid) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "lookupconfig " << mid << endl;
  }

  if(moduleMap.find(mid) == moduleMap.end()) {
    cacheModuleConfig(mid);
  }

  ABCDModule *result = 0;

  if(moduleMap.find(mid) != moduleMap.end()) {
    result = &(moduleMap.find(mid)->second);
  }

  return result;
}

void SctApi::cacheModuleConfig(UINT32 mid) {
  unsigned int partition, crate, rod, channel;
  getpcrc(mid, partition, crate, rod, channel);

  RodLabel label(partition, crate, rod);

  std::string moduleName;
  unsigned int mur, module;

  // First find out what this module is called
  try {
    if(checkDebugOption(DEBUG_DIAG))
      cout << "Looking for partition " << partition << " crate " << crate << " rod " << rod << " channel " << channel << endl;
    config->translateFromROD(partition, crate, rod, channel, 
                             mur, module);

    if(checkDebugOption(DEBUG_DIAG2))
      cout << "Found MUR " << mur << " module number " << module << endl;
  } catch(ConfigurationException &c) {
    cout << "Can't find module in MUR map: \n";
    cout << c.what() << endl;

    {
      boost::mutex::scoped_lock lock(logMutex);
      log << "  (Not found)\n";
    }

    throw;
  }

  try {
    config->translateToSN(mur, module, moduleName);

    if(checkDebugOption(DEBUG_DIAG2))
      cout << "Found module name " << moduleName << endl;
    //        config->getModuleRodPosition(moduleName, partition, crate, rod, channel);
  } catch(ConfigurationException &c) {
    cout << "Module name not found: \n";
    cout << c.what() << endl;

    {
      boost::mutex::scoped_lock lock(logMutex);
      log << "  (Not found)\n";
    }

    throw;
  }

  bool redundantOnROD = false;

  unsigned int rchannel;

  try {
    unsigned int rpartition, rcrate, rrod;
    config->translateToRROD(mur, module, 
                            rpartition, rcrate, rrod, rchannel);

    if(partition == rpartition && crate == rcrate && rod == rrod) {
      redundantOnROD = true;
    } else {
      RodLabel label(partition, crate, rod);
      RodLabel rlabel(rpartition, rcrate, rrod);

      RodInfo info(getRodInfo(label));
      RodInfo rinfo(getRodInfo(rlabel));

      unsigned int rchannel = 48 + rinfo.offRODTTC.size();

      unsigned long rmid = Utility::packpcrc(rpartition, rcrate, rrod, rchannel);

      info.offRODTTC[mid] = rmid;
      rinfo.offRODRX[rmid] = mid;
    }
  } catch(ConfigurationException &c) {
    cout << "Module redundancy not found: \n";
    cout << c.what() << endl;

    {
      boost::mutex::scoped_lock lock(logMutex);
      log << "  (Not found)\n";
    }

    // We can live with this (as long as select not set...?)
  }

  // Got the module serial number, now get the configuration
  ABCDModule configuration; 
  try {
    configuration = config->getModuleConfig(moduleName);

    // Fix up chip address for select
    char *mappings = config->getFibreMappings(partition, crate, rod);

    // Only modules with sn like 20220nn0200nnn are guaranteed to have select line connectly correctly
    // Others like 20220nn0100nnn may not
//     if(moduleName[8] == '2') {
      if(configuration.select) {
        for(int c=0; c<12; c++) {
          configuration.chip[c].address |= 0x10;
        }
      } else {
        for(int c=0; c<12; c++) {
          configuration.chip[c].address &= ~0x10;
        }
      } 
//     } // Otherwise select line on module might be hard-wired

    // Fix up for off ROD redundancy
    configuration.pTTC  = mappings[channel * 3 + 0];
    if(redundantOnROD)
      configuration.rTTC = mappings[rchannel * 3 + 0];
    else 
      configuration.rTTC = OFF_ROD_TTC;

    // Fix formatter addressing
    unsigned char rx0 = mappings[channel * 3 + 1];
    if(rx0 != DATA_LINK_OFF) {
      configuration.rx[0] = (rx0/12)*16 + rx0%12;

      if(rx0 < getRodInfo(label).minLink) {
        getRodInfo(label).minLink = rx0;
      }
    } else {
      configuration.rx[0] = DATA_LINK_OFF;
    }

    // Fix formatter addressing (receive 1)
    unsigned char rx1 = mappings[channel * 3 + 2];
    if(rx1 != DATA_LINK_OFF) {
      configuration.rx[1] = (rx1/12)*16 + rx1%12;

      if(rx1 < getRodInfo(label).minLink) {
        getRodInfo(label).minLink = rx1;
      }
    } else {
      configuration.rx[1] = DATA_LINK_OFF;
    }

    // Get Module group
    configuration.groupId = config->getModuleGroup(moduleName);

    cout << "\tpTTC = " << (int)configuration.pTTC << " rTTC = " << (int)configuration.rTTC 
         << " rx[0] = " << (int)configuration.rx[0] << " rx[1] = " << (int)configuration.rx[1] 
         << " group = " << (int)configuration.groupId << endl;

    // Put fixed configuration into cache
    moduleMap.insert(make_pair(mid, configuration));

    if(mrs) {
      int link0 = configuration.rx[0];
      int link1 = configuration.rx[1];

      // Reverse translate formatter encoding
      if(link0 != DATA_LINK_OFF) 
        link0 = (link0/16) * 12 + link0%16;
      if(link1 != DATA_LINK_OFF) 
        link1 = (link1/16) * 12 + link1%16;

      *mrs << "MODULE_LOADED" << MRS_INFORMATION << MRS_QUALIF("SCTAPI") 
           << MRS_PARAM<const char *>("sn", moduleName.c_str())
           << MRS_PARAM<int>("group", configuration.groupId)
           << MRS_PARAM<int>("select", configuration.select)
           << MRS_PARAM<int>("pTTC", configuration.pTTC)
           << MRS_PARAM<int>("rTTC", configuration.rTTC)
           << MRS_PARAM<int>("rx0", link0)
           << MRS_PARAM<int>("rx1", link1)
           << MRS_TEXT("Module configuration loaded")
           << ENDM;
    }

    delete [] mappings;
  } catch(ConfigurationException &c) {
    cout << "Module not initialised: \n";
    cout << c.what() << endl;

    {
      boost::mutex::scoped_lock lock(logMutex);
      log << "  (Not found)\n";
    }

    throw;
  }
}

ABCDModule *SctApi::retrieveModule(UINT32 mid) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "retrieveModule " << mid << endl;
  }
  return lookupConfig(mid);
}

// Create a prim list
void SctApi::createDebugPrimList() {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "createDebugPrimList\n";
  }
  debugPrimList->clear();
}

// Append a primitive to the prim list
void SctApi::addDebugPrimList(unsigned long length, long index, long id, long version,
                              unsigned long * body) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "addDebugPrimList ID: " << id << " length: " << length << endl;
  }
  // Don't pass the body because we don't know where it came from...
  debugPrimList->addPrimitive(RodPrimitive((long)length, index, id, version, (long *)body));
}

// Send primlist
void SctApi::sendDebugPrimList(unsigned int partition, unsigned int crate, unsigned int rod) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "sendDebugPrimList " << partition << " " << crate << " " << rod << endl;
  }
  sendPrimList(partition, crate, rod, debugPrimList);
}

// Send primlist to all rods
void SctApi::sendDebugPrimListAll() {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "sendDebugPrimListAll\n";
  }

  // For each crate
  for(map<pair<unsigned int,unsigned int>, Crate*>::const_iterator i=crateMap.begin();
      i!=crateMap.end();
      i++){
    sendPrimListAll(i->first.first, i->first.second, debugPrimList);
  }
}

void SctApi::sendDebugSlavePrimList(unsigned int partition, unsigned int crate, unsigned int rod,
                                    unsigned int slave, bool await, bool response) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "sendDebugSlavePrimList \n";
  }
  sendSlavePrimList(partition, crate, rod, debugPrimList, slave, await, response);
}

void SctApi::debugPrimListFromFile(string fileName) {
  ifstream fin(fileName.c_str(), ios::binary);

  unsigned long fileSize;

  // Get size of file
  fin.seekg(0, std::ios::end);  // go to end of file
  fileSize = fin.tellg();  // file size is current location
  fin.seekg(0, std::ios::beg);  // go back to beginning of file

  cout << "Loading file size: " << fileSize << endl;

  long *buffer = new long[fileSize];

  fin.read((char *)(buffer), fileSize);

  long listLength = buffer[0];
//   long listIndex = buffer[1];
  long listNumPrims = buffer[2];
//   long listPrimVersion = buffer[3];

  int offset = 4;

  debugPrimList->clear();

  for(int i=0; i<listNumPrims; i++) {
    long primLength = buffer[offset + 0];
    long primIndex = buffer[offset + 1];
    long primId = buffer[offset + 2];
    long primVersion = buffer[offset + 3];

    // Copy data to primitive
    debugPrimList->addPrimitive(primLength - 4, primIndex, primId, primVersion, buffer + offset + 4);

    offset += primLength;
    if(offset > listLength) {
      cout << "offset too big\n";
      return;
    }
  }
}

void SctApi::dumpDebugPrimList() {
  dumpPrimList(debugPrimList);
}

/*
  Initialise a module:
  Load configuration from database
  Send configuration to ROD

  return 0 for success
*/
int SctApi::initialiseModule(string modulename) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "InitialiseModule " << modulename << endl << flush;
  }

  unsigned int partition, crate, rod, channel;

  // First find out where this module is
  try {
    unsigned int MUR, module;
    config->translateFromSN(modulename, MUR, module);
    config->translateToROD(MUR, module, partition, crate, rod, channel);
  } catch(ConfigurationException &c) {
    cout << "Module not initialised: \n";
    cout << c.what() << endl;
    return -1;
  }

  cout << "Initialise module " << modulename << " at " <<
    partition << ", " << crate << ", " << rod << ", " << channel << " \n";

  {
    boost::mutex::scoped_lock lock(logMutex);
    log << " found at " <<
      partition << ", " << crate << ", " << rod << ", " << channel << " \n" << flush;
  }

  try {
    cacheModuleConfig(packpcrc(partition, crate, rod, channel));
  } catch(ConfigurationException &c) {
    cout << "Module not initialised: \n";
    cout << c.what() << endl;
    return -2;
  }

  // Install configuration in ROD
  setABCDModule(packpcrc(partition, crate, rod, channel), SCTAPI_BANK_PHYSICS);
  setABCDModule(packpcrc(partition, crate, rod, channel), SCTAPI_BANK_SCAN);

  return 0;
}

/*
   Initialise a rod from the configuration
 */
int SctApi::initialiseRod(unsigned int partition, unsigned int crate, unsigned int rod) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "InitialiseRod\n";
  }
  Crate *myCrate = getCrate(partition, crate);
  if(!myCrate) {
    cout << "Request for non-existent crate " << partition << " " << crate << endl;
    return -1;
  } else if(!getCrate(partition, crate)->RODPresent(rod)) {
    cout << "ROD can't be initialised further: " << partition << " " << crate << " " << rod << endl;
    return -2;
  } else {
    // Set up and start slaves
    SctConfiguration::RodConfig rodConf;
    try {
      rodConf = config->getRodConfig(partition, crate, rod);
    } catch(ConfigurationException &c) {
      cout << "No configuration for rod " << rod << endl;
      return -3;
    }

    int startedSlaves = 0;
    int configuredSlaves = 0;
    for(int slave=0;  slave < numSlaves; slave++) {
      try {
        string ip;
        string id;
        string ext;

        if(rodConf.slaves[slave].ipramFile != "") {
          ip = rodConf.slaves[slave].ipramFile;
          id = rodConf.slaves[slave].idramFile;
          ext = rodConf.slaves[slave].extFile;
        } else {
          cout << " Trying defaults\n";

          if(getCrate(partition, crate)->getRodRevision(rod) == 0xe) {
            cout << " RevE\n";
            SlaveConfig revE = config->getDefaultSlaveConfig('E');
            ip = revE.ipramFile;
            id = revE.idramFile;
            ext = revE.extFile;
          } else {
            SlaveConfig revC = config->getDefaultSlaveConfig('C');
            ip = revC.ipramFile;
            id = revC.idramFile;
            ext = revC.extFile;
          }
        }

        std::cerr << "Starting slave " << slave << std::endl;
        std::cerr << "(" << ip << ", " << id << ", " << ext << ")\n";

        configuredSlaves ++;

        // These changed for Rev E!
        if(getCrate(partition, crate)->getRodRevision(rod) == 0xe) {
          writeSlaveFile(partition, crate, rod, slave, ip, 0);
          writeSlaveFile(partition, crate, rod, slave, id, 0x10000);
          writeSlaveFile(partition, crate, rod, slave, ext, 0xa0000000);
        } else {
          writeSlaveFile(partition, crate, rod, slave, ip, 0);
          writeSlaveFile(partition, crate, rod, slave, id, 0x80000000);
          writeSlaveFile(partition, crate, rod, slave, ext, 0x2000000);
        }

        startSlave(partition, crate, rod, slave);

        myCrate->slaveStarted(rod, slave);

        startedSlaves ++;

        cout << "Slave " << slave << " on rod " << rod << " configured " << endl;
      } catch(ConfigurationException &c) {
        cout << "No configuration for slave " << slave << ": " << c.what() << endl;

        if(mrs) {
          *mrs << "ROD_SLAVE_INIT" << MRS_ERROR
               << MRS_TEXT((string("ROD slave configuration not found: ") + c.what()).c_str())
               << MRS_PARAM<int>("partition", partition)
               << MRS_PARAM<int>("crate", crate)
               << MRS_PARAM<int>("rod", rod)
               << MRS_PARAM<int>("slave", slave) 
               << ENDM;
        }
      } catch(SctApiException &s) {
        cout << "Slave start failed: " << s.what() << endl;

        if(mrs) {
          *mrs << "ROD_SLAVE_INIT" << MRS_ERROR
               << MRS_TEXT((string("ROD slave initialisation failed: ") + s.what()).c_str())
               << MRS_PARAM<int>("partition", partition)
               << MRS_PARAM<int>("crate", crate)
               << MRS_PARAM<int>("rod", rod)
               << MRS_PARAM<int>("slave", slave) 
               << ENDM;
        }
      }
    }

    if(mrs) {
      *mrs << "ROD_SLAVE_INIT";
      if(startedSlaves != configuredSlaves) {
        *mrs << MRS_ERROR
             << MRS_TEXT("ROD slave initialisation incomplete");
      } else {
        *mrs << MRS_INFORMATION
             << MRS_TEXT("All ROD slaves initialised");
      }
      *mrs << MRS_QUALIF("SCTAPI") 
           << MRS_PARAM<int>("partition", partition)
           << MRS_PARAM<int>("crate", crate)
           << MRS_PARAM<int>("rod", rod)
           << MRS_PARAM<int>("started", startedSlaves)
           << MRS_PARAM<int>("configuration", configuredSlaves)
           << ENDM;
    }

    cout << "Started " << startedSlaves << " slave DSPs\n";

    RodLabel currRod(partition, crate, rod);
    addRodToList(currRod);
  }

  return 0;
}

/*
   Initialise everything from the configuration.
   Uses initialiseRod and initialiseModule to do actual configuration
 */
void SctApi::initialiseAll(int run) {
  runNumber = run;
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "InitialiseAll " << runNumber << "\n";
  }

  cout << "********** SCT API initialisation started *************\n";
  {
    time_t startTime = time(NULL);
    struct tm broke = *(gmtime(&startTime));

    char buffer[25];
    strftime(buffer, 25, "%H:%M:%S %Y/%m/%d", &broke);

    cout << "***************** " << buffer << " ******************\n";
  }

  unsigned int totalPartitions = 0;
  unsigned int totalCrates = 0;
  unsigned int totalRods = 0;
  unsigned int totalModules = 0;

  // Clear up from previous time if necessary
  for(map<pair<unsigned int,unsigned int>, Crate*>::const_iterator i=crateMap.begin();i!=crateMap.end();i++){
    delete i->second;
  }

  crateMap.clear();
  moduleMap.clear();
  debugPrimList->clear();
  rodList.clear();

  // Read in configuration and act accordingly
  list<unsigned int> partitions; 
  try {
    partitions = config->listPartitions();
  } catch(ConfigurationException &c) {
    cout << "No partitions to initialise\n";
    cout << c.what() << endl;
    return;
  }

  cout << "Found configuration for " << partitions.size() << " partitions\n";

  for(list<unsigned int>::const_iterator p=partitions.begin();
      p!=partitions.end(); 
      p++) {

    list<unsigned int> crates; 

    try {
      crates = config->listCratesInPartition(*p);
    } catch(ConfigurationException &c) {
      cout << "No crates in this partition\n";
      cout << c.what() << endl;
      continue;
    }

    cout << "Found configuration for " << crates.size() << " crates\n";

    for(list<unsigned int>::const_iterator c=crates.begin(); 
        c!=crates.end(); 
        c++) {
      try {
        crateMap.insert(make_pair(make_pair(*p, *c), new CrateImpl(*p, *c, config)));

        if(mrs)
          getCrate(*p, *c)->setMrsStream(mrs);

        // Setup VME and say HI to the TIM and all the RODs and BOCs (includes reset of the RODs)
        totalRods += getCrate(*p, *c)->initialiseCrate();

        list<unsigned int> rods;
        try {
          rods = config->listRodsInCrate(*p, *c);
        } catch(ConfigurationException &c) {
          cout << "No RODs in this crate\n";
          cout << c.what() << endl;
          continue;
        }

        for(list<unsigned int>::const_iterator r=rods.begin(); 
            r!=rods.end(); 
            r++) {
          RodLabel label(*p, *c, *r);

          // This also does the slave setup
          if(initialiseRod(*p, *c, *r) != 0) {
            cout << "Initialisation of rod  " << *r << " failed!\n";
            // Don't load modules...
            continue;
          }

          list<unsigned int> MURs;
          try {
            MURs = config->listMURSInRod(*p, *c, *r);
          } catch(ConfigurationException &c) {
            cout << "No MURs for this ROD\n";
            cout << c.what() << endl;
            continue;
          }

          for(list<unsigned int>::const_iterator mur=MURs.begin(); 
              mur!=MURs.end(); 
              mur++) {

            list<string> modules;
            try {
              modules = config->listModulesInMUR(*p, *mur);
            } catch(ConfigurationException &c) {
              cout << "No modules for this MUR\n";
              cout << c.what() << endl;
              continue;
            }

            for(list<string>::const_iterator m=modules.begin(); 
                m!=modules.end(); 
                m++) {
              if(!initialiseModule(*m)) {
                totalModules ++;
              }
            }
          }
          cout << "Sending lowest module link to front panel: " << getRodInfo(label).minLink << endl;
          testLinkOutSelect(*p, *c, *r, getRodInfo(label).minLink);
        }

        getCrate(*p, *c)->status();
        totalCrates++;
      } catch(...) {
        // VME initialisation failed (crate already told MRS)
      }
    }
    totalPartitions ++;
  }

  // Load modules first so we know which channels to unmask
  cout << "Setup for calibration...\n";
  calib_init();
  cout << "... Done\n";

  cout << "Initialisation counts:\n";
  cout << "\ttotalPartitions: " << totalPartitions << endl;
  cout << "\ttotalCrates: " << totalCrates << endl;
  cout << "\ttotalRods: " << totalRods << endl;
  cout << "\ttotalModules: " << totalModules << endl;

  if(mrs) {
    *mrs << "SCTAPIINIT_COMPLETE" << MRS_INFORMATION << MRS_QUALIF("SCTAPI") 
         << MRS_TEXT("SCTAPI Initialisation counts")
         << MRS_PARAM<int>("partitions", totalPartitions)
         << MRS_PARAM<int>("crates", totalCrates)
         << MRS_PARAM<int>("rods", totalRods)
         << MRS_PARAM<int>("modules", totalModules)
         << ENDM;
  }

  cout << "************** SCT API initialised ****************\n";
  {
    time_t endTime = time(NULL);
    struct tm broke = *(gmtime(&endTime));

    char buffer[25];
    strftime(buffer, 25, "%H:%M:%S %Y/%m/%d", &broke);

    cout << "************** " << buffer << " ****************\n";
  }
}

void SctApi::startSlave(unsigned int partition, unsigned int crate, unsigned int rod, unsigned int slave)
{
  cout << "Starting slave on " << slave << endl;

  shared_ptr<PrimListWrapper> primList(new PrimListWrapper(1));

  PrimBuilder::instance().startSlave(primList, slave);

  sendPrimList(partition, crate, rod, primList);
  awaitResponse(partition, crate, rod, 5);

  unsigned long length;
  unsigned long *result = getResponse(partition, crate, rod, length);

  if(result) {
    unsigned int slaveRet = result[8];
    delete [] result;
    if(slaveRet != slave) {
      cout << "Bad return (" << slaveRet << ") from start slave " << slave << " \n";
      throw SctApiException("Bad return value from start slave");
    }
  } else {
    cout << "Bad response from start slave\n";
    throw SctApiException("Missing return primitive from start slave");
  }
}

void SctApi::setRunNumber(UINT32 newRun) {
  if(newRun < runNumber) {
    if(mrs) {
      *mrs << "SCTAPI_RUNNUMBER" << MRS_INFORMATION << MRS_QUALIF("SCTAPI") 
           << MRS_TEXT("SCTAPI runnumber decrease")
           << MRS_PARAM<int>("oldRunNumber", runNumber)
           << MRS_PARAM<int>("newRunNumber", newRun)
           << MRS_PARAM<int>("newScanNumber", 0)
           << ENDM;
    }
    cout << "****  Run number has been decreased!!\n";
    runNumber = newRun;
    scanNumber = 0;
  } else if(newRun == runNumber) {
    if(mrs) {
      *mrs << "SCTAPI_RUNNUMBER" << MRS_INFORMATION << MRS_QUALIF("SCTAPI") 
           << MRS_TEXT("SCTAPI runnumber unchanged")
           << MRS_PARAM<int>("oldRunNumber", runNumber)
           << MRS_PARAM<int>("newRunNumber", newRun)
           << MRS_PARAM<int>("newScanNumber", scanNumber)
           << ENDM;
    }
    cout << "**** Run number is unchanged scan number not reset!!!\n";
  } else {
    if(mrs) {
      *mrs << "SCTAPI_RUNNUMBER" << MRS_INFORMATION << MRS_QUALIF("SCTAPI") 
           << MRS_TEXT("SCTAPI runnumber changed")
           << MRS_PARAM<int>("oldRunNumber", runNumber)
           << MRS_PARAM<int>("newRunNumber", newRun)
           << MRS_PARAM<int>("newScanNumber", 0)
           << ENDM;
    }
    runNumber = newRun;
    scanNumber = 0;
  }
}

void SctApi::setScanNumber(UINT32 newScan) {
  if(mrs) {
    *mrs << "SCTAPI_SCANNUMBER" << MRS_INFORMATION << MRS_QUALIF("SCTAPI");
  }

  if(newScan < scanNumber) {
    cout << "**** Scan number has been decreased!!\n";
    if(mrs) {
      *mrs << MRS_TEXT("SCTAPI scannumber decreased");
    }
  } else {
    if(mrs) {
      *mrs << MRS_TEXT("SCTAPI scannumber changed");
    }
  }

  if(mrs) {
    *mrs << MRS_PARAM<int>("oldScanNumber", scanNumber)
         << MRS_PARAM<int>("newScanNumber", newScan)
         << ENDM;
  }

  scanNumber = newScan;
}

UINT32 SctApi::getRunNumber() {
  return runNumber;
}

UINT32 SctApi::getScanNumber() {
  return scanNumber;
}

std::list<std::string> SctApi::getModuleList()  
{
  std::list<std::string> result;

  // Set up masks for modules
  for(map<UINT32, ABCDModule>::const_iterator mi = moduleMap.begin();
      mi!=moduleMap.end();
      mi ++) {
    UINT32 currMid = mi->first;
    result.push_back(convertToString(currMid));
  }

  return result;
}

bool SctApi::isRODPresent(unsigned int partition, unsigned int crate, unsigned int rod) {
  bool result = false;
  if(getCrate(partition, crate)) {
    if(getCrate(partition, crate)->RODPresent(rod)) {
      result = true;
    }
  }

  return result;
}

void SctApi::startupModules() {
  // Turn power to standby1
  dcsAccess->toStandby1();

  // Check for clk/2 or event (its still running from last time)
  checkAllModulesProbe("2E");

  // Send configs
  sendAllABCDModules(SCTAPI_BANK_PHYSICS);

  // Check for event headers
  checkAllModulesProbe("E");

  // Power to standby2
  dcsAccess->toStandby2();
}

void SctApi::autoConfigure() {
  ::SctApi::AutoConf::AutoConfigurer configurer(*this);
  configurer.run();
}

/*
   Send a PrimList to a ROD

   This function exposes RodPrimList so make it private.
   Therefore to be used internally to this class.
*/
void SctApi::sendPrimList(unsigned int partition, unsigned int crate, unsigned int rod,
                          shared_ptr<PrimListWrapper> prim) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "SendPrimList (" << partition << " " << crate << " " << rod << ")\n" << flush;
  }

  dumpPrimList(prim);

  if(getCrate(partition, crate)) {
    getCrate(partition, crate)->sendPrimList(rod, prim);
  } else {
    cout << "Request for non-existent crate " << partition << " " << crate << endl;
  }
}

/*
   Send a PrimList to all RODs in a crate

   This function exposes RodPrimList so make it private.
   Therefore to be used internally to this class.
*/
void SctApi::sendPrimListAll(unsigned int partition, unsigned int crate,
                             shared_ptr<PrimListWrapper> prim) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "SendPrimListAll (" << partition << " " << crate << ")\n" << flush;
  }

  dumpPrimList(prim);

  if(getCrate(partition, crate)) {
    getCrate(partition, crate)->sendPrimListAll(prim);
  } else {
    cout << "Request for non-existent crate " << partition << " " << crate << endl;
  }
}

int SctApi::synchSendPrimListAllCrates(shared_ptr<PrimListWrapper> primList, int timeout) {
  // Send list
  for(map<pair<unsigned int,unsigned int>, Crate*>::const_iterator i=crateMap.begin();i!=crateMap.end();i++){
    sendPrimListAll(i->first.first, i->first.second, primList);
  }

  int result = 0;

  // Wait for list, with no threads the awaitResponse starts the primitive...
  for(map<pair<unsigned int,unsigned int>, Crate*>::const_iterator i=crateMap.begin();i!=crateMap.end();i++){
    int responseCode = awaitResponseAll(i->first.first, i->first.second, timeout);

    if(responseCode != 0) {
      cout << "Synchronous primitive list failed on crate (" << i->first.first << " " << i->first.second << ") failed!\n";
      cout << " code = " << responseCode << endl;

      result = 1;
    }
  }

  return result;
}

void SctApi::dumpPrimList(shared_ptr<PrimListWrapper> prim) {
  prim->list->bufferBuild();

  long buffLength = prim->list->getBufferLength();
  unsigned long* bufferStart = prim->list->getBuffer();

  if(checkDebugOption(DEBUG_DUMP_PRIM_BINARY))
    printMemoryBlock(log, bufferStart, buffLength, 8);

  if(checkDebugOption(DEBUG_SAVE_PRIM)) {
    static int primCount = 0;

    string dirName = Sct::SctNames::getTempDir() + "/PrimLists";

    // Ignore success (0) or failure
    mkdir(dirName.c_str(), S_IRUSR|S_IWUSR|S_IXUSR);

    char *saveDir = getcwd(NULL, 0);
    chdir(dirName.c_str());

    char fileName[100];
    sprintf(fileName, "PrimLists_%05d.bin", primCount++);
    ofstream fout(fileName, ios::binary | ios::trunc);
    fout.write((char*)&bufferStart[0], buffLength*4);

    chdir(saveDir);

    free(saveDir);
  }

  if(checkDebugOption(DEBUG_PRINT_IN_PRIM)) {
    // Pick between log and cout
    if(checkDebugOption(DEBUG_LOG_PRINT_PRIM)) {
      ::SctApi::printOutList(bufferStart, buffLength, true, 0, log, 
                             checkDebugOption(DEBUG_PRINT_UNKNOWN), checkDebugOption(DEBUG_PRINT_RAW));
    } else {
      ::SctApi::printOutList(bufferStart, buffLength, true, 0, cout, 
                             checkDebugOption(DEBUG_PRINT_UNKNOWN), checkDebugOption(DEBUG_PRINT_RAW));
    }
  }
}

/*
   Shutdown anything that needs shutting down
   Send DCS to standby message
 */
void SctApi::shutdownAll() {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "SctApi told to shutdown\n";
    log << "\t" << second_clock::universal_time() << endl;
  }
  // What does shutdown involve?
  lasersOff(); //   ????

  if(checkDebugOption(DEBUG_EXTRA_DUMPS)) {
    standardRegisterDumpAll();
  }

  if(mrs) {
    *mrs << "SCTAPI_SHUTDOWN" << MRS_INFORMATION << MRS_QUALIF("SCTAPI") << MRS_TEXT("First ROD shut down") << ENDM;
  }

  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "Shutdown sequence complete\n";
  }
}

/*
   Create a RodPrimList to send to the Master DSP which contains the RodPrimList which is passed as an argument
 */
void SctApi::sendSlavePrimList(unsigned int partition, unsigned int crate, unsigned int rod,
                               shared_ptr<PrimListWrapper> prim, unsigned int slaveNumber, bool await, bool response) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "SendSlavePrimlist (" << partition << " " << crate << " " << rod << ")\n" << flush;
  }

  if((!getCrate(partition, crate))
     || (!getCrate(partition, crate)->slavePresent(rod, slaveNumber))) {
    cout << "***** Slave DSP (" << slaveNumber << ") was not initialised, please check the configuration and the Slave Image files\n";
    return;
  }

  shared_ptr<PrimListWrapper> container(new PrimListWrapper(1));

  PrimBuilder::instance().slavePrimList(container, prim, slaveNumber, await, response);

  try {
    container->list->bufferBuild();
  }
  catch (PrimListException &p) {
    cout << p.getDescriptor() << " ";
    cout << p.getData1() << ", " << p.getData2() << "\n";
  }

  sendPrimList(partition, crate, rod, container); 
}

/*
    Lookup the Crate object to send instructions to.
    Return 0 if not found.
 */
Crate *SctApi::getCrate(unsigned int partition, unsigned int crate) const {
  map<pair<unsigned int, unsigned int>, Crate* >::const_iterator found;

  found = crateMap.find(make_pair(partition, crate));
  if(found == crateMap.end()) {
    cerr << "Crate not found in getCrate " << partition << ", " << crate << endl;
    throw SctApiException("Missing crate");
  }
  else return found->second;
}

int SctApi::awaitResponse(unsigned int partition, unsigned int crate, unsigned int rod, int timeout) {
  int retVal = 0;
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "AwaitResponse\n";
  }

  if(getCrate(partition, crate)) {
    try {
      retVal = getCrate(partition, crate)->awaitResponse(rod, timeout);
    } catch(CrateException &c) {
      cout << "Crate exception in await response\n" << c.what() << endl;
      retVal = -1;
    } catch(BaseException &b) {
      cout << "Base exception in await response\n" << b << endl;
      retVal = -1;
    } catch(...) {
      cout << "Exception in await response\n";
      retVal = -1;
    }

    if(retVal != 0) {
      if(config->isDummyCrate(partition, crate)) {
        //We can't wait if there is nothing there so don't signal an error
        //We only check here to prevent doing potentially a lot of unnecessary IPC calls...
        retVal = 0;
      } else {
        cout << "Problem during await response\n";
        cout << "Crate status " << partition << " " << 
          crate << endl;
        getCrate(partition, crate)->status();
      }
    }
  } else {
    cout << "Request for non-existent crate " << partition << " " << crate << endl;
  }

  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "\tAwaitResponse: " << retVal << endl;
  }

  return retVal;
}

int SctApi::awaitResponseAll(unsigned int partition, unsigned int crate, int timeout) {
  int retVal = 0;
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "AwaitResponse\n";
  }

  if(getCrate(partition, crate)) {
    try {
      retVal = getCrate(partition, crate)->awaitResponseAll(timeout);
    } catch(CrateException &c) {
      cout << "Crate exception in await response all\n" << c.what() << endl;
      retVal = -1;
    } catch(BaseException &b) {
      cout << "Base exception in await response all\n" << b << endl;
      retVal = -1;
    } catch(...) {
      cout << "Exception in await response all\n";
      retVal = -1;
    }

    if(retVal != 0) {
      cout << "Problem during await response all\n";
    }
  } else {
    cout << "Request for non-existent crate " << partition << " " << crate << endl;
  }

  return retVal;
}

unsigned long *SctApi::getResponse(unsigned int partition, unsigned int crate, unsigned int rod, 
                                   unsigned long &length) {
  unsigned long *result;

  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "GetResponse\n";
  }
  if(getCrate(partition, crate)) {
    boost::shared_ptr<RodOutList> output = getCrate(partition, crate)->getResponse(rod);

    if(!output) {
      cout << "No response from crate\n";
      return NULL;
    }

    length = (unsigned long) output->getLength();

    if(checkDebugOption(DEBUG_DIAG_RESPONSE))
      cout << "Got response from crate length = " << length << endl;

    unsigned long *body = output->getBody();

    result = new unsigned long[length];

    for(unsigned int i=0; i<length; i++) {
      result[i] = body[i];
    }

    if(body[0] != length) {
      cout << "Bad outlist/outlength mismatch in returned outlist (" << partition << " " << crate << " " << rod << "): " << body[0] << " != " << length << endl;
      return NULL;
    }

    if(checkDebugOption(DEBUG_SAVE_PRIM)) {
      static int primCount = 0;

      string dirName = Sct::SctNames::getTempDir() + "/PrimLists";

      // Ignore success (0) or failure
      mkdir(dirName.c_str(), S_IRUSR|S_IWUSR|S_IXUSR);

      char *saveDir = getcwd(NULL, 0);
      chdir(dirName.c_str());

      char fileName[100];
      sprintf(fileName, "OutLists_%05d.bin", primCount++);
      ofstream fout(fileName, ios::binary | ios::trunc);
      fout.write((char*)result, length*4);

      chdir(saveDir);

      free(saveDir);
    }

    if(checkDebugOption(DEBUG_PRINT_OUT_PRIM)) {
    // Pick between log and cout
      if(checkDebugOption(DEBUG_LOG_PRINT_PRIM)) {
        ::SctApi::printOutList(result, length, false, 0, log, 
                               checkDebugOption(DEBUG_PRINT_UNKNOWN), checkDebugOption(DEBUG_PRINT_RAW));
      } else {
        ::SctApi::printOutList(result, length, false, 0, cout, 
                               checkDebugOption(DEBUG_PRINT_UNKNOWN), checkDebugOption(DEBUG_PRINT_RAW));
      }
    }
  } else {
    cout << "Request for non-existent crate " << partition << " " << crate << endl;
    return NULL;
  }

  return result;
}

/* Get a message from a ROD text buffer
   Returns whether the action was completed
*/
bool SctApi::getRodMessage(unsigned int partition, unsigned int crate, unsigned int rod,
                           char *buffer, unsigned long &length) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "Get message\n";
  }
  if(getCrate(partition, crate)) {
    return getCrate(partition, crate)->getRodMessage(rod, buffer, length);
  } else {
    buffer[0] = 0;
    length = 0;
    return false;
  }
}

// Get ID number of module
UINT32 SctApi::findModule(string sn) { 
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "findModule " << sn << endl;
  }

  unsigned int partition, crate, rod, channel;

  try {
    unsigned int mur, module;
    config->translateFromSN(sn, mur, module);
    config->translateToROD(mur, module, partition, crate, rod, channel);
  } catch(ConfigurationException &c) {
    cout << "Module not found\n";
    return BAD_MODULE;
  }

  return packpcrc(partition, crate, rod, channel);
}

string SctApi::convertToString(UINT32 mid) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "convertToString " << mid << endl;
  }

  unsigned int partition, crate, rod, channel;
  getpcrc(mid, partition, crate, rod, channel);

  string sn;

  try {
    unsigned int mur, module;
    config->translateFromROD(partition, crate, rod, channel, mur, module);
    config->translateToSN(mur, module, sn);
  } catch(ConfigurationException &c) {
    cout << "Module not found\n";
    return "BAD_MODULE";
  }

  return sn;
}

UINT32 SctApi::findModule(INT32 mur, INT32 module) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "findModule " << mur << " " << module << endl;
  }

  unsigned int partition, crate, rod, channel;

  try {
    config->translateToROD(mur, module,
                           partition, crate, rod, channel);
  } catch(ConfigurationException &c) {
    cout << "Module not found\n";
    return BAD_MODULE;
  }

  return packpcrc(partition, crate, rod, channel);
}

pair<INT32, INT32> SctApi::convertToMUR(UINT32 mid) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "convertToMUR " << mid << endl;
  }

  unsigned int partition, crate, rod, channel;
  getpcrc(mid, partition, crate, rod, channel);

  UINT32 mur;
  UINT32 module;

  try {
    config->translateFromROD(partition, crate, rod, channel, mur, module);
  } catch(ConfigurationException &c) {
    cout << "Module not found\n";
    return make_pair(BAD_MODULE, 0);
  }

  return make_pair(mur, module);
}

UINT32 SctApi::findBarrelModule(INT32 barrel, INT32 row, INT32 number) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "findBarrelModule " << barrel << " " << row << " " << number << endl;
  }

  unsigned int partition, crate, rod, channel;

  try {
    unsigned int mur, module;
    config->translateFromBarrel(barrel, row, number, mur, module);
    config->translateToROD(mur, module,
                           partition, crate, rod, channel);
  } catch(ConfigurationException &c) {
    cout << "Module not found\n";
    return BAD_MODULE;
  }

  return packpcrc(partition, crate, rod, channel); 
}

void SctApi::convertToBarrelModule(UINT32 mid, UINT32 &barrel, UINT32 &row, int &number) { 
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "convertToBarrelModule " << mid << endl;
  }

  unsigned int partition, crate, rod, channel;
  getpcrc(mid, partition, crate, rod, channel);

  try {
    unsigned int mur, module;
    config->translateFromROD(partition, crate, rod, channel, mur, module);
    config->translateToBarrel(mur, module, barrel, row, number);
  } catch(ConfigurationException &c) {
    cout << "Module not found\n";

    barrel = BAD_MODULE;
    return;
  }
}

UINT32 SctApi::findEndcapModule(INT32 disk, INT32 ring, INT32 number) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "findEndcapModule " << disk << " " << ring << " " << number << endl;
  }
  unsigned int partition, crate, rod, channel;

  try {
    unsigned int mur, module;
    config->translateFromEndcap(disk, ring, number, mur, module);
    config->translateToROD(mur, module,
                           partition, crate, rod, channel);
  } catch(ConfigurationException &c) {
    cout << "Module not found\n";
    return BAD_MODULE;
  }

  return packpcrc(partition, crate, rod, channel);
}

void SctApi::convertToEndcapModule(UINT32 mid, INT32 &disk, UINT32 &ring, UINT32 &number) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "convertToEndcapModule " << mid << endl;
  }

  unsigned int partition, crate, rod, channel;
  getpcrc(mid, partition, crate, rod, channel);

  try {
    unsigned int mur, module;
    config->translateFromROD(partition, crate, rod, channel, mur, module);
    config->translateToEndcap(mur, module, disk, ring, number);
  } catch(ConfigurationException &c) {
    cout << "Module not found\n";
    disk = BAD_MODULE;
    return;
  }
}

#warning "These should be due to notifications from the configuration???"

void SctApi::loadConfiguration() {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "loadConfiguration (default)\n";
  }

  cout << "Load configuration\n";
#warning "Using feature of ConfigurationXmlImpl"
  loadConfiguration("");
}

void SctApi::loadConfiguration(string filename) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "loadConfiguration (" << filename << ")\n";
  }

  try {
    cout << "Load configuration\n";
    config->loadConfiguration(filename); 
  } catch(ConfigurationException &c) {
    cout << "Problem loading new configuration, check configuration service\n";
  }

  cout << "Remove module configuration cache\n";
  moduleMap.clear();

  loadModuleConfigurations();
}

int SctApi::loadModuleConfigurations() {
  int moduleConfigurations = 0;

  // For each ROD
  for(list<RodLabel>::const_iterator rl = rodList.begin();
      rl!=rodList.end();
      rl++){
    list<unsigned int> MURs;
    try {
      MURs = config->listMURSInRod(rl->partition, rl->crate, rl->rod);
    } catch(ConfigurationException &c) {
      cout << "No MURs for this ROD\n";
      cout << c.what() << endl;
      continue;
    }

    for(list<unsigned int>::const_iterator mur=MURs.begin(); 
        mur!=MURs.end(); 
        mur++) {

      list<string> modules;
      try {
        modules = config->listModulesInMUR(rl->partition, *mur);
      } catch(ConfigurationException &c) {
        cout << "No modules for this MUR\n";
        cout << c.what() << endl;
        continue;
      }

      for(list<string>::const_iterator m=modules.begin(); 
          m!=modules.end(); 
          m++) {
        if(!initialiseModule(*m)) {
          moduleConfigurations ++;
        }
      }
    }
  }

  cout << "Setup for calibration...\n";
  calib_init();

  return moduleConfigurations;
}

void SctApi::storeModuleConfigurations() {
  // Current cache is now the PHYSICS bank
  setABCDModules(SCTAPI_BANK_PHYSICS);

  for(std::map<UINT32, ABCDModule>::const_iterator i = moduleMap.begin(); 
      i != moduleMap.end(); 
      i++) {
    const ABCDModule &moduleConf = i->second;
    std::string sn = convertToString(i->first);
    config->configureModuleFromStructure(sn, moduleConf);
  }
}

void SctApi::configureBOC(unsigned int partition, unsigned int crate, unsigned int rod) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "ConfigureBOC called\n";
    log << "\t" << second_clock::universal_time() << endl;
  }

  try {
    if(getCrate(partition, crate))
      getCrate(partition, crate)->configureBOC(rod);
  } catch(CrateException &c) {
    throw SctApiException(c);
  }
}

std::vector<double> SctApi::getBOCMonitorArray(unsigned int partition, unsigned int crate, unsigned int rod) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "getBOCMonitorArray called\n";
  }

  if(getCrate(partition, crate)) 
    return getCrate(partition, crate)->getBOCMonitorArray(rod);
  else
    throw SctApiException("Failed to getBOCMonitorArray");
}

void SctApi::status() {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "print all status\n";
  }

  cout << "SCTAPI status:\n";
  cout << "Run " << runNumber << " Scan " << scanNumber << endl;

#if USE_THREADS
  cout << "Primitive thread engine enabled\n";
#else
  cout << "Primitive thread engine disabed (dangerous in IPC)\n";
#endif
#if USE_SCAN_THREAD
  cout << "Scan polling thread enabled\n";
#else
  cout << "Scan polling thread disabled (doScan can timeout with IPC)\n";
#endif
  if(mrs) {
    *mrs << "SCTAPI_STATUS" << MRS_INFORMATION << MRS_QUALIF("SCTAPI") 
         << MRS_PARAM<int>("run", runNumber) << MRS_PARAM<int>("scan", scanNumber) 
         << MRS_TEXT("Run numbers") << ENDM;
  }

  if(crateMap.size() == 0) {
    if(mrs) {
      *mrs << "SCTAPI_STATUS" << MRS_INFORMATION << MRS_QUALIF("SCTAPI") 
           << MRS_TEXT("No crates loaded") << ENDM;
    }
  }

  for(map<pair<unsigned int, unsigned int>, Crate* >::const_iterator crate = crateMap.begin();
      crate != crateMap.end();
      crate++) {

    cout << "Crate " << crate->first.first << " " << 
      crate->first.second << endl;
    crate->second->status();
  }

  cout << "Module configuration for " << moduleMap.size() << " modules:\n";
  for(map<UINT32, ABCDModule>::const_iterator mi = moduleMap.begin();
      mi!=moduleMap.end();
      mi ++) {
    UINT32 mid = mi->first;

    string sn = convertToString(mid);
    int group = lookupConfig(mid)->groupId;
    int select = lookupConfig(mid)->select;
    int prim = lookupConfig(mid)->pTTC;
    int redun = lookupConfig(mid)->rTTC;
    int link0 = lookupConfig(mid)->rx[0];
    int link1 = lookupConfig(mid)->rx[1];

    // Reverse translate formatter encoding
    if(link0 != DATA_LINK_OFF) 
      link0 = (link0/16) * 12 + link0%16;
    if(link1 != DATA_LINK_OFF) 
      link1 = (link1/16) * 12 + link1%16;

    cout.width(8);
    cout << mid << " " << sn << " Group: " << group << " Select: " << select;
    cout << "\t Pr: ";
    cout.width(2);
    cout << prim << " Red: ";
    cout.width(3);
    cout << redun << " Link 0/1: ";
    cout.width(3);
    cout << link0 << " ";
    cout.width(3);
    cout << link1 << endl;

    if(mrs) {
      *mrs << "MODULE_INFO" << MRS_INFORMATION << MRS_QUALIF("SCTAPI") 
           << MRS_PARAM<const char *>("sn",sn.c_str()) << MRS_PARAM<int>("internalId",mid) 
           << MRS_PARAM<int>("group",group) << MRS_PARAM<int>("select",select)  
           << MRS_PARAM<int>("primary",prim) << MRS_PARAM<int>("redundancy",redun) 
           << MRS_PARAM<int>("link0rx",link0) << MRS_PARAM<int>("link1rx",link1) 
           << MRS_TEXT("Module status ")
           << ENDM;
    }
  }

  if(mrs) {
    *mrs << "SCTAPI_STATUS" << MRS_INFORMATION << MRS_QUALIF("SCTAPI") 
         << MRS_PARAM<int>("run", runNumber) << MRS_PARAM<int>("scan", scanNumber) 
         << MRS_TEXT("Run numbers") << ENDM;
  }
}

// ****  Histogramming  ****
void SctApi::defaultScan(int type) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "Send default scan number " << type << endl;
  }
  ScanDefImpl scan;

  long *evSetup;
  long *histSetup;
  long *histStart;

  shared_ptr<PrimListWrapper> primList(new PrimListWrapper(1));

  int histStartLength;

  switch(type) {
  default:
  case 0:
    {
      // Event Trap Setup
      long setup[24] = {0x1,   0, 0x2000, 0, 
                        0x3, 0x1,      0, 0,
                          0,   0,      0, 0,
                        0x3,   0,      0, 1,               // Config, exclusion
                        0x1,   0,      0, 1,               // function, match
                          1,   0,      0, 0};              // modulus, remainder

      evSetup = new long[24];
      for(int i=0; i<24; i++) evSetup[i] = setup[i];

      // Setup histogram
      long setupH[] = {
        0x20d2000, 0, 0x60, 0x01010000, 0x20200101, 1, 0, 1, 
        0,         0,    0, 0x03f00000, 0x30f01000};

      histSetup = new long[13];
      for(int i=0; i<13; i++) histSetup[i] = setupH[i];

      // Start histogram
      long startHist[4 + 63] = {
        16, 107, 1, 1,

        // Task params
        // 
        0x01010001, 0x000f0001, 0x08040201,        0xf, 0x020d2000, 0x00200101, 0x0100000f,       1000, 
                50,        0x0, 0x40a00000, 0x40a00000, 0x41200000, 0x40a00000, 0x42c80000, 0x40200000, 
        0x43160000, 0x40a00000, 0x43480000, 0x40a00000, 0x40a00000, 0x40a00000, 0x41200000, 0x40a00000,

        0x42c80000, 0x40200000, 0x43160000, 0x40a00000, 0x43480000, 0x40a00000,        0x0,        0x0,
        0x0070006a, 0x00640065, 0x00640064, 0x00000000, 0x81,              0x0,        0x0,        0x0,
        0x0,      // End of trigger sequence (1)
        0x00640064, 0x00640064, 0x00640064, 0, 0, 0, 0, 
        0, 0,      // End of trigger sequence (2)
        0,         // incCmd (2 bytes) etc 
        0, 0,      // incData
        0, 0, 0,
        0, 0, 0, 0, 0,      // End of control data (rest is up to max task structure size)
      };

      histStart = new long[4+63];
      for(int i=0; i<4+63; i++) histStart[i] = startHist[i];

      histStartLength = 67;
    }
    break;
  case 1:
    // A short nmask scan on module 0
    {
      // Event Trap Setup
      long setup[24] = {
        0x00000001, 0x00000000, 0x00002000, 0x00000000, 0x00000003, 0x00000000, 0x00000001, 0x00000000,
        0x00000000, 0x00000000, 0x00000001, 0x00000002, 0x00000003, 0x00000000, 0x00000000, 0x00000001, 
        0x00000001, 0x00000000, 0x00000000, 0x00000000, 0x00000001, 0x00000000, 0x00000000, 0x00000000};

      evSetup = new long[24];
      for(int i=0; i<24; i++) evSetup[i] = setup[i];

      // Setup histogram
      long setupH[] = {
        0x020d2000, 0x00000000, 0x00000008, 0x18090000, 0x00200101, 0x00000001, 0x00000000, 0x00000001, 
        0x00000000, 0x00000000, 0x00000000, 0x03f00000, 0x03f01000};

      histSetup = new long[13];
      for(int i=0; i<13; i++) histSetup[i] = setupH[i];

      // Start histogram
      long startHist[79] = {
        0x18090001, 0x000f0001, 
        0x000000ff, 0x0000f00f, 0x020d2000, 0x00200101, 0x0000000f, 0x0000040a, 0x00000008, 0x00000000, 
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0xffffffff, 0x0070006a, 0x00640065, 
        0x00640064, 0x00000000, 0x00000082, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00640064, 
        0x00640064, 0x00640064, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 
        0x00000000, 0x00000000, 0x00000000, 0x0000000f, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x41800000, 0x42000000, 
        0x42400000, 0x42800000, 0x42a00000, 0x42c00000, 0x42e00000, 0x00000000, 0x41800000, 0x42000000, 
        0x42400000, 0x42800000, 0x42a00000, 0x42c00000, 0x42e00000 
      };

      histStart = new long[79];
      for(int i=0; i<79; i++) histStart[i] = startHist[i];

      histStartLength = 79;


      /* 000000ff 00000001 00000006 0000006f 0000001c 00000000 00000003 00000067 

         00000001 00000000 00002000 00000000 00000003 00000000 00000001 00000000 
         00000000 00000000 00000001 00000002 00000003 00000000 00000000 00000001 
         00000001 00000000 00000000 00000000 00000001 00000000 00000000 00000000 

         00000012 00000001 00002007 00000067 00000000 0000000a ffffffff ffffffff 
         0000000a 00000001 00000001 0000006f 00000004 00000000 00001000 00000065 
         0000000a 0000100e 00000007 00000002 00002008 00000067 00000000 00000001 
         00000001 00000066 00000001 00002007 00000067 00000000 0000005e ffffffff 
         ffffffff 0000005e 00000004 00000002 0000006f 00000011 00000002 00001002 
         00000068 020d2000 00000000 00000008 18090000 00200101 00000001 00000000 
         00000001 00000000 00000000 00000000 03f00000 03f01000 

         00000047 00000003 
         0000000c 00000065 00000020 00000064 00000001 00000001 00000000 00000000 
         00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000 
         00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000 
         00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000 
         00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000 
         00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000 
         00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000 
         00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000 
         00000000 00000000 00000000 00000000 00000000 0000005e 1a242170 00000007 
         00000002 00002008 00000067 00000000 00000001 00000001 00000057 00000000 
         0000000c 00000065 00000010 0000006b 00000001 00000001 18090001 000f0001 
         000000ff 0000f00f 020d2000 00200101 0000000f 0000040a 00000008 00000000 
         00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000 
         00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000 
         00000000 00000000 00000000 00000000 ffffffff ffffffff 0070006a 00640065 
         00640064 00000000 00000082 00000000 00000000 00000000 00000000 00640064 
         00640064 00640064 00000000 00000000 00000000 00000000 00000000 00000000 
         00000000 00000000 00000000 0000000f 00000000 00000000 00000000 00000000 
         00000000 00000000 00000000 00000000 00000000 00000000 41800000 42000000 
         42400000 42800000 42a00000 42c00000 42e00000 00000000 41800000 42000000 
         42400000 42800000 42a00000 42c00000 42e00000 000000ff 1a3ff509 
      */
      break;
    }
  }

  if(!getCrate(0, 0) || !getCrate(0, 0)->RODPresent(0)) {
    cout << "No configured ROD ! Aborting default scan\n";
    if(mrs) {
      *mrs << "ROD_NONEXIST" << MRS_ERROR << MRS_QUALIF("SCTAPI") 
           << MRS_PARAM<int>("crate", 0) 
           << MRS_PARAM<int>("rod", 0)
           << MRS_TEXT("Aborting default scan (ROD not configured)") << ENDM;
    }

    return;
  }

  if(!getCrate(0, 0)->checkBOCLasersOn(0)) {
    cout << "Trying to send default scan using BOC that has its lasers cut out (continuing)\n";
    if(mrs) {
      *mrs << "BOC_INTERLOCKED" << MRS_WARNING << MRS_QUALIF("SCTAPI") 
           << MRS_PARAM<int>("crate", 0) 
           << MRS_PARAM<int>("rod", 0)
           << MRS_TEXT("Default scan continuing even though BOC interlocked") << ENDM;
    }
  }

  primList->addPrimitive(RodPrimitive(4+24, 1, 3, 103, evSetup),
                         evSetup);

  // Start event trap
  long trap[0] = {};
  shared_ptr<PrimListWrapper> sPrimList(new PrimListWrapper(2));
  sPrimList->addPrimitive(RodPrimitive(4, 0, 4096, 101, trap),
                          trap);

  PrimBuilder &builder = PrimBuilder::instance();

  builder.slavePrimList(primList, sPrimList, 0, true, true);

  // Can clear slave list because its fully built!
  sPrimList->clear();
  sPrimList->addPrimitive(RodPrimitive(4 + 13 /* + 2*3 */ , 3, 4098, 104, histSetup),
                          histSetup);
  builder.slavePrimList(primList, sPrimList, 0, true, true);
  
  long task[] = {32, 100, 1, 1, 0, 0};
  sPrimList->clear();
  sPrimList->addPrimitive(RodPrimitive(4 + 6, 4, 12, 101, task),
                          task);
  builder.slavePrimList(primList, sPrimList, 0, true, true);

  primList->addPrimitive(RodPrimitive(histStartLength + 4, 3, 12, 101, histStart),
                         histStart);

  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "Send default scan list\n";
  }
  sendPrimList(0, 0, 0, primList);
  awaitResponse(0, 0, 0, 2);


  // Get result
  unsigned long length;

  unsigned long *result = getResponse(0, 0, 0, length);

  if(result) {
    cout << "Got response from default scan setup\n";
    // Pick between log and cout
    if(checkDebugOption(DEBUG_LOG_PRINT_PRIM)) {
      ::SctApi::printOutList(result, length, true, 0, log, 
                   checkDebugOption(DEBUG_PRINT_UNKNOWN), checkDebugOption(DEBUG_PRINT_RAW));
    } else {
      ::SctApi::printOutList(result, length, true, 0, cout, 
                   checkDebugOption(DEBUG_PRINT_UNKNOWN), checkDebugOption(DEBUG_PRINT_RAW));
    }

    delete [] result;
  } else {
    cout << "No response from default scan setup\n";
  }

  // Poll histogram
//    pollHistogramming(extra, scan.getScanPoints1().size(), scan.getNTrigs(), 5);

//    // Read histogram
//  #ifdef USE_IS
//    readHistogramsToIS(scan, extra);
//  #endif
//    readHistogramsToFile(scan, extra);
}

void SctApi::tidyHistogramming() {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "Brutal histogram stopping\n";
  }

  shared_ptr<PrimListWrapper> primList(new PrimListWrapper(2));

  PrimBuilder &builder = PrimBuilder::instance();

  builder.taskOp(primList, HISTOGRAM_CTRL_TASK, TASK_STOP, 0);

  for(list<RodLabel>::const_iterator rl = rodList.begin();
      rl!=rodList.end();
      rl++){

    unsigned int partition = rl->partition, crate = rl->crate, rod = rl->rod;

    {
      boost::mutex::scoped_lock lock(logMutex);
      log << "Tidy histogram on (" << partition << ", " << crate << ", " << rod << ")\n";
      log << "Stop HCT\n";
    }
    sendPrimList(partition, crate, rod, primList);

    awaitResponse(partition, crate, rod, 2);
    primList->clear();

    cout << "Done kill control task\n";

    for(int slaveNumber=0; slaveNumber<4; slaveNumber++) {
      shared_ptr<PrimListWrapper> slList(new PrimListWrapper(1));

      // Stop Slave histogramming task

      builder.taskOp(slList, HISTOGRAM_TASK, TASK_STOP, 0);

      builder.slavePrimList(primList, slList, slaveNumber, true, true);
      {
        boost::mutex::scoped_lock lock(logMutex);
        log << "Stop histo task (" << slaveNumber << ")\n";
      }
      sendPrimList(partition, crate, rod, primList);
      awaitResponse(partition, crate, rod, 2);
      slList->clear();
      primList->clear();

      cout << "Done kill histogram task\n";

      builder.stopEvTrap(slList);

      builder.slavePrimList(primList, slList, slaveNumber, true, true);
      {
        boost::mutex::scoped_lock lock(logMutex);
        log << "Stop ETS (" << slaveNumber << ")\n";
      }
      sendPrimList(partition, crate, rod, primList);

      awaitResponse(partition, crate, rod, 2);
      primList->clear();

      cout << "Done kill event trapping\n";
    }
//     sendPrimList(partition, crate, rod, primList);
//    awaitResponse(partition, crate, rod, 2);
  }
}

void SctApi::rodMode(unsigned int partition, unsigned int crate, unsigned int rod,
                     int mode, int flag, int fifoSetup, int nBits, int delay, int message) {
  shared_ptr<PrimListWrapper> rodModeList(new PrimListWrapper(2));

  PrimBuilder::instance().rodMode(rodModeList, mode, flag, fifoSetup, nBits, delay, message);

  sendPrimList(partition, crate, rod, rodModeList);

  int responseCode = awaitResponse(partition, crate, rod, 1);

  if(responseCode != 0) {
    cout << "Set ROD mode failed!\n";
  }
}

void SctApi::setupModuleMask(unsigned int partition, unsigned int crate, unsigned int rod,
                             int port, int slvs) {
  shared_ptr<PrimListWrapper> maskList(new PrimListWrapper(1));

  setupModuleMask(port, slvs, maskList);

  sendPrimList(0, 0, 0, maskList);
  int responseCode = awaitResponse(0, 0, 0, 10);

  if(responseCode != 0) {
    cout << "Module mask setup unsuccessful\n";
  }
}

void SctApi::setupModuleMask(int port, int slvs, boost::shared_ptr<PrimListWrapper> primList) { 
  PrimBuilder::instance().masksFromConfig(primList, port);
  PrimBuilder::instance().masksToSlaves(primList, slvs);
}

void SctApi::calib_init() {
#if (R_MODULE_MASK < 101)
  #error "Unsupported old MODULE_MASK version"
#else 
  shared_ptr<PrimListWrapper> calibList(new PrimListWrapper(1));

  setupModuleMask(SP0, 0xf, calibList);

  PrimBuilder &builder = PrimBuilder::instance();
  builder.writeRegister(calibList, RRIF_CMND_0, 0, 8, 0xff);    // Reset everything
  builder.writeRegister(calibList, RRIF_CMND_0, 0, 8, 0x78);

  if(synchSendPrimListAllCrates(calibList) != 0) {
    cout << "Calib list moduleMask failed!\n";
  }

  shared_ptr<PrimListWrapper> rodModeList(new PrimListWrapper(1));
  PrimBuilder::instance().rodMode(rodModeList, CALIBRATION_MODE + CALIBRATION_SLINK_OVERRIDE_MODE, 
                                  0, 1, 1, 1, 1);

  if(synchSendPrimListAllCrates(rodModeList) != 0) {
    cout << "Calib list Rod mode on failed!\n";
  }
#endif  /// new style calib_init
}

void SctApi::print_calib(unsigned int partition, unsigned int crate, unsigned int rod) {
#ifndef FMT_LINK_EN
  cout << "Register ids not known\n";
#else
  cout << "Formatters\n";

  cout << hex;

  cout << " Enables\n";

  for(int a=0; a<8; a++) {
    int data = readRODRegister(partition, crate, rod, FMT_LINK_EN(a)) & 0xffff;
    cout.width(4);
    cout << data << " ";
  }
  cout << endl;

  cout << " Expanded\n";
  for(int a=0; a<8; a++) {
    int data = readRODRegister(partition, crate, rod, FMT_EXP_MODE_EN(a)) & 0xffff;
    cout.width(4);
    cout << hex;
    cout << data << " ";
  }
  cout << endl;

//   cout << " Config mode\n";
//   for(int a=0; a<8; a++) {
//     int data = readRegister(a+0x10) & 0xffff;
//     cout.width(4);
//     cout << hex;
//     cout << data << " ";
//   }
//   cout << endl;

//   cout << " Edge mode\n";
//   for(int a=0; a<8; a++) {
//     int data = readRegister(a+0x18) & 0xffff;
//     cout.width(4);
//     cout << hex;
//     cout << data << " ";
//   }
//   cout << endl;

//   cout << " Time out\n";
//   for(int a=0; a<8; a++) {
//     int data = readRegister(a+0x20, false) & 0xff;
//     cout.width(4);
//     cout << hex;
//     cout << data << " ";
//   }
//   cout << endl;

//   cout << " Overflow limit\n";
//   for(int a=0; a<8; a++) {
//     int data = readRegister(a+0x28, false) & 0x7ff;
//     cout.width(4);
//     cout << hex;
//     cout << data << " ";
//   }
//   cout << endl;

//   cout << " H/T limit\n";
//   for(int a=0; a<8; a++) {
//     int data = readRegister(a+0x30, false) & 0x3ff;
//     cout.width(4);
//     cout << hex;
//     cout << data << " ";
//   }
//   cout << endl;

//   cout << " ROD busy limit\n";
//   for(int a=0; a<8; a++) {
//     int data = readRegister(a+0x38, false) & 0x3ff;
//     cout.width(4);
//     cout << hex;
//     cout << data << " ";
//   }
//   cout << endl;

//   cout << " Diagnostic link to LEMO\n";
//   for(int a=0; a<8; a++) {
//     int data = readRegister(a+0x70, false) & 0xf;
//     cout.width(4);
//     cout << hex;
//     cout << data << " ";
//   }
//   cout << endl;

//   cout << " Diagnostic mode bits read enable\n";
//   for(int a=0; a<8; a++) {
//     int data = readRegister(a+0x78, false) & 0x1;
//     cout.width(4);
//     cout << hex;
//     cout << data << " ";
//   }
//   cout << endl;

  cout << "Link oocupancies\n";
  for(int fmt=0; fmt<8; fmt++) {
    for(int link=0; link<12; link++) {
      int val = readRODRegister(partition, crate, rod, FMT_LINK_OCC_CNT(fmt, link)) & 0x7ff;
      cout << " ";
      Utility::printHex(val, 3);
    }
    cout << endl;
  }

  cout << " Timeout error\n";
  for(int a=0; a<8; a++) {
    int data = readRODRegister(partition, crate, rod, FMT_TIMEOUT_ERR(a)) & 0xfff;
    cout.width(4);
    cout << hex;
    cout << data << " ";
  }
  cout << endl;

  cout << " Overflow error\n";
  for(int a=0; a<8; a++) {
    int data = readRODRegister(partition, crate, rod, FMT_DATA_OVERFLOW_ERR(a)) & 0xfff;
    cout.width(4);
    cout << hex;
    cout << data << " ";
  }
  cout << endl;

  cout << " H/T error\n";
  for(int a=0; a<8; a++) {
    int data = readRODRegister(partition, crate, rod, FMT_HEADER_TRAILER_ERR(a)) & 0xfff;
    cout.width(4);
    cout << hex;
    cout << data << " ";
  }
  cout << endl;

  cout << " ROD busy error\n";
  for(int a=0; a<8; a++) {
    int data = readRODRegister(partition, crate, rod, FMT_ROD_BUSY_ERR(a)) & 0xfff;
    cout.width(4);
    cout << hex;
    cout << data << " ";
  }
  cout << endl;

//   cout << " Parsing mode (1 = Raw, 0 = normal) \n";
//   for(int a=0; a<8; a++) {
//     int data = readRegister(a+0x100, false) & 0xfff;
//     cout.width(4);
//     cout << hex;
//     cout << data << " ";
//   }
//   cout << endl;

//   cout << " Formatter status\n";
//   for(int a=0; a<8; a++) {
//     int data = readRegister(a+0x108, false) & 0xffff;
//     tapi.printHex(data, 4);
//     cout << " ";
// //     cout.width(4);
// //     cout << hex;
// //     cout << data << " ";
//   }
//   cout << endl;

//   cout << " Formatter version\n";
//   for(int a=0; a<8; a++) {
//     int data = readRegister(a+0x110, false) & 0xffff;
//     cout.width(4);
//     cout << hex;
//     cout << data << " ";
//   }
//   cout << endl;

//   cout << "EFB version\n";
//   readRegister(0x1b2);

//   cout << "RTR version\n";
//   readRegister(0x205);

  cout << "RRIF version\n";
  cout << readRODRegister(partition, crate, rod, RRIF_CODE_VERSION) << endl;
  cout << "RRIF status 0\n";
  cout << readRODRegister(partition, crate, rod, RRIF_STATUS_0) << endl;
  cout << "RRIF status 1\n";
  cout << readRODRegister(partition, crate, rod, RRIF_STATUS_1) << endl;

  cout << "RRIF command 0\n";
  cout << readRODRegister(partition, crate, rod, RRIF_CMND_0) << endl;
  cout << "RRIF command 1\n";
  cout << readRODRegister(partition, crate, rod, RRIF_CMND_1) << endl;

  cout << "Output masks\n";
  cout << " ";
  Utility::printHex(readRODRegister(partition, crate, rod, FE_CMND_MASK_0_LO), 8);
  cout << endl;
  cout << " ";
  Utility::printHex(readRODRegister(partition, crate, rod, FE_CMND_MASK_0_HI), 8);
  cout << endl;
  cout << " ";
  Utility::printHex(readRODRegister(partition, crate, rod, FE_CMND_MASK_1_LO), 8);
  cout << endl;
  cout << " ";
  Utility::printHex(readRODRegister(partition, crate, rod, FE_CMND_MASK_1_HI), 8);
  cout << endl;

  cout << "L1ID for ports 0 and 1\n";
  Utility::printHex(readRODRegister(partition, crate, rod, CAL_L1_ID_0), 8);
  Utility::printHex(readRODRegister(partition, crate, rod, CAL_L1_ID_1), 8);
  cout << "BCID for ports 0 and 1\n";
  Utility::printHex(readRODRegister(partition, crate, rod, CAL_BCID), 8);

  cout << dec;
#endif
}

void SctApi::sendTimTrigger(unsigned int partition, unsigned int crate, const Trigger *trig) {
#warning "Convert to TIM trigger and send"
}

void SctApi::sendTimBurst(unsigned int partition, unsigned int crate, unsigned int count) {
  getCrate(partition, crate)->sendTimBurst(count);
}

void SctApi::sendTrigger(unsigned int partition, unsigned int crate, unsigned int rod, const Trigger *trig) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "Send Trigger\n";
  }

  long *data = new long[sizeof(BUILD_STREAM_IN)/4];

  BUILD_STREAM_IN &buildPrim = *(BUILD_STREAM_IN*)data;

#if (R_BUILD_STREAM == 102)
  Trigger::RODTriggers points = trig->getRODTriggers();

  for(unsigned int i=0; i<points.size(); i++) {
    buildPrim.cmdList.cmd[i] = points[i].first;        // Set a trigger sequence
    buildPrim.cmdList.data[i] = points[i].second;      // Set a trigger sequence
  }

#ifdef CMD_BUFFER_0
  buildPrim.cmdBuff = CMD_BUFFER_0;
#else
  buildPrim.cmdBuff = SP0;
#endif
  buildPrim.reset = 1;
  buildPrim.chip = 63;
  buildPrim.fibre = 3;
  buildPrim.dataLen = 0;
  buildPrim.data = 0;    // Should be ignored?
 
  shared_ptr<PrimListWrapper> sendList(new PrimListWrapper(3));

  sendList->addPrimitive(RodPrimitive(4 + sizeof(BUILD_STREAM_IN)/4, 
                                      2, BUILD_STREAM, R_BUILD_STREAM, data),
                         data);
#elif R_BUILD_STREAM == 101
#error "Build stream not compiled (unsupported old version)" 
#else   // Unknown R_BUILD_STREAM
#error "Build stream not compiled (unknown primitive version, check primParams.h)" 
#endif

#ifdef CMD_BUFFER_BOTH
  PrimBuilder::instance().sendStream(sendList, CMD_BUFFER_BOTH, 0);
#else
  PrimBuilder::instance().sendStream(sendList, SP_BOTH, 0);
#endif

  sendPrimList(partition, crate, rod, sendList);

  int responseCode = awaitResponse(partition, crate, rod, 2);

  if(responseCode != 0) {
    cout << "Send trigger failed!\n";
  }
}

void SctApi::sendL1A(unsigned int partition, unsigned int crate, unsigned int rod, bool capture) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "Send L1A\n";
  }

  long *data = new long[sizeof(BUILD_STREAM_IN)/4];

  BUILD_STREAM_IN &buildPrim = *(BUILD_STREAM_IN*)data;

#if (R_BUILD_STREAM == 102)
  for(int i=0; i<6; i++) {
    buildPrim.cmdList.cmd[i] = 0x64;
    buildPrim.cmdList.data[i] = 0;
  }

  buildPrim.cmdList.cmd[0] = 0x65;

#ifdef CMD_BUFFER_0
  buildPrim.cmdBuff = CMD_BUFFER_0;
#else
  buildPrim.cmdBuff = SP0;
#endif
  buildPrim.reset = 1;
  buildPrim.chip = 63;
  buildPrim.fibre = 3;
  buildPrim.dataLen = 0;
  buildPrim.data = 0;    // Should be ignored?
 
  shared_ptr<PrimListWrapper> sendList(new PrimListWrapper(3));

  sendList->addPrimitive(RodPrimitive(4 + sizeof(BUILD_STREAM_IN)/4, 
                                      2, BUILD_STREAM, R_BUILD_STREAM, data),
                         data);
#elif R_BUILD_STREAM == 101
#error "Build stream not compiled (unsupported old version)" 
#else   // Unknown R_BUILD_STREAM
#error "Build stream not compiled (unknown primitive version, check primParams.h)" 
#endif

#ifdef CMD_BUFFER_BOTH
  PrimBuilder::instance().sendStream(sendList, CMD_BUFFER_BOTH, capture);
#else
  PrimBuilder::instance().sendStream(sendList, SP_BOTH, capture);
#endif

  sendPrimList(partition, crate, rod, sendList);

  awaitResponse(partition, crate, rod, 2);
}

void SctApi::bocHistogram(unsigned int partition, unsigned int crate, unsigned int rod, 
                          unsigned int samples, unsigned int numLoops) {
  shared_ptr<PrimListWrapper> bocList(new PrimListWrapper(3));

  PrimBuilder::instance().bocHistogram(bocList, samples, numLoops);

  sendPrimList(partition, crate, rod, bocList);

  int responseCode = awaitResponse(partition, crate, rod, 10);

  if(responseCode != 0) {
    cout << "BOC histogram failed!\n";
    return;
  }

  unsigned long length;
  unsigned long *mem = getResponse(partition, crate, rod, length);

  if(mem) {
    BOC_HISTOGRAM_OUT &histoOut = *((BOC_HISTOGRAM_OUT*)(mem + 8));

    for(int l=0; l<96; l++) {
      for(int c=0; c<2; c++) {
        cout << " " << histoOut.histo[l][c];
      }
      cout << endl;
    }

    delete [] mem;
  } else {
    cout << "Bad result from BOC_HISTOGRAM\n";
  }
}

void SctApi::printABCDModule(int mid) {
  ABCDModule *mConf = lookupConfig(mid);

  if(mConf) {
    config->printModuleConfig(*mConf);
  } else {
    cout << "Invalid module id " << mid << endl;
  }
}

void SctApi::printABCDRodModule(int mid, BankType bank) {
  getABCDModule(mid, bank);

  ABCDModule *mConf = lookupConfig(mid);

  config->printModuleConfig(*mConf);
}

void SctApi::modifyBOCParam(unsigned int type, unsigned int val) {
  for(map<pair<unsigned int, unsigned int>, Crate* >::const_iterator ci = crateMap.begin();
      ci != crateMap.end();
      ci++) {
    getCrate(ci->first.first, ci->first.second)->modifyBOCParam(type, val, true);
  }
}

void SctApi::modifyBOCParam(unsigned int partition, unsigned int crate, unsigned int rod,
                            unsigned int channel, unsigned int type, unsigned int val) {
  getCrate(partition, crate)->modifyBOCParam(rod, channel, type, val, true);
}

void SctApi::printBOCSetup(unsigned int partition, unsigned int crate, unsigned int rod) {
  getCrate(partition, crate)->printBOCSetup(rod);
}

vector<BOCChannelConfig> SctApi::currentBOCSetup(unsigned int partition, unsigned int crate, unsigned int rod) {
  return getCrate(partition, crate)->currentBOCSetup(rod);
}

void SctApi::printBOCRegisters(unsigned int partition, unsigned int crate, unsigned int rod) {
  getCrate(partition, crate)->printBOCRegisters(rod);
}

BOCGlobalConfig SctApi::currentBOCRegisters(unsigned int partition, unsigned int crate, unsigned int rod) {
  return getCrate(partition, crate)->currentBOCRegisters(rod);
}

void SctApi::saveBOCSetup(unsigned int partition, unsigned int crate, unsigned int rod, BankType bank) {
  getCrate(partition, crate)->saveBOCSetup(rod, bank);
}

void SctApi::saveBOCRegisters(unsigned int partition, unsigned int crate, unsigned int rod, BankType bank) {
  getCrate(partition, crate)->saveBOCRegisters(rod, bank);
}

void SctApi::restoreBOCSetup(unsigned int partition, unsigned int crate, unsigned int rod, BankType bank) {
  getCrate(partition, crate)->restoreBOCSetup(rod, bank);
}

void SctApi::restoreBOCRegisters(unsigned int partition, unsigned int crate, unsigned int rod, BankType bank) {
  getCrate(partition, crate)->restoreBOCRegisters(rod, bank);
}

void SctApi::lasersOff() {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "Lasers off called\n";
    log << "\t" << second_clock::universal_time() << endl;
  }

  std::cout << " *** Trying to turn lasers off  ***\n";
  for(map<pair<unsigned int, unsigned int>, Crate* >::const_iterator ci = crateMap.begin();
      ci != crateMap.end();
      ci++) {
    ci->second->lasersOff();
  }
}

void SctApi::timSetFrequency(unsigned int partition, unsigned int crate, double trigFreq, double rstFreq) {
  getCrate(partition, crate)->timSetFrequency(trigFreq, rstFreq);
}

void SctApi::freeTriggers(unsigned int partition, unsigned int crate) {
  shared_ptr<PrimListWrapper> timList(new PrimListWrapper(1));

#ifdef RRIF_CMND_1
  PrimBuilder &builder = PrimBuilder::instance();
  // Use TIM triggers
  builder.writeRegister(timList, RRIF_CMND_1, 1, 1, 1);
  // Enable TIM trigger event ID decoding
  builder.writeRegister(timList, RRIF_CMND_1, 8, 1, 1);
#else
#error "Unsupported no registers"
#endif

  sendPrimListAll(partition, crate, timList);

  awaitResponseAll(partition, crate, 2);

  getCrate(partition, crate)->freeTriggers();
}

void SctApi::stopTriggers(unsigned int partition, unsigned int crate) {
  shared_ptr<PrimListWrapper> timList(new PrimListWrapper(1));

#ifdef RRIF_CMND_1
  PrimBuilder &builder = PrimBuilder::instance();
  // Use TIM triggers
  builder.writeRegister(timList, RRIF_CMND_1, 1, 1, 0);
  // Enable TIM trigger event ID decoding
  builder.writeRegister(timList, RRIF_CMND_1, 8, 1, 0);
#else
#error "Unsupported no registers"
#endif

  sendPrimListAll(partition, crate, timList);

  awaitResponseAll(partition, crate, 2);

  getCrate(partition, crate)->stopTriggers();
}

void SctApi::timL1A(unsigned int partition, unsigned int crate) {
  getCrate(partition, crate)->timL1A();
}

void SctApi::timCalL1A(unsigned int partition, unsigned int crate, int delay) {
  getCrate(partition, crate)->timCalL1A(delay);
}

void SctApi::timSoftReset(unsigned int partition, unsigned int crate) {
  // FER doesn't reset TIM L1 count
  getCrate(partition, crate)->timECR();
}

void SctApi::timBCReset(unsigned int partition, unsigned int crate) {
  getCrate(partition, crate)->timBCR();
}

void SctApi::timVerbose(unsigned int partition, unsigned int crate) {
  getCrate(partition, crate)->timVerbose();
}

void SctApi::timWriteRegister(unsigned int partition, unsigned int crate, int reg, UINT16 val) {
  getCrate(partition, crate)->timRegLoad(reg, val);
}

UINT16 SctApi::timReadRegister(unsigned int partition, unsigned int crate, int reg) {
  return getCrate(partition, crate)->timRegRead(reg);
}

void SctApi::decodeConfig(unsigned int partition, unsigned int crate, unsigned int rod, 
                          bool skipTrim, bool bypass) {
  unsigned long readlength;
  unsigned long *config = dspBlockRead(partition, crate, rod, 0x2102000, 270 * 12, -1, readlength);

  unsigned int pos = 0;
  while(pos < readlength) {
    if((config[pos] & 0x5760000) != 0x5700000) {
      cout << "Strange data: ";
      cout << hex << " " << pos << " " << config[pos] << dec << endl;
      break;
    } 

    unsigned int f3 = (config[pos] & 0xff000) >> 12;
    unsigned int f4 = (config[pos] & 0x00fc0) >> 6;
    unsigned int f5 = (config[pos] & 0x0003f) >> 0;

    pos ++;

    cout << "Chip address " << f4 << ": ";
    if(!bypass) {
      cout << hex << f3 << "/" << f5 << dec << ": ";
    }

    switch(f3) {
    case 0x1c:
      {
        unsigned short data = (0xffff0000 & config[pos]) >> 16;
        switch(f5) {
        case 0:
          // Configuration reg
          if(!bypass) {
            cout << "Config: " << hex << data << dec;
            cout << " comp " << (data & 3) 
                 << " cal "  << ((data & 0xc) >> 2)
                 << " trim "  << ((data & 0x30) >> 4)
                 << " edge "  << ((data & 0x40) >> 6)
                 << " mask "  << ((data & 0x80) >> 7)
                 << " acc "  << ((data & 0x100) >> 8)
                 << " in_b " << ((data & 0x200) >> 9)
                 << " out_b " << ((data & 0x400) >> 10)
                 << " mast " << ((data & 0x800) >> 11)
                 << " end " << ((data & 0x1000) >> 12)
                 << " feed " << ((data & 0x2000) >> 13)
                 << endl;
          } else {
            cout << " in_b " << ((data & 0x200) >> 9)
                 << " out_b " << ((data & 0x400) >> 10)
                 << " mast " << ((data & 0x800) >> 11)
                 << " end " << ((data & 0x1000) >> 12)
                 << " feed " << ((data & 0x2000) >> 13)
                 << endl;
          }
          break;
        case 0x10:
          if(!bypass) {
            cout << "Strobe delay: " << hex 
                 << (data & 0x3f) << dec << endl;
          }
          break;
        case 0x18:
          if(!bypass) {
            cout << "Threshold/Cal: " << hex
                 << ((data & 0xff00) >> 8) << "/" 
                 << (data & 0xff) << dec << endl;
          }
          break;
        case 0x38:
          if(!bypass) {
            cout << "preamp/shaper: " << hex
                 << ((data & 0xff00) >> 8) << "/" 
                 << (data & 0xff) << dec << endl;
          }
          break;
        case 0x04:
          if(skipTrim || bypass) {
            cout << "\r                   \r";
          } else {
            cout << "TRIM DAC: " << hex
                 << (data & 0xf) << " " 
                 << ((data & 0x7f0) >> 4) << dec << endl;
          }
          break;
        }
        pos ++;
        break;
      }
    case 0x8c:
      if(!bypass) {
        // Mask reg
        cout << "Mask ";
        cout << hex;
        for(int i=0; i<4; i++) {
          cout << config[pos++] << " ";
        }
        cout << dec;
        cout << endl;
        pos++;
      } else {
        pos += 5;
      }
      break;
    case 0x0c:
      cout << "Something else\n";
      break;
    }
  }
}

void SctApi::getrpcrc(UINT32 mid, unsigned int &rpartition, unsigned int &rcrate, unsigned int &rrod, unsigned int &rchannel) {
  unsigned int partition, crate, rod, channel;

  getpcrc(mid, partition, crate, rod, channel);

  unsigned int mur, module;

  try {
    if(checkDebugOption(DEBUG_DIAG))
      cout << "Looking for module in partition " << partition << " crate " << crate << " rod " << rod << " channel " << channel << endl;
    config->translateFromROD(partition, crate, rod, channel, 
                             mur, module);

    if(checkDebugOption(DEBUG_DIAG2))
      cout << "Found MUR " << mur << " module number " << module << endl;
  } catch(ConfigurationException &c) {
    cout << "Can't find module in MUR map: \n";
    cout << c.what() << endl;

    {
      boost::mutex::scoped_lock lock(logMutex);
      log << "  (Not found)\n";
    }
    throw;
  }

  try {
    config->translateToRROD(mur, module, 
                            rpartition, rcrate, rrod, rchannel);
  } catch(ConfigurationException &c) {
    cout << "Module redundancy not found: \n";
    cout << c.what() << endl;

    {
      boost::mutex::scoped_lock lock(logMutex);
      log << "  (Not found)\n";
    }
    throw;
  }
}

long SctApi::getRodSlot(unsigned int partition, unsigned int crate, unsigned int rod) 
{
  if(getCrate(partition, crate)) {
    return getCrate(partition, crate)->getRodSlot(rod);
  } else {
    return -1;
  }
}

int SctApi::getRodRevision(unsigned int partition, unsigned int crate, unsigned int rod) 
{
  if(getCrate(partition, crate)) {
    return getCrate(partition, crate)->getRodRevision(rod);
  } else {
    return -1;
  }
}

int SctApi::getRodRevision(const RodLabel &label) 
{
  if(getCrate(label.partition, label.crate)) {
    return getCrate(label.partition, label.crate)->getRodRevision(label.rod);
  } else {
    return -1;
  }
}

void SctApi::testLinkOutSelect(unsigned int partition, unsigned int crate, unsigned int rod,
                               unsigned int link) 
{
#ifdef FMT_LINK_DATA_TEST_MUX
  int formatter = link/12;
  int intLink = link%12;

  if(formatter > 8 || formatter < 0) {
    cout << "formatter out of range: " << formatter << endl;
    return;
  }

  if(intLink > 12 || intLink < 0) {
    cout << "link out of range: " << intLink << endl;
    return;
  }

  shared_ptr<PrimListWrapper> primList(new PrimListWrapper(1));

  PrimBuilder &builder = PrimBuilder::instance();
  builder.writeRegister(primList, FMT_LINK_DATA_TEST_MUX(formatter), 0, 4, intLink);   // 0x070 + formatter
  builder.writeRegister(primList, EFB_CMND_0, 4, 3, formatter & 0x7);                  // 0x1a4

  sendPrimList(partition, crate, rod, primList);
  awaitResponse(partition, crate, rod, 2);
#else 
#warning "Link out select not compiled"
  cout << "Unsupported action\n";
#endif
}

unsigned int SctApi::testLinkOutRetrieve(unsigned int partition, unsigned int crate, unsigned int rod) 
{
#ifdef FMT_LINK_DATA_TEST_MUX
  int fReg = readRODRegister(partition, crate, rod, EFB_CMND_0);
  int formatter = (fReg >> 4) & 0x7;

  int lReg = readRODRegister(partition, crate, rod, FMT_LINK_DATA_TEST_MUX(formatter));
  int intLink = lReg & 0xf;

  return formatter * 12 + intLink;
#else 
#error "Link out select not compiled"
#endif
}

void SctApi::setDebugOption(std::string opt) {
  Debug::getInstance()->setDebugOption(opt);
}

void SctApi::unsetDebugOption(std::string opt) {
  Debug::getInstance()->unsetDebugOption(opt);
}

std::list<std::string> SctApi::listEnabledDebugOptions() {
  return Debug::getInstance()->listEnabledDebugOptions();
}

std::vector<std::string> SctApi::listDebugOptions() {
  return Debug::getInstance()->listDebugOptions();
}

bool SctApi::checkDebugOption(std::string opt) {
  return Debug::getInstance()->checkDebugOption(opt);
}

bool SctApi::checkDebugOption(int intOpt) {
  return Debug::getInstance()->checkDebugOption((DebugOptions)intOpt);
}

void SctApi::debugStepHistogram()
{
  if(!lastDebugScanEx) {
    cout << "No last scan stored\n";
    return;
  }

  // This can only work as a single ROD method!
  RodLabel zeroRod = lastDebugScanEx->rodInfo.begin()->first;

  // Do step
  dspSingleWrite(zeroRod.partition, zeroRod.crate, zeroRod.rod, 
                 0x80000010, lastDebugScanEx->diagnosticReg | (1<<7) | (1<<6), -1); 

#if USE_THREADS
  sleep(1);
#else
  // Read the text buffers
  awaitResponse(zeroRod.partition, zeroRod.crate, zeroRod.rod, 1);
#endif

  if(checkDebugOption(DEBUG_EXTRA_DUMPS)) {
    standardRegisterDump(zeroRod.partition, zeroRod.crate, zeroRod.rod);
  }
}

void SctApi::debugContinueHistogram()
{
  // This can only work as a single ROD method!
  RodLabel zeroRod = lastDebugScanEx->rodInfo.begin()->first;

  dspSingleWrite(zeroRod.partition, zeroRod.crate, zeroRod.rod, 
                 0x80000010, lastDebugScanEx->diagnosticReg | (1<<7), -1); 

#if USE_THREADS
  sleep(1);
#else
  // Read the text buffers
  awaitResponse(zeroRod.partition, zeroRod.crate, zeroRod.rod, 
                1);
#endif

  if(checkDebugOption(DEBUG_EXTRA_DUMPS)) {
    standardRegisterDump(zeroRod.partition, zeroRod.crate, zeroRod.rod);
  }
}

void SctApi::debugAbortHistogram()
{
#warning "How to do this... tidyHistogramming? (not when in stalled mode!) AUTO_STALL ends at next bin!"
  // killing histogram task then step?
  time_t start_time = time(0);

  // This can only work as a single ROD method!
  RodLabel zeroRod = lastDebugScanEx->rodInfo.begin()->first;

  while(1) {
    cout << "** Loop\n";
    dspSingleWrite(zeroRod.partition, zeroRod.crate, zeroRod.rod, 
                   0x80000010, (1<<7), -1);
    while(dspSingleRead(zeroRod.partition, zeroRod.crate, zeroRod.rod, 
                        0x80000010, -1) & (1<<7)) {
      if((time(0) - start_time) > 2) {
        goto breakOut;
      }
    }
  }

 breakOut:
  tidyHistogramming();
}

void SctApi::standardRegisterDump(RodLabel rl) {
  standardRegisterDump(rl.partition, rl.crate, rl.rod);
}

void SctApi::standardRegisterDump(unsigned int partition, unsigned int crate, unsigned int rod) {
  static int dumpCount = 0;

  if(!getCrate(partition, crate) 
     || !getCrate(partition, crate)->RODPresent(rod)) {
    cout << "Invalid crate or ROD in standardRegisterDump\n";
    return;
  }

  cout << "Making dump number " << dumpCount << endl;

  // Awkward to do %05d with strings...
  char *dirName = new char[Sct::SctNames::getTempDir().size() + 20];
  sprintf(dirName, "%s/Dump_%05d", Sct::SctNames::getTempDir().c_str(), dumpCount ++);

  // Ignore result, chdir succeeds or fails anyway (0 is success)
  mkdir(dirName, S_IRUSR|S_IWUSR|S_IXUSR);

  char *saveDir = getcwd(NULL, 0);
  chdir(dirName);

  dspBlockDumpFile(partition, crate, rod, 0x02400000, 0x80000, -1, "BigMDSPDump.bin");

  dspBlockDumpFile(partition, crate, rod, 0x00400000, 0x800, -1, "StandardDumpFRM.bin");
  dspBlockDumpFile(partition, crate, rod, 0x00402000, 0x100, -1, "StandardDumpEFB.bin");
  dspBlockDumpFile(partition, crate, rod, 0x00402400, 0x060, -1, "StandardDumpRTR.bin");
  dspBlockDumpFile(partition, crate, rod, 0x00404400, 0x300, -1, "StandardDumpRCF.bin");

  dspBlockDumpFile(partition, crate, rod, 0x02000000, 0x40000, -1, "StandardDumpMDSP_XPROG.bin");
  dspBlockDumpFile(partition, crate, rod, 0x80000000, 0x02000, -1, "StandardDumpMDSP_IDRAM.bin");

  ofstream moduleList("StandardModuleList.txt");

  moduleList << "Module configuration for " << moduleMap.size() << " modules:\n";
  for(map<UINT32, ABCDModule>::const_iterator mi = moduleMap.begin();
      mi!=moduleMap.end();
      mi ++) {
    UINT32 mid = mi->first;

    string sn = convertToString(mid);
    int group = lookupConfig(mid)->groupId;
    int select = lookupConfig(mid)->select;
    int prim = lookupConfig(mid)->pTTC;
    int redun = lookupConfig(mid)->rTTC;
    int link0 = lookupConfig(mid)->rx[0];
    int link1 = lookupConfig(mid)->rx[1];

    // Reverse translate formatter encoding
    if(link0 != DATA_LINK_OFF) 
      link0 = (link0/16) * 12 + link0%16;
    if(link1 != DATA_LINK_OFF) 
      link1 = (link1/16) * 12 + link1%16;

    moduleList.width(8);
    moduleList << mid << " " << sn << " Group: " << group << " Select: " << select;
    moduleList << "\t Pr: ";
    moduleList.width(2);
    moduleList << prim << " Red: ";
    moduleList.width(3);
    moduleList << redun << " Link 0/1: ";
    moduleList.width(3);
    moduleList << link0 << " ";
    moduleList.width(3);
    moduleList << link1 << endl;
  }

  if(getCrate(partition, crate)->getRodRevision(rod) == 0xE) {
    for(int s=0; s<4; s++) {
      // Event data in slave
      string idramFile = "StandardDumpSDSP0_IDRAM.bin";
      idramFile[16] = s + '0';
      dspBlockDumpFile(partition, crate, rod, 0x00010000, 0x2000, s, idramFile);
      string burstFile = "StandardDumpSDSP0_BURST.bin";
      burstFile[16] = s + '0';
      dspBlockDumpFile(partition, crate, rod, 0x00018000, 0x2000, s, burstFile);
      string sdramFile = "StandardDumpSDSP0_SDRAM.bin";
      sdramFile[16] = s + '0';
      dspBlockDumpFile(partition, crate, rod, 0xa0040000, 0x28800, s, sdramFile);
    }
  } else {  // Rev C
    for(int s=0; s<4; s++) {
      // Event data in slave
      string idramFile = "StandardDumpSDSP0_IDRAM.bin";
      idramFile[16] = s + '0';
      dspBlockDumpFile(partition, crate, rod, 0x80000000, 0x2000, s, idramFile);
      string burstFile = "StandardDumpSDSP0_BURST.bin";
      burstFile[16] = s + '0';
      dspBlockDumpFile(partition, crate, rod, 0x80008000, 0x2000, s, burstFile);
      string sdramFile = "StandardDumpSDSP0_SDRAM.bin";
      sdramFile[16] = s + '0';
      dspBlockDumpFile(partition, crate, rod, 0x02040000, 0x28800, s, sdramFile);
    }
  }

  chdir(saveDir);
}

void SctApi::standardRegisterDumpAll() {
  for(list<RodLabel>::const_iterator rl = rodList.begin();
      rl!=rodList.end();
      rl++){
    standardRegisterDump(*rl);
  }
}

void SctApi::setupScanMasks(ScanEx &extra, int distSlave, bool dual) {
  RodScanEx defaultInfo;

  if(distSlave == 0) {
    defaultInfo.slaves = 1;
    defaultInfo.bitFieldDSP = 0x1;
  } else if(distSlave == 1) {
    defaultInfo.slaves = 0;
    defaultInfo.bitFieldDSP = 0x0; // Fill in which slaves are used later
  } else if(distSlave == 2) {
    defaultInfo.slaves = 4;
    defaultInfo.bitFieldDSP = 0xf;
  }

  ModuleMask zeroMask = {0, 0};

  defaultInfo.channels = zeroMask;

  for(int i=0; i<8; i++) {
    defaultInfo.groupChannels[i] = zeroMask;
  }
  for(int i=0; i<4; i++) {
    defaultInfo.slaveChannels[i] = zeroMask;
  }

  // Group to DSP mappings
  if(distSlave == 0) {               // Which groups for each DSP to use
    // All groups to slave 0
    extra.groupDspMap[0] = 0xff;
    extra.groupDspMap[1] = 0x0;
    extra.groupDspMap[2] = 0x0;
    extra.groupDspMap[3] = 0x0;
  } else if(distSlave == 2) {
    // All groups to all slaves
    extra.groupDspMap[0] = 0xff;
    extra.groupDspMap[1] = 0xff;
    extra.groupDspMap[2] = 0xff;
    extra.groupDspMap[3] = 0xff;
  } else {
    if(!dual) {
      // Each group to a separate DSP
      extra.groupDspMap[0] = 0x11;
      extra.groupDspMap[1] = 0x22;
      extra.groupDspMap[2] = 0x44;
      extra.groupDspMap[3] = 0x88;
    } else {
      // Each Dsp has to service one trigger
      extra.groupDspMap[0] = 0x03;
      extra.groupDspMap[1] = 0x0c;
      extra.groupDspMap[2] = 0x30;
      extra.groupDspMap[3] = 0xc0;
    }
  }

  if(!dual) {
    extra.groupSpMap[0] = 0xff;        // All groups use port 0
    extra.groupSpMap[1] = 0x00;
  } else {
    extra.groupSpMap[0] = 0xf;         // Groups 0-3 use port 0
    extra.groupSpMap[1] = 0xf0;        // Groups 4-7 use port 1
  }

  if(!dual) {
    extra.groupRangeMap[0] = 0xff;         // All groups use range list 0
    extra.groupRangeMap[1] = 0x0;
  } else {
    extra.groupRangeMap[0] = 0x0f;         // Groups 0-3 -> range list 0
    extra.groupRangeMap[1] = 0xf0;         // Groups 4-7 -> range list 1
  }

  // Set up masks for modules
  for(map<UINT32, ABCDModule>::const_iterator mi = moduleMap.begin();
      mi!=moduleMap.end();
      mi ++) {
    UINT32 currMid = mi->first;

    ABCDModule *moduleConfig = lookupConfig(currMid);
    unsigned int group = moduleConfig->groupId;

    unsigned int newPartition, newCrate, newRod, newChannel;
    getpcrc(currMid, newPartition, newCrate, newRod, newChannel);

    RodLabel thisRod(newPartition, newCrate, newRod);

    RodScanEx &thisInfo = extra.getOrNewRod(thisRod, defaultInfo);

    // Check if redundancy in use
    if(moduleConfig->select) {
      // does clk/cmd go from same ROD
      try {
        unsigned int rpartition, rcrate, rrod, rchannel;
        getrpcrc(currMid, rpartition, rcrate, rrod, rchannel);

        if(newPartition != rpartition || newCrate != rcrate || newRod != rrod) {
          // Nothing to do as mask are rx
        }
      } catch(ConfigurationException &c) {
        // Redundant conversion failed, this should have been picked up in the checks
      }
    }

    if(newChannel<32) {
      if(thisInfo.channels.mask0 & 1<<newChannel) {
        cout << "Module already using this channel in this group\n";
      }
      thisInfo.channels.mask0 |= 1<<newChannel;

      if(distSlave == 1) {
        thisInfo.groupChannels[group].mask0 |= 1<<newChannel;
      } else if(distSlave == 0) {
        thisInfo.groupChannels[0].mask0 |= 1<<newChannel;
      } else {
        // Same as distSlave == 1...
        thisInfo.groupChannels[group].mask0 |= 1<<newChannel;
      }
    } else {
      if(thisInfo.channels.mask1 & 1<<(newChannel-32)) {
        cout << "Module already using this channel in this group\n";
      }
      thisInfo.channels.mask1 |= 1<<(newChannel-32);

      if(distSlave == 1) {
        thisInfo.groupChannels[group].mask1 |= 1<<(newChannel-32);
      } else if(distSlave == 0) {
        thisInfo.groupChannels[0].mask1 |= 1<<(newChannel-32);
      } else {
        // Same as distSlave == 1...
        thisInfo.groupChannels[group].mask1 |= 1<<(newChannel-32);
      }
    }

    if(group+1 > extra.groupLists.size()) {
      extra.groupLists.resize(group + 1);
    }
    extra.groupLists[group].push_back(convertToString(currMid));
  }

  // Fix slave maps
  for(list<RodLabel>::const_iterator rl = rodList.begin();
      rl!=rodList.end();
      rl++){

    RodScanEx &thisInfo = extra.getRodScanInfo(*rl);

    thisInfo.slaves = 0;

    for(int slave=0; slave<numSlaves; slave++) {
      for(int group=0; group<8; group++) {
        if(extra.groupDspMap[slave] & (1<<group)) {
          if(thisInfo.groupChannels[group].mask0 || thisInfo.groupChannels[group].mask1) {
            thisInfo.bitFieldDSP |= 1<<slave;
          }

          thisInfo.slaveChannels[slave].mask0 |= thisInfo.groupChannels[group].mask0;
          thisInfo.slaveChannels[slave].mask1 |= thisInfo.groupChannels[group].mask1;
        }
      }

      if(thisInfo.bitFieldDSP & (1<<slave)) {
        thisInfo.slaves++; 
      }
    }
  }
}

// Check lists are valid
bool SctApi::checkModuleListsForScan() {
  if(moduleMap.begin() == moduleMap.end()) {
    cout << "No modules to scan!\n";
    if(mrs) {
      *mrs << "SCAN_NO_MODULES" << MRS_ERROR 
           << MRS_QUALIF("SCTAPI") << MRS_QUALIF("doScan") 
           << MRS_TEXT("No modules to scan")
           << ENDM;
    }
    return false;
  }

  // Initialise with first module
  UINT32 mid = moduleMap.begin()->first;
  unsigned int partition, crate, rod, channel;
  getpcrc(mid, partition, crate, rod, channel);

  for(map<UINT32, ABCDModule>::const_iterator mi = moduleMap.begin();
      mi!=moduleMap.end();
      mi ++) {
    UINT32 currMid = mi->first;

    unsigned int newPartition, newCrate, newRod, newChannel;

    getpcrc(currMid, newPartition, newCrate, newRod, newChannel);

    if(newPartition != partition) {
      cout << "Can't do scans on modules in different partitions (ever?)\n";
      if(mrs) {
        *mrs << "SCAN_MISMATCH" << MRS_ERROR 
             << MRS_QUALIF("SCTAPI") << MRS_QUALIF("doScan") 
             << MRS_TEXT("Mismatch partitions. Can't do scans on modules in different partitions (ever?)")
             << ENDM;
      }
      return false;
    }

    if(newCrate != crate) {
      cout << "Can't do scans on modules in different crates (yet)\n";
      if(mrs) {
        *mrs << "SCAN_MISMATCH" << MRS_ERROR 
             << MRS_QUALIF("SCTAPI") << MRS_QUALIF("doScan") 
             << MRS_TEXT("Crate mismatch, can't do scans on modules in different crates (yet)")
             << ENDM;
      }
      return false;
    }

    ABCDModule *config = lookupConfig(currMid);

    if(!config) {
      cout << "Trying to do a scan on a module with no configuration FAILED!\n";
      if(mrs) {
        *mrs << "SCAN_NOCONFIG" << MRS_ERROR 
             << MRS_QUALIF("SCTAPI") << MRS_QUALIF("doScan") 
             << MRS_TEXT("Trying to do a scan on a module with no configuration FAILED!")
             << ENDM;
      }
      return false;
    }

    // Check if redundancy in use
    if(config->select) {
      // does clk/cmd go from same ROD
      try {
        unsigned int rpartition, rcrate, rrod, rchannel;
        getrpcrc(currMid, rpartition, rcrate, rrod, rchannel);

        if(partition != rpartition || crate != rcrate || rod != rrod) {
          cout << "Can't do redundant stuff on split RODs at the moment, SCAN CANCELLED\n";
          return false;
        }
      } catch(ConfigurationException &c) {
        cout << "No mapping to redundant channel for module with select, SCAN CANCELLED\n";
        return false;
      }
    }
  }

  return true;
}

bool SctApi::preScanHardwareCheck(Scan& scan, ScanEx& extra) {
  // Check that the appropriate slaves were initialised 
  for(ScanEx::RodInfoMap::const_iterator ri = extra.rodInfo.begin();
      ri != extra.rodInfo.end();
      ri++) {
    const RodLabel rod = ri->first;
    for(int i=0; i<4; i++) {
      if(ri->second.bitFieldDSP & (1<<i) 
         && getCrate(rod.partition, rod.crate)
         && !getCrate(rod.partition, rod.crate)->slavePresent(rod.rod, i)) {
        cout << "***** A slave DSP necessary for the requested scan was not initialised\n";
        cout << "*****  please check the configuration and the Slave Image files\n";
        if(mrs) {
          *mrs << MRS_ERROR 
               << "SCAN_NODSP" 
               << MRS_QUALIF("SCTAPI") << MRS_QUALIF("doScan") 
               << MRS_PARAM<int>("slavesNeeded", ri->second.bitFieldDSP)
               << MRS_PARAM<int>("slaveNotStarted", i)
               << MRS_TEXT("Required DSPs not initialised for the requested scan")
               << ENDM;
        }
        return false;
      }
    }

    // Don't bother checking if in debug mode
    if(!getCrate(rod.partition, rod.crate)->checkBOCLasersOn(rod.rod)) {
      if(!scan.getOption(Scan::DEBUG)) {
        cout << "Trying to do scan using BOC that has its lasers cut out (aborting)\n";
        if(mrs) {
          *mrs << "BOC_INTERLOCKED" << MRS_ERROR << MRS_QUALIF("SCTAPI") 
               << MRS_PARAM<int>("crate", rod.crate) 
               << MRS_PARAM<int>("rod", rod.rod)
               << MRS_TEXT("Scan aborting (BOC interlocked)") << ENDM;
        }

        return false;
      } else {
        cout << "Trying to do scan using BOC that has its lasers cut out (continuing for debug)\n";
      }
    }

    if(!checkAllModulesProbe("E")) {
#warning "Possibly disable the ones that aren't returning events? (with warnings!)"
      if(!scan.getOption(Scan::DEBUG)) {
        cout << "Check all modules in scan are returning events! (aborting)\n";
        if(mrs) {
          *mrs << "SCAN_ABORTED" << MRS_ERROR << MRS_QUALIF("SCTAPI") 
               << MRS_PARAM<int>("crate", rod.crate) 
               << MRS_PARAM<int>("rod", rod.rod)
               << MRS_TEXT("Scan aborting (Modules not returning events)") << ENDM;
        }

        return false;
      } else {
        cout << "All modules in scan are not returning events! (continuing for DEBUG)\n";

        if(mrs) {
          *mrs << "SCAN_WARNING" << MRS_WARNING << MRS_QUALIF("SCTAPI") 
               << MRS_PARAM<int>("crate", rod.crate) 
               << MRS_PARAM<int>("rod", rod.rod)
               << MRS_TEXT("Modules not all returning events (Scan continuing anyway (DEBUG))") << ENDM;
        }
      }
    }
  }
  
  return true;
}

/* This probably should be in ScanControlRODHisto, but 
   it has to be done before the main scan and so doesn't 
   fit into the current plan...
 */
void SctApi::preScanModuleSetup(Scan &scan) {
  /* Copy memory cache to SCAN bank (possible modifications) */
  setABCDModules(SCTAPI_BANK_SCAN);

  cout << "Configuration for all modules uploaded to SCAN\n";

  cout << "Do ROD configuration (masks etc)\n"; 
  calib_init();

  if(checkDebugOption(DEBUG_PRINT_CALIB)) {
    UINT32 mid = moduleMap.begin()->first;
    unsigned int partition, crate, rod, channel;
    getpcrc(mid, partition, crate, rod, channel);

    print_calib(partition, crate, rod);
  }
  // Send configuration to the module (this should be redundant as it the first thing the histogramming does)
  sendAllABCDModules(SCTAPI_BANK_SCAN);

  cout << "Module Configurations for scan uploaded to ROD\n";

  PrimBuilder &builder = PrimBuilder::instance();


  // For full histograms the formatters need to be told to produce expanded events
  if(scan.getOption(Scan::FULL)) {
    shared_ptr<PrimListWrapper> expandFormatList(new PrimListWrapper(1));
    for(int i=0; i<8; i++) {
      builder.writeRegister(expandFormatList, FMT_EXP_MODE_EN(i), 0, 12, 0x3ff);
    }

    if(synchSendPrimListAllCrates(expandFormatList) != 0) {
      cout << "Expand mode list failed!\n";
    }

    cout << "Formatters set for expanded mode\n";
  }

  if(scan.getScanVariable1() == ST_TOKEN
     || scan.getScanVariable1() == ST_BYPASS) {
    PrimBuilder &builder = PrimBuilder::instance();

    // Turn off chip sequence check in EFB
    shared_ptr<PrimListWrapper> chipSeqList(new PrimListWrapper(1));
    for(int i=0; i<48; i++) {
      builder.writeRegister(chipSeqList, ERROR_MASK(0, i), 10, 1, 1);
      builder.writeRegister(chipSeqList, ERROR_MASK(1, i), 10, 1, 1);
    }
    
    if(synchSendPrimListAllCrates(chipSeqList) != 0) {
      cout << "Chip sequence error list failed!\n";
    }

    cout << "EFB chip sequence detection turned off\n";
  }

  // Check for double triggers which aren't supported by DSP code
  {
    Trigger::RODTriggers points = scan.getTrigger1()->getRODTriggers();

    int triggerCount = 0;

    for(unsigned int i=0; i<points.size(); i++) {
      // Count triggers
      if(points[i].first == L1_TRIGGER) {
        triggerCount++;
      }
    }

    // Double triggers... none of current options will cope...
    if(triggerCount > 1 && scan.getOption(Scan::DISTSLAVE) < 3) {
      shared_ptr<PrimListWrapper> dblTrigFixList(new PrimListWrapper(1));

      // Mask L1 and BCID checks
      for(int f=0; f<48; f++) {
        // L1ID
        builder.writeRegister(dblTrigFixList, ERROR_MASK(0, f), 5, 1, 1);
        builder.writeRegister(dblTrigFixList, ERROR_MASK(1, f), 5, 1, 1);

        // BCID
        builder.writeRegister(dblTrigFixList, ERROR_MASK(0, f), 6, 1, 1);
        builder.writeRegister(dblTrigFixList, ERROR_MASK(1, f), 6, 1, 1);
      }

      if(synchSendPrimListAllCrates(dblTrigFixList)) {
        cout << "Double trigger BC check list failed!\n";
      }
    }
  }

  cout << "Done module set-up\n";

  if(checkDebugOption(DEBUG_PRINT_CALIB)) {
    // A bit over the top just to find 0, 0, 0!
    UINT32 mid = moduleMap.begin()->first;
    unsigned int partition, crate, rod, channel;
    getpcrc(mid, partition, crate, rod, channel);

    print_calib(partition, crate, rod);
  }
}

void SctApi::resumePolling() {
  for(map<pair<unsigned int,unsigned int>, Crate*>::const_iterator i=crateMap.begin();i!=crateMap.end();i++)
    i->second->resumePolling();
}

void SctApi::stopPolling() {
  for(map<pair<unsigned int,unsigned int>, Crate*>::const_iterator i=crateMap.begin();i!=crateMap.end();i++)
    i->second->stopPolling();
}

Scan::~Scan() {}
Trigger::~Trigger() {}

}  // End of namespace SctApi

// Some static exception handlers documented only in this file
/**
   Catch various exceptions (just so its known what happened, can't recover anything)
   Hopefully VME will get shut down properly???

   *** Only works for exception prototypes with missing exceptions ***
 */
static void handle_unexpected(void) {
  cerr << "Unexpected exception thrown in unknown place\n";

  try {
    throw;
  } catch (RodException &r) {
    cerr << "Rod Exception\n";
    cerr << r.getDescriptor() << ", " << r.getData1() << ", " << r.getData2() << endl;
  } catch (NoImageFile &f) {
    cerr << "No image file exception " << f.getFileName() << endl;
  } catch (VmeException &v) {
    SctApi::Utility::decodeVme(v);
  } catch (HpiException &h) {
    cerr << "HpiException:\n";
    hex(cerr);
    cerr << h.getDescriptor() << '\n';
    cerr << "calcAddr: " << h.getCalcAddr() << ", readAddr: " <<
      h.getReadAddr() << '\n';
    dec(cerr);
  } catch (PrimListException &p) {
    cerr << "Primlist Exception:\n";
    cerr << p.getDescriptor() << " " << p.getData1() << ", " << p.getData2() << endl;
  } catch(std::exception &e) {
    cerr << "std::exception with what: " << e.what() << endl;
  } catch(...) {
    cerr << "Unknown exception type\n";
  }

  // New version of VmeInterface should cope and shutdown RCC interface properly
  exit(1);

  //  Rethrow exception, function _must_ catch (and therefore method with specification, 
  //   if no exceptions specified then no problem in the first place...) std::bad_exception to recover.
// throw;
}

#if 0

void buildChannelMasks() {
  UINT32 txchannels[2];
  UINT32 rxchannels[4];

  txchannels[0] = 0;
  txchannels[1] = 0;
  rxchannels[0] = 0;
  rxchannels[1] = 0;
  rxchannels[2] = 0;
  rxchannels[3] = 0;

  // Build channel map for modules
  for(map<UINT32, ABCDModule>::const_iterator mi = moduleMap.begin();
      mi!=moduleMap.end();
      mi ++) {
    UINT32 currMid = mi->first;

    cout << "Set up module " << currMid << endl;

    unsigned int newPartition, newCrate, newRod;
    // If requires look up redundant channel instead
#warning "How is off-ROD redundancy handled?... (old calib_init)"

    unsigned int rcvChannel, sndChannel;
    getpcrc(currMid, newPartition, newCrate, newRod, rcvChannel);

    if(lookupConfig(currMid)->select == 1) {
      try {
        unsigned int redPartition, redCrate, redRod;
        getrpcrc(currMid, redPartition, redCrate, redRod, sndChannel);
#warning "Check same ROD? (old calib_init)"
      } catch(ConfigurationException &c) {
        sndChannel = rcvChannel;
      }
    } else {
      sndChannel = rcvChannel;
    }

    char *mappings = config->getFibreMappings(newPartition, newCrate, newRod);

    int txChannel = mappings[sndChannel * 3 + 0];
    int rx0Channel = mappings[rcvChannel * 3 + 1];
    int rx1Channel = mappings[rcvChannel * 3 + 2];

    int txi, txval, rx0i, rx0val, rx1i, rx1val;

    if(txChannel<32) { txi = 0; txval = 1<<txChannel; }
    else { txi = 1; txval = 1<<(txChannel-32); }

    if(rx0Channel<32) { rx0i = 0; rx0val = 1<<rx0Channel; }
    else if(rx0Channel<64) { rx0i = 1; rx0val = 1<<(rx0Channel-32); }
    else if(rx0Channel<96) { rx0i = 2; rx0val = 1<<(rx0Channel-64); }
    else { rx0i = 3; rx0val = 1<<(rx0Channel-96); }

    if(rx1Channel<16) { rx1i = 0; rx1val = 1<<rx1Channel; }
    else if(rx1Channel<32) { rx1i = 1; rx1val = 1<<(rx1Channel-32); }
    else if(rx1Channel<64) { rx1i = 2; rx1val = 1<<(rx1Channel-64); }
    else { rx1i = 3; rx1val = 1<<(rx1Channel-96); }

    if(txchannels[txi] & txval) {
      cout << "Module already using this channel in this group\n";
    } else {
      txchannels[txi] |= txval;
    }

    if(rxchannels[rx0i] & rx0val) {
      cout << "Module already using this channel in this group\n";
    } else {
      rxchannels[rx0i] |= rx0val;
    }

    if(rxchannels[rx1i] & rx1val) {
      cout << "Module already using this channel in this group\n";
    } else {
      rxchannels[rx1i] |= rx1val;
    }

    delete [] mappings;
  }

  cout << "Calculated channel masks:\n";
  cout << hex;
  cout << "tx: 0x" << txchannels[0] << " 0x" << txchannels[1] << endl;
  cout << "rx: 0x" << rxchannels[0] << " 0x" << rxchannels[1] << " 0x" << rxchannels[2] << " 0x" << rxchannels[3] << endl;
  cout << dec;
}

#endif
