#include <is/isinfo.h>
#include <is/isinfotmpl.h>
#include <ipc/object.h>
#include <ipc/server.h>
#include <iostream>
#include <boost/shared_ptr.hpp>

#include <TH1.h>

#include "sctConf/configipc.h"
#include "TestApi.h"
#include "Sct/SctNames.h"
#include "Sct/SctParameters.h"

#include "DummyHardware.h"
#include "ScanMap.h"
#include "Exceptions.h"
#include "Configuration.h"
#include "SctDaqRootFile.h"

using namespace ::SctApi;
using namespace Sct;
using namespace std;

namespace SctTest {

TestApi::TestApi() : SctApi(configServer = shared_ptr<SctConfiguration::Configuration>(new SctConfiguration::ConfigIPC())) {
}

TestApi::TestApi(boost::shared_ptr<SctConfiguration::Configuration> newConf) : SctApi(newConf) {
}

int TestApi::initialiseRod(unsigned int partition, unsigned int crate, unsigned int rod) {
    return 0;
}

int TestApi::initialiseTim(unsigned int partition, unsigned int crate) {
    return 0;
}

int TestApi::initialiseBOC(unsigned int partition, unsigned int crate, unsigned int rod) {
    return 0;
}

void TestApi::getABCDModule(UINT32 mid, BankType bank) {
    if(mid==0xffffffff) {
        cout <<  "TestApi::getABCDModule called for all modules.  Bank: " << bank << endl;
	for(map<UINT32, ABCDModule>::const_iterator iter = moduleMap.begin(); iter != moduleMap.end(); iter ++) {	
	    getABCDModule(iter->first, bank);
	}
	return;
    }
    
    cout << "SctTestApi::getABCDModule called, mid: " << mid << " bank: " << bank <<endl;
    ABCDModule *previous = lookupConfig(mid);
    *previous = DummyHardware::instance().getConfig(mid, bank);
}

void TestApi::setABCDModule(UINT32 mid, BankType bank) {
    if(mid==0xffffffff) {
	cout <<  "TestApi::setABCDModule called for all modules.  Bank: " << bank << endl;
	for(map<UINT32, ABCDModule>::const_iterator iter = moduleMap.begin(); iter != moduleMap.end(); iter ++) {	
	    setABCDModule(iter->first, bank);
	}
	return;
    }
    
    cout << "SctTestApi::setABCDModule called, mid: " << mid << " bank: " << bank <<endl;
    DummyHardware::instance().setConfig(mid, bank, *lookupConfig(mid));
}

void TestApi::sendABCDModule(UINT32 mid, BankType bank, UINT32 type) {
    if(mid==0xffffffff) {
	cout <<  "TestApi::sendABCDModule called for all modules.  Bank: " << bank << " Type:" << type << endl;
	for(map<UINT32, ABCDModule>::const_iterator iter = moduleMap.begin(); iter != moduleMap.end(); iter ++) {	
	    sendABCDModule(iter->first, bank, type);
	}
	return;
    }
    
    cout << "SctTestApi::sendABCDModule called, mid: " << mid << " bank: " << bank << " type: " << type << endl;
    DummyHardware::instance().sendConfig(mid, bank);
}

void TestApi::sendAllABCDModules(BankType bank, UINT32 type) {
    cout << "SctTestApi::sendAllABCDModules called, bank: " << bank << endl;
    for(map<UINT32, ABCDModule>::const_iterator iter = moduleMap.begin(); iter != moduleMap.end(); iter ++) {	
	sendABCDModule(iter->first, bank, type);
    }
}

//Scan methods

bool TestApi::preScanHardwareCheck(Scan& scan, ScanEx& extra) {

    return true;
}

/**
A handly little routine that counts the number of bits in an unsigned int
*/
unsigned int countBits(unsigned int bits) {
    unsigned int count = 0;
    while (bits) {
    if (bits & 1) ++count;
        bits >>= 1;
    }
    return count;
}

unsigned int countBits(char bits) {
    unsigned int count = 0;
    while (bits) {
    if (bits & 1) ++count;
        bits >>= 1;
    }
    return count;
}

double TestApi::calculateScanTime(const ScanEx& ex) {
    unsigned int maxModules = 0;
    bool multiDsp = false;

    //Loop over all RODs
    for (ScanEx::RodInfoMap::const_iterator i=ex.rodInfo.begin(); i!=ex.rodInfo.end(); ++i) {
        RodScanEx r = i->second;
        if (countBits(r.bitFieldDSP) > 1) multiDsp = true;
        //Loop over DSPs
        for (unsigned int dsp=0; dsp<4; ++dsp) {
            // The second 4 "dsp" groups are for the second scan point set
            unsigned int nModules = countBits(r.groupChannels[dsp].mask0) + countBits(r.groupChannels[dsp].mask1) 
                                  + countBits(r.groupChannels[dsp + 4].mask0) + countBits(r.groupChannels[dsp + 4].mask1);
            if (nModules > maxModules) maxModules = nModules;
        }
    }
    multiDsp = true; 	///@note Force multiDsp for now.
    if (multiDsp) return Configuration::instance().getScanTime() * maxModules / (4*Configuration::instance().getMultiDspEfficiency());
    else return Configuration::instance().getScanTime() * maxModules;
}

int TestApi::pollHistogramming(ScanEx &ex, int bins, int events, int timeout) {
    cout << "SctTestApi::pollHistogramming called with " << bins << " bins." << endl;

    int currBin = 0;

    ISInfoDictionary& isDict = SctNames::getISDictionary();

    ISInfoInt binNumber(0);
    isDict.insert("SCTAPIServer.currentBin", binNumber);

    ISInfoInt maxNumber(bins-1);
    isDict.insert("SCTAPIServer.maxBin", maxNumber);

    //Calculate scan time:
    double scanTime = calculateScanTime(ex);
    for (int slept=0; slept<scanTime+1; ) {
        slept += 1 - sleep(1);

        currBin = (int)((double)slept/scanTime * bins);

        ISInfoInt binNumber(currBin);
        isDict.update("SCTAPIServer.currentBin", binNumber);
    }

    isDict.remove("SCTAPIServer.currentBin");
    isDict.remove("SCTAPIServer.maxBin");
    
    return 0;
}

char* TestApi::readHistogramRawData(const Scan& scan, const ScanEx& extra, UINT32 mid, int frame) {
    cout << "SctTestApi::readHistogramRawData called for module: " << mid << endl;
    try {
        string serialNum = convertToString(mid);
        string dataFile = ScanMap::instance().getDataFile(serialNum, DummyHardware::instance().getCurrentConfig(mid), scan.getScanVariable1());
        //cout << "Datafile: " << dataFile << endl;

        SctDaqRootFile file(dataFile);
        string acSerialNum = ScanMap::convertSerial(serialNum);
        //cout << "Actual Serial  " << acSerialNum << endl;
        int cycleNum = file.getCycleNum(acSerialNum);
        //cout << "Cycle: " << cycleNum << endl;

        auto_ptr<TH1F> hist0Data = file.getHistData(cycleNum, 0);
        auto_ptr<TH1F> hist1Data = file.getHistData(cycleNum, 1);
        auto_ptr<TH1F> triggerData = file.getTriggerData();

        file.dealWithOccupancy(*hist0Data, *hist1Data, *triggerData, scan.getScanVariable1());

        modifyScanInformation(scan, *triggerData);

        if (scan.getOption(Scan::BITS32))
            return file.fillData32(*hist0Data, *hist1Data);
        else
            return file.fillData16(*hist0Data, *hist1Data);

    } catch (SctTest::SctTestException& ioe) {
        cout << "Catch" << endl;
        ioe.sendToMrs(MRS_DIAGNOSTIC);

        int nBins = scan.getScanPoints1().size();
        int binSize = 0x800*4;    // In bytes, so works for 16bit histograms er?
        char *data = new char[nBins * binSize];
        for (int i=0; i<nBins * binSize; ++i) {
            data[i] = 0;
        }
        return data;
    }
}

void TestApi::modifyScanInformation(const Scan& c_scan, const TH1F& triggerData) {
    Scan& scan = const_cast<Scan&>(c_scan);
    Scan::ScanPoints points;
    for (int i=1; i<=triggerData.GetNbinsX(); ++i) {
        points.push_back(triggerData.GetBinCenter(i));
    }
    scan.setScanPoints1(points);
    scan.setNTrigs(triggerData.GetBinContent(1));
}


}
