#ifndef SCTDATA_TRIMRANGETESTRESULT_H
#define SCTDATA_TRIMRANGETESTRESULT_H
#include <boost/shared_ptr.hpp>
#include <cmath>

#include "TestResult.h"
#include "Sct/Streamable.h"
#include "Stat.h"

class TGraph;

namespace SctData{

/**
   Holds the results of the trim range test. Namely: 
   <p> TrimRangeTestResult::ChipTrim objects, containing TrimRangeTestResult::Trim s for the best trim.
   <BR>(each of which has the value of the trim, vthr and trim target).
   <p> TrimRangeTestResult::ChipTrimData objects, containing TrimRangeTestResult::TrimData s.
   <BR>(each of which has a representation of the graph and fit to the trim vs vt50 graph.
   @author Alan Barr
   @date 21 July 2003
 */
    
class TrimRangeTestResult : public TestResult {
public:
    TrimRangeTestResult() throw();
    
    /// Constructor.
    TrimRangeTestResult(unsigned int runNumber, const string& moduleName) throw();
    
    // Destructor
    virtual ~TrimRangeTestResult() throw();
    
    //Override
    virtual string getClassName() const throw();
    
    /** Represents the trim setting for an individual channel for a particular trim target. */
    class Trim : public Sct::Streamable {
    public:
	Trim(const short unsigned trim, const float vthr=0.) throw() : trim(trim), vthr(vthr) {;}
	/** trim 0->15 */
	short unsigned trim;
	/** trimmed value of threshold in mV */
	float vthr;
	
	Trim& operator+= (const Trim& t) throw()    {trim+=t.trim ; vthr+=t.vthr; return *this;}
	Trim& operator/= (const unsigned i) throw()  {trim/=i;       vthr/=i;      return *this;}
	Trim operator- (const Trim& t) const throw() {return Trim(trim-t.trim , vthr-t.vthr); }
	Trim operator* (const Trim& t) const throw() {return Trim(trim*t.trim , vthr*t.vthr); }
	Trim() throw() : trim(0), vthr(0.) {;}
        virtual string getClassName() const throw();
    };
    
    /** 
	The whole trim setting for a chip at a particular trim target.
	Example usage :<BR>ChipTrim.setting[iChannelInChip]=(Trim(3,140.), true)<BR>
	Where 3 is the trim, 140. is the threshold, and true indicates that it can be trimmed.
    */
    class ChipTrim : public Sct::Streamable {
    public:
	ChipTrim() throw() : channelTrim(nChannelChip), target(0) {;}
	/** pointer to the trims of each channel */
	Stats<Trim> channelTrim;
	/** the trim target with which this trim was produced */
	float target;
	/** the trim range with which this trim was produced */
	short range;
	virtual string getClassName() const throw();
    };
    
    
    /** 
	Contains (representation of a) graph of TrimDAQ against target, 
	and fitparameters, p0, and p1, such that trim=(int) p0 + target * p1
    */
    class TrimData : public Sct::Streamable {
    public:
	TrimData(float p0, float p1) throw() : p0(p0), p1(p0) {;}
	/** representation of the graph . first is x, second is y*/
	vector<pair<float, float> > graph;
	/** offset of slope to graph */
	float p0;
	/** slope of fit to graph */
	float p1;
	/** convert trim value to threshold in mV */
	float getVthr(const short unsigned trim) const throw();
	/** convert threshold in mV to integer value of trim */
	int getTrim(const float vthr) const throw();

	TrimData& operator+= (const TrimData& d) throw()    { p0+=d.p0; p1+=d.p1; return *this;}
	TrimData& operator/= (const unsigned i) throw()     { p0/=i;    p1/=i;    return *this;}
	TrimData operator* (const TrimData& d) const throw() { return TrimData(p0*d.p0, p1*d.p1);}
	TrimData() throw() : p0(0), p1(0) {;}
	virtual string getClassName() const throw();
    };
    
    /** Represents all the data for a single chip and a single trim range */
    class ChipTrimData : public Sct::Streamable {
    public:
	ChipTrimData() throw() : 
	    channelData(nChannelChip) {;}
	
	/** Get all values of p1, ie. */
	const Stats<double> getOffsets() const throw() ;
	/** Get all values of 1/p1, i.e. the mV step from changing trim by one unit */
	const Stats<double> getSteps() const throw() ;
	/** Add a list of defects so that serious ones can be ignored */
	void addDefects(shared_ptr<DefectList> list) {fitDefects=list ;}
	Stats<TrimData> channelData;
	/** module defect list */
	shared_ptr<DefectList> fitDefects;
	virtual string getClassName() const throw();
    };

    vector<shared_ptr<const ChipTrimData> > chipTrimData;
    vector<shared_ptr<const ChipTrim> > chipTrim;

    float charge;              ///< injected charge
    short signed type;         ///< range 0->3 or -1 for optimum 
    short unsigned algorithm;  ///< index indicates which chip-to-chip variations allowed in TrimRangeAlgorithm
};// end of TrimRangeTestResult class


//INLINES
    inline float TrimRangeTestResult::TrimData::getVthr(const short unsigned trim) const throw() {
	return (trim-p0) / p1;
    }

    inline int TrimRangeTestResult::TrimData::getTrim(const float vthr) const throw() {
	return static_cast<int>(floor((p0 + vthr * p1) + 0.5));
    }



} // end of namespace SctData

#endif //#ifndef SCTDATA_TRIMRANGETESTRESULT_H
