#ifndef SCT_IOEXCEPTIONS_H
#define SCT_IOEXCEPTIONS_H

#include "Exception.h"

namespace Sct {

/**
 * Thrown when an IO error has occured
 * e.g. file cannot be opened. 
 * @ingroup Exceptions     
 * @author Alan Barr
 */
class IoException : public Exception {
public:
    /**
      Creates an Exception with message msg and no cause
      */
    IoException(const string& msg, const string& file, int line) throw();
    
    /**
      Creates an Exception with a cause.  Useful for simply wrapping
      exceptions.
      */
    IoException(Throwable& cause, const string& file, int line) throw();
    
    /**
      Creates an Exception with a message and a cause
      */
    IoException(const string& msg, Throwable& cause, const string& file, int line) throw();
    
protected:
    IoException() throw() {}        
};    

///To maintain backward compatibility
typedef IoException IoError;

/**
 * Thrown to indicate an error from the Information Service
 * @ingroup Exceptions     
 * @author Alan Barr
 * @todo Add meaningful information here.
 */
class IsException : public IoException {
public:
    /**
      Creates an Exception with message msg and no cause
      */
    IsException(int status, const string& msg, const string& file, int line) throw();
    
    /**
      Creates an Exception with a cause.  Useful for simply wrapping
      exceptions.
      */
    IsException(int status, Throwable& cause, const string& file, int line) throw();
    
    /**
      Creates an Exception with a message and a cause
      */
    IsException(int status, const string& msg, Throwable& cause, const string& file, int line) throw();
    
    ///Get the status code that IS returned
    int getStatus() const throw();
    virtual shared_ptr<Throwable> clone() const throw();
    virtual string getMessage() const throw();
    
protected:
    IsException() throw() {}
    
    int status;
};


/**
 * Thrown to indicate a file IO error
 * @ingroup Exceptions     
 * @author Alan Barr
 */
class FileException : public IoException {
public:    
    /**
      Creates an Exception with message msg and no cause
      */
    FileException(const string& fileName, const string& msg, const string& file, int line) throw();
    
    /**
      Creates an Exception with a cause.  Useful for simply wrapping
      exceptions.
      */
    FileException(const string& fileName, Throwable& cause, const string& file, int line) throw();
    
    /**
      Creates an Exception with a message and a cause
      */
    FileException(const string& fileName, const string& msg, Throwable& cause, const string& file, int line) throw();
    
    ~FileException() throw() {}
    
    ///Get the filename of the file being accessed which caused the exception
    const string& getFileName() const throw();
    virtual shared_ptr<Throwable> clone() const throw();
    virtual string getMessage() const throw();
    
protected:
    FileException() throw() {}       
    
    string fileName; ///< the name of the file which had the problem
};

/**
  Thrown to indicate that the stream is somehow not in the state the I/O
  code thought it ought to be in.
  @ingroup Exceptions
  @author Matthew Palmer
  */
class StreamCorruptedException : public IoException {
public:
    /**
      Creates an Exception with message msg and no cause
      */
    StreamCorruptedException(const string& msg, const string& file, int line) throw();
    
    /**
      Creates an Exception with a cause.  Useful for simply wrapping
      exceptions.
      */
    StreamCorruptedException(Throwable& cause, const string& file, int line) throw();
    
    /**
      Creates an Exception with a message and a cause
      */
    StreamCorruptedException(const string& msg, Throwable& cause, const string& file, int line) throw();
    
protected:
    StreamCorruptedException() throw() {}    
};


/**
  Indicates that there is no Streamer for a given className.
  @ingroup Exceptions
  @author Matthew Palmer
  */
class NoSuchStreamerException : public IoException {
public:
    /**
      Creates an Exception with message msg and no cause
      */
    NoSuchStreamerException(const string& className, const string& msg, const string& file, int line) throw();
    
    /**
      Creates an Exception with a cause.  Useful for simply wrapping
      exceptions.
      */
    NoSuchStreamerException(const string& className, Throwable& cause, const string& file, int line) throw();
    
    /**
      Creates an Exception with a message and a cause
      */
    NoSuchStreamerException(const string& className, const string& msg, Throwable& cause, const string& file, int line) throw();
    
    ~NoSuchStreamerException() throw() {}
    
    ///Get the name of the class that there was no streamer for
    const string& getClassName() const throw();
    virtual shared_ptr<Throwable> clone() const throw();
    virtual string getMessage() const throw();
    
protected:
    NoSuchStreamerException() throw() {}   
    
    string className;
};

}

#endif //SCT_IOEXCEPTIONS_H
