#ifndef SCT_IOMANAGER_H
#define SCT_IOMANAGER_H

#include "IoExceptions.h"
#include <map>
#include <boost/shared_ptr.hpp>
#include <boost/utility.hpp>

using std::map;
using boost::shared_ptr;

namespace Sct {

class Streamer;
class Streamable;
class Serializable;
class OStream;
class IStream;

/**
  A tag class.  Sub-classes of IOManager are expected to sub-class
  this if necessary to allow extra parameters to be passed.
  */
class IOParams {
public:
    virtual ~IOParams(){}
};

/**
  An abstract class that represents an I/O backend.  Possibilities are
  raw file, IS, some other form of persistency.  Perhaps even a viewer.
  
  Some methods here deal with a name.  This is a backend specific identifier - 
  e.g. a filename or an IS name.  In order to retrieve a name, a backend specific method
  or an appropriate IOName sub-class should be used.
  
  shared_ptr is returned sometimes, rather than auto_ptr to allow singletons to be returned
  which might be indicated by some numeric ID for instance.
  
  @author Matthew Palmer
  @date 15 July 2003
  */
class IOManager : private boost::noncopyable {
public:
    virtual ~IOManager() {};
    
    ///{
    ///@name Methods for users

    /**
      Write an object.  
      @param ob The Serializable to write
      @param params A backend specific sub-class of IOParams.
      @throw IoException if there is a problem with the I/O
      */
    virtual void write(const Serializable& ob, const IOParams* params = 0) const = 0;

    /**
      Read an object.  Will determine the class by first reading a string assumed to contain the className.
      @param name The name of the object to read
      @param params A backend specific sub-class of IOParams.
      @throw IoException if there is a problem with the I/O
      */
    virtual shared_ptr<Serializable> read(const string& name, const IOParams* params = 0) const = 0;
    ///}


    //{
    ///@name Methods for Streamable implementors
    ///@note These methods are named readImpl and writeImpl to avoid a gcc 2.9x bug.

    /**
      Writes a Streamable to an output stream.
      @param ob the Streamable to write.
      @param out the OStream to write to.
      @param writeClassName if true, write the class name, otherwise don't.
      @throw IoException if something bad happens in the back-end.
      */
    virtual void writeImpl(OStream& out, const Streamable& ob, bool writeClassName) const ;

     /**
      Writes a Streamable to an output stream with a different name - to be used for base classes.
      @param ob the Streamable to write.
      @param out the OStream to write to.
      @param writeClassName if true, write the class name, otherwise don't.
      @throw IoException if something bad happens in the back-end.
      */
    virtual void writeImpl(OStream& out, const Streamable& ob, const string& className) const ;
    
    /** explicitly calls same with string as final argument, to prevent implicit conversion to bool */
    virtual void writeImpl(OStream& out, const Streamable& ob, const char* className) const ;

    /**
      Read a Streamable from an input stream.  Reads the class name from the stream first.
      Corresponds to write with writeClassName=true.
      @param in the IStream to read from
      @return the newly created Streamable
      @throw IoException if something bad happens in the back-end.
      */
    virtual shared_ptr<Streamable> readImpl(IStream& in) const;

    /**
      Read a Streamable from an input stream.
      Corresponds to write with writeClassName=false.
      @param in the IStream to read from
      @param className
      @return the newly created Streamable
      @throw IoException if something bad happens in the back-end.
      */
    virtual shared_ptr<Streamable> readImpl(IStream& in, const string& className) const;
    /** explicit conversion to string for the final argument 
	prevents accidental implicit conversion to bool.
     */
    virtual shared_ptr<Streamable> readImpl(IStream& in, const char* className) const;

    /**
      Read a Streamable from an input stream. 
      @param in the IStream to read from.
      @param ob the object to read into.
      @param readClassName if true, read a string that should match ob.getClassName() first
      @throw IoException if something bad happens in the back-end.
      */
    virtual void readImpl(IStream& in, Streamable& ob, bool readClassName) const;

     /**
      Read a Streamable from an input stream with different classname - to be used for derrived classes.
      @param in the IStream to read from.
      @param ob the object to read into.
      @param readClassName if true, read a string that should match ob.getClassName() first
      @throw IoException if something bad happens in the back-end.
      */
    virtual void readImpl(IStream& in, Streamable& ob, const string& className) const;
    
    /** calls method with explicit string in last arg to prevent implicit conversion to bool */
    virtual void readImpl(IStream& in, Streamable& ob, const char* className) const;

    /**
      Writes a class name to an OStream.  Should only be overriden if absolutely necessary.
      Sub-classes should use this method if ever they need to write a class name and avoid
      proprietry solutions as they may break Java compatability.
      @param out the OStream to be written to
      @param className the class name to write
      @throws IoException if something bad happens in OStream.
    */
    virtual void writeClassName(OStream& out, const string& className) const;
    
    /**
      Read a class name from an IStream.  Should only be overriden if absolutely necessary.
      Sub-classes should use this method if ever they need to write a class name and avoid
      proprietry solutions as they may break Java compatability.
      @param in the IStream to be written to
      @return the className
      @throws IoException if something bad happens in OStream or there is no className to be read      
      */    
    virtual string readClassName(IStream& in) const;    
    
    /**
    Add a Streamer to the map.  The Streamer will be used to do IO on classes named className.
    @param className The name of the class that s can do IO on.
    @param s An Streamer capable of doing IO on classes named className.
    @return True if successful
    */
    static bool addToMap(const std::string& className, std::auto_ptr<Streamer> s) throw();


    //}
protected:
    //{
    ///@name Methods for sub-class authors
    
    /**
      A sub-class can override this if it wants to influence the Streamer that
      will be used for a particular class.
      @return an appropriate Streamer.
      @throw InvalidArgumentError if an appropriate Streamer can't be found.
      */
    virtual Streamer& getStreamer(const string& className) const;
    /** 
	Get a particular verion of a streamer, to be used e.g. by archival when reading data 
	saved with old streamers.
    */

    virtual Streamer& getStreamer(const string& className, const unsigned version) const;
    //}
    typedef map<string, shared_ptr<Streamer> > StreamerMap;
    static StreamerMap& getMap() throw(); ///< map of classname, streamer
    static StreamerMap& getFullVersionMap() throw(); ///<map of mapname (=classname/version), streamer
    static std::string getMapName(const std::string& classname, const unsigned version) throw();
    friend class Streamer;
};

}

#endif //SCT_IOMANAGER_H
