/*
 * SearchClassLoader.java
 *
 * Created on 13 June 2003, 22:21
 */

package Sct;

import java.util.StringTokenizer;
import java.util.jar.*;
import java.util.zip.*;
import java.util.Enumeration;
import java.util.*;
import java.io.*;

/**
 * A SearchClassLoader is a bit different to a normal one
 * It looks for classes with a given name in any package.
 * It caches results by the asked for name and prototype to
 * improve performance.
 * @todo Should we cache failure too????
 * @author  Matthew Palmer
 */
public class SearchClassLoader extends ClassLoader {
    private static SearchClassLoader loader = new SearchClassLoader();
    //A Map of CacheData to Class
    private HashMap cache = new HashMap();
    
    class CacheData {
        String name;
        Class[] prototype;
        
        CacheData(String name, Class[] prototype) {
            this.name = name;
            this.prototype = prototype;
        }                
        public boolean equals(Object ob) {
            CacheData cd = (CacheData) ob;
            if (cd == null) return false;
            boolean equal = name.equals(cd.name) && prototype.length == cd.prototype.length;
            
            if (!equal) return false;
            for (int i=0; i<prototype.length && equal; ++i) {
                equal &= prototype[i].equals(cd.prototype[i]);
            }
            return equal;
        }
        
        public int hashCode() {
            int hash = name.hashCode();
            for (int i=0; i<prototype.length; ++i) {
                hash += prototype[i].hashCode();
            }
            return hash;
        }
    }
    
    public static SearchClassLoader instance() {
        return loader;
    }    

    //Note that it is quicker to search ClassPath first - presumably
    //because you avoid going through all the gazillions of system packages...
    //if necessary, it would be possible to speed that up to I guess - just
    //don't check if it starts with java or sun.
    public Class loadClass(String name, Class[] prototype) throws ClassNotFoundException {
        Class c = findInMap(name, prototype);
        if (c!=null) return c;
        c = findClassInParent(name, prototype);
        if (c == null) c = findClassInClassPath(name, prototype);                         
        if (c == null) c = findClassInPackages(name, prototype); 
        
        if (c != null) {
            addToMap(name, c, prototype);
            return c;
        }
        else throw new ClassNotFoundException("ClassLoaderIS could not find class: " + name);
    }
    
    private void addToMap(String name, Class c, Class[] prototype) {
        CacheData cd = new CacheData(name, prototype);
        cache.put(cd, c);
    }
    
    private Class findInMap(String name, Class[] prototype) {
        CacheData cd = new CacheData(name, prototype);
        return (Class)cache.get(cd);
    }
    
    private boolean checkClass(Class c, Class[] prototype) {
        //System.out.println("Checking class: " + c.getName());
        try {            
            //Will throw if fails.
            c.getConstructor(prototype);
            return true;
        } catch (NoSuchMethodException nsme) {
            //System.out.println("checkClass failed for: " + c.getName());
            return false;
        }
    }
    
    private Class findClassInParent(String name, Class[] prototype) {
        //System.out.println("findClassInParent: " + name);
        try {             
            Class c = loadClass(name);
            if (checkClass(c, prototype)) return c;
        } catch (ClassNotFoundException e) {}
        return null;        
    }
    
    private Class findClassInPackages(String name, Class[] prototype) {
        Class c = null;
        Package[] p = getPackages();            
        int i = 0;
        while (c==null && i<p.length) {                
            c = findClassInParent(p[i].getName() + "." + name, prototype);
            ++i;
        }
        return c;    
    }
    
    //Looks in all jar files in classpath
    ///@note This does not follow additional class paths in Manifest files
    ///@note probably also doesn't work with extension directories.    
    private Class findClassInClassPath(String name, Class[] prototype) {
        Class c = null;
        String classPath = System.getProperty("java.class.path");
        //System.out.println(classPath "+\n");        
        String classFileName = name + ".class";
        StringTokenizer st = new StringTokenizer(classPath, ":");
        
        while (st.hasMoreTokens() && c==null) {
            String cpElement = st.nextToken();
            //System.out.println(cpElement);            
            if (cpElement.endsWith(".jar")) c = findInJarFile(classFileName, cpElement, prototype);
            else if (cpElement.endsWith(".zip")) c = findInZipFile(classFileName, cpElement, prototype);
            else c = findInDir(classFileName, cpElement, prototype);
            //System.out.println("FindClassInClassPath: " + c);
        }    
        
        return c;
    }
    
    private Class findInDir(String fileName, String dirName, Class[] prototype) {
        File f = new File(dirName);
        //System.out.println("DirName: " + dirName);
        if (!f.isDirectory()) return null;
        return findInDir(fileName, f.getAbsolutePath(), f, prototype);
    }    

    private Class findInDir(String fileName, String parentDir, File dir, Class[] prototype) {
        File[] files = dir.listFiles(new MyFileNameFilter(fileName));
        
        if (files.length > 0) {            
            String className = files[0].getAbsolutePath();
            //Make sure we remove the /!
            className = className.substring(parentDir.length() + 1, className.length() - 6);
            //System.out.println("findInDir candidate: " + className);
            Class c = findClassInParent(className, prototype);
            if (c!=null) return c;
        }
        
        //loop to find class
        files = dir.listFiles();
        Class c;
        for (int i=0; i<files.length; ++i) {
            if (files[i].isDirectory()) {
                c = findInDir(fileName, parentDir, files[i], prototype);
                if (c!=null) return c;
            }
        }
        return null;
    }
    
    private class MyFileNameFilter implements FilenameFilter {
        private String name;
        MyFileNameFilter(String name) {
            this.name = name;
        }
        
        public boolean accept(File dir, String name) {
            return this.name.equals(name);
        }
    }

    
    private Class findInZipFile(String classFileName, String cpElement, Class[] prototype) {
        ZipFile f = null;
        try {
            f = new ZipFile(cpElement); 
            return findInZipFile(f, classFileName, prototype);
        } catch (IOException e) {
        } finally {
            try {
                if (f != null) f.close();
            } catch (IOException e) {}
        } 
        return null;    
    }
    
    private Class findInJarFile(String classFileName, String cpElement, Class[] prototype) {
        JarFile f = null;
        try {
            f = new JarFile(cpElement); 
            return findInZipFile(f, classFileName, prototype);
        } catch (IOException e) {
        } finally {
            try {
                if (f != null) f.close();
            } catch (IOException e) {}
        } 
        return null;
    }
    
    private Class findInZipFile(ZipFile f, String classFileName, Class[] prototype) {
        Enumeration e = f.entries();
        while (e.hasMoreElements()) {                
            String entry = ((ZipEntry)e.nextElement()).getName();
            if (entry.endsWith("/" + classFileName)) {                   
                String className = entry.substring(0, entry.length() - 6);
                className.replace('/', '.');
                Class c = findClassInParent(className, prototype);
                if (c != null) return c;
            }
        }                               
        return null;
    }

     
    public static void main(String[] args) {
        find("Test");
        find("SctControl.Test");
        find("SctControl/Test");
        find("SctControl/Test.class");
    }
     
    public static void find(String name) {        
        System.out.println("\n\nSearching for:" + name +"\n");
        SearchClassLoader cl = SearchClassLoader.instance();
        Class[] argTypes = {IStream.class, ObjectManager.class};
        try {
            Class c = cl.loadClass(name, argTypes);
            System.out.println("Success!!! : " + c.getName());
        } catch (ClassNotFoundException e) {
            System.out.println(e.getMessage());            
        }         
    }
}
