/*
 * TestObjectManager.java
 *
 * Created on 16 September 2003, 10:49
 */

package Sct.TestFramework;
import Sct.*;
import java.io.IOException;
import java.io.StreamCorruptedException;
import java.io.InvalidClassException;
import java.io.FileNotFoundException;
import java.io.InvalidObjectException;
import java.util.*;

/**
 * Extends AbstractObjectManager to get the createObject method.
 * Should be used first to write then to read the same object
 * @author  Matthew Palmer
 */
public class TestObjectManager extends AbstractObjectManager {
    
    private LinkedList list = new LinkedList();
    private String uniqueID;
    private String className;
       
    /**
     * Assumes that name is "<uniqueID>$<className>".  Then calls readObject(uniqueID, className)
     */
    public Serializable readObject(String name) throws IOException {
        int index = name.indexOf('$');
        if (index == -1) throw new FileNotFoundException("Invalid name - expected $ separator: " + name);
        return readObject(name.substring(0, index), name.substring(index+1));
    }
        
    public Serializable readObject(String uniqueID, String className) throws IOException {
        if (!this.uniqueID.equals(uniqueID))
            throw new StreamCorruptedException("Incorrect uniqueID.\nWas asked for uniqueID: " + uniqueID + "\tClassName: " 
                                                    + className + "\nHave className: " + this.className + "\tuniqueID: " + this.uniqueID);

        Serializable s = (Serializable)readObject(null, "PrimarySerializableObject", className);
        
        //Check against className actually returned.
        if (!this.className.equals(s.getClassName()))
            throw new InvalidClassException(className, "Incorrect className.\nWas asked for uniqueID: " + uniqueID + "\tClassName: " 
                                                    + className + "\nHave className: " + this.className + "\tuniqueID: " + this.uniqueID + 
                                                    "\nGot class: " + s.getClass().getName()); 
        
        return s;
    }    
    
    public Streamable readObject(IStream s, String name)  throws IOException{
        String className = readClassName(s);
        return readObject(s, name, className);
    }
    
    public Streamable readObject(IStream s, String name, String className) throws IOException{
        try {
            ClassData data = (ClassData)list.removeFirst();
            if (!data.name.equals(name)) 
                throw new StreamCorruptedException("Incorrect name.\nWas asked for className: " + className + "\tName: " 
                                                    + name + "\nHave className: " + data.className + "\tName: " + data.name);            
                        
            TestIStream is = new TestIStream(data.data);
            Streamable ob = createObject(is, className);        
            
            //Check className of object returned not name asked for in case using the static method.
            if (!data.className.equals(ob.getClassName())) 
                throw new InvalidClassException(className, "Incorrect className.\nWas asked for className: " + className + "\tName: " 
                                                    + name + "\nHave className: " + data.className + "\tName: " + data.name + 
                                                    "\nGot className: " + ob.getClassName());
            
            return ob;
            
        } catch (NoSuchElementException nsee) {
            throw new StreamCorruptedException("Tried to read in an object that wasn't written.\nName: " + name + "\nClassName: " + className);
        }
    }
    
    public void writeObject(OStream s, String name, Streamable object, boolean writeClassName) throws IOException{
        //Check non-null
        if (object == null) throw new InvalidObjectException("Cannot write a null object.\nName: " + name);
        
        //Write className to parent stream.
        if (writeClassName) writeClassName(s, object.getClassName());
        
        TestOStream os = new TestOStream();
        ClassData data = new ClassData();
        data.className = object.getClassName();
        data.name = name;
        data.data = os.getData();
        list.add(data);
                
        object.write(os, this);
    }
        
    /** 
     * Break the return contract!
     */
    public Name writeObject(Serializable object) throws IOException {
        uniqueID = object.getUniqueID();
        className = object.getClassName();
        writeObject(null, "PrimarySerializableObject", object, false);
        return null;    
    }
    
    
    public Streamable readStreamable() throws IOException {
        return readObject(null, "PrimaryStreamableObject", className);        
    }
    
    public void writeStreamable(Streamable object)  throws IOException {
        className = object.getClassName();
        writeObject(null, "PrimaryStreamableObject", object, false);
    }
    
    
    private class ClassData { 
        String className;
        String name;
        TestStreamData data;
    }
}
