// Test program to send a single ECHO primitive to a slave DSP
//
// The output consists of a buffer of three primitives:
//    1. An ECHO reply from the MDSP with "beef" data
//    2. A reply from a SEND_SLAVE_LIST primitive with data in the following format:
//         a. a primitive list with one primitive
//         b. an ECHO reply from the slave with "feed" data
//    3. An ECHO reply from the MDSP with "cafe" data

#include <iostream>
using namespace std;

#include <ctype.h>

#include "RodModule.h"

#include "primParams.h"
#include "RCCVmeInterface.h"
#include "parameters.h"

int main(int argc, char *argv[]) {

using namespace SctPixelRod;

  int slaveNumber=-1;
  RodPrimList primList(1);                      // Primitive List
  long myTextLength;                            // Actual length of text message
  TEXT_BUFFER_TYPE myTextType;                  // Buffer type for latest message
  PrimState returnPState;
  TextBuffState returnTState;
  std::string ipramFile("../Dsp/Binary/slaveRun_IPRAM_exp.bin");
  std::string idramFile("../Dsp/Binary/slaveRun_IDRAM_exp.bin");
  std::string extFile("../Dsp/Binary/slaveRun_xcode_exp.bin");
  std::string fileName(""), option;
  bool initRod = true;
  int slot = -1;

  unsigned long baseAddress;
  
  if (argc > 1) {
    for (int i=1; i<argc; i++) {
      option = argv[i];
      if (option[0] != '-') break;
      switch (option[1]) {
        case 'n': {
          initRod = false;
          break;
        }
        case 's': {
          slot = atoi(option.substr(2).c_str());
          break;
        }
        case 'v': {
          slaveNumber = atoi(option.substr(2).c_str());
          break;
        }
        default: {
          break;
        }
      }
    }
  }
// Prompt for slot number
  if (slot < 0 ) {
    cout << "Enter slot number (decimal):"; 
    cin >> slot;
    while ((slot < 1) || (slot > 21)) {
      cout << "Slot number out or range [1:21], re-enter: ";
      cin >> slot;
    }
  }
  baseAddress = slot << 24;

// Create VME interface
  RCCVmeInterface *vme1 = new RCCVmeInterface();
  
// Create RodModule and initialize it
  RodModule* rod0 = new RodModule(baseAddress, mapSize, *vme1, numSlaves);
  if (initRod) {
    try{
      rod0->initialize();
    }
    catch (HpiException &h) {
      cout << h;
    }
    catch (VmeException &v) {
      cout << "VmeException creating RodModule." << endl;
      cout << "ErrorClass = " << v.getErrorClass() << endl;
      cout << "ErrorCode = " << v.getErrorCode() << endl; 
    }
    catch (RodException &r) {
    cout << r;
    }
  }

// Initialize slave
  if (slaveNumber < 0 ) {
    cout << "Enter slave number (0-3):"; 
    cin >> slaveNumber;
    while ((slaveNumber < 0) || (slaveNumber > 3)) {
      cout << "Slave number out or range [0:3], re-enter: ";
      cin >> slaveNumber;
    }
  }

  try {
    rod0->initSlaveDsp(ipramFile, idramFile, extFile, slaveNumber, 'v');
  }
  catch (RodException & r) {
  cout << r.getDescriptor() <<", " << r.getData1() << ", " << r.getData2()
          << '\n';
  };

// Create a buffer for text messages
  char * myTextBuffer;
  try {
    myTextBuffer= new char[TEXT_BUFF_SIZE];
  }
  catch (std::bad_alloc) {
    cout << "Unable to allocate text buffer in main.";
  }
  
// Create and Send an ECHO primitive to a slave DSP

  long sdataLength;
//  cout << "Enter number of data words for Slave DSP: ";
//  cin >> sdataLength;
  sdataLength = 16;
  long* echoData;
  try {
    echoData = new long[sdataLength];
  }
  catch (std::bad_alloc) {
    cout << "Unable to allocate echoData in main.";
  }
  
  for (long i=0; i<sdataLength; i++) {
    echoData[i] = 0xfeed0000 | i;
  }
  RodPrimitive* echo;
  try {
    echo= new RodPrimitive(sdataLength+4, 1, ECHO, R_ECHO, echoData);
  }
  catch (std::bad_alloc) {
    cout << "Unable to allocate Slave DSP ECHO primitive in main.";
  }

  primList.push_back(*echo);
  try {
    primList.bufferBuild();
  }
  catch (PrimListException &p) {
    cout << p.getDescriptor() << " ";
    cout << p.getData1() << ", " << p.getData2() << "\n";
  };
  
// Save a copy of primlist buffer to use as data in SEND_SLAVE_LIST and then clear buffer
  long buffLength = primList.getBufferLength();
  unsigned long* slavePrimList;
  try {
    slavePrimList = new unsigned long[buffLength];
  }
  catch (bad_alloc) {
    cout << "Unable to allocate slavePrimList in main." << endl;
  };
  
  unsigned long* bufferStart = primList.getBuffer();
  for (int i=0; i< buffLength; i++) {
    slavePrimList[i] = bufferStart[i];
  }
  primList.clear();
  delete [] echoData;
  delete echo;
  
// Create 1st simple ECHO primitive to the MDSP
  long dataLength;
  dataLength = 16;
  long* mechoData;
  try {
    mechoData = new long[dataLength];
  }
  catch (std::bad_alloc) {
    cout << "Unable to allocate mechoData in main.";
  }
  
  for (long i=0; i<dataLength; i++) {
    mechoData[i] = 0xbeef0000 | i;
  }
  RodPrimitive* mecho;
  try {
    mecho= new RodPrimitive(dataLength+4, 1, ECHO, R_ECHO, mechoData);
  }
  catch (std::bad_alloc) {
    cout << "Unable to allocate Master DSP ECHO primitive in main.";
  }

  primList.push_back(*mecho);
  
// Use SEND_SLAVE_LIST/START_SLAVE_LIST primitives to have MDSP send "feed" ECHO to the 
// slave
  long slaveData[4+buffLength]; 
  slaveData[0] = slaveNumber;
  slaveData[1] = buffLength;
  slaveData[2] = DEFAULT;
  slaveData[3] = DEFAULT;
  for (int i=0; i< buffLength; i++) {
    slaveData[i+4] = slavePrimList[i];
  }
  RodPrimitive* send0;
  try {
    send0 = new RodPrimitive(8+buffLength, 0, SEND_SLAVE_LIST, 
                R_SEND_SLAVE_LIST, slaveData);
  }
  catch (bad_alloc) {
    cout << "Unable to allocate send0 primitive in main." << endl;
  };
  
  primList.push_back(*send0);
  long startData[3] = {slaveNumber, 1, 1};
  RodPrimitive* start0;
  try {
    start0 = new RodPrimitive(7, 0, START_SLAVE_LIST, 
                R_START_SLAVE_LIST, startData);
  }
  catch (bad_alloc) {
    cout << "Unable to allocate start0 primitive in main." << endl;
  };
  primList.push_back(*start0);

// Create 2nd simple ECHO primitive to the MDSP
  long* mechoData2;
  try {
    mechoData2 = new long[dataLength];
  }
  catch (std::bad_alloc) {
    cout << "Unable to allocate mechoData2 in main.";
  }
  
  for (long i=0; i<dataLength; i++) {
    mechoData2[i] = 0xcafe0000 | i;
  }
  RodPrimitive* mecho2;
  try {
    mecho2= new RodPrimitive(dataLength+4, 1, ECHO, R_ECHO, mechoData2);
  }
  catch (std::bad_alloc) {
    cout << "Unable to allocate Master DSP ECHO primitive in main.";
  }

  primList.push_back(*mecho2);

  try {
    primList.bufferBuild();
  }
  catch (PrimListException *p) {
    cout << p->getDescriptor() << " ";
    cout << p->getData1() << ", " << p->getData2() << "\n";
  };
  delete [] mechoData;
  delete [] mechoData2;

  try {
    rod0->sendPrimList(&primList);
  }
  catch (HpiException *h) {
    hex(cout);
    cout << h->getDescriptor() << '\n';
    cout << "calcAddr: " << h->getCalcAddr() << ", readAddr: " << 
            h->getReadAddr() << '\n';
    dec(cout);
  }
  catch (VmeException &v) {
    cout << "VmeException in SendPrimList." << endl;
    cout << "ErrorClass = " << v.getErrorClass() << endl;
    cout << "ErrorCode = " << v.getErrorCode() << endl; 
  };

// Wait for ROD to begin executing and then wait for it to finish executing
// Check for error messages in text buffer and read them out if they exist.
// Note: this is NOT how primHandler and textHandler will be used once we
// go to threads.  This is for debugging the code only.
  do {
    try {
      returnPState = rod0->primHandler();
    }
    catch (VmeException &v) {
      cout << "VmeException in first primHandler call." << endl;
      cout << "ErrorClass = " << v.getErrorClass() << endl;
      cout << "ErrorCode = " << v.getErrorCode() << endl; 
    }
    try {
      returnTState = rod0->textHandler();
    }
    catch (VmeException &v) {
      cout << "VmeException in first textHandler call." << endl;
      cout << "ErrorClass = " << v.getErrorClass() << endl;
      cout << "ErrorCode = " << v.getErrorCode() << endl; 
    }
    if (returnTState == TEXT_RQ_SET) {
      do {
        try {
          returnTState = rod0->textHandler();
          }
        catch (VmeException &v) {
          cout << "VmeException in second textHandler call." << endl;
          cout << "ErrorClass = " << v.getErrorClass() << endl;
          cout << "ErrorCode = " << v.getErrorCode() << endl; 
        }
      } while (returnTState != TEXT_READOUT);
      rod0->getTextBuffer(myTextBuffer, myTextLength, myTextType);
      rod0->clearTextBuffer();
      for (int i=0; i<myTextLength; i++) {
        cout << myTextBuffer[i];
        if (0==(i+1)%64) cout << endl;
        }
      cout << endl; 
    }
  } while (returnPState != PRIM_EXECUTING); 
  do {
    try {
      returnPState = rod0->primHandler();
    }
    catch (RodException *r) {
    cout << r->getDescriptor() <<", " << r->getData1() << ", " << r->getData2()
            << '\n';
    }
    catch (VmeException &v) {
      cout << "VmeException in second primHandler call." << endl;
      cout << "ErrorClass = " << v.getErrorClass() << endl;
      cout << "ErrorCode = " << v.getErrorCode() << endl; 
    }
  } while ((returnPState != PRIM_WAITING)&&(returnPState != PRIM_IDLE)); 
    
// Retrieve output buffer
  RodOutList* outList = rod0->getOutList();
  
// Print results (User processing of outList)
  UINT32 outLength = UINT32(outList->getLength());
  unsigned long* outBody = outList->getBody();
  UINT32 outIndex = UINT32(outBody[1]);
  UINT32 outNumPrims = outBody[2];
  UINT32 outPrimVersion = outBody[3];
  cout << "outLength = " << outLength << ", outIndex = " << outIndex << 
          ", outNumPrims = " << outNumPrims << ", outPrimVersion = " << 
          outPrimVersion <<'\n';
  int outPtr = 4;
  for (UINT32 j=0; j<outNumPrims; j++) {
    UINT32 primLength = outBody[outPtr++];
    UINT32 primIndex = outBody[outPtr++];
    UINT32 primId = outBody[outPtr++];
    UINT32 primVersion = outBody[outPtr++];
    cout << "primLength = " << primLength << ", primIndex = " << primIndex << 
            ", primId = 0x" << hex << primId << ", primVersion = " << primVersion << '\n';
    cout << "Primitive data: " << endl;
    hex(cout);
    for (UINT32 i=0; i<primLength-4; i++) {
      cout.width(8);
      cout << outBody[i+outPtr] <<" ";
      if (0 == (i+1)%8) cout << endl; 
    };
    outPtr += primLength-4;
    if (0 != (primLength-4)%8) cout << endl;
    dec(cout);
  };
  

// Clean up: clear primList, delete primitive, delete the outList
  primList.clear();
  delete send0;
  delete start0;
  delete mecho;
  delete mecho2;
  delete [] slavePrimList;
  delete [] myTextBuffer;
  rod0->deleteOutList();
  
// Delete the ROD and VME objects before exiting
  delete rod0;
  delete vme1;

  return 0;  
}


