//File: TimModule.cxx

/*! \file
 * \brief TimModule: A derived class for VME TIM modules.
 *
 * This is the implementation of a TIM class derived from the VmeModule base
 * class. It should be the sole interface for VME communication with TIM.
 *
 * Contributors: John Lane <jbl@hep.ucl.ac.uk> - originator
 *
 * $Id: TimModule.cxx,v 1.6 2004/04/01 13:26:04 gallop Exp $
 *
 * $Log: TimModule.cxx,v $
 * Revision 1.6  2004/04/01 13:26:04  gallop
 * BJG 1/4/04
 *  Restore decimal mode after status
 *
 * Revision 1.5  2003/12/04 19:10:49  jbl
 * TimModule uses BaseException
 *
 * Revision 1.4  2003/05/20 19:26:25  jbl
 * TimModule UINT8 & UINT16
 *
 * Revision 1.3  2002/12/11 21:30:50  jbl
 * TimModule major update
 *
 */

#include <iostream>  // for debug only

using namespace std;

#include "TimModule.h"

// Namespace for the common routines for SCT and PIXEL ROD software.

namespace SctPixelRod {

// ************************* Class TimException ******************************

//! Constructors. Use defaults for destructor, copy, and assignment.

TimException::TimException( std::string descriptor, int data1, int data2 ) :
             BaseException( descriptor ) {

  m_data1 = data1;
  m_data2 = data2;
  setType( TIM );
}

void TimException::what( std::ostream& os ) {

  os << "TimException: " << getDescriptor() << endl;
  os << "Data1: " << getData1() << endl;
  os << "Data2: " << getData2() << endl;
}

// ************************* Class TimModule *********************************

// Constructors, destructor, and assignment

// ------------------------- Constructor -------------------------------------

//! This is the only constructor to use.
/*! Objects of this class cannot be copied or assigned.
 *  This could lead to VME conflicts.
 */

TimModule::TimModule( UINT32 baseAddr, UINT32 mapSize, VmeInterface & vme ) :
	   VmeModule( baseAddr, mapSize, vme ) {

  m_serialNumber = 0xFFFFFFFF;  // Set in initialize

  if (&vme == 0) throw TimException( "TimModule has no VmeInterface", 0, 0 );

  try {
    m_vmePort = new VmePort( baseAddr, mapSize, VmeInterface::A32, vme );

    m_vmePort->setExceptionTrapping( true );
  }
  catch (bad_alloc) {
    throw TimException( "TimModule caught bad_alloc", 0, 0 );
  }
  catch (VmeException x) {
    throw TimException( "TimModule caught VmeException",
                        x.getErrorClass(), x.getErrorCode() );
  }
}

// ------------------------- Destructor --------------------------------------

/*! This deletes all its VME Ports.
 */

TimModule::~TimModule() {

  delete m_vmePort;
  m_vmePort = 0;
}

// ---------------------------------------------------------------------------

// Member methods

// ------------------------- initialize --------------------------------------

/*! This method configures the TIM into its initialized state.
 */

void TimModule::initialize() {

  // enable serial output streams
  regLoad( TIM_REG_RUN_ENABLES, TIM_BIT_EN_ID | TIM_BIT_EN_TYPE );
  regLoad( TIM_REG_COMMAND, 0 );                    // ensure stand-alone mode
  regLoad( TIM_REG_TRIGGER_BCID, TIM_BCID_OFFSET << 12 );  // set BCID offset
  fetchTimID();                                          // read serial number
}

// ------------------------- reset -------------------------------------------

/*! This method issues a reset to the TIM.
 */

void TimModule::reset() {

  regLoad( TIM_REG_COMMAND, TIM_BIT_VRESET );
}

// ------------------------- status ------------------------------------------

/*! This method reports the status of the TIM.
 *  For now, it simply prints to standard output.
 */

void TimModule::status() {

  cout << "status" << endl;

  cout << " Serial Number: " << m_serialNumber;
  cout << " Version: "       << m_firmware;
  cout << endl;
  hex(cout);
  cout << " L1ID: "   << fetchL1ID();
  cout << " BCID: "   << regFetch( TIM_REG_TRIGGER_BCID );
  cout << " status: " << regFetch( TIM_REG_STATUS );
  cout << endl;
  dec(cout);
}

// ------------------------- fetchL1ID ---------------------------------------
/*! This method reads the last TIM L1ID value.
 */

UINT32 TimModule::fetchL1ID() {

  const UINT32 lo = regFetch( TIM_REG_TRIGGER_IDLO );
  const UINT32 hi = regFetch( TIM_REG_TRIGGER_IDHI );
  UINT32 l1id;

  if (m_firmware < 9) l1id = (lo & 0x0FFF) + ((hi & 0x0FFF) << 12);
  else                l1id = (lo & 0xFFFF) + ((hi & 0x00FF) << 16);
  return l1id;
}

// ------------------------- fetchTimID --------------------------------------

/*! This method reads the TIM ID register.
 */

UINT16 TimModule::fetchTimID() {

  const UINT16 timID = regFetch( TIM_REG_TIM_ID );

  m_serialNumber = timID & 0xFF;
  m_firmware     = timID >> 8;
  return timID;
}

// ------------------------- intTrigStart ------------------------------------

/*! This method Enables Internal Triggers with the given repeat frequency.
 */

void TimModule::intTrigStart( const TimMaskFrequency frequency ) {

  loadByteLo( TIM_REG_FREQUENCY, frequency );
  loadBitSet( TIM_REG_ENABLES, TIM_BIT_EN_INT_TRIG );
}

// ------------------------- intTrigStop -------------------------------------

/*! This method stops Internal Triggers.
 */

void TimModule::intTrigStop(void) { //*** void

  loadBitClear( TIM_REG_ENABLES, TIM_BIT_EN_INT_TRIG );
}

// ------------------------- issueCommand ------------------------------------

/*! This method issues a TIM command edge-mode bit - experimental!
 */

void TimModule::issueCommand( const TimBitCommand mask ) {

  if (mask >  TIM_VSPA) loadBitClear( TIM_REG_COMMAND, mask );
  loadBitSet( TIM_REG_COMMAND, mask );
  if (mask >  TIM_VSPA) loadBitClear( TIM_REG_COMMAND, mask );
}

// ------------------------- issueVCAL ---------------------------------------

/*! This method issues a VME CAL command followed by an L1A,
 *  after the given pipeline delay.
 */

void TimModule::issueVCAL( const UINT8 pipelineDelay ) {

  loadByteLo(   TIM_REG_DELAY, pipelineDelay );
  issueCommand( TIM_VCAL );
}

// ------------------------- loadBitSet --------------------------------------

/*! This method writes the bits set in a 16-bit bit mask into a VME register,
 *  leaving the other bits unchanged.
 */

void TimModule::loadBitSet( const TimRegister addr, const UINT16 mask ) {

  UINT16 value = vmeFetch( addr );
  vmeLoad( addr, value | mask );
}

// ------------------------- loadBitClear ------------------------------------

/*! This method clears the bits set in a 16-bit bit mask into a VME register,
 *  leaving the other bits unchanged.
 */

void TimModule::loadBitClear( const TimRegister addr, const UINT16 mask ) {

  UINT16 value = vmeFetch( addr );
  vmeLoad( addr, value & ~mask );
}

// ------------------------- loadByteHi --------------------------------------

/*! This method writes a byte into the upper half of a 16-bit VME register,
 *  leaving the other byte unchanged.
 */

void TimModule::loadByteHi( const TimRegister addr, const UINT8 byte ) {

  UINT16 value = vmeFetch( addr );
  vmeLoad( addr, (value & 0x00FF) | (byte << 8) );
}

// ------------------------- loadByteLo --------------------------------------

/*! This method writes a byte into the lower half of a 16-bit VME register,
 *  leaving the other byte unchanged.
 */

void TimModule::loadByteLo( const TimRegister addr, const UINT8 byte ) {

  UINT16 value = vmeFetch( addr );
  vmeLoad( addr, (value & 0xFF00) | byte );
}

// ------------------------- regFetch ----------------------------------------

/*! This method reads a 16-bit value from a VME register.
 */

UINT16 TimModule::regFetch( const TimRegister addr ) {

  UINT16 data = vmeFetch( addr );
  return data;
}

// ------------------------- regLoad -----------------------------------------

/*! This method writes a 16-bit value into a VME register.
 */

void TimModule::regLoad( const TimRegister addr, const UINT16 data ) {

  vmeLoad( addr, data );
}

// ------------------------- seqRun ------------------------------------------

/*! This method starts the Sequencer executing.
 */

void TimModule::seqRun( const UINT16 size ) {

  regLoad( TIM_REG_SEQ_END,     size - 1 );
  regLoad( TIM_REG_SEQ_CONTROL, TIM_BIT_SEQ_RESET );
  regLoad( TIM_REG_SEQ_CONTROL, 0 );
  regLoad( TIM_REG_SEQ_CONTROL, TIM_BIT_SEQ_GO | TIM_BIT_SEQ_EN_ALL );
  // all outputs enabled
}

// ------------------------- seqFetch ----------------------------------------

/*! This method reads the Sequencer memory into a buffer.
 */

void TimModule::seqFetch( const UINT16 size, UINT16 buffer[] ) {

  for (int i = 0; i < size; i++) {
    buffer[i] = vmeFetch( TIM_SEQ_ADDR + i*2 );
  }
}

// ------------------------- seqLoad -----------------------------------------

/*! This method writes a buffer into the Sequencer memory.
 */

void TimModule::seqLoad( const UINT16 size, const UINT16 buffer[] ) {

  UINT16 data;
  for (int i = 0; i < size; i++) {
    data = buffer[i];
    vmeLoad( TIM_SEQ_ADDR + i*2, data );
  }
}

// ------------------------- vmeFetch ----------------------------------------

/*! This method reads a 16-bit value from a VME register.
 */

UINT16 TimModule::vmeFetch( const UINT32 addr )
		  throw (VmeException &) {

  UINT16 data = 0;
  try {
    data = m_vmePort->read16( addr );
  }
  catch (VmeException & vmeFailed) {
    // Leave critical section before re-throwing the error
    throw vmeFailed;
  }
  return data;
}

// ------------------------- vmeLoad -----------------------------------------

/*! This method writes a 16-bit value into a VME register.
 */

void TimModule::vmeLoad( const UINT32 addr, const UINT16 data )
		throw (VmeException & ) {

  try {
    m_vmePort->write16( addr, data );
//debug  throw VmeException(VmeException::BUS_ERROR, 1, m_vmePort);
  }
  catch (VmeException & vmeFailed) {
    // Leave critical section before re-throwing the error
    throw vmeFailed;
  }
}

} // End namespace SctPixelRod

// ------------------------- Overload operator<< -----------------------------

/*! This overloaded operator lets us use cout to print the status of the TIM.
 *  It overloads std::operator<< when used with an object of class TimModule.
 */

using namespace SctPixelRod;

namespace std {

ostream& operator<<( ostream& os, TimModule& tim ) {

  os << "TIM status" << endl;

  os << " Serial Number: " << tim.getSerialNumber();
  os << " Version: "       << tim.getFirmware() << endl;
  hex(cout);
  os << " L1ID: "   << tim.fetchL1ID();
  os << " BCID: "   << tim.regFetch( TIM_REG_TRIGGER_BCID );
  os << " status: " << tim.regFetch( TIM_REG_STATUS );
  os << endl;

  return os;
}

} // End namespace std
