/*
 * sct_structures.h
 *
 * A fresh look at what may be needed for the 
 * SCT ROD configuration data structures
 *
 * This is a revised and expanded version of John Hill's "sct_structure.h"
 * which can be found here:
 * http://sctpixel.home.cern.ch/sctpixel/Cambridge/sct_structure.txt 
 *
 * From our point of view, the layout of the structures is less important
 * than their content, so the ROD developers should feel free to suggest
 * alternative layouts and/or naming schemes.
 *
 * Version 0.8 PWP/GFM 2002.06.04
 * Version 1.0 PWP     2002.06.17
 * Version 1.1 PWP     2003.03.25
 * Version 1.2 DPSF    2003.03.27
 */
#include "processor.h"
#include "bocStructure.h"

#ifndef SCT_STRUCT_H	 /* multiple inclusion protection */
#define SCT_STRUCT_H

#define N_SCT_MODULES     48   /*   48 modules directly attached +12 modules  */
#define N_SCT_EXTMODULES  12   /*   using redundant command links OFF ROD     */
#define N_SCT_TOTMODULES  ((N_SCT_MODULES)+(N_SCT_EXTMODULES))

#define N_SCT_CHANS    128
#define N_SCT_CHIPS    12

/* Default & off-ROD positions of a module's TTC fibres. The default primary
   TTC fibre location for a module is simply the module's position inside the
   structure set. */
#define DEFAULT_TTC    0x80
#define OFF_ROD_TTC    0xff

/* Data Links which are turned off: */
#define DATA_LINK_OFF  0xff

/* Structure Definitions */
typedef struct {

  /* This structure corresponds to the ABCD chip configuration register.
   * Aside from one correction to suit the ABCD3T chip (trimRange) it
   * is essentially the same as that proposed by John Hill. */

  UINT16 readoutMode  : 2;
  UINT16 calibMode    : 2;
  UINT16 trimRange    : 2;
  UINT16 edgeDetect   : 1;
  UINT16 mask         : 1;
  UINT16 accumulate   : 1;
  UINT16 inputBypass  : 1;
  UINT16 outputBypass : 1;
  UINT16 master       : 1;
  UINT16 end          : 1;
  UINT16 feedThrough  : 1;
  UINT16 padding      : 2;  /*pad to 16 bits */

} ABCDConfig;

typedef struct{

 /* This structure holds calibration information about this
  * ABCD chip.  None of these parameters was present in
  * John Hill's version - we probably hadn't thought of it
  * two years ago! */

 /* We store a functional representation of the response 
  * curve (equivalent threshold as a function of charge)
  * so that we may request ROD to set the threshold to
  * a value specified in fC. */

  UINT8 rc_function;     /* Used to denote the type of function which
                          * has been fitted to the response curve, eg.
                          *  0 - no calibration information
                          *  1 - (can't remember, probably exponential)
                          *  2 - second order polynomial 
                          *  3 - (can't remember, probably "Grillo" func)
                          *  4 - straight line fit */
  FLOAT32 rc_params[3];  /* Response Curve Fit */

  FLOAT32 c_factor;      /* Capacitor Correction factor */

  UINT32  unused[3];     /* For data alignment            */
 /* Other calibration data is available from the wafer test -
  * but it's probably irrelevant. */

 /* We had a discussion about putting temperature correction
  * coefficients in here, but to avoid the complication of 
  * delivering recent temperature data to ROD, we conclude 
  * that temperature corrections are best done offline. */

} ABCDCaldata;

typedef struct {

  /* All registers except the TrimDAC registers.  The contents of
   * these registers change more often than the TrimDAC settings
   * and will most often be loaded as a group.  */

  ABCDConfig   config;   /* Configuration Register 16  bits */

  UINT16 unused;          /* For data alignment            */
  UINT8  vthr;            /* Threshold,             8 bits */
  UINT8  vcal;            /* Calibration Amplitude, 8 bits */
  UINT8  delay;           /* Strobe Delay,          6 bits */
  UINT8  preamp;          /* Preamp Bias Current,   5 bits */
  UINT8  shaper;          /* Shaper Current,        5 bits */

  UINT32 mask[4];        /* Mask Register,         128 bits */
} ABCDBasic;

typedef struct {

  /* This structure includes all register settings and
   * calibration information for one ABCD chip. */

  UINT8        active;   /* 1 -> participates in scans */
                         /* 0 -> register contents unchanged during scanning
                          *      (although the fixed values are still loaded) */
  UINT8        address;  /* In SCTDAQ we chose to associate a hardcoded
                          * chip address with each location on a module.
                          * That's fine as long as things work correctly.
                          * If we allow the address to be configurable
                          * we can recover from (a subset of possible)
                          * address failure modes. */

  UINT8        unused;   /* For data alignment            */

  UINT8        target;   /* <= I think this should be correct?  */

  ABCDBasic    basic;    /* All registers apart from the TrimDACs */

  ABCDCaldata  caldata;  /* Calibration data for this chip */

  UINT8  trim[N_SCT_CHANS];  /* TrimDAC registers, 4 bits/channel
                          * This could be more tightly packed, but
                          * unless you need the space, this sounds
                          * like an unnecessary complication... */

} ABCDChip;


typedef struct {

  /* This structure includes all register settings and
   * calibration information for one ABCD module. */

  ABCDChip chip[N_SCT_CHIPS];
 
  UINT8 present;  /* Module is physically present. Does not need setting
                     externally; handled by the Master DSP. */

  UINT8 active;   /* 1 -> participates in scans */
                  /* 0 -> registers unchanged during scanning */
  UINT8 select;   /* 1 -> use redundant clk/com; ID4=1 */
                  /* 0 -> use primary clk/com; ID4=0 */

  UINT8 groupId;   /* The ID of the module's group. This is used to indicate
                      which slave DSP will receive the module's data (if group
                      based distribution is set), and also to allow different
                      module groups to be triggered independently (for
                      cross-talk studies). Valid range: [0,7] */

  BOCConfig *bocConfig;  /* Pointer to the BOC configuration structure. 
                            Does not need setting externally; handled by the MDSP. */
  UINT8 pTTC;     /* primary TX channel   (used when SELECT = 0) */
  UINT8 rTTC;     /* redundant TX channel (used when SELECT = 1) */

  UINT8 rx[2];  /* data links used by module: SCT: 0,1  Pixel 0, 0 & 1, or all. */

  UINT8  unused1[4];
  UINT32 unused2[4]; /* align module structures on 8 word boundary */

 /* Each of a module's RX datalinks may or may not be in use 
  * dependent upon the bypass configuration.  ROD should 
  * consider the settings of the MASTER bit of the
  * configuration register of each master chip when
  * choosing to enable or disable input streams, 
  * reallocating buffers as necessary. */

} ABCDModule;

#endif /*SCT_STRUCT_H */
