#include "RxDelayAlgorithm.h"
#include "AnalysisAlgorithmMap.h"
#include "SctData/RxDelayTestResult.h"
#include "SctData/RawScanResult.h"
#include "SctData/DefectList.h"
#include "SctData/StandardDefects.h"
#include <string>
#include "TH2.h"

using namespace std;
using namespace boost;
using namespace SctData;
using namespace Sct;

namespace SctAnalysis {
    
bool RxDelayAlgorithm::inMap = AnalysisAlgorithmMap::instance().setAlgorithm("RxDelayTest", auto_ptr<AnalysisAlgorithm>(new RxDelayAlgorithm()));

shared_ptr<AnalysisAlgorithm> RxDelayAlgorithm::clone(const TestData& testData, const string& moduleName) const throw(){
    return shared_ptr<AnalysisAlgorithm>(new RxDelayAlgorithm(testData, moduleName, *this));
}

shared_ptr<TestResult> RxDelayAlgorithm::createTestResult() const {
    return shared_ptr<RxDelayTestResult> (new RxDelayTestResult()) ;
}

void RxDelayAlgorithm::loadData() {
    loadAllRaws();
}

bool RxDelayAlgorithm::canAnalyze() const {
    return hasAllRaws();
}

// Choose a point far from where the raw scan switches from all 0 to all 1.
void RxDelayAlgorithm::analyze() {
    RxDelayTestResult& result = *dynamic_pointer_cast<RxDelayTestResult>(getTestResult());
    const RawScanResult& raw = *getRaw(0);
    result.setScanVariable(raw.getHeader().getVariable());	
    
    bool pass = true;
    const SctData::ScanPoints points = raw.getPoints();
    bool ascending = points.ascending();
    result.setNOptima(nLinkModule);

    for (short unsigned int ilink=0; ilink<nLinkModule; ++ilink ) {
      const TH2& data = raw.getScanData(ilink);
      const unsigned nx = data.GetNbinsX();
      const unsigned ny = data.GetNbinsY();
      double* projection = new double[ny];
      int rise=-1;
      int fall=-1;
      for (unsigned iy=0; iy<ny; ++iy){
	const unsigned ipt = ascending ? iy : ny-iy-1;
	projection[ipt]=0;
	for (unsigned ix=0; ix<nx; ++ix){
	  projection[iy]+=data.GetBinContent(ix+1,ipt+1);
	}
	// half occupancy:
        double half = points.getNEvents(ipt) * nx/2.;
	if (ipt>0 && projection[ipt]>half && projection[ipt-1]<half) {
//	  std::cout << ipt << " Rise " << projection[ipt-1] <<  " -> " 
//		    << projection[ipt] << " : " << half << std::endl; 
	  rise=ipt;
	}
	if (ipt>0 && projection[ipt]<half && projection[ipt-1]>half) {
//	  std::cout << ipt << " Fall " << projection[ipt-1] <<  " -> " 
//		    << projection[ipt] << " : " << half << std::endl; 
	  fall=ipt;
	}
      }
      
      double optimum=-1;
      double fraction=0.5;
      const unsigned zero   = ascending ? 0 : ny-1;
      const unsigned end    = ascending ? ny-1 : 0;

//      cout << result.getUniqueID() << "rise = " << rise << ", fall = " << fall << endl;

      if (rise==-1&&fall==-1) {
	// no rise or fall - put it in the middle
	optimum = points[zero] + fraction*(points[end]-points[zero]);
      } else if (rise==-1){
	// only a fall - put it far from the fall
	optimum = points[fall] + (1-fraction)*25.;
      } else if (fall==-1){
	// only a rise - put it far from the rise
	optimum = points[rise] + fraction*25.;
      } else {
	// both a rise and a fall- very odd!
	pass=false;
      }
      while (optimum>25.) {optimum-=25.;}
//      cout << result.getUniqueID() << "optimum=" << optimum << endl;
      result.setOptimum(ilink, optimum);
    }
    result.setPassed(pass);
}


}
