#include <iostream>
#include <boost/shared_ptr.hpp>

#include "configXMLImpl.h"

using namespace std;
using namespace SctConfiguration;

boost::shared_ptr<SctConfiguration::Configuration> config;


/**
   Write a CSV file (or whatever you call it with tabs) to filename
   @param headers List of header names and bool saying that the contents should be enclosed in ""
*/
static void writeCSV(string filename, 
                     vector<pair<string, bool> > headers, 
                     vector<vector<string> > body);

/** Main function */
static void writePowerSupplyConfiguration(boost::shared_ptr<SctConfiguration::Configuration> config, 
                                          string directory);

static void writeMappings(boost::shared_ptr<SctConfiguration::Configuration> config, string filename);
static void writeCrateParams(boost::shared_ptr<SctConfiguration::Configuration> config, int crate, string state, string filename);

static string intToString(int i);
static string floatToString(double f);

int main() {
  config.reset(new ConfigurationXMLImpl);
  cout << "Hello world\n";

  try {
    writePowerSupplyConfiguration(config, ".");
  } catch(SctConfiguration::ConfigurationException &e) {
    cout << "Exception writing configuration\n";
    cout << e.what() << endl;
  }

  cout << "Wrote dump\n";
}

/*
  8 Tables:
  Cnf_Card
  Cnf_Crate
  Cnf_Host
  Cnf_Iden
  Cnf_Node
  Cnf_Para
  Cnf_Syst
  Cnf_Unit

  Cnf_Card
    Card Parameters
  Cnf_Crate
    Crate parameters
  (Cnf_Host)
    Which hosts are in the DCS system
  Cnf_Iden
    Mappings!!!
  (Cnf_Node)
    CAN-bus nodes?
  Cnf_Para
    Channel Parameters
  (Cnf_Syst)
    Global variables...
  (Cnf_Unit)
    Info about units in system

*/

void writePowerSupplyConfiguration(boost::shared_ptr<SctConfiguration::Configuration> config, string directory) {
  writeMappings(config, directory + "/Cnf_Map.txt");
  
  const char *states[] = {"OFF", "STBY", "ON"};
  char fname[64];

  for(unsigned int crate=0; crate<1; crate++){
    for(unsigned int state=0; state<3; state++){
      sprintf(fname,"/Cnf_Crate%02d_%s.txt",crate,states[state]);
      writeCrateParams(config, 0, states[state], directory + fname);
    }
  }
}

void writeMappings(boost::shared_ptr<SctConfiguration::Configuration> config, string filename) 
{
  const char *headerNames[] = {"MoPS", "Serial Number", "Group", "Alias"};

  vector<pair<string, bool> > headers;

  unsigned int numHeaders = sizeof(headerNames)/4;

  for(unsigned int i=0; i<numHeaders; i++) {
    bool isString;
    switch(i){
      default:
        isString = false;
        break;
    }
    headers.push_back(make_pair(headerNames[i], isString));
  }

  char geom[64];
  char power[64];

  int line = 0;
  vector<vector<string> > body;

  list<string> modules = config->listAllModules();

  for(unsigned int crate=0; crate<88; crate++){
    for(unsigned int channel=0; channel<48; channel++) {

      unsigned int MUR, module;
      std::string sn;
      short group;
      MURType murType;
      try {
        config->translateFromPowerSupply((unsigned int)0,crate,channel,MUR,module);
        config->translateToSN(MUR, module, sn);
        group = config->getModuleGroup(sn);
        murType = config->getMURType(MUR);

        if(murType == BARREL){
          unsigned int barrel, row;
          int b_number;
          config->translateToBarrel(MUR, module, barrel, row, b_number);
          sprintf(geom,"Barrel%02d.Row%02d.Module%02d", barrel,row,b_number);
        } else if(murType == ENDCAP){
          int disk;
          unsigned int quadrant, e_number;
	
          config->translateToEndcap(MUR, module, disk, quadrant, e_number);
          sprintf(geom,"Disk%02d.Quadrant%02d.Module%02d", disk, quadrant, e_number);
        } else {
          throw ConfigurationException("no MUR mapping");
	}

        body.push_back(vector<string>());

        // MoPS dp name
        sprintf(power,"Crate%02d.Channel%02d",crate,channel);
        body[line].push_back(power);

        // serial number
        body[line].push_back(sn);

        // module group
        body[line].push_back(intToString(group));

        // geographical alias
        body[line].push_back(geom);

        line ++;

      } catch(ConfigurationException &e) {
        // cout << "Crate " << crate << " channel " << channel << " not fully mapped" << endl;
      } 

    } // end for(channel)
  } // end for(crate)

  cout << "Writing " << filename << endl;
  writeCSV(filename.c_str(), headers, body);
}

void writeCrateParams(boost::shared_ptr<SctConfiguration::Configuration> config, int crate, string state, string filename) {

  const char *headerNames[] = {"MoPS", "value", "loAlarm", "loWarn", "hiWarn", "hiAlarm"}; 

  const char *channelParams[] = {
    "LVch_Vcc", "LVps_Vcc", "LVretVcc", "LVch_Icc", "LVch_Vdd", "LVps_Vdd", "LVretVdd", "LVch_Idd",
    "LVchVCSV", "LVchVCSI", "LVchPINV", "LVchPINI", "MOch_Tm0", "MOch_Tm1", "HVchVolt", "HVchCurr", 
    "LVchDTrp", "LVchCTrp", "LVchTLim", "LVchTTrp", "LVchCLKS", "LVchVOut", "HVchITrp", "HVchRamp"};

  const char *channelFormats[] = {
    "%3.2f",    "%3.2f",    "%3.2f",    "%1.0f",    "%3.2f",    "%3.2f",    "%3.2f",    "%1.0f",
    "%2.1f",    "%3.2f",    "%2.1f",    "%3.2f",    "%2.1f",    "%2.1f",    "%2.1f",    "%3.2f", 
    "%1.0f",    "%1.0f",    "%1.0f",    "%1.0f",    "%1.0f",    "%1.0f",    "%3.2f",    "%1.0f"};

  const char *channelType[] = {      /*         M - moni    C - comm    B - both    */
    "B",         "M",        "M",       "M",        "B",        "M",         "M",       "M",
    "B",         "M",        "B",       "M",        "M",        "M",         "B",       "M",
    "C",         "C",        "C",       "C",        "C",        "C",         "C",       "C"};

  const char *hvCardParams[] = { "HVcaMask"};
  const char *hvCardFormats[] ={ "%1.0f"};
  const char *hvCardType[] =   { "C"};

  const char *lvCardParams[] = { "LVcaPowr", "LVcaTemp", "LVcaTrip", "LVcaMask"};
  const char *lvCardFormats[] ={ "%2.1f",    "%1.0f",    "%1.0f",    "%1.0f"};
  const char *lvCardType[] =   { "M",        "M",        "C",        "C"       };


  float valueFloat;
  char  valueString[128];
  int line = 0;

  std::string name, type;

  // Column headers
  
  vector<pair<string, bool> > headers;
  vector<vector<string> > body;

  for(unsigned int i=0; i<sizeof(headerNames)/4; i++) {
    headers.push_back(make_pair(headerNames[i], 0));
  }

  // MoPS channel parameters

  for(unsigned int channel=0; channel<48; channel++){
    for(unsigned int param=0; param<sizeof(channelParams)/4; param++) {

      // datapoint name
      sprintf(valueString,"Crate%02d.Channel%02d.%s",crate,channel,channelParams[param]);
      body.push_back(vector<string>());
      body[line].push_back(valueString);

      unsigned int stop = 5;
      if(channelType[param]=="C") stop = 1;

      // if first column is empty,
      for(unsigned int column=0; column<stop; column++){
        if((column==0)&&(channelType[param]=="M")){
          // first column not used
          sprintf(valueString,"-");
        }else{
          name = channelParams[param];
          type = headerNames[column+1];
          valueFloat = config->getPowerParamCC(crate, channel, state, name, type);
          sprintf(valueString,channelFormats[param], valueFloat);
	}
        body[line].push_back(valueString);
      }

      line ++;

    } // end for(param)
  } // end for(channel)

  // Card parameters are only written to the OFF state file

  if(state=="OFF"){  
    // HV card parameters

    for(unsigned int card=0; card<6; card++){
      for(unsigned int param=0; param<sizeof(hvCardParams)/4; param++) {

        // datapoint name
        sprintf(valueString,"Crate%02d.HV_card%02d.%s",crate,card,hvCardParams[param]);
        body.push_back(vector<string>());
        body[line].push_back(valueString);

        unsigned int stop = 5;
        if(hvCardType[param]=="C") stop = 1;

        // if first column is empty,
        for(unsigned int column=0; column<stop; column++){
          if((column==0)&&(hvCardType[param]=="M")){
            // first column not used
            sprintf(valueString,"-");
          }else{
            name = hvCardParams[param];
            type = headerNames[column+1];
            valueFloat = config->getHVCardParam(crate, card, name, type);
            sprintf(valueString,hvCardFormats[param], valueFloat);
  	}
          body[line].push_back(valueString);
        }

        line ++;

      } // end for(param)
    } // end for(card)

    // LV card parameters

    for(unsigned int card=0; card<12; card++){
      for(unsigned int param=0; param<sizeof(lvCardParams)/4; param++) {

        // datapoint name
        sprintf(valueString,"Crate%02d.LV_card%02d.%s",crate,card,lvCardParams[param]);
        body.push_back(vector<string>());
        body[line].push_back(valueString);

        unsigned int stop = 5;
        if(lvCardType[param]=="C") stop = 1;

        // if first column is empty,
        for(unsigned int column=0; column<stop; column++){
          if((column==0)&&(lvCardType[param]=="M")){
            // first column not used
            sprintf(valueString,"-");
          }else{
            name = lvCardParams[param];
            type = headerNames[column+1];
            valueFloat = config->getLVCardParam(crate, card, name, type);
            sprintf(valueString,lvCardFormats[param], valueFloat);
  	}
          body[line].push_back(valueString);
        }

        line ++;

      } // end for(param)
    } // end for(card)

  } // end if(OFF)
  cout << "Writing " << filename << endl;
  writeCSV(filename.c_str(), headers, body);
}


/**
   Headers is a vector of pairs. First is the header name, second is a bool specifying that the body should be enclosed in quotes
*/
void writeCSV(string filename, 
              vector<pair<string, bool> > headers, 
              vector<vector<string> > body) 
{
  ofstream outFile(filename.c_str(), ios::out);

  cout << "Size of headers is " << headers.size() << endl;

  // Write headers
  for(unsigned int i=0; i<headers.size(); i++) {
    outFile << "\"" << headers[i].first << "\"\t";
  }

  // outFile << endl;
  outFile << "\r\n"; // DOS EOL

  cout << "Size of body is " << body.size() << endl;

  for(unsigned int line = 0; line < body.size(); line++) {
//    if(body[line].size() != headers.size()) {
//      cout << "Header size mismatch in line " << line << " (" << body[line].size() << " not " << headers.size() << ")\n";
//      return;
//    }
    for(unsigned int param=0; param < body[line].size(); param++) {
      if(headers[param].second) {
        outFile << "\"" << body[line][param] << "\"\t";
      } else {
        outFile << body[line][param] << "\t";
      }
    }

    // outFile << endl;
    outFile << "\r\n"; // DOS EOL
  }
}

inline string intToString(int i) {
  char buffer[40];
  sprintf(buffer, "%d", i);
  return string(buffer);
}

inline string floatToString(double f) {
  char buffer[40];
  sprintf(buffer, "%.2f", f);
  return string(buffer);
}
