#include <libxml/tree.h>
#include <libxml/xpath.h>
#include <libxml/xinclude.h>
#include <libxml/xmlversion.h>

#include <boost/lexical_cast.hpp>

#include <fstream.h>

#include <mrs/message.h>

#include <stdexcept>

#include <unistd.h> // For getcwd
#include <time.h>

#include "configXMLImpl.h"
#include "Sct/SctNames.h"

using namespace SctConfiguration;

/*
  Parse the xml configuration and do XInclude processing
  Configuration file loaded from environment variable SCTDAQ_ROD_CONFIGURATION_PATH
*/
ConfigurationXMLImpl::ConfigurationXMLImpl() : document(0), xpathContext(0), log(), lastFilename(""), is_dict(0), isLookupDone(false), serialToMurNodeBuilt(false), serialToConfigNodeBuilt(false) {
  const char *envname = "SCTDAQ_ROD_CONFIGURATION_PATH";

  char *filename = getenv(envname);

  if(filename) {
    log << "Configuration loading from " << filename << std::endl;
    loadConfiguration(filename);
  } else {
    std::cerr << "Configuration Environment: " << envname << " not set, don't know where to load config from\n";
    log << "Configuration Environment: " << envname << " not set, don't know where to load config from\n";
    throw ConfigurationException("No configuration");
  }

  IPCPartition partition("SCT");

  if(partition.isValid()) {
    MRSStream mrsStream(partition);

    mrsStream << "SCT_CONFIG_START" << MRS_INFORMATION << MRS_QUALIF("sctConf") << MRS_TEXT("SCT configuration service started") << ENDM;

    ISInfoDictionary dict(partition);
    ISInfoString isFilename(filename);

    //   ISInfoString getFilename;
    //   dict.findValue("ConfigurationServer.Filename", getFilename);
    //   std::cout << "Found " << getFilename.getValue() << " in IS\n";

    if(dict.contains("ConfigurationServer.Filename")) {
      dict.update("ConfigurationServer.Filename", isFilename);

      std::cout << "*** Filename already in IS, will run anyway but there could be another service running?\n";
    } else {
      dict.insert("ConfigurationServer.Filename", isFilename);
    }

    std::cout << "Put " << isFilename.getValue() << " into  IS\n";
  }
}

/*
   Modify the default value for any MoPS (Module Power Supply) parameter
*/
void ConfigurationXMLImpl::modifyDefaultPowerParam(std::string state, std::string name, std::string type, float value) {
  log << "Change Default Power Param (" << name << ") to " << value << std::endl;

  char expression[1000];
  sprintf(expression, 
          "/configuration/power/defaults[@state=\"%s\"]/param[@name=\"%s\"]",
          state.c_str(), name.c_str());
  xmlNodePtr paramNode;

  try {
    paramNode = getQueryNode(expression, "Modify default power param");
  } catch(ConfigurationException &c) {
    // Create param node 
    xmlNodePtr root = xmlDocGetRootElement(document);
    xmlNodePtr power = getOrNewChildNode(root, "power");
    xmlNodePtr defaults = getOrNewChildWithStringAttr(power, "defaults", "state", state);
    paramNode = getOrNewChildWithStringAttr(defaults, "param", "name", name);
  }

  setFloatAttr(paramNode, type.c_str(), value);
}

/*
   Modify the value of any MoPS (Module Power Supply) HV card parameter
*/
void ConfigurationXMLImpl::modifyHVCardParam(unsigned int crate, unsigned int card, 
                                             std::string name, std::string type, float value) {
  log << "Change MoPS Crate" << crate << ".HV_card" << card << "." << name << "." << type;
  log << " to " << value << std::endl;

  if(crate < 0 || crate> 87) {
    throw ConfigurationException("MoPS crate out of range on input to modifyHVCardParam");
  }
  if(card < 0 || card > 5) {
    throw ConfigurationException("MoPS HVCard out of range on input to modifyHVCardParam");
  }

  char expression[1000];
  sprintf(expression, 
            "/configuration/power/crate[@id=\"%d\"]/hvcard[@id=\"%d\"]/param[@name=\"%s\"]", 
          crate, card, name.c_str());

  xmlNodePtr paramNode;

  try {
    paramNode = getQueryNode(expression, "Modify power param");
  } catch(ConfigurationException &c) {
    // Create param node 
    xmlNodePtr root = xmlDocGetRootElement(document);
    xmlNodePtr power = getOrNewChildNode(root, "power");
    xmlNodePtr crateNode = getOrNewChildWithIntAttr(power, "crate", "id", crate);
    xmlNodePtr cardNode = getOrNewChildWithIntAttr(crateNode, "hvcard", "id", card);
    paramNode = getOrNewChildWithStringAttr(cardNode, "param", "name", name);
  }

  setFloatAttr(paramNode, type.c_str(), value);
}

/*
   Modify the value of any MoPS (Module Power Supply) LV card parameter
*/
void ConfigurationXMLImpl::modifyLVCardParam(unsigned int crate, unsigned int card, 
                                             std::string name, std::string type, float value) {
  log << "Change MoPS Crate" << crate << ".LV_card" << card << "." << name << "." << type;
  log << " to " << value << std::endl;

  if(crate < 0 || crate> 87) {
    throw ConfigurationException("MoPS crate out of range on input to modifyLVCardParam");
  }
  if(card < 0 || card > 5) {
    throw ConfigurationException("MoPS HVCard out of range on input to modifyLVCardParam");
  }

  char expression[1000];
  sprintf(expression, 
          "/configuration/power/crate[@id=\"%d\"]/lvcard[@id=\"%d\"]/param[@name=\"%s\"]", 
          crate, card, name.c_str());

  xmlNodePtr paramNode;

  try {
    paramNode = getQueryNode(expression, "Modify power param");
  } catch(ConfigurationException &c) {
    // Create param node 
    xmlNodePtr root = xmlDocGetRootElement(document);
    xmlNodePtr power = getOrNewChildNode(root, "power");
    xmlNodePtr crateNode = getOrNewChildWithIntAttr(power, "crate", "id", crate);
    xmlNodePtr cardNode = getOrNewChildWithIntAttr(crateNode, "lvcard", "id", card);
    paramNode = getOrNewChildWithStringAttr(cardNode, "param", "name", name);
  }

  setFloatAttr(paramNode, type.c_str(), value);
}

/*
   Modify the value of any MoPS (Module Power Supply) channel parameter
   (referenced by MUR/number)
*/
void ConfigurationXMLImpl::modifyPowerParam(unsigned int MUR, unsigned int number, 
                                            std::string state, std::string name, std::string type, float value) {
  log << "Change module's (" << MUR << ", " << number << ") Power Param (" << name << ") to " << value << std::endl;

  if(number < 1 || number > 6) {
    throw ConfigurationException("MUR position out of range (1-6) on input to modifyPowerParam");
  }

  unsigned int partition, crate, channel;
  translateToPowerSupply(MUR, number, partition, crate, channel);

  char expression[1000];
  sprintf(expression, 
            "/configuration/power/crate[@id=\"%d\"]/channel[@id=\"%d\"]/state[@name=\"%s\"]/param[@name=\"%s\"]", 
          crate, channel, state.c_str(), name.c_str());

  xmlNodePtr paramNode;

  try {
    paramNode = getQueryNode(expression, "Modify power param");
  } catch(ConfigurationException &c) {
    // Create param node 
    xmlNodePtr root = xmlDocGetRootElement(document);
    xmlNodePtr power = getOrNewChildNode(root, "power");
    xmlNodePtr crateNode = getOrNewChildWithIntAttr(power, "crate", "id", crate);
    xmlNodePtr channelNode = getOrNewChildWithIntAttr(crateNode, "channel", "id", channel);
    xmlNodePtr stateNode = getOrNewChildWithStringAttr(channelNode, "state", "name", state);
    paramNode = getOrNewChildWithStringAttr(stateNode, "param", "name", name);
  }

  setFloatAttr(paramNode, type.c_str(), value);
}

/*
   Modify the value of any MoPS (Module Power Supply) channel parameter
   (referenced by MoPS crate/channel)
*/
void ConfigurationXMLImpl::modifyPowerParamCC(unsigned int crate, unsigned int channel, 
                                              std::string state, std::string name, std::string type, float value) {
  log << "Change MoPS Crate" << crate << ".Channel" << channel << "." << name << "." << type;
  log << " to " << value << " for " << state << " state" << std::endl;

  if(crate < 0 || crate> 87) {
    throw ConfigurationException("MoPS crate out of range on input to ModifyPowerParamCC");
  }
  if(channel < 0 || channel > 47) {
    throw ConfigurationException("MoPS channel out of range on input to ModifyPowerParamCC");
  }

  char expression[1000];
  sprintf(expression, 
            "/configuration/power/crate[@id=\"%d\"]/channel[@id=\"%d\"]/state[@name=\"%s\"]/param[@name=\"%s\"]", 
          crate, channel, state.c_str(), name.c_str());

  xmlNodePtr paramNode;

  try {
    paramNode = getQueryNode(expression, "Modify power param");
  } catch(ConfigurationException &c) {
    // Create param node 
    xmlNodePtr root = xmlDocGetRootElement(document);
    xmlNodePtr power = getOrNewChildNode(root, "power");
    xmlNodePtr crateNode = getOrNewChildWithIntAttr(power, "crate", "id", crate);
    xmlNodePtr channelNode = getOrNewChildWithIntAttr(crateNode, "channel", "id", channel);
    xmlNodePtr stateNode = getOrNewChildWithStringAttr(channelNode, "state", "name", state);
    paramNode = getOrNewChildWithStringAttr(stateNode, "param", "name", name);
  }

  setFloatAttr(paramNode, type.c_str(), value);
}

/*
   Return the default value of any MoPS (Module Power Supply) parameter
*/
float ConfigurationXMLImpl::getDefaultPowerParam(std::string state, std::string name, std::string type) {
  log << "Get default power param " << name << " type " << type;
  log << " for " << state << " state" << std::endl;

  char expression[1000];

  try {
    sprintf(expression, 
            "/configuration/power/defaults[@state=\"%s\"]/param[@name=\"%s\"]", 
            state.c_str(), name.c_str());

    xmlNodePtr paramNode = getQueryNode(expression, "Get default power param");
    return getAttrAsFloat(paramNode, type.c_str());

  } catch(ConfigurationException &c) {
    return 0;
  }
}

/*
   Return the value of any MoPS (Module Power Supply) HV card parameter
*/
float ConfigurationXMLImpl::getHVCardParam(unsigned int crate, unsigned int card,
		                           std::string name, std::string type) {
  log << "Get MoPS Crate" << crate << ".HV_card" << card << "." << name << "." << type << std::endl;

  if(crate < 0 || crate> 87) {
    throw ConfigurationException("MoPS crate out of range on input to getHVCardParam");
  }
  if(card < 0 || card > 5) {
    throw ConfigurationException("MoPS HVcard out of range on input to getHVCardParam");
  }

  char expression[1000];

  try {
    sprintf(expression, 
            "/configuration/power/crate[@id=\"%d\"]/hvcard[@id=\"%d\"]/param[@name=\"%s\"]", 
            crate, card, name.c_str());

    xmlNodePtr paramNode = getQueryNode(expression, "Get HVcard param");
    return getAttrAsFloat(paramNode, type.c_str());

  } catch(ConfigurationException &c) {
    sprintf(expression, 
            "/configuration/power/defaults[@state=\"OFF\"]/param[@name=\"%s\"]", 
            name.c_str());
    
    xmlNodePtr paramNode = getQueryNode(expression, "Get default power param");
    return getAttrAsFloat(paramNode, type.c_str());
  }
}

/*
   Return the value of any MoPS (Module Power Supply) LV card parameter
*/
float ConfigurationXMLImpl::getLVCardParam(unsigned int crate, unsigned int card,
		                           std::string name, std::string type) {
  log << "Get MoPS Crate" << crate << ".LV_card" << card << "." << name << "." << type << std::endl;

  if(crate < 0 || crate> 87) {
    throw ConfigurationException("MoPS crate out of range on input to getLVCardParam");
  }
  if(card < 0 || card > 11) {
    throw ConfigurationException("MoPS LVcard out of range on input to getLVCardParam");
  }

  char expression[1000];

  try {
    sprintf(expression, 
            "/configuration/power/crate[@id=\"%d\"]/lvcard[@id=\"%d\"]/param[@name=\"%s\"]", 
            crate, card, name.c_str());

    xmlNodePtr paramNode = getQueryNode(expression, "Get LVcard param");
    return getAttrAsFloat(paramNode, type.c_str());

  } catch(ConfigurationException &c) {
    sprintf(expression, 
            "/configuration/power/defaults[@state=\"OFF\"]/param[@name=\"%s\"]", 
            name.c_str());
    
    xmlNodePtr paramNode = getQueryNode(expression, "Get default power param");
    return getAttrAsFloat(paramNode, type.c_str());
  }
}

/*
   Return the value of any MoPS (Module Power Supply) channel parameter
   (referenced by MUR/number)
*/
float ConfigurationXMLImpl::getPowerParam(unsigned int MUR, unsigned int number, 
                                          std::string state, std::string name, std::string type) {
  log << "Get power param " << MUR << ", " << number << ", " << name << ", " << type;
  log << " for " << state << " state" << std::endl;

  if(number < 1 || number > 6) {
    throw ConfigurationException("MUR position out of range (1-6) on input to getPowerParam");
  }

  char expression[1000]; 

  try {
    unsigned int partition, crate, channel;
    translateToPowerSupply(MUR, number, partition, crate, channel);

    sprintf(expression, 
            "/configuration/power/crate[@id=\"%d\"]/channel[@id=\"%d\"]/state[@name=\"%s\"]/param[@name=\"%s\"]", 
            crate, channel, state.c_str(), name.c_str());

    xmlNodePtr paramNode = getQueryNode(expression, "Get power param");
    return getAttrAsFloat(paramNode, type.c_str());

  } catch(ConfigurationException &c) {
    sprintf(expression, 
            "/configuration/power/defaults[@state=\"%s\"]/param[@name=\"%s\"]", 
            state.c_str(),name.c_str());
    
    xmlNodePtr paramNode = getQueryNode(expression, "Get default power param");
    return getAttrAsFloat(paramNode, type.c_str());
  }
}

/*
   Return the value of any MoPS (Module Power Supply) channel parameter
   (referenced by MoPS crate/channel)
*/
float ConfigurationXMLImpl::getPowerParamCC(unsigned int crate, unsigned int channel,
		                            std::string state, std::string name, std::string type) {
  log << "Get MoPS Crate" << crate << ".channel" << channel << "." << name << "." << type;
  log << " for " << state << " state" << std::endl;

  if(crate < 0 || crate> 87) {
    throw ConfigurationException("MoPS crate out of range on input to getPowerParamCC");
  }
  if(channel < 0 || channel > 47) {
    throw ConfigurationException("MoPS channel out of range on input to getPowerParamCC");
  }

  char expression[1000];

  try {
    sprintf(expression, 
            "/configuration/power/crate[@id=\"%d\"]/channel[@id=\"%d\"]/state[@name=\"%s\"]/param[@name=\"%s\"]", 
            crate, channel, state.c_str(), name.c_str());

    xmlNodePtr paramNode = getQueryNode(expression, "Get power param");
    return getAttrAsFloat(paramNode, type.c_str());

  } catch(ConfigurationException &c) {
    sprintf(expression, 
            "/configuration/power/defaults[@state=\"%s\"]/param[@name=\"%s\"]", 
            state.c_str(),name.c_str());
    
    xmlNodePtr paramNode = getQueryNode(expression, "Get default power param");
    return getAttrAsFloat(paramNode, type.c_str());
  }
}


std::list<std::string> ConfigurationXMLImpl::listDefaultPowerParams() {
  log << "List default Power params\n";

  char expression[1000];
  sprintf(expression, 
          "/configuration/power/defaults/param");

  std::list<std::string> result;

  xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)expression, xpathContext);

  if(queryResult) {
    if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
      std::cout << "XPath found " << queryResult->nodesetval->nodeNr << " params\n";
      for(int i=0; i<queryResult->nodesetval->nodeNr; i++) {
        xmlNodePtr curr = queryResult->nodesetval->nodeTab[i];
        result.push_back((const char *)xmlGetProp(curr, (xmlChar*)"name"));
      }
    }
  }

  std::cout << "Found " << result.size() << " parameters\n";
  return result;
}

float ConfigurationXMLImpl::getCratePowerParam(unsigned int crate, std::string name) {
  log << "getCratePowerParam (**unimpl**) " << crate << ", " << name << std::endl;
#warning "Implement more stuff"
  return 0.0;
}

static const char *DCSChannelParams[] = {
  "LVchStat", "LVch_Vcc", "LVps_Vcc", "LVretVcc", "LVch_Icc", "LVch_Vdd", "LVps_Vdd", "LVretVdd", "LVch_Idd", "LVchVCSV", 
  "LVchVCSI", "LVchPINV", "LVchPINI", "MOch_Tm0", "MOch_Tm1", "LVchDTrp", "LVchCTrp", "LVchTLim", "LVchTTrp", "LVchCLKS", 
  "LVchCLKR", "LVchVOut", "HVchStat", "HVchVolt", "HVchCurr", "HVchITrp", "HVchRamp", 0};

std::list<std::string> ConfigurationXMLImpl::listDCSChannelParams() {
  log << "listDCSChannelParams\n";

  std::list<std::string> result;
  for(int i=0; DCSChannelParams[i] != 0; i++) {
    result.push_back(DCSChannelParams[i]);
  }

  return result;
}

static const char *DCSCardParams[] = {"LVcaStat", "LVcaTemp", "LVcaPowr", "LVcaMask", "LVcaTrip", 
                                      "HVcaStat", "HVcaMask", 0};

std::list<std::string> ConfigurationXMLImpl::listDCSCardParams() {
  log << "listDCSCardParams\n";

  std::list<std::string> result;
  for(int i=0; DCSCardParams[i] != 0; i++) {
    result.push_back(DCSCardParams[i]);
  }

  return result;
}

static const char *DCSCrateParams[] = {0};

std::list<std::string> ConfigurationXMLImpl::listDCSCrateParams() {
  log << "listDCSCrateParams\n";

  std::list<std::string> result;
  for(int i=0; DCSCrateParams[i] != 0; i++) {
    result.push_back(DCSCrateParams[i]);
  }

  return result;
}

/*
  Load configuration from named file
*/
void ConfigurationXMLImpl::loadConfiguration(const std::string &fname) {
  log << "loadConfiguration " << fname << std::endl;

  std::string realFilename = fname;
  if(realFilename == "") {
    if(lastFilename != "") {
      realFilename = lastFilename;
    } else {
      log << "Passed bad filename \"\"!\n";
      std::cerr << "Bad filename!\n";
      throw ConfigurationException("Empty file name");
    }
  }

  log << "Load configuration from " << realFilename << std::endl;

  xmlDocPtr newDocument;

  newDocument = xmlParseFile(realFilename.c_str());

  if(newDocument) {
    //    processFileLinks(document);
    xmlXIncludeProcess(newDocument);
    //    std::cout << "XIncludes : " << includeCount << std::endl; //processFileLinks(document);

    //saveConfiguration();
  } else {
    log << "Configuration Document \"" << realFilename << "\" not parsed!\n";
    std::cerr << "Configuration Document \"" << realFilename << "\" not parsed!\n";
    throw ConfigurationException("File not found or bad file");
  }

  xmlXPathInit();
  if(xpathContext) {
    xmlXPathFreeContext(xpathContext);
  }
  xpathContext = xmlXPathNewContext(newDocument);

  lastFilename = realFilename;

  if(document) {
    // Delete previous one first
    xmlFreeDoc(document);
  }

  document = newDocument;

  serialToMurNodeBuilt = false;
  serialToConfigNodeBuilt = false;

  notifySystemStructureChange();
}

/*
  Free the configuration xml
*/
ConfigurationXMLImpl::~ConfigurationXMLImpl() {
  log << "~ConfigurationXMLImpl\n";

  //saveConfiguration();
  if(xpathContext) {
    xmlXPathFreeContext(xpathContext);
  }
  if(document) {
    xmlFreeDoc(document);
  }
  xmlCleanupParser();

  ISInfoDictionary *dict = getISDict();

  if(dict) {
    if(dict->contains("ConfigurationServer.Filename")) {
      dict->remove("ConfigurationServer.Filename");
    } else {
      std::cout << "*** No Filename in IS, don't need to remove it?\n";
    }
  }
}

/*
  Lookup the partitions defined in the configuration and return a list of integer ID's
*/
std::list<unsigned int> ConfigurationXMLImpl::listPartitions() {
  log << "List partitions\n";

  std::list<unsigned int> result;

  char expression[1000];

  sprintf(expression, "/configuration/partition");

  xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)expression, xpathContext);

  if(queryResult) {
    if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
      for(int i=0; i<queryResult->nodesetval->nodeNr; i++) {
        xmlNodePtr curr = queryResult->nodesetval->nodeTab[i];

        try {
          int thisId = getAttrAsInt(curr, "id");
          result.push_back(thisId);
        } catch(ConfigurationException &c) {
          log << "This partition has no id attribute\n";
        }
      }
    } else {
      log << "Bad partition data\n";
    }

    xmlXPathFreeObject(queryResult);
  } else {
    log << "No result\n";
  }

  return result;
}

/*
  Lookup the crates defined in the specified partition and return a list of integer ID's
*/
std::list<unsigned int> ConfigurationXMLImpl::listCratesInPartition(unsigned int partition) {
  log << "List crates in partition " << partition << std::endl;

  std::list<unsigned int> result;

  char expression[1000];

  sprintf(expression, "/configuration/partition[@id=%d]/crate", partition);

  xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)expression, xpathContext);

  if(queryResult) {
    if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
      for(int i=0; i<queryResult->nodesetval->nodeNr; i++) {
        xmlNodePtr curr = queryResult->nodesetval->nodeTab[i];
        try {
          result.push_back(getAttrAsInt(curr, "id"));
        } catch(ConfigurationException &c) {
          log << "This crate has no id attribute\n";
        }
      }
    } else {
      log << "Bad crate data\n";
    }

    xmlXPathFreeObject(queryResult);
  } else {
    log << "No result\n";
  }

  return result;
}

bool ConfigurationXMLImpl::isDummyCrate(unsigned int partition, unsigned int crate) 
{
  log << "Check for dummy crate...\n";

  char expression[1000];
  sprintf(expression, 
          "/configuration/partition[@id=%d]/crate[@id=%d]", partition, crate);

  try {
    xmlNodePtr crateNode = getQueryNode(expression, "Is dummy crate");
    int i = getAttrAsInt(crateNode, "dummy");
    return i;
  } catch(ConfigurationException &c) {
    return false;
  }
}

/*
  Lookup the RODs defined in the specified crate and return a list of integer ID's
*/
std::list<unsigned int> ConfigurationXMLImpl::listRodsInCrate(unsigned int partition, unsigned int crate) {
  log << "List rods in (" << partition << ", " << crate << ")\n";

  std::list<unsigned int> result;

  char expression[1000];

  sprintf(expression, "/configuration/partition[@id=%d]/crate[@id=%d]/rod", partition, crate);

  xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)expression, xpathContext);

  if(queryResult) {
    if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
      for(int i=0; i<queryResult->nodesetval->nodeNr; i++) {
        xmlNodePtr curr = queryResult->nodesetval->nodeTab[i];

        try {
          result.push_back(getAttrAsInt(curr, "id"));
        } catch(ConfigurationException &c) {
          log << "This rod has no id attribute\n";
        }
      }
    } else {
      log << "Bad ROD data\n";
    }
    xmlXPathFreeObject(queryResult);
  } else {
    log << "No result rod\n";
  }

  return result;
}

/*
  Return integer id's of the MURs in the specified rod.
*/
std::list<unsigned int> ConfigurationXMLImpl::listMURSInRod(unsigned int partition, unsigned int crate, unsigned int rod) {
  log << "List MURs in (" << partition << ", " << crate << ", " << rod << ")\n";
  std::list<unsigned int> result;

  char expression[1000];

  sprintf(expression, "/configuration/partition[@id=%d]/crate[@id=%d]/rod[@id=%d]/MUR", partition, crate, rod);

  xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)expression, xpathContext);

  if(queryResult) {
    if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
      for(int i=0; i<queryResult->nodesetval->nodeNr; i++) {
        xmlNodePtr curr = queryResult->nodesetval->nodeTab[i];

        try {
          result.push_back(getAttrAsInt(curr, "id"));
        } catch(ConfigurationException &c) {
          log << "This MUR has no id attribute\n";
        }
      }
    } else {
      log << "Bad MUR data\n";
    }

    xmlXPathFreeObject(queryResult);
  } else {
    log << "No MUR result\n";
  }

  return result;
}

/*
  Return a list of serial numbers associated with the specified MUR
*/
std::list<std::string> ConfigurationXMLImpl::listModulesInMUR(unsigned int partition, unsigned int MUR) {
  log << "List modules in (" << partition << ", " << MUR << ")\n";
  std::list<std::string> result;

  char expression[1000];

  sprintf(expression, "/configuration/partition[@id=%d]/crate/rod/MUR[@id=%d]/module/text()", partition, MUR);

  xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)expression, xpathContext);

  if(queryResult) {
    if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
      if(queryResult->nodesetval->nodeNr <= 6) {
        for(int i=0; i<queryResult->nodesetval->nodeNr; i++) {
          xmlNodePtr curr = queryResult->nodesetval->nodeTab[i];

          result.push_back((char *)curr->content);
        }
      } else {
        log << "Config: Too many modules in MUR " << MUR << " (" << queryResult->nodesetval->nodeNr << ")\n";
      }
    } else {
      log << "Config: Can't find modules in MUR " << MUR << " of partition " << partition << "\n";
    }
    xmlXPathFreeObject(queryResult);
  } else {
    log << "No module/MUR result\n";
  }

  return result;
}

/*
  Return a list of serial numbers of all modules that have configuration

  This includes those that are not attached to an MUR 
*/
std::list<std::string> ConfigurationXMLImpl::listAllModules() {
  log << "List all modules\n";
  std::list<std::string> result;

  char expression[1000];

  sprintf(expression, "/configuration/modules/module/sn/text()");

  xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)expression, xpathContext);

  if(queryResult) {
    if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
      if(queryResult->nodesetval->nodeNr > 0) {
        for(int i=0; i<queryResult->nodesetval->nodeNr; i++) {
          xmlNodePtr curr = queryResult->nodesetval->nodeTab[i];

          result.push_back((char *)curr->content);
        }
      } else {
        log << "Config: Found 0 modules!\n";
      }
    } else {
      log << "Config: Can't find any modules!\n";
    }
    xmlXPathFreeObject(queryResult);
  } else {
    log << "No module/MUR result\n";
  }

  return result;
}

/*
  Return a list of serial numbers of modules that have configuration but are not attached to an MUR.
*/
std::list<std::string> ConfigurationXMLImpl::listUnusedModules() {
  log << "List unattached modules\n";
  std::list<std::string> result;

  char expression[1000];

  sprintf(expression, "/configuration/modules/module/sn/text()");

  xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)expression, xpathContext);

  if(queryResult) {
    if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
      if(queryResult->nodesetval->nodeNr > 0) {
        for(int i=0; i<queryResult->nodesetval->nodeNr; i++) {
          xmlNodePtr curr = queryResult->nodesetval->nodeTab[i];

          std::string sn = (char *)curr->content;

          try {
            unsigned int MUR, module;
            translateFromSN(sn, MUR, module);
          } catch(ConfigurationException &e) {
            result.push_back(sn);
          }
        }
      } else {
        log << "Config: Found 0 modules!\n";
      }
    } else {
      log << "Config: Can't find any modules!\n";
    }
    xmlXPathFreeObject(queryResult);
  } else {
    log << "No module/MUR result\n";
  }

  return result;
}

/*
  Return a list of serial numbers associated with the redundancy links to the specified MUR
*/
std::list<std::string> ConfigurationXMLImpl::listRModulesInMUR(unsigned int partition, unsigned int MUR) {
  log << "List rmodules in MUR (" << partition << ", " << MUR << ")\n";
  std::list<std::string> result;

  char expression[1000];

  sprintf(expression, "/configuration/partition[@id=%d]/crate/rod/MUR[@id=%d]/rmodule/text()", partition, MUR);

  xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)expression, xpathContext);

  if(queryResult) {
    if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
      if(queryResult->nodesetval->nodeNr <= 6) {
        for(int i=0; i<queryResult->nodesetval->nodeNr; i++) {
          xmlNodePtr curr = queryResult->nodesetval->nodeTab[i];

          result.push_back((char *)curr->content);
        }
      } else {
        log << "Config: Too many redundant modules in MUR " << MUR << " (" << queryResult->nodesetval->nodeNr << ")\n";
      }
    } else {
      log << "Config: Can't find redundant modules for MUR " << MUR << " of partition " << partition << "\n";
    }
    xmlXPathFreeObject(queryResult);
  } else {
    log << "No redundant module/MUR result\n";
  }

  return result;
}

/*
  Find the configuration associated with the specified ROD.
  Return a RodConfig structure.
*/
RodConfig ConfigurationXMLImpl::getRodConfig(unsigned int partition, unsigned int crate, unsigned int rod) {
  log << "Get Rod Configuration " << partition << ", " << crate << ", " << rod << std::endl;
  RodConfig result;

  char expression[1000];

  sprintf(expression, "/configuration/partition[@id=%d]/crate[@id=%d]/rod[@id=%d]", partition, crate, rod);

  xmlNodePtr c = getQueryNode(expression, "ROD configuration"); // queryResult->nodesetval->nodeTab[0];

  result = parseRodConfig(c);

  return result;
}

/*
  Find the configuration associated with the specified ROD.
  Return a RodConfig structure.
*/
SlaveConfig ConfigurationXMLImpl::getDefaultSlaveConfig(char type) {
  log << "Get default slave configuration " << type << std::endl;
  SlaveConfig result;

  char expression[1000];
  sprintf(expression, "/configuration/defaults/slave[@type=\"%c\"]", type);

  xmlNodePtr slaveNode = getQueryNode(expression, "Slave configuration");

  for(xmlNodePtr slaveChild = slaveNode->children;
      slaveChild;
      slaveChild = slaveChild->next) {
    char* content = (char *)xmlNodeGetContent(slaveChild);

    if(strcmp((char *)slaveChild->name, "ipramFile") == 0) {
      result.ipramFile = content;
    } else if(strcmp((char *)slaveChild->name, "idramFile") == 0) {
      result.idramFile = content;
    } else if(strcmp((char *)slaveChild->name, "extFile") == 0) {
      result.extFile = content;
    }
    free(content);
  }

  return result;
}

/*
  Find the configuration associated with the specified TIM.
*/
TimConfig ConfigurationXMLImpl::getTimConfig(unsigned int partition, unsigned int crate) {
  log << "Get TIM Configuration " << partition << ", " << crate << std::endl;
  TimConfig result;

  char expression[1000];

  sprintf(expression, "/configuration/partition[@id=%d]/crate[@id=%d]/tim", partition, crate);

  xmlNodePtr c = getQueryNode(expression, "TIM configuration");

  result = parseTimConfig(c);

  return result;
}

MURType ConfigurationXMLImpl::getMURType(unsigned int MUR) {
  log << "Find type of MUR: " << MUR << std::endl;

  char expression[1000];
  sprintf(expression, "/configuration/geography/barrel/row/MUR[@id=%d]", MUR);

  try {
    getQueryNode(expression, "Check MUR is Barrel");

    return BARREL;
  } catch (ConfigurationException &c) {
    // Not a barrel MUR
  }

  sprintf(expression, "/configuration/geography/endcap/quadrant/MUR[@id=%d]", MUR);

  try {
    getQueryNode(expression, "Check MUR is Endcap");

    return ENDCAP;
  } catch(ConfigurationException &c) {
    // Not an endcap MUR!
  }

  sprintf(expression, "/configuration//MUR[@id=%d]", MUR);

  try {
    getQueryNode(expression, "Is MUR defined");

    return UNMAPPED;
  } catch(ConfigurationException &c) {
    // Not defined!
    return UNKNOWN;
  }
}

// Very simple way of finding mur...
unsigned int ConfigurationXMLImpl::getFreeMurId() {
  log << "getFreeMurId\n";

  unsigned int mur = 0;

  while(getMURType(mur) != UNKNOWN) {
    mur ++;
  }

  return mur;
}

void ConfigurationXMLImpl::translateToROD(unsigned int MUR, unsigned int module,
                                          unsigned int &partition, unsigned int &crate,
                                          unsigned int &rod, unsigned int &channel) {
  log << "Translate to Rod " << MUR << ", " << module << std::endl;

  if(module < 1 || module > 6) {
    throw ConfigurationException("MUR position out of range (1-6) on input to translateToROD");
  }

  char expression[1000];
  sprintf(expression, "/configuration/partition/crate/rod/MUR[@id=%d]/module[@id=\"%d\"]", MUR, module);
  xmlNodePtr moduleNode = getQueryNode(expression, "Translate to ROD");

  // MUR order
  int order = getAttrAsInt(moduleNode->parent, "order");

  rod = getAttrAsInt(moduleNode->parent->parent, "id");
  crate = getAttrAsInt(moduleNode->parent->parent->parent, "id");
  partition = getAttrAsInt(moduleNode->parent->parent->parent->parent, "id");

  channel = order * 6 + module - 1;

  log << " Translate to Rod " << partition << ", " << crate << ", " << rod << ", " << channel << "\n";
}

void ConfigurationXMLImpl::translateFromROD(unsigned int partition, unsigned int crate,
                                            unsigned int rod, unsigned int channel,
                                            unsigned int &MUR, unsigned int &module) {
  log << "Translate from Rod " << partition << ", " << crate << ", " << rod << ", " << channel << "\n";

  if(channel < 0 || channel > 47) {
    throw ConfigurationException("Channel not in range (0-47) on input to translateFromROD");
  }

  unsigned int order = channel / 6;
  unsigned int id = (channel % 6) + 1;

  char expression[1000];
  sprintf(expression,
          "/configuration/partition[@id=%d]/crate[@id=%d]/rod[@id=%d]/MUR[@order=%d]",
          partition, crate, rod, order);

  xmlNodePtr moduleNode = getQueryNode(expression, "Translation from ROD");

  module = id; // getAttrAsInt(moduleNode, "id");
  MUR = getAttrAsInt(moduleNode, "id");

  log << " Translate from Rod " << MUR << ", " << module << std::endl;
}

void ConfigurationXMLImpl::translateToRROD(unsigned int MUR, unsigned int module,
                                    unsigned int &partition, unsigned int &crate,
                                    unsigned int &rod, unsigned int &channel) {
  log << "Translate to redundant Rod " << MUR << ", " << module << std::endl;

  if(module < 1 || module > 6) {
    throw ConfigurationException("MUR position out of range (1-6) on input to translateToRROD");
  }

  std::string sn;

  translateToSN(MUR, module, sn);

  char expression[1000];
  sprintf(expression, "/configuration/partition/crate/rod/MUR/rmodule[text()=\"%s\"]", sn.c_str());
  xmlNodePtr moduleNode = getQueryNode(expression, "Translate to Redundant ROD");

  // MUR order
  module = getAttrAsInt(moduleNode, "id");
  int order = getAttrAsInt(moduleNode->parent, "order");

  rod = getAttrAsInt(moduleNode->parent->parent, "id");
  crate = getAttrAsInt(moduleNode->parent->parent->parent, "id");
  partition = getAttrAsInt(moduleNode->parent->parent->parent->parent, "id");

  channel = order * 6 + module - 1;

  log << " Translate to redundant Rod " << partition << ", " << crate << ", " << rod << ", " << channel << "\n";
}

void ConfigurationXMLImpl::translateFromRROD(unsigned int partition, unsigned int crate,
                                      unsigned int rod, unsigned int channel,
                                      unsigned int &MUR, unsigned int &module) {
  log << "Translate from redundant Rod " << partition << ", " << crate << ", " << rod << ", " << channel << "\n";

  if(channel < 0 || channel > 47) {
    throw ConfigurationException("Channel not in range (0-47) on input to translateFromRROD");
  }

  unsigned int order = channel / 6;
  unsigned int id = (channel % 6) + 1;

  char expression[1000];
  sprintf(expression,
          "/configuration/partition[@id=%d]/crate[@id=%d]/rod[@id=%d]/MUR[@order=%d]/rmodule[@id=%d]/text()",
          partition, crate, rod, order, id);

  xmlNodePtr moduleNode = getQueryNode(expression, "Translation from Redundant ROD");

//   std::cout << (char *)(moduleNode->content) << std::endl;

  translateFromSN((char *)moduleNode->content, MUR, module);

  log << " Translate from redundant Rod " << MUR << ", " << module << "\n";
}

void ConfigurationXMLImpl::translateToSN(unsigned int MUR, unsigned int module,
                                         std::string &sn) {
  log << "Translate to serial number " << MUR << ", " << module << std::endl;

  if(module < 1 || module > 6) {
    throw ConfigurationException("MUR position out of range (1-6) on input to translateToSN");
  }

  char expression[1000];
  sprintf(expression, "/configuration/partition/crate/rod/MUR[@id=%d]/module[@id=\"%d\"]/text()", MUR, module);
  xmlNodePtr moduleNode = getQueryNode(expression, "Translation to SN");

  sn = (char *)moduleNode->content;

  log << " Translate to serial number " << sn << std::endl;
}

void ConfigurationXMLImpl::translateFromSN(std::string sn,
                                           unsigned int &MUR, unsigned int &module) {
  log << "Translate from serial number " << sn << "\n";

  if(!serialToMurNodeBuilt) 
    buildSerialToMurNode();

  xmlNodePtr murNode = serialToMurNode[sn];

  if(!murNode) 
    throw ConfigurationException(sn + " not mapped to MUR");

  xmlNodePtr moduleNode = getChildNodeWithContent(murNode, "module", sn.c_str());
  if(!moduleNode)
    throw ConfigurationException("Can't find " + sn);

  module = getAttrAsInt(moduleNode, "id");
  MUR = getAttrAsInt(murNode, "id");

  log << " Translate From serial number " << MUR << ", " << module << std::endl;
}

void ConfigurationXMLImpl::buildSerialToMurNode() {
  char expression[1000];
  sprintf(expression, "/configuration/partition/crate/rod/MUR/module");

  std::list<xmlNodePtr> moduleNodes = getQueryNodeList(expression, "Build translation from SN");

  serialToMurNode.clear();

  for(std::list<xmlNodePtr>::const_iterator iter = moduleNodes.begin(); 
      iter != moduleNodes.end();
      iter ++) {
    xmlNodePtr current = *iter;

    char *cSerial = (char *)xmlNodeGetContent(current);

    std::string serial(cSerial);

    serialToMurNode[serial] = current->parent;

    free(cSerial);
  }

  serialToMurNodeBuilt = true;
}

void ConfigurationXMLImpl::translateToRMUR(unsigned int MUR, unsigned int module,
                                           unsigned int &RMUR, unsigned int &rmodule) {
  log << "Translate to rmur " << MUR << " " << module << "\n";

  if(module < 1 || module > 6) {
    throw ConfigurationException("MUR position out of range (1-6) on input to translateToRMUR");
  }

  std::string sn;

  translateToSN(MUR, module, sn);

  char expression[1000];
  sprintf(expression,
          "/configuration/partition/crate/rod/MUR/rmodule[text()=\"%s\"]",
          sn.c_str());

  xmlNodePtr moduleNode = getQueryNode(expression, "Translate to Redundant MUR");

  rmodule = getAttrAsInt(moduleNode, "id");
  RMUR = getAttrAsInt(moduleNode->parent, "id");

  log << " Translate to redundant MUR " << RMUR << ", " << rmodule << "\n";
}

void ConfigurationXMLImpl::translateFromRMUR(unsigned int RMUR, unsigned int rmodule,
                                             unsigned int &MUR, unsigned int &module) {
  log << "Translate from rmur " << RMUR << " " << rmodule << "\n";

  if(rmodule < 1 || rmodule > 6) {
    throw ConfigurationException("MUR position (redundant) out of range (1-6) on input to translateFromRMUR");
  }

  char expression[1000];
  sprintf(expression,
          "/configuration/partition/crate/rod/MUR[@id=%d]/rmodule[@id=%d]/text()",
          RMUR, rmodule);

  xmlNodePtr moduleNode = getQueryNode(expression, "Translate from Redundant MUR");

  translateFromSN((char *)moduleNode->content, MUR, module);

  log << " Translate from redundant MUR " << MUR << ", " << module << "\n";
}

void ConfigurationXMLImpl::translateToBarrel(unsigned int MUR, unsigned int module,
                                             unsigned int &barrel, unsigned int &row, int &number) {
  log << "Translate to Barrel " << MUR << ", " << module << std::endl;

  if(module < 1 || module > 6) {
    throw ConfigurationException("MUR position out of range (1-6) on input to translateToBarrel");
  }

  char expression[1000];
  sprintf(expression, "/configuration/geography/barrel/row/MUR[@id=%d]", MUR);
  xmlNodePtr moduleNode = getQueryNode(expression, "Translation to Barrel");

  int position = getAttrAsInt(moduleNode, "position");
  row = getAttrAsInt(moduleNode->parent, "id");
  barrel = getAttrAsInt(moduleNode->parent->parent, "id");
  number = position * module;

  log << " Translate to Barrel " << barrel << ", " << row << ", " << number << std::endl;
}

void ConfigurationXMLImpl::translateFromBarrel(unsigned int barrel, unsigned int row, int number,
                                               unsigned int &MUR, unsigned int &module) {
  log << "Translate from barrel " << barrel << ", " << row << ", " << number << "\n";

  if(barrel < 3 || barrel > 6) {
    throw ConfigurationException("Barrel not in range (3-6) on input to translateFromBarrel");
  }

  if(row < 0 || row > 55) {
    throw ConfigurationException("Barrel row not in range (0-55) on input to translateFromBarrel");
  }

  if(number < -6 || number > 6 || number == 0) {
    throw ConfigurationException("Barrel number not in range (-6 to -1 or 1 to 6) on input to translateFromBarrel");
  }

  int position;

  if(number < 0) {
    position = -1;
  } else {
    position = 1;
  }

  char expression[1000];
  sprintf(expression,
          "/configuration/geography/barrel[@id=%d]/row[@id=%d]/MUR[@position=%d]",
          barrel, row, position);

  xmlNodePtr moduleNode = getQueryNode(expression, "Translation from Barrel");

  module = (unsigned int)(number * position);
  MUR = getAttrAsInt(moduleNode, "id");

  log << " Translate from Barrel " << MUR << ", " << module << std::endl;
}

/* Endcap Internals:
   position of MUR in quadrant
   number is left to right top to bottom (from 0)
      0000001222222   (0-12)
        4444455555    (13-22)
        1111133333    (23-32)
*/
void ConfigurationXMLImpl::translateToEndcap(unsigned int MUR, unsigned int module,
                                             int &disk, unsigned int &quadrant, unsigned int &number) {
  log << "Translate to Endcap " << MUR << ", " << module << std::endl;

  if(module < 1 || module > 6) {
    throw ConfigurationException("MUR position out of range (1-6) on input to translateToEndcap");
  }

  char expression[1000];
  sprintf(expression, "/configuration/geography/endcap/quadrant/MUR[@id=%d]", MUR);
  xmlNodePtr moduleNode = getQueryNode(expression, "Translation to Endcap");

  quadrant = getAttrAsInt(moduleNode->parent, "id");
  disk = getAttrAsInt(moduleNode->parent->parent, "id");

  if((disk == 9 || disk == -9) && quadrant == 4) {
    quadrant = module - 1;
    number = 6;
  } else {
    // Position isn't needed by "quadrant 4"
    int position = getAttrAsInt(moduleNode, "position");

    switch(position) {
    case 0: number = module - 1; break;
    case 1: number = (module==1)?6:module + 21; break;
    case 2: number = module + 6; break;
    case 3: number = module + 27; break;
    case 4: number = module + 12; break;
    case 5: number = module + 17; break;
    }
  }

  log << " Translate to endcap " << disk << ", " << quadrant << ", " << number << "\n";
}

void ConfigurationXMLImpl::translateFromEndcap(int disk, unsigned int quadrant, unsigned int number,
                                               unsigned int &MUR, unsigned int &module) {
  log << "Translate from endcap " << disk << ", " << quadrant << ", " << number << "\n";

  if(disk < -9 || disk > 9 || disk == 0) {
    throw ConfigurationException("Endcap disk not in range (-9 to -1 or 1 to 9) on input to translateFromEndcap");
  }

  if(quadrant < 0 || quadrant > 3) {
    throw ConfigurationException("Endcap quadrant not in range (0 to 3) on input to translateFromEndcap");
  }

  if(number < 0 || number > 32) {
    throw ConfigurationException("Endcap module number not in range (0 to 32) on input to translateFromEndcap");
  }

  int position;

  if((disk == 9 || disk == -9) && number == 6) {
    position = -1; 
    module = quadrant + 1;
    quadrant = 4; 
  } else if(number < 6) {
    position = 0; module = number + 1;
  } else if(number == 6) {
    position = 1; module = 1;
  } else if(number < 13) {
    position = 2; module = number - 6;
  } else if(number < 18) {
    position = 4; module = number - 12;
  } else if(number < 23) {
    position = 5; module = number - 17;
  } else if(number < 28) {
    position = 1; module = number - 21;
  } else if(number < 33) {
    position = 3; module = number - 27;
  } else {
    log << "Number too big in translate from endcap " << number << "\n";
    throw ConfigurationException("Number to big in translate from endcap " + boost::lexical_cast<std::string>(number));
  }

  char expression[1000];
  if(position == -1) {
    sprintf(expression,
            "/configuration/geography/endcap[@id=%d]/quadrant[@id=%d]/MUR",
            disk, quadrant);
  } else {
    sprintf(expression,
            "/configuration/geography/endcap[@id=%d]/quadrant[@id=%d]/MUR[@position=%d]",
            disk, quadrant, position);
  }

  xmlNodePtr moduleNode = getQueryNode(expression, "Translation from Endcap");

  MUR = getAttrAsInt(moduleNode, "id");

  log << " Translate from Endcap " << MUR << ", " << module << std::endl;
}

void ConfigurationXMLImpl::translateToPowerSupply(unsigned int MUR, unsigned int module,
                                           unsigned int &partition, 
                                           unsigned int &crate, unsigned int &channel) {  
  log << "Translate to Power Supply " << MUR << ", " << module << std::endl;

  if(module < 1 || module > 6) {
    throw ConfigurationException("MUR position out of range (1-6) on input to translateToPowerSupply");
  }

  char expression[1000];
  sprintf(expression, "/configuration/power/crate/channel[@MUR=%d and @module=%d]", MUR, module);
  xmlNodePtr moduleNode = getQueryNode(expression, "Translation to Power Supply");

  channel = getAttrAsInt(moduleNode, "id");
  crate = getAttrAsInt(moduleNode->parent, "id");
  partition = 0; // getAttrAsInt(moduleNode->parent->parent, "id");

  log << " Translate to Power Supply " << partition << ", " << crate << ", " << channel << std::endl;
}

void ConfigurationXMLImpl::translateFromPowerSupply(unsigned int partition, 
                                                    unsigned int crate, unsigned int channel, 
                                                    unsigned int &MUR, unsigned int &module) {
  log << "Translate from Power Supply " << partition << ", " << crate << ", " << channel << std::endl;

  char expression[1000];
  sprintf(expression, "/configuration/power/crate[@id=%d]/channel[@id=%d]", crate, channel);
  xmlNodePtr moduleNode = getQueryNode(expression, "Translation from Power Supply");

  MUR = getAttrAsInt(moduleNode, "MUR");
  module = getAttrAsInt(moduleNode, "module");

  log << " Translate from Power Supply " << MUR << ", " << module << std::endl;
}


/*
  Return the configuration for the given module
*/
ABCDModule ConfigurationXMLImpl::getModuleConfig(std::string module) {
  log << "Get module configuration " << module << std::endl;

  xmlNodePtr moduleNode = getModuleConfigurationNode(module, "getModuleConfig");
  if(!moduleNode) 
    throw ConfigurationException("Failure to look up module");

  return parseModuleConfig(moduleNode);
}

short ConfigurationXMLImpl::getModuleGroup(const std::string module) {
  log << "Get module group " << module << std::endl;

  if(!serialToMurNodeBuilt) 
    buildSerialToMurNode();

  xmlNodePtr murNode = serialToMurNode[module];

  if(!murNode) 
    throw ConfigurationException(module + " not mapped to MUR (for group)");

  xmlNodePtr moduleNode = getChildNodeWithContent(murNode, "module", module.c_str());
  if(!moduleNode)
    throw ConfigurationException("Can't find module! " + module);

  try {
    return getAttrAsInt(moduleNode, "group");
  } catch(ConfigurationException &c) {
    return 0;
  }
}

void ConfigurationXMLImpl::setModuleGroup(const std::string module, short group) {
  log << "Set module group " << module << ", " << group << std::endl;

  char expression[1000];
  sprintf(expression, 
          "/configuration//MUR/module[text()=\"%s\"]", // /@group", 
          module.c_str());

  xmlNodePtr moduleNode = getQueryNode(expression, "Set module group");

  setIntAttr(moduleNode, "group", group);
}

/*
  Parse the xml tree provided and return the module configuration
*/
const ABCDModule ConfigurationXMLImpl::parseModuleConfig(xmlNodePtr moduleNode) {
  ABCDModule result;

  for(unsigned int i=0; i<sizeof(result); i++) {
    ((char*)&result)[i] = 0;
  }

  if(moduleNode) {
    //    xmlElemDump(stdout, document, moduleNode);

    for(xmlNodePtr child = moduleNode->children;
        child;
        child = child->next) {
      if(child->type == XML_ELEMENT_NODE) {
        char * content = (char *)xmlNodeGetContent(child);

        //        std::cout << "Tag in module " << (char *)child->name << std::endl;;

        if(strcmp((char *)child->name, "active") == 0) {
          result.active = atoi(content);
        } else if(strcmp((char *)child->name, "group") == 0) {
          result.groupId = atoi(content);
        } else if(strcmp((char *)child->name, "select") == 0) {
          result.select = atoi(content);
        } else if(strcmp((char *)child->name, "config") == 0) {
          log << "Using old active/select format\n";
          // Old config format
          if(xmlGetProp(child, (xmlChar *)"active") &&
             xmlGetProp(child, (xmlChar *)"select")) {
            result.active = atoi((char *)xmlGetProp(child, (xmlChar *)"active"));
            result.select = atoi((char *)xmlGetProp(child, (xmlChar *)"select"));
          } else {
            log << "module config without active or select tags\n";
          }
        } else if(strcmp((char *)child->name, "chip") == 0) {
          try {
            int id = getAttrAsInt(child, "id", 10);   // This is stored with a leading 0 so it shouldn't be octal!!!
            int active = getAttrAsInt(child, "active");
            int address = getAttrAsInt(child, "address");

            result.chip[id].active = active;
            result.chip[id].address = address;

            for(xmlNodePtr chipChild = child->children;
                chipChild;
                chipChild = chipChild->next) {
              char * chipContent = (char *)xmlNodeGetContent(chipChild);

              //            std::cout << "Tag in chip " << (char *)chipChild->name << std::endl;

              if(chipChild->type != XML_ELEMENT_NODE) {
              } else if(strcmp((char *)chipChild->name, "config") == 0) {
                char *eptr = chipContent;
                result.chip[id].basic.config.padding = 0;
                for(int i=0; i<11; i++) {
                  int value;
                  char *cptr = eptr;
                  value = strtol(cptr, &eptr, 0);
//                perror("Error converting");
                  switch(i) {
                  case 0: result.chip[id].basic.config.readoutMode = value; break;
                  case 1: result.chip[id].basic.config.calibMode = value; break;
                  case 2: result.chip[id].basic.config.trimRange = value; break;
                  case 3: result.chip[id].basic.config.edgeDetect = value; break;
                  case 4: result.chip[id].basic.config.mask = value; break;
                  case 5: result.chip[id].basic.config.accumulate = value; break;
                  case 6: result.chip[id].basic.config.inputBypass = value; break;
                  case 7: result.chip[id].basic.config.outputBypass = value; break;
                  case 8: result.chip[id].basic.config.master = value; break;
                  case 9: result.chip[id].basic.config.end = value; break;
                  case 10: result.chip[id].basic.config.feedThrough = value; break;
                  }
                }
              } else if(strcmp((char *)chipChild->name, "mask") == 0) {
                // Should be more robust?
                char *eptr = chipContent;
                for(int m=3; m>=0; m--) {
                  char *cptr = eptr;
                  result.chip[id].basic.mask[m] = strtoul(cptr, &eptr, 0);
                }
              } else if(strcmp((char *)chipChild->name, "reg") == 0) {
                try {
                  std::string name = getAttrAsString(chipChild, "name");

                  if(strcmp(name.c_str(), "vthr") == 0) {
                    result.chip[id].basic.vthr = strtol(chipContent, NULL, 0);
                  } else if(strcmp(name.c_str(), "vcal") == 0) {
                    result.chip[id].basic.vcal = strtol(chipContent, NULL, 0);
                  } else if(strcmp(name.c_str(), "delay") == 0) {
                    result.chip[id].basic.delay = strtol(chipContent, NULL, 0);
                  } else if(strcmp(name.c_str(), "preamp") == 0) {
                    result.chip[id].basic.preamp = strtol(chipContent, NULL, 0);
                  } else if(strcmp(name.c_str(), "shaper") == 0) {
                    result.chip[id].basic.shaper = strtol(chipContent, NULL, 0);
                  } else {
                    log << "Unknown tag for reg value " << name << std::endl;
                  }
                } catch(ConfigurationException &c) {
                  log << "chip reg with no name attribute\n";
                }
              } else if(strcmp((char *)chipChild->name, "rc_function") == 0) {
                try {
                  result.chip[id].caldata.rc_function = getAttrAsInt(chipChild, "type");

                  char *token = strtok(chipContent, " ");
                  int nextVal = -1;
                  do {
                    if(strcmp(token, "p0") == 0) {
                      nextVal = 0;
                    } else if(strcmp(token, "p1") == 0) {
                      nextVal = 1;
                    } else if(strcmp(token, "p2") == 0) {
                      nextVal = 2;
                    } else {
                      if(nextVal < 0) {
                        log << "Bad rc function parameters " << token << std::endl;
                      } else {
                        result.chip[id].caldata.rc_params[nextVal] = strtod(token, NULL);
                      }
                      nextVal = -1;
                    }
                  } while((token = strtok(NULL, " ")) != 0);
                } catch(ConfigurationException &c) {
                  log << "No type attribute on rc_function\n";
                }
              } else if(strcmp((char *)chipChild->name, "c_factor") == 0) {
                result.chip[id].caldata.c_factor = strtod(chipContent, NULL);
              } else if(strcmp((char *)chipChild->name, "target") == 0) {
                double val = strtod(chipContent, NULL);
                result.chip[id].target = ((int)((val/2.5)+0.5));
              } else if(strcmp((char *)chipChild->name, "trim") == 0) {
                try {
                  int start = getAttrAsInt(chipChild, "start", 16);
                  int end = getAttrAsInt(chipChild, "end", 16);

                  int length = end-start+1;

                  if((length%8) != 0) log << "Can't interpret trim values not in blocks of 8\n";
                  else {
                    char *endPtr = chipContent;
                    char *startPtr = chipContent;
                    int curr = start;
                    do {
                      startPtr = endPtr;
                      unsigned long trimValues = strtoul(startPtr, &endPtr, 0);

                      if(curr > end) {
                        if(startPtr != endPtr) {
                          log << "Too many numbers in trim (range " << start << "-" << end << ")\n";
                        }
                        break;
                      }

                      for(int i=0; i<8; i++) {
                        unsigned char val = (trimValues >> ((7-i)*4)) & 0x0000000f;

                        result.chip[id].trim[curr+i] = val;
//                          if(val>9) val += 'a'-10;
//                          else val += '0';
                      }
                      curr += 8;
                    } while(*endPtr!=0);
                  }
                } catch(ConfigurationException &c) {
                  log << "No start and end tags on trim\n";
                } // End trim attribute check
              } // End if over chip tags

              free(chipContent);
            } // End loop over chip tags
          } catch(ConfigurationException &c) {
            log << "Bad attributes to chip\n";
          } // End chip attribute check
        } else {
//           std::cout << "Unrecognised tag in module " << child->name << "\n";
        } // End if over module tags

        free(content);
      } // Tag not an element
    } // End loop over module tags
  } else {
    log << "Module configuration not found\n";
    throw ConfigurationException("Module configuration not found");
  }

  return result;
}

char *ConfigurationXMLImpl::getFibreMappings(unsigned int partition, unsigned int crate, unsigned int rod) {
  log << "Get BOC fibre mappings\n";
  
  char *result = new char[48*3];

  for(int channel = 0; channel < 48; channel++) {
    char expression[1000];

    int order = channel/6;
    int chan = (channel % 6) + 1;

    // Defaults
    result[channel * 3 + 0] = channel;
    result[channel * 3 + 1] = channel * 2;
    result[channel * 3 + 2] = channel * 2 + 1;

    sprintf(expression,
            "/configuration/partition[@id=%d]/crate[@id=%d]/rod[@id=%d]/MUR[@order=%d]/channel[@id=%d]",
            partition, crate, rod, order, chan);

    try {
      xmlNodePtr c = getQueryNode(expression, "Get fibre mappings");
      
      try {
        xmlNodePtr node = getChildNode(c, "output");
        result[channel * 3 + 0] = getAttrAsInt(node, "fibre");
      } catch(ConfigurationException &c) {
        // Leave as default
      }

      try {
        xmlNodePtr node = getChildNode(c, "stream0");
        result[channel * 3 + 1] = getAttrAsInt(node, "fibre");
      } catch(ConfigurationException &c) {
        //        std::cout << "Bad stream0 tag in channel " << channel << std::endl;
      }

      try {
        xmlNodePtr node = getChildNode(c, "stream1");
        result[channel * 3 + 2] = getAttrAsInt(node, "fibre");
      } catch(ConfigurationException &c) {
        //        std::cout << "Bad stream1 tag in channel " << channel << std::endl;
      }
    } catch(ConfigurationException &c) {
      //      std::cout << "Bad configuration for channel " << channel << std::endl;
    }
  }

  return result;
}

void ConfigurationXMLImpl::setFibreMapping(unsigned int partition, unsigned int crate, unsigned int rod, 
                                           unsigned int channel, unsigned int tx, unsigned int rx0, unsigned int rx1) {
  log << "Set BOC fibre mappings for " << partition << ", " << crate << ", " << rod << ", " << channel << "\n";

  if(channel < 0 || channel > 47) {
    throw ConfigurationException("Channel not in range (0 to 47) on input to setFibreMapping");
  }

  unsigned int MUR, module;
  translateFromROD(partition, crate, rod, channel, MUR, module);

  char expression[1000];

  sprintf(expression,
          "/configuration/partition[@id=%d]/crate[@id=%d]/rod[@id=%d]/MUR[@id=%d]",
          partition, crate, rod, MUR);

  // If this fails just propogate the exception
  xmlNodePtr murNode = getQueryNode(expression, "Set fibre mapping");

  xmlNodePtr channelNode = getOrNewChildWithIntAttr(murNode, "channel", "id", module);

  xmlNodePtr outputNode = getOrNewChildNode(channelNode, "output");
  setIntAttr(outputNode, "fibre", tx);

  xmlNodePtr stream0 = getOrNewChildNode(channelNode, "stream0");
  setIntAttr(stream0, "fibre", rx0);

  xmlNodePtr stream1 = getOrNewChildNode(channelNode, "stream1");
  setIntAttr(stream1, "fibre", rx1);

#warning "Is this the right one?"
  notifyModuleConfChange();
}

BOCChannelConfig ConfigurationXMLImpl::getBOCConfig(unsigned int partition, unsigned int crate, unsigned int rod,
                                                    unsigned int channel) {
  log << "Get BOC configuration\n";

  if(channel < 0 || channel > 47) {
    throw ConfigurationException("Channel not in range (0 to 47) on input to getBOCConfig");
  }

  BOCChannelConfig result;

  char expression[1000];

  int order = channel/6;
  int chan = (channel % 6) + 1;

  sprintf(expression,
          "/configuration/partition[@id=%d]/crate[@id=%d]/rod[@id=%d]/MUR[@order=%d]/channel[@id=%d]",
          partition, crate, rod, order, chan);

  xmlNodePtr c = getQueryNode(expression, "Get BOC Channel Configuration");

  xmlNodePtr node = getChildNode(c, "output");

  result.current = getAttrAsInt(node, "current");
  result.delay = getAttrAsInt(node, "delay");
  result.markSpace = getAttrAsInt(node, "markSpace");

  node = getChildNode(c, "stream0");
  result.threshold0 = getAttrAsInt(node, "threshold");
  result.delay0 = getAttrAsInt(node, "delay");

  node = getChildNode(c, "stream1");
  result.threshold1 = getAttrAsInt(node, "threshold");
  result.delay1 = getAttrAsInt(node, "delay");

  return result;
}

BOCGlobalConfig ConfigurationXMLImpl::getBOCGlobalConfig(unsigned int partition, unsigned int crate, unsigned int rod) {
  log << "Get BOC global configuration\n";

  BOCGlobalConfig result;

  char expression[1000];

  result.validMask = 0;

  sprintf(expression,
          "/configuration/partition[@id=%d]/crate[@id=%d]/rod[@id=%d]/BOC",
          partition, crate, rod);

  xmlNodePtr bocNode;
  try {
    bocNode = getQueryNode(expression, "Get BOC Global Configuration");
  } catch(ConfigurationException &c) {
    // All variables masked off
    return result;
  }

  for(xmlNodePtr child = bocNode->children;
      child;
      child = child->next) {

    if(child->type == XML_ELEMENT_NODE) {
      std::string regName = getAttrAsString(child, "name");
      int value = getAttrAsInt(child, "value");
      if(regName == "clockControl") {
        result.clockControl = value;
        result.validMask |= 1<<0;
      } else if(regName == "rxDataMode") {
        result.rxDataMode = value;
        result.validMask |= 1<<1;
      } else if(regName == "rxDacClear") {
        result.rxDacClear = value;
        result.validMask |= 1<<2;
      } else if(regName == "txDacClear") {
        result.txDacClear = value;
        result.validMask |= 1<<3;
      } else if(regName == "vernierFinePhase") {
        result.vernierFinePhase = value;
        result.validMask |= 1<<4;
      } else if(regName == "vernierClockPhase0") {
        result.vernierClockPhase0 = value;
        result.validMask |= 1<<5;
      } else if(regName == "vernierClockPhase1") {
        result.vernierClockPhase1 = value;
        result.validMask |= 1<<6;
      } else if(regName == "bpmClockPhase") {
        result.bpmClockPhase = value;
        result.validMask |= 1<<7;
      } else if(regName == "bregClockPhase") {
        result.bregClockPhase = value;
        result.validMask |= 1<<8;
      }
    }
  }

  return result;
}

void ConfigurationXMLImpl::mapBarrelMUR(unsigned int MUR, unsigned int barrel, unsigned int row, int position) {
  log << "Map barrel MUR " << MUR << ", " << barrel << ", " << row << ", " << position << std::endl;

  if(!(position == 1 || position == -1)) {
    throw ConfigurationException("Barrel position one of -1 (left) and 1 (right)");
  }

  if(barrel < 3 || barrel > 6) {
    throw ConfigurationException("Barrel must be one of 3, 4, 5, 6");
  }

  if(row < 0 || row >= 56) {
    throw ConfigurationException("Barrel row must be in range (0-55)");
  }

  char expression[1000];

  // Check if MUR already in map
  sprintf(expression,
          "/configuration/geography/barrel/row/MUR[@id=%d]",
          MUR);

  try {
    xmlNodePtr node = getQueryNode(expression, "Check MUR for barrel remap");

    xmlUnlinkNode(node);
    xmlFreeNode(node);
  } catch (ConfigurationException &e) {
    // This is OK, we don't have to remove it
  }

  // Get barrel and row
  sprintf(expression,
          "/configuration/geography/barrel[@id=%d]/row[@id=%d]",
          barrel, row);

  xmlNodePtr rowNode = 0;
  try {
    rowNode = getQueryNode(expression, "Check if barrel geography exist");

    // If MUR with this position there then delete it
    for(xmlNodePtr murNode = rowNode->children;
        murNode;
        ) {

      if(murNode->type == XML_ELEMENT_NODE &&
         (getAttrAsInt(murNode, "position") == position)) {
        xmlNodePtr nextNode = murNode->next;
        xmlUnlinkNode(murNode);
        xmlFreeNode(murNode);
        murNode = nextNode;
      } else {
        murNode = murNode->next;
      }
    }
  } catch(ConfigurationException &e) {
    xmlNodePtr root = xmlDocGetRootElement(document);

    xmlNodePtr geog = getOrNewChildNode(root, "geography");

    xmlNodePtr barrelNode = getOrNewChildWithIntAttr(geog, "barrel", "id", barrel);
    rowNode = getOrNewChildWithIntAttr(barrelNode, "row", "id", row);
  }

  if(rowNode) {
    // Check not already a node at this position
    xmlNodePtr testMURPosition = getNodeWithIntAttr(rowNode, "MUR", "position", position);
    if(testMURPosition && (getAttrAsInt(testMURPosition, "id", 10) != MUR)) {
      xmlUnlinkNode(testMURPosition);
      xmlFreeNode(testMURPosition);
    }

    // MUR insert
    xmlNodePtr newMUR = getOrNewChildWithIntAttr(rowNode, "MUR", "id", MUR);
    setIntAttr(newMUR, "position", position);
  }

  notifyModuleMappingChange();
}

void ConfigurationXMLImpl::unmapBarrelMUR(unsigned int MUR) {
  log << "Unmap barrel MUR " << MUR << std::endl;

  char expression[1000];

  // Get MUR
  sprintf(expression,
          "/configuration/geography/barrel/row/MUR[@id=%d]",
          MUR);

  try {
    xmlNodePtr node = getQueryNode(expression, "Get barrel MUR for unmap");

    xmlUnlinkNode(node);
    xmlFreeNode(node);

    notifyModuleMappingChange();
  } catch (ConfigurationException &e) {
    // This is OK, we don't have to remove it
  }
}

void ConfigurationXMLImpl::mapEndcapMUR(unsigned int MUR, int disk, unsigned int quadrant, unsigned int position) {
  log << "Map endcap MUR " << MUR << ", " << disk << ", " << quadrant << ", " << position << std::endl;

  if(position < 0 || position > 5) {
    throw ConfigurationException("Endcap position must be (0-5)");
  }

  if(disk < -9 || disk > 9 || disk == 0) {
    throw ConfigurationException("Endcap disk must be (-9 to -1, 1 to 9)");
  }

  if(quadrant < 0 || quadrant > 3) {
    if(disk == 9 || disk == -9) {
      if(!(quadrant == 4)) {
        throw ConfigurationException("Endcap quadrant must be (0-4)");
      }
    } else {
      throw ConfigurationException("Endcap quadrant must be (0-3)");
    }
  }

  char expression[1000];

  // Check if MUR already in map
  sprintf(expression,
          "/configuration/geography/endcap/quadrant/MUR[@id=%d]",
          MUR);

#warning "Should check if was a barrel MUR?"

  try {
    xmlNodePtr node = getQueryNode(expression, "Check Endcap MUR for remap");

    xmlUnlinkNode(node);
    xmlFreeNode(node);
  } catch (ConfigurationException &e) {
    // OK, MUR was unmapped before
  }

  // Get barrel and row
  sprintf(expression,
          "/configuration/geography/endcap[@id=%d]/quadrant[@id=%d]",
          disk, quadrant);

  xmlNodePtr quadrantNode = 0;
  try {
    quadrantNode = getQueryNode(expression, "Check if endcap geography exist");

    // If MUR with this position there then delete it
    for(xmlNodePtr murNode = quadrantNode->children;
        murNode;
        ) { // Increment doesn't work if node gets deleted

      if((murNode->type == XML_ELEMENT_NODE) 
         && (getAttrAsInt(murNode, "position") == position)) {
        xmlNodePtr nextNode = murNode->next;
        xmlUnlinkNode(murNode);
        xmlFreeNode(murNode);
        murNode = nextNode;
      } else {
        murNode = murNode->next;
      }
    }
  } catch(ConfigurationException &e) {
    xmlNodePtr root = xmlDocGetRootElement(document);

    xmlNodePtr geog = getOrNewChildNode(root, "geography");

    xmlNodePtr endcapNode = 0;

    endcapNode = getOrNewChildWithIntAttr(geog, "endcap", "id", disk);

    if(!endcapNode) {
      endcapNode = getOrNewChildNode(geog, "endcap");

      char buffer[5];
      snprintf(buffer, 5, "%d", disk);
      xmlSetProp(endcapNode, (xmlChar *)"id", (xmlChar *)buffer);
    }

    quadrantNode = getOrNewChildWithIntAttr(endcapNode, "quadrant", "id", quadrant);
  }

  if(quadrantNode) {
    // Check not already a node at this position
    xmlNodePtr testMURPosition = getNodeWithIntAttr(quadrantNode, "MUR", "position", position);
    if(testMURPosition && (getAttrAsInt(testMURPosition, "id", 10) != MUR)) {
      xmlUnlinkNode(testMURPosition);
      xmlFreeNode(testMURPosition);
    }

    // MUR insert
    xmlNodePtr newMUR = getOrNewChildWithIntAttr(quadrantNode, "MUR", "id", MUR);
    setIntAttr(newMUR, "position", position);
  }

  notifyModuleMappingChange();
}

void ConfigurationXMLImpl::unmapEndcapMUR(unsigned int MUR) {
  log << "Unmap endcap MUR " << MUR << std::endl;

  char expression[1000];

  // Get MUR
  sprintf(expression,
          "/configuration/geography/endcap/quadrant/MUR[@id=%d]",
          MUR);

  try {
    xmlNodePtr node = getQueryNode(expression, "Get Endcap MUR for unmap");

    xmlUnlinkNode(node);
    xmlFreeNode(node);

    notifyModuleMappingChange();
  } catch (ConfigurationException &e) {
    // This is OK, we don't have to remove it
  }
}

void ConfigurationXMLImpl::swapMURNames(unsigned int MUR1, unsigned int MUR2) {
  log << "swapMURNames " << MUR1 << ", " << MUR2 << std::endl;

  // Find everything to do with MUR1
  std::list<xmlNodePtr> idPointers1;

  {
    char expression[1000];
    sprintf(expression, "//MUR[@id=%d]", MUR1);
  
    xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)expression, xpathContext);

    if(queryResult) {
      if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
        for(int i=0; i<queryResult->nodesetval->nodeNr; i++) {
          xmlNodePtr curr = queryResult->nodesetval->nodeTab[i];

          idPointers1.push_back(curr);
        }
      } else {
        log << "Bad query result data\n";
      }

      xmlXPathFreeObject(queryResult);
    } else {
      log << "No id pointers to MUR1\n";
    }
  }

  // Power mappings have channel elements with MUR attribute!
  std::list<xmlNodePtr> channelPointers1;

  {
    char expression[1000];
    sprintf(expression, "//channel[@MUR=%d]", MUR1);
  
    xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)expression, xpathContext);

    if(queryResult) {
      if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
        for(int i=0; i<queryResult->nodesetval->nodeNr; i++) {
          xmlNodePtr curr = queryResult->nodesetval->nodeTab[i];

          channelPointers1.push_back(curr);
        }
      } else {
        log << "Bad query result data\n";
      }

      xmlXPathFreeObject(queryResult);
    } else {
      log << "No id pointers to MUR1\n";
    }
  }

  // Find everything to do with MUR2
  std::list<xmlNodePtr> idPointers2;

  {
    char expression[1000];
    sprintf(expression, "//MUR[@id=%d]", MUR2);
  
    xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)expression, xpathContext);

    if(queryResult) {
      if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
        for(int i=0; i<queryResult->nodesetval->nodeNr; i++) {
          xmlNodePtr curr = queryResult->nodesetval->nodeTab[i];

          idPointers2.push_back(curr);
        }
      } else {
        log << "Bad query result data\n";
      }

      xmlXPathFreeObject(queryResult);
    } else {
      log << "No id pointers to MUR1\n";
    }
  }

  // Power mappings have channel elements with MUR attribute!
  std::list<xmlNodePtr> channelPointers2;

  {
    char expression[1000];
    sprintf(expression, "//channel[@MUR=%d]", MUR2);
  
    xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)expression, xpathContext);

    if(queryResult) {
      if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
        for(int i=0; i<queryResult->nodesetval->nodeNr; i++) {
          xmlNodePtr curr = queryResult->nodesetval->nodeTab[i];

          channelPointers2.push_back(curr);
        }
      } else {
        log << "Bad query result data\n";
      }

      xmlXPathFreeObject(queryResult);
    } else {
      log << "No id pointers to MUR1\n";
    }
  }

  // Write over MUR1 with MUR2
  for(std::list<xmlNodePtr>::const_iterator pointer = idPointers1.begin();
      pointer != idPointers1.end();
      pointer ++) {
    setIntAttr(*pointer, "id", MUR2);
  }

  for(std::list<xmlNodePtr>::const_iterator pointer = channelPointers1.begin();
      pointer != channelPointers1.end();
      pointer ++) {
    setIntAttr(*pointer, "MUR", MUR2);
  }

  // Write over MUR2 with MUR1
  for(std::list<xmlNodePtr>::const_iterator pointer = idPointers2.begin();
      pointer != idPointers2.end();
      pointer ++) {
    setIntAttr(*pointer, "id", MUR1);
  }

  for(std::list<xmlNodePtr>::const_iterator pointer = channelPointers2.begin();
      pointer != channelPointers2.end();
      pointer ++) {
    setIntAttr(*pointer, "MUR", MUR1);
  }

  notifyModuleMappingChange();
}

void ConfigurationXMLImpl::mapPowerChannel(unsigned int MUR, unsigned int number, 
                                           unsigned int partition, unsigned int crate, unsigned int channel) {
  log << "mapPowerChannel " << MUR << ", " << number << ", " << partition << ", " << crate << ", " << channel << std::endl;

  if(number < 1 || number > 6) {
    throw ConfigurationException("MUR position out of range (1-6) on input to mapPowerChannel");
  }

  if(channel < 0 || channel > 47) {
    throw ConfigurationException("Power supply channel must be 0-47 on input to mapPowerChannel");
  }

  char expression[1000];

#warning "Should check the MUR/number exists elsewhere?"

  // Find old map if there
  sprintf(expression, "/configuration/power/crate/channel[@MUR=%d and @module=%d]", MUR, number);
  try {
    xmlNodePtr oldChannelNode = getQueryNode(expression, "Check MUR power supply channel");
    clearAttr(oldChannelNode, "MUR");
    clearAttr(oldChannelNode, "module");
  } catch(ConfigurationException &c) {
    // Doesn't matter if not found
  }

  // Get crate to put channel in
  sprintf(expression, "/configuration/power/crate[@id=%d]/channel[@id=%d]", crate, channel);

  xmlNodePtr channelNode = 0;
  try {
    channelNode = getQueryNode(expression, "Map Power supply channel");
  } catch(ConfigurationException &e) {
    xmlNodePtr root = xmlDocGetRootElement(document);

    xmlNodePtr power = getOrNewChildNode(root, "power");

    //    xmlNodePtr partitionNode = getOrNewChildWithIntAttr(power, "partition", "id", partition);
    xmlNodePtr crateNode = getOrNewChildWithIntAttr(power /* partitionNode */, "crate", "id", crate);
    channelNode = getOrNewChildWithIntAttr(crateNode, "channel", "id", channel);
  }

  setIntAttr(channelNode, "MUR", MUR);
  setIntAttr(channelNode, "module", number);

  notifyModuleMappingChange();
}

void ConfigurationXMLImpl::unmapPowerChannel(unsigned int MUR, unsigned int number) {
  log << "unmapPowerChannel " << MUR << ", " << number << std::endl;

  if(number < 1 || number > 6) {
    throw ConfigurationException("Number not in range (1 to 6) on input to unmapPowerChannel");
  }

  char expression[1000];

  // Get MUR
  sprintf(expression,
          "/configuration/power/crate/channel[@MUR=%d and module=%d]",
          MUR, number);

  try {
    xmlNodePtr node = getQueryNode(expression, "Get Power channel for unmap");

    xmlUnlinkNode(node);
    xmlFreeNode(node);

    notifyModuleMappingChange();
  } catch (ConfigurationException &e) {
    // This is OK, we don't have to remove it
  }
}

// If its already there then steal the MUR from the owning ROD!
void ConfigurationXMLImpl::mapRODMUR(unsigned int partition, unsigned int crate, unsigned int rod, unsigned int order, unsigned int number) {
  log << "mapRODMUR " << partition << ", " << crate << ", " << rod << ", " << order << ", " << number << std::endl;

  if(order < 0 || order > 7) {
    throw ConfigurationException("Order must be (0-7) on input to mapRODMUR");
  }

  char expression[1000];

  // Check if MUR already in map
  sprintf(expression, "/configuration/partition/crate/rod/MUR[@id=%d]", number);

  xmlNodePtr node = 0;
  try {
    node = getQueryNode(expression, "Check MUR for remap");
    xmlUnlinkNode(node);
  } catch (ConfigurationException &e) {
    node = xmlNewNode(NULL, (xmlChar *)"MUR");

    setIntAttr(node, "id", number);
    setIntAttr(node, "order", order);
  }

  // We now have a detached node to put somewhere else that isn't a duplicate of an existing MUR id

  // Find parent ROD node
  sprintf(expression, "/configuration/partition[@id=%d]/crate[@id=%d]/rod[@id=%d]", partition, crate, rod);

  xmlNodePtr rodNode = 0;
  try {
    rodNode = getQueryNode(expression, "Get ROD for MUR map");
  } catch (ConfigurationException &e) {
    createROD(partition, crate, rod);
    rodNode = getQueryNode(expression, "Get ROD for MUR map (2)");
  }

  // Check there's not already an mur with this "order"
  for(xmlNodePtr murNode = rodNode->children;
      murNode;
      ) {

    if((murNode->type == XML_ELEMENT_NODE) 
       && (strcmp((const char *)murNode->name, "MUR") == 0) 
       && (getAttrAsInt(murNode, "order") == order)) {
      // Delete this MUR!
#warning "What if there's useful stuff in it!"

      xmlNodePtr nextNode = murNode->next;
      xmlUnlinkNode(murNode);
      xmlFreeNode(murNode);
      murNode = nextNode;
    } else {
      murNode = murNode->next;
    }
  }

  xmlAddChild(rodNode, node);
  setIntAttr(node, "order", order);

//   if(rodNode) {
//     // MUR insert
//     xmlNodePtr newMUR = xmlNewChild(rodNode, NULL, (xmlChar *)"MUR", NULL);

//     char buffer[5];
//     snprintf(buffer, 5, "%d", number);
//     xmlSetProp(newMUR, (xmlChar *)"id", (xmlChar *)buffer);
//     snprintf(buffer, 5, "%d", order);
//     xmlSetProp(newMUR, (xmlChar *)"order", (xmlChar *)buffer);
//   }

  // May call create ROD
  notifySystemStructureChange();
}

#warning "This has lots of other information in it!"
void ConfigurationXMLImpl::unmapRODMUR(unsigned int MUR) {
  log << "unmapRODMUR " << MUR << std::endl;

  char expression[1000];

  // Get MUR
  sprintf(expression,
          "/configuration/partition/crate/rod/MUR[@id=%d]",
          MUR);

  try {
    xmlNodePtr node = getQueryNode(expression, "Get MUR in ROD for unmap");

    xmlUnlinkNode(node);
    xmlFreeNode(node);

    notifyModuleMappingChange();
  } catch (ConfigurationException &e) {
    // This is OK, we don't have to remove it
  }
}

void ConfigurationXMLImpl::getMapMURROD(unsigned int MUR, unsigned int &partition, unsigned int &crate, unsigned int &rod, unsigned int &order) {
  log << "Get mapping from MUR to Rod " << MUR << std::endl;

  char expression[1000];
  sprintf(expression, "/configuration/partition/crate/rod/MUR[@id=%d]", MUR);
  xmlNodePtr murNode = getQueryNode(expression, "Map MUR to ROD");

  // MUR order
  order = getAttrAsInt(murNode, "order");
  rod = getAttrAsInt(murNode->parent, "id");
  crate = getAttrAsInt(murNode->parent->parent, "id");
  partition = getAttrAsInt(murNode->parent->parent->parent, "id");
}

void ConfigurationXMLImpl::getMapRODMUR(unsigned int partition, unsigned int crate, unsigned int rod, unsigned int order, unsigned int &MUR) {
  log << "Get mapping from ROD to MUR " << partition << ", " << crate << ", " << rod << ", " << order << std::endl;

  if(order < 0 || order > 7) {
    throw ConfigurationException("Order of MUR not in range (0 to 7) on input to getMapRODMUR");
  }

  char expression[1000];
  sprintf(expression, 
          "/configuration/partition[@id=%d]/crate[@id=%d]/rod[@id=%d]/MUR[@order=%d]", 
          partition, crate, rod, order);
  xmlNodePtr moduleNode = getQueryNode(expression, "Map ROD to MUR");

  // MUR id
  MUR = getAttrAsInt(moduleNode, "id");
}

void ConfigurationXMLImpl::getMapMURPhysical(unsigned int MUR, MURType &type, int &plane, int &section, int &position) {
  log << "Get mapping from MUR " << MUR << " to physical\n";

  type = getMURType(MUR);

  char expression[1000];

  switch(type) {
  case BARREL:
    {
      sprintf(expression, "/configuration/geography/barrel/row/MUR[@id=%d]", MUR);
      xmlNodePtr moduleNode = getQueryNode(expression, "Get translation to Barrel");

      position = getAttrAsInt(moduleNode, "position");
      section = getAttrAsInt(moduleNode->parent, "id");
      plane = getAttrAsInt(moduleNode->parent->parent, "id");
      break;
    }
  case ENDCAP:
    {
      sprintf(expression, "/configuration/geography/endcap/quadrant/MUR[@id=%d]", MUR);
      xmlNodePtr moduleNode = getQueryNode(expression, "Get translation to Endcap");

      try {
        position = getAttrAsInt(moduleNode, "position");
      } catch(ConfigurationException &c) {
        position = -1;
      }
      section = getAttrAsInt(moduleNode->parent, "id");
      plane = getAttrAsInt(moduleNode->parent->parent, "id");
      break;
    }
  default:
    throw ConfigurationException("No mapping to physical structure");
  }
}

void ConfigurationXMLImpl::printModuleConfig(const ABCDModule &conf) {
  log << "Print module configuration\n";

  std::cout << "Module global settings\n";
  std::cout << " Present " << (int)conf.present << " Active " << (int)conf.active << " Select " << (int)conf.select << " Group " << (int)conf.groupId << std::endl;
  std::cout << " TX fibre " << (int)conf.pTTC << " rx fibre0 " << (int)conf.rx[0] << " rx fibre1 " << (int)conf.rx[1] << std::endl;

//   std::cout << " BOC configuration " << conf.bocConfig << std::endl;

  for(int c=0; c<N_SCT_CHIPS; c++) {
    std::cout << "Chip: " << c << (conf.chip[c].active?" active ":" inactive ") ; 
    if(conf.chip[c].active) {
      std::cout << std::hex << "0x" << (int)conf.chip[c].address << std::dec << std::endl;
      std::cout << "Config 0x" << std::hex << *(UINT16*)(&conf.chip[c].basic.config) << std::dec << std::endl;

      std::cout << "R" << conf.chip[c].basic.config.readoutMode << " ";
      std::cout << "C" << conf.chip[c].basic.config.calibMode << " ";
      std::cout << "T" << conf.chip[c].basic.config.trimRange << " ";
      std::cout << "D" << conf.chip[c].basic.config.edgeDetect << " ";
      std::cout << "K" << conf.chip[c].basic.config.mask << " ";
      std::cout << "A" << conf.chip[c].basic.config.accumulate << " ";
      std::cout << "I" << conf.chip[c].basic.config.inputBypass << " ";
      std::cout << "O" << conf.chip[c].basic.config.outputBypass << " ";
      std::cout << "M" << conf.chip[c].basic.config.master << " ";
      std::cout << "E" << conf.chip[c].basic.config.end << " ";
      std::cout << "F" << conf.chip[c].basic.config.feedThrough << " ";

      std::cout << std::endl;

      std::cout << "Mask " << std::hex
           << " 0x" << conf.chip[c].basic.mask[3]
           << " 0x" << conf.chip[c].basic.mask[2]
           << " 0x" << conf.chip[c].basic.mask[1]
           << " 0x" << conf.chip[c].basic.mask[0] << std::dec << std::endl;

      std::cout << "Rc function type " << (int)conf.chip[c].caldata.rc_function << " :";
      for(int r=0; r<3; r++) {
        std::cout << " " << conf.chip[c].caldata.rc_params[r];
      }
      std::cout << std::endl;
      std::cout << "Regs ";
      std::cout << " vthr " << (int)conf.chip[c].basic.vthr;
      std::cout << " vcal " << (int)conf.chip[c].basic.vcal;
      std::cout << " delay " << (int)conf.chip[c].basic.delay;
      std::cout << " preamp " << (int)conf.chip[c].basic.preamp;
      std::cout << " shaper " << (int)conf.chip[c].basic.shaper;
      std::cout << std::endl;

      std::cout << "Trim target = " << (int)conf.chip[c].target;
      for(int i=0; i<128; i++) {
        if(!(i%32)) std::cout << std::endl;
        std::cout << std::hex << (int)conf.chip[c].trim[i] << std::dec;
      }
      std::cout << std::endl;
    } else {
      std::cout << std::endl;
    }
  }
  std::cout << std::endl;
}


/*
  Configure a module using the configuration

  XXX throws exception if module not configured previously ?
*/
void ConfigurationXMLImpl::configureModuleFromStructure(std::string module, ABCDModule conf) {
  log << "Configure for module " << module << std::endl;

  xmlNodePtr moduleNode;

  try {
    moduleNode = getModuleConfigurationNode(module, "from structure");
  } catch(ConfigurationException &c) {
//     std::cerr << "Recovered\n";
    xmlNodePtr modulesNode = 0;

    try {
      modulesNode = getQueryNode("/configuration/modules", "Create module (1)");
    } catch(ConfigurationException &c) {
      xmlNodePtr root = xmlDocGetRootElement(document);

      xmlNewChild(root, NULL, (xmlChar *)"modules", NULL);

      modulesNode = getQueryNode("/configuration/modules", "Create Module (2)");
    }

    moduleNode = xmlNewChild(modulesNode, NULL, (xmlChar *)"module", NULL);
  }

  moduleNode = replaceModuleConfig(moduleNode, conf, module);

  serialToConfigNode[module] = moduleNode;

  notifyModuleConfChange();
}

void ConfigurationXMLImpl::configureModuleFromFile(std::string filename) {
  log << "Configure module from file " << filename << std::endl;

  // Put <xi::include href="filename" /> into configuration/modules

  // Move to directory of file in order to find includes...
  std::string lastDir(lastFilename, 0, lastFilename.rfind('/'));
//   std::cout << "Directory = " << lastDir << std::endl;
  char *saveDir = getcwd(NULL, 0);
  chdir(lastDir.c_str());

  // Copy existing document so get proper settings
  xmlDocPtr moduleDocument = xmlCopyDoc(document, 0);
  xmlNodePtr moduleRoot = xmlNewNode(NULL, (const xmlChar*)"old");
  xmlDocSetRootElement(moduleDocument, moduleRoot);

  xmlNsPtr xiNameSpace = xmlSearchNsByHref(moduleDocument, moduleRoot, (const xmlChar*)"http://www.w3.org/2001/XInclude");

  if(!xiNameSpace) {
    xiNameSpace = xmlNewNs(moduleRoot, (xmlChar*)"http://www.w3.org/2001/XInclude", (xmlChar*)"xi");
  }

  xmlNodePtr includeNode = xmlNewChild(moduleRoot, xiNameSpace, (xmlChar *)"include", NULL);
  xmlSetProp(includeNode, (xmlChar *)"href", (xmlChar *)filename.c_str());

//   xmlSaveFormatFile("PreXIncludeModule.xml", moduleDocument, 0);
  xmlXIncludeProcess(moduleDocument);
//   xmlSaveFormatFile("PostXIncludeModule.xml", moduleDocument, 0);

  chdir(saveDir);
  free(saveDir);

  xmlNodePtr moduleNode = getChildNode(moduleRoot, "module");
  xmlNodePtr snNode = getChildNode(moduleNode, "sn");

  char *snString = (char *)xmlNodeGetContent(snNode);
  try {
    // Is there an old configuration?
    xmlNodePtr oldModuleNode = getModuleConfigurationNode(snString, "Check for old config");

    // Remove old module configuration
    xmlUnlinkNode(oldModuleNode);
    xmlFreeNode(oldModuleNode);
  } catch(ConfigurationException &c) {
    // Don't do anything
  }

  // Get modules list
  xmlNodePtr modulesNode;
  try {
    modulesNode = getQueryNode("/configuration/modules", "Modules for configure file");
  } catch(ConfigurationException &c) {
    xmlNodePtr root = xmlDocGetRootElement(document);

    xmlNewChild(root, NULL, (xmlChar *)"modules", NULL);

    modulesNode = getQueryNode("/configuration/modules", "Modules for configure file (2)");
  }

  // Attach new configuration to document
  xmlUnlinkNode(moduleNode);
  xmlAddChild(modulesNode, moduleNode);

  serialToConfigNode[snString] = moduleNode;

//   saveConfiguration("PreXInclude.xml");
// #if (LIBXML_VERSION < 20504)   // bugzilla 116095 
// #warning "There's a bug about calling xmlXIncludeProcess more than once!"
// #endif
//   int count = xmlXIncludeProcess(document);
//   std::cout << count << " XInclude substitutions in configureModuleFromFile\n";
//   saveConfiguration("PostXInclude.xml");
  notifyModuleConfChange();

  free(snString);
}

/*
  Replace the configuration in the given node with the data
  stored in the given configuration
*/
xmlNodePtr ConfigurationXMLImpl::replaceModuleConfig(xmlNodePtr node, ABCDModule conf, std::string name) {
  xmlNodePtr newNode = xmlNewNode(NULL, (xmlChar *)"module");

  const int BUF_SIZE = 200;

  char buffer[BUF_SIZE];

  xmlNewChild(newNode, NULL, (xmlChar *)"sn", (xmlChar *)name.c_str());
  xmlNewChild(newNode, NULL, (xmlChar *)"source", (xmlChar *)"Module configuration saved during running");
  xmlNewChild(newNode, NULL, (xmlChar *)"location", (xmlChar *)"RAL");
  snprintf(buffer, BUF_SIZE, "%d", conf.active);
  xmlNewChild(newNode, NULL, (xmlChar *)"active", (xmlChar *)buffer);
  snprintf(buffer, BUF_SIZE, "%d", conf.groupId);
  xmlNewChild(newNode, NULL, (xmlChar *)"group", (xmlChar *)buffer);
  snprintf(buffer, BUF_SIZE, "%d", conf.select);
  xmlNewChild(newNode, NULL, (xmlChar *)"select", (xmlChar *)buffer);

  for(int chipNum=0; chipNum<12; chipNum++) {
    ABCDChip &currChip = conf.chip[chipNum]; 

    xmlNodePtr chip = xmlNewChild(newNode, NULL, (xmlChar *)"chip", (xmlChar *)"\n");
    snprintf(buffer, BUF_SIZE, "%02d", chipNum);
    xmlSetProp(chip, (xmlChar *)"id", (xmlChar *)buffer);
    snprintf(buffer, BUF_SIZE, "0x%x", currChip.address);
    xmlSetProp(chip, (xmlChar *)"address", (xmlChar *)buffer);
    snprintf(buffer, BUF_SIZE, "%d", currChip.active);
    xmlSetProp(chip, (xmlChar *)"active", (xmlChar *)buffer);

    ABCDConfig &chipConf = currChip.basic.config;

    snprintf(buffer, BUF_SIZE, "%d %d %d %d %d %d %d %d %d %d %d",
             chipConf.readoutMode, chipConf.calibMode, chipConf.trimRange,
             chipConf.edgeDetect, chipConf.mask, chipConf.accumulate,
             chipConf.inputBypass, chipConf.outputBypass, chipConf.master,
             chipConf.end, chipConf.feedThrough);
    xmlNewChild(chip, NULL, (xmlChar *)"config", (xmlChar *)buffer);

    snprintf(buffer, BUF_SIZE, "0x%08x 0x%08x 0x%08x 0x%08x",
             currChip.basic.mask[3], currChip.basic.mask[2],
             currChip.basic.mask[1], currChip.basic.mask[0]);
    xmlNewChild(chip, NULL, (xmlChar *)"mask", (xmlChar *)buffer);

    snprintf(buffer, BUF_SIZE, "0x%02x", currChip.basic.vthr);
    xmlNodePtr reg = xmlNewChild(chip, NULL, (xmlChar *)"reg", (xmlChar *)buffer);
    xmlSetProp(reg, (xmlChar *)"name", (xmlChar *)"vthr");
    snprintf(buffer, BUF_SIZE, "0x%02x", currChip.basic.vcal);
    reg = xmlNewChild(chip, NULL, (xmlChar *)"reg", (xmlChar *)buffer);
    xmlSetProp(reg, (xmlChar *)"name", (xmlChar *)"vcal");
    snprintf(buffer, BUF_SIZE, "0x%02x", currChip.basic.delay);
    reg = xmlNewChild(chip, NULL, (xmlChar *)"reg", (xmlChar *)buffer);
    xmlSetProp(reg, (xmlChar *)"name", (xmlChar *)"delay");
    snprintf(buffer, BUF_SIZE, "0x%02x", currChip.basic.preamp);
    reg = xmlNewChild(chip, NULL, (xmlChar *)"reg", (xmlChar *)buffer);
    xmlSetProp(reg, (xmlChar *)"name", (xmlChar *)"preamp");
    snprintf(buffer, BUF_SIZE, "0x%02x", currChip.basic.shaper);
    reg = xmlNewChild(chip, NULL, (xmlChar *)"reg", (xmlChar *)buffer);
    xmlSetProp(reg, (xmlChar *)"name", (xmlChar *)"shaper");

    snprintf(buffer, BUF_SIZE, "p0 %e p1 %e p2 %e",
             currChip.caldata.rc_params[0], currChip.caldata.rc_params[1], currChip.caldata.rc_params[2]);
    reg = xmlNewChild(chip, NULL, (xmlChar *)"rc_function",
                      (xmlChar *)buffer);
    snprintf(buffer, BUF_SIZE, "%d", currChip.caldata.rc_function); 
    xmlSetProp(reg, (xmlChar *)"type", (xmlChar *)buffer);

    snprintf(buffer, BUF_SIZE, "%f", currChip.caldata.c_factor); 
    xmlNewChild(chip, NULL, (xmlChar *)"c_factor", (xmlChar *)buffer);

    snprintf(buffer, BUF_SIZE, "%f", currChip.target * 2.5); 
    xmlNewChild(chip, NULL, (xmlChar *)"target", (xmlChar *)buffer);

    for(int u=0; u<4; u++) {
      buffer[0] = '\0';
      for(int t=u*32; t<(u+1)*32; t+=8) {
        snprintf(buffer + strlen(buffer), BUF_SIZE, "0x%x%x%x%x%x%x%x%x ",
                 currChip.trim[t+0], currChip.trim[t+1], currChip.trim[t+2], currChip.trim[t+3],
                 currChip.trim[t+4], currChip.trim[t+5], currChip.trim[t+6], currChip.trim[t+7]);
      }
      reg = xmlNewChild(chip, NULL, (xmlChar *)"trim",
                        (xmlChar *)buffer);
      snprintf(buffer, BUF_SIZE, "0x%02x", u*32);
      xmlSetProp(reg, (xmlChar *)"start", (xmlChar *)buffer);
      snprintf(buffer, BUF_SIZE, "0x%02x", u*32+31);
      xmlSetProp(reg, (xmlChar *)"end", (xmlChar *)buffer);
    }
  }

  xmlReplaceNode(node, newNode);
  xmlFreeNode(node);

  return newNode;
}

ISInfoDictionary *ConfigurationXMLImpl::getISDict() {
  if(!is_dict) {
    if(!isLookupDone) {
      IPCPartition partition("SCT");
      if(partition.isValid()) 
        is_dict = new ISInfoDictionary(partition);
      isLookupDone = true;
    }
  }

  return is_dict;
}

/*
  Replace the in memory configuration for the given ROD
  with the configuration provided.
*/
void ConfigurationXMLImpl::updateRodConfig(unsigned int partition, unsigned int crate, unsigned int rod, RodConfig conf) {
  log << "Update configuration for ROD\n";

  char expression[1000];

  sprintf(expression, "/configuration/partition[@id=%d]/crate[@id=%d]/rod[@id=%d]", partition, crate, rod);

  xmlNodePtr oldConf = getQueryNode(expression, "ROD update"); // queryResult->nodesetval->nodeTab[0];

  if(oldConf) {
    xmlNodePtr firstNode = 0;

    char buffer[200];

    xmlNodePtr child = oldConf->children;
    while(child) {
      char *name = (char *)child->name;

      xmlNodePtr next = child->next;
      if((strcmp("baseAddress", name) == 0)) {
        xmlUnlinkNode(child);
        xmlFreeNode(child);
      } else if(strcmp("slot", name) == 0) {
        xmlUnlinkNode(child);
        xmlFreeNode(child);
      } else if(strcmp("mapSize", name) == 0) {
        xmlUnlinkNode(child);
        xmlFreeNode(child);
      } else if(strcmp("resetLevel", name) == 0) {
        xmlUnlinkNode(child);
        xmlFreeNode(child);
      } else if(strcmp("numSlaves", name) == 0) {
        xmlUnlinkNode(child);
        xmlFreeNode(child);
      } else if(strcmp("slave", name) == 0) {
        xmlUnlinkNode(child);
        xmlFreeNode(child);
      } else if(strcmp("MUR", name) == 0) {
        if(!firstNode) firstNode = child;
      }

      child = next;
    }

    xmlNodePtr newChild = 0;
    sprintf(buffer, "%d", (int)(conf.baseAddress >> 24));
    newChild = xmlNewChild(oldConf, NULL, (xmlChar *)"slot", (xmlChar *)buffer);

//     sprintf(buffer, "0x%lx", conf.mapSize);
//     newChild = xmlNewChild(oldConf, NULL, (xmlChar *)"mapSize", (xmlChar *)buffer);

    sprintf(buffer, "%ld", conf.resetLevel);
    newChild = xmlNewChild(oldConf, NULL, (xmlChar *)"resetLevel", (xmlChar *)buffer);

    for(int s=3; s>=0; s--) {
      xmlNodePtr slave = xmlNewChild(oldConf, NULL, (xmlChar *)"slave", (xmlChar *)"\n");
          
      sprintf(buffer, "%d", s);
      xmlSetProp(slave, (xmlChar *)"id", (xmlChar *)buffer);

      xmlNewChild(slave, NULL, (xmlChar *)"ipramFile", (xmlChar *)conf.slaves[s].ipramFile.c_str());
      xmlNewChild(slave, NULL, (xmlChar *)"idramFile", (xmlChar *)conf.slaves[s].idramFile.c_str());
      xmlNewChild(slave, NULL, (xmlChar *)"extFile", (xmlChar *)conf.slaves[s].extFile.c_str());

      xmlAddPrevSibling(firstNode, slave);
      firstNode = slave;
    }
  }

  notifySystemStructureChange();
}

/*
  Write configuration to disc.
  This creates a unique filename based on the date and time
  and saves the current configuration to it.
*/
void ConfigurationXMLImpl::saveConfiguration(std::string filename) {
  log << "Save configuration " << std::flush;

  int useCompress = 1;

  if(filename.size() == 0) {
    filename = Sct::SctNames::getPersistentDir() + "/";
    const int BUFF_SIZE = 50;
    char buffer[BUFF_SIZE];

    time_t secs;
    secs = time(NULL);

    struct tm broke;
    broke = *(gmtime(&secs));
    // 20021112143712
    if(useCompress) {
      strftime(buffer, BUFF_SIZE, "Conf_dump_%Y%m%d%H%M%S.xml.gz", &broke);
    } else {
      strftime(buffer, BUFF_SIZE, "Conf_dump_%Y%m%d%H%M%S.xml", &broke);
    }

    filename += buffer;
  } else {
    // Compress if filename ends with .gz
    if(filename.substr(filename.size()-3, 3).compare(".gz") == 0) {
      useCompress = 1;
    } else {
      useCompress = 0;
    }
  }

  log << " to " << filename << (useCompress?" compressed":" uncompressed") << std::endl;

  if(useCompress)
    xmlSetDocCompressMode(document, 5);
  else
    xmlSetDocCompressMode(document, 0);
  xmlSaveFormatFile(filename.c_str(), document, 0);
//   std::cout << "Saved the configuration as " << filename.c_str() << std::endl;
  log << "Saved the configuration as " << filename.c_str() << std::endl;
  log << " Using compression " << xmlGetCompressMode() << " " << xmlGetDocCompressMode(document) << std::endl;
}

void ConfigurationXMLImpl::saveModuleConfiguration(std::string sn, std::string filename) {
  log << "saveModuleConfiguration " << sn << " to " << filename << std::endl;

  xmlNodePtr moduleNode = getModuleConfigurationNode(sn, "Save config");
  xmlNodePtr moduleCopy = xmlCopyNode(moduleNode, 1);

  xmlDocPtr moduleDocument = xmlNewDoc((const xmlChar *)"1.0");

  xmlDocSetRootElement(moduleDocument, moduleCopy);
  xmlSaveFile(filename.c_str(), moduleDocument);

  xmlFreeDoc(moduleDocument);
}

std::string ConfigurationXMLImpl::getModuleConfigurationString(const std::string modsn, const ABCDModule config)
{
  log << "getModuleConfigurationString " << modsn << std::endl;

  xmlDocPtr moduleDocument = xmlNewDoc((const xmlChar *)"1.0");
  xmlNodePtr moduleRoot = xmlNewNode(NULL, (const xmlChar*)"old");
  xmlDocSetRootElement(moduleDocument, moduleRoot);

  // Needs something to replace it
  replaceModuleConfig(moduleRoot, config, modsn);

  xmlChar *buffer;
  int length;
  xmlDocDumpMemory(moduleDocument, &buffer, &length);

  xmlFreeDoc(moduleDocument);

  std::string result((const char *)buffer);

  xmlFree(buffer);

  return result;
}

const RodConfig ConfigurationXMLImpl::parseRodConfig(xmlNodePtr c) {
  RodConfig result;

  result.resetLevel = 0;
//   result.bocPresent = true;

  if(c) {
    for(xmlNodePtr child = c->children;
        child;
        child = child->next) {
      if(child->type == XML_ELEMENT_NODE) {
        char * content = (char *)xmlNodeGetContent(child);

        log << "Node " << child->name << std::endl;

        if(strcmp((char *)child->name, "baseAddress") == 0) {
          result.baseAddress = strtol(content, NULL, 0);
        } else if(strcmp((char *)child->name, "slot") == 0) {
          result.baseAddress = strtol(content, NULL, 0) << 24;
//         } else if(strcmp((char *)child->name, "mapSize") == 0) {
//           result.mapSize = strtol(content, NULL, 0);
        } else if(strcmp((char *)child->name, "resetLevel") == 0) {
          result.resetLevel = strtol(content, NULL, 0);
        } else if(strcmp((char *)child->name, "bocNotPresent") == 0) {
          log << "BOC not present obsolete\n";
//           result.bocPresent = false;
        } else if(strcmp((char *)child->name, "slave") == 0) {
          try {
            unsigned int slaveId = getAttrAsInt(child, "id");
            //                std::cout << "Found data for slave " << slaveId << std::endl;
            for(xmlNodePtr slaveChild = child->children;
                slaveChild;
                slaveChild = slaveChild->next) {
              char* content = (char *)xmlNodeGetContent(slaveChild);

              if(strcmp((char *)slaveChild->name, "ipramFile") == 0) {
                result.slaves[slaveId].ipramFile = content;
              } else if(strcmp((char *)slaveChild->name, "idramFile") == 0) {
                result.slaves[slaveId].idramFile = content;
              } else if(strcmp((char *)slaveChild->name, "extFile") == 0) {
                result.slaves[slaveId].extFile = content;
              }
              free(content);
            }
          } catch(ConfigurationException &c) {
            log << "This slave has no id attribute\n";
          }
        } else {
          log << "Unknown tag in rod conf " << child->name << " \n";
        }

        free(content);
      } else {
        // log << "Node of a different type\n";
      }
    } // End while child
  //      result.push_back(atoi((char *)xmlGetProp(curr, (xmlChar *)"id")));
  }
  return result;
}

const TimConfig ConfigurationXMLImpl::parseTimConfig(xmlNodePtr c) {
  TimConfig result;

  result.baseAddress = 0xd << 24;
  result.trigFrequency = 100;
  result.resetFrequency = 0.01;

  if(c) {
    for(xmlNodePtr child = c->children;
        child;
        child = child->next) {
      if(child->type == XML_ELEMENT_NODE) {
        char * content = (char *)xmlNodeGetContent(child);

        if(strcmp((char *)child->name, "baseAddress") == 0) {
          result.baseAddress = strtol(content, NULL, 0);
        } else if(strcmp((char *)child->name, "slot") == 0) {
          result.baseAddress = strtol(content, NULL, 0) << 24;
        } else if(strcmp((char *)child->name, "triggerFrequency") == 0) {
          result.trigFrequency = strtod(content, NULL);
        } else if(strcmp((char *)child->name, "resetFrequency") == 0) {
          result.resetFrequency = strtod(content, NULL);
//         } else if(strcmp((char *)child->name, "mapSize") == 0) {
//           result.mapSize = strtol(content, NULL, 0);
        } else {
          log << "Unknown tag in tim conf " << child->name << " \n";
        }

        free(content);
      } else {
        // log << "Node of a different type\n";
      }
    } // End while child
  }
  return result;
}

// Very dangerous method!!!!
void ConfigurationXMLImpl::clearAll() {
  log << "CLEAR ALL CONFIGURATION!!!!!!\n";

  xmlNodePtr conf = xmlNewNode(NULL, (const xmlChar*)"configuration");

  xmlNodePtr oldRoot = xmlDocGetRootElement(document);

  xmlReplaceNode(oldRoot, conf);

//    xmlNewChild(conf, NULL, (xmlChar *)"modules", NULL);

  xmlFreeNode(oldRoot);

  serialToConfigNodeBuilt = false;
  serialToMurNodeBuilt = false;

  notifySystemStructureChange();
}

// Is this necessary?
void ConfigurationXMLImpl::namePartition(unsigned int partition, std::string name) {
  log << "Name partition\n";

  char buffer[100];

  xmlNodePtr root;

  sprintf(buffer, "/configuration/partition[@id=%d]", partition);
  try {
    root = getQueryNode(buffer, "Name partition (1)");
  } catch(ConfigurationException &e) {
//     std::cerr << "Recovered\n";
    createPartition(partition);
    root = getQueryNode(buffer, "Name partition (2)");
  }

  xmlSetProp(root, (xmlChar *)"name", (xmlChar *)name.c_str());
}

void ConfigurationXMLImpl::createPartition(unsigned int partition) {
  log << "Create partition\n";

  char buffer[100];
 
  sprintf(buffer, "/configuration");
  xmlNodePtr root;
  try {
    root = getQueryNode(buffer, "Create partition");
  } catch(ConfigurationException &e) {
    xmlNodePtr conf = xmlNewNode(NULL, (const xmlChar*)"configuration");
    xmlNodePtr oldRoot = xmlDocGetRootElement(document);
    xmlReplaceNode(oldRoot, conf);

    root = getQueryNode(buffer, "Create partition (2)");
  }

  getOrNewChildWithIntAttr(root, "partition", "id", partition);
//      xmlSetProp(newNode, (xmlChar *)"name", (xmlChar *)name.c_str());
}

void ConfigurationXMLImpl::createCrate(unsigned int partition, unsigned int crate) {
  log << "Create crate\n";

  char buffer[1000];

  xmlNodePtr part;
  sprintf(buffer, "/configuration/partition[@id=%d]", partition);
  try {
    part = getQueryNode(buffer, "Create crate");
  } catch(ConfigurationException &e) {
//     std::cerr << "Recovered\n";
    createPartition(partition);
    part = getQueryNode(buffer, "Create crate (2)");
  }

  getOrNewChildWithIntAttr(part, "crate", "id", crate);
}

void ConfigurationXMLImpl::createROD(unsigned int partition, unsigned int crate, unsigned int rod) {
  log << "Create ROD\n";

  char buffer[1000];

  xmlNodePtr crateNode;
  sprintf(buffer, "/configuration/partition[@id=%d]/crate[@id=%d]", partition, crate);
  try {
    crateNode = getQueryNode(buffer, "Create ROD");
  } catch(ConfigurationException &e) {
//     std::cerr << "Recovered\n";
    createCrate(partition, crate);
    crateNode = getQueryNode(buffer, "Create ROD (2)");
  }

  getOrNewChildWithIntAttr(crateNode, "rod", "id", rod);
}

void ConfigurationXMLImpl::configureROD(unsigned int partition, unsigned int crate, unsigned int rod, RodConfig conf) {
  log << "Configure ROD " << partition << " " << crate << " " << rod << std::endl;

  char buffer[1000];

//   std::cout << "Configure a ROD!\n";

  sprintf(buffer, "/configuration/partition[@id=%d]/crate[@id=%d]/rod[@id=%d]", partition, crate, rod);
  xmlNodePtr node;
  try {
    node = getQueryNode(buffer, "Configure ROD");
  } catch(ConfigurationException &e) {
//     std::cerr << "Recovered\n";
    createROD(partition, crate, rod);
    node = getQueryNode(buffer, "Configure ROD (2)");
  }

  updateRodConfig(partition, crate, rod, conf);
}

void ConfigurationXMLImpl::configureTIM(unsigned int partition, unsigned int crate, TimConfig conf) {
  log << "Configure TIM " << partition << " " << crate << std::endl;

  char buffer[1000];

  sprintf(buffer, "/configuration/partition[@id=%d]/crate[@id=%d]/tim", partition, crate);
  xmlNodePtr node;
  try {
    node = getQueryNode(buffer, "Configure TIM");
  } catch(ConfigurationException &e) {
    // Create TIM node 
    xmlNodePtr config = xmlDocGetRootElement(document);
    xmlNodePtr partNode = getOrNewChildWithIntAttr(config, "partition", "id", partition);
    xmlNodePtr crateNode = getOrNewChildWithIntAttr(partNode, "crate", "id", crate);
    node = getOrNewChildNode(crateNode, "tim");
  }

  sprintf(buffer, "%d", (int)(conf.baseAddress >> 24));
  setOrCreateChildNodeWithContent(node, "slot", buffer);

  // Minimum 0.15, 0.12, 0.10
  sprintf(buffer, "%.2f", conf.trigFrequency);
  setOrCreateChildNodeWithContent(node, "triggerFrequency", buffer);

  // Minimums 0.015, 0.012, 0.010
  sprintf(buffer, "%.3f", conf.resetFrequency);
  setOrCreateChildNodeWithContent(node, "resetFrequency", buffer);

//   sprintf(buffer, "0x%lx", conf.mapSize);
//   setOrCreateChildNodeWithContent(node, "mapSize", buffer);
}

void ConfigurationXMLImpl::configureBOC(unsigned int partition, unsigned int crate, unsigned int rod, const BOCGlobalConfig &conf) {
  log << "Configure BOC " << partition << " " << crate << " " << rod << std::endl;

  char buffer[1000];

  sprintf(buffer, "/configuration/partition[@id=%d]/crate[@id=%d]/rod[@id=%d]", partition, crate, rod);
  xmlNodePtr node;
  try {
    node = getQueryNode(buffer, "Configure BOC");
  } catch(ConfigurationException &e) {
    throw ConfigurationException("Trying to configure BOC on non-existant ROD");
  }

  xmlNodePtr bocNode = getOrNewChildNode(node, "BOC");

  for(int r=0; r<9; r++) {
    if(conf.validMask & 1<<r) {
      std::string name;
      int value;

      switch(r) {
      case 0:
        name = "clockControl";
        value = conf.clockControl;
        break;
      case 1:
        name = "rxDataMode";
        value = conf.rxDataMode;
        break;
      case 2:
        name = "rxDacClear";
        value = conf.rxDacClear;
        break;
      case 3:
        name = "txDacClear";
        value = conf.txDacClear;
        break;
      case 4:
        name = "vernierFinePhase";
        value = conf.vernierFinePhase;
        break;
      case 5:
        name = "vernierClockPhase0";
        value = conf.vernierClockPhase0;
        break;
      case 6:
        name = "vernierClockPhase1";
        value = conf.vernierClockPhase1;
        break;
      case 7:
        name = "bpmClockPhase";
        value = conf.bpmClockPhase;
        break;
      case 8:
        name = "bregClockPhase";
        value = conf.bregClockPhase;
        break;
      default:
        // Ignore invalid number
        continue;
      }

      xmlNodePtr regNode = xmlNewChild(bocNode, NULL, (xmlChar *)"reg", NULL);

      xmlSetProp(regNode, (xmlChar *)"name", (xmlChar *)name.c_str());
      char buffer[11];
      sprintf(buffer, "0x%x", value);
      xmlSetProp(regNode, (xmlChar *)"value", (xmlChar *)buffer);
    }
  }
}

void ConfigurationXMLImpl::mapModuleMUR(unsigned int MUR, unsigned int order, unsigned int RMUR, unsigned int rorder, std::string number) {
  log << "Map module onto MUR\n";

  if(order < 1 || order > 6) {
    throw ConfigurationException("MUR position out of range (1-6) on input to mapModuleMUR");
  }

  if(rorder < 1 || rorder > 6) {
    throw ConfigurationException("Redundant MUR position out of range (1-6) on input to mapModuleMUR");
  }

  char buffer[1000];

  try {
    unsigned int testMUR, testOrder;
    translateFromSN(number, testMUR, testOrder);

    if(testMUR != MUR) {
      sprintf(buffer, "/configuration/partition/crate/rod/MUR[@id=%d]/module[text()=\"%s\"]", testMUR, number.c_str());
      xmlNodePtr oldMurNode = getQueryNode(buffer, "Clear module from MUR");
      xmlUnlinkNode(oldMurNode);
      xmlFreeNode(oldMurNode);
    }
  } catch(ConfigurationException &c) {
    // Serial number not already mapped
  }

  try {
    sprintf(buffer, "/configuration/partition/crate/rod/MUR/rmodule[text()=\"%s\"]", number.c_str());
    xmlNodePtr oldRMurNode = getQueryNode(buffer, "Clear rmodule from MUR");
    if(getAttrAsInt(oldRMurNode->parent, "id", 10) != RMUR) {
      xmlUnlinkNode(oldRMurNode);
      xmlFreeNode(oldRMurNode);
    }
  } catch(ConfigurationException &c) {
    // Serial number not mapped as redundant
    // Ignore problems finding MUR id
  }

  try {
    sprintf(buffer, "/configuration/partition/crate/rod/MUR[@id=%d]", MUR);
    xmlNodePtr murNode = getQueryNode(buffer, "Assign module to MUR");

    xmlNodePtr newNode = getOrNewChildWithIntAttr(murNode, "module", "id", order);
    xmlNodeSetContent(newNode, (xmlChar *)number.c_str());

    serialToMurNode[number] = murNode;
  } catch(ConfigurationException &e) {
    log << "No MUR " << MUR << " for module " << number << std::endl;
    throw;
  }

  try {
    sprintf(buffer, "/configuration/partition/crate/rod/MUR[@id=%d]", RMUR);
    xmlNodePtr node = getQueryNode(buffer, "Assign module to RMUR");

    xmlNodePtr newNode = getOrNewChildWithIntAttr(node, "rmodule", "id", rorder);
    xmlNodeSetContent(newNode, (xmlChar *)number.c_str());
  } catch(ConfigurationException &e) {
    log << "No MUR " << RMUR << " for rmodule " << number << std::endl;
  }

  notifyModuleMappingChange();
}

void ConfigurationXMLImpl::unmapModuleMUR(unsigned int MUR, unsigned int order) {
  log << "unmapModuleMUR " << MUR << ", " << order << std::endl;

  if(order < 1 || order > 6) {
    throw ConfigurationException("Order within MUR not in range (1 to 6) on input to unmapModuleMUR");
  }

  char expression[1000];

  // Get MUR
  sprintf(expression,
          "/configuration/partition/crate/rod/MUR[@id=%d]/module[@id=%d]",
          MUR, order);

  try {
    xmlNodePtr node = getQueryNode(expression, "Get module in MUR for unmap");

    xmlUnlinkNode(node);
    xmlFreeNode(node);

    notifyModuleMappingChange();
  } catch (ConfigurationException &e) {
    // This is OK, we don't have to remove it
  }
 
}

void ConfigurationXMLImpl::configureBOCChannel(unsigned int MUR, unsigned int order, const BOCChannelConfig &bConf) {
  log << "Create BOC configuration\n";

  if(order < 1 || order > 6) {
    throw ConfigurationException("MUR position out of range (1-6) on input to configureBOCChannel");
  }

  char buffer[1000];

  sprintf(buffer, "/configuration/partition/crate/rod/MUR[@id=%d]", MUR);
  xmlNodePtr node = getQueryNode(buffer, "Set BOC Channel Configs");

  xmlNodePtr ourChannelNode = 0;

  for(xmlNodePtr channelNode = node->children;
        channelNode;
        channelNode = channelNode->next) {

    if(channelNode->type != XML_ELEMENT_NODE) {
    } else {
      if(strcmp((char*)channelNode->name, "channel") == 0 &&  getAttrAsInt(channelNode, "id") == order) {
        ourChannelNode = channelNode;
        break;
      }
    }
  }

  if(!ourChannelNode) {
    ourChannelNode = xmlNewChild(node, NULL, (xmlChar *)"channel", NULL);
  }

  sprintf(buffer, "%d", order);
  xmlSetProp(ourChannelNode, (xmlChar *)"id", (xmlChar *)buffer);

  xmlNodePtr newNode = getOrNewChildNode(ourChannelNode, "output");
  sprintf(buffer, "0x%x", bConf.current);
  xmlSetProp(newNode, (xmlChar *)"current", (xmlChar *)buffer);
  sprintf(buffer, "0x%x", bConf.delay);
  xmlSetProp(newNode, (xmlChar *)"delay", (xmlChar *)buffer);
  sprintf(buffer, "0x%x", bConf.markSpace);
  xmlSetProp(newNode, (xmlChar *)"markSpace", (xmlChar *)buffer);

  newNode = getOrNewChildNode(ourChannelNode, "stream0");
  sprintf(buffer, "0x%x", bConf.threshold0);
  xmlSetProp(newNode, (xmlChar *)"threshold", (xmlChar *)buffer);
  sprintf(buffer, "0x%x", bConf.delay0);
  xmlSetProp(newNode, (xmlChar *)"delay", (xmlChar *)buffer);

  newNode = getOrNewChildNode(ourChannelNode, "stream1");
  sprintf(buffer, "0x%x", bConf.threshold1);
  xmlSetProp(newNode, (xmlChar *)"threshold", (xmlChar *)buffer);
  sprintf(buffer, "0x%x", bConf.delay1);
  xmlSetProp(newNode, (xmlChar *)"delay", (xmlChar *)buffer);

  notifyModuleConfChange();
}

xmlNodePtr ConfigurationXMLImpl::getModuleConfigurationNode(std::string sn, std::string reason) {
  if(!serialToConfigNodeBuilt) 
    buildSerialToConfigNode();

  xmlNodePtr moduleNode = serialToConfigNode[sn];

  if(!moduleNode)
    throw ConfigurationException("No module config for " + sn);

  char *cSn = (char*)xmlNodeGetContent(getChildNode(moduleNode, "sn"));
  if(sn != std::string(cSn)) {
    free(cSn);
    throw ConfigurationException("Mismatched configuration nodes (internal error)");
  }
  free(cSn);

  return moduleNode;
}

void ConfigurationXMLImpl::buildSerialToConfigNode() {
  char expression[1000];
  sprintf(expression, "/configuration/modules/module/sn");

  std::list<xmlNodePtr> moduleNodes = getQueryNodeList(expression, "Build config translation from SN");

  serialToConfigNode.clear();

  for(std::list<xmlNodePtr>::const_iterator iter = moduleNodes.begin(); 
      iter != moduleNodes.end();
      iter ++) {
    xmlNodePtr current = *iter;

    char *cSerial = (char *)xmlNodeGetContent(current);
    std::string serial(cSerial);

    serialToConfigNode[serial] = current->parent;

    free(cSerial);
  }

  serialToConfigNodeBuilt = true;
}


void ConfigurationXMLImpl::writePowerSupplyConfiguration(std::string directory) {
  log << "writePowerSupplyConfiguration called\n";
  // Call external program?
}

void ConfigurationXMLImpl::notifyModuleConfChange() {
  static int changeCount = 0;

  ISInfoDictionary *dict = getISDict();
  if(dict) {
    ISInfoInt changeValue(changeCount ++);

    const char *isName = "ConfigurationServer.ModuleConfChangeCount";

    if(dict->contains(isName)) {
      dict->update(isName, changeValue);
    } else {
      dict->insert(isName, changeValue);
    }
  }
}

void ConfigurationXMLImpl::notifyModuleMappingChange() {
  static int changeCount = 0;

  ISInfoDictionary *dict = getISDict();
  if(dict) {
    ISInfoInt changeValue(changeCount ++);

    const char *isName = "ConfigurationServer.ModuleMappingChangeCount";

    if(dict->contains(isName)) {
      dict->update(isName, changeValue);
    } else {
      dict->insert(isName, changeValue);
    }
  }
}

void ConfigurationXMLImpl::notifySystemStructureChange() {
  static int changeCount = 0;

  ISInfoDictionary *dict = getISDict();
  if(dict) {
    ISInfoInt changeValue(changeCount ++);

    const char *isName = "ConfigurationServer.SystemStructureChangeCount";

    if(dict->contains(isName)) {
      dict->update(isName, changeValue);
    } else {
      dict->insert(isName, changeValue);
    }
  }
}

// ********   XML Stuff ************


xmlNodePtr ConfigurationXMLImpl::getQueryNode(std::string query, std::string task) {
  xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)(query.c_str()), xpathContext);

  xmlNodePtr nodeResult = 0;

  try {
    if(queryResult) {
      if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
        if(queryResult->nodesetval->nodeNr > 1) {
          log << "Too many matches (" << queryResult->nodesetval->nodeNr << ") doing " << task << ": " << query << std::endl;
          throw ConfigurationException("Query: too many matches (" + task + "): " + query);
        } else if(queryResult->nodesetval->nodeNr == 0) {
          log << "Too few matches doing " << task << ": " << query << std::endl;
          throw ConfigurationException("Query: too few matches (" + task + "): " + query);
        } else {
          nodeResult = queryResult->nodesetval->nodeTab[0];
        }
      } else {
        log << "Bad result type (" << task << "): " << query << std::endl;
        throw ConfigurationException("Query: Bad result type (" + task + "): " + query);
      }

//       xmlXPathFreeNodeSetList(queryResult);
    } else {
      log << "No result (" << task << "): " << query << std::endl;
      throw ConfigurationException("Query (" + task + "): No result");
    }
  } catch(ConfigurationException &c) {
    if(queryResult)
      xmlXPathFreeObject(queryResult);
    throw;
  } catch(...) {
    log << "Unexpected exception in getQueryNode doing: " << task << std::endl;
    if(queryResult)
      xmlXPathFreeObject(queryResult);
    throw;
  }

  xmlXPathFreeObject(queryResult);

  return nodeResult;
}

std::list<xmlNodePtr> ConfigurationXMLImpl::getQueryNodeList(std::string query, std::string task) {
  xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)(query.c_str()), xpathContext);

  std::list<xmlNodePtr> nodesResult;

  try {
    if(queryResult) {
      if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
        for(int i=0; i<queryResult->nodesetval->nodeNr; i++) {
          nodesResult.push_back(queryResult->nodesetval->nodeTab[i]);
        }
      } else {
        log << "Bad result type (" << task << "): " << query << std::endl;
        throw ConfigurationException("Query List: Bad result type (" + task + "): " + query);
      }

//       xmlXPathFreeNodeSetList(queryResult);
    } else {
      log << "No result (" << task << "): " << query << std::endl;
      throw ConfigurationException("Query (" + task + "): No result");
    }
  } catch(ConfigurationException &c) {
    if(queryResult)
      xmlXPathFreeObject(queryResult);
    throw;
  } catch(...) {
    log << "Unexpected exception in getQueryNodeList doing: " << task << std::endl;
    if(queryResult)
      xmlXPathFreeObject(queryResult);
    throw;
  }

  xmlXPathFreeObject(queryResult);

  return nodesResult;
}

xmlNodePtr ConfigurationXMLImpl::getChildNode(xmlNodePtr node, const char *name) {
  xmlNodePtr curr = node->children;

  while(curr != NULL) {
    if(curr->type == XML_ELEMENT_NODE) {
      if(strcmp((const char *)curr->name, name) == 0) {
        return curr;
      }
    }

    curr = curr->next;
  }

  log << "No child " << name << " of parent " << node->name << std::endl;
  throw ConfigurationException(std::string("Child node not found: ") + name);
}

xmlNodePtr ConfigurationXMLImpl::getOrNewChildNode(xmlNodePtr node, const char *name) {
  xmlNodePtr curr = node->children;

  while(curr != NULL) {
    if(curr->type == XML_ELEMENT_NODE) {
      if(strcmp((const char *)curr->name, name) == 0) {
        return curr;
      }
    }

    curr = curr->next;
  }

  xmlNodePtr newNode = xmlNewChild(node, NULL, (xmlChar *)name, NULL);

  return newNode;
}

void ConfigurationXMLImpl::setOrCreateChildNodeWithContent(xmlNodePtr node, const char *name, const char *content) {
  xmlNodePtr curr = node->children;

  while(curr != NULL) {
    if(curr->type == XML_ELEMENT_NODE) {
      if(strcmp((const char *)curr->name, name) == 0) {
        // Set content 
        xmlNodeSetContent(curr, (xmlChar *)content);
        return;
      }
    }

    curr = curr->next;
  }

  xmlNewChild(node, NULL, (xmlChar *)name, (xmlChar *)content);
}

xmlNodePtr ConfigurationXMLImpl::getChildNodeWithContent(xmlNodePtr node, const char *name, const char *content) {
  xmlNodePtr curr = node->children;

  while(curr != NULL) {
    if(curr->type == XML_ELEMENT_NODE) {
      if(strcmp((const char *)curr->name, name) == 0) {
        char *testContent = (char *)xmlNodeGetContent(curr);
        if(strcmp(testContent, content) == 0) {
          free(testContent);
          return curr;
        }
        free(testContent);
      }
    }

    curr = curr->next;
  }

  return 0;
}

xmlNodePtr ConfigurationXMLImpl::getOrNewChildWithIntAttr(xmlNodePtr parent, const char *name, const char *attr, int integer) {
  xmlNodePtr result = getNodeWithIntAttr(parent, name, attr, integer);

  if(!result) {
    result = xmlNewChild(parent, NULL, (xmlChar*)name, NULL);

    setIntAttr(result, attr, integer);
  }

  return result;
}

xmlNodePtr ConfigurationXMLImpl::getOrNewChildWithStringAttr(xmlNodePtr parent, const char *name, const char *attr, std::string value) {
  xmlNodePtr result = getNodeWithStringAttr(parent, name, attr, value);

  if(!result) {
    result = xmlNewChild(parent, NULL, (xmlChar*)name, NULL);

    xmlSetProp(result, (xmlChar*)attr, (xmlChar*)value.c_str());
  }

  return result;
}

void ConfigurationXMLImpl::setFloatAttr(xmlNodePtr node, const char *attr, float floater) {
  char buffer[25];
  snprintf(buffer, 25, "%f", floater);

  xmlSetProp(node, (xmlChar*)attr, (xmlChar*)buffer);
}

void ConfigurationXMLImpl::setIntAttr(xmlNodePtr node, const char *attr, int integer) {
  char buffer[10];
  snprintf(buffer, 10, "%d", integer);

  xmlSetProp(node, (xmlChar*)attr, (xmlChar*)buffer);
}

void ConfigurationXMLImpl::clearAttr(xmlNodePtr node, const char *attr) {
//   char buffer[10];
//   snprintf(buffer, 10, "%d", integer);

  xmlUnsetProp(node, (xmlChar*)attr);
}

float ConfigurationXMLImpl::getAttrAsFloat(xmlNodePtr node, const char *attr) {
  const char *str;

  str = (const char *)xmlGetProp(node, (xmlChar *)attr);

  if(str) {
    float result = (float)strtod(str, NULL);
    xmlFree((void *)str);
    return result;
  } else {
    log << "Can't find attribute " << std::string(attr) << " in node called " << std::string((const char *)node->name) << std::endl;
    throw ConfigurationException("Can't find attribute " + std::string(attr));
  }
}

int ConfigurationXMLImpl::getAttrAsInt(xmlNodePtr node, const char *attr, int base) {
  const char *str;

  str = (const char *)xmlGetProp(node, (xmlChar *)attr);

  if(str) {
    int result = strtol(str, NULL, base);
    xmlFree((void *)str);
    return result;
  } else {
    log << "Can't find attribute " << std::string(attr) << " in node called " << std::string((const char *)node->name) << std::endl;
    throw ConfigurationException("Can't find attribute " + std::string(attr));
  }
}

std::string ConfigurationXMLImpl::getAttrAsString(xmlNodePtr node, const char *attr) {
  const char *str;

  str = (const char *)xmlGetProp(node, (xmlChar *)attr);

  if(str) {
    std::string result = str;
    xmlFree((void *)str);
    return result;
  } else {
    log << "Can't find attribute " << std::string(attr) << std::endl;
    throw ConfigurationException("Can't find attribute" + std::string(attr));
  }
}

xmlNodePtr ConfigurationXMLImpl::getNodeWithIntAttr(xmlNodePtr node, const char *name, const char *attr, int integer) {
  for(xmlNodePtr iter = node->children;
        iter;
        iter = iter->next) {

    if(iter->type != XML_ELEMENT_NODE) {
    } else {
      if(strcmp((char*)iter->name, name) == 0 && getAttrAsInt(iter, attr) == integer) {
        return iter;
      }
    }
  }

  return 0;
}

xmlNodePtr ConfigurationXMLImpl::getNodeWithStringAttr(xmlNodePtr node, const char *name, const char *attr, std::string value) {
  for(xmlNodePtr iter = node->children;
        iter;
        iter = iter->next) {

    if(iter->type != XML_ELEMENT_NODE) {
    } else {
      if(strcmp((char*)iter->name, name) == 0 && getAttrAsString(iter, attr) == value) {
        return iter;
      }
    }
  }

  return 0;
}

//  ************* End XML functions **************
