#include "SummaryWriter.h"

#include "Sct/SctParameters.h"
#include "Sct/SctNames.h"

#include "SctData/DefectList.h"
#include "SctData/TestResult.h"
#include "SctData/ConfigurationVariable.h"
#include "SctData/DcsData.h"
#include "SctData/Stat.h"
#include "Sct/Version.h"
#include <boost/date_time/posix_time/posix_time.hpp>
#include <unistd.h>

using namespace std;

namespace SctData {
    namespace TestSummary{

      const unsigned geneva_upload_points=16;

	SummaryWriter::SummaryWriter() {}

	SummaryWriter::~SummaryWriter() {}
        
        string getHostName() {
            char name[255];
	    if (gethostname(name, 255)) return "";
            return string(name);
        }

	void SummaryWriter::writeHeader(const TestResult& r, ostream& out) const throw() {
	    boost::posix_time::ptime t(boost::posix_time::second_clock::local_time());
	    out << "#\n%NewTest\n#\n" ;
	    out << "SERIAL NUMBER    : " << r.getModuleName() <<endl;
	    out << "TEST MADE BY     : " << SctNames::getUser() << endl;
	    out << "LOCATION MADE    : " << SctNames::getLocation() << endl;
	    out << "Run number       : " << r.getRunNumber() << "-" << r.getScanNumberAt(0) << endl;
	    out << "TEST_DATE        : " << (short)t.date().day() << "/" << (short)t.date().month() << "/" << (short)t.date().year() << endl;
	    out << "PASSED           : " ;
	    if (r.getPassed()) out << "YES";
	    else out << "NO";
	    out << endl; 
	    out << "PROBLEM          : ";
	    if (r.getProblem()) out << "YES";
	    else out << "NO";
	    out << endl;
	    out << "#\n%DAQ_INFO\n#HOST\n\"" << getHostName() << "\"" << endl;
	    out << "#VERSION \n\"" << Sct::VersionString << "\"" << endl;
	    out << "#DUT\n\"module\"" << endl;
	    out << "#TIME\n\"" << boost::posix_time::to_simple_string(t.time_of_day()) << "\"\n#" << endl;
	    
	    vector<string> comments = r.getComments();
	    out << endl;
	    for (unsigned int i=0; i<comments.size(); ++i) {
		out << "%Comment" << endl << "COMMENT          : " << comments[i] << endl;
	    }
	    
	    boost::shared_ptr<const DcsData> dcs = r.getDcsData();
            if (dcs.get()) {
	      out << "#\n%DCS_INFO\n#\n#T0     T1" << std::endl
		  << dcs->getT0() << "\t" << dcs->getT1() << std::endl;

	      out << "#VDET   IDET" << std::endl
		  << dcs->getVbias() << "\t" << dcs->getIbias() << std::endl;

	      out << "#VCC    ICC" << std::endl
		  << dcs->getVcc() << "\t" << dcs->getIcc() << std::endl;

	      out <<"#VDD    IDD"  << std::endl
		  << dcs->getVdd() << "\t" << dcs->getIdd() << std::endl;

	      out << "#TIME_POWERED\n"
		  << "." 
		  << "\n#\n";
	    }
 
	    if (r.getNScans()>1){
		out << "%SCAN_INFO\n#"<<endl;
		out << "#POINT_TYPE\n\"" << r.getTestVariable().getVariableName() << "\"" << endl;
		out << "#N_POINTS\n" << r.getNScans() << endl;
		out << "#POINTS";
		for (unsigned ipt=0; ipt<geneva_upload_points; ++ipt){
		    if (ipt%8==0) out << "\n";
		    if (ipt<r.getNScans()){
		      out << r.getTestPointAt(ipt) << "\t";
		    } else {
		      out << "." << "\t";
		    }
		}
	    out << "\n#" << endl;
	    }
	}
	
	void SummaryWriter::write(const DefectList& list, std::ostream& out) const throw(Sct::IoError, Sct::LogicError){
	    const std::list<Defect>& defects = list.getAllDefects(); 

	    if ( defects.empty() ){
		out << "#No defects found\n#" << endl; return;
	    }
	    
	    unsigned largestDefect=0;
	    
	    Stats<int> channels(nChannelModule, Stat<int>(0, true) );
	    
	    for (std::list<Defect>::const_iterator i =defects.begin(); i!=defects.end(); ++i){
	      const ModuleElement& e = (*i).getModuleElement();
	      
	      for (unsigned ichannel = e.getFirst(); ichannel <= e.getLast(); ++ichannel ){
		channels.modifyAt(ichannel).valid=false;
	      }
	      
	      if (e.getNChannels() > largestDefect) largestDefect=e.getNChannels() ;
	      
	      out << "\n%Defect" << endl;
	      out << "DEFECT NAME           : " << (*i).getPrototype().getName()<<endl;
	      out << "FIRST CHANNEL         : " << (*i).getModuleElement().getFirst()<<endl;
	      out << "LAST CHANNEL          : " << (*i).getModuleElement().getLast()<<endl;
	      out << "#"<<endl;
	    }
	    out << "#" << defects.size() << " defects found" 
		<< " affecting " << nChannelModule-channels.n() << " strips\n"
		<<  "#" << largestDefect << " maximum consecutive defects\n#"<< endl;
	}
      
	string SummaryWriter::getChipName(const unsigned short ichip) throw() {
	    short unsigned id=ichip;
	    if (id>=6) id+=2;
	    switch(id){
	    case (0) : return string("M0"); break;
	    case (5) : return string("E5"); break;
	    case (8) : return string("M8"); break;
	    case (13) : return string("E13"); break;
	    default : {
		char name[5];
		sprintf(name,"S%d",id);
		return string(name);
		break; 
	    }
	    }
	}
    }
}
