#include "DisplayManager.h"
#include "DisplayInfo.h"
#include "Displayer.h"
#include "Sct/SctParameters.h"
#include "Sct/Serializable.h"
#include "Sct/IoExceptions.h"
#include "Sct/IS/IOManagerIS.h"
#include "Sct/File/IOManagerFile.h"
#include "Sct/Archive/IOManagerArchiveFile.h"
#include <iostream>
#include <vector>
#include <string>
#include <TApplication.h>

using namespace std;
using namespace Sct;
using namespace Sct::IS;
using namespace Sct::File;
using namespace Sct::Archive;
using namespace SctDataDisplay;

//Holds the list of things to display
vector<string> isNames;

/**
  Creates the default DisplayInfo
  */
DisplayInfo getDefaultInfo() {
    DisplayInfo info;
    for (unsigned int i=0; i<nChipModule; ++i) {
	info.displayChips.push_back(i);
    }
    
    return info;
}


void printHelp() {
    cerr << "Usage: DataDisplayer [-batch] [-chip <start> <end>]* [-chan <start> <end>]* <ISNames> ..." << endl << endl;
    cerr << "DataDisplayer is a program that shows data that has been stored in IS" << endl;
    cerr << "If no -chip or -chan arguments are given, it will show information about all 12 chips and no channels by default"<< endl;
    cerr << "Overall module information is always shown." << endl;
    cerr << "Arguments: " << endl;
    cerr << "-batch (flag)                     : Don't display windows, but instead create postscript files" << endl;
    cerr << "-chip <start> <end> (optional)    : A range of chips to display (may be specified more than once)" << endl;
    cerr << "-chan <start> <end> (optional)    : A range of channels to display (may be specified more than once)" << endl;
    cerr << "<Names> (at least 1)	       : A list of names in IS or filenames of archived data" << endl;
    cerr << "--help			       : Show this help message" << endl;
    cerr << "All numbers are in base 0" << endl;
    
    exit(-1);
}

void parseChip(DisplayInfo& info, string start, string end) {
    istringstream iss1(start);
    unsigned int chipStart = nChipModule;
    iss1 >> chipStart;
    if (chipStart >= nChipModule) {
	cerr << "Invalid argument: " << start << " to -chip" << endl;
	printHelp();
    }
    
    istringstream iss2(end);
    unsigned int chipEnd = nChipModule;
    iss2 >> chipEnd;
    if (chipEnd >= nChipModule) {
	cerr << "Invalid argument: " << end << " to -chip" << endl;
	printHelp();
    }
    
    for (unsigned int i=chipStart; i<=chipEnd; ++i) {
	info.displayChips.push_back(i);
    }
}

void parseChan(DisplayInfo& info, string start, string end) {    
    istringstream iss1(start);
    unsigned int chanStart = nChannelModule;
    iss1 >> chanStart;
    if (chanStart >= nChannelModule) {
	cerr << "Invalid argument: " << start << " to -chan" << endl;
	printHelp();
    }
    
    istringstream iss2(end);
    unsigned int chanEnd = nChannelModule;
    iss2 >> chanEnd;
    if (chanEnd >= nChannelModule) {
	cerr << "Invalid argument: " << end << " to -chan" << endl;
	printHelp();
    }
    
    for (unsigned int i=chanStart; i<=chanEnd; ++i) {
	info.displayChannels.push_back(i);
    }    
}

DisplayInfo handleArgs(int argc, char** argv) {
   if (argc == 1) printHelp();
    
    DisplayInfo info;
    
    for (int i=1; i<argc; ++i) {
	if (string(argv[i]) == "--help") printHelp();
	
	if (string(argv[i]) == "-chip") {
	    if (i+2 >= argc) {
		cerr << "Invalid arguments for option -chip" << endl;
		printHelp();
	    }
	    parseChip(info, argv[i+1], argv[i+2]);	    
	    i += 2;
	} else if (string(argv[i]) == "-chan") {
	    if (i+2 >= argc) {
		cerr << "Invalid arguments for option -chip" << endl;
		printHelp();
	    }
	    parseChan(info, argv[i+1], argv[i+2]);	    	
	    i += 2;
	} else if (string(argv[i]) == "-batch") {
	  DisplayManager::setBatchMode();
	} else {
	    isNames.push_back(argv[i]);
	}
    }
    
    if (info.displayChannels.size() == 0 && info.displayChips.size() == 0)
	info = getDefaultInfo();
    
    return info;
}

shared_ptr<Serializable> readObject(string name) {
    FILE* f = fopen(name.c_str(), "rb");
    if (f) {
	fclose(f);
	if (name.substr(name.size()-3) == ".gz"){
	  cout << "Reading from archive file" << std::endl;
	  return IOManagerArchiveFile::instance().read(name);
	} else {
	  cout << "Reading from scratch space" << std::endl;	  
	  return IOManagerFile::instance().read(name);
	}
    } else {
	  cout << "Reading from information service" << std::endl;	  
	return IOManagerIS::instance().read(name);
    }
}

//The main for the program...
//Handles input that passes the rest to the Displayers

int main(int argc, char** argv) {
    setExceptionHandlers(argv[0]);

    DisplayInfo info = handleArgs(argc, argv);

    //Create TApplication - necessary to prevent batch mode and no graphics on later ROOT versions
    TApplication* myapp=0;
    if (!DisplayManager::batchMode()) myapp=new TApplication("myapp", 0, 0);
    
    int errorCount = 0;
    
    vector<shared_ptr<DisplayData> > displayObjs;
    for (unsigned int i=0; i<isNames.size(); ++i) {
	try {
	    shared_ptr<Serializable> ob = readObject(isNames[i]);
	    if (!ob) 
		throw IoException("No object `" + isNames[i] + "' in IS, scratch space or data archive", __FILE__, __LINE__);

	      if (DisplayManager::batchMode()){
		std::ostringstream oss;
		displayObjs.push_back(DisplayManager::display(ob, info, oss));
		if (oss.str()!=""){
		  std::string filename=ob->getClassName();
		  filename+=".";
		  filename+=ob->getUniqueID();
		  std::ofstream f(filename.c_str());
		  f << oss.str();
		  std::cout << "output to file " << filename << std::endl;
		}
	      }else{
		displayObjs.push_back(DisplayManager::display(ob, info, std::cout));
	      }
	} catch (Sct::Exception& e) {
	  e.sendToMrs(MRS_DIAGNOSTIC);
	  ++errorCount;
	}
    }
    
    if (myapp) { 
      myapp->Run();
    } else {
      DisplayManager::OutputCanvases();
    }
 
    return errorCount;
}
