#include "SctApi.h"
#include "crate.h"
#include "utility.h"

#include "SctApiDebug.h"

using namespace std;
using namespace SctApi::Utility;

namespace {
  void decodeEventWord(UINT16 currWord, int errorType, bool &linkCondensed, 
                       string &type, vector<pair<string, int> > &arguments, vector<string> &errors);

  vector<string> decodeEventFlags(UINT32 flags);

  list<pair<int, int> > getFragmentList(UINT32 *buffer);
}

namespace SctApi {
  const int eventSize = 0x400;
  const int eventWords = eventSize / 4;
  const int frameCount = 32;

  inline UINT32 eventWord(UINT32 *buffer, int frame, int index) {
    unsigned int offset = (eventWords * frame + index) % (frameCount * eventWords); 
    return buffer[offset];
  }

  inline int findEOF(UINT32 *buffer, int startFrame) {
    int eofAt = 0;
    while(eventWord(buffer, startFrame, eofAt) != 0xe0f00000 && eofAt < (eventWords * frameCount)) eofAt ++;

    if(eofAt < (eventWords * frameCount)) {
      return eofAt;
    } else {
      return -1;
    }
  }

void SctApi::scanEvents(unsigned int partition, unsigned int crate, unsigned int rod, int sl, bool extFlag, bool errorType) {
  int beginSl, endSl;

  if(sl == -1) {
    beginSl = 0;
    endSl = 4;
  } else {
    beginSl = sl;
    endSl = sl+1;
  }

  for(int slave = beginSl; slave<endSl; slave++) {
    if(sl == -1) {
      cout << "Slave: " << slave << endl;
    }

    int lastFrame = getLastEventFrame(partition, crate, rod, slave);

    if(lastFrame == -1) {
      cout << " No frames in buffer\n";
    } else {
      cout << " Start of last event at frame: " << lastFrame << endl;
    }

    UINT32 *frameBuffer = readEventBuffer(partition, crate, rod, slave, extFlag, errorType);

    if(frameBuffer == 0) {
      cout << "Couldn't load memory\n";
      return;
    }

    int trailerLength;
    int headerLength;

    if(errorType) {
      headerLength = 6;
      trailerLength = 8;
    } else {
      headerLength = 9;
      trailerLength = 5;
    }

    list<pair<int, int> > fragments = getFragmentList(frameBuffer);

    for(list<pair<int, int> >::const_iterator iter = fragments.begin();
        iter != fragments.end();
        iter++) {
      cout << "Fragment from ";
      cout.width(2);
      cout << iter->first << "-";
      cout.width(2);
      cout << iter->second;

      int L1ID, BCID;
      if(errorType) {
        L1ID = eventWord(frameBuffer, iter->first, 2);
        BCID = eventWord(frameBuffer, iter->first, 3);
      } else {
        L1ID = eventWord(frameBuffer, iter->first, 5);
        BCID = eventWord(frameBuffer, iter->first, 6);
      }

      cout << " L1 ";
      cout.width(4);
      cout << L1ID << " BCID ";
      cout.width(4);
      cout << BCID << endl;

      int eofAt = findEOF(frameBuffer, iter->first);
      int errorCount = eventWord(frameBuffer, iter->first, eofAt-trailerLength);
      if(errorCount > 0) {
        cout << " Error count = " << errorCount; 

        int errorFlags = eventWord(frameBuffer, iter->first, eofAt-trailerLength + 1);

        cout << " flags = 0x" << hex << errorFlags << dec;

        vector<string> flagStrings = decodeEventFlags(errorFlags);
        for(vector<string>::const_iterator iter = flagStrings.begin();
            iter != flagStrings.end();
            iter++) {
          cout << " " << *iter;
        }

        cout << endl;
      }
    }
  }
}

void SctApi::decodeEvent(unsigned int partition, unsigned int crate, unsigned int rod, 
                         int slave, int index, bool extFlag, bool errorType) {
  UINT32 *frameBuffer = readEventBuffer(partition, crate, rod, slave, extFlag, errorType);

  if(frameBuffer == 0) {
    cout << "Couldn't load memory\n";
    return;
  }

  int lastFrame = -1;

  // Finds one beyond last
  for(int i=index+1; i<frameCount+index; i++) {
    if((eventWord(frameBuffer, i, -1) & 0xff0e0000) == 0x400e0000) {
      lastFrame = i;
      break;
    } 
  }

  if(lastFrame == -1) {
    cout << "Can't find end of frame\n";
    return;
  } else {
    cout << "Found " << (lastFrame - index) << " frames of event data\n";
  }

  if((eventWord(frameBuffer, lastFrame, -1) & 0xff0e0000) != 0x400e0000) {
    cout << "Event trailer marker not found " << eventWord(frameBuffer, lastFrame, -1) << endl;
    return;
  } 

  if(eventWord(frameBuffer, index, 0) == 0xb0f00000) { 
    cout << "Found header\n";
  } else {
    cout << "Bad header 0x" << hex << eventWord(frameBuffer, index, 0) << dec << endl;
    return;
  }

  if(eventWord(frameBuffer, index, 1) == 0xee1234ee) {
    cout << "Valid header\n";
  } else {
    cout << "Bad check 0x" << hex << eventWord(frameBuffer, index, 1) << dec << endl;
  }

  UINT32 headerLength;     // Including bof
  UINT32 trailerLength;    // Excluding eof

  if(errorType) {
    headerLength = 6;
    trailerLength = 8;

    cout << "L1ID = 0x" << hex << eventWord(frameBuffer, index, 2) 
         << " BCID = 0x" << eventWord(frameBuffer, index, 3) << dec << endl;
    cout << "Trigger type = " << eventWord(frameBuffer, index, 4) 
         << " Event type = " << eventWord(frameBuffer, index, 5) << endl;
  } else {
    headerLength = 9;
    trailerLength = 5;

    if(eventWord(frameBuffer, index, 2) != headerLength) {
      cout << "Unknown header length (" << eventWord(frameBuffer, index, 2) << ")\n";
    } 

    cout << "Version: " << eventWord(frameBuffer, index, 3) 
         << " ID: " << eventWord(frameBuffer, index, 4) << endl;
    cout << "L1ID = " << eventWord(frameBuffer, index, 5) 
         << " BCID = " << eventWord(frameBuffer, index, 6) 
         << " TType = " << eventWord(frameBuffer, index, 7) 
         << " det type = " << eventWord(frameBuffer, index, 8) << endl;
  } 

  int eofAt = findEOF(frameBuffer, index);

  if(eofAt != -1) {
    cout << "Found end of frame!\n";
  } else {
    cout << "EOF not found\n";
    return;
  }

  int eventLength = eofAt-headerLength-trailerLength;

  UINT32 rawOffset = headerLength;

  // Something different to the first word
  UINT16 lastWord = 1 + ((eventWord(frameBuffer, index, rawOffset) & 0xffff0000) >> 16);
  UINT32 repeats = 0;

  bool linkCondensed = true;

  for(int i=0; i<eventLength * 2; i++) {
    UINT16 currWord;
    {
      UINT32 rawWord = eventWord(frameBuffer, index, rawOffset + i/2);
      if(i&1) {
        currWord = rawWord & 0x00ffff;
      } else {
        currWord = (rawWord & 0xffff0000) >> 16;
      }
    }

    if(currWord == lastWord) {
      repeats ++;
      continue;
    } else if(repeats) {
      cout << "        Repeated " << repeats << " times\n";
      repeats = 0;
    }

    lastWord = currWord;

    cout.width(4);
    cout.fill('0');
    cout << hex << currWord << dec;
    cout.fill(' ');

    string type;
    vector<pair<string, int> > arguments;
    vector<string> errors;

    decodeEventWord(currWord, errorType, linkCondensed, 
                    type, arguments, errors);

    cout << " " << type << " ";
    for(vector<pair<string, int> > :: const_iterator iter = arguments.begin();
        iter != arguments.end();
        iter ++) {
      cout << "  " << iter->first << ": " << iter->second;
    }

    if(errors.size() > 0) {
      cout << "  ERRORS: ";
    }
    for(vector<string> :: const_iterator iter = errors.begin();
        iter != errors.end();
        iter ++) {
      cout << " " << *iter;
    }
    cout << endl;

  }

  if(repeats) {
    cout << "        Repeated " << repeats << " times\n";
    repeats = 0;
  }

  cout << "Error count = " << eventWord(frameBuffer, index, eofAt-trailerLength) << endl;
  cout << "Error flags = 0x" << hex << eventWord(frameBuffer, index, eofAt-trailerLength + 1) << dec << endl;

  int flags = eventWord(frameBuffer, index, eofAt-trailerLength + 1);
  vector<string> flagStrings = decodeEventFlags(flags);
  for(vector<string>::const_iterator iter = flagStrings.begin();
      iter != flagStrings.end();
      iter++) {
    cout << *iter << " ";
  }

  if(flagStrings.size() > 0) 
    cout << endl;

  if(errorType) {
    cout << "Error Status:  0: No data 1: Good data 2: BC or L1 error 3: Timeout\n" << hex;

    for(int es = 0; es < 6; es++) {
      UINT16 esWord = eventWord(frameBuffer, index, eofAt-trailerLength + 2 + es);
      for(int i=0; i<8; i++) {
        cout << (esWord & 0x3) << " ";
        esWord >>= 2;
      }
    }

    cout << dec << endl;
  } else {
    cout << "nData = " << eventWord(frameBuffer, index, eofAt-2) 
         << " words found = " << (eofAt - (headerLength + trailerLength)) << endl;
  }

  cout << "EOF = 0x" << hex << eventWord(frameBuffer, index, eofAt) << dec << endl;

  if(checkDebugOption(DEBUG_DUMP_RAW_EVENT)) {
    // Print out all the raw data afterwards
    int wordsPerLine = 8;

#warning "Is it more useful as it was (only print out the event data)"
    for(int i=index; i<lastFrame; i++) {
      printMemoryBlock(cout, (long unsigned *)frameBuffer[(i % frameCount) * eventWords], eventWords, wordsPerLine);
    }
  }
  if(checkDebugOption(DEBUG_DIAG)) {
    cout << (lastFrame - index) << " frames finished in frame " << lastFrame << " with 0x";
    cout.fill('0'); cout.width(8);
    cout << hex << int(eventWord(frameBuffer, lastFrame, -1)) << dec << endl;
    cout.fill(' ');
  }
}

void SctApi::reportEventErrors() {
  // If no mrs then don't bother!
  if(!mrs) 
    return;

  // In a standard scan these aren't set
  bool extFlag = false;
  bool errorType = false;

  for(list<RodLabel>::const_iterator rl = rodList.begin();
      rl!=rodList.end();
      rl++){
    for(int slave=0; slave<4; slave++) {
      int lastFrame = getLastEventFrame(rl->partition, rl->crate, rl->rod, slave);

      *mrs << "EVENT_REPORT" << MRS_DIAGNOSTIC << MRS_QUALIF("SCTAPI") 
           << MRS_PARAM<int>("partition", rl->partition)
           << MRS_PARAM<int>("crate", rl->crate)
           << MRS_PARAM<int>("rod", rl->rod)
           << MRS_PARAM<int>("slave", slave);
      if(lastFrame == -1) {
        *mrs << MRS_TEXT("Event trap empty");
      } else {
        *mrs << MRS_PARAM<int>("lastFrame", lastFrame)
             << MRS_TEXT("Start of last event at frame");
      }
      *mrs << ENDM;

      UINT32 *eventBuffer = readEventBuffer(rl->partition, rl->crate, rl->rod, slave, extFlag, errorType);

      if(eventBuffer == 0) {
        cout << "Couldn't load memory\n";
        return;
      }

      int trailerLength;
      int headerLength;
      if(errorType) {
        headerLength = 6;
        trailerLength = 8;
      } else {
        headerLength = 9;
        trailerLength = 5;
      }

      list<pair<int, int> > fragments = getFragmentList(eventBuffer);

      for(list<pair<int, int> >::const_iterator iter = fragments.begin();
          iter != fragments.end();
          iter++) {

        int L1ID, BCID;
        if(errorType) {
          L1ID = eventWord(eventBuffer, iter->first, 2);
          BCID = eventWord(eventBuffer, iter->first, 3);
        } else {
          L1ID = eventWord(eventBuffer, iter->first, 5);
          BCID = eventWord(eventBuffer, iter->first, 6);
        }

        *mrs << "FRAGMENT_REPORT" << MRS_DIAGNOSTIC << MRS_QUALIF("SCTAPI") 
             << MRS_PARAM<int>("partition", rl->partition)
             << MRS_PARAM<int>("crate", rl->crate)
             << MRS_PARAM<int>("rod", rl->rod)
             << MRS_PARAM<int>("slave", slave)
             << MRS_PARAM<int>("bcid", BCID)
             << MRS_PARAM<int>("l1id", L1ID);

        int eofAt = findEOF(eventBuffer, iter->first);

        int eventLength = eofAt-headerLength-trailerLength;

        bool linkCondensed = false;

        int errorCount = eventWord(eventBuffer, iter->first, eofAt-trailerLength);

        if(errorCount > 0) {
          *mrs << MRS_PARAM<int>("errorCount", errorCount);

          int errorFlags = eventWord(eventBuffer, iter->first, eofAt-trailerLength + 1);
          vector<string> flagStrings = decodeEventFlags(errorFlags);

          string flagString;

          for(vector<string>::const_iterator iter = flagStrings.begin();
              iter != flagStrings.end();
              iter++) {
            flagString += *iter;
            flagString += " ";
          }

          *mrs << MRS_PARAM<const char *>("errorFlags", flagString.c_str());
        }

        *mrs << ENDM;

        for(int i=0; i<eventLength * 2; i++) {
          UINT16 currWord;
          {
            UINT32 rawWord = eventWord(eventBuffer, iter->first, headerLength + i/2);
            if(i&1) {
              currWord = rawWord & 0x00ffff;
            } else {
              currWord = (rawWord & 0xffff0000) >> 16;
            }
          }

          string type;
          vector<pair<string, int> > arguments;
          vector<string> errors;

          decodeEventWord(currWord, errorType, linkCondensed, 
                          type, arguments, errors);

          if(errors.size() > 0) {
            *mrs << "EVENT_WORD" << MRS_DIAGNOSTIC << MRS_QUALIF("SCTAPI") 
                 << MRS_TEXT("Event word with errors")
                 << MRS_PARAM<const char *>("wordType", type.c_str());

            for(vector<pair<string, int> > :: const_iterator iter = arguments.begin();
                iter != arguments.end();
                iter ++) {
              *mrs << MRS_PARAM<int>(iter->first.c_str(), iter->second);
            }

            string errorString;
            for(vector<string> :: const_iterator iter = errors.begin();
                iter != errors.end();
                iter ++) {
              errorString += " " + *iter;
            }

            *mrs << MRS_PARAM<const char *>("errors", errorString.c_str())
                 << ENDM;
          } // End of word
        }
      }
    }
  }
}

UINT32 *SctApi::readEventBuffer(unsigned int partition, unsigned int crate, unsigned int rod, 
                                int slave, bool extFlag, bool errorType) {
  long bufferBase;
  if(extFlag)
    bufferBase = 0x2092000;
  else {
    if(getCrate(partition, crate)->getRodRevision(rod) == 0xe) {
      bufferBase = 0x18000; // BURST_BFR_BASE;
    } else {
      bufferBase = 0x80008000;
    }
  }

  // Storage for 32 frames
  UINT32 *frameBuffer = new UINT32[eventWords * 32];

  unsigned long *mem = primReadSlaveDsp(partition, crate, rod,
                                        slave, bufferBase, eventWords * 32);

  if(mem) {
    copy(mem, mem + eventWords*32, frameBuffer);
  } else {
    return 0;
  }

  return frameBuffer;
}

int SctApi::getLastEventFrame(unsigned int partition, unsigned int crate, unsigned int rod, int slave) {
  unsigned long IDRAM_BASE;
  if(getCrate(partition, crate)->getRodRevision(rod) == 0xe) {
    IDRAM_BASE = 0x10000;
  } else {
    IDRAM_BASE = 0x80000000;
  }

  unsigned long *regs = primReadSlaveDsp(partition, crate, rod, slave, IDRAM_BASE + 0x10, 20);
  unsigned long trapStat0 = regs[1]; 
//   unsigned long trapStat1 = regs[2];

  int eventWordCount = trapStat0 & 0xffff;
  int frameTail = (trapStat0 & 0xff0000) >> 16;

  int eventFrames = (eventWordCount / eventWords) + 1;

  return (frameTail - eventFrames) % frameCount;
}

} // Close SctApi namespace

namespace {

void decodeEventWord(UINT16 currWord, int errorType, bool &linkCondensed, 
                     string &type, vector<pair<string, int> > &arguments, vector<string> &errors) {
  type = "UNKNOWN";
  arguments.clear();
  errors.clear();

  switch((currWord & 0xe000) >> 13) {
  case 0:
    {
      if(currWord & 0x1f80) 
        type = "INVALID";
      else {
        // Flagged error
        type = "Flagged error";
        errors.push_back("FLAGGED");
        arguments.push_back(make_pair("chip", (currWord & 0x78) >> 3));
        arguments.push_back(make_pair("value", currWord & 0x7));
      }
    }
    break;
  case 1:
    {
      // Header
      type = "Header";
      if(errorType) {
        // No indication of whether its condensed...??
        arguments.push_back(make_pair("L1", ((currWord & 0x0f00) >> 8)));
        arguments.push_back(make_pair("BCID", ((currWord & 0xff))));
        if(currWord & 0x1000) {
          errors.push_back("Preamble err");
        }
      } else {
        arguments.push_back(make_pair("Link", (currWord & 0x7f)));
        if(currWord & 0x100) {
          arguments.push_back(make_pair("Condensed mode", 1));
          linkCondensed = true;
        } else {
          linkCondensed = false;
        }
        if(currWord & 0x200) 
          errors.push_back("BC err");
        if(currWord & 0x400)
          errors.push_back("L1 err");
        if(currWord & 0x800)
          errors.push_back("Time out err");
        if(currWord & 0x1000)
          errors.push_back("Preamble err");
      }
    }
    break;
  case 2:
    {
      if(currWord & 0x3ff) 
        type = "INVALID";
      else {
        // Trailer
        type = "Trailer";
        if(currWord & 0x400)
          errors.push_back("Data overflow err");
        if(currWord & 0x800)
          errors.push_back("H/T limit err");
        if(currWord & 0x1000)
          errors.push_back("Trailer bit err");
      }
    }
    break;
  case 3:
    {
      if(currWord & 0x300) 
        type = "INVALID";
      else {
        // Raw data
        type = "Raw";
        arguments.push_back(make_pair("bits", ((currWord & 0x1c00) >> 10) + 1));
        arguments.push_back(make_pair("value", (currWord & 0xff)));
      }
    }
    break;
  default:
    // Everything else (hits)
    {
      if((currWord & 0x2) == 0) {
        // Check if it should be condensed
        if(linkCondensed || errorType) {
          arguments.push_back(make_pair("Chip", ((currWord & 0x7e00) >> 11)));
          arguments.push_back(make_pair("Channel", ((currWord & 0x7f0) >> 4)));
          if(currWord & 1) {
            type = "Condensed double hit";
            if(currWord & 0x4) errors.push_back("Error in hit1");
            if(currWord & 0x8) errors.push_back("Error in hit2");
          } else {
            type = "Condensed hit";
            if(currWord & 0x4) errors.push_back("Error in hit");
          }
        }
      }

      // Only check if expanded is a posibility
      if(!linkCondensed || errorType) {
        if(currWord & 0x8 == 0) {
          type = "1st hit clust exp";
          arguments.push_back(make_pair("Chip", ((currWord & 0x7e00) >> 11)));
          arguments.push_back(make_pair("Channel", ((currWord & 0x7f0) >> 4)));
          arguments.push_back(make_pair("hits", currWord & 0x7));
        } else {
          if((currWord & 0x7f00) == 0) {
            type = "Clust exp";
            arguments.push_back(make_pair("hits", currWord & 0x7));
            if(currWord & 0x80) {
              arguments.push_back(make_pair("hits2", (currWord & 0x70) >> 4));
            }
            cout << endl;
          }
        }
      }
    }
  }
}

vector<string> decodeEventFlags(UINT32 flags) {
  vector<string> result;

  if(flags & 0x1) result.push_back("HEADER");
  if(flags & 0x2) result.push_back("TRAILER");
  if(flags & 0x4) result.push_back("FLAGGED");
  if(flags & 0x8) result.push_back("\"HIT PATTERN\"");
  if(flags & 0x10) result.push_back("SYNC");
  if(flags & 0x20) result.push_back("L1ID");
  if(flags & 0x40) result.push_back("BCID");
  if(flags & 0x80) result.push_back("TIMEOUT");
  if(flags & 0x100) result.push_back("\"ALMOST FULL\"");
  if(flags & 0x200) result.push_back("OVERFLOW");
  if(flags & 0x400) result.push_back("\"CHIP SEQ\"");
  if(flags & 0x800) result.push_back("\"BAD CHIP\"");

  return result;
}

list<pair<int, int> > getFragmentList(UINT32 *eventBuffer) {
  list<pair<int, int> > fragments;
  int firstEnd = -1; 

  int lastBegin = -1;
  int lastEnd = -1;

  for(int i=0; i<SctApi::frameCount; i++) {
    if((SctApi::eventWord(eventBuffer, i, 0) == 0xb0f00000)
       && (SctApi::eventWord(eventBuffer, i, 1) == 0xee1234ee)) {
      lastBegin = i;
    }

    if((SctApi::eventWord(eventBuffer, i+1, -1) & 0xff0e0000) == 0x400e0000) {
      if(lastBegin == -1) 
        firstEnd = i;
      lastEnd = i;
      if(lastBegin != -1) {
        fragments.push_back(make_pair(lastBegin, lastEnd));
        lastBegin = lastEnd = -1;
      }
    }
  }

  if(firstEnd != -1 && lastBegin != -1) {
    fragments.push_back(make_pair(lastBegin, firstEnd));
  }

  return fragments;
}

} // Close null namespace
