#include <iostream>

#include <boost/lexical_cast.hpp>

#include "Sct/Env.h"

#include "SctApi.h"
#include "SctApiDebug.h"
#include "crate.h"
#include "utility.h"

#include "primListWrapper.h"
#include "PrimBuilder.h"

#include "VmeDecode.h"

// For exceptions
#include "RodCrate/RodModule.h"

using namespace std;
using namespace SctPixelRod;
using namespace boost;
using namespace SctApi::Utility;

namespace SctApi {
#warning "Should some of these return smart pointers?"

/*
  Dump contents of a block of DSP memory to cout.
  Use dspNumber -1 to refer to the master DSP.
*/
int SctApi::dspBlockDump(unsigned int partition, unsigned int crate, unsigned int rod,
                         long dspStart, long numWords, long dspNumber, bool usePrim) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "DspBlockDump (" << partition << " " << crate << " " << rod << ") "
        << dspStart << " " << numWords << " " << dspNumber << endl;
  }

  // read data
  unsigned long length;
  unsigned long *buffer = dspBlockRead(partition, crate, rod, dspStart, numWords, dspNumber, length, usePrim);

  if(!buffer) {
    // Something didn't work
    return 1;
  }

  printMemoryBlock(cout, buffer, numWords, 8, 0);

  char *name = tempnam("./", "textB");
  if(checkDebugOption(DEBUG_DIAG))
    cout << "Write output to " << name << endl;
  ofstream output(name);
  free(name);

  output.write((char *)buffer, numWords * 4);
 
  delete [] buffer;

  return 0;
}

/*
  Dump contents of a block of DSP memory to a file.
  Use dspNumber -1 to refer to the master DSP.
*/
int SctApi::dspBlockDumpFile(unsigned int partition, unsigned int crate, unsigned int rod,
                             long dspStart, long numWords, long dspNumber, string filename, bool usePrim){
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "DspBlockDumpFile (" << partition << " " << crate << " " << rod << ") "
        << hex << "0x" << dspStart << hex << " " << numWords << " " << dspNumber << " to " << filename << endl;
  }

  unsigned long length;
  unsigned long * buffer = dspBlockRead(partition, crate, rod, dspStart, numWords, dspNumber, length, usePrim);

  if(buffer) {
    if(checkDebugOption(DEBUG_DIAG2)) 
      cout << "Writing output to " << filename << endl;
    ofstream output(filename.c_str());

    output.write((char *)buffer, numWords * 4);

    // delete the buffer
    delete [] buffer;
  } else {
    cout << "Block read failed!\n";
    return 1;
  }

  return 0;
}

/*
  Read contents of a block of DSP memory 

  Use dspNumber -1 to refer to the master DSP.
*/
unsigned long *SctApi::dspBlockRead(unsigned int partition, unsigned int crate, unsigned int rod,
                                    long dspStart, long numWords, long dspNumber, unsigned long &length, bool usePrim){
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "DspBlockRead (" << partition << " " << crate << " " << rod << ") "
        << hex << "0x" << dspStart << dec << " " << numWords << " " << dspNumber << endl << flush;
  }
  Crate* myCrate = getCrate(partition, crate);

  if(!myCrate) {
    cout << "Request for non-existent crate " << partition << " " << crate << endl;
    return 0;
  }

  if(dspNumber < -1 || dspNumber > 3) {
    cout << "dspBlockRead slave out of range " << dspNumber << endl;
    throw SctApiException("DspBlockRead slave out of range");
  }

  // create buffer
  unsigned long * buffer = new unsigned long[numWords];

  for(int i=0; i<numWords; i++) {
    buffer[i] = 0;
  }

  // read data
  if (dspNumber == -1) {
    if(checkDebugOption(DEBUG_DIAG2))
      cout << "Reading from mdsp\n";
    myCrate->mdspBlockRead(rod, dspStart, buffer, numWords);
  } else {
    if(checkDebugOption(DEBUG_DIAG2))
      cout << "Reading from dsp " << dspNumber << endl;

    if(usePrim) {
      delete [] buffer;
      buffer = primReadSlaveDsp(partition, crate, rod, dspNumber, dspStart, numWords);
    } else {
      myCrate->slvBlockRead(rod, dspStart, buffer, numWords, dspNumber);
    }
  }

  length = numWords;

  // Don't delete the buffer!!!
  return buffer;
}

int SctApi::dspBlockWrite(unsigned int partition, unsigned int crate, unsigned int rod,
                          unsigned long *buffer, unsigned long dspAddress, long numWords, 
                          long dspNumber, bool usePrim){
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "DspBlockWrite (" << partition << " " << crate << " " << rod << ") "
        << dspAddress << " " << numWords << " " << dspNumber << endl << flush;
  }

  Crate* myCrate = getCrate(partition, crate);

  if(!myCrate) {
    cout << "Request for non-existent crate " << partition << " " << crate << endl;
    return 0;
  }

  // Write data
  if (dspNumber == -1) {
    if(checkDebugOption(DEBUG_DIAG))
      cout << "Writing to mdsp\n";
    myCrate->mdspBlockWrite(rod, dspAddress, buffer, numWords);
  } else {
    if(checkDebugOption(DEBUG_DIAG))
      cout << "Writing to dsp " << dspNumber << endl;
    if(usePrim) {
      primWriteSlaveDsp(partition, crate, rod, dspNumber, dspAddress, numWords, buffer);
    } else {
      myCrate->slvBlockWrite(rod, dspAddress, buffer, numWords, dspNumber);
    }
  }

  return 0;
}

void SctApi::setSlaveBlock(unsigned int partition, unsigned int crate, unsigned int rod, int s, 
                           long sAdd, int words, long value, bool usePrim) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "SetSlaveBlock to " << value << endl;
  }

  if(usePrim) {
    shared_ptr<PrimListWrapper> slList(new PrimListWrapper(1));
    PrimBuilder::instance().setMemory(slList, sAdd, words, value);

    shared_ptr<PrimListWrapper> primList(new PrimListWrapper(1));
    PrimBuilder::instance().slavePrimList(primList, slList, s, true, false);

    sendPrimList(partition, crate, rod, primList);

    int responseCode = awaitResponse(partition, crate, rod, 10);
    if(responseCode == 0) {
      cout << "Slave " << s << " block set to " << value << " at address 0x" << hex << sAdd << dec << "\n";
    } else {
      cout << "Slave set failed\n";
    }
  } else {
    // "Safe" way
    unsigned long *buffer = new unsigned long[words];
    for(int i=0; i<words; i++) {
      buffer[i+5] = value;
    }
    dspBlockWrite(partition, crate, rod, buffer, sAdd, words, s, usePrim);
  }
}

unsigned long *SctApi::primReadSlaveDsp(unsigned int partition, unsigned int crate, unsigned int rod,
                                        int s, int add, int words) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "Read slave (" << s << ") memory 0x" << hex << add << dec << endl;
  }

  unsigned long *result = new unsigned long[words];

  // The 6 is the header and trailer of the primlist
  const int maxLength = (REPLY_BUFF_SIZE)/4 - 6 - sizeof(RW_SLAVE_MEMORY_IN)/4;

  int wordsDone = 0;
  while(wordsDone < words) {
    int primLength = min(maxLength, words-wordsDone);

    shared_ptr<PrimListWrapper> rwList(new PrimListWrapper(2));

    PrimBuilder::instance().readSlaveMemory(rwList, s, add+wordsDone*4, primLength);

    sendPrimList(partition, crate, rod, rwList);

    int responseCode = awaitResponse(partition, crate, rod, 10);

    if(responseCode != 0) {
      cout << "Read slave memory failed!\n";
      return 0;
    }

    unsigned long length;
    unsigned long *mem = getResponse(partition, crate, rod, length);

    if(!mem) {
      return 0;
    }

    copy(mem+8, mem+8+primLength, result + wordsDone);
    delete [] mem;

    wordsDone += primLength;
  }

  return result;
}

void SctApi::primWriteSlaveDsp(unsigned int partition, unsigned int crate, unsigned int rod,
                               int s, int add, int numWords, unsigned long *data) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "Write slave (" << s << ") memory 0x" << hex << add << dec << endl;
  }

  const int maxLength = (REPLY_BUFF_SIZE)/4 - 6 - sizeof(RW_SLAVE_MEMORY_IN)/4;

  int wordsDone = 0;
  while(wordsDone < numWords) {
    int primLength = min(maxLength, numWords-wordsDone);

    shared_ptr<PrimListWrapper> rwList(new PrimListWrapper(2));

    long *primData = new long[sizeof(RW_SLAVE_MEMORY_IN)/4 + primLength];

    RW_SLAVE_MEMORY_IN &rwPrim = *(RW_SLAVE_MEMORY_IN*)primData;

#if (R_RW_SLAVE_MEMORY == 100)
    rwPrim.slaveNumber = s;
    rwPrim.readNotWrite = 0;
    rwPrim.slaveAddress = (UINT32 *)add + wordsDone;
    rwPrim.masterAddress = (UINT32 *)0xffffffff;
    rwPrim.numWords = primLength;
#else
#error "RW_SLAVE_MEMORY revision change"
#endif

    copy(data + wordsDone, data + wordsDone + primLength, primData + 5);

    rwList->addPrimitive(RodPrimitive(4 + sizeof(RW_SLAVE_MEMORY_IN)/4 + primLength, 
                                      1, RW_SLAVE_MEMORY, R_RW_SLAVE_MEMORY, primData),
                         primData);

    sendPrimList(partition, crate, rod, rwList);
    int responseCode = awaitResponse(partition, crate, rod, 10);

    if(responseCode != 0) {
      cout << "Write slave memory failed!\n";
      return;
    }

    wordsDone += primLength;
  }

  return;
}

shared_array<unsigned long> SctApi::loadFile(string fileName, unsigned long & length) {
  shared_array<unsigned long> buffer;
  ifstream fin;
  fin.open(fileName.c_str(), ios::binary);
  if (fin.is_open()) {
    unsigned long fileSize;

    // Get size of file
    fin.seekg(0, std::ios::end);  // go to end of file
    fileSize = fin.tellg();  // file size is current location
    fin.seekg(0, std::ios::beg);  // go back to beginning of file

    buffer.reset(new unsigned long[fileSize / 4]);

    fin.read((char *)buffer.get(), fileSize);

    length = fileSize / 4;
  }

  return buffer;
}

// Mainly for starting slaves
void SctApi::writeSlaveFile(unsigned int partition, unsigned int crate, unsigned int rod, unsigned int slave, 
                            const string fileName, unsigned long address, bool usePrim) {
  cout << "Writing " << fileName << " to slave " << slave << " at 0x" << hex << address << dec << endl;

  unsigned long length;
  string nameCopy(fileName);
  nameCopy = Sct::Env::substituteVariables(nameCopy);

  shared_array<unsigned long> buffer = loadFile(nameCopy, length);

  if(buffer) {
    if(usePrim) {
      primWriteSlaveDsp(partition, crate, rod,
                        slave, address, length, buffer.get());
    } else {
      dspBlockWrite(partition, crate, rod,
                    buffer.get(), address, length, slave, usePrim);
    }
  } else {
    cout << "Load of file \"" << nameCopy << "\" failed\n";
    throw SctApiException("Load of file \"" + nameCopy + "\" failed");
  }

  shared_array<unsigned long> readBuffer(primReadSlaveDsp(partition, crate, rod, 
                                                          slave, address, length));

  for(unsigned int i=0; i<length; i++) {
    if(buffer[i] != readBuffer[i]) {
      cout << "Slave write verification failed at " << i << endl;
      cout << "\t0x" << hex << buffer[i] << " != 0x" << readBuffer[i] << dec << endl;

      throw SctApiException("Slave verification error: word " + boost::lexical_cast<string>(i));
    }
  }
}

} // End namespace
