// File: SctApi.h
// Library interface to SCTAPI
#ifndef SCTROD_SCTAPI_H
#define SCTROD_SCTAPI_H

// Make sure all compilations have same SctApi structure!
#include "config.h"

#include <list>
#include <map>
#include <string>
#include <vector>

#include <boost/thread.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/shared_array.hpp>
#include <boost/date_time/posix_time/posix_time_types.hpp>

#include <mrs/message.h>
#if USE_IS
#include <is/isinfotmpl.h>
#endif

class SCTDCSCommand;
#include "log.h"

#include "DCSAccess.h"

// This defines ABCDModule (a typedef)
#include <CommonWithDsp/sctStructure.h>
// Unfortunately this is necessary
#include <CommonWithDsp/processor.h>

struct scan_result_ptrs;

namespace SctConfiguration {
  class Configuration;
  struct BOCChannelConfig;
  struct BOCGlobalConfig;
}

#include "SctApiFwd.h"

namespace SctApi {

class CrateException;

/**
   Exception to be thrown by public API methods
*/
class SctApiException : public std::exception {
  std::string desc;
 public:
  /// Provide a description on construction
  SctApiException(const std::string &str) throw() : desc(str) {}

  /// Copy a crate exception
  SctApiException(const CrateException &c) throw();

  ~SctApiException() throw() {}

  /// Override std::exception method
  virtual const char* what () const throw() { return desc.c_str(); }
};

class Crate;

class PrimListWrapper;

class ScanEx;

/// Trigger interface
class Trigger {
 public:
  virtual ~Trigger();

  /// Type for representation as a series of pairs as defined for ROD DSP
  typedef std::vector<std::pair<UINT16, UINT32> > RODTriggers;

  /// Set to a single L1A
  virtual void singleL1A() = 0;
  /// Set to a double L1A
  virtual void doubleL1A(unsigned short delay) = 0;
  /// Set to a delayed L1A
  virtual void delayedL1A(unsigned short delay) = 0;
  /// Set to a Calibration pulse followed by an L1A BC0s later
  virtual void calL1A(unsigned short delay) = 0;
  virtual void pulseL1A(unsigned short delay) = 0;
  virtual void softL1A(unsigned short delay) = 0;
  virtual void softCalL1A(unsigned short delay, unsigned short delay2) = 0;
  virtual void softPulseL1A(unsigned short delay, unsigned short delay2) = 0;
  virtual void bcL1A(unsigned short delay) = 0;
  virtual void bcCalL1A(unsigned short delay, unsigned short delay2) = 0;
  virtual void bcPulseL1A(unsigned short delay, unsigned short delay2) = 0;

  virtual void setCommIncr(unsigned short command, unsigned short incr) = 0;
  virtual void getCommIncr(unsigned short &command, unsigned short &incr) const = 0;

  /// Get list of triggers in ROD format
  virtual const RODTriggers getRODTriggers() const = 0; 

  virtual void print() const = 0;
};

/**
    Scan class

    Describes a scan request

    There are two module sets defined by the module group 
    number in the configuration.
*/ 
class Scan {
 public:
  virtual ~Scan();

  /// Typedef for list of scan points
  typedef std::vector<FLOAT32> ScanPoints;

  /// Typedef for list of scan points
  typedef std::vector<UINT32> TrigPoints;

  /// Current options for a scan request
  enum ScanOptions {FULL, BITS32, LOOPCALLINE, DISTSLAVE, DEBUG, TIM};

  virtual void configure(unsigned short type, float start, float stop, float step) = 0;
  virtual void configure2(unsigned short type, float start, float stop, float step) = 0;

  virtual const boost::shared_ptr<Trigger> getTrigger1() const = 0;
  virtual const boost::shared_ptr<Trigger> getTrigger2() const = 0;

  virtual boost::shared_ptr<Trigger> getTrigger1() = 0;
  virtual boost::shared_ptr<Trigger> getTrigger2() = 0;

  virtual void setTrigger1(boost::shared_ptr<Trigger> trigger) = 0;
  virtual void setTrigger2(boost::shared_ptr<Trigger> trigger) = 0;

  virtual void setScanVariable1(unsigned short var) = 0;
  virtual void setScanVariable2(unsigned short var) = 0;

  virtual unsigned short getScanVariable1() const = 0;
  virtual unsigned short getScanVariable2() const = 0;

  virtual void setNTrigs(unsigned long nTrigs) = 0;
  virtual unsigned long getNTrigs() const = 0;

  virtual const TrigPoints getVariableTrigs() const = 0;
  virtual void setVariableTrigs(const TrigPoints &scans) = 0;
  virtual void setVariableTrigRange(unsigned short start, unsigned short end, unsigned long value) = 0;

  virtual const ScanPoints getScanPoints1() const = 0;
  virtual const ScanPoints getScanPoints2() const = 0;
  virtual void setScanPoints1(const ScanPoints &scans) = 0;
  virtual void setScanPoints2(const ScanPoints &scans) = 0;

  virtual int getOption(enum ScanOptions) const = 0;
  virtual void setOption(enum ScanOptions, int option) = 0;

  /// Print the scan
  virtual void print() const = 0;

  virtual void setScanNumber(unsigned int scan) = 0;
  virtual unsigned int getScanNumber() const = 0;
  virtual void setRunNumber(unsigned int scan) = 0;
  virtual unsigned int getRunNumber() const = 0;

  virtual std::list<std::string> getModuleList(unsigned int group) const = 0;
  virtual void setModuleList(unsigned int group, std::list<std::string> newList) = 0;
  virtual unsigned int getNGroups() const = 0;
  virtual void setNGroups(unsigned int val) = 0;
  
  virtual boost::posix_time::ptime getStartTime() const = 0;
  virtual void setStartTime(boost::posix_time::ptime t) = 0;
  
  virtual boost::posix_time::ptime getEndTime() const = 0;
  virtual void setEndTime(boost::posix_time::ptime t) = 0;
};

/** 
    SctApi class.

    This is the API used by the rest of the world to interact with
    SCT Modules using RODs.

    Functionality that should be included in this class includes.

    Modules are referred to by an "mid" which is a 32bit number 
    packed in hex as PPCCRRCH where:
       PP is the partition
       CC is the crate
       RR is the ROD
       CH is the ROD channel

    DAQ run level transitions:
       Boot
           Load and run the crate controller (Done automatically in DAQ-1 from configuration)
       Load
           Load config?
       Config
           Send config to Modules
       Start
           Mode for execution of scans

    Configuration:
      loading from file
      loading to modules
      fine grained control on modules
      recovering module configuration from the ROD
      Save configuration

      Reconfiguring from new configuration file
    
    Scanning:
      perform scans over configuration variable(s)

    DCS:
      minimal DCS communication

    ROD diagnostics:
      Flash LEDs
      Text buffer retrieval
      Memory dumps
      Send arbitrary primitives to a single ROD

    Indexing:
      There are several ways of addressing a ROD or module
      the canonical method is
              partition, crate, rod, module

    Other methods include:
       Geographical:
           barrel/disc MUR, number
       MUR:
           Partition, MUR, number
*/
class SctApi {
  SctApi(const SctApi &);
  SctApi &operator=(const SctApi&);
 public:

  /** The constant returned as an unknown module */ 
  static const unsigned int BAD_MODULE;

  /**
     Default to local xmlConfig
  */
  SctApi();

  /**
     Specify configuration
  */
  SctApi(boost::shared_ptr<SctConfiguration::Configuration> newConf);

  virtual ~SctApi();

 private:
  /**
     This sets the unexpected handler to interpret all sorts of exceptions

     Called by constructor
   */
  void setup();

  /**
     If requested, dump prim list to file and/or log to output
   */
  void dumpPrimList(boost::shared_ptr<PrimListWrapper> prim);

  /// Save raw event to file
  void dumpRawEvent(unsigned int partition, unsigned int crate, unsigned int rod, int units, unsigned long *bufferA, unsigned long *bufferB);

public:
  /**
     C++ only method to tell a MRS aware program where to write to
   */
  void setMrsStream(MRSStream *stream);

#if USE_IS
  /**
     C++ only method to tell a IS aware program where to write to
   */
  void setIsDictionary(ISInfoDictionary *dict);
#endif

  /***************** Global configuration functions ***********************/

  /// Initialise all software components
  /** Initialise the crates and create the Rod/Vme objects.

      Load the system configuration from the database.
      Create crates and RODs as necessary.
      Calls initialiseRod and initialiseModule as necessary.

      Store the module configuration in the RODs (UNIMPLEMENTED)

      Return: 0 for success
              Possibly error code for failure

      IPC: Block everything else!
  */
  void initialiseAll(int run);

  /// Tidy up and shutdown
  /** Leave everything in a suitable state and delete all the objects

      Shutdown anything that needs shutting down
      This is where we should send DCS to standby message??

      IPC: Block everything else!
  */
  void shutdownAll();

  /// Change the run number
  /**
     IPC: Don't do during a scan
  */
  void setRunNumber(UINT32 newRun);

  /// Change the next scan number
  /**
     IPC: Don't do during a scan
  */
  void setScanNumber(UINT32 newScan);

  /// Return the run number
  /**
     IPC: Don't do during a scan
  */
  UINT32 getRunNumber();

  /// Return the number of the next scan
  /**
     IPC: Don't do during a scan
  */
  UINT32 getScanNumber();

  /// Check if a ROD has been configured
  /**
     Return true if ROD can accept commands
     (Doesn't go so far as to access ROD hardware)
   */
  bool isRODPresent(unsigned int partition, unsigned int crate, unsigned int rod);

  /// Send the configurations from the RODs to the modules.
  /**
     DDC: Check modules in configurable state.
          Correct run state and no warnings

     IPC:  Don't do during a scan
     Conf: Maybe interaction
   */
  void startupModules();

  /// Return list of configured modules
  std::list<std::string> getModuleList();

  /*********************** Module configuration **********************/

  /// (Re)Load configuration from the "configuration database"
  /**
     IPC:  Don't do during a scan
     Conf: Definite interaction!
  */
  void loadConfiguration();

  /// Load new configuration from a "configuration database"
  /**
     IPC:  Don't do during a scan
     Conf: Definite interaction!
  */
  void loadConfiguration(std::string filename);

  /// Load module configurations from database
  /**
     Return number of module loaded
   */
  int loadModuleConfigurations();

  /// Store the current module configurations into database
  /**
     IPC:  Don't do during a scan
     Conf: Definite interaction!
  */
  void storeModuleConfigurations();

  /// Reset BOC configuration from current 
  /**
     IPC:  Don't do during a scan
     Conf: Definite interaction!
  */
  void configureBOC(unsigned int partition, unsigned int crate, unsigned int rod);

  /// Wait for a ROD to finish a primlist
  /**
     IPC:  Allow during a scan (only expert mode)
     Conf: no interaction!
  */
  int awaitResponse(unsigned int partition, unsigned int crate, unsigned int rod, int timeout);

  /// Wait for all RODs in a crate to finish a primlist
  /**
     IPC:  Allow during a scan (only expert mode)
     Conf: no interaction!
  */
  int awaitResponseAll(unsigned int partition, unsigned int crate, int timeout);

  /// Return the out list
  /**
     (No interaction with VME)
     IPC:  Allow during a scan (only expert mode)
     Conf: No interaction!
  */
  unsigned long *getResponse(unsigned int partition, unsigned int crate, unsigned int rod, 
                             unsigned long &length);

  /** Get a message in the text buffer

      Get a message from a ROD text buffer
      Returns whether the action was completed
  */
  bool getRodMessage(unsigned int partition, unsigned int crate, unsigned int rod,
                     char *buffer, unsigned long &length);

  /*************** ROD Diagnostics ********************/

  /// Flash the LED associated with a slave DSP. 
  /**
     This creates its own primitive list and sends it to 
     a slave DSP on the ROD.

     @param period Flash period given in ms
     @param slaveNumber Slave number or -1 for Master

     IPC:  Possible during a scan?
     Conf: No interaction
  */
  void flashLED(unsigned int partition, unsigned int crate, unsigned int rod,
                long slaveNumber, long period = 1000, long flashes = 10);

  /// Echo data to a ROD
  /**
     IPC:  Possible during a scan?
     Conf: No interaction
  */
  void echo(unsigned int partition, unsigned int crate, unsigned int rod,
            unsigned int length, const unsigned long *data);

  /// Echo data to all RODs
  /**
     IPC:  Possible during a scan?
     Conf: No interaction
  */
  void echoAll(unsigned int length, const unsigned long *data);

  /// Echo data to a Slave in a ROD
  /**
     IPC:  Possible during a scan?
     Conf: No interaction
  */
  void echoSlave(unsigned int partition, unsigned int crate, unsigned int rod,
                 unsigned int slave, unsigned int length, const unsigned long *data);


  /***************** Direct ROD access (Debug only) ********************/
  /* Should be able to do anything with these commands */

  /// Dump a block of memory direct from a DSP
  /**
     Dump contents of a block of DSP memory to cout.

     @param dspNumber Slave number, -1 for the Master DSP.
     @param usePrim Use a primitive instead of direct VME access

     IPC:  Possible during a scan
     Conf: No interaction
  */
  int dspBlockDump(unsigned int partition, unsigned int crate, unsigned int rod,
                   long dspStart, long numWords, long dspNumber, bool usePrim = true);

  /// Dump a block of memory direct from a DSP to a file
  /**
     Dump contents of a block of DSP memory to a file.
     Use dspNumber -1 to refer to the master DSP.

     @param dspNumber Slave number, -1 for the Master DSP.
     @param usePrim Use a primitive instead of direct VME access

     IPC:  Possible during a scan
     Conf: No interaction
  */
  int dspBlockDumpFile(unsigned int partition, unsigned int crate, unsigned int rod,
                       long dspStart, long numWords, long dspNumber, std::string filename, bool usePrim = true);

  /// Perform a dump of the ROD with standard file names
  /**
     Dump to ${tempDir}/Dump_%05d

     @param rl Refers to the ROD to dump
   */
  void standardRegisterDump(RodLabel rl);

  /// Perform a dump of the ROD with standard file names
  /**
     Dump to ${tempDir}/Dump_%05d
   */
  void standardRegisterDump(unsigned int partition, unsigned int crate, unsigned int rod);

  /// Dump registers of all RODs to files
  /**
     Dump to ${tempDir}/Dump_%05d
     RODs sent to consecutive %d
   */
  void standardRegisterDumpAll();

  /// Read a block of memory direct from a DSP
  /**
     Use dspNumber -1 to refer to the master DSP.

     IPC:  Possible during a scan
     Conf: No interaction
  */
  unsigned long *dspBlockRead(unsigned int partition, unsigned int crate, unsigned int rod,
                              long dspStart, long numWords, long dspNumber, unsigned long &length, bool usePrim = true);

  /// Write a block direct to a DSP
  /**
     Use dspNumber -1 to refer to the master DSP.

     IPC:  Possible during a scan
     Conf: No interaction
  */
  int dspBlockWrite(unsigned int partition, unsigned int crate, unsigned int rod,
                    unsigned long *buffer, unsigned long dspAddress, long numWords, long dspNumber, bool usePrim = true);

  /// Read a block of slave DSP memory using RW_SLAVE_MEMORY
  /**
     Don't use if important to preserve primitive state!

     Returns new[]ed block of memory

     IPC:  Possible during a scan
     Conf: No interaction
  */
  unsigned long *primReadSlaveDsp(unsigned int partition, unsigned int crate, unsigned int rod,
                                  int s, int add, int words);

  /// Write a block of slave DSP memory using RW_SLAVE_MEMORY
  /**
     Don't use if important to preserve primitive state!

     IPC:  Possible during a scan
     Conf: No interaction
  */
  void primWriteSlaveDsp(unsigned int partition, unsigned int crate, unsigned int rod,
                         int s, int add, int numWords, unsigned long *data);

  void writeSlaveFile(unsigned int partition, unsigned int crate, unsigned int rod, unsigned int slave, 
                      std::string fileName, unsigned long address, bool usePrim = true);

  /// Send the start slave primitive to a DSP
  void startSlave(unsigned int partition, unsigned int crate, unsigned int rod, unsigned int slave);

  /// Get block of memory from file
  boost::shared_array<unsigned long> loadFile(std::string filename, unsigned long & length);

  /// Read one ROD Status Register via HPI
  /**
     IPC:  Immediate response
     Conf: No interaction
  */
  unsigned long readRodStatusReg(unsigned int partition, unsigned int crate, unsigned int rod,
                                 long regNumber);

  /// Read one ROD Command Register via HPI
  /**
     IPC:  Immediate response
     Conf: No interaction
  */
  unsigned long readRodCommandReg(unsigned int partition, unsigned int crate, unsigned int rod,
                                  long regNumber);

  /// Read a single 32b word from MasterDSP SDRAM via HPI
  /**
     IPC:  Immediate response
     Conf: No interaction
  */
  unsigned long dspSingleRead(unsigned int partition, unsigned int crate, unsigned int rod,
                              const unsigned long dspAddr, long dspNumber);

  /// Write a single 32b word to MasterDSP SDRAM via HPI
  /**
     IPC:  Immediate response (expert only)
     Conf: No interaction
  */
  void dspSingleWrite(unsigned int partition, unsigned int crate, unsigned int rod,
                      unsigned long dspAddr, unsigned long value, long dspNumber);

  /// Retrieve module configuration data
  /**
     Module data as cached from Configuration

     IPC:  Immediate response
     Conf: Interaction
  */
  ABCDModule *retrieveModule(UINT32 mid);

  /// Create a prim list
  /**
     IPC:  Immediate response (expert)
     Conf: No Interaction
  */
  void createDebugPrimList();

  /// Insert a primitive in a prim list
  /**
     IPC:  Immediate response (expert)
     Conf: No Interaction
  */
  void addDebugPrimList(unsigned long length, long index, long id, long version,
                        unsigned long * body);

  /// Send the previously created prim list
  /**
     IPC:  Immediate response (expert)
     Conf: No Interaction
  */
  void sendDebugPrimList(unsigned int partition, unsigned int crate, unsigned int rod);

  /// Send the previously created prim list to all RODs
  /**
     IPC:  Immediate response (expert)
     Conf: No Interaction
  */
  void sendDebugPrimListAll();

  /// Send the "Debug" primitive to a slave dsp
  /**
     Send the previously created prim list to a slave dsp
     @param await Master prim list waits for slave list to complete 
     @param response Expect a reply 

     IPC:  Immediate response (expert)
     Conf: No Interaction
  */
  void sendDebugSlavePrimList(unsigned int partition, unsigned int crate, unsigned int rod,
                              unsigned int slave, bool await, bool response);

  /// Load a binary dump of a primitive list into the debug prim list
  /**
     This doesn't check the primlist checksum (but slave lists will need the correct sum)
   */
  void debugPrimListFromFile(std::string fileName);

  /// Perform a binary dump of the debug prim list
  void dumpDebugPrimList();

  /// Return a list of the RODs that have been configured
  std::list<RodLabel> listRods();

  /**** Official SCTAPI bit ******/

  /// Return unique identifier for a module
  /**
     @param sn Module serial number
     @return The unique identifier, BAD_MODULE if not found

     IPC:  Immediate response
     Conf: Lookup
  */
  UINT32 findModule(std::string sn);

  /// Return std::string for uid
  /**
     IPC:  Immediate response
     Conf: Lookup
  */
  std::string convertToString(UINT32 mid);

  /// Return unique identifier for mur/module 
  /**
     IPC:  Immediate response
     Conf: Lookup
  */
  UINT32 findModule(INT32 mur, INT32 module);

  /// Return mur/module pair for uid
  /**
     IPC:  Immediate response
     Conf: Lookup
  */
  std::pair<INT32, INT32> convertToMUR(UINT32 mid);

  /// Return unique identifer from barrel geometry
  /**
     IPC:  Immediate response
     Conf: Lookup
  */
  UINT32 findBarrelModule(INT32 barrel, INT32 row, INT32 number);

  /// Convert uid to barrel geometry
  /**
     IPC:  Immediate response
     Conf: Lookup
  */
  void convertToBarrelModule(UINT32 mid, UINT32 &barrel, UINT32 &row, int &number);

  /// Return unique identifer from endcap geometry
  /**
     IPC:  Immediate response
     Conf: Lookup
  */
  UINT32 findEndcapModule(INT32 disk, INT32 quadrant, INT32 number);

  /// Convert uid to endcap geometry
  /**
     IPC:  Immediate response
     Conf: Lookup
  */
  void convertToEndcapModule(UINT32 mid, INT32 &disk, UINT32 &quadrant, UINT32 &number);

  /// Get all modules from ROD to memory
  /**
     IPC:  Immediate response
     Conf: Lookup
  */
  void getABCDModules(BankType bank);

  /// Get module from ROD to memory
  /**
     IPC:  Immediate response
     Conf: Lookup
  */
  virtual void getABCDModule(UINT32 mid, BankType bank);

  /// Get module configuration as stored in ROD, for debugging
  /**
     IPC:  Immediate response
     Conf: Lookup
  */
  virtual boost::shared_ptr<ABCDModule> getABCDModuleRaw(unsigned int partition, unsigned int crate, unsigned int rod, 
                                                         UINT32 slot, BankType bank);
  
  /// Set all module configs in a ROD bank
  /**
     IPC:  Don't do during scan
     Conf: Possible cache
  */
  virtual void setABCDModules(BankType bank);

  /// Set module config in ROD bank
  /**
     IPC:  Don't do during scan
     Conf: Possible cache
  */
  virtual void setABCDModule(UINT32 mid, BankType bank);

  /// Send configuration for all modules from the ROD bank to the module
  /**
     @param bank Configuration bank in the ROD
     @param type How much configuration to transmit

     DDC:  Check module state
           Set select line
     IPC:  Not during scan
     Conf: No Lookup
  */
  virtual void sendAllABCDModules(BankType bank, ConfigType type = SCTAPI_CONFIG_ALL);

  /// Send configuration for one module from the ROD bank to the module
  /**
     @param mid Unique identifier of the module to send
     @param bank Configuration bank in the ROD
     @param type How much configuration to transmit

     DDC:  Check module state
           Set select line
     IPC:  Not during scan
     Conf: No Lookup
  */
  virtual void sendABCDModule(UINT32 mid, BankType bank, ConfigType type = SCTAPI_CONFIG_ALL);

  /// Modify the mask of a module configuration
  /**
     Set the channel mask of all chips in a module

     IPC:  Not during scan
     Conf: No communication
  */
  void modifyABCDMask(UINT32 mid, UINT32* mask);

  /**
     Modify the mask of a list of module configuration
  */
/*    void modifyABCDListMask(mList* mlist, UINT32* mask); */

  /**
     Modify a module's trim configuration

     IPC:  Not during scan
     Conf: No communication
  */
  void modifyABCDTrims(UINT32 mid, UINT8* trims);

  /**
     Modify a module list's trim configuration
  */
/*    void modifyABCDTrims(mList* mlist, UINT8* trims); */

  /// Modify all module configurations
  /**
     IPC:  Not during scan
     Conf: No communication
  */
  void modifyABCDVar(UINT32 typ, FLOAT32 var);

  /// Modify variable in module configuration
  /**
     IPC:  Not during scan
     Conf: No communication
  */
  void modifyABCDVar(UINT32 mid, UINT32 typ, FLOAT32 var);

  /// Modify variable in list of modules
/*    void modifyABCDVar(mList* mlist, UINT32 typ, FLOAT32 var); */

  /// Modify variable on one chip of module configuration 
  /**
     IPC:  Not during scan
     Conf: No communication
  */
  void modifyABCDVar(UINT32 mid, UINT32 chip, UINT32 typ, FLOAT32 var);

  /// Modify variable on one chip of the module configuration on the ROD
  /**
     IPC:  Not during scan
     Conf: No communication
  */
  void modifyABCDVarROD(UINT32 mid, UINT32 chip, UINT32 typ, FLOAT32 var, BankType bank);

  /// Modify variable on one module configuration on the ROD
  /**
     IPC:  Not during scan
     Conf: No communication
  */
  void modifyABCDVarROD(UINT32 mid, UINT32 typ, FLOAT32 var, BankType bank);

  /// Modify variable of all module's configuration on the ROD
  /**
     IPC:  Not during scan
     Conf: No communication
  */
  void modifyABCDVarROD(UINT32 type, FLOAT32 value, BankType bank);

  /// Modify variable on one chip of module configurations in a list
/*    void modifyABCDVar(mList* mlist, UINT32 chip, UINT32 typ, FLOAT32 var); */

  /// Modify response curve setting on one chip
  /**
     IPC:  Not during scan
     Conf: No communication
  */
  void modifyABCDRC(UINT32 mid, UINT32 chip, UINT16 function, FLOAT32 p0, FLOAT32 p1, FLOAT32 p2);

  /// Modify BOC parameter
  /**
     IPC:  Not during scan
     Conf: No communication
  */
  void modifyBOCParam(unsigned int partition, unsigned int crate, unsigned int rod,
                      unsigned int channel, unsigned int type, unsigned int val);

  /// Modify all BOC parameters
  /**
     IPC:  Not during scan
     Conf: No communication
  */
  void modifyBOCParam(unsigned int type, unsigned int val);

  /// Print BOC parameters for all channels
  /**
     IPC:  Immediate response
     Conf: No communication
  */
  void printBOCSetup(unsigned int partition, unsigned int crate, unsigned int rod);

  /// Return the current BOC parameters for all channels
  /**
     IPC:  Immediate response (Needs results by IPC)
     Conf: No communication
  */
  std::vector<SctConfiguration::BOCChannelConfig> currentBOCSetup(unsigned int partition, unsigned int crate, unsigned int rod);

  /// Print BOC registers (independent of channels)
  /**
     IPC:  Immediate response
     Conf: No communication
  */
  void printBOCRegisters(unsigned int partition, unsigned int crate, unsigned int rod);

  /// Return the current BOC global registers
  /**
     IPC:  Immediate response (Needs results by IPC)
     Conf: No communication
  */
  SctConfiguration::BOCGlobalConfig currentBOCRegisters(unsigned int partition, unsigned int crate, unsigned int rod);

  /// Save BOC setup
  void saveBOCSetup(unsigned int partition, unsigned int crate, unsigned int rod);

  /// Save BOC registers
  void saveBOCRegisters(unsigned int partition, unsigned int crate, unsigned int rod);

  /// Response BOC setup
  void restoreBOCSetup(unsigned int partition, unsigned int crate, unsigned int rod);

  /// Restore BOC registers
  void restoreBOCRegisters(unsigned int partition, unsigned int crate, unsigned int rod);

  /// Turn all the BOC laser currents off (modules lose clock, not a good idea)
  /**
     IPC:  Immediate response
     Conf: No communication
  */
  void lasersOff();

  /* *************  TIM functs **************/

  /// Set TIM trigger frequency
  /**
     Freq is approximated by divider circuit
     Available frequencies are
        600, 60, 6, 0.6  kHz
     Divided by:
        1, 2, 3, 4, 5, 6, (10, 12)
   */
  void timSetFrequency(unsigned int partition, unsigned int crate, double trigFreq, double rstFreq);

  /// Start the TIM generating regular triggers 
  /**

     IPC:  Not during scan
     Conf: No communication
  */
  void freeTriggers(unsigned int partition, unsigned int crate);

  /// Stop the TIM generating regular triggers
  /**
     IPC:  Not during scan
     Conf: No communication
  */
  void stopTriggers(unsigned int partition, unsigned int crate);

  /// Tell the TIM to send an L1A 
  /**
     IPC:  Not during scan
     Conf: No communication
  */
  void timL1A(unsigned int partition, unsigned int crate);

  /// Tell TIM to send Cal + L1A
  /**
     IPC:  Not during scan
     Conf: No communication
  */
  void timCalL1A(unsigned int partition, unsigned int crate, int delay);

  /// Tell TIM to send soft Reset
  /**
     IPC:  Not during scan
     Conf: No communication
  */
  void timSoftReset(unsigned int partition, unsigned int crate);

  /// Tell TIM to send BC Reset
  /**
     IPC:  Not during scan
     Conf: No communication
  */
  void timBCReset(unsigned int partition, unsigned int crate);

  /// Send burst of triggers to TIM
  void sendTimBurst(unsigned int partition, unsigned int crate, unsigned int count);

  /// Print all TIM registers to stdout
  void timVerbose(unsigned int partition, unsigned int crate);

  /// Write a TIM register
  /**
     @param reg VME address of the register
   */
  void timWriteRegister(unsigned int partition, unsigned int crate, int reg, UINT16 val);

  /// Read a TIM register
  /**
     @param reg VME address of the register
   */
  UINT16 timReadRegister(unsigned int partition, unsigned int crate, int reg);

  /// Print status to cout
  /**
     IPC:  Immediate response (sane in all eventualities!)
     Conf: No communication
  */
  void status();

  /// Do a scan (hardcoded for testing)
  /**
     IPC:  Not during any other scan
     Conf: No communication
  */
  void defaultScan(int type);

  /// Coordinate a scan
  /**
     DDC:  Check modules configured...
           Check select
     IPC:  Not during any other scan
     Conf: Lookup
  */
  void doScan(boost::shared_ptr<Scan> scan);

  /**
     Await completion of a scan (alternative to checking IS running entry)
   */
  void awaitScan();

  /**
     Abort any long term activity

     Unimplemented

     DDC:  No action
     IPC:  Immediate response
     Conf: No communication
   */
  void abortScan();

  /// Scan returning histograms of raw data
  /**
     IPC:  Not during any other scan
     Conf: Lookup
  */
  void doRawScan(boost::shared_ptr<Scan> scan, int delay, int width, 
                 bool configureModules = false, bool clkBy2 = false);

  /// Send one trigger to all modules in a crate using the TIM
  /**
     IPC:  Not during any other scan
     Conf: No communication
  */
  void sendTimTrigger(unsigned int partition, unsigned int crate, const Trigger *trig);

  /// Send one trigger to modules in one ROD
  /**
     IPC:  Not during any other scan
     Conf: No communication
  */
  void sendTrigger(unsigned int partition, unsigned int crate, unsigned int rod, const Trigger *trig);

  /// Print the module configuration in memory
  /** 
     Uses the sctConf method

     IPC:  Not very useful without feedback
     Conf: Possible lookup
  */
  void printABCDModule(int mid);

  /// Print the module configuration from the ROD 
  /**
     First retrieve the configuration from the ROD
     Uses the sctConf method

     IPC:  Not very useful without feedback
     Conf: Possible lookup
   */
  void printABCDRodModule(int mid, BankType bank);

  /// Print decoded event
  /**
     IPC:  Needs feedback, immediate response
     Conf: No Communication
  */
  void decodeEvent(unsigned int partition, unsigned int crate, unsigned int rod, 
                   int sl, int index, bool extFlag = false, bool errorType = false);

  /// Scan event buffer
  /**
     IPC:  Needs feedback, immediate response
     Conf: No Communication
  */
  void scanEvents(unsigned int partition, unsigned int crate, unsigned int rod, int sl, bool extFlag, bool errorType);

  /**
     Suggest problems with modules based on events in event buffer
   */
  void reportEventErrors();

  /**
     Decode config bit stream (only valid after SEND_CONFIG
   */
  void decodeConfig(unsigned int partition, unsigned int crate, unsigned int rod, 
                    bool skipTrim = false, bool bypass = false);

  /// Read raw link data (send L1A to trigger some data)
  /**
     IPC:  Needs feedback, immediate response
     Conf: No Communication
  */
  void rawData(unsigned int partition, unsigned int crate, unsigned int rod, int delay, int units, bool setMask = true, const Trigger *trig = 0);

  /// Send an L1A to all modules
  /**
     The mask is unchanged in this function and it is set up to go to all modules

     IPC:  Not during scan
     Conf: No Communication
  */
  void sendL1A(unsigned int partition, unsigned int crate, unsigned int rod, bool capture = false);

  /// Make guess at what's attached to different channels
  /**
     Do quick scan of channels
     Recognises all the same, clock by 2 and an event header
     Return a char for each link examined
        0 All zero's
        1 All one's
        2 Clock by 2
        4 Clock by 4
        L L1A
        E Event Header
        S Soft Reset
        B BC Reset
        J Junk (ie anything else)

     IPC:  Needs feedback, some delay to response
     Conf: Some lookup?!
  */
  std::vector<char> probe(unsigned int partition, unsigned int crate, unsigned int rod, 
                          signed int harness = -1);

  /// Send a trigger to all channels and see what comes back
  /**
     Do quick scan of channels
     Recognises all the same, clock by 2 and an event header
     Return a char for each link examined
        0 All zero's
        1 All one's
        2 Clock by 2
        4 Clock by 4
        L L1A
        E Event Header
        S Soft Reset
        B BC Reset
        J Junk (ie anything else)

     IPC:  Needs feedback, some delay to response
     Conf: Some lookup?!
  */
  std::vector<char> probeWithTrigger(unsigned int partition, unsigned int crate, unsigned int rod, 
                                     const Trigger *trigger, signed int harness = -1);

  /// Scan over a variable and perform a probe for each
  std::vector<std::vector<char> > SctApi::probeScan(unsigned int partition, unsigned int crate, unsigned int rod, 
                                                    boost::shared_ptr<Scan> scan, signed int harness = -1);

  /**
     Probe all modules and check that value returned by probe 
     is one of the characters specified

     @param value Set of chars allowed as response to probe
   */
  bool checkAllModulesProbe(std::string value);

  /**
     Suggest values for configuration

     1) Perform probe scan of tx channels and interpret giving connections of modules to tx channels
   */
  void autoConfigure();

  /**
     Run the ROD "BOC histogram" primitive
   */
  void bocHistogram(unsigned int partition, unsigned int crate, unsigned int rod, 
                    unsigned int samples, unsigned int numLoops);

  /*** DCS functionality */

  /**
     Request hard reset on all modules

     DDC:  Primary purpose
     IPC:  Not during scan
     Conf: Some lookup?!
  */
  void requestHardResetAll();

  /**
     Request hard reset of module mid

     DDC:  Primary purpose
     IPC:  Not during scan
     Conf: Some lookup?!
   */
  void requestHardReset(UINT32 mid);

  /**
     Request initialisation of an IV scan

     DDC:  Primary purpose
     IPC:  Not during scan, result will be asynchronous
     Conf: Some lookup?!
   */
  void requestIVCurve(UINT32 mid, FLOAT32 start, FLOAT32 stop, FLOAT32 step, UINT16 delay, FLOAT32 currentLim);

  /**
     Request initialisation of an IV scan on all modules

     DDC:  Primary purpose
     IPC:  Not during scan, result will be asynchronous
     Conf: Some lookup?!
   */
  void requestIVCurveAll(FLOAT32 start, FLOAT32 stop, FLOAT32 step, UINT16 delay, FLOAT32 currentLim);

  /**
     Set the select value for all known power supply channels in a crate

     This is a debug method for DCS. Primary route to this functionality is 
     via modifyABCDVar and sendModule???

     DDC:  Primary purpose
     IPC:  Not during scan
     Conf: Some lookup?!
   */
  void setSelectAllInCrate(int crate, bool value, int timeout = 5);

  /**
     Set the select value for module mid

     This is a debug method for DCS. Primary route to this functionality is 
     via modifyABCDVar and sendModule!

     DDC:  Primary purpose
     IPC:  Not during scan
     Conf: Some lookup?!
   */
  // #warning "Should this just change DCS?"
  void setSelect(UINT32 mid, bool value); 

  /**
     Change run state of all power modules

     One of Off, Standby, On

     DDC:  Primary purpose
     IPC:  Not during scan
     Conf: No communication
  */
  void changeRunState(int state);

  /************ Direct DDC (no translations) ****************/

/*   /\** Get a channel parameter from DDC *\/ */
/*   void ddcGetChannelParameter(int crate, int channel, std::string dpeName, */
/*                               int& dpeValue, int timeout); */

  /// Change state on one channel 
  void ddcChangeState (int crate, int channel, short state, int timeout = 2);
  /// Change state on all power supply channels
  void ddcChangeStateAll (int crate, short state, int timeout = 2);
  /// Load power supply crate configuration for particular state
  void ddcLoadConfiguration (int crate, short state, int timeout = 2);
  /// Send hard reset to power supply channel
  void ddcHardReset (int crate, int channel, int timeout = 2);
  /// Send hard reset to all channels in power supply crate
  void ddcHardResetAll (int crate, int timeout = 2);

  /// Get a parameter that should be interpreted as an integer
  int ddcGetChannelParameterInt (int crate, int channel, std::string name, int timeout = 2);
  /// Get a parameter that should be interpreted as a float
  float ddcGetChannelParameterFloat (int crate, int channel, std::string name, int timeout = 2);
  /// Get a parameter that should be interpreted as a string (most likely an error)
  std::string ddcGetChannelParameterString (int crate, int channel, std::string name, int timeout = 2);
  /// Get a card parameter that should be interpreted as an integer
  int ddcGetCardParameterInt (int crate, int card, std::string name, int timeout = 2);
  /// Get a card parameter that should be interpreted as a float
  float ddcGetCardParameterFloat (int crate, int card, std::string name, int timeout = 2);
  /// Get a card parameter that should be interpreted as a string (probably an error)
  std::string ddcGetCardParameterString (int crate, int card, std::string name, int timeout = 2);
  /// Get a parameter of a power supply crate that should be interpreted as an integer
  int ddcGetCrateParameterInt (int crate, std::string name, int timeout = 2);
  /// Get a parameter of a power supply crate that should be interpreted as a float
  float ddcGetCrateParameterFloat (int crate, std::string name, int timeout = 2);
  /// Get a parameter of a power supply crate that should be interpreted as a string
  std::string ddcGetCrateParameterString (int crate, std::string name, int timeout = 2);

  /// Get some parameters of a power supply crate (interpreted as floats)
  /** 
      @param crate The crate id
      @param names A list of strings representing parameters
      @param dpeValues The return value, map parameter name strings to values
   */
  void ddcGetCrateParameters(int crate, std::vector<std::string> names, 
                             std::map<std::string, float>& dpeValues, int timeout);

  /// Set a parameter on a power supply channel
  void ddcSetChannelParameter (int crate, int channel, std::string name, float value, int timeout = 2);
  /// Set a parameter on a power supply card
  void ddcSetCardParameter (int crate, int card, std::string name, float value, int timeout = 2);
  /// Set a parameter on a power supply crate
  void ddcSetCrateParameter (int crate, std::string name, float value, int timeout = 2);



  /**
     Stop all tasks to do with histogramming

     This is a lower level function than abort,
     which returns to a know state

     DDC:  If changed?
     IPC:  Any time
     Conf: No communication
   */
  void tidyHistogramming();

  /**
     Return slot this ROD is in
   */
  long getRodSlot(unsigned int partition, unsigned int crate, unsigned int rod);

  /**
     Return slot this ROD is in
   */
  int getRodRevision(unsigned int partition, unsigned int crate, unsigned int rod);

  /**
     Return slot this ROD is in
   */
  int getRodRevision(const RodLabel &label);

  /**
     Select link to send to ROD front panel LEMO
  */
  void testLinkOutSelect(unsigned int partition, unsigned int crate, unsigned int rod,
                         unsigned int link);

  /**
     Return selected link to send to ROD front panel LEMO
  */
  unsigned int testLinkOutRetrieve(unsigned int partition, unsigned int crate, unsigned int rod);

  /**
     Disable a debug option

     DDC:  No action 
     IPC:  Class action
     Conf: No communication
  */
  void unsetDebugOption(std::string opt);

  /**
     Enable a debug option

     DDC:  No action 
     IPC:  Class action
     Conf: No communication
  */
  void setDebugOption(std::string opt);

  /**
     Get the currently enabled debug options

     DDC:  No action 
     IPC:  Class action, immediate response
     Conf: No communication
  */
  std::list<std::string> listEnabledDebugOptions();

  /**
     List possible debug options

     DDC:  No action 
     IPC:  Class action, immediate response
     Conf: No communication
  */
  std::vector<std::string> listDebugOptions();

  /**
     Perform a step in a histogram which is in a stalled state
   */
  void debugStepHistogram();

  /**
     Continue a histogram which was previously stalled

     This only works if the histogram is at the end of a bin
   */
  void debugContinueHistogram();

  /**
     Try to abort the current histogram

     This may not work. The DSP can be in a state where the abort signal is 
     delayed in its interpretation
   */
  void debugAbortHistogram();

  /**
     Setup event trapping

     A reduced set of parameters compared to the primitive is used.
   */
  void eventTrapSetup(int slaveMask, int trapMatch, int trapMod, int trapRemain, bool tim, bool error, 
                      boost::shared_ptr<PrimListWrapper> primList);

  /// Start event trap on a slave
  void startEventTrap(int slave, boost::shared_ptr<PrimListWrapper> primList);

  /// Stop event trap on a slave
  void stopEventTrap(int slave, boost::shared_ptr<PrimListWrapper> primList);

  /// Direct write of ROD mode!
  void rodMode(unsigned int partition, unsigned int crate, unsigned int rod,
               int mode, int flag, int fifoSetup, int nBins, int delay, int message);

  /// Set module masks
  void setupModuleMask(unsigned int partition, unsigned int crate, unsigned int rod,
                       int port, int slvs);

  /// Add set module masks command to primitive list
  void setupModuleMask(int port, int slvs, boost::shared_ptr<PrimListWrapper> list);

  /// Send a SEND_DATA primitive and retrieve the (pointer, length) result 
  std::pair<UINT32, UINT32> sendData(unsigned int partition, unsigned int crate, unsigned int rod, int type, int dsp);

  /**
     Resume the primitive polling thread
  */
  void resumePolling();

  /**
     Pause the primitive polling thread
  */
  void stopPolling();

  /// Print ROD calibration registers
  void print_calib(unsigned int partition, unsigned int crate, unsigned int rod);

  /// Send a primitive list to all the RODs in a crate
  /**
     A helper function for other methods in this class,
     to be used only internally to this class as
     it exposes the RodPrimList class
  */
  int synchSendPrimListAllCrates(boost::shared_ptr<PrimListWrapper> primList, int timeout = 10);

  /// Stop the histogramming and event trapping tasks
  void stopHistogramming(const ScanEx &ex);

  /// Set block of slave memory to value (using RW_SLAVE_MEMORY)
  void setSlaveBlock(unsigned int partition, unsigned int crate, unsigned int rod, 
                     int s, long sAdd, int words, long value = 0, bool usePrim = true);

  /// Write to a ROD register
  void writeRODRegister(unsigned int partition, unsigned int crate, unsigned int rod,
                        int reg, int off, int width, int value);

  /// Read a register from the ROD
  unsigned int readRODRegister(unsigned int partition, unsigned int crate, unsigned int rod, int r);

  /// Poll a register
  int pollRegister(unsigned int partition, unsigned int crate, unsigned int rod, int r, int off, int width, int val, int timeout = 1000000);

  virtual unsigned long *readFifo(unsigned int partition, unsigned int crate, unsigned int rod, int id, int bank, int elems);

  /// Create histogram task primitive
  void startHistogramTask(const Scan &scan, const ScanEx &ex, const RodLabel, 
                          unsigned int startBin, unsigned int nBins, unsigned int nTrigs,
                          boost::shared_ptr<PrimListWrapper> list);

  /// Send a primitive list to a ROD
  /**
     A helper function for other methods in this class,
     to be used only internally to this class as 
     it exposes the RodPrimList class
  */
  void sendPrimList(unsigned int partition, unsigned int crate, unsigned int rod,
                    boost::shared_ptr<PrimListWrapper> prim);

  MRSStream *mrs;
#if USE_IS
  ISInfoDictionary *m_isDict;
#endif
 protected:
  
  /********** Should these be part of the API? **************/

  ///Initialise a ROD from the configuration
  /**
      Proxy for Crate

      Return 0 for success
  */
  virtual int initialiseRod(unsigned int partition, unsigned int crate, unsigned int rod);

  /// Initialise a Module from the configuration.
  /**  
       Load module configuration from database
       Send module configuration to ROD

       Return 0 for success
  */
  virtual int initialiseModule(std::string module);

  /// Lookup module configuration in the cache
  ABCDModule *lookupConfig(UINT32 mid);

  /// Scan methods
  /**
    Perform necessary hardware checks before a scan
    @return true if successful, false otherwise
    */
  virtual bool preScanHardwareCheck(Scan& scan, ScanEx& extra);
  
  virtual void doHistogramSetup(const Scan &scan, const ScanEx &extra);
  
  /**
     IPC:  Needs feedback, immediate response
     Conf: No Communication
  */
  UINT32 *readEventBuffer(unsigned int partition, unsigned int crate, unsigned int rod, 
                        int sl, bool extFlag, bool errorType);

  /**
     Return the frame in which the last event starts
   */
  int getLastEventFrame(unsigned int partition, unsigned int crate, unsigned int rod, int slave);

  /// Poll the histogramming
  /**
     Wait for the number of bins to equal (bins - 1) on all master 
     dsps involved in the scan.

     Then wait for the number of events on the slave to equal to events

     Timeout if the bin count doesn't change, and if the number 
     of events doesn't change
  */
  virtual int pollHistogramming(boost::shared_ptr<ScanControl> controller, int timeout = 5);
 
  void scanLoop();

  void scanPollingThread();

  void addRodToList(RodLabel newRod);

  RodInfo &SctApi::getRodInfo(const RodLabel &l);

  /// List all the RODs
  std::list<RodLabel> rodList;

  /// List them again, but with additional info...
  std::map<RodLabel, RodInfo> rodInfoList;
  
  /// Cache of module configurations
  std::map<UINT32, ABCDModule> moduleMap;
  
  // Controller for current scan
  boost::shared_ptr<ScanControl> scanController;

  // Possibly to be used for continuing a scan
  boost::shared_ptr<ScanControl> lastScanController;

 private:

  /*************************************************************
   *          Internal functions (shouldn't be exported,       *
   *              probably able to keep them private )         *
   *************************************************************/

  /// Send a primitive list to a slave DSP
  /** 
      Send a RodPrimList to the Master DSP which contains the RodPrimList which is passed as an argument

      @param await If true master waits for completion of slave primlist
      @param response If true master expects reponse from slave
   */
  void sendSlavePrimList(unsigned int partition, unsigned int crate, unsigned int rod,
                         boost::shared_ptr<PrimListWrapper> prim, unsigned int slave, bool await, bool response);

  /// Send a primitive list to all the RODs in a crate
  /**
     A helper function for other methods in this class,
     to be used only internally to this class as 
     it exposes the RodPrimList class
  */
  void sendPrimListAll(unsigned int partition, unsigned int crate,
                       boost::shared_ptr<PrimListWrapper> prim);

  /// Setup event trapping according to scan and ex
  void setupEventTrapping(const Scan &scan, const ScanEx &ex, const RodLabel, boost::shared_ptr<PrimListWrapper> list);

  /// Start event trapping according to scan and ex
  void startEventTrapping(const Scan &scan, const ScanEx &ex, const RodLabel, boost::shared_ptr<PrimListWrapper> primList);

  /// Setup an event dump according to scan and ex
  void setupEventDump(const Scan &scan, const ScanEx &ex, const RodLabel, boost::shared_ptr<PrimListWrapper> list);

  /// Setup histogramming according to scan and ex
  void setupHistogramming(const Scan &scan, const ScanEx &ex, const RodLabel, bool tim, 
                          boost::shared_ptr<PrimListWrapper> list);

  /// Setup ROD for calibration
  void calib_init();

  /// Read raw data from all links into one "histogram"
  void readRawData(unsigned int partition, unsigned int crate, unsigned int rod, 
                   int delay, bool setMask);

  ///  Put raw data into FIFOs ready to read out
  virtual int setupRawData(unsigned int partition, unsigned int crate, unsigned int rod, 
			   int delay, int units, bool setMask, const Trigger *trig = 0);

  /// Utility function to get the Crate object from the map
  /** 
      Lookup the Crate object to send instructions to.
      Return 0 if not found.
   */
  Crate *getCrate(unsigned int partition, unsigned int crate) const;

  /// Put module mid into the cache (if present in the configuration)
  void cacheModuleConfig(UINT32 mid);

  /// Translate mid to pcrc of redundant connection
  /**
     XXX What does it do on failure!!!
   */
  void getrpcrc(UINT32 mid, unsigned int &rpartition, unsigned int &rcrate, unsigned int &rrod, unsigned &rchannel);

  /// I think I have to do this after ILU is running (otherwise SCTDCSCommand can't do the stuff it needs to initialise)
  void initDDC();

  /// Setup module, dsp masks etc
  void setupScanMasks(ScanEx &extra, int distSlave, bool dual);

  /// Return whether check passed
  bool checkModuleListsForScan();

  /// Setup modules before scanning
  /**
     Needs to know whether to set formatters into expanded mode
   */
  void preScanModuleSetup(Scan &scan);

  bool checkDebugOption(std::string opt);
  // This should be DebugOptions but that would mean including SctApiDebug.h here
  bool checkDebugOption(int opt);

  /*********** Internal variables *************/

  /// The configuration database
  boost::shared_ptr<SctConfiguration::Configuration> config;

  IPCPartition *ddcPartition;
  std::string *ddcCtrls[2];
  SCTDCSCommand *ddcCmd;

  boost::shared_ptr<DCSAccess> dcsAccess;

  /// The current debug primitive list
  boost::shared_ptr<PrimListWrapper> debugPrimList;

  /// Map to retrieve crate objects
  std::map<std::pair<unsigned int, unsigned int>, Crate* > crateMap;

  /// Log everything that is called
  Log log;

  /// For debug only, save the scan information
  boost::shared_ptr<const Scan> lastDebugScan;
  boost::shared_ptr<const ScanEx> lastDebugScanEx;

  /// Current scan number
  UINT32 scanNumber;

  /// Current run number
  UINT32 runNumber;

#if USE_SCAN_THREAD
  boost::condition scanQueue_notEmpty;
  boost::mutex scanQueue_mutex;

  /* ******* Threads ******/
  std::auto_ptr<boost::thread> scanPollThread;

  bool m_stopPolling;
  bool m_inScanLoop;
#endif

  /* **********  Mutexes  ********** */
  boost::mutex logMutex;
};
}  // End of namespace SctApi
#endif
