#ifndef SCTDATA_STAT_H
#define SCTDATA_STAT_H
#include <vector>

namespace SctData{

    /** 
	Represents an individual statistic. The templated class
	must be able to do T+=T, T/=unsigned, T*T, T-T
	and must have a default constructor T() which sets values to zero.
    */
    
template <class T>
class Stat {
public:
    Stat(T value, bool valid) : value(value), valid(valid) {}
    Stat() : valid(true) {;}
    T value;
    bool valid;
};

/// holds lots of Stat<T> objects, and has useful functions
template <class T>
class Stats : public vector<Stat<T> >{
public:
    /** default constructor */
    Stats<T>();
    /** default fill constructor */
    Stats<T>(const unsigned n);
    /** fill constructor */
    Stats<T>(const unsigned n, const T t);
    /** fill constructor */
    Stats<T> (const unsigned n, const Stat<T> s);
    /** default resize */
    void resize(const unsigned n);
    /** default resize */
    void resize(const unsigned n, const T t);
    /** clear */
    void clear();
    /** The mean value of the valid Stat s -- E(t) */
    T mean() const ;
    /** The variance value of the valid Stat s E(t**2) - E(t)**2 */
    T var() const ;
    /** The number of valid Stat s */
    unsigned n() const ;
    /** The sum of the valid Stat s */
    T sum() const ;
    /** The sum of the squares of valid Stat s */
    T sumSq() const ;
    /** The meam of the squares of valid Stat s */
    T meanSq() const ;
    /** const index operator */
    const Stat<T>& getAt(const unsigned i) const;
    /** non-const index operator */
    Stat<T>& modifyAt(const unsigned i);
 protected:
    // called if cache is modified.
    void modified(){ mean_calc=false; var_calc=false, n_calc=false;}
    // Cache
    mutable bool mean_calc;
    mutable bool var_calc;
    mutable bool n_calc;
    mutable T m_mean;
    mutable T m_var;
    mutable unsigned m_n;
private:
    const Stat<T> operator[](const unsigned i) const; ///< use getAt instead
    Stat<T>& operator[](const unsigned i); ///< use modifyAt() instead
};

//INLINES
    
    template <class T>
    inline Stats<T>::Stats<T>() {
	modified();
    }

    template <class T>
    inline Stats<T>::Stats<T>(const unsigned n) : vector<Stat<T> >(n, Stat<T>()) {
	modified();
    }

    template <class T>
    inline Stats<T>::Stats<T>(const unsigned n, const T t) : vector<Stat<T> >(n, Stat<T>(t, true)) {
	modified();
    }

    
    template <class T>
    inline Stats<T>::Stats<T>(const unsigned n, const Stat<T> s) : vector<Stat<T> >(n, s) {
	modified();
    }

    template <class T>
    inline void Stats<T>::clear() {
	vector<Stat<T> >::clear();
	modified();
    }

    template <class T>
    inline void Stats<T>::resize(const unsigned n) {
	if (n!=size()){
	    modified();
	    vector<Stat<T> >::resize(n, Stat<T>() );
	}
    }

    template <class T>
    inline void Stats<T>::resize(const unsigned n, const T t) {
	if (n!=size()){
	    modified();
	    vector<Stat<T> >::resize(n, t);
	}
    }

    template <class T>
    inline T Stats<T>::mean() const {
	if (!mean_calc){
	    m_mean=sum();
	    if (n()!=0) m_mean/=n();
	    mean_calc=true;
	}
	return m_mean;
    }
    
    template <class T>
    inline unsigned Stats<T>::n() const {
	if (!n_calc){
	    m_n=0;
	    for (unsigned i=0; i<size(); ++i){
		if (getAt(i).valid) ++m_n;
	    }
	    n_calc=true;
	}
	return m_n;
    }
    
    template <class T>
    inline T Stats<T>::sum() const  {
        T sum(0);
	for (unsigned i=0; i<size(); ++i){
	    if (getAt(i).valid) sum+=getAt(i).value;
	}
	return sum;
    }
    
    template <class T>
    inline T Stats<T>::sumSq() const  {
        T sumSq(0);
	for (unsigned i=0; i<size(); ++i){
	    if (getAt(i).valid) {
		T val=getAt(i).value;
		sumSq += val*val;
	    }
	}
	return sumSq;
    }

    template <class T>
    inline T Stats<T>::meanSq() const  {
	T meansq = sumSq();
	if (n()!=0) meansq/=n();
	return meansq;
    }


    template <class T>
    T Stats<T>::var() const  {
	if (!var_calc){
	    m_mean=mean();
	    m_var=meanSq()-m_mean*m_mean;
	    var_calc=true;
	}
	return m_var;
    }

    template <class T>
    const Stat<T>& Stats<T>::getAt(unsigned i) const{
	return vector<Stat<T> >::operator[](i);
    }

    template <class T>
    Stat<T>& Stats<T>::modifyAt(const unsigned i){
	modified();
	return vector<Stat<T> >::operator[](i);
    }

} //end of namespace SctData

#endif //#ifndef SCTDATA_STAT_H
