#ifndef MODULEELEMENT_H
#define MODULEELEMENT_H
#include "Sct/Exception.h"
#include "Sct/SctParameters.h"
#include "Sct/OutOfRangeError.h"

using namespace Sct;

namespace SctData {
    /**
     * Represents an contiguous element of a module : may be a chip, link or channel.
     * @author Alan Barr
     * @date 3 July 2003
     */
    class ModuleElement{
    public:
	//@{ @name Named Constructors:
	/**
	  Returns a ModuleElement representing a whole Module
	  */
	static ModuleElement Module() throw();	
	
	/**
	  Returns a ModuleElement representing Link link.
	  @throws OutOfRangeError if link>=nLinkModule
	  */
	static ModuleElement Link(unsigned int link);
	
	/**
	  Returns a ModuleElement representing Chip chip.
	  @throws OutOfRangeError if chip>=nChipModule
	  */
	static ModuleElement Chip(unsigned int chip);
	
	/**
	  Returns a ModuleElement representing Channel channel.
	  @throws OutOfRangeError if channel>nChannelModule
	  */
	static ModuleElement Channel(unsigned int channel);	
	//@}
	
	
	/** Constructor takes the first and last channel of the element.
	 *   0,127 for chip 0 link 0, or perhaps 768, 895 for chip 0 link 1.
	 * @param first channel contained @param last channel contained
	 * @throw OutOfRangeError if last<first or last>1535.
	 */
	ModuleElement(const unsigned first, const unsigned last);
	/// Destructor
	~ModuleElement() throw() {;}
	/// First channel contained (0->1535)
	unsigned getFirst() const throw() {return first;}
	/// Last channel contained (0->1535)
	unsigned getLast() const throw() {return last;}
	/// @return the number of channels (=last-first+1)
	unsigned getNChannels() const throw();
	/// @return true if only contains one element
	bool isChannel() const throw();
	/// @return true if the channels contained precisely match one chip
	bool isChip() const throw();
	/// @return true if the channels contained precisely match one link
	bool isLink() const throw();
	/// @return true if all of the channels are contained
	bool isModule() const throw();
	/// explicit equality operator @return true if first and last are equal
	bool operator == (const ModuleElement&) const throw();
	/// @return true if this element is contained within or equal to the argument
	bool isWithin(const ModuleElement&) const throw();
	/// @return true if this element contains or is equal to the RHS element
	bool contains(const ModuleElement&) const throw();
	/// @return true if this element overlaps any part of RHS element
	bool overlaps(const ModuleElement&) const throw();
	/// @return true if this element is logically next to RHS element
	/// @note includes `adjacent' across chip and even link boundries.
	/// @note overlapping elements are <B>not</B> adjacent.
	bool adjacentTo(const ModuleElement&) const throw();
    private:
	unsigned first, last;
	ModuleElement() throw(); ///<can't do this
    };

//==========================================================================
// inlines
//==========================================================================
    inline ModuleElement::ModuleElement(const unsigned lo, const unsigned hi) {
#ifndef NDEBUG
	if (hi>nChannelModule ) {
	    throw OutOfRangeError<unsigned>("ModuleElement last", __FILE__, __LINE__, hi, 0, nChannelModule);
	}
	if (lo>hi ){
	    throw OutOfRangeError<unsigned>("ModuleElement first ", __FILE__, __LINE__, lo, 0, hi);
	}
#endif
	last=hi;
	first=lo;

    }
    inline unsigned int ModuleElement::getNChannels() const throw() {
	return getLast() - getFirst() + 1;
    }
    inline bool ModuleElement::isChannel() const throw(){
	return (getNChannels() == 1);
    }
    inline bool ModuleElement::isChip() const throw(){
	return (getFirst() % nChannelChip == 0 && getNChannels() == nChannelChip);
    }
    inline bool ModuleElement::isLink() const throw(){
	return (getFirst() % nChannelLink == 0 && getNChannels() == nChannelLink);
    }
     inline bool ModuleElement::isModule() const throw(){
	return (getFirst() % nChannelModule == 0 && getNChannels() == nChannelModule);
    }

    inline bool ModuleElement::operator == (const ModuleElement& el) const throw(){
	return (getFirst()==el.getFirst() && getLast()==el.getLast() );
    }

    inline bool ModuleElement::isWithin(const ModuleElement& el) const throw(){
	return (getFirst()>=el.getFirst() && getLast()<=el.getLast() );
    }
    inline bool ModuleElement::contains(const ModuleElement& el) const throw(){
	return (getFirst()<=el.getFirst() && getLast()>=el.getLast() );
    }
    
    inline bool ModuleElement::overlaps(const ModuleElement& el) const throw(){
	return ( (getFirst()<=el.getFirst() && getLast()>=el.getFirst() )  // my first element contained
		 || ( getFirst()>el.getFirst() && getFirst()<=el.getLast() ) ); // my last element contained
    }
    
    inline bool ModuleElement::adjacentTo(const ModuleElement& el) const throw(){
      if ( overlaps(el) ) return false;
      return ( getFirst()==el.getLast()+1 || getLast()==el.getFirst()-1 );
    }

    inline ModuleElement ModuleElement::Chip(const unsigned ichip) { 
	return ModuleElement(ichip*nChannelChip, (ichip+1)*nChannelChip-1);
    }
    
    inline ModuleElement ModuleElement::Link(const unsigned ilink) { 
	return ModuleElement(ilink*nChannelLink, (ilink+1)*nChannelLink-1);
    }
    
    inline ModuleElement ModuleElement::Channel(const unsigned ichannel) { 
	return ModuleElement(ichannel, ichannel);
    }
    
    inline ModuleElement ModuleElement::Module() throw() { 
	return ModuleElement(0, nChannelModule-1);
    }

} // end of namespace SctData
#endif //#ifndef MODULEELEMENT_H
