package SctData;

import Sct.*;
import sctConf.*;

/**
 * Represents the configuration of a module. Just a placeholder for now, but will probably be an interface to ABCDModule.
 * It is worth noting that this class and its components define access to the data in terms of the physical setting e.g. DAC
 * settings.  If a user wants to think of a variable in terms of a higher level quantity, such as charge, mV etc
 * then he should use an appropriate strategy for performing the conversion.
 * This is motivated by the realisation that typically DAC step sizes are only approximate
 * so expressing the setting in terms of e.g. mV is just as meaningless as expressing it in terms of DAC steps.  However, we
 * have decided to stick with functions in terms of nominal values for now.
 * @author Matthew Palmer
 */
public class ModuleConfiguration implements Streamable {
 
    public ModuleConfiguration() {
        data = new ABCDModule();
        data.chip = new ABCDChip[Parameters.nChipModule];
        for (int i=0; i<data.chip.length; ++i) {
            data.chip[i] = new ABCDChip();
            data.chip[i].basic = new ABCDBasic();
            data.chip[i].caldata = new ABCDCaldata();
            data.chip[i].trim = new byte[Parameters.nChannelChip];
            data.chip[i].caldata.rc_params = new float[3];
            data.chip[i].basic.config = new ABCDConfig();
            data.chip[i].basic.mask = new int[4];            
        }
        data.rx = new byte[2];
        
        for (int i=0; i<chips.length; ++i) {
            chips[i] = new ChipConfiguration(data.chip[i]);
        }
    }
    
    public ModuleConfiguration(ABCDModule data) {
        this.data = data;
        for (int i=0; i<chips.length; ++i) {
            chips[i] = new ChipConfiguration(data.chip[i]);
        }
    }
    
    public ABCDModule getData() {
        return data;
    }
    
    public ChipConfiguration getChipConfiguration(int chip) {
        return chips[chip];
    }

    /**
     * presence means module configurations are present in the
     * ROD and therefore sent triggers and read out by the histogramming code
     */
    public boolean isPresent() {
        return data.present!=0;
    }

    /** active means a module "takes part in a scan" ie has its configuration modified by the histogramming engine */
    public boolean isActive() {
        return data.active != 0;
    }

    public void setActive(boolean active) {
        if (active) data.active = 1;
        else data.active = 0;
    }

    public boolean usingPrimaryClock() {
        return data.select != 0;
    }

    public void setUsingPrimaryClock(boolean usePrimaryClock) {
        data.select = usePrimaryClock ? (byte)1 : (byte)0;
    }

    public boolean channelIsMasked(int ichannel) {
        return getChipConfiguration(ichannel/Parameters.nChannelChip).isMasked(ichannel%Parameters.nChannelChip);
    }

    public void maskChannel(int ichannel) {
        getChipConfiguration(ichannel/Parameters.nChannelChip).mask(ichannel%Parameters.nChannelChip);
    }

    /**
     * Group ID This is used to indicate which slave DSP will receive the module's data (if group
     * based distribution is set), and also to allow different module groups to be triggered independently (for
     * cross-talk studies). valid range: [0,3]
     */
    public byte getGroupId() {
        return data.groupId;
    }

    public void setGroupId(byte param) {
        data.groupId = param;
    }

    public String getClassName() {
        return "SctData.ModuleConfiguration";
    }    
    
    public static Streamable read(IStream s, ObjectManager o) throws java.io.IOException {
        ModuleConfiguration mc = new ModuleConfiguration();
        for (int i=0; i<mc.chips.length; ++i) {
            mc.chips[i] = (ChipConfiguration)o.readObject(s, "ChipConfiguration", "SctData.ChipConfiguration");
            mc.data.chip[i] = mc.chips[i].getData();
        }
        mc.data.present = s.readByte("Present");
        mc.data.active = s.readByte("Active");
        mc.data.select = s.readByte("Select");
        mc.data.groupId = s.readByte("GroupId");
        mc.data.pTTC = s.readByte("pTTC");
        mc.data.rTTC = s.readByte("rTTC");
        mc.data.rx = s.readByteArray("rx");
        
        return mc;
    }    
    
    public void write(OStream s, ObjectManager o) throws java.io.IOException {
        for (int i=0; i<chips.length; ++i) {
            o.writeObject(s, "ChipConfiguration", chips[i], false);
        }
        s.writeByte("Present", data.present, false);
        s.writeByte("Active", data.active, false);
        s.writeByte("Select", data.select, false);
        s.writeByte("GroupId", data.groupId, false);
        s.writeByte("pTTC", data.pTTC, false);
        s.writeByte("rTTC", data.rTTC, false);
        s.writeByteArray("rx", data.rx, false);
    }
    
    //Data
    private ABCDModule data;
    private ChipConfiguration[] chips = new ChipConfiguration[Parameters.nChipModule];
}
