#ifndef SCT_ISSTREAMERWRAPPER_H
#define SCT_ISSTREAMERWRAPPER_H
#include "Streamer.h"
#include "LogicErrors.h"
#include "IoExceptions.h"
#include "ISOStreamWrapper.h"
#include "ISIStreamWrapper.h"
#include "ISSerializableWrapper.h"
#include "is/isinfo.h"

namespace Sct{
class OStream;
class IStream;
class Streamable;
class IOManager;
/** 
    Templated class for streaming objects derriving from the oniline isinfo 
    with the various Sct::IOManager classes.
    The templated class should be an isinfo with a public 
    unsigned data member `version'.
    Allows one to write isinfo to other IOManager back-ends.
    To be used in conjuction with ISSerializableWrapper class.
    Overrides read and write to use the isinfo publishGuts 
    and refreshGuts methods.
    The templated class must have a member called `version', which 
    is returned by the getVersion() method.
    Streamer version control for this class is via this isinfo version member.
    Used for example in Archiving IS objects, such as TestData and 
    SequenceData.
    @todo find a way of doing auto-addition to IOManager map
*/
template<class T>
class ISStreamerWrapper : public virtual Streamer {
 public:
  /** 
      write method uses publishGuts()
  */
  virtual void write(OStream& out, const Streamable& ob, const IOManager& manager) const;

  /** 
      Read method uses refreshGuts() and returns ISSerializableWrapper<T> 
  */
  virtual shared_ptr<Streamable> read(IStream& in, const IOManager& manager) const;

  /**  
       Read method uses refreshGuts() and takes as second arg ISSerializableWrapper<T>
  */
  virtual void read(IStream& in, Streamable& ob, const IOManager& manager) const;

  /** 
      constructor 
  */
  ISStreamerWrapper() {; }

  /** 
      final method - version control is via templated class `version' data member */
  virtual unsigned getVersion() const throw();
 protected:
 /**
     Small class inherits from T since publishGuts 
     and refreshGuts are protected by is_generator.sh
  */
  class InnerClass : public ISInfo {
  public:
    /** constructor */
    InnerClass(ISInfo& ob) : m_ob(ob) {}
    /** make method public */
    virtual void publishGuts( ISostream & out ) {m_ob.publishGuts(out);}
    /** make method public */
    virtual void refreshGuts( ISistream & in ) {m_ob.refreshGuts(in);}
  private:
    InnerClass();
    ISInfo& m_ob;
  };
};

/// INLINES

template<class T>
inline unsigned ISStreamerWrapper<T>::getVersion() const throw() {
  T t;
  return t.version;
}

template<class T>
inline void ISStreamerWrapper<T>::write(OStream& out, const Streamable& ob, const IOManager& manager) const {
  // wrap the stream
  //#warning probable slow memory leak forced by using Sct/isstream_bugfix.h
  ISOStreamWrapper isosw(out);

  // cast the Streamable as a ISSerializableWrapper
  const ISSerializableWrapper<T>& wrapper = 
    dynamic_cast <const ISSerializableWrapper<T>&> ( ob );

  // use the inner class to make the publishGuts method public:
  InnerClass inner( const_cast<T&> ( *wrapper.getWrappedObject()) );

  // use the publishGuts method to stream.
  inner.publishGuts(isosw);
}

template<class T> 
 inline void ISStreamerWrapper<T>::read(IStream& in, Sct::Streamable& ob, const IOManager& manager) const{
    // cast the Streamable as a ISSerializableWrapper
  const ISSerializableWrapper<T>& wrapper = 
    dynamic_cast <const ISSerializableWrapper<T>&> ( ob );

   // make an InnerClass to make the refreshGuts public:
  InnerClass inner( const_cast<T&> (*wrapper.getWrappedObject()) );

  //#warning forced slow memory leak below. The ISIStreamWrapper destructor has problems because we have kludged the ISistream in file isstream_bugfix.h
   // convert the stream
  ISIStreamWrapper isisw(in);
  
   // use the refreshGuts to update.
  inner.refreshGuts(isisw);
  wrapper.constructUniqueID();
 }

template<class T>
inline shared_ptr<Streamable> ISStreamerWrapper<T>::read(IStream& in, const IOManager& manager) const {
  // wrap it as a serializable
  shared_ptr<ISSerializableWrapper<T> > ob = shared_ptr<ISSerializableWrapper<T> >( new ISSerializableWrapper<T>() ) ;
  read(in, *ob, manager);
  return ob;
}

}

#endif //SCT_ISSTREAMERWRAPPER_H
