#ifndef SCT_ISSERIALIZABLEWRAPPER_H
#define SCT_ISSERIALIZABLEWRAPPER_H
#include "Serializable.h"
#include "SctNames.h"
#include "is/isinfo.h"
#include <boost/shared_ptr.hpp>

using boost::shared_ptr;

namespace Sct{
  /** interface for templated class ISSerializableWrapper */
  class ISSerializableWrapperI : public virtual Serializable {
  public:
    /** publish the wrapped object direct to the ISServer of this name */
    virtual void publishWrappedObjectToServer(const string& isServerName)const =0;
  };
  
  /** 
      wraps a isinfo object (class as a template) as a Serializable
      allowing it to be read and written using IOManager classes.
      and getUniqueID in the usual way for a Serializable.
      Class is templated so that ISStreamerWrapper can make a new
      isinfo of the correct type on read.
      The templated class must have a public `version' member.
      @todo is there a better way of getting the unique ID?
  */
  template<class T> class ISSerializableWrapper : public ISSerializableWrapperI {
  public:
    /** default constructor */
    ISSerializableWrapper() : ob(new T), m_uniqueID("unknown") {
//      std::cout << "SerWrap Default constructor " << std::endl;
    }
    /** 
	constructor with shared_ptr to ISInfo to wrap
	@param shared_ptr to the ISInfo
	@param unique id of this ISInfo
     */
    ISSerializableWrapper(shared_ptr<T> ob) : ob(ob) {
//        std::cout << "SerWrap Non Default Contstructor" << std::endl;
      	    constructUniqueID(); 
    }
      /** destructor */
    virtual ~ISSerializableWrapper() { /*std::cout << "~SerWrap" << std::endl;*/}
    /** to be implimented by concrete class */
    virtual string getClassName() const;
    /** to be implimented by concrete class */
    virtual string getUniqueID() const; 
    /** access wrapped isinfo ob */
    shared_ptr<T> getWrappedObject() {return ob;}
    /** const access to wrapped isinfo ob */
    shared_ptr<const T> getWrappedObject() const {return ob;}
    /** publish wrapped object to IS directly, with server name
        @param the name of the server to publish to.
        @throw ISException if there is a problem with the publishing.
    */
    virtual void publishWrappedObjectToServer(const string& isServerName) const;
    /** To be implimented by user code. Use to set m_uniqueID from object of type T */
    void constructUniqueID() const;
  private:
    shared_ptr<T> ob; ///< shared pointer to the wrapped ISInfo 
    mutable string m_uniqueID; ///< unique ID of the ISInfo
  };
  /// INLINE FUNCTIONS
  template<class T>
  inline string ISSerializableWrapper<T>::getClassName() const {
    return ob->type().name();
  }
  
  template<class T>
  inline string ISSerializableWrapper<T>::getUniqueID() const {
    return m_uniqueID;
  }

  template<class T>
    inline void ISSerializableWrapper<T>::publishWrappedObjectToServer(const string& serverName) const {
//    std::cout << "publish wrapped" << std::endl;
    constructUniqueID();
    ISInfoDictionary& is = SctNames::getISDictionary();
    string name=serverName;
    name+=".";
    name+=getClassName();
    name+=".";
    name+=getUniqueID();
//    std::cout << getUniqueID() << ", " << getClassName() << std::endl;
    ISInfo::Status s = is.insert(name.c_str(), const_cast<T&>(*getWrappedObject()) );
    if (s != ISInfo::Success) throw IsException(s, name + " publish failed", __FILE__, __LINE__);
  }

}

#endif //SCT_ISSERIALIZABLEWRAPPER_H
