package GuiComponents.Inspector;

import java.util.*;
import java.lang.reflect.*;
import Sct.*;

interface Rep {
    public String getName();
    public String getClassName();
    public String getValue();
}

class PrimitiveRep implements Rep {
    protected String name;
    protected String typeName;
    protected Object value;
    
    PrimitiveRep(String name, Object value) {
        this.name = name;
        this.value = value;
        typeName = value.getClass().getName();
    }
    
    public String toString() {
        return getName();
    }
    
    public String getName() {
        return name;
    }
    
    public String getClassName() {
        return typeName;
    }
    
    public String getValue() {
        return value.toString();
    }
}

class NullRep extends PrimitiveRep {
   NullRep(String name) {
        super(name, "null");
        typeName = "Unknown";
   }
}

class ArrayRep extends PrimitiveRep {
   ArrayRep(String name, Object value) {
        super(name, value);
        typeName = value.getClass().getComponentType().getName() + "[" + Array.getLength(value) + "]";        
   }
   
   
   public String getValue() {
       int count = 10;
       if (Array.getLength(value) < 10) count = Array.getLength(value);
       String s = "";
       for (int i=0; i<count; ++i) {
           if(getClassName().startsWith("int")) { 
               Integer intVal = (Integer)Array.get(value, i);
               String hexString = "0x" + Integer.toHexString(intVal.intValue());
               s += hexString;
           } else {
               s += Array.get(value, i);
           }

           if(i<count-1) s += ", ";
       }
       return s;
   }
}

class StreamableRep implements Rep, Sct.OStream, Sct.ObjectManager {
    protected ArrayList streamables = new ArrayList();
    protected ArrayList primitives = new ArrayList();
    protected String className;
    protected String name;
    private Streamable object;
    private boolean written = false;    ///Used to implement lazy writing
    
    StreamableRep(String name, Serializable ob) {
        className = ob.getClassName();
        object = ob;
        this.name = name;    
        writeObject(ob);
        written = true;
    }
    
    StreamableRep(String name, Streamable ob) {        
        className = ob.getClassName();
        object = ob;
        this.name = name;
        
        if (ob instanceof Serializable) {
            this.name += " (" + ob.getClassName() + "." + ((Serializable)ob).getUniqueID() + ")";
        }
    }   
    
    public String toString() {
        return getName();
    }
    
    public String getName() {
        return name;
    }
    
    public String getClassName() {
        return className;
    }
    
    public String getValue() {
        return "";
    }
    
    int getChildCount() {
        if (!written) writeChildren();
        return streamables.size() + primitives.size();
    }
    
    Rep getChild(int index) {
        if (index >= streamables.size()) {
            return (Rep)primitives.get(index - streamables.size());
        } else {
            return (Rep)streamables.get(index);
        }
    }
    
    private void writeChildren() {
        if (written) return;
        try {
            object.write(this, this);
            written = true;
        } catch (java.io.IOException ioe) {
            throw new RuntimeException(ioe);        
        }        
    }
    
    //Import ObjectManager stuff
        
    //Break return contract
    public Name writeObject(Serializable object) {
        try {
            object.write(this, this);
        } catch (java.io.IOException ioe) {
            throw new RuntimeException(ioe);
        }
        return null;
    }
    
    public void writeObject(OStream s, String name, Streamable object, boolean writeClassName) {
        if (object == null) primitives.add(new NullRep(name));
        else {
            StreamableRep srep = new StreamableRep(name, object);
            streamables.add(srep);        
        }
    }
    

    //OStream stuff
    public void writeBoolean(String name, boolean b) {
        primitives.add(new PrimitiveRep(name, new Boolean(b)));
    }
    
    public void writeBooleanArray(String name, boolean[] b) {
        primitives.add(new ArrayRep(name, b));
    }
    
    public void writeByte(String name, byte b, boolean sign) {
        primitives.add(new PrimitiveRep(name, new Byte(b)));
    }
    
    public void writeByteArray(String name, byte[] b, boolean sign) {
        primitives.add(new ArrayRep(name, b));
    }
    
    public void writeDouble(String name, double d) {
        primitives.add(new PrimitiveRep(name, new Double(d)));
    }
    
    public void writeDoubleArray(String name, double[] d) {
        primitives.add(new ArrayRep(name, d));
    }
    
    public void writeFloat(String name, float f) {
        primitives.add(new PrimitiveRep(name, new Float(f)));
    }
    
    public void writeFloatArray(String name, float[] f) {
        primitives.add(new ArrayRep(name, f));
    }
    
    public void writeInt(String name, int i, boolean sign) {
        primitives.add(new PrimitiveRep(name, new Integer(i)));
    }
    
    public void writeIntArray(String name, int[] i, boolean sign) {
        primitives.add(new ArrayRep(name, i));
    }
    
    public void writeShort(String name, short s, boolean sign) {
        primitives.add(new PrimitiveRep(name, new Short(s)));
    }
    
    public void writeShortArray(String name, short[] s, boolean sign) {
        primitives.add(new ArrayRep(name, s));
    }
    
    public void writeString(String name, String s)  {
        primitives.add(new PrimitiveRep(name, s));
    }
    
    public void writeStringArray(String name, String[] s) {
        primitives.add(new ArrayRep(name, s));
    }      
    
    public Sct.Serializable readObject(String name) throws UnsupportedOperationException{
        throw new UnsupportedOperationException("TestGUI.Inspector.ObjectManager only supports write operations");
    }
    
    public Sct.Streamable readObject(IStream s, String name) throws UnsupportedOperationException {
        throw new UnsupportedOperationException("TestGUI.Inspector.ObjectManager only supports write operations");
    }
    
    public Sct.Serializable readObject(String uniqueID, String className) throws UnsupportedOperationException {
        throw new UnsupportedOperationException("TestGUI.Inspector.ObjectManager only supports write operations");
    }
    
    public Sct.Streamable readObject(IStream s, String name, String className) throws UnsupportedOperationException {
        throw new UnsupportedOperationException("TestGUI.Inspector.ObjectManager only supports write operations");
    }
    
    public void writeLong(String name, long i, boolean sign) throws java.io.IOException {
    }
    
    public void writeLongArray(String name, long[] i, boolean sign) throws java.io.IOException {
    }
    
}


