package GuiComponents.Console;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;
import java.util.*;
import java.awt.Font;
import java.awt.Insets;
import java.awt.Component;
import java.awt.event.*;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeEvent;
import java.io.*;

/**
 * This is a simple GUI console.  It generates ActionEvents whenever the user presses the Enter key
 * and TextEvents when the name completion key is pressed.
 * It has a history, a prompt and that's about it.  More complex things should be done by 
 * whatever understands the input!  It can also print text in lots of different ways and icons.  
 * It is based very loosely on the JConsole class in the BeanShell scripting framework, but is 
 * extremely modified.
 * @author  Matthew Palmer
 * @date 9/12/02
 */
public class JConsole extends JEditorPane 
implements KeyListener, ActionListener, PropertyChangeListener, MouseListener, CaretListener {
    
    //Variables    
    protected JPopupMenu mPopup;                    ///<The popup menu
    protected String sPrompt;                       ///<The prompt
    protected String sPostPrompt;                   ///<A String that is printed after the prompt, but can be edited by the user.
    protected char cEnterChar= '\n';                ///<The char that signifies "enter"
    private int lineStartPos = 0;                   ///<The place where the current command entry line starts
    private int startPos = 0;                       ///<The place where the current command starts
    private int currPos = 0;                        ///<The current text entry point
    protected Vector vListeners = new Vector();     ///<The action listeners
    protected Vector nameListeners = new Vector();  ///<The name completion listeners
    protected Vector vHistory = new Vector();       ///<Holds the history
    private int historyPos = 0;                     ///<Current history position
    private String curCmd;                          ///<Holds the current command
    protected AbstractDocument doc;            ///<The Document for the output
    
    //Constants
    private final static String CUT = "Cut";
    private final static String COPY = "Copy";
    private final static String PASTE = "Paste";
    
    /** Creates a new instance of JConsole */
    public JConsole() {
        initComponents();
        requestFocus();        
    }
    
    protected void initComponents() {
        //A slightly modified document to prevent deletions before the startPos
        //This is mostly to overcome the backspace not consumed issue.
        doc = new PlainDocument() {
            public void remove(int offs, int len) throws BadLocationException {                
                if (offs>=startPos) super.remove(offs, len);
            }
        };
        doc.setAsynchronousLoadPriority(1);

        setDocument(doc);
        Font font = new Font("Monospaced",Font.PLAIN,14);   
        setText("");
        setFont( font );
        setMargin( new Insets(7,5,7,5) );
        addKeyListener(this);                
        setDoubleBuffered(true);

        // create popup menu
        mPopup = new JPopupMenu("JConsole Menu");
        mPopup.add(new JMenuItem(CUT)).addActionListener(this);
        mPopup.add(new JMenuItem(COPY)).addActionListener(this);
        mPopup.add(new JMenuItem(PASTE)).addActionListener(this);

        addMouseListener(this);
        addCaretListener(this);

        // make sure popup menu follows Look & Feel
        UIManager.addPropertyChangeListener(this);
    }
    
    public void initialize() {
        resetCommandStart();
    }
    
    public InputStream getInputStream() {
        return new ActionStream(this);
    }
    
    public void cut() {
        if (getCaretPosition() < startPos) {
            super.copy();
        } else {
            super.cut();                    
        }
    }

    public void paste() {
        //System.out.println("paste: " + getCaretPosition());        
        ensureCaretPosition();
        super.paste();
    }

    public void setCaretPosition(int position) {
        //System.out.println("setCaretPosition to : " + position + " from "+ getCaretPosition() + "  " + lineStartPos + "  " + startPos + "  " + currPos + "  " + doc.getLength());
        if (position < startPos) return;
        super.setCaretPosition(position);
    }

    //
    //Listeners
    //
    
    
    /** Invoked when an action occurs.
     *
     */
    public void actionPerformed(ActionEvent e) {
        String cmd = e.getActionCommand();
        if (cmd.equals(CUT)) {
            cut();
        } else if (cmd.equals(COPY)) {
            copy();
        } else if (cmd.equals(PASTE)) {
            paste();
        }
    }
    
    /** Invoked when a key has been pressed.
     * See the class description for {@link KeyEvent} for a definition of
     * a key pressed event.
     *
     */
    public void keyPressed(KeyEvent e) {
        switch ( e.getKeyCode() ) {       
            case ( KeyEvent.VK_UP ):
                if (e.getID() == KeyEvent.KEY_PRESSED) {
                    historyUp();
                }
                e.consume();
                break;

            case ( KeyEvent.VK_DOWN ):
                if (e.getID() == KeyEvent.KEY_PRESSED) {
                    historyDown();
                }
                e.consume();
                break;
                
            //Prevent moving before the prompt        
            case ( KeyEvent.VK_LEFT ):
            case ( KeyEvent.VK_BACK_SPACE ):
            case ( KeyEvent.VK_DELETE ):                
                if (getCaretPosition() <= startPos) e.consume();
                break;
                
            case (KeyEvent.VK_END):
                setCaretPosition(doc.getLength());
                e.consume();
                break;
                
            case ( KeyEvent.VK_HOME ):
                setCaretPosition(startPos);
                e.consume();
                break;
                
            case (KeyEvent.VK_TAB):
            case (KeyEvent.VK_ENTER):
                e.consume();
                break;

        }
       
    }
    
    
    /** Invoked when a key has been released.
     * See the class description for {@link KeyEvent} for a definition of
     * a key released event.
     *
     */
    public void keyReleased(KeyEvent e) {
    }
    
    /** Invoked when a key has been typed.
     * See the class description for {@link KeyEvent} for a definition of
     * a key typed event.
     *
     */
    public void keyTyped(KeyEvent e) {        
        if (e.getKeyChar() == cEnterChar) {
            enter();
            e.consume();
        }
        
        if (e.getKeyChar() == '\t') {
            nameCompletionEvent();
            e.consume();
        }
        //Default behaviour:
        ensureCaretPosition();
    }
    
    /** 
     * Invoked when the caret changes position
     */
    public void caretUpdate(CaretEvent e) {
        //System.out.println("Caret: " + e.getDot() + "  " + e.getMark());
        if (e.getDot() == e.getMark()) {
            if (e.getDot() < startPos) {
                setCaretPosition(currPos);
            } else {
                //System.out.println("Caret update " + lineStartPos + "  " + startPos + "  " + currPos + "  " + doc.getLength());
                currPos = e.getDot();
                //System.out.println("Caret update " + lineStartPos + "  " + startPos + "  " + currPos + "  " + doc.getLength());
            }
        }
    }
    
    protected void ensureCaretPosition() {
        if (getCaretPosition() != currPos) setCaretPosition(currPos);
    }   
    
    /** Invoked when the mouse button has been clicked (pressed
     * and released) on a component.
     *
     */
    public void mouseClicked(MouseEvent e) {
    }
    
    /** Invoked when the mouse enters a component.
     *
     */
    public void mouseEntered(MouseEvent e) {
    }
    
    /** Invoked when the mouse exits a component.
     *
     */
    public void mouseExited(MouseEvent e) {
    }
    
    /** Invoked when a mouse button has been pressed on a component.
     *
     */
    public void mousePressed(MouseEvent e) {
        showMenu(e);
    }
    
    /** Invoked when a mouse button has been released on a component.
     *
     */
    public void mouseReleased(MouseEvent e) {
        showMenu(e);
    }
    
    private void showMenu(MouseEvent e) {
        if (e.isPopupTrigger()) {
            mPopup.show((Component)e.getSource(), e.getX(), e.getY());
        }
    }
    
    /** This method gets called when a bound property is changed.
     * @param evt A PropertyChangeEvent object describing the event source
     *   	and the property that has changed.
     *
     */
    public void propertyChange(PropertyChangeEvent evt) {
        if (evt.getPropertyName().equals("lookAndFeel")) {
            SwingUtilities.updateComponentTreeUI(mPopup);
        }
    }

    //ActionListener functions
    public void addActionListener(ActionListener al) {
        vListeners.add(al);
    }
    
    public void removeActionListener(ActionListener al) {
        vListeners.remove(al);
    }
    
    //TextListener functions
    public void addNameCompletionListener(NameCompletionListener tl) {
        nameListeners.add(tl);
    }
    
    public void removeNameCompletionListener(NameCompletionListener tl) {
        nameListeners.remove(tl);
    }
    
    //Called whenever the namec completion char is pressed
    //Generates TextEvents
    protected void nameCompletionEvent() {
	ArrayList allCompletions = new ArrayList();
	String cmd = getCmd();
	for (int i=0; i<nameListeners.size(); i++) {
            List l = ((NameCompletionListener)nameListeners.get(i)).completeName(cmd);
	    if (l!= null) allCompletions.addAll(l);
	}
	
	//Now do something with the completions
        if (allCompletions.size() == 0) return;          //Didn't find anything
        
        String complete = completeInline(cmd, allCompletions);
	//System.out.println(complete);
        if (allCompletions.size() == 1) {                //Only found 1 thing - complete inline
            complete = complete.substring(cmd.length());
            append(complete);
        } else {                            //Show options to user
            println();
	    append("\n");
            Iterator i = allCompletions.iterator();	    
            while (i.hasNext()) {
                append(i.next() + "\n");
            }
            resetCommandStart();
            append(complete);                        
        }
	//System.out.println("Cmd:" + getCmd());
    }
    
    /**
      Do as much completing as possible inline...
      */
    protected String completeInline(String part, List l) {
        //Trivial for List size 0 or 1.
        if (l.size() == 0) return part;
        if (l.size() == 1) return l.get(0).toString();
        
        //Non-trivial bit.  We know all the strings in l begin with part.
        //Algorithm is to take the first one and then add chars until startsWith on 
        //any of the others return false.
        
        String complete = part;
        String temp;
        boolean retVal = true;
        int index = part.length() + 1;
        while (index <= ((String)l.get(0)).length()) {            
            temp = ((String)l.get(0)).substring(0, index);
            for (int j=1; j<l.size(); j++) {
                retVal &= ((String)l.get(j)).startsWith(temp);                
            }
            if (!retVal) break;
            index++;
            complete = temp;
        }
        
        return complete;
    }

    
    
    //Text functions
    
    /**
     * Prints the prompt and prepairs for input.
     * Should be called by a client if they want to force a new command.
     */
    public void resetCommandStart() {
        //System.out.println("Reset " + lineStartPos + "  " + startPos + "  " + currPos + "  " + doc.getLength());
        int pos = doc.getLength();
        append(sPrompt);
        lineStartPos = pos;
        pos = doc.getLength();        
        append(sPostPrompt);
        startPos = pos;
        currPos = doc.getLength();
        //System.out.println("Reset " + lineStartPos + "  " + startPos + "  " + currPos + "  " + doc.getLength());
    }

    public void append(String string) {
        insert(string, doc.getLength());
    }
    
    private void insert(String string, int pos) {
        String s = (string==null) ? string = "" : string;
        try {
            //System.out.println(lineStartPos + "  " + startPos + "  " + currPos + "  " + string.length() + "  " + doc.getLength());
            doc.insertString(pos, s, null);
            if (pos < lineStartPos) lineStartPos += s.length();
            if (pos < startPos) startPos += s.length();
            //if (pos <= currPos) currPos += s.length();  Not needed - set by caretUpdate
            //System.out.println(lineStartPos + "  " + startPos + "  " + currPos + "  " + string.length() + "  " + doc.getLength());
        } catch(javax.swing.text.BadLocationException ble) {
            //Should never get here, but still
            ble.printStackTrace();
        }    
    }

    String replaceRange(Object s, int start, int end) {
        String st = s.toString();
        select(start, end);
        replaceSelection(st);
        
        return st;
    }
    
    //Command functions
    protected void enter() {
        String s = getCmd();

        if (s.length() == 0) s = "\n";     //Empty return...do we want this???
        else {
            vHistory.addElement(s);
            s = s +"\n";
        }

        append("\n");
        historyPos = 0;
        resetCommandStart(); 
        dispatchEvent(s);                       
    }

    private String getCmd() {
        String s = "";
        try {
            s = getText(startPos, getText().length() - startPos);
        } catch (BadLocationException e) {
            // should not happen
            System.out.println("Internal Error: JConsole getCmd: "+e);
        }
        return s;
    }
    
    protected void dispatchEvent(String cmd) {
        ActionEvent e = new ActionEvent(this, ActionEvent.ACTION_FIRST, cmd);
        for (int i=0; i<vListeners.size(); i++) {
            ((ActionListener)vListeners.get(i)).actionPerformed(e);
        }
    }
    
    
    //Prompt functions
    public String getPrompt() {
        return sPrompt;
    }
    
    public void setPrompt(String prompt) {
        sPrompt = prompt;
    }
    
    public String getPostPrompt() {
        return sPostPrompt;
    }
    
    public void setPostPrompt(String postPrompt) {
        sPostPrompt = postPrompt;
    }
    
    
    
    //History functions
    protected void historyUp() {
        if (vHistory.size() == 0) return;
        if (historyPos == 0) curCmd = getCmd();
        if (historyPos < vHistory.size()) {
            historyPos++;
            showHistoryLine();
        }
    }
    
    protected void historyDown() {
        if (historyPos == 0) return;

        historyPos--;
        showHistoryLine();
    }
    
    protected void showHistoryLine() {
	String line;
        if (historyPos == 0) line = curCmd;
        else line = (String)vHistory.elementAt(vHistory.size() - historyPos);

        replaceRange(line, startPos, getText().length() );
        //moveCaretToEnd();
        repaint();
    }
     
    public String toString() {
        return "JConsole";
    }     
     
    //
    //Millions of print routines
    //
    public void println(String string) {
        print( string + "\n" );       
    }

    public void print(String string) {
        insert(string, lineStartPos );
        //moveCaretToEnd();
    }

    public void println() {
        print("\n");
    }

    public void println(Object object) {
        print(new StringBuffer(String.valueOf(object)).append("\n"));
    }

    public void print(Object object) {
        print(String.valueOf(object));
    }
    
    //Provides a stream interface to the console commands
    class ActionStream extends InputStream implements ActionListener {
        protected JConsole console;
        protected java.util.List list;
        private byte[] current;
        private int pos;
        
        public ActionStream(JConsole console) {
            this.console = console;
            console.addActionListener(this);
            list = Collections.synchronizedList(new LinkedList());
        }
        
        /** Invoked when an action occurs.
         *
         */
        public synchronized void actionPerformed(ActionEvent e) {
            list.add(e.getActionCommand());
        }        
        
        /** Close the stream.  Once a stream has been closed, further read(),
         * ready(), mark(), or reset() invocations will throw an IOException.
         * Closing a previously-closed stream, however, has no effect.
         *
         * @exception  IOException  If an I/O error occurs
         *
         */
        public void close() throws IOException {
            console.removeActionListener(this);
        }
        
       
        /** Reads the next byte of data from the input stream. The value byte is
         * returned as an <code>int</code> in the range <code>0</code> to
         * <code>255</code>. If no byte is available because the end of the stream
         * has been reached, the value <code>-1</code> is returned. This method
         * blocks until input data is available, the end of the stream is detected,
         * or an exception is thrown.
         *
         * <p> A subclass must provide an implementation of this method.
         *
         * @return     the next byte of data, or <code>-1</code> if the end of the
         *             stream is reached.
         * @exception  IOException  if an I/O error occurs.
         *
         */
        public int read() throws IOException {
            if (current == null || pos == current.length) {
                if (list.size() > 0) {
                    current = ((String)list.get(0)).getBytes();
                    pos = 0;
                } else {    //Block until some input!
                    while (list.size() == 0) {
                        try {
                            Thread.currentThread().sleep(100);
                        } catch (java.lang.InterruptedException e){}
                    }
                }
            }
            return current[pos++];
        }
        
    }
}
