#include "Fitter.h"
#include "Sct/SctNames.h"
#include "Sct/IpcObjectException.h"
#include "FitterI.hh"
#include <mrs/message.h>
#include <ipc/core.h>
#include <pmg/pmg_initSync.h>

using namespace Sct;

/** This function returns non-zero if something else is already doing its job:
 * (a) if there is already a server with this name and context running.
 * (b) if any fitting service is subscribing to the same input IS server.
 * @author Alan Barr
 */

int findOther(const char*const partition, const char*const sid, 
	       const char*const cid, const char*const inputServerName) {
    IPCPartition p(partition);
    IPCServerIterator list( p, 0 );

    IPCObjectVar<FitterI_T_FitterInterface> handle;

    while ( list() ){
	if ( list.name()==string(sid) && list.context()==string(cid) ) {
	    std::ostringstream txt;
	    txt << "Server already exists in partition " << partition 
		<< " in context " << list.context() 
		<< " with name " <<  list.name();	    
	    SctNames::Mrs() << MRS_TEXT(txt.str().c_str()) << MRS_INFORMATION << MRS_QUALIF("FITTER")
		<< "PROCESS_DIED" << ENDM;
	    return 1;
	}
	
	handle = p.lookup( list.name(), list.context(), 
			   FitterI_T_FitterInterface::ILUClassRecord, false)  ;
	if ( handle ){
	    FitterIStatus s;
	    s.returnCode=0;
	    if (string( handle->getInputServerName(&s) ) == string (inputServerName) ) {
		std::ostringstream txt;
		txt << "Fitter already exists subscribing to the input server:" 
		    << inputServerName << ":";
		SctNames::Mrs() << "`" << list.name() << ":" << list.context() << "'"
		                << MRS_TEXT(txt.str().c_str()) << MRS_INFORMATION << MRS_QUALIF("FITTER")
                                << "PROCESS_DIED" <<  ENDM;
		return 2;
	    }
	}
    }
    return 0;
}

void pmgSynch(void *) {
    pmg_initSync();
}

/**
 * A main program to generate a FittingService executable.
 * @author Alan Barr, @date 20 Feb 2003.
 * <BR>
 */

int main(int argc, char** argv) throw (){
  using namespace SctFitter;
  using namespace std;

  setExceptionHandlers(argv[0]);
  
  bool multiThreaded = true; //Choose single/multi-threaded
  IPCCore::init(multiThreaded);

  try{
      //Initialize the fitter with the default FitStrategy
      Fitter& f = Fitter::initialize();
      
      // Try the Nag fit strategy first. If this doesn't work 
      // i.e. (NagFitStrategy hasn't compiled and isn't in the factor map), use Root
      try{
	  f.setFitStrategy("NagFitStrategy");
      }
      catch(LogicError& e){
	  //Do nothing - stick with default
      }

      f.setFitOptions("NQR");
      if (!f.publish()){
	  throw IpcObjectException("FittingService failed to publish", __FILE__, __LINE__);
      }

      // launch with one worker
      f.go(1);

      Fitter::getFitterServer().doSoon(pmgSynch, NULL);
      Fitter::getFitterServer().run();
  }
  catch (Throwable& e){e.sendToMrs(MRS_ERROR) ; terminate(); }
}
