#ifndef ARCHIVER_H
#define ARCHIVER_H

#include <is/isinfo.h>
#include <ipc/object.h>
#include <ipc/server.h>

#include <string>
#include <memory>
#include <sys/types.h>
#include <unistd.h>

#include <boost/thread.hpp>
#include "ArchivingServiceI.hh"
#include "Sct/Exception.h"
#include "Sct/ConfigurationException.h"
#include "Sct/LogicErrors.h"
#include "Sct/IoExceptions.h"
#include "Sct/Archive/IOManagerArchive.h"

using namespace Sct;
namespace SctArchiving {
  class ArchivingCommand;
  class ArchivingWorkerGroup;
  /**
   * Class definition of the SCT archiving service.
   * @author Alan Barr
   * @date 10 Feb 2004
   */
  class Archiver: public IPCObject, public virtual ArchivingServiceI_T_ArchivingServiceInterface, private boost::noncopyable{
  private:
    /// singleton
    Archiver() throw(ConfigurationException);
    /// private destructor
    ~Archiver() throw();
    static Archiver* archiver; ///< pointer to singleton used in lazy initialization.
    
  public:
    /**
       Gets a reference to the instance of the Archiver.
       If it hasn't been initialized, it calls initialize() first
    */
    static Archiver& instance();
    
    /**
      Initialize the Archiver instance
      @throws ConfigurationException if
    */
    static Archiver& initialize() throw(ConfigurationException);
    
    /**
     * The command to start the workers. @param nWorker the number of worker threads.
     */
    void go(unsigned nWorker) throw(IsException);
    /**
     * Get the status information
     */
    const char* getStatus() const throw();
    /**
     * As for getStatus, but available over IPC. Makes a new char[] which ipc then deletes.
     * @param the status of the IPC request.
     */
    virtual char* status(ArchivingServiceIStatus* status);
    /** busy method available over IPC */
    virtual ilu_ShortInteger busy(ArchivingServiceIStatus *_status);
     /** queue method available over IPC */
    virtual ilu_ShortInteger queueLength(ArchivingServiceIStatus *_status);
    /** call this to suspend callbacks when retrieving to data servers */
    virtual void suspendCallbacks (ArchivingServiceIStatus *_status, ilu_Boolean doSuspend);
    /**
     * Static function used as callback to archive data
     */
    static void archive_callback(ISCallbackInfo * isc);
    
    /** wrap method to get server */
    IPCServer& getArchiverServer() throw();
    
    /** Archive a IONameIS */
    virtual void archiveISName (ArchivingServiceIStatus *status, char* ioNameIS);
    /** Retrieve a IONameIS */
    virtual void retrieveISName (ArchivingServiceIStatus *status, char* ioNameIS);
    /** Retrieve a IONameArchiveFile*/
    virtual void retrieveArchName (ArchivingServiceIStatus *status, char* archivingName);
    /** Retrieve objects matching a particular file glob pattern */
    virtual void retrieve(ArchivingServiceIStatus *_status, char* runNumber, char* scanNumber, char* className, char* specifier); 
    /** get the gzip compression level */
    virtual ilu_ShortInteger getCompressionLevel (ArchivingServiceIStatus *_status);
    /** set the gzip compression level */
    virtual void setCompressionLevel (ArchivingServiceIStatus *_status, ilu_ShortInteger level);
    /** set the retrieval IS server name */
    virtual void setRetrieveIsServer (ArchivingServiceIStatus *_status, ilu_T_CString serverName);
    /*
      Add an ArchivingCommand to the workergroup.
    */
    virtual void addCommand(boost::shared_ptr<ArchivingCommand> command) const;
    /**
     * Incriment the number of archivals by 1.
     */
    void incrimentNArchived() throw();
    /**
     * Incriment the number of retrievals
     */
    void incrimentNRetrieved() throw();
    /**
     * Incriment the number of validations.
     */
    void incrimentNValidated() throw();
    /**
       Called to add to time for reading/writing to IS.  
    */
    void addISTime(double time) throw();
    /** 
	Called to add to time for reading/writing to archive file.
    */
    void addFileTime(double time) throw();
    /**
       get the archive manager 
    */
    Sct::Archive::IOManagerArchive& Archiver::getIOManagerArchive() const;
    /** get the IS server name to retrieve data to*/
    string getRetrieveIsServer() const throw();
  private:
    /** pointer to its ArchivingWorkerGroup */
    ArchivingWorkerGroup* workergroup;
    /** subscribe to IS, throwing IsExceptions if there are problems */
    void subscribe(const string& servername, const string& regexp, ISCallbackInfo::Callback callback);
    /** pointer to the IS infoReceiver object */
    std::auto_ptr<ISInfoReceiver> infoReceiver;
    /* counters */
    long nArchived;
    long nRetrieved;
    long nValidated;
    double isTimeTaken;
    double fileTimeTaken;
    boost::mutex counterMutex; ///< used to lock access to counters
    double getAverageTime(double time) const throw();
    /** iomanager to use for achiving */
    Sct::Archive::IOManagerArchive* m_archive_manager;

    std::string m_retrieval_is_server;
    bool m_suspend_callback;
}; // end of definition of class Archiver;
 
}// end of namespace SctArchiving
#endif // #ifndef ARCHIVER_H
