#include "AnalysisAlgorithm.h"
#include "SctData/TestResult.h"
#include "SctData/FitScanResult.h"
#include "SctData/RawScanResult.h"
#include "SctData/UniqueID.h"
#include "SctData/ConfigurationVariable.h"
#include "Sct/SctNames.h"
#include "Sct/LogicErrors.h"
#include "Sct/IOName.h"
#include "Sct/IS/IOManagerIS.h"
#include "Sct/ISProxy/IOManagerISProxy.h"
#include "SummaryWriter/SummaryManager.h"
#include "SummaryWriter/SummaryWriter.h"
#include "CalibrationController/IS/TestData.h"
#include "SummaryWriter/TestSummaryIS.h"

#include <boost/date_time/posix_time/posix_time_types.hpp>
#include <sstream>
#include <string>

using namespace std;
using namespace boost;
using namespace Sct;
using namespace SctData;
using namespace Sct::IS;
using namespace boost::posix_time;

namespace SctAnalysis {

AnalysisAlgorithm::AnalysisAlgorithm() : m_test(*new TestData), prototype(*this), ioTime(0), analysisTime(0), 
    overheadTime(0), nCalls(0), nAnalyses(0) {
}

AnalysisAlgorithm::AnalysisAlgorithm(const TestData& testData, const string& moduleName, const AnalysisAlgorithm& prototype) throw()
    : m_modulename(moduleName), m_test(testData), scanData(testData.nScans), m_done(false),  prototype(prototype),
    ioTime(0), analysisTime(0), overheadTime(0), nCalls(0), nAnalyses(0) {
}

AnalysisAlgorithm::~AnalysisAlgorithm() throw() {
    boost::recursive_mutex::scoped_lock lock(m_access);
    //Need to explicitly delete so that lock is in force
    m_result.reset();
}

void AnalysisAlgorithm::checkTestResultExists() {
    if (m_result.get()==0){
	m_result = createTestResult();
	if (m_result.get()==0)
	    throw IllegalStateError("AnalysisAlgorithm cannot initialize - no test exists", __FILE__, __LINE__);
	initializeTestResult();
    }
}

void AnalysisAlgorithm::initializeTestResult() {
    m_result->setRunNumber(m_test.runNumber);
    m_result->setModuleName(m_modulename);
    m_result->setTestVariable(*SctData::ConfigurationVariableIOHelper::getFromTypeRep(m_test.testVariable));
    m_result->getHeader().setStartTimeString(m_test.startTime);

    //Now add all the scans:
    for (unsigned int i=0; i<m_test.testPoints_size; ++i) {
        m_result->addScan(m_test.startScanNumber+i, m_test.testPoints[i]);
        //cout << "Analysis algorithm adding scan number ... "<<m_test->startScanNumber+i<<" to test"<<endl;
    }
}

void AnalysisAlgorithm::addFitScanResult(shared_ptr<Sct::IOName> name) {
    UniqueID id(name->getUniqueID());
    unsigned int index = id.getScanNumber() - getTestData().startScanNumber;
    scanData[index].fitAvailable = true;
    scanData[index].fitName = name->getIOName();
}

void AnalysisAlgorithm::addRawScanResult(shared_ptr<Sct::IOName> name) {
    UniqueID id(name->getUniqueID());
    unsigned int index = id.getScanNumber() - getTestData().startScanNumber;
    scanData[index].rawAvailable = true;
    scanData[index].rawName = name->getIOName();
}

bool AnalysisAlgorithm::hasAllRaws() const {
    for (unsigned int i=0; i<scanData.size(); ++i) {
	if (!scanData[i].rawAvailable) return false;
    }
    return true;
}

bool AnalysisAlgorithm::hasAllFits() const {
    for (unsigned int i=0; i<scanData.size(); ++i) {
	if (!scanData[i].fitAvailable) return false;
    }
    return true;
}

void AnalysisAlgorithm::loadAllRaws() {
    for (unsigned int i=0; i<scanData.size(); ++i) {
	if (scanData[i].rawAvailable && !scanData[i].raw) 
	    scanData[i].raw = dynamic_pointer_cast<RawScanResult>(IOManagerIS::instance().read(scanData[i].rawName, 0));
    }
}


void AnalysisAlgorithm::loadAllFits() {
    for (unsigned int i=0; i<scanData.size(); ++i) {
	if (scanData[i].fitAvailable && !scanData[i].fit) 
	    scanData[i].fit = dynamic_pointer_cast<FitScanResult>(IOManagerIS::instance().read(scanData[i].fitName, 0));
    }
}

shared_ptr<RawScanResult> AnalysisAlgorithm::getRaw(unsigned int index) const {
    if (!scanData[index].rawAvailable || !scanData[index].raw) throw IllegalStateError("AnalysisAlgorithm::getRaw.  RawScanResult not available", __FILE__, __LINE__);
    return scanData[index].raw;
}

shared_ptr<FitScanResult> AnalysisAlgorithm::getFit(unsigned int index) const {
    if (!scanData[index].fitAvailable || !scanData[index].fit) throw IllegalStateError("AnalysisAlgorithm::getFit.  FitScanResult not available", __FILE__, __LINE__);
    return scanData[index].fit;
}

void AnalysisAlgorithm::mergeDefects() {
    if (!m_result) return;
    for (unsigned int i=0; i<scanData.size(); ++i) {
	if (scanData[i].fit) {
	    m_result->getDefects() += scanData[i].fit->getDefects();
	}
    }
}


void AnalysisAlgorithm::finish() {
    checkTestResultExists();
    m_done=true;
    m_result->getHeader().setEndTime(second_clock::universal_time());

    m_result->getPassed() ? addPass() : addFailure();

    cout <<"Algorithm done"<<endl;
    try {
        Sct::IS::IOParamsIS param(Sct::SctNames::getTestDataName());
        Sct::ISProxy::IOManagerISProxy::instance().write(*m_result, &param);
        cout <<"TestResult published"<<endl;
    } catch (Throwable& e) {
        e.sendToMrs(MRS_ERROR);
    }

    try{
        TestSummaryIS t;
        t.dataString = SctData::TestSummary::SummaryManager::instance().write(*m_result);
        string name = SctNames::getTestDataName();
        name += ".";
        name += "Summary.";
        name += m_result->getUniqueID();
        ISInfoDictionary& is = SctNames::getISDictionary();
        ISInfo::Status stat;
        /*ofstream f(name.c_str());
	f << t.dataString;
	f.close();*/
        if (is.contains(name.c_str())) {
        stat = is.update(name.c_str(), t);
        } else {
        stat = is.insert(name.c_str(), t);
        }
    } catch (Throwable& e) {
        e.sendToMrs(MRS_ERROR);
    }
}

AnalysisAlgorithm::ScanData::ScanData() : rawAvailable(false), fitAvailable(false){
}
    
string AnalysisAlgorithm::getStatus() const {
  boost::recursive_mutex::scoped_lock lock(m_status_access);
    ostringstream oss;
    oss << "Status: \n" << nAnalyses << " analyses done in " << analysisTime << "s\n" 
	<< "I/O: " << ioTime << "s.  Overhead time: " << overheadTime << "s and called " << nCalls << " times\n"
	<< "Total time: " << (analysisTime+ioTime+overheadTime) << "s average: " << ((analysisTime+ioTime+overheadTime)/nAnalyses) << endl;
    oss << "Passed : " << prototype.nPassed << ", Failed : " << prototype.nFailed << endl;
    return oss.str();
}
}
