#ifndef SCTAPI_OKS_IMPL_CONFIGURATION
#define SCTAPI_OKS_IMPL_CONFIGURATION

#include <list>
#include <string>
#include <stdexcept>

#include <oks/class.h>
#include <oks/object.h>
#include <oks/kernel.h>

#include "RodDaq/CommonWithDsp/sctStructure.h"

#include "sctConf/configuration.h"

/** 
    The configuration files needed to initialise a ROD slave DSP
 */
/*  struct SlaveConfig { */
/*    std::string emifFile; */
/*    std::string ipramFile; */
/*    std::string idramFile; */
/*    std::string extFile; */
/*  }; */

/**
   Configuration needed for ROD initialisation
 */
/*  struct RodConfig { */
/*    /// Is there a BOC? (Shouldn't be necessary) */
/*    bool bocPresent; */
/*    /// VME base address */
/*    unsigned long baseAddress; */
/*    /// VME map size */
/*    unsigned long mapSize; */
/*    /// Number of slaves */
/*    long numSlaves; */
/*    /// Slave file configuration */
/*    SlaveConfig slaves[4]; */
/*  }; */

/**
   Configuration needed for TIM initialisation
 */
/*  struct TimConfig { */
/*    /// VME base address */
/*    unsigned long baseAddress; */
/*    /// VME map size */
/*    unsigned long mapSize; */
/*  }; */

/**
   Configuration of a "channel" on a BOC

   This is the information associated with one module
   ie. one output channel for primary data and
   two input channels for streams 0 and 1
 */
/*  struct BOCChannelConfig { */
/*    /// Laser current */
/*    int current; */
/*    /// BPM delay  */
/*    int delay; */
/*    /// BPM markspace */
/*    int markSpace; */
/*    /// Input threshold (stream 0) */
/*    int threshold0; */
/*    /// Strobe delay (stream 0) */
/*    int delay0; */
/*    /// Input threshold (stream 1) */
/*    int threshold1; */
/*    /// Strobe delay (stream 1) */
/*    int delay1; */

/*    /// Fibre used for tranimission */
/*    int txFibre; */
/*    /// Fibre used for reception of link 0 */
/*    int rxFibre0; */
/*    /// Fibre used for reception of link 1 */
/*    int rxFibre1; */
/*  }; */

/**
   Exception thrown by public configuration methods
 */
/*  class ConfigurationException : std::exception { */
/*    std::string desc; */
/*   public: */
/*    /// Provide a description on construction */
/*    ConfigurationException(const std::string &str) : desc(str) {} */
/*    /// Override std::exception method */
/*    virtual const char* what () const { return desc.c_str(); } */
/*  }; */

/**
   Interface to XML configuration file.
   This will be replaced by an interface to the DAQ confDB system.

   The primary function of this class is the readout of configuration
   data from the database.

   Translation functions:

   Barrel
   barrel, row,     number
   3 to 6  0 to 55  -1 to -6 and 1 to 6

   Internals:
   position on row given as -1 for -ve numbers and 1 for +ve numbers

   Endcap
   disc     quadrant number 
   0 to 17  0 to 3   0 to 33

   Internals:
   position of MUR in quadrant
   number is left to right top to bottom 
      00000012222223
        4444455555
        1111133333
*/
class OksImpl : public SctConfiguration::Configuration {
  OksImpl(const OksImpl &);
  OksImpl &operator=(const OksImpl &);
public:
  /**
     Initialise system
  */
  OksImpl(bool readOnly = true);
  /**
     Free resources
  */
  ~OksImpl();

  /************* Get physical configuration ****************/

  /**
     List partition ids
     Lookup the partitions defined in the configuration and return a list of integer ID's
  */
  std::list<unsigned int> listPartitions();

  /**
     List crate ids with reference to the given partition
     Lookup the crates defined in the specified partition and return a list of integer ID's
  */
  std::list<unsigned int> listCratesInPartition(unsigned int partition);

  /**
     List the rods present in a crate
     Lookup the RODs defined in the specified crate and return a list of integer ID's
  */
  std::list<unsigned int> listRodsInCrate(unsigned int partition, unsigned int crate);

  /**
     List the MURs associated with a ROD
     Return integer id's of the MURs in the specified rod.

     FIXME:  Should be in numbered order
  */
  std::list<unsigned int> listMURSInRod(unsigned int partition, unsigned int crate, unsigned int rod);

  /**
     List the modules in the given MUR
     Return a list of serial numbers associated with the specified MUR

     FIXME:  Should be in numbered order
  */
  std::list<std::string> listModulesInMUR(unsigned int partition, unsigned int MUR);

  /**
     List the modules in the given MUR
     Return a list of serial numbers associated with the redundancy links of the specified MUR

     FIXME:  Should be in numbered order (array?)
  */
  std::list<std::string> listRModulesInMUR(unsigned int partition, unsigned int MUR);

  /**
     Return a list of serial numbers of all modules that have configuration

     This includes those that are not attached to a ROD or MUR
  */
  std::list<std::string> listAllModules();

  /**
     Return a list of serial numbers of modules that have configuration but are not attached to an MUR.
  */
  std::list<std::string> listUnusedModules();

  /************** Get configurations *****************/

  /**
     Return the configuration for the module

     Deprecated: Should take MUR argument
  */
  ABCDModule getModuleConfig(const std::string module);

  /**
     Return the Rod configuration
     Find the configuration associated with the specified ROD.
     Return a RodConfig structure.
  */
  SctConfiguration::RodConfig getRodConfig(unsigned int partition, unsigned int crate, unsigned int rod);

  /**
     Get the BOC configuration of one of 48 channels 

     Input channels are paired as one.
  */
  SctConfiguration::BOCChannelConfig getBOCConfig(unsigned int partition, unsigned int crate, unsigned int rod, 
                                                  unsigned int channel);

  /**
     Get the BOC fibre mappings

     An array of 48*3 chars (out, in0, in1)
  */
  char *getFibreMappings(unsigned int partition, unsigned int crate, unsigned int rod);

  /**
     Return the TIM configuration
     Find the configuration associated with the specified TIM.
     Return a TimConfig structure.
  */
  SctConfiguration::TimConfig getTimConfig(unsigned int partition, unsigned int crate);

  /**
     Set fibre mappings for one channel

     Any value over 48/96 leaves current value unchanged
   */
  void setFibreMapping(unsigned int partition, unsigned int crate, unsigned int rod, 
                       unsigned int channel, unsigned int tx, unsigned int rx0, unsigned int rx1);

  /**
     Print module configuration to cout
  */
  void printModuleConfig(const ABCDModule &conf);

  SctConfiguration::MURType getMURType(unsigned int MUR);

  /************** Change indexing schemes ****************/

  /**
     Translate from MUR / module(1-6) number to p/k/r/c
  */
  void translateToROD(unsigned int MUR, unsigned int module,
		      unsigned int &partition, unsigned int &crate, 
		      unsigned int &rod, unsigned int &channel);
 
  /**
     Translate from MUR / redundant module(1-6) number to p/k/r/c
  */
  void translateToRROD(unsigned int MUR, unsigned int module,
		       unsigned int &partition, unsigned int &crate, 
		       unsigned int &rod, unsigned int &channel);
 

  /**
     Translate from MUR / module(1-6) number to serial number
  */
  void translateToSN(unsigned int MUR, unsigned int module,
                     std::string &sn);

  /**
     Translate from MUR / module(1-6) number to geographical barrel system
  */
  void translateToBarrel(unsigned int MUR, unsigned int module,
                         unsigned int &barrel, unsigned int &row, int &number);

  /**
     Translate from MUR / module(1-6) number to geographical endcap system
   */
  void translateToEndcap(unsigned int MUR, unsigned int module,
                         unsigned int &disk, unsigned int &ring, unsigned int &number);

  /**
     Translate from MUR / module(1-6) number to power supply channel address
   */
  void translateToPowerSupply(unsigned int MUR, unsigned int module,
                              unsigned int &partition, 
                              unsigned int &crate, unsigned int &channel);

  /**
     Translate to MUR / module(1-6) number from p/k/r/c
  */
  void translateFromROD(unsigned int partition, unsigned int crate, 
                        unsigned int rod, unsigned int channel,
                        unsigned int &MUR, unsigned int &module);

  /**
     Translate to MUR / module(1-6) number from address of redundant module

     This returns the MUR/module of the module connected using its
     redundant link to the p/k/r/c address
  */
  void translateFromRROD(unsigned int partition, unsigned int crate,
                         unsigned int rod, unsigned int channel,
                         unsigned int &MUR, unsigned int &module);

  /**
     Translate to MUR / module(1-6) number from serial number
  */
  void translateFromSN(const std::string sn,
                       unsigned int &MUR, unsigned int &module);

  /**
     Translate to MUR / module(1-6) number from geographical barrel system
  */
  void translateFromBarrel(unsigned int barrel, unsigned int row, int number,
                         unsigned int &MUR, unsigned int &module);

  /**
     Translate to MUR / module(1-6) number from geographical endcap system
   */
  void translateFromEndcap(unsigned int disk, unsigned int ring, unsigned int number,
                           unsigned int &MUR, unsigned int &module);

  /**
     Translate from power supply channel address to MUR / module(1-6) number 
   */
  void translateFromPowerSupply(unsigned int partition, 
                                unsigned int crate, unsigned int channel, 
                                unsigned int &MUR, unsigned int &module);


  /************** Change configuration ******************/

  /**
     Update or add the configuration of a module stored in memory
     Replace the configuration for the given module with the data
     stored in the given configuration
  */
  void configureModuleFromStructure(const std::string module, const ABCDModule conf);

  /**
     Update or add the configuration of a module stored in memory
     Replace the configuration for the given module with the data
     stored in the given file (the serial number is read from the file)
  */
  void configureModuleFromFile(const std::string filename);

  /**
     Configure a ROD

     Replace the in memory configuration for the given ROD
     with the configuration provided.

     If no configuration then construct it
   */
  void configureROD(unsigned int partition, unsigned int crate, unsigned int rod, SctConfiguration::RodConfig conf);

  /**
     Configure a TIM

     Replace the in memory configuration for the given TIM
     with the configuration provided.

     If no configuration then construct it
   */
  void configureTIM(unsigned int partition, unsigned int crate, SctConfiguration::TimConfig conf);

  /**
     Remove all configuration.
     Very dangerous!!
   */
  void clearAll();

  /**
     Configuration creation:

     Create module (write configuration)

     Change translations

     Change MUR translations
   */

  /** 
     Name a partition (for information, any use?)
   */
  void namePartition(int partition, const std::string name);

  /**
     Create a partition 
  */
  void createPartition(int partition);

  /**
     Create a crate 
  */
  void createCrate(int partition, int crate);

  /**
     Create a ROD
   */
  void createROD(int partition, int crate, int rod, SctConfiguration::RodConfig conf);

  /** 
     Create an MUR
  */
  void createMUR(int partition, int crate, int rod, int order, int number);

  /**
     Create a module
  */
  void createModule(int MUR, int order, int RMUR, int rorder, string number, const ABCDModule &mConf);

  /**
     Set a BOC configuration
  */
  void configureBOCChannel(int MUR, int position, const SctConfiguration::BOCChannelConfig &bConf);

  /**
     Map a module (by serial number) onto a position in an MUR
  */
  void mapModuleMUR(int MUR, int order, int RMUR, int rorder, std::string number);

  /** 
     Map an MUR
  */
  void mapRODMUR(int partition, int crate, int rod, int order, int number);

  /**
     Map Barrel MUR
  */
  void mapBarrelMUR(int MUR, int barrel, int row, int position);

  /**
     Swap two MUR names around

     This means in regards to everything about one MUR you can use the other name
     If either doesn't exist then just one MUR is renamed
  */
  void swapMURNames(int MUR1, int MUR2);

  /**
     Map Endcap MUR
  */
  void mapEndcapMUR(int MUR, int disk, int quadrant, int position);

  /**
     Map Power channel
  */
  void mapPowerChannel(int MUR, int number, unsigned int partition, unsigned int crate, unsigned int channel);

  /**
     Set power supply parameter
  */
  void modifyPowerParam(int MUR, int number, std::string name, float value);

  /**
     Set default power supply parameter
  */
  void modifyDefaultPowerParam(std::string name, float value);

  /**
     Get power supply parameter
  */
  float getPowerParam(int MUR, int number, std::string name);

  /** 
     Load configuration from named file
   */
  void loadConfiguration(const std::string &filename);

  /************** Save configuration *********************/

  /**
     Save the current configuration

     Write configuration to disc.
     This creates a unique filename based on the date and time
     and saves the current configuration to it.

     Filename eg Conf_dump_20021210142311.xml.gz (in current directory)
  */
  void saveConfiguration(std::string filename = "");

  /**
     Save the current configuration for a module

     Write the configuration for the module modsn to disc.
  */
  void saveModuleConfiguration(const std::string modsn, const std::string filename);

  /**
     Construct the XML for a module configuration and return as a string
  */
  std::string getModuleConfigurationString(const std::string modsn, const ABCDModule config);

  /**
     Dump power supply configuration to a file
  */
  void writePowerSupplyConfiguration(std::string filename);

 private:
/*   /\** */
/*      Parse the xml tree provided and return the module configuration */
/*   *\/ */
/*   const ABCDModule parseModuleConfig(xmlNodePtr node); */

/*   /\** */
/*      Replace the configuration in the given node with the data */
/*      stored in the given configuration */
/*   *\/ */
/*   void replaceModuleConfig(xmlNodePtr node, ABCDModule conf, std::string name); */

 private:
  OksObject *getPartition(unsigned int p);
  OksObject *getCrate(unsigned int p, unsigned int c);
  OksObject *getRod(unsigned int p, unsigned int c, unsigned int r);
  OksObject *getMUR(unsigned int m);
  OksObject *getModule(unsigned short mur, unsigned char position);

  OksObject *getObjectFromRelationship(const OksObject *obj, const std::string rel);
  unsigned long getULongData(const OksObject *obj, const std::string attr);
  unsigned short getUShortData(const OksObject *obj, const std::string attr);
  unsigned char getUCharData(const OksObject *obj, const std::string attr);
  std::string getStringData(const OksObject *obj, const std::string attr);

  OksClass *partClass;
  OksClass *crateClass;
  OksClass *rodClass;
  OksClass *murClass;
  OksClass *moduleClass;

/*    ConfdbConfiguration configuration; */

  OksKernel kernel;
};
#endif
